#ifndef _ELM_CODE_WIDGET_EO_H_
#define _ELM_CODE_WIDGET_EO_H_

#ifndef _ELM_CODE_WIDGET_EO_CLASS_TYPE
#define _ELM_CODE_WIDGET_EO_CLASS_TYPE

typedef Eo Elm_Code_Widget;

#endif

#ifndef _ELM_CODE_WIDGET_EO_TYPES
#define _ELM_CODE_WIDGET_EO_TYPES


#endif
#define ELM_CODE_WIDGET_CLASS elm_code_widget_class_get()

EAPI const Eo_Class *elm_code_widget_class_get(void) EINA_CONST;

/**
 * @brief Set the underlying code object that this widget renders. This can
 * only be set during construction, once the widget is created the backing code
 * object cannot be changed.
 *
 * @param[in] code Our underlying Elm_Code object
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI void elm_obj_code_widget_code_set(Elm_Code *code);

/**
 * @brief Get the underlying code object we are rendering
 *
 * @return Our underlying Elm_Code object
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI Elm_Code *elm_obj_code_widget_code_get(void);

/**
 * @brief Set the font that this widget uses, the font should be a monospaced
 * scalable font. Passing NULL will load the default system monospaced font.
 *
 * @param[in] name The name of the font to load
 * @param[in] size The font size for the widget
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI void elm_obj_code_widget_font_set(const char *name, Evas_Font_Size size);

/**
 * @brief Get the font currently in use. The font name is a copy ad should be
 * freed once it is no longer needed
 *
 * @param[out] name The name of the font to load
 * @param[out] size The font size for the widget
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI void elm_obj_code_widget_font_get(const char **name, Evas_Font_Size *size);

/**
 * @brief Set how this widget's scroller should respond to new lines being
 * added.
 *
 * An x value of 0.0 will maintain the distance from the left edge, 1.0 will
 * ensure the rightmost edge (of the longest line) is respected With 0.0 for y
 * the view will keep it's position relative to the top whereas 1.0 will scroll
 * downward as lines are added.
 *
 * @param[in] x The horizontal value of the scroller gravity - valid values are
 * 0.0 and 1.0
 * @param[in] y The vertical gravity of the widget's scroller - valid values
 * are 0.0 and 1.0
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI void elm_obj_code_widget_gravity_set(double x, double y);

/**
 * @brief Get the current x and y gravity of the widget's scroller
 *
 * @param[out] x The horizontal value of the scroller gravity - valid values
 * are 0.0 and 1.0
 * @param[out] y The vertical gravity of the widget's scroller - valid values
 * are 0.0 and 1.0
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI void elm_obj_code_widget_gravity_get(double *x, double *y);

/**
 * @brief Set the policy for scrollbar visibility.
 *
 * @param[in] policy_h The horizontal scrollbar visibility policy
 * @param[in] policy_v The vertical scrollbar visibility policy
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI void elm_obj_code_widget_policy_set(Elm_Scroller_Policy policy_h, Elm_Scroller_Policy policy_v);

/**
 * @brief Get the widget's policy for scrollbar visibility.
 *
 * @param[out] policy_h The horizontal scrollbar visibility policy
 * @param[out] policy_v The vertical scrollbar visibility policy
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI void elm_obj_code_widget_policy_get(Elm_Scroller_Policy *policy_h, Elm_Scroller_Policy *policy_v);

/**
 * @brief Set the width of a tab stop, used purely for visual layout of tab
 * characters.
 *
 * Recommended value is between 2 and 8.
 *
 * @param[in] tabstop Maximum width of a tab character
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI void elm_obj_code_widget_tabstop_set(unsigned int tabstop);

/**
 * @brief Get the current width of a tab stop. This is used to determine where
 * characters after a tab should appear in the line.
 *
 * @return Maximum width of a tab character
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI unsigned int elm_obj_code_widget_tabstop_get(void);

/**
 * @brief Set whether this widget allows editing
 *
 * If editable then the widget will allow user input to manipulate the
 * underlying Elm_Code_File of this Elm_Code instance. Any other
 * Elm_Code_Widget's connected to this Elm_Code will update to reflect the
 * changes.
 *
 * @param[in] editable The editable state of the widget
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI void elm_obj_code_widget_editable_set(Eina_Bool editable);

/**
 * @brief Get the current editable state of this widget
 *
 * returns EINA_TRUE if the widget is editable, EINA_FALSE otherwise. If this
 * widget is not editable the underlying Elm_Code_File could still be
 * manipulated by a different widget or the filesystem.
 *
 * @return The editable state of the widget
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI Eina_Bool elm_obj_code_widget_editable_get(void);

/**
 * @brief Set whether line numbers should be displayed in the left gutter.
 *
 * Passing EINA_TRUE will reserve a space for showing line numbers, EINA_FALSE
 * will turn this off.
 *
 * @param[in] line_numbers Whether or not line numbers (or their placeholder)
 * should be shown
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI void elm_obj_code_widget_line_numbers_set(Eina_Bool line_numbers);

/**
 * @brief Get the status of line number display for this widget.
 *
 * @return Whether or not line numbers (or their placeholder) should be shown
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI Eina_Bool elm_obj_code_widget_line_numbers_get(void);

/**
 * @brief Set where the line width market should be shown.
 *
 * Passing a non-zero value will set which line width to mark with a vertical
 * line. Passing 0 will hide this marker.
 *
 * @param[in] line_width_marker Where to display a line width marker, if at all
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI void elm_obj_code_widget_line_width_marker_set(unsigned int line_width_marker);

/**
 * @brief Get the position of the line width marker, any positive return
 * indicates where the marker appears.
 *
 * @return Where to display a line width marker, if at all
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI unsigned int elm_obj_code_widget_line_width_marker_get(void);

/**
 * @brief Set where white space should be shown.
 *
 * @param[in] show_whitespace Whether or not we show whitespace characters
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI void elm_obj_code_widget_show_whitespace_set(Eina_Bool show_whitespace);

/**
 * @brief Get whether or not white space will be visible.
 *
 * @return Whether or not we show whitespace characters
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI Eina_Bool elm_obj_code_widget_show_whitespace_get(void);

/**
 * @brief Set whether space characters should be inserted instead of tabs.
 *
 * @param[in] tab_inserts_spaces EINA_TRUE if we should insert space characters
 * instead of a tab when the Tab key is pressed
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI void elm_obj_code_widget_tab_inserts_spaces_set(Eina_Bool tab_inserts_spaces);

/**
 * @brief Get whether or not space characters will be inserted instead of tabs.
 *
 * @return EINA_TRUE if we should insert space characters instead of a tab when
 * the Tab key is pressed
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI Eina_Bool elm_obj_code_widget_tab_inserts_spaces_get(void);

/**
 * @brief Set the current location of the text cursor.
 *
 * @param[in] col The horizontal position of the cursor, starting from column 1
 * @param[in] line The vertical position of the cursor - the top row is 1
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI void elm_obj_code_widget_cursor_position_set(unsigned int col, unsigned int line);

/**
 * @brief Get the current x and y position of the widget's cursor.
 *
 * @param[out] col The horizontal position of the cursor, starting from column
 * 1
 * @param[out] line The vertical position of the cursor - the top row is 1
 *
 * @ingroup Elm_Code_Widget
 */
EOAPI void elm_obj_code_widget_cursor_position_get(unsigned int *col, unsigned int *line);

EOAPI void elm_obj_code_widget_line_refresh(Elm_Code_Line *line);

EOAPI Eina_Bool elm_obj_code_widget_line_visible_get(Elm_Code_Line *line);

EOAPI unsigned int elm_obj_code_widget_lines_visible_get(void);

EOAPI int elm_obj_code_widget_text_left_gutter_width_get(void);

EOAPI int elm_obj_code_widget_text_line_number_width_get(void);

EOAPI char *elm_obj_code_widget_text_between_positions_get(unsigned int start_col, unsigned int start_line, unsigned int end_col, unsigned int end_line);

EOAPI unsigned int elm_obj_code_widget_line_text_column_width_to_position(Elm_Code_Line *line, unsigned int position);

EOAPI unsigned int elm_obj_code_widget_line_text_column_width_get(Elm_Code_Line *line);

EOAPI unsigned int elm_obj_code_widget_line_text_position_for_column_get(Elm_Code_Line *line, unsigned int column);

EOAPI unsigned int elm_obj_code_widget_text_tabwidth_at_column_get(unsigned int column);

EOAPI void elm_obj_code_widget_undo(void);

EOAPI extern const Eo_Event_Description _ELM_CODE_WIDGET_EVENT_LINE_CLICKED;
EOAPI extern const Eo_Event_Description _ELM_CODE_WIDGET_EVENT_LINE_GUTTER_CLICKED;
EOAPI extern const Eo_Event_Description _ELM_CODE_WIDGET_EVENT_CURSOR_CHANGED;
EOAPI extern const Eo_Event_Description _ELM_CODE_WIDGET_EVENT_CHANGED_USER;
EOAPI extern const Eo_Event_Description _ELM_CODE_WIDGET_EVENT_SELECTION_CHANGED;
EOAPI extern const Eo_Event_Description _ELM_CODE_WIDGET_EVENT_SELECTION_CLEARED;

/**
 * No description
 */
#define ELM_CODE_WIDGET_EVENT_LINE_CLICKED (&(_ELM_CODE_WIDGET_EVENT_LINE_CLICKED))

/**
 * No description
 */
#define ELM_CODE_WIDGET_EVENT_LINE_GUTTER_CLICKED (&(_ELM_CODE_WIDGET_EVENT_LINE_GUTTER_CLICKED))

/**
 * No description
 */
#define ELM_CODE_WIDGET_EVENT_CURSOR_CHANGED (&(_ELM_CODE_WIDGET_EVENT_CURSOR_CHANGED))

/**
 * No description
 */
#define ELM_CODE_WIDGET_EVENT_CHANGED_USER (&(_ELM_CODE_WIDGET_EVENT_CHANGED_USER))

/**
 * No description
 */
#define ELM_CODE_WIDGET_EVENT_SELECTION_CHANGED (&(_ELM_CODE_WIDGET_EVENT_SELECTION_CHANGED))

/**
 * No description
 */
#define ELM_CODE_WIDGET_EVENT_SELECTION_CLEARED (&(_ELM_CODE_WIDGET_EVENT_SELECTION_CLEARED))

#endif
