/*
 * Decompiled with CFR 0.152.
 */
package git4idea.repo;

import com.intellij.dvcs.DvcsUtil;
import com.intellij.dvcs.repo.RepoStateException;
import com.intellij.dvcs.repo.Repository;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.util.Condition;
import com.intellij.openapi.util.Pair;
import com.intellij.openapi.util.io.FileUtil;
import com.intellij.openapi.util.text.LineTokenizer;
import com.intellij.util.Function;
import com.intellij.util.Processor;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.vcs.log.Hash;
import com.intellij.vcs.log.impl.HashImpl;
import git4idea.GitBranch;
import git4idea.GitLocalBranch;
import git4idea.GitRemoteBranch;
import git4idea.GitStandardRemoteBranch;
import git4idea.GitSvnRemoteBranch;
import git4idea.GitUtil;
import git4idea.branch.GitBranchUtil;
import git4idea.repo.GitBranchState;
import git4idea.repo.GitRemote;
import java.io.File;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

class GitRepositoryReader {
    private static final Logger LOG = Logger.getInstance(GitRepositoryReader.class);
    private static final Processor<File> NOT_HIDDEN_DIRECTORIES = new Processor<File>(){

        public boolean process(File dir) {
            return !GitRepositoryReader.isHidden(dir);
        }
    };
    private static Pattern BRANCH_PATTERN = Pattern.compile(" *(?:ref:)? */?((?:refs/heads/|refs/remotes/)?\\S+)");
    private static final String REFS_HEADS_PREFIX = "refs/heads/";
    private static final String REFS_REMOTES_PREFIX = "refs/remotes/";
    private final File myGitDir;
    private final File myHeadFile;
    private final File myRefsHeadsDir;
    private final File myRefsRemotesDir;
    private final File myPackedRefsFile;

    GitRepositoryReader(File gitDir) {
        this.myGitDir = gitDir;
        DvcsUtil.assertFileExists((File)this.myGitDir, (String)(".git directory not found in " + gitDir));
        this.myHeadFile = new File(this.myGitDir, "HEAD");
        DvcsUtil.assertFileExists((File)this.myHeadFile, (String)(".git/HEAD file not found in " + gitDir));
        this.myRefsHeadsDir = new File(new File(this.myGitDir, "refs"), "heads");
        this.myRefsRemotesDir = new File(new File(this.myGitDir, "refs"), "remotes");
        this.myPackedRefsFile = new File(this.myGitDir, "packed-refs");
    }

    GitBranchState readState(Collection<GitRemote> remotes) {
        String currentRevision;
        GitLocalBranch currentBranch;
        Pair<Set<GitLocalBranch>, Set<GitRemoteBranch>> branches = this.readBranches(remotes);
        Set localBranches = (Set)branches.first;
        HeadInfo headInfo = this.readHead();
        Repository.State state = this.readRepositoryState(headInfo);
        if (!headInfo.isBranch) {
            currentBranch = null;
            currentRevision = headInfo.content;
        } else if (!localBranches.isEmpty()) {
            currentBranch = this.findCurrentBranch(headInfo, state, localBranches);
            currentRevision = GitRepositoryReader.getCurrentRevision(headInfo, currentBranch);
        } else if (headInfo.content != null) {
            currentBranch = new GitLocalBranch(headInfo.content, GitBranch.DUMMY_HASH);
            currentRevision = null;
        } else {
            currentBranch = null;
            currentRevision = null;
        }
        return new GitBranchState(currentRevision, currentBranch, state, localBranches, (Collection)branches.second);
    }

    private static String getCurrentRevision(HeadInfo headInfo, GitLocalBranch currentBranch) {
        Object currentRevision = !headInfo.isBranch ? headInfo.content : (currentBranch == null ? null : currentBranch.getHash().asString());
        return currentRevision;
    }

    private GitLocalBranch findCurrentBranch(HeadInfo headInfo, Repository.State state, Set<GitLocalBranch> localBranches) {
        final String currentBranchName = this.findCurrentBranchName(state, headInfo);
        if (currentBranchName == null) {
            return null;
        }
        return (GitLocalBranch)ContainerUtil.find(localBranches, (Condition)new Condition<GitLocalBranch>(){

            public boolean value(GitLocalBranch branch) {
                return branch.getFullName().equals(currentBranchName);
            }
        });
    }

    private Repository.State readRepositoryState(HeadInfo headInfo) {
        if (this.isMergeInProgress()) {
            return Repository.State.MERGING;
        }
        if (this.isRebaseInProgress()) {
            return Repository.State.REBASING;
        }
        if (!headInfo.isBranch) {
            return Repository.State.DETACHED;
        }
        return Repository.State.NORMAL;
    }

    private String findCurrentBranchName(Repository.State state, HeadInfo headInfo) {
        String currentBranch = null;
        if (headInfo.isBranch) {
            currentBranch = headInfo.content;
        } else if (state == Repository.State.REBASING && (currentBranch = this.readRebaseDirBranchFile("rebase-apply")) == null) {
            currentBranch = this.readRebaseDirBranchFile("rebase-merge");
        }
        return GitRepositoryReader.addRefsHeadsPrefixIfNeeded(currentBranch);
    }

    private String readRebaseDirBranchFile(String rebaseDirName) {
        File headName;
        File rebaseDir = new File(this.myGitDir, rebaseDirName);
        if (rebaseDir.exists() && (headName = new File(rebaseDir, "head-name")).exists()) {
            return DvcsUtil.tryLoadFileOrReturn((File)headName, null);
        }
        return null;
    }

    private static String addRefsHeadsPrefixIfNeeded(String branchName) {
        if (branchName != null && !branchName.startsWith(REFS_HEADS_PREFIX)) {
            return REFS_HEADS_PREFIX + branchName;
        }
        return branchName;
    }

    private boolean isMergeInProgress() {
        File mergeHead = new File(this.myGitDir, "MERGE_HEAD");
        return mergeHead.exists();
    }

    private boolean isRebaseInProgress() {
        File f = new File(this.myGitDir, "rebase-apply");
        if (f.exists()) {
            return true;
        }
        f = new File(this.myGitDir, "rebase-merge");
        return f.exists();
    }

    private Map<String, String> readPackedBranches() {
        if (!this.myPackedRefsFile.exists()) {
            return Collections.emptyMap();
        }
        try {
            String content = DvcsUtil.tryLoadFile((File)this.myPackedRefsFile);
            return ContainerUtil.map2MapNotNull((Object[])LineTokenizer.tokenize((CharSequence)content, (boolean)false), (Function)new Function<String, Pair<String, String>>(){

                public Pair<String, String> fun(String line) {
                    return GitRepositoryReader.parsePackedRefsLine(line);
                }
            });
        }
        catch (RepoStateException e) {
            return Collections.emptyMap();
        }
    }

    private Pair<Set<GitLocalBranch>, Set<GitRemoteBranch>> readBranches(Collection<GitRemote> remotes) {
        Map<String, String> data = this.readBranchRefsFromFiles();
        Map<String, Hash> resolvedRefs = GitRepositoryReader.resolveRefs(data);
        return GitRepositoryReader.createBranchesFromData(remotes, resolvedRefs);
    }

    private Map<String, String> readBranchRefsFromFiles() {
        HashMap result = ContainerUtil.newHashMap(this.readPackedBranches());
        result.putAll(this.readFromBranchFiles(this.myRefsHeadsDir));
        result.putAll(this.readFromBranchFiles(this.myRefsRemotesDir));
        result.remove("refs/remotes/origin/HEAD");
        return result;
    }

    private static Pair<Set<GitLocalBranch>, Set<GitRemoteBranch>> createBranchesFromData(Collection<GitRemote> remotes, Map<String, Hash> data) {
        HashSet localBranches = ContainerUtil.newHashSet();
        HashSet remoteBranches = ContainerUtil.newHashSet();
        for (Map.Entry<String, Hash> entry : data.entrySet()) {
            String refName = entry.getKey();
            Hash hash = entry.getValue();
            if (refName.startsWith(REFS_HEADS_PREFIX)) {
                localBranches.add(new GitLocalBranch(refName, hash));
                continue;
            }
            if (refName.startsWith(REFS_REMOTES_PREFIX)) {
                GitRemoteBranch remoteBranch = GitRepositoryReader.parseRemoteBranch(refName, hash, remotes);
                if (remoteBranch == null) continue;
                remoteBranches.add(remoteBranch);
                continue;
            }
            LOG.warn("Unexpected ref format: " + refName);
        }
        return Pair.create((Object)localBranches, (Object)remoteBranches);
    }

    private static String loadHashFromBranchFile(File branchFile) {
        return DvcsUtil.tryLoadFileOrReturn((File)branchFile, null);
    }

    private Map<String, String> readFromBranchFiles(File rootDir) {
        if (!rootDir.exists()) {
            return Collections.emptyMap();
        }
        final HashMap<String, String> result = new HashMap<String, String>();
        FileUtil.processFilesRecursively((File)rootDir, (Processor)new Processor<File>(){

            public boolean process(File file) {
                String relativePath;
                if (!file.isDirectory() && !GitRepositoryReader.isHidden(file) && (relativePath = FileUtil.getRelativePath((File)GitRepositoryReader.this.myGitDir, (File)file)) != null) {
                    String branchName = FileUtil.toSystemIndependentName((String)relativePath);
                    String hash = GitRepositoryReader.loadHashFromBranchFile(file);
                    if (hash != null) {
                        result.put(branchName, hash);
                    }
                }
                return true;
            }
        }, NOT_HIDDEN_DIRECTORIES);
        return result;
    }

    private static boolean isHidden(File file) {
        return file.getName().startsWith(".");
    }

    private static GitRemoteBranch parseRemoteBranch(String fullBranchName, Hash hash, Collection<GitRemote> remotes) {
        String stdName = GitBranchUtil.stripRefsPrefix(fullBranchName);
        int slash = stdName.indexOf(47);
        if (slash == -1) {
            return new GitSvnRemoteBranch(fullBranchName, hash);
        }
        String remoteName = stdName.substring(0, slash);
        String branchName = stdName.substring(slash + 1);
        GitRemote remote = GitUtil.findRemoteByName(remotes, remoteName);
        if (remote == null) {
            LOG.debug(String.format("No remote found with the name [%s]. All remotes: %s", remoteName, remotes));
            GitRemote fakeRemote = new GitRemote(remoteName, ContainerUtil.emptyList(), Collections.<String>emptyList(), Collections.<String>emptyList(), Collections.<String>emptyList());
            return new GitStandardRemoteBranch(fakeRemote, branchName, hash);
        }
        return new GitStandardRemoteBranch(remote, branchName, hash);
    }

    private HeadInfo readHead() {
        String headContent;
        try {
            headContent = DvcsUtil.tryLoadFile((File)this.myHeadFile);
        }
        catch (RepoStateException e) {
            LOG.error((Throwable)e);
            return new HeadInfo(false, null);
        }
        Hash hash = GitRepositoryReader.parseHash(headContent);
        if (hash != null) {
            return new HeadInfo(false, headContent);
        }
        String target = GitRepositoryReader.getTarget(headContent);
        if (target != null) {
            return new HeadInfo(true, target);
        }
        LOG.error((Throwable)new RepoStateException("Invalid format of the .git/HEAD file: [" + headContent + "]"));
        return new HeadInfo(false, null);
    }

    private static Pair<String, String> parsePackedRefsLine(String line) {
        int i;
        if ((line = line.trim()).isEmpty()) {
            return null;
        }
        char firstChar = line.charAt(0);
        if (firstChar == '#') {
            return null;
        }
        if (firstChar == '^') {
            return null;
        }
        String hash = null;
        for (i = 0; i < line.length(); ++i) {
            char c = line.charAt(i);
            if (Character.isLetterOrDigit(c)) continue;
            hash = line.substring(0, i);
            break;
        }
        if (hash == null) {
            LOG.warn("Ignoring invalid packed-refs line: [" + line + "]");
            return null;
        }
        String branch = null;
        int start = i;
        if (start < line.length() && line.charAt(start++) == ' ') {
            char c;
            for (i = start; i < line.length() && !Character.isWhitespace(c = line.charAt(i)); ++i) {
            }
            branch = line.substring(start, i);
        }
        if (branch == null || !branch.startsWith(REFS_HEADS_PREFIX) && !branch.startsWith(REFS_REMOTES_PREFIX)) {
            return null;
        }
        return Pair.create((Object)GitRepositoryReader.shortBuffer(branch), (Object)GitRepositoryReader.shortBuffer(hash.trim()));
    }

    private static String shortBuffer(String raw) {
        return new String(raw);
    }

    private static Map<String, Hash> resolveRefs(Map<String, String> data) {
        final Map<String, Hash> resolved = GitRepositoryReader.getResolvedHashes(data);
        Map unresolved = ContainerUtil.filter(data, (Condition)new Condition<String>(){

            public boolean value(String refName) {
                return !resolved.containsKey(refName);
            }
        });
        boolean progressed = true;
        while (progressed && !unresolved.isEmpty()) {
            progressed = false;
            Iterator iterator = unresolved.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry entry = iterator.next();
                String refName = (String)entry.getKey();
                String refValue = (String)entry.getValue();
                String link = GitRepositoryReader.getTarget(refValue);
                if (link != null) {
                    if (GitRepositoryReader.duplicateEntry(resolved, refName, refValue)) {
                        iterator.remove();
                        continue;
                    }
                    if (!resolved.containsKey(link)) {
                        LOG.debug("Unresolved symbolic link [" + refName + "] pointing to [" + refValue + "]");
                        continue;
                    }
                    Hash targetValue = resolved.get(link);
                    resolved.put(refName, targetValue);
                    iterator.remove();
                    progressed = true;
                    continue;
                }
                LOG.warn("Unexpected record [" + refName + "] -> [" + refValue + "]");
                iterator.remove();
            }
        }
        if (!unresolved.isEmpty()) {
            LOG.warn("Cyclic symbolic links among .git/refs: " + unresolved);
        }
        return resolved;
    }

    private static Map<String, Hash> getResolvedHashes(Map<String, String> data) {
        HashMap resolved = ContainerUtil.newHashMap();
        for (Map.Entry<String, String> entry : data.entrySet()) {
            String refName = entry.getKey();
            Hash hash = GitRepositoryReader.parseHash(entry.getValue());
            if (hash == null || GitRepositoryReader.duplicateEntry(resolved, refName, hash)) continue;
            resolved.put(refName, hash);
        }
        return resolved;
    }

    private static String getTarget(String refName) {
        Matcher matcher = BRANCH_PATTERN.matcher(refName);
        if (!matcher.matches()) {
            return null;
        }
        String target = matcher.group(1);
        if (!target.startsWith(REFS_HEADS_PREFIX) && !target.startsWith(REFS_REMOTES_PREFIX)) {
            target = REFS_HEADS_PREFIX + target;
        }
        return target;
    }

    private static Hash parseHash(String value) {
        try {
            return HashImpl.build((String)value);
        }
        catch (Exception e) {
            return null;
        }
    }

    private static boolean duplicateEntry(Map<String, Hash> resolved, String refName, Object newValue) {
        if (resolved.containsKey(refName)) {
            LOG.error("Duplicate entry for [" + refName + "]. resolved: [" + resolved.get(refName).asString() + "], current: " + newValue + "]");
            return true;
        }
        return false;
    }

    private static class HeadInfo {
        private final String content;
        private final boolean isBranch;

        HeadInfo(boolean branch, String content) {
            this.isBranch = branch;
            this.content = content;
        }
    }
}

