#!/usr/bin/env php
<?php

$basedir = dirname(dirname(__FILE__));
include($basedir . '/lib/Constants.inc');
include($basedir . '/lib/mysql_bman.inc');
include($basedir . '/lib/myEnv.inc');

$rc = OK;

ini_set('date.timezone', 'Europe/Zurich');

// This variable is only for test automatization
$gTesting       = false;
$gaBackupPolicy = array('daily', 'weekly', 'monthly', 'quarterly', 'yearly');
$gLogFile       = './mysql_bman.log';;
$gBackupDir     = './bck';


// Check requirements

$ret = OK;
$cmd = 'which mysqldump';
$stdout = exec("$cmd 2>&1", $output, $ret);
if ( $ret != OK ) {
	print $stdout . "\n";
	print "WARNING: The MySQL utility mysqldump seems to be not installed. Please install it first:\n";
	print "RedHat/CentOS: shell> sudo yum install mysql\n";
	print "Ubuntu/Debian: shell> sudo apt-get install mysql-client\n";
	print "SLES/OpenSuSE: shell> sudo zypper install mysql-client\n";
}
else {
	$gMysqlDump = trim($stdout);
}

$ret += checkMyEnvRequirements();

if ( $ret != OK ) {
	$rc = 9999;
  if ( $ret == ERR ) {
    exit($rc);
  }
  else {
    // continue
  }
}


// Parse command line

$shortopts  = '';

$longopts  = array(
	'archive'
, 'archivedir:'
, 'backupdir:'
, 'cleanup'
, 'config:'
, 'help'
, 'log:'
, 'instance-name:'
, 'mode:'
, 'no-compress'
, 'per-schema'
, 'policy:'
, 'retention:'
, 'schema:'
, 'simulate'
, 'target:'
, 'type:'
);

$aParameter = getopt($shortopts, $longopts);

if ( isset($aParameter['help']) ) {
  $rc = 0;
  printUsage();
  exit($rc);
}

if ( ! isset($aOptions['config']) ) {
  $aOptions['config'] = '';
}


// Search for default MySQL configuration files
// */my.cnf, ~/.my.cnf, etc.
// https://dev.mysql.com/doc/refman/5.6/en/option-files.html

$aMyCnfFiles = array(
	'/etc/my.cnf'
, '/etc/mysql/my.cnf'
, $_ENV['HOME'] . '/.my.cnf'
);

$aMyCnf = array();
foreach ( $aMyCnfFiles as $filename ) {

	if ( is_readable($filename) ) {
		// print "Reading configuation from $filename\n";
		$aConfig = @parse_ini_file($filename, TRUE);
		// An error on parsing!
		if ( $aConfig === false ) {
			$rc = 164;
			$err = error_get_last(); 
			print trim($err['message']) . " (rc=$rc)\n";
			exit($rc);
		}
		
		// sections [client] [mysqldump]
		if ( array_key_exists('client', $aConfig) ) {
			if ( array_key_exists('user', $aConfig['client']) ) {
				$aMyCnf['user'] = $aConfig['client']['user'];
			}
			if ( array_key_exists('password', $aConfig['client']) ) {
				$aMyCnf['password'] = $aConfig['client']['password'];
			}
		}
		if ( array_key_exists('mysqldump', $aConfig) ) {
			if ( array_key_exists('user', $aConfig['mysqldump']) ) {
				$aMyCnf['user'] = $aConfig['mysqldump']['user'];
			}
			if ( array_key_exists('password', $aConfig['mysqldump']) ) {
				$aMyCnf['password'] = $aConfig['mysqldump']['password'];
			}
		}
	}
	else {
		// print "No file $filename found.\n";
	}
}

foreach ( $aMyCnf as $key => $value ) {
  $aOptions[$key] = $value;
}


// Do we have a configuration file?
$gMysqlBman = '';
if ( isset($aParameter['config']) && is_readable($aParameter['config']) ) {
  $gMysqlBman = $aParameter['config'];
  list($rc, $aConfig) = readConfigurationFile($gMysqlBman);
	foreach ( $aConfig as $key => $value ) {
		$aOptions[$key] = $value;
	}
}

// Options = Config + Parameter
// Rule: Parameters overwrite Config
foreach ( $aParameter as $key => $value ) {
  $aOptions[$key] = $value;
}

// print_r($aOptions); exit;

// Set some mandatory defaults
if ( (! isset($aOptions['log'])) || ($aOptions['log'] == '') ) {
  $aOptions['log'] = './mysql_bman.log';
}
if ( (! isset($aOptions['backupdir'])) || ($aOptions['backupdir'] == '') ) {
  $aOptions['backupdir'] = './bck';
}

// Minimum requirements is a log file
if ( ! isset($aOptions['log']) ) {
  // tee does not work here yet!
  $rc = 140;
  print "ERR: No log file specified!\n";
  exit($rc);
}

$gLogFile = $aOptions['log'];
$gBackupDir = $aOptions['backupdir'];
$ret = &checkArguments($aOptions);

if ( $ret != 0 ) {
  $rc = 127;
	print "More help can be found with mysql_bman --help\n";
  exit($rc);
}

if ( ! array_key_exists('mode', $aOptions) ) {
  $aOptions['mode'] = 'logical';
}

// If type = schema and no schema specified add empty schema option
if ( $aOptions['type'] == 'schema' ) {
  if ( ! isset($aOptions['schema']) ) {
    $aOptions['schema'] = '';
  }
}

// $gCatalog = '';
// $aAuxiliary = '';
// Target is not needed when type is cleanup
$aTarget = array();
if ( isset($aOptions['target']) ) {
  list($ret, $aTarget) = parseConnectString($aOptions['target']);
  
  if ( $ret != 0 ) {
    $rc = 109;
    tee("  ERROR wrong connect string (rc=$rc)!");
    exit($rc);
  }
}


// If user or password was set in .my.cnf update target

if ( isset($aOptions['user']) ) {
	$aTarget['user'] = $aOptions['user'];
}
if ( isset($aOptions['password']) ) {
	$aTarget['password'] = $aOptions['password'];
}
// Adjust target again
$aOptions['target'] = $aTarget['user'] . ':' . '******' . '@' . $aTarget['host'] . ':' . $aTarget['port'];


if ( $gMysqlDump == '' ) {
  $rc = 110;
  tee("  ERROR: Cannot find mysqldump utility. Please set path accordingly (rc=$rc).");
  exit($rc);
}

if ( isset($aOptions['retention']) ) {
  $aOptions['retention'] = intval($aOptions['retention']);
}

$gHostname = gethostname();

// ----------------------------------------------------------------------
// MAIN
// ----------------------------------------------------------------------

if ($gMysqlBman != '' ) {
  tee("\nConfiguration from " . $gMysqlBman);
  // Display parameter from config file
  $fh = fopen($gMysqlBman, 'r');
  if ( $fh ) {
    while ( ($buffer = fgets($fh, 4096)) !== false ) {
      if ( ! preg_match('/^$/', $buffer)
        && ! preg_match('/^\s*#.*$/', $buffer)
        ) {
        tee('  ' . trim($buffer));
      }
    }
    if ( ! feof($fh) ) {
      $rc = 136;
      tee("  ERROR: Unexpected fgets() fail (rc=$rc).");
      exit($rc);
    }
    fclose($fh);
  }
}
else {
  tee("\nNo configuration file.\n");
}

// Display parameter from command line options
tee("Configuration from command line options");
foreach ( $aParameter as $key => $value ) {
	// Password in target is OK here...
  tee(sprintf("  %-13s = %s", $key, $value));
}

// Display resulting configuration
tee("Resulting options");
foreach ( $aOptions as $key => $value ) {
	if ( $key == 'password' ) {
		$value = '******';
	}
  tee(sprintf("  %-13s = %s", $key, $value));
}

tee("\nLogging to " . $gLogFile);
tee("Backupdir is " . $gBackupDir);
tee("Hostname is " . $gHostname);
if ( array_key_exists('simulate', $aOptions) ) {
  tee("This is a SIMULATION only!");
}
$gBackupTimestamp = date("Y-m-d_H-i-s");
tee("\nStart " . $aOptions['type'] . ' ' . $aOptions['policy'] . " backup at $gBackupTimestamp");

// Is empty for type cleanup
$gTargetConfiguration = array();
if ( (count($aTarget) > 0) && ($aOptions['type'] != 'cleanup') ) {
  $gTargetConfiguration = getTargetConfiguration($aTarget);
}

if ( checkBackupStructure($gBackupDir) != 0 ) {
  $rc = 129;
  tee("  ERROR: Problems creating backup structure (rc=$rc).");
  exit($rc);
}

if ( array_key_exists('archive', $aOptions) ) {
  if ( ! isset($aOptions['archivedir']) || ($aOptions['archivedir'] == '' ) ) {
    $rc = 134;
    tee("  ERROR: Archivedir is not specified (rc=$rc).");
    exit($rc);
  }
}

if ( array_key_exists('archive', $aOptions) && (checkArchiveStructure($aOptions['archivedir']) != 0) ) {
  $rc = 147;
  tee("  ERROR: Problems creating archive structure (rc=$rc).");
  exit($rc);
}

if ( $aOptions['type'] == 'full' ) {
  if ( $aOptions['mode'] == 'logical' ) {
    $rc = doFullLogicalBackup($aTarget, $aOptions);
  }
  elseif ( $aOptions['mode'] == 'physical' ) {
    $rc = doFullPhysicalBackup($aTarget, $aOptions);
  }
}
elseif ( $aOptions['type'] == 'structure' ) {
  $rc = doStructureBackup($aTarget, $aOptions);
}
elseif ( $aOptions['type'] == 'binlog' ) {
  $rc = doBinlogBackup($aTarget, $aOptions);
}
elseif ( $aOptions['type'] == 'config' ) {
  $rc = doConfigBackup($aTarget, $aOptions);
}
elseif ( $aOptions['type'] == 'cleanup' ) {
  $rc = doCleanup($aTarget, $aOptions);
}
elseif ( $aOptions['type'] == 'schema' ) {
  $rc = doSchemaBackup($aTarget, $aOptions);
}
else {
  $rc = 163;
  tee("  ERROR: Unexpected error. This should never happen (" . $aOptions['type'] . ")! Please report a bug! (rc=$rc).");
  exit($rc);
}

$lEndTs = date("Y-m-d_H-i-s");
tee("End " . $aOptions['type'] . " " . $aOptions['policy'] . " backup at $lEndTs (rc=$rc)");
exit($rc);

?>
