<?php

/*

  debug($s)
  error($s)
  getConfiguration($pConfigurationFile)
  extractVersion($s)
  startDatabase($aInstance)
  checkDatabase($aInstance)
  wait_for_pid($lMysqldPid, $pidfile)
  stopDatabase($aInstance)
  getSchemaNames($pDatadir)
  addDirectoryToPath($lPath, $lDir, $lOldDir, $location = 'left')
  deleteDirectoryFromPath($lPath, $lDir)
  getSectionTitles($a)
  printStartingDatabases()
  readline($lne = '')
  getCurrentUser()
  checkMyEnvRequirements()

*/

// ----------------------------------------------------------------------
function debug($s)
// ----------------------------------------------------------------------
{
  if ( isset($_ENV['MYENV_DEBUG']) && ($_ENV['MYENV_DEBUG'] != '') ) {
    echo 'MYENV_DEBUG: ' . $s . "\n";
  }
}

// ----------------------------------------------------------------------
function error($s)
// ----------------------------------------------------------------------
{
  echo 'ERROR: ' . $s . "\n";
}

// ----------------------------------------------------------------------
function getConfiguration($pConfigurationFile)
// ----------------------------------------------------------------------
{

  $aConfiguration = array();

  if ( ! file_exists($pConfigurationFile) ) {
    $rc = 221;
    error("Configuration file $pConfigurationFile does not exist (rc=$rc).");
    return $aConfiguration; 
  }

  $aConfiguration = parse_ini_file($pConfigurationFile, true, INI_SCANNER_RAW);

  // Process default section first
  $lInstance = 'default';

  // Define default behaviour here

  if ( ! isset($aConfiguration[$lInstance]['angel']) ) {
    $aConfiguration[$lInstance]['angel'] = 'yes';
  }

  if ( ! isset($aConfiguration[$lInstance]['user']) ) {
    $aConfiguration[$lInstance]['user'] = 'mysql';
  }

  // Process all other sections != default
  foreach ( $aConfiguration as $db => $value ) {

    // add section name to section
    $aConfiguration[$db]['name'] = $db;

    if ( $db != 'default' ) {

      if ( (! isset($aConfiguration[$db]['hideschema'])) && (isset($aConfiguration['default']['hideschema'])) ) {
        $aConfiguration[$db]['hideschema'] = $aConfiguration['default']['hideschema'];
      }
      if ( (! isset($aConfiguration[$db]['angel'])) && (isset($aConfiguration['default']['angel'])) ) {
        $aConfiguration[$db]['angel'] = $aConfiguration['default']['angel'];
      }
      // Set default mysql user
      if ( (! isset($aConfiguration[$db]['user'])) && (isset($aConfiguration['default']['user'])) ) {
        $aConfiguration[$db]['user'] = $aConfiguration['default']['user'];
      }
    }
  }

  return $aConfiguration;
}

// ----------------------------------------------------------------------
function extractVersion($s)
// $s = basedir
// ----------------------------------------------------------------------
{
  $version = 'unknown';

  // Expected patterns: see tst/extractVersion.php
  if ( preg_match("/([a-zA-Z]+)\d?\-/", $s, &$matches) ) {

    if ( ($matches[1] == 'mysql')
      || ($matches[1] == 'percona')
      || ($matches[1] == 'mariadb')
       ) {

      if ( preg_match("/(\d{1,2}\.\d{1,2}\.\d{1,2})/", $s, &$matches) ) {
        $version = $matches[1];
      }
      elseif ( preg_match("/(\d{1,2}\.\d{1,2}\.\d{1,2}\-ndb\-\d{1,2}\.\d{1,2}\.\d{1,2}[a-z]?)/", $s, &$matches) ) {
        $version = $matches[1];
      }
      elseif ( preg_match("/(\d{1,2}\.\d{1,2})/", $s, &$matches) ) {
        $version = $matches[1];
      }
      else {
      }
    }
    elseif ( $matches[1] == 'Percona' ) {
      if ( preg_match("/(\d{1,2}\.\d{1,2}\.\d{1,2})/", $s, &$matches) ) {
        $version = 'Percona-' . $matches[1];
      }
    }
    else {
      // print '1';
    }
  }
  else {
    // Case: 5.6.12
    if ( preg_match("/(\d{1,2}\.\d{1,2}\.\d{1,2})/", $s, &$matches) ) {
      $version = $matches[1];
    }
  }

  return $version;
}

// ----------------------------------------------------------------------
function startDatabase($aInstance, $pOptions = '')
// ----------------------------------------------------------------------
{
  $rc = 0;
  global $lDebug;

  // Check if db is started with correct user
  // Also root should be allowed to start DB (Bug #99)
  $lCurrentUser = getCurrentUser();
  if ( ($lCurrentUser != $aInstance['user'])
    && ($lCurrentUser != 'root')
     ) {
    $rc = 222;
    error("Database is started with the wrong user ($lCurrentUser) but should should be strated with " . $aInstance['user'] . " (rc=$rc).");
    return $rc;
  }

  // Check if mysql.user table exists
  if ( ! file_exists($aInstance['datadir'] . '/mysql/user.frm') ) {
    $rc = 223;
    error("MySQL user table (user.frm) does not exist. Are you sure you have installed a MySQL instance already? (rc=$rc)");
    error("Create a MySQL instance with the command: mysql_install_db --datadir=" . $aInstance['datadir'] . " --basedir=" . $aInstance['basedir']);
    return $rc;
  }

  $aMyCnf = parse_ini_file($aInstance['my.cnf'], true, INI_SCANNER_RAW);

  // Check if there is any missmatch between my.cnf and myenv.conf
  foreach ( $aInstance as $key => $value ) {
    if ( isset($aMyCnf['mysqld'][$key]) ) {
      if ( $aMyCnf['mysqld'][$key] != $value ) {
        $cnf = '/etc/myenv/myenv.conf';
        print "WARNING: $key differs between " . $aInstance['my.cnf'] . " and " . $cnf . ". Please fix to avoid troubles...\n";
      }
    }
  }


  // It can happen, that client section is missing in my.cnf
  if ( isset($aMyCnf['client']) ) {

    // Check also missmatches between mysqld and client section in my.cnf
    foreach ( $aMyCnf['client'] as $key => $value ) {
      if ( isset($aMyCnf['mysqld'][$key]) ) {
        if ( $aMyCnf['mysqld'][$key] != $value ) {
          print "WARNING: $key differs between mysqld and client section in my.cnf. Please fix to avoid troubles...\n";
        }
      }
    }
  }

  // Check if the database is started already
  $ret = checkDatabase($aInstance);
  if ( $ret == 0 ) {
    $rc = 233;
    error("Database is alreay started (rc=$rc).");
    return $rc;
  }


  $cmd = '';
  $cmd .= 'cd ' . $aInstance['basedir'] . " ; ";

  // Start mysqld with angel process
  if ( $aInstance['angel'] == 'yes' ) {

    $cmd .= "nohup bin/mysqld_safe --defaults-file=" . $aInstance['my.cnf'] . " --datadir=" . $aInstance['datadir'] . " --basedir=" . $aInstance['basedir'] . " " . $pOptions;

    if ( $lDebug != '' ) {
      $cmd .= ' & echo $!';
    }
    else {
      $cmd .= ' >/dev/null 2>&1 & echo $!';
    }
  }
  // Start mysqld WITHOUT angel process
  else {
    // Should be better redirected to the error log!

    $cmd .= $aInstance['basedir'] . '/bin/mysqld --defaults-file=' . $aInstance['my.cnf'] . ' --basedir=' . $aInstance['basedir'] . ' --datadir=' . $aInstance['datadir'] . ' --user=' . $aInstance['user']. ' ' . $pOptions;

    if ( $lDebug != '' ) {
      $cmd .= ' & echo $!';
    }
    else {
      $cmd .= ' >/dev/null 2>&1 & echo $!';
    }
  }

  debug($cmd);
  $stdout = shell_exec($cmd);


  # No return code here!
  #if ( $ret != 0 ) {
  #  error("Command $cmd failed.");
  #  $rc = 224;
  #  // no return here
  #}

  $lDebugOrig = $lDebug;
  $lDebug = '';
  for ( $i = 1; $i <= 60 ; $i++ ) {

    sleep(1);
    echo ".";
    $ret = checkDatabase($aInstance);
    $rc = $ret;
    if ( $ret == 0 ) {
      break;
    }
  }
  $lDebug = $lDebugOrig;

  if ( $rc == 0 ) {
    print " SUCCESS!\n";
    return($rc);
  }
  else {
    print " ERROR!\n";
    return($rc);
  }
}

// ----------------------------------------------------------------------
function checkDatabase($aInstance)
// ----------------------------------------------------------------------
{
  $rc = 0;

  $lTimeout = 3;

  // todo: This strategy is not good because it requires access to
  // defaults-file which is not given to non privileged users
  // but for just ping we do not really need defaults-file?
  // Timeout is for hanging mysqld (happend with Galera Bug #56)
  $cmd = $aInstance['basedir'] . "/bin/mysqladmin --socket=" . $aInstance['socket'] . " ping --connect-timeout=$lTimeout >/dev/null 2>&1";

  debug($cmd);
  exec($cmd, &$output, &$ret);

  if ( $ret != 0 ) {
    $rc = 225;
    // no return here
  }

  return $rc;
}

// ----------------------------------------------------------------------
function wait_for_pid($lMysqldPid, $pidfile)
// ----------------------------------------------------------------------
{
  $rc = 0;

  $i = 0;
  $service_shutdown_timeout = 900;   // 15 min

  while ( $i <= $service_shutdown_timeout ) {

    sleep(1);
    print '.';
    // Process seems dead
    if ( posix_kill($lMysqldPid, SIG_IGN) === false ) {
      break;
    }

    $i++;
  }

  if ( file_exists($pidfile) ) {
    $rc = 226;
    print "\nThe server quit without removing PID file ($pidfile) (rc=$rc).\n";
  }

  if ( $rc == 0 ) {
    print " SUCCESS!\n";
    return($rc);
  }
  else {
    print " ERROR!\n";
    return($rc);
  }
}

// ----------------------------------------------------------------------
function stopDatabase($aInstance)
// ----------------------------------------------------------------------
{
  $rc = 0;

  $lDebug         = isset($_ENV['MYENV_DEBUG']) ? intval($_ENV['MYENV_DEBUG']) : 0;
  $basedir        = $_ENV['MYENV_BASE'];
  $user           = 'root';

  // We need defaults file here to support root password in client
  // section for shutdown service
  // $cmd = $aInstance['basedir'] . "/bin/mysqladmin --defaults-file=" . $aInstance['my.cnf'] . " --user=$user shutdown --socket=" . $aInstance['socket'];

  // debug($cmd);
  // $stdout = exec($cmd, &$output, &$ret);
  // if ( $ret != 0 ) {
  //   $rc = 227;
  //   error("Command $cmd failed (rc=$rc).");
  //   error("Please use password in ~/.my.cnf.");
  //   // no return here
  // }


  // Find the PID file

  $lMysqldPidFile = '';
  // We do not look into the database because this requires some account
  // which we do not have yet...
  $aMyCnf = parse_ini_file($aInstance['my.cnf'], true, INI_SCANNER_RAW);
  // print_r($aMyCnf);
  // Use PID file from my.cnf
  if ( isset($aMyCnf['mysqld']['pid_file']) && ($aMyCnf['mysqld']['pid_file'] != '') ) {
    $lMysqldPidFile = $aMyCnf['mysqld']['pid_file'];
  }
  elseif ( isset($aMyCnf['mysqld']['pid-file']) && ($aMyCnf['mysqld']['pid-file'] != '') ) {
    $lMysqldPidFile = $aMyCnf['mysqld']['pid-file'];
  }
  // Guess PID file
  // Guessing on /var/run/mysqld/mysqld.pid is NOT allowed because this could kill the wrong Instance!
  else {

    $pf = $aInstance['datadir'] . '/mysqld.pid';
    if ( is_file($pf) ) {
      $lMysqldPidFile = $pf;
    }

    $pf = $aInstance['datadir'] . '/' . php_uname('n') . '.pid';
    if ( is_file($pf) ) {
      $lMysqldPidFile = $pf;
    }
  }

  if ( $lMysqldPidFile == '' ) {
    $rc = 228;
    error("Cannot find nor guess PID file (rc=$rc). Is it possible that the database is already stopped? (rc=$rc)");
    return $rc;
  }


  // Stop daemon. We use a signal here to avoid having to know the
  // root password.

  if ( file_exists($lMysqldPidFile) ) {

    if ( ! is_readable($lMysqldPidFile) ) {
      $rc = 229;
      error("Cannot read PID file $lMysqldPidFile. Wrong privileges? (rc=$rc).");
      return $rc;
    }

    $lMysqldPid = intval(file_get_contents($lMysqldPidFile));
    if ( $lMysqldPid == 0 ) {
      $rc = 230;
      error("PID from $lMysqldPidFile is wrong or empty (rc=$rc).");
      return $rc;
    }

    if ( posix_kill($lMysqldPid, SIG_IGN) === true ) {

      // print 'Shutting down MySQL';
      posix_kill($lMysqldPid, SIGTERM);
      // mysqld should remove the PID file when it exits, so wait for it.
      $rc = wait_for_pid($lMysqldPid, $lMysqldPidFile);
    }
    else {
      print "MySQL server process $lMysqldPid is not running!\n";
      unlink($lMysqldPidFile);
    }

    // Delete lock for RedHat / SuSE
    $lLockDir  = '/var/lock/subsys';
    $lLockFile = $lLockDir . '/mysql';

    if ( is_file($lLockFile) ) {
      unlink($lLockFile);
    }
  }
  else {
    $rc = 231;
    print "MySQL server PID file could not be found at $lMysqldPidFile. Database already stopped? (rc=$rc)!\n";
    return $rc;
  }

  return $rc;
}

# ----------------------------------------------------------------------
function getSchemaNames($pDatadir)
# Parameter     : $_[0] data directory
# Return values : hash of schema names
# ----------------------------------------------------------------------
{
  $aSchema = array();

  foreach ( glob("$pDatadir/*", GLOB_ONLYDIR) as $schema ) {
    if ( preg_match(':([^/]+)$:', $schema, &$match) ) {
      array_push($aSchema, $match[1]);
    }
  }

  return $aSchema;
}

# ----------------------------------------------------------------------
function addDirectoryToPath($lPath, $lDir, $lOldDir, $location = 'left')
# Parameter     : $_[0] $PATH
#                 $_[1] new dir
# Return values : new $PATH
# ----------------------------------------------------------------------
{
  global $lDebug;

  if ( $lDebug != '' ) {
    print "PATH = " . $lPath . "\n";
    print "DIR  = " . $lDir . "\n";
    print "OLD  = " . $lOldDir . "\n";
  }

  $lPath = str_replace($lDir, '', $lPath);
  $lPath = str_replace($lOldDir, '', $lPath);
  if ( $location == 'left' ) {
    $lPath = $lDir . ':' . $lPath;
  }
  elseif ( $location == 'right' ) {
    $lPath = $lPath . ':' . $lDir;
  }
  else {
    $rc = 232;
    fprint(STDERR, "Invalid location $location (rc=$rc).\n");
  }
  // some clean-up
  $lPath = preg_replace('/:+/', ':', $lPath);
  $lPath = trim($lPath, ':');
  if ( $lDebug != '' ) {
    print "PATH = " . $lPath . "\n";
  }

  return $lPath;
}

# ----------------------------------------------------------------------
function deleteDirectoryFromPath($lPath, $lDir)
# ----------------------------------------------------------------------
{
  // delete
  $lPath = str_replace($lDir, '', $lPath);
  // remove ::
  $lPath = preg_replace('/:+/', ':', $lPath);
  // clean : at left or right side of PATH
  $lPath = trim($lPath, ':');

  return $lPath;
}

// ----------------------------------------------------------------------
function getSectionTitles($a)
// ----------------------------------------------------------------------
{
  $aDbNames = array();
  foreach ( $a as $db => $value ) {

    if ( $db != 'default' ) {
      array_push($aDbNames, $db);
    }
  }
  return $aDbNames;
}

// ----------------------------------------------------------------------
function printStartingDatabases()
// ----------------------------------------------------------------------
{
  $lDebug         = isset($_ENV['MYENV_DEBUG']) ? strval($_ENV['MYENV_DEBUG']) : '';
  $basedir        = $_ENV['MYENV_BASE'];
  $lConfFile      = '/etc/myenv/myenv.conf';
  $aConfiguration = getConfiguration($lConfFile);

  if ( $lDebug ) {
    var_dump($aConfiguration);
  }

  $aDbNames = array();
  foreach ( $aConfiguration as $db => $parameter ) {

    if ( $db != 'default' ) {
      foreach ( $parameter as $key => $value ) {

        if ( ($key == 'start') && ((strtoupper($value) == 'YES' ) || ($value == 1)) ) {
          array_push($aDbNames, $db);
        }
      }
    }
  }
  return $aDbNames;
}

if ( ! function_exists('readline') ) {

  function readline($lne = '') {

    print $lne;
    $fh = fopen('php://stdin', 'r');
    $input = fgets($fh, 1024);
    fclose($fh);
   return rtrim($input);
  }
}

// ----------------------------------------------------------------------
function getCurrentUser()
// ----------------------------------------------------------------------
{
  return trim(shell_exec('whoami'));
}

// ----------------------------------------------------------------------
function checkMyEnvRequirements()
// Return values:
// 0 - OK
// 1 - Warning
// 2 - Error
// ----------------------------------------------------------------------
{
  $ret = 0;

  // Check if environment variables are enabled

  if ( count($_ENV) == 0 ) {

    $vo = ini_get('variables_order');
    print "ERROR: Environment variables cannot be used. Please set variables_order = 'EGPCS' in /etc/php/cli/php.ini or /etc/php.ini or /etc/php5/cli/php.ini (or similar) first and then try again. Current value is: variables_order = " . $vo . "\n";
    return 2;
  }

  // Check if posix functions are installed

  if ( ! function_exists('posix_kill') ) {
    print "WARNING: The function posix_kill is not installed. You cannot stop MySQL instances without. Please install the package containg posix_kill.\n";
    print "Try to install packages {apt-get|yum|zypper} install {php-posix|php5-posix}.\n";
    $ret = 1;
  }
  if ( ! function_exists('posix_getpwuid') ) {
    print "WARNING: The function posix_getpwuid is not installed. You cannot continue installing without. Please install the package containg posix_getpwuid first.\n";
    print "Try to install packages {apt-get|yum|zypper} install {php-posix|php5-posix}.\n";
    return 2;
  }

  if ( (! defined('SIG_IGN')) || (! defined('SIGTERM')) ) {
    print "WARNING: The constants SIG_IGN and/or SIGTERM are not known. You cannot stop MySQL instances without. Please install the package containing these constants.\n";
    print "Try to install the package php5-pcntl. We will help you in the meanwhile with a fake...\n";
    define('SIG_IGN', 0);
    define('SIGTERM', 15);
    $ret = 1;
  }

  return $ret;
}

?>
