package InnoDbStatus;

# %LICENSE%

use strict;
use warnings;

use File::Copy;
use Digest::SHA;
use File::Which;

use FromDualMySQLagent ':stooges';
use writeDataToCacheFile;

# sub handleDeadlockMessage
# sub handleForeignKeyMessage
# sub getInnoDbStatus
# sub parseInnoDbStatus

# ---------------------------------------------------------------------
sub handleDeadlockMessage
# ---------------------------------------------------------------------
{
  my $lDeadlockMessage = shift;
  my $lTimestamp = time();   # Seems to be not needed at all???

  my $rc = 0;

  if ( $main::gParameter{'Debug'} >= INFO ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, INFO, '      ' . (caller(0))[3]); }

  # Cleanup leading and trailing ----
  $lDeadlockMessage =~ s/-+\n//g;

  my $lSendFile = 0;
  # Does an old Deadlock message exist?
  my $lDeadlockFile = "/tmp/DeadLockMessage." . $main::gParameter{'Hostname'};
  if ( -f $lDeadlockFile ) {

    if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      Old Deadlock message file $lDeadlockFile exists."); }

    # Write temporary Deadlock file
    my $lTmpDeadlockFile = '/tmp/DeadlockMessage.tmp.' . $$;
    if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      Write temporary Deadlock file to $lTmpDeadlockFile."); }
    if ( ! open(FILE, '>' . $lTmpDeadlockFile) ) {
      $rc = 1705;
      &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, ERR, "    Cannot open file $lTmpDeadlockFile (rc=$rc).\n$!\n");
      return $rc;
    }
    print FILE "$lDeadlockMessage";
    close(FILE);

    my $sha1;
    # Compare with old Deadlock file
    $sha1 = Digest::SHA1->new;
    if ( ! open(FILE, $lTmpDeadlockFile) ) {
      $rc = 1700;
      &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, ERR, "    Cannot open file $lTmpDeadlockFile (rc=$rc).\n$!\n");
      return $rc;
    }
    binmode FILE;
    $sha1->addfile(*FILE);
    close(FILE);
    my $hash1 = $sha1->hexdigest;
    if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      Hash of new Deadlock file is $hash1."); }

    $sha1 = Digest::SHA1->new;
    if ( ! open(FILE, $lDeadlockFile) ) {
      $rc = 1701;
      &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, ERR, "    Cannot open file $lDeadlockFile (rc=$rc).\n$!\n");
      return $rc;
    }
    binmode FILE;
    $sha1->addfile(*FILE);
    close(FILE);
    my $hash2 = $sha1->hexdigest;
    if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      Hash of old Deadlock file is $hash2."); }

    # Both Deadlocks are qual
    if ( $hash1 eq $hash2 ) {
      if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      Hashes are equal. Clean up new Deadlock."); }
      unlink($lTmpDeadlockFile);
    }
    # the 2 Deadlocks differ
    else {
      if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      Hashes are different. Propagate new Deadlock."); }
      # todo: this should be caught. causes a bug when file is owned by root!
      if ( ! move($lTmpDeadlockFile, $lDeadlockFile) ) {
        $rc = 1710;
        my $msg = $!;
        &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, ERR, "      Cannot move file $lTmpDeadlockFile to $lDeadlockFile. Please check permissions of both of them (rc=$rc).");
        &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, ERR, "      Error message: $msg");
        return $rc;
      }
      else {
        $lSendFile = 1;
      }
    }

  }
  # no, old Deadlock message does NOT exists: then write it
  else {

    if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      Old Deadlock message files does NOT exist."); }
    if ( ! open(FILE, '>' . $lDeadlockFile) ) {
      $rc = 1702;
      &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, ERR, "   Cannot open file $lDeadlockFile (rc=$rc).\n$!\n");
      return $rc;
    }
    print FILE "$lDeadlockMessage";
    close(FILE);
    $lSendFile = 1;
  }   # old Deadlock message file exists?

  # --------

  my %hGlobalStatus;

  my @aGlobalStatusToSend = (
    'Deadlock_new'
  # , 'Deadlock_message'   # only push message if there is one!
  );

  # Only push message if there is one!
  if ( $lSendFile == 1 ) {
    push(@aGlobalStatusToSend, 'Deadlock_message');
  }

  &FromDualMySQLagent::initValues(\%hGlobalStatus, \@aGlobalStatusToSend);
  $hGlobalStatus{'Deadlock_message'} = 'none';

  # MPM can be run in 2 modes now: locally or MaaS
  # We run MPM locally
  if ( lc($main::gParameter{'MaaS'}) ne 'on' ) {

    # Deadlock does not exist yet or is a new Deadlock, so send it
    if ( $lSendFile == 1 ) {

      if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      New Deadlock found."); }

      $hGlobalStatus{'Deadlock_new'} = 1;
      my $rc = &FromDualMySQLagent::checkConnectionToZabbixServer($main::gParameter{'ZabbixServer'}, $main::gParameter{'ZabbixServerPort'}, $main::gParameter{'Hostname'});

      # Connection to zabbix server seems OK
      if ( $rc == 0 ) {

        if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "Connection to zabbix server seems OK."); }

        my $key = 'Deadlock_message';

        my $exe = 'zabbix_sender';
        $exe = which($exe);
        if ( ! defined($exe) ) {
          $rc = 1317;
          &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, ERR, "Cannot find program $exe.");
        }

        # This is an exception because with cache file we have not found a way
        # how to send the data with CR/LF
        my $cmd = "$exe --zabbix-server " . $main::gParameter{'ZabbixServer'} . " --port $main::gParameter{'ZabbixServerPort'} --host '" . $main::gParameter{'Hostname'}. "' --key " . $main::gParameter{'Domain'} . '.' . $key . " --value '" . $lDeadlockMessage . "'";

        if ( $main::gParameter{'Debug'} == DBG ) {
          $cmd .= ' -vv';
          &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, $cmd);
        }

        if ( $main::gParameter{'Debug'} == DBG ) {
          system("$cmd >>" . $main::gParameter{'LogFile'} . " 2>&1");
        }
        else {
          system("$cmd >/dev/null 2>&1");
        }
        my $ret = $?;
        $rc = 1308;
        if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "(ret=$ret / rc=$rc)."); }
        # If zabbix_sender fails we should roll back the stuff to get all Deadlocks!
      }
      # No success connecting to zabbix server write to Cache File
      else {
        $rc = 1709;
        &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, WARN, "Connection to zabbix server failed (rc=$rc).");

        # Transform \n into marker 'xxx' because it will cause an error otherwise!
        # We could think about if it is sufficient to send just a Deadlock without the timestamp
        # this would cause only the first time a wrong information
        $lDeadlockMessage =~ s/\n/xxx/g;
        $hGlobalStatus{'Deadlock_message'}  = $lDeadlockMessage;
      }
    }
    # There was NO new Deadlock
    else {

      if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      NO new Deadlock found."); }
      $hGlobalStatus{'Deadlock_new'} = 0;
    } # Deadlock does not exist or is new, so send it...
  }
  # We run mpm as MaaS
  else {

    # Deadlock does not exist yet or is a new Deadlock, so send it
    if ( $lSendFile == 1 ) {
      if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      New Deadlock found."); }

      $hGlobalStatus{'Deadlock_new'} = 1;
      # Transform \n into marker 'xxx' because it will cause an error otherwise!
      # We could think about if it is sufficient to send just a Deadlock without the timestamp
      # this would cause only the first time a wrong information
      $lDeadlockMessage =~ s/\n/xxx/g;
      $hGlobalStatus{'Deadlock_message'}  = $lDeadlockMessage;
    }
    # There was NO new Deadlock
    else {
      if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      NO new Deadlock found."); }
      $hGlobalStatus{'Deadlock_new'} = 0;
    } # Deadlock does not exist or is new, so send it...
  }

  &writeDataToCacheFile::writeDataToCacheFile(\%hGlobalStatus, \@aGlobalStatusToSend);
  return $rc;
}

# ---------------------------------------------------------------------
sub handleForeignKeyMessage
# ---------------------------------------------------------------------
{
  my $lForeignKeyMessage = shift;
  my $lTimestamp = time();

  my $rc = 0;

  if ( $main::gParameter{'Debug'} >= INFO ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, INFO, '      ' . (caller(0))[3]); }

  # Cleanup leading and trailing ----
  $lForeignKeyMessage =~ s/-+\n//g;

  my $lSendFile = 0;
  # Does an old Foreign Key message exist?
  my $lForeignKeyFile = "/tmp/ForeignKeyMessage." . $main::gParameter{'Hostname'};
  if ( -f $lForeignKeyFile ) {

    if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      Old FK message files exists."); }

    # Write temporary Foreign Key file
    my $lTmpForeignKeyFile = '/tmp/ForeignKeyMessage.tmp.' . $$;
    if ( ! open(FILE, '>' . $lTmpForeignKeyFile) ) {
      $rc = 1706;
      &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, ERR, "   Cannot open file $lTmpForeignKeyFile (rc=$rc).\n$!\n");
      return $rc;
    }
    print FILE "$lForeignKeyMessage";
    close (FILE);

    my $sha1;
    # Compare with old Foreign Key file
    $sha1 = Digest::SHA1->new;
    if ( ! open(FILE, $lTmpForeignKeyFile) ) {
      $rc = 1703;
      &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, ERR, "   Cannot open file $lTmpForeignKeyFile (rc=$rc).\n$!\n");
      return $rc;
    }
    binmode FILE;
    $sha1->addfile(*FILE);
    close(FILE);
    my $hash1 = $sha1->hexdigest;

    $sha1 = Digest::SHA1->new;
    if ( ! open(FILE, $lForeignKeyFile) ) {
      $rc = 1704;
      &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, ERR, "   Cannot open file $lForeignKeyFile (rc=$rc)\n$!\n");
      return $rc;
    }
    binmode FILE;
    $sha1->addfile(*FILE);
    close(FILE);
    my $hash2 = $sha1->hexdigest;

    # Both Foreign Key messages are qual
    if ( $hash1 eq $hash2 ) {
      unlink($lTmpForeignKeyFile);
    }
    # the 2 Foreign Key messagess differ
    else {
      move($lTmpForeignKeyFile, $lForeignKeyFile);
      $lSendFile = 1;
    }

  }
  # no, old Foreign Key message does NOT exist: then write it
  else {

    if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      Old FK message files does NOT exist."); }
    if ( ! open(FILE, '>' . $lForeignKeyFile) ) {
      $rc = 1707;
      &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, ERR, "   Cannot open file $lForeignKeyFile (rc=$rc)\n$!\n");
      return $rc;
    }
    print FILE "$lForeignKeyMessage";
    close(FILE);
    $lSendFile = 1;
  }   # old Foreign Key message file exists?

  # --------

  my %hGlobalStatus;

  my @aGlobalStatusToSend = (
    'ForeignKey_new'
  # , 'ForeignKey_message'   # only push messages if there is one!
  );

  # Only push message if there is one!
  if ( $lSendFile == 1 ) {
    push(@aGlobalStatusToSend, 'ForeignKey_message');
  }

  &FromDualMySQLagent::initValues(\%hGlobalStatus, \@aGlobalStatusToSend);
  $hGlobalStatus{'ForeignKey_message'} = 'none';

  # MPM can be run in 2 modes now: locally or MaaS
  # We run MPM locally
  if ( lc($main::gParameter{'MaaS'}) ne 'on' ) {

    # Deadlock does not exist yet or is a new Deadlock, so send it
    if ( $lSendFile == 1 ) {
      if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      New foreign key message found."); }

      $hGlobalStatus{'ForeignKey_new'} = 1;
      my $rc = &FromDualMySQLagent::checkConnectionToZabbixServer($main::gParameter{'ZabbixServer'}, $main::gParameter{'ZabbixServerPort'}, $main::gParameter{'Hostname'});

      # Connection to zabbix server seems OK
      if ( $rc == 0 ) {

        if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "Connection to zabbix server seems OK."); }

        my $key = 'ForeignKey_message';

        my $exe = 'zabbix_sender';
        $exe = which($exe);
        if ( ! defined($exe) ) {
          $rc = 1320;
          &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, ERR, "Cannot find program $exe.");
        }

        # This is an exception because with cache file we have not found a way
        # how to send the data with CR/LF
        my $cmd = "$exe --zabbix-server " . $main::gParameter{'ZabbixServer'} . " --port $main::gParameter{'ZabbixServerPort'} --host '" . $main::gParameter{'Hostname'}. "' --key " . $main::gParameter{'Domain'} . '.' . $key . " --value '" . $lForeignKeyMessage . "'";

        if ( $main::gParameter{'Debug'} == DBG ) {
          $cmd .= ' -vv';
          &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, $cmd);
        }

        if ( $main::gParameter{'Debug'} == DBG ) {
          system("$cmd >>$main::gParameter{'LogFile'} 2>&1");
        }
        else {
          system("$cmd >/dev/null 2>&1");
        }
        my $ret = $?;
        $rc = 1309;
        if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "(ret=$ret / rc=$rc)"); }
        # If zabbix_sender fails we should roll back the stuff to get all foreign key errors!
      }
      # No success connecting to zabbix server write to Cache File
      else {
        $rc = 1708;
        &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, WARN, "Connection to zabbix server failed (rc=$rc).");

        # Transform \n into marker 'xxx' because it will cause an error otherwise!
        # We could think about if it is sufficient to send just a Deadlock without the timestamp
        # this would cause only the first time a wrong information
        $lForeignKeyMessage =~ s/\n/xxx/g;
        $hGlobalStatus{'ForeignKey_message'}  = $lForeignKeyMessage;
      }   # Connection to zabbix_server ok/nok
    }
    # There was NO new Foreign Key message
    else {
      if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      NO new foreign key found."); }
      $hGlobalStatus{'ForeignKey_new'} = 0;
    } # Foreign Key message does not exist or is new, so send it...
  }
  # We run mpm as MaaS
  else {
    if ( $lSendFile == 1 ) {
      if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      New foreign key message found."); }

      $hGlobalStatus{'ForeignKey_new'} = 1;
      # Transform \n into marker 'xxx' because it will cause an error otherwise!
      # We could think about if it is sufficient to send just a Deadlock without the timestamp
      # this would cause only the first time a wrong information
      $lForeignKeyMessage =~ s/\n/xxx/g;
      $hGlobalStatus{'ForeignKey_message'}  = $lForeignKeyMessage;
    }
    # There was NO new Foreign Key message
    else {
      if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      NO new foreign key found."); }
      $hGlobalStatus{'ForeignKey_new'} = 0;
    } # Foreign Key message does not exist or is new, so send it...
  }

  &writeDataToCacheFile::writeDataToCacheFile(\%hGlobalStatus, \@aGlobalStatusToSend);
  return $rc;
}

# ---------------------------------------------------------------------
sub getInnoDbStatus
# ---------------------------------------------------------------------
{
  my $dbh = shift;
  my $status_ref = shift;

  my $rc = 0;

  if ( $main::gParameter{'Debug'} >= INFO ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, INFO, '    ' . (caller(0))[3]); }

  my $sql = 'SHOW /*!50000 ENGINE */ INNODB STATUS';
  my $lInnodbStatus;

  my $sth = $dbh->prepare($sql);
  if ( $sth->execute() ) {
    if ( my $ref = $sth->fetchrow_hashref() ) {
      $lInnodbStatus = $ref->{'Status'};
    }
    $sth->finish();
  }

  # It seems like we do not have access rights!
  # Needs possibly some more investigation here?
  if ( ! defined($lInnodbStatus) || ($lInnodbStatus eq '') ) {
    return $rc;
  }

  &parseInnoDbStatus($dbh, $lInnodbStatus, $status_ref);
  return $rc;
}

# ---------------------------------------------------------------------
sub parseInnoDbStatus
# ---------------------------------------------------------------------
{
  my $dbh = shift;
  my $lInnodbStatus = shift;
  my $status_ref = shift;

  my $rc = 0;

  if ( $main::gParameter{'Debug'} >= INFO ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, INFO, '    ' . (caller(0))[3]); }

  # Loop over all lines of output
  my $section = '';
  my $lDeadlockMessage   = '';
  my $lForeignKeyMessage = '';
  foreach ( split("\n", $lInnodbStatus) ) {

    # print $_ . "\n";

    if ( $_ =~ /^BACKGROUND THREAD$/ ) {
      $section = 'BACKGROUND THREAD';
      next;
    }
    elsif ( ($_ =~ /^DEAD LOCK ERRORS$/)
         || ($_ =~ /^LATEST DETECTED DEADLOCK$/)
          ) {
      $section = 'DEAD LOCK ERRORS';
      next;
    }
    elsif ( ($_ =~ /^FOREIGN KEY CONSTRAINT ERRORS$/)
         || ($_ =~ /^LATEST FOREIGN KEY ERROR$/)
          ) {
      $section = 'FOREIGN KEY CONSTRAINT ERRORS';
      next;
    }
    elsif ( $_ =~ /^SEMAPHORES$/ ) {
      $section = 'SEMAPHORES';
      next;
    }
    elsif ( $_ =~ /^TRANSACTIONS$/ ) {
      $section = 'TRANSACTIONS';
      next;
    }
    elsif ( $_ =~ /^FILE I\/O$/ ) {
      $section = 'FILE I/O';
      next;
    }
    elsif ( $_ =~ /^INSERT BUFFER AND ADAPTIVE HASH INDEX$/ ) {
      $section = 'INSERT BUFFER AND ADAPTIVE HASH INDEX';
      next;
    }
    elsif ( $_ =~ /^LOG$/ ) {
      $section = 'LOG';
      next;
    }
    elsif ( $_ =~ /^BUFFER POOL AND MEMORY$/ ) {
      $section = 'BUFFER POOL AND MEMORY';
      next;
    }
    elsif ( $_ =~ /^ROW OPERATIONS$/ ) {
      $section = 'ROW OPERATIONS';
      next;
    }

    # --------

    if ( $section eq 'INSERT BUFFER AND ADAPTIVE HASH INDEX' ) {

      # Hash table size 276671, node heap has 810 buffer(s)
      if ( $_ =~ /node heap has (\d+) buffer/ ) {
        $$status_ref{'Innodb_hash_node_size'} = 16384 * $1;
        next;
      }

      # 0.00 hash searches/s, 0.00 non-hash searches/s
      if ( $_ =~ /^(\d+\.\d+)\s+hash searches\/s,\s+(\d+.\d+)\s+non-hash searches\/s/ ) {
        $$status_ref{'Innodb_hash_seaches'} = $1;
        $$status_ref{'Innodb_non_hash_searches'} = $2;
        next;
      }
    }

    if ( $section eq 'SEMAPHORES' ) {

      # Mutex spin waits 27329, rounds 60032, OS waits 929
      if ( $_ =~ /^Mutex spin waits\s+(\d+),\s+rounds\s+(\d+),\s+OS waits\s+(\d+)/ ) {
        $$status_ref{'Innodb_mutex_spin_waits'} = $1;
        $$status_ref{'Innodb_mutex_spin_rounds'} = $2;
        $$status_ref{'Innodb_mutex_os_waits'} = $3;
        next;
      }
    }

    if ( $section eq 'TRANSACTIONS' ) {

      # History list length 177
      if ( $_ =~ /^History list length\s+(\d+)/ ) {
        $$status_ref{'Innodb_trx_history_list_length'} = $1;
        next;
      }
    }

    if ( $section eq 'LOG' ) {

      # Log sequence number 13392740025
      # Log flushed up to   13392740025
      # Pages flushed up to 13392679774
      # Last checkpoint at  13392679671
      if ( $_ =~ /^Log sequence number\s+(\d+)/ ) {
        $$status_ref{'Innodb_log_sequence_number'} = $1;
        next;
      }
      if ( $_ =~ /^Log flushed up to\s+(\d+)/ ) {
        $$status_ref{'Innodb_log_flushed_up_to'} = $1;
        next;
      }
      # 5.6 feature?
      if ( $_ =~ /^Pages flushed up to\s+(\d+)/ ) {
        $$status_ref{'Innodb_pages_flushed_up_to'} = $1;
        next;
      }
      if ( $_ =~ /^Last checkpoint at\s+(\d+)/ ) {
        $$status_ref{'Innodb_last_checkpoint_at'} = $1;
        next;
      }
    }

    if ( $section eq 'DEAD LOCK ERRORS' ) {
      $lDeadlockMessage .= $_ . "\n";
    }

    if ( $section eq 'FOREIGN KEY CONSTRAINT ERRORS' ) {
      $lForeignKeyMessage .= $_ . "\n";
    }

    if ( $section eq 'ROW OPERATIONS' ) {

      # 0 queries inside InnoDB, 0 queries in queue
      if ( $_ =~ /^(\d+)\s+queries inside InnoDB,\s+(\d+)\s+queries in queue/ ) {
        $$status_ref{'Innodb_queries_inside_innodb'} = $1;
        $$status_ref{'Innodb_queries_in_queue'} = $2;
        next;
      }

      # 1 read views open inside InnoDB
      if ( $_ =~ /^(\d+)\s+read views open inside InnoDB/ ) {
        $$status_ref{'Innodb_open_read_views_inside_innodb'} = $1;
        next;
      }
    }
  }   # loop over InnoDB status

  # Do we have a Deadlock Message?
  if ( $lDeadlockMessage ne '' ) {
    &handleDeadlockMessage($lDeadlockMessage);
  }
  else {
    if ( $main::gParameter{'Debug'} >= DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, '      No Deadlock message.'); }
  }

  # Do we have a ForeignKey Message?
  if ( $lForeignKeyMessage ne '' ) {
    &handleForeignKeyMessage($lForeignKeyMessage);
  }
  else {
    if ( $main::gParameter{'Debug'} >= DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, '      No Deadlock message.'); }
  }

  # Get InnoDB Compression Time
  my $sql = 'SELECT SUM(compress_time) AS compress_time, SUM(uncompress_time) AS uncompress_time FROM information_schema.INNODB_CMP';

  my $sth = $dbh->prepare($sql);
  if ( $sth->execute() ) {
    if ( my $ref = $sth->fetchrow_hashref() ) {
      $$status_ref{'Innodb_compress_time'}   = $ref->{'compress_time'};
      $$status_ref{'Innodb_uncompress_time'} = $ref->{'uncompress_time'};
    }
    $sth->finish();
  }

  # Get InnoDB Row Lock Details and InnoDB Transcation Lock Memory
  $sql = 'SELECT SUM(trx_rows_locked) AS rows_locked, SUM(trx_rows_modified) AS rows_modified, SUM(trx_lock_memory_bytes) AS lock_memory FROM information_schema.INNODB_TRX';

  $sth = $dbh->prepare($sql);
  if ( $sth->execute() ) {
    if ( my $ref = $sth->fetchrow_hashref() ) {
      $$status_ref{'Innodb_rows_locked'}     = defined($ref->{'rows_locked'}) ? $ref->{'rows_locked'} : 0;
      $$status_ref{'Innodb_rows_modified'}   = defined($ref->{'rows_modified'}) ? $ref->{'rows_modified'} : 0;
      $$status_ref{'Innodb_trx_lock_memory'} = defined($ref->{'lock_memory'}) ? $ref->{'lock_memory'} : 0;
    }
    $sth->finish();
  }

  # Get InnoDB Transaction states
  $sql = 'SELECT LOWER(REPLACE(trx_state, " ", "_")) AS state, count(*) AS cnt from information_schema.INNODB_TRX GROUP BY state';

  $sth = $dbh->prepare($sql);
  if ( $sth->execute() ) {
    if ( my $ref = $sth->fetchrow_hashref() ) {
      $$status_ref{'Innodb_trx_' . $ref->{'state'}} = $ref->{'cnt'};
    }
    $sth->finish();
  }

  if ( $main::gParameter{'Debug'} == DBG ) {
    &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, '    InnoDB Status:');
    foreach my $key ( keys %$status_ref ) {
      &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      key: $key, value: $$status_ref{$key}");
    }
  }

  return $rc;
}

1;
__END__
