package FromDualMySQLagent;

#
# Copyright (C) 2010, 2011, 2012 FromDual GmbH
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.
# If not, see <http://www.gnu.org/licenses/>.
#

# cleanAndQuit
# setAgentLock
# clearCacheFile
# checkConnectionToZabbixServer
# readConfigurationFile
# getSections
# readSection
# getDefaultParameter
# getParameter
# mylog
# getDatabaseConnection
# releaseDatabaseConnection
# initValues
# getGlobalVariables
# getGlobalStatus
# removeAgentLock

use strict;
use warnings;

use Time::HiRes qw( gettimeofday );
use File::Which;
use Config;

use base 'Exporter';

use constant DBG  => 4;
use constant INFO => 3;
use constant WARN => 2;
use constant ERR  => 1;

our @EXPORT_OK = ('DBG', 'INFO', 'WARN', 'ERR');
our %EXPORT_TAGS = ( stooges => [ 'DBG', 'INFO', 'WARN', 'ERR' ] );
our $gRemoveLockFile;

# ----------------------------------------------------------------------
sub cleanAndQuit
# do a proper exit here
# This should be the only exit for the whole mpm!
# ----------------------------------------------------------------------
{
  my $pRc = $_[0];

  # Some exits have not defined:
  # $FromDualMySQLagent::gRemoveLockFile
  # $main::gParameter{"LogLevel"}
  # This causes some exits in the agent where we do not see the reason
  # todo: Ugly hack should be fixed in the next release
  if ( ! defined($gRemoveLockFile) ) {
		$gRemoveLockFile = 0;
  }
	if ( ! defined($main::gParameter{'LogLevel'}) ) {
		$main::gParameter{'LogLevel'} = ERR;
	}

  # Remove Lock file if exists.
  if ( $gRemoveLockFile == 1 ) {

    if ( defined($main::gParameter{'AgentLockFile'}) ) {

      if ( -e $main::gParameter{'AgentLockFile'} ) {
        # Todo: somehow LogLevel is not defined here thus
        # Code it hard until problem is found!
        #&removeAgentLock($main::gParameter{'LogLevel'}, $main::gParameter{'LogFile'});
        &removeAgentLock(INFO, $main::gParameter{'LogFile'});
      }
    }
  }
  else {
    if ( $main::gParameter{'LogLevel'} >= INFO ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, INFO, "    gRemoveLockFile is not set."); }
  }

  # Print exit message to log if exists or to stdout.
  my $msg = "FromDual Performance Monitor for MySQL run finshed (rc=$pRc).";
  if ( defined($main::gParameter{'LogFile'})
    && $main::gParameter{'LogFile'} ne ''
    && -e $main::gParameter{'LogFile'}
    && -w $main::gParameter{'LogFile'} ) {
    &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, INFO, $msg);
  }
  else {
    print "$msg\n";
  }

  print "1\n";
  exit $pRc;
}

# ----------------------------------------------------------------------
sub setAgentLock
# ----------------------------------------------------------------------
{
  my $pLogLevel = $_[0];
  my $pLogFile  = $_[1];

  my $rc = 0;

  if ( $pLogLevel >= INFO ) { &FromDualMySQLagent::mylog($pLogFile, INFO, '  ' . (caller(0))[3]); }

  my $lAgentLockFile = $main::gParameter{'AgentLockFile'};
  # Check if lock file already exists and complain if yes
  if ( -e $lAgentLockFile ) {
    $rc = 1314;
    if ( $pLogLevel >= WARN ) { &FromDualMySQLagent::mylog($pLogFile, WARN, "  Agent Lock file $lAgentLockFile already exists."); }

    # Check the mtime of the lock file
    # If agent was running too long, kill it!
    my $mtime = (stat($lAgentLockFile))[9];
    $mtime = defined($mtime) ? $mtime : 0;
    my $now = time();

    # This is possibly too long for a local installation?
    my $timeout = 60;
    if ( ($now - $mtime) > $timeout ) {
      $rc = 1315;
      &FromDualMySQLagent::mylog($pLogFile, ERR, "  Agent was running for more than $timeout seconds (now: $now, mtime: $mtime). Killing it now! (rc=$rc).");

      # cat $lAgentLockFile
      # todo: Could be done in function like getPidFromFile or so. (see also getMySQLPid function)
      if ( open(FILE, '<' . $lAgentLockFile) ) {
        my $pid = int(<FILE>);
        close(FILE);
        # It looks like this situation happens during log rotate!!!
        if ( $pid > 0 ) {
          chomp($pid);
          # Terminate program
          &FromDualMySQLagent::mylog($pLogFile, ERR, "  Kill (-TERM) agent with PID $pid (our PID is $$).");
          my $cnt = kill 15, $pid;
          if ( $cnt == 0 ) {
            &FromDualMySQLagent::mylog($pLogFile, ERR, "  Kill (-KILL) agent with PID $pid (our PID is $$).");
            $cnt = kill 9, $pid;
          }
          &FromDualMySQLagent::mylog($pLogFile, ERR, "  Remove agent lock file.");
          &removeAgentLock($pLogLevel, $pLogFile);
        }
        else {
          # It looks like this situation happens during log rotate!!!
          &FromDualMySQLagent::mylog($pLogFile, ERR, "  PID file was empty again!");
        }
      }
    }
    else {
      $rc = 1316;
      my $pid = 0;
      if ( open(FILE, '<' . $lAgentLockFile) ) {
        $pid = <FILE>;
        chomp($pid);
        close(FILE);
      }
      &FromDualMySQLagent::mylog($pLogFile, WARN, "  Another agent with PID $pid (our PID is $$) is running. We will abort now.");
      # This message seems to be redundant with cleanAndQuit
      # &FromDualMySQLagent::mylog($pLogFile, WARN, "FromDual Performance Monitor for MySQL run finshed (rc=$rc).");
      $gRemoveLockFile = 0;
      $main::gParameter{'LogLevel'} = $pLogLevel;
      &cleanAndQuit($rc);
    }
  }

  # Create a new agent lock file
  # todo: this is possibly a bug! should be in the else clause?
  if ( ! open(FH, '>', $lAgentLockFile) ) {
    $rc = 1310;
    &FromDualMySQLagent::mylog($pLogFile, ERR, "  Cannot write to Lock File $lAgentLockFile (rc=$rc).\n$!\n");
    # This message seems to be redundant with cleanAndQuit
    # &FromDualMySQLagent::mylog($pLogFile, WARN, "FromDual Performance Monitor for MySQL run finshed (rc=$rc).");
    &cleanAndQuit($rc);
  }
  printf(FH "$$\n");
  close(FH);
  $gRemoveLockFile = 1;
}

# ----------------------------------------------------------------------
sub clearCacheFile
# ----------------------------------------------------------------------
{
  my $pCacheFile = $_[0];

  my $rc = 0;

  if ( $main::gParameter{'Debug'} >= INFO ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, INFO, '      ' . (caller(0))[3]); }

  # Clear Cache File
  if ( ! open(FH, '>', $pCacheFile) ) {
    $rc = 1311;
    &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, ERR, "  Cannot write to Cache File $pCacheFile (rc=$rc).\n$!\n");
    return $rc;
  }
  close(FH);
  return $rc;
}

# ----------------------------------------------------------------------
sub checkConnectionToZabbixServer
# ----------------------------------------------------------------------
{
  my $pServer = $_[0];
  my $pPort   = $_[1];
  my $pHost   = $_[2];

  my $rc = 0;

  if ( $main::gParameter{'Debug'} >= INFO ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, INFO, '      ' . (caller(0))[3]); }

  # This tag does NOT exist in templates!!!
  my $lKey    = 'FromDual.server.check';
  my $lValue  = 1;

  if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      Check connection to zabbix server."); }

  my $prg = 'zabbix_sender';
  my $exe = which($prg);
  if ( ! defined($exe) ) {
    $rc = 1319;
    &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, ERR, "      Cannot find program $prg (rc=$rc).");
    return $rc;
  }

  my $cmd = "$exe --zabbix-server $pServer --port $pPort --host '$pHost' --key $lKey --value '$lValue'";
  if ( $main::gParameter{'Debug'} == DBG ) {
    $cmd .= ' -vv';
    &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, '      ' . $cmd);
  }

  if ( $main::gParameter{'Debug'} == DBG ) {
    system("$cmd >>" . $main::gParameter{'LogFile'} . " 2>&1");
  }
  else {
    system("$cmd >/dev/null 2>&1");
  }
  my $ret = $?;
  if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      (ret=$ret)."); }

  if ( ($ret >> 8) == 0 ) {
  	# Everything is OK
  }
  elsif ( ($ret >> 8) == 2 ) {
  	# Change in Zabbix 2.1.7
  	# [ZBXNEXT-935] zabbix sender exit status now better reflects the operation result:
  	# success: 0
  	# partial success: 2
  	# failure: 1
    if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      Partial success (ret=$ret / rc=$rc).\n"); }
  }
  elsif ( $ret == -1 ) {
    $rc = 1327;
    if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, ERR, "      Failed to execute (ret=$ret / rc=$rc).\n$!\n"); }
    return $rc;
  }
  elsif ( $ret & 127 ) {
    $rc = 1328;
    if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, ERR, "      " . sprintf("Child died with signal %d, %s coredump", ($ret & 127), ($ret & 128) ? 'with' : 'without') . " (ret=$ret / rc=$rc)."); }
    return $rc;
  }
  else {
    $rc = 1329;
    if ( $main::gParameter{'Debug'} == DBG ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, ERR, "      " . sprintf("Child exited with value %d", $ret >> 8) . " (ret=$ret / rc=$rc)."); }
    return $rc;
  }

  return $rc;
}

# ----------------------------------------------------------------------
sub readConfigurationFile
# ----------------------------------------------------------------------
{
  my $file = $_[0];
  my $rc = 0;

  if ( ! -e "$file" ) {
    $rc = 1323;
    print "Configuration file $file does not exist (rc=$rc).\n";
    &cleanAndQuit($rc);
  }

  if ( ! -r "$file" ) {
    $rc = 1322;
    print "Cannot read configuration file $file (rc=$rc).\n";
    &cleanAndQuit($rc);
  }

  my @config;
  if ( ! open(CONFIG, $file) ) {
    $rc = 1321;
    print "Cannot open file $file (rc=$rc).\n$!\n";
    &cleanAndQuit($rc);
  }
  while ( <CONFIG> ) {
    chomp;                  # remove newlines
    s/^\s*#.*//;            # remove comment
    s/^\s+//;               # remove leading white space
    s/\s+$//;               # remove trailing white space
    next unless length;     # anything left?
    push(@config, $_);
  }
  close(CONFIG);
  return @config;
}

# ----------------------------------------------------------------------
sub getSections
# ----------------------------------------------------------------------
{
  my @config = @_;

  my @sections;
  foreach my $line ( @config ) {

    if ( $line =~ /^\[(.+)\]/ ) {
      push(@sections, $1);
    }
  }

  return @sections;
}

# ----------------------------------------------------------------------
sub readSection
# ----------------------------------------------------------------------
{
  my $config = shift;
  my $section = shift;

  my %section;
  my $in_section = 0;
  foreach my $line ( @$config ) {

    if ( $line =~ /^\[(.+)\]/ ) {
      if ( $1 eq $section ) {
        $in_section = 1;
        next;
      }
      else {
        if ( $in_section == 1 ) {
          last;
        }
        $in_section = 0;
        next;
      }
    }

    if ( $in_section && ( $line =~ /^(\S+)\s*=\s*(.+)\s*$/ ) ) {
      $section{$1} = $2;
    }
  }

  return %section;
}

# ----------------------------------------------------------------------
sub getDefaultParameter
# ----------------------------------------------------------------------
{
  my $config = shift;

  my %hDefaults;
  my $rc = 0;

  if ( $Config{'osname'} =~ /^MSWin/ ) {
    $hDefaults{'ClusterLog'}       = 'C:\mysql\bin\cluster-logs\ndb_1_cluster.log';
    $hDefaults{'Socket'}           = 'C:\Program Files\MySQL\MySQL Server 5.5\data\mysql.sock';
    $hDefaults{'LogFile'}          = 'C:\Temp\FromDualMySQLagent.log';
    $hDefaults{'CacheFileBase'}    = 'C:\Temp\FromDualAgentCache';
    $hDefaults{'AgentLockFile'}    = 'C:\Temp\FromDualMySQLagent.lock';
  }
  elsif ( $Config{'osname'} eq 'linux' ) {
    $hDefaults{'ClusterLog'}       = '/var/lib/mysql-cluster/ndb_1_cluster.log';
    $hDefaults{'Socket'}           = '/var/lib/mysql/mysql.sock';
    $hDefaults{'LogFile'}          = '/var/log/zabbix/FromDualMySQLagent.log';
    $hDefaults{'CacheFileBase'}    = '/tmp/FromDualAgentCache';
    $hDefaults{'AgentLockFile'}    = '/tmp/FromDualMySQLagent.lock';
  }
  else {
    $rc = 1331;
    print 'O/S ' . $Config{'osname'} . " is not supported yet (rc=$rc).\n";
    &cleanAndQuit($rc);
  }

  $hDefaults{'Debug'}            = WARN;
  $hDefaults{'TimeShift'}        = 0.0;
  $hDefaults{'Disabled'}         = 'false';
  $hDefaults{'Modules'}          = 'mysql myisam process';
  $hDefaults{'MysqlHost'}        = '127.0.0.1';
  $hDefaults{'MysqlPort'}        = '3306';
  $hDefaults{'Password'}         = '';
  $hDefaults{'PidFile'}          = '';
  $hDefaults{'Type'}             = 'mysqld';
  $hDefaults{'Username'}         = 'root';
  $hDefaults{'ZabbixServer'}     = '';

  $hDefaults{'ZabbixServerPort'} = '10051';

  $hDefaults{'MaaS'}             = 'off';
  $hDefaults{'Hash'}             = '';
  $hDefaults{'Methode'}          = 'https';
  $hDefaults{'Url'}              = 'https://support.fromdual.com/maas/receiver.php';

  # Read default from configuration
  my %default = &readSection(\@$config, 'default');

  foreach my $key ( keys %default ) {

    if ( $default{$key} ne '' ) {
      $hDefaults{$key} = $default{$key};
    }
  }

  return %hDefaults
}

# ----------------------------------------------------------------------
sub getParameter
# ----------------------------------------------------------------------
{
  my $config = shift;
  my $section = shift;

  # The logic is:
  # - Get defaults
  # - Get section parameters
  # - Overwrite defaults by section values
  # - Return defaults as section parameters!

  my %defaults = &getDefaultParameter(\@$config);

  if ( $defaults{'Debug'} == DBG ) {
    &FromDualMySQLagent::mylog($defaults{'LogFile'}, DBG, "The defaults parameters of this configuration file are:");
    foreach my $p ( keys %defaults ) {

      # This would be possibly a security problem...
      if ( $p eq 'Password' ) {
        &FromDualMySQLagent::mylog($defaults{'LogFile'}, DBG, "  $p - ********");
      }
      else {
        &FromDualMySQLagent::mylog($defaults{'LogFile'}, DBG, "  $p - $defaults{$p}");
      }
    }
  }

  my %parameter = &readSection(\@$config, $section);

  if ( $defaults{'Debug'} == DBG ) {
    &FromDualMySQLagent::mylog($defaults{'LogFile'}, DBG, "The parameters of section $section are:");
    foreach my $p ( keys %parameter ) {

      # This would be possibly a security problem...
      if ( $p eq 'Password' ) {
        &FromDualMySQLagent::mylog($defaults{'LogFile'}, DBG, "  $p - ********");
      }
      else {
        &FromDualMySQLagent::mylog($defaults{'LogFile'}, DBG, "  $p - $parameter{$p}");
      }
    }
  }

  foreach my $key ( keys %parameter ) {

    if ( $parameter{$key} ne '' ) {
      $defaults{$key} = $parameter{$key};
    }
  }
  $defaults{'Hostname'} = $section;
  $defaults{'CacheFile'} = $defaults{'CacheFileBase'} . '.' . $section . '.cache';
  return %defaults;
}

# ----------------------------------------------------------------------
sub mylog
# ----------------------------------------------------------------------
{
  my $logfile = shift;
  my $lvl = shift;
  my $str = shift;

  my $rc = 0;

  if ( (! defined $logfile) || ($logfile eq '') ) {
    $rc = 1306;

    my ($seconds, $microseconds) = gettimeofday();
    my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = localtime($seconds);
    printf("%5d:%4d-%02d-%02d %02d:%02d:%02d.%03d - %s: %s\n", $$, $year+1900,$mon+1,$mday,$hour,$min,$sec, $microseconds / 1000, 'ERR', "Logfile is not defined or not specified (rc=$rc).");
    &cleanAndQuit($rc);
  }

  # Logfile does not exist try to create it
  # This has to be done before the -w check because otherwiese -w would fail the
  # time!
  if ( ! -e $logfile ) {
    $rc = 1324;
    if ( ! open(LOG, ">>", $logfile) ) {
      print "Cannot open file $logfile (rc=$rc).\n$!\n";
      &cleanAndQuit($rc);
    }
    close(LOG);
  }

  if ( ! -w $logfile ) {
    $rc = 1307;
    print "Cannot write to logfile $logfile. Please check permissions (rc=$rc).\n";
    &cleanAndQuit($rc);
  }

  my $severity = '';
  if ( $lvl == INFO ) {
    $severity = 'INFO';
  }
  elsif ( $lvl == WARN ) {
    $severity = 'WARN';
  }
  elsif ( $lvl == ERR ) {
    $severity = 'ERR ';
  }
  elsif ( $lvl == DBG ) {
    $severity = 'DBG ';
  }
  else {
    $severity = 'UNKN';
  }

  my $PID = $$;

  if ( ! open(LOG, ">>", $logfile) ) {
    $rc = 1325;
    &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, ERR, "   Cannot open file $logfile\n$!\n");
    &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, ERR, "FromDual Performance Monitor for MySQL run finshed (rc=$rc).");
    &cleanAndQuit($rc);
  }
  my ($seconds, $microseconds) = gettimeofday();
  my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = localtime($seconds);
  printf(LOG "%5d:%4d-%02d-%02d %02d:%02d:%02d.%03d - %s: %s\n", $PID, $year+1900,$mon+1,$mday,$hour,$min,$sec, $microseconds / 1000, $severity, $str);
  close(LOG);

  # Do a log rotate if necessary if logfile is > 10M
  my $filesize = -s $logfile;
  if ( $filesize > 10*1024*1024 ) {
    my $new = $logfile . '.rotated';
    rename($logfile, $new) or system("mv", $logfile, $new);
  }
}

# ----------------------------------------------------------------------
sub getDatabaseConnection
# ----------------------------------------------------------------------
{
  my $rc = 0;

  # Check if perl DBD/mysql is available
  # Should be done on a higher level?
  eval {
    require DBD::mysql;
  };
  if ( $@ ) {
    $rc = 1312;
    &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, WARN, "Warning: No DBD::mysql found. Please install DBD::mysql (rc=$rc).");
    # Should we return an error instead?
    &FromDualMySQLagent::removeAgentLock($main::gParameter{'LogLevel'}, $main::gParameter{'LogFile'});
    &cleanAndQuit($rc);
  }

  # We do not specify a database, it could be we do not have the
  # privileges for
  my $dbh = DBI->connect('DBI:mysql:database=;host=' . $main::gParameter{'MysqlHost'} . ';port=' . $main::gParameter{'MysqlPort'}
                       , $main::gParameter{'Username'}, $main::gParameter{'Password'}
                       , { 'RaiseError' => 0, 'PrintError' => 0 }
                         );

  if ( defined $DBI::err ) {
    &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, ERR, 'DBI connect with database=mysql, host=' . $main::gParameter{'MysqlHost'} . ', port=' . $main::gParameter{'MysqlPort'} . ' and user=' . $main::gParameter{'Username'} . ' failed: ' . $DBI::errstr);
  }

  return $dbh;
}

# ----------------------------------------------------------------------
sub releaseDatabaseConnection()
# ----------------------------------------------------------------------
{
  my $dbh = shift;
  $dbh->disconnect();
}

# ----------------------------------------------------------------------
sub initValues
# ----------------------------------------------------------------------
{
  my $variables_ref = shift;
  my $variables_to_send_ref = shift;

  foreach my $key ( @$variables_to_send_ref ) {
    $$variables_ref{$key} = 0;
  }
}

# ----------------------------------------------------------------------
sub getGlobalVariables
# ----------------------------------------------------------------------
{
  my $dbh = shift;
  my $variables_ref = shift;

  if ( $main::gParameter{'Debug'} >= INFO ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, INFO, '    ' . (caller(0))[3]); }

  my $sql = 'SHOW GLOBAL VARIABLES';
  my $sth = $dbh->prepare($sql);
  if ( $sth->execute() ) {
    while ( my $ref = $sth->fetchrow_hashref() ) {
      $$variables_ref{$ref->{'Variable_name'}} = $ref->{'Value'};
    }
    $sth->finish();
  }

  if ( $main::gParameter{'Debug'} == DBG ) {
    &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, '    Global Variables:');
    foreach my $key ( keys %$variables_ref ) {
      &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      key: $key, value: $$variables_ref{$key}");
    }
  }
}

# ----------------------------------------------------------------------
sub getGlobalStatus
# ----------------------------------------------------------------------
{
  my $dbh = shift;
  my $status_ref = shift;

  if ( $main::gParameter{'Debug'} >= INFO ) { &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, INFO, '    ' . (caller(0))[3]); }

  my $sql = 'SHOW /*!50000 GLOBAL */ STATUS';
  my $sth = $dbh->prepare($sql);
  if ( $sth->execute() ) {
    while ( my $ref = $sth->fetchrow_hashref() ) {
      $$status_ref{$ref->{'Variable_name'}} = $ref->{'Value'};
    }
    $sth->finish();
  }

  if ( $main::gParameter{'Debug'} == DBG ) {
    &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, '    Global Status:');
    foreach my $key ( keys %$status_ref ) {
      &FromDualMySQLagent::mylog($main::gParameter{'LogFile'}, DBG, "      key: $key, value: $$status_ref{$key}");
    }
  }
}

# ----------------------------------------------------------------------
sub removeAgentLock
# ----------------------------------------------------------------------
{
  my $pLogLevel = $_[0];
  my $pLogFile  = $_[1];

  if ( $pLogLevel >= INFO ) { &FromDualMySQLagent::mylog($pLogFile, INFO, '  ' . (caller(0))[3]); }

  my $rc = 0;

  if ( $gRemoveLockFile == 1 ) {

    my $lAgentLockFile = $main::gParameter{'AgentLockFile'};
    if ( ! unlink($lAgentLockFile) ) {
      $rc = 1330;
      if ( $pLogLevel >= ERR ) { &FromDualMySQLagent::mylog($pLogFile, ERR, "    Cannot delete lock file $lAgentLockFile (rc=$rc).\n$!\n"); }
      # no exit here!
    }
  }
  else {
    if ( $pLogLevel >= INFO ) { &FromDualMySQLagent::mylog($pLogFile, INFO, "    gRemoveLockFile is not set."); }
  }
  return $rc;
}

1;
__END__
