<?php

//
// Copyright (C) 2013, 2014 FromDual GmbH
//
// This program is commercial software. Use of this software is governed
// by your applicable license agreement with FromDual GmbH
//

/*

  safeServer($aServer)
  readServerById($pServerId)
  testServer($dbh, $aServer, $abort = false)
  deleteServer($dbh, $pServerId)
  getServersWithFilter($dbh, $aFilter)
  addServer($dbh, $aServer)
  setServer($dbh, $aServer)
	printHostnameCheck($aServer, $arr)
	printHostPingCheck($aServer, $arr)
	printSshConnectCheck($aServer, $aServer, $arr)
	retrieveShortFile($pUser, $pHost, $pFile)

*/

// -------------------------------------------------------------------
function safeServer($aServer)
// -------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  list($rc, $dbh) = openRepositoryDatabase();

  // Section was newly added
  if ( $aServer['server_id'] == 0 ) {
    list($rc, $cnt) =  addServer($dbh, $aServer);
  }
  // Section was modified
  else {
    list($rc, $cnt) = updateServer($dbh, $aServer);
  }

  closeRepositoryDatabase($dbh);
  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $cnt);
}

// -------------------------------------------------------------------
function readServerById($pServerId)
// -------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $aServer = array();

  list($ret, $dbh) = openRepositoryDatabase();
  if ( $ret != OK ) {
    $rc = 101;
    $msg = "Cannot open configuration database (rc=$rc).";
    print "<p>$msg</p>\n";
    logMessage(LOG_ERR, "$msg");
    logMessage(LOG_ERR, print_r(debug_backtrace(), true));
  }
  else {
    list($ret, $aServer) = getServerById($dbh, $pServerId);

    if ( ($ret != OK) || (count($aServer) == 0) ) {
      $rc = 113;
      $msg = "Server $pServerId does not exist (rc=$rc).";
      print "<p>$msg</p>\n";
      logMessage(LOG_ERR, "$msg");
      logMessage(LOG_ERR, print_r(debug_backtrace(), true));
    }

    closeRepositoryDatabase($dbh);
  }
  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $aServer);
}

// -------------------------------------------------------------------
function testServer($dbh, $aServer, $abort = false)
// parameter: abort = {true|false}
// return   : array(failed_checks, array('check', array($rc, $msg)))
// -------------------------------------------------------------------
{
	global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
	$gIndention += 2;

  $rc = OK;
  $aChecks = array();
  $failed_checks = 0;

  list($ret, $aServer) = getServerById($dbh, $aServer['server_id']);

  // Check hostname

  $check = 'hostname';
  list($rc, $msg) = checkHostname($aServer);
  $aChecks[$check] = array($rc, $msg);
  if ( $rc != 0 ) {
    $failed_checks++;
    if ( $abort === true ) {
      return array($failed_checks, $aChecks);
    }
  }


  // Check host ping

  $check = 'host_ping';
  list($rc, $msg) = checkPing($aServer['name']);
  // Skip nodes which are not created yet!
  if ( $aServer['server_id'] != 0 ) {
    $ret = writeCheck($aServer['server_id'], 'server', 'host_ping', $rc);
  }
  $aChecks[$check] = array($rc, $msg);
  if ( $rc != 0 ) {
    $failed_checks++;
    if ( $abort === true ) {
      return array($failed_checks, $aChecks);
    }
  }


  // Check ssh connect

  $check = 'ssh_connect';
  list($rc, $msg) = checkSshConnect($aServer);
  $aChecks[$check] = array($rc, $msg);
  if ( $rc != 0 ) {
    $failed_checks++;
    if ( $abort === true ) {
      return array($failed_checks, $aChecks);
    }
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__);
  return array($failed_checks, $aChecks);
}

// -------------------------------------------------------------------
function deleteServer($dbh, $pServerId)
// -------------------------------------------------------------------
{
	global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
	$gIndention += 2;

  $rc = OK;
  $sql = "DELETE FROM servers WHERE server_id = :server_id";

  $stmt = $dbh->prepare($sql);
  $stmt->bindValue(':server_id', $pServerId, PDO::PARAM_INT);

  try {
    $stmt->execute();
    $cnt = $stmt->rowCount();
  }
  catch (PDOException $e) {
    $rc = 889;
    $cnt = $e->getMessage();
  }
  $stmt->closeCursor();

	$gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $cnt);
}

// ---------------------------------------------------------------------
function getServersWithFilter($dbh, $aFilter = array())
// ---------------------------------------------------------------------
{
	global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
	$gIndention += 2;

  $rc = OK;
  $aServers = array();

  $sql = "SELECT * FROM servers WHERE 1\n";

  foreach ( $aFilter as $key => $value ) {
    $sql .= "  AND $key = :$key\n";
  }
  logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

  $stmt = $dbh->prepare($sql);

  foreach ( $aFilter as $key => $value ) {
    $stmt->bindValue(":$key", $value);
  }

  try {
    $stmt->execute();

    while ( $result = $stmt->fetch(PDO::FETCH_ASSOC) ) {
      $aServers[$result['server_id']] = $result;
    }
  }
  catch (PDOException $e) {
    $rc = 815;
    logMessage(LOG_ERR, $e->getMessage() . " (rc=$rc)");
    $aServers = $e->getMessage() . " (rc=$rc)";
  }
  $stmt->closeCursor();

	$gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $aServers);
}

// ---------------------------------------------------------------------
function getAllServers($dbh)
// ---------------------------------------------------------------------
{
  return getServersWithFilter($dbh, array());
}

// ---------------------------------------------------------------------
function getServerById($dbh, $pServerId)
// ---------------------------------------------------------------------
{
  list($ret, $aServers) = getServersWithFilter($dbh, array('server_id' => $pServerId));
  return array($ret, array_shift($aServers));
}

// ---------------------------------------------------------------------
function getServerByName($dbh, $pServerName)
// ---------------------------------------------------------------------
{
  list($ret, $aServers) = getServersWithFilter($dbh, array('name' => $pServerName));
  return array($ret, array_shift($aServers));
}

// ---------------------------------------------------------------------
function getServerByIpAddress($dbh, $pDefaultIp)
// ---------------------------------------------------------------------
{
  list($ret, $aServers) = getServersWithFilter($dbh, array('default_ip' => $pDefaultIp));
  return array($ret, array_shift($aServers));
}

// ---------------------------------------------------------------------
function addServer($dbh, $aServer)
// ---------------------------------------------------------------------
{
	global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
	$gIndention += 2;

  $rc = OK;
  $server_id = 0;

  $sql = "INSERT INTO servers ('name', 'default_ip', 'os_user', 'cluster_id', 'last_change_ts')
VALUES (:name, :default_ip, :os_user, :cluster_id, :last_change_ts)";
  $stmt = $dbh->prepare($sql);
  $stmt->bindValue(':name', $aServer['name']);
  $stmt->bindValue(':default_ip', $aServer['default_ip']);
  $stmt->bindValue(':os_user', $aServer['os_user']);
  $stmt->bindValue(':cluster_id', $aServer['cluster_id']);
  $stmt->bindValue(':last_change_ts', time());

  try {
    $stmt->execute();
    $server_id =  $dbh->lastInsertId();
  }
  catch (PDOException $e) {
    $rc = 914;
    return array($rc, $e->getMessage());
  }
  $stmt->closeCursor();

	$gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $server_id);
}

// ---------------------------------------------------------------------
function setServer($dbh, $aServer)
// ---------------------------------------------------------------------
{
	global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
	$gIndention += 2;

  $rc = OK;

  $sql = "UPDATE servers
   SET name = :name
     , default_ip = :default_ip
     , os_user = :os_user
     , cluster_id = :cluster_id
     , last_change_ts = :last_change_ts
 WHERE server_id = :server_id";
  $stmt = $dbh->prepare($sql);
  $stmt->bindValue(':server_id', $aServer['server_id']);
  $stmt->bindValue(':name', $aServer['name']);
  $stmt->bindValue(':default_ip', $aServer['default_ip']);
  $stmt->bindValue(':os_user', $aServer['os_user']);
  $stmt->bindValue(':cluster_id', $aServer['cluster_id']);
  $stmt->bindValue(':last_change_ts', time());

  try {
    $stmt->execute();
  }
  catch (PDOException $e) {
    $rc = 915;
    return array($rc, $e->getMessage());
  }
  $stmt->closeCursor();

	$gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, 'OK');
}

// ---------------------------------------------------------------------
function printHostnameCheck($aServer, $arr)
// ---------------------------------------------------------------------
{
	global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
	$gIndention += 2;
	
	$rc = OK;

  print "<p>Checking hostname " . $aServer['name'] . "...<br />\n";
  if ( $arr[0] == 0 ) {
    print '&nbsp;&nbsp;Hostname ' . $aServer['name'] . " is <span style='color:green;'><strong>OK</strong></span>.<br />\n";
    print "&nbsp;&nbsp;Reply is: " . $arr[1] . "</p>\n";
  }
  else {
    print "&nbsp;&nbsp;<span style='color:red;'><strong>ERROR</strong></span>: " . $arr[1] . "</p>\n";
  }
	$gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return $rc;
}

// ---------------------------------------------------------------------
function printHostPingCheck($aServer, $arr)
// ---------------------------------------------------------------------
{
	global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
	$gIndention += 2;

	$rc = OK;
	
  print "<p>Pinging host " . $aServer['name'] . "...<br />\n";
  if ( $arr[0] == 0 ) {
    print '&nbsp;&nbsp;Ping to ' . $aServer['name'] . " is <span style='color:green;'><strong>OK</strong></span>.</p>\n";
  }
  else {
    print "&nbsp;&nbsp;<span style='color:red;'><strong>ERROR</strong></span>: " . $arr[1] . "</p>\n";
  }
	$gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return $rc;
}

// ---------------------------------------------------------------------
function printSshConnectCheck($aServer, $aServer, $arr)
// ---------------------------------------------------------------------
{
	global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
	$gIndention += 2;

	$rc = OK;
	
  print "<p>Checking ssh connect to host " . $aServer['name'] . "...<br />\n";
  if ( $arr[0] == 0 ) {
    print '&nbsp;&nbsp;Ssh connect to host ' . $aServer['name'] . " is <span style='color:green;'><strong>OK</strong></span>.<br />\n";
    print "&nbsp;&nbsp;Reply is " . $arr[1] . ".</p>\n";
  }
  else {
    print "&nbsp;&nbsp;<span style='color:red;'><strong>ERROR</strong></span>: " . $arr[1] . "<br />\n";

    // Check for customers public key file first

    $whoami = whoami();
    $home = getHomeDir($whoami);

    $aSshPublicKeyFiles = array(
      0 => $home . '/.ssh/id_rsa.pub'
    , 1 => $home . '/.ssh/id_dsa.pub'
    );

    $pk_exists = false;
    foreach ( $aSshPublicKeyFiles as $publicKeyFile ) {

      if ( file_exists($publicKeyFile) ) {
        $pk_exists = true;
        break;
      }
    }

    // RSA or DSA key exists
    if ( $pk_exists !== true) {
      print "&nbsp;&nbsp;You do NOT have a public key file yet. Generate it <a href='index.php?f=create&o=public_key'>here</a>...<br /></p>\n";
    }
    // User seems to have a public key file but it is not distributed.
    else {

			$aOperatingSystem = getOperatingSystem();

			// Get users shell
			$cmd = "grep $whoami /etc/passwd | cut -d: -f7";
			$shell = exec($cmd, $stdout, $ret);

			if ( $shell == '/bin/false'
				|| $shell == '/usr/sbin/nologin'
				|| $shell == '/sbin/nologin'
					) {
				?>
				<p>Please change your shell first as follows:</p>
				<?php
				// We should switch to root not to apache user
				if ( $aOperatingSystem['Distributor ID'] == 'CentOS' ) {
					print "<pre>su - root\n";
				}
				else {
					print "<pre>sudo su - root\n";
				}
				print "chsh -s /bin/bash $whoami</pre>";
			}

      print "&nbsp;&nbsp;Copy the ssh public key file of user $whoami to the authorized_keys file of user " . $aServer['os_user'] . " on host " . $aServer['name'] . ".<br />\n";

      if ( $aOperatingSystem['Distributor ID'] == 'CentOS' ) {
        print "<pre>su - $whoami\n";
      }
      else {
        print "<pre>sudo su - $whoami\n";
      }
      print "ssh-copy-id -i ~/.ssh/id_?sa.pub " . $aServer['os_user'] . "@" . $aServer['name'] . "</pre>\n";
    }   // no public key file
  }
	$gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
	return $rc;
}

// -------------------------------------------------------------------
function retrieveShortFile($pUser, $pHost, $pFile)
// -------------------------------------------------------------------
{
	global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
	$gIndention += 2;

  $rc = OK;

  $cmd = "cat $pFile 2>&1";
  list($rc, $output) = runSshCommand($pUser, $pHost, $cmd);
	$gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
	return array($rc, $output);
}

?>
