<?php

//
// Copyright (C) 2013, 2014 FromDual GmbH
//
// This program is commercial software. Use of this software is governed
// by your applicable license agreement with FromDual GmbH
//

/*

	openRepositoryDatabase()
	closeRepositoryDatabase($dbh)
	upgrade100to110($dbh)
	upgradeRepositoryDatabase()
	createTableCluster($dbh)
	createTableVip($dbh)
	createTableInterface($dbh)
	createTableServer($dbh)
	createRepositoryDatabase()
	createTableUser($dbh)
	setEncryptedLicense($dbh, $pLicense)
	getEncryptedLicense($dbh)
	getMocIdentifier($dbh)
	decryptLicense($pKey, $pEncryptedLicense)
	readEncryptLicenseFromFile($pFile)
	generateMocIdentifier()
	createTableMocIdentifier($dbh)
  getRepositoryVersion($dbh)

*/

// ---------------------------------------------------------------------
function openRepositoryDatabase()
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  global $gRepository;

  $rc = OK;
  $dbh = null;
  try {

		// Check if file exists
		if ( ! file_exists($gRepository) ) {
			$rc = 936;
			$msg = "Repository database $gRepository does not exist (rc=$rc).";
			logMessage(LOG_ERR, $msg);
			throw new Exception($msg);
		}

		// Check if file is writeables
		if ( ! is_writeable($gRepository) ) {
			$rc = 937;
			$msg = "Repository database $gRepository is not writeable for user " . whoami() . " (rc=$rc).";
			logMessage(LOG_ERR, $msg);
			throw new Exception($msg);
		}

		$user     = null;
		$password = null;
		$dsn      = 'sqlite:' . $gRepository;

		try {
			$dbh = new PDO($dsn, $user, $password);
		}
		catch (PDOException $e) {
			$rc = 803;
			$msg = 'Connection to repository database failed: ' . $e->getMessage() . " (rc=$rc).";
			logMessage(LOG_ERR, $msg);
			throw new Exception($msg);
		}

		$dbh->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
		$dbh->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
	}
	catch (Exception $e) {
		print "<p>" . $e->getMessage() . "</p>\n";
		$dbh = null;
	}

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $dbh);
}

// ---------------------------------------------------------------------
function closeRepositoryDatabase($dbh)
// ---------------------------------------------------------------------
{
  $dbh = null;
}

// ---------------------------------------------------------------------
function upgrade100to110($dbh)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;

  // http://sqlite.org/lang_altertable.html


  $sql = "BEGIN TRANSACTION";
  logMessage(LOG_DEBUG, $sql);

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 980;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }

  // ----

  $sql = "ALTER TABLE nodes ADD COLUMN master_id INTEGER";
  logMessage(LOG_DEBUG, $sql);

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 981;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }

  $sql = "UPDATE nodes SET master_id = 0 WHERE role_id != :role_id";
  logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

  $stmt = $dbh->prepare($sql);

  $role = MASTER;
  $stmt->bindParam(':role_id', $role, PDO::PARAM_INT);
  try {
    $stmt->execute();
  }
  catch (PDOException $e) {
    $rc = 969;
    $msg = $e->getMessage() . " (rc=$rc).";
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }
  $stmt->closeCursor();

  $sql = "SELECT slave.node_id AS slave_node_id, master.node_id AS master_node_id
  FROM nodes AS slave
  JOIN nodes AS master ON slave.cluster_id = master.cluster_id
 WHERE slave.role_id = :slave_role_id
   AND master.role_id = :master_role_id
";
  logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

  $stmt = $dbh->prepare($sql);

  $role = SLAVE;
  $stmt->bindParam(':slave_role_id', $role, PDO::PARAM_INT);
  $role = MASTER;
  $stmt->bindParam(':master_role_id', $role, PDO::PARAM_INT);

  try {
    $stmt->execute();
  }
  catch (PDOException $e) {
    $rc = 968;
    $msg = $e->getMessage() . " (rc=$rc).";
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }

  while ( $result = $stmt->fetch(PDO::FETCH_ASSOC) ) {

    $sql2 = "UPDATE nodes SET master_id = :master_id where node_id = :node_id";
		logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql2)));

    $stmt2 = $dbh->prepare($sql2);

    $stmt2->bindParam(':master_id', $result['master_node_id'], PDO::PARAM_INT);
    $stmt2->bindParam(':node_id', $result['slave_node_id'], PDO::PARAM_INT);
    try {
      $stmt2->execute();
    }
    catch (PDOException $e) {
      $rc = 967;
      $msg = $e->getMessage() . " (rc=$rc).";
      logMessage(LOG_ERR, $msg);
      print "<p>$msg</p>\n";
    }
    $stmt2->closeCursor();
  }
  $stmt->closeCursor();

  // ----

  $sql = "ALTER TABLE servers ADD COLUMN last_change_ts INTEGER";
  logMessage(LOG_DEBUG, $sql);

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 982;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }

  $sql = "UPDATE servers SET last_change_ts = :now";
  logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));
 
  $stmt = $dbh->prepare($sql);
  $t = time();
  $stmt->bindParam(':now', $t, PDO::PARAM_INT);
  try {
    $stmt->execute();
  }
  catch (PDOException $e) {
    $rc = 966;
    $msg = $e->getMessage() . " (rc=$rc).";
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }
  $stmt->closeCursor();

  // ----

  $sql = "ALTER TABLE vips ADD COLUMN last_change_ts INTEGER";
  logMessage(LOG_DEBUG, $sql);

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 983;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }

  $sql = "UPDATE vips SET last_change_ts = :now";
  logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

  $stmt = $dbh->prepare($sql);
  $t = time();
  $stmt->bindParam(':now', $t, PDO::PARAM_INT);
  try {
    $stmt->execute();
  }
  catch (PDOException $e) {
    $rc = 965;
    $msg = $e->getMessage() . " (rc=$rc).";
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }
  $stmt->closeCursor();

  // ----

  // checks: node_id -> unit_id

  $sql = "ALTER TABLE checks RENAME TO checks_100";
  logMessage(LOG_DEBUG, $sql);

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 984;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }

  $sql = "CREATE TABLE checks (
  unit_id                      INTEGER
, type                         TEXT
, name                         TEXT
, last_check_ts                INTEGER
, last_check_status            TEXT
, last_successful_check_ts     INTEGER
, last_successful_check_status TEXT
, PRIMARY KEY (unit_id, name)
)";
  logMessage(LOG_DEBUG, $sql);

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 985;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }

  $sql = "DROP INDEX name_dc";
  logMessage(LOG_DEBUG, $sql);

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 973;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }

  $sql = "CREATE INDEX name_dc ON checks (name)";
  logMessage(LOG_DEBUG, $sql);

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 976;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }

  $sql = "INSERT INTO checks
SELECT node_id, type, name, last_check_ts, last_check_status, last_successful_check_ts, last_successful_check_status
FROM checks_100";
  logMessage(LOG_DEBUG, $sql);

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 977;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }

  $sql = "DROP TABLE checks_100";
  logMessage(LOG_DEBUG, $sql);

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 978;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }

  // ----

  $sql = "CREATE TABLE jobs (
  job_id         INTEGER
, name           TEXT
, server         TEXT
, pid            INTEGER
, start_ts       INTEGER
, status         TEXT
, check_interval INTEGER
, last_check_ts  INTEGER
, end_ts         INTEGER
, error_code     INTEGER
, error_message  TEXT
, command        TEXT
, PRIMARY KEY (job_id)
)";
  logMessage(LOG_DEBUG, $sql);

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 979;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }

  // ----

  $sql = "INSERT INTO versions (version, mr_version, timestamp) VALUES (:version, :mr_version, :timestamp)";
  logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

  $stmt = $dbh->prepare($sql);

  $version    = 'v1.1';
  $mr_version = 110;
  $timestamp  = time();

  $stmt->bindParam(':version', $version, PDO::PARAM_STR);
  $stmt->bindParam(':mr_version', $mr_version, PDO::PARAM_INT);
  $stmt->bindParam(':timestamp', $timestamp, PDO::PARAM_INT);
  try {
    $stmt->execute();
  }
  catch (PDOException $e) {
    $rc = 974;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }
  $stmt->closeCursor();

  // ----

  // table vips
  // fo_sync_only INTEGER: Start failover only if both nodes are in sync.
  //                       --> Failover possibly never happens
  // fo_wait_sync INTEGER: Start failover anyway but wait with starting
	//                       VIP until both nodes are in sync.
	//                       --> Service is possibly down for long time!
  
  $sql = "ALTER TABLE vips ADD COLUMN fo_sync_only INTEGER";
  logMessage(LOG_DEBUG, $sql);

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 943;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }

	$sql = "ALTER TABLE vips ADD COLUMN fo_wait_sync INTEGER";
  logMessage(LOG_DEBUG, $sql);

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 944;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }

  $sql = 'UPDATE vips SET fo_sync_only = 1, fo_wait_sync = 1';
  logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

  $stmt = $dbh->prepare($sql);
  try {
    $stmt->execute();
  }
  catch (PDOException $e) {
    $rc = 945;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }
  $stmt->closeCursor();
  
  // ----

	$sql = "ALTER TABLE servers ADD COLUMN myenv_basedir TEXT";
  logMessage(LOG_DEBUG, $sql);

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 986;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }

  // ----

	$sql = "CREATE UNIQUE INDEX name_ui ON nodes ( name )";
  logMessage(LOG_DEBUG, $sql);

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 966;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }

	// ----

  if ( $rc == OK ) {
    $sql = "COMMIT";
  }
  else {
    $sql = "ROLLBACK";
  }
  logMessage(LOG_DEBUG, $sql);

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 975;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return $rc;
}

// ---------------------------------------------------------------------
function upgradeRepositoryDatabase($dbh)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  global $gRepository;

  $rc = OK;

  list($ret, $aVersion) = getRepositoryVersion($dbh);
  if ( $ret != OK ) {
    return $ret;
  }
	logMessage(LOG_DEBUG, 'Repository version is: ' . $aVersion['version']);

	// todo: We have to loop here over many versions if necessary!
	
  switch ( $aVersion['mr_version'] ) {
  // Upgrade 1.0 to 1.1
  case 100:
    logMessage(LOG_INFO, 'Start upgrading from 1.0 to 1.1');

    // Backup repository
    if ( ! copy($gRepository, $gRepository . '.100') ) {
      $rc = 972;
      logMessage(LOG_ERR, "Backup of $gRepository failed (rc=$rc).");
      return $rc;
    }

    $ret = upgrade100to110($dbh);
    if ( $ret != OK ) {
      $rc = 971;
      $msg = "Upgrade from 1.0 to 1.1 failed (rc=$rc).";
      logMessage(LOG_ERR, $msg);
      print "<p>$msg</p>";
      if ( ! rename($gRepository . '.100', $gRepository) ) {
        $rc = 970;
        logMessage(LOG_ERR, "Rollback of $gRepository failed (rc=$rc).");
      }
    }
    else {
      if ( ! unlink($gRepository . '.100') ) {
        $rc = 964;
        logMessage(LOG_ERR, "Delete of $gRepository failed (rc=$rc).");
      }
      $msg = "Upgrade from 1.0 to 1.1 successful.";
      print "<p>$msg</p>";
      logMessage(LOG_INFO, $msg);
    }

    break;
  // case 110:
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return $rc;
}

// ---------------------------------------------------------------------
function createTableCluster($dbh)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;

  // Type:
  // 1: Master/Slave
  // 2: Master/Master
  // 3: Galera
  // 4: a/p Failover Cluster
  // 5: MySQL Cluster

  $sql = "CREATE TABLE IF NOT EXISTS clusters (
  cluster_id     INTEGER
, name           TEXT
, last_change_ts INTEGER
, type           INTEGER
, PRIMARY KEY (cluster_id)
)";

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 900;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    print "<p>$msg</p>\n";
  }

  $sql = "CREATE UNIQUE INDEX cluster_name ON clusters (name)";

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 901;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    print "<p>$msg</p>\n";
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return $rc;
}

// ---------------------------------------------------------------------
function createTableVip($dbh)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;

  // ipaddr_type:
  // 1: ifconfig
  // 2: ip addr show

  $sql = "CREATE TABLE IF NOT EXISTS vips (
  vip_id      INTEGER
, ip_address  TEXT
, name        TEXT
, ipaddr_type INTEGER
, interface   TEXT
, alias       INTEGER
, primary_id  INTEGER
, failover_id INTEGER
, location_id INTEGER
, cluster_id  INTEGER
, PRIMARY KEY (vip_id)
)";

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 902;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    print "<p>$msg</p>\n";
  }

  $sql = "CREATE UNIQUE INDEX ip_address ON vips (ip_address)";

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 903;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    print "<p>$msg</p>\n";
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return $rc;
}

// ---------------------------------------------------------------------
function createTableInterface($dbh)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;

  $sql = "CREATE TABLE IF NOT EXISTS interfaces (
  interface_id   INTEGER
, name           TEXT
, server_id      INTEGER
, PRIMARY KEY (interface_id)
)";

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 929;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    print "<p>$msg</p>\n";
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return $rc;
}

// ---------------------------------------------------------------------
function createTableServer($dbh)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;

  $sql = "CREATE TABLE IF NOT EXISTS servers (
  server_id      INTEGER
, name           TEXT
, default_ip     TEXT
, os_user        TEXT
, cluster_id     INTEGER
, PRIMARY KEY (server_id)
)";

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 928;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    print "<p>$msg</p>\n";
  }


  $sql = "CREATE INDEX server_name ON servers (name)";

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 821;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    print "<p>$msg</p>\n";
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return $rc;
}

// ---------------------------------------------------------------------
function createRepositoryDatabase()
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

	global $gRepository;
  $rc = OK;

  try {

		// This is only for the very fist time!
    if ( ! touch($gRepository) ) {
			$rc = 868;
			$msg = "Cannot create repository database (rc=$rc).";
			logMessage(LOG_ERR, $msg);
			throw new Exception($msg);
    }

		list($ret, $dbh) = openRepositoryDatabase();
	
		if ( $ret != OK ) {
			$rc = 874;
			$msg = "Cannot open repository database (rc=$rc).";
			logMessage(LOG_ERR, $msg);
			throw new Exception($msg);
		}

		
		// Table for version history

		$sql = "CREATE TABLE IF NOT EXISTS versions (
  version    TEXT PRIMARY KEY
, mr_version INTEGER
, timestamp  INTEGER
)";
		logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

		try {
			$dbh->exec($sql);
		}
		catch (PDOException $e) {
			$rc = 804;
			$msg = $e->getMessage() . " (rc=$rc).";
			logMessage(LOG_ERR, $msg);
			throw new Exception($msg);
		}

		$sql = "INSERT INTO versions (version, mr_version, timestamp) VALUES (:version, :mr_version, :timestamp)";
		logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

		$stmt = $dbh->prepare($sql);

		$version    = 'v1.0';
		$mr_version = 100;
		$timestamp  = time();

		$stmt->bindParam(':version', $version, PDO::PARAM_STR);
		$stmt->bindParam(':mr_version', $mr_version, PDO::PARAM_INT);
		$stmt->bindParam(':timestamp', $timestamp, PDO::PARAM_INT);
		try {
			$stmt->execute();
		}
		catch (PDOException $e) {
			$rc = 805;
			$msg = $e->getMessage() . " (rc=$rc).";
			logMessage(LOG_ERR, $msg);
			throw new Exception($msg);
		}
		$stmt->closeCursor();


		// Table for license

		$sql = "CREATE TABLE IF NOT EXISTS licenses (license TEXT)";
		logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

		try {
			$dbh->exec($sql);
		}
		catch (PDOException $e) {
			$rc = 806;
			$msg = $e->getMessage() . " (rc=$rc).";
			logMessage(LOG_ERR, $msg);
			throw new Exception($msg);
		}


		// Table for nodes

		// todo: name should be UNIQUE
		// add index on, INDEX (name)
		// create unique index name on node (name);

		$sql = "CREATE TABLE IF NOT EXISTS nodes (
  node_id                INTEGER
, name                   TEXT
, last_change_ts         INTEGER
, status                 TEXT
, node_type              TEXT
, hostname               TEXT
, basedir                TEXT
, datadir                TEXT
, my_cnf                 TEXT
, port                   INTEGER
, database_user          TEXT
, database_user_password TEXT
, error_log              TEXT
, pid_file               TEXT
, read_only              INTEGER
, server_id              INTEGER
, role_id                INTEGER
, cluster_id             INTEGER
, PRIMARY KEY (node_id)
)";
		logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

		try {
			$dbh->exec($sql);
		}
		catch (PDOException $e) {
			$rc = 802;
			$msg = $e->getMessage() . " (rc=$rc).";
			logMessage(LOG_ERR, $msg);
			throw new Exception($msg);
		}

		// Table for last database checks

		// lc  = last check
		// lsc = last successful check

		$sql = "CREATE TABLE IF NOT EXISTS checks (
  node_id                      INTEGER
, type                         TEXT
, name                         TEXT
, last_check_ts                INTEGER
, last_check_status            TEXT
, last_successful_check_ts     INTEGER
, last_successful_check_status TEXT
, PRIMARY KEY (node_id, name)
)";
		logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

		try {
			$dbh->exec($sql);
		}
		catch (PDOException $e) {
			$rc = 807;
			$msg = $e->getMessage() . " (rc=$rc).";
			logMessage(LOG_ERR, $msg);
			throw new Exception($msg);
		}

		$sql = "CREATE INDEX name_dc ON checks (name)";
		logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

		try {
			$dbh->exec($sql);
		}
		catch (PDOException $e) {
			$rc = 810;
			$msg = $e->getMessage() . " (rc=$rc).";
			logMessage(LOG_ERR, $msg);
			throw new Exception($msg);
		}

		$ret = createTableCluster($dbh);
		if ( $ret != OK ) {
			$rc = 814;
			$msg = "Cannot create table clusters (rc=$rc).";
			logMessage(LOG_ERR, $msg);
			throw new Exception($msg);
		}

		$ret = createTableVip($dbh);
		if ( $ret != OK ) {
			$rc = 811;
			$msg = "Cannot create table vips (rc=$rc).";
			logMessage(LOG_ERR, $msg);
			throw new Exception($msg);
		}

		$ret = createTableServer($dbh);
		if ( $ret != OK ) {
			$rc = 812;
			$msg = "Cannot create table servers (rc=$rc).";
			logMessage(LOG_ERR, $msg);
			throw new Exception($msg);
		}

		$ret = createTableUser($dbh);
		if ( $ret != OK ) {
			$rc = 813;
			$msg = "Cannot create table users (rc=$rc).";
			logMessage(LOG_ERR, $msg);
			throw new Exception($msg);
		}

		$ret = createTableMocIdentifier($dbh);
		if ( $ret != OK ) {
			$rc = 816;
			$msg = "Cannot create table moc_identifier (rc=$rc).";
			logMessage(LOG_ERR, $msg);
			throw new Exception($msg);
		}


		// generate MOC identifier

		$moc_identifier = generateMocIdentifier();

		$sql = "INSERT INTO moc_identifier (moc_identifier) VALUES (:moc_identifier)";
		logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

		$stmt = $dbh->prepare($sql);
		$stmt->bindValue(':moc_identifier', $moc_identifier, PDO::PARAM_STR);

		try {
			$stmt->execute();
		}
		catch (PDOException $e) {
			$rc = 875;
			$msg = $e->getMessage() . " (rc=$rc).";
			logMessage(LOG_ERR, $msg);
			throw new Exception($msg);
		}

		closeRepositoryDatabase($dbh);
  }
  catch (Exception $e) {
		// noop
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return $rc;
}

// ---------------------------------------------------------------------
function createTableUser($dbh)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;

  $sql = "CREATE TABLE IF NOT EXISTS users (
  user_id        INTEGER
, login_name     TEXT
, password_hash  TEXT
, email_address  TEXT
, first_name     TEXT
, last_name      TEXT
, mobile         TEXT
, role_id        INTEGER
, PRIMARY KEY (user_id)
)";

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 892;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    return $rc;
  }

  $sql = "CREATE UNIQUE INDEX login_name ON users (login_name)";

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 893;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    return $rc;
  }

  $sql = "INSERT INTO users (login_name, password_hash)
VALUES (:login_name, :password_hash)";
  logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

  $stmt = $dbh->prepare($sql);

  $user = 'admin';
  $hash = md5('admin');

  $stmt->bindValue(':login_name', $user, PDO::PARAM_STR);
  $stmt->bindValue(':password_hash', $hash, PDO::PARAM_STR);

  try {
    $stmt->execute();
  }
  catch (PDOException $e) {
    $rc = 888;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    return $rc;
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return $rc;
}

// ---------------------------------------------------------------------
function setEncryptedLicense($dbh, $pLicense)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $cnt = 0;

  $sql = "UPDATE licenses SET license = :license WHERE 1";
  logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

  $stmt = $dbh->prepare($sql);
  $stmt->bindValue(':license', $pLicense);
  try {
    $stmt->execute();
    $cnt = $stmt->rowCount();
  }
  catch (PDOException $e) {
    $rc = 994;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    $cnt = $msg;
  }
  $stmt->closeCursor();

  // We do not have a license yet
  if ( $cnt == 0 ) {

    $sql = "INSERT INTO licenses VALUES (:license)";
		logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

		$stmt = $dbh->prepare($sql);
    $stmt->bindValue(':license', $pLicense);
    try {
      $stmt->execute();
      $cnt = $stmt->rowCount();
    }
    catch (PDOException $e) {
      $rc = 993;
			$msg = $e->getMessage() . " (rc=$rc)."; 
      $cnt = $msg;
    }
    $stmt->closeCursor();
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $cnt);
}

// ---------------------------------------------------------------------
function getEncryptedLicense($dbh)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $lLicense = '';

	try {
  
		$sql = "SELECT * FROM licenses";
		logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));
		// This seems to be the first part of the code
		// where we access the database and check if
		// it is working...
		try {
			$stmt = $dbh->prepare($sql);
			$stmt->execute();
		}
		catch (PDOException $e) {
			$rc = 992;
			$msg = $e->getMessage() . " (rc=$rc)."; 
			throw new Exception($msg);
		}

		if ( $result = $stmt->fetch(PDO::FETCH_ASSOC) ) {
			$lLicense = $result['license'];
		}
		$stmt->closeCursor();

		if ( $lLicense == '' ) {
			$rc = 991;
			$msg = 'No License found. Please get a new License.';
			throw new Exception($msg);
		}
	}
	catch (Exception $e) {
		$lLicense = $msg;
	}

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $lLicense);
}

// ---------------------------------------------------------------------
function getMocIdentifier($dbh)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $lMocIndentifier = '';

  $sql = "SELECT * FROM moc_identifier";
  logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

  try {
    $stmt = $dbh->prepare($sql);
    $stmt->execute();
  }
  catch (PDOException $e) {
    $rc = 998;
    $msg = $e->getMessage() . " (rc=$rc)."; 
		logMessage(LOG_ERR, $msg);
    return $lMocIndentifier;
  }

  if ( $result = $stmt->fetch(PDO::FETCH_ASSOC) ) {
    $lMocIndentifier = $result['moc_identifier'];
  }
  $stmt->closeCursor();

  if ( $lMocIndentifier == '' ) {
    $rc = 988;
    $msg = 'No License found. Please get a new License.';
		logMessage(LOG_ERR, $msg . " (rc=$rc)");
    return $lMocIndentifier;
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return $lMocIndentifier;
}

// ---------------------------------------------------------------------
function decryptLicense($pKey, $pEncryptedLicense)
// redundant code with admin/lib/encryptLicense.inc
// Alternative: http://de1.php.net/mcrypt
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;

  $lDecryptedLicense = '';

  for ( $i = 0; $i < strlen($pEncryptedLicense); $i = $i + 2 ) {
    $lDecryptedLicense .= chr(hexdec($pEncryptedLicense[$i] . $pEncryptedLicense[$i+1]) ^ $pKey[13]);
  }

  $aFields = array();
  foreach ( explode(';', $lDecryptedLicense) as $key => $value ) {
    switch ( $key ) {
		case 0:
			$aFields['company'] = $value;
			break;
		case 1:
			$aFields['type'] = $value;
			break;
		case 2:
			$aFields['mysql_nodes'] = $value;
			break;
		case 3:
			$aFields['master_nodes'] = $value;
			break;
		case 4:
			$aFields['slave_nodes'] = $value;
			break;
		case 5:
			$aFields['galera_nodes'] = $value;
			break;
		case 6:
			$aFields['expiry_date'] = $value;
			break;
		case 7:
			$aFields['moc_key'] = $value;
			break;
		default:
    }
  }
  
  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
	return $aFields;
}

// ---------------------------------------------------------------------
function readEncryptLicenseFromFile($pFile)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $lEncryptedLicense = '';

  $handle = @fopen($pFile, 'r');
  if ( $handle === false ) {
    $rc = 990;
    $err = error_get_last();
    $msg = "Failed opening file: Error was '" . $err['message'] . "'";
    return array($rc, $msg);
  }

  while ( ($buffer = fgets($handle, 256)) !== false ) {
    $lEncryptedLicense = trim($buffer);
  }
  if ( ! feof($handle) ) {
    $rc = 989;
    $msg = "Error: unexpected fgets() fail.";
    return array($rc, $msg);
  }
  fclose($handle);

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $lEncryptedLicense);
}

// ---------------------------------------------------------------------
function generateMocIdentifier()
// ---------------------------------------------------------------------
{
	$characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
	$moc_identifier = '';
	for ($i = 0; $i < 32; $i++) {
		$moc_identifier .= $characters[rand(0, strlen($characters)-1)];
	}

	return $moc_identifier;
}

// ---------------------------------------------------------------------
function createTableMocIdentifier($dbh)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;

  $sql = "CREATE TABLE IF NOT EXISTS moc_identifier (
	moc_identifier     TEXT
, PRIMARY KEY (moc_identifier)
)";

  try {
    $dbh->exec($sql);
  }
  catch (PDOException $e) {
    $rc = 907;
    $msg = $e->getMessage() . " (rc=$rc)."; 
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return $rc;
}

// ---------------------------------------------------------------------
function getRepositoryVersion($dbh)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;

  $sql = "SELECT * FROM versions ORDER BY mr_version DESC LIMIT 1";
  logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

  try {
    $stmt = $dbh->prepare($sql);
    $stmt->execute();
  }
  catch (PDOException $e) {
    $rc = 987;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    logMessage(LOG_ERR, $msg);
    return array($rc, $msg);
  }

  $result = $stmt->fetch(PDO::FETCH_ASSOC);
  $stmt->closeCursor();

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $result);
}

?>
