<?php

//
// Copyright (C) 2013, 2014 FromDual GmbH
//
// This program is commercial software. Use of this software is governed
// by your applicable license agreement with FromDual GmbH
//

/*

  backupNode($aNode)
  readNodeById($pNodeId)
  startNodeRemote($aNode)
  stopNodeRemote($aServer, $aNode)
  testNode($dbh, $aNode, $abort = false)
  startNodeLocal($aServer, $aNode)
  stopNodeLocal($aNode)
  getNodesWithFilter($dbh, $aFilter = array())
  makeNodeReadOnly($aNode)
  makeNodeReadWrite($aNode)
  getNodesOfClusterById($dbh, $pClusterId)
  getMastersOfReplication($dbh, $pClusterId)
  getSlavesOfMasterSlaveReplication($dbh, $pClusterId)
  addDatabaseNode($dbh, $aDatabaseNode)
  setDatabaseNode($dbh, $aDatabaseNode)
  deleteDatabaseNode($dbh, $pNodeId)
  addSlaveToCluster($dbh, $pClusterId, $pSlaveId, $pMasterId)
  addMasterToCluster($dbh, $pClusterId, $pNodeId, $pMasterId)
  getDatabaseNodeByName($dbh, $pName)
  getDatabaseNodeById($dbh, $pNodeId)
  getDatabaseById($dbh, $pNodeId)
  getAllDatabaseNodes($dbh)
  getErrorLog($mysqli, $aNode)
	printHostnameCheck($aNode, $arr)
	printHostPingCheck($aNode, $arr)
	printSshConnectCheck($aServer, $aNode, $arr)
	printDatabaseConnectCheck($aNode, $arr)
	printDatabasePrivilegesCheck($aNode, $arr)
	printDatabaseVariablesCheck($aNode, $arr)
	printMyCnfCheck($aNode, $arr)
	printReadOnlyCheck($aNode, $rc, $msg)

*/

// No require/included here!
// require_once("lib/Check.inc");
// require_once("lib/System.inc");
// require_once("lib/Database.inc");
// require_once("lib/Repository.inc");

// ---------------------------------------------------------------------
function backupNode($aNode)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  global $gMyNameBase;

  $msg = '';

  // Check if non-InnoDB tables are available

  list($rc, $mysqli) = openDatabaseConnection($aNode['database_user'], $aNode['database_user_password'], $aNode['hostname'], $aNode['port']);
  if ( $rc != 0 ) {
    $rc = 103;
    return array($rc, $mysqli);
  }

  $sql = sprintf("SELECT COUNT(*) AS count
  FROM information_schema.tables
 WHERE table_schema NOT IN ('performance_schema', 'mysql', 'information_schema')
   AND engine != 'InnoDB'");

  if ( ! ($result = $mysqli->query($sql)) ) {
    $rc = 104;
    $msg = "Invalid query: $sql, " . $mysqli->error . ".";
    return array($rc, $msg);
  }

  $record = $result->fetch_assoc();
  closeDatabaseConnection($mysqli);

  // Non InnoDB tables found
  if ( $record['count'] > 0 ) {
    $mode = ' --lock-all-tables';
  }
  else {
    $mode = ' --single-transaction';
  }

  // do a backup
  $lNodeName    = $aNode['name'];
  $lBackupDir   = $gMyNameBase . '/' . 'bck' . '/' . $lNodeName;
  $time = date('Y-m-d-H-i-s');
  $lBackupFile  = $lBackupDir . '/' . 'full-dump-' . $time . '.sql';
  $lErrorOutput = $gMyNameBase . '/' . 'tmp/mysqldump.err';
  // todo: hard coded!
  $lMysqldump   = '/home/mysql/product/mysql-5.6.9/bin/mysqldump';

  if ( ! is_dir($lBackupDir) ) {

    if ( mkdir($lBackupDir) === false ) {
      $rc = 105;
      print "Directory " . $lBackupDir . " (rc=$rc).\n";
      exit($rc);
    }
  }

  $cmd = $lMysqldump . ' --user=' . $aNode['database_user'] . ' --password=' . $aNode['database_user_password'] . ' --host=' . $aNode['hostname'] . ' --port=' . $aNode['port'] . ' --all-databases --master-data' . $mode;

  $stdout = exec("$cmd > " . $lBackupFile . " 2>$lErrorOutput", $output, $ret);

  if ( $ret != 0 ) {
    $stdout = exec("cat $lErrorOutput 2>&1", $output, $ret2);
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($ret, $stdout);
}

// -------------------------------------------------------------------
function readNodeById($pNodeId)
// -------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $aNode = array();

  list($rc, $dbh) = openRepositoryDatabase();
  if ( $rc != 0 ) {
    $rc = 102;
    print "<p>Cannot open configuration database (rc=$rc).</p>\n";
    return array($rc, $aNode);
  }

  list($rc, $aNode) = getDatabaseNodeById($dbh, $pNodeId);

  if ( count($aNode) == 0 ) {
    $rc = 114;
    print "<p>Database node $pNodeId does not exist (rc=$rc).</p>\n";
    return array($rc, $aNode);
  }

  closeRepositoryDatabase($dbh);

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $aNode);
}

// -------------------------------------------------------------------
function startNodeRemote($aNode)
// In the meaning of: over the network. We need some kind of agent for this!
// -------------------------------------------------------------------
{
  // dummy
}

// -------------------------------------------------------------------
function stopNodeRemote($aServer, $aNode)
// In the meaning of: over the network (mysqladmin shutdown)
// -------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  // todo: question if you are really sure! for production only
  // todo: check if node is still running! for production only

	try {
  
		// get Location of pid mysqld from pid file

		list($rc, $mysqli) = openDatabaseConnection($aNode['database_user'], $aNode['database_user_password'], $aNode['hostname'], $aNode['port']);
		if ( $rc != OK ) {
			printf("<p>ERROR: Connect failed: (%d) %s (rc=$rc).</p>\n", mysqli_connect_errno(), mysqli_connect_error());
			throw new Exception(null);
		}
		list($rc, $lPidFile) = getPidFile($mysqli);
		if ( $rc != OK ) {
			$rc = $lPidFile;
			throw new Exception(null);
		}
		closeDatabaseConnection($mysqli);

		list($rc, $pid) = getPidFromPidFile($aServer['os_user'], $aServer['name'], $lPidFile);
		if ( $rc != OK ) {
			print "<p>Cannot get PID from file. Please check connection to server " . $aServer['name'] . ".</p>\n";
			throw new Exception(null);
		}

		// Check if mysqladmin exists on MOCenter server
		$exe = '/usr/bin/mysqladmin';
		if ( ! is_executable($exe) ) {
			$rc = 184;
			$msg = $exe . " does not exist (rc=$rc).";
			print "<p>$msg<br />Please install with <code>yum install mysql</code> or <code>apt-get install mysql-client</code></p>\n";
			logMessage(LOG_ERR, $msg);
			throw new Exception(null);
		}

		// todo: This should be excuted remotely!
		$cmd = $exe. " --user=" . $aNode['database_user'] . " --password=" . $aNode['database_user_password'] . " --host=" . $aNode['hostname'] . " --port=" . $aNode['port'] . " shutdown 2>&1";
		// print $cmd . "<br />\n";
		exec($cmd, $output, $ret);

		if ( $ret != OK ) {
			foreach ( $output as $line ) {
				print $line . "<br />\n";
			}
		}
	//   else {
	//     print "<p>Stop initiated...</p>\n";
	//   }

		// Ping until gone
		// should be done remotely
		// is there not a ping equivalent for the mysql api?

		$cmd = $exe . " --user=" . $aNode['database_user'] . " --password=" . $aNode['database_user_password'] . " --host=" . $aNode['hostname'] . " --port=" . $aNode['port'] . " ping 2>&1";

		$ping_timeout = 10;
		$cnt = $ping_timeout;
		while ( $cnt > 0 ) {
			$stdout = exec($cmd, $output, $ret);
			if ( $ret != 0 ) {
				break;
			}
			sleep(1);
			$cnt--;
		}

		print "<p>Pinging database stopped after " . ($ping_timeout-$cnt) . " seconds.<p>\n";

		// Check if pid is still there
		$cmd = "ps -p $pid --no-headers";

		$pid_check_timeout = 10;
		$cnt = $pid_check_timeout;
		while ( $cnt > OK ) {

			list($ret, $stdout) = runSshCommand($aServer['os_user'], $aServer['name'], $cmd);

			if ( $ret != OK ) {
				break;
			}
			sleep(1);
			$cnt--;
		}

		print "<p>PID disappeared after " . ($pid_check_timeout-$cnt) . " seconds.<p>\n";
	}
	catch (Exception $e) {
		// noop $e->getMessage()
	}

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return $rc;
}

// -------------------------------------------------------------------
// see also testCluster
function testNode($dbh, $aNode, $abort = false)
// parameter: abort = {true|false}
// return   : array(failed_checks, array('check', array($rc, $msg)))
// -------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $aChecks = array();
  $failed_checks = 0;

  list($ret, $aServer) = getServerById($dbh, $aNode['server_id']);

  // Possibly also check database ping?

  // Check database connect

  $check = 'database_connect';
  list($ret, $msg) = checkDatabaseConnect($aNode);
  $aChecks[$check] = array($rc, $msg);
  if ( $ret != OK ) {
    $failed_checks++;
    if ( $abort === true ) {
			$gIndention -= 2;
			logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
      return array($failed_checks, $aChecks);
    }
  }


  // Open database connection for the following checks
  list($ret, $mysqli) = openDatabaseConnection($aNode['database_user'], $aNode['database_user_password'], $aNode['hostname'], $aNode['port']);
  if ( $ret != OK ) {
    $aChecks[$check] = array($ret, $mysqli);
    $failed_checks++;
    if ( $abort === true ) {
			$gIndention -= 2;
			logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
      return array($failed_checks, $aChecks);
    }
  }


  // Get the error log path here for later investigation when we have no
  // connection to the database any more

  if ( $ret == OK ) {

    list($ret, $lErrorLog) = getErrorLog($mysqli, $aNode);
    if ( $ret == OK ) {

      // Store error_log now:
      $aNode['error_log'] = $lErrorLog;
      list($ret, $cnt) = setDatabaseNode($dbh, $aNode);
      logMessage(LOG_INFO, "(rc=$ret), $cnt");
    }
  }


  // Check database privileges

  $check = 'database_privileges';
  if ( gettype($mysqli) == 'object' ) {

    list($ret, $msg) = checkDatabasePrivileges($mysqli, $aNode);
    $aChecks[$check] = array($ret, $msg);
    if ( $ret != OK ) {
      $failed_checks++;
      if ( $abort === true ) {
        closeDatabaseConnection($mysqli);
				$gIndention -= 2;
				logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
        return array($failed_checks, $aChecks);
      }
    }
  }
  else {
    $failed_checks++;
    $rc = 655;
		$ret = writeCheck($aNode['node_id'], 'mysqld', 'database_privileges', $rc);
    if ( $abort === true ) {
      // no closeDatabaseConnection
			$gIndention -= 2;
			logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
      return array($failed_checks, $aChecks);
    }
  }


  // Check database variables

  $check = 'database_variables';
  if ( gettype($mysqli) == 'object' ) {

    $aVariables = array(
      'datadir'  => $aNode['datadir']
    , 'basedir'  => $aNode['basedir']
    , 'port'     => $aNode['port']
    , 'hostname' => $aNode['hostname']
    );
    list($rc, $msg) = checkDatabaseVariables($mysqli, $aNode, $aVariables);
    $aChecks[$check] = array($rc, $msg);
    if ( $rc != OK ) {
      $failed_checks++;
      if ( $abort === true ) {
        closeDatabaseConnection($mysqli);
				$gIndention -= 2;
				logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
        return array($failed_checks, $aChecks);
      }
    }
  }
  else {
    $failed_checks++;
    $rc = 656;
		$ret = writeCheck($aNode['node_id'], 'mysqld', 'database_variables', $rc);
    if ( $abort === true ) {
      // no closeDatabaseConnection
			$gIndention -= 2;
			logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
      return array($failed_checks, $aChecks);
    }
  }


  // Check my.cnf

  $check = 'my_cnf';
  if ( gettype($mysqli) == 'object' ) {
    list($rc, $msg) = checkMyCnf($aServer, $aNode);
    $aChecks[$check] = array($rc, $msg);
    if ( $rc != OK ) {
      $failed_checks++;
      if ( $abort === true ) {
        closeDatabaseConnection($mysqli);
				$gIndention -= 2;
				logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
        return array($failed_checks, $aChecks);
      }
    }
  }
  else {
    $aChecks[$check] = array($rc, $mysqli);
    $failed_checks++;
    $rc = 657;
		$ret = writeCheck($aNode['node_id'], 'mysqld', 'my_cnf', $rc);
    if ( $abort === true ) {
			$gIndention -= 2;
			logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
      return array($failed_checks, $aChecks);
    }
  }


  // Check if database is read_only

  $check = 'read_only';
  if ( gettype($mysqli) == 'object' ) {
    list($rc, $msg) = checkDatabaseReadOnly($mysqli, $aNode);
    $aChecks[$check] = array($rc, $msg);
    if ( $rc != OK ) {
      $failed_checks++;
      if ( $abort === true ) {
        closeDatabaseConnection($mysqli);
				$gIndention -= 2;
				logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
        return array($failed_checks, $aChecks);
      }
    }
  }
  else {
    $aChecks[$check] = array($rc, $mysqli);
    $failed_checks++;
    $rc = 658;
		$ret = writeCheck($aNode['node_id'], 'mysqld', 'read_only', $rc);
    if ( $abort === true ) {
			$gIndention -= 2;
			logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
      return array($failed_checks, $aChecks);
    }
  }

  // Close databse connection for all...
  if ( gettype($mysqli) == 'object' ) {
    closeDatabaseConnection($mysqli);
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($failed_checks, $aChecks);
}

// -------------------------------------------------------------------
function startNodeLocal($aServer, $aNode)
// local in the meaning of: go to the server and start mysqld
// -------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;

  // todo: question if you are really sure! for production only

  $ret = '';
  $val = '';

  // Check database connect
  list($ret, $val) = checkDatabaseConnect($aNode);

  if ( $ret == OK ) {
    return array($rc, "Database already started: $val (rc=$rc).");
  }

// todo: possibly other checks are needed like
// my.cnf pid file, pid etc.
// | pid_file      | /home/mysql/data/mysql-5.6.7/laptop3.pid | $datadir/<server>.pid
// ll /proc/29277
// ps -p 29277
//   PID TTY          TIME CMD
// 29277 ?        00:00:01 mysqld
// see stopNode there it is alrady implemented

  // start database via ssh
  $ssh = 'ssh -o ConnectTimeout=10 ' . $aServer['os_user'] . '@' . $aServer['name'] . ' ';
  logMessage(LOG_DEBUG, $ssh);


  // http://www.snailbook.com/faq/background-jobs.auto.html
	$out = '&/dev/null';
  $cmd = $ssh . "'cd " . $aNode['basedir'] . " ; bin/mysqld_safe --defaults-file=" . $aNode['my_cnf'] . " --datadir=" . $aNode['datadir'] . ' --basedir=' . $aNode['basedir'] . " 2>&1 >$out &' 2>&1";
  logMessage(LOG_DEBUG, $cmd);

	// todo: node is not properly cachted here!!
  $stdout = exec($cmd, $output, $ret);
  logMessage(LOG_INFO, "ret=$ret");
  logMessage(LOG_DEBUG, "stdout=$stdout");
  logMessage(LOG_DEBUG, "output=");
  foreach ( $output as $line ) {
    logMessage(LOG_DEBUG, $line);
  }

  if ( $ret != OK ) {
    $rc = 118;
    foreach ( $output as $line ) {
      $val .= $line . "\n";
    }
  }

  // ping or check until server is up for production only

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $val);
}

// -------------------------------------------------------------------
function stopNodeLocal($aNode)
// In the meaning of: go to the server and start the server with mysqld_safe
// -------------------------------------------------------------------
{
  // SIGTERM
  // http://dev.mysql.com/doc/refman/5.5/en/server-shutdown.html
}

// ---------------------------------------------------------------------
function getNodesWithFilter($dbh, $aFilter = array())
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $aNodes = array();

  $sql = "SELECT * FROM nodes WHERE 1\n";

  foreach ( $aFilter as $key => $value ) {
    $sql .= "  AND $key = :$key\n";
  }
  logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

  $stmt = $dbh->prepare($sql);

  foreach ( $aFilter as $key => $value ) {
    $stmt->bindValue(":$key", $value);
  }

  try {
    $stmt->execute();

    while ( $result = $stmt->fetch(PDO::FETCH_ASSOC) ) {
      $aNodes[$result['node_id']] = $result;
      $aNodes[$result['node_id']]['master_id'] = intval($result['master_id']);
    }
  }
  catch (PDOException $e) {
    $rc = 179;
    logMessage(LOG_ERR, $e->getMessage() . " (rc=$rc)");
    $aNodes = $e->getMessage() . " (rc=$rc)";
  }
  $stmt->closeCursor();

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc)");
  return array($rc, $aNodes);
}

// ---------------------------------------------------------------------
function makeNodeReadOnly($aNode)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $msg = '';

  list($ret, $mysqli) = openDatabaseConnection($aNode['database_user'], $aNode['database_user_password'], $aNode['hostname'], $aNode['port']);
  if ( $ret != 0 ) {
    $rc = 180;
    return array($rc, $mysqli);
  }

  $sql = sprintf("SET GLOBAL read_only = 1");

  if ( ! ($mysqli->query($sql)) ) {
    $rc = 181;
    $msg = "Invalid query: $sql, " . $mysqli->error;
    return array($rc, $msg);
  }

  closeDatabaseConnection($mysqli);

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($ret, 'OK');
}

// ---------------------------------------------------------------------
function makeNodeReadWrite($aNode)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $msg = '';

  list($ret, $mysqli) = openDatabaseConnection($aNode['database_user'], $aNode['database_user_password'], $aNode['hostname'], $aNode['port']);
  if ( $ret != OK ) {
    $rc = 182;
    return array($rc, $mysqli);
  }

  $sql = sprintf("SET GLOBAL read_only = 0");

  if ( ! ($mysqli->query($sql)) ) {
    $rc = 183;
    $msg = "Invalid query: $sql, " . $mysqli->error;
    return array($rc, $msg);
  }

  closeDatabaseConnection($mysqli);

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($ret, 'OK');
}

// ---------------------------------------------------------------------
function getNodesOfClusterById($dbh, $pClusterId)
// ---------------------------------------------------------------------
{
  $rc = OK;
  return getNodesWithFilter($dbh, array('cluster_id' => $pClusterId));
}

// ---------------------------------------------------------------------
function getMastersOfReplication($dbh, $pClusterId)
// ---------------------------------------------------------------------
{
  $rc = OK;
  list($rc, $aNodes) = getNodesWithFilter($dbh, array('cluster_id' => $pClusterId, 'role_id' => MASTER));
  return array($rc, array_keys($aNodes));
}

// ---------------------------------------------------------------------
function getSlavesOfMasterSlaveReplication($dbh, $pClusterId)
// ---------------------------------------------------------------------
{
  $rc = OK;
  list($rc, $aNodes) = getNodesWithFilter($dbh, array('cluster_id' => $pClusterId, 'role_id' => SLAVE));
  return array($rc, array_keys($aNodes));
}

// ---------------------------------------------------------------------
function addDatabaseNode($dbh, $aDatabaseNode)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $cnt = 0;

  $aDatabaseNode['last_change_ts'] = time();
  $aDatabaseNode['status']         = 'active';

  if ( $aDatabaseNode['name'] == '' ) {
    $rc = 128;
    return array($rc, $cnt);
  }

  $aColumns = array('name', 'last_change_ts', 'status', 'node_type', 'hostname'
                  , 'basedir', 'datadir', 'my_cnf', 'port', 'database_user'
                  , 'database_user_password', 'error_log', 'pid_file'
                  , 'read_only', 'server_id', 'role_id', 'cluster_id', 'master_id');

  $aHelp = array();
  foreach ( $aColumns as $column ) {
    array_push($aHelp, ':' . $column);
  }

  $sql = "INSERT INTO nodes (" . implode(', ', $aColumns) . ') VALUES (' . implode(', ', $aHelp) . ")";
  logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));
  $stmt = $dbh->prepare($sql);
  foreach ( $aColumns as $column ) {
    $stmt->bindValue(':' . $column, $aDatabaseNode[$column]);
  }
  try {
    $stmt->execute();
    $node_id = $dbh->lastInsertId();
  }
  catch (PDOException $e) {
    $rc = 189;
    $msg = $e->getMessage() . " (rc=$rc)\n";
    return array($rc, $msg);
  }
  $stmt->closeCursor();

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $node_id);
}

// ---------------------------------------------------------------------
function setDatabaseNode($dbh, $aDatabaseNode)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $cnt = 0;

  $aDatabaseNode['last_change_ts'] = time();

  $aColumns = array('name', 'last_change_ts', 'status', 'node_type', 'hostname'
                  , 'basedir', 'datadir', 'my_cnf', 'hostname', 'port', 'database_user'
                  , 'database_user_password', 'error_log', 'pid_file'
                  , 'read_only', 'server_id', 'role_id', 'cluster_id', 'master_id'
                   );

  $aHelp = array();
  foreach ( $aColumns as $column ) {
    array_push($aHelp, $column . ' = :' . $column);
  }
  $sql = "UPDATE nodes SET " . implode(', ', $aHelp) . " WHERE node_id = :node_id";
  logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

  $stmt = $dbh->prepare($sql);

  // logMessage(LOG_DEBUG, print_r($aDatabaseNode, true));
  foreach ( $aColumns as $column ) {
    $stmt->bindValue(':' . $column, $aDatabaseNode[$column]);
  }
  $stmt->bindValue(':node_id', $aDatabaseNode['node_id']);
  try {
    $stmt->execute();
    $cnt = $stmt->rowCount();
  }
  catch (PDOException $e) {
    $rc = 190;
    $cnt = $e->getMessage();
  }

  $stmt->closeCursor();

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $cnt);
}

// ---------------------------------------------------------------------
function deleteDatabaseNode($dbh, $pNodeId)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $cnt = 0;

  // todo: Later should be moved or updated status not deleted for sync with other moc node

  $sql = "DELETE FROM nodes WHERE node_id = :node_id";

  logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));
  $stmt = $dbh->prepare($sql);
  $stmt->bindValue(':node_id', $pNodeId);
  try {
    $stmt->execute();
    $cnt = $stmt->rowCount();
  }
  catch (PDOException $e) {
    $rc = 191;
    print $e->getMessage() . " (rc=$rc)";
  }
  $stmt->closeCursor();

  // todo: also delete checks!

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $cnt);
}

// ---------------------------------------------------------------------
function addSlaveToCluster($dbh, $pClusterId, $pNodeId, $pMasterId)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;

  $sql = "UPDATE nodes
   SET cluster_id = :cluster_id
     , role_id = :role_id
     , master_id = :master_id
 WHERE node_id = :node_id";

	logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

  $stmt = $dbh->prepare($sql);
  $stmt->bindValue(':node_id', $pNodeId);
  $stmt->bindValue(':role_id', SLAVE);
  $stmt->bindValue(':cluster_id', $pClusterId);
  $stmt->bindValue(':master_id', $pMasterId);

  try {
    $stmt->execute();
  }
  catch (PDOException $e) {
    $rc = 921;
    $msg = $e->getMessage() . " (rc=$rc)."; 
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }
  $stmt->closeCursor();

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return $rc;
}

// ---------------------------------------------------------------------
function addMasterToCluster($dbh, $pClusterId, $pNodeId, $pMasterId)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;

  $sql = "UPDATE nodes
   SET cluster_id = :cluster_id
     , role_id = :role_id
     , master_id = :master_id
 WHERE node_id = :node_id";

  logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

  $stmt = $dbh->prepare($sql);
  $stmt->bindValue(':node_id', $pNodeId);
  $stmt->bindValue(':role_id', MASTER);
  $stmt->bindValue(':cluster_id', $pClusterId);
  $stmt->bindValue(':master_id', $pMasterId);

  try {
    $stmt->execute();
  }
  catch (PDOException $e) {
    $rc = 920;
    $msg = $e->getMessage() . " (rc=$rc).";
    logMessage(LOG_ERR, $msg);
    print "<p>$msg</p>\n";
  }
  $stmt->closeCursor();

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return $rc;
}

// ---------------------------------------------------------------------
function getDatabaseNodeByName($dbh, $pName)
// ---------------------------------------------------------------------
{
  $rc = OK;
  list($rc, $aNodes) = getNodesWithFilter($dbh, array('name' => $pName));
  return array($rc, array_shift($aNodes));
}

// ---------------------------------------------------------------------
function getDatabaseNodeById($dbh, $pNodeId)
// ---------------------------------------------------------------------
{
  $rc = OK;
  list($rc, $aNodes) = getNodesWithFilter($dbh, array('node_id' => $pNodeId));
  return array($rc, array_shift($aNodes));
}

// ---------------------------------------------------------------------
function getDatabaseById($dbh, $pNodeId)
// ---------------------------------------------------------------------
{
  $rc = OK;
  list($rc, $aNodes) = getNodesWithFilter($dbh, array('node_id' => $pNodeId));
  return array($rc, array_shift($aNodes));
}

// ---------------------------------------------------------------------
function getAllDatabaseNodes($dbh)
// ---------------------------------------------------------------------
{
  return getNodesWithFilter($dbh, array());
}

// ---------------------------------------------------------------------
function getErrorLog($mysqli, $aNode)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;

  $log_error = '';
  list($ret, $log_error) = getGlobalVariable($mysqli, 'log_error');
  logMessage(LOG_INFO, "(rc=$ret), $log_error");

  if ( $ret == OK ) {

    // Absolute path
    // Example: /home/mysql/data/mm-5.1.b/error.log
    if ( substr($log_error, 0, 1) == '/' ) {
      $lErrorLog = $log_error;
    }
    // Relative path
    // Example: ./error.log
    else {
      $lErrorLog = rtrim($aNode['datadir'], '/') . '/' . ltrim($log_error, './');
    }
  }
  else {
    $lErrorLog = '';
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $lErrorLog);
}

// ---------------------------------------------------------------------
function printHostnameCheckNode($aNode, $arr)
// ---------------------------------------------------------------------
{
  print "<p>Checking hostname " . $aNode['hostname'] . "...<br />\n";
  if ( $arr[0] == 0 ) {
    print '&nbsp;&nbsp;Hostname ' . $aNode['hostname'] . " is <span style='color:green;'><strong>OK</strong></span>.<br />\n";
    print "&nbsp;&nbsp;Reply is: " . $arr[1] . "</p>\n";
  }
  else {
    print "&nbsp;&nbsp;<span style='color:red;'><strong>ERROR</strong></span>: " . $arr[1] . "</p>\n";
  }
}

// ---------------------------------------------------------------------
function printHostPingCheckNode($aNode, $arr)
// ---------------------------------------------------------------------
{
  print "<p>Pinging host " . $aNode['hostname'] . "...<br />\n";
  if ( $arr[0] == 0 ) {
    print '&nbsp;&nbsp;Ping to ' . $aNode['hostname'] . " is <span style='color:green;'><strong>OK</strong></span>.</p>\n";
  }
  else {
    print "&nbsp;&nbsp;<span style='color:red;'><strong>ERROR</strong></span>: " . $arr[1] . "</p>\n";
  }
}

// ---------------------------------------------------------------------
function printSshConnectCheckNode($aServer, $aNode, $arr)
// ---------------------------------------------------------------------
{
  print "<p>Checking ssh connect to host " . $aNode['hostname'] . "...<br />\n";
  if ( $arr[0] == 0 ) {
    print '&nbsp;&nbsp;Ssh connect to host ' . $aNode['hostname'] . " is <span style='color:green;'><strong>OK</strong></span>.<br />\n";
    print "&nbsp;&nbsp;Reply is " . $arr[1] . ".</p>\n";
  }
  else {
    print "&nbsp;&nbsp;<span style='color:red;'><strong>ERROR</strong></span>: " . $arr[1] . "<br />\n";

    // Check for customers public key file first

    $whoami = whoami();
    $home = getHomeDir($whoami);

    $aSshPublicKeyFiles = array(
      0 => $home . '/.ssh/id_rsa.pub'
    , 1 => $home . '/.ssh/id_dsa.pub'
    );

    $pk_exists = false;
    foreach ( $aSshPublicKeyFiles as $publicKeyFile ) {

      if ( file_exists($publicKeyFile) ) {
        $pk_exists = true;
        break;
      }
    }

    // RSA or DSA key exists
    if ( $pk_exists !== true) {
      print "&nbsp;&nbsp;You do NOT have a public key file yet <a href='index.php?f=create&o=public_key'>generate it here</a>...<br /></p>\n";
    }
    // User seems to have a public key file but it is not distributed.
    else {

      print "&nbsp;&nbsp;Copy the ssh public key file of user $whoami to the authorized_keys file of user " . $aServer['os_user'] . " on host " . $aServer['name'] . ".<br />\n";

      $aOperatingSystem = getOperatingSystem();
      if ( $aOperatingSystem['Distributor ID'] == 'CentOS' ) {
        print "<pre>su - $whoami\n";
      }
      else {
        print "<pre>sudo su - $whoami\n";
      }
      print "ssh-copy-id -i ~/.ssh/id_?sa.pub " . $aServer['os_user'] . "@" . $aServer['name'] . "</pre>\n";
    }   // no public key file
  }
}

// ---------------------------------------------------------------------
function printDatabaseConnectCheck($aNode, $arr)
// ---------------------------------------------------------------------
{
  print "<p>Checking database connection...<br />\n";
  if ( $arr[0] == 0 ) {
    print "&nbsp;&nbsp;Connect to database is <span style='color:green;'><strong>OK</strong></span>.<br />\n";
    print "&nbsp;&nbsp;Reply is $arr[1].</p>\n";
  }
  else {
    print "&nbsp;&nbsp;<span style='color:red;'><strong>ERROR</strong></span>: " . $arr[1] . "\n";
    print "&nbsp;&nbsp;Please create user as follows:<br />\n";
    print "&nbsp;&nbsp;<code>mysql&gt; GRANT ALL ON *.* TO '" . $aNode['database_user'] . "'@'" . gethostname() . "' IDENTIFIED BY '" . $aNode['database_user_password'] . "';</code></p>\n";
  }
}

// ---------------------------------------------------------------------
function printDatabasePrivilegesCheck($aNode, $arr)
// ---------------------------------------------------------------------
{
  print "<p>Checking database privileges...<br />\n";
  if ( $arr[0] == 0 ) {
    print "&nbsp;&nbsp;Database privileges are <span style='color:green;'><strong>OK</strong></span>.</p>\n";
  }
  else {
    print "&nbsp;&nbsp;<span style='color:red;'><strong>ERROR</strong></span>: " . $arr[1] . "</p>\n";
  }
}

// ---------------------------------------------------------------------
function printDatabaseVariablesCheck($aNode, $arr)
// ---------------------------------------------------------------------
{
  print "<p>Checking database variables...<br />\n";
  if ( $arr[0] == 0 ) {
    print "&nbsp;&nbsp;Database variables are <span style='color:green;'><strong>OK</strong></span>.</p>\n";
  }
  else {
    print "&nbsp;&nbsp;<span style='color:red;'><strong>ERROR</strong></span>: " . $arr[1] . "</p>\n";
  }
}

// ---------------------------------------------------------------------
function printMyCnfCheck($aNode, $arr)
// ---------------------------------------------------------------------
{
  print "<p>Checking my.cnf...<br />\n";
  if ( $arr[0] == 0 ) {
    print "&nbsp;&nbsp;Configuration file " . $aNode['my_cnf'] . " is <span style='color:green;'><strong>OK</strong></span>.</p>\n";
  }
  else {
    print "&nbsp;&nbsp;<span style='color:red;'><strong>ERROR</strong></span>: " . $arr[1] . "</p>\n";
  }
}

// ---------------------------------------------------------------------
function printReadOnlyCheck($aNode, $rc, $msg)
// ---------------------------------------------------------------------
{
  print "<p>Checking if database is read_only...<br />\n";

  if ( $rc == OK ) {
    print "&nbsp;&nbsp;Database is " . $msg . " which is <span style='color:green;'><strong>OK</strong></span>.</p>\n";
  }
  else {
    print "&nbsp;&nbsp;Database is $msg which is an <span style='color:red;'><strong>ERROR</strong></span>.</p>\n";
  }
}

?>
