<?php

//
// Copyright (C) 2013, 2014 FromDual GmbH
//
// This program is commercial software. Use of this software is governed
// by your applicable license agreement with FromDual GmbH
//

/*

  getJobsWithFilter($dbh, $aFilter = array())
  setJob($dbh, $aJob)
  addJob($dbh, $aJob)
  deleteJob($dbh, $pJobId)
  startJob($dbh, $pJobId)
  killJob($dbh, $pJobId)
  checkJob($dbh, $pJobId)

*/

// ---------------------------------------------------------------------
function getJobsWithFilter($dbh, $aFilter = array())
// ---------------------------------------------------------------------
{
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);

  $rc = OK;
  $aJobs = array();

  $sql = "SELECT * FROM jobs WHERE 1\n";

  foreach ( $aFilter as $key => $value ) {
    $sql .= "  AND $key = :$key\n";
  }
  logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

  $stmt = $dbh->prepare($sql);

  foreach ( $aFilter as $key => $value ) {
    $stmt->bindValue(":$key", $value);
  }

  try {
    $stmt->execute();

    while ( $result = $stmt->fetch(PDO::FETCH_ASSOC) ) {
      $aJobs[$result['job_id']] = $result;
    }
  }
  catch (PDOException $e) {
    $rc = 755;
    logMessage(LOG_ERR, $e->getMessage() . " (rc=$rc)");
    $aJobs = $e->getMessage() . " (rc=$rc)";
  }
  $stmt->closeCursor();

  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__);
  return array($rc, $aJobs);
}

// ---------------------------------------------------------------------
function setJob($dbh, $aJob)
// ---------------------------------------------------------------------
{
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);

  $rc = OK;
  $cnt = 0;

  $aColumns = array('name', 'server', 'pid', 'start_ts', 'status', 'check_interval', 'last_check_ts', 'end_ts'
                  , 'error_code', 'error_message', 'command'
                   );

  $aHelp = array();
  foreach ( $aColumns as $column ) {
    array_push($aHelp, $column . ' = :' . $column);
  }
  $sql = "UPDATE jobs SET " . implode(', ', $aHelp) . " WHERE job_id = :job_id";
  logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));
  $stmt = $dbh->prepare($sql);

  logMessage(LOG_DEBUG, print_r($aJob, true));
  foreach ( $aColumns as $column ) {
    $stmt->bindValue(':' . $column, $aJob[$column]);
  }
  $stmt->bindValue(':job_id', $aJob['job_id']);
  try {
    $stmt->execute();
    $cnt = $stmt->rowCount();
  }
  catch (PDOException $e) {
    $rc = 187;
    $cnt = $e->getMessage();
  }

  $stmt->closeCursor();

  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__);
  return array($rc, $cnt);
}

// ---------------------------------------------------------------------
function addJob($dbh, $aJob)
// ---------------------------------------------------------------------
{
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);

  $rc = OK;
  $job_id = 0;

  $aJob['start_ts']      = time();
  $aJob['last_check_ts'] = 0;
  $aJob['end_ts']        = 0;
  $aJob['error_code']    = 999;
  $aJob['error_message'] = '';

  if ( $aJob['name'] == '' ) {
    $rc = 185;
    return array($rc, $cnt);
  }

  $aColumns = array('name', 'server', 'pid', 'status', 'start_ts', 'check_interval', 'last_check_ts', 'end_ts'
                  , 'error_code', 'error_message', 'command'
                   );

  $aHelp = array();
  foreach ( $aColumns as $column ) {
    array_push($aHelp, ':' . $column);
  }

  $sql = "INSERT INTO jobs (" . implode(', ', $aColumns) . ') VALUES (' . implode(', ', $aHelp) . ")";
  logMessage(LOG_DEBUG, $sql);

  try {
    $stmt = $dbh->prepare($sql);
    foreach ( $aColumns as $column ) {
      $stmt->bindValue(':' . $column, $aJob[$column]);
    }
    $stmt->execute();
    $job_id = $dbh->lastInsertId();
  }
  catch (PDOException $e) {
    $rc = 186;
    $msg = $e->getMessage() . " (rc=$rc)\n";
    logMessage(LOG_ERR, $msg);
    return array($rc, $msg);
  }
  $stmt->closeCursor();

  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__);
  return array($rc, $job_id);
}

// ---------------------------------------------------------------------
function deleteJob($dbh, $pJobId)
// ---------------------------------------------------------------------
{
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);

  $rc = OK;
  $cnt = 0;

  $sql = "DELETE FROM jobs WHERE job_id = :job_id";

  $stmt = $dbh->prepare($sql);
  $stmt->bindValue(':job_id', $pJobId);
  try {
    $stmt->execute();
    $cnt = $stmt->rowCount();
  }
  catch (PDOException $e) {
    $rc = 188;
    print $e->getMessage() . " (rc=$rc)";
  }
  $stmt->closeCursor();

  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__);
  return array($rc, $cnt);
}

// ---------------------------------------------------------------------
function startJob($dbh, $pName, $pUser, $pServer, $pCommand, $pCheckInterval, $pUseStdout = false)
// ---------------------------------------------------------------------
{
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);

  global $gTmpDir;

  $rc = OK;
  $lJobId = 0;

  // Prepare Job

  $aJob['pid']            = 0;
  $aJob['name']           = $pName;
  $aJob['server']         = $pServer;
  $aJob['command']        = $pCommand;
  $aJob['check_interval'] = $pCheckInterval;
  $aJob['status']         = 'Preparing';

  list($ret, $lJobId) = addJob($dbh, $aJob);

  if ( $ret != OK ) {
    $rc = 963;
    $msg = "Adding job to repository failed (rc=$rc)\n";
    print "<p>$msg</p>";
    logMessage(LOG_ERR, $msg);
    return array($rc, $lJobId);
  }

  // Copy and start on target

  // nohup command 1>/tmp/mocenter/stdout.00000001 2>/tmp/mocenter/stderr.00000001 </dev/null &
  // pid=$!
  // echo $pid >/tmp/mocenter/pid.00000001
  // wait $pid
  // echo $? >/tmp/mocenter/ret.00000001

  $jobname    = $gTmpDir . '/' . sprintf("job.%08d", $lJobId);
  $pidname    = $gTmpDir . '/' . sprintf("pid.%08d", $lJobId);
  $retname    = $gTmpDir . '/' . sprintf("ret.%08d", $lJobId);
  $stdoutname = $gTmpDir . '/' . sprintf("stdout.%08d", $lJobId);
  $stderrname = $gTmpDir . '/' . sprintf("stderr.%08d", $lJobId);

  $data = 'nohup ' . $aJob['command'];
  // Stdout is used by the command!
  if ( $pUseStdout === false ) {
    $data .=  ' 1>' . $stdoutname;
  }
  $data .= ' 2>' . $stderrname . ' </dev/null &
pid=$!
echo $pid >' . $pidname . '
wait $pid
echo $? >' . $retname . "\n";

  @mkdir($gTmpDir, 0777, true);
  if ( ! file_put_contents($jobname, $data) ) {
    $rc = 959;
    $msg = "Creating job failed (rc=$rc)";
    print "<p>$msg</p>";
    logMessage(LOG_ERR, $msg);
    return array($rc, $lJobId);
  }

  list($ret, $output) = runRemoteCommand($pUser, $pServer, "mkdir -p $gTmpDir");
  if ( $ret != OK ) {
    $rc = 955;
    print $ret . " (rc=$rc)";
    print_r($output);
  }

// var_dump($jobname, $pUser, $pServer, $gTmpDir);


  list($ret, $output) = copyFileToRemoteServer($jobname, $pUser, $pServer, $gTmpDir);
  if ( $ret != OK ) {
    $rc = 956;
    print $ret . " (rc=$rc)";
    print_r($output);
  }

  $cmd = "sh $jobname";
  list($ret, $output) = runRemoteCommand($pUser, $pServer, $cmd);
  if ( $ret != OK ) {
    $rc = 958;
    print $cmd;
    print $ret . " (rc=$rc)";
    print_r($output);
  }

  // job.00000001
  // pid.00000001
  // ret.00000001
  // stderr.00000001
  // stdout.00000001

  $aJob['job_id']        = $lJobId;
  $aJob['status']        = 'Started';
  $aJob['command']       = $data;
  $aJob['start_ts']      = time();
  $aJob['end_ts']        = 0;
  $aJob['last_check_ts'] = 0;

  list($ret, $pid) = catRemoteFile($pUser, $pServer, $pidname);
  $aJob['pid']           = $pid;

  list($ret, $returncode) = catRemoteFile($pUser, $pServer, $retname);
  $aJob['error_code']    = $returncode;

  list($ret, $stderr) = catRemoteFile($pUser, $pServer, $stderrname);
  list($ret, $stdout) = catRemoteFile($pUser, $pServer, $stdoutname);
  $aJob['error_message'] = $stdout . "\n" . $stderr;

  list($ret, $cnt) = setJob($dbh, $aJob);

  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__);
  return array($rc, $lJobId);
}

// ---------------------------------------------------------------------
function killJob($dbh, $pJobId)
// ---------------------------------------------------------------------
{
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);

  $rc = OK;
  // we cannot kill job atm so we return an error
  $rc = 177;

  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__);
  return $rc;
}

// ---------------------------------------------------------------------
function checkJob($dbh, $pJobId)
// ---------------------------------------------------------------------
{
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);

  $rc = OK;
  $rc = 176;

  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__);
  return $rc;
}
     
?>
