<?php

//
// Copyright (C) 2013, 2014 FromDual GmbH
//
// This program is commercial software. Use of this software is governed
// by your applicable license agreement with FromDual GmbH
//

/*

  getNodeCheckTypes()
  getServerCheckTypes()
  getCheckSummaryById($pNodeId)
  getNodeCheckSummary($pNodeId)
  getServerCheckSummary($pServerId)
  getCheckSummaryByName($pNode)
  getDatabaseVariables($mysqli)
  renderCheck ($ts, $status)
  checkHostname($aServer)
  checkPing($pIpOrName)
  checkSshConnect($aServer)
  checkDatabaseConnect($aNode)
  checkDatabasePrivileges($mysqli, $aNode)
  checkDatabaseVariables($mysqli, $aNode, $aVariables)
  checkMyCnf($aServer, $aNode)
  checkDatabaseReadOnly($mysqli, $aNode)
  writeCheck($pId, $pType, $pCheckName, $pStatus)
  getChecksWithFilter($dbh, $aFilter = array())
  getDatabaseCheck($dbh, $pNodeId, $pType, $pName)
  getAllDatabaseChecks($dbh, $pNodeId)
  getAllServerChecks($dbh, $pNodeId)
  addDatabaseCheck($dbh, $aDatabaseCheck)
  updateDatabaseCheck($dbh, $aDatabaseCheck)
  deleteChecks($dbh, $pNodeId, $pNodeType)

*/

// No require/included here!
// require_once("lib/Database.inc");

// ---------------------------------------------------------------------
function getNodeCheckTypes()
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $aNodeCheckTypes = array(
  'database_connect'
, 'database_privileges'
, 'database_variables'
, 'read_only'
);

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__);
  return $aNodeCheckTypes;
}

// ---------------------------------------------------------------------
function getServerCheckTypes()
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $aNodeCheckTypes = array(
  'hostname'
, 'host_ping'
, 'ssh_connect'
);

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__);
  return $aNodeCheckTypes;
}

// ---------------------------------------------------------------------
// do not use anymore!
function getCheckSummaryById($pNodeId)
// ---------------------------------------------------------------------
{
  return getNodeCheckSummary($pNodeId);
}

// ---------------------------------------------------------------------
function getNodeCheckSummary($pNodeId)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $ret = OK;

  // 0 = OK
  // 1 = WRN
  // 2 = ERR
  // 9 = UNK

	list($ret, $dbh) = openRepositoryDatabase();
	list($ret, $aChecks) = getAllDatabaseChecks($dbh, $pNodeId);
	// print "<pre>"; print_r($aChecks); print "</pre>";
	closeRepositoryDatabase($dbh);

  if ( count($aChecks) == 0 ) {
    $ret = 9;
  }
  else {

    $aNodeCheckTypes = getNodeCheckTypes();

    $err = 0;
    $wrn = 0;
    $unk = 0;
    $ok  = 0;

    foreach ( $aNodeCheckTypes as $check ) {

      if ( ! isset($aChecks[$check]) ) {
        $unk++;
        continue;
      }
      else {

        if ( $aChecks[$check]['last_check_status'] == 'ERR'
        && (($check == 'database_privileges')
        || ($check == 'database_variables')
        || ($check == 'my_cnf'))
          ) {
          $wrn++;
          continue;
        }
        elseif ( $aChecks[$check]['last_check_status'] == 'ERR' ) {
          $err++;
          continue;
        }
        else {
          $ok++;
        }
      }
    }

    if ( $unk > 0 ) {
      $ret = 9;
    }
    if ( $wrn > 0 ) {
      $ret = 1;
    }
    if ( $err > 0 ) {
      $ret = 2;
    }
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (ret=$ret).");
  return $ret;
}

// ---------------------------------------------------------------------
function getServerCheckSummary($pServerId)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $ret = OK;

  // 0 = OK
  // 1 = WRN
  // 2 = ERR
  // 9 = UNK

  list($ret, $dbh) = openRepositoryDatabase();
  list($ret, $aChecks) = getAllServerChecks($dbh, $pServerId);
  closeRepositoryDatabase($dbh);

  if ( count($aChecks) == 0 ) {
    $ret = 9;
  }
  else {

    $aServerCheckTypes = getServerCheckTypes();

    $err = 0;
    $wrn = 0;
    $unk = 0;
    $ok  = 0;

    foreach ( $aServerCheckTypes as $check ) {

      if ( ! isset($aChecks[$check]) ) {
        $unk++;
        continue;
      }
      else {

        if ( $aChecks[$check]['last_check_status'] == 'ERR' ) {
          $err++;
          continue;
        }
        else {
          $ok++;
        }
      }
    }

    if ( $unk > 0 ) {
      $ret = 9;
    }
    if ( $wrn > 0 ) {
      $ret = 1;
    }
    if ( $err > 0 ) {
      $ret = 2;
    }
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (ret=$ret).");
  return $ret;
}

// ---------------------------------------------------------------------
function getCheckSummaryByName($pNode)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;

  // 0 = OK
  // 1 = WRN
  // 2 = ERR
  // 9 = UNK

  list($ret, $dbh) = openRepositoryDatabase();
  // todo: this could be a bug. should be $pNodeId?
  list($ret, $aChecks) = getAllDatabaseChecks($dbh, $pNode);
  closeRepositoryDatabase($dbh);

  if ( count($aChecks) == 0 ) {
    $rc = 153;
  }
  else {

    $aNodeCheckTypes = getNodeCheckTypes();

    $err = 0;
    $wrn = 0;

    foreach ( $aNodeCheckTypes as $check ) {

      if ( $aChecks[$check ] == 'ERR'
      && (($check == 'database_privileges')
      || ($check == 'database_variables')
      || ($check == 'my_cnf'))
        ) {
        $wrn++;
      }
      elseif ( $aChecks[$check] == 'ERR' ) {
        $err++;
      }
    }

    if ( $wrn > 0 ) {
      $rc = 115;
    }
    if ( $err > 0 ) {
      $rc = 116;
    }
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return $rc;
}

// ---------------------------------------------------------------------
function getDatabaseVariables($mysqli)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $msg = '';

  // SHOW GLOBAL VARIABLES;
  // +--------------------------+-------+
  // | Variable_name            | Value |
  // +--------------------------+-------+
  // | auto_increment_increment | 1     |
  // | auto_increment_offset    | 1     |
  // | autocommit               | ON    |
  // | automatic_sp_privileges  | ON    |
  // | read_only                | OFF   |
  // +--------------------------+-------+
  $sql = sprintf("SHOW GLOBAL VARIABLES");

  if ( ! ($result = $mysqli->query($sql)) ) {
    $rc = 154;
    $aVariables = "Invalid query: $sql, " . $mysqli->error . ", (rc=$rc).";
    logMessage(LOG_ERR, $aVariables);
  }
  else {

    // Copy all variabels into array
    $aVariables = array();
    while ( $record = $result->fetch_assoc() ) {
      // Trim directory names
      $aVariables[$record['Variable_name']] = rtrim($record['Value'], '/');
    }
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $aVariables);
}

// ---------------------------------------------------------------------
function renderCheck ($ts, $status)
// and calculates time
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $color = 'red';
  if ( $status == 'OK' ) {
    $color = 'green';
  }
  elseif ( $status == 'unknown' ) {
    $color = 'orange';
  }
  elseif ( $status == 'ERROR' ) {
    $color = 'red';
  }
  elseif ( $status == '' ) {
    $status = 'unknown';
    $color = 'orange';
  }

  if ( $ts > 0 ) {
    $sago = time() - $ts;

    if ( $sago >= 86400 ) {
      $ago = round($sago/3600) . ' h ' . round(($sago/60)%60, 0) . ' min ' . $sago%60%60 . ' s ago';
    }
    elseif ( $sago >= 3600  ) {
      $ago = round($sago/3600) . ' h ' . round(($sago/60)%60, 0) . ' min ' . $sago%60%60 . ' s ago';
    }
    elseif ( $sago >= 60  ) {
      $ago = round($sago/60, 0) . ' min ' . $sago%60 . ' s ago';
    }
    else {
      $ago = $sago . ' s ago';
    }
  }
  else {
    $ago = 'unknown';
    // This is because of earlier bugs. Possibly can be removed later
    $ts = 0;
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__);
  return array($ts, $status, $color, $ago);
}

// ---------------------------------------------------------------------
function checkHostname($aServer)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;

  $ip = getHostByName($aServer['name']);

  if ( $ip == $aServer['name'] ) {
    $rc = 155;
    $ip = 'Cannot resolve hostname ' . $aServer['name'] . '.';
  }

  // Skip nodes which are not created yet!
  if ( $aServer['server_id'] != 0 ) {
    $ret = writeCheck($aServer['server_id'], 'server', 'hostname', $rc);
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $ip);
}

// ---------------------------------------------------------------------
function checkPing($pIpOrName)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $msg = 'OK';

  // count 1, timeout 3
  $cmd = 'ping -c 1 -W 3 ' . $pIpOrName . ' 2>&1';
  logMessage(LOG_DEBUG, $cmd);
  $stdout = exec($cmd, $output, $ret);
  logMessage(LOG_DEBUG, $stdout);
  logMessage(LOG_DEBUG, $ret);
  logMessage(LOG_DEBUG, print_r($output, true));

  if ( $ret != 0 ) {
    $rc = 156;
    $msg = "Host " . $pIpOrName . " does not answer to ping: $stdout (rc=$rc).";
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $msg);
}

// ---------------------------------------------------------------------
function checkSshConnect($aServer)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  // var_dump($aServer);

  $ssh = 'ssh -o ConnectTimeout=10 ' . $aServer['os_user'] . '@' . $aServer['name'] . ' ';

  // http://www.snailbook.com/faq/background-jobs.auto.html
  $cmd = $ssh . "'hostname 2>&1' 2>&1";
  $stdout = exec($cmd, $output, $ret);

  $msg = $stdout;
  if ( $ret != 0 ) {
    $rc  = 799;
    $msg = "Ssh connect failed: $stdout";
  }

  // Skip nodes which are not created yet!
  if ( $aServer['server_id'] != 0 ) {
    $ret = writeCheck($aServer['server_id'], 'server', 'ssh_connect', $rc);
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $stdout);
}

// ---------------------------------------------------------------------
function checkDatabaseConnect($aNode)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $val = '';

  list($ret, $dbh) = openRepositoryDatabase();
  list($ret, $aServer) = getServerById($dbh, $aNode['server_id']);
  closeRepositoryDatabase($dbh);

  list($ret, $mysqli) = openDatabaseConnection($aNode['database_user'], $aNode['database_user_password'], $aNode['hostname'], $aNode['port']);

  if ( $ret == OK ) {
    $msg = $mysqli->host_info;
    closeDatabaseConnection($mysqli);
  }
  else {
    $rc = 157;
    $msg = $mysqli;
  }

  // Skip nodes which are not created yet!
  if ( $aNode['node_id'] != 0 ) {
    $ret = writeCheck($aNode['node_id'], 'mysqld', 'database_connect', $rc);
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $msg);
}

// ---------------------------------------------------------------------
function checkDatabasePrivileges($mysqli, $aNode)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $msg = '';

  // Get the username we are currently connected with
  // admin@%
  // root@localhost

  $sql = sprintf("SELECT current_user");
  if ( ! ($result = $mysqli->query($sql)) ) {
    $rc  = 798;
    $msg = "Invalid query: $sql, " . $mysqli->error . ", (rc=$rc).";
    logMessage(LOG_ERR, $msg);
    // Skip nodes which are not created yet!
    if ( $aNode['node_id'] != 0 ) {
      $ret = writeCheck($aNode['node_id'], 'mysqld', 'database_privileges', $rc);
    }
    return array($rc, $msg);
  }

  if ( ! ($record = $result->fetch_assoc()) ) {
    $rc = 158;
    $msg = "No current_user found.";
    logMessage(LOG_ERR, $msg);
    // Skip nodes which are not created yet!
    if ( $aNode['node_id'] != 0 ) {
      $ret = writeCheck($aNode['node_id'], 'mysqld', 'database_privileges', $rc);
    }
    return array($rc, $msg);
  }

  list($user, $host) = explode('@', $record['current_user']);

  // SHUTDOWN privilege
  // SHOW GRANTS FOR 'root'@'master';
  // +------------------------------------------------------------------+
  // | Grants for root@master                                           |
  // +-----------------------------------------------------------------+
  // | GRANT ALL PRIVILEGES ON *.* TO 'root'@'master' WITH GRANT OPTION |
  // | GRANT PROXY ON ''@'' TO 'root'@'master' WITH GRANT OPTION        |
  // +------------------------------------------------------------------+

  // SELECT Shutdown_priv FROM mysql.user WHERE host = 'master' AND user = 'root';
  // +---------------+
  // | Shutdown_priv |
  // +---------------+
  // | Y             |
  // +---------------+

  $sql = sprintf("SELECT Shutdown_priv
  FROM mysql.user
WHERE host = '%s'
  AND user = '%s'"
  , $mysqli->real_escape_string($host)
  , $mysqli->real_escape_string($user)
  );

  if ( ! ($result = $mysqli->query($sql)) ) {
    $rc = 159;
    $msg = "Error in query: $sql, " . $mysqli->error . ", (rc=$rc).";
    logMessage(LOG_ERR, $msg);
    // Skip nodes which are not created yet!
    if ( $aNode['node_id'] != 0 ) {
      $ret = writeCheck($aNode['node_id'], 'mysqld', 'database_privileges', $rc);
    }
    return array($rc, $msg);
  }

  if ( $record = $result->fetch_assoc() ) {

    if ( $record['Shutdown_priv'] == 'Y' ) {
      // No print out here!
      // print "&nbsp;&nbsp;SHUTDOWN privilege <span style='color:green;'><strong>OK</strong></span>.<br />\n";
    }
    else {
      $rc = 160;
      $msg = "SHUTDOWN privlege MISSING.";
      logMessage(LOG_ERR, $msg);
      // Skip nodes which are not created yet!
      if ( $aNode['node_id'] != 0 ) {
        $ret = writeCheck($aNode['node_id'], 'mysqld', 'database_privileges', $rc);
      }
      return array($rc, $msg);
    }
  }
  else {
    $rc = 161;
    $msg = "No record retrieved.";
    logMessage(LOG_ERR, $msg);
    // Skip nodes which are not created yet!
    if ( $aNode['node_id'] != 0 ) {
      $ret = writeCheck($aNode['node_id'], 'mysqld', 'database_privileges', $rc);
    }
    return array($rc, $msg);
  }

  // Skip nodes which are not created yet!
  if ( $aNode['node_id'] != 0 ) {
    $ret = writeCheck($aNode['node_id'], 'mysqld', 'database_privileges', $rc);
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $msg);
}

// ---------------------------------------------------------------------
function checkDatabaseVariables($mysqli, $aNode, $aVariables)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $msg = '';

  list($ret, $aVariables) = getDatabaseVariables($mysqli);

  // Check variables
  $err = 0;
  $msg = '';
  foreach ( $aVariables as $key => $value ) {

    if ( array_key_exists($key, $aVariables) ) {

      if ( $value == $aVariables[$key] ) {
        $msg .= "Variable $key ($value) is <span style='color:green;'><strong>OK</strong></span>.<br />";
      }
      else {
        $err++;
        $msg .= "Variable $key has different value in configuration and database ($value vs " . $aVariables[$key] . ").<br />";
        $rc = 162;
      }
    }
    else {
      $rc = 163;
      $msg .= "Variable $key does not exist.<br />";
      logMessage(LOG_ERR, $msg);
      // Skip nodes which are not created yet!
      if ( $aNode['node_id'] != 0 ) {
        $ret = writeCheck($aNode['node_id'], 'mysqld', 'database_variables', $rc);
      }
      return array($rc, $msg);
    }
  }

  if ( $err != 0 ) {
    $rc = 164;
    $msg .= "Not all variables are as expected.<br />";
    logMessage(LOG_ERR, $msg);
    // Skip nodes which are not created yet!
    if ( $aNode['node_id'] != 0 ) {
      $ret = writeCheck($aNode['node_id'], 'mysqld', 'database_variables', $rc);
    }
    return array($rc, $msg);
  }

  // Skip nodes which are not created yet!
  if ( $aNode['node_id'] != 0 ) {
    $ret = writeCheck($aNode['node_id'], 'mysqld', 'database_variables', $rc);
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $msg);
}

// ---------------------------------------------------------------------
function checkMyCnf($aServer, $aNode)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;

  $ssh = 'ssh -o ConnectTimeout=10 ' . $aServer['os_user'] . '@' . $aServer['name'] . ' ';

  // http://www.snailbook.com/faq/background-jobs.auto.html
  $cmd = $ssh . "'stat " . $aNode['my_cnf'] . " 2>&1' 2>&1";

  $stdout = exec($cmd, $output, $ret);
  $msg = $stdout;

  if ( $ret != 0 ) {
    $rc = 165;
    $msg = "Reading my.cnf was a problem: $stdout";
  }

  // Skip nodes which are not created yet!
  if ( $aNode['node_id'] != 0 ) {
    $ret = writeCheck($aNode['node_id'], 'mysqld', 'my_cnf', $rc);
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $msg);
}

// ---------------------------------------------------------------------
function checkDatabaseReadOnly($mysqli, $aNode)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $msg = '';

  list($ret, $aVariables) = getDatabaseVariables($mysqli);
  if ( $ret != OK ) {
    $msg = $aVariables;
    // Skip nodes which are not created yet!
    if ( $aNode['node_id'] != 0 ) {
      $ret = writeCheck($aNode['node_id'], 'mysqld', 'read_only', $rc);
    }
    return array($rc, $msg);
  }

  if ( $aVariables['read_only'] == 'ON' ) {
    $msg = 'read_only';
  }
  elseif ( $aVariables['read_only'] == 'OFF' ) {
    $msg = 'read_write';
  }
  else {
    $rc = 166;
    $msg = "Database is NOT read_only but " . $aVariables['read_only'] . " (rc=$rc).";
    logMessage(LOG_ERR, $msg);
    $ret = writeCheck($aNode['node_id'], 'mysqld', 'read_only', $rc);
    return array($rc, $msg);
  }

  // Skip nodes which are not created yet!
  if ( $aNode['node_id'] != 0 ) {

    // Database is read_only and should be read_only
    // OR database is read_write and should be read_write
    if ( (($msg == 'read_only') && ($aNode['read_only'] == 1))
      || (($msg == 'read_write') && ($aNode['read_only'] == 0))
      ) {
      $rc = OK;
      $ret = writeCheck($aNode['node_id'], 'mysqld', 'read_only', $rc);
    }
    else {
      $rc = 167;
      $ret = writeCheck($aNode['node_id'], 'mysqld', 'read_only', $rc);
    }
  }

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $msg);
}

// ---------------------------------------------------------------------
function writeCheck($pId, $pType, $pCheckName, $pStatus)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  list($ret, $dbh) = openRepositoryDatabase();
  list($ret, $aDatabaseCheck) = getDatabaseCheck($dbh, $pId, $pType, $pCheckName);

  $ts = time();
  if ( $pStatus == 0 ) {
    $status = 'OK';
  }
  else {
    $status = 'ERR';
  }

  // print "<pre>"; print_r($aDatabaseCheck); print "</pre>";

  // Check found
  if ( count($aDatabaseCheck) > 0 ) {

    if ( $aDatabaseCheck['last_check_status'] == 'OK' ) {
      $aDatabaseCheck['last_successful_check_ts'] = $aDatabaseCheck['last_check_ts'];
      $aDatabaseCheck['last_successful_status'] = $aDatabaseCheck['last_check_status'];
    }
    $aDatabaseCheck['last_check_ts'] = $ts;
    $aDatabaseCheck['last_check_status'] = $status;
    list($ret, $cnt) = updateDatabaseCheck($dbh, $aDatabaseCheck);
  }
  // Check for this node does not exist yet
  else {

    $aDatabaseCheck['unit_id']                      = $pId;
    $aDatabaseCheck['name']                         = $pCheckName;
    $aDatabaseCheck['type']                         = $pType;
    $aDatabaseCheck['last_check_ts']                = $ts;
    $aDatabaseCheck['last_check_status']            = $status;
    $aDatabaseCheck['last_successful_check_ts']     = $ts;
    $aDatabaseCheck['last_successful_check_status'] = $status;

    list($ret, $cnt) = addDatabaseCheck($dbh, $aDatabaseCheck);
  }
  closeRepositoryDatabase($dbh);

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return $rc;
}

// ---- Database Checks ----

// ---------------------------------------------------------------------
function getChecksWithFilter($dbh, $aFilter = array())
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $aChecks = array();

  $sql = "SELECT * FROM checks WHERE 1\n";

  foreach ( $aFilter as $key => $value ) {
    $sql .= "  AND $key = :$key\n";
  }
  logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

  $stmt = $dbh->prepare($sql);

  foreach ( $aFilter as $key => $value ) {
    $stmt->bindValue(":$key", $value);
  }

  try {
    $stmt->execute();

    while ( $result = $stmt->fetch(PDO::FETCH_ASSOC) ) {
      $aChecks[$result['name']] = $result;
    }
  }
  catch (PDOException $e) {
    $rc = 751;
    logMessage(LOG_ALERT, $e->getMessage() . " (rc=$rc)");
    $aChecks = $e->getMessage() . " (rc=$rc)";
  }
  $stmt->closeCursor();

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $aChecks);
}

// ---------------------------------------------------------------------
function getDatabaseCheck($dbh, $pUnitId, $pType, $pName)
// ---------------------------------------------------------------------
{
  list($rc, $aChecks) = getChecksWithFilter($dbh, array('unit_id' => $pUnitId, 'type' => $pType, 'name' => $pName));
  return array($rc, array_shift($aChecks));
}

// ---------------------------------------------------------------------
function getAllDatabaseChecks($dbh, $pUnitId)
// ---------------------------------------------------------------------
{
  return getChecksWithFilter($dbh, array('type' => 'mysqld', 'unit_id' => $pUnitId));
}

// ---------------------------------------------------------------------
function getAllServerChecks($dbh, $pUnitId)
// ---------------------------------------------------------------------
{
  return getChecksWithFilter($dbh, array('type' => 'server', 'unit_id' => $pUnitId));
}

// ---------------------------------------------------------------------
function addDatabaseCheck($dbh, $aDatabaseCheck)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $cnt = 0;

  if ( $aDatabaseCheck['name'] == '' ) {
    $rc = 168;
    logMessage(LOG_ERR, "$rc");
    return array($rc, $cnt);
  }

  // print "<pre>"; print_r($aDatabaseCheck); print "</pre>";

  $aDatabaseCheck['last_check_ts'] = isset($aDatabaseCheck['last_check_ts']) ? $aDatabaseCheck['last_check_ts'] : 0;
  $aDatabaseCheck['last_check_status'] = isset($aDatabaseCheck['last_check_status']) ? $aDatabaseCheck['last_check_status'] : '';
  $aDatabaseCheck['last_successful_check_ts'] = isset($aDatabaseCheck['last_successful_check_ts']) ? $aDatabaseCheck['last_successful_check_ts'] : 0;
  $aDatabaseCheck['last_successful_check_status'] = isset($aDatabaseCheck['last_successful_check_status']) ? $aDatabaseCheck['last_successful_check_status'] : '';

  $sql = "INSERT INTO checks (
  unit_id
, type
, name
, last_check_ts
, last_check_status
, last_successful_check_ts
, last_successful_check_status
) VALUES (
  :unit_id
, :type
, :name
, :last_check_ts
, :last_check_status
, :last_successful_check_ts
, :last_successful_check_status
)";

  $stmt = $dbh->prepare($sql);
  $stmt->bindValue(':unit_id', $aDatabaseCheck['unit_id']);
  $stmt->bindValue(':type', $aDatabaseCheck['type']);
  $stmt->bindValue(':name', $aDatabaseCheck['name']);
  $stmt->bindValue(':last_check_ts', $aDatabaseCheck['last_check_ts']);
  $stmt->bindValue(':last_check_status', $aDatabaseCheck['last_check_status']);
  $stmt->bindValue(':last_successful_check_ts', $aDatabaseCheck['last_successful_check_ts']);
  $stmt->bindValue(':last_successful_check_status', $aDatabaseCheck['last_successful_check_status']);

  try {
    $stmt->execute();
    $cnt = $stmt->rowCount();
  }
  catch (PDOException $e) {
    $rc = 169;
    print $e->getMessage() . " (rc=$rc)\n";
  }
  $stmt->closeCursor();

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $cnt);
}

// ---------------------------------------------------------------------
function updateDatabaseCheck($dbh, $aDatabaseCheck)
// ---------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $cnt = 0;

  $sql = "UPDATE checks
   SET last_check_ts                = :last_check_ts
     , last_check_status            = :last_check_status
     , last_successful_check_ts     = :last_successful_check_ts
     , last_successful_check_status = :last_successful_check_status
     , type                         = :type
WHERE name = :name
  AND unit_id = :unit_id";
  logMessage(LOG_DEBUG, trim(preg_replace('/\s+/', ' ', $sql)));

  $stmt = $dbh->prepare($sql);
  $stmt->bindValue(':name', $aDatabaseCheck['name']);
  $stmt->bindValue(':unit_id', $aDatabaseCheck['unit_id']);
  $stmt->bindValue(':type', $aDatabaseCheck['type']);
  $stmt->bindValue(':last_check_ts', $aDatabaseCheck['last_check_ts']);
  $stmt->bindValue(':last_check_status', $aDatabaseCheck['last_check_status']);
  $stmt->bindValue(':last_successful_check_ts', $aDatabaseCheck['last_successful_check_ts']);
  $stmt->bindValue(':last_successful_check_status', $aDatabaseCheck['last_successful_check_status']);

  try {
    $stmt->execute();
    $cnt = $stmt->rowCount();
  }
  catch (Exception $e) {
    $rc = 170;
    logMessage(LOG_ALERT, $e->getMessage() . " (rc=$rc)");
  }
  $stmt->closeCursor();

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $cnt);
}

// -------------------------------------------------------------------
function deleteChecks($dbh, $pUnitId, $pNodeType)
// -------------------------------------------------------------------
{
  global $gIndention;
  logMessage(LOG_NOTICE, 'Begin function: ' . basename(__FILE__) . '/' . __FUNCTION__);
  $gIndention += 2;

  $rc = OK;
  $sql = "DELETE FROM checks WHERE unit_id = :unit_id AND type = :node_type";

  $stmt = $dbh->prepare($sql);
  $stmt->bindValue(':unit_id', $pUnitId, PDO::PARAM_INT);
  $stmt->bindValue(':node_type', $pNodeType, PDO::PARAM_STR);

  try {
    $stmt->execute();
    $cnt = $stmt->rowCount();
  }
  catch (PDOException $e) {
    $rc = 171;
    $cnt = $e->getMessage();
  }
  $stmt->closeCursor();

  $gIndention -= 2;
  logMessage(LOG_NOTICE, 'End function: ' . __FUNCTION__ . " (rc=$rc).");
  return array($rc, $cnt);
}

?>
