#!/usr/bin/env php
<?php

//
// Copyright (C) 2013, 2014 FromDual GmbH
//
// This program is commercial software. Use of this software is governed
// by your applicable license agreement with FromDual GmbH
//

$gMyNameBase = dirname(dirname(__FILE__));
set_include_path(get_include_path() . PATH_SEPARATOR . $gMyNameBase);

require_once('lib/Node.inc');
require_once('lib/Repository.inc');
require_once('lib/Database.inc');
require_once('lib/Log.inc');
require_once('lib/Constants.inc');
require_once('lib/Check.inc');
require_once('lib/Configuration.inc');
require_once('lib/Server.inc');

// ---------------------------------------------------------------------
function processRunning($pid)
// ---------------------------------------------------------------------
{

  if ( file_exists('/proc/' . $pid) ) {
    return true;
  }
  else {
    return false;
  }
}

$rc = OK;

logMessage(LOG_INFO, 'Start ' . __FILE__);
$gIndention += 2;

// Check for lock file
$lLockDir  = $gMyNameBase . '/tmp/check_nodes.lock';
$lLockFile = $lLockDir . '/pid';
// mkdir is the only atomic operation!
if ( @mkdir($lLockDir) === false ) {

  // Check if lock file still exists
  if ( file_exists($lLockFile) ) {

    $pid = trim(file_get_contents($lLockFile));
    logMessage(LOG_WARNING, "Lock file $lLockFile exists and belongs to pid $pid.");

    $epoch = filemtime($lLockFile);
    $timeout = 300;
    // Process was running for more than $timeout seconds
    $runtime = time() - $epoch;
    if ( $runtime > $timeout ) {
      logMessage(LOG_INFO, "Process with pid $pid was running for $runtime seconds. Killing it and cleaning up.");
      posix_kill($pid, 9);
      unlink($lLockFile);
    }
    // It seems like we were started too early
    else {
      $rc = 100;
      logMessage(LOG_INFO, "We started too early.");
      $gIndention -= 2;
      logMessage(LOG_INFO, 'Finish ' . __FILE__ . " (rc=$rc).");
      exit($rc);
    }
  }
  // Lock file does not exist anymore
  else {
    // do nothing
  }
}

// Set Lock
$pid = getmypid() . "\n";
file_put_contents($lLockFile, $pid);

list($rc, $dbh) = openRepositoryDatabase();
list($rc, $aSections) = getAllDatabaseNodes($dbh);
list($rc, $aServers) = getServersWithFilter($dbh, array());


// Loop over all database nodes

foreach ( array_keys($aSections) as $name ) {

  logMessage(LOG_INFO, 'Check mysqld node ' . $name);

  $aNode = array();
  try {
    $aNode = $aSections[$name];
    list($failed_checks, $aChecks) = testNode($dbh, $aNode, false);
    if ( $failed_checks != 0 ) {
      $rc = 150;
      logMessage(LOG_ERR, 'Check of mysqld node ' . $name . " failed (failed checks=$failed_checks) (rc=$rc).\n");
    }
  }
  catch (Exception $e) {
    $rc = 151;
    logMessage(LOG_ERR, 'Check of mysqld node ' . $name . ' failed: ' . $e->getMessage() . " (rc=$rc)\n");
  }
}

closeRepositoryDatabase($dbh);


// Loop over all servers

foreach ( $aServers as $id => $server ) {

  logMessage(LOG_INFO, 'Check server ' . $server['name']);

  try {
		list($failed_checks, $aChecks) = testServer($dbh, $server, false);
    if ( $failed_checks != 0 ) {
      $rc = 175;
      logMessage(LOG_ERR, 'Check of server ' . $server['name'] . " failed (failed checks=$failed_checks) (rc=$rc).\n");
    }
  }
  catch (Exception $e) {
    $rc = 174;
    logMessage(LOG_ERR, 'Check of server ' . $server['name'] . ' failed: ' . $e->getMessage() . " (rc=$rc).\n");
  }
}


// remove lock file
unlink($lLockFile);
rmdir($lLockDir);

$gIndention -= 2;
logMessage(LOG_INFO, 'Finish ' . __FILE__ . " (rc=$rc)");
exit($rc);

?>
