#!/usr/bin/env php
<?php

$basedir = dirname(dirname(__FILE__));
include($basedir . '/lib/Constants.inc');
include($basedir . '/lib/mysql_bman.inc');
include($basedir . '/lib/mysql_bman_catalog.inc');
include($basedir . '/lib/myEnv.inc');

$rc = OK;

ini_set('date.timezone', 'Europe/Zurich');

// This variable is only for test automatization
$gTesting       = false;
$gaBackupPolicy = array('daily', 'weekly', 'monthly', 'quarterly', 'yearly', 'binlog');
$gLogFile       = './mysql_bman.log';;
$gBackupDir     = './bck';
$gRelease       = '1.1.0';

// Check requirements

$ret = OK;
$cmd = 'which mysqldump';
$stdout = exec("$cmd 2>&1", $output, $ret);
if ( $ret != OK ) {
	print $stdout . "\n";
	print "WARNING: The MySQL utility mysqldump seems to be not installed. Please install it first:\n";
	print "RedHat/CentOS: shell> sudo yum install mysql\n";
	print "Ubuntu/Debian: shell> sudo apt-get install mysql-client\n";
	print "SLES/OpenSuSE: shell> sudo zypper install mysql-client\n";
}
else {
	$gMysqlDump = trim($stdout);
}

$ret += checkMyEnvRequirements();

if ( $ret != OK ) {
	$rc = 120;
  if ( $ret == ERR ) {
    exit($rc);
  }
  else {
    // continue
  }
}


// Parse command line

$shortopts  = '';

$longopts  = array(
	'archive'
, 'archivedir:'
, 'backupdir:'
, 'cleanup'
, 'config:'
, 'catalog:'
, 'catalog-name:'
, 'create'
, 'help'
, 'log:'
, 'instance-name:'
, 'mode:'
, 'no-compress'
, 'no-memory-table-check'
, 'per-schema'
, 'policy:'
, 'retention:'
, 'schema:'
, 'simulate'
, 'target:'
, 'type:'
, 'upgrade'
);

// The parsing of options will end at the first non-option found, any-
// thing that follows is discarded. 
$aParameter = getopt($shortopts, $longopts);

if ( isset($aParameter['help']) ) {
  $rc = OK;
  printUsage();
  exit($rc);
}

// Check if options are entered correctly
// There should be one more argv (0) than Parameters
if ( (count($argv) - 1) != count($aParameter) ) {

	$rc = 180;

	print "ERROR: Options were not entered correctly. Please fix it (rc=$rc).\n";
	
	// Check and show which variables are not correct

	// Remove 1st option which is the filename
	unset($argv[0]);
	// Remove all options which were detected corretly from argv
	foreach ( $aParameter as $option => $v ) {
		foreach ( $argv as $key => $value ) {
		  $pattern = "/^\-\-$option/";
			if ( preg_match($pattern, $value) ) {
				unset($argv[$key]);
			}
		}
	}

	print "       I could not interprete the following options:\n";
	print_r($argv);
	exit($rc);
}

if ( ! isset($aOptions['config']) ) {
  $aOptions['config'] = '';
}


// Search for default MySQL configuration files
// */my.cnf, ~/.my.cnf, etc.
// https://dev.mysql.com/doc/refman/5.6/en/option-files.html

$aMyCnfFiles = array(
	'/etc/my.cnf'
, '/etc/mysql/my.cnf'
, $_ENV['HOME'] . '/.my.cnf'
);

$aMyCnf = array();
foreach ( $aMyCnfFiles as $filename ) {

	if ( is_readable($filename) ) {
		print "\nReading configuation from $filename\n";
		list($ret, $aConfig) = parseConfigFile($filename);
		// An error on parsing!
		if ( $aConfig === false ) {
			$rc = 164;
			$err = error_get_last(); 
			print trim($err['message']) . " (rc=$rc)\n";
			exit($rc);
		}
		
		// sections [client] [mysqldump]
		if ( array_key_exists('client', $aConfig) ) {
			if ( array_key_exists('user', $aConfig['client']) ) {
				$aMyCnf['user'] = $aConfig['client']['user'];
			}
			if ( array_key_exists('password', $aConfig['client']) ) {
				$aMyCnf['password'] = $aConfig['client']['password'];
			}
		}
		if ( array_key_exists('mysqldump', $aConfig) ) {
			if ( array_key_exists('user', $aConfig['mysqldump']) ) {
				$aMyCnf['user'] = $aConfig['mysqldump']['user'];
			}
			if ( array_key_exists('password', $aConfig['mysqldump']) ) {
				$aMyCnf['password'] = $aConfig['mysqldump']['password'];
			}
		}
	}
	else {
		// print "No config file $filename found.\n";
	}
}

foreach ( $aMyCnf as $key => $value ) {
  $aOptions[$key] = $value;
}


// Do we have a configuration file?
$gMysqlBman = '';
if ( isset($aParameter['config']) && is_readable($aParameter['config']) ) {
  $gMysqlBman = $aParameter['config'];
  list($rc, $aConfig) = readConfigurationFile($gMysqlBman);
  if ( $rc != OK ) {
		exit($rc);
	}
	foreach ( $aConfig as $key => $value ) {
		$aOptions[$key] = $value;
	}
}

// Options = Config + Parameter
// Rule: Parameters overwrite Config
foreach ( $aParameter as $key => $value ) {
  $aOptions[$key] = $value;
}

// print_r($aOptions); exit;

// Set some mandatory defaults
if ( (! isset($aOptions['log'])) || ($aOptions['log'] == '') ) {
  $aOptions['log'] = './mysql_bman.log';
}
if ( (! isset($aOptions['backupdir'])) || ($aOptions['backupdir'] == '') ) {
  $aOptions['backupdir'] = './bck';
}

if ( (! isset($aOptions['catalog-name'])) || ($aOptions['catalog-name'] == '') ) {
	$aOptions['catalog-name'] = 'bman_catalog';
}

// Minimum requirements is a log file
if ( ! isset($aOptions['log']) ) {
  // tee does not work here yet!
  $rc = 140;
  print "ERR: No log file specified!\n";
  exit($rc);
}

$gLogFile = $aOptions['log'];
$gBackupDir = $aOptions['backupdir'];
$ret = checkArguments($aOptions);

if ( $ret != OK ) {
  $rc = 127;
	print "More help can be found with mysql_bman --help\n";
  exit($rc);
}

// todo: Should be done above checkArguments!
if ( ! array_key_exists('mode', $aOptions) ) {
  $aOptions['mode'] = 'logical';
}

// Set type and policy to '' if catalog operations are wanted to avoid
// Undefined index: type error messages...
if ( ! isset($aOptions['type']) && (array_key_exists('create', $aOptions) || array_key_exists('upgrade', $aOptions)) ) {
	$aOptions['type'] = 'catalog';
	$aOptions['policy'] = array_key_exists('create', $aOptions) ? 'create' : 'upgrade';
	$aOptions['mode'] = '';
}

// todo: Should be done above checkArguments!
// If type = schema and no schema specified add empty schema option
if ( $aOptions['type'] == 'schema' ) {
  if ( ! isset($aOptions['schema']) ) {
    $aOptions['schema'] = '';
  }
}


// $gCatalog = '';
// $aAuxiliary = '';
// Target is not needed when type is cleanup
$aTarget = array();
if ( isset($aOptions['target']) ) {
  list($ret, $aTarget) = parseConnectString($aOptions['target']);
  
  if ( $ret != OK ) {
    $rc = 109;
    tee("  ERROR wrong connect string in target (rc=$rc)!");
    exit($rc);
  }
}

$aCatalog = array();
if ( isset($aOptions['catalog']) ) {
  list($ret, $aCatalog) = parseConnectString($aOptions['catalog']);
  
  if ( $ret != OK ) {
    $rc = 196;
    tee("  ERROR wrong connect string in catalog(rc=$rc)!");
    exit($rc);
  }
}


// If user or password was set in .my.cnf update target

// todo: Should be done above checkArguments!
if ( isset($aOptions['user']) ) {
	$aTarget['user'] = $aOptions['user'];
}
if ( isset($aOptions['password']) ) {
	$aTarget['password'] = $aOptions['password'];
}
// Adjust target again

if ( isset($aTarget['host']) ) {
	$aOptions['target'] = $aTarget['user'] . ':' . '******' . '@' . $aTarget['host'] . ':' . $aTarget['port'];
}

if ( isset($aCatalog['host']) ) {
	$aOptions['catalog'] = $aCatalog['user'] . ':' . '******' . '@' . $aCatalog['host'] . ':' . $aCatalog['port'];
}


if ( $gMysqlDump == '' ) {
  $rc = 110;
  tee("  ERROR: Cannot find mysqldump utility. Please set path accordingly (rc=$rc).");
  exit($rc);
}

// todo: Should be done above checkArguments!
// possibly becomes obsolete later when d/h is introduced
if ( isset($aOptions['retention']) ) {
  $aOptions['retention'] = intval($aOptions['retention']);
}

$gHostname = gethostname();

// ----------------------------------------------------------------------
// MAIN
// ----------------------------------------------------------------------

if ($gMysqlBman != '' ) {
  tee("\nConfiguration from " . $gMysqlBman);
  // Display parameter from config file
  $fh = fopen($gMysqlBman, 'r');
  if ( $fh ) {
    while ( ($buffer = fgets($fh, 4096)) !== false ) {
      if ( ! preg_match('/^$/', $buffer)
        && ! preg_match('/^\s*#.*$/', $buffer)
        ) {
        tee('  ' . trim($buffer));
      }
    }
    if ( ! feof($fh) ) {
      $rc = 136;
      tee("  ERROR: Unexpected fgets() fail (rc=$rc).");
      exit($rc);
    }
    fclose($fh);
  }
}
else {
	tee("\nNo mysql_bman configuration file.\n");
}


// Prepare command line
$cmdln = '';
foreach ( $argv as $val ) {
	if ( preg_match('/(--target=)(.*)(@.*)/', $val, $matches)
	  || preg_match('/(--catalog=)(.*)(@.*)/', $val, $matches)
	  ) {

		$val = $matches[1];
		if ( preg_match('|(.*[/:])(.*)|', $matches[2], $ms) ) {
			$val .= $ms[1] . '******';
		}
		$val .= $matches[3];
	}
	$cmdln .= $val . ' ';
}

tee('Command line: ' . $cmdln . "\n");


// Display parameter from command line options

tee("Configuration from command line options");
foreach ( $aParameter as $key => $value ) {
	if ( ($key == 'target') || ($key == 'catalog') ) {
		list($ret, $aT) = parseConnectString($value);
		tee(sprintf("  %-13s = %s", $key, $aT['user'] . ':' . '******' . '@' . $aT['host'] . ':' . $aT['port']   ));
		continue;
	}
	tee(sprintf("  %-13s = %s", $key, $value));
}

// Display resulting configuration
tee("Resulting options");
foreach ( $aOptions as $key => $value ) {
	if ( $key == 'password' ) {
		$value = '******';
	}
	tee(sprintf("  %-13s = %s", $key, $value));
}

list($ret, $aVersion) = getBmanVersion();

tee("\nLogging to   " . $gLogFile);
tee("Backupdir is " . $gBackupDir);
tee("Hostname is  " . $gHostname);
tee("Version is   " . $gRelease . ' (catalog v' . $aVersion['hr_version'] . ')');
if ( array_key_exists('simulate', $aOptions) ) {
	tee("This is a SIMULATION only!");
}


// We will use a catalog
if ( array_key_exists('catalog', $aOptions) ) {

	list($rc, $mysqli) = openCatalog($aCatalog, $aOptions['catalog-name']);
	if ( $rc != OK ) {
		$rc = 224;
		$msg = "Connection to catalog failed (rc=$rc).";
		tee("  ERROR: $msg");
		exit($rc);
	}

	// On catalog creation
	if ( array_key_exists('create', $aOptions) ) {
		// noop
	}
	// Otherwise : upgrade or normal backup
	else {

		list($ret, $cnt) = checkIfCatalogExists($mysqli, $aOptions['catalog-name']);
		if ( ($ret == OK) && ($cnt != 0) ) {
			list($rc, $aCatalogVersion) = getCatalogVersion($mysqli, $aOptions['catalog-name']);
			tee("Catalog is   " . $aOptions['catalog-name'] . " version " . $aCatalogVersion['hr_version']);
		}
		// Catalog does NOT exist
		else {
			$rc = 203;
			$msg = "Catalog does NOT exist. Please create catalog first with --create (rc=$rc).";
			tee("  ERROR: $msg");
			exit($rc);
		}

		// Check if catalog version matches

		list($ret, $aBmanVersion) = getBmanVersion();

		if ( $aCatalogVersion['mr_version'] != $aBmanVersion['mr_version'] ) {
			$rc = 199;
			$msg = "Catalog has not the same version as software. (rc=$rc).";
			tee(" ERROR: $msg");
			exit($rc);
		}
	}
}


// Not necessary if it is cleanup or catalog

if ( ($aOptions['type'] != 'catalog') && ($aOptions['type'] != 'cleanup') ) {

	// Should be check and create Backup structure
	$rc = checkBackupStructure($gBackupDir);
	if ( $rc != OK ) {
		$rc = 129;
		tee("  ERROR: Problems creating backup structure (rc=$rc).");
		exit($rc);
	}
}


// todo: should go to optionCheck
if ( array_key_exists('archive', $aOptions) ) {
	if ( ! isset($aOptions['archivedir']) || ($aOptions['archivedir'] == '' ) ) {
		$rc = 134;
		tee("  ERROR: Archivedir is not specified (rc=$rc).");
		exit($rc);
	}
}

if ( array_key_exists('archive', $aOptions) ) {
	$rc = checkArchiveStructure($aOptions['archivedir']);
	if ( $rc != OK ) {
		$rc = 147;
		tee("  ERROR: Problems creating archive structure (rc=$rc).");
		exit($rc);
	}
}


// Is empty for type cleanup

$gTargetConfiguration = array();
if ( (count($aTarget) > 0) && ($aOptions['type'] != 'cleanup') ) {
	list($rc, $gTargetConfiguration) = getTargetConfiguration($aTarget);
	if ( $rc != OK ) {
		exit($rc);
	}
}


// Start here:

$gBackupTimestamp = date("Y-m-d_H-i-s");

// Write backup start tag into catalog
if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('create', $aOptions)) && (! array_key_exists('upgrade', $aOptions)) ) {

  $aBackupInformation = array(
    'id'               => 0
	, 'instance_name'    => $aOptions['instance-name']
	, 'start_ts'         => time()
	, 'stop_ts'          => 0
	, 'rc'               => 9999
	, 'backup_type_id'   => getBackupTypeId($aOptions['type'])
	, 'backup_policy_id' => getBackupPolicyId($aOptions['policy'])
	, 'backup_mode_id'   => getBackupModeId($aOptions['mode'])
	, 'compressed'       => isset($aOptions['no-compress']) ? 0 : 1
	, 'archived'         => isset($aOptions['archive']) ? 1 : 0
	, 'cleanedup'        => isset($aOptions['cleanup']) ? 1 : 0
	, 'version'          => $gRelease
	, 'hostname'         => $gHostname
	, 'schema_name'      => ''
	, 'binlog_file'      => ''
	, 'binlog_pos'       => 0
	, 'command'          => ''
	, 'log'              => ''
	, 'mysqli'           => $mysqli
  );
  
	list($rc, $id) = createBackupInCatalog($mysqli, $aBackupInformation);
	$aBackupInformation['id'] = $id;
}

tee("\nStart " . $aOptions['type'] . ' ' . $aOptions['policy'] . " backup at $gBackupTimestamp");

// Do backups

if ( $aOptions['type'] == 'full' ) {
  if ( $aOptions['mode'] == 'logical' ) {
    $rc = doFullLogicalBackup($aTarget, $aOptions, $aBackupInformation);
  }
  elseif ( $aOptions['mode'] == 'physical' ) {
    $rc = doFullPhysicalBackup($aTarget, $aOptions, $aBackupInformation);
  }
}
elseif ( $aOptions['type'] == 'structure' ) {
  $rc = doStructureBackup($aTarget, $aOptions, $aBackupInformation);
}
elseif ( $aOptions['type'] == 'binlog' ) {
  $rc = doBinlogBackup($aTarget, $aOptions, $aBackupInformation);
}
elseif ( $aOptions['type'] == 'config' ) {
  $rc = doConfigBackup($aTarget, $aOptions, $aBackupInformation);
}
elseif ( $aOptions['type'] == 'cleanup' ) {
  $rc = doCleanup($aOptions, $aBackupInformation);
}
elseif ( $aOptions['type'] == 'schema' ) {
  $rc = doSchemaBackup($aTarget, $aOptions, $aBackupInformation);
}
elseif ( $aOptions['type'] == 'privilege' ) {
  $rc = doPrivilegeBackup($aTarget, $aOptions, $aBackupInformation);
}
elseif ( $aOptions['type'] == 'catalog' ) {

	if ( $aOptions['policy'] == 'create' ) {
		$rc = createCatalog($mysqli, $aOptions['catalog-name']);
		if ( $rc == OK ) {
			tee("  Catalog successfully created.");
		}
	}
	elseif ( $aOptions['policy'] == 'upgrade' ) {
		$rc = upgradeCatalog($mysqli, $aOptions['catalog-name']);
		if ( $rc == OK ) {
			tee("  Catalog successfully upgraded.");
		}
	}
	else {
		$rc = 197;
		tee("  ERROR: Unexpected error. This should never happen (type=" . $aOptions['type'] . ', policy=' . $aOptions['policy'] . ")! Please report a bug! (rc=$rc).");
	}
}
else {
	$rc = 163;
	tee("  ERROR: Unexpected error. This should never happen (" . $aOptions['type'] . ")! Please report a bug! (rc=$rc).");
}

$lEndTs = date("Y-m-d_H-i-s");

// Write backup start tag into catalog
if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('create', $aOptions)) && (! array_key_exists('upgrade', $aOptions)) ) {
	$aBackupInformation['stop_ts'] = time();
	$aBackupInformation['rc']      = $rc;
	list($ret, $cnt) = setBackupInCatalog($mysqli, $aBackupInformation);
	list($ret, $cnt) = setBackupDetailsInCatalog($mysqli, $aBackupInformation);
}

if ( array_key_exists('catalog', $aOptions) ) {
	$ret = closeCatalog($mysqli);
}

tee("End " . $aOptions['type'] . " " . $aOptions['policy'] . " backup at $lEndTs (rc=$rc)");
exit($rc);

?>
