<?php

/*

	openCatalog($aCatalog, $pCatalogName)
	closeCatalog($mysqli)
	checkIfCatalogExists($mysqli, $pCatalogName)
	createCatalog($mysqli, $pCatalogName)
	createTableMetadata($mysqli)
	createTableBackups($mysqli)
	createTableBackupDetails($mysqli)
	createTableFiles($mysqli)
	createTableBinaryLogs($mysqli)
	getCatalogVersion($mysqli)
	getBmanVersion()
	upgradeCatalog($mysqli, $pCatalogName)
	upgradeCatalogFrom010to020($mysqli)
	createBackupInCatalog($mysqli, $aBackup)
	setBackupInCatalog($mysqli, $aBackup)
	setBackupDetailsInCatalog($mysqli, $aBackup)
	setFileInCatalog($mysqli, $aFile)
	setBinaryLogInCatalog($mysqli, $aBinaryLog)

*/

// ---------------------------------------------------------------------
function openCatalog($aCatalog, $pCatalogName)
// ---------------------------------------------------------------------
{
	$rc = OK;
  
	list($rc, $mysqli) = openConnection($aCatalog);
	if ( $rc != OK ) {
		$rc = 182;
		$msg = "Opening Catalog connecton failed (rc=$rc).\n";
		$msg .= "  Please create the catalog user as follows:\n  CREATE USER '" . $aCatalog['user'] . "'@'%' IDENTIFIED BY '<some password>';\n";
		$msg .= "  GRANT ALL ON `$pCatalogName`.* to '" . $aCatalog['user'] . "'@'%';";
		tee("  ERROR: $msg");
		$mysqli = $msg;
		return array($rc, $mysqli);
	}

	$ret = $mysqli->select_db($pCatalogName);
	if ( $ret !== true ) {
		$rc = 204;
		$msg  = "Failed to access catalog schema $pCatalogName: " . $mysqli->error . ' (' . $mysqli->errno . " / rc=$rc)" . "\n";
		$msg .= "  Please create catalog schema manually with:\n  CREATE SCHEMA `$pCatalogName` CHARACTER SET utf8;";
		tee("  ERROR: $msg");
		$mysqli = null;
		return array($rc, $mysqli);
	}

		return array($rc, $mysqli);
}

// ---------------------------------------------------------------------
function closeCatalog($mysqli)
// ---------------------------------------------------------------------
{
	$rc = OK;
	closeConnection($mysqli);
	return $rc;
}

// ---------------------------------------------------------------------
function checkIfCatalogExists($mysqli, $pCatalogName)
// ---------------------------------------------------------------------
{
	$rc = OK;

	$sql = sprintf("SELECT COUNT(*) AS cnt FROM information_schema.tables WHERE table_schema = '%s' AND table_name = '%s'", $pCatalogName, 'metadata');

	$cnt = 0;
	if ( ! ($result = $mysqli->query($sql)) ) {
		$rc = 183;
		$msg =  "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
		tee(" ERROR: $msg");
	}
	else {

		if ( $record = $result->fetch_assoc() ) {
			$cnt = $record['cnt'];
		}
	}

	return array($rc, $cnt);
}

// ---------------------------------------------------------------------
function createCatalog($mysqli, $pCatalogName)
// ---------------------------------------------------------------------
{
	$rc = OK;

	list($ret, $cnt) = checkIfCatalogExists($mysqli, $pCatalogName);
	if ( ($ret == OK) && ($cnt != 0) ) {
		$rc = 184;
		$msg = "Catalog $pCatalogName already exists (rc=$rc).";
		tee(" ERROR: $msg");
		return $rc;
	}
	else {
		$ret = createTableMetadata($mysqli);
		if ( $ret != OK ) {
			$rc = 185;
			return $rc;
		}
		$ret = createTableBackups($mysqli);
		if ( $ret != OK ) {
			$rc = 186;
			return $rc;
		}
		$ret = createTableBackupDetails($mysqli);
		if ( $ret != OK ) {
			$rc = 221;
			return $rc;
		}
		$ret = createTableFiles($mysqli);
		if ( $ret != OK ) {
			$rc = 187;
			return $rc;
		}
		$ret = createTableBinaryLogs($mysqli);
		if ( $ret != OK ) {
			$rc = 188;
			return $rc;
		}
	}

	return $rc;
}

// ---------------------------------------------------------------------
function createTableMetadata($mysqli)
// ---------------------------------------------------------------------
{
	$rc = OK;

	try {

		$sql = sprintf("CREATE TABLE metadata (
  id      TINYINT UNSIGNED NOT NULL AUTO_INCREMENT
, `key`   VARCHAR(32) NOT NULL
, `value` VARCHAR(255) NULL
, PRIMARY KEY (id)
, UNIQUE KEY (`key`)
)");
		if ( ! ($result = $mysqli->query($sql)) ) {
			$rc = 189;
			$msg =  "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
			throw new Exception($msg);
		}

		$sql = sprintf("INSERT INTO metadata (`key`, `value`) VALUES ('hr_version', '0.1.0')");
		if ( ! ($result = $mysqli->query($sql)) ) {
			$rc = 190;
			$msg =  "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
			throw new Exception($msg);
		}

		$sql = sprintf("INSERT INTO metadata (`key`, `value`) VALUES ('mr_version', '010')");
		if ( ! ($result = $mysqli->query($sql)) ) {
			$rc = 191;
			$msg =  "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
			throw new Exception($msg);
		}
	}
	catch (Exception $e) {
		tee(" ERROR: " . $e->getMessage());
	}

	return $rc;
}

// ---------------------------------------------------------------------
function createTableBackups($mysqli)
// ---------------------------------------------------------------------
{
	$rc = OK;

	// command is not known at the begin, thus NULL
	$sql = sprintf("CREATE TABLE backups (
  `id`               INT UNSIGNED NOT NULL AUTO_INCREMENT
, `instance_name`    VARCHAR(64) NOT NULL
, `start_ts`         INT UNSIGNED NOT NULL
, `stop_ts`          INT UNSIGNED NULL
, `rc`               SMALLINT UNSIGNED
, `backup_type_id`   TINYINT UNSIGNED NOT NULL
, `backup_policy_id` TINYINT UNSIGNED NOT NULL
, `backup_mode_id`   TINYINT UNSIGNED NOT NULL
, `version`          VARCHAR(12) NOT NULL
, `compressed`       TINYINT UNSIGNED NOT NULL
, `archived`         TINYINT UNSIGNED NOT NULL
, `cleanedup`        TINYINT UNSIGNED NOT NULL
, PRIMARY KEY (`id`)
, INDEX (`instance_name`)
)");

	if ( ! ($result = $mysqli->query($sql)) ) {
		$rc = 222;
		$msg =  "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
		tee(" ERROR: $msg");
		// No exit here
	}

	return $rc;
}

// ---------------------------------------------------------------------
function createTableBackupDetails($mysqli)
// ---------------------------------------------------------------------
{
	$rc = OK;

	// command is not known at the begin, thus NULL
	$sql = sprintf("CREATE TABLE backup_details (
  `backup_id`   INT UNSIGNED NOT NULL
, `hostname`    VARCHAR(255) NULL
, `binlog_file` VARCHAR(255) NULL
, `binlog_pos`  INT UNSIGNED NULL
, `command`     VARCHAR(4096) NULL
, `log`         TEXT NULL
, PRIMARY KEY (`backup_id`)
)");

	if ( ! ($result = $mysqli->query($sql)) ) {
		$rc = 192;
		$msg =  "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
		tee(" ERROR: $msg");
		// No exit here
	}

	return $rc;
}

// ---------------------------------------------------------------------
function createTableFiles($mysqli)
// ---------------------------------------------------------------------
{
	$rc = OK;

	$sql = sprintf("CREATE TABLE files (
  `id`                INT UNSIGNED NOT NULL AUTO_INCREMENT
, `schema_name`       VARCHAR(64) NULL
, `original_name`     VARCHAR(255) NOT NULL
, `original_location` VARCHAR(1024) NOT NULL
, `original_size`     BIGINT UNSIGNED NOT NULL
, `file_type_id`      TINYINT UNSIGNED NOT NULL
, `mtime`             INT UNSIGNED NOT NULL
, `backup_name`       VARCHAR(255) NOT NULL
, `backup_location`   VARCHAR(1024) NOT NULL
, `md5`               CHAR(32) NOT NULL
, `compressed`        TINYINT UNSIGNED NOT NULL
, `compressed_name`   VARCHAR(1024) NOT NULL
, `compressed_size`   BIGINT UNSIGNED NOT NULL
, `archived`          TINYINT UNSIGNED NULL
, `archive_location`  VARCHAR(1024) NOT NULL
, `cleanedup`         TINYINT UNSIGNED NOT NULL
, `backup_id`         INT UNSIGNED NOT NULL
, PRIMARY KEY (`id`)
, INDEX (`backup_id`)
)");

	if ( ! ($result = $mysqli->query($sql)) ) {
		$rc = 193;
		$msg =  "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
		tee(" ERROR: $msg");
		// No exit here
	}

	return $rc;
}

// ---------------------------------------------------------------------
function createTableBinaryLogs($mysqli)
// ---------------------------------------------------------------------
{
	$rc = OK;

	$sql = sprintf("CREATE TABLE binary_logs (
  `id`             INT UNSIGNED NOT NULL AUTO_INCREMENT
, `filename`       VARCHAR(255) NOT NULL
, `begin_ts`       INT UNSIGNED NOT NULL
, `end_ts`         INT UNSIGNED NOT NULL
, `begin_position` BIGINT UNSIGNED NOT NULL
, `end_position`   BIGINT UNSIGNED NOT NULL
, `file_id`        INT UNSIGNED NOT NULL
, PRIMARY KEY (`id`)
, INDEX (`file_id`)
)");

	if ( ! ($result = $mysqli->query($sql)) ) {
		$rc = 194;
		$msg =  "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
		tee(" ERROR: $msg");
		// No exit here
	}

	return $rc;
}

// ---------------------------------------------------------------------
function getCatalogVersion($mysqli)
// ---------------------------------------------------------------------
{
	$rc = OK;

	$aVersion = array();

	$sql = sprintf("SELECT `key`, `value` FROM metadata WHERE `key` IN ('mr_version', 'hr_version')");

	if ( ! ($result = $mysqli->query($sql)) ) {
		$rc = 195;
		$msg =  "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
		tee(" ERROR: $msg");
	}
	else {

		while ( $record = $result->fetch_assoc() ) {
			$aVersion[$record['key']] = $record['value'];
		}
	}

	return array($rc, $aVersion);
}

// ---------------------------------------------------------------------
function getBmanVersion()
// ---------------------------------------------------------------------
{
	$rc = OK;
	return array($rc, array('mr_version' => '010', 'hr_version' => '0.1.0'));
}

// ---------------------------------------------------------------------
function upgradeCatalog($mysqli, $pCatalogName)
// ---------------------------------------------------------------------
{
	$rc = OK;

	list($ret, $cnt) = checkIfCatalogExists($mysqli, $pCatalogName);
	if ( ($ret == OK) && ($cnt == 0) ) {
		$rc = 198;
		$msg = "Catalog $pCatalogName does NOT exists (rc=$rc).";
		tee(" ERROR: $msg");
		return $rc;
	}

	list($ret, $aBmanVersion) = getBmanVersion();
	list($ret, $aCatalogVersion) = getCatalogVersion($mysqli);

	// 3 cases: Catalog is older same or newer
	if ( $aCatalogVersion['mr_version'] < $aBmanVersion['mr_version'] ) {

		$cnt = 0;
		while ( $aCatalogVersion['mr_version'] < $aBmanVersion['mr_version'] ) {

			switch ( $aCatalogVersion['mr_version'] ) {
				case '010':
					$rc = upgradeCatalogFrom010to020($mysqli);
					if ( $rc != OK ) {
						tee(" ERROR: Upgrade from version " . $aCatalogVersion['hr_version'] . " to next higher version failed, aborting... (rc=$rc).");
						exit($rc);
					}
					break;
				default:
					$rc = 179;
					tee(" ERROR: We should never reach here! (" . $aCatalogVersion['hr_version'] . "), aborting... (rc=$rc).");
					exit($rc);
			}

			list($ret, $aCatalogVersion) = getCatalogVersion($mysqli);

			// We hope we never get in an endless loop here
			if ( $cnt > 100 ) {
				$rc = 181;
				tee(" ERROR: we caugth an endless loop, aborting... (rc=$rc).");
				tee(" Catalog version: " . $aCatalogVersion['hr_version'] . ", software version: " . $aBmanVersion['hr_version']);
				exit($rc);
			}
			$cnt++;
		}   // while
	}
	elseif ( $aCatalogVersion['mr_version'] == $aBmanVersion['mr_version'] ) {
		tee(" INFO: No catalog upgrade required. Catalog has already same version as software (" . $aBmanVersion['hr_version'] . ").");
	}
	else {
		$rc = 176;
		tee(" WARNING: Catalog (" . $aCatalogVersion['hr_version'] . ") is already newer than software (" . $aBmanVersion['hr_version'] . ") (rc=$rc).");
	}

	return $rc;
}

// ---------------------------------------------------------------------
function upgradeCatalogFrom010to020($mysqli)
// ---------------------------------------------------------------------
{
	$rc = OK;

	try {

		$sql = sprintf("UPDATE metadata SET `value` = '0.2.0' WHERE `key` = 'hr_version'");
		if ( ! ($result = $mysqli->query($sql)) ) {
			$rc = 141;
			$msg =  "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
			throw new Exception($msg);
		}

		$sql = sprintf("UPDATE metadata SET `value` = '020' WHERE `key` = 'mr_version'");
		if ( ! ($result = $mysqli->query($sql)) ) {
			$rc = 178;
			$msg =  "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
			throw new Exception($msg);
		}
	}
	catch (Exception $e) {
		tee(" ERROR: " . $e->getMessage());
	}

	return $rc;
}

// ---------------------------------------------------------------------
function createBackupInCatalog($mysqli, $aBackup)
// ---------------------------------------------------------------------
{
	$rc = OK;

	$sql = sprintf("INSERT INTO backups (
	  `instance_name`, `start_ts`, `stop_ts`, `rc`
	, `backup_type_id`, `backup_policy_id`, `backup_mode_id`
	, `compressed`, `archived`, `cleanedup`
	, `version`)
	VALUES ('%s', %d, %d, %d, %d, %d, %d, %d, %d, %d, '%s')"
	, $aBackup['instance_name'], $aBackup['start_ts'], $aBackup['stop_ts'], $aBackup['rc']
	, $aBackup['backup_type_id'], $aBackup['backup_policy_id'], $aBackup['backup_mode_id']
	, $aBackup['compressed'], $aBackup['archived'], $aBackup['cleanedup']
	, $aBackup['version']);

	if ( ! ($result = $mysqli->query($sql)) ) {
		$rc = 145;
		$msg =  "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
		tee(" ERROR: " . $msg);
		return array($rc, 0);
	}
	$id = $mysqli->insert_id;

	$sql = sprintf("INSERT INTO backup_details (
	  `backup_id`
	, `hostname`, `binlog_file`, `binlog_pos`
	, `command`, `log`)
	VALUES (%d, '%s', '%s', %d, '%s', '%s')"
	, $id
	, $aBackup['hostname'], $aBackup['binlog_file'], $aBackup['binlog_pos']
	, $aBackup['command'], $aBackup['log']);

	if ( ! ($result 	= $mysqli->query($sql)) ) {
		$rc = 157;
		$msg =  "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
		tee(" ERROR: " . $msg);
		return array($rc, 0);
	}

	return array($rc, $id);
}

// ---------------------------------------------------------------------
function setBackupInCatalog($mysqli, $aBackup)
// ---------------------------------------------------------------------
{
	$rc = OK;

	$sql = sprintf("UPDATE backups SET
	  `instance_name`    = '%s'
	, `start_ts`         = %d
	, `stop_ts`          = %d
	, `rc`               = %d
	, `backup_type_id`   = %d
	, `backup_policy_id` = %d
	, `backup_mode_id`   = %d
	, `compressed`       = %d
	, `archived`         = %d
	, `cleanedup`        = %d
	, `version`          = '%s'
	WHERE id = %d"
	, $aBackup['instance_name'], $aBackup['start_ts'], $aBackup['stop_ts'], $aBackup['rc']
	, $aBackup['backup_type_id'], $aBackup['backup_policy_id'], $aBackup['backup_mode_id']
	, $aBackup['compressed'], $aBackup['archived'], $aBackup['cleanedup']
	, $aBackup['version']
	, $aBackup['id']);

	if ( ! ($result = $mysqli->query($sql)) ) {
		$rc = 177;
		$msg =  "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
		tee(" ERROR: " . $msg);
		return array($rc, 0);
	}

	return array($rc, $mysqli->affected_rows);
}

// ---------------------------------------------------------------------
function setBackupDetailsInCatalog($mysqli, $aBackup)
// ---------------------------------------------------------------------
{
	$rc = OK;

	$sql = sprintf("UPDATE backup_details SET
	  `hostname`    = '%s'
	, `binlog_file` = '%s'
	, `binlog_pos`  = %d
	, `command`     = '%s'
	, `log`         = '%s'
	WHERE `backup_id` = %d"
	, $aBackup['hostname'], $aBackup['binlog_file'], $aBackup['binlog_pos']
	, $aBackup['command'], $aBackup['log'], $aBackup['id']);

	if ( ! ($result = $mysqli->query($sql)) ) {
		$rc = 218;
		$msg =  "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
		tee(" ERROR: " . $msg);
		return array($rc, 0);
	}

	return array($rc, $mysqli->affected_rows);
}

// ---------------------------------------------------------------------
function setFileInCatalog($mysqli, &$aFile)
// ---------------------------------------------------------------------
{
	$rc = OK;
	$id = 0;

	// Creater a new record
	if ( $aFile['id'] == 0 ) {
	
		$sql = sprintf("INSERT INTO files (
			`schema_name`, `original_name`, `original_location`, `original_size`, `file_type_id`
		, `mtime`, `backup_name`, `backup_location`, `md5`
		, `compressed`, `compressed_name`, `compressed_size`
		, `archived`, `archive_location`, `cleanedup`
		, `backup_id`)
		VALUES ('%s', '%s', '%s', %d, %d, %d, '%s', '%s', '%s', %d, '%s', %d, %d, '%s', %d, %d)"
		, $aFile['schema_name'], $aFile['original_name'], $aFile['original_location'], $aFile['original_size'], $aFile['file_type_id']
		, $aFile['mtime'], $aFile['backup_name'], $aFile['backup_location'], $aFile['md5']
		, $aFile['compressed'], $aFile['compressed_name'], $aFile['compressed_size']
		, $aFile['archived'], $aFile['archive_location'], $aFile['cleanedup']
		, $aFile['backup_id']);

		if ( ! ($result = $mysqli->query($sql)) ) {
			$rc = 214;
			$msg =  "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
			tee(" ERROR: " . $msg);
			return array($rc, 0);
		}
		$id = $mysqli->insert_id;
		$aFile['id'] = $id;
	}
	// Update existing record
	else {
	
		$sql = sprintf("UPDATE files SET
		  `schema_name`       = '%s'
		, `original_name`     = '%s'
		, `original_location` = '%s'
		, `original_size`     = %d
		, `file_type_id`      = %d
		, `mtime`             = %d
		, `backup_name`       = '%s'
		, `backup_location`   = '%s'
		, `md5`               = '%s'
		, `compressed`        = %d
		, `compressed_name`   = '%s'
		, `compressed_size`   = %d
		, `archived`          = %d
		, `archive_location`  = '%s'
		, `cleanedup`         = %d
		, `backup_id`         = %d
		WHERE `id` = %d"
		, $aFile['schema_name'], $aFile['original_name'], $aFile['original_location'], $aFile['original_size'], $aFile['file_type_id']
		, $aFile['mtime'], $aFile['backup_name'], $aFile['backup_location'], $aFile['md5']
		, $aFile['compressed'], $aFile['compressed_name'], $aFile['compressed_size']
		, $aFile['archived'], $aFile['archive_location'], $aFile['cleanedup']
		, $aFile['backup_id'], $aFile['id']);

		if ( ! ($result = $mysqli->query($sql)) ) {
			$rc = 217;
			$msg =  "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
			tee(" ERROR: " . $msg);
			return array($rc, 0);
		}
		$id = $mysqli->affected_rows;
	}
	
	return array($rc, $id);
}

// ---------------------------------------------------------------------
function setBinaryLogInCatalog($mysqli, &$aBinaryLog)
// ---------------------------------------------------------------------
{
	$rc = OK;
	$id = 0;

	// Creater a new record
	if ( $aBinaryLog['id'] == 0 ) {
	
		$sql = sprintf("INSERT INTO binary_logs (
			`filename`
		, `begin_ts`, `end_ts`
		, `begin_position`, `end_position`
		, `file_id`
		)
		VALUES ('%s', %d, %d, %d, %d, %d)"
		, $aBinaryLog['filename']
		, $aBinaryLog['begin_ts'], $aBinaryLog['end_ts']
		, $aBinaryLog['begin_position'], $aBinaryLog['end_position']
		, $aBinaryLog['file_id']
		);

		if ( ! ($result = $mysqli->query($sql)) ) {
			$rc = 215;
			$msg =  "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
			tee(" ERROR: " . $msg);
			return array($rc, 0);
		}
		$id = $mysqli->insert_id;
		$aBinaryLog['id'] = $id;
	}
	// Update existing record
	else {
	
		$sql = sprintf("UPDATE binary_logs
		  `filename`       = '%s'
		, `begin_ts`       = %d
		, `end_ts`         = %d
		, `begin_position` = %d
		, `end_position`   = %d
		, `file_id`        = %d"
		, $aBinaryLog['filename']
		, $aBinaryLog['begin_ts'], $aBinaryLog['end_ts']
		, $aBinaryLog['begin_position'], $aBinaryLog['end_position']
		, $aBinaryLog['file_id']
		, $aBinaryLog['id']);

		if ( ! ($result = $mysqli->query($sql)) ) {
			$rc = 216;
			$msg =  "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
			tee(" ERROR: " . $msg);
			return array($rc, 0);
		}
		$id = $mysqli->affected_rows;
	}
	
	return array($rc, $id);
}

// ---------------------------------------------------------------------
function setFileCleanedupInCatalog($mysqli, $pInstanceName, $pFile, $pMtime)
// ---------------------------------------------------------------------
{
	$rc = OK;

	$sql = sprintf("UPDATE backups AS b
  JOIN files AS f ON f.backup_id = b.id
   SET f.cleanedup = 1
 WHERE b.instance_name = '%s'
   AND (f.original_name = '%s' OR f.compressed_name = '%s' OR f.backup_name = '%s')
   AND mtime = %d"
	, $pInstanceName
	, basename($pFile)
	, basename($pFile)
	, $pMtime
	);
   
	if ( ! ($result = $mysqli->query($sql)) ) {
		$rc = 119;
		$msg =  "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
		tee(" ERROR: " . $msg);
		return array($rc, 0);
	}
	$cnt = $mysqli->affected_rows;

	return array($rc, $cnt);
}

// ---------------------------------------------------------------------
function setBackupCleanedupInCatalog($mysqli)
// ---------------------------------------------------------------------
{
	$rc = OK;

	$sql = sprintf("SELECT f.backup_id, b.instance_name, b.cleanedup, SUM(f.cleanedup) AS sum , COUNT(*) AS cnt
  FROM files AS f
  JOIN backups AS b ON b.id = f.backup_id
 GROUP BY f.backup_id, b.instance_name, b.cleanedup
 HAVING SUM(f.cleanedup) = COUNT(*) 
    AND b.cleanedup = 0");
   
	if ( ! ($result = $mysqli->query($sql)) ) {
		$rc = 219;
		$msg =  "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
		tee(" ERROR: " . $msg);
		return array($rc, 0);
	}

	while ( $record = $result->fetch_assoc() ) {

		$sql2 = sprintf("UPDATE backups SET cleanedup = 1 WHERE id = %d", $record['backup_id']);
		if ( ! ($result2 = $mysqli->query($sql2)) ) {
			$rc = 220;
			$msg =  "Invalid query: $sql2, " . $mysqli->error . " (rc=$rc).";
			tee(" ERROR: " . $msg);
			// no return here
		}
		else {
			print '  Backup #' . $record['backup_id'] . ' of instance ' . $record['instance_name'] . ' cleanedup completely now.' . "\n";
		}
	}
	$cnt = $result->num_rows;

	return array($rc, $cnt);
}

?>
