<?php

/*

	printUsage()
	readConfigurationFile($pFile)
	getMasterPosFromDump($lDump)
	checkArguments($aOptions)
	closeConnection($mysqli)
	openConnection($aConnection)
	getTargetConfiguration($aTarget)
	parseConnectString($pConnectString)
	checkBackupStructure($pBackupDir)
	checkArchiveStructure($pArchiveDir)
	tee($s)
	doStructureBackup($aTarget, $aOptions)
	getMD5sum($file, $aOptions)
	doFullLogicalBackup($aTarget, $aOptions, &$aBackupInformation)
	checkIfBinaryLoggingIsEnabled($mysqli)
	checkForNonTransactionalTables($mysqli, $aOptions, $aSchemas)
	doMysqldump($aOptions, $cmd, $file, $aTarget, $mysqldump)
	doChecksumOfFile($aOptions, $file)
	compressFile($aOptions, $file)
	archiveFileToArchivedir($aOptions, $file)
	doFullPhysicalBackup($aTarget, $aOptions, &$aBackupInformation)
	doConfigBackup($aTarget, $aOptions)
	doCleanup($aTarget, $aOptions)
	doSchemaBackup($aTarget, $aOptions, &$aBackupInformation)
	doBinlogBackup($aTarget, $aOptions)
	getAllSchemas($mysqli)
	getDumpOverview($mysqli, $aSchemas)
	doPrivilegeBackup($aTarget, $aOptions)
	getBackupTypeId($pType)
	getBackupPolicyId($pPolicy)
	getBackupModeId($pMode)
	getFileTypeId($pExt)
	getAllGlobalVariables($mysqli)

*/

// ---------------------------------------------------------------------
function printUsage()
// ---------------------------------------------------------------------
{
  global $gLogFile, $gBackupDir;

  $script = 'mysql_bman';

  print "
usage: $script --target=connectstring --type=backup_type
               --policy=backup_policy [--retention=n] [--simulate]
               [--help] [--log=logfile] [--backupdir]
               [--mode={ logical | physical }] --per-schema
               [--catalog=connectstring [--create | --upgrade]
               --instace-name=name]

Options:
  target        Specifies the database (=target) to backup. A connectstring
                looks as follows: user/password@host:port
  type          See below.
  policy        See below.
  retention     Retention in days. All backups which are older than
                retention days will be deleted when backup type cleanup is
                executed.
  simulate      If simulate is called no operation is done but only
                simulated.
  help          Prints this help.
  log           Specifies the logfile where results are reported to (default:
                $gLogFile).
  backupdir     Directory where backups are written to (default: $gBackupDir).
  mode          See below. (default: logical)        
  archive       Archive (copy) files after backup to archivedir.
  cleanup       Cleanup files after archive.
  archivedir    Directory where backups are archived to (default: none).
  config        Configuration file to read parameter from.
  schema        Schemas to backup or not to backup. +a,+b means backup schema
                a and b. -a,-b means backup all schemas except a and b.
  per-schema    Backup is done per schema. Keep in mind: This breaks
                consistency among schemas.
  no-compress   Do not compress backup.
  instance-name Provide instance name to add to backup name.
  catalog       Specifies the database (=catalog) to store the backup meta
                data.
  catalog-name  Name of the catalog database (default bman_catalog).

Special options:
  no-memory-table-check  Do NOT check for non-transactional MEMORY tables.
                This is the default MySQL behaviour and bad. You can
                easily get inconsistencies in your backup.
  create        Create backup meta data catalog.
  upgrade       Upgrade backup meta data catalog.

Backup type:
  full          full logical backup (mysqldump) of all schemas
  binlog        binlog backup
  config        configuration file backup (my.cnf)
  structure     structure backup
  cleanup       cleanup of backup pieces older than n days
  schema        backup of one or more schemas
  privilege     privilege dump (SHOW GRANTS FOR)

Backup policy:
  daily         directory to store daily backups
  weekly                  \"        weekly backups
  monthly                 \"        monthly backups
  quarterly               \"        quarterly backups
  yearly                  \"        yearly backups
  binlog                  \"        binlog backups

Backup mode:
  logical       do a logical backup (mysqldump).
  physical      do a physical backup (mysqlbackup/innobackup/xtrabackup)

Config file example:
  policy                = daily
  target                = root/secret@127.0.0.1:3306 
  type                  = schema 
  schema                = -mysql 
  policy                = daily 
  archive               = on
  archivedir            = /mnt/tape
  per-schema            = on
  no-compress           = on
  no-memory-table-check = on


Examples:

  Full logical backup with mysqldump:
  $script --target=root:secret@127.0.0.1:3306 --type=full --policy=daily

  Full physical backup with xtrabackup:
  $script --target=root/secret@127.0.0.1 --type=full --mode=physical --policy=daily

  Binary log backup without a password:
  $script --target=bman@192.168.1.42:3307 --type=binlog --policy=daily

  Configuation backup:
  $script --target=root/secret@127.0.0.1:3306 --type=config --policy=weekly

  Structure backup:
  $script --target=root:secret@127.0.0.1:3306 --type=structure --policy=weekly
  
  Cleanup old backups:
  $script --target=root/secret@127.0.0.1:3306 --type=cleanup --policy=daily --retention=30

  Cleanup old backups from archive location:
  $script --target=root/secret@127.0.0.1:3306 --type=cleanup --policy=daily --retention=30 --archive

  Archive backup to other location:
  $script --target=root/secret@127.0.0.1:3306 --type=structure --policy=weekly --archive --archivedir=/mnt/tape --cleanup

  Selection of schema backup:
  $script --target=root/secret@127.0.0.1:3306 --type=schema --schema=-mysql --policy=daily --archive --archivedir=/mnt/tape

  Selection of schema backup with separated files per schema:
  $script --target=root:secret@127.0.0.1:3306 --type=schema --schema=+foodmart,+world --per-schema --policy=daily --no-compress

  Creation of a backup catalog:
  $script --catalog=root/secret@127.0.0.1:3306 --create
  
  Backups against catalog:
  $script --target=root/secret@127.0.0.1:3306 --catalog=root/secret@127.0.0.1:3306 --instance-name=test --type=full --policy=daily
  
  Privilege backup:
  $script --target=root/secret@127.0.0.1:3306 --type=privilege --policy=daily --mode=logical
  
  Privilege backup per schema:
  $script --target=root:secret@127.0.0.1:3306 --type=privilege --policy=daily --per-schema
";
}

// ---------------------------------------------------------------------
function readConfigurationFile($pFile)
// ---------------------------------------------------------------------
{

// print "Here we are\n";

  $rc = OK;
  $aConfig = array();

  $fh = fopen($pFile, 'r');
  if ( ! $fh ) {
    $rc = 137;
    tee("  ERROR: Cannot open file $pFile (rc=$rc).");
    return array($rc, $aConfig);
  }

  while ( ($buffer = fgets($fh, 4096)) != false ) {

		$buffer = trim($buffer);

    // Skip empty line
    $pattern = "/^\s*$/";
    if ( preg_match($pattern, $buffer) ) {
      continue;
    }

    // Skip comment line
    $pattern = "/^\s*#.*$/";
    if ( preg_match($pattern, $buffer) ) {
      continue;
    }

    // Line must have format: bla = bla
    $pattern = "/^([^\\s]+)\s*=\s*(.+)$/";
    if ( preg_match($pattern, $buffer, $matches) ) {
      $aConfig[trim($matches[1])] = trim($matches[2]);
    }
    else {
      $rc = 138;
      tee("  ERROR: Wrong line $buffer in configuration file (rc=$rc).");
      return array($rc, $aConfig);
    }
  }
  if ( ! feof($fh) ) {
    $rc = 139;
    tee("  ERROR: Unexpected fgets() fail (rc=$rc).");
    return array($rc, $aConfig);
  }
  fclose($fh);

  return array($rc, $aConfig);
}

// ---------------------------------------------------------------------
function getMasterPosFromDump($lDump)
// ---------------------------------------------------------------------
{
  $rc = OK;

  if( ( $fh = fopen($lDump, 'r') ) === FALSE ) {
    $rc = 150;
    tee("  ERROR: Failed to open file $lDump for reading (rc=$rc)!");
    return array($rc, array());
  }

  $binlog = '';
  $pos    = 0;

  $cnt = 1;
  while ( $line = fgets($fh) ) {
    if ( $cnt > 30 ) {
      break;
    }

    // -- CHANGE MASTER TO MASTER_LOG_FILE='bin_log.000057', MASTER_LOG_POS=106;

    if ( preg_match("/.*CHANGE MASTER TO MASTER_LOG_FILE='([\w-]*\.\d*)'.*MASTER_LOG_POS=(\d*)/", $line, $matches) ) {
      $binlog = $matches[1];
      $pos    = $matches[2];
      break;
    }
    $cnt++;
  }
  fclose($fh);

  return array($rc, array($binlog, $pos));
}

// ---------------------------------------------------------------------
function checkArguments($aOptions)
// ---------------------------------------------------------------------
{
  global $gLogFile, $gBackupDir;

  $rc = OK;
  $ret = OK;

//   var_dump($aOptions);

  if ( array_key_exists('help', $aOptions) ) {
    printUsage();
    // exit MUST be here
    exit($rc);
  }

	// Catalog operations
	if ( array_key_exists('catalog', $aOptions) ) {
		// Catalog must exist and nothing else!
		if ( array_key_exists('create', $aOptions) || array_key_exists('upgrade', $aOptions) ) {
			if ( array_key_exists('instance-name', $aOptions) ) {
				$ret = 181;
				tee("Creating or upgrading a catalog does not allow any other option than --catalog and --create (ret=$ret).");
				return $ret;
			}
			return $ret;
		}
		else {
			if ( ! array_key_exists('instance-name', $aOptions) ) {
				$ret = 180;
				tee("For using a catalog an --instance-name must be specified (ret=$ret).");
				return $ret;
			}
		}
		// not return here. Continue with normal checking...
	}

  if ( ! array_key_exists('type', $aOptions) ) {
    $ret = 171;
    tee("Backup type is missing (ret=$ret).");
    return $ret;
  }

  if ( ! array_key_exists('policy', $aOptions) ) {
    $ret = 174;
    tee("Backup policy is missing (ret=$ret).");
    return $ret;
  }

  // target is always required unless type is cleanup
  if ( (! array_key_exists('target', $aOptions))
    && ($aOptions['type'] != 'cleanup')
     ) {
    $ret = 168;
    tee("Target is missing (ret=$ret).");
    return $ret;
  }

  foreach ( $aOptions as $arg => $val ) {

		// Check if option was given twice
		if ( gettype($aOptions[$arg]) == 'array' ) {
			$ret = 157;
			tee("Option $arg was passed several times (ret=$ret).");
			return $ret;
		}

    if ( $arg == 'type' ) {

      if ( ( $val != 'full' )
        && ( $val != 'binlog' )
        && ( $val != 'config' )
        && ( $val != 'structure' )
        && ( $val != 'cleanup' )
        && ( $val != 'schema' )
        && ( $val != 'privilege' )
         ) {
        $ret = 174;
        tee("Wrong or missing backup type $val. Please choose one of full, binlog, config, structure, cleanup (ret=$ret).");
        return $ret;
      }
    }

    if ( $arg == 'policy' ) {

      if ( ( $val != 'daily' )
        && ( $val != 'weekly' )
        && ( $val != 'monthly' )
        && ( $val != 'quarterly' )
        && ( $val != 'yearly' )
        && ( $val != 'binlog' )
        ) {
        $ret = 167;
        tee("Wrong or missing backup policy $val. Please choose one of daily, weekly, monthly, quarterly, yearly, binlog (ret=$ret).");
        return $ret;
      }
    }

    if ( ($arg == 'type') && ($aOptions['type'] == 'cleanup') && (! isset($aOptions['retention'])) ) {
      $ret = 170;
      tee("Retention is needed if type is cleanup (ret=$ret).");
      return $ret;
    }

		if ( ($arg == 'type') && ($val == 'privilege') ) {
			if ( array_key_exists('mode', $aOptions) && ($aOptions['mode'] != 'logical') ) {
				$ret = 180;
				tee("Privilege backup is only allowed with --mode=logical (ret=$ret).");
				return $ret;
			}
		}

    if ( $arg == 'retention' ) {

      if ( ! is_numeric($val) ) {
        $ret = 169;
        tee("Value for retention must be numeric but is $val (ret=$ret).");
        return $ret;
      }
    }

    if ( $arg == 'log' ) {
      if ( strval($val) != '' ) {
        $gLogFile = strval($val);
      }
      else {
        tee("  ERROR: Logfile not specified correctly: $val");
        tee("         Using $gLogFile instead.");
      }
    }

    if ( $arg == 'backupdir' ) {
      if ( strval($val) != '' ) {
        $gBackupDir = strval($val);
      }
      else {
        tee("  ERROR: Backupdir not specified correctly: $val");
        tee("         Using $gBackupDir instead.");
      }
    }

    if ( $arg == 'archivedir' ) {
      if ( strval($val) != '' ) {
        $gArchiveDir = strval($val);
      }
      else {
        $ret = 177;
        tee("  ERROR: Archivedir not specified correctly: $val (ret=$ret).");
        return $ret;
      }
    }

    if ( $arg == 'mode' ) {

      if ( ( $val != 'logical' )
        && ( $val != 'physical' ) ) {
        $ret = 178;
        tee("Wrong or missing backup mode $val. Please choose one of logical, physical (ret=$ret).");
        return $ret;
      }
    }

    if ( $arg == 'per-schema' ) {
      if ( ($aOptions['type'] != 'schema') && ($aOptions['type'] != 'privilege') ) {
        $ret = 179;
        tee("Per-schema backup is only allowed with --backup-type=schema or privilege (ret=$ret).");
        return $ret;
      }
    }

		// With catalog (exception --create or --upgrade) --instance-name must be given
		if ( $arg == 'catalog' ) {
			if ( ! array_key_exists('instance-name', $aOptions) ) {
				$ret = 180;
				tee("When backup is performed against a catalog --instance-name must be provided (ret=$ret).");
				return $ret;
			}
		}

  }   // foreach
  return $ret;
}

// ---------------------------------------------------------------------
function closeConnection($mysqli)
// ---------------------------------------------------------------------
{
  $mysqli->close();
}

// ---------------------------------------------------------------------
function openConnection($aConnection)
// ---------------------------------------------------------------------
{
  $rc = OK;

  // todo: very bad idea to use schema mysql in this generic function. Shot in the foot with catalog!
  // should be replaced later...
  @$mysqli = new mysqli($aConnection['host'], $aConnection['user'], $aConnection['password'], null, $aConnection['port'], null);

  // This is the "official" OO way to do it,
  // BUT $connect_error was broken until PHP 5.2.9 and 5.3.0.
  
  // if ( $mysqli->connect_error ) {
  //   die('Connect Error (' . $mysqli->connect_errno . ') ' . $mysqli->connect_error);
  // }

  // Use this instead of $connect_error if you need to ensure
  // compatibility with PHP versions prior to 5.2.9 and 5.3.0.

	if ( mysqli_connect_error() ) {
		$rc = 175;
		$msg = mysqli_connect_error() . " (rc=$rc).";
		tee('  ERROR: ' . $msg);
		return array($rc, null);
	}

	$mysqli->query('SET NAMES utf8');
	$mysqli->query('SET SESSION SQL_MODE = "TRADITIONAL,ANSI,ONLY_FULL_GROUP_BY"');

	return array($rc, $mysqli);
}

// ----------------------------------------------------------------------
function getTargetConfiguration($aTarget)
// ----------------------------------------------------------------------
{
  $rc = OK;

  $aConf = array();

  list($rc, $mysqli) = openConnection($aTarget);
  if ( $rc != OK ) {
    $rc = 113;
    tee("  ERROR: Connection error (rc=$rc).");
    tee("  Please create a backup user as follows:");
    tee("  CREATE USER '" . $aTarget['user'] . "'@'" . $aTarget['host'] . "' IDENTIFIED BY '<your password>';");
    tee("  GRANT ALL ON *.* TO '" . $aTarget['user'] . "'@'" . $aTarget['host'] . "';");
    $aTarget['password'] = '******';
    // tee('  ' . print_r($aTarget, true));
    // debug_print_backtrace();
    return array($rc, $aConf);
  }

  // This is NOT 5.0 compatible syntax!
  // $sql = sprintf("SELECT @@datadir AS datadir, @@log_bin AS log_bin");

  $sql = sprintf("SHOW GLOBAL VARIABLES WHERE variable_name IN('log_bin', 'datadir', 'version')");

  if ( ! ($result = $mysqli->query($sql)) ) {
    $rc = 112;
    tee("Invalid query: $sql, " . $mysqli->error . " (rc=$rc).");
    return array($rc, $aConf);
  }

  while ( $record = $result->fetch_assoc() ) {

    if ( $record['Variable_name'] == 'datadir' ) {
      $aConf['datadir'] = rtrim($record['Value'], '/');
    }
    elseif ( $record['Variable_name'] == 'version' ) {
      $pattern='/(\d+)\.(\d+).(\d+)/';
      preg_match($pattern, $record['Value'], $matches);
      $aConf['mr_version'] = sprintf("%02d%02d%02d", $matches[1], $matches[2], $matches[3]);
    }
  }

  closeConnection($mysqli);

  return array($rc, $aConf);
}

// ----------------------------------------------------------------------
function parseConnectString($pConnectString)
// ----------------------------------------------------------------------
{
  // Connect string style:
  // user/password@host:port

  // http://en.wikipedia.org/wiki/URI_scheme
  // foo://username:password@example.com:8042

  // Use the last @ for splitting host and user
  // Use the first / or : for splitting user and password
  // Characters like : @ and / are allowed in password!
  // Allowed combinations for user are:
  // user/password
  // user:password
  // user
  // Allowed combinations for host are:
  // host:port
  // host

  $aTarget = array();
  $ret = OK;


  // Split first into user and host part: @ is mandatory

  $pos = strrpos($pConnectString, '@');

  if ( $pos == 0 ) {
    $ret = 173;
    // This is/was a security issue! Removed: $pConnectString
    tee("Connect string does not match pattern: user/password@host:port (ret=$ret).");
    return array($ret, $aTarget);
  }

  $user_part = substr($pConnectString, 0, $pos);
  $host_part = substr($pConnectString, $pos+1);


  // User part

  $u = preg_split("/[\/:]/", $user_part, 2);
  $aTarget['user']     = $u[0];
  $aTarget['password'] = isset($u[1]) ? $u[1] : '';

  // Host part

  $h = preg_split("/:/", $host_part);
  $aTarget['host'] = $h[0];
  $aTarget['port'] = 3306;
  
  if ( isset($h[1]) ) {
    if ( $h[1] == '' ) {
      $aTarget['port'] = 3306;
    }
    else {
      $aTarget['port'] = intval($h[1]);
    }

    if ( $aTarget['port'] == 0 ) {
      $ret = 172;
      tee("Port " . $h[1] . " is zero or not an integer (ret=$ret).");
      $aTarget['port'] = '';
      return array($ret, $aTarget);
    }
  }

  return array($ret, $aTarget);
}

// ----------------------------------------------------------------------
function checkBackupStructure($pBackupDir)
// ----------------------------------------------------------------------
{
  global $gaBackupPolicy;
  $rc = OK;

  if ( file_exists($pBackupDir) === false ) {
    tee("\nBackup directory $pBackupDir does NOT exist. Creating it...");
    if ( mkdir($pBackupDir, 0755, true) !==  true ) {
      $rc = 123;
      tee("  ERROR: Cannot create $pBackupDir (rc=$rc).");
      return $rc;
    }
  }

  foreach ( $gaBackupPolicy as $policy ) {

    if ( ! file_exists("$pBackupDir/$policy") ) {
      tee("Policy directory $pBackupDir/$policy does NOT exist. Creating it...");
      if ( @mkdir("$pBackupDir/$policy", 0755, true) === false ) {
        $rc = 124;
        tee("  ERROR cannot create directory $pBackupDir/$policy (rc=$rc).");
        return $rc;
      }
    }
  }

  return $rc;
}

// ----------------------------------------------------------------------
function checkArchiveStructure($pArchiveDir)
// ----------------------------------------------------------------------
{
  global $gaBackupPolicy;
  $rc = OK;

  if ( file_exists($pArchiveDir) === false ) {
    tee("\nArchive directory $pArchiveDir does NOT exist. Creating it...");
    if ( @mkdir($pArchiveDir, 0755, true) !==  true ) {
      $rc = 122;
      tee("  ERROR: Cannot create $pArchiveDir (rc=$rc).");
      return $rc;
    }
  }

  foreach ( $gaBackupPolicy as $policy ) {

    if ( ! file_exists("$pArchiveDir/$policy") ) {
      tee("Policy directory $pArchiveDir/$policy does NOT exist. Creating it...");
      if ( mkdir("$pArchiveDir/$policy", 0755, true) === false ) {
        $rc = 146;
        tee("  ERROR cannot create directory $pArchiveDir/$policy (rc=$rc)");
        return $rc;
      }
    }
  }

  return $rc;
}

// ----------------------------------------------------------------------
function tee($s)
// ----------------------------------------------------------------------
{
  global $gLogFile, $gTesting;

  $rc = OK;

  print "$s\n";

  // todo: not really the optimal way to do it
  if ( ! ($fh = @fopen($gLogFile, 'a')) ) {
    $rc = 118;
    $err = error_get_last(); 
    print "  ERROR: Can't open file $gLogFile (rc=$rc).\n";
    print "         " . $err['message'] . "\n";
    if ( $gTesting === false ) {
			// exit MUST be here
			exit($rc);
    }
  }
  else {
    fwrite($fh, "$s\n");
    fclose($fh);
  }
  return $rc;
}

// ----------------------------------------------------------------------
function archiveFileToArchivedir($aOptions, $file)
// file is the full path
// ----------------------------------------------------------------------
{
	$rc = OK;

	tee("  Archiving $file to " . $aOptions['archivedir'] . '/' . $aOptions['policy'] . '/.');

	$cmd = "cp $file " . $aOptions['archivedir'] . '/' . $aOptions['policy'] . '/';
	if ( ! array_key_exists('simulate', $aOptions) ) {

		system($cmd, $ret);
		if ( $ret != OK ) {
			$rc = 143;
			tee("  ERROR: archiving failed (rc=$rc).");
			return $rc;
		}

	}
	else {
		tee("  $cmd");
	}

	return $rc;
}

// ----------------------------------------------------------------------
function cleanupFileAfterArchiveToArchivedir($aOptions, $file)
// file is the full path
// ----------------------------------------------------------------------
{
	$rc = OK;

	// this is redundant with outer call. Revmove after check!
	if ( array_key_exists('cleanup', $aOptions) ) {

		tee("  Cleanup $file after archiving.");

		$cmd = "rm -f $file";
		if ( ! array_key_exists('simulate', $aOptions) ) {

			system($cmd, $ret);
			if ( $ret != OK ) {
				$rc = 106;
				tee("  ERROR: cleanup after archiving failed (rc=$rc).");
				return $rc;
			}
		}
		else {
			tee("  $cmd");
		}
	}
	else {
		tee("  No cleanup after archiving.");
	}

	return $rc;
}

// ----------------------------------------------------------------------
function doStructureBackup($aTarget, $aOptions, &$aBackupInformation)
// ----------------------------------------------------------------------
{
  global $gMysqlDump, $gBackupDir, $gBackupTimestamp;

  $rc = OK;

  $name = '';
  if ( isset($aOptions['instance-name']) && ($aOptions['instance-name'] != '') ) {
    $name = $aOptions['instance-name'] . '_';
  }

  $cmd = "$gMysqlDump --user=". $aTarget['user'] . " --host=". $aTarget['host'] . " --port=" . $aTarget['port'] . " --all-databases --no-data";
  $ext = 'sql';
  $file = 'bck_' . $name . $aOptions['type'] . '_' . $gBackupTimestamp . '.' . $ext;
  $dst = $gBackupDir . '/' . $aOptions['policy'] . '/' . $file;
  tee("  $cmd");
  tee("  to Destination: $dst");

  if ( ! array_key_exists('simulate', $aOptions) ) {
    $cmd = "$cmd --password='" . $aTarget['password'] . "' >$dst";
    // todo: possibly better use doMysqldump?
    passthru($cmd, $ret);
    if ( $ret != OK ) {
      $rc = 125;
      tee("  ERROR: Command failed (rc=$rc).");
      return $rc;
    }


		// Log file information into catalog

		if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
			$aFile = array (
				'id'                => 0
			, 'schema_name'       => ''
			, 'original_name'     => ''
			, 'original_location' => ''
			, 'original_size'     => filesize($dst)
			, 'file_type_id'      => getFileTypeId($ext)
			, 'mtime'             => filemtime($dst)
			, 'backup_name'       => $file
			, 'backup_location'   => $gBackupDir . '/' . $aOptions['policy']
			, 'md5'               => ''
			, 'compressed'        => 0
			, 'compressed_name'   => ''
			, 'compressed_size'   => 0
			, 'archived'          => 0
			, 'archive_location'  => ''
			, 'cleanedup'         => 0
			, 'backup_id'         => $aBackupInformation['id']
			);
			list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
		}

    
    tee("  Do MD5 checksum of $dst");
    list($ret, $md5) = getMD5sum($dst, $aOptions);
    if ( $ret == OK ) {
      tee("  md5sum = $md5");

			// Log file information into catalog
			if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
				$aFile['md5'] = $md5;
				list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
			}
    }
    else {
      $rc = 151;
      tee("  ERROR: Error building MD5 sum of file $dst (rc=$rc).");
      return $rc;
    }
    
    // compression

		if ( ! isset($aOptions['no-compress']) ) {
			// Compress file
			$ret = compressFile($aOptions, $dst);
			if ( $ret != OK ) {
				return $ret;
			}
			$dst .= '.gz';

			// Log file information into catalog
			if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
				$aFile['compressed_name'] = $file .= '.gz';
				$aFile['compressed_size'] = filesize($dst);
				$aFile['compressed']      = 1;
				list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
			}
		}


		// Archive file

		if ( array_key_exists('archive', $aOptions) ) {
			$rc = archiveFileToArchivedir($aOptions, $dst);

			// Log file information into catalog
			if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
				$aFile['archived']         = 1;
				$aFile['archive_location'] = $aOptions['archivedir'] . '/' . $aOptions['policy'];
				list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
			}
		}


		// Cleanup file after archive

		if ( array_key_exists('cleanup', $aOptions) ) {

			if (! array_key_exists('simulate', $aOptions)) {
				$rc = cleanupFileAfterArchiveToArchivedir($aOptions, $dst);
			}

			// Log file information into catalog
			if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
				$aFile['cleanedup'] = 1;
				list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
			}
		}
    
  }   // if simulate, todo: a bit to general! should be per block above?
  return $rc;
}

// ----------------------------------------------------------------------
function getMD5sum($file, $aOptions)
// ----------------------------------------------------------------------
{
  $rc = OK;
  $ret = OK;
  $md5sum = '';
  $cmd = "md5sum --binary $file";
  tee("  $cmd");

  if ( ! array_key_exists('simulate', $aOptions) ) {
    exec("$cmd 2>&1", $output, $ret);
    $s = implode("\n", $output);

    if ( $ret != OK ) {
      $rc = 130;
      tee("  ERROR: Command failed (rc=$rc).");
      return array($rc, $md5sum);
    }
    if ( preg_match("/(\w{32})\s.*$/", $s, $matches) ) {
      $md5sum = $matches[1];
    }
    else {
      $rc = 131;
      tee("  ERROR: error building md5sum (rc=$rc).");
      return array($rc, $md5sum);
    }
  }
  return array($ret, $md5sum);
}

// ----------------------------------------------------------------------
function doFullLogicalBackup($aTarget, $aOptions, &$aBackupInformation)
// ----------------------------------------------------------------------
{
	global $gMysqlDump, $gBackupDir, $gBackupTimestamp, $gTargetConfiguration;

	$rc = OK;

	// This backup would be useless if we have non transactional tables
	// So check for those and if exists, abort

	list($rc, $mysqli) = openConnection($aTarget);
	if ( $rc != OK ) {
		$rc = 107;
		tee("  ERROR: Connection error (rc=$rc).");
		return $rc;
	}

	// Check for non transactional tables
	$ret = checkForNonTransactionalTables($mysqli, $aOptions, array());
	if ( $ret != OK ) {
		return $ret;
	}
  // We have not implemented (yet) backup for MyISAM
	$lConsistency = '--single-transaction';

	// Check if binary logging is enabled
	list($rc, $lBinaryLogging) = checkIfBinaryLoggingIsEnabled($mysqli);
	if ( $rc != OK ) {
		return $rc;
	}
	if ( $lBinaryLogging == '' ) {
		tee("  Binary logging is disabled");
	}

	// Get all schemas for display output only
	list ($ret, $aSchemas) = getAllSchemas($mysqli);
	if ( $ret != OK ) {
		return $ret;
	}

	tee("  Schema to backup: " . implode($aSchemas, ', '));

	list($ret, $output) = getDumpOverview($mysqli, $aSchemas);
	tee($output);

	closeConnection($mysqli);


	// do mysqldump command

	$cmd = $gMysqlDump . " --user=" . $aTarget['user'] . " --host=". $aTarget['host'] . " --port=" . $aTarget['port'] . " --all-databases $lBinaryLogging --quick $lConsistency --flush-logs --triggers --routines --hex-blob";
	// MySQL >= 5.1
	if ( $gTargetConfiguration['mr_version'] >= '050100' ) {
		$cmd .= ' --events';
	}

	$name = '';
	if ( isset($aOptions['instance-name']) && ($aOptions['instance-name'] != '') ) {
		$name = $aOptions['instance-name'] . '_';
	}

	$ext = 'sql';
	$file = 'bck_' . $name . $aOptions['type'] . '_' . $gBackupTimestamp . '.' . $ext;
	$dst = $gBackupDir . '/' . $aOptions['policy'] . '/' . $file;
	// todo: parameter list has to be cleaned-up!
	$ret = doMysqldump($aOptions, $cmd, $dst, $aTarget, $gMysqlDump);
	if ( $ret != OK ) {
		return $ret;
	}

	// Log file information into catalog
	if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
		$aFile = array (
		  'id'               => 0
		, 'schema_name'      => ''
		, 'original_name'    => ''
		, 'original_location' => ''
		, 'original_size'    => filesize($dst)
		, 'file_type_id'     => getFileTypeId($ext)
		, 'mtime'            => filemtime($dst)
		, 'backup_name'      => $file
		, 'backup_location'  => $gBackupDir . '/' . $aOptions['policy']
		, 'md5'              => ''
		, 'compressed'       => 0
		, 'compressed_name'  => ''
		, 'compressed_size'  => 0
		, 'archived'         => 0
		, 'archive_location' => ''
		, 'cleanedup'        => 0
		, 'backup_id'        => $aBackupInformation['id']
		);
		list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
	}

	// Determine binary log file and binary log pos
	if ( ! array_key_exists('simulate', $aOptions) ) {
		// get binary log file/pos
		list($rc, $aBinlogPos) = getMasterPosFromDump($dst);
		if ( $rc != OK ) {
			return $rc;
		}
		$binlog = $aBinlogPos[0];
		$pos    = $aBinlogPos[1];
		if ( $binlog == '' ) {
			tee("  Backup does NOT contain any binary log information.");
		}
    else {
      tee("  Binlog file is " . $binlog . " and position is " . $pos);
      $aBackupInformation['binlog_file'] = $binlog;
      $aBackupInformation['binlog_pos'] = $pos;
    }
  }

	// Do checksum of file
	list($ret, $md5) = doChecksumOfFile($aOptions, $dst);
	if ( $ret != OK ) {
		return $ret;
	}

	// Log file information into catalog
	if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
		$aFile['md5'] = $md5;
		list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
	}

	if ( ! isset($aOptions['no-compress']) ) {
		// Compress file
		$ret = compressFile($aOptions, $dst);
		if ( $ret != OK ) {
			return $ret;
		}
		$dst .= '.gz';

		// Log file information into catalog
		if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
			$aFile['compressed_name'] = $file .= '.gz';
			$aFile['compressed_size'] = filesize($dst);
			$aFile['compressed']      = 1;
			list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
		}
	}

	// Archive file
	if ( array_key_exists('archive', $aOptions) ) {
		$rc = archiveFileToArchivedir($aOptions, $dst);

		// Log file information into catalog
		if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
			$aFile['archived']         = 1;
			$aFile['archive_location'] = $aOptions['archivedir'] . '/' . $aOptions['policy'];
			list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
		}
	}

	// Cleanup file after archive
	if ( array_key_exists('cleanup', $aOptions) ) {
		if (! array_key_exists('simulate', $aOptions)) {
			$rc = cleanupFileAfterArchiveToArchivedir($aOptions, $dst);
		}

		// Log file information into catalog
		if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
			$aFile['cleanedup'] = 1;
			list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
		}
	}

	return $rc;
}

// ----------------------------------------------------------------------
function checkIfBinaryLoggingIsEnabled($mysqli)
// ----------------------------------------------------------------------
{
	$rc = OK;

	$lBinaryLogging = '--master-data=2';

	$sql = 'SHOW MASTER STATUS';
	if ( ! ($result = $mysqli->query($sql)) ) {
		$rc = 155;
		tee("Invalid query: $sql, " . $mysqli->error . " (rc=$rc).");
		return array($rc, '');
	}

	$record = $result->fetch_assoc();
	if ( $record == NULL ) {
		$lBinaryLogging = '';
	}

	return array($rc, $lBinaryLogging);
}

// ----------------------------------------------------------------------
function checkForNonTransactionalTables($mysqli, $aOptions, $aSchemas)
// ----------------------------------------------------------------------
{
	$rc = OK;

  $sql = sprintf("SELECT table_schema, engine, COUNT(*) AS count
  FROM information_schema.tables
 WHERE table_type = 'BASE TABLE'
   AND table_schema NOT IN ('mysql', 'information_schema', 'PERFORMANCE_SCHEMA')
   AND engine NOT IN ('InnoDB', 'PBXT')");

	// Ignore MEMORY table in check
	if ( isset($aOptions['no-memory-table-check']) ) {
		tee("  WARNING: Ignore MEMORY tables possibly causing inconsistencies.");
		$sql .= "\n   AND engine != 'MEMORY'";
	}

  if ( count($aSchemas) > 0 ) {
    $sql .= "\n   AND table_schema IN ('" . implode($aSchemas, "', '") . "')";
  }
  $sql .= "\n GROUP BY table_schema, engine";

  if ( ! ($result = $mysqli->query($sql)) ) {
    $rc = 149;
    tee("Invalid query: $sql, " . $mysqli->error . " (rc=$rc).");
    return $rc;
  }

  while ( $record = $result->fetch_assoc() ) {
    $rc = 128;
    tee("  ERROR: We have non-transactional tables. So this backup would be worthless.");
    tee("         In schema " . $record['table_schema'] . ' ' .  $record['count'] . ' table(s) of type ' . $record['engine']);
    tee("         Aborting (rc=$rc).");
    // No return here, show all schemas with troubles...
  }

  return $rc;
}

// ----------------------------------------------------------------------
function doMysqldump($aOptions, $cmd, $file, $aTarget, $mysqldump)
// ----------------------------------------------------------------------
{
	$rc = OK;

  tee("  $cmd");
  tee("  to Destination: $file");
  $cmd = "$cmd --password='" . $aTarget['password'] . "' >$file";
  if ( ! array_key_exists('simulate', $aOptions) ) {
    $stdout = exec($cmd, $output, $ret);
    if ( $ret != OK ) {
      $rc = 121;
      tee("  ERROR: $mysqldump command failed: $stdout (rc=$rc).");
      tee(print_r($output, true));
      return $rc;
    }
  }
  return $rc;
}

// ----------------------------------------------------------------------
function doChecksumOfFile($aOptions, $file)
// ----------------------------------------------------------------------
{
	$rc = OK;

  tee("  Do MD5 checksum of uncompressed file $file");
  if ( ! array_key_exists('simulate', $aOptions) ) {
    list($ret, $md5) = getMD5sum($file, $aOptions);
    if ( $ret == OK ) {
      tee("  md5sum = $md5");
    }
    else {
      $rc = 135;
      tee("  ERROR: Error building MD5 sum of file $dst (rc=$rc).");
      return array($rc, '');
    }
  }
  else {
    tee("  md5sum = simulation");
  }
  return array($rc, $md5);
}

// ----------------------------------------------------------------------
function compressFile($aOptions, $file)
// ----------------------------------------------------------------------
{
	$rc = OK;

	// todo: pbzip2 and pigz
	// todo: compression rate -9
	$rate = 6;
	$cmd = "gzip -$rate $file";
	tee("  $cmd");
	if ( ! array_key_exists('simulate', $aOptions) ) {
		passthru($cmd, $ret);
		if ( $ret != OK ) {
			$rc = 142;
			tee("  ERROR: gzip command failed (ret=$ret, rc=$rc).");
			return $rc;
		}
	}

	return $rc;
}

// ----------------------------------------------------------------------
function doFullPhysicalBackup($aTarget, $aOptions, &$aBackupInformation)
// ----------------------------------------------------------------------
{
	global $gMysqlDump, $gBackupDir, $gBackupTimestamp, $gTargetConfiguration;

	$rc = OK;

	// Check if Percona Xtrabackup or old InnoBackup is available

	$cmd = 'which innobackupex';
	$stdout = exec("$cmd 2>&1", $output, $ret);
	if ( $ret != OK ) {
		$rc = 223;
		tee("  WARNING: The utility innobackupex (xtrabackup) is not installed. Please install it first (rc=$rc).");
		// No return here yet!
	}
	else {
		$lInnoBackupEx = trim($stdout);
	}

	if ( $rc != OK ) {
		$cmd = 'which mysqlbackup';
		$stdout = exec("$cmd 2>&1", $output, $ret);
		if ( $ret != OK ) {
			$rc = 206;
			tee("  WARNING: The utility mysqlbackup is not installed either. Please install it first (rc=$rc).");
			return $rc;
		}
		else {
			$lInnoBackupEx = trim($stdout);
		}
	}


	list($rc, $mysqli) = openConnection($aTarget);
	if ( $rc != OK ) {
		$rc = 148;
		tee("  ERROR: Connection error (rc=$rc).");
		return $rc;
	}


	// Check if binary logging is enabled

	list($rc, $lBinaryLogging) = checkIfBinaryLoggingIsEnabled($mysqli);
	if ( $rc != OK ) {
		return $rc;
	}
	if ( $lBinaryLogging == '' ) {
		tee("  Binary logging is disabled");
	}


	// Get all schemas for display output only

	list ($ret, $aSchemas) = getAllSchemas($mysqli);
	if ( $ret != OK ) {
		return $ret;
	}

	tee("  Schema to backup: " . implode($aSchemas, ', '));

	list($ret, $output) = getDumpOverview($mysqli, $aSchemas);
	tee($output);


	// Get information to create backup configuration file

	$sql = sprintf("SELECT CONCAT(LOWER(variable_name), ' = ', IF(variable_value = ''
		, @@global.datadir, IF(variable_value = './', @@global.datadir
		, variable_value))) AS line
	FROM information_schema.global_variables
WHERE variable_name IN ('datadir', 'innodb_data_home_dir'
		, 'innodb_data_file_path'
		, 'innodb_log_group_home_dir', 'innodb_log_files_in_group'
		, 'innodb_log_file_size')
ORDER BY variable_name");

	if ( ! ($result = $mysqli->query($sql)) ) {
		$rc = 108;
		tee("Invalid query: $sql, " . $mysqli->error . " (rc=$rc).");
		return $rc;
	}
	
	list($ret, $aGlobalVariables) = getAllGlobalVariables($mysqli);
	
	// to early?
	closeConnection($mysqli);


	// Create backup.conf

	$name = '';
	if ( isset($aOptions['instance-name']) && ($aOptions['instance-name'] != '') ) {
		$name = $aOptions['instance-name'] . '_';
	}

	// We cannot use $dst path because ibbackup would otherwise complain
	// that directory already exists.
	$lBackupConf = '/tmp/' . 'bck_' . $name . $aOptions['type'] . '_' . $gBackupTimestamp . '.cnf';

	if ( ! ($fh = fopen($lBackupConf, 'w')) ) {
		$rc = 144;
		tee("  ERROR: Cannot open file $lBackupConf (rc=$rc).");
		return $rc;
	}

	fwrite($fh, "#\n");
	fwrite($fh, "# $lBackupConf\n");
	fwrite($fh, "#\n");
	fwrite($fh, "[mysqld]\n");
	while ( $record = $result->fetch_assoc() ) {
		fwrite($fh, $record['line'] . "\n");
	}
	fclose($fh);


	// Do physical backup with innobackupex

	$dir = 'bck_' . $name . $aOptions['type'] . '_' . $gBackupTimestamp;
	$dst = $gBackupDir . '/' . $aOptions['policy'] . '/' . $dir;
	// todo: password is bad, redo better later...
	$cmd = $lInnoBackupEx . ' --defaults-file=' . $lBackupConf . ' --no-timestamp' . ' --user=' . $aTarget['user'] . ' --host=' . $aTarget['host'] . ' --port=' . $aTarget['port'] . ' ' . $dst;
	tee("  $cmd");
	$cmd .= ' --password=' . $aTarget['password'];

	if ( ! array_key_exists('simulate', $aOptions) ) {

		$stdout = exec("$cmd 2>&1", $output, $ret);
		// stdout shoud contain: 140602 17:53:00  innobackupex: completed OK!
		// output contains the whole log
		foreach ( $output as $line ) {
			tee($line);
		}
		tee('');
		if ( $ret != OK ) {
			$rc = 205;
			tee("  ERROR: Backup failed. See logfile " . $aOptions['log'] . " (rc=$rc).");
			return $rc;
		}
		if ( ! preg_match('/innobackupex: completed OK!/', $stdout) ) {
			$rc = 207;
			tee("  ERROR: Backup failed. See logfile (rc=$rc).");
			return $rc;
		}
	}
	
	// Do apply log phase with innobackupex

	$dir = 'bck_' . $name . $aOptions['type'] . '_' . $gBackupTimestamp;
	$dst = $gBackupDir . '/' . $aOptions['policy'] . '/' . $dir;
	// todo: password is bad, redo better later...
	$cmd = $lInnoBackupEx . ' --defaults-file=' . $lBackupConf . ' --no-timestamp --use-memory=256M --apply-log' . ' ' . $dst;
	tee("  $cmd");

	if ( ! array_key_exists('simulate', $aOptions) ) {
		$stdout = exec("$cmd 2>&1", $output, $ret);
		// stdout shoud contain: 140602 17:53:00  innobackupex: completed OK!
		// output contains the whole log
		foreach ( $output as $line ) {
			tee($line);
		}
		tee('');
		if ( $ret != OK ) {
			$rc = 208;
			tee("  ERROR: Apply log failed. See logfile (rc=$rc).");
			return $rc;
		}
		if ( ! preg_match('/innobackupex: completed OK!/', $stdout) ) {
			$rc = 209;
			tee("  ERROR: Apply log failed. See logfile (rc=$rc).");
			return $rc;
		}
	}


	// Safe backup configuation

	if ( ! array_key_exists('simulate', $aOptions) ) {
		rename($lBackupConf, $dst . '/' . 'bck_' . $name . $aOptions['type'] . '_' . $gBackupTimestamp . '.cnf');
	}
	tee("  Move $lBackupConf to $dst/");

	if ( ! array_key_exists('simulate', $aOptions) ) {

		if ( $lBinaryLogging != '' ) {
			// Get binlog file and pos
			// File 1: xtrabackup_binlog_info
			$line = trim(file_get_contents($dst . '/xtrabackup_binlog_info'));
			if ( preg_match('/(.+)\s+(\d+)/', $line, $matches) ) {
				$aBinlog1 = array($matches[1], $matches[2]);
			}
			else {
				$rc = 210;
				$msg = "No binlog file and position (rc=$rc)."; 
				tee("  ERROR: $msg");
				return $rc;
			}

			// File 2: xtrabackup_binlog_pos_innodb
			$line = trim(file_get_contents($dst . '/xtrabackup_binlog_pos_innodb'));
			if ( preg_match('/(.+)\s+(\d+)/', $line, $matches) ) {
				$aBinlog2 = array($matches[1], $matches[2]);
			}
			else {
				$rc = 211;
				$msg = "No binlog file and position (rc=$rc)."; 
				tee("  ERROR: $msg");
				return $rc;
			}

			if ( $aBinlog1 != $aBinlog2 ) {
				$rc = 212;
				$msg = "Two different binary log positions (rc=$rc)."; 
				tee("  INFO: $msg");
				tee("  " . $aBinlog1[0] . '/' . $aBinlog1[1]);
				tee("  " . $aBinlog2[0] . '/' . $aBinlog2[1]);
				// No return here. This is NO error!
			}

			
			// https://www.percona.com/doc/percona-xtrabackup/2.1/xtrabackup_bin/working_with_binary_logs.html
			// If other storage engines are used (i.e. MyISAM), you should use
			// the xtrabackup_binlog_info file to retrieve the position.

			tee("  Binlog file is " . $aBinlog1[0] . " and position is " . $aBinlog1[1]);
			$aBackupInformation['binlog_file'] = $aBinlog1[0];
			$aBackupInformation['binlog_pos']  = $aBinlog1[1];
		}
	}

	// Do over all files
	foreach ( array_merge(glob("$dst/*"), glob("$dst/*/*")) as $file ) {

		if ( ! is_file($file) ) {
			continue;
		}


		$relname           = substr($file, strlen($gBackupDir . '/' . $aOptions['policy'] . '/' . 'bck_' . $name . $aOptions['type'] . '_' . $gBackupTimestamp . '/'));
		$lSchemaName       = rtrim(dirname($relname), '.');
		$lOriginalName     = basename($file);
		$lOriginalLocation = rtrim(rtrim($aGlobalVariables['datadir'], '/') . '/' . $lSchemaName, '/');
		$lBackupName       = basename($file);
		$lBackupLocation   = dirname($file);

		$ext = '';
		// todo: This is not perfect yet in case of 'bla.bla.txt'
		if ( preg_match('/\.(.{1,5})?$/', $file, $matches) ) {
			$ext = $matches[1];
		}

		if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {

			$aFile = array (
				'id'                => 0
			, 'schema_name'       => $lSchemaName
			, 'original_name'     => $lOriginalName
			, 'original_location' => $lOriginalLocation
			, 'original_size'     => filesize($file)
			, 'file_type_id'      => getFileTypeId($ext)
			, 'mtime'             => filemtime($file)
			, 'backup_name'       => $lBackupName
			, 'backup_location'   => $lBackupLocation
			, 'md5'               => ''
			, 'compressed'        => 0
			, 'compressed_name'   => ''
			, 'compressed_size'   => 0
			, 'archived'          => 0
			, 'archive_location'  => ''
			, 'cleanedup'         => 0
			, 'backup_id'         => $aBackupInformation['id']
			);
			list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
		}

		tee("  Do MD5 checksum of uncompressed file $file");
		list($ret, $md5) = getMD5sum($file, $aOptions);
		if ( $ret == OK ) {
			tee("  md5sum = $md5");

			// Log file information into catalog
			if ( array_key_exists('catalog', $aOptions) && ( ! array_key_exists('simulate', $aOptions) ) ) {
				$aFile['md5'] = $md5;
				list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
			}
		}
		else {
			$rc = 133;
			tee("  ERROR: Error building MD5 sum of file $dst (rc=$rc).");
			return $rc;
		}

		if ( ! isset($aOptions['no-compress']) ) {

			// Compress file

			tee("  Compress file $file");
			$ret = compressFile($aOptions, $file);
			if ( $ret != OK ) {
				return $ret;
			}
			$file .= '.gz';
			$lCompressedName = $lOriginalName . '.gz';

			// Log file information into catalog
			if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
				$aFile['compressed_name'] = $lCompressedName;
				$aFile['compressed_size'] = filesize($file);
				$aFile['compressed']      = 1;
				list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
			}
		}
		
		// Archive file to archivedir

		if ( array_key_exists('archive', $aOptions) ) {

			$lArchiveLocation = rtrim($aOptions['archivedir'] . '/' . $aOptions['policy'] . '/' . 'bck_' . $name . $aOptions['type'] . '_' . $gBackupTimestamp . '/' . dirname($relname), './');

			tee("  Archiving $file to $lArchiveLocation");
			$cmd = "cp $file $lArchiveLocation";
			tee("  $cmd");
			if ( ! array_key_exists('simulate', $aOptions) ) {
				if ( ! is_dir($lArchiveLocation) ) {
					@mkdir($lArchiveLocation, 0777, true);
				}
				passthru($cmd, $ret);
				if ( $ret != OK ) {
					$rc = 153;
					tee("  ERROR: archiving failed (rc=$rc).");
					return $rc;
				}

				// Log file information into catalog
				if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
					$aFile['archived']         = 1;
					$aFile['archive_location'] = $lArchiveLocation;
					list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
				}
			}
		}
		
		// Cleanup file after archive

		if ( array_key_exists('cleanup', $aOptions) ) {

			if ( ! array_key_exists('simulate', $aOptions)) {

				$rc = cleanupFileAfterArchiveToArchivedir($aOptions, $file);

				// Log file information into catalog
				if ( array_key_exists('catalog', $aOptions) ) {
					$aFile['cleanedup'] = 1;
					list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
				}
			}
			else {
				tee(  "Simulate cleanup of file $file");
			}
		}
	}

	return $rc;
}

// ----------------------------------------------------------------------
function doConfigBackup($aTarget, $aOptions, &$aBackupInformation)
// ----------------------------------------------------------------------
{
  $rc = OK;

  global $gMysqlDump, $gBackupDir, $gBackupTimestamp, $gTargetConfiguration;

  // Guess my.cnf
  // todo: this is NOT a good aproach!!!

  $src = '';

  $cnf = '/etc/mysql/my.cnf';
  if ( is_readable($cnf) ) {
    $src = $cnf;
  }

  $src = '/etc/my.cnf';
  if ( is_readable($cnf) ) {
    $src = $cnf;
  }

  $src = $gTargetConfiguration['datadir'] . '/my.cnf';
  if ( is_readable($cnf) ) {
    $src = $cnf;
  }

  if ( $src == '' ) {
    $rc = 100;
    tee("  ERROR: No my.cnf found (rc=$rc).");
    return $rc;
  }

  $name = '';
  if ( isset($aOptions['instance-name']) && ($aOptions['instance-name'] != '') ) {
    $name = $aOptions['instance-name'] . '_';
  }

	$ext = 'cnf';
	$file = 'bck_' . $name . $aOptions['type'] . '_' . $gBackupTimestamp . '.' . $ext;
  $dst = $gBackupDir . '/' . $aOptions['policy'] . '/' . $file;
  tee("  Copy $src to $dst");
  if ( ! array_key_exists('simulate', $aOptions) ) {

    if ( copy($src, $dst) !== true ) {
      $rc = 126;
      tee("  ERROR: Cannot copy $src (rc=$rc).");
      return $rc;
    }

		// Log file information into catalog
		if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
			$aFile = array (
				'id'                => 0
			, 'schema_name'       => ''
			, 'original_name'     => basename($src)
			, 'original_location' => dirname($src)
			, 'original_size'     => filesize($src)
			, 'file_type_id'      => getFileTypeId('cnf')
			, 'mtime'             => filemtime($src)
			, 'backup_name'       => $file
			, 'backup_location'   => $gBackupDir . '/' . $aOptions['policy']
			, 'md5'               => ''
			, 'compressed'        => 0
			, 'compressed_name'   => ''
			, 'compressed_size'   => 0
			, 'archived'          => 0
			, 'archive_location'  => ''
			, 'cleanedup'         => 0
			, 'backup_id'         => $aBackupInformation['id']
			);
			list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
		}
    
    tee("  Do MD5 checksum of $dst");
    list($ret, $md5) = getMD5sum($dst, $aOptions);
    if ( $ret == OK ) {
      tee("  md5sum = $md5");
      
			// Log file information into catalog
			if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
				$aFile['md5'] = $md5;
				list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
			}
    }
    else {
      $rc = 152;
      tee("  ERROR: Error building MD5 sum of file $dst (rc=$rc).");
      return $rc;
    }
  }


	// We do NOT compress the .cnf file (not worth)

	$aBackupInformation['compressed'] = 0;
	list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);


	// Archive file to archivedir

	if ( array_key_exists('archive', $aOptions) ) {
		tee("  Archiving $dst to " . $aOptions['archivedir'] . '.');
		$cmd = "cp $dst " . $aOptions['archivedir'] . '/' . $aOptions['policy'] . '/';
		tee("  $cmd");
		if ( ! array_key_exists('simulate', $aOptions) ) {
			passthru($cmd, $ret);
			if ( $ret != OK ) {
				$rc = 153;
				tee("  ERROR: archiving failed (rc=$rc).");
				return $rc;
			}

			// Log file information into catalog
			if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
				$aFile['archived']         = 1;
				$aFile['archive_location'] = $aOptions['archivedir'] . '/' . $aOptions['policy'];
				list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
			}
		}
	}


	// Cleanup file after archive

	if ( array_key_exists('cleanup', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {

		$rc = cleanupFileAfterArchiveToArchivedir($aOptions, $dst);

		// Log file information into catalog
		if ( array_key_exists('catalog', $aOptions) ) {
			$aFile['cleanedup'] = 1;
			list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
		}
	}
	else {
		tee(  "Simulate cleanup of file $dst");
	}

	return $rc;
}

// ----------------------------------------------------------------------
function doCleanup($aOptions, &$aBackupInformation)
// ----------------------------------------------------------------------
{
  global $gMysqlDump, $gBackupDir, $gBackupTimestamp;

  $rc = OK;

  if ( $aOptions['retention'] == 0 ) {
    $rc = 103;
    tee("  Missing backup retention policy. Should be a number of days (rc=$rc).");
    tee("  More help can be found with mysql_bman --help");
    return $rc;
  }

  if ( ( $aOptions['retention'] < 1 )
    || ( $aOptions['retention'] > 800 ) ) {
    tee("  Your retention policy is OUT of range (1..800).");
    tee("  Retention policy is NOT numeric. Please choose a number.");
    $ret = 176;
    return $rc;
  }
  
	// Compression does not make sense for cleanup
	if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
		$aBackupInformation['compressed']     = 0;
		$aBackupInformation['backup_mode_id'] = 2;
		list($ret, $cnt) = setBackupInCatalog($aBackupInformation['mysqli'], $aBackupInformation);
	}

  $deletionDate = date("Y-m-d_H-i-s", (time() - ($aOptions['retention'] * 86400)));


  // We should do a cleanup of archive here as well...

  if ( array_key_exists('archive', $aOptions) ) {
    $lDir = $aOptions['archivedir'];
  }
  else {
    $lDir = $gBackupDir;
  }


	tee("  Retention policy is " . $aOptions['retention'] . " days. Deleting all files which are older than $deletionDate...");
	// ls /data/backup/*/bck_*_[0-9][0-9][0-9][0-9]-[0-9][0-9]-[0-9][0-9]*
	tee('  Deleting the files from: ' . $lDir . '/' . $aOptions['policy'] . '/');
	$cnt = 0;
	// only delete our own files called bck_*{sql|cnf}[.gz]
	foreach ( glob($lDir . '/' . $aOptions['policy'] . '/' . 'bck_*') as $file ) {

		$aStat = stat($file);
		if ( $aStat['mtime'] < (time() - ($aOptions['retention'] * 86400)) ) {
			tee("  Remove $file");
			if ( ! array_key_exists('simulate', $aOptions) ) {
				unlink($file);
				if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
					list($ret, $cnt) = setFileCleanedupInCatalog($aBackupInformation['mysqli'], $aOptions['instance-name'], $file, $aStat['mtime']);
				}
			}
			$cnt++;
		}
	}   // foreach


	if ( $cnt == 0 ) {
		tee("  No files found to delete.");
	}
	else {
		// todo: If all files of a backup are cleaned up, set cleanedup flag in backup
		
				if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
					list($ret, $cnt) = setBackupCleanedupInCatalog($aBackupInformation['mysqli']);
				}
		
	}

	return $rc;
}


// ----------------------------------------------------------------------
function doSchemaBackup($aTarget, $aOptions, &$aBackupInformation)
// ----------------------------------------------------------------------
{
  global $gMysqlDump, $gBackupDir, $gBackupTimestamp, $gTargetConfiguration;

  $rc = OK;

  // Get schemata
  list($rc, $mysqli) = openConnection($aTarget);
  if ( $rc != OK ) {
    $rc = 156;
    tee("  ERROR: Connection error (rc=$rc).");
    return $rc;
  }

	list($ret, $aSchema) = getAllSchemas($mysqli);
	if ( $ret != OK ) {
		return $ret;
	}

	// No schema specified then use all schema
	if ( $aOptions['schema'] == '' ) {
		$aOptions['schema'] = '+' . implode(',+', $aSchema);
	}

	// print_r($aOptions);

  // Parse string
  $mode = '';
  $aSchemaToBackup = array();
  foreach ( explode(',', $aOptions['schema']) as $schema ) {

    // subtractive -test
    if ( substr($schema, 0, 1) == '-' ) {
      if ( ($mode == '-') || ($mode == '') ) {
        $mode = '-';
      }
      else {
        $rc = 158;
        tee("  ERROR: Subtractive and additive schema dump cannot be used together: " . $aOptions['schema'] . " (rc=$rc).");
        return $rc;
      }
    }
    // additive +test or test
    else {
      if ( ($mode == '+') || ($mode == '') ) {
        $mode = '+';
      }
      else {
        $rc = 159;
        tee("  ERROR: Subtractive and additive schema dump cannot be used together: " . $aOptions['schema'] . " (rc=$rc).");
        return $rc;
      }
    }

    if ( preg_match('/^[+-]?(.*)$/', $schema, $match) ) {
      array_push($aSchemaToBackup, $match[1]);
    }
    else {
      $rc = 160;
      tee("  ERROR: Unexpected error in $schema. Please report a bug (rc=$rc).");
      return $rc;
    }
  }

  // check if schemata are available
  foreach ( $aSchemaToBackup as $schema ) {
    if ( ! in_array($schema, $aSchema) ) {
      $rc = 161;
      tee("  ERROR: Schema $schema does not exist in database (rc=$rc).");
      // Do not exit here yet...
    }
  }
  if ( $rc != OK ) {
    return $rc;
  }

  // Create final Schema list
  if ( $mode == '+' ) {
    // SchemaToBackup is already correct
  }
  elseif ( $mode == '-' ) {
    $aSchemaToBackup = array_diff($aSchema, $aSchemaToBackup);
  }
  else {
    $rc = 162;
    tee("  ERROR: Unexpected error in $mode. Please report a bug (rc=$rc).");
    return $rc;
  }
  tee("  Schema to backup: " . implode($aSchemaToBackup, ', '));


  // Do the schema backup now

  list($ret, $output) = getDumpOverview($mysqli, $aSchemaToBackup);
  tee($output);

	// Check for non transactional tables
	$ret = checkForNonTransactionalTables($mysqli, $aOptions, $aSchemaToBackup);
	if ( $ret != OK ) {
		return $ret;
	}
	$lConsistency = '--single-transaction';

	// Check if binary logging is enabled
	list($rc, $lBinaryLogging) = checkIfBinaryLoggingIsEnabled($mysqli);
	if ( $rc != OK ) {
		return $rc;
	}
	if ( $lBinaryLogging == '' ) {
		tee("  Binary logging is disabled");
	}

	closeConnection($mysqli);


  // Do backup per schema

  if ( isset($aOptions['per-schema']) ) {

    foreach ( $aSchemaToBackup as $schema ) {

      // do mysqldump command
      // master_data=n does not make sense in this case
      // Make sure drop/create schema is not included. this is default behaviour
      // and allows to "rename" schemas
      $cmd = $gMysqlDump . " --user=" . $aTarget['user'] . " --host=". $aTarget['host'] . " --port=" . $aTarget['port'] . " $lBinaryLogging $lConsistency --flush-logs --triggers --routines --hex-blob " . $schema;
      // MySQL >= 5.1
      if ( $gTargetConfiguration['mr_version'] >= '050100' ) {
        $cmd .= ' --events';
      }

      $name = '';
      if ( isset($aOptions['instance-name']) && ($aOptions['instance-name'] != '') ) {
        $name = $aOptions['instance-name'] . '_';
      }

      $ext = 'sql';
      $file = 'bck_' . $name . $aOptions['type'] . '_' . $schema . '_' . $gBackupTimestamp . '.' . $ext;
      $dst = $gBackupDir . '/' . $aOptions['policy'] . '/' . $file;
      // todo: parameter list has to be cleaned-up!
      // todo: simulate is missing
      $ret = doMysqldump($aOptions, $cmd, $dst, $aTarget, $gMysqlDump);
			if ( $ret != OK ) {
				return $rc;
			}


			// Determine binary log file and binary log pos

			if ( ! array_key_exists('simulate', $aOptions) ) {
				// get binary log file/pos
				list($rc, $aBinlogPos) = getMasterPosFromDump($dst);
				if ( $rc != OK ) {
					return $rc;
				}
				$binlog = $aBinlogPos[0];
				$pos    = $aBinlogPos[1];
				if ( $binlog == '' ) {
					tee("  Backup does NOT contain any binary log information.");
				}
				else {
					tee("  Binlog file is " . $binlog . " and position is " . $pos);
					$aBackupInformation['binlog_file'] = $binlog;
					$aBackupInformation['binlog_pos'] = $pos;
				}
			}


			// Log file information into catalog

			if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
				$aFile = array (
					'id'                => 0
				, 'schema_name'       => $schema
				, 'original_name'     => ''
				, 'original_location' => ''
				, 'original_size'     => filesize($dst)
				, 'file_type_id'      => getFileTypeId($ext)
				, 'mtime'             => filemtime($dst)
				, 'backup_name'       => $file
				, 'backup_location'   => $gBackupDir . '/' . $aOptions['policy']
				, 'md5'               => ''
				, 'compressed'        => 0
				, 'compressed_name'   => ''
				, 'compressed_size'   => 0
				, 'archived'          => 0
				, 'archive_location'  => ''
				, 'cleanedup'         => 0
				, 'backup_id'         => $aBackupInformation['id']
				);
				list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
			}


			// Do checksum of file

			list($ret, $md5) = doChecksumOfFile($aOptions, $dst);
			if ( $ret != OK ) {
				return $ret;
			}


			// Log file information into catalog

			if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
				$aFile['md5'] = $md5;
				list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
			}


			// Compress file

			if ( ! isset($aOptions['no-compress']) ) {
				// Compress file
				$ret = compressFile($aOptions, $dst);
				if ( $ret != OK ) {
					return $ret;
				}
				$dst .= '.gz';


				// Log file information into catalog

				if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
					$aFile['compressed_name'] = $file .= '.gz';
					$aFile['compressed_size'] = filesize($dst);
					$aFile['compressed']      = 1;
					list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
				}
			}


			// Archive file

			if ( array_key_exists('archive', $aOptions) ) {
				$rc = archiveFileToArchivedir($aOptions, $dst);

				// Log file information into catalog
				if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
					$aFile['archived']         = 1;
					$aFile['archive_location'] = $aOptions['archivedir'] . '/' . $aOptions['policy'];
					list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
				}
			}


			// Cleanup file after archive

			if ( array_key_exists('cleanup', $aOptions) ) {
				if (! array_key_exists('simulate', $aOptions)) {
					$rc = cleanupFileAfterArchiveToArchivedir($aOptions, $dst);
				}

				// Log file information into catalog
				if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
					$aFile['cleanedup'] = 1;
					list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
				}
			}
			tee("");
		}   // foreach
	}

  // Do backup for all schemas together

  else {

    // do mysqldump command
    $cmd = $gMysqlDump . " --user=" . $aTarget['user'] . " --host=". $aTarget['host'] . " --port=" . $aTarget['port'] . " $lBinaryLogging --single-transaction --flush-logs --triggers --routines --hex-blob --databases " . implode($aSchemaToBackup, ' ');
    // MySQL >= 5.1
    if ( $gTargetConfiguration['mr_version'] >= '050100' ) {
      $cmd .= ' --events';
    }

    $name = '';
    if ( isset($aOptions['instance-name']) && ($aOptions['instance-name'] != '') ) {
      $name = $aOptions['instance-name'] . '_';
    }

		$ext = 'sql';
		$file = 'bck_' . $name . $aOptions['type'] . '_' . $gBackupTimestamp . '.' . $ext;
		$dst = $gBackupDir . '/' . $aOptions['policy'] . '/' . $file;
		// todo: parameter list has to be cleaned-up!
		$ret = doMysqldump($aOptions, $cmd, $dst, $aTarget, $gMysqlDump);
		if ( $ret != OK ) {
			return $rc;
		}


		// Log file information into catalog

		if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
			$aFile = array (
				'id'                => 0
			// , 'schema_name'       => count($aSchemaToBackup) . ' different schema(ta).'
			, 'schema_name'       => ''
			, 'original_name'     => ''
			, 'original_location' => ''
			, 'original_size'     => filesize($dst)
			, 'file_type_id'      => getFileTypeId($ext)
			, 'mtime'             => filemtime($dst)
			, 'backup_name'       => $file
			, 'backup_location'   => $gBackupDir . '/' . $aOptions['policy']
			, 'md5'               => ''
			, 'compressed'        => 0
			, 'compressed_name'   => ''
			, 'compressed_size'   => 0
			, 'archived'          => 0
			, 'archive_location'  => ''
			, 'cleanedup'         => 0
			, 'backup_id'         => $aBackupInformation['id']
			);
			list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
		}


		// Determine binary log file and binary log pos

		if ( ! array_key_exists('simulate', $aOptions) ) {
			// get binary log file/pos
			list($rc, $aBinlogPos) = getMasterPosFromDump($dst);
			if ( $rc != OK ) {
				return $rc;
			}
			$binlog = $aBinlogPos[0];
			$pos    = $aBinlogPos[1];
			if ( $binlog == '' ) {
				tee("  Backup does NOT contain any binary log information.");
			}
			else {
				tee("  Binlog file is " . $binlog . " and position is " . $pos);
				$aBackupInformation['binlog_file'] = $binlog;
				$aBackupInformation['binlog_pos']  = $pos;
			}
		}

		// Do checksum of file

		list($ret, $md5) = doChecksumOfFile($aOptions, $dst);
		if ( $ret != OK ) {
			return $ret;
		}


		// Log file information into catalog

		if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
			$aFile['md5'] = $md5;
			list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
		}


		// Compress file

		if ( ! isset($aOptions['no-compress']) ) {
			// Compress file
			$ret = compressFile($aOptions, $dst);
			if ( $ret != OK ) {
				return $ret;
			}
			$dst .= '.gz';

			// Log file information into catalog

			if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
				$aFile['compressed_name'] = $file .= '.gz';
				$aFile['compressed_size'] = filesize($dst);
				$aFile['compressed']      = 1;
				list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
			}
		}


		// Archive file

		if ( array_key_exists('archive', $aOptions) ) {
			$rc = archiveFileToArchivedir($aOptions, $dst);

				// Log file information into catalog
				if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
					$aFile['archived']         = 1;
					$aFile['archive_location'] = $aOptions['archivedir'] . '/' . $aOptions['policy'];
					list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
				}
		}


		// Cleanup files after archive

		if ( array_key_exists('cleanup', $aOptions) ) {
			if (! array_key_exists('simulate', $aOptions)) {
				$rc = cleanupFileAfterArchiveToArchivedir($aOptions, $dst);
			}

			// Log file information into catalog
			if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
				$aFile['cleanedup'] = 1;
				list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
			}
		}
	}

  return $rc;
}

// ----------------------------------------------------------------------
function doBinlogBackup($aTarget, $aOptions, &$aBackupInformation)
// ----------------------------------------------------------------------
{
  global $gMysqlDump, $gBackupDir, $gBackupTimestamp, $gTargetConfiguration;

  $rc = OK;

  list($rc, $mysqli) = openConnection($aTarget);
  if ( $rc != OK ) {
    $rc = 115;
    tee("  ERROR: open datbase failed (rc=$rc).");
    return $rc;
  }

  // check if binary logging is enabled!!!
  $sql = sprintf("SHOW MASTER STATUS");

  if ( ! ($result = $mysqli->query($sql)) ) {
    $rc = 105;
    tee("  ERROR: Invalid query: $sql, " . $mysqli->error . " (rc=$rc).");
    return $rc;
  }

  if ( $record = $result->fetch_assoc() ) {
    $lFile = $record['File'];
  }
  else {
    $rc = 116;
    tee("  ERROR: Binary logging seems to be NOT enabled on your database (rc=$rc).");
    return $rc;
  }


	/*
	
	Guess the Binary Log Index File
	This is the file where all known binary logs should be listed
	Before 5.6.4 we HAVE to guess:
	"The log_bin_index system variable was added in MySQL 5.6.4."
	There is NO variables to find out (only my.cnf --log-bin-index[=file_name])
	"If you omit the file name, and if you did not specify one with --log-bin,
	MySQL uses host_name-bin.index as the file name."
	
	So rule is:
	
	1. get log_bin_index
	2. If log_bin_index == '' or log_bin_index does NOT exist
	   Use value from log-bin (SHOW MASTER STATUS) + '.index'
	3. If file does not exist
	   Complain that we cannot guess and feature request must be added.
	
	*/

	list($ret, $aGlobalVariables) = getAllGlobalVariables($mysqli);

	$lBinlogIndexFile = '';
	if ( isset($aGlobalVariables['log_bin_index']) && $aGlobalVariables['log_bin_index'] != '' ) {
		$lBinlogIndexFile = $aGlobalVariables['log_bin_index'];
	}
	else {

		if ( preg_match("/^(.*)\.\d{4,6}$/", $lFile, $matches) ) {
			$lBinlogIndexFile = $gTargetConfiguration['datadir'] . '/' . $matches[1] . '.index';
		}
		else {
			$rc = 101;
			tee("  ERROR: We cannot figure out the binlog.index file. Please report this as an error (rc=$rc).");
			return $rc;
		}
	}


	// Read the bin_log.index file
	// Should be the same as with SHOW BINARY LOGS (without path)

	tee('  Binlog Index file is: ' . $lBinlogIndexFile);
	
	if( ( $fh = @fopen($lBinlogIndexFile, 'r') ) === FALSE ) {
		$rc = 114;
		tee("  ERROR: Failed to open file $lBinlogIndexFile for reading!");
		$a = error_get_last();
		tee('         ' . $a['message']);
		tee("         Is binary logging enabled on your database?");
		tee("         Possibly we cannot guess binlog index file good enough in older MySQL versions.");
		tee("         Please report at least a feature request (rc=$rc).");
		return $rc;
	}
  
	$aBinlogList = array();
	while ( $line = trim(fgets($fh)) ) {
	
		// print $line . "\n";
		
		// File can be:
		// * ./binary-log.nnnnnn
		// * /absolute/path/binary-log.nnnnnn

		$file = basename($line);
		$dir  = dirname($line);
		if ( $dir == '.' ) {
			$dir = $gTargetConfiguration['datadir'];
		}
		
		array_push($aBinlogList, $dir . '/' . $file);
	}
	fclose($fh);


	$name = '';
	if ( isset($aOptions['instance-name']) && ($aOptions['instance-name'] != '') ) {
		$name = $aOptions['instance-name'] . '_';
	}


	// Get the binary logs from SHOW BINARY LOGS to copy and purge

	$sql = sprintf("SHOW BINARY LOGS");

	if ( ! ($result = $mysqli->query($sql)) ) {
		$rc = 111;
		tee("Invalid query: $sql, " . $mysqli->error);
		tee("  ERROR: with $sql (rc=$rc)");
		return $rc;
	}


	// Do a FLUSH BINARY LOGS here
	// must be done AFTER readinging file and SHOW BINARY LOGS

	$sql3 = sprintf("FLUSH /*!50503 BINARY */ LOGS");
	tee("  $sql3");
	if ( ! array_key_exists('simulate', $aOptions) ) {
		$mysqli->query($sql3);
	}


	// do binary log backup file per file

	while ( $record = $result->fetch_assoc() ) {

		// find binary log in the BinlogList of bin_log.index to get the full path
		foreach ( $aBinlogList as $binlog ) {

			if ( preg_match("/" . $record['Log_name'] ."$/", $binlog, $matches) ) {

				$src = $binlog;
				$ext = '';
				$file = 'bck_' . $name . $record['Log_name'];
				$dst = $gBackupDir . '/' . $aOptions['policy'] . '/' . $file;

				// Before we touch/copy the binary log we have to get stats information for end ts.
				// this is a rough guessing but I have no clue how to do it better atm.
				$aStat = stat($src);

				tee("  Copy $src to $dst");
				if ( ! array_key_exists('simulate', $aOptions) ) {
					if ( copy($src, $dst) === false ) {
						$rc = 102;
						tee("  ERROR: Copying files... (rc=$rc)");
						return $rc;
					}
				}

				// Stop-position must be bigger than 4 to get the 1st timestamp
				$cmd = "mysqlbinlog --stop-position=8 $src";
				// If output is not set to '' begin ts is the same for all binary logs!!!
				$output = '';
				$stdout = exec("$cmd 2>&1", $output, $ret);
				if ( $ret != OK ) {
					$rc = 213;
					$msg = "Extracting timestamp from binary log $src failed: " . $output[0] . " (rc=$rc).";
					tee("  ERROR: $msg");
					return $rc;
				}

				$ts = '';
				foreach ( $output as $line ) {
					if ( preg_match('/^#(\d{6}\s+\d{1,2}:\d{2}:\d{2})/', $line, $matches) ) {
						$ts = $matches[1];
						break;
					}
				}

				$year   = '20' . substr($ts, 0, 2);
				$month  = substr($ts, 2, 2);
				$day    = substr($ts, 4, 2);
				$hour   = substr($ts, 7, 2);
				$minute = substr($ts, 10, 2);
				$second = substr($ts, 13, 2);

				$epoch = mktime( $hour, $minute, $second, $month, $day, $year );


				// Log file information into catalog

				if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
					$aFile = array (
						'id'                => 0
					, 'schema_name'       => ''
					, 'original_name'     => basename($src)
					, 'original_location' => dirname($src)
					, 'original_size'     => filesize($dst)
					, 'file_type_id'      => getFileTypeId($ext)
					, 'mtime'             => filemtime($dst)
					, 'backup_name'       => $file
					, 'backup_location'   => $gBackupDir . '/' . $aOptions['policy']
					, 'md5'               => ''
					, 'compressed'        => 0
					, 'compressed_name'   => ''
					, 'compressed_size'   => 0
					, 'archived'          => 0
					, 'archive_location'  => ''
					, 'cleanedup'         => 0
					, 'backup_id'         => $aBackupInformation['id']
					);
					list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
				}


				// Log binlog information into catalog

				if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
					$aBinaryLog = array (
						'id'             => 0
					, 'filename'       => basename($binlog)
					, 'begin_ts'       => $epoch
					, 'end_ts'         => $aStat['mtime']
					, 'begin_position' => 4
					, 'end_position'   => $aFile['original_size']
					, 'file_id'        => $aFile['id']
					);
					list($ret, $cnt) = setBinaryLogInCatalog($aBackupInformation['mysqli'], $aBinaryLog);
				}


				tee("  Do MD5 checksum of $dst");
				list($ret, $md5) = getMD5sum($dst, $aOptions);
				if ( $ret == OK ) {
					tee("  md5sum = $md5");

					// Log file information into catalog

					if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
						$aFile['md5'] = $md5;
						list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
					}
				}
				else {
					$rc = 154;
					tee("  ERROR: Error building MD5 sum of file $dst (rc=$rc).");
					return $rc;
				}


				// Compress file

				if ( ! isset($aOptions['no-compress']) ) {

					// Compress file
					$ret = compressFile($aOptions, $dst);
					if ( $ret != OK ) {
						return $ret;
					}
					$dst .= '.gz';


					// Log file information into catalog

					if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
						$aFile['compressed_name'] = $file .= '.gz';
						$aFile['compressed_size'] = filesize($dst);
						$aFile['compressed']      = 1;
						list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
					}
				}


				// Archive file to archivedir

				if ( array_key_exists('archive', $aOptions) ) {
					tee("  Archiving $dst to " . $aOptions['archivedir'] . '.');
					$cmd = "cp $dst " . $aOptions['archivedir'] . '/' . $aOptions['policy'] . '/';
					tee("  $cmd");
					if ( ! array_key_exists('simulate', $aOptions) ) {
						passthru($cmd, $ret);
						if ( $ret != OK ) {
							$rc = 132;
							tee("  ERROR: archiving failed (rc=$rc).");
							return $rc;
						}

						// Log file information into catalog

						if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
							$aFile['archived']         = 1;
							$aFile['archive_location'] = $aOptions['archivedir'] . '/' . $aOptions['policy'];
							list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
						}
          }
        }

				// Be carefull with this step if you have slaves!
				// This purges binary logs BEFORE xx

				if ( preg_match("/^(.*)\.(\d{4,6})$/", $record['Log_name'], $matches) ) {
					$lLogPlusOne = sprintf("%s.%06d", $matches[1], $matches[2]+1);
				}
				else {
					$rc = 117;
					tee("  ERROR: Problem parsing the binary logfile name. Please report this as an error.");
					tee("         " . $record['Log_name'] . " (rc=$rc).");
					return $rc;
				}

				$sql2 = sprintf("PURGE BINARY LOGS TO '%s'", $lLogPlusOne);
				tee("  $sql2");
				if ( ! array_key_exists('simulate', $aOptions) ) {
					$mysqli->query($sql2);
				}


				// Cleanup files after archive

				if ( array_key_exists('cleanup', $aOptions) ) {
					if (! array_key_exists('simulate', $aOptions)) {
						$rc = cleanupFileAfterArchiveToArchivedir($aOptions, $dst);
					}

					// Log file information into catalog
					if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
						$aFile['cleanedup'] = 1;
						list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
					}
				}


				// remove file from the binlog list
				if ( ($key = array_search($binlog, $aBinlogList)) !== false ) {
					unset($aBinlogList[$key]);
				}
				break;

			}   // binlog seems ok
			else {
				$rc = 226;
				$msg = 'Cannot parse binary log file name: ' . $record['Log_name'] . " (rc=$rc).";
				tee('  ERROR: ' . $msg);
				// no return here!
			}
		}   // foreach binlog
	}   // while binlog

	closeConnection($mysqli);
	return $rc;
}

// ----------------------------------------------------------------------
function getAllSchemas($mysqli)
// ----------------------------------------------------------------------
{
  $rc = OK;
  $aSchemas = array();

  $sql = sprintf("SELECT schema_name
  FROM information_schema.schemata
 WHERE schema_name NOT IN ('information_schema', 'PERFORMANCE_SCHEMA')");

  if ( ! ($result = $mysqli->query($sql)) ) {
    $rc = 165;
    tee("Invalid query: $sql, " . $mysqli->error . " (rc=$rc).");
    return array($rc, $aSchemas);
  }

  while ( $record = $result->fetch_assoc() ) {
    array_push($aSchemas, $record['schema_name']);
  }

  return array($rc, $aSchemas);
}

// ----------------------------------------------------------------------
function getDumpOverview($mysqli, $aSchemas)
// ----------------------------------------------------------------------
{
  $rc = OK;
  $overview = '';

  // +--------------+--------+-----+------------+-------------+------------+
  // | table_schema | engine | cnt | data_bytes | index_bytes | table_rows |
  // +--------------+--------+-----+------------+-------------+------------+
  // | download     | InnoDB |  12 |     196608 |       16384 |         85 |
  // | erp          | InnoDB |  22 |    1032192 |      327680 |       9581 |
  // | mysql        | CSV    |   2 |          0 |           0 |          4 |
  // | mysql        | InnoDB |   5 |     360448 |           0 |       2042 |
  // | mysql        | MyISAM |  21 |     613920 |       97280 |       2102 |
  // | test         | InnoDB |   6 |   67289088 |      245760 |    1053191 |
  // | tutorial     | InnoDB |   1 |    1589248 |      393216 |       8418 |
  // | world        | InnoDB |   3 |     655360 |      507904 |       5411 |
  // | zabbix       | InnoDB | 103 |   29605888 |    18923520 |     162575 |
  // +--------------+--------+-----+------------+-------------+------------+

  // -- This cannot be set per session :-(
  // -- requires SUPER privilege
  // /*!050504 SET GLOBAL innodb_stats_on_metadata = 0; */

  $sql = sprintf("SELECT table_schema, engine
     , COUNT(*) AS cnt, SUM(data_length) AS data_bytes, SUM(index_length) AS index_bytes, SUM(table_rows) AS table_rows
  FROM information_schema.tables
 WHERE table_type = 'BASE TABLE'
   AND table_schema NOT IN ('information_schema', 'PERFORMANCE_SCHEMA')\n");
  $sql .= sprintf("  AND table_schema IN('%s')\n", implode("', '", $aSchemas));
  $sql .= " GROUP BY table_schema, engine";

  if ( ! ($result = $mysqli->query($sql)) ) {
    $rc = 166;
    tee("Invalid query: $sql, " . $mysqli->error . " (rc=$rc).");
    return $rc;
  }

  $overview .= sprintf("  %-12s %-6s %-4s %-13s %-13s %-11s\n", 'table_schema', 'engine', 'cnt', 'data_bytes', 'index_bytes', 'table_rows');

  while ( $record = $result->fetch_assoc() ) {
    $overview .= sprintf("  %-12s %-6s %4d %13d %13d %11d\n", $record['table_schema'], $record['engine'], $record['cnt'], $record['data_bytes'], $record['index_bytes'], $record['table_rows']);
  }

  // -- This cannot be set per session :-(
  // -- requires SUPER privilege
  // /*!050504 SET GLOBAL innodb_stats_on_metadata = 1; */

  return array($rc, $overview);
}

// ----------------------------------------------------------------------
function doPrivilegeBackup($aTarget, $aOptions, &$aBackupInformation)
// ----------------------------------------------------------------------
{
	global $gBackupDir, $gBackupTimestamp;

	$rc = OK;

	list($rc, $mysqli) = openConnection($aTarget);
	if ( $rc != OK ) {
		$rc = 167;
		$msg = "Open datbase failed (rc=$rc).";
		tee("  ERROR: $msg");
		return $rc;
	}


	// Per schema privilege backup

	if ( isset($aOptions['per-schema']) ) {
		list($ret, $aSchemas) = getAllSchemas($mysqli);
		if ( $ret != OK ) {
			return $ret;
		}

		// todo: possibly later we have to support --schema=+/-schema functionality
		foreach ( $aSchemas as $schema ) {

			// Get all users of schema
			// First line to avoid error on non-hardened MySQL databases:
			// ERROR 1141 (42000): There is no such grant defined for user '' on host '%'
			$sql = sprintf("SELECT user, host FROM mysql.db WHERE db = '%s' AND user != ''
UNION
SELECT user, host FROM mysql.tables_priv WHERE db = '%s'
UNION
SELECT user, host FROM mysql.columns_priv WHERE db = '%s'
UNION
SELECT user, host FROM mysql.procs_priv WHERE db = '%s'", $schema, $schema, $schema, $schema);

			if ( ! ($result = $mysqli->query($sql)) ) {
				$rc = 168;
				$msg = "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
				tee("  ERROR: $msg");
				return $rc;
			}

			$aUser = array();
			while ( $record = $result->fetch_assoc() ) {
				array_push($aUser, $record);
			}

			if ( count($aUser) == 0 ) {
				tee("  No schema specific privileges for schema $schema");
				continue;
			}
				tee("  Schema specific privileges for schema $schema");

			// Now get all privileges of each user per schema

			$name = '';
			if ( isset($aOptions['instance-name']) && ($aOptions['instance-name'] != '') ) {
				$name = $aOptions['instance-name'] . '_';
			}

			$ext = 'sql';
			$file = 'bck_' . $name . $aOptions['type'] . '_' . $schema . '_' . $gBackupTimestamp . '.' . $ext;
			$dst = $gBackupDir . '/' . $aOptions['policy'] . '/' . $file;

			tee("  Writing privileges to $dst");
			if ( ! ($fh = fopen($dst, 'w')) ) {
				$rc = 169;
				tee("  ERROR: Can't open file $dst (rc=$rc).");
				return $rc;
			}

			fwrite($fh, "--\n");
			fwrite($fh, "-- Mysql bman privilege backup for schema $schema\n");
			fwrite($fh, "--\n");
			fwrite($fh, "\n");

			foreach ( $aUser as $user ) {

				fwrite($fh, "-- '" . $user['user'] . "'@'" . $user['host'] . "'\n");

				$sql = sprintf("SHOW GRANTS FOR '%s'@'%s'", $user['user'], $user['host']);

				if ( ! ($result = $mysqli->query($sql)) ) {
					$rc = 170;
					$msg = "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
					tee("  ERROR: $msg");
					return $rc;
				}

				while ( $record = $result->fetch_assoc() ) {
					foreach ( $record as $value) {
						fwrite($fh, $value . "\n");
					}
				}
				fwrite($fh, "\n");
			}
			fclose($fh);


			// Log file information into catalog

			if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
				$aFile = array (
					'id'               => 0
				, 'schema_name'      => $schema
				, 'original_name'    => ''
				, 'original_location' => ''
				, 'original_size'    => filesize($dst)
				, 'file_type_id'     => getFileTypeId($ext)
				, 'mtime'            => filemtime($dst)
				, 'backup_name'      => $file
				, 'backup_location'  => $gBackupDir . '/' . $aOptions['policy']
				, 'md5'              => ''
				, 'compressed'       => 0
				, 'compressed_name'  => ''
				, 'compressed_size'  => 0
				, 'archived'         => 0
				, 'archive_location' => ''
				, 'cleanedup'        => 0
				, 'backup_id'        => $aBackupInformation['id']
				);
				list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
			}


			tee("  Do MD5 checksum of $dst");
			list($ret, $md5) = getMD5sum($dst, $aOptions);
			if ( $ret == OK ) {
				tee("  md5sum = $md5");

				// Log file information into catalog

				if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
					$aFile['md5'] = $md5;
					list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
				}
			}
			else {
				$rc = 171;
				tee("  ERROR: Error building MD5 sum of file $dst (rc=$rc).");
				return $rc;
			}


			// We do NOT compress the file (not worth it)

			$aBackupInformation['compressed'] = 0;
			list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);


			// Archive file to archivedir

			if ( array_key_exists('archive', $aOptions) ) {
				tee("  Archiving $dst to " . $aOptions['archivedir'] . '.');
				// todo: Possibly better use: archiveFileToArchivedir? Also on other locations!
				$cmd = "cp $dst " . $aOptions['archivedir'] . '/' . $aOptions['policy'] . '/';
				tee("  $cmd");
				if ( ! array_key_exists('simulate', $aOptions) ) {
					passthru($cmd, $ret);
					if ( $ret != OK ) {
						$rc = 172;
						tee("  ERROR: archiving failed (rc=$rc).");
						return $rc;
					}
				}

				// Log file information into catalog

				if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
					$aFile['archived']         = 1;
					$aFile['archive_location'] = $aOptions['archivedir'] . '/' . $aOptions['policy'];
					list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
				}
			}


			// Cleanup files after archive

			if ( array_key_exists('cleanup', $aOptions) ) {
				if (! array_key_exists('simulate', $aOptions)) {
					$rc = cleanupFileAfterArchiveToArchivedir($aOptions, $dst);
				}

				// Log file information into catalog
				if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
					$aFile['cleanedup'] = 1;
					list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
				}
			}

		}   // foreach
	}

	// All privilege backup

	else {

		// Get all users
		$sql = sprintf("SELECT user, host FROM mysql.user");

		if ( ! ($result = $mysqli->query($sql)) ) {
			$rc = 173;
			$msg = "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
			tee("  ERROR: $msg");
			return $rc;
		}

		$aUser = array();
		while ( $record = $result->fetch_assoc() ) {
			array_push($aUser, $record);
		}


		// Now get all privileges of each user

		$name = '';
		if ( isset($aOptions['instance-name']) && ($aOptions['instance-name'] != '') ) {
			$name = $aOptions['instance-name'] . '_';
		}

		$ext  = 'sql';
		$file = 'bck_' . $name . $aOptions['type'] . '_' . $gBackupTimestamp . '.' . $ext;
		$dst  = $gBackupDir . '/' . $aOptions['policy'] . '/' . $file;

		tee("  Writing privileges to $dst");
		if ( ! ($fh = fopen($dst, 'w')) ) {
			$rc = 174;
			tee("  ERROR: Can't open file $dst (rc=$rc).");
			return $rc;
		}

		fwrite($fh, "--\n");
		fwrite($fh, "-- Mysql bman privilege backup\n");
		fwrite($fh, "--\n");
		fwrite($fh, "\n");

		foreach ( $aUser as $user ) {

			fwrite($fh, "-- '" . $user['user'] . "'@'" . $user['host'] . "'\n");

			$sql = sprintf("SHOW GRANTS FOR '%s'@'%s'", $user['user'], $user['host']);

			if ( ! ($result = $mysqli->query($sql)) ) {
				$rc = 201;
				$msg = "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
				tee("  ERROR: $msg");
				return $rc;
			}

			while ( $record = $result->fetch_assoc() ) {
				foreach ( $record as $value) {
					fwrite($fh, $value . "\n");
				}
			}
			fwrite($fh, "\n");
		}
		fclose($fh);


		// Log file information into catalog

		if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
			$aFile = array (
				'id'               => 0
			, 'schema_name'      => ''
			, 'original_name'    => ''
			, 'original_location' => ''
			, 'original_size'    => filesize($dst)
			, 'file_type_id'     => getFileTypeId($ext)
			, 'mtime'            => filemtime($dst)
			, 'backup_name'      => $file
			, 'backup_location'  => $gBackupDir . '/' . $aOptions['policy']
			, 'md5'              => ''
			, 'compressed'       => 0
			, 'compressed_name'  => ''
			, 'compressed_size'  => 0
			, 'archived'         => 0
			, 'archive_location' => ''
			, 'cleanedup'        => 0
			, 'backup_id'        => $aBackupInformation['id']
			);
			list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
		}


		// todo: possibly use: doChecksumOfFile here? and also other locations?
		tee("  Do MD5 checksum of $dst");
		list($ret, $md5) = getMD5sum($dst, $aOptions);
		if ( $ret == OK ) {
			tee("  md5sum = $md5");

			// Log file information into catalog

			if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
				$aFile['md5'] = $md5;
				list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
			}
		}
		else {
			$rc = 200;
			tee("  ERROR: Error building MD5 sum of file $dst (rc=$rc).");
			return $rc;
		}


		// We do NOT compress the file (not worth it)

		$aBackupInformation['compressed'] = 0;
		list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);


		// Archive file to archivedir

		if ( array_key_exists('archive', $aOptions) ) {
			tee("  Archiving $dst to " . $aOptions['archivedir'] . '.');
			$cmd = "cp $dst " . $aOptions['archivedir'] . '/' . $aOptions['policy'] . '/';
			tee("  $cmd");
			if ( ! array_key_exists('simulate', $aOptions) ) {
				passthru($cmd, $ret);
				if ( $ret != OK ) {
					$rc = 202;
					tee("  ERROR: archiving failed (rc=$rc).");
					return $rc;
				}

				// Log file information into catalog

				if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
					$aFile['archived']         = 1;
					$aFile['archive_location'] = $aOptions['archivedir'] . '/' . $aOptions['policy'];
					list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
				}
			}
		}


		// Cleanup files after archive

		if ( array_key_exists('cleanup', $aOptions) ) {
			if (! array_key_exists('simulate', $aOptions)) {
				$rc = cleanupFileAfterArchiveToArchivedir($aOptions, $dst);
			}

			// Log file information into catalog
			if ( array_key_exists('catalog', $aOptions) && (! array_key_exists('simulate', $aOptions)) ) {
				$aFile['cleanedup'] = 1;
				list($ret, $cnt) = setFileInCatalog($aBackupInformation['mysqli'], $aFile);
			}
		}

	}   // per-schema backup
	closeConnection($mysqli);

	return $rc;
}

// ---------------------------------------------------------------------
function getBackupTypeId($pType)
// ---------------------------------------------------------------------
{
	$aBackupType = array(
	  1 => 'full'
	, 2 => 'binlog'
	, 3 => 'config'
	, 4 => 'structure'
	, 5 => 'cleanup'
	, 6 => 'schema'
	, 7 => 'privilege'
	);
	return intval(array_search($pType, $aBackupType));
}

// ---------------------------------------------------------------------
function getBackupPolicyId($pPolicy)
// ---------------------------------------------------------------------
{
	$aBackupPolicy = array(
	  1 => 'daily'
	, 2 => 'weekly'
	, 3 => 'monthly'
	, 4 => 'quarterly'
	, 5 => 'yearly'
	, 6 => 'binlog'
	);
	return intval(array_search($pPolicy, $aBackupPolicy));
}

// ---------------------------------------------------------------------
function getBackupModeId($pMode)
// ---------------------------------------------------------------------
{
	$aBackupMode = array(
	  1 => 'logical'
	, 2 => 'physical'
	);
	return intval(array_search($pMode, $aBackupMode));
}

// ---------------------------------------------------------------------
function getFileTypeId($pExt)
// ---------------------------------------------------------------------
{
	$aFileType = array(
	  1 => 'sql'
	, 2 => 'bin'
	, 3 => 'frm'
	, 4 => 'MYD'
	, 5 => 'MYI'
	, 6 => 'ibd'
	);
	return intval(array_search(ltrim($pExt, '.'), $aFileType));
}

// ----------------------------------------------------------------------
function getAllGlobalVariables($mysqli)
// ----------------------------------------------------------------------
{
	$rc = OK;

	$aGlobalVariables = array();

	$sql = sprintf("SHOW GLOBAL VARIABLES");

	if ( ! ($result = $mysqli->query($sql)) ) {
		$rc = 225;
		$msg = "Invalid query: $sql, " . $mysqli->error . " (rc=$rc).";
		tee("  ERROR: $msg");
		return array($rc, null);
	}

	while ( $record = $result->fetch_assoc() ) {

		$aGlobalVariables[$record['Variable_name']] = $record['Value'];
	}

	return array($rc, $aGlobalVariables);
}

?>
