<?php

/*

  debug($s)
  error($s)
  getConfiguration($pConfigurationFile)
  extractVersion($s)
  startDatabase($aInstance)
  checkDatabase($aInstance)
  wait_for_pid($lMysqldPid, $pidfile)
  stopDatabase($aInstance)
  getSchemaNames($pDatadir)
  addDirectoryToPath($lPath, $lDir, $lOldDir, $location = 'left')
  deleteDirectoryFromPath($lPath, $lDir)
  getSectionTitles($a)
  printStartingDatabases()
  readline($lne = '')
  getCurrentUser()
  checkMyEnvRequirements()
  parseConfigFile($pConfigurationFile, $scanner_mode = INI_SCANNER_NORMAL)

*/

require_once('Constants.inc');

// ----------------------------------------------------------------------
function debug($s)
// ----------------------------------------------------------------------
{
  if ( isset($_ENV['MYENV_DEBUG']) && ($_ENV['MYENV_DEBUG'] != '') ) {
    echo 'MYENV_DEBUG: ' . $s . "\n";
  }
}

// ----------------------------------------------------------------------
function error($s)
// ----------------------------------------------------------------------
{
  echo 'ERROR: ' . $s . "\n";
}

// ----------------------------------------------------------------------
function getConfiguration($pConfigurationFile)
// ----------------------------------------------------------------------
{
	$rc = OK;

  $aConfiguration = array();

  if ( ! file_exists($pConfigurationFile) ) {
    $rc = 221;
    error("Configuration file $pConfigurationFile does not exist (rc=$rc).");
    return array($rc, $aConfiguration);
  }

  list($ret, $aConfiguration) = parseConfigFile($pConfigurationFile, INI_SCANNER_RAW);

  // Process default section first
  $lInstance = 'default';

  // Define default behaviour here

  if ( ! isset($aConfiguration[$lInstance]['angel']) ) {
    $aConfiguration[$lInstance]['angel'] = 'yes';
  }

  if ( ! isset($aConfiguration[$lInstance]['user']) ) {
    $aConfiguration[$lInstance]['user'] = 'mysql';
  }

  // Process all other sections != default
  foreach ( $aConfiguration as $db => $value ) {

    // add section name to section
    $aConfiguration[$db]['name'] = $db;

    if ( $db != 'default' ) {

      if ( (! isset($aConfiguration[$db]['hideschema'])) && (isset($aConfiguration['default']['hideschema'])) ) {
        $aConfiguration[$db]['hideschema'] = $aConfiguration['default']['hideschema'];
      }
      if ( (! isset($aConfiguration[$db]['angel'])) && (isset($aConfiguration['default']['angel'])) ) {
        $aConfiguration[$db]['angel'] = $aConfiguration['default']['angel'];
      }
      // Set default mysql user
      if ( (! isset($aConfiguration[$db]['user'])) && (isset($aConfiguration['default']['user'])) ) {
        $aConfiguration[$db]['user'] = $aConfiguration['default']['user'];
      }
    }
  }

  return array($rc, $aConfiguration);
}

// ----------------------------------------------------------------------
function extractVersion($s)
// $s = basedir
// ----------------------------------------------------------------------
{
  $version = 'unknown';

  // Expected patterns: see tst/extractVersion.php
  if ( preg_match("/([a-zA-Z]+)\d?\-/", $s, $matches) ) {

    if ( ($matches[1] == 'mysql')
      || ($matches[1] == 'percona')
      || ($matches[1] == 'mariadb')
       ) {

      if ( preg_match("/(\d{1,2}\.\d{1,2}\.\d{1,2})/", $s, $matches) ) {
        $version = $matches[1];
      }
      elseif ( preg_match("/(\d{1,2}\.\d{1,2}\.\d{1,2}\-ndb\-\d{1,2}\.\d{1,2}\.\d{1,2}[a-z]?)/", $s, $matches) ) {
        $version = $matches[1];
      }
      elseif ( preg_match("/(\d{1,2}\.\d{1,2})/", $s, $matches) ) {
        $version = $matches[1];
      }
      else {
      }
    }
    elseif ( $matches[1] == 'Percona' ) {
      if ( preg_match("/(\d{1,2}\.\d{1,2}\.\d{1,2})/", $s, $matches) ) {
        $version = 'Percona-' . $matches[1];
      }
    }
    else {
      // print '1';
    }
  }
  else {
    // Case: 5.6.12
    if ( preg_match("/(\d{1,2}\.\d{1,2}\.\d{1,2})/", $s, $matches) ) {
      $version = $matches[1];
    }
  }

  return $version;
}

// ----------------------------------------------------------------------
function startDatabase($aInstance, $pOptions = '')
// ----------------------------------------------------------------------
{
  $rc = 0;
  global $lDebug;

  // Check if db is started with correct user
  // Also root should be allowed to start DB (Bug #99)
  $lCurrentUser = getCurrentUser();
  if ( ($lCurrentUser != $aInstance['user'])
    && ($lCurrentUser != 'root')
     ) {
    $rc = 222;
    error("Database is started with the wrong user ($lCurrentUser) but should should be strated with " . $aInstance['user'] . " (rc=$rc).");
    return $rc;
  }

  // Check if mysql.user table exists
  if ( ! file_exists($aInstance['datadir'] . '/mysql/user.frm') ) {
    $rc = 223;
    error("MySQL user table (user.frm) does not exist. Are you sure you have installed a MySQL instance already? (rc=$rc)");
    error("Create a MySQL instance with the command: mysql_install_db --datadir=" . $aInstance['datadir'] . " --basedir=" . $aInstance['basedir']);
    return $rc;
  }

  list($ret, $aMyCnf) = parseConfigFile($aInstance['my.cnf'], INI_SCANNER_RAW);

  // Check if there is any missmatch between my.cnf and myenv.conf
  foreach ( $aInstance as $key => $value ) {
    if ( isset($aMyCnf['mysqld'][$key]) ) {
      if ( $aMyCnf['mysqld'][$key] != $value ) {
        $cnf = '/etc/myenv/myenv.conf';
        print "WARNING: $key differs between " . $aInstance['my.cnf'] . " and " . $cnf . ". Please fix to avoid troubles...\n";
      }
    }
  }


  // It can happen, that client section is missing in my.cnf
  if ( isset($aMyCnf['client']) ) {

    // Check also missmatches between mysqld and client section in my.cnf
    foreach ( $aMyCnf['client'] as $key => $value ) {
      if ( isset($aMyCnf['mysqld'][$key]) ) {
        if ( $aMyCnf['mysqld'][$key] != $value ) {
          print "WARNING: $key differs between mysqld and client section in my.cnf. Please fix to avoid troubles...\n";
        }
      }
    }
  }

  // Check if the database is started already
  $ret = checkDatabase($aInstance);
  if ( $ret == 0 ) {
    $rc = 233;
    error("Database is alreay started (rc=$rc).");
    return $rc;
  }


  $cmd = '';
  $cmd .= 'cd ' . $aInstance['basedir'] . " ; ";

  // Start mysqld with angel process
  if ( $aInstance['angel'] == 'yes' ) {

    $cmd .= "nohup bin/mysqld_safe --defaults-file=" . $aInstance['my.cnf'] . " --datadir=" . $aInstance['datadir'] . " --basedir=" . $aInstance['basedir'] . " " . $pOptions;

    if ( $lDebug != '' ) {
      $cmd .= ' & echo $!';
    }
    else {
      $cmd .= ' >/dev/null 2>&1 & echo $!';
    }
  }
  // Start mysqld WITHOUT angel process
  else {
    // Should be better redirected to the error log!

    $cmd .= $aInstance['basedir'] . '/bin/mysqld --defaults-file=' . $aInstance['my.cnf'] . ' --basedir=' . $aInstance['basedir'] . ' --datadir=' . $aInstance['datadir'] . ' --user=' . $aInstance['user']. ' ' . $pOptions;

    if ( $lDebug != '' ) {
      $cmd .= ' & echo $!';
    }
    else {
      $cmd .= ' >/dev/null 2>&1 & echo $!';
    }
  }

  debug($cmd);
  $stdout = shell_exec($cmd);


  # No return code here!
  #if ( $ret != 0 ) {
  #  error("Command $cmd failed.");
  #  $rc = 224;
  #  // no return here
  #}

  $lDebugOrig = $lDebug;
  $lDebug = '';
  for ( $i = 1; $i <= 60 ; $i++ ) {

    sleep(1);
    echo ".";
    $ret = checkDatabase($aInstance);
    $rc = $ret;
    if ( $ret == 0 ) {
      break;
    }
  }
  $lDebug = $lDebugOrig;

  if ( $rc == 0 ) {
    print " SUCCESS!\n";
    return($rc);
  }
  else {
    print " ERROR!\n";
    return($rc);
  }
}

// ----------------------------------------------------------------------
function checkDatabase($aInstance)
// ----------------------------------------------------------------------
{
  $rc = OK;

  $lTimeout = 3;

  // Sometimes it can happen, that mysqladmin is missing (Galera)
  $exe = $aInstance['basedir'] . '/bin/mysqladmin';
	$stdout = exec("which $exe 2>&1", $output, $ret);
	if ( $ret != OK ) {
		$rc = 224;
		$msg = "$exe does no exist (rc=$rc).";
		print "$msg\n";
		return $rc;
	}

	// todo: This strategy is not good because it requires access to
  // defaults-file which is not given to non privileged users
  // but for just ping we do not really need defaults-file?
  // Timeout is for hanging mysqld (happend with Galera Bug #56)
  $err = '/tmp/checkDatabase.' . posix_getpid() . '.tmp';
	$cmd = "$exe --socket=" . $aInstance['socket'] . " ping --connect-timeout=$lTimeout >/dev/null 2>$err";

  debug($cmd);
  exec($cmd, $output, $ret);

  if ( $ret != OK ) {
    $rc = 225;
    if ( $ret != 1 ) {
			readfile($err);
		}
    
    // no return here
  }
  @unlink($err);

  return $rc;
}

// ----------------------------------------------------------------------
function wait_for_pid($lMysqldPid, $pidfile)
// ----------------------------------------------------------------------
{
  $rc = 0;

  $i = 0;
  $service_shutdown_timeout = 900;   // 15 min

  while ( $i <= $service_shutdown_timeout ) {

    sleep(1);
    print '.';
    // Process seems dead
    if ( posix_kill($lMysqldPid, SIG_IGN) === false ) {
      break;
    }

    $i++;
  }

  if ( file_exists($pidfile) ) {
    $rc = 226;
    print "\nThe server quit without removing PID file ($pidfile) (rc=$rc).\n";
  }

  if ( $rc == 0 ) {
    print " SUCCESS!\n";
    return($rc);
  }
  else {
    print " ERROR!\n";
    return($rc);
  }
}

// ----------------------------------------------------------------------
function stopDatabase($aInstance)
// ----------------------------------------------------------------------
{
  $rc = 0;

  $lDebug         = isset($_ENV['MYENV_DEBUG']) ? intval($_ENV['MYENV_DEBUG']) : 0;
  $basedir        = $_ENV['MYENV_BASE'];
  $user           = 'root';

  // We need defaults file here to support root password in client
  // section for shutdown service
  // $cmd = $aInstance['basedir'] . "/bin/mysqladmin --defaults-file=" . $aInstance['my.cnf'] . " --user=$user shutdown --socket=" . $aInstance['socket'];

  // debug($cmd);
  // $stdout = exec($cmd, $output, $ret);
  // if ( $ret != 0 ) {
  //   $rc = 227;
  //   error("Command $cmd failed (rc=$rc).");
  //   error("Please use password in ~/.my.cnf.");
  //   // no return here
  // }


  // Find the PID file

  $lMysqldPidFile = '';
  // We do not look into the database because this requires some account
  // which we do not have yet...
  list($ret, $aMyCnf) = parseConfigFile($aInstance['my.cnf'], INI_SCANNER_RAW);
  // print_r($aMyCnf);
  // Use PID file from my.cnf
  if ( isset($aMyCnf['mysqld']['pid_file']) && ($aMyCnf['mysqld']['pid_file'] != '') ) {
    $lMysqldPidFile = $aMyCnf['mysqld']['pid_file'];
  }
  elseif ( isset($aMyCnf['mysqld']['pid-file']) && ($aMyCnf['mysqld']['pid-file'] != '') ) {
    $lMysqldPidFile = $aMyCnf['mysqld']['pid-file'];
  }
  // Guess PID file
  // Guessing on /var/run/mysqld/mysqld.pid is NOT allowed because this could kill the wrong Instance!
  else {

    $pf = $aInstance['datadir'] . '/mysqld.pid';
    if ( is_file($pf) ) {
      $lMysqldPidFile = $pf;
    }

    $pf = $aInstance['datadir'] . '/' . php_uname('n') . '.pid';
    if ( is_file($pf) ) {
      $lMysqldPidFile = $pf;
    }
  }

  if ( $lMysqldPidFile == '' ) {
    $rc = 228;
    error("Cannot find nor guess PID file (rc=$rc). Is it possible that the database is already stopped? (rc=$rc)");
    return $rc;
  }


  // Stop daemon. We use a signal here to avoid having to know the
  // root password.

  if ( file_exists($lMysqldPidFile) ) {

    if ( ! is_readable($lMysqldPidFile) ) {
      $rc = 229;
      error("Cannot read PID file $lMysqldPidFile. Wrong privileges? (rc=$rc).");
      return $rc;
    }

    $lMysqldPid = intval(file_get_contents($lMysqldPidFile));
    if ( $lMysqldPid == 0 ) {
      $rc = 230;
      error("PID from $lMysqldPidFile is wrong or empty (rc=$rc).");
      return $rc;
    }

    if ( posix_kill($lMysqldPid, SIG_IGN) === true ) {

      // print 'Shutting down MySQL';
      posix_kill($lMysqldPid, SIGTERM);
      // mysqld should remove the PID file when it exits, so wait for it.
      $rc = wait_for_pid($lMysqldPid, $lMysqldPidFile);
    }
    else {
      print "MySQL server process $lMysqldPid is not running!\n";
      unlink($lMysqldPidFile);
    }

    // Delete lock for RedHat / SuSE
    $lLockDir  = '/var/lock/subsys';
    $lLockFile = $lLockDir . '/mysql';

    if ( is_file($lLockFile) ) {
      unlink($lLockFile);
    }
  }
  else {
    $rc = 231;
    print "MySQL server PID file could not be found at $lMysqldPidFile. Database already stopped? (rc=$rc)!\n";
    return $rc;
  }

  return $rc;
}

# ----------------------------------------------------------------------
function getSchemaNames($pDatadir)
# Parameter     : $_[0] data directory
# Return values : hash of schema names
# ----------------------------------------------------------------------
{
  $aSchema = array();

  foreach ( glob("$pDatadir/*", GLOB_ONLYDIR) as $schema ) {
    if ( preg_match(':([^/]+)$:', $schema, $match) ) {
      array_push($aSchema, $match[1]);
    }
  }

  return $aSchema;
}

# ----------------------------------------------------------------------
function addDirectoryToPath($lPath, $lDir, $lOldDir, $location = 'left')
# Parameter     : $_[0] $PATH
#                 $_[1] new dir
# Return values : new $PATH
# ----------------------------------------------------------------------
{
  global $lDebug;

  if ( $lDebug != '' ) {
    print "PATH = " . $lPath . "\n";
    print "DIR  = " . $lDir . "\n";
    print "OLD  = " . $lOldDir . "\n";
  }

  $lPath = str_replace($lDir, '', $lPath);
  $lPath = str_replace($lOldDir, '', $lPath);
  if ( $location == 'left' ) {
    $lPath = $lDir . ':' . $lPath;
  }
  elseif ( $location == 'right' ) {
    $lPath = $lPath . ':' . $lDir;
  }
  else {
    $rc = 232;
    fprint(STDERR, "Invalid location $location (rc=$rc).\n");
  }
  // some clean-up
  $lPath = preg_replace('/:+/', ':', $lPath);
  $lPath = trim($lPath, ':');
  if ( $lDebug != '' ) {
    print "PATH = " . $lPath . "\n";
  }

  return $lPath;
}

# ----------------------------------------------------------------------
function deleteDirectoryFromPath($lPath, $lDir)
# ----------------------------------------------------------------------
{
  // delete
  $lPath = str_replace($lDir, '', $lPath);
  // remove ::
  $lPath = preg_replace('/:+/', ':', $lPath);
  // clean : at left or right side of PATH
  $lPath = trim($lPath, ':');

  return $lPath;
}

// ----------------------------------------------------------------------
function getSectionTitles($a)
// ----------------------------------------------------------------------
{
  $aDbNames = array();
  foreach ( $a as $db => $value ) {

    if ( $db != 'default' ) {
      array_push($aDbNames, $db);
    }
  }
  return $aDbNames;
}

// ----------------------------------------------------------------------
function printStartingDatabases()
// ----------------------------------------------------------------------
{
	$rc = OK;
  $aDbNames = array();

  $lDebug         = isset($_ENV['MYENV_DEBUG']) ? strval($_ENV['MYENV_DEBUG']) : '';
  $basedir        = $_ENV['MYENV_BASE'];
  // todo: hard coded here:
  $lConfFile      = '/etc/myenv/myenv.conf';
  list($ret, $aConfiguration) = getConfiguration($lConfFile);
  if ( $lDebug ) {
    var_dump($aConfiguration);
  }
  if ( $ret != OK ) {
		return array($ret, $aDbNames);
  }

  foreach ( $aConfiguration as $db => $parameter ) {

    if ( $db != 'default' ) {
      foreach ( $parameter as $key => $value ) {

        if ( ($key == 'start') && ((strtoupper($value) == 'YES' ) || ($value == 1)) ) {
          array_push($aDbNames, $db);
        }
      }
    }
  }
  return array($rc, $aDbNames);
}

// ----------------------------------------------------------------------
if ( ! function_exists('readline') ) {

  function readline($lne = '') {
// ----------------------------------------------------------------------

    print $lne;
    $fh = fopen('php://stdin', 'r');
    $input = fgets($fh, 1024);
    fclose($fh);
   return rtrim($input);
  }
}

// ----------------------------------------------------------------------
function getCurrentUser()
// ----------------------------------------------------------------------
{
  return trim(shell_exec('whoami'));
}

// ----------------------------------------------------------------------
function checkMyEnvRequirements()
// ----------------------------------------------------------------------
{
  $ret = OK;

  // print_r(get_defined_functions());
  // print_r(get_loaded_extensions());

  // Check if environment variables are enabled

  if ( count($_ENV) == 0 ) {
    $vo = ini_get('variables_order');
    print "ERROR: Environment variables cannot be used. Please set variables_order = 'EGPCS' in PHP cli config file. Current value is: variables_order = " . $vo . "\n";
		print "RedHat/CentOS: shell> sudo vi /etc/php.ini\n";
		print "Ubuntu/Debian: shell> sudo vi /etc/php5/cli/php.ini\n";
		print "SLES/OpenSuSE: shell> sudo vi /etc/php5/cli/php.ini\n";
    return ERR;
  }

  // Check if posix functions are installed

  if ( ! function_exists('posix_kill') ) {
    print "WARNING: The function posix_kill is not installed. You cannot stop MySQL instances without. Please install the package containing posix_kill:\n";
    print "RedHat/CentOS: shell> sudo yum install php-posix\n";
    print "Ubuntu/Debian: shell> sudo apt-get install php5-posix\n";
    print "SLES/OpenSuSE: shell> sudo zypper install php5-posix\n";
    $ret = WARN;
  }
  if ( ! function_exists('posix_getpwuid') ) {
    print "WARNING: The function posix_getpwuid is not installed. You cannot continue installing without. Please install the package containing posix_getpwuid first:\n";
    print "RedHat/CentOS: shell> sudo yum install php-posix\n";
    print "Ubuntu/Debian: shell> sudo apt-get install php5-posix\n";
    print "SLES/OpenSuSE: shell> sudo zypper install php5-posix\n";
    return ERR;
  }

  if ( (! defined('SIG_IGN')) || (! defined('SIGTERM')) ) {
    print "WARNING: The constants SIG_IGN and/or SIGTERM are not known. You cannot stop MySQL instances without. Please install the package containing these constants first:\n";
    print "Try to install the package php5-pcntl. We will help you in the meanwhile with a fake...\n";
    print "RedHat/CentOS: shell> sudo yum install php-pcntl\n";
    print "Ubuntu/Debian: shell> sudo apt-get install php5-pcntl\n";
    print "SLES/OpenSuSE: shell> sudo zypper install php5-pcntl\n";
    define('SIG_IGN', 0);
    define('SIGTERM', 15);
    $ret = WARN;
  }

  // Check if php-mysqli is installed (for mysql_bman)

  if ( ! in_array('mysqli', get_loaded_extensions()) ) {
    print "WARNING: The PHP extension mysqli is not installed. You cannot run somte tools without it. Please install the package containing mysqli first:\n";
    print "RedHat/CentOS: shell> sudo yum install php-mysqli\n";
    print "Ubuntu/Debian: shell> sudo apt-get install php5-mysql\n";
    print "SLES/OpenSuSE: shell> sudo zypper install php5-mysqli\n";
    $ret = ERR;
  }

  return $ret;
}

// ----------------------------------------------------------------------
function parseConfigFile($pConfigurationFile, $scanner_mode = INI_SCANNER_NORMAL)
/*

  we have different problems parsing mysql config files:
	parse_ini_file() seems to be not a good idea to use with mysql my.cnf files
	Comments starting with '#' are deprecated in /home/mysql/data/mysql-5.6.15/my.cnf on line 42 in /home/mysql/product/myenv-rev392/lib/myEnv.inc on line 365
	Comments starting with '#' are deprecated in /home/mysql/data/mysql-5.6.15/my.cnf on line 42 in /home/mysql/product/myenv-rev392/lib/myEnv.inc on line 166
	syntax error, unexpected '!' in /etc/mysql/my.cnf on line 127 (rc=164)
	127 !includedir /etc/mysql/conf.d/
	Seems to have problems with !include directive!!!
	Seems to have problems with passwords with strange characaters
	core-file is not shown!
	The my.cnf has this line at the end "!includedir /etc/mysql/conf.d/" which caused the error "syntax error, unexpected '!' in /etc/mysql/my.cnf on line 132 (rc=164)".
	mysql_bman.php
	
*/
// ----------------------------------------------------------------------
{
	$rc = OK;

	// Replace all comments like:
	// \w*#$
	
	if ( ! file_exists($pConfigurationFile) ) {
	  $rc = 227;
	  $msg = "File $pConfigurationFile does not exist (rc=$rc).";
	  return array($rc, $msg);
	}
	
	if ( ($aRawConf = file($pConfigurationFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES)) === false ) {
	  $rc = 228;
	  $msg = "Cannot read file $pConfigurationFile (rc=$rc).";
	  return array($rc, $msg);
	}

	$tmp = '/tmp' . '/' . posix_getpid() . '.cnf';
	$fh = fopen($tmp, 'w');
	
	$pattern = array(
	  '/^#.*$/'
	, '/\s+#.*$/'
	, '/^!include/'
	);
	foreach ( $aRawConf as $line ) {
	  $line = preg_replace($pattern, '', $line) . "\n";
	  $n = fwrite($fh, $line);
	}
	fclose($fh);

	// Do not hide error outputs!
	if ( ($aConfig = @parse_ini_file($tmp, true, $scanner_mode)) === false ) {
	  $rc = 230;
	  parse_ini_file($tmp, true, $scanner_mode);
	  $msg = "Troubles reading configuration file $pConfigurationFile (rc=$rc).";
		unlink($tmp);
	  return array($rc, $msg);
	}
	unlink($tmp);

	return array($rc, $aConfig);
}

?>
