from mercurial.node import hex
from mercurial import bookmarks
from mercurial.cmdutil import getlogrevs
from mercurial.graphmod import dagwalker

def _isfile(repo, node, path):
    try:
        cctx = repo[node]
        cctx.filenode(path)
        return True
    except LookupError:
        return False

def _determinefilerevs(repo, node, path):
    changelog = repo.changelog
    # detect all heads for which we are trying to compose the log
    # it's important to have our target head first because this
    # is the anchor into the log
    heads = list()
    heads.append(changelog.rev(node))
    for node in changelog.heads():
        heads.append(changelog.rev(node))
    filelog = set()
    revtoinfo = dict()
    for headrev in heads:
        try:
            filectx = repo[headrev][path]
        except LookupError:
            continue

        # find the first real changed rev
        filectx = repo[filectx.introrev()][path]
        filectxs = [filectx]
        for ctx in filectx.ancestors():
            ctx.rev()  # TODO: if this call is active, log is 10-100 times faster
            filectxs.append(ctx)

        headlog = set()
        for ctx in filectxs:
            rev = ctx.rev()
            headlog.add(rev)

            revparents = list()
            for parentctx in ctx.parents():
                revparents.append(parentctx.rev())
            revtoinfo[rev] = (ctx._path, revparents)

        if not filelog:
            filelog |= headlog
        # we will only add the log of an additional head if
        # it's actually the same file
        elif filelog & headlog:
            filelog |= headlog
    return filelog, revtoinfo

def _determinedirrevs(repo, path):
    revs, expr, filematcher = getlogrevs(repo, (path,), dict())
    revdag = dagwalker(repo, revs)
    revs = set(revs)
    filelog = set()
    revtoinfo = dict()
    for rev, type, ctx, parents in revdag:
        dagparents = list()
        for p in parents: # dag parents may not be contained in dag itself
            if p in revs:
                dagparents.append(p)

        filelog.add(rev)
        revtoinfo[rev] = (path, dagparents)

    return filelog, revtoinfo


def _calcrevtorefs(repo, head, pathlog):
    changelog = repo.changelog
    revtotags = {}
    headRev = changelog.rev(head)
    revtotags.setdefault(headRev, set()).add(('head', 'HEAD', hex(head), headRev))
    for (branch, nodes) in repo.branchmap().items():
        for node in nodes:
            type = 'branch-closed' if repo[node].closesbranch() else 'branch'
            rev = changelog.rev(node)
            revtotags.setdefault(rev, set()).add((type, branch, hex(node), rev))

    for tag, node in repo.tags().items():
        rev = changelog.rev(node)
        type = repo.tagtype(tag)
        if type is not None:
            revtotags.setdefault(rev, set()).add(('tag-' + type, tag, hex(node), rev))

    marks = bookmarks.bmstore(repo)
    for (bookmark, node) in sorted(marks.iteritems()):
        rev = changelog.rev(node)
        revtotags.setdefault(rev, set()).add(('bookmark', bookmark, hex(node), rev))

    tip = changelog.rev(changelog.tip())
    logrevtotag = {}
    processedtags = set()
    for rev in reversed(range(0, tip + 1)):
        revtags = revtotags.get(rev, set())
        if rev in pathlog:
            logrevtotag[rev] = list(revtags.difference(processedtags))
            processedtags = processedtags.union(revtags)
        elif len(revtags) > 0:
            for parent in changelog.parentrevs(rev):
                revtotags.setdefault(parent, set())
                revtotags[parent] = revtags.union(revtotags[parent]).difference(processedtags)
        revtotags.pop(rev, None)
    return logrevtotag


def sglogpath(ui, repo, node, path, **opts):
    node = repo.changelog.lookup(node)
    isfile = _isfile(repo, node, path)

    if isfile:
        pathlog, revtoinfo = _determinefilerevs(repo, node, path)
    else:
        pathlog, revtoinfo = _determinedirrevs(repo, path)

    logrevtotag = _calcrevtorefs(repo, node, pathlog)

    for rev in reversed(sorted(list(pathlog))):
        (revpath, revparents) = revtoinfo.get(rev)
        ctx = repo[rev]

        p1Node = hex(ctx.parents()[0].node()) if len(ctx.parents()) >= 1 else ''
        p1Rev = ctx.parents()[0].rev() if len(ctx.parents()) >= 1 else '-1'
        p2Node = hex(ctx.parents()[1].node()) if len(ctx.parents()) >= 2 else ''
        p2Rev = ctx.parents()[1].rev() if len(ctx.parents()) >= 2 else '-1'
        revRefs = logrevtotag.get(rev, [])
        ui.write('node\t', len(revparents), '\t', len(revRefs), '\t', revpath, '\t', hex(ctx.node()), '\t', rev, '\t', ctx.phase(), '\t', ctx.user(),
                 '\t%d %d' % ctx.date(), '\t',
                 ctx.branch(), '\t', p1Node, '\t', p1Rev, '\t', p2Node, '\t', p2Rev, '\t\0', ctx.description().rstrip(), '\0\n')

        for p in revparents:
            ui.write('ancestor\t', p, '\t', hex(repo[p].node()), '\n')

        for (type, ref, commit, rev) in revRefs:
            ui.write('ref\t', type, '\t', ref, '\t', commit, '\t', rev, '\n')
    ui.write('eof\n')

cmdtable = {
    'sglogpath': (sglogpath, [], '[options] REV FILE')
}
