/*******************************************************************************
 * Copyright (c) 2000, 2011 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Syntevo GmbH    - major redesign
 *******************************************************************************/
package com.syntevo.q.swt;

import java.util.*;
import java.util.List;

import org.eclipse.swt.*;
import org.eclipse.swt.events.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.widgets.*;

public class QTabFolder extends Composite {

	// Constants ==============================================================

	private static final int SELECTION_FOREGROUND = SWT.COLOR_LIST_FOREGROUND;
	private static final int SELECTION_BACKGROUND = SWT.COLOR_LIST_BACKGROUND;

	// Fields =================================================================

	private final List<QTabFolderListener> folderListeners = new ArrayList<>(2);
	private final List<QTabFolderItem> items = new ArrayList<>(2);

	private final Rectangle chevronRect = new Rectangle(0, 0, 0, 0);
	private final Rectangle topRightRect = new Rectangle(0, 0, 0, 0);

	private final QTabFolderRenderer renderer;

	private int fixedTabHeight = SWT.DEFAULT;
	private int tabHeight;
	private final boolean borderVisible;
	private int firstIndex = -1; // index of the left most visible tab.
	private int selectedIndex = -1;
	private int[] priority = new int[0];
	private boolean mru;
	private final Listener listener;

	/* Selected item appearance */
	private Color selectionForeground;
	private Color selectionBackground;

	// close, min/max and chevron buttons
	private final boolean showClose;
	private boolean showUnselectedClose = true;
	private int chevronImageState = SWT.NONE;
	private boolean showChevron;
	private Menu showMenu;

	private Control topRight;

	private boolean inDispose;

	// keep track of size changes in order to redraw only affected area
	// on Resize
	private Point oldSize;
	private Font oldFont;
	private Color borderColor;

	// Setup ==================================================================

	public QTabFolder(Composite parent, int style) {
		this(parent, style, parent.getDisplay().getSystemColor(SWT.COLOR_WIDGET_NORMAL_SHADOW));
	}

	public QTabFolder(Composite parent, int style, Color borderColor) {
		super(parent, checkStyle(parent, style));

		this.borderColor = borderColor;

		super.setLayout(new QTabFolderLayout());
		final int style2 = super.getStyle();
		oldFont = getFont();
		showClose = (style2 & SWT.CLOSE) != 0;
		borderVisible = (style & SWT.BORDER) != 0;
		//set up default colors
		final Display display = getDisplay();
		selectionForeground = display.getSystemColor(SELECTION_FOREGROUND);
		selectionBackground = display.getSystemColor(SELECTION_BACKGROUND);
		renderer = new QTabFolderRenderer(this);
		updateTabHeight(false);

		// Add all listeners
		listener = new Listener() {
			@Override
			public void handleEvent(Event event) {
				switch (event.type) {
				case SWT.Dispose:
					onDispose(event);
					break;
				case SWT.DragDetect:
					onDragDetect(event);
					break;

				case SWT.MouseDoubleClick:
					mouseDoubleClick(event);
					break;
				case SWT.MouseDown:
					if (event.button == 1) {
						mouseDown(event.x, event.y);
					}
					break;
				case SWT.MouseEnter:
					mouseEnter();
					break;
				case SWT.MouseExit:
					mouseExit();
					break;
				case SWT.MouseMove:
					mouseMove(event.x, event.y);
					break;
				case SWT.MouseUp:
					if (event.button == 1) {
						mouseUp(event.x, event.y, event.time);
					}
					break;

				case SWT.Paint:
					onPaint(event);
					break;
				case SWT.Resize:
					onResize();
					break;
				}
			}
		};

		final int[] folderEvents = new int[] {
				SWT.Dispose,
				SWT.DragDetect,
				SWT.MouseDoubleClick,
				SWT.MouseDown,
				SWT.MouseEnter,
				SWT.MouseExit,
				SWT.MouseMove,
				SWT.MouseUp,
				SWT.Paint,
				SWT.Resize
		};
		for (int folderEvent : folderEvents) {
			addListener(folderEvent, listener);
		}
	}

	// Implemented ============================================================

	@Override
	public String toString() {
		String string = "*Disposed*";
		if (!isDisposed()) {
			final StringBuilder buffer = new StringBuilder();
			for (QTabFolderItem item : items) {
				if (buffer.length() > 0) {
					buffer.append(' ');
				}
				if (item == getSelection()) {
					buffer.append('*');
				}

				buffer.append(item.getText());
			}
			string = buffer.toString();
		}
		return "QTabFolder {" + string + "}";
	}

	@Override
	public Rectangle computeTrim(int x, int y, int width, int height) {
		checkWidget();
		return renderer.computeBodyTrim(x, y, width, height);
	}

	@Override
	public Rectangle getClientArea() {
		checkWidget();
		final Rectangle trim = renderer.computeBodyTrim(0, 0, 0, 0);
		final Point size = getSize();
		final int width = size.x - trim.width;
		final int height = size.y - trim.height;
		return new Rectangle(-trim.x, -trim.y, width, height);
	}

	@Override
	public int getStyle() {
		int style = super.getStyle();
		if (borderVisible) {
			style |= SWT.BORDER;
		}
		style &= ~SWT.CLOSE;
		if (showClose) {
			style |= SWT.CLOSE;
		}
		return style;
	}

	@Override
	public void reskin(int flags) {
		super.reskin(flags);
		for (QTabFolderItem item : items) {
			item.reskin(flags);
		}
	}

	@Override
	public void setForeground(Color color) {
		super.setForeground(color);
		redraw();
	}

	/**
	 * Sets the layout which is associated with the receiver to be
	 * the argument which may be null.
	 * <p>
	 * Note: No Layout can be set on this Control because it already
	 * manages the size and position of its children.
	 * </p>
	 *
	 * @param layout the receiver's new layout or null
	 * @throws SWTException <ul>
	 *                      <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
	 *                      <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
	 *                      </ul>
	 */
	@Override
	public void setLayout(Layout layout) {
		checkWidget();
	}

	// Accessing ==============================================================

	public void addTabFolderListener(QTabFolderListener listener) {
		checkWidget();
		if (listener == null) {
			SWT.error(SWT.ERROR_NULL_ARGUMENT);
		}
		if (folderListeners.contains(listener)) {
			SWT.error(SWT.ERROR_UNSPECIFIED);
		}

		folderListeners.add(listener);
	}

	public void removeTabFolderListener(QTabFolderListener listener) {
		checkWidget();
		if (listener == null) {
			SWT.error(SWT.ERROR_NULL_ARGUMENT);
		}
		if (!folderListeners.contains(listener)) {
			SWT.error(SWT.ERROR_UNSPECIFIED);
		}

		folderListeners.remove(listener);
	}

	/**
	 * Adds the listener to the collection of listeners who will
	 * be notified when the user changes the receiver's selection, by sending
	 * it one of the messages defined in the <code>SelectionListener</code>
	 * interface.
	 * <p>
	 * <code>widgetSelected</code> is called when the user changes the selected tab.
	 * <code>widgetDefaultSelected</code> is not called.
	 * </p>
	 *
	 * @param listener the listener which should be notified when the user changes the receiver's selection
	 * @throws IllegalArgumentException <ul>
	 *                                  <li>ERROR_NULL_ARGUMENT - if the listener is null</li>
	 *                                  </ul>
	 * @throws SWTException             <ul>
	 *                                  <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
	 *                                  <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
	 *                                  </ul>
	 * @see SelectionListener
	 * @see #removeSelectionListener
	 * @see SelectionEvent
	 */
	public void addSelectionListener(SelectionListener listener) {
		checkWidget();
		if (listener == null) {
			SWT.error(SWT.ERROR_NULL_ARGUMENT);
		}
		final TypedListener typedListener = new TypedListener(listener);
		addListener(SWT.Selection, typedListener);
		addListener(SWT.DefaultSelection, typedListener);
	}

	void createItem(QTabFolderItem item, int index) {
		if (item == null) {
			SWT.error(SWT.ERROR_NULL_ARGUMENT);
		}
		if (index < 0) {
			SWT.error(SWT.ERROR_UNSPECIFIED);
		}
		if (index > getItemCount()) {
			SWT.error(SWT.ERROR_UNSPECIFIED);
		}

		items.add(index, item);
		if (selectedIndex >= index) {
			selectedIndex++;
		}
		final int[] newPriority = new int[priority.length + 1];
		int next = 0;
		int priorityIndex = priority.length;
		for (int aPriority : priority) {
			if (!mru && aPriority == index) {
				priorityIndex = next++;
			}
			newPriority[next++] = aPriority >= index ? aPriority + 1 : aPriority;
		}
		newPriority[priorityIndex] = index;
		priority = newPriority;

		if (items.size() == 1) {
			if (!updateTabHeight(false)) {
				updateItems();
			}
			redraw();
		}
		else {
			updateItems();
			redrawTabs();
		}
	}

	void destroyItem(QTabFolderItem item) {
		if (item == null) {
			SWT.error(SWT.ERROR_NULL_ARGUMENT);
		}
		if (inDispose) {
			return;
		}

		final int index = indexOf(item);
		if (index == -1) {
			return;
		}

		items.remove(index);
		if (items.size() == 1) {
			priority = new int[0];
			firstIndex = -1;
			selectedIndex = -1;

			item.getControl().setVisible(false);
			setToolTipText(null);
			setButtonBounds();
			redraw();
			return;
		}

		final int[] newPriority = new int[priority.length - 1];
		int next = 0;
		for (int aPriority : priority) {
			if (aPriority == index) {
				continue;
			}
			newPriority[next++] = aPriority > index ? aPriority - 1 : aPriority;
		}
		priority = newPriority;

		// move the selection if this item is selected
		if (selectedIndex == index) {
			final Control control = item.getControl();
			selectedIndex = -1;
			final int nextSelection = mru ? priority[0] : Math.max(0, index - 1);
			setSelection(nextSelection, true);
			control.setVisible(false);
		}
		else if (selectedIndex > index) {
			selectedIndex--;
		}

		updateItems();
		redrawTabs();
	}

	/**
	 * Return the tab that is located at the specified index.
	 *
	 * @param index the index of the tab item
	 * @return the item at the specified index
	 * @throws IllegalArgumentException <ul>
	 *                                  <li>ERROR_INVALID_RANGE - if the index is out of range</li>
	 *                                  </ul>
	 * @throws SWTException             <ul>
	 *                                  <li>ERROR_THREAD_INVALID_ACCESS when called from the wrong thread</li>
	 *                                  <li>ERROR_WIDGET_DISPOSED when the widget has been disposed</li>
	 *                                  </ul>
	 */
	public QTabFolderItem getItem(int index) {
		return items.get(index);
	}

	/**
	 * Return the number of tabs in the folder.
	 *
	 * @return the number of tabs in the folder
	 * @throws SWTException <ul>
	 *                      <li>ERROR_THREAD_INVALID_ACCESS when called from the wrong thread</li>
	 *                      <li>ERROR_WIDGET_DISPOSED when the widget has been disposed</li>
	 *                      </ul>
	 */
	public int getItemCount() {
		return items.size();
	}

	public List<QTabFolderItem> getItemsList() {
		return Collections.unmodifiableList(items);
	}

	QTabFolderRenderer getRenderer() {
		checkWidget();
		return renderer;
	}

	int getRightItemEdge() {
		final Rectangle trim = renderer.computeBorderTrim();
		int x = getSize().x - (trim.width + trim.x) - 3; //TODO: add setter for spacing?
		if (showChevron) {
			x -= renderer.getChevronButtonSize().x;
		}
		if (topRight != null) {
			final Point rightSize = topRight.computeSize(SWT.DEFAULT, SWT.DEFAULT);
			x -= rightSize.x + QTabFolderLayout.SPACE_RIGHT_BESIDE_TOPRIGHT;
		}
		return Math.max(0, x);
	}

	/**
	 * Return the selected tab item, or null if there is no selection.
	 *
	 * @return the selected tab item, or null if none has been selected
	 * @throws SWTException <ul>
	 *                      <li>ERROR_THREAD_INVALID_ACCESS when called from the wrong thread</li>
	 *                      <li>ERROR_WIDGET_DISPOSED when the widget has been disposed</li>
	 *                      </ul>
	 */
	public QTabFolderItem getSelection() {
		//checkWidget();
		if (selectedIndex == -1) {
			return null;
		}

		return items.get(selectedIndex);
	}

	/**
	 * Returns the receiver's selection background color.
	 *
	 * @return the selection background color of the receiver
	 * @throws SWTException <ul>
	 *                      <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
	 *                      <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
	 *                      </ul>
	 * @since 3.0
	 */
	public Color getSelectionBackground() {
		checkWidget();
		return selectionBackground;
	}

	/**
	 * Returns the receiver's selection foreground color.
	 *
	 * @return the selection foreground color of the receiver
	 * @throws SWTException <ul>
	 *                      <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
	 *                      <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
	 *                      </ul>
	 * @since 3.0
	 */
	public Color getSelectionForeground() {
		checkWidget();
		return selectionForeground;
	}

	/**
	 * Return the index of the selected tab item, or -1 if there
	 * is no selection.
	 *
	 * @return the index of the selected tab item or -1
	 * @throws SWTException <ul>
	 *                      <li>ERROR_THREAD_INVALID_ACCESS when called from the wrong thread</li>
	 *                      <li>ERROR_WIDGET_DISPOSED when the widget has been disposed</li>
	 *                      </ul>
	 */
	public int getSelectionIndex() {
		//checkWidget();
		return selectedIndex;
	}

	/**
	 * Returns the height of the tab
	 *
	 * @return the height of the tab
	 * @throws SWTException <ul>
	 *                      <li>ERROR_THREAD_INVALID_ACCESS when called from the wrong thread</li>
	 *                      <li>ERROR_WIDGET_DISPOSED when the widget has been disposed</li>
	 *                      </ul>
	 */
	public int getTabHeight() {
		checkWidget();
		if (fixedTabHeight != SWT.DEFAULT) {
			return fixedTabHeight;
		}
		return tabHeight - 1; // -1 for line drawn across top of tab //TODO: replace w/ computeTrim of tab area?
	}

	/**
	 * Returns the control in the top right corner of the tab folder.
	 * Typically this is a close button or a composite with a menu and close button.
	 *
	 * @return the control in the top right corner of the tab folder or null
	 * @throws SWTException <ul>
	 *                      <li>ERROR_THREAD_INVALID_ACCESS when called from the wrong thread</li>
	 *                      <li>ERROR_WIDGET_DISPOSED when the widget has been disposed</li>
	 *                      </ul>
	 * @since 2.1
	 */
	public Control getTopRight() {
		checkWidget();
		return topRight;
	}

	public void setTopRight(Control control) {
		checkWidget();
		if (control != null && control.getParent() != this) {
			SWT.error(SWT.ERROR_INVALID_ARGUMENT);
		}
		topRight = control;
		if (updateItems()) {
			redraw();
		}
	}

	/**
	 * Returns <code>true</code> if the close button appears
	 * when the user hovers over an unselected tabs.
	 *
	 * @return <code>true</code> if the close button appears on unselected tabs
	 * @since 3.0
	 */
	public boolean getUnselectedCloseVisible() {
		checkWidget();
		return showUnselectedClose;
	}

	/**
	 * Return the index of the specified tab or -1 if the tab is not
	 * in the receiver.
	 *
	 * @param item the tab item for which the index is required
	 * @return the index of the specified tab item or -1
	 * @throws IllegalArgumentException <ul>
	 *                                  <li>ERROR_NULL_ARGUMENT - if the listener is null</li>
	 *                                  </ul>
	 * @throws SWTException             <ul>
	 *                                  <li>ERROR_THREAD_INVALID_ACCESS when called from the wrong thread</li>
	 *                                  <li>ERROR_WIDGET_DISPOSED when the widget has been disposed</li>
	 *                                  </ul>
	 */
	public int indexOf(QTabFolderItem item) {
		checkWidget();
		if (item == null) {
			SWT.error(SWT.ERROR_NULL_ARGUMENT);
		}

		return items.indexOf(item);
	}

	void redrawTabs() {
		final Point size = getSize();
		final Rectangle trim = renderer.computeBodyTrim(0, 0, 0, 0);
		redraw(0, 0, size.x, -trim.y + 1, false);
	}

	/**
	 * Removes the listener from the collection of listeners who will
	 * be notified when the user changes the receiver's selection.
	 *
	 * @param listener the listener which should no longer be notified
	 * @throws IllegalArgumentException <ul>
	 *                                  <li>ERROR_NULL_ARGUMENT - if the listener is null</li>
	 *                                  </ul>
	 * @throws SWTException             <ul>
	 *                                  <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
	 *                                  <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
	 *                                  </ul>
	 * @see SelectionListener
	 * @see #addSelectionListener
	 */
	public void removeSelectionListener(SelectionListener listener) {
		checkWidget();
		if (listener == null) {
			SWT.error(SWT.ERROR_NULL_ARGUMENT);
		}
		removeListener(SWT.Selection, listener);
		removeListener(SWT.DefaultSelection, listener);
	}

	/**
	 * When there is not enough horizontal space to show all the tabs,
	 * by default, tabs are shown sequentially from left to right in
	 * order of their index.  When the MRU visibility is turned on,
	 * the tabs that are visible will be the tabs most recently selected.
	 * Tabs will still maintain their left to right order based on index
	 * but only the most recently selected tabs are visible.
	 * <p>
	 * For example, consider a QTabFolder that contains "Tab 1", "Tab 2",
	 * "Tab 3" and "Tab 4" (in order by index).  The user selects
	 * "Tab 1" and then "Tab 3".  If the QTabFolder is now
	 * compressed so that only two tabs are visible, by default,
	 * "Tab 2" and "Tab 3" will be shown ("Tab 3" since it is currently
	 * selected and "Tab 2" because it is the previous item in index order).
	 * If MRU visibility is enabled, the two visible tabs will be "Tab 1"
	 * and "Tab 3" (in that order from left to right).</p>
	 *
	 * @param show the new visibility state
	 * @throws SWTException <ul>
	 *                      <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
	 *                      <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
	 *                      </ul>
	 * @since 3.1
	 */
	public void setMRUVisible(boolean show) {
		checkWidget();
		if (mru == show) {
			return;
		}
		mru = show;
		if (!mru) {
			final int idx = firstIndex;
			int next = 0;
			for (int i = firstIndex; i < items.size(); i++) {
				priority[next++] = i;
			}
			for (int i = 0; i < idx; i++) {
				priority[next++] = i;
			}
			if (updateItems()) {
				redrawTabs();
			}
		}
	}

	/**
	 * Set the selection to the tab at the specified item.
	 *
	 * @param item the tab item to be selected
	 * @throws IllegalArgumentException <ul>
	 *                                  <li>ERROR_NULL_ARGUMENT - if the item is null</li>
	 *                                  </ul>
	 * @throws SWTException             <ul>
	 *                                  <li>ERROR_THREAD_INVALID_ACCESS when called from the wrong thread</li>
	 *                                  <li>ERROR_WIDGET_DISPOSED when the widget has been disposed</li>
	 *                                  </ul>
	 */
	public void setSelection(QTabFolderItem item) {
		checkWidget();
		if (item == null) {
			SWT.error(SWT.ERROR_NULL_ARGUMENT);
		}

		final int index = indexOf(item);
		setSelection(index);
	}

	/**
	 * Sets the receiver's selection background color to the color specified
	 * by the argument, or to the default system color for the control
	 * if the argument is null.
	 *
	 * @param color the new color (or null)
	 * @throws IllegalArgumentException <ul>
	 *                                  <li>ERROR_INVALID_ARGUMENT - if the argument has been disposed</li>
	 *                                  </ul>
	 * @throws SWTException             <ul>
	 *                                  <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
	 *                                  <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
	 *                                  </ul>
	 * @since 3.0
	 */
	public void setSelectionBackground(Color color) {
		checkWidget();
		if (selectionBackground.equals(color)) {
			return;
		}
		if (color == null) {
			color = getDisplay().getSystemColor(SELECTION_BACKGROUND);
		}
		selectionBackground = color;
		if (selectedIndex > -1) {
			redraw();
		}
	}

	/**
	 * Set the foreground color of the selected tab.
	 *
	 * @param color the color of the text displayed in the selected tab
	 * @throws SWTException <ul>
	 *                      <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
	 *                      <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
	 *                      </ul>
	 */
	public void setSelectionForeground(Color color) {
		checkWidget();
		if (selectionForeground.equals(color)) {
			return;
		}
		if (color == null) {
			color = getDisplay().getSystemColor(SELECTION_FOREGROUND);
		}
		selectionForeground = color;
		if (selectedIndex > -1) {
			redraw();
		}
	}

	/**
	 * Specify a fixed height for the tab items.  If no height is specified,
	 * the default height is the height of the text or the image, whichever
	 * is greater. Specifying a height of -1 will revert to the default height.
	 *
	 * @param height the pixel value of the height or -1
	 * @throws SWTException <ul>
	 *                      <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
	 *                      <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
	 *                      <li>ERROR_INVALID_ARGUMENT - if called with a height of less than 0</li>
	 *                      </ul>
	 */
	public void setTabHeight(int height) {
		checkWidget();
		if (height < -1) {
			SWT.error(SWT.ERROR_INVALID_ARGUMENT);
		}
		fixedTabHeight = height;
		updateTabHeight(false);
	}

	/**
	 * Specify whether the close button appears
	 * when the user hovers over an unselected tabs.
	 *
	 * @param visible <code>true</code> makes the close button appear
	 * @throws SWTException <ul>
	 *                      <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
	 *                      <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
	 *                      </ul>
	 * @since 3.0
	 */
	public void setUnselectedCloseVisible(boolean visible) {
		checkWidget();
		if (showUnselectedClose == visible) {
			return;
		}
		// display close button when mouse hovers
		showUnselectedClose = visible;
		updateItems();
		redraw();
	}

	/**
	 * Shows the selection.  If the selection is already showing in the receiver,
	 * this method simply returns.  Otherwise, the items are scrolled until
	 * the selection is visible.
	 *
	 * @throws SWTException <ul>
	 *                      <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
	 *                      <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
	 *                      </ul>
	 * @see QTabFolder#showItem(QTabFolderItem)
	 * @since 2.0
	 */
	public void showSelection() {
		checkWidget();
		final QTabFolderItem selection = getSelection();
		if (selection != null) {
			showItem(selection);
		}
	}

	void _setToolTipText(int x, int y) {
		final String oldTip = getToolTipText();
		final String newTip = _getToolTip(x, y);
		if (newTip == null || !newTip.equals(oldTip)) {
			setToolTipText(newTip);
		}
	}

	boolean updateItems() {
		return updateItems(selectedIndex);
	}

	boolean updateItems(int showIndex) {
		final GC gc = new GC(this);
		if (!mru && showIndex != -1) {
			// make sure selected item will be showing
			int firstIndex = showIndex;
			if (priority[0] < showIndex) {
				final Rectangle trim = renderer.computeBorderTrim();
				final int borderLeft = -trim.x;
				final int maxWidth = getRightItemEdge() - borderLeft;
				int width = 0;
				final int[] widths = new int[items.size()];
				for (int i = priority[0]; i <= showIndex; i++) {
					final QTabFolderItem item = getItem(i);
					widths[i] = renderer.getMinTabWidth(item, gc);
					width += widths[i];
					if (width > maxWidth) {
						break;
					}
				}
				if (width > maxWidth) {
					width = 0;
					for (int i = showIndex; i >= 0; i--) {
						if (widths[i] == 0) {
							final QTabFolderItem item = getItem(i);
							widths[i] = renderer.getMinTabWidth(item, gc);
						}
						width += widths[i];
						if (width > maxWidth) {
							break;
						}
						firstIndex = i;
					}
				}
				else {
					firstIndex = priority[0];
					for (int i = showIndex + 1; i < items.size(); i++) {
						final QTabFolderItem item = getItem(i);
						widths[i] = renderer.getMinTabWidth(item, gc);
						width += widths[i];
						if (width >= maxWidth) {
							break;
						}
					}
					if (width < maxWidth) {
						for (int i = priority[0] - 1; i >= 0; i--) {
							if (widths[i] == 0) {
								final QTabFolderItem item = getItem(i);
								widths[i] = renderer.getMinTabWidth(item, gc);
							}
							width += widths[i];
							if (width > maxWidth) {
								break;
							}
							firstIndex = i;
						}
					}
				}
			}
			if (firstIndex != priority[0]) {
				int index = 0;
				for (int i = firstIndex; i < items.size(); i++) {
					priority[index++] = i;
				}
				for (int i = 0; i < firstIndex; i++) {
					priority[index++] = i;
				}
			}
		}

		final boolean oldShowChevron = showChevron;
		boolean changed = setItemSize(gc);
		changed |= setItemLocation();
		setButtonBounds();
		changed |= showChevron != oldShowChevron;
		if (changed && getToolTipText() != null) {
			Point pt = getDisplay().getCursorLocation();
			pt = toControl(pt);
			_setToolTipText(pt.x, pt.y);
		}
		gc.dispose();
		return changed;
	}

	boolean updateTabHeight(boolean force) {
		final int oldHeight = tabHeight;
		tabHeight = renderer.getHeaderHeight();
		if (!force && tabHeight == oldHeight) {
			return false;
		}
		oldSize = null;
		notifyListeners(SWT.Resize, new Event());
		return true;
	}

	String _getToolTip(int x, int y) {
		if (showChevron && chevronRect.contains(x, y)) {
			return SWT.getMessage("SWT_ShowList"); //$NON-NLS-1$
		}
		final QTabFolderItem item = getItem(x, y);
		if (item == null) {
			return null;
		}
		if (!item.isShowing()) {
			return null;
		}
		if ((showClose || item.isShowClose()) && item.getCloseRect().contains(x, y)) {
			return SWT.getMessage("SWT_Close"); //$NON-NLS-1$
		}
		return item.getToolTipText();
	}

	int getFixedTabHeight() {
		return fixedTabHeight;
	}

	boolean isBorderVisible() {
		return borderVisible;
	}

	int getFirstIndex() {
		return firstIndex;
	}

	int getSelectedIndex() {
		return selectedIndex;
	}

	int[] getPriority() {
		return priority;
	}

	boolean isShowClose() {
		return showClose;
	}

	boolean isShowUnselectedClose() {
		return showUnselectedClose;
	}

	public QTabFolderItem getClickedItem(int x, int y) {
		for (QTabFolderItem item : items) {
			final Rectangle bounds = item.getBounds();
			if (bounds.contains(x, y)) {
				return item;
			}
		}
		return null;
	}

	public Point getPreferredSize(Control control, boolean flushCache) {
		return control.computeSize(SWT.DEFAULT, SWT.DEFAULT, flushCache);
	}

	protected void setFocus(Control control) {
		control.setFocus();
	}

	boolean isSelected(QTabFolderItem item) {
		final int i = indexOf(item);
		if (i < 0) {
			return false;
		}
		return i == selectedIndex;
	}

	Color getBorderColor() {
		return borderColor;
	}

	public void setBorderColor(Color borderColor) {
		if (this.borderColor == borderColor) {
			return;
		}

		this.borderColor = borderColor;

		redraw();
	}

	// Utils ==================================================================

	private QTabFolderItem getItem(int x, int y) {
		if (items.isEmpty()) {
			return null;
		}
		final Point size = getSize();
		final Rectangle trim = renderer.computeBorderTrim();
		if (size.x <= trim.width) {
			return null;
		}
		if (showChevron && chevronRect.contains(x, y)) {
			return null;
		}
		for (int aPriority : priority) {
			final QTabFolderItem item = getItem(aPriority);
			final Rectangle rect = item.getBounds();
			if (rect.contains(x, y)) {
				return item;
			}
		}
		return null;
	}

	private void onDispose(Event event) {
		removeListener(SWT.Dispose, listener);
		notifyListeners(SWT.Dispose, event);
		event.type = SWT.None;
		/*
			 * Usually when an item is disposed, destroyItem will change the size of the items array,
			 * reset the bounds of all the tabs and manage the widget associated with the tab.
			 * Since the whole folder is being disposed, this is not necessary.  For speed
			 * the inDispose flag is used to skip over this part of the item dispose.
			 */
		inDispose = true;

		if (showMenu != null && !showMenu.isDisposed()) {
			showMenu.dispose();
			showMenu = null;
		}
		for (QTabFolderItem item : items) {
			item.dispose();
		}

		selectionBackground = null;
		selectionForeground = null;
	}

	private void onDragDetect(Event event) {
		boolean consume = false;
		if (chevronRect.contains(event.x, event.y)) {
			consume = true;
		}
		else {
			for (QTabFolderItem item : items) {
				if (item.getCloseRect().contains(event.x, event.y)) {
					consume = true;
					break;
				}
			}
		}
		if (consume) {
			event.type = SWT.None;
		}
	}

	private void mouseDoubleClick(Event event) {
		if (event.button != 1 ||
				(event.stateMask & SWT.BUTTON2) != 0 ||
				(event.stateMask & SWT.BUTTON3) != 0) {
			return;
		}
		final Event e = new Event();
		e.item = getItem(event.x, event.y);
		if (e.item != null) {
			notifyListeners(SWT.DefaultSelection, e);
		}
	}

	private void mouseEnter() {
		setToolTipText(null);
	}

	private void mouseExit() {
		if (chevronImageState != SWT.NONE) {
			chevronImageState = SWT.NONE;
			redraw(chevronRect.x, chevronRect.y, chevronRect.width, chevronRect.height, false);
		}

		for (int i = 0; i < items.size(); i++) {
			final QTabFolderItem item = items.get(i);
			final Rectangle closeRect = item.getCloseRect();
			if (i != selectedIndex && item.getCloseImageState() != SWT.BACKGROUND) {
				item.setCloseImageState(SWT.BACKGROUND);
				redraw(closeRect.x, closeRect.y, closeRect.width, closeRect.height, false);
			}
			if ((item.getState() & SWT.HOT) != 0) {
				item.setState(item.getState() & ~SWT.HOT);
				redraw(item.getX(), item.getY(), item.getWidth(), item.getHeight(), false);
			}
			if (i == selectedIndex && item.getCloseImageState() != SWT.NONE) {
				item.setCloseImageState(SWT.NONE);
				redraw(closeRect.x, closeRect.y, closeRect.width, closeRect.height, false);
			}
		}
	}

	private void mouseDown(int x, int y) {
		if (chevronRect.contains(x, y)) {
			if (chevronImageState != SWT.HOT) {
				chevronImageState = SWT.HOT;
			}
			else {
				chevronImageState = SWT.SELECTED;
			}
			redraw(chevronRect.x, chevronRect.y, chevronRect.width, chevronRect.height, false);
			update();
			return;
		}

		final QTabFolderItem clickedItem = getClickedItem(x, y);
		if (clickedItem == null) {
			return;
		}

		final Rectangle closeRect = clickedItem.getCloseRect();
		if (closeRect.contains(x, y)) {
			clickedItem.setCloseImageState(SWT.SELECTED);
			redraw(closeRect.x, closeRect.y, closeRect.width, closeRect.height, false);
			update();
			return;
		}

		if (clickedItem.isShowing()) {
			setSelection(indexOf(clickedItem), true);
			setFocus(clickedItem.getControl());
		}
	}

	private void mouseMove(int x, int y) {
		_setToolTipText(x, y);

		boolean chevron = false;
		if (chevronRect.contains(x, y)) {
			chevron = true;
			if (chevronImageState != SWT.SELECTED && chevronImageState != SWT.HOT) {
				chevronImageState = SWT.HOT;
				redraw(chevronRect.x, chevronRect.y, chevronRect.width, chevronRect.height, false);
			}
		}
		if (chevronImageState != SWT.NONE && !chevron) {
			chevronImageState = SWT.NONE;
			redraw(chevronRect.x, chevronRect.y, chevronRect.width, chevronRect.height, false);
		}

		for (int i = 0; i < items.size(); i++) {
			final QTabFolderItem item = getItem(i);
			final Rectangle closeRect = item.getCloseRect();
			if (item.getBounds().contains(x, y)) {
				final int closeImageState = item.getCloseImageState();
				if (closeRect.contains(x, y)) {
					if (closeImageState != SWT.SELECTED && closeImageState != SWT.HOT) {
						item.setCloseImageState(SWT.HOT);
						redraw(closeRect.x, closeRect.y, closeRect.width, closeRect.height, false);
					}
				}
				else {
					item.setCloseImageState(SWT.NONE);
					redraw(closeRect.x, closeRect.y, closeRect.width, closeRect.height, false);
				}

				if ((item.getState() & SWT.HOT) == 0) {
					item.setState(item.getState() | SWT.HOT);
					redraw(item.getX(), item.getY(), item.getWidth(), item.getHeight(), false);
				}
			}
			else {
				if (i != selectedIndex && item.getCloseImageState() != SWT.BACKGROUND) {
					item.setCloseImageState(SWT.BACKGROUND);
					redraw(closeRect.x, closeRect.y, closeRect.width, closeRect.height, false);
				}
				if ((item.getState() & SWT.HOT) != 0) {
					item.setState(item.getState() & ~SWT.HOT);
					redraw(item.getX(), item.getY(), item.getWidth(), item.getHeight(), false);
				}
				if (i == selectedIndex && item.getCloseImageState() != SWT.NONE) {
					item.setCloseImageState(SWT.NONE);
					redraw(closeRect.x, closeRect.y, closeRect.width, closeRect.height, false);
				}
			}
		}
	}

	private void mouseUp(int x, int y, int eventTime) {
		if (chevronRect.contains(x, y)) {
			if (chevronImageState != SWT.SELECTED) {
				return;
			}

			final QTabFolderEvent e = new QTabFolderEvent(this);
			e.widget = this;
			e.time = eventTime;
			e.x = chevronRect.x;
			e.y = chevronRect.y;
			e.width = chevronRect.width;
			e.height = chevronRect.height;
			e.doit = true;
			for (QTabFolderListener folderListener : folderListeners) {
				folderListener.showList(e);
			}
			if (e.doit && !isDisposed()) {
				showList(chevronRect);
			}
			return;
		}

		final QTabFolderItem clickedItem = getClickedItem(x, y);
		if (clickedItem == null) {
			return;
		}

		final Rectangle clickedItemCloseRect = clickedItem.getCloseRect();
		if (!clickedItemCloseRect.contains(x, y)) {
			return;
		}

		final boolean selected = clickedItem.getCloseImageState() == SWT.SELECTED;
		clickedItem.setCloseImageState(SWT.HOT);
		redraw(clickedItemCloseRect.x, clickedItemCloseRect.y, clickedItemCloseRect.width, clickedItemCloseRect.height, false);
		if (!selected) {
			return;
		}

		final QTabFolderEvent e = new QTabFolderEvent(this);
		e.widget = this;
		e.time = eventTime;
		e.item = clickedItem;
		e.doit = true;
		for (QTabFolderListener listener : folderListeners) {
			listener.close(e);
		}

		if (e.doit) {
			clickedItem.dispose();
		}

		if (isDisposed() || !clickedItem.isDisposed()) {
			return;
		}

		final Display display = getDisplay();
		Point pt = display.getCursorLocation();
		pt = display.map(null, this, pt.x, pt.y);
		final QTabFolderItem nextItem = getItem(pt.x, pt.y);
		if (nextItem == null) {
			return;
		}

		final Rectangle nextItemCloseRect = nextItem.getCloseRect();
		final int nextCloseImageState = nextItem.getCloseImageState();
		if (nextItemCloseRect.contains(pt)) {
			if (nextCloseImageState != SWT.SELECTED && nextCloseImageState != SWT.HOT) {
				nextItem.setCloseImageState(SWT.HOT);
				redraw(nextItemCloseRect.x, nextItemCloseRect.y, nextItemCloseRect.width, nextItemCloseRect.height, false);
			}
		}
		else {
			if (nextCloseImageState != SWT.NONE) {
				nextItem.setCloseImageState(SWT.NONE);
				redraw(nextItemCloseRect.x, nextItemCloseRect.y, nextItemCloseRect.width, nextItemCloseRect.height, false);
			}
		}
	}

	private void onPaint(Event event) {
		if (event.gc.isDisposed() || inDispose) {
			return;
		}

		final Font font = getFont();
		if (oldFont == null || !oldFont.equals(font)) {
			// handle case where  default font changes
			oldFont = font;
			if (!updateTabHeight(false)) {
				updateItems();
				redraw();
				return;
			}
		}

		final GC gc = event.gc;
		final Font gcFont = gc.getFont();
		final Color gcBackground = gc.getBackground();
		final Color gcForeground = gc.getForeground();

		renderer.drawBody(getSize(), gc);

		gc.setFont(gcFont);
		gc.setForeground(gcForeground);
		gc.setBackground(gcBackground);

		renderer.drawHeader(gc);

		gc.setFont(gcFont);
		gc.setForeground(gcForeground);
		gc.setBackground(gcBackground);

		for (int i = 0; i < items.size(); i++) {
			final QTabFolderItem item = getItem(i);
			final Rectangle itemBounds = item.getBounds();
			if (i != selectedIndex && event.getBounds().intersects(itemBounds)) {
				renderer.drawTab(item, gc);
			}
		}

		gc.setFont(gcFont);
		gc.setForeground(gcForeground);
		gc.setBackground(gcBackground);

		if (selectedIndex != -1) {
			final QTabFolderItem selectedItem = getItem(selectedIndex);
			renderer.drawTab(selectedItem, gc);
		}

		gc.setFont(gcFont);
		gc.setForeground(gcForeground);
		gc.setBackground(gcBackground);

		renderer.drawChevron(chevronImageState, chevronRect, gc);

		gc.setFont(gcFont);
		gc.setForeground(gcForeground);
		gc.setBackground(gcBackground);
	}

	private void onResize() {
		if (updateItems()) {
			redrawTabs();
		}

		final Point size = getSize();
		if (oldSize == null) {
			redraw();
		}
		else {
			int x1 = Math.min(size.x, oldSize.x);
			final Rectangle trim = renderer.computeBodyTrim(0, 0, 0, 0);
			if (size.x != oldSize.x) {
				x1 -= trim.width + trim.x + 2;
			}
			int y1 = Math.min(size.y, oldSize.y);
			if (size.y != oldSize.y) {
				y1 -= trim.height + trim.y;
			}
			final int x2 = Math.max(size.x, oldSize.x);
			final int y2 = Math.max(size.y, oldSize.y);
			redraw(0, y1, x2, y2 - y1, false);
			redraw(x1, 0, x2 - x1, y2, false);
		}
		oldSize = size;
	}

	private void setButtonBounds() {
		final Point size = getSize();
		int oldX;
		int oldY;
		int oldWidth;
		int oldHeight;
		final Rectangle trim = renderer.computeBorderTrim();
		final int borderRight = trim.width + trim.x;
		final int borderTop = -trim.y;

		// top right control
		oldX = topRightRect.x;
		oldY = topRightRect.y;
		oldWidth = topRightRect.width;
		oldHeight = topRightRect.height;
		topRightRect.x = 0;
		topRightRect.y = 0;
		topRightRect.width = 0;
		topRightRect.height = 0;
		if (topRight != null) {
			final Point topRightSize = topRight.computeSize(SWT.DEFAULT, tabHeight, false);
			final int rightEdge = size.x - borderRight - QTabFolderLayout.SPACE_RIGHT_BESIDE_TOPRIGHT;
			topRightRect.x = rightEdge - topRightSize.x;
			topRightRect.width = topRightSize.x;
			topRightRect.y = borderTop + 1;
			topRightRect.height = tabHeight - 1;
			topRight.setBounds(topRightRect);
		}
		if (oldX != topRightRect.x || oldWidth != topRightRect.width ||
				oldY != topRightRect.y || oldHeight != topRightRect.height) {
			final int left = Math.min(oldX, topRightRect.x);
			final int right = Math.max(oldX + oldWidth, topRightRect.x + topRightRect.width);
			final int top = borderTop + 1;
			redraw(left, top, right - left, tabHeight, false);
		}

		// chevron button
		oldX = chevronRect.x;
		oldY = chevronRect.y;
		oldWidth = chevronRect.width;
		oldHeight = chevronRect.height;
		chevronRect.x = 0;
		chevronRect.y = 0;
		chevronRect.height = 0;
		chevronRect.width = 0;
		if (showChevron) {
			final Point chevronSize = renderer.getChevronButtonSize();
			chevronRect.width = chevronSize.x;
			chevronRect.height = chevronSize.y;

			int i = 0;
			int lastIndex = -1;
			while (i < priority.length && getItem(priority[i]).isShowing()) {
				lastIndex = Math.max(lastIndex, priority[i++]);
			}

			if (lastIndex == -1) {
				lastIndex = firstIndex;
			}

			final QTabFolderItem lastItem = getItem(lastIndex);
			final int w = lastItem.getX() + lastItem.getWidth() + 3;
			chevronRect.x = Math.min(w, getRightItemEdge());
			chevronRect.y = borderTop + (tabHeight - chevronRect.height) / 2;
		}

		if (oldX != chevronRect.x || oldWidth != chevronRect.width ||
				oldY != chevronRect.y || oldHeight != chevronRect.height) {
			final int left = Math.min(oldX, chevronRect.x);
			final int right = Math.max(oldX + oldWidth, chevronRect.x + chevronRect.width);
			final int top = borderTop + 1;
			redraw(left, top, right - left, tabHeight, false);
		}
	}

	private boolean setItemLocation() {
		boolean changed = false;
		if (items.size() == 0) {
			return false;
		}
		final Rectangle trim = renderer.computeBorderTrim();
		final int borderLeft = -trim.x;
		final int borderTop = -trim.y;
		final Point closeButtonSize = renderer.getCloseButtonSize();
		final int rightItemEdge = getRightItemEdge();
		final int maxWidth = rightItemEdge - borderLeft;
		int width = 0;
		for (int i = 0; i < priority.length; i++) {
			final QTabFolderItem item = getItem(priority[i]);
			width += item.getWidth();
			if (i == 0) {
				item.setShowing(true);
			}
			else {
				item.setShowing(item.getWidth() > 0 && width <= maxWidth);
			}
		}
		int x = 0;
		final int defaultX = getDisplay().getBounds().width + 10; // off screen
		firstIndex = items.size() - 1;
		for (int i = 0; i < items.size(); i++) {
			final QTabFolderItem item = getItem(i);
			if (!item.isShowing()) {
				if (item.getX() != defaultX) {
					changed = true;
				}
				item.setX(defaultX);
			}
			else {
				firstIndex = Math.min(firstIndex, i);
				if (item.getX() != x || item.getY() != borderTop) {
					changed = true;
				}
				item.setX(x);
				item.setY(borderTop);
				final Rectangle edgeTrim = renderer.computeTabTrim(0, 0);
				final Rectangle closeRect = item.getCloseRect();
				closeRect.x = item.getX() + item.getWidth() - (edgeTrim.width + edgeTrim.x) - closeButtonSize.x;
				closeRect.y = borderTop + (tabHeight - closeButtonSize.y) / 2;
				x = x + item.getWidth();
			}
		}
		return changed;
	}

	private boolean setItemSize(GC gc) {
		boolean changed = false;
		if (isDisposed()) {
			return changed;
		}
		final Point size = getSize();
		if (size.x <= 0 || size.y <= 0) {
			return changed;
		}

		final Rectangle trim = renderer.computeBorderTrim();
		final int borderRight = trim.width + trim.x;
		final int borderLeft = -trim.x;

		showChevron = false;

		if (items.size() == 0) {
			return changed;
		}

		final int[] widths;
		int tabAreaWidth = size.x - borderLeft - borderRight - 3;
		if (topRight != null) {
			final Point rightSize = topRight.computeSize(SWT.DEFAULT, SWT.DEFAULT, false);
			tabAreaWidth -= rightSize.x + QTabFolderLayout.SPACE_RIGHT_BESIDE_TOPRIGHT;
		}
		tabAreaWidth = Math.max(0, tabAreaWidth);

		// First, try the minimum tab size at full compression.
		int minWidth = 0;
		final int[] minWidths = new int[items.size()];
		for (int index : priority) {
			final QTabFolderItem item = getItem(index);
			minWidths[index] = renderer.getMinTabWidth(item, gc);
			minWidth += minWidths[index];
			if (minWidth > tabAreaWidth) {
				break;
			}
		}
		if (minWidth > tabAreaWidth) {
			// full compression required and a chevron
			showChevron = items.size() > 1;
			if (showChevron) {
				tabAreaWidth -= renderer.getChevronButtonSize().x;
			}
			widths = minWidths;
			final int index = selectedIndex != -1 ? selectedIndex : 0;
			if (tabAreaWidth < widths[index]) {
				widths[index] = Math.max(0, tabAreaWidth);
			}
		}
		else {
			int maxWidth = 0;
			final int[] maxWidths = new int[items.size()];
			for (int i = 0; i < items.size(); i++) {
				final QTabFolderItem item = getItem(i);
				maxWidths[i] = renderer.getTabWidth(item, gc);
				maxWidth += maxWidths[i];
			}
			if (maxWidth <= tabAreaWidth) {
				// no compression required
				widths = maxWidths;
			}
			else {
				// determine compression for each item
				int extra = (tabAreaWidth - minWidth) / items.size();
				while (true) {
					int large = 0;
					int totalWidth = 0;
					for (int i = 0; i < items.size(); i++) {
						if (maxWidths[i] > minWidths[i] + extra) {
							totalWidth += minWidths[i] + extra;
							large++;
						}
						else {
							totalWidth += maxWidths[i];
						}
					}
					if (totalWidth >= tabAreaWidth) {
						extra--;
						break;
					}
					if (large == 0 || tabAreaWidth - totalWidth < large) {
						break;
					}
					extra++;
				}
				widths = new int[items.size()];
				for (int i = 0; i < items.size(); i++) {
					widths[i] = Math.min(maxWidths[i], minWidths[i] + extra);
				}
			}
		}

		for (int i = 0; i < items.size(); i++) {
			final QTabFolderItem tab = getItem(i);
			final int width = widths[i];
			if (tab.getHeight() != tabHeight || tab.getWidth() != width) {
				changed = true;
				tab.setShortenedText(null);
				tab.setShortenedTextWidth(0);
				tab.setHeight(tabHeight);
				tab.setWidth(width);
				final Rectangle closeRect = tab.getCloseRect();
				closeRect.width = closeRect.height = 0;
				if (showClose || tab.isShowClose()) {
					if (i == selectedIndex || showUnselectedClose) {
						final Point closeSize = renderer.getCloseButtonSize();
						closeRect.width = closeSize.x;
						closeRect.height = closeSize.y;
					}
				}
			}
		}
		return changed;
	}

	/**
	 * Set the selection to the tab at the specified index.
	 *
	 * @param index the index of the tab item to be selected
	 * @throws SWTException <ul>
	 *                      <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
	 *                      <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
	 *                      </ul>
	 */
	private void setSelection(int index) {
		checkWidget();
		if (index < 0 || index >= items.size()) {
			return;
		}
		final QTabFolderItem selection = getItem(index);
		if (selectedIndex == index) {
			showItem(selection);
			return;
		}

		final int oldIndex = selectedIndex;
		selectedIndex = index;
		if (oldIndex != -1) {
			final QTabFolderItem oldIndexItem = getItem(oldIndex);
			oldIndexItem.setCloseImageState(SWT.BACKGROUND);
			oldIndexItem.setState(oldIndexItem.getState() & ~SWT.SELECTED);
		}
		selection.setCloseImageState(SWT.NONE);
		selection.setShowing(false);
		selection.setState(selection.getState() | SWT.SELECTED);

		final Control newControl = selection.getControl();
		Control oldControl = null;
		if (oldIndex != -1) {
			final QTabFolderItem oldIndexItem = getItem(oldIndex);
			oldControl = oldIndexItem.getControl();
		}

		if (newControl != oldControl) {
			newControl.setBounds(getClientArea());
			newControl.setVisible(true);

			if (oldControl != null) {
				oldControl.setVisible(false);
			}
		}
		showItem(selection);
		redraw();
	}

	private void setSelection(int index, boolean notify) {
		final int oldSelectedIndex = selectedIndex;
		setSelection(index);
		if (notify && selectedIndex != oldSelectedIndex && selectedIndex != -1) {
			final Event event = new Event();
			event.item = getItem(selectedIndex);
			notifyListeners(SWT.Selection, event);
		}
	}

	private void showItem(QTabFolderItem item) {
		checkWidget();
		if (item == null) {
			SWT.error(SWT.ERROR_NULL_ARGUMENT);
		}

		if (item.isDisposed()) {
			SWT.error(SWT.ERROR_INVALID_ARGUMENT);
		}

		final int index = indexOf(item);
		if (index == -1) {
			SWT.error(SWT.ERROR_INVALID_ARGUMENT);
		}
		int idx = -1;
		for (int i = 0; i < priority.length; i++) {
			if (priority[i] == index) {
				idx = i;
				break;
			}
		}
		if (mru) {
			// move to front of mru order
			final int[] newPriority = new int[priority.length];
			System.arraycopy(priority, 0, newPriority, 1, idx);
			System.arraycopy(priority, idx + 1, newPriority, idx + 1, priority.length - idx - 1);
			newPriority[0] = index;
			priority = newPriority;
		}
		if (item.isShowing()) {
			return;
		}
		updateItems(index);
		redrawTabs();
	}

	private void showList(Rectangle rect) {
		if (items.isEmpty() || !showChevron) {
			return;
		}
		if (showMenu == null || showMenu.isDisposed()) {
			showMenu = new Menu(getShell(), getStyle() & (SWT.LEFT_TO_RIGHT | SWT.RIGHT_TO_LEFT));
		}
		else {
			final MenuItem[] items = showMenu.getItems();
			for (MenuItem item : items) {
				item.dispose();
			}
		}
		final String id = "QTabFolder_showList_Index"; //$NON-NLS-1$
		for (QTabFolderItem tab : items) {
			if (tab.isShowing()) {
				continue;
			}
			final MenuItem item = new MenuItem(showMenu, SWT.NONE);
			item.setText(tab.getText());
			item.setImage(tab.getImage());
			item.setData(id, tab);
			item.addSelectionListener(new SelectionAdapter() {
				@Override
				public void widgetSelected(SelectionEvent e) {
					final MenuItem menuItem = (MenuItem)e.widget;
					final int index = indexOf((QTabFolderItem)menuItem.getData(id));
					setSelection(index, true);
				}
			});
		}
		final int x = rect.x;
		final int y = rect.y + rect.height;
		final Point location = getDisplay().map(this, null, x, y);
		showMenu.setLocation(location.x, location.y);
		showMenu.setVisible(true);
	}

	private static int checkStyle(Composite parent, int style) {
		final int mask = SWT.CLOSE | SWT.FLAT | SWT.LEFT_TO_RIGHT | SWT.RIGHT_TO_LEFT;
		style = style & mask;
		// reduce the flash by not redrawing the entire area on a Resize event
		style |= SWT.NO_REDRAW_RESIZE;

		//TEMPORARY CODE
		/*
			 * In Right To Left orientation on Windows, all GC calls that use a brush are drawing
			 * offset by one pixel.  This results in some parts of the QTabFolder not drawing correctly.
			 * To alleviate some of the appearance problems, allow the OS to draw the background.
			 * This does not draw correctly but the result is less obviously wrong.
			 */
		if ((style & SWT.RIGHT_TO_LEFT) != 0) {
			return style;
		}
		if ((parent.getStyle() & SWT.MIRRORED) != 0 && (style & SWT.LEFT_TO_RIGHT) == 0) {
			return style;
		}

		style |= SWT.NO_FOCUS;
		return style | SWT.DOUBLE_BUFFERED;
	}
}
