/*******************************************************************************
 * Copyright (c) 2005, 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Hannes Erven <hannes@erven.at> - Bug 293841 - [FieldAssist] NumLock keyDown event should not close the proposal popup [with patch]
 *     Syntevo GmbH    - major refactoring
 *******************************************************************************/
package com.syntevo.q.swt;

import java.util.*;
import java.util.List;

import org.eclipse.swt.*;
import org.eclipse.swt.widgets.*;

/**
 * Refactored org.eclipse.jface.fieldassist.ContentProposalAdapter
 */
public abstract class QContentProposalAdapter {

	// Abstract ===============================================================

	protected abstract void proposalAccepted(QContentProposal proposal);

	protected abstract boolean supportsAutoActivation();

	protected abstract boolean isAutoActivateChar(char chr);

	// Fields =================================================================

	private final QContentProposalProvider proposalProvider;
	private final Control control;
	private final int triggerKeyStroke;
	private final QControlContentAdapter controlContentAdapter;

	private ContentProposalPopup popup;
	private boolean watchModify;

	// Setup ==================================================================

	protected QContentProposalAdapter(QControlContentAdapter controlContentAdapter, QContentProposalProvider proposalProvider, int triggerKeyStroke) {
		this.controlContentAdapter = controlContentAdapter;
		this.proposalProvider = proposalProvider;
		this.triggerKeyStroke = triggerKeyStroke;

		control = controlContentAdapter.getControl();

		final Listener controlListener = new Listener() {
			@Override
			public void handleEvent(Event e) {
				handleControlEvent(e);
			}
		};
		control.addListener(SWT.KeyDown, controlListener);
		control.addListener(SWT.Traverse, controlListener);
		control.addListener(SWT.Modify, controlListener);
	}

	// Accessing ==============================================================

	protected void openProposalPopup() {
		openProposalPopup(false);
	}

	public List<? extends QContentProposal> getProposals() {
		if (control.isDisposed()) {
			return Collections.emptyList();
		}

		final int position = controlContentAdapter.getCaretPosition();
		final String contents = controlContentAdapter.getText();
		return proposalProvider.getProposals(contents, position);
	}

	public void setWatchModify(boolean watchModify) {
		this.watchModify = watchModify;
	}

	protected void initializeItem(QContentProposal proposal, TableItem item) {
		final String label = proposal.getLabel();
		item.setText(label != null ? label : proposal.getContent());
	}

	// Utils ==================================================================

	private void closeProposalPopup() {
		if (popup != null) {
			popup.close();
		}
	}

	private void openProposalPopup(boolean autoActivated) {
		if (control.isDisposed()) {
			return;
		}
		if (popup != null) {
			return;
		}

		final List<? extends QContentProposal> proposals = getProposals();
		if (proposals.size() > 0) {
			popup = new ContentProposalPopup(this, control, controlContentAdapter, proposals);
			popup.getShell().addListener(SWT.Dispose, new Listener() {
				@Override
				public void handleEvent(Event event) {
					popup = null;
				}
			});
		}
		else if (!autoActivated) {
			control.getDisplay().beep();
		}
	}

	private void autoActivate() {
		control.getDisplay().asyncExec(new Runnable() {
			@Override
			public void run() {
				if (control.isDisposed()) {
					return;
				}
				openProposalPopup(true);
			}
		});
	}

	private boolean isControlContentEmpty() {
		return controlContentAdapter.getText().length() == 0;
	}

	private boolean shouldPopupRemainOpen() {
		// If we always autoactivate or never autoactivate, it should remain open
		if (!supportsAutoActivation()) {
			return true;
		}

		final String content = controlContentAdapter.getText();
		for (int i = 0; i < content.length(); i++) {
			if (isAutoActivateChar(content.charAt(i))) {
				return true;
			}
		}
		return false;
	}

	private boolean allowsAutoActivate() {
		if (supportsAutoActivation()) {
			return true;
		}

		return triggerKeyStroke == 0;
	}

	private void handleControlEvent(Event e) {
		switch (e.type) {
		case SWT.Traverse:
		case SWT.KeyDown:
			if (!e.doit) {
				return;
			}

			// If the popup is open, it gets first shot at the
			// keystroke and should set the doit flags appropriately.
			if (popup != null) {
				popup.handleTraverseAndKeyDownEvent(e);
				// See https://bugs.eclipse.org/bugs/show_bug.cgi?id=192633
				// If the popup is open and this is a valid character, we
				// want to watch for the modified text.
				if (e.character != 0) {
					setWatchModify(true);
				}

				return;
			}

			// We were only listening to traverse events for the popup
			if (e.type == SWT.Traverse) {
				return;
			}

			// The popup is not open. We are looking at keydown events
			// for a trigger to open the popup.
			if (triggerKeyStroke != 0) {
				// Either there are no modifiers for the trigger and we
				// check the character field...
				final int modifierKeys = triggerKeyStroke & SWT.MODIFIER_MASK;
				final int naturalKey = triggerKeyStroke - modifierKeys;
				if ((modifierKeys == 0 && naturalKey == e.character)
				    ||
				    // ...or there are modifiers, in which case the
				    // keycode and state must match
				    (naturalKey == e.keyCode && ((modifierKeys & e.stateMask) == modifierKeys))) {
					// We never propagate the keystroke for an explicit
					// keystroke invocation of the popup
					e.doit = false;
					openProposalPopup(false);
					return;
				}
			}
			/*
			 * The triggering keystroke was not invoked. If a character
			 * was typed, compare it to the autoactivation characters.
			 */
			if (e.character != 0) {
				if (supportsAutoActivation()) {
					if (isAutoActivateChar(e.character)) {
						autoActivate();
					}
					else {
						// No autoactivation occurred, so record the key
						// down as a means to interrupt any
						// autoactivation that is pending due to
						// autoactivation delay.
						// watch the modify so we can close the popup in
						// cases where there is no longer a trigger
						// character in the content
						setWatchModify(true);
					}
				}
				else {
					// The autoactivate string is null. If the trigger
					// is also null, we want to act on any modification
					// to the content. Set a flag so we'll catch this
					// in the modify event.
					if (triggerKeyStroke == 0) {
						setWatchModify(true);
					}
				}
			}
			else {
				// A non-character key has been pressed. Interrupt any
				// autoactivation that is pending due to autoactivation delay.
			}
			break;

		// There are times when we want to monitor content changes
		// rather than individual keystrokes to determine whether
		// the popup should be closed or opened based on the entire
		// content of the control.
		// The watchModify flag ensures that we don't autoactivate if
		// the content change was caused by something other than typing.
		// See https://bugs.eclipse.org/bugs/show_bug.cgi?id=183650
		case SWT.Modify:
			if (allowsAutoActivate() && watchModify) {
				setWatchModify(false);
				// We are in autoactivation mode, either for specific
				// characters or for all characters. In either case,
				// we should close the proposal popup when there is no
				// content in the control.
				if (isControlContentEmpty()) {
					// see https://bugs.eclipse.org/bugs/show_bug.cgi?id=192633
					closeProposalPopup();
				}
				else {
					// See https://bugs.eclipse.org/bugs/show_bug.cgi?id=147377
					// Given that we will close the popup when there are
					// no valid proposals, we must consider reopening it on any
					// content change when there are no particular autoActivation
					// characters
					if (supportsAutoActivation()) {
						// Autoactivation characters are defined, but this
						// modify event does not involve one of them.  See
						// if any of the autoactivation characters are left
						// in the content and close the popup if none remain.
						if (!shouldPopupRemainOpen()) {
							closeProposalPopup();
						}
					}
					else {
						autoActivate();
					}
				}
			}
			break;
		default:
			break;
		}
	}
}
