// Copyright (C) 2014 David Sugar, Tycho Softworks.
// Copyright (C) 2015 Cherokees of Idaho.
//
// This file is part of siot.
//
// Siot is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 of the License, or
// (at your option) any later version.
//
// Siot is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with siot. If not, see <http://www.gnu.org/licenses/>.

#ifndef SIORT_H
#define SIORT_H
#include <ucommon/ucommon.h>

namespace ucommon {
/**
 * Low level console i/o related functions.  This is used for some basic
 * console operations that might be used, for example, to do a crude
 * serial terminal.  In particular, siot will be using it for this
 * purpose.
 * @author David Sugar <dyfet@gnutelephony.org>
 */
class __EXPORT Console
{
private:
    friend class Serial;

    __DELETE_DEFAULTS(Console);

    /**
     * Internal unicode processing.
     * @param buffer to evaluate
     * @return true if contains unicode
     */
    static bool unicode(char *buffer);

public:
    /**
     * Setup console for use with getch.  This may disable character
     * echo and set various other termios properties as needed.  Should
     * be called before any console operations.
     */
    static void setup(void);

    /**
     * Restore terminal to original state.  This should be called before
     * exit, and maybe in an atexit() handler.
     */
    static void cleanup(void);

    /**
     * Fetch input from console.  This will normally fetch a single
     * character code, but also can parse console control escape
     * sequences, so that they can be fetched and sent in a single
     * operation.  This will also by default fetch a complete unicode
     * utf-8 encoded codepoint as well.  At some point we may add
     * a flag to set or disable unicode parsing.
     * @param buffer to save result
     * @param size of result buffer.  Ideally 16 is a good size.
     * @return number of bytes returned from console into buffer.
     */
    static int getch(char *buffer, size_t size);
};

/**
 * Serial device manager interface.  This offers abstract base classes which
 * are used to manipulate underlying serial devices and hardware.  The actual
 * objects are created by factory methods in derived classes.  Some of these
 * things are used by clients, various servers and services, and utilities.
 * @author David Sugar <dyfet@gnutelephony.org>
 */
class __EXPORT Serial
{
private:
    __DELETE_COPY(Serial);

protected:
    int error;
    char prior;

    Serial();

public:
    virtual ~Serial();

    inline int err(void)
        {return error;}

    virtual operator bool() = 0;
    virtual bool operator!() = 0;
    virtual void restore(void) = 0;
    virtual bool set(const char *format) = 0;
    virtual void dtr(timeout_t toggle_time) = 0;
    virtual size_t get(void *addr, size_t len) = 0;
    virtual size_t put(void *addr, size_t len) = 0;
    virtual void clear(void) = 0;
    virtual bool flush(timeout_t timeout) = 0;
    virtual bool wait(timeout_t timeout) = 0;
    virtual void text(char nl1 = 0x13, char nl2 = 0x10) = 0;
    virtual void bin(size_t size, timeout_t timeout = 0) = 0;
    virtual void sync(void) = 0;
    virtual fd_t handle(void) = 0;

    inline bool is_open(void)
        {return operator bool();}

    static Serial *create(const char *name, bool exclusive = true);

    static stringlist_t *list(void);
};

typedef Serial  *serial_t;

} // namespace ucommon
#endif
