/****************************************************************************/
/// @file    GNEPoly.cpp
/// @author  Jakob Erdmann
/// @date    Sept 2012
/// @version $Id: GNEPoly.cpp 25658 2017-08-18 11:52:28Z palcraft $
///
// A class for visualizing and editing POIS in netedit (adapted from
// GUIPolygon and NLHandler)
/****************************************************************************/
// SUMO, Simulation of Urban MObility; see http://sumo.dlr.de/
// Copyright (C) 2001-2017 DLR (http://www.dlr.de/) and contributors
/****************************************************************************/
//
//   This file is part of SUMO.
//   SUMO is free software: you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation, either version 3 of the License, or
//   (at your option) any later version.
//
/****************************************************************************/


// ===========================================================================
// included modules
// ===========================================================================
#ifdef _MSC_VER
#include <windows_config.h>
#else
#include <config.h>
#endif

#include <string>
#include <utility>
#include <foreign/polyfonts/polyfonts.h>
#include <utils/foxtools/MFXImageHelper.h>
#include <utils/geom/Position.h>
#include <utils/geom/GeomConvHelper.h>
#include <utils/geom/GeoConvHelper.h>
#include <utils/common/MsgHandler.h>
#include <utils/xml/XMLSubSys.h>
#include <utils/gui/windows/GUIAppEnum.h>
#include <utils/gui/windows/GUIMainWindow.h>
#include <utils/gui/windows/GUISUMOAbstractView.h>
#include <utils/gui/globjects/GUIGLObjectPopupMenu.h>
#include <utils/gui/div/GUIGlobalSelection.h>
#include <utils/gui/div/GUIParameterTableWindow.h>
#include <utils/gui/div/GLHelper.h>
#include <utils/gui/images/GUITexturesHelper.h>
#include <utils/gui/images/GUIIconSubSys.h>
#include <utils/gui/globjects/GUIGlObjectStorage.h>
#include <utils/gui/globjects/GUIGLObjectPopupMenu.h>
#include <netimport/NIImporter_SUMO.h>
#include <netwrite/NWWriter_SUMO.h>

#include "GNENet.h"
#include "GNEEdge.h"
#include "GNEUndoList.h"
#include "GNEViewNet.h"
#include "GNEChange_Attribute.h"
#include "GNEPoly.h"
#include "GNEJunction.h"


// ===========================================================================
// static members
// ===========================================================================

// ===========================================================================
// method definitions
// ===========================================================================
GNEPoly::GNEPoly(GNENet* net, GNEJunction* junction, const std::string& id, const std::string& type, const PositionVector& shape, bool fill,
                 const RGBColor& color, double layer, double angle, const std::string& imgFile) :
    GUIPolygon(id, type, color, shape, fill, layer, angle, imgFile),
    GNEShape(net, SUMO_TAG_POLY, ICON_LOCATEPOLY),
    myJunction(junction) {
}


GNEPoly::~GNEPoly() {}


void GNEPoly::writeShape(OutputDevice &device) {
    writeXML(device);
}


void 
GNEPoly::moveGeometry(const Position &newPosition) {
    for (auto i : myShape) {
        i.set(i.x() + newPosition.x(), i.y() + newPosition.y());
    }
}


void 
GNEPoly::commitGeometryMoving(const PositionVector& oldShape, GNEUndoList* undoList) {
    undoList->p_begin("moving " + toString(SUMO_ATTR_SHAPE) + " of " + toString(getTag()));
    undoList->p_add(new GNEChange_Attribute(this, SUMO_ATTR_SHAPE, toString(myShape)));
    undoList->p_end();
}


void 
GNEPoly::updateGeometry() {
}


Position 
GNEPoly::getPositionInView() const {
    return getCenteringBoundary().getCenter();
}


const std::string& 
GNEPoly::getParentName() const {
    if(myJunction != NULL) {
        return myJunction->getMicrosimID();
    } else {
        return myNet->getMicrosimID();
    }
}


GUIGLObjectPopupMenu*
GNEPoly::getPopUpMenu(GUIMainWindow& app, GUISUMOAbstractView& parent) {
    GUIGLObjectPopupMenu* ret = GUIPolygon::getPopUpMenu(app, parent);
    new FXMenuSeparator(ret);
    new FXMenuCommand(ret, "Set custom shape (ENTER)", 0, &app, MID_GNE_HOTKEY_ENTER);
    new FXMenuCommand(ret, "Discard custom shape (ESC)", 0, &app, MID_GNE_ABORT);
    new FXMenuCommand(ret, "Simplify Shape\t\tReplace shape with a rectangle", 0, &parent, MID_GNE_SIMPLIFY_SHAPE);
    new FXMenuCommand(ret, "Remove geometry point\t\tRemove the closest geometry point", 0, &parent, MID_GNE_DELETE_GEOMETRY);
    // let the GNEViewNet store the popup position
    (dynamic_cast<GNEViewNet&>(parent)).markPopupPosition();
    return ret;
}


GUIParameterTableWindow* 
GNEPoly::getParameterWindow(GUIMainWindow& app, GUISUMOAbstractView& parent) {
    return GUIPolygon::getParameterWindow(app, parent);
}


Boundary 
GNEPoly::getCenteringBoundary() const {
    return GUIPolygon::getCenteringBoundary();
}


void
GNEPoly::drawGL(const GUIVisualizationSettings& s) const {
    const double hintSize = 0.8;
    GUIPolygon::drawGL(s);
    // draw geometry hints
    if (s.scale * hintSize > 1.) { // check whether it is not too small
        RGBColor current = GLHelper::getColor();
        RGBColor darker = current.changedBrightness(-32);
        GLHelper::setColor(darker);
        glPushName(getGlID());
        for (int i = 0; i < (int)myShape.size() - 1; i++) {
            Position pos = myShape[i];
            glPushMatrix();
            glTranslated(pos.x(), pos.y(), GLO_POLYGON + 0.01);
            GLHelper:: drawFilledCircle(hintSize, 32);
            glPopMatrix();
        }
        glPopName();
    }
}


Position
GNEPoly::changeShapeGeometry(const Position& oldPos, const Position& newPos, bool relative) {
    PositionVector geom = myShape;
    bool changed = GNEEdge::changeGeometry(geom, getMicrosimID(), oldPos, newPos, relative, true);
    if (changed) {
        myShape = geom;
        myNet->refreshElement(this);
        return newPos;
    } else {
        return oldPos;
    }
}


void
GNEPoly::simplifyShape() {
    const Boundary b =  myShape.getBoxBoundary();
    myShape.clear();
    myShape.push_back(Position(b.xmin(), b.ymin()));
    myShape.push_back(Position(b.xmin(), b.ymax()));
    myShape.push_back(Position(b.xmax(), b.ymax()));
    myShape.push_back(Position(b.xmax(), b.ymin()));
    myShape.push_back(myShape[0]);
}


void
GNEPoly::deleteGeometryNear(const Position& pos) {
    if (myShape.size() <= 3) {
        return;
    }
    int index = myShape.indexOfClosest(pos);
    if ((index == 0 || index == (int)myShape.size() - 1) && myShape.front() == myShape.back()) {
        myShape.erase(myShape.begin());
        myShape.erase(myShape.end() - 1);
        myShape.push_back(myShape.front());
    } else {
        myShape.erase(myShape.begin() + index);
    }
}


GNEJunction* 
GNEPoly::getEditedJunction() const {
    return myJunction;
}


std::string
GNEPoly::getAttribute(SumoXMLAttr key) const {
    switch (key) {
        case SUMO_ATTR_ID:
            return myID;
        case SUMO_ATTR_SHAPE:
            return toString(myShape);
        case SUMO_ATTR_COLOR:
            return toString(myColor);
        case SUMO_ATTR_FILL:
            return toString(myFill);
        case SUMO_ATTR_LAYER:
            return toString(myLayer);
        case SUMO_ATTR_TYPE:
            return myType;
        case SUMO_ATTR_IMGFILE:
            return myImgFile;
        case GNE_ATTR_BLOCK_MOVEMENT:
            return toString(myBlockMovement);
        case GNE_ATTR_BLOCK_SHAPE:
            return toString(myBlockShape);
        default:
            throw InvalidArgument(toString(getTag()) + " doesn't have an attribute of type '" + toString(key) + "'");
    }
}


void
GNEPoly::setAttribute(SumoXMLAttr key, const std::string& value, GNEUndoList* undoList) {
    if (value == getAttribute(key)) {
        return; //avoid needless changes, later logic relies on the fact that attributes have changed
    }
    switch (key) {
        case SUMO_ATTR_ID:
        case SUMO_ATTR_SHAPE:
        case SUMO_ATTR_COLOR:
        case SUMO_ATTR_FILL:
        case SUMO_ATTR_LAYER:
        case SUMO_ATTR_TYPE:
        case SUMO_ATTR_IMGFILE:
        case GNE_ATTR_BLOCK_MOVEMENT:
        case GNE_ATTR_BLOCK_SHAPE:
            undoList->p_add(new GNEChange_Attribute(this, key, value));
            break;
        default:
            throw InvalidArgument(toString(getTag()) + " doesn't have an attribute of type '" + toString(key) + "'");
    }
}


bool
GNEPoly::isValid(SumoXMLAttr key, const std::string& value) {
    switch (key) {
        case SUMO_ATTR_ID:
            return isValidID(value) && (myNet->retrievePolygon(value, false) == 0);
        case SUMO_ATTR_SHAPE: {
            bool ok = true;
            PositionVector shape = GeomConvHelper::parseShapeReporting(value, "user-supplied position", 0, ok, true);
            return ok;
        }
        case SUMO_ATTR_COLOR:
            return canParse<RGBColor>(value);
        case SUMO_ATTR_FILL:
            return canParse<bool>(value);
        case SUMO_ATTR_LAYER:
            return canParse<double>(value);
        case SUMO_ATTR_TYPE:
            return true;
        case SUMO_ATTR_IMGFILE:
            return isValidFilename(value);
        case GNE_ATTR_BLOCK_MOVEMENT:
            return canParse<bool>(value);
        case GNE_ATTR_BLOCK_SHAPE:
            return canParse<bool>(value);
        default:
            throw InvalidArgument(toString(getTag()) + " doesn't have an attribute of type '" + toString(key) + "'");
    }
}


// ===========================================================================
// private
// ===========================================================================

void
GNEPoly::setAttribute(SumoXMLAttr key, const std::string& value) {
    switch (key) {
        case SUMO_ATTR_ID: {
            std::string oldID = myID;
            myID = value;
            myNet->changePolygonID(this, oldID);
            break;
        }
        case SUMO_ATTR_SHAPE: {
            bool ok = true;
            myShape = GeomConvHelper::parseShapeReporting(value, "netedit-given", 0, ok, true);
            break;
        }
        case SUMO_ATTR_COLOR:
            myColor = parse<RGBColor>(value);
            break;
        case SUMO_ATTR_FILL:
            myFill = parse<bool>(value);
            break;
        case SUMO_ATTR_LAYER:
            myLayer = parse<double>(value);
            break;
        case SUMO_ATTR_TYPE:
            myType = value;
            break;
        case SUMO_ATTR_IMGFILE:
            myType = value;
            break;
        case GNE_ATTR_BLOCK_MOVEMENT:
            myBlockMovement = parse<bool>(value);
            break;
        case GNE_ATTR_BLOCK_SHAPE:
            myBlockShape = parse<bool>(value);
            break;
        default:
            throw InvalidArgument(toString(getTag()) + " doesn't have an attribute of type '" + toString(key) + "'");
    }
}


/****************************************************************************/
