/****************************************************************************/
/// @file    GNECrossing.cpp
/// @author  Jakob Erdmann
/// @date    June 2011
/// @version $Id: GNECrossing.cpp 25270 2017-07-19 11:59:16Z palcraft $
///
// A class for visualizing Inner Lanes (used when editing traffic lights)
/****************************************************************************/
// SUMO, Simulation of Urban MObility; see http://sumo.dlr.de/
// Copyright (C) 2001-2017 DLR (http://www.dlr.de/) and contributors
/****************************************************************************/
//
//   This file is part of SUMO.
//   SUMO is free software: you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation, either version 3 of the License, or
//   (at your option) any later version.
//
/****************************************************************************/


// ===========================================================================
// included modules
// ===========================================================================
#ifdef _MSC_VER
#include <windows_config.h>
#else
#include <config.h>
#endif

#include <string>
#include <iostream>
#include <utility>
#include <time.h>
#include <foreign/polyfonts/polyfonts.h>
#include <utils/foxtools/MFXUtils.h>
#include <utils/geom/PositionVector.h>
#include <utils/gui/windows/GUISUMOAbstractView.h>
#include <utils/common/ToString.h>
#include <utils/gui/windows/GUIAppEnum.h>
#include <utils/gui/div/GUIParameterTableWindow.h>
#include <utils/gui/globjects/GUIGLObjectPopupMenu.h>
#include <utils/gui/div/GUIGlobalSelection.h>
#include <utils/gui/div/GLHelper.h>
#include <utils/gui/globjects/GLIncludes.h>

#include "GNECrossing.h"
#include "GNEJunction.h"
#include "GNEUndoList.h"
#include "GNENet.h"
#include "GNEEdge.h"
#include "GNEViewNet.h"
#include "GNEChange_Attribute.h"

// ===========================================================================
// method definitions
// ===========================================================================
GNECrossing::GNECrossing(GNEJunction* parentJunction, NBNode::Crossing* crossing) :
    GNENetElement(parentJunction->getNet(), crossing->id, GLO_CROSSING, SUMO_TAG_CROSSING, ICON_CROSSING),
    myParentJunction(parentJunction),
    myCrossing(crossing) {
    // Update geometry
    updateGeometry();
}


GNECrossing::~GNECrossing() {}


void
GNECrossing::updateGeometry() {
    // Clear Shape rotations and segments
    myShapeRotations.clear();
    myShapeLengths.clear();
    // only rebuild shape if junction's shape isn't in Buuble mode
    if (myParentJunction->getNBNode()->getShape().size() > 0) {
        // Obtain segments of size and calculate it
        int segments = (int) myCrossing->shape.size() - 1;
        if (segments >= 0) {
            myShapeRotations.reserve(segments);
            myShapeLengths.reserve(segments);
            for (int i = 0; i < segments; ++i) {
                const Position& f = myCrossing->shape[i];
                const Position& s = myCrossing->shape[i + 1];
                myShapeLengths.push_back(f.distanceTo2D(s));
                myShapeRotations.push_back((double) atan2((s.x() - f.x()), (f.y() - s.y())) * (double) 180.0 / (double) PI);
            }
        }
    }
}


GNEJunction*
GNECrossing::getParentJunction() const {
    return myParentJunction;
}


NBNode::Crossing*
GNECrossing::getNBCrossing() const {
    return myCrossing;
}


void
GNECrossing::drawGL(const GUIVisualizationSettings& s) const {
    // only draw if option drawCrossingsAndWalkingareas is enabled and size of shape is greather than 0
    if (s.drawCrossingsAndWalkingareas && myShapeRotations.size() > 0 && myShapeLengths.size() > 0) {
        // push first draw matrix
        glPushMatrix();
        // push name
        glPushName(getGlID());
        // must draw on top of junction
        glTranslated(0, 0, GLO_JUNCTION + 0.1);
        // set color depending of selection and priority
        if (gSelected.isSelected(getType(), getGlID())) {
            glColor3d(0.118, 0.565, 1.000);
        } else if (!myCrossing->valid) {
            glColor3d(1.0, 0.1, 0.1);
        } else if (myCrossing->priority) {
            glColor3d(0.9, 0.9, 0.9);
        } else {
            glColor3d(0.1, 0.1, 0.1);
        }
        // traslate to front
        glTranslated(0, 0, .2);
        // set default values
        double length = 0.5;
        double spacing = 1.0;
        double halfWidth = myCrossing->width * 0.5;
        // push second draw matrix
        glPushMatrix();
        // draw on top of of the white area between the rails
        glTranslated(0, 0, 0.1);
        for (int i = 0; i < (int)myCrossing->shape.size() - 1; ++i) {
            // push three draw matrix
            glPushMatrix();
            // traslete and rotate
            glTranslated(myCrossing->shape[i].x(), myCrossing->shape[i].y(), 0.0);
            glRotated(myShapeRotations[i], 0, 0, 1);
            // draw crossing
            for (double t = 0; t < myShapeLengths[i]; t += spacing) {
                glBegin(GL_QUADS);
                glVertex2d(-halfWidth, -t);
                glVertex2d(-halfWidth, -t - length);
                glVertex2d(halfWidth, -t - length);
                glVertex2d(halfWidth, -t);
                glEnd();
            }
            // pop three draw matrix
            glPopMatrix();
        }
        // XXX draw junction index / tls index
        // pop second draw matrix
        glPopMatrix();
        // traslate to back
        glTranslated(0, 0, -.2);
        // pop name
        glPopName();
        // pop draw matrix
        glPopMatrix();
    }
}


GUIGLObjectPopupMenu*
GNECrossing::getPopUpMenu(GUIMainWindow& app, GUISUMOAbstractView& parent) {
    myPopup = new GUIGLObjectPopupMenu(app, parent, *this);
    buildPopupHeader(myPopup, app);
    return myPopup;
}


GUIParameterTableWindow*
GNECrossing::getParameterWindow(GUIMainWindow& app, GUISUMOAbstractView&) {
    GUIParameterTableWindow* ret =
        new GUIParameterTableWindow(app, *this, 2);
    // add items
    // close building
    ret->closeBuilding();
    return ret;
}


Boundary
GNECrossing::getCenteringBoundary() const {
    Boundary b = myCrossing->shape.getBoxBoundary();
    b.grow(10);
    return b;
}


std::string
GNECrossing::getAttribute(SumoXMLAttr key) const {
    switch (key) {
        case SUMO_ATTR_ID:
            return getMicrosimID();
            break;
        case SUMO_ATTR_WIDTH:
            return toString(myCrossing->width);
            break;
        case SUMO_ATTR_PRIORITY:
            return myCrossing->priority ? "true" : "false";
            break;
        case SUMO_ATTR_EDGES:
            return toString(myCrossing->edges);
            break;
        default:
            throw InvalidArgument(toString(getTag()) + " doesn't have an attribute of type '" + toString(key) + "'");
    }
}


void
GNECrossing::setAttribute(SumoXMLAttr key, const std::string& value, GNEUndoList* undoList) {
    if (value == getAttribute(key)) {
        return; //avoid needless changes, later logic relies on the fact that attributes have changed
    }
    switch (key) {
        case SUMO_ATTR_ID:
            throw InvalidArgument("Modifying attribute '" + toString(key) + "' of " + toString(getTag()) + " isn't allowed");
        case SUMO_ATTR_EDGES:
        case SUMO_ATTR_WIDTH:
        case SUMO_ATTR_PRIORITY:
            undoList->add(new GNEChange_Attribute(this, key, value), true);
            break;
        default:
            throw InvalidArgument(toString(getTag()) + " doesn't have an attribute of type '" + toString(key) + "'");
    }
}


bool
GNECrossing::isValid(SumoXMLAttr key, const std::string& value) {
    switch (key) {
        case SUMO_ATTR_ID:
            return false;
        case SUMO_ATTR_EDGES: {
            std::vector<std::string> NBEdgeIDs = GNEAttributeCarrier::parse<std::vector<std::string> > (value);
            // Obtain NBEdges of GNENet and check if exists
            for (std::vector<std::string>::iterator i = NBEdgeIDs.begin(); i != NBEdgeIDs.end(); i++) {
                if (myNet->retrieveEdge((*i), false) == NULL) {
                    return false;
                }
            }
            return true;
        }
        case SUMO_ATTR_WIDTH:
            return canParse<double>(value) && isPositive<double>(value);
        case SUMO_ATTR_PRIORITY:
            return canParse<bool>(value);
        default:
            throw InvalidArgument(toString(getTag()) + " doesn't have an attribute of type '" + toString(key) + "'");
    }
}


bool 
GNECrossing::checkEdgeBelong(GNEEdge* edge) const {
    if(std::find(myCrossing->edges.begin(), myCrossing->edges.end(), edge->getNBEdge()) !=  myCrossing->edges.end()) {
        return true;
    } else {
        return false;
    }
}


bool 
GNECrossing::checkEdgeBelong(const std::vector<GNEEdge*> &edges) const {
    for(std::vector<GNEEdge*>::const_iterator i = edges.begin(); i != edges.end(); i++) {   
        if(checkEdgeBelong(*i)) {
            return true;
        }
    }
    return false;
}

// ===========================================================================
// private
// ===========================================================================

void
GNECrossing::setAttribute(SumoXMLAttr key, const std::string& value) {
    switch (key) {
        case SUMO_ATTR_ID:
            throw InvalidArgument("Modifying attribute '" + toString(key) + "' of " + toString(getTag()) + " isn't allowed");
        case SUMO_ATTR_EDGES: {
            // remove edges of crossing
            myCrossing->edges.clear();
            std::vector<std::string> NBEdgeIDs = GNEAttributeCarrier::parse<std::vector<std::string> > (value);
            // Obtain NBEdges of GNENet and insert it in the crossing
            for (std::vector<std::string>::iterator i = NBEdgeIDs.begin(); i != NBEdgeIDs.end(); i++) {
                myCrossing->edges.push_back(myNet->retrieveEdge(*i)->getNBEdge());
            }
            // update geometry of parent junction
            myParentJunction->updateGeometry();
            break;
        }
        case SUMO_ATTR_WIDTH:
            // Change width an refresh element
            myCrossing->width = parse<double>(value);
            myNet->refreshElement(this);
            break;
        case SUMO_ATTR_PRIORITY:
            myCrossing->priority = parse<bool>(value);
            break;
        default:
            throw InvalidArgument(toString(getTag()) + " doesn't have an attribute of type '" + toString(key) + "'");
    }
}

/****************************************************************************/
