/***********************************************************************
 *                                                                     *
 * Copyright 2016  Lorenzo Porta (Vindex17) <vindex17@outlook.it>      *
 *                                                                     *
 * This program is free software; you can redistribute it and/or       *
 * modify it under the terms of the GNU General Public License as      *
 * published by the Free Software Foundation; either version 3 of      *
 * the License or any later version accepted by the membership of      *
 * KDE e.V. (or its successor approved by the membership of KDE        *
 * e.V.), which shall act as a proxy defined in Section 14 of          *
 * version 3 of the license.                                           *
 *                                                                     *
 * This program is distributed in the hope that it will be useful,     *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of      *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the       *
 * GNU General Public License for more details.                        *
 *                                                                     *
 * You should have received a copy of the GNU General Public License   *
 * along with this program. If not, see <http://www.gnu.org/licenses/> *
 *                                                                     *
 ***********************************************************************/


#include "kfoldersync.hxx"

#include <KAboutData>
#include <KMessageBox>
#include <KUser>
#include <QCommandLineParser>


__attribute__((cold)) static void message_handler(QtMsgType type, const QMessageLogContext &context, const QString &msg)
{

    switch (type) {
    case QtDebugMsg:
    case QtInfoMsg:
        fprintf(stdout, "%s\n(%s: %u)\n", qPrintable(msg), context.file, context.line);
        break;
    case QtWarningMsg:
    case QtCriticalMsg:
        fprintf(stderr, "%s\n(%s: %u)\n", qPrintable(msg), context.file, context.line);
        break;
    case QtFatalMsg:
        qApp->closeAllWindows();
        fprintf(stderr, "%s\n(%s: %u)\n", qPrintable(msg), context.file, context.line);
        KMessageBox::error(nullptr, QStringLiteral("%1\n(%2: %3)").arg(msg, QString::fromLocal8Bit(context.file), QString::number(context.line)), i18nc("@title:window", "Fatal error"));
        qApp->exit(EXIT_FAILURE);
        break;
    }
}


int main(int argc, char **argv)
{
    // Init
    qInstallMessageHandler(message_handler);
    QApplication application(argc, argv);
    const QString &component_name = QStringLiteral(PROJECT_NAME);
    application.setWindowIcon(QIcon::fromTheme(component_name));
    KLocalizedString::setApplicationDomain(PROJECT_NAME);
    KAboutData aboutData(component_name,
                         i18n("KFolderSync"),
                         QStringLiteral(PROJECT_VERSION),
                         i18n("Folder synchronization and backup tool"),
                         KAboutLicense::GPL_V3,
                         QStringLiteral("© 2014-2016 Lorenzo Porta (Vindex17)"),
                         xi18nc("@info", "Translations at: <link>https://www.transifex.com/projects/p/kfoldersync/</link>"),
                         QStringLiteral("https://www.linux-apps.com/p/1127677/"));
    aboutData.setCustomAuthorText(QString(), QString());
    aboutData.addAuthor(QStringLiteral("Lorenzo Porta (Vindex17)"),
                        i18nc("@info:credit", "Developer"),
                        QStringLiteral("vindex17@outlook.it"),
                        QStringLiteral("https://www.linux-apps.com/member/338265/"),
                        QStringLiteral("Vindex17"));
    const auto &icon_helper = i18nc("@info:credit", "Application icon prototypes");
    aboutData.addCredit(QStringLiteral("Paul Von Zimmerman"), icon_helper, QStringLiteral("pcvonz@gmail.com"), QString(), QStringLiteral("pcvonz"));
    aboutData.addCredit(QStringLiteral("Andreas Kainz"),
                        icon_helper,
                        QStringLiteral("kainz.a@gmail.com"),
                        QStringLiteral("https://identity.kde.org/index.php?r=people/view&uid=andreask"),
                        QStringLiteral("andreaskai"));
    aboutData.setTranslator(i18nc("NAME OF TRANSLATORS", "Your names"), i18nc("EMAIL OF TRANSLATORS", "Your emails"));
    KAboutData::setApplicationData(aboutData);
    QCommandLineParser cmdline_parser;
    cmdline_parser.addHelpOption();
    cmdline_parser.addVersionOption();
    aboutData.setupCommandLine(&cmdline_parser);
    cmdline_parser.process(application);
    aboutData.processCommandLine(&cmdline_parser);

    // Check super user privileges
    if (Q_UNLIKELY(KUser(KUser::UseEffectiveUID).isSuperUser()))
        if (KMessageBox::warningYesNo(nullptr, xi18nc("@info", "<warning>It's strongly recommended not to run <application>%1</application> as super user (root)."
                                                      "</warning><nl/><nl/>Are you absolutely sure of what you are doing?", application.applicationDisplayName())) != KMessageBox::Yes)
            return EXIT_SUCCESS;

    // Open main window
    new KFolderSync(&application);

    // Enter main event loop
    return application.exec();
}
