#! /usr/bin/env python2
# Copyright (c) 2012 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

"""Unit tests for git_cl.py."""

import contextlib
import json
import os
import StringIO
import sys
import unittest
import urlparse

sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from testing_support.auto_stub import TestCase

import git_cl
import git_common
import git_footers
import subprocess2

def callError(code=1, cmd='', cwd='', stdout='', stderr=''):
  return subprocess2.CalledProcessError(code, cmd, cwd, stdout, stderr)


CERR1 = callError(1)


class ChangelistMock(object):
  # A class variable so we can access it when we don't have access to the
  # instance that's being set.
  desc = ""
  def __init__(self, **kwargs):
    pass
  def GetIssue(self):
    return 1
  def GetDescription(self):
    return ChangelistMock.desc
  def UpdateDescription(self, desc, force=False):
    ChangelistMock.desc = desc


class PresubmitMock(object):
  def __init__(self, *args, **kwargs):
    self.reviewers = []
  @staticmethod
  def should_continue():
    return True


class RietveldMock(object):
  def __init__(self, *args, **kwargs):
    pass

  @staticmethod
  def get_description(issue):
    return 'Issue: %d' % issue

  @staticmethod
  def get_issue_properties(_issue, _messages):
    return {
      'reviewers': ['joe@chromium.org', 'john@chromium.org'],
      'messages': [
        {
          'approval': True,
          'sender': 'john@chromium.org',
        },
      ],
      'patchsets': [1, 20001],
    }

  @staticmethod
  def close_issue(_issue):
    return 'Closed'

  @staticmethod
  def get_patch(issue, patchset):
    return 'patch set from issue %s patchset %s' % (issue, patchset)

  @staticmethod
  def update_description(_issue, _description):
    return 'Updated'

  @staticmethod
  def add_comment(_issue, _comment):
    return 'Commented'




class GitCheckoutMock(object):
  def __init__(self, *args, **kwargs):
    pass

  @staticmethod
  def reset():
    GitCheckoutMock.conflict = False

  def apply_patch(self, p):
    if GitCheckoutMock.conflict:
      raise Exception('failed')


class WatchlistsMock(object):
  def __init__(self, _):
    pass
  @staticmethod
  def GetWatchersForPaths(_):
    return ['joe@example.com']


class CodereviewSettingsFileMock(object):
  def __init__(self):
    pass
  # pylint: disable=R0201
  def read(self):
    return ("CODE_REVIEW_SERVER: gerrit.chromium.org\n" +
            "GERRIT_HOST: True\n")


class AuthenticatorMock(object):
  def __init__(self, *_args):
    pass
  def has_cached_credentials(self):
    return True
  def authorize(self, http):
    return http


def CookiesAuthenticatorMockFactory(hosts_with_creds=None, same_cookie=False):
  """Use to mock Gerrit/Git credentials from ~/.netrc or ~/.gitcookies.

  Usage:
    >>> self.mock(git_cl.gerrit_util, "CookiesAuthenticator",
                  CookiesAuthenticatorMockFactory({'host1': 'cookie1'}))

  OR
    >>> self.mock(git_cl.gerrit_util, "CookiesAuthenticator",
                  CookiesAuthenticatorMockFactory(cookie='cookie'))
  """
  class CookiesAuthenticatorMock(git_cl.gerrit_util.CookiesAuthenticator):
    def __init__(self):  # pylint: disable=W0231
      # Intentionally not calling super() because it reads actual cookie files.
      pass
    @classmethod
    def get_gitcookies_path(cls):
      return '~/.gitcookies'
    @classmethod
    def get_netrc_path(cls):
      return '~/.netrc'
    def get_auth_header(self, host):
      if same_cookie:
        return same_cookie
      return (hosts_with_creds or {}).get(host)
  return CookiesAuthenticatorMock

class MockChangelistWithBranchAndIssue():
  def __init__(self, branch, issue):
    self.branch = branch
    self.issue = issue
  def GetBranch(self):
    return self.branch
  def GetIssue(self):
    return self.issue


class SystemExitMock(Exception):
  pass


class TestGitClBasic(unittest.TestCase):
  def _test_ParseIssueUrl(self, func, url, issue, patchset, hostname, fail):
    parsed = urlparse.urlparse(url)
    result = func(parsed)
    if fail:
      self.assertIsNone(result)
      return None
    self.assertIsNotNone(result)
    self.assertEqual(result.issue, issue)
    self.assertEqual(result.patchset, patchset)
    self.assertEqual(result.hostname, hostname)
    return result

  def test_ParseIssueURL_rietveld(self):
    def test(url, issue=None, patchset=None, hostname=None, fail=None):
      self._test_ParseIssueUrl(
          git_cl._RietveldChangelistImpl.ParseIssueURL,
          url, issue, patchset, hostname, fail)

    test('http://codereview.chromium.org/123',
         123, None, 'codereview.chromium.org')
    test('https://codereview.chromium.org/123',
         123, None, 'codereview.chromium.org')
    test('https://codereview.chromium.org/123/',
         123, None, 'codereview.chromium.org')
    test('https://codereview.chromium.org/123/whatever',
         123, None, 'codereview.chromium.org')
    test('https://codereview.chromium.org/123/#ps20001',
         123, 20001, 'codereview.chromium.org')
    test('http://codereview.chromium.org/download/issue123_4.diff',
         123, 4, 'codereview.chromium.org')
    # This looks like bad Gerrit, but is actually valid Rietveld.
    test('https://chrome-review.source.com/123/4/',
         123, None, 'chrome-review.source.com')

    test('https://codereview.chromium.org/deadbeaf', fail=True)
    test('https://codereview.chromium.org/api/123', fail=True)
    test('bad://codereview.chromium.org/123', fail=True)
    test('http://codereview.chromium.org/download/issue123_4.diffff', fail=True)

  def test_ParseIssueURL_gerrit(self):
    def test(url, issue=None, patchset=None, hostname=None, fail=None):
      self._test_ParseIssueUrl(
          git_cl._GerritChangelistImpl.ParseIssueURL,
          url, issue, patchset, hostname, fail)

    test('http://chrome-review.source.com/c/123',
         123, None, 'chrome-review.source.com')
    test('https://chrome-review.source.com/c/123/',
         123, None, 'chrome-review.source.com')
    test('https://chrome-review.source.com/c/123/4',
         123, 4, 'chrome-review.source.com')
    test('https://chrome-review.source.com/#/c/123/4',
         123, 4, 'chrome-review.source.com')
    test('https://chrome-review.source.com/c/123/4',
         123, 4, 'chrome-review.source.com')
    test('https://chrome-review.source.com/123',
         123, None, 'chrome-review.source.com')
    test('https://chrome-review.source.com/123/4',
         123, 4, 'chrome-review.source.com')

    test('https://chrome-review.source.com/c/123/1/whatisthis', fail=True)
    test('https://chrome-review.source.com/c/abc/', fail=True)
    test('ssh://chrome-review.source.com/c/123/1/', fail=True)

  def test_ParseIssueNumberArgument(self):
    def test(arg, issue=None, patchset=None, hostname=None, fail=False):
      result = git_cl.ParseIssueNumberArgument(arg)
      self.assertIsNotNone(result)
      if fail:
        self.assertFalse(result.valid)
      else:
        self.assertEqual(result.issue, issue)
        self.assertEqual(result.patchset, patchset)
        self.assertEqual(result.hostname, hostname)

    test('123', 123)
    test('', fail=True)
    test('abc', fail=True)
    test('123/1', fail=True)
    test('123a', fail=True)
    test('ssh://chrome-review.source.com/#/c/123/4/', fail=True)
    # Rietveld.
    test('https://codereview.source.com/123',
         123, None, 'codereview.source.com')
    test('https://codereview.source.com/www123', fail=True)
    # Gerrrit.
    test('https://chrome-review.source.com/c/123/4',
         123, 4, 'chrome-review.source.com')
    test('https://chrome-review.source.com/bad/123/4', fail=True)

  def test_get_bug_line_values(self):
    f = lambda p, bugs: list(git_cl._get_bug_line_values(p, bugs))
    self.assertEqual(f('', ''), [])
    self.assertEqual(f('', '123,v8:456'), ['123', 'v8:456'])
    self.assertEqual(f('v8', '456'), ['v8:456'])
    self.assertEqual(f('v8', 'chromium:123,456'), ['v8:456', 'chromium:123'])
    # Not nice, but not worth carying.
    self.assertEqual(f('v8', 'chromium:123,456,v8:123'),
                     ['v8:456', 'chromium:123', 'v8:123'])

  def _test_git_number(self, parent_msg, dest_ref, child_msg,
                       parent_hash='parenthash'):
    desc = git_cl.ChangeDescription(child_msg)
    desc.update_with_git_number_footers(parent_hash, parent_msg, dest_ref)
    return desc.description

  def assertEqualByLine(self, actual, expected):
    self.assertEqual(actual.splitlines(), expected.splitlines())

  def test_git_number_bad_parent(self):
    with self.assertRaises(ValueError):
      self._test_git_number('Parent', 'refs/heads/master', 'Child')

  def test_git_number_bad_parent_footer(self):
    with self.assertRaises(AssertionError):
      self._test_git_number(
          'Parent\n'
          '\n'
          'Cr-Commit-Position: wrong',
          'refs/heads/master', 'Child')

  def test_git_number_bad_lineage_ignored(self):
    actual = self._test_git_number(
        'Parent\n'
        '\n'
        'Cr-Commit-Position: refs/heads/master@{#1}\n'
        'Cr-Branched-From: mustBeReal40CharHash-branch@{#pos}',
        'refs/heads/master', 'Child')
    self.assertEqualByLine(
        actual,
        'Child\n'
        '\n'
        'Cr-Commit-Position: refs/heads/master@{#2}\n'
        'Cr-Branched-From: mustBeReal40CharHash-branch@{#pos}')

  def test_git_number_same_branch(self):
    actual = self._test_git_number(
        'Parent\n'
        '\n'
        'Cr-Commit-Position: refs/heads/master@{#12}',
        dest_ref='refs/heads/master',
        child_msg='Child')
    self.assertEqualByLine(
        actual,
        'Child\n'
        '\n'
        'Cr-Commit-Position: refs/heads/master@{#13}')

  def test_git_number_same_branch_with_originals(self):
    actual = self._test_git_number(
        'Parent\n'
        '\n'
        'Cr-Commit-Position: refs/heads/master@{#12}',
        dest_ref='refs/heads/master',
        child_msg='Child\n'
        '\n'
        'Some users are smart and insert their own footers\n'
        '\n'
        'Cr-Whatever: value\n'
        'Cr-Commit-Position: refs/copy/paste@{#22}')
    self.assertEqualByLine(
        actual,
        'Child\n'
        '\n'
        'Some users are smart and insert their own footers\n'
        '\n'
        'Cr-Original-Whatever: value\n'
        'Cr-Original-Commit-Position: refs/copy/paste@{#22}\n'
        'Cr-Commit-Position: refs/heads/master@{#13}')

  def test_git_number_new_branch(self):
    actual = self._test_git_number(
        'Parent\n'
        '\n'
        'Cr-Commit-Position: refs/heads/master@{#12}',
        dest_ref='refs/heads/branch',
        child_msg='Child')
    self.assertEqualByLine(
        actual,
        'Child\n'
        '\n'
        'Cr-Commit-Position: refs/heads/branch@{#1}\n'
        'Cr-Branched-From: parenthash-refs/heads/master@{#12}')

  def test_git_number_lineage(self):
    actual = self._test_git_number(
        'Parent\n'
        '\n'
        'Cr-Commit-Position: refs/heads/branch@{#1}\n'
        'Cr-Branched-From: somehash-refs/heads/master@{#12}',
        dest_ref='refs/heads/branch',
        child_msg='Child')
    self.assertEqualByLine(
        actual,
        'Child\n'
        '\n'
        'Cr-Commit-Position: refs/heads/branch@{#2}\n'
        'Cr-Branched-From: somehash-refs/heads/master@{#12}')

  def test_git_number_moooooooore_lineage(self):
    actual = self._test_git_number(
        'Parent\n'
        '\n'
        'Cr-Commit-Position: refs/heads/branch@{#5}\n'
        'Cr-Branched-From: somehash-refs/heads/master@{#12}',
        dest_ref='refs/heads/mooore',
        child_msg='Child')
    self.assertEqualByLine(
        actual,
        'Child\n'
        '\n'
        'Cr-Commit-Position: refs/heads/mooore@{#1}\n'
        'Cr-Branched-From: parenthash-refs/heads/branch@{#5}\n'
        'Cr-Branched-From: somehash-refs/heads/master@{#12}')


  def test_git_number_cherry_pick(self):
    actual = self._test_git_number(
        'Parent\n'
        '\n'
        'Cr-Commit-Position: refs/heads/branch@{#1}\n'
        'Cr-Branched-From: somehash-refs/heads/master@{#12}',
        dest_ref='refs/heads/branch',
        child_msg='Child, which is cherry-pick from master\n'
        '\n'
        'Cr-Commit-Position: refs/heads/master@{#100}\n'
        '(cherry picked from commit deadbeef12345678deadbeef12345678deadbeef)')
    self.assertEqualByLine(
        actual,
        'Child, which is cherry-pick from master\n'
        '\n'
        '(cherry picked from commit deadbeef12345678deadbeef12345678deadbeef)\n'
        '\n'
        'Cr-Original-Commit-Position: refs/heads/master@{#100}\n'
        'Cr-Commit-Position: refs/heads/branch@{#2}\n'
        'Cr-Branched-From: somehash-refs/heads/master@{#12}')


class TestGitCl(TestCase):
  def setUp(self):
    super(TestGitCl, self).setUp()
    self.calls = []
    self._calls_done = []
    self.mock(subprocess2, 'call', self._mocked_call)
    self.mock(subprocess2, 'check_call', self._mocked_call)
    self.mock(subprocess2, 'check_output', self._mocked_call)
    self.mock(subprocess2, 'communicate',
              lambda *a, **kw: ([self._mocked_call(*a, **kw), ''], 0))
    self.mock(git_cl.gclient_utils, 'CheckCallAndFilter', self._mocked_call)
    self.mock(git_common, 'is_dirty_git_tree', lambda x: False)
    self.mock(git_common, 'get_or_create_merge_base',
              lambda *a: (
                  self._mocked_call(['get_or_create_merge_base']+list(a))))
    self.mock(git_cl, 'BranchExists', lambda _: True)
    self.mock(git_cl, 'FindCodereviewSettingsFile', lambda: '')
    self.mock(git_cl, 'ask_for_data', self._mocked_call)
    self.mock(git_cl, 'write_json', lambda path, contents:
        self._mocked_call('write_json', path, contents))
    self.mock(git_cl.presubmit_support, 'DoPresubmitChecks', PresubmitMock)
    self.mock(git_cl.rietveld, 'Rietveld', RietveldMock)
    self.mock(git_cl.rietveld, 'CachingRietveld', RietveldMock)
    self.mock(git_cl.checkout, 'GitCheckout', GitCheckoutMock)
    GitCheckoutMock.reset()
    self.mock(git_cl.upload, 'RealMain', self.fail)
    self.mock(git_cl.watchlists, 'Watchlists', WatchlistsMock)
    self.mock(git_cl.auth, 'get_authenticator_for_host', AuthenticatorMock)
    self.mock(git_cl.gerrit_util.GceAuthenticator, 'is_gce',
              classmethod(lambda _: False))
    self.mock(git_cl, 'DieWithError',
              lambda msg: self._mocked_call(['DieWithError', msg]))
    # It's important to reset settings to not have inter-tests interference.
    git_cl.settings = None


  def tearDown(self):
    try:
      # Note: has_failed returns True if at least 1 test ran so far, current
      # included, has failed. That means current test may have actually ran
      # fine, and the check for no leftover calls would be skipped.
      if not self.has_failed():
        self.assertEquals([], self.calls)
    finally:
      super(TestGitCl, self).tearDown()

  def _mocked_call(self, *args, **_kwargs):
    self.assertTrue(
        self.calls,
        '@%d  Expected: <Missing>   Actual: %r' % (len(self._calls_done), args))
    top = self.calls.pop(0)
    expected_args, result = top

    # Also logs otherwise it could get caught in a try/finally and be hard to
    # diagnose.
    if expected_args != args:
      N = 5
      prior_calls  = '\n  '.join(
          '@%d: %r' % (len(self._calls_done) - N + i, c[0])
          for i, c in enumerate(self._calls_done[-N:]))
      following_calls = '\n  '.join(
          '@%d: %r' % (len(self._calls_done) + i + 1, c[0])
          for i, c in enumerate(self.calls[:N]))
      extended_msg = (
          'A few prior calls:\n  %s\n\n'
          'This (expected):\n  @%d: %r\n'
          'This (actual):\n  @%d: %r\n\n'
          'A few following expected calls:\n  %s' %
          (prior_calls, len(self._calls_done), expected_args,
           len(self._calls_done), args, following_calls))
      git_cl.logging.error(extended_msg)

      self.fail('@%d\n'
                '  Expected: %r\n'
                '  Actual:   %r' % (
          len(self._calls_done), expected_args, args))

    self._calls_done.append(top)
    if isinstance(result, Exception):
      raise result
    return result

  def test_LoadCodereviewSettingsFromFile_gerrit(self):
    codereview_file = StringIO.StringIO('GERRIT_HOST: true')
    self.calls = [
      ((['git', 'config', '--unset-all', 'rietveld.cc'],), CERR1),
      ((['git', 'config', '--unset-all', 'rietveld.private'],), CERR1),
      ((['git', 'config', '--unset-all', 'rietveld.tree-status-url'],), CERR1),
      ((['git', 'config', '--unset-all', 'rietveld.viewvc-url'],), CERR1),
      ((['git', 'config', '--unset-all', 'rietveld.bug-prefix'],), CERR1),
      ((['git', 'config', '--unset-all', 'rietveld.cpplint-regex'],), CERR1),
      ((['git', 'config', '--unset-all', 'rietveld.force-https-commit-url'],),
        CERR1),
      ((['git', 'config', '--unset-all', 'rietveld.cpplint-ignore-regex'],),
        CERR1),
      ((['git', 'config', '--unset-all', 'rietveld.project'],), CERR1),
      ((['git', 'config', '--unset-all', 'rietveld.pending-ref-prefix'],),
        CERR1),
      ((['git', 'config', '--unset-all', 'rietveld.run-post-upload-hook'],),
        CERR1),
      ((['git', 'config', 'gerrit.host', 'true'],), ''),
    ]
    self.assertIsNone(git_cl.LoadCodereviewSettingsFromFile(codereview_file))

  @classmethod
  def _is_gerrit_calls(cls, gerrit=False):
    return [((['git', 'config', 'rietveld.autoupdate'],), ''),
            ((['git', 'config', 'gerrit.host'],), 'True' if gerrit else '')]

  @classmethod
  def _upload_calls(cls, similarity, find_copies, private):
    return (cls._git_base_calls(similarity, find_copies) +
            cls._git_upload_calls(private))

  @classmethod
  def _upload_no_rev_calls(cls, similarity, find_copies):
    return (cls._git_base_calls(similarity, find_copies) +
            cls._git_upload_no_rev_calls())

  @classmethod
  def _git_base_calls(cls, similarity, find_copies):
    if similarity is None:
      similarity = '50'
      similarity_call = ((['git', 'config',
                         'branch.master.git-cl-similarity'],), CERR1)
    else:
      similarity_call = ((['git', 'config',
                           'branch.master.git-cl-similarity', similarity],), '')

    if find_copies is None:
      find_copies = True
      find_copies_call = ((['git', 'config', '--bool',
                          'branch.master.git-find-copies'],), CERR1)
    else:
      val = str(find_copies).lower()
      find_copies_call = ((['git', 'config', '--bool',
                          'branch.master.git-find-copies', val],), '')

    if find_copies:
      stat_call = ((['git', 'diff', '--no-ext-diff', '--stat',
                     '-l100000', '-C'+similarity,
                     'fake_ancestor_sha', 'HEAD'],), '+dat')
    else:
      stat_call = ((['git', 'diff', '--no-ext-diff', '--stat',
                   '-M'+similarity, 'fake_ancestor_sha', 'HEAD'],), '+dat')

    return [
      ((['git', 'symbolic-ref', 'HEAD'],), 'master'),
      similarity_call,
      ((['git', 'symbolic-ref', 'HEAD'],), 'master'),
      find_copies_call,
    ] + cls._is_gerrit_calls() + [
      ((['git', 'symbolic-ref', 'HEAD'],), 'master'),
      ((['git', 'config', 'branch.master.rietveldissue'],), CERR1),
      ((['git', 'config', 'branch.master.gerritissue'],), CERR1),
      ((['git', 'config', 'rietveld.server'],),
       'codereview.example.com'),
      ((['git', 'config', 'branch.master.merge'],), 'master'),
      ((['git', 'config', 'branch.master.remote'],), 'origin'),
      ((['get_or_create_merge_base', 'master', 'master'],),
       'fake_ancestor_sha'),
    ] + cls._git_sanity_checks('fake_ancestor_sha', 'master') + [
      ((['git', 'rev-parse', '--show-cdup'],), ''),
      ((['git', 'rev-parse', 'HEAD'],), '12345'),
      ((['git', 'diff', '--name-status', '--no-renames', '-r',
         'fake_ancestor_sha...', '.'],),
        'M\t.gitignore\n'),
      ((['git', 'config', 'branch.master.rietveldpatchset'],), CERR1),
      ((['git', 'log', '--pretty=format:%s%n%n%b',
         'fake_ancestor_sha...'],),
       'foo'),
      ((['git', 'config', 'user.email'],), 'me@example.com'),
      stat_call,
      ((['git', 'log', '--pretty=format:%s\n\n%b',
         'fake_ancestor_sha..HEAD'],),
       'desc\n'),
      ((['git', 'config', 'rietveld.bug-prefix'],), ''),
    ]

  @classmethod
  def _git_upload_no_rev_calls(cls):
    return [
      ((['git', 'config', 'core.editor'],), ''),
    ]

  @classmethod
  def _git_upload_calls(cls, private):
    if private:
      cc_call = []
      private_call = []
    else:
      cc_call = [((['git', 'config', 'rietveld.cc'],), '')]
      private_call = [
          ((['git', 'config', 'rietveld.private'],), '')]

    return [
        ((['git', 'config', 'core.editor'],), ''),
    ] + cc_call + private_call + [
        ((['git', 'config', 'branch.master.base-url'],), ''),
        ((['git', 'config', 'rietveld.pending-ref-prefix'],), ''),
        ((['git',
           'config', '--local', '--get-regexp', '^svn-remote\\.'],),
         (('', None), 0)),
        ((['git', 'rev-parse', '--show-cdup'],), ''),
        ((['git', 'svn', 'info'],), ''),
        ((['git', 'config', 'rietveld.project'],), ''),
        ((['git', 'config', 'branch.master.rietveldissue', '1'],), ''),
        ((['git', 'config', 'branch.master.rietveldserver',
           'https://codereview.example.com'],), ''),
        ((['git',
           'config', 'branch.master.rietveldpatchset', '2'],), ''),
    ] + cls._git_post_upload_calls()

  @classmethod
  def _git_post_upload_calls(cls):
    return [
        ((['git', 'rev-parse', 'HEAD'],), 'hash'),
        ((['git', 'symbolic-ref', 'HEAD'],), 'hash'),
        ((['git',
           'config', 'branch.hash.last-upload-hash', 'hash'],), ''),
        ((['git', 'config', 'rietveld.run-post-upload-hook'],), ''),
    ]

  @staticmethod
  def _git_sanity_checks(diff_base, working_branch, get_remote_branch=True):
    fake_ancestor = 'fake_ancestor'
    fake_cl = 'fake_cl_for_patch'
    return [
      ((['git',
         'rev-parse', '--verify', diff_base],), fake_ancestor),
      ((['git',
         'merge-base', fake_ancestor, 'HEAD'],), fake_ancestor),
      ((['git',
         'rev-list', '^' + fake_ancestor, 'HEAD'],), fake_cl),
      # Mock a config miss (error code 1)
      ((['git',
         'config', 'gitcl.remotebranch'],), CERR1),
    ] + ([
      # Call to GetRemoteBranch()
      ((['git',
         'config', 'branch.%s.merge' % working_branch],),
       'refs/heads/master'),
      ((['git',
         'config', 'branch.%s.remote' % working_branch],), 'origin'),
    ] if get_remote_branch else []) + [
      ((['git', 'rev-list', '^' + fake_ancestor,
         'refs/remotes/origin/master'],), ''),
    ]

  @classmethod
  def _dcommit_calls_1(cls):
    return [
      ((['git', 'config', 'rietveld.autoupdate'],),
       ''),
      ((['git', 'config', 'rietveld.pending-ref-prefix'],),
       ''),
      ((['git',
         'config', '--local', '--get-regexp', '^svn-remote\\.'],),
       ((('svn-remote.svn.url svn://svn.chromium.org/chrome\n'
          'svn-remote.svn.fetch trunk/src:refs/remotes/origin/master'),
         None),
        0)),
      ((['git', 'symbolic-ref', 'HEAD'],), 'refs/heads/working'),
      ((['git', 'config',
        'branch.working.git-cl-similarity'],), CERR1),
      ((['git', 'symbolic-ref', 'HEAD'],), 'refs/heads/working'),
      ((['git', 'config', '--bool',
        'branch.working.git-find-copies'],), CERR1),
      ((['git', 'symbolic-ref', 'HEAD'],), 'refs/heads/working'),
      ((['git',
         'config', 'branch.working.rietveldissue'],), '12345'),
      ((['git',
         'config', 'rietveld.server'],), 'codereview.example.com'),
      ((['git',
         'config', 'branch.working.merge'],), 'refs/heads/master'),
      ((['git', 'config', 'branch.working.remote'],), 'origin'),
      ((['git', 'config', 'branch.working.merge'],),
       'refs/heads/master'),
      ((['git', 'config', 'branch.working.remote'],), 'origin'),
      ((['git', 'rev-list', '--merges',
         '--grep=^SVN changes up to revision [0-9]*$',
         'refs/remotes/origin/master^!'],), ''),
      ((['git', 'rev-list', '^refs/heads/working',
         'refs/remotes/origin/master'],),
         ''),
      ((['git',
         'log', '--grep=^git-svn-id:', '-1', '--pretty=format:%H'],),
         '3fc18b62c4966193eb435baabe2d18a3810ec82e'),
      ((['git',
         'rev-list', '^3fc18b62c4966193eb435baabe2d18a3810ec82e',
         'refs/remotes/origin/master'],), ''),
      ((['git',
         'merge-base', 'refs/remotes/origin/master', 'HEAD'],),
       'fake_ancestor_sha'),
    ]

  @classmethod
  def _dcommit_calls_normal(cls):
    return [
      ((['git', 'rev-parse', '--show-cdup'],), ''),
      ((['git', 'rev-parse', 'HEAD'],),
          '00ff397798ea57439712ed7e04ab96e13969ef40'),
      ((['git',
         'diff', '--name-status', '--no-renames', '-r', 'fake_ancestor_sha...',
         '.'],),
        'M\tPRESUBMIT.py'),
      ((['git',
         'config', 'branch.working.rietveldpatchset'],), '31137'),
      ((['git', 'config', 'branch.working.rietveldserver'],),
         'codereview.example.com'),
      ((['git', 'config', 'user.email'],), 'author@example.com'),
      ((['git', 'config', 'rietveld.tree-status-url'],), ''),
  ]

  @classmethod
  def _dcommit_calls_bypassed(cls):
    return [
      ((['git', 'config', 'branch.working.rietveldserver'],),
         'codereview.example.com'),
  ]

  @classmethod
  def _dcommit_calls_3(cls):
    return [
      ((['git',
         'diff', '--no-ext-diff', '--stat', '-l100000', '-C50',
         'fake_ancestor_sha', 'refs/heads/working'],),
       (' PRESUBMIT.py |    2 +-\n'
        ' 1 files changed, 1 insertions(+), 1 deletions(-)\n')),
      ((['git', 'show-ref', '--quiet', '--verify',
         'refs/heads/git-cl-commit'],), ''),
      ((['git', 'branch', '-D', 'git-cl-commit'],), ''),
      ((['git', 'show-ref', '--quiet', '--verify',
         'refs/heads/git-cl-cherry-pick'],), CERR1),
      ((['git', 'rev-parse', '--show-cdup'],), '\n'),
      ((['git', 'checkout', '-q', '-b', 'git-cl-commit'],), ''),
      ((['git', 'reset', '--soft', 'fake_ancestor_sha'],), ''),
      ((['git', 'commit', '-m',
         'Issue: 12345\n\nR=john@chromium.org\n\n'
         'Review-Url: https://codereview.example.com/12345 .'],),
       ''),
      ((['git', 'config', 'rietveld.force-https-commit-url'],), ''),
      ((['git',
         'svn', 'dcommit', '-C50', '--no-rebase', '--rmdir'],),
       (('', None), 0)),
      ((['git', 'checkout', '-q', 'working'],), ''),
      ((['git', 'branch', '-D', 'git-cl-commit'],), ''),
  ]

  @staticmethod
  def _cmd_line(description, args, similarity, find_copies, private, cc):
    """Returns the upload command line passed to upload.RealMain()."""
    return [
        'upload', '--assume_yes', '--server',
        'https://codereview.example.com',
        '--message', description
    ] + args + [
        '--cc', ','.join(['joe@example.com'] + cc),
    ] + (['--private'] if private else []) + [
        '--git_similarity', similarity or '50'
    ] + (['--git_no_find_copies'] if find_copies == False else []) + [
        'fake_ancestor_sha', 'HEAD'
    ]

  def _run_reviewer_test(
      self,
      upload_args,
      expected_description,
      returned_description,
      final_description,
      reviewers,
      private=False,
      cc=None):
    """Generic reviewer test framework."""
    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())
    try:
      similarity = upload_args[upload_args.index('--similarity')+1]
    except ValueError:
      similarity = None

    if '--find-copies' in upload_args:
      find_copies = True
    elif '--no-find-copies' in upload_args:
      find_copies = False
    else:
      find_copies = None

    private = '--private' in upload_args
    cc = cc or []

    self.calls = self._upload_calls(similarity, find_copies, private)

    def RunEditor(desc, _, **kwargs):
      self.assertEquals(
          '# Enter a description of the change.\n'
          '# This will be displayed on the codereview site.\n'
          '# The first line will also be used as the subject of the review.\n'
          '#--------------------This line is 72 characters long'
          '--------------------\n' +
          expected_description,
          desc)
      return returned_description
    self.mock(git_cl.gclient_utils, 'RunEditor', RunEditor)

    def check_upload(args):
      cmd_line = self._cmd_line(final_description, reviewers, similarity,
                                find_copies, private, cc)
      self.assertEquals(cmd_line, args)
      return 1, 2
    self.mock(git_cl.upload, 'RealMain', check_upload)

    git_cl.main(['upload'] + upload_args)

  def test_no_reviewer(self):
    self._run_reviewer_test(
        [],
        'desc\n\nBUG=',
        '# Blah blah comment.\ndesc\n\nBUG=',
        'desc\n\nBUG=',
        [])

  def test_keep_similarity(self):
    self._run_reviewer_test(
        ['--similarity', '70'],
        'desc\n\nBUG=',
        '# Blah blah comment.\ndesc\n\nBUG=',
        'desc\n\nBUG=',
        [])

  def test_keep_find_copies(self):
    self._run_reviewer_test(
        ['--no-find-copies'],
        'desc\n\nBUG=',
        '# Blah blah comment.\ndesc\n\nBUG=\n',
        'desc\n\nBUG=',
        [])

  def test_private(self):
    self._run_reviewer_test(
        ['--private'],
        'desc\n\nBUG=',
        '# Blah blah comment.\ndesc\n\nBUG=\n',
        'desc\n\nBUG=',
        [])

  def test_reviewers_cmd_line(self):
    # Reviewer is passed as-is
    description = 'desc\n\nR=foo@example.com\nBUG='
    self._run_reviewer_test(
        ['-r' 'foo@example.com'],
        description,
        '\n%s\n' % description,
        description,
        ['--reviewers=foo@example.com'])

  def test_reviewer_tbr_overriden(self):
    # Reviewer is overriden with TBR
    # Also verifies the regexp work without a trailing LF
    description = 'Foo Bar\n\nTBR=reviewer@example.com'
    self._run_reviewer_test(
        ['-r' 'foo@example.com'],
        'desc\n\nR=foo@example.com\nBUG=',
        description.strip('\n'),
        description,
        ['--reviewers=reviewer@example.com'])

  def test_reviewer_multiple(self):
    # Handles multiple R= or TBR= lines.
    description = (
        'Foo Bar\nTBR=reviewer@example.com\nBUG=\nR=another@example.com\n'
        'CC=more@example.com,people@example.com')
    self._run_reviewer_test(
        [],
        'desc\n\nBUG=',
        description,
        description,
        ['--reviewers=another@example.com,reviewer@example.com'],
        cc=['more@example.com', 'people@example.com'])

  def test_reviewer_send_mail(self):
    # --send-mail can be used without -r if R= is used
    description = 'Foo Bar\nR=reviewer@example.com'
    self._run_reviewer_test(
        ['--send-mail'],
        'desc\n\nBUG=',
        description.strip('\n'),
        description,
        ['--reviewers=reviewer@example.com', '--send_mail'])

  def test_reviewer_send_mail_no_rev(self):
    # Fails without a reviewer.
    stdout = StringIO.StringIO()
    self.calls = self._upload_no_rev_calls(None, None) + [
        ((['DieWithError', 'Must specify reviewers to send email.'],),
          SystemExitMock())
    ]

    def RunEditor(desc, _, **kwargs):
      return desc
    self.mock(git_cl.gclient_utils, 'RunEditor', RunEditor)
    self.mock(sys, 'stdout', stdout)
    with self.assertRaises(SystemExitMock):
      git_cl.main(['upload', '--send-mail'])
    self.assertEqual(
        'Using 50% similarity for rename/copy detection. Override with '
        '--similarity.\n',
        stdout.getvalue())

  def test_bug_on_cmd(self):
    self._run_reviewer_test(
        ['--bug=500658,proj:123'],
        'desc\n\nBUG=500658\nBUG=proj:123',
        '# Blah blah comment.\ndesc\n\nBUG=500658\nBUG=proj:1234',
        'desc\n\nBUG=500658\nBUG=proj:1234',
        [])

  def test_dcommit(self):
    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())
    self.calls = (
        self._dcommit_calls_1() +
        self._git_sanity_checks('fake_ancestor_sha', 'working') +
        self._dcommit_calls_normal() +
        self._dcommit_calls_3())
    git_cl.main(['dcommit'])

  def test_dcommit_bypass_hooks(self):
    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())
    self.calls = (
        self._dcommit_calls_1() +
        self._dcommit_calls_bypassed() +
        self._dcommit_calls_3())
    git_cl.main(['dcommit', '--bypass-hooks'])

  def _land_rietveld_common(self, debug=False):
    if debug:
      # Very useful due to finally clause in git cl land raising exceptions and
      # shadowing real cause of failure.
      self.mock(git_cl, '_IS_BEING_TESTED', True)
    else:
      self.mock(git_cl.sys, 'stdout', StringIO.StringIO())

    self.mock(git_cl._GitNumbererState, 'load', classmethod(lambda _, url, ref:
        self._mocked_call(['_GitNumbererState', url, ref])))
    self.mock(RietveldMock, 'update_description', staticmethod(
              lambda i, d: self._mocked_call(['update_description', i, d])))
    self.mock(RietveldMock, 'add_comment', staticmethod(
              lambda i, c: self._mocked_call(['add_comment', i, c])))
    self.calls = [
      ((['git', 'config', 'rietveld.autoupdate'],), ''),
      ((['git', 'config', 'rietveld.pending-ref-prefix'],), CERR1),
      ((['git', 'config', '--local', '--get-regexp', '^svn-remote\\.'],),
       CERR1),
      ((['git', 'symbolic-ref', 'HEAD'],), 'feature'),
      ((['git', 'config', 'branch.feature.git-cl-similarity'],), CERR1),
      ((['git', 'symbolic-ref', 'HEAD'],), 'feature'),
      ((['git', 'config', '--bool', 'branch.feature.git-find-copies'],),
       CERR1),
      ((['git', 'symbolic-ref', 'HEAD'],), 'feature'),
      ((['git', 'config', 'branch.feature.rietveldissue'],), '123'),
      ((['git', 'config', 'rietveld.server'],),
       'https://codereview.chromium.org'),
      ((['git', 'config', 'branch.feature.merge'],), 'refs/heads/master'),
      ((['git', 'config', 'branch.feature.remote'],), 'origin'),
      ((['git', 'config', 'branch.feature.merge'],), 'refs/heads/master'),
      ((['git', 'config', 'branch.feature.remote'],), 'origin'),
      ((['git', 'rev-list', '--merges',
         '--grep=^SVN changes up to revision [0-9]*$',
         'refs/remotes/origin/master^!'],), ''),
      ((['git', 'rev-list', '^feature', 'refs/remotes/origin/master'],),
       ''),  # No commits to rebase, according to local view of origin.
      ((['git', 'merge-base', 'refs/remotes/origin/master', 'HEAD'],),
       'fake_ancestor_sha'),
    ] + self._git_sanity_checks('fake_ancestor_sha', 'feature') + [
      ((['git', 'rev-parse', '--show-cdup'],), ''),
      ((['git', 'rev-parse', 'HEAD'],), 'fake_sha'),
      ((['git', 'diff', '--name-status', '--no-renames', '-r',
         'fake_ancestor_sha...', '.'],),
       'M\tfile1.cpp'),
      ((['git', 'config', 'branch.feature.rietveldpatchset'],), '20001'),
      ((['git', 'config', 'branch.feature.rietveldserver'],),
       'https://codereview.chromium.org'),
      ((['git', 'config', 'user.email'],), 'user@e.mail'),
      ((['git', 'config', 'rietveld.tree-status-url'],), CERR1),
      ((['git', 'diff', '--no-ext-diff', '--stat', '-l100000', '-C50',
         'fake_ancestor_sha', 'feature'],),
       # This command just prints smth like this:
       # file1.cpp   |  53 ++++++--
       # 1 file changed, 33 insertions(+), 20 deletions(-)\n
       ''),
      ((['git', 'show-ref', '--quiet', '--verify',
         'refs/heads/git-cl-commit'],),
         ''),   # 0 return code means branch exists.
      ((['git', 'branch', '-D', 'git-cl-commit'],), ''),
      ((['git', 'show-ref', '--quiet', '--verify',
         'refs/heads/git-cl-cherry-pick'],),
       CERR1),  # This means git-cl-cherry-pick branch does not exist.
      ((['git', 'rev-parse', '--show-cdup'],), ''),
      ((['git', 'checkout', '-q', '-b', 'git-cl-commit'],), ''),
      ((['git', 'reset', '--soft', 'fake_ancestor_sha'],), ''),
      ((['git', 'commit', '-m',
         'Issue: 123\n\nR=john@chromium.org\n\n'
         'Review-Url: https://codereview.chromium.org/123 .'],), ''),
      ((['git', 'config', 'branch.feature.merge'],), 'refs/heads/master'),
      ((['git', 'config', 'branch.feature.remote'],), 'origin'),
      ((['git', 'config', '--get', 'remote.origin.url'],),
       'https://chromium.googlesource.com/infra/infra'),
    ]

  def test_land_rietveld(self):
    self._land_rietveld_common(debug=False)
    self.calls += [
      ((['git', 'config', 'remote.origin.url'],),
       'https://chromium.googlesource.com/infra/infra'),
      ((['_GitNumbererState',
         'https://chromium.googlesource.com/infra/infra',
         'refs/heads/master'],),
       git_cl._GitNumbererState(None, False)),
      ((['git', 'push', '--porcelain', 'origin', 'HEAD:refs/heads/master'],),
       ''),
      ((['git', 'rev-parse', 'HEAD'],), 'fake_sha_rebased'),
      ((['git', 'checkout', '-q', 'feature'],), ''),
      ((['git', 'branch', '-D', 'git-cl-commit'],), ''),
      ((['git', 'config', 'rietveld.viewvc-url'],),
       'https://chromium.googlesource.com/infra/infra/+/'),
      ((['update_description', 123,
         'Issue: 123\n\nR=john@chromium.org\n\nCommitted: '
         'https://chromium.googlesource.com/infra/infra/+/fake_sha_rebased'],),
       ''),
      ((['add_comment', 123, 'Committed patchset #2 (id:20001) manually as '
                             'fake_sha_rebased (presubmit successful).'],), ''),
    ]
    git_cl.main(['land'])

  def test_land_rietveld_gnumbd(self):
    self._land_rietveld_common(debug=False)
    self.mock(git_cl, 'WaitForRealCommit',
              lambda *a: self._mocked_call(['WaitForRealCommit'] + list(a)))
    self.calls += [
      ((['git', 'config', 'remote.origin.url'],),
       'https://chromium.googlesource.com/chromium/src'),
      ((['_GitNumbererState',
         'https://chromium.googlesource.com/chromium/src',
         'refs/heads/master'],),
       git_cl._GitNumbererState('refs/pending', True)),
      ((['git', 'rev-parse', 'HEAD'],), 'fake_sha_rebased'),
      ((['git', 'retry', 'fetch', 'origin',
         '+refs/pending/heads/master:refs/git-cl/pending/heads/master'],), ''),
      ((['git', 'checkout', 'refs/git-cl/pending/heads/master'],), ''),
      ((['git', 'cherry-pick', 'fake_sha_rebased'],), ''),

      ((['git', 'retry', 'push', '--porcelain', 'origin',
        'HEAD:refs/pending/heads/master'],),''),
      ((['git', 'rev-parse', 'HEAD'],), 'fake_sha_rebased_on_pending'),

      ((['git', 'checkout', '-q', 'feature'],), ''),
      ((['git', 'branch', '-D', 'git-cl-commit'],), ''),

      ((['WaitForRealCommit', 'origin', 'fake_sha_rebased_on_pending',
         'refs/remotes/origin/master', 'refs/heads/master'],),
         'fake_sha_gnumbded'),

      ((['git', 'config', 'rietveld.viewvc-url'],),
       'https://chromium.googlesource.com/infra/infra/+/'),
      ((['update_description', 123,
         'Issue: 123\n\nR=john@chromium.org\n\nCommitted: '
         'https://chromium.googlesource.com/infra/infra/+/fake_sha_gnumbded'],),
       ''),
      ((['add_comment', 123, 'Committed patchset #2 (id:20001) manually as '
                             'fake_sha_gnumbded (presubmit successful).'],),
       ''),
    ]
    git_cl.main(['land'])

  def test_land_rietveld_git_numberer(self):
    self._land_rietveld_common(debug=False)

    # Special mocks to check validity of timestamp.
    original_git_amend_head = git_cl._git_amend_head
    def _git_amend_head_mock(msg, tstamp):
      self._mocked_call(['git_amend_head committer timestamp', tstamp])
      return original_git_amend_head(msg, tstamp)
    self.mock(git_cl, '_git_amend_head', _git_amend_head_mock)

    self.calls += [
      ((['git', 'config', 'remote.origin.url'],),
       'https://chromium.googlesource.com/chromium/src'),
      ((['_GitNumbererState',
         'https://chromium.googlesource.com/chromium/src',
         'refs/heads/master'],),
       git_cl._GitNumbererState(None, True)),

      ((['git', 'show', '-s', '--format=%B', 'fake_ancestor_sha'],),
       'This is parent commit.\n'
       '\n'
       'Cr-Commit-Position: refs/heads/master@{#543}\n'
       'Cr-Branched-From: refs/svn/2014@{#2208}'),
      ((['git', 'show', '-s', '--format=%ct', 'fake_ancestor_sha'],),
       '1480022355'),  # Committer's unix timestamp.
      ((['git', 'show', '-s', '--format=%ct', 'HEAD'],),
       '1480024000'),

      ((['git_amend_head committer timestamp', 1480024000],), None),
      ((['git', 'commit', '--amend', '-m',
        'Issue: 123\n\nR=john@chromium.org\n'
        '\n'
        'Review-Url: https://codereview.chromium.org/123 .\n'
        'Cr-Commit-Position: refs/heads/master@{#544}\n'
        'Cr-Branched-From: refs/svn/2014@{#2208}'],), ''),

      ((['git', 'push', '--porcelain', 'origin', 'HEAD:refs/heads/master'],),
       ''),
      ((['git', 'rev-parse', 'HEAD'],), 'fake_sha_rebased'),
      ((['git', 'checkout', '-q', 'feature'],), ''),
      ((['git', 'branch', '-D', 'git-cl-commit'],), ''),
      ((['git', 'config', 'rietveld.viewvc-url'],),
       'https://chromium.googlesource.com/infra/infra/+/'),
      ((['update_description', 123,
         'Issue: 123\n\nR=john@chromium.org\n'
         '\n'
         'Review-Url: https://codereview.chromium.org/123 .\n'
         'Cr-Commit-Position: refs/heads/master@{#544}\n'
         'Cr-Branched-From: refs/svn/2014@{#2208}\n'
         'Committed: '
         'https://chromium.googlesource.com/infra/infra/+/fake_sha_rebased'],),
       ''),
      ((['add_comment', 123, 'Committed patchset #2 (id:20001) manually as '
                             'fake_sha_rebased (presubmit successful).'],), ''),
    ]
    git_cl.main(['land'])

  def test_land_rietveld_git_numberer_bad_parent(self):
    self._land_rietveld_common(debug=False)
    self.calls += [
      ((['git', 'config', 'remote.origin.url'],),
       'https://chromium.googlesource.com/v8/v8'),
      ((['_GitNumbererState',
         'https://chromium.googlesource.com/v8/v8', 'refs/heads/master'],),
       git_cl._GitNumbererState(None, True)),

      ((['git', 'show', '-s', '--format=%B', 'fake_ancestor_sha'],),
       'This is parent commit with no footer.'),

      ((['git', 'checkout', '-q', 'feature'],), ''),
      ((['git', 'branch', '-D', 'git-cl-commit'],), ''),
    ]
    with self.assertRaises(ValueError) as cm:
      git_cl.main(['land'])
    self.assertEqual(cm.exception.message,
                     'Unable to infer commit position from footers')

  def test_GitNumbererState_not_whitelisted_repo(self):
    self.calls = [
        ((['git', 'config', 'rietveld.autoupdate'],), CERR1),
        ((['git', 'config', 'rietveld.pending-ref-prefix'],), CERR1),
    ]
    res = git_cl._GitNumbererState.load(
        remote_url='https://chromium.googlesource.com/chromium/tools/build',
        remote_ref='refs/whatever')
    self.assertEqual(res.pending_prefix, None)
    self.assertEqual(res.should_git_number, False)

  def test_GitNumbererState_fail_fetch(self):
    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())
    self.calls = [
        ((['git', 'fetch', 'https://chromium.googlesource.com/chromium/src',
           '+refs/meta/config:refs/git_cl/meta/config',
           '+refs/gnumbd-config/main:refs/git_cl/gnumbd-config/main'],), CERR1),
        ((['git', 'config', 'rietveld.autoupdate'],), CERR1),
        ((['git', 'config', 'rietveld.pending-ref-prefix'],),
         'refs/pending-prefix'),
    ]
    res = git_cl._GitNumbererState.load(
        remote_url='https://chromium.googlesource.com/chromium/src',
        remote_ref='refs/whatever')
    self.assertEqual(res.pending_prefix, 'refs/pending-prefix/')
    self.assertEqual(res.should_git_number, False)

  def test_GitNumbererState_fail_gnumbd_and_validator(self):
    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())
    self.calls = [
        ((['git', 'fetch', 'https://chromium.googlesource.com/chromium/src',
           '+refs/meta/config:refs/git_cl/meta/config',
           '+refs/gnumbd-config/main:refs/git_cl/gnumbd-config/main'],), ''),
        ((['git', 'show', 'refs/git_cl/gnumbd-config/main:config.json'],),
         'ba d conig'),
        ((['git', 'config', 'rietveld.autoupdate'],), CERR1),
        ((['git', 'config', 'rietveld.pending-ref-prefix'],), CERR1),
        ((['git', 'show', 'refs/git_cl/meta/config:project.config'],), CERR1),
    ]
    res = git_cl._GitNumbererState.load(
        remote_url='https://chromium.googlesource.com/chromium/src',
        remote_ref='refs/whatever')
    self.assertEqual(res.pending_prefix, None)
    self.assertEqual(res.should_git_number, False)

  def test_GitNumbererState_valid_configs(self):
    class NamedTempFileStab(StringIO.StringIO):
      @classmethod
      @contextlib.contextmanager
      def create(cls, *_, **__):
        yield cls()
      name = 'tempfile'
    self.mock(git_cl.tempfile, 'NamedTemporaryFile', NamedTempFileStab.create)
    self.calls = [
        ((['git', 'fetch', 'https://chromium.googlesource.com/chromium/src',
           '+refs/meta/config:refs/git_cl/meta/config',
           '+refs/gnumbd-config/main:refs/git_cl/gnumbd-config/main'],), ''),
        ((['git', 'show', 'refs/git_cl/gnumbd-config/main:config.json'],),
         '''{
              "pending_tag_prefix": "refs/pending-tags",
              "pending_ref_prefix": "refs/pending",
              "enabled_refglobs": [
                "refs/heads/m*"
              ]
            }
         '''),
        ((['git', 'show', 'refs/git_cl/meta/config:project.config'],),
         '''
          [plugin "git-numberer"]
            validate-enabled-refglob = refs/else/*
            validate-enabled-refglob = refs/heads/*
            validate-disabled-refglob = refs/heads/disabled
            validate-disabled-refglob = refs/branch-heads/*
         '''),
        ((['git', 'config', '-f', 'tempfile', '--get-all',
           'plugin.git-numberer.validate-enabled-refglob'],),
         'refs/else/*\n'
         'refs/heads/*\n'),
        ((['git', 'config', '-f', 'tempfile', '--get-all',
           'plugin.git-numberer.validate-disabled-refglob'],),
         'refs/heads/disabled\n'
         'refs/branch-heads/*\n'),
    ] * 4  # 4 tests below have exactly same IO.

    res = git_cl._GitNumbererState.load(
        remote_url='https://chromium.googlesource.com/chromium/src',
        remote_ref='refs/heads/master')
    self.assertEqual(res.pending_prefix, 'refs/pending/')
    self.assertEqual(res.should_git_number, False)

    res = git_cl._GitNumbererState.load(
        remote_url='https://chromium.googlesource.com/chromium/src',
        remote_ref='refs/heads/test')
    self.assertEqual(res.pending_prefix, None)
    self.assertEqual(res.should_git_number, True)

    res = git_cl._GitNumbererState.load(
        remote_url='https://chromium.googlesource.com/chromium/src',
        remote_ref='refs/heads/disabled')
    self.assertEqual(res.pending_prefix, None)
    self.assertEqual(res.should_git_number, False)

    # Validator is disabled by default, even if it's not explicitely in disabled
    # refglobs.
    res = git_cl._GitNumbererState.load(
        remote_url='https://chromium.googlesource.com/chromium/src',
        remote_ref='refs/arbitrary/ref')
    self.assertEqual(res.pending_prefix, None)
    self.assertEqual(res.should_git_number, False)

  @classmethod
  def _gerrit_ensure_auth_calls(cls, issue=None, skip_auth_check=False):
    cmd = ['git', 'config', '--bool', 'gerrit.skip-ensure-authenticated']
    if skip_auth_check:
      return [((cmd, ), 'true')]

    calls = [((cmd, ), CERR1)]
    if issue:
      calls.extend([
          ((['git', 'config', 'branch.master.gerritserver'],), ''),
      ])
    calls.extend([
        ((['git', 'config', 'branch.master.merge'],), 'refs/heads/master'),
        ((['git', 'config', 'branch.master.remote'],), 'origin'),
        ((['git', 'config', 'remote.origin.url'],),
         'https://chromium.googlesource.com/my/repo'),
        ((['git', 'config', 'remote.origin.url'],),
         'https://chromium.googlesource.com/my/repo'),
    ])
    return calls

  @classmethod
  def _gerrit_base_calls(cls, issue=None):
    return [
        ((['git', 'symbolic-ref', 'HEAD'],), 'master'),
        ((['git', 'config', 'branch.master.git-cl-similarity'],),
         CERR1),
        ((['git', 'symbolic-ref', 'HEAD'],), 'master'),
        ((['git', 'config', '--bool', 'branch.master.git-find-copies'],),
         CERR1),
      ] + cls._is_gerrit_calls(True) + [
        ((['git', 'symbolic-ref', 'HEAD'],), 'master'),
        ((['git', 'config', 'branch.master.rietveldissue'],), CERR1),
        ((['git', 'config', 'branch.master.gerritissue'],),
          CERR1 if issue is None else str(issue)),
        ((['git', 'config', 'branch.master.merge'],), 'refs/heads/master'),
        ((['git', 'config', 'branch.master.remote'],), 'origin'),
        ((['get_or_create_merge_base', 'master',
           'refs/remotes/origin/master'],),
         'fake_ancestor_sha'),
        # Calls to verify branch point is ancestor
      ] + (cls._gerrit_ensure_auth_calls(issue=issue) +
           cls._git_sanity_checks('fake_ancestor_sha', 'master',
                                  get_remote_branch=False)) + [
        ((['git', 'rev-parse', '--show-cdup'],), ''),
        ((['git', 'rev-parse', 'HEAD'],), '12345'),

        ((['git',
           'diff', '--name-status', '--no-renames', '-r',
           'fake_ancestor_sha...', '.'],),
         'M\t.gitignore\n'),
        ((['git', 'config', 'branch.master.gerritpatchset'],), CERR1),
      ] + ([] if issue else [
        ((['git',
           'log', '--pretty=format:%s%n%n%b', 'fake_ancestor_sha...'],),
         'foo'),
      ]) + [
        ((['git', 'config', 'user.email'],), 'me@example.com'),
        ((['git',
           'diff', '--no-ext-diff', '--stat', '-l100000', '-C50',
           'fake_ancestor_sha', 'HEAD'],),
         '+dat'),
      ]

  @classmethod
  def _gerrit_upload_calls(cls, description, reviewers, squash,
                           squash_mode='default',
                           expected_upstream_ref='origin/refs/heads/master',
                           ref_suffix='', notify=False,
                           post_amend_description=None, issue=None, cc=None):
    if post_amend_description is None:
      post_amend_description = description
    calls = []
    cc = cc or []

    if squash_mode == 'default':
      calls.extend([
        ((['git', 'config', '--bool', 'gerrit.override-squash-uploads'],), ''),
        ((['git', 'config', '--bool', 'gerrit.squash-uploads'],), ''),
      ])
    elif squash_mode in ('override_squash', 'override_nosquash'):
      calls.extend([
        ((['git', 'config', '--bool', 'gerrit.override-squash-uploads'],),
         'true' if squash_mode == 'override_squash' else 'false'),
      ])
    else:
      assert squash_mode in ('squash', 'nosquash')

    # If issue is given, then description is fetched from Gerrit instead.
    if issue is None:
      calls += [
          ((['git', 'log', '--pretty=format:%s\n\n%b',
                   'fake_ancestor_sha..HEAD'],),
                  description)]
    if not git_footers.get_footer_change_id(description) and not squash:
      calls += [
          # DownloadGerritHook(False)
          ((False, ),
            ''),
          # Amending of commit message to get the Change-Id.
          ((['git', 'log', '--pretty=format:%s\n\n%b',
             'fake_ancestor_sha..HEAD'],),
           description),
          ((['git', 'commit', '--amend', '-m', description],),
           ''),
          ((['git', 'log', '--pretty=format:%s\n\n%b',
             'fake_ancestor_sha..HEAD'],),
           post_amend_description)
          ]
    if squash:
      if not issue:
        # Prompting to edit description on first upload.
        calls += [
            ((['git', 'config', 'core.editor'],), ''),
            ((['RunEditor'],), description),
        ]
      ref_to_push = 'abcdef0123456789'
      calls += [
          ((['git', 'config', 'branch.master.merge'],),
           'refs/heads/master'),
          ((['git', 'config', 'branch.master.remote'],),
           'origin'),
          ((['get_or_create_merge_base', 'master',
             'refs/remotes/origin/master'],),
           'origin/master'),
          ((['git', 'rev-parse', 'HEAD:'],),
           '0123456789abcdef'),
          ((['git', 'commit-tree', '0123456789abcdef', '-p',
             'origin/master', '-m', description],),
           ref_to_push),
          ]
    else:
      ref_to_push = 'HEAD'

    calls += [
        ((['git', 'rev-list',
            expected_upstream_ref + '..' + ref_to_push],), ''),
        ]

    notify_suffix = 'notify=%s' % ('ALL' if notify else 'NONE')
    if ref_suffix:
      ref_suffix += ',' + notify_suffix
    else:
      ref_suffix = '%' + notify_suffix

    if reviewers:
      ref_suffix += ',' + ','.join('r=%s' % email
                                   for email in sorted(reviewers))
    calls += [
        ((['git', 'push', 'origin',
           ref_to_push + ':refs/for/refs/heads/master' + ref_suffix],),
         ('remote:\n'
         'remote: Processing changes: (\)\n'
         'remote: Processing changes: (|)\n'
         'remote: Processing changes: (/)\n'
         'remote: Processing changes: (-)\n'
         'remote: Processing changes: new: 1 (/)\n'
         'remote: Processing changes: new: 1, done\n'
         'remote:\n'
         'remote: New Changes:\n'
         'remote:   https://chromium-review.googlesource.com/123456 XXX.\n'
         'remote:\n'
         'To https://chromium.googlesource.com/yyy/zzz\n'
         ' * [new branch]      hhhh -> refs/for/refs/heads/master\n')),
        ]
    if squash:
      calls += [
          ((['git', 'config', 'branch.master.gerritissue', '123456'],),
           ''),
          ((['git', 'config', 'branch.master.gerritserver',
             'https://chromium-review.googlesource.com'],), ''),
          ((['git', 'config', 'branch.master.gerritsquashhash',
             'abcdef0123456789'],), ''),
      ]
    calls += [
        ((['git', 'config', 'rietveld.cc'],), ''),
        ((['AddReviewers', 'chromium-review.googlesource.com',
           123456 if squash else None,
          ['joe@example.com'] + cc, False],), ''),
    ]
    calls += cls._git_post_upload_calls()
    return calls

  def _run_gerrit_upload_test(
      self,
      upload_args,
      description,
      reviewers=None,
      squash=True,
      squash_mode=None,
      expected_upstream_ref='origin/refs/heads/master',
      ref_suffix='',
      notify=False,
      post_amend_description=None,
      issue=None,
      cc=None):
    """Generic gerrit upload test framework."""
    if squash_mode is None:
      if '--no-squash' in upload_args:
        squash_mode = 'nosquash'
      elif '--squash' in upload_args:
        squash_mode = 'squash'
      else:
        squash_mode = 'default'

    reviewers = reviewers or []
    cc = cc or []
    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())
    self.mock(git_cl.gerrit_util, 'CookiesAuthenticator',
              CookiesAuthenticatorMockFactory(same_cookie='same_cred'))
    self.mock(git_cl._GerritChangelistImpl, '_GerritCommitMsgHookCheck',
              lambda _, offer_removal: None)
    self.mock(git_cl.gclient_utils, 'RunEditor',
              lambda *_, **__: self._mocked_call(['RunEditor']))
    self.mock(git_cl, 'DownloadGerritHook', self._mocked_call)
    self.mock(git_cl.gerrit_util, 'AddReviewers',
              lambda h, i, add, is_reviewer: self._mocked_call(
                  ['AddReviewers', h, i, add, is_reviewer]))

    self.calls = self._gerrit_base_calls(issue=issue)
    self.calls += self._gerrit_upload_calls(
        description, reviewers, squash,
        squash_mode=squash_mode,
        expected_upstream_ref=expected_upstream_ref,
        ref_suffix=ref_suffix, notify=notify,
        post_amend_description=post_amend_description,
        issue=issue, cc=cc)
    # Uncomment when debugging.
    # print '\n'.join(map(lambda x: '%2i: %s' % x, enumerate(self.calls)))
    git_cl.main(['upload'] + upload_args)

  def test_gerrit_upload_without_change_id(self):
    self._run_gerrit_upload_test(
        ['--no-squash'],
        'desc\n\nBUG=\n',
        [],
        squash=False,
        post_amend_description='desc\n\nBUG=\n\nChange-Id: Ixxx')

  def test_gerrit_upload_without_change_id_override_nosquash(self):
    self.mock(git_cl, 'DownloadGerritHook', self._mocked_call)
    self._run_gerrit_upload_test(
        [],
        'desc\n\nBUG=\n',
        [],
        squash=False,
        squash_mode='override_nosquash',
        post_amend_description='desc\n\nBUG=\n\nChange-Id: Ixxx')

  def test_gerrit_no_reviewer(self):
    self._run_gerrit_upload_test(
        [],
        'desc\n\nBUG=\n\nChange-Id: I123456789\n',
        [],
        squash=False,
        squash_mode='override_nosquash')

  def test_gerrit_patch_bad_chars(self):
    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())
    self._run_gerrit_upload_test(
        ['-f', '-t', 'Don\'t put bad cha,.rs'],
        'desc\n\nBUG=\n\nChange-Id: I123456789',
        squash=False,
        squash_mode='override_nosquash',
        ref_suffix='%m=Dont_put_bad_chars')
    self.assertIn(
        'WARNING: Patchset title may only contain alphanumeric chars '
        'and spaces. Cleaned up title:\nDont put bad chars\n',
        git_cl.sys.stdout.getvalue())

  def test_gerrit_reviewers_cmd_line(self):
    self._run_gerrit_upload_test(
        ['-r', 'foo@example.com', '--send-mail'],
        'desc\n\nBUG=\n\nChange-Id: I123456789',
        ['foo@example.com'],
        squash=False,
        squash_mode='override_nosquash',
        notify=True)

  def test_gerrit_reviewer_multiple(self):
    self._run_gerrit_upload_test(
        [],
        'desc\nTBR=reviewer@example.com\nBUG=\nR=another@example.com\n'
        'CC=more@example.com,people@example.com\n\n'
        'Change-Id: 123456789\n',
        ['reviewer@example.com', 'another@example.com'],
        squash=False,
        squash_mode='override_nosquash',
        ref_suffix='%l=Code-Review+1',
        cc=['more@example.com', 'people@example.com'])

  def test_gerrit_upload_squash_first_is_default(self):
    # Mock Gerrit CL description to indicate the first upload.
    self.mock(git_cl.Changelist, 'GetDescription',
              lambda *_: None)
    self._run_gerrit_upload_test(
        [],
        'desc\nBUG=\n\nChange-Id: 123456789',
        [],
        expected_upstream_ref='origin/master')

  def test_gerrit_upload_squash_first(self):
    # Mock Gerrit CL description to indicate the first upload.
    self.mock(git_cl.Changelist, 'GetDescription',
              lambda *_: None)
    self._run_gerrit_upload_test(
        ['--squash'],
        'desc\nBUG=\n\nChange-Id: 123456789',
        [],
        squash=True,
        expected_upstream_ref='origin/master')

  def test_gerrit_upload_squash_reupload(self):
    description = 'desc\nBUG=\n\nChange-Id: 123456789'
    # Mock Gerrit CL description to indicate re-upload.
    self.mock(git_cl.Changelist, 'GetDescription',
              lambda *args: description)
    self.mock(git_cl.Changelist, 'GetMostRecentPatchset',
              lambda *args: 1)
    self.mock(git_cl._GerritChangelistImpl, '_GetChangeDetail',
              lambda *args: {'change_id': '123456789'})
    self._run_gerrit_upload_test(
        ['--squash'],
        description,
        [],
        squash=True,
        expected_upstream_ref='origin/master',
        issue=123456)

  def test_upload_branch_deps(self):
    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())
    def mock_run_git(*args, **_kwargs):
      if args[0] == ['for-each-ref',
                       '--format=%(refname:short) %(upstream:short)',
                       'refs/heads']:
        # Create a local branch dependency tree that looks like this:
        # test1 -> test2 -> test3   -> test4 -> test5
        #                -> test3.1
        # test6 -> test0
        branch_deps = [
            'test2 test1',    # test1 -> test2
            'test3 test2',    # test2 -> test3
            'test3.1 test2',  # test2 -> test3.1
            'test4 test3',    # test3 -> test4
            'test5 test4',    # test4 -> test5
            'test6 test0',    # test0 -> test6
            'test7',          # test7
        ]
        return '\n'.join(branch_deps)
    self.mock(git_cl, 'RunGit', mock_run_git)

    class RecordCalls:
      times_called = 0
    record_calls = RecordCalls()
    def mock_CMDupload(*args, **_kwargs):
      record_calls.times_called += 1
      return 0
    self.mock(git_cl, 'CMDupload', mock_CMDupload)

    self.calls = [
        (('[Press enter to continue or ctrl-C to quit]',), ''),
      ]

    class MockChangelist():
      def __init__(self):
        pass
      def GetBranch(self):
        return 'test1'
      def GetIssue(self):
        return '123'
      def GetPatchset(self):
        return '1001'
      def IsGerrit(self):
        return False

    ret = git_cl.upload_branch_deps(MockChangelist(), [])
    # CMDupload should have been called 5 times because of 5 dependent branches.
    self.assertEquals(5, record_calls.times_called)
    self.assertEquals(0, ret)

  def test_gerrit_change_id(self):
    self.calls = [
        ((['git', 'write-tree'], ),
          'hashtree'),
        ((['git', 'rev-parse', 'HEAD~0'], ),
          'branch-parent'),
        ((['git', 'var', 'GIT_AUTHOR_IDENT'], ),
          'A B <a@b.org> 1456848326 +0100'),
        ((['git', 'var', 'GIT_COMMITTER_IDENT'], ),
          'C D <c@d.org> 1456858326 +0100'),
        ((['git', 'hash-object', '-t', 'commit', '--stdin'], ),
          'hashchange'),
    ]
    change_id = git_cl.GenerateGerritChangeId('line1\nline2\n')
    self.assertEqual(change_id, 'Ihashchange')

  def test_desecription_append_footer(self):
    for init_desc, footer_line, expected_desc in [
      # Use unique desc first lines for easy test failure identification.
      ('foo', 'R=one', 'foo\n\nR=one'),
      ('foo\n\nR=one', 'BUG=', 'foo\n\nR=one\nBUG='),
      ('foo\n\nR=one', 'Change-Id: Ixx', 'foo\n\nR=one\n\nChange-Id: Ixx'),
      ('foo\n\nChange-Id: Ixx', 'R=one', 'foo\n\nR=one\n\nChange-Id: Ixx'),
      ('foo\n\nR=one\n\nChange-Id: Ixx', 'TBR=two',
       'foo\n\nR=one\nTBR=two\n\nChange-Id: Ixx'),
      ('foo\n\nR=one\n\nChange-Id: Ixx', 'Foo-Bar: baz',
       'foo\n\nR=one\n\nChange-Id: Ixx\nFoo-Bar: baz'),
      ('foo\n\nChange-Id: Ixx', 'Foo-Bak: baz',
       'foo\n\nChange-Id: Ixx\nFoo-Bak: baz'),
      ('foo', 'Change-Id: Ixx', 'foo\n\nChange-Id: Ixx'),
    ]:
      desc = git_cl.ChangeDescription(init_desc)
      desc.append_footer(footer_line)
      self.assertEqual(desc.description, expected_desc)

  def test_update_reviewers(self):
    data = [
      ('foo', [], 'foo'),
      ('foo\nR=xx', [], 'foo\nR=xx'),
      ('foo\nTBR=xx', [], 'foo\nTBR=xx'),
      ('foo', ['a@c'], 'foo\n\nR=a@c'),
      ('foo\nR=xx', ['a@c'], 'foo\n\nR=a@c, xx'),
      ('foo\nTBR=xx', ['a@c'], 'foo\n\nR=a@c\nTBR=xx'),
      ('foo\nTBR=xx\nR=yy', ['a@c'], 'foo\n\nR=a@c, yy\nTBR=xx'),
      ('foo\nBUG=', ['a@c'], 'foo\nBUG=\nR=a@c'),
      ('foo\nR=xx\nTBR=yy\nR=bar', ['a@c'], 'foo\n\nR=a@c, xx, bar\nTBR=yy'),
      ('foo', ['a@c', 'b@c'], 'foo\n\nR=a@c, b@c'),
      ('foo\nBar\n\nR=\nBUG=', ['c@c'], 'foo\nBar\n\nR=c@c\nBUG='),
      ('foo\nBar\n\nR=\nBUG=\nR=', ['c@c'], 'foo\nBar\n\nR=c@c\nBUG='),
      # Same as the line before, but full of whitespaces.
      (
        'foo\nBar\n\n R = \n BUG = \n R = ', ['c@c'],
        'foo\nBar\n\nR=c@c\n BUG =',
      ),
      # Whitespaces aren't interpreted as new lines.
      ('foo BUG=allo R=joe ', ['c@c'], 'foo BUG=allo R=joe\n\nR=c@c'),
    ]
    expected = [i[2] for i in data]
    actual = []
    for orig, reviewers, _expected in data:
      obj = git_cl.ChangeDescription(orig)
      obj.update_reviewers(reviewers)
      actual.append(obj.description)
    self.assertEqual(expected, actual)

  def test_get_target_ref(self):
    # Check remote or remote branch not present.
    self.assertEqual(None, git_cl.GetTargetRef('origin', None, 'master', False))
    self.assertEqual(None, git_cl.GetTargetRef(None,
                                               'refs/remotes/origin/master',
                                               'master', False))

    # Check default target refs for branches.
    self.assertEqual('refs/heads/master',
                     git_cl.GetTargetRef('origin', 'refs/remotes/origin/master',
                                         None, False))
    self.assertEqual('refs/heads/master',
                     git_cl.GetTargetRef('origin', 'refs/remotes/origin/lkgr',
                                         None, False))
    self.assertEqual('refs/heads/master',
                     git_cl.GetTargetRef('origin', 'refs/remotes/origin/lkcr',
                                         None, False))
    self.assertEqual('refs/branch-heads/123',
                     git_cl.GetTargetRef('origin',
                                         'refs/remotes/branch-heads/123',
                                         None, False))
    self.assertEqual('refs/diff/test',
                     git_cl.GetTargetRef('origin',
                                         'refs/remotes/origin/refs/diff/test',
                                         None, False))
    self.assertEqual('refs/heads/chrome/m42',
                     git_cl.GetTargetRef('origin',
                                         'refs/remotes/origin/chrome/m42',
                                         None, False))

    # Check target refs for user-specified target branch.
    for branch in ('branch-heads/123', 'remotes/branch-heads/123',
                   'refs/remotes/branch-heads/123'):
      self.assertEqual('refs/branch-heads/123',
                       git_cl.GetTargetRef('origin',
                                           'refs/remotes/origin/master',
                                           branch, False))
    for branch in ('origin/master', 'remotes/origin/master',
                   'refs/remotes/origin/master'):
      self.assertEqual('refs/heads/master',
                       git_cl.GetTargetRef('origin',
                                           'refs/remotes/branch-heads/123',
                                           branch, False))
    for branch in ('master', 'heads/master', 'refs/heads/master'):
      self.assertEqual('refs/heads/master',
                       git_cl.GetTargetRef('origin',
                                           'refs/remotes/branch-heads/123',
                                           branch, False))

    # Check target refs for pending prefix.
    self.mock(git_cl._GitNumbererState, 'load',
              classmethod(lambda *_: git_cl._GitNumbererState('prefix', False)))
    self.assertEqual('prefix/heads/master',
                     git_cl.GetTargetRef('origin', 'refs/remotes/origin/master',
                                         None, True,
                                         'https://remote.url/some.git'))

  def test_patch_when_dirty(self):
    # Patch when local tree is dirty
    self.mock(git_common, 'is_dirty_git_tree', lambda x: True)
    self.assertNotEqual(git_cl.main(['patch', '123456']), 0)

  def test_diff_when_dirty(self):
    # Do 'git cl diff' when local tree is dirty
    self.mock(git_common, 'is_dirty_git_tree', lambda x: True)
    self.assertNotEqual(git_cl.main(['diff']), 0)

  def _patch_common(self, is_gerrit=False, force_codereview=False,
                    new_branch=False):
    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())
    self.mock(git_cl._RietveldChangelistImpl, 'GetMostRecentPatchset',
              lambda x: '60001')
    self.mock(git_cl._GerritChangelistImpl, '_GetChangeDetail',
              lambda *args: {
                'current_revision': '7777777777',
                'revisions': {
                  '1111111111': {
                    '_number': 1,
                    'fetch': {'http': {
                      'url': 'https://chromium.googlesource.com/my/repo',
                      'ref': 'refs/changes/56/123456/1',
                    }},
                  },
                  '7777777777': {
                    '_number': 7,
                    'fetch': {'http': {
                      'url': 'https://chromium.googlesource.com/my/repo',
                      'ref': 'refs/changes/56/123456/7',
                    }},
                  },
                },
              })
    self.mock(git_cl.Changelist, 'GetDescription',
              lambda *args: 'Description')
    self.mock(git_cl, 'IsGitVersionAtLeast', lambda *args: True)

    if new_branch:
      self.calls = [((['git', 'new-branch', 'master'],), ''),]
    else:
      self.calls = [((['git', 'symbolic-ref', 'HEAD'],), 'master')]
    if not force_codereview:
      # These calls detect codereview to use.
      self.calls += [
        ((['git', 'symbolic-ref', 'HEAD'],), 'master'),
        ((['git', 'config', 'branch.master.rietveldissue'],), CERR1),
        ((['git', 'config', 'branch.master.gerritissue'],), CERR1),
        ((['git', 'config', 'rietveld.autoupdate'],), CERR1),
      ]

    if is_gerrit:
      if not force_codereview:
        self.calls += [
          ((['git', 'config', 'gerrit.host'],), 'true'),
        ]
    else:
      self.calls += [
        ((['git', 'config', 'gerrit.host'],), CERR1),
        ((['git', 'config', 'rietveld.server'],), 'codereview.example.com'),
        ((['git', 'config', 'branch.master.rietveldserver',],), CERR1),
        ((['git', 'rev-parse', '--show-cdup'],), ''),
      ]

  def _common_patch_successful(self, new_branch=False):
    self._patch_common(new_branch=new_branch)
    self.calls += [
      ((['git', 'commit', '-m',
         'Description\n\n' +
         'patch from issue 123456 at patchset 60001 ' +
         '(http://crrev.com/123456#ps60001)'],), ''),
      ((['git', 'config', 'branch.master.rietveldissue', '123456'],),
       ''),
      ((['git', 'config', 'branch.master.rietveldserver',
         'https://codereview.example.com'],), ''),
      ((['git', 'config', 'branch.master.rietveldpatchset', '60001'],),
       ''),
    ]

  def test_patch_successful(self):
    self._common_patch_successful()
    self.assertEqual(git_cl.main(['patch', '123456']), 0)

  def test_patch_successful_new_branch(self):
    self._common_patch_successful(new_branch=True)
    self.assertEqual(git_cl.main(['patch', '-b', 'master', '123456']), 0)

  def test_patch_conflict(self):
    self._patch_common()
    GitCheckoutMock.conflict = True
    self.assertNotEqual(git_cl.main(['patch', '123456']), 0)

  def test_gerrit_patch_successful(self):
    self._patch_common(is_gerrit=True)
    self.calls += [
      ((['git', 'fetch', 'https://chromium.googlesource.com/my/repo',
         'refs/changes/56/123456/7'],), ''),
      ((['git', 'cherry-pick', 'FETCH_HEAD'],), ''),
      ((['git', 'config', 'branch.master.gerritissue', '123456'],),
       ''),
      ((['git', 'config', 'branch.master.gerritserver'],), ''),
      ((['git', 'config', 'branch.master.merge'],), 'master'),
      ((['git', 'config', 'branch.master.remote'],), 'origin'),
      ((['git', 'config', 'remote.origin.url'],),
       'https://chromium.googlesource.com/my/repo'),
      ((['git', 'config', 'branch.master.gerritserver',
        'https://chromium-review.googlesource.com'],), ''),
      ((['git', 'config', 'branch.master.gerritpatchset', '7'],), ''),
    ]
    self.assertEqual(git_cl.main(['patch', '123456']), 0)

  def test_patch_force_codereview(self):
    self._patch_common(is_gerrit=True, force_codereview=True)
    self.calls += [
      ((['git', 'fetch', 'https://chromium.googlesource.com/my/repo',
         'refs/changes/56/123456/7'],), ''),
      ((['git', 'cherry-pick', 'FETCH_HEAD'],), ''),
      ((['git', 'symbolic-ref', 'HEAD'],), 'master'),
      ((['git', 'config', 'branch.master.gerritissue', '123456'],),
       ''),
      ((['git', 'config', 'branch.master.gerritserver'],), ''),
      ((['git', 'config', 'branch.master.merge'],), 'master'),
      ((['git', 'config', 'branch.master.remote'],), 'origin'),
      ((['git', 'config', 'remote.origin.url'],),
       'https://chromium.googlesource.com/my/repo'),
      ((['git', 'config', 'branch.master.gerritserver',
        'https://chromium-review.googlesource.com'],), ''),
      ((['git', 'config', 'branch.master.gerritpatchset', '7'],), ''),
    ]
    self.assertEqual(git_cl.main(['patch', '--gerrit', '123456']), 0)

  def test_gerrit_patch_url_successful(self):
    self._patch_common(is_gerrit=True)
    self.calls += [
      ((['git', 'fetch', 'https://chromium.googlesource.com/my/repo',
         'refs/changes/56/123456/1'],), ''),
      ((['git', 'cherry-pick', 'FETCH_HEAD'],), ''),
      ((['git', 'config', 'branch.master.gerritissue', '123456'],),
       ''),
      ((['git', 'config', 'branch.master.gerritserver',
        'https://chromium-review.googlesource.com'],), ''),
      ((['git', 'config', 'branch.master.gerritpatchset', '1'],), ''),
    ]
    self.assertEqual(git_cl.main(
      ['patch', 'https://chromium-review.googlesource.com/#/c/123456/1']), 0)

  def test_gerrit_patch_conflict(self):
    self._patch_common(is_gerrit=True)
    self.calls += [
      ((['git', 'fetch', 'https://chromium.googlesource.com/my/repo',
         'refs/changes/56/123456/1'],), ''),
      ((['git', 'cherry-pick', 'FETCH_HEAD'],), CERR1),
      ((['DieWithError', 'Command "git cherry-pick FETCH_HEAD" failed.\n'],),
       SystemExitMock()),
    ]
    with self.assertRaises(SystemExitMock):
      git_cl.main(['patch',
                   'https://chromium-review.googlesource.com/#/c/123456/1'])

  def test_gerrit_patch_not_exists(self):
    def notExists(_issue, *_, **kwargs):
      self.assertFalse(kwargs['ignore_404'])
      raise git_cl.gerrit_util.GerritError(404, '')
    self.mock(git_cl.gerrit_util, 'GetChangeDetail', notExists)

    url = 'https://chromium-review.googlesource.com'
    self.calls = [
      ((['git', 'symbolic-ref', 'HEAD'],), 'master'),
      ((['git', 'symbolic-ref', 'HEAD'],), 'master'),
      ((['git', 'config', 'branch.master.rietveldissue'],), CERR1),
      ((['git', 'config', 'branch.master.gerritissue'],), CERR1),
      ((['git', 'config', 'rietveld.autoupdate'],), CERR1),
      ((['git', 'config', 'gerrit.host'],), 'true'),
      ((['DieWithError', 'change 123456 at ' + url + ' does not exist '
                         'or you have no access to it'],), SystemExitMock()),
    ]
    with self.assertRaises(SystemExitMock):
      self.assertEqual(1, git_cl.main(['patch', url + '/#/c/123456/1']))

  def _checkout_calls(self):
    return [
        ((['git', 'config', '--local', '--get-regexp',
           'branch\\..*\\.rietveldissue'], ),
           ('branch.retrying.rietveldissue 1111111111\n'
            'branch.some-fix.rietveldissue 2222222222\n')),
        ((['git', 'config', '--local', '--get-regexp',
           'branch\\..*\\.gerritissue'], ),
           ('branch.ger-branch.gerritissue 123456\n'
            'branch.gbranch654.gerritissue 654321\n')),
    ]

  def test_checkout_gerrit(self):
    """Tests git cl checkout <issue>."""
    self.calls = self._checkout_calls()
    self.calls += [((['git', 'checkout', 'ger-branch'], ), '')]
    self.assertEqual(0, git_cl.main(['checkout', '123456']))

  def test_checkout_rietveld(self):
    """Tests git cl checkout <issue>."""
    self.calls = self._checkout_calls()
    self.calls += [((['git', 'checkout', 'some-fix'], ), '')]
    self.assertEqual(0, git_cl.main(['checkout', '2222222222']))

  def test_checkout_not_found(self):
    """Tests git cl checkout <issue>."""
    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())
    self.calls = self._checkout_calls()
    self.assertEqual(1, git_cl.main(['checkout', '99999']))

  def test_checkout_no_branch_issues(self):
    """Tests git cl checkout <issue>."""
    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())
    self.calls = [
        ((['git', 'config', '--local', '--get-regexp',
           'branch\\..*\\.rietveldissue'], ), CERR1),
        ((['git', 'config', '--local', '--get-regexp',
           'branch\\..*\\.gerritissue'], ), CERR1),
    ]
    self.assertEqual(1, git_cl.main(['checkout', '99999']))

  def _test_gerrit_ensure_authenticated_common(self, auth,
                                               skip_auth_check=False):
    self.mock(git_cl.gerrit_util, 'CookiesAuthenticator',
              CookiesAuthenticatorMockFactory(hosts_with_creds=auth))
    self.mock(git_cl, 'DieWithError',
              lambda msg: self._mocked_call(['DieWithError', msg]))
    self.mock(git_cl, 'ask_for_data',
              lambda msg: self._mocked_call(['ask_for_data', msg]))
    self.calls = self._gerrit_ensure_auth_calls(skip_auth_check=skip_auth_check)
    cl = git_cl.Changelist(codereview='gerrit')
    cl.branch = 'master'
    cl.branchref = 'refs/heads/master'
    cl.lookedup_issue = True
    return cl

  def test_gerrit_ensure_authenticated_missing(self):
    cl = self._test_gerrit_ensure_authenticated_common(auth={
      'chromium.googlesource.com': 'git is ok, but gerrit one is missing',
    })
    self.calls.append(
        ((['DieWithError',
           'Credentials for the following hosts are required:\n'
           '  chromium-review.googlesource.com\n'
           'These are read from ~/.gitcookies (or legacy ~/.netrc)\n'
           'You can (re)generate your credentails by visiting '
           'https://chromium-review.googlesource.com/new-password'],), ''),)
    self.assertIsNone(cl.EnsureAuthenticated(force=False))

  def test_gerrit_ensure_authenticated_conflict(self):
    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())
    cl = self._test_gerrit_ensure_authenticated_common(auth={
      'chromium.googlesource.com': 'one',
      'chromium-review.googlesource.com': 'other',
    })
    self.calls.append(
        ((['ask_for_data', 'If you know what you are doing, '
                           'press Enter to continue, Ctrl+C to abort.'],), ''))
    self.assertIsNone(cl.EnsureAuthenticated(force=False))

  def test_gerrit_ensure_authenticated_ok(self):
    cl = self._test_gerrit_ensure_authenticated_common(auth={
      'chromium.googlesource.com': 'same',
      'chromium-review.googlesource.com': 'same',
    })
    self.assertIsNone(cl.EnsureAuthenticated(force=False))

  def test_gerrit_ensure_authenticated_skipped(self):
    cl = self._test_gerrit_ensure_authenticated_common(
        auth={}, skip_auth_check=True)
    self.assertIsNone(cl.EnsureAuthenticated(force=False))

  def test_cmd_set_commit_rietveld(self):
    self.mock(git_cl._RietveldChangelistImpl, 'SetFlags',
              lambda _, v: self._mocked_call(['SetFlags', v]))
    self.calls = [
        ((['git', 'symbolic-ref', 'HEAD'],), 'feature'),
        ((['git', 'config', 'branch.feature.rietveldissue'],), '123'),
        ((['git', 'config', 'rietveld.autoupdate'],), ''),
        ((['git', 'config', 'rietveld.server'],), ''),
        ((['git', 'config', 'rietveld.server'],), ''),
        ((['git', 'config', 'branch.feature.rietveldserver'],),
         'https://codereview.chromium.org'),
        ((['SetFlags', {'commit': '1', 'cq_dry_run': '0'}], ), ''),
    ]
    self.assertEqual(0, git_cl.main(['set-commit']))

  def _cmd_set_commit_gerrit_common(self, vote, notify=None):
    self.mock(git_cl.gerrit_util, 'SetReview',
              lambda h, i, labels, notify=None:
                  self._mocked_call(['SetReview', h, i, labels, notify]))

    self.calls = [
        ((['git', 'symbolic-ref', 'HEAD'],), 'feature'),
        ((['git', 'config', 'branch.feature.rietveldissue'],), CERR1),
        ((['git', 'config', 'branch.feature.gerritissue'],), '123'),
        ((['git', 'config', 'branch.feature.gerritserver'],),
         'https://chromium-review.googlesource.com'),
        ((['SetReview', 'chromium-review.googlesource.com', 123,
           {'Commit-Queue': vote}, notify],), ''),
    ]

  def test_cmd_set_commit_gerrit_clear(self):
    self._cmd_set_commit_gerrit_common(0)
    self.assertEqual(0, git_cl.main(['set-commit', '-c']))

  def test_cmd_set_commit_gerrit_dry(self):
    self._cmd_set_commit_gerrit_common(1, notify='NONE')
    self.assertEqual(0, git_cl.main(['set-commit', '-d']))

  def test_cmd_set_commit_gerrit(self):
    self._cmd_set_commit_gerrit_common(2)
    self.assertEqual(0, git_cl.main(['set-commit']))

  def test_description_display(self):
    out = StringIO.StringIO()
    self.mock(git_cl.sys, 'stdout', out)

    self.mock(git_cl, 'Changelist', ChangelistMock)
    ChangelistMock.desc = 'foo\n'

    self.assertEqual(0, git_cl.main(['description', '-d']))
    self.assertEqual('foo\n', out.getvalue())

  def test_description_rietveld(self):
    out = StringIO.StringIO()
    self.mock(git_cl.sys, 'stdout', out)
    self.mock(git_cl.Changelist, 'GetDescription', lambda *args: 'foobar')

    self.assertEqual(0, git_cl.main([
        'description', 'https://code.review.org/123123', '-d', '--rietveld']))
    self.assertEqual('foobar\n', out.getvalue())

  def test_StatusFieldOverrideIssueMissingArgs(self):
    out = StringIO.StringIO()
    self.mock(git_cl.sys, 'stderr', out)

    try:
      self.assertEqual(git_cl.main(['status', '--issue', '1']), 0)
    except SystemExit as ex:
      self.assertEqual(ex.code, 2)
      self.assertRegexpMatches(out.getvalue(), r'--issue must be specified')

    out = StringIO.StringIO()
    self.mock(git_cl.sys, 'stderr', out)

    try:
      self.assertEqual(git_cl.main(['status', '--issue', '1', '--rietveld']), 0)
    except SystemExit as ex:
      self.assertEqual(ex.code, 2)
      self.assertRegexpMatches(out.getvalue(), r'--field must be specified')

  def test_StatusFieldOverrideIssue(self):
    out = StringIO.StringIO()
    self.mock(git_cl.sys, 'stdout', out)

    def assertIssue(cl_self, *_args):
      self.assertEquals(cl_self.issue, 1)
      return 'foobar'

    self.mock(git_cl.Changelist, 'GetDescription', assertIssue)
    self.calls = [
      ((['git', 'config', 'rietveld.autoupdate'],), ''),
      ((['git', 'config', 'rietveld.server'],), ''),
      ((['git', 'config', 'rietveld.server'],), ''),
    ]
    self.assertEqual(
      git_cl.main(['status', '--issue', '1', '--rietveld', '--field', 'desc']),
      0)
    self.assertEqual(out.getvalue(), 'foobar\n')

  def test_SetCloseOverrideIssue(self):
    def assertIssue(cl_self, *_args):
      self.assertEquals(cl_self.issue, 1)
      return 'foobar'

    self.mock(git_cl.Changelist, 'GetDescription', assertIssue)
    self.mock(git_cl.Changelist, 'CloseIssue', lambda *_: None)
    self.calls = [
      ((['git', 'config', 'rietveld.autoupdate'],), ''),
      ((['git', 'config', 'rietveld.server'],), ''),
      ((['git', 'config', 'rietveld.server'],), ''),
    ]
    self.assertEqual(
      git_cl.main(['set-close', '--issue', '1', '--rietveld']), 0)

  def test_SetCommitOverrideIssue(self):
    def assertIssue(cl_self, *_args):
      self.assertEquals(cl_self.issue, 1)
      return 'foobar'

    self.mock(git_cl.Changelist, 'GetDescription', assertIssue)
    self.mock(git_cl.Changelist, 'SetCQState', lambda *_: None)
    self.calls = [
      ((['git', 'config', 'rietveld.autoupdate'],), ''),
      ((['git', 'config', 'rietveld.server'],), ''),
      ((['git', 'config', 'rietveld.server'],), ''),
      ((['git', 'symbolic-ref', 'HEAD'],), ''),
      ((['git', 'config', 'rietveld.server'],), ''),
      ((['git', 'config', 'rietveld.server'],), ''),
    ]
    self.assertEqual(
      git_cl.main(['set-close', '--issue', '1', '--rietveld']), 0)

  def test_description_gerrit(self):
    out = StringIO.StringIO()
    self.mock(git_cl.sys, 'stdout', out)
    self.mock(git_cl.Changelist, 'GetDescription', lambda *args: 'foobar')

    self.assertEqual(0, git_cl.main([
        'description', 'https://code.review.org/123123', '-d', '--gerrit']))
    self.assertEqual('foobar\n', out.getvalue())

  def test_description_set_raw(self):
    out = StringIO.StringIO()
    self.mock(git_cl.sys, 'stdout', out)

    self.mock(git_cl, 'Changelist', ChangelistMock)
    self.mock(git_cl.sys, 'stdin', StringIO.StringIO('hihi'))

    self.assertEqual(0, git_cl.main(['description', '-n', 'hihi']))
    self.assertEqual('hihi', ChangelistMock.desc)

  def test_description_appends_bug_line(self):
    current_desc = 'Some.\n\nChange-Id: xxx'

    def RunEditor(desc, _, **kwargs):
      self.assertEquals(
          '# Enter a description of the change.\n'
          '# This will be displayed on the codereview site.\n'
          '# The first line will also be used as the subject of the review.\n'
          '#--------------------This line is 72 characters long'
          '--------------------\n'
          'Some.\n\nBUG=\n\nChange-Id: xxx',
          desc)
      # Simulate user changing something.
      return 'Some.\n\nBUG=123\n\nChange-Id: xxx'

    def UpdateDescriptionRemote(_, desc, force=False):
      self.assertEquals(desc, 'Some.\n\nBUG=123\n\nChange-Id: xxx')

    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())
    self.mock(git_cl.Changelist, 'GetDescription',
              lambda *args: current_desc)
    self.mock(git_cl._GerritChangelistImpl, 'UpdateDescriptionRemote',
              UpdateDescriptionRemote)
    self.mock(git_cl.gclient_utils, 'RunEditor', RunEditor)

    self.calls = [
        ((['git', 'symbolic-ref', 'HEAD'],), 'feature'),
        ((['git', 'config', 'branch.feature.gerritissue'],), '123'),
        ((['git', 'config', 'rietveld.autoupdate'],), CERR1),
        ((['git', 'config', 'rietveld.bug-prefix'],), CERR1),
        ((['git', 'config', 'core.editor'],), 'vi'),
    ]
    self.assertEqual(0, git_cl.main(['description', '--gerrit']))

  def test_description_set_stdin(self):
    out = StringIO.StringIO()
    self.mock(git_cl.sys, 'stdout', out)

    self.mock(git_cl, 'Changelist', ChangelistMock)
    self.mock(git_cl.sys, 'stdin', StringIO.StringIO('hi \r\n\t there\n\nman'))

    self.assertEqual(0, git_cl.main(['description', '-n', '-']))
    self.assertEqual('hi\n\t there\n\nman', ChangelistMock.desc)

  def test_archive(self):
    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())

    self.calls = \
        [((['git', 'for-each-ref', '--format=%(refname)', 'refs/heads'],),
          'refs/heads/master\nrefs/heads/foo\nrefs/heads/bar'),
         ((['git', 'config', 'branch.master.rietveldissue'],), '1'),
         ((['git', 'config', 'rietveld.autoupdate'],), CERR1),
         ((['git', 'config', 'rietveld.server'],), 'codereview.example.com'),
         ((['git', 'config', 'branch.foo.rietveldissue'],), '456'),
         ((['git', 'config', 'branch.bar.rietveldissue'],), CERR1),
         ((['git', 'config', 'branch.bar.gerritissue'],), '789'),
         ((['git', 'symbolic-ref', 'HEAD'],), 'master'),
         ((['git', 'tag', 'git-cl-archived-456-foo', 'foo'],), ''),
         ((['git', 'branch', '-D', 'foo'],), '')]

    self.mock(git_cl, 'get_cl_statuses',
              lambda branches, fine_grained, max_processes:
              [(MockChangelistWithBranchAndIssue('master', 1), 'open'),
               (MockChangelistWithBranchAndIssue('foo', 456), 'closed'),
               (MockChangelistWithBranchAndIssue('bar', 789), 'open')])

    self.assertEqual(0, git_cl.main(['archive', '-f']))

  def test_archive_current_branch_fails(self):
    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())
    self.calls = \
        [((['git', 'for-each-ref', '--format=%(refname)', 'refs/heads'],),
          'refs/heads/master'),
         ((['git', 'config', 'branch.master.rietveldissue'],), '1'),
         ((['git', 'config', 'rietveld.autoupdate'],), CERR1),
         ((['git', 'config', 'rietveld.server'],), 'codereview.example.com'),
         ((['git', 'symbolic-ref', 'HEAD'],), 'master')]

    self.mock(git_cl, 'get_cl_statuses',
              lambda branches, fine_grained, max_processes:
              [(MockChangelistWithBranchAndIssue('master', 1), 'closed')])

    self.assertEqual(1, git_cl.main(['archive', '-f']))

  def test_archive_dry_run(self):
    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())

    self.calls = \
        [((['git', 'for-each-ref', '--format=%(refname)', 'refs/heads'],),
          'refs/heads/master\nrefs/heads/foo\nrefs/heads/bar'),
         ((['git', 'config', 'branch.master.rietveldissue'],), '1'),
         ((['git', 'config', 'rietveld.autoupdate'],), CERR1),
         ((['git', 'config', 'rietveld.server'],), 'codereview.example.com'),
         ((['git', 'config', 'branch.foo.rietveldissue'],), '456'),
         ((['git', 'config', 'branch.bar.rietveldissue'],), CERR1),
         ((['git', 'config', 'branch.bar.gerritissue'],), '789'),
         ((['git', 'symbolic-ref', 'HEAD'],), 'master'),]

    self.mock(git_cl, 'get_cl_statuses',
              lambda branches, fine_grained, max_processes:
              [(MockChangelistWithBranchAndIssue('master', 1), 'open'),
               (MockChangelistWithBranchAndIssue('foo', 456), 'closed'),
               (MockChangelistWithBranchAndIssue('bar', 789), 'open')])

    self.assertEqual(0, git_cl.main(['archive', '-f', '--dry-run']))

  def test_archive_no_tags(self):
    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())

    self.calls = \
        [((['git', 'for-each-ref', '--format=%(refname)', 'refs/heads'],),
          'refs/heads/master\nrefs/heads/foo\nrefs/heads/bar'),
         ((['git', 'config', 'branch.master.rietveldissue'],), '1'),
         ((['git', 'config', 'rietveld.autoupdate'],), CERR1),
         ((['git', 'config', 'rietveld.server'],), 'codereview.example.com'),
         ((['git', 'config', 'branch.foo.rietveldissue'],), '456'),
         ((['git', 'config', 'branch.bar.rietveldissue'],), CERR1),
         ((['git', 'config', 'branch.bar.gerritissue'],), '789'),
         ((['git', 'symbolic-ref', 'HEAD'],), 'master'),
         ((['git', 'branch', '-D', 'foo'],), '')]

    self.mock(git_cl, 'get_cl_statuses',
              lambda branches, fine_grained, max_processes:
              [(MockChangelistWithBranchAndIssue('master', 1), 'open'),
               (MockChangelistWithBranchAndIssue('foo', 456), 'closed'),
               (MockChangelistWithBranchAndIssue('bar', 789), 'open')])

    self.assertEqual(0, git_cl.main(['archive', '-f', '--notags']))

  def test_cmd_issue_erase_existing(self):
    out = StringIO.StringIO()
    self.mock(git_cl.sys, 'stdout', out)
    self.calls = [
        ((['git', 'symbolic-ref', 'HEAD'],), 'feature'),
        ((['git', 'config', 'branch.feature.rietveldissue'],), CERR1),
        ((['git', 'config', 'branch.feature.gerritissue'],), '123'),
        # Let this command raise exception (retcode=1) - it should be ignored.
        ((['git', 'config', '--unset', 'branch.feature.last-upload-hash'],),
         CERR1),
        ((['git', 'config', '--unset', 'branch.feature.gerritissue'],), ''),
        ((['git', 'config', '--unset', 'branch.feature.gerritpatchset'],), ''),
        ((['git', 'config', '--unset', 'branch.feature.gerritserver'],), ''),
        ((['git', 'config', '--unset', 'branch.feature.gerritsquashhash'],),
         ''),
    ]
    self.assertEqual(0, git_cl.main(['issue', '0']))

  def test_cmd_issue_json(self):
    out = StringIO.StringIO()
    self.mock(git_cl.sys, 'stdout', out)
    self.calls = [
        ((['git', 'symbolic-ref', 'HEAD'],), 'feature'),
        ((['git', 'config', 'branch.feature.rietveldissue'],), '123'),
        ((['git', 'config', 'rietveld.autoupdate'],), ''),
        ((['git', 'config', 'rietveld.server'],),
         'https://codereview.chromium.org'),
        ((['git', 'config', 'branch.feature.rietveldserver'],), ''),
        (('write_json', 'output.json',
          {'issue': 123, 'issue_url': 'https://codereview.chromium.org/123'}),
         ''),
    ]
    self.assertEqual(0, git_cl.main(['issue', '--json', 'output.json']))

  def test_git_cl_try_default_cq_dry_run(self):
    self.mock(git_cl.Changelist, 'GetChange',
              lambda _, *a: (
                self._mocked_call(['GetChange']+list(a))))
    self.mock(git_cl.presubmit_support, 'DoGetTryMasters',
              lambda *_, **__: (
                self._mocked_call(['DoGetTryMasters'])))
    self.mock(git_cl._RietveldChangelistImpl, 'SetCQState',
              lambda _, s: self._mocked_call(['SetCQState', s]))
    self.calls = [
        ((['git', 'symbolic-ref', 'HEAD'],), 'feature'),
        ((['git', 'config', 'branch.feature.rietveldissue'],), '123'),
        ((['git', 'config', 'rietveld.autoupdate'],), ''),
        ((['git', 'config', 'rietveld.server'],),
         'https://codereview.chromium.org'),
        ((['git', 'config', 'branch.feature.rietveldserver'],), ''),
        ((['git', 'config', 'branch.feature.merge'],), 'feature'),
        ((['git', 'config', 'branch.feature.remote'],), 'origin'),
        ((['get_or_create_merge_base', 'feature', 'feature'],),
         'fake_ancestor_sha'),
        ((['GetChange', 'fake_ancestor_sha', None], ),
         git_cl.presubmit_support.GitChange(
           '', '', '', '', '', '', '', '')),
        ((['git', 'rev-parse', '--show-cdup'],), '../'),
        ((['DoGetTryMasters'], ), None),
        ((['SetCQState', git_cl._CQState.DRY_RUN], ), None),
    ]
    out = StringIO.StringIO()
    self.mock(git_cl.sys, 'stdout', out)
    self.assertEqual(0, git_cl.main(['try']))
    self.assertEqual(
        out.getvalue(),
        'scheduled CQ Dry Run on https://codereview.chromium.org/123\n')

  def test_git_cl_try_default_cq_dry_run_gerrit(self):
    self.mock(git_cl.Changelist, 'GetChange',
              lambda _, *a: (
                self._mocked_call(['GetChange']+list(a))))
    self.mock(git_cl.presubmit_support, 'DoGetTryMasters',
              lambda *_, **__: (
                self._mocked_call(['DoGetTryMasters'])))
    self.mock(git_cl._GerritChangelistImpl, 'SetCQState',
              lambda _, s: self._mocked_call(['SetCQState', s]))

    def _GetChangeDetail(gerrit_change_list_impl, opts=None):
      # Get realistic expectations.
      gerrit_change_list_impl._GetGerritHost()
      return self._mocked_call(['_GetChangeDetail', opts or []])
    self.mock(git_cl._GerritChangelistImpl, '_GetChangeDetail',
              _GetChangeDetail)

    self.calls = [
        ((['git', 'symbolic-ref', 'HEAD'],), 'feature'),
        ((['git', 'config', 'branch.feature.rietveldissue'],), CERR1),
        ((['git', 'config', 'branch.feature.gerritissue'],), '123456'),
        ((['git', 'config', 'branch.feature.gerritserver'],),
         'https://chromium-review.googlesource.com'),
        ((['_GetChangeDetail', []],), {'status': 'OPEN'}),
        ((['git', 'config', 'branch.feature.merge'],), 'feature'),
        ((['git', 'config', 'branch.feature.remote'],), 'origin'),
        ((['get_or_create_merge_base', 'feature', 'feature'],),
         'fake_ancestor_sha'),
        ((['GetChange', 'fake_ancestor_sha', None], ),
         git_cl.presubmit_support.GitChange(
           '', '', '', '', '', '', '', '')),
        ((['git', 'rev-parse', '--show-cdup'],), '../'),
        ((['DoGetTryMasters'], ), None),
        ((['SetCQState', git_cl._CQState.DRY_RUN], ), None),
    ]
    out = StringIO.StringIO()
    self.mock(git_cl.sys, 'stdout', out)
    self.assertEqual(0, git_cl.main(['try']))
    self.assertEqual(
        out.getvalue(),
        'scheduled CQ Dry Run on '
        'https://chromium-review.googlesource.com/123456\n')

  def test_git_cl_try_buildbucket_with_properties_rietveld(self):
    self.mock(git_cl._RietveldChangelistImpl, 'GetIssueProperties',
              lambda _: {
                'owner_email': 'owner@e.mail',
                'private': False,
                'closed': False,
                'project': 'depot_tools',
                'patchsets': [20001],
              })
    self.mock(git_cl.uuid, 'uuid4', lambda: 'uuid4')
    self.calls = [
        ((['git', 'symbolic-ref', 'HEAD'],), 'feature'),
        ((['git', 'config', 'branch.feature.rietveldissue'],), '123'),
        ((['git', 'config', 'rietveld.autoupdate'],), CERR1),
        ((['git', 'config', 'rietveld.server'],),
         'https://codereview.chromium.org'),
        ((['git', 'config', 'branch.feature.rietveldpatchset'],), '20001'),
        ((['git', 'config', 'branch.feature.rietveldserver'],), CERR1),
    ]

    def _buildbucket_retry(*_, **kw):
      # self.maxDiff = 10000
      body = json.loads(kw['body'])
      self.assertEqual(len(body['builds']), 1)
      build = body['builds'][0]
      params = json.loads(build.pop('parameters_json'))
      self.assertEqual(params, {
        u'builder_name': u'win',
        u'changes': [{u'author': {u'email': u'owner@e.mail'},
                      u'revision': None}],
        u'properties': {
          u'category': u'git_cl_try',
           u'issue': 123,
           u'key': u'val',
           u'json': [{u'a': 1}, None],
           u'master': u'tryserver.chromium',
           u'patch_project': u'depot_tools',
           u'patch_storage': u'rietveld',
           u'patchset': 20001,
           u'rietveld': u'https://codereview.chromium.org',
        }
      })
      self.assertEqual(build, {
        u'bucket': u'master.tryserver.chromium',
        u'client_operation_id': u'uuid4',
        u'tags': [u'builder:win',
                  u'buildset:patch/rietveld/codereview.chromium.org/123/20001',
                  u'user_agent:git_cl_try',
                  u'master:tryserver.chromium'],
      })

    self.mock(git_cl, '_buildbucket_retry', _buildbucket_retry)

    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())
    self.assertEqual(0, git_cl.main([
        'try', '-m', 'tryserver.chromium', '-b', 'win',
        '-p', 'key=val', '-p', 'json=[{"a":1}, null]']))
    self.assertRegexpMatches(
        git_cl.sys.stdout.getvalue(),
        'Tried jobs on:\nBucket: master.tryserver.chromium')

  def test_git_cl_try_buildbucket_with_properties_gerrit(self):
    self.mock(git_cl.Changelist, 'GetMostRecentPatchset', lambda _: 7)
    self.mock(git_cl.uuid, 'uuid4', lambda: 'uuid4')

    def _GetChangeDetail(gerrit_change_list_impl, opts=None):
      # Get realistic expectations.
      gerrit_change_list_impl._GetGerritHost()
      return self._mocked_call(['_GetChangeDetail', opts or []])
    self.mock(git_cl._GerritChangelistImpl, '_GetChangeDetail',
              _GetChangeDetail)

    self.calls = [
        ((['git', 'symbolic-ref', 'HEAD'],), 'feature'),
        ((['git', 'config', 'branch.feature.rietveldissue'],), CERR1),
        ((['git', 'config', 'branch.feature.gerritissue'],), '123456'),
        ((['git', 'config', 'branch.feature.gerritserver'],),
         'https://chromium-review.googlesource.com'),
        ((['_GetChangeDetail', []],), {'status': 'OPEN'}),
        ((['_GetChangeDetail', ['DETAILED_ACCOUNTS']],),
         {'owner': {'email': 'owner@e.mail'}}),
        ((['_GetChangeDetail', ['ALL_REVISIONS']],), {
          'project': 'depot_tools',
          'revisions': {
            'deadbeaf':  {
              '_number': 6,
            },
            'beeeeeef': {
              '_number': 7,
              'fetch': {'http': {
                'url': 'https://chromium.googlesource.com/depot_tools',
                'ref': 'refs/changes/56/123456/7'
              }},
            },
          },
        }),
    ]

    def _buildbucket_retry(*_, **kw):
      # self.maxDiff = 10000
      body = json.loads(kw['body'])
      self.assertEqual(len(body['builds']), 1)
      build = body['builds'][0]
      params = json.loads(build.pop('parameters_json'))
      self.assertEqual(params, {
        u'builder_name': u'win',
        u'changes': [{u'author': {u'email': u'owner@e.mail'},
                      u'revision': None}],
        u'properties': {
          u'category': u'git_cl_try',
           u'key': u'val',
           u'json': [{u'a': 1}, None],
           u'master': u'tryserver.chromium',

           u'patch_gerrit_url':
             u'https://chromium-review.googlesource.com',
           u'patch_issue': 123456,
           u'patch_project': u'depot_tools',
           u'patch_ref': u'refs/changes/56/123456/7',
           u'patch_repository_url':
             u'https://chromium.googlesource.com/depot_tools',
           u'patch_set': 7,
           u'patch_storage': u'gerrit',
        }
      })
      self.assertEqual(build, {
        u'bucket': u'master.tryserver.chromium',
        u'client_operation_id': u'uuid4',
        u'tags': [
          u'builder:win',
          u'buildset:patch/gerrit/chromium-review.googlesource.com/123456/7',
          u'user_agent:git_cl_try',
          u'master:tryserver.chromium'],
      })

    self.mock(git_cl, '_buildbucket_retry', _buildbucket_retry)

    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())
    self.assertEqual(0, git_cl.main([
        'try', '-m', 'tryserver.chromium', '-b', 'win',
        '-p', 'key=val', '-p', 'json=[{"a":1}, null]']))
    self.assertRegexpMatches(
        git_cl.sys.stdout.getvalue(),
        'Tried jobs on:\nBucket: master.tryserver.chromium')

  def test_git_cl_try_buildbucket_bucket_flag(self):
    self.mock(git_cl._RietveldChangelistImpl, 'GetIssueProperties',
              lambda _: {
                'owner_email': 'owner@e.mail',
                'private': False,
                'closed': False,
                'project': 'depot_tools',
                'patchsets': [20001],
              })
    self.mock(git_cl.uuid, 'uuid4', lambda: 'uuid4')
    self.calls = [
        ((['git', 'symbolic-ref', 'HEAD'],), 'feature'),
        ((['git', 'config', 'branch.feature.rietveldissue'],), '123'),
        ((['git', 'config', 'rietveld.autoupdate'],), CERR1),
        ((['git', 'config', 'rietveld.server'],),
         'https://codereview.chromium.org'),
        ((['git', 'config', 'branch.feature.rietveldpatchset'],), '20001'),
        ((['git', 'config', 'branch.feature.rietveldserver'],), CERR1),
    ]

    def _buildbucket_retry(*_, **kw):
      body = json.loads(kw['body'])
      self.assertEqual(len(body['builds']), 1)
      build = body['builds'][0]
      params = json.loads(build.pop('parameters_json'))
      self.assertEqual(params, {
        u'builder_name': u'win',
        u'changes': [{u'author': {u'email': u'owner@e.mail'},
                      u'revision': None}],
        u'properties': {
          u'category': u'git_cl_try',
           u'issue': 123,
           u'patch_project': u'depot_tools',
           u'patch_storage': u'rietveld',
           u'patchset': 20001,
           u'rietveld': u'https://codereview.chromium.org',
        }
      })
      self.assertEqual(build, {
        u'bucket': u'test.bucket',
        u'client_operation_id': u'uuid4',
        u'tags': [u'builder:win',
                  u'buildset:patch/rietveld/codereview.chromium.org/123/20001',
                  u'user_agent:git_cl_try'],
      })

    self.mock(git_cl, '_buildbucket_retry', _buildbucket_retry)

    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())
    self.assertEqual(0, git_cl.main([
        'try', '-B', 'test.bucket', '-b', 'win']))
    self.assertRegexpMatches(
        git_cl.sys.stdout.getvalue(),
        'Tried jobs on:\nBucket: test.bucket')

  def test_git_cl_try_bots_on_multiple_masters(self):
    self.mock(git_cl.Changelist, 'GetMostRecentPatchset', lambda _: 20001)
    self.calls = [
      ((['git', 'symbolic-ref', 'HEAD'],), 'feature'),
      ((['git', 'config', 'branch.feature.rietveldissue'],), '123'),
      ((['git', 'config', 'rietveld.autoupdate'],), CERR1),
      ((['git', 'config', 'rietveld.server'],),
        'https://codereview.chromium.org'),
      ((['git', 'config', 'branch.feature.rietveldserver'],), CERR1),
      ((['git', 'config', 'branch.feature.rietveldpatchset'],), '20001'),
    ]

    def _buildbucket_retry(*_, **kw):
      body = json.loads(kw['body'])
      self.assertEqual(len(body['builds']), 2)

      first_build_params = json.loads(body['builds'][0]['parameters_json'])
      self.assertEqual(first_build_params['builder_name'], 'builder1')
      self.assertEqual(first_build_params['properties']['master'], 'master1')

      first_build_params = json.loads(body['builds'][1]['parameters_json'])
      self.assertEqual(first_build_params['builder_name'], 'builder2')
      self.assertEqual(first_build_params['properties']['master'], 'master2')

    self.mock(git_cl, '_buildbucket_retry', _buildbucket_retry)

    self.mock(git_cl.urllib2, 'urlopen', lambda _: StringIO.StringIO(
      json.dumps({'builder1': ['master1'], 'builder2': ['master2']})))

    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())
    self.assertEqual(
        0, git_cl.main(['try', '-b', 'builder1', '-b', 'builder2']))
    self.assertEqual(
        git_cl.sys.stdout.getvalue(),
        'Tried jobs on:\n'
        'Bucket: master.master1\n'
        '  builder1: []\n'
        'Bucket: master.master2\n'
        '  builder2: []\n'
        'To see results here, run:        git cl try-results\n'
        'To see results in browser, run:  git cl web\n')

  def _common_GerritCommitMsgHookCheck(self):
    self.mock(git_cl.sys, 'stdout', StringIO.StringIO())
    self.mock(git_cl.os.path, 'abspath',
              lambda path: self._mocked_call(['abspath', path]))
    self.mock(git_cl.os.path, 'exists',
              lambda path: self._mocked_call(['exists', path]))
    self.mock(git_cl.gclient_utils, 'FileRead',
              lambda path: self._mocked_call(['FileRead', path]))
    self.mock(git_cl.gclient_utils, 'rm_file_or_tree',
              lambda path: self._mocked_call(['rm_file_or_tree', path]))
    self.calls = [
        ((['git', 'rev-parse', '--show-cdup'],), '../'),
        ((['abspath', '../'],), '/abs/git_repo_root'),
    ]
    return git_cl.Changelist(codereview='gerrit', issue=123)

  def test_GerritCommitMsgHookCheck_custom_hook(self):
    cl = self._common_GerritCommitMsgHookCheck()
    self.calls += [
        ((['exists', '/abs/git_repo_root/.git/hooks/commit-msg'],), True),
        ((['FileRead', '/abs/git_repo_root/.git/hooks/commit-msg'],),
         '#!/bin/sh\necho "custom hook"')
    ]
    cl._codereview_impl._GerritCommitMsgHookCheck(offer_removal=True)

  def test_GerritCommitMsgHookCheck_not_exists(self):
    cl = self._common_GerritCommitMsgHookCheck()
    self.calls += [
        ((['exists', '/abs/git_repo_root/.git/hooks/commit-msg'],), False),
    ]
    cl._codereview_impl._GerritCommitMsgHookCheck(offer_removal=True)

  def test_GerritCommitMsgHookCheck(self):
    cl = self._common_GerritCommitMsgHookCheck()
    self.calls += [
        ((['exists', '/abs/git_repo_root/.git/hooks/commit-msg'],), True),
        ((['FileRead', '/abs/git_repo_root/.git/hooks/commit-msg'],),
         '...\n# From Gerrit Code Review\n...\nadd_ChangeId()\n'),
        (('Do you want to remove it now? [Yes/No]',), 'Yes'),
        ((['rm_file_or_tree', '/abs/git_repo_root/.git/hooks/commit-msg'],),
         ''),
    ]
    cl._codereview_impl._GerritCommitMsgHookCheck(offer_removal=True)

  def test_GerritCmdLand(self):
    self.calls += [
      ((['git', 'symbolic-ref', 'HEAD'],), 'feature'),
      ((['git', 'config', 'branch.feature.gerritsquashhash'],),
       'deadbeaf'),
      ((['git', 'diff', 'deadbeaf'],), ''),  # No diff.
      ((['git', 'config', 'branch.feature.gerritserver'],),
       'chromium-review.googlesource.com'),
    ]
    cl = git_cl.Changelist(issue=123, codereview='gerrit')
    cl._codereview_impl._GetChangeDetail = lambda _: {
      'labels': {},
      'current_revision': 'deadbeaf',
    }
    cl._codereview_impl._GetChangeCommit = lambda: {
      'commit': 'deadbeef',
      'web_links': [{'name': 'gerrit',
                     'url': 'https://git.googlesource.com/test/+/deadbeef'}],
    }
    cl._codereview_impl.SubmitIssue = lambda wait_for_merge: None
    out = StringIO.StringIO()
    self.mock(sys, 'stdout', out)
    self.assertEqual(0, cl.CMDLand(force=True, bypass_hooks=True, verbose=True))
    self.assertRegexpMatches(out.getvalue(), 'Issue.*123 has been submitted')
    self.assertRegexpMatches(out.getvalue(), 'Landed as .*deadbeef')

  BUILDBUCKET_BUILDS_MAP = {
        '9000': {
            'id': '9000',
            'status': 'STARTED',
            'url': 'http://build.cr.org/p/x.y/builders/my-builder/builds/2',
            'result_details_json': '{"properties": {}}',
            'bucket': 'master.x.y',
            'created_by': 'user:someone@chromium.org',
            'created_ts': '147200002222000',
            'parameters_json': '{"builder_name": "my-builder", "category": ""}',
        },
        '8000': {
            'id': '8000',
            'status': 'COMPLETED',
            'result': 'FAILURE',
            'failure_reason': 'BUILD_FAILURE',
            'url': 'http://build.cr.org/p/x.y/builders/my-builder/builds/1',
            'result_details_json': '{"properties": {}}',
            'bucket': 'master.x.y',
            'created_by': 'user:someone@chromium.org',
            'created_ts': '147200001111000',
            'parameters_json': '{"builder_name": "my-builder", "category": ""}',
        },
    }

  def test_write_try_results_json(self):
    expected_output = [
        {
            'buildbucket_id': '8000',
            'bucket': 'master.x.y',
            'builder_name': 'my-builder',
            'status': 'COMPLETED',
            'result': 'FAILURE',
            'failure_reason': 'BUILD_FAILURE',
            'url': 'http://build.cr.org/p/x.y/builders/my-builder/builds/1',
        },
        {
            'buildbucket_id': '9000',
            'bucket': 'master.x.y',
            'builder_name': 'my-builder',
            'status': 'STARTED',
            'result': None,
            'failure_reason': None,
            'url': 'http://build.cr.org/p/x.y/builders/my-builder/builds/2',
        }
    ]
    self.calls = [(('write_json', 'output.json', expected_output), '')]
    git_cl.write_try_results_json('output.json', self.BUILDBUCKET_BUILDS_MAP)

  def _setup_fetch_try_jobs(self, most_recent_patchset=20001):
    out = StringIO.StringIO()
    self.mock(sys, 'stdout', out)
    self.mock(git_cl.Changelist, 'GetMostRecentPatchset',
              lambda *args: most_recent_patchset)
    self.mock(git_cl.auth, 'get_authenticator_for_host', lambda host, _cfg:
              self._mocked_call(['get_authenticator_for_host', host]))
    self.mock(git_cl, '_buildbucket_retry', lambda *_, **__:
              self._mocked_call(['_buildbucket_retry']))

  def _setup_fetch_try_jobs_rietveld(self, *request_results):
    self._setup_fetch_try_jobs(most_recent_patchset=20001)
    self.calls += [
      ((['git', 'symbolic-ref', 'HEAD'],), 'feature'),
      ((['git', 'config', 'branch.feature.rietveldissue'],), '1'),
      ((['git', 'config', 'rietveld.autoupdate'],), CERR1),
      ((['git', 'config', 'rietveld.server'],), 'codereview.example.com'),
      ((['git', 'config', 'branch.feature.rietveldpatchset'],), '20001'),
      ((['git', 'config', 'branch.feature.rietveldserver'],),
       'codereview.example.com'),
      ((['get_authenticator_for_host', 'codereview.example.com'],),
       AuthenticatorMock()),
    ] + [((['_buildbucket_retry'],), r) for r in request_results]

  def test_fetch_try_jobs_none_rietveld(self):
    self._setup_fetch_try_jobs_rietveld({})
    # Simulate that user isn't logged in.
    self.mock(AuthenticatorMock, 'has_cached_credentials', lambda _: False)
    self.assertEqual(0, git_cl.main(['try-results']))
    self.assertRegexpMatches(sys.stdout.getvalue(),
                             'Warning: Some results might be missing')
    self.assertRegexpMatches(sys.stdout.getvalue(), 'No try jobs')

  def test_fetch_try_jobs_some_rietveld(self):
    self._setup_fetch_try_jobs_rietveld({
      'builds': self.BUILDBUCKET_BUILDS_MAP.values(),
    })
    self.assertEqual(0, git_cl.main(['try-results']))
    self.assertRegexpMatches(sys.stdout.getvalue(), '^Failures:')
    self.assertRegexpMatches(sys.stdout.getvalue(), 'Started:')
    self.assertRegexpMatches(sys.stdout.getvalue(), '2 try jobs')

  def _setup_fetch_try_jobs_gerrit(self, *request_results):
    self._setup_fetch_try_jobs(most_recent_patchset=13)
    self.calls += [
      ((['git', 'symbolic-ref', 'HEAD'],), 'feature'),
      ((['git', 'config', 'branch.feature.rietveldissue'],), CERR1),
      ((['git', 'config', 'branch.feature.gerritissue'],), '1'),
      # TODO(tandrii): Uncomment the below if we decide to support checking
      # patchsets for Gerrit.
      # Simulate that Gerrit has more patchsets than local.
      # ((['git', 'config', 'branch.feature.gerritpatchset'],), '12'),
      ((['git', 'config', 'branch.feature.gerritserver'],),
       'https://x-review.googlesource.com'),
      ((['get_authenticator_for_host', 'x-review.googlesource.com'],),
       AuthenticatorMock()),
    ] + [((['_buildbucket_retry'],), r) for r in request_results]

  def test_fetch_try_jobs_none_gerrit(self):
    self._setup_fetch_try_jobs_gerrit({})
    self.assertEqual(0, git_cl.main(['try-results']))
    # TODO(tandrii): Uncomment the below if we decide to support checking
    # patchsets for Gerrit.
    # self.assertRegexpMatches(
    #     sys.stdout.getvalue(),
    #     r'Warning: Codereview server has newer patchsets \(13\)')
    self.assertRegexpMatches(sys.stdout.getvalue(), 'No try jobs')

  def test_fetch_try_jobs_some_gerrit(self):
    self._setup_fetch_try_jobs_gerrit({
      'builds': self.BUILDBUCKET_BUILDS_MAP.values(),
    })
    # TODO(tandrii): Uncomment the below if we decide to support checking
    # patchsets for Gerrit.
    # self.calls.remove(
    #     ((['git', 'config', 'branch.feature.gerritpatchset'],), '12'))
    self.assertEqual(0, git_cl.main(['try-results', '--patchset', '5']))

    # ... and doesn't result in warning.
    self.assertNotRegexpMatches(sys.stdout.getvalue(), 'Warning')
    self.assertRegexpMatches(sys.stdout.getvalue(), '^Failures:')
    self.assertRegexpMatches(sys.stdout.getvalue(), 'Started:')
    self.assertRegexpMatches(sys.stdout.getvalue(), '2 try jobs')


if __name__ == '__main__':
  git_cl.logging.basicConfig(
      level=git_cl.logging.DEBUG if '-v' in sys.argv else git_cl.logging.ERROR)
  unittest.main()
