#
# -*- coding: utf-8 -*-
#
#       Copyright 2011 Voldemar Khramtsov <harestomper@gmail.com>
#       
#       This program is free software; you can redistribute it and/or modify
#       it under the terms of the GNU General Public License as published by
#       the Free Software Foundation; either version 2 of the License, or
#       (at your option) any later version.
#       
#       This program is distributed in the hope that it will be useful,
#       but WITHOUT ANY WARRANTY; without even the implied warranty of
#       MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#       GNU General Public License for more details.
#       
#       You should have received a copy of the GNU General Public License
#       along with this program; if not, write to the Free Software
#       Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
#       MA 02110-1301, USA.

import os
import re
import dbus
import socket
import pickle
import urllib
import dbus.service

from lxml import etree
from dbus.mainloop.glib import DBusGMainLoop

from itmagesd.common import *

dbus_mainloop = DBusGMainLoop(set_as_default=True)


#-------------------------------------------------------------------------------
def check_uri(uri):
    try:
        fp = urllib.urlopen(uri)
        mime = fp.info().type

        if mime in MIMETYPES:
            filename = urllib.unquote(fp.url).replace('file://', '')
            fsize = os.path.getsize(filename)

            if  fsize > 5242880:
                return None, None, None
            else:
                return filename, fsize, mime
        else:
            return None, None, None
    except IOError, e:
        echo (e, "e")
        return None, None, None
        

#-------------------------------------------------------------------------------
def is_xml(value):
    try:
        root = etree.fromstring(convert_item(value))
    except (etree.XMLSyntaxError, ValueError):
        return False
    else:
        return True


#-------------------------------------------------------------------------------
typealiases = {
                str: "string",
                bool: "bool",
                int: "int",
                long: "int",
                unicode: "string",
                tuple: "list",
                list: "list",
                dict: "dict",
                float: "float",
                None: "none",

                "s": str,
                "srt": str,
                "string": str,
                "b": bool,
                "bool": bool,
                "i": int,
                "int": int,
                "integer": int,
                "byte": int,
                "d": float,
                "double": float,
                "f": float,
                "float": float,
                "u": unicode,
                "unicode": unicode,
                "uni": unicode,
                "none": "none",
                "NONE": "none",
                "NULL": "none",
                "null": "none",
                "nill": "none",
                "n": "none",

                "x": "xml",
                "xml": "xml",
                "l": "list",
                "list": "list",
                "dict": "dict",
              }
             
def message_dump(method, **kwargs):

    def newdict(item, parent_key):
        root = None
        elements = []

        for key, value in item.items():
            elem = None

            if isinstance(value, (list, tuple)):
                elem = newlist(value)
                elem.tag = "kwarg"

            elif isinstance(value, dict):
                elem = newdict(value, key)

            elif is_xml(value):
                elem = newxml(value, "kwarg")
                
            else:
                tp = typealiases.get(type(value))
                if tp is None:
                    tp = "none"
                elem = etree.Element("kwarg", type=tp)
                if isinstance(value, (str, unicode)):
                    value = urllib.quote(value)
                if value is None:
                    elem.text = ""
                else:
                    elem.text = str(value)

            if elem is not None:
                elem.set("name", key)
                elements.append(elem)

        if root is None:
            root = etree.Element('arg', name = "", type="dict")

        if parent_key:
            root.tag = "kwarg"
            root.set("name", parent_key)

        if elements:    
            [root.append(e) for e in elements]
            return root
        else:
            return None
        

    def newlist(item):
        root = None
        elements = []

        for value in item:
            elem = None
            if isinstance(value, (tuple, list)):
                elem = newlist(value)

            elif isinstance(value, dict):
                elem = newdict(value, None)

            elif is_xml(value):
                elem = newxml(value, "arg")
                
            else:
                elem = etree.Element("arg", type=typealiases.get(type(value)))
                if isinstance(value, (str, unicode)):
                    value = urllib.quote(value)
                if value is None:
                    elem.text = ""
                else:
                    elem.text = str(value)

            if elem is not None:
                elements.append(elem)

        if elements:
            root = etree.Element('arg', type="list")
            [root.append(elem) for elem in elements]

        return root

    def newxml(item, tag):
        root = etree.fromstring(item)
        elem = etree.Element(tag, type = "xml")
        elem.append(root)
        return elem


    assert isinstance(method, str), "First argement should be string"
    method = convert_item(method)
    tree = etree.Element("message")
    root = etree.Element('method', name=method)
    tree.append(root)
    
    newkw = convert_item(kwargs)
    elem = newdict(newkw, None)

    if elem is not None:
        elem.tag = "kwarg"
        root.append(elem)

    return etree.tostring(
                            tree,
                            pretty_print = True,
                            encoding = "UTF-8",
                            xml_declaration = True
                          )
        

#-------------------------------------------------------------------------------
def xmlsplit(xmldoc):
    samples = []
    try:
        for sample in xmldoc.split("<?xml version='1.0' encoding='UTF-8'?>"):
            samples.append(sample)
    except TypeError:
        return [xmldoc,]
    else:
        return samples
    
#-------------------------------------------------------------------------------
def message_load(xmldoc):
    xmldoc = convert_item(xmldoc)
    items = []
    
    if is_xml(xmldoc):

        def get(elem):
            tp = elem.get("type")
            if tp == "list":
                val = getlist(elem)
            elif tp == "dict":
                val = getdict(elem)
            elif tp == "xml":
                val = etree.tostring(elem[0], encoding="UTF-8")
            elif tp == "none":
                val = None
            else:
                func = typealiases.get(tp)
                if callable(func):
                    try:
                        val = func(elem.text)
                    except ValueError:
                        val = elem.text

                if func is str:
                    val = urllib.unquote(val)

            return val
                

        def getdict(elem):
            dic = {}
            for row in elem.findall("kwarg"):
                if row.get("name"):
                    dic[row.get("name")] = get(row)
                else:
                    dic = get(row)
            return dic


        def getlist(elem):
            lst = []
            if elem.get("type") == "list" and len(elem) == 1 and elem[0].get("type"):
                return None
            elif elem.get("type") == "list":
                for row in elem.findall("arg"):
                    lst.append(get(row))
            if len(lst) == 1:
                return lst[0]
            else:
                return lst


        root = etree.fromstring(xmldoc)

        if root.tag in ("message", "entry"):
            for elem in root.findall("method") + root.findall("object"):
                if elem.get("name"):
                    kwargs = getdict(elem)
                    item = elem.get("name"), (), kwargs
                    items.append(item)

    if len(items) == 1:
        return items[0]
    else:
        return items

        
def unpack(message):
    vlist = []
    try:
        data = re.findall('<start>(.+?)<end>', message, re.DOTALL)
    except Exception:
        pass
    else:
        for content in data:
            try:
                loads = pickle.loads(content)
            except Exception:
                pass
            else:
                vlist.append(loads)
    return vlist


#-------------------------------------------------------------------------------
def convert_item(item, quote=None):
    if isinstance(item, (str, dbus.String,
                                dbus.ByteArray,
                                unicode,
                                dbus.UTF8String
                         )
                    ):

        if quote == "quote":
            return urllib.quote(item)

        elif quote == "unquote":
            return urllib.unquote(item)

        elif isinstance(item, (unicode, dbus.UTF8String)):
            return item.encode("UTF-8")

        else:
            return str(item)

    elif isinstance(item, (long, dbus.Byte,
                                    dbus.Int16,
                                    dbus.Int32,
                                    dbus.Int64,
                                    dbus.UInt16,
                                    dbus.UInt32,
                                    dbus.UInt64,
                            )
                      ):
        return int(item)

    elif isinstance(item, (dbus.Double, float)):
        return float(item)

    elif item is None:
        return ""

    elif isinstance(item, (dbus.Array, list, tuple)):

        if isinstance(item, dbus.Array):
            item = list(item)
            func = list
        else:
            func = type(item)

        newitem = []
        [newitem.append(convert_item(i)) for i in item]
        return func(newitem)

    elif isinstance(item, (dbus.Dictionary, dict)):
        newdict = {}
        [newdict.__setitem__(convert_item(k), convert_item(v)) for k, v in item.items()]
        return newdict
        
    elif isinstance(item, (dbus.Boolean)):
        return bool(item.conjugate())

    else:
        return item


#-------------------------------------------------------------------------------
class ImageItem(object):
    def __init__(self):
        self.name = None
        self.info = {}
        self.status = None
        self.reason = None
        self.index = 0
        self.hash = None
        self.obj = ""


#-------------------------------------------------------------------------------
class ResponseItem(object):

    def __init__(self, **kwargs):
        self.name = None
        self.code = None
        self.data = None
        self.action = None
        self.addition = None

        if kwargs:
            for key, value in kwargs.items():
                try:
                    setattr(self, key, value)
                except AttributeError:
                    pass


#-------------------------------------------------------------------------------
class ProgressItem(object):

    def __init__(self, **kwargs):
        self.dt = 0.0
        self.dr = 0.0
        self.ut = 0.0
        self.ur = 0.0
        self.addition = None
        self.name = None
        self.action = None

        if kwargs:
            for key, value in kwargs.items():
                try:
                    setattr(self, key, value)
                except AttributeError:
                    pass


#-------------------------------------------------------------------------------
class ErrorItem(object):

    def __init__(self, **kwargs):
        self.code = kwargs.get('code')
        self.message = kwargs.get('message')
        self.method = kwargs.get('method')
        
#-------------------------------------------------------------------------------
aliases = {
            'date': 'date',
            'creared': 'date',
            'server': 'server',
            'thumb': 'small',
            'turi': 'small',
            'iuri': 'full',
            'full': 'full',
            'size': 'size',
            'mime': 'mime',
            'width': 'width',
            'height': 'height',
            'id': 'id',
            'key': 'key'
          }
            
#-------------------------------------------------------------------------------
def parse_response(xmldoc, getobject=False, info_only=False):
    status, reason, info = 'failed', '', {}
    root = etree.fromstring(xmldoc)
    obj = "<obj></obj>"

    def iterator(element):
        for subelement in element:
            tag = aliases.get(subelement.tag)
            yield tag, subelement

    def getdesc(tag):
        for e in root.iterdescendants(tag):
            return e

    elem = getdesc("status")
    if elem is not None:
        status = elem.text

    elem = getdesc("reason")
    if elem is not None:
        reason = elem.text

    elem = getdesc("obj")
    if elem is not None:
        info = {}
        obj = etree.tostring(elem, pretty_print = True)
        newroot = etree.Element("getpictures")

        for tag, subelement in iterator(elem):
            if tag:
                info[tag] = subelement.text

    status = status in ("ok", True)
    reason = reason or ""

    if getobject:
        if info_only:
            return info, obj
        else:
            return status, reason, info, obj
    else:
        if info_only:
            return info
        else:
            return status, reason, info


#-------------------------------------------------------------------------------
def convert_getpictures_data(xmldata):
    newroot = etree.Element("getpictures")
    root = etree.fromstring(xmldata)
    obj_elem = None

    # find root tag "obj"
    if root.tag == "obj":
        obj_elem = root
    else:
        for o in root:
            if o.tag == "obj":
                obj_elem = o
                break
        
    for elem in obj_elem:
        newobj = etree.SubElement(newroot, "obj")

        for child in elem:
            tag = aliases.get(child.tag)
            newchild = etree.SubElement(newobj, tag)
            newchild.text = child.text

    return etree.tostring( newroot,
                            pretty_print=True,
                            encoding="UTF-8",
                            xml_declaration=True
                          )


#-------------------------------------------------------------------------------
def parse_string(value):
    value = convert_item(value)
    return unpack_message(value)

#-------------------------------------------------------------------------------
def find_alive_socket(conn_name = "none"):
    try:
        listdir = os.listdir(TEMPFOLDER)
    except Exception:
        return None

    for fname in listdir:
        path = os.path.join(TEMPFOLDER, fname)

        if path.endswith('.socket'):
            sock = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)

            try:
                sock.connect(path)
            except Exception:
                continue

            sock.settimeout(3)
            msg = message_dump('ping', result="pid", conn_name = conn_name)
            sock.send(msg)
            length = sock.fileno()
            try:
                pid = sock.recv(length)
            except (socket.error, socket.timeout), e:
                echo ("Exception in 'find_alive_socket' %s"  % e)
                return None

            if pid:
                return sock
    else:
        return None
            

#-------------------------------------------------------------------------------
def create_new_socket(sockname=SOCKNAME):
    sock = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)

    try:
        sock.bind(sockname)
    except Exception:
        return None

    sock.setblocking(False)
    sock.listen(5)
    return sock

#-------------------------------------------------------------------------------
def tempclear(with_socket=False):
    if os.access(TEMPFOLDER, os.R_OK|os.W_OK):
        for filename in os.listdir(TEMPFOLDER):
            if not with_socket and filename.endswith('.socket'): continue
            try: os.remove(os.path.join(TEMPFOLDER, filename))
            except Exception: pass
        try: os.rmdir(TEMPFOLDER)
        except Exception: pass


#-------------------------------------------------------------------------------
def conf_get(key):
    """
    Read config file.
    """
    try:
        xmldoc = etree.parse(DAEMON_CONF)
    except Exception:
        return None
    else:
        try:
            pydata = message_load(etree.tostring(xmldoc))
        except Exception, e:
            echo ("Config Error: %s" % e, "e")
            return None
        else:
            if pydata and len(pydata) == 3 and isinstance(pydata[2], dict):
                if key == "**all**":
                    return pydata[2]
                else:
                    return pydata[2].get(key)
            else:
                return None


#-------------------------------------------------------------------------------
def conf_set(key, value):
    """
    Write config file
    """
    pydata = conf_get("**all**")
    if pydata:
        pydata[key] = value
    else:
        pydata = {key: value}

    xml = apply(message_dump, ("DaemonConfig",), pydata)
    open(DAEMON_CONF, "w").write(xml)

    
#path = os.path.join(os.path.environ['HOME'], '.config', 'deadbeef', 'socket')
#sock = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
#sock.connect(path) # для клиента
#sock.bind(path) # для сервера

####
