#
# -*- coding: utf-8 -*-
#
#       Без имени.py
#       
#       Copyright 2011 Voldemar Khramtsov <harestomper@gmail.com>
#       
#       This program is free software; you can redistribute it and/or modify
#       it under the terms of the GNU General Public License as published by
#       the Free Software Foundation; either version 2 of the License, or
#       (at your option) any later version.
#       
#       This program is distributed in the hope that it will be useful,
#       but WITHOUT ANY WARRANTY; without even the implied warranty of
#       MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#       GNU General Public License for more details.
#       
#       You should have received a copy of the GNU General Public License
#       along with this program; if not, write to the Free Software
#       Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
#       MA 02110-1301, USA.



import os
import copy
import glib
import Queue
import hashlib
import tempfile
import threading

from itmagesd import iomod
from itmagesd.tools import *
from itmagesd.common import *


class _stat:
    cancel = False
    quiting = False


#-------------------------------------------------------------------------------
STATUS_DONE = 0
STATUS_WAIT = 1
STATUS_FAIL = 2
STATUS_RUN = 3
STATUS_CNCL = 4



glib.threads_init()

def sha256(filename):
    hasher = hashlib.sha256()
    data = open(filename, 'r').read()
    hasher.update(data)
    return hasher.hexdigest()
    

#-------------------------------------------------------------------------------
class DbusServer(dbus.service.Object):

    def __init__(self, parent):
        self.__session_bus = dbus.SessionBus(dbus_mainloop)
        self.__bus_name = dbus.service.BusName(DBUS_INTERFACE, bus=self.__session_bus)
        super(DbusServer, self).__init__(self.__bus_name, DBUS_PATH)
        self.parent = parent


    # Shared methods
    @dbus.service.method(DBUS_INTERFACE, out_signature='a{sv}', in_signature='a{sv}')
    def Shared(self, args):
        kwargs = convert_item(args)
        output_info = {"status": False}
        methodname = kwargs.get("method") or ""
        output_info["method"] = methodname

        try:
            hasattr(self.parent, methodname)
        except TypeError, e:
            echo (e, "e")
            output_info["status"] = False
            output_info["reason"] = e or ""
        else:
            if hasattr(self.parent, str(methodname)) \
                    and methodname in self.parent.shared_methods:
                method = getattr(self.parent, methodname)

                try:
                    if kwargs.get('async') in (None, 1, 'true', True):
                        glib.idle_add(lambda: apply(method, ("dbus",), kwargs) and 0)
                        return {"status": True, "method": methodname}

                    else:
                        response = apply(method, ('dbus',), kwargs)

                except Exception, e:
                    output_info["reason"] = e.message
                else:
                    data = []

                    if isinstance(response, dict):
                        data = response.items()
                        output_info["status"] = True
                        output_info["reason"] = "Done"

                    elif isinstance(response, (tuple, list)):

                        try:
                            data = [v for v in response if len(v) == 2]
                        except Exception, e:
                            output_info["reason"] = e.message
                        else:
                            output_info["reason"] = "Done"
                            output_info["status"] = True

                    for k, v in data:
                        output_info[k] = v

            else:
                output_info["reason"] = Messages.IMPOSSIBLE_METHOD
            [output_info.__setitem__(k, '') for k, v in output_info.items() if v is None]
        return output_info
        


    @dbus.service.method(DBUS_INTERFACE, out_signature='b', in_signature='s')
    def AddNewItem(self, arg):
        argument = convert_item(arg)
        try:
            status = self.parent.add_new_item('dbus', argument)
        except Exception, e:
            return False
        else:
            return status


    @dbus.service.method(DBUS_INTERFACE, out_signature='b', in_signature='s')
    def DelItem(self, arg):
        argument = convert_item(arg)
        try:
            status = self.parent.remove_item('dbus', argument)
        except Exception:
            return False
        else:
            return status

    @dbus.service.method(DBUS_INTERFACE, out_signature='as', in_signature='')
    def ListMethodsGet(self):
        return self.parent.shared_methods

    @dbus.service.method(DBUS_INTERFACE, out_signature='as', in_signature='')
    def ItemsListGet(self):
        lst = [i.name for i in self.parent.itemlist]
        return lst


    @dbus.service.method(DBUS_INTERFACE, out_signature='a{sv}', in_signature='s')
    def ItemInfoGet(self, filename):
        info = {}
        item = self.parent.get_by_name(None, convert_item(filename))
        if item:
            info = item.info.copy()
            info["obj"] = item.obj
            info["name"] = item.name
            info["hash"] = item.hash
            info["status"] = item.status
            info["reason"] = item.reason
            info["id"] = item.info.get("id")
            info["key"] = item.info.get("key")
            info["size"] = item.info.get("size")
            info["date"] = item.info.get("date")
            info["full"] = item.info.get("full")
            info["small"] = item.info.get("small")
            info["server"] = item.info.get("server")
            info["short"] = item.info.get("short")
        return convert_item(info)


    @dbus.service.method(DBUS_INTERFACE, out_signature='s', in_signature='s')
    def ItemInfoGetLine(self, filename):
        line = ""
        for k, w in self.ItemInfoGet(filename).items():
            if line:
                line = line + ";;" + str(k) + ":" + str(w)
            else:
                line = str(k) + ":" + str(w)
        return line

    
    @dbus.service.method(DBUS_INTERFACE, out_signature='b', in_signature='si')
    def SetItemStatus (self, name, status):
        status = convert_item (status)
        item = self.parent.get_by_name(None, convert_item(name))
        if item and status in (STATUS_WAIT, STATUS_CNCL, STATUS_FAIL):
            if item.status != status:
                self._item_status_changed (name, item.status, status) 
                item.status = status
                return True

        return False
        

    # Shared events
    @dbus.service.signal(DBUS_INTERFACE, signature='isdddd')
    def progress_event(self, action, name, dt, dr, ut, ur):
        pass

        
    def _progress_event(self, *args):
        args = convert_item(args)
        apply(self.progress_event, args)
        

    @dbus.service.signal(DBUS_INTERFACE, signature='b')
    def service_status_changed(self, new_status):
        pass


    @dbus.service.signal(DBUS_INTERFACE, signature='b')
    def connection_status_changed(self, new_status):
        pass


    @dbus.service.signal(DBUS_INTERFACE, signature='syy')
    def item_status_changed(self, filename, old_status, new_status):
        return False

    def _item_status_changed(self, *args):
        args = convert_item(args)
        apply(self.item_status_changed, args)


    @dbus.service.signal(DBUS_INTERFACE, signature='s')
    def item_deleted(self, filename):
        return False

    def _item_deleted(self, filename):
        filename = convert_item(filename)
        glib.idle_add(self.item_deleted, filename)


    @dbus.service.signal(DBUS_INTERFACE, signature='a{sv}')
    def new_item_added(self, imageinfo):
        return False

    def _new_item_added(self, name):
        name = convert_item(name)
        glib.idle_add(self.new_item_added, name)


    @dbus.service.signal(DBUS_INTERFACE, signature='sa{sv}')
    def action_complete(self, action, values):
        return False

    
    @dbus.service.signal(DBUS_INTERFACE, signature='s')
    def action_complete_string_line(self, line):
        return False
        

    def _action_complete(self, action, values):
        values = convert_item(values)
        action = convert_item(action)
        glib.idle_add(self.action_complete, action, values)
        strret = "action:%s" % action
        if isinstance(values, dict):
            for k, v in values.items():
                strret += ";;" + "%s:%s" % (k, v)
        glib.idle_add(self.action_complete_string_line, strret)
        

    @dbus.service.signal(DBUS_INTERFACE, signature='ss')
    def action_started(self, action, name):
        return False

    def _action_started(self, action, name):
        name = convert_item(name)
        action = convert_item(action)
        glib.idle_add(self.action_started, action, name)


    @dbus.service.signal(DBUS_INTERFACE, signature='')
    def quiting(self):
        pass

        
#-------------------------------------------------------------------------------
class Engine(object):

    def __init__(self):
        super(Engine, self).__init__()
        self.server = DbusServer(self)
        self.started = False
        self.loops = {}
        self.queue = []
        self.__oids = {}
        self.itemlist = []
        self.cancelled = {}
        self.__connections = {}
        self.event = threading.Event()
        self.mainloop = glib.MainLoop()
        sock = create_new_socket()
        glib.io_add_watch(sock, glib.IO_IN, self.listener, sock)
        self.__allowed_methods = [
                                    "add_new_item", "remove_item", "check",
                                    "login", "logout", "upload", "delete",
                                    "getpictures", "check_connection",
                                    "quiting", "get_all_items", "get_queue",
                                    "get_uploaded", "get_by_name", "cancel"
                                  ]

    @property
    def shared_methods(self):
        return self.__allowed_methods
        

    def check_for_reply(self, sha, fname = None):
        for i in self.itemlist:
            if isinstance (i.hash, str) and i.hash == sha:
                if fname and fname != i.name:
                    item = copy.deepcopy (i)
                    item.name = fname
                    self.itemlist.append (item)
                return True
        return False


    def get_from_name(self, oid, name, *args, **kwargs):
        new_item = None
        for item in self.itemlist:
            if item.name == name:
                new_item = item
                break

        if oid is None:
            return new_item
        else:
            self.send_message(oid, 'get_from_name', new_item)



    def send_message(self, oid, method, value):
        dump = apply(message_dump, (method,), value)

        if oid == -1:
            for conn in self.__connections.values():
                try:
                    conn.send(dump)
                except Exception, e:
                    pass

        elif oid in self.__connections:
            try:
                self.__connections[oid].send(dump)
            except Exception, e:
                pass


    def add_new_item(self, oid, item, *args, **kwargs):
        new_item = None
        value = convert_item(item)
        if isinstance(value, (str, unicode)):
            
            fname, fsize, mime = check_uri(value)

            if fname: 
                sha = sha256(fname)

                if not self.check_for_reply(sha, fname):
                    item = ImageItem()
                    item.name = fname
                    item.hash = sha
                    item.info['size'] = fsize
                    item.info['mime'] = mime
                    item.status = STATUS_WAIT
                    self.itemlist.append(item)
                    self.queue.append(item)
                    new_item = item
                    status = True
                    reason = "Done"
                    self.server.new_item_added(
                                                {
                                                  "hash": sha,
                                                  "mime": mime,
                                                  "size": fsize,
                                                  "filename": fname,
                                                  "status": STATUS_WAIT
                                                 }
                                                )

                else:
                    new_item = self.get_from_name(None, fname)
                    status = not (new_item.status in (STATUS_WAIT, STATUS_RUN, STATUS_CNCL))

                    if status:
                        reason = "Done"
                    else:
                        reason = Messages.ITEM_EXISTS
            else:
                status = False
                reason = Messages.CAN_NOT_BE_PUT
        else:
            status = False
            reason = Messages.UNSUPPORTED

        if isinstance(oid, str) and oid == 'dbus':
            return {"status": status, "reason": reason}

        elif oid:
            self.send_message(oid, 'add_new_file', new_item)

        else:
            return new_item


    def remove_item(self, oid, item, *args, **kwargs):
        value = convert_item(item)

        if isinstance(value,ImageItem):
            name = value.name
            new_item = self.get_from_name(None, value.name)
        else:
            name = value
            new_item = self.get_from_name(None, value)

        if new_item:
            index = self.itemlist.index(new_item)
            del self.itemlist[index]

            try:
                del self.queue[self.queue.index(value)]
            except ValueError:
                pass
            else:
                self.send_message(oid, 'remove_item', new_item)

            self.server.item_deleted(name)
            return name


    def perform_socket_command(self, method, **kwargs):
        name = kwargs.get("name")
        loop = self.loops.get(name)

        if name in self.cancelled:
            self.cancelled.get(name).set()
            del self.cancelled[name]
            
        if method == ActionType.IOMOD_RESPONSE:
            self.loops[name] = kwargs
            try:
                loop.quit()
            except Exception, e:
                echo('Exception in "pipe_monitor": %s' % e.message, "e")

        elif method == ActionType.IOMOD_PROGRESS:
            apply(self.progress_handler, (), kwargs)

        return True
            
    
    def listener(self, source, condition, sock):
        conn, addr = sock.accept()
        conn.setblocking(False)
        oid = id(conn)
        self.__connections[oid] = conn
        self.__oids[oid] = None
        glib.io_add_watch(conn,
                             glib.IO_IN |
                             glib.IO_HUP,
                             self._data_callback,
                             oid)
        return True


    def _data_callback(self, conn, condition, oid):
        mask = glib.IO_IN | glib.IO_HUP

        if condition in (glib.IO_IN, mask):
            try:
                data = conn.recv(10000000)
            except socket.error:
                pass
            else:
                self.extract_data(oid, data)

        if condition in (glib.IO_HUP, mask):
            conn = self.__connections[oid]
            try:
                conn.close()
            except Exception:
                pass
            self.__connections.pop(oid)
            self.__oids.pop(oid)

        return oid in self.__connections


    def extract_data(self, oid, source):
        for sample in xmlsplit(source):
            data = message_load(sample)
            command, arg, content = None, (), {}

            if data:
                command, arg, content = data
            else:
                continue

            if command == "ping":
                result = str(os.getpid())
                name = content.get("conn_name")
                self.__oids[name] = oid
                self.__oids[oid] = name
                self.__connections[oid].send(result)

            elif command in (ActionType.IOMOD_RESPONSE, ActionType.IOMOD_PROGRESS):
                apply(self.perform_socket_command, (command,), content)

            else:
                try:
                    handler = getattr(self, command)
                except (AttributeError, TypeError):
                    pass
                else:                        

                    if isinstance(content, (list, tuple)):
                        args = (oid,) + content
                        kwargs = {}
                    elif isinstance(content, dict):
                        kwargs = content
                        args = (oid,)
                    elif content is None:
                        kwargs = {}
                        args = ()
                    else:
                        args = oid, content
                        kwargs = {}

                    try:
                        if handler:
                            glib.idle_add(lambda *a:apply(handler, args, kwargs) and False)
                    except Exception:
                        pass
#                        sys.stderr.write('Exception in "extract_data": %s\n' % e)


    def progress_handler(self, *args, **kwargs):
        dump = apply(message_dump, ('progress',), kwargs)
        self.server.progress_event(
                                        kwargs.get("action"),
                                        kwargs.get("name"),
                                        kwargs.get("dt"),
                                        kwargs.get("dr"),
                                        kwargs.get("ut"),
                                        kwargs.get("ur")
                                    )

        for connection in self.__connections.values():
            try:
                connection.send(dump)
            except Exception:
                pass


    def new_interface(self, name, **kwargs):
        interface = iomod.Interface(name)
        if kwargs:
            apply(interface.set_default, (), kwargs)
        return interface


    def _wait_response(self, action, kwargs):
        filename = kwargs.get("filename")
        interface = apply(self.new_interface, (filename,), kwargs)
        name = interface.name
        interface.set_action(action)
        loop = glib.MainLoop()
        self.loops[name] = loop
        self.server._action_started(action, name)
        interface.start()
        loop.run()
        response = self.loops.get(name)
        del self.loops[name]
        return response



    def login(self, oid, **kwargs):
        response = apply(self.check, (None,), kwargs)
        status = response.get("status")
        reason = response.get("reason")

        if not status:
            response = self._wait_response("login", kwargs)
            status, reason, info = parse_response(response.get("data"))

        status = status in ('ok', True)
        values = {"status": status, "reason": reason or ""}
        self.server._action_complete('login', values)
        self.send_message(oid, 'login', values)
        return values
        

    def logout(self, oid, **kwargs):
        pass


    def check(self, oid, **kwargs):
        if None in [kwargs.get(i) for i in ('user', 'passwd')]:
            status = False
            reason = ""
        else:
            response = self._wait_response("check", kwargs)
            status, reason, info = parse_response(response.get("data"))

        status = status in ('ok', True)
        values = {"status": status, "reason": reason or ""}
        self.server._action_complete('check', values)
        self.send_message(oid, 'check', values)
        return values


    def upload(self, oid, **kwargs):
        filename = kwargs.get('filename')
        item = self.add_new_item(None, filename)
        status = False
        reason = ""
        obj = "<obj></obj>"
        info = {}

        can_upload = True
        autoauth = kwargs.get("autoauth") and "user" in kwargs and "passwd" in kwargs
        
        if item and item.status == STATUS_DONE and item.obj:
            status, reason, obj, info = True, 'Done', item.obj, item.info
            can_upload = False

        elif autoauth:
            values = apply(self.login, (oid,), kwargs)
            if not values.get("status"):
                reason = values.get("reason")
                status = values.get("status")
                can_upload = False

        if can_upload and item:
            values = item.name, item.status, STATUS_RUN
            item.status = STATUS_RUN
            apply(self.server._item_status_changed, values)
            response = self._wait_response("upload", kwargs)
            strdata = response.get("data")
            status, reason, info, obj = parse_response(strdata, True)

            try:
                index = self.queue.index(item)
            except (ValueError, IndexError):
                pass
            else:
                del self.queue[index]

            if status in ('ok', True):
                kwargs["id"] = info.get ("id")
                response = self._wait_response("short", kwargs)
                try:
                    info["short"] = re.findall("<url>(.*)</url>", response.get("data"))[0]
                except Exception: pass
                
                item.status = STATUS_DONE
                item.info = info
                item.obj = obj

            else:
                item.reason = reason
                item.status = status == "failed" and STATUS_FAILED or STATUS_CNCL

            self.server.item_status_changed(item.name, STATUS_RUN, item.status)

        status = status in ('ok', True)
        values = {"status": status, "reason": reason or "", "filename": filename}
        [values.__setitem__(k, v) for k, v in info.items()]
        self.server._action_complete('upload', values)
        self.send_message(oid, "upload", values)
        return values



    def delete(self, oid, **kw):
        user = kw.get('user')
        passwd = kw.get('passwd')
        key = kw.get('key')
        id = kw.get('id')

        if None in (user, passwd, key, id):
            status = False
            reason = Messages.ARGEMENTS_REQUIRES
        else:
            status, reason = self.login(None, user=user, passwd=passwd)

            if status:
                response = self._wait_response("delete", kw)
                status, reason, info = parse_response(response.get("data"))

        values = {"status": status, "reason": reason or "", "key": key, "id": id}
        self.server._action_complete('delete', values)
        self.send_message(oid, 'delete', values)
        return values
        


    def getpictures(self, oid, **kw):
        user = kw.get('user')
        passwd = kw.get('passwd')
        data = ""

        if user and passwd:
            status, reason = self.login(None,
                                         user = user,
                                         passwd = passwd,
                                         autoauth = True,
                                         proxytype = kw.get("proxytype"),
                                         proxynode = kw.get("proxynode"),
                                         proxyport = kw.get("proxyport"),
                                         proxyuser = kw.get("proxyuser"),
                                         proxypass = kw.get("proxypass")
                                        )
            if status:
                response = self._wait_response("getpictures", kw)
                status, reason, info, obj = parse_response(response.get("data"), True)
                data = convert_getpictures_data(obj)
                
        else:
            status = False
            reason = Messages.USERDATA_REQUIRED

        status = status in ('ok', True)
        values = {"status": status, "reason": reason or "", "info": data}
        self.server._action_complete('getpictures', values)
        self.send_message(oid,'getpictures', values)
        return values

            

    def check_connection(self, oid, **kwargs):
        response = self._wait_response("check_connection", kwargs)
        code_ok = response.get("code") in (0, 200)

        values = {"status": code_ok, "reason": response.get("data") or ""}
        self.server._action_complete('check_connection', values)
        self.send_message(oid, "chack_connection", values)
        return code_ok
            


    def get_all_items(self, oid, *args, **kwargs):
        if oid in (None, 'dbus'):
            return self.itemlist
        else:
            self.send_message(oid, 'get_all_items', self.itemlist)
        

    def get_queue(self, oid, *args, **kwargs):
        vlist = []
        if self.queue:
            vlist = self.queue
        else:
            [vlist.append(i) for i in self.itemlist if i.status == STATUS_WAIT]

        if oid in (None, 'dbus'):
            return vlist
        else:
            self.send_message(oid, 'get_queue', vlist)
        

    def get_uploaded(self, oid, *args, **kwargs):
        vlist = []
        [vlist.append(i) for i in self.itemlist if i.status == STATUS_DONE]

        if oid in (None, 'dbus'):
            return vlist
        else:
            self.send_message(oid, 'get_uploaded', vlist)
        

    def get_by_name(self, oid, name, *args, **kwargs):
        item = self.get_from_name(None, name)
        if oid in (None, 'dbus'):
            return item
        else:
            self.send_message(oid, 'get_by_name', item)




    def start(self, oid, *args, **kwargs):
        pass


    def stop(self, oid, *args, **kwargs):
        pass


    def cancel(self, oid, *args, **kwargs):
        name = kwargs.get('name')
        if name in self.__oids or oid == -1:
            if oid == -1:
                conn_list = self.__connections.values()
            else:
                conn_oid = self.__oids.get(name)
                conn = self.__connections.get(conn_oid)
                conn_list = [conn,]

            for conn in conn_list:
                try:
                    conn.shutdown(2)
                except Exception, e:
                    status = False
                    reason = e.message
                else:
                    status = True
                    reason = "Action cancelled"
                    self.perform_socket_command(
                                                    ActionType.IOMOD_RESPONSE,
                                                    name = name,
                                                    data = RESP_CANCELLED
                                                )
            if oid == -1:
                return 0
                
        else:
            status = False
            reason = "Name '%s' is not found." % name

        item = self.get_by_name(None, name)
        if item and item.status == STATUS_RUN:
            item.status = STATUS_WAIT

        values = {"status": status, "reason": reason, "name": name}
        self.server._action_complete("cancel", values)
        self.send_message(oid,'cancel', values)
        return values


    def get_status(self, oid, *args, **kwargs):
        pass


    def quiting(self, oid, **kw):
        self.mainloop.quit()
        self.cancel(-1)
        status = self.mainloop.is_running()

        if isinstance(oid, str) and oid == "dbus":
            return {'status': status, 'reason': ""}


    def main(self):
        self.mainloop.run()
        self.server.quiting()
        for loop, event in self.loops.values():
            loop.quit()
            event.set()
        for sock in self.__connections.values():
            try:
                sock.close()
            except Exception:
                pass
        tempclear(True)
            
        
if __name__ == '__main__':
    loop = Kernel()
    loop.main()

    
###
