// Cyphesis Online RPG Server and AI Engine
// Copyright (C) 2000,2001 Alistair Riddoch
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA


#ifndef RULESETS_EXTERNAL_MIND_H
#define RULESETS_EXTERNAL_MIND_H

#include "common/Router.h"
#include <Atlas/Objects/Operation.h>
#include <modules/Ref.h>

class Link;

class Entity;

class LocatedEntity;

/// \brief This class connects in-game entities to the Link of the client
/// controlling it.
///
/// Essentially used to relay in-game operations that pass to the mind on
/// to the client.
class ExternalMind : public Router
{
    protected:

        /// \brief Stores relay data, as generated by a Relay operation.
        ///
        /// Whenever a Relay operation is used it's with the intention that any response
        /// should be relayed back to the originating entity (using serialno and refno).
        /// This struct is used to keep track of this routing data, so that the operation
        /// can be correctly sent on to its destination.
        struct Relay
        {
            Atlas::Objects::Operation::RootOperation op;

            std::string from_id;
        };

        /**
         * Serial number generation to use when generating new serial numbers.
         */
        static long s_serialNumberNext;


    Link* m_link;
        Ref<LocatedEntity> m_entity;

        /**
         * \brief A store of registered relays for this character, both outgoing and incoming.
         *
         * Whenever a relay is set up between two entities entries are created in this
         * map. The refno of the operations is used as a key.
         */
        std::map<long, Relay> m_relays;

        void deleteEntity(const std::string& id, bool forceDelete);

        void purgeEntity(const LocatedEntity& ent, bool forceDelete = false);

        void RelayOperation(const Operation& op, OpVector& res);

        void externalRelayedOperation(const Operation& op);

    public:
        static int s_numberOfMinds;


        explicit ExternalMind(const std::string& strId, long id, Ref<LocatedEntity> entity);

        ~ExternalMind() override;

        void externalOperation(const Operation& op, Link&) override;

        void operation(const Operation&, OpVector&) override;

        bool isLinked()
        { return m_link != nullptr; }

        bool isLinkedTo(Link* c)
        { return m_link == c; }

        const Ref<LocatedEntity>& getEntity() const;

        const std::string& connectionId();

        void linkUp(Link* c);

        Link* getLink() const
        {
            return m_link;
        }

        void addToEntity(const Atlas::Objects::Entity::RootEntity&) const override;

        virtual void GetOperation(const Operation& smartPtr, OpVector& res);

};

#endif // RULESETS_EXTERNAL_MIND_H
