/*
Boost Software License - Version 1.0 - August 17th, 2003

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
 */

#ifndef WAITFREEQUEUE_H_
#define WAITFREEQUEUE_H_

#include <atomic>

namespace Eris
{

/**
 * @brief A queue optimized for insertion from background threads and consumption from one main thread.
 */
template<typename T>
class WaitFreeQueue
{
public:
    struct node
    {
        T data;
        node * next;
    };

    WaitFreeQueue() :
            _head(nullptr)
    {
    }

    void push(const T& data)
    {
        node* n = new node;
        n->data = data;
        node * stale_head = _head.load(std::memory_order_relaxed);
        do {
            n->next = stale_head;
        } while (!_head.compare_exchange_weak(stale_head, n,
                std::memory_order_release));
    }

    node* pop_all(void)
    {
        node* last = pop_all_reverse(), *first = nullptr;
        while (last) {
            node * tmp = last;
            last = last->next;
            tmp->next = first;
            first = tmp;
        }
        return first;
    }

    node * pop_all_reverse(void)
    {
        return _head.exchange(nullptr, std::memory_order_acquire);
    }

private:
    std::atomic<node*> _head;
};

}

#endif /* WAITFREEQUEUE_H_ */
