//
// C++ Interface: ActionBarIconManager
//
// Description:
//
//	Author Tiberiu Paunescu <tpa12@sfu.ca>, (C) 2010
//	Based on the EntityIcon class by Erik Ogenvik <erik@ogenvik.org>, (C) 2007
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.//
//
#ifndef EMBEROGRE_GUIACTIONBARICONMANAGER_H
#define EMBEROGRE_GUIACTIONBARICONMANAGER_H

#include <sigc++/signal.h>
#include <string>
#include <vector>

namespace Eris {
class ServerInfo;
}

namespace Ember {
class EmberEntity;
namespace OgreView {
class GUIManager;
namespace Gui {

namespace Icons
{
class Icon;
}

class ActionBarIconSlot;
class ActionBarIcon;

/**
@author Erik Ogenvik <erik@ogenvik.org>
@brief Handles and manages action bar icons.

An action bar icon is an icon which can be used to execute actions on a number of different objects. The icons can be dragged and dropped onto slots and created when other icons are dragged on the action bar.
This class is responsible for handling all of them in the UI, creating new and destroying old.
You normally don't create instances of neither ActionBarIconSlot nor ActionBarIcon yourself, but use the create* and destroy* methods in this class instead.
*/
class ActionBarIconManager
{
public:

	/**
	 * @brief A type which encompasses both a server info instance as well as an avatar id.
	 *
	 * An instance of this is used for saving avatar bound data.
	 */
	typedef std::pair<Eris::ServerInfo, std::string> AvatarIdType;

	/**
	 * @brief Ctor.
	 * @param guiManager The main GUI manager.
	 */
	ActionBarIconManager(GUIManager& guiManager);

	/**
	 * @brief Dtor.
	 *
	 * All icons and slots will be destroyed.
	 */
	virtual ~ActionBarIconManager();

	/**
	 * @brief Creates a new action bar icon slot.
	 * @param pixelSize The size of the slot in pixels.
	 * @returns A new slot instance.
	 */
	ActionBarIconSlot* createSlot(unsigned int pixelSize = 64);

	/**
	 * @brief Destroys an action bar icon slot.
	 * @param slot The slot to destroy.
	 */
	void destroySlot(ActionBarIconSlot* slot);

	/**
	 * @brief Creates a new action bar icon.
	 * @param icon An empty icon instance which will be used by the entity icon.
	 * @param pixelSize The size of the icon in pixels.
	 * @returns An action bar icon instance.
	 */
	ActionBarIcon* createIcon(Gui::Icons::Icon* icon, unsigned int pixelSize = 64);

	/**
	 * @brief Destroys an action bar icon.
	 *
	 * @param The action bar icon to destroy.
	 */
	void destroyIcon(ActionBarIcon* icon);

	/**
	 * @brief Gets a saved value from the server settings, specific for the supplied avatar.
	 *
	 * @param avatarId The id of the avatar.
	 * @param name The name of the key for the value to get.
	 *
	 * @returns The saved value.
	 */
	const std::string getSavedValue(const AvatarIdType& avatarId, const std::string& name) const;

	/**
	 * @brief Saves a value specific for the supplied avatar.
	 *
	 * @param avatarId The id of the avatar.
	 * @param name The name of the key for the value to save.
	 * @param value The value to save.
	 */
	void saveValue(const AvatarIdType& avatarId, const std::string& key, const std::string& value);

	/**
	 * @brief Erases a value specific for the supplied avatar.
	 *
	 * @param avatarId The id of the avatar.
	 * @param name The name of the key for the value to erase.
	 */
	void eraseValue(const AvatarIdType& avatarId, const std::string& key);

	/**
	 * @brief Emitted when a drag action of an entity icon has started.
	 */
	sigc::signal<void, ActionBarIcon*> EventIconDragStart;

	/**
	 * @brief Emitted when a drag action of an entity icon has ended.
	 */
	sigc::signal<void, ActionBarIcon*> EventIconDragStop;

protected:
	typedef std::vector<ActionBarIconSlot*> ActionBarIconSlotStore;
	typedef std::vector<ActionBarIcon*> ActionBarIconStore;

	/**
	 * @brief All of the action bar icon slots managed by this instance.
	 */
	ActionBarIconSlotStore mSlots;

	/**
	 * @brief All of the action bar icons managed by this instance.
	 */
	ActionBarIconStore mIcons;

	/**
	 * @brief The main GUI manager.
	 */
	GUIManager& mGuiManager;

	/**
	 * @brief A counter used for providing unique names for each icon window.
	 */
	int mIconsCounter;

	/**
	 * @brief A counter used for providing unique names for each slot window.
	 */
	int mSlotsCounter;
};

}

}

}

#endif
