/****************************************************************************

Copyright (c) 2000 - 2010 Novell, Inc.
All Rights Reserved.

This program is free software; you can redistribute it and/or
modify it under the terms of version 2 of the GNU General Public License as
published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.   See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, contact Novell, Inc.

To contact Novell about this file by physical or electronic mail,
you may find current contact information at www.novell.com

****************************************************************************


  File:		YCPMacroRecorder.cc

  Author:	Stefan Hundhammer <sh@suse.de>

/-*/


#include <stdio.h>
#include <sys/time.h>
#include <ycp/YCPSymbol.h>
#include <ycp/YCPString.h>
#include <ycp/YCPTerm.h>
#include <ycp/YCPVoid.h>
#include <ycp/ExecutionEnvironment.h>

#define YUILogComponent "ui-macro"
#include <yui/YUILog.h>

#include <yui/YUISymbols.h>
#include <yui/YWidget.h>
#include <yui/YInputField.h>
#include "YCPMacroRecorder.h"
#include "YUIComponent.h"
#include "YCPValueWidgetID.h"
#include <yui/YUI.h>
#include "YCP_util.h"
#include "YCP_UI.h"

#ifndef Y2LOG_DATE
#   define Y2LOG_DATE	"%Y-%m-%d %H:%M:%S"	/* The date format */
#endif



#define YMACRO_INDENT "    "	// 4 blanks


YCPMacroRecorder::YCPMacroRecorder()
    : YMacroRecorder()
    , _macroFile(0)
    , _screenShotCount(0)
    , _recording( false )
{
}


YCPMacroRecorder::~YCPMacroRecorder()
{
    writeMacroFileFooter();
    closeMacroFile();
}


void YCPMacroRecorder::record( const string & macroFileName )
{
    _screenShotCount = 0;
    openMacroFile( macroFileName );
    writeMacroFileHeader();
}


void YCPMacroRecorder::endRecording()
{
    writeMacroFileFooter();
    closeMacroFile();
}


bool YCPMacroRecorder::recording() const
{
    return _recording;
}


void YCPMacroRecorder::openMacroFile( const string & macroFileName )
{
    _macroFile = fopen( macroFileName.c_str(), "w" );

    if ( _macroFile )
    {
	_recording = true;
	yuiMilestone() << "Recording macro to " << macroFileName << endl;
    }
    else
    {
	yuiError() << "Can't record to macro file " << macroFileName << endl;
    }
}


void YCPMacroRecorder::closeMacroFile()
{
    if ( _macroFile )
    {
	fclose( _macroFile );
	_macroFile = 0;
	yuiMilestone() << "Macro recording done." << endl;
    }

    _recording = false;
}


void YCPMacroRecorder::writeMacroFileHeader()
{
    if ( ! _macroFile )
	return;

    fprintf( _macroFile,
	     "// YaST2 UI macro file generated by UI macro recorder\n"
	     "//\n"
	     "//     Qt UI: Alt-Ctrl-Shift-M: start/stop Macro recorder\n"
	     "//	    Alt-Ctrl-Shift-P: Play macro\n"
	     "//\n"
	     "// Each block will be executed just before the next UserInput().\n"
	     "// 'return' before the closing brace ( '}' ) of each block relinquishes control\n"
	     "// back to the YCP source.\n"
	     "// Inside each block arbitrary YCP code can be added manually.\n"
	     "\n"
	     "{\n"
	     );
}


void YCPMacroRecorder::writeMacroFileFooter()
{
    if ( ! _macroFile )
	return;

    fprintf( _macroFile, "}\n" );
}


void YCPMacroRecorder::recordYcpCodeLocation()
{
    YaST::ExecutionEnvironment::CallStack callStack(YaST::ee.callstack());

    if ( ! callStack.empty() )
    {
	const YaST::CallFrame* frame = callStack.back();
	string functionName;

	if ( frame && frame->function->entry()->toString().find( "Wizard::UserInput" ) == string::npos  )
	    functionName = frame->function->entry()->toString();

	if ( frame )
	{
	    if ( functionName.empty() )
	    {
		fprintf( _macroFile, "%s%s// Source: %s:%d\n",
			 YMACRO_INDENT, YMACRO_INDENT,
			 frame->filename.c_str(),
			 frame->linenumber );
	    }
	    else
	    {
		fprintf( _macroFile, "%s%s// Source: %s( %s ):%d\n",
			 YMACRO_INDENT, YMACRO_INDENT,
			 frame->filename.c_str(),
			 functionName.c_str(),
			 frame->linenumber );
	    }
	}
    }
}


void YCPMacroRecorder::recordTimeStamp()
{
    time_t now_seconds = time (NULL);
    struct tm *tm_now = localtime( &now_seconds );
    char timeStamp[80];		// that's big enough
    strftime( timeStamp, sizeof( timeStamp ), Y2LOG_DATE, tm_now );

    fprintf( _macroFile, "%s%s// %s\n",
	     YMACRO_INDENT, YMACRO_INDENT,
	     timeStamp );
}


void YCPMacroRecorder::recordComment( string text )
{
    fprintf( _macroFile, "%s%s// %s\n",
	     YMACRO_INDENT, YMACRO_INDENT,
	     text.c_str() );
}


void YCPMacroRecorder::beginBlock()
{
    if ( ! _macroFile )
	return;

    fprintf( _macroFile, "%s{\n", YMACRO_INDENT );
    fprintf( _macroFile, "%s%s//\n", YMACRO_INDENT, YMACRO_INDENT );
    recordYcpCodeLocation();
    recordTimeStamp();
    fprintf( _macroFile, "\n" );
}


void YCPMacroRecorder::endBlock()
{
    if ( ! _macroFile )
	return;

    fprintf( _macroFile, "\n" );
    fprintf( _macroFile, "%s%sreturn;\n", YMACRO_INDENT, YMACRO_INDENT );
    fprintf( _macroFile, "%s}\n\n", YMACRO_INDENT );
}


void YCPMacroRecorder::recordUserInput( const YCPValue & input )
{
    if ( ! _macroFile )
	return;

    fprintf( _macroFile, "\n" );

    recordMakeScreenShot();

    if ( input->isVoid() )
    {
	fprintf( _macroFile, "%s%sUI::%s();\n",
		 YMACRO_INDENT, YMACRO_INDENT,
		 YUIBuiltin_FakeUserInput );
    }
    else
    {
	fprintf( _macroFile, "%s%sUI::%s( %s );\n",
		 YMACRO_INDENT, YMACRO_INDENT,
		 YUIBuiltin_FakeUserInput,
		 input->toString().c_str() );
    }

    fflush( _macroFile );	// sync to disk at this point - for debugging

    yuiDebug() << "Input: " << input << endl;
}


void YCPMacroRecorder::recordMakeScreenShot( bool enabled, const string & fname )
{
    if ( ! _macroFile )
	return;

    // Automatically add a (commented out) UI::MakeScreenShot() statement.
    //
    // The screenshot goes to /tmp/yast2-*.png , but since that statement is
    // commented out anyway this is not a security hazard - the user has to
    // remove the comment characters to actually trigger any action upon macro
    // replay, and while he is at it, he can also chose some other directory.
    //
    // All this is mainly for the SuSE documentation department anyway who will
    // use that in a testing environment, not on some ultra-sensitive
    // server. We need a good default directory here, not some academic
    // bullshit discussion about creating temp directories with awkward names
    // and unusable permissions on the fly. Been there, done that, works only
    // for security theoreticians, not in real life. Those who devise such
    // schemes never seem to use them in reality.
    //
    // End of discussion before it even starts. ;-)

    string filename = fname;

    if ( filename.empty() )
    {
	char buffer[256];
	sprintf( buffer, "/tmp/yast2-%04d", _screenShotCount++ );
	filename = string( buffer );
    }

    fprintf( _macroFile, "%s%s%sUI::%s( \"%s\" );\n",
	     YMACRO_INDENT, YMACRO_INDENT,
	     enabled ? "" : "// ",
	     YUIBuiltin_MakeScreenShot, filename.c_str() );
}


void YCPMacroRecorder::recordWidgetProperty( YWidget *    widget,
					     const char * propertyName )
{
    if ( ! _macroFile )
	return;

    if ( ! widget )
    {
	yuiError() << "Null widget" << endl;
	return;
    }

    if ( ! widget->isValid() )
    {
	yuiError() << "Invalid widget" << endl;
	return;
    }

    if ( ! propertyName )
    {
	yuiError() << "Null property name" << endl;
	return;
    }

    if ( ! widget->hasId() )
    {
	// It's pointless to save properties if the widget doesn't have an ID -
	// there is no way to restore the property without an ID.

	return;
    }

    YCPValueWidgetID * widgetId = dynamic_cast<YCPValueWidgetID *> ( widget->id() );

    if ( ! widgetId )
	return;

    YCPTerm idTerm( YUISymbol_id );	// `id()
    idTerm->add( widgetId->value() );	// `id( `something )

    YInputField * inputField = dynamic_cast<YInputField *> (widget);

    if ( inputField && inputField->passwordMode() )
    {
	// Don't record passwords in the macro file

	string text = "UI::";
	text += YUIBuiltin_ChangeWidget;
	text += "( " + idTerm->toString() + ", \t`";
	text += YUIProperty_Value;
	text += ", \"<not recording password in plain text>\" );\t// ";
	text += widget->widgetClass();
	text += " \"" + widget->debugLabel() + "\"";

	recordComment( text );
    }
    else
    {
	YCPValue val = YCP_UI::QueryWidget( idTerm, YCPSymbol( propertyName ) );

	fprintf( _macroFile, "%s%sUI::%s( %s,\t`%s,\t%s );\t// %s \"%s\"\n",
		 // UI::ChangeWidget( `id( `something ), `Value, 42 ) // YWidget
		 YMACRO_INDENT, YMACRO_INDENT,
		 YUIBuiltin_ChangeWidget,
		 idTerm->toString().c_str(),
		 propertyName,
		 val->toString().c_str(),
		 widget->widgetClass(),
		 widget->debugLabel().c_str() );

	yuiDebug() << "Recording " 	<< widget->widgetClass()
		   << " status: "  	<< propertyName
		   << ": "		<< val
		   << endl;
    }
}

