/*
 * install.c
 * 
 * This is the first half of the install. It does just enough to get the
 * second half going. It, and everything it needs, has to fit on one floppy
 * along with a kernel and modules. Needless to say, it's a bit tight.
 *
 * Erik Troan (ewt@redhat.com)
 *
 * Copyright 1999 Red Hat Software 
 *
 * This software may be freely redistributed under the terms of the GNU
 * public license.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <alloca.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <netinet/in.h>		/* for ntohl */
#include <popt.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/mount.h>
#include <sys/stat.h>
#include <sys/sysmacros.h>
#include <sys/wait.h>
#include <unistd.h>
#include <zlib.h>

#include "cpio.h"
#include "devices.h"
#include "fs.h"
#include "install.h"
#include "otherinsmod.h"
#include "intl.h"
#include "kickstart.h"
#include "log.h"
#include "methods.h"
#include "net.h"
#include "newt.h"
#include "run.h"
#include "windows.h"
#ifdef __i386__
#include "pcmcia-probing/probe.h"
#endif

#define KS_NONE		0
#define KS_FLOPPY	1
#define KS_BOOTP	2
#define KS_FILE		3

#ifdef SINGLE_INSTALL
#define main install_main
#endif
int debug = 1;
int testing = 0;
int expert = 0;
int kickstart = 0;
int rescue = 0;
#ifdef __i386__
int pcmcia = 0;
#endif
char * ksFile;

/* hack */
int rmmod_main(int argc, char ** argv);
#ifdef INSTALL_INCLUDE_CARDMGR
int cardmgr_main(int argc, char ** argv);
#endif

void mountFloppy() {
    if (!testing) {
	logMessage("mounting ext2 fs on floppy");
	doMount("fd0", "/tmp/bootdisk", "ext2", 1, 0);
	logMessage("floppy filesystem mounted on /tmp/bootdisk");
    }
}

#ifdef __i386__
extern int loadPCMCIADisk(void);

int setupPCMCIA(char ** arg, int direction) {
    int rc;
    struct driversLoaded * dl = NULL;
    int status;
    char *pcic;
    int argc;
    char **argv;
    poptContext optCon;
    struct poptOption ksPcOptions[] = {
	{ 0, 0, 0, 0, 0 }
    };

    /* just probe and if pcmcia controller exists we load support
       automatically */
    if ((pcic = pcmciaProbeController()) == NULL) {
	if (direction < 0) return INST_CANCEL;
	return INST_OKAY;
    }
    logMessage("pcmcia pcic type: %s", pcic);

    /* go ahead and tell the rest of the install we found a
       PCMCIA controller */
    *arg = pcic;

    if (kickstart) {
	if (!ksGetCommand(KS_CMD_PCMCIA, argv, &argc, &argv)) {
	    const char *t;
	    
	    optCon = poptGetContext(NULL, argc, (const char **) argv, ksPcOptions, 0);
	    poptGetNextOpt(optCon);
	    t = poptGetArg(optCon);
	    if (t && *t && (!strcasecmp(t,"no") || !strcasecmp(t,"off") ||
			    !strcmp(t, "0")))
		return INST_OKAY;
	    else if (!(t && *t && (!strcasecmp(t,"yes") ||
				   !strcasecmp(t,"on") || !strcmp(t, "1")))) {
		newtWinMessage("PCMCIA Kickstart", "Ok",
			       "bad argument to kickstart "
			       "pcmcia command: %s.\nMust be "
			       "'on', '1', or 'yes' to enable, "
			       "or 'off', '0', or 'no' to disable.",t);
                kickstart=0;
	    }
	}
    } else {
      if (!pcmcia) {
	/* no need to ask anymore */
	/*
	rc = newtWinTernary(_("PCMCIA Support"), _("Yes"), _("No"), _("Back"),
			    _("Do you need to use PCMCIA devices during the "
			      "install? Answer no to this question if only "
			      "need PCMCIA support after the install. You do "
			      "not need install-time PCMCIA support if you "
			      "are installing Linux Mandrake on a laptop with "
			      "a built-in CDROM drive."));

        if (rc == 2)*/ return INST_OKAY;
	if (rc == 3) return INST_CANCEL;
      }
    }
    if (!pcmcia) {
	do {
            if (!kickstart) {
		rc = newtWinChoice(_("PCMCIA Support Disk"), _("Ok"), _("Back"),
			_("PCMCIA support requires a PCMCIA support disk. "
			 "Please remove the boot disk currently in your "
			 "drive and replace it with the Mandrake "
			 "PCMCIA support disk."));
		if (rc == 2) return INST_CANCEL;
	    }
	    if ((rc = loadPCMCIADisk()))
		kickstart = 0;
	} while (rc);
    }

    winStatus(40, 3, "PCMCIA", _("Starting PCMCIA services..."));

    loadModule("pcmcia_core", DRIVER_PCMCIA, DRIVER_MINOR_NONE, &dl);
    loadModule(pcic, DRIVER_PCMCIA, DRIVER_MINOR_NONE, &dl);
    loadModule("ds", DRIVER_PCMCIA, DRIVER_MINOR_NONE, &dl);

    if (!fork()) {
	pid_t child;

	if (!(child = fork())) {
	    execl("/sbin/cardmgr", "/sbin/cardmgr", "-m",
#ifdef INSTALL_INCLUDE_CARDMGR
		  pcmcia ? "/modules" : "/tmp/pcmcia/modules",
#else
		  "/tmp/pcmcia/lib/modules/preferred",
#endif
		  NULL);
	    exit(-1);
	}
	exit(child);
    }

    wait(&status);


    /* if cardmgr a chance to get going */
    sleep(5);

    newtPopWindow();

    return 0;
}
#endif

#ifdef SINGLE_INSTALL
extern void doSuspend(void);
#else
void doSuspend(void *data) {
    pid_t pid;
    int status;

    if (testing) {
	newtFinished();
	exit(1);
    } else if (access("/bin/sh", X_OK)) {
	return;
    }

    newtSuspend();
    if (!(pid = fork())) {
	printf(_("\n\nType <exit> to return to the install program.\n\n"));
	execl("/bin/sh", "-/bin/sh", NULL);
	perror("error execing /bin/sh");
	sleep(5);
	exit(1);
    }
    waitpid(pid, &status, 0);
    newtResume();
}
#endif

void doKickstart(struct intfInfo * intf, struct netInfo * netc,
		 struct driversLoaded ** dl) {
#ifndef DISABLE_NETWORK
    char * file;
    char * ksPath;
    int rc;

    if (kickstart == KS_BOOTP) {
	if ((bringUpNetworking(intf, netc, dl, 1))) {
	    kickstart = 0;
	} else if (!(intf->set & INTFINFO_HAS_BOOTSERVER)) {
	    newtWinMessage(_("Kickstart Error"), _("Ok"), _("No kickstart "
			   "configuration file server can be found."));
	    kickstart = 0;
	} else {
	    if (!(intf->set & INTFINFO_HAS_BOOTFILE)) {
		file = "/kickstart/";
		logMessage("bootp: no bootfile received");
	    } else {
		file = intf->bootFile;
	    }

	    ksPath = malloc(strlen(file) + strlen(netc->hostname) + 70);
	    strcpy(ksPath, inet_ntoa(intf->bootServer));
	    strcat(ksPath, ":");
	    strcat(ksPath, file);

	    if (ksPath[strlen(ksPath) - 1] == '/') {
		ksPath[strlen(ksPath) - 1] = '\0';
		file = malloc(30);
		sprintf(file, "%s-kickstart", inet_ntoa(intf->ip));
	    } else {
		file = strrchr(ksPath, '/');
		if (!file) {
		    file = ksPath;
		    ksPath = "/";
		} else {
		    *file++ = '\0';
		}
	    }

	    logMessage("ks server: %s file: %s", ksPath, file);

	    loadFilesystem("nfs", "nfs", dl);

	    if ((rc = doMount(ksPath, "/tmp/ks", "nfs", 1, 0))) {
		newtWinMessage(_("Error"), _("Ok"), 
			_("I could not mount the kickstart path %s.\n"),
			ksPath);
		kickstart = 0;
	    } else {
		ksFile = malloc(strlen(file) + 20);
		sprintf(ksFile, "/tmp/ks/%s", file);

		if (ksReadCommands(ksFile)) 
		    kickstart = 0;
	    }
	}
    }
#endif
}

#ifdef SPAWN_SHELL
void spawnShell(void) {
    pid_t pid;
    int fd;

    if (!testing) {
	fd = open("/dev/tty2", O_RDWR);
	if (fd < 0) {
	    logMessage("cannot open /dev/tty2 -- no shell will be provided");
	    return;
	} else if (access("/bin/sh",  X_OK))  {
	    logMessage("cannot open shell - /usr/bin/sh doesn't exist");
	    return;
	}

	if (!(pid = fork())) {
	    dup2(fd, 0);
	    dup2(fd, 1);
	    dup2(fd, 2);

	    close(fd);
	    setsid();
	    if (ioctl(0, TIOCSCTTY, NULL)) {
	       perror("could not set new controlling tty");
	    }

	    execl("/bin/sh", "-/bin/sh", NULL);
	    logMessage("exec of /bin/sh failed: %s", strerror(errno));
	}

	close(fd);
    }
}
#endif


static void ls(char *dir) {
  DIR *o = opendir(dir);
  
  if (o == NULL) printf("bad directory %s\n", dir);
  else {
    struct dirent *e;
    printf("%s:\n", dir);
    while ((e = readdir(o))) printf("- %s\n", e->d_name);
  }
}


int main(int argc, char ** argv) {
    char ** argptr;
    const char* arg;
    struct installMethod * method;
    int rc;
#ifdef __i386__
    char * pcmciaArg = NULL;
#endif
    char * childArgs[30];
    struct intfInfo intf;
    struct netInfo netc;
    struct driversLoaded * dl = NULL;
    char * ksMode = NULL;
    int stage, direction;
    poptContext optCon;
    int cont = 0;
    int hd = 0;
    int cdrom = 0;
    int nfs = 0;
    int ftp = 0;
    struct poptOption optionTable[] = {
	    { "expert", '\0', POPT_ARG_NONE, &expert, 0 },
	    { "kickstart", '\0', POPT_ARG_STRING, &ksMode, 0 },
	    { "ks", '\0', POPT_ARG_STRING, &ksMode, 0 },
	    { "test", '\0', POPT_ARG_NONE, &testing, 0 },
	    { "hd", '\0', POPT_ARG_NONE, &hd, 0 },
	    { "cdrom", '\0', POPT_ARG_NONE, &cdrom, 0},
	    { "nfs", '\0', POPT_ARG_NONE, &nfs, 0 },
	    { "ftp", '\0', POPT_ARG_NONE, &ftp, 0 },
	    { "rescue", '\0', POPT_ARG_NONE, &rescue, 0},
#ifdef __i386__
	    { "pcmcia", '\0', POPT_ARG_NONE, &pcmcia, 0},
#endif
	    { 0, 0, 0, 0, 0 }
    };

#ifdef SPAWN_SHELL
    spawnShell();
#endif

    if (!strcmp(argv[0] + strlen(argv[0]) - 6, "insmod") ||
	!strcmp(argv[0] + strlen(argv[0]) - 8, "modprobe")) {
	return ourInsmodCommand(argc, argv);
    } else if (!strcmp(argv[0] + strlen(argv[0]) - 5, "rmmod")) {
	return rmmod_main(argc, argv);
#ifdef INSTALL_INCLUDE_CARDMGR
    } else if (!strcmp(argv[0] + strlen(argv[0]) - 7, "cardmgr")) {
	return cardmgr_main(argc, argv);
#endif
    } else if (!strcmp(argv[0], "install-continue")) {
	cont = 1;
    }

    memset(&intf, 0, sizeof(intf));
    memset(&netc, 0, sizeof(netc));

    optCon = poptGetContext(NULL, argc, (const char **) argv, optionTable, 0);

    if ((rc = poptGetNextOpt(optCon)) < -1) {
	fprintf(stderr, "bad option %s: %s\n",
		       poptBadOption(optCon, POPT_BADOPTION_NOALIAS), 
		       poptStrerror(rc));
	exit(1);
    }

    if ((arg = poptGetArg(optCon))) {
	fprintf(stderr, "unexpected argument: %s\n", arg);
	exit(1);
    }

    if (ksMode) {
	if (testing) {
	    kickstart = KS_FILE;
	    ksFile = ksMode;
	} else if (!strcmp(ksMode, "floppy"))
	    kickstart = KS_FLOPPY;
	else if (!strcmp(ksMode, "bootp"))
	    kickstart = KS_BOOTP;
	else {
	    fprintf(stderr, "unknown kickstart option %s\n", ksMode);
	    exit(1);
	}
    }

    openLog(testing);

    logMessage("welcome to the Mandrake install "
	       "(first stage, version " INSTALL_VERSION " built " __DATE__ " "
	       __TIME__")");

    /* Speed up all IDE drives... */
    /*speeddisks();*/

    /* Because termcap isn't in standard place */
    setenv("TERMINFO", "/etc/terminfo", 1);
    
    newtInit();
    newtCls();

    newtSetSuspendCallback(doSuspend, NULL);
    newtDrawRootText(0, 0, _("Welcome to Linux Mandrake"));

    newtPushHelpLine(_("  <Tab>/<Alt-Tab> between elements  | <Space> selects | <F12> next screen "));
    
    loadModuleDep("/modules/modules.dep");

    loadFilesystem("vfat", "vfat", &dl); /* here for PCMCIA disk (vfat) */
    /* kickstart from floppy needs to be handled before PCMCIA */
    if (!cont && kickstart == KS_FLOPPY) {
	if (devMakeInode("fd0", "/tmp/fd0"))
	    kickstart = 0;
	else if (doMount("/tmp/fd0", "/tmp/ks", "vfat", 1, 0)) {
	    newtWinMessage(_("Error"), _("Ok"), 
		    _("I could not mount the boot floppy."));
	    kickstart = 0;
	} else if (access("/tmp/ks/ks.cfg", R_OK)) {
	    newtWinMessage(_("Error"), _("Ok"), 
		    _("Cannot find ks.cfg on boot floppy."));
	    kickstart = 0;
	} else {
	    int infd = -1, outfd = -1;
	    char buf[4096];
	    int i;

	    if ((outfd = open("/tmp/ks.cfg", O_CREAT | O_RDWR, 0666)) < 0 
	         || (infd = open("/tmp/ks/ks.cfg", O_RDONLY)) < 0) {
		newtWinMessage(_("Error"), _("Ok"), _("Error opening files "
			       "for kickstart copy: %s\n"), strerror(errno));
		kickstart = 0;
	    } else {
		while ((i = read(infd, buf, sizeof(buf))) > 0) {
		    if (write(outfd, buf, i) != i) break;
		}

		if (infd >= 0) close(infd);
		if (outfd >= 0) close(outfd);

		if (!i) {
		    ksFile = alloca(30);
		    strcpy(ksFile, "/tmp/ks.cfg");

		    if (ksReadCommands(ksFile)) 
			kickstart = 0;
		} else {
		    newtWinMessage(_("Error"), _("Ok"), 
			       _("Error copying kickstart file from floppy."));
		    kickstart = 0;
		}
	    }

	    umount("/tmp/ks");
	    devRemoveInode("/tmp/fd0");
	}
    } else if (cont && kickstart == KS_FLOPPY) {
	 /* continue install after language re-exec - all data was lost */
	ksFile = strdup("/tmp/ks.cfg");

	if (ksReadCommands(ksFile)) 
	    kickstart = 0;
    } else if (kickstart == KS_FILE) {
	rc = 0;
	while (!rc)
	    if ((rc = ksReadCommands(ksFile))) 
		kickstart = 0;
    }

    direction = 1;
    if (cont)
	stage = 1;
    else
	stage = 0;

    do {
	switch (stage) {
	  case 0:
	    newtFinished();
	    argv[0] = "install-continue";
	    execv(testing ? "./install" : "/sbin/install", argv);
	    break;

	  case 1:
	    stage++;
	    break;

	  case 2:
#if defined(__i386__)
	    rc = setupPCMCIA(&pcmciaArg, direction);
	    direction = (rc == INST_CANCEL) ? -1 : 1;
#endif
	    stage += direction;
	    break;

	  case 3:
	    doKickstart(&intf, &netc, &dl);
	    stage += direction;
	    break;

	  case 4:
	    if (cdrom) winStatus(20, 3, _("Starting"), _("Please wait..."));


	    if (!ftp && !nfs && !hd && !cdrom) /* if they didn't say, assume all */
		ftp = nfs = hd = cdrom = 1;

	    rc = chooseInstallMethod(hd, cdrom, nfs, ftp, &method, &netc, &intf, &dl);
	    direction = (rc == INST_CANCEL) ? -1 : 1;
	    stage += direction;
	    break;
	}
    } while (stage < 5);

    logMessage("method selection completed");

#ifndef DISABLE_NETWORK
    /* FIXME: is this a decent test? */
    if (intf.set) {
	writeNetInterfaceConfig("/tmp", &intf);
	writeNetConfig("/tmp", &netc, &intf, 1);
    }
#endif
    if (dl) writeModuleConf("/tmp", dl, 0);

    logMessage("state saved to /tmp");

    newtFinished();

    if (symlink("/tmp/rhimage/Mandrake/mdkinst", "/tmp/stage2") != 0)
      logMessage("symlink /tmp/stage2 failed");

    closeLog();

    if (testing) exit(0);

    if (rescue) {
      int fd = open("/proc/sys/kernel/real-root-dev", O_RDWR);
      write(fd, "0x103", sizeof("0x103")); /* ram3 */
      close(fd);
      exit(0);
    }

    argptr = childArgs;
    *argptr++ = "/usr/bin/runinstall2";
    *argptr++ = "--method";
    *argptr++ = method->abbrev;

    if (expert)
	*argptr++ = "--expert";

#ifdef __i386__
    if (pcmciaArg) {
	*argptr++ = "--pcmcia";
	*argptr++ = pcmciaArg;
    }
#endif

    *argptr++ = NULL;

    execv(childArgs[0], childArgs);

    rc = errno;
    openLog(testing);
    logMessage("error in exec of second stage loader :-(");
    logMessage("\terror:%s", strerror(rc));
 
    childArgs[0] = "-";
    childArgs[1] = NULL;
    execv("/bin/ash", childArgs); /* at least to do something */
    while (1) ;

    return 0;
}
