unit SDUGeneral;
// Description: Sarah Dean's General Utils
// By Sarah Dean
// Email: sdean12@mailcity.com
// WWW:   http://www.fortunecity.com/skyscraper/true/882/
//
// -----------------------------------------------------------------------------
// Version history:
// 1.0.0 - 6th November 1999
//         Initial release
//


interface

uses forms, controls, stdctrls,
     Windows, // needed for TWIN32FindData in ConvertSFNPartToLFN, and THandle
     classes,
     comctrls, // required in SDUEnableControl to enable/disable TRichedit controls
     dialogs; // needed for SDUOpenSaveAsDialogSetup

// Setup a Open/Save dialog with supplied default filename & path
// Fixes problem with just setting "filename" for these dialogs before
// Execute()ing them
procedure SDUOpenSaveDialogSetup(dlg: TCommonDialog; defaultFilename: string);
// Convert short filename to LFN
function  SDUConvertSFNToLFN(sfn: String): String;
// Convert LFN to short filename
function  SDUConvertLFNToSFN(lfn: string): string;
// Enable/disable the specified control, with correct colors
procedure SDUEnableControl(control: TControl; enable: boolean);
// Set "value" to the value of the command line parameter "-<parameter> value". Returns TRUE/FALSE on success/failure
function  SDUCommandLineParameter(parameter: string; var value: string): boolean;
// Returns TRUE if the specified command line switch could be found, otherwise FALSE
function  SDUCommandLineSwitch(parameter: string): boolean;
// Returns the parameter number in the command line of the specified parameter. Returns -1 on failure
function  SDUCommandLineSwitchNumber(parameter: string): integer;
// Returns the executables version numbers as set in Project|Options|Version Info
function SDUGetVersionInfo(filename: string; var majorVersion, minorVersion, revisionVersion, buildVersion: integer): boolean;
// As SDUGetVersionInfo, but returns a nicely formatted string
function SDUGetVersionInfoString(filename: string): string;
// Pause for the given number of ms
procedure SDUPause(delayLen: integer);
// Execute the specified commandline and return when the command line returns
function  SDUWinExecAndWait32(cmdLine: string; cmdShow: integer; workDir: string = ''): integer;
// Returns the control within parentControl which has the specified tag value
function  SDUGetControlWithTag(tag: integer; parentControl: TControl): TControl;
// Display the Windows shell dialog displaying the properties for the specified item
procedure SDUShowFileProperties(const filename: String);
// Get a list of all environment variables
function SDUGetEnvironmentStrings(envStrings: TStringList): boolean;
// Get the value of the specified environment variable
function SDUGetEnvironmentVar(envVar: string; var value: string): boolean;
// Set the specified time/datestamps on the file referred to by Handle
// Identical to "SetFileTime", but updates all 3 timestamps (created, last
// modified and last accessed)
function SDUSetAllFileTimes(Handle: Integer; Age: Integer): integer;
// Counts the number of instances of theChar in theString, and returns this
// count
function SDUCountCharInstances(theChar: char; theString: string): integer;
// These two functions ripped from FileCtrl.pas - see Delphi 4 source
function SDUVolumeID(DriveChar: Char): string;
function SDUNetworkVolume(DriveChar: Char): string;
// This calls SDUVolumeID/SDUNetworkVolume as appropriate
// Returns '3.5" Floppy'/'Removable Disk' instead of volume label for these
// type of drives
function SDUGetVolumeID(drive: char): string;
// Detect if the current application is already running, and if it is, return
// a handle to it's main window.
// Returns 0 if the application is not currently running
function SDUDetectExistingApp(): THandle;
// Split the string supplied into two parts, before and after the split char
function SDUSplitString(wholeString: string; var firstItem: string; var theRest: string; splitOn: char = ' '): boolean;
// Convert a hex number into an integer
function SDUHexToInt(hex: string): integer;

implementation

uses SysUtils, graphics,
     extctrls,
     Spin, // needed for TSpinEdit control disabling (Under the "samples" tab on Delphi 4.0)
     ShellAPI, // needed for SDUShowFileProperties
     FileCtrl; // needed for TDriveType


var
  MyAppName   : Array[0..255] of Char;
  MyClassName : Array[0..255] of Char;
  NumFound    : Integer;
  LastFound   : HWnd;


// Setup a Open/Save dialog with supplied default filename & path
// Fixes problem with just setting "filename" for these dialogs before
// Execute()ing them
procedure SDUOpenSaveDialogSetup(dlg: TCommonDialog; defaultFilename: string);
var
  filename: string;
  initDir: string;
begin
  // uses FileCtrl
  defaultFilename := trim(defaultFilename);
  if not(directoryexists(defaultFilename)) then
    begin
    filename := extractfilename(defaultFilename);
    initDir := extractfilepath(defaultFilename);
    end
  else
    begin
    filename := '';
    initDir := defaultFilename;
    end;

  if dlg is TOpenDialog then
    begin
    TOpenDialog(dlg).filename := filename;
    TOpenDialog(dlg).initialdir := initDir;
    end
  else if dlg is TSaveDialog then
    begin
    TSaveDialog(dlg).filename := filename;
    TSaveDialog(dlg).initialdir := initDir;
    end;

end;

// Convert a short path & filename to it's LFN version
// [IN] sfn - the short filename to be converted into a long filename
// Returns - the long filename
function SDUConvertSFNToLFN(sfn: String): String;
  function SDUConvertSFNPartToLFN(sfn: String): String;
  var
    temp: TWIN32FindData;
    searchHandle: THandle;
  begin
    searchHandle := FindFirstFile(PChar(sfn), temp);
    if searchHandle <> ERROR_INVALID_HANDLE then
      begin
      Result := String(temp.cFileName);
      if Result = '' then
        begin
        Result := String(temp.cAlternateFileName);
        end;
      end
    else
      begin
      Result := '';
      end;
    Windows.FindClose(searchHandle);
  end;

var
  lastSlash: PChar;
  tempPathPtr: PChar;
  copySfn: string;
begin
  sfn := SDUConvertLFNToSFN(sfn);

  Result := '';

  if not(FileExists(sfn)) and not(DirectoryExists(sfn)) then
    begin
    // Result already set to '' so just exit.
    exit;
    end;

  copySfn := copy(sfn, 1, length(sfn));
  tempPathPtr := PChar(copySfn);
  lastSlash := StrRScan(tempPathPtr, '\');
  while lastSlash <> nil do
    begin
    Result := '\' + SDUConvertSFNPartToLFN(tempPathPtr) + Result;
    if lastSlash <> nil then
      begin
      lastSlash^ := char(0);
      lastSlash := StrRScan(tempPathPtr, '\');

      // This bit is required to take into account the possibility of being
      // passed a UNC filename (e.g. \\computer_name\share_name\path\filename)
      if ( (Pos('\\', tempPathPtr) = 1) and
           (SDUCountCharInstances('\', tempPathPtr)=3) ) then
        begin
        lastSlash := nil;
        end;
      end
      
    end;

  if tempPathPtr[1]=':' then
    begin
    tempPathPtr[0] := upcase(tempPathPtr[0]);
    end;

  Result := tempPathPtr + Result;
end;


// Convert a LFN to it's short version
// [IN] lfn - the LFN to be converted into a short filename
// Returns - the short filename
function SDUConvertLFNToSFN(lfn: string): string;
var
  sfn: string;
begin
  if not(FileExists(lfn)) and not(DirectoryExists(lfn)) then
    begin
    Result := '';
    exit;
    end;

  sfn := ExtractShortPathName(lfn);
  if sfn[2]=':' then
    begin
    sfn[1] := upcase(sfn[1]);
    end;
  Result := sfn;
end;


procedure SDUEnableControl(control: TControl; enable: boolean);
var
  i: integer;
begin
  if not(control is TPageControl) and
     not(control is TForm)        and
     not(control is TPanel)       then
    begin
    control.enabled := enable;
    end;

  if (control is TEdit)     OR
     (control is TSpinEdit) OR
     (control is TRichedit) OR
     (control is TDateTimePicker) OR
     (control is TComboBox) then
    begin
    if enable then
      begin
      TEdit(control).color := clWindow;
      end
    else
      begin
      TEdit(control).color := clBtnFace;
      end
    end
  else if control is TWinControl then
    begin
    for i:=0 to (TWinControl(control).ControlCount-1) do
      begin
      SDUEnableControl(TWinControl(control).Controls[i], enable);
      end;
    end;

  if control is TRichedit then
    begin
    TRichedit(control).enabled := TRUE;
    TRichedit(control).readonly := not(enable);
    end;

end;

function SDUCommandLineParameter(parameter: string; var value: string): boolean;
var
  i: integer;
  testParam: string;
begin
  Result := FALSE;
  parameter := uppercase(parameter);
  for i:=1 to (ParamCount-1) do
    begin
    testParam := uppercase(ParamStr(i));
    if ((testParam=('-'+parameter)) OR
        (testParam=('/'+parameter))) then
      begin
      value := ParamStr(i+1);
      Result := TRUE;
      break;
      end;
    end;

end;


function SDUCommandLineSwitch(parameter: string): boolean;
var
  i: integer;
begin
  Result := FALSE;
  parameter := uppercase(parameter);
  for i:=1 to ParamCount do
    begin
    if (uppercase(ParamStr(i))=('-'+parameter)) OR
       (uppercase(ParamStr(i))=('/'+parameter)) then
      begin
      Result := TRUE;
      break;
      end;
    end;

end;

function SDUCommandLineSwitchNumber(parameter: string): integer;
var
  i: integer;
begin
  Result := -1;
  parameter := uppercase(parameter);
  for i:=1 to ParamCount do
    begin
    if (uppercase(ParamStr(i))=('-'+parameter)) OR
       (uppercase(ParamStr(i))=('/'+parameter)) then
      begin
      Result := i;
      break;
      end;
    end;

end;

// set filename to '' to get version info on the currently running executable
function SDUGetVersionInfo(filename: string; var majorVersion, minorVersion, revisionVersion, buildVersion: integer): boolean;
var
  vsize: Integer;
  puLen: Cardinal;
  dwHandle: DWORD;
  pBlock: Pointer;
  pVPointer: Pointer;
  tvs: PVSFixedFileInfo;
begin
  Result := FALSE;

  if filename='' then
    begin
    filename := Application.ExeName;
    end;

  vsize := GetFileVersionInfoSize(PChar(filename),dwHandle);
  if vsize = 0 then
    begin
    exit;
    end;

  GetMem(pBlock,vsize);
  try
    if GetFileVersionInfo(PChar(filename),dwHandle,vsize,pBlock) then
      begin
      VerQueryValue(pBlock,'\',pVPointer,puLen);
      if puLen > 0 then
        begin
        tvs := PVSFixedFileInfo(pVPointer);
        majorVersion    := tvs^.dwFileVersionMS shr 16;
        minorVersion    := tvs^.dwFileVersionMS and $ffff;
        revisionVersion := tvs^.dwFileVersionLS shr 16;
        buildVersion    := tvs^.dwFileVersionLS and $ffff;
        Result := TRUE;
        end;
      end;
  finally
    FreeMem(pBlock);
  end;

end;

function SDUGetVersionInfoString(filename: string): string;
var
  majorVersion: integer;
  minorVersion: integer;
  revisionVersion: integer;
  buildVersion: integer;
begin
  Result := '';
  if SDUGetVersionInfo(filename, majorVersion, minorVersion, revisionVersion, buildVersion) then
    begin
    Result := Format('%d.%.2d.%.2d.%.4d', [majorVersion, minorVersion, revisionVersion, buildVersion]);
    end;

end;

procedure SDUPause(delayLen: integer);
var
  delay: TTimeStamp;
begin
  delay := DateTimeToTimeStamp(now);
  delay.Time := delay.Time+delayLen;
  while (delay.time>DateTimeToTimeStamp(now).time) do
    begin
    // Nothing - just pause
    end;

end;


// !! WARNING !! cmdLine must contain no whitespaces, otherwise the first
//               parameter in the CreateProcess call must be set
// xxx - sort out that warning will sort this out later
// This function ripped from UDDF
// [IN] cmdLine - the command line to execute
// [IN] cmsShow - see the description of the nCmdShow parameter of the
//                ShowWindow function
// [IN] workDir - the working dir of the cmdLine (default is ""; no working dir)
// Returns: The return value of the command, or $FFFFFFFF on failure
function SDUWinExecAndWait32(cmdLine: string; cmdShow: integer; workDir: string = ''): integer;
var
  zAppName:array[0..512] of char;
//  zCurDir:array[0..255] of char;
//  WorkDir:String;
  StartupInfo:TStartupInfo;
  ProcessInfo:TProcessInformation;
  retVal: DWORD;
  pWrkDir: PChar;
begin
  retVal := $FFFFFFFF;

  StrPCopy(zAppName, cmdLine);
  FillChar(StartupInfo, Sizeof(StartupInfo), #0);
  StartupInfo.cb := Sizeof(StartupInfo);

  StartupInfo.dwFlags := STARTF_USESHOWWINDOW;
  StartupInfo.wShowWindow := cmdShow;

  pWrkDir := nil;
  if workDir<>'' then
    begin
    pWrkDir := PChar(workDir);
    end;

  if CreateProcess(nil,
                   zAppName,              { pointer to command line string }
                   nil,                   { pointer to process security attributes }
                   nil,                   { pointer to thread security attributes }
                   false,                 { handle inheritance flag }
                   CREATE_NEW_CONSOLE or  { creation flags }
                   NORMAL_PRIORITY_CLASS,
                   nil,                   { pointer to new environment block }
                   pWrkDir,               { pointer to current directory name }
                   StartupInfo,           { pointer to STARTUPINFO }
                   ProcessInfo) then      { pointer to PROCESS_INF }
    begin
    WaitforSingleObject(ProcessInfo.hProcess, INFINITE);
    GetExitCodeProcess(ProcessInfo.hProcess, retVal);
    CloseHandle(ProcessInfo.hProcess);
    CloseHandle(ProcessInfo.hThread);
    end;

  Result := retVal;

end;

function SDUGetControlWithTag(tag: integer; parentControl: TControl): TControl;
var
  i: integer;
begin
  Result := nil;

  for i:=0 to (TWinControl(parentControl).ControlCount-1) do
    begin
    if TWinControl(parentControl).Controls[i].tag = tag then
      begin
      Result := TWinControl(parentControl).Controls[i];
      end;
    end;

end;


procedure SDUShowFileProperties(const filename: String);
var
  sei: TShellExecuteinfo;
begin
  FillChar(sei,sizeof(sei),0);
  sei.cbSize := sizeof(sei);
  sei.lpFile := Pchar(filename);
  sei.lpVerb := 'properties';
  sei.fMask  := SEE_MASK_INVOKEIDLIST;
  ShellExecuteEx(@sei);
end;

  function SDUVolumeID(DriveChar: Char): string;
  var
    OldErrorMode: Integer;
    NotUsed, VolFlags: DWORD;
    Buf: array [0..MAX_PATH] of Char;
  begin
    OldErrorMode := SetErrorMode(SEM_FAILCRITICALERRORS);
    try
      Buf[0] := #$00;
      if GetVolumeInformation(PChar(DriveChar + ':\'), Buf, DWORD(sizeof(Buf)),
        nil, NotUsed, VolFlags, nil, 0) then
        SetString(Result, Buf, StrLen(Buf))
      else Result := '';
    finally
      SetErrorMode(OldErrorMode);
    end;
  end;

  function SDUNetworkVolume(DriveChar: Char): string;
  var
    Buf: Array [0..MAX_PATH] of Char;
    DriveStr: array [0..3] of Char;
    BufferSize: DWORD;
  begin
    BufferSize := sizeof(Buf);
    DriveStr[0] := UpCase(DriveChar);
    DriveStr[1] := ':';
    DriveStr[2] := #0;
    if WNetGetConnection(DriveStr, Buf, BufferSize) = WN_SUCCESS then
    begin
      SetString(Result, Buf, BufferSize);
      if pos(#0, Result)>0 then
        begin
        delete(Result, pos(#0, Result), length(Result)-pos(#0, Result)+1);
        end;
      if DriveChar < 'a' then
        Result := AnsiUpperCaseFileName(Result)
      else
        Result := AnsiLowerCaseFileName(Result);
    end
    else
      Result := SDUVolumeID(DriveChar);
  end;

function SDUGetVolumeID(drive: char): string;
var
  DriveType: TDriveType;
begin
  Result := '';

  drive := upcase(drive);
  DriveType := TDriveType(GetDriveType(PChar(drive + ':\')));

  case DriveType of
    dtFloppy:
      begin
      if (drive='A') OR (drive='B') then
        begin
        Result := '3.5" Floppy';
        end
      else
        begin
        Result := 'Removable Disk';
        end;
      end;
    dtFixed:
      begin
      Result := SDUVolumeID(drive);
      end;
    dtNetwork:
      begin
      Result := SDUNetworkVolume(drive);
      end;
    dtCDROM:
      begin
      Result := SDUVolumeID(drive);
      end;
    dtRAM:
      begin
      Result := SDUVolumeID(drive);
      end;
  end;

end;


// Populate a TStringsList with environment variables
// [OUT] envStrings - a TStringsList to be populated with the names of all environment variables
// Returns TRUE/FALSE on success/failure
function SDUGetEnvironmentStrings(envStrings: TStringList): boolean;
var
  pEnvPtr, pSavePtr: PChar;
begin
  pEnvPtr := GetEnvironmentStrings;
  pSavePtr := pEnvPtr;
  repeat
    envStrings.add(Copy(StrPas(pEnvPtr),1,Pos('=',StrPas (pEnvPtr))-1));
    inc(pEnvPtr, StrLen(pEnvPtr)+1);
  until pEnvPtr^ = #0;

  FreeEnvironmentStrings(pSavePtr);

  Result := TRUE;
end;


// Get the value of the specified environment variable
// [IN] envVar - the name of the environment variable
// [OUT] value - set to the value of the environment variable on success
// Returns TRUE/FALSE on success/failure
function SDUGetEnvironmentVar(envVar: string; var value: string): boolean;
var
  buffer: string;
  buffSize: integer;
  i: integer;
begin
  Result := FALSE;

  SetString(buffer, nil, 1);
  buffSize := GetEnvironmentVariable(PChar(envVar), PChar(buffer), 0);
  if buffSize<>0 then
    begin
    SetString(buffer, nil, buffSize);
    GetEnvironmentVariable(PChar(envVar), PChar(buffer), buffSize);
    value := '';
    for i:=1 to buffSize-1 do
      begin
      value := value + buffer[i];
      end;

    Result := TRUE;
    end;

end;


// Set the specified time/datestamps on the file referred to by Handle
// Identical to "SetFileTime", but updates all 3 timestamps (created, last
// modified and last accessed)
function SDUSetAllFileTimes(Handle: Integer; Age: Integer): integer;
var
  LocalFileTime, FileTime: TFileTime;
begin
  Result := 0;
  if DosDateTimeToFileTime(LongRec(Age).Hi, LongRec(Age).Lo, LocalFileTime) and
    LocalFileTimeToFileTime(LocalFileTime, FileTime) and
    SetFileTime(Handle, @FileTime, @FileTime, @FileTime) then Exit;
  Result := GetLastError;
end;


// Counts the number of instances of theChar in theString, and returns this
// count
function SDUCountCharInstances(theChar: char; theString: string): integer;
var
  i: integer;
  count: integer;
begin
  count := 0;
  for i:=1 to length(theString) do
    begin
    if theString[i]=theChar then
      begin
      inc(count);
      end;
    end;

  Result := count;
end;



// This is used by SDUDetectExistingApp and carries out a check on the window
// supplied to see if it matches the current one
function SDUDetectExistingAppCheckWindow(Handle: HWND; Temp: LongInt): BOOL; stdcall;
var
  WindowName : Array[0..255] of Char;
  ClassName  : Array[0..255] of Char;
begin
  // Go get the windows class name
  // Is the window class the same?
  if GetClassName(Handle,ClassName,SizeOf(ClassName)) > 0 then
    begin
    if StrComp(ClassName,MyClassName) = 0 then
      begin
      // Get its window caption
      // Does this have the same window title?
      if GetWindowText(Handle,WindowName,SizeOf(WindowName)) > 0 then
        begin
        if StrComp(WindowName,MyAppName)=0 then
          begin
          inc(NumFound);
          // Are the handles different?
          if Handle <> Application.Handle then
            begin
            // Save it so we can bring it to the top later.
            LastFound := Handle;
            end;
          end;
        end;
      end;
    end;

  Result := TRUE;
end;


// Detect if the current application is already running, and if it is, return
// a handle to it's main window.
// Returns 0 if the application is not currently running
function SDUDetectExistingApp(): THandle;
begin
  // Determine what this application's name is...
  GetWindowText(Application.Handle, MyAppName, SizeOf(MyAppName));

  // ...then determine the class name for this application...
  GetClassName(Application.Handle, MyClassName, SizeOf(MyClassName));

  // ...and count how many others out there are Delphi apps with this title
  NumFound := 0;
  LastFound := 0;
  EnumWindows(@SDUDetectExistingAppCheckWindow, 0);

  Result := LastFound;
end;


// Split the string supplied into two parts, before and after the split char
function  SDUSplitString(wholeString: string; var firstItem: string; var theRest: string; splitOn: char = ' '): boolean;
begin
  Result := FALSE;
  firstItem := wholeString;
  if pos(splitOn, wholeString)>0 then
    begin
    firstItem := copy(wholeString, 1, (pos(splitOn, wholeString)-1));
    theRest := copy(wholeString, length(firstItem)+length(splitOn)+1, (length(wholeString)-(length(firstItem)+length(splitOn))));
    Result := TRUE;
    end;

end;


// Convert a hex number into an integer
function SDUHexToInt(hex: string): integer;
begin
  Result := StrToInt('$' + hex);

end;


END.

