#include <alloca.h>
#include <newt.h>
#include <popt.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/mount.h>

#include "devices.h"
#include "install.h"
#include "intl.h"
#include "lilo.h"
#include "log.h"
#include "run.h"
#include "windows.h"
#include "smp.h"

#define KERNEL_IMAGE "/boot/vmlinuz-%s"
#define KERNEL_SMPIMAGE "/boot/vmlinuz-%ssmp"

#ifdef __sparc__
#define LILO_BINARY "/mnt/sbin/silo"
#else
#define LILO_BINARY "/mnt/sbin/lilo"
#endif

static int mkinitrd(char * kernelVersion, char * initrdImage, int issmp) {
    char * argv[] = { "/sbin/mkinitrd", "-f", initrdImage, "--ifneeded", 
		      kernelVersion, NULL };
    int rc = 0;
    static int alreadyHappened[2] = { 0, 0 };
    char ver[60];

    #ifdef __powerpc__
        return 0;
    #endif    

    if (alreadyHappened[issmp]) return 0;
    
    if (issmp)
	sprintf(ver, "%ssmp", kernelVersion);
    else
	sprintf(ver, "%s", kernelVersion);
    argv[4] = ver;
	
    if (loadModule("loop", DRIVER_OTHER, DRIVER_MINOR_NONE, NULL)) 
	return INST_ERROR;
    
    winStatus(32, 3, 
#ifdef __sparc__
	      _("SILO"),
#else    
	      _("LILO"),
#endif
	      _("Creating initial ramdisk..."));
    rc = runProgramRoot(RUN_LOG, "/mnt", "/sbin/mkinitrd", argv);
    newtPopWindow();
	
    removeModule("loop");

    if (rc) {
	unlink("/mnt/boot/initrd");
    } else {
	alreadyHappened[issmp] = 1;
    }

    return rc;
}

#ifndef __powerpc__
static int noSpaceFilter(newtComponent entry, void * data, int ch, int cursor) {
    if (ch == ' ')
	return 0;
    return ch;
}
#endif /* !powerpc */

#define SKIP_LILO 1000

#if defined(__i386__) || defined(__sparc__)
static int liloWhere(char * hdName, char * bootDevice, char ** where) {
    char * format = "/dev/%-11s    %s";
    char ** items;
    int which = 0, rc;

    items = alloca(sizeof(*items) * 3);
    items[0] = alloca(80);
    items[1] = alloca(80);
    items[2] = NULL;

    sprintf(items[0], format, hdName, _("Master Boot Record"));
    sprintf(items[1], format, bootDevice, _("First sector of boot partition"));

#ifdef __i386__
    rc = newtWinMenu(_("Lilo Installation"), _("Where do you want to install "
		       "the bootloader?"), 50, 5, 5, 4, items, &which,
		     _("Ok"), _("Skip"), _("Back"), NULL);
#elif defined(__sparc__)
    rc = newtWinMenu(_("SILO Installation"), _("Where do you want to install "
		       "the bootloader?"), 50, 5, 5, 4, items, &which,
		     _("Ok"), _("Skip"), _("Back"), NULL);
#endif		     

    if (rc == 3) return INST_CANCEL;
    if (rc == 2) return SKIP_LILO;

    switch (which) {
      case 0: 		*where = hdName; break;
      case 1: 		*where = bootDevice; break;
    }

    return 0;
}
#elif defined(__powerpc__)
/*
 * lilo patches for LinuxPPC - pmac were written by Paul Mackerras
 * (paulus@cs.anu.edu.au)
 *
 * lilo patches for MkLinux (and other LinuxPPC systems) and final
 * code merge were written by Mark Hatle (fray@kernel.crashing.org)
 *
 * 10/22/98 Troy Benjegerdes (hozer@drgw.net)
 * Merged into RedHat 5.1/Rawhide installer and added PReP support
 * 10/22/98 Mark Hatle (fray@kernel.crashing.org)
 * Merge cleanup - MkLinux install cleaned up a bit
 * 01/19/99 Mark Hatle (fray@kernel.crashing.org
 * We're #if 0ing all of the quik/nvram initialization
 * quik/OF should be obsolete for newbies because of BootX!
 */
static int prepWhere(char * hdName, char * bootDevice, char ** where,
                       struct partitionTable table) {
    newtComponent text, label, form, okay, listbox, cancel, answer, skip;
    char buf[200];
    struct partition * which;
    int i;

    umount("/tmp/rhimage"); // Should be safe to unmount this right now,
                            // this drive might also be the bootdrive.

    newtOpenWindow(10, 2, 64, 19, _("PReP Boot Loader Configuration"));

    for (i = 0; i < table.count; i++)
        if (table.parts[i].type == PART_PPC_BOOT) break;

    if ( i == table.count ) {

        newtWinMessage(_("Error"), _("Ok"), _("I can not find any "
                        "PReP boot partitions! "
                        "You will have to boot from floppy or network")
                        );

        return SKIP_LILO;
        }

     while (1) {
     text = newtTextbox(1, 1, 62, 4, NEWT_TEXTBOX_WRAP);
     newtTextboxSetText(text,   _("On which PReP boot partition would you "
                                "like to place the kernel?"));

     okay = newtButton(6, 15, _("Ok"));
     skip = newtButton(22, 15, _("Skip"));
     cancel = newtButton(38, 15, _("Cancel"));

     form = newtForm(NULL, NULL, 0);

     listbox = newtListbox(7, 7, 7, NEWT_LISTBOX_RETURNEXIT);
     label = newtLabel(7, 6, _("Device"));

     for (i = 0; i < table.count; i++) {
        if (table.parts[i].type == PART_PPC_BOOT) {
                sprintf(buf, "/dev/%-5s", table.parts[i].device);
                newtListboxAddEntry(listbox, buf, &table.parts[i]);
        }
     }

     newtFormAddComponents(form, label, listbox, NULL);
     newtFormAddComponents(form, text, okay, skip, cancel, NULL);

     answer = newtRunForm(form);

     which = newtListboxGetCurrent(listbox);

     newtFormDestroy(form);
     newtPopWindow();

     if (answer == cancel) return INST_CANCEL;
     if (answer == skip) {
         newtWinMessage(_("Info"), _("Ok"), _("You must manually "
                        "install the kernel "
                        "by issueing a 'dd if=/boot/vmlinuz of=/dev/%s' "
                        "command.")
                       , table.parts[which - table.parts].device);

         return SKIP_LILO;
     }

     if ( ! strcmp(bootDevice,"sda1") &&  ! strcmp(bootDevice,"hda1") )
         newtWinMessage(_("Warning"), _("Ok"), _("The root partition "
                        "is not sda1 or hda1. The kernel will not be "
                        "able to find the root partition unless you "
                        "manually enter it at boot time. "
                        "(Yes, this needs to be fixed.)"));

     (*where) = table.parts[which - table.parts].device;

    return 0;
    }

}

#if 0
static int quikWhere(char * hdName, char * bootDevice, char ** where) {
    newtComponent text, yes, no, cancel, f, answer;
    int rc;

    newtOpenWindow(18, 6, 42, 10, "Quik Installation");

    text = newtTextbox(1, 1, 40, 3, NEWT_TEXTBOX_WRAP);
    newtTextboxSetText(text,
        _("Would you like to install or configure the quik bootloader on "
        "your system?"));

    yes = newtButton(3, 6, _("Yes"));
    no = newtButton(16, 6, _("No"));
    cancel = newtButton(29, 6, _("Cancel"));

    f = newtForm(NULL, NULL, 0);
    newtFormAddComponents(f, text, yes, no, cancel, NULL);

    answer = newtRunForm(f);
    if (answer == f)
      answer = newtFormGetCurrent(f);

    newtPopWindow();
    newtFormDestroy(f);
    if (answer == yes) {
        *where = bootDevice;
        rc = 0;
    } else if (answer == cancel) {
        rc = INST_CANCEL;
        return INST_CANCEL;
    } else {
        rc = SKIP_LILO;
    }

    return rc;
}
#endif

static int liloWhere(char * hdName, char * bootDevice, char ** where,
                     struct partitionTable table) {

    switch (machine_type) {
    case ID_MkLinux:
        newtWinMessage(_("MkLinux"), _("Ok"), _("You must manually edit "
                        "your lilo.conf file in MacOS.  You can edit "
                        "this file from the MkLinux Control-Panel.  Set "
                        "rootdev equal to /dev/%s."), bootDevice);

        return SKIP_LILO;

    case ID_PMac:
#if 0
       return quikWhere(hdName, bootDevice, where);
#endif
        newtWinMessage(_("BootX Configuration"), _("Ok"), _("To boot "
                        "into Linux you must set the 'Root device' "
                        "in BootX to /dev/%s.  You must also uncheck "
                        "the ramdisk option."), bootDevice);

        return SKIP_LILO;

    case ID_PReP:
       return prepWhere(hdName, bootDevice, where, table);

    case ID_CHRP:
    case ID_Unknown:
    default:
        newtWinMessage(_("Bootloader Installation"), _("Ok"),
                        _("Bootloader installation "
                        "is not currently available for your platform.  "
                        "You will have to manually install the bootloader. "
                        "When setting up the bootloader your boot device is "
                        "/dev/%s."), bootDevice);

    return SKIP_LILO;

    }
}
#endif

#ifndef __powerpc__
static void editBootLabel(struct partition * item) {
    newtComponent form, entry, okay, cancel, clear, answer;
    char buf[50];
    char * entryValue;
    newtGrid subgrid, grid, buttons;
 
    form = newtForm(NULL, NULL, 0);

    strcpy(buf, "/dev/");
    strcat(buf, item->device);

    entry = newtEntry(-1, -1, item->bootLabel, 20, &entryValue, 
		      NEWT_ENTRY_SCROLL | NEWT_ENTRY_RETURNEXIT);
    newtEntrySetFilter(entry, noSpaceFilter, NULL);


    subgrid = newtCreateGrid(2, 2);
    newtGridSetField(subgrid, 0, 0, NEWT_GRID_COMPONENT,
		     newtLabel(-1, -1, _("Device:")),
		     0, 0, 0, 0, NEWT_ANCHOR_LEFT, 0);
    newtGridSetField(subgrid, 1, 0, NEWT_GRID_COMPONENT,
		     newtLabel(-1, -1, buf),
		     1, 0, 0, 0, NEWT_ANCHOR_LEFT, 0);
    newtGridSetField(subgrid, 0, 1, NEWT_GRID_COMPONENT,
		     newtLabel(-1, -1, _("Boot label:")),
		     0, 0, 0, 0, NEWT_ANCHOR_LEFT, 0);
    newtGridSetField(subgrid, 1, 1, NEWT_GRID_COMPONENT, entry,
		     1, 0, 0, 0, NEWT_ANCHOR_LEFT, 0);

    buttons = newtButtonBar(_("Ok"), &okay, _("Clear"), &clear,
			    _("Cancel"), &cancel, NULL);

    grid = newtCreateGrid(1, 2);
    newtGridSetField(grid, 0, 0, NEWT_GRID_SUBGRID, subgrid,
		     0, 0, 0, 0, 0, 0);
    newtGridSetField(grid, 0, 1, NEWT_GRID_SUBGRID, buttons,
		     0, 1, 0, 0, 0, NEWT_GRID_FLAG_GROWX);

    newtGridAddComponentsToForm(grid, form, 1);
    newtGridWrappedWindow(grid, _("Edit Boot Label"));
    newtGridFree(grid, 1);

    do {
	answer = newtRunForm(form);

	if (answer == clear)
	    newtEntrySet(entry, "", 1);
    } while (answer == clear);

    if (answer != cancel) {
	if (item->bootLabel) free(item->bootLabel);

	if (strlen(entryValue))
	    item->bootLabel = strdup(entryValue);
	else
	    item->bootLabel = NULL;
    }

    newtPopWindow();
}	    
#endif /* !powerpc */

#ifdef __powerpc__
static int doinstallPRePlo(char * prefix, char * dev, char * rootdev,
                           struct partitionTable table,
                           char * append, char * kernelVersion,
                           char * hdname, int linear) {
    int rc;
    char * argv[] = { "/mnt/sbin/lilo", NULL, NULL, NULL };

    if ( access("/mnt/boot/vmlinuz", R_OK)) {
        newtWinMessage(_("Error"), _("Ok"), _("Can't find "
                        "/mnt/boot/vmlinuz"));
        return INST_ERROR;
    }

    argv[0] = "dd";
    argv[1] = "if=/boot/vmlinuz";
    argv[2] = malloc(128);
    sprintf(argv[2], "of=/dev/%s", dev);
    argv[3] = NULL;
    if (!testing){
       rc = runProgramRoot(RUN_LOG, "/mnt", "bin/dd", argv);
       if (rc)
           return INST_ERROR;
       else
           return 0;
    } else {
       newtWinMessage(_("Testing"), _("Ok"), _("I would run %s %s %s"),
                        argv[0], argv[1], argv[2]);
        return 0;
    }

}

#endif

static int doinstallLilo(char * prefix, char * dev, char * rootdev,
			 char * bootdev,
			 struct partitionTable table,
			 char * append, char * kernelVersion, 
			 char * hdname, int linear) {
    char filename[100];
    FILE * f;
    char * argv[] = { LILO_BINARY, NULL, NULL };
    char * temp;
    int i;
    int rc;
    struct stat sb;
    int useInitrd = 0;
    char relinitrdImage[50], absinitrdImage[55];
    char buff[1024];
#ifndef __powerpc__
    int pass;
#endif
    static int firstTime = 1;
    static int issmp = 0;
    char relsmpinitrdImage[50];
    int usesmpInitrd = 0;

    #ifdef __powerpc__
    if (machine_type == ID_PReP)
      return doinstallPRePlo(prefix, dev, rootdev, table, append,
                             kernelVersion, hdname, linear);
    #endif

    sprintf(relinitrdImage, "/boot/initrd-%s.img", kernelVersion);
    strcpy(absinitrdImage, "/mnt");
    strcat(absinitrdImage, relinitrdImage);

    if (mkinitrd(kernelVersion, relinitrdImage, 0))
	return INST_ERROR;

    if (testing) return 0;

    if (!stat(absinitrdImage, &sb))
	useInitrd = 1;

    if ((issmp = detectSMP()) == -1) {
	logMessage("Error trying to detect SMP machine");
	issmp = 0;
    }
    if (issmp) {
	sprintf(buff, "/mnt" KERNEL_SMPIMAGE, kernelVersion);
	if (stat(buff, &sb)) {
	    logMessage("SMP machine, but no SMP kernel found");
	    issmp = 0;
	}
    }
    if (issmp) {
	char abssmpinitrdImage[55];
	sprintf(relsmpinitrdImage, "/boot/initrd-%ssmp.img", kernelVersion);
	if (mkinitrd(kernelVersion, relsmpinitrdImage, 1))
	    issmp = 0;
	else {
	    strcpy(abssmpinitrdImage, "/mnt");
	    strcat(abssmpinitrdImage, relsmpinitrdImage);
	    
	    if (!stat (abssmpinitrdImage, &sb))
		usesmpInitrd = 1;
	}
    }
#ifdef __sparc__
	sprintf(filename, "%s/silo.conf", prefix);
#elif __powerpc__
        sprintf(filename, "%s/quik.conf", prefix);
#else /* !__sparc__ || __powerpc__ */
	sprintf(filename, "%s/lilo.conf", prefix);
#endif 

    /* why not? */
    if (firstTime) {
	rename("/mnt/etc/lilo.conf", "/mnt/etc/lilo.conf.rpmsave");
	rename("/mnt/etc/silo.conf", "/mnt/etc/silo.conf.rpmsave");
	firstTime = 0;
    }
    
    f = fopen(filename, "w");
    if (!f) {
	errorWindow("cannot create [ls]ilo config file: %s");
	return INST_ERROR;
    }

    logMessage("writing [sl]ilo config to %s", filename);

#ifdef __i386__
	fprintf(f, "boot=/dev/%s\n", dev);
	fprintf(f, "map=/boot/map\n");
	fprintf(f, "install=/boot/boot.b\n");
	fprintf(f, "prompt\n");
	if (linear) fprintf(f, "linear\n");
	fprintf(f, "timeout=50\n");
#elif __sparc__
	fprintf(f, "timeout=50\n");
	fprintf(f, "partition=%s\n", rootdev + 3);
	fprintf(f, "root=/dev/%s\n", rootdev);
#elif __powerpc__
        fprintf(f, "timeout=50\n");
        fprintf(f, "image=%s\n", KERNEL_IMAGE);
        fprintf(f, "\tlabel=linux\n");
        if (append) fprintf(f, "\tappend=\"%s\"\n", append);
#else /* !__sparc__ && !__i386__  && !__powerpc__ */
#error "unsupported architecture";
#endif /* __sparc__ */

#ifndef __powerpc__
    for (pass = 0; pass < 2; pass++) {
	for (i = 0; i < table.count; i++) {
	    if (!table.parts[i].bootLabel) continue;
	    if (pass == 0 && !table.parts[i].defaultBoot) continue;
	    if (pass == 1 && table.parts[i].defaultBoot) continue;

	    if (table.parts[i].type == PART_EXT2) {
          	if (issmp) {
		    fprintf(f, "image=" KERNEL_SMPIMAGE "\n", kernelVersion);
		    fprintf(f, "\tlabel=%s\n", table.parts[i].bootLabel);

	            /* remap /boot to root=/, but make sure we set up booting
		       from old root partitions as well */
		    if (strcmp(bootdev, table.parts[i].device))
			fprintf(f, "\troot=/dev/%s\n", table.parts[i].device);
#ifndef __sparc__		    
		    else
			fprintf(f, "\troot=/dev/%s\n", rootdev);
#endif
		    if (usesmpInitrd)
		        fprintf(f, "\tinitrd=%s\n", relsmpinitrdImage);
		    if (append) fprintf(f, "\tappend=\"%s\"\n", append);
			fprintf(f, "\tread-only\n");
          	}
		fprintf(f, "image=" KERNEL_IMAGE "\n", kernelVersion);
		if (issmp)
                    fprintf(f, "\tlabel=linux-up\n");
		else
                    fprintf(f, "\tlabel=%s\n", table.parts[i].bootLabel);

		/* remap /boot to root=/, but make sure we set up booting
		   from old root partitions as well */
		if (strcmp(bootdev, table.parts[i].device))
		    fprintf(f, "\troot=/dev/%s\n", table.parts[i].device);
#ifndef __sparc__		    
		else
		    fprintf(f, "\troot=/dev/%s\n", rootdev);
#endif
		if (useInitrd)
		    fprintf(f, "\tinitrd=%s\n", relinitrdImage);
		if (append) fprintf(f, "\tappend=\"%s\"\n", append);
		fprintf(f, "\tread-only\n");
#ifdef __i386__
	    } else {
		fprintf(f, "other=/dev/%s\n", table.parts[i].device);
		fprintf(f, "\tlabel=%s\n", table.parts[i].bootLabel);
		fprintf(f, "\ttable=/dev/%.3s\n", table.parts[i].device);

		if (strncmp(table.parts[i].device, hdname, 3)) {
		    /* boot off the second drive, so reverse the BIOS maps */
		    fprintf(f, "\tmap-drive=0x80\n");
		    fprintf(f, "\t   to = 0x81\n");
		    fprintf(f, "\tmap-drive=0x81\n");
		    fprintf(f, "\t   to = 0x80\n");
		}
#endif
	    }
	}
    }
#endif

    fclose(f);

    winStatus(35, 3, _("Running"), _("Installing boot loader..."));

#ifdef __i386__
	rc = runProgramRoot(RUN_LOG, "/mnt", "sbin/lilo", argv);
#elif __sparc__ 
        if (strcmp(hdname, dev))
            argv[1] = "-t";
	umount("/mnt/proc");
	rc = doMount("/proc", "/mnt/proc", "proc", 0, 0);
	if (rc) {
	    newtPopWindow();
	    return rc;
	}
	rc = runProgramRoot(RUN_LOG, "/mnt", "sbin/silo", argv);
	umount("/mnt/proc");
    #elif __powerpc__
        if (machine_type != ID_PReP )
            rc = doMount("/proc", "/mnt/proc", "proc", 0, 0);
            if (rc) {
                newtPopWindow();
                return rc;
            }
            rc = runProgramRoot(RUN_LOG, "/mnt", "sbin/quik", argv);
            argv[0] = "nvsetenv";
            argv[1] = "boot-file";
            argv[2] = NULL;
            runProgramIORoot(RUN_NOLOG, "/mnt", "sbin/nvsetenv", argv, "", &temp);
            if (strlen(temp) == 1) /* empty string w/ \n */
            {
                argv[0] = "nvsetenv";
                argv[1] = "boot-file";
                argv[2] = malloc(128);
                sprintf(argv[2], " root=/dev/%s noinitrd", rootdev);
                runProgramRoot(RUN_NOLOG, "/mnt", "sbin/nvsetenv", argv);
            }
            umount("/mnt/proc");
            editBootDevice();

#else
#error unsupported architectures
#endif

    newtPopWindow();

    if (rc)
	return INST_ERROR;

    return 0;
}

#ifndef __powerpc__
static void formatEntry(char * buf, struct partition * part) {
    sprintf(buf, "/dev/%-5s  %-25s %-7s %-10s", part->device, 
	    part->tagName, 
	    part->defaultBoot ? "   *" : "",
	    part->bootLabel ? part->bootLabel : "");
}
#endif /* !powerpc */


static int getBootLabels(struct partitionTable table, struct fstab fstab) {

#ifndef __powerpc__

    newtComponent f, okay, text, listbox, label, cancel, edit;
    struct newtExitStruct answer;
    char buf[80];
    int i, j;
    int foundDos = 0;
    int mustAsk = 0;
    int * map;
    struct partition * curr;
    int * currNum;
    int count;
    int done;
    int defaultBootPart = 0;
    char * reflowedText;
    int width, height;
    newtGrid buttons, grid;

    reflowedText = newtReflowText(
		       _("The boot manager Mandrake uses can boot other " 
                         "operating systems as well. You need to tell me " 
                         "what partitions you would like to be able to boot " 
                         "and what label you want to use for each of them."),
			 60, -5, -5, &width, &height);
    text = newtTextbox(-1, -1, width, height, NEWT_TEXTBOX_WRAP);
    newtTextboxSetText(text, reflowedText);
    free(reflowedText);

    f = newtForm(NULL, NULL, 0);

    sprintf(buf, "%-10s  %-25s %-7s %-10s", _("Device"), _("Partition type"), 
			_("Default"), _("Boot label"));
    label = newtLabel(-1, -1, buf);

    listbox = newtListbox(-1, -1, 10 - height, 
			  NEWT_LISTBOX_RETURNEXIT | NEWT_FLAG_SCROLL);
    map = alloca(sizeof(int) * table.count);
    
    for (i = 0, count = 0; i < table.count; i++) {
	if (table.parts[i].type != PART_SWAP && 
	    table.parts[i].type != PART_IGNORE &&
#ifdef __sparc__
	    table.parts[i].type != PART_OTHER &&
#endif
	    (table.parts[i].type != PART_FAT32 || !foundDos) &&
	    (table.parts[i].type != PART_DOS || !foundDos)) {

	    if (table.parts[i].type == PART_DOS || 
	        table.parts[i].type == PART_FAT32) {
		foundDos = 1;
	    }

	    if (table.parts[i].type == PART_EXT2) {
		for (j = 0; j < fstab.numEntries; j++) {
		    if (!strcmp(table.parts[i].device, fstab.entries[j].device))
			break;
		}

		if (j < fstab.numEntries && !table.parts[i].bootLabel) 
		    continue;
	    }

	    if (!table.parts[i].bootLabel ||
		strcmp(table.parts[i].bootLabel, "linux")) mustAsk = 1;

	    if (table.parts[i].defaultBoot)
		defaultBootPart = count;

	    map[count] = i;
	    formatEntry(buf, table.parts + i);
	    newtListboxAddEntry(listbox, buf, map + count++);
	} 
    }

    /* we add stuff to the form here so we can actually destroy it if
       we don't need this window after all */
    newtFormAddComponents(f, text, label, listbox, NULL);

    if (!mustAsk) {
	newtFormDestroy(f);
	return 0;
    }

    newtPushHelpLine("<F2> Selects the default partition");

    buttons = newtButtonBar(_("Ok"), &okay, _("Edit"), &edit,
			    _("Back"), &cancel, NULL);

    newtFormAddComponents(f, okay, edit, cancel, NULL);

    newtFormAddHotKey(f, NEWT_KEY_F2);

    grid = newtCreateGrid(1, 4);

    newtGridSetField(grid, 0, 0, NEWT_GRID_COMPONENT, text,
		     0, 0, 0, 1, NEWT_ANCHOR_LEFT, 0);
    newtGridSetField(grid, 0, 1, NEWT_GRID_COMPONENT, label,
		     0, 0, 0, 0, NEWT_ANCHOR_LEFT, 0);
    newtGridSetField(grid, 0, 2, NEWT_GRID_COMPONENT, listbox,
		     0, 0, 0, 0, NEWT_ANCHOR_LEFT, 0);
    newtGridSetField(grid, 0, 3,  NEWT_GRID_SUBGRID, buttons,
		     0, 1, 0, 0, 0, NEWT_GRID_FLAG_GROWX);

    newtGridWrappedWindow(grid, _("Bootable Partitions"));
    newtGridFree(grid, 1);

    done = 0;
    while (!done) {
	newtFormRun(f, &answer);

	if (answer.reason == NEWT_EXIT_HOTKEY) {
	    if (answer.u.key == NEWT_KEY_F12) {
		done = 1;
	    } else if (answer.u.key == NEWT_KEY_F2) {
		currNum = newtListboxGetCurrent(listbox);
		curr = table.parts + *currNum;

		if (!curr->bootLabel) {
		    newtWinMessage("Boot Partition", "Ok", "You cannot mark a "
				  "partition as the default partition to "
				  "boot from unless that partition has "
				  "been assigned a boot label.");
		} else{
		    for (i = 0; i < count; i++) {
			if (table.parts[map[i]].defaultBoot) {
			    table.parts[map[i]].defaultBoot = 0;
			    formatEntry(buf, table.parts + map[i]);
			    newtListboxSetEntry(listbox, i, buf);
			    break;
			}
		    }

		    curr->defaultBoot = 1;
		    formatEntry(buf, curr);
		    newtListboxSetEntry(listbox, currNum - map, buf);
		}
	    }
	} else {
	    if (answer.u.co == edit || answer.u.co== listbox) {
		currNum = newtListboxGetCurrent(listbox);
		curr = table.parts + *currNum;
		editBootLabel(curr);

		if (!curr->bootLabel && curr->defaultBoot) {
		    curr->defaultBoot = 0;
		    if (table.parts[map[defaultBootPart]].bootLabel) {
			table.parts[map[defaultBootPart]].defaultBoot = 1;

			formatEntry(buf, table.parts + map[defaultBootPart]);
			newtListboxSetEntry(listbox, defaultBootPart, buf);
		    }
		}

		formatEntry(buf, curr);
		newtListboxSetEntry(listbox, currNum - map, buf);
	    } else 
		done = 1;
	}
    }

    newtPopHelpLine();

    newtFormDestroy(f);
    newtPopWindow();

    if (answer.reason == NEWT_EXIT_COMPONENT && answer.u.co == cancel)
	return INST_CANCEL;
    else
#endif /* !powerpc */
	return 0;
}

static int getAppendLine(char ** line, int * linear) {
    newtComponent form, text, entry, okay, cancel, answer;
    newtComponent linearCheck;
    char * result = NULL;
    char linearChar = (*linear) ? '*' : ' ';
    newtGrid grid, buttons, subgrid;

    text = newtTextboxReflowed(0, 0,
		     _("A few systems will need to pass special options "
		       "to the kernel at boot time for the system to function "
		       "properly. If you need to pass boot options to the "
		       "kernel, enter them now. If you don't need any or "
		       "aren't sure, leave this blank."), 53, 10, 0, 0);

    form = newtForm(NULL, NULL, 0);

    entry = newtEntry(-1, -1, *line, 48, &result, 
			NEWT_ENTRY_SCROLL | NEWT_ENTRY_RETURNEXIT);

#if !defined( __sparc__ ) && !defined (__powerpc__)
	linearCheck = newtCheckbox(-1, -1, 
			     _("Use linear mode (needed for some SCSI drives)"),
			     linearChar, NULL, &linearChar);
#endif

    buttons = newtButtonBar(_("Ok"), &okay, _("Back"), &cancel, NULL);
    subgrid = newtGridVStacked(NEWT_GRID_COMPONENT, entry,
#if !defined( __sparc__ ) && !defined (__powerpc__)
				  NEWT_GRID_COMPONENT, linearCheck,
#endif
			       NULL);
    grid = newtGridBasicWindow(text, subgrid, buttons);

    newtGridAddComponentsToForm(grid, form, 1);
    newtFormSetCurrent(form, okay);

#ifdef __sparc__
	newtGridWrappedWindow(grid, _("Silo Installation"));
#elif defined(__powerpc__)
        switch (machine_type) {
            case ID_MkLinux:
               newtGridWrappedWindow(grid, _("MkLinux Installation"));
               break;
            case ID_PMac:
               newtGridWrappedWindow(grid, _("Quik Installation"));
               break;
            case ID_PReP:
               newtGridWrappedWindow(grid, _("PReP Installation"));
               break;
            case ID_CHRP:
               newtGridWrappedWindow(grid, _("CHRP Installation"));
               break;
            case ID_Unknown:
            default:
               newtGridWrappedWindow(grid, _("LinuxPPC Installation"));
        }
#else
	newtGridWrappedWindow(grid, _("Lilo Installation"));
#endif

    newtGridFree(grid, 1);

    answer = newtRunForm(form);

    newtPopWindow();

    if (answer == cancel) {
	newtFormDestroy(form);
	return INST_CANCEL;
    }

    *linear = linearChar != ' ';

    if (!strlen(result)) 
	*line = NULL;
    else 
	*line = strdup(result);

    newtFormDestroy(form);

    return 0;
}

#define LILO_WHERE 2
#define LILO_LABELS 3
#define LILO_INSTALL 4
#define LILO_APPEND 5
#define LILO_DONE 20
    
int installLilo(char * prefix, struct partitionTable table, struct fstab fstab,
		char * kernelVersion, int flags, char * options) {
    char * rootDevice, * bootDevice = NULL;
    char * hdName;
    char * where = NULL;
    char * append = NULL;
    char * chptr = NULL;
    int i;
    int rc;
    int stage = LILO_WHERE;
    static int linear = 0;
    int foundDos = 0;

    hdName = default_boot_hd;

    for (i = 0; i < fstab.numEntries; i++) {
	if (!strcmp(fstab.entries[i].mntpoint, "/boot")) break;
    }

    if (i < fstab.numEntries) 
	bootDevice = fstab.entries[i].device;

    for (i = 0; i < fstab.numEntries; i++) {
	if (!strcmp(fstab.entries[i].mntpoint, "/")) break;
    }

    rootDevice = fstab.entries[i].device;
    if (!bootDevice) {
	bootDevice = rootDevice;
    }

    for (i = 0; i < table.count; i++) {
	if (!strcmp(table.parts[i].device, bootDevice)) {
	    table.parts[i].bootLabel = strdup("linux");
	    table.parts[i].defaultBoot = 1;
	} else if ((table.parts[i].type == PART_DOS || 
	    table.parts[i].type == PART_FAT32) && !foundDos) {
	    table.parts[i].bootLabel = strdup("dos");
	    foundDos = 1;
	}
    }

    if (flags & LILO_ON_MBR)
	where = hdName;
    else if (flags & LILO_ON_PARTITION)
	where = bootDevice;

    if (flags & LILO_USE_OPTIONS)
	append = options;

    if (where) {
	rc = doinstallLilo(prefix, where, rootDevice, bootDevice, table, 
			    append, kernelVersion, hdName, linear);
	if (rc == INST_ERROR) return INST_ERROR;
	stage = LILO_DONE;
    }

    while (stage != LILO_DONE) {
	switch (stage) {
	  case LILO_WHERE:
            #ifndef __powerpc__
            rc = liloWhere(hdName, bootDevice, &where);
            #else
            rc = liloWhere(hdName, bootDevice, &where, table);
            #endif
            if (rc == SKIP_LILO ) return 0;
	    if (rc) return rc;
           #ifdef __powerpc__
            if (machine_type == ID_PReP)
               stage = LILO_INSTALL;
            else
               stage = LILO_APPEND;

           #else
            stage = LILO_APPEND;
           #endif
	    break;

	  case LILO_APPEND:
	    chptr = append;
	    rc = getAppendLine(&chptr, &linear);

	    if (rc == INST_ERROR) return INST_ERROR;
	    if (rc == INST_CANCEL)
		stage = LILO_WHERE;
	    else {
		stage = LILO_LABELS;

		if (append) free(append);
		if (chptr) {
		    append = alloca(strlen(chptr) + 1);
		    strcpy(append, chptr);
		    free(chptr);
		} else {
		    append = NULL;
		}
	    }

	    break;

	  case LILO_LABELS:
	    rc = getBootLabels(table, fstab);
	    if (rc == INST_ERROR) return INST_ERROR;
	    if (rc == INST_CANCEL)
		stage = LILO_APPEND;
	    else
		stage = LILO_INSTALL;
	    break;

	  case LILO_INSTALL:
	    rc = doinstallLilo(prefix, where, rootDevice, bootDevice,
				table, append, 
				kernelVersion, hdName, linear);
	    if (rc == INST_ERROR) return INST_ERROR;
	    stage = LILO_DONE;
	    break;
	}
    }

    return 0;
}

int makeBootdisk(char * prefix, char * kernelVersion) {
    char * argv[] = { "/sbin/mkbootdisk", "--noprompt", "--device", "/dev/fd0", kernelVersion, NULL };
    int rc;

#ifdef __sparc__
    if (loadModule("romfs", DRIVER_OTHER, DRIVER_MINOR_NONE, NULL)) 
        return INST_ERROR;
    /* Find out if the only floppy in the system is /dev/fd1 */
    rc = open("/mnt/dev/fd0", O_RDONLY);
    if (rc < 0 && errno == ENXIO)
        argv[3] = "/dev/fd1";
    else if (rc >= 0)
    	close(rc);
#endif
    if (loadModule("loop", DRIVER_OTHER, DRIVER_MINOR_NONE, NULL)) 
	return INST_ERROR;
	
    winStatus(32, 3, _("Bootdisk"), _("Creating bootdisk..."));

#ifdef __sparc__	
    umount("/mnt/proc");
    rc = doMount("/proc", "/mnt/proc", "proc", 0, 0);
    if (rc) {
	newtPopWindow();
	removeModule("romfs");
	removeModule("loop");
	return rc;
    }
#endif
    
    rc = runProgramRoot(RUN_LOG, "/mnt", "/sbin/mkbootdisk", argv);
    newtPopWindow();

#ifdef __sparc__
    umount("/mnt/proc");
#endif
    removeModule("loop");

    return rc;
}

int editBootDevice(void)
{
    newtComponent form, text, dev_entry, file_entry, okay, cancel, answer;
    char *boot_device, *boot_file, *orig_dev, *orig_file;
    char *argv[] = { NULL, NULL, NULL , NULL};
    int rc = 0;
    
    if ((rc = doMount("/proc", "/mnt/proc", "proc", 0, 0))) {
        newtPopWindow();
        return rc;
    }
    
    argv[0]="nvsetenv";  
    argv[1]="boot-device";
    runProgramIORoot(RUN_NOLOG, "/mnt", "sbin/nvsetenv",  argv, "", &orig_dev);
    orig_dev[strlen(orig_dev)-1]='\0';
        
    argv[1]="boot-file";
    runProgramIORoot(RUN_NOLOG, "/mnt", "sbin/nvsetenv",  argv, "", &orig_file);
    orig_file[strlen(orig_file)-1]='\0';
    
    newtOpenWindow(12, 5, 55, 15, "Change Boot Variables");
    form = newtForm(NULL, NULL, 0);
    text = newtTextbox(1, 1, 53, 5, NEWT_TEXTBOX_WRAP);
    newtTextboxSetText(text,
                     "The current boot settings for Open Firmware "
                     "are shown below.  You may like to change them "
                     "in order to boot from your newly-installed disk "
                     "(or set boot-device to /AAPL,ROM to boot MacOS).");
    
    newtFormAddComponent(form, newtLabel(2, 7, "boot-device :"));
    newtFormAddComponent(form, newtLabel(2, 9, "boot-file   :"));
    
    dev_entry = newtEntry(17,7, orig_dev, 30, &boot_device,
                        NEWT_ENTRY_SCROLL | NEWT_ENTRY_RETURNEXIT);
    file_entry = newtEntry(17, 9, orig_file, 30, &boot_file,
                         NEWT_ENTRY_SCROLL | NEWT_ENTRY_RETURNEXIT); 
    okay = newtButton(12, 11, "Ok");
    cancel = newtButton(35, 11, "Cancel");
    
    newtFormAddComponents(form, text, dev_entry, file_entry,
                        okay, cancel, NULL);
    
    answer = newtRunForm(form);
    if (answer != cancel) {
        argv[0] = "nvsetenv";
        argv[1] = "boot-device";
        argv[2] = boot_device;
        runProgramRoot(RUN_NOLOG, "/mnt", "sbin/nvsetenv",  argv);
        argv[1] = "boot-file";
        argv[2] = boot_file;
        runProgramRoot(RUN_NOLOG, "/mnt", "sbin/nvsetenv",  argv);
    }
                     
    logMessage("boot-device: '%s'",boot_device);
    logMessage("boot-file: '%s'",boot_file);
    
    newtPopWindow();
                        
    umount("/mnt/proc");
    return answer == cancel? INST_CANCEL: 0;
}
        

