/*
pwavecat  Jan Panteltje's Sound Editor copyright 2004-always Jan Panteltje.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "pwavecat.h"

//struct
//	{						/* header for WAV-Files */
//	char main_chunk[4];		/* 'RIFF' */
//	DWORD length;			/* length of file */
//	char chunk_type[4];		/* 'WAVE' */
//	char sub_chunk[4];		/* 'fmt ' */
//	DWORD length_chunk;		/* length sub_chunk, always 16 bytes */
//	WORD format;			/* always 1 = PCM-Code */

//	WORD modus;				/* 1 = Mono, 2 = Stereo */
//	DWORD sample_fq;		/* Sample Freq */
//	DWORD byte_p_sec;		/* Data per sec */
//	WORD byte_p_spl;		/* bytes per sample, 1=8 bit, 2=16 bit (mono)
//						     2=8 bit, 4=16 bit (stereo) */
//	WORD bit_p_spl;			/* bits per sample, 8, 12, 16 */
//	char data_chunk[4];		/* 'data' */
//	DWORD data_length;		/* length of data */
//	} wave_header;

int modus;
int sample_fq;
int byte_p_sec;
int byte_p_spl;
int bit_p_spl;
int have_format_flag;		// set if first wave header is read, now file format is set.



char *strsave(char *s) /* save string s somewhere */
{
char *p;

p = malloc(strlen(s) + 1);
if(!p) return 0;
strcpy(p, s);
return p;
} /* end function strsave */


int print_usage()
{
fprintf(stderr, "\
\npwavecat-%s Copyright 2004 Jan Panteltje.\n\
Usage:               pwavecat [-d] [-h] [-v] file1.wav file2.wav file3.wav ..... filen.wav > outfile.wav\n\n\
-d                   debug (prints functions and arguments).\n\
-h                   help (this help).\n\
-v                   display progress, ID and version.\n\
", VERSION);

return 1;
} /* end function print_usage */


int main(int argc, char **argv)
{
int i;
int input_files;

/* defaults */
debug_flag = 0;
verbose_flag = 0;
/* end defaults */

/* get command line arguments */
for(i = 1; i < argc; i++)
	{
	if(strcmp(argv[i], "-d") == 0)
		{
		debug_flag = 1;
		}
	else if(strcmp(argv[i], "-h") == 0)
		{
		print_usage();

		exit(1);
		}
	else if(strcmp(argv[i], "-v") == 0)
		{
		verbose_flag = 1;
		}
	} 

if(verbose_flag)
	{
	fprintf(stderr, "Panteltje pwavecat-%s\n", VERSION);
	}

/* get filenames, skip command line arguments */
input_files = 0;
for(i = 1; i < argc; i++)
	{
	if(strcmp(argv[i], "-d") == 0) continue;
	if(strcmp(argv[i], "-h") == 0) continue;
	if(strcmp(argv[i], "-v") == 0) continue;

	if(! insert_file(argv[i], input_files) )
		{
		fprintf(stderr, "pwavecat: aborting.\n");

		exit(1);
		}

	input_files++;

	if(input_files > MAX_FILES)
		{
		fprintf(stderr, "pwavecat: max %d input files exceeded, aborting.\n", MAX_FILES);

		exit(1);
		}

	} /* end while all arguments */

if(input_files == 0)
	{
	fprintf(stderr, "pwavecat: no input file(s) specified, aborting.\n");

	print_usage();

	exit(1);
	}

if(input_files == 1)
	{
	fprintf(stderr, "pwavecat: only 1 input file specified, aborting.\n");

	print_usage();

	exit(1);
	}

if(! write_all("/dev/stdout") )
	{
	fprintf(stderr, "pwavecat: aborting.\n");

	exit(1);
	}

if(verbose_flag)
	{
	fprintf(stderr, "Ready\n");
	}

/* exit OK */
exit(0);
} /* end function main */



int insert_file(char *filename, size_t position)
{
int a;
FILE *fptr;
int header_size;
wave_header *file_header;
int error_flag;
struct part *pa;
char temp[10];
int wave_file_flag;

if(debug_flag)
	{
	fprintf(stderr, "insert_file(): arg filename=%s position=%lu\n", filename, (unsigned long)position); 
	}

if(! filename) return 0;

fptr = fopen(filename, "r");
if(! fptr)
	{
	fprintf(stderr, "insert_file(): ERROR could not open file %s for read\n", filename);

	return 0;
	}

/* read header */
/* get size of wave header */
header_size = sizeof(wave_header);

if(debug_flag)
	{
	fprintf(stderr, "insert_file(): wave header_size=%d\n", header_size);
	}

/* allocate space for wave header */
file_header = malloc( sizeof(wave_header) );
if(! file_header)
	{
	fprintf(stderr, "pwavecat: write_file(): could not allocate space for file_header.\n");

	fclose(fptr);
	return 0;
	}

a = fread(file_header, sizeof(char), header_size, fptr);
if(a < header_size)
	{   
	fprintf(stderr, "insert_file(): ERROR could not read wave header for file %s\n", filename);
    
	fclose(fptr);
	free(file_header);
	return 0;
	}

/* get data length */
if(debug_flag)
	{
	fprintf(stderr, "insert_file(): data_length=%lu\n", (unsigned long) file_header -> data_length);
	}

if(file_header -> data_length == 0)
	{
	fprintf(stderr, "pwavecat: ERROR file %s has zero length\n", filename);

	fclose(fptr);
	free(file_header);
	return 0;
	}

/* test if a wave file */
wave_file_flag = 1;
//	char main_chunk[4];		/* 'RIFF' */
if(strncmp(file_header -> main_chunk, "RIFF", 4) != 0)
	{
	strncpy(temp, file_header -> main_chunk, 4);
	temp[4] = 0;

	fprintf(stderr,\
	"pwavecat: ERROR file %s is not a wave file, header field 'RIFF' reads: '%s'.\n", filename, temp);

	wave_file_flag = 0;
	}

//	char chunk_type[4];		/* 'WAVE' */
if(strncmp(file_header -> chunk_type, "WAVE", 4) != 0)
	{
	strncpy(temp, file_header -> chunk_type, 4);
	temp[4] = 0;
	
	fprintf(stderr,\
	"pwavecat: ERROR file %s is not a wave file, header field 'WAVE' reads: '%s'.\n", filename, temp);

	wave_file_flag = 0;
	}

//	char sub_chunk[4];		/* 'fmt ' */
if(strncmp(file_header -> sub_chunk, "fmt ", 4) != 0)
	{
	strncpy(temp, file_header -> sub_chunk, 4);
	temp[4] = 0;

	fprintf(stderr,\
	"pwavecat: WARNING file %s header field 'fmt ' reads: '%s'.\n", filename, temp);
	}

//	char data_chunk[4];		/* 'data' */
if(strncmp(file_header -> data_chunk, "data", 4) != 0)
	{
	strncpy(temp, file_header -> data_chunk, 4);
	temp[4] = 0;

	fprintf(stderr,\
	"pwavecat: WARNING file %s header field 'data' reads: '%s'.\n", filename, temp);
	}

if(! wave_file_flag)
	{
	fclose(fptr);
	free(file_header);
	return 0;
	}

/* if this is the first insert, then it sets the format, else check if next insert is same format */
if(! have_format_flag)
	{
	/* set format */
	modus = file_header -> modus;
	sample_fq = file_header -> sample_fq;
	byte_p_sec = file_header -> byte_p_sec;
	byte_p_spl = file_header -> byte_p_spl;
	bit_p_spl = file_header -> bit_p_spl;
	have_format_flag = 1;
	}
else
	{
	error_flag = 0;
	/* check format */
	if(file_header -> modus != modus) error_flag = 1; 
	if(file_header -> sample_fq !=sample_fq) error_flag = 1;
	if(file_header -> byte_p_sec != byte_p_sec) error_flag = 1;
	if(file_header -> byte_p_spl != byte_p_spl) error_flag = 1;
	if(file_header -> bit_p_spl != bit_p_spl) error_flag = 1;

	if(error_flag)
		{
		fprintf(stderr, "insert_file(): ERROR file %s is in a different format\n", filename);

		fprintf(stderr, "Current format is:\n");
		fprintf(stderr, "modus=%d\n", modus);
		fprintf(stderr, "sample_fq=%d\n", sample_fq);
		fprintf(stderr, "byte_p_sec=%d\n", byte_p_sec);
		fprintf(stderr, "byte_p_spl=%d\n", byte_p_spl);
		fprintf(stderr, "bit_p_spl=%d\n", bit_p_spl);
		fprintf(stderr, "\n");

		fprintf(stderr, "This file has format:\n");
		fprintf(stderr, "modus=%d\n", file_header -> modus);
		fprintf(stderr, "sample_fq=%lu\n", (unsigned long)file_header -> sample_fq);
		fprintf(stderr, "byte_p_sec=%lu\n", (unsigned long)file_header -> byte_p_sec);
		fprintf(stderr, "byte_p_spl=%d\n", file_header -> byte_p_spl);
		fprintf(stderr, "bit_p_spl=%d\n", file_header -> bit_p_spl);

		fclose(fptr);
		free(file_header);
		return 0;
		}
	}

fclose(fptr);

/* data to structure */
pa = install_part_at_end_of_list(filename);
if(! pa)
	{
	fprintf(stderr, "insert_file(): ERROR could not install file in list\n");
	
	return 0;
	}

pa -> length = file_header -> data_length;

if(debug_flag)
	{
	fprintf(stderr, "insert_file(): pa->length=%lu\n", (long unsigned) pa -> length);
	}

free(file_header);

return 1;
} /* end function insert_file */


