#!/usr/bin/env bash
# SPDX-License-Identifier: LGPL-2.1-or-later
# shellcheck disable=SC2016
set -eux
set -o pipefail

# Verify our own unit files (where applicable)

# This is generated by meson during build
UNIT_FILE_LIST="/usr/lib/systemd/tests/testdata/installed-unit-files.txt"

if [[ ! -f "$UNIT_FILE_LIST" ]]; then
    echo "Couldn't find the list of installed units, skipping the test"
    exit 0
fi

if ! command -v systemd-analyze >/dev/null; then
    echo "Built without systemd-analyze, skipping the test"
    exit 0
fi

mapfile -t UNIT_FILES <"$UNIT_FILE_LIST"

if [[ "${#UNIT_FILES[@]}" -le 0 ]]; then
    echo >&2 "The unit file list is empty, this is most likely a bug"
    exit 1
fi

for unit_file in "${UNIT_FILES[@]}"; do
    # Skip the check for a couple of units, namely:
    #   - syslog.socket: the corresponding syslog.service might not be installed
    #   - rc-local.service: compat API, /etc/rc.d/rc.local most likely won't be present
    if [[ "$unit_file" =~ /(syslog.socket|rc-local.service)$ ]]; then
        continue
    fi

    # Skip missing unit files - this is useful for $NO_BUILD runs, where certain unit files might be dropped
    # in distro packaging
    if [[ ! -e "$unit_file" ]]; then
        echo "$unit_file not found, skipping"
        continue
    fi

    systemd-analyze --recursive-errors=no --man=no verify "$unit_file"
done
