
#include "recman.h"

#include "tools.h"

// STL headers need to be before VDR tools.h (included by <vdr/videodir.h>)
#include <fstream>
#include <stack>
#include <algorithm>

#include <vdr/videodir.h>

#define INDEXFILESUFFIX   "/index.vdr"
#define LENGTHFILESUFFIX  "/length.vdr"

using std::string;
using std::vector;
using std::pair;

namespace vdrlive {

	/**
	 *  Implementation of class RecordingsManager:
	 */
	stdext::weak_ptr< RecordingsManager > RecordingsManager::m_recMan;
	stdext::shared_ptr< RecordingsTree > RecordingsManager::m_recTree;
	stdext::shared_ptr< RecordingsList > RecordingsManager::m_recList;
	stdext::shared_ptr< DirectoryList > RecordingsManager::m_recDirs;
#if VDRVERSNUM >= 20301
	cStateKey RecordingsManager::m_recordingsStateKey;
#else
	int RecordingsManager::m_recordingsState = 0;
#endif

	// The RecordingsManager holds a VDR lock on the
	// Recordings. Additionally the singleton instance of
	// RecordingsManager is held in a weak pointer. If it is not in
	// use any longer, it will be freed automaticaly, which leads to a
	// release of the VDR recordings lock. Upon requesting access to
	// the RecordingsManager via LiveRecordingsManger function, first
	// the weak ptr is locked (obtaining a stdext::shared_ptr from an possible
	// existing instance) and if not successfull a new instance is
	// created, which again locks the VDR Recordings.
	//
	// RecordingsManager provides factory methods to obtain other
	// recordings data structures. The data structures returned keep if
	// needed the instance of RecordingsManager alive until destructed
	// themselfs. This way the use of LIVE::recordings is straight
	// forward and does hide the locking needs from the user.

#if VDRVERSNUM >= 20301
	RecordingsManager::RecordingsManager()
#else
	RecordingsManager::RecordingsManager() :
		m_recordingsLock(&Recordings)
#endif
	{
	}

	RecordingsTreePtr RecordingsManager::GetRecordingsTree() const
	{
		RecordingsManagerPtr recMan = EnsureValidData();
		if (! recMan) {
			return RecordingsTreePtr(recMan, stdext::shared_ptr< RecordingsTree >());
		}
		return RecordingsTreePtr(recMan, m_recTree);
	}

	RecordingsListPtr RecordingsManager::GetRecordingsList(bool ascending) const
	{
		RecordingsManagerPtr recMan = EnsureValidData();
		if (! recMan) {
			return RecordingsListPtr(recMan, stdext::shared_ptr< RecordingsList >());
		}
		return RecordingsListPtr(recMan, stdext::shared_ptr< RecordingsList >(new RecordingsList(m_recList, ascending)));
	}

	RecordingsListPtr RecordingsManager::GetRecordingsList(time_t begin, time_t end, bool ascending) const
	{
		RecordingsManagerPtr recMan = EnsureValidData();
		if (! recMan) {
			return RecordingsListPtr(recMan, stdext::shared_ptr< RecordingsList >());
		}
		return RecordingsListPtr(recMan, stdext::shared_ptr< RecordingsList >(new RecordingsList(m_recList, ascending)));
	}

	DirectoryListPtr RecordingsManager::GetDirectoryList() const
	{
		RecordingsManagerPtr recMan = EnsureValidData();
		if (!recMan) {
			return DirectoryListPtr(recMan, stdext::shared_ptr< DirectoryList >());
		}
		return DirectoryListPtr(recMan, m_recDirs);
	}

	string RecordingsManager::Md5Hash(cRecording const * recording) const
	{
		return "recording_" + MD5Hash(recording->FileName());
	}

	cRecording const * RecordingsManager::GetByMd5Hash(string const & hash) const
	{
		if (!hash.empty()) {
#if VDRVERSNUM >= 20301
			LOCK_RECORDINGS_READ;
			for (cRecording* rec = (cRecording *)Recordings->First(); rec; rec = (cRecording *)Recordings->Next(rec)) {
#else
			for (cRecording* rec = Recordings.First(); rec; rec = Recordings.Next(rec)) {
#endif
				if (hash == Md5Hash(rec))
					return rec;
			}
		}
		return 0;
	}

	bool RecordingsManager::MoveRecording(cRecording const * recording, string const & name, bool copy) const
	{
		if (!recording)
			return false;

		string oldname = recording->FileName();
		size_t found = oldname.find_last_of("/");

		if (found == string::npos)
			return false;

#if APIVERSNUM > 20101
		string newname = string(cVideoDirectory::Name()) + "/" + name + oldname.substr(found);
#else
		string newname = string(VideoDirectory) + "/" + name + oldname.substr(found);
#endif

		if (!MoveDirectory(oldname.c_str(), newname.c_str(), copy)) {
			esyslog("live: renaming failed from '%s' to '%s'", oldname.c_str(), newname.c_str());
			return false;
		}

#if VDRVERSNUM >= 20301
		LOCK_RECORDINGS_WRITE;
		if (!copy)
			Recordings->DelByName(oldname.c_str());
		Recordings->AddByName(newname.c_str());
#else
		if (!copy)
			Recordings.DelByName(oldname.c_str());
		Recordings.AddByName(newname.c_str());
#endif
		cRecordingUserCommand::InvokeCommand(*cString::sprintf("rename \"%s\"", *strescape(oldname.c_str(), "\\\"$'")), newname.c_str());

		return true;
	}

	void RecordingsManager::DeleteResume(cRecording const * recording) const
	{
		if (!recording)
			return;

		cResumeFile ResumeFile(recording->FileName(), recording->IsPesRecording());
		ResumeFile.Delete();
	}

	void RecordingsManager::DeleteMarks(cRecording const * recording) const
	{
		if (!recording)
			return;

		cMarks marks;
		marks.Load(recording->FileName());
		if (marks.Count()) {
			cMark *mark = marks.First();
			while (mark) {
				cMark *nextmark = marks.Next(mark);
				marks.Del(mark);
				mark = nextmark;
			}
			marks.Save();
		}
	}

	void RecordingsManager::DeleteRecording(cRecording const * recording) const
	{
		if (!recording)
			return;

		string name(recording->FileName());
		const_cast<cRecording *>(recording)->Delete();
#if VDRVERSNUM >= 20301
		LOCK_RECORDINGS_WRITE;
		Recordings->DelByName(name.c_str());
#else
		Recordings.DelByName(name.c_str());
#endif
	}

	int RecordingsManager::GetArchiveType(cRecording const * recording)
	{
		string filename = recording->FileName();

		string dvdFile = filename + "/dvd.vdr";
		if (0 == access(dvdFile.c_str(), R_OK)) {
			return 1;
		}
		string hddFile = filename + "/hdd.vdr";
		if (0 == access(hddFile.c_str(), R_OK)) {
			return 2;
		}
		return 0;
	}

	string const RecordingsManager::GetArchiveId(cRecording const * recording, int archiveType)
	{
		string filename = recording->FileName();

		if (archiveType==1) {
			string dvdFile = filename + "/dvd.vdr";
			std::ifstream dvd(dvdFile.c_str());

		if (dvd) {
			string archiveDisc;
			string videoDisc;
			dvd >> archiveDisc;
			if ("0000" == archiveDisc) {
				dvd >> videoDisc;
				return videoDisc;
			}
			return archiveDisc;
		}
		} else if(archiveType==2) {
			string hddFile = filename + "/hdd.vdr";
			std::ifstream hdd(hddFile.c_str());

			if (hdd) {
				string archiveDisc;
				hdd >> archiveDisc;
				return archiveDisc;
			}
		}
		return "";
	}

	string const RecordingsManager::GetArchiveDescr(cRecording const * recording)
	{
		int archiveType;
		string archived;
		archiveType = GetArchiveType(recording);
		if (archiveType==1) {
			archived += " [";
			archived += tr("On archive DVD No.");
			archived += ": ";
			archived += GetArchiveId(recording, archiveType);
			archived += "]";
		} else if (archiveType==2) {
			archived += " [";
			archived += tr("On archive HDD No.");
			archived += ": ";
			archived += GetArchiveId(recording, archiveType);
			archived += "]";
		}
		return archived;
	}

#if VDRVERSNUM >= 20301
	bool RecordingsManager::StateChanged ()
	{
		bool result = false;

		// will return != 0 only, if the Recordings List has been changed since last read
		if (cRecordings::GetRecordingsRead(m_recordingsStateKey)) {
			result = true;
			m_recordingsStateKey.Remove();
		}

		return result;
	}
#endif

	RecordingsManagerPtr RecordingsManager::EnsureValidData()
	{
		// Get singleton instance of RecordingsManager.  'this' is not
		// an instance of stdext::shared_ptr of the singleton
		// RecordingsManager, so we obtain it in the overall
		// recommended way.
		RecordingsManagerPtr recMan = LiveRecordingsManager();
		if (! recMan) {
			// theoretically this code is never reached ...
			esyslog("live: lost RecordingsManager instance while using it!");
			return RecordingsManagerPtr();
		}

		// StateChanged must be executed every time, so not part of
		// the short cut evaluation in the if statement below.
#if VDRVERSNUM >= 20301
		bool stateChanged = StateChanged();
#else
		bool stateChanged = Recordings.StateChanged(m_recordingsState);
#endif
		if (stateChanged || (!m_recTree) || (!m_recList) || (!m_recDirs)) {
			if (stateChanged) {
				m_recTree.reset();
				m_recList.reset();
				m_recDirs.reset();
			}
			if (stateChanged || !m_recTree) {
				m_recTree = stdext::shared_ptr< RecordingsTree >(new RecordingsTree(recMan));
			}
			if (!m_recTree) {
				esyslog("live: creation of recordings tree failed!");
				return RecordingsManagerPtr();
			}
			if (stateChanged || !m_recList) {
				m_recList = stdext::shared_ptr< RecordingsList >(new RecordingsList(RecordingsTreePtr(recMan, m_recTree)));
			}
			if (!m_recList) {
				esyslog("live: creation of recordings list failed!");
				return RecordingsManagerPtr();
			}
			if (stateChanged || !m_recDirs) {
				m_recDirs = stdext::shared_ptr< DirectoryList >(new DirectoryList(recMan));
			}
			if (!m_recDirs) {
				esyslog("live: creation of directory list failed!");
				return RecordingsManagerPtr();
			}

		}
		return recMan;
	}


        ShortTextDescription::ShortTextDescription(const char * ShortText, const char * Description) :
            m_short_text(ShortText),
            m_description(Description)
           {  }
        char ShortTextDescription::getNextCharShortText() {
          if (!m_short_text ||  !m_short_text[m_counter] ) {
             m_in_short_text = false;
             m_counter = 0;
             return 0;
          }
          return m_short_text[m_counter ++];
        }
        char ShortTextDescription::getNextCharDescription() {
          if (!m_description ||  !m_description[m_counter] ) return 0;
          return m_description[m_counter ++];
        }
        char ShortTextDescription::getNextChar() {
          char result;
          if( m_in_short_text) result = getNextCharShortText();
          if(!m_in_short_text) result = getNextCharDescription();
          return result;
        }
        char ShortTextDescription::getNextNonPunctChar() {
           char result;
           do {
              result = getNextChar();
           } while (result && ispunct(result));
           return tolower(result);
        }


        int RecordingsItemPtrCompare::Compare2(int &numEqualChars, const RecordingsItemPtr & first, const RecordingsItemPtr & second) {
          numEqualChars = 0;
          int i = first->NameForCompare().compare(second->NameForCompare() );
          if(i != 0) return i;
// compare short text & description
          ShortTextDescription chfirst (first ->ShortText() , first ->Description() );
          ShortTextDescription chsecond(second->ShortText() , second->Description() );
          char flc;
          char slc;
          do {
            flc = chfirst.getNextNonPunctChar();
            slc = chsecond.getNextNonPunctChar();
            if ( flc < slc ) return -1;
            if ( flc > slc ) return  1;
            ++numEqualChars;
          } while ( flc && slc );
          --numEqualChars;

          if (slc ) return -1;
          if (flc ) return  1;
          return 0;
        }

  int RecordingsItemPtrCompare::FindBestMatch(RecordingsItemPtr & BestMatch, const std::list<RecordingsItemPtr>::iterator & First, const std::list<RecordingsItemPtr>::iterator & Last, const RecordingsItemPtr & EPG_Entry){
// d: length of movie in minutes, without commercial breaks
// Assumption: the maximum length of commercial breaks is cb% * d
// Assumption: cb = 34%
   const long cb = 34;
// lengthLowerBound: minimum of d, if EPG_Entry->Duration() has commercial breaks
   long lengthLowerBound = EPG_Entry->Duration() * 100 / ( 100 + cb) ;
// lengthUpperBound: if EPG_Entry->Duration() is d (no commercial breaks), max length of recording with commercial breaks
// Add VDR recording margins to this value
   long lengthUpperBound = EPG_Entry->Duration() * (100 + cb) / 100;
   lengthUpperBound += Setup.MarginStart + Setup.MarginStop;
   if(EPG_Entry->Duration() >= 90 && lengthLowerBound < 70) lengthLowerBound = 70;

   int numRecordings = 0;
   int min_deviation = 100000;
   std::list<RecordingsItemPtr>::iterator bestMatchIter;
   for ( std::list<RecordingsItemPtr>::iterator iter = First; iter != Last; ++iter)
     if ( (*iter)->Duration() >= lengthLowerBound && (*iter)->Duration() <= lengthUpperBound ) {
        int deviation = abs( (*iter)->Duration() - EPG_Entry->Duration() );
        if (deviation < min_deviation || numRecordings == 0) {
          min_deviation = deviation;
          bestMatchIter = iter;
        }
        ++numRecordings;
   }
   if (numRecordings > 0) BestMatch = *bestMatchIter;
   return numRecordings;
}


	/**
	 * Implemetation of class RecordingsItemPtrCompare
	 */
	bool RecordingsItemPtrCompare::ByAscendingDate(const RecordingsItemPtr & first, const RecordingsItemPtr & second)
	{
		return (first->StartTime() < second->StartTime());
	}

	bool RecordingsItemPtrCompare::ByDescendingDate(const RecordingsItemPtr & first, const RecordingsItemPtr & second)
	{
		return (first->StartTime() >= second->StartTime());
	}

	int RecordingsItemPtrCompare::Compare(int &numEqualChars, const RecordingsItemPtr &first, const RecordingsItemPtr &second)
	{
                numEqualChars = 0;
                int i = first->NameForCompare().compare(second->NameForCompare() );
                if(i != 0) return i;
             // name is identical, compare short text
                i = RecordingsItemPtrCompare::compareLC(numEqualChars, first->ShortText(), second->ShortText() );
                if(i != 0) return i;
                i = RecordingsItemPtrCompare::compareLC(numEqualChars, first->Description(), second->Description() );
                return i;
	}

	bool RecordingsItemPtrCompare::ByAscendingName(const RecordingsItemPtr & first, const RecordingsItemPtr & second)  // return first < second
	{
           int i = 0;
           return RecordingsItemPtrCompare::Compare2(i, first, second) < 0;
	}

	bool RecordingsItemPtrCompare::ByDescendingName(const RecordingsItemPtr & first, const RecordingsItemPtr & second)  // return first > second
	{
           return RecordingsItemPtrCompare::ByAscendingName(second, first);
	}

        void RecordingsItemPtrCompare::getNameForCompare(string &NameForCompare, const string &Name){
// remove punctuation characters at the beginning of the string
            unsigned int start;
            for(start = 0; start < Name.length() && ispunct( Name[ start ] ); start++ );
            NameForCompare = Name;
            NameForCompare.erase(0, start);
// convert to lower case
            transform(NameForCompare.begin(), NameForCompare.end(), NameForCompare.begin(), ::tolower);
        }

        int RecordingsItemPtrCompare::compareLC(int &numEqualChars, const char *first, const char *second) {
          bool fe = !first  || !first[0];    // is first  string empty string?
          bool se = !second || !second[0];   // is second string empty string?
          if (fe && se) return 0;
          if (se) return  1;
          if (fe) return -1;
// compare strings case-insensitive
          int len;
          for(len = 0; first[len] && second[len] ; ++len) {
            int flc = tolower(first[len]);
            int slc = tolower(second[len]);
            if ( flc < slc ) { numEqualChars += len; return -1; }
            if ( flc > slc ) { numEqualChars += len; return  1; }
          }
          numEqualChars += len;
          if (second[len] ) return -1;
          if (first [len] ) return  1;
          return 0;
        }


	/**
	 *  Implementation of class RecordingsItem:
	 */
	RecordingsItem::RecordingsItem(string const & name, RecordingsItemPtr parent) :
		m_level((parent != NULL) ? parent->Level() + 1 : 0),
		m_name(name),
		m_entries(),
		m_parent(parent)
	{
                RecordingsItemPtrCompare::getNameForCompare(m_name_for_compare, name);
	}

	RecordingsItem::~RecordingsItem()
	{
	}


	/**
	 *  Implementation of class RecordingsItemDir:
	 */
	RecordingsItemDir::RecordingsItemDir(const string& name, int level, RecordingsItemPtr parent) :
		RecordingsItem(name, parent),
		m_level(level)
	{
		// dsyslog("live: REC: C: dir %s -> %s", name.c_str(), parent ? parent->Name().c_str() : "ROOT");
	}

	RecordingsItemDir::~RecordingsItemDir()
	{
		// dsyslog("live: REC: D: dir %s", Name().c_str());
	}


	/**
	 *  Implementation of class RecordingsItemRec:
	 */
	RecordingsItemRec::RecordingsItemRec(const string& id, const string& name, const cRecording* recording, RecordingsItemPtr parent) :
		RecordingsItem(name, parent),
		m_recording(recording),
		m_id(id)
	{
		// dsyslog("live: REC: C: rec %s -> %s", name.c_str(), parent->Name().c_str());
	}

	RecordingsItemRec::~RecordingsItemRec()
	{
		// dsyslog("live: REC: D: rec %s", Name().c_str());
	}

	time_t RecordingsItemRec::StartTime() const
	{
		return m_recording->Start();
	}

	long RecordingsItemRec::Duration() const
	{
		if (!m_recording->FileName()) return 0;
		return m_recording->LengthInSeconds() / 60;
	}

	/**
	 * Implemetation of class RecordingsItemDummy
	 */
        RecordingsItemDummy::RecordingsItemDummy(const std::string &Name, const std::string &ShortText, const std::string &Description, long Duration):
                RecordingsItem(Name, RecordingsItemPtr() ),
                m_short_text(ShortText.c_str() ),
                m_description(Description.c_str() ),
                m_duration( Duration / 60 )
                { }

	/**
	 *  Implementation of class RecordingsTree:
	 */
	RecordingsTree::RecordingsTree(RecordingsManagerPtr recMan) :
		m_maxLevel(0),
		m_root(new RecordingsItemDir("", 0, RecordingsItemPtr()))
	{
		// esyslog("live: DH: ****** RecordingsTree::RecordingsTree() ********");
#if VDRVERSNUM >= 20301
		LOCK_RECORDINGS_READ;
		for (cRecording* recording = (cRecording *)Recordings->First(); recording; recording = (cRecording *)Recordings->Next(recording)) {
#else
		for (cRecording* recording = Recordings.First(); recording; recording = Recordings.Next(recording)) {
#endif
			if (m_maxLevel < recording->HierarchyLevels()) {
				m_maxLevel = recording->HierarchyLevels();
			}

			RecordingsItemPtr dir = m_root;
			string name(recording->Name());

			// esyslog("live: DH: recName = '%s'", recording->Name());
			int level = 0;
			size_t index = 0;
			size_t pos = 0;
			do {
				pos = name.find('~', index);
				if (pos != string::npos) {
					string dirName(name.substr(index, pos - index));
					index = pos + 1;
					RecordingsMap::iterator i = findDir(dir, dirName);
					if (i == dir->m_entries.end()) {
						RecordingsItemPtr recPtr (new RecordingsItemDir(dirName, level, dir));
						dir->m_entries.insert(pair< string, RecordingsItemPtr > (dirName, recPtr));
						i = findDir(dir, dirName);
#if 0
						if (i != dir->m_entries.end()) {
							// esyslog("live: DH: added dir: '%s'", dirName.c_str());
						}
						else {
							// esyslog("live: DH: panic: didn't found inserted dir: '%s'", dirName.c_str());
						}
#endif
					}
					dir = i->second;
					// esyslog("live: DH: current dir: '%s'", dir->Name().c_str());
					level++;
				}
				else {
					string recName(name.substr(index, name.length() - index));
					RecordingsItemPtr recPtr (new RecordingsItemRec(recMan->Md5Hash(recording), recName, recording, dir));
					dir->m_entries.insert(pair< string, RecordingsItemPtr > (recName, recPtr));
					// esyslog("live: DH: added rec: '%s'", recName.c_str());
				}
			} while (pos != string::npos);
		}
		// esyslog("live: DH: ------ RecordingsTree::RecordingsTree() --------");
	}

	RecordingsTree::~RecordingsTree()
	{
		// esyslog("live: DH: ****** RecordingsTree::~RecordingsTree() ********");
	}

	RecordingsMap::iterator RecordingsTree::begin(const vector< string >& path)
	{
		if (path.empty()) {
			return m_root->m_entries.begin();
		}

		RecordingsItemPtr recItem = m_root;
		for (vector< string >::const_iterator i = path.begin(); i != path.end(); ++i)
		{
			pair< RecordingsMap::iterator, RecordingsMap::iterator> range = recItem->m_entries.equal_range(*i);
			for (RecordingsMap::iterator iter = range.first; iter != range.second; ++iter) {
				if (iter->second->IsDir()) {
					recItem = iter->second;
					break;
				}
			}
		}
		return recItem->m_entries.begin();
	}

	RecordingsMap::iterator RecordingsTree::end(const vector< string >&path)
	{
		if (path.empty()) {
			return m_root->m_entries.end();
		}

		RecordingsItemPtr recItem = m_root;
		for (vector< string >::const_iterator i = path.begin(); i != path.end(); ++i)
		{
			pair< RecordingsMap::iterator, RecordingsMap::iterator> range = recItem->m_entries.equal_range(*i);
			for (RecordingsMap::iterator iter = range.first; iter != range.second; ++iter) {
				if (iter->second->IsDir()) {
					recItem = iter->second;
					break;
				}
			}
		}
		return recItem->m_entries.end();
	}

	RecordingsMap::iterator RecordingsTree::findDir(RecordingsItemPtr& dir, const string& dirName)
	{
		pair< RecordingsMap::iterator, RecordingsMap::iterator > range = dir->m_entries.equal_range(dirName);
		for (RecordingsMap::iterator i = range.first; i != range.second; ++i) {
			if (i->second->IsDir()) {
				return i;
			}
		}
		return dir->m_entries.end();
	}


	/**
	 *  Implementation of class RecordingsTreePtr:
	 */
	RecordingsTreePtr::RecordingsTreePtr() :
		stdext::shared_ptr<RecordingsTree>(),
		m_recManPtr()
	{
	}

	RecordingsTreePtr::RecordingsTreePtr(RecordingsManagerPtr recManPtr, stdext::shared_ptr< RecordingsTree > recTree) :
		stdext::shared_ptr<RecordingsTree>(recTree),
		m_recManPtr(recManPtr)
	{
	}

	RecordingsTreePtr::~RecordingsTreePtr()
	{
	}


	/**
	 *  Implementation of class RecordingsList:
	 */
	RecordingsList::RecordingsList(RecordingsTreePtr recTree) :
		m_pRecVec(new RecVecType())
	{
		if (!m_pRecVec) {
			return;
		}

		std::stack< RecordingsItemPtr > treeStack;
		treeStack.push(recTree->Root());

		while (!treeStack.empty()) {
			RecordingsItemPtr current = treeStack.top();
			treeStack.pop();
			for (RecordingsMap::const_iterator iter = current->begin(); iter != current->end(); ++iter) {
				RecordingsItemPtr recItem = iter->second;
				if (recItem->IsDir()) {
					treeStack.push(recItem);
				}
				else {
					m_pRecVec->push_back(recItem);
				}
			}
		}
	}

	RecordingsList::RecordingsList(stdext::shared_ptr< RecordingsList > recList, bool ascending) :
		m_pRecVec(new RecVecType(recList->size()))
	{
		if (!m_pRecVec) {
			return;
		}
		if (ascending) {
			partial_sort_copy(recList->begin(), recList->end(), m_pRecVec->begin(), m_pRecVec->end(), Ascending());
		}
		else {
			partial_sort_copy(recList->begin(), recList->end(), m_pRecVec->begin(), m_pRecVec->end(), Descending());
		}
	}

	RecordingsList::RecordingsList(stdext::shared_ptr< RecordingsList > recList, time_t begin, time_t end, bool ascending) :
		m_pRecVec(new RecVecType())
	{
		if (end > begin) {
			return;
		}
		if (!m_pRecVec) {
			return;
		}
		remove_copy_if(recList->begin(), recList->end(), m_pRecVec->end(), NotInRange(begin, end));

		if (ascending) {
			sort(m_pRecVec->begin(), m_pRecVec->end(), Ascending());
		}
		else {
			sort(m_pRecVec->begin(), m_pRecVec->end(), Descending());
		}
	}

	RecordingsList::~RecordingsList()
	{
		if (m_pRecVec) {
			delete m_pRecVec, m_pRecVec = 0;
		}
	}


	RecordingsList::NotInRange::NotInRange(time_t begin, time_t end) :
		m_begin(begin),
		m_end(end)
	{
	}

	bool RecordingsList::NotInRange::operator()(RecordingsItemPtr const &x) const
	{
		return (x->StartTime() < m_begin) || (m_end >= x->StartTime());
	}


	/**
	 *  Implementation of class RecordingsList:
	 */
	RecordingsListPtr::RecordingsListPtr(RecordingsManagerPtr recManPtr, stdext::shared_ptr< RecordingsList > recList) :
		stdext::shared_ptr< RecordingsList >(recList),
		m_recManPtr(recManPtr)
	{
	}

	RecordingsListPtr::~RecordingsListPtr()
	{
	}


	/**
	 *  Implementation of class DirectoryList:
	 */
	DirectoryList::DirectoryList(RecordingsManagerPtr recManPtr) :
		m_pDirVec(new DirVecType())
	{
		if (!m_pDirVec) {
			return;
		}

		m_pDirVec->push_back(""); // always add root directory
		for (cNestedItem* item = Folders.First(); item; item = Folders.Next(item)) { // add folders.conf entries
			InjectFoldersConf(item);
		}
#if VDRVERSNUM >= 20301
		LOCK_RECORDINGS_READ;
		for (cRecording* recording = (cRecording *)Recordings->First(); recording; recording = (cRecording*)Recordings->Next(recording)) {
#else
		for (cRecording* recording = Recordings.First(); recording; recording = Recordings.Next(recording)) {
#endif
			string name = recording->Name();
			size_t found = name.find_last_of("~");

			if (found != string::npos) {
				m_pDirVec->push_back(StringReplace(name.substr(0, found), "~", "/"));
			}
		}
		m_pDirVec->sort();
		m_pDirVec->unique();
	}

	DirectoryList::~DirectoryList()
	{
		if (m_pDirVec) {
			delete m_pDirVec, m_pDirVec = 0;
		}
	}

	void DirectoryList::InjectFoldersConf(cNestedItem * folder, string parent)
	{
		if (!folder) {
			return;
		}

		string dir = string((parent.size() == 0) ? "" : parent + "/") + folder->Text();
		m_pDirVec->push_back(StringReplace(dir, "_", " "));

		if (!folder->SubItems()) {
			return;
		}

		for(cNestedItem* item = folder->SubItems()->First(); item; item = folder->SubItems()->Next(item)) {
			InjectFoldersConf(item, dir);
		}
	}

	/**
	 *  Implementation of class DirectoryListPtr:
	 */
	DirectoryListPtr::DirectoryListPtr(RecordingsManagerPtr recManPtr, stdext::shared_ptr< DirectoryList > recDirs) :
		stdext::shared_ptr< DirectoryList >(recDirs),
		m_recManPtr(recManPtr)
	{
	}

	DirectoryListPtr::~DirectoryListPtr()
	{
	}


	/**
	 *  Implementation of function LiveRecordingsManager:
	 */
	RecordingsManagerPtr LiveRecordingsManager()
	{
		RecordingsManagerPtr r = RecordingsManager::m_recMan.lock();
		if (r) {
			return r;
		}
		else {
			RecordingsManagerPtr n(new RecordingsManager);
			RecordingsManager::m_recMan = n;
			return n;
		}
	}

} // namespace vdrlive
