/*
 * 86Box    A hypervisor and IBM PC system emulator that specializes in
 *          running old operating systems and software designed for IBM
 *          PC systems and compatibles from 1981 through fairly recent
 *          system designs based on the PCI bus.
 *
 *          This file is part of the 86Box distribution.
 *
 *          Implementation of PS/2 series Mouse devices.
 *
 * Authors: Miran Grca, <mgrca8@gmail.com>
 *          Fred N. van Kempen, <decwiz@yahoo.com>
 *
 *          Copyright 2016-2023 Miran Grca.
 *          Copyright 2017-2023 Fred N. van Kempen.
 */
#include <stdarg.h>
#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <stdlib.h>
#include <wchar.h>
#define HAVE_STDARG_H
#include <86box/86box.h>
#include <86box/device.h>
#include <86box/keyboard.h>
#include <86box/mouse.h>

#define FLAG_PS2       0x08  /* dev is AT or PS/2 */
#define FLAG_AT        0x00  /* dev is AT or PS/2 */
#define FLAG_TYPE_MASK 0x07  /* mask for type     */

#define FIFO_SIZE      16

#define BAT_COUNT      1000

enum {
    KBD_84_KEY = 0,
    KBD_101_KEY,
    KBD_102_KEY,
    KBD_JIS,
    KBD_KOREAN
};

#define FLAG_ENABLED 0x10  /* dev is enabled for use */
#define FLAG_CTRLDAT 0x08  /* ctrl or data mode */

const uint8_t id_bytes[16][4] = { { 0x00, 0x00, 0x00, 0x00 },    /* AT 84-key */
                                  { 0x00, 0x00, 0x00, 0x00 },    /* AT 101/102/106-key */
                                  { 0x00, 0x00, 0x00, 0x00 },
                                  { 0x00, 0x00, 0x00, 0x00 },
                                  { 0x00, 0x00, 0x00, 0x00 },    /* AT Korean */
                                  { 0x00, 0x00, 0x00, 0x00 },
                                  { 0x00, 0x00, 0x00, 0x00 },
                                  { 0x00, 0x00, 0x00, 0x00 },
                                  { 0x00, 0x00, 0x00, 0x00 },
                                  { 0xab, 0x83, 0x00, 0x00 },    /* PS/2 101-key */
                                  { 0xab, 0x83, 0x00, 0x00 },    /* PS/2 102-key */
                                  { 0xab, 0x90, 0x00, 0x00 },    /* PS/2 106-key JIS */
                                  /* Japanese keyboard ID - TODO: Find the actual Korean one. */
                                  { 0xab, 0x90, 0x00, 0x00 },    /* PS/2 Korean */
                                  { 0x00, 0x00, 0x00, 0x00 },
                                  { 0x00, 0x00, 0x00, 0x00 },
                                  { 0x00, 0x00, 0x00, 0x00 } };

/* Global keyboard flags for scan code set 3:
   bit 0 = repeat, bit 1 = makes break code? */
uint8_t keyboard_set3_flags[512];
uint8_t keyboard_set3_all_repeat;
uint8_t keyboard_set3_all_break;

/* Global keyboard mode:
   Bits 0 - 1 = scan code set. */
uint8_t keyboard_mode = 0x02;

static atkbc_dev_t *SavedKbd                        = NULL;

static uint8_t     inv_cmd_response                 = 0xfa;

static uint16_t    bat_counter                      = 0;

static const scancode scancode_set1[512] = {
  // clang-format off
    { .mk = {            0 }, .brk = {                   0 } }, /* 000 */
    { .mk = {      0x01, 0 }, .brk = {             0x81, 0 } }, /* 001 */
    { .mk = {      0x02, 0 }, .brk = {             0x82, 0 } }, /* 002 */
    { .mk = {      0x03, 0 }, .brk = {             0x83, 0 } }, /* 003 */
    { .mk = {      0x04, 0 }, .brk = {             0x84, 0 } }, /* 004 */
    { .mk = {      0x05, 0 }, .brk = {             0x85, 0 } }, /* 005 */
    { .mk = {      0x06, 0 }, .brk = {             0x86, 0 } }, /* 006 */
    { .mk = {      0x07, 0 }, .brk = {             0x87, 0 } }, /* 007 */
    { .mk = {      0x08, 0 }, .brk = {             0x88, 0 } }, /* 008 */
    { .mk = {      0x09, 0 }, .brk = {             0x89, 0 } }, /* 009 */
    { .mk = {      0x0a, 0 }, .brk = {             0x8a, 0 } }, /* 00a */
    { .mk = {      0x0b, 0 }, .brk = {             0x8b, 0 } }, /* 00b */
    { .mk = {      0x0c, 0 }, .brk = {             0x8c, 0 } }, /* 00c */
    { .mk = {      0x0d, 0 }, .brk = {             0x8d, 0 } }, /* 00d */
    { .mk = {      0x0e, 0 }, .brk = {             0x8e, 0 } }, /* 00e */
    { .mk = {      0x0f, 0 }, .brk = {             0x8f, 0 } }, /* 00f */
    { .mk = {      0x10, 0 }, .brk = {             0x90, 0 } }, /* 010 */
    { .mk = {      0x11, 0 }, .brk = {             0x91, 0 } }, /* 011 */
    { .mk = {      0x12, 0 }, .brk = {             0x92, 0 } }, /* 012 */
    { .mk = {      0x13, 0 }, .brk = {             0x93, 0 } }, /* 013 */
    { .mk = {      0x14, 0 }, .brk = {             0x94, 0 } }, /* 014 */
    { .mk = {      0x15, 0 }, .brk = {             0x95, 0 } }, /* 015 */
    { .mk = {      0x16, 0 }, .brk = {             0x96, 0 } }, /* 016 */
    { .mk = {      0x17, 0 }, .brk = {             0x97, 0 } }, /* 017 */
    { .mk = {      0x18, 0 }, .brk = {             0x98, 0 } }, /* 018 */
    { .mk = {      0x19, 0 }, .brk = {             0x99, 0 } }, /* 019 */
    { .mk = {      0x1a, 0 }, .brk = {             0x9a, 0 } }, /* 01a */
    { .mk = {      0x1b, 0 }, .brk = {             0x9b, 0 } }, /* 01b */
    { .mk = {      0x1c, 0 }, .brk = {             0x9c, 0 } }, /* 01c */
    { .mk = {      0x1d, 0 }, .brk = {             0x9d, 0 } }, /* 01d */
    { .mk = {      0x1e, 0 }, .brk = {             0x9e, 0 } }, /* 01e */
    { .mk = {      0x1f, 0 }, .brk = {             0x9f, 0 } }, /* 01f */
    { .mk = {      0x20, 0 }, .brk = {             0xa0, 0 } }, /* 020 */
    { .mk = {      0x21, 0 }, .brk = {             0xa1, 0 } }, /* 021 */
    { .mk = {      0x22, 0 }, .brk = {             0xa2, 0 } }, /* 022 */
    { .mk = {      0x23, 0 }, .brk = {             0xa3, 0 } }, /* 023 */
    { .mk = {      0x24, 0 }, .brk = {             0xa4, 0 } }, /* 024 */
    { .mk = {      0x25, 0 }, .brk = {             0xa5, 0 } }, /* 025 */
    { .mk = {      0x26, 0 }, .brk = {             0xa6, 0 } }, /* 026 */
    { .mk = {      0x27, 0 }, .brk = {             0xa7, 0 } }, /* 027 */
    { .mk = {      0x28, 0 }, .brk = {             0xa8, 0 } }, /* 028 */
    { .mk = {      0x29, 0 }, .brk = {             0xa9, 0 } }, /* 029 */
    { .mk = {      0x2a, 0 }, .brk = {             0xaa, 0 } }, /* 02a */
    { .mk = {      0x2b, 0 }, .brk = {             0xab, 0 } }, /* 02b */
    { .mk = {      0x2c, 0 }, .brk = {             0xac, 0 } }, /* 02c */
    { .mk = {      0x2d, 0 }, .brk = {             0xad, 0 } }, /* 02d */
    { .mk = {      0x2e, 0 }, .brk = {             0xae, 0 } }, /* 02e */
    { .mk = {      0x2f, 0 }, .brk = {             0xaf, 0 } }, /* 02f */
    { .mk = {      0x30, 0 }, .brk = {             0xb0, 0 } }, /* 030 */
    { .mk = {      0x31, 0 }, .brk = {             0xb1, 0 } }, /* 031 */
    { .mk = {      0x32, 0 }, .brk = {             0xb2, 0 } }, /* 032 */
    { .mk = {      0x33, 0 }, .brk = {             0xb3, 0 } }, /* 033 */
    { .mk = {      0x34, 0 }, .brk = {             0xb4, 0 } }, /* 034 */
    { .mk = {      0x35, 0 }, .brk = {             0xb5, 0 } }, /* 035 */
    { .mk = {      0x36, 0 }, .brk = {             0xb6, 0 } }, /* 036 */
    { .mk = {      0x37, 0 }, .brk = {             0xb7, 0 } }, /* 037 */
    { .mk = {      0x38, 0 }, .brk = {             0xb8, 0 } }, /* 038 */
    { .mk = {      0x39, 0 }, .brk = {             0xb9, 0 } }, /* 039 */
    { .mk = {      0x3a, 0 }, .brk = {             0xba, 0 } }, /* 03a */
    { .mk = {      0x3b, 0 }, .brk = {             0xbb, 0 } }, /* 03b */
    { .mk = {      0x3c, 0 }, .brk = {             0xbc, 0 } }, /* 03c */
    { .mk = {      0x3d, 0 }, .brk = {             0xbd, 0 } }, /* 03d */
    { .mk = {      0x3e, 0 }, .brk = {             0xbe, 0 } }, /* 03e */
    { .mk = {      0x3f, 0 }, .brk = {             0xbf, 0 } }, /* 03f */
    { .mk = {      0x40, 0 }, .brk = {             0xc0, 0 } }, /* 040 */
    { .mk = {      0x41, 0 }, .brk = {             0xc1, 0 } }, /* 041 */
    { .mk = {      0x42, 0 }, .brk = {             0xc2, 0 } }, /* 042 */
    { .mk = {      0x43, 0 }, .brk = {             0xc3, 0 } }, /* 043 */
    { .mk = {      0x44, 0 }, .brk = {             0xc4, 0 } }, /* 044 */
    { .mk = {      0x45, 0 }, .brk = {             0xc5, 0 } }, /* 045 */
    { .mk = {      0x46, 0 }, .brk = {             0xc6, 0 } }, /* 046 */
    { .mk = {      0x47, 0 }, .brk = {             0xc7, 0 } }, /* 047 */
    { .mk = {      0x48, 0 }, .brk = {             0xc8, 0 } }, /* 048 */
    { .mk = {      0x49, 0 }, .brk = {             0xc9, 0 } }, /* 049 */
    { .mk = {      0x4a, 0 }, .brk = {             0xca, 0 } }, /* 04a */
    { .mk = {      0x4b, 0 }, .brk = {             0xcb, 0 } }, /* 04b */
    { .mk = {      0x4c, 0 }, .brk = {             0xcc, 0 } }, /* 04c */
    { .mk = {      0x4d, 0 }, .brk = {             0xcd, 0 } }, /* 04d */
    { .mk = {      0x4e, 0 }, .brk = {             0xce, 0 } }, /* 04e */
    { .mk = {      0x4f, 0 }, .brk = {             0xcf, 0 } }, /* 04f */
    { .mk = {      0x50, 0 }, .brk = {             0xd0, 0 } }, /* 050 */
    { .mk = {      0x51, 0 }, .brk = {             0xd1, 0 } }, /* 051 */
    { .mk = {      0x52, 0 }, .brk = {             0xd2, 0 } }, /* 052 */
    { .mk = {      0x53, 0 }, .brk = {             0xd3, 0 } }, /* 053 */
    { .mk = {      0x54, 0 }, .brk = {             0xd4, 0 } }, /* 054 */
    { .mk = {      0x55, 0 }, .brk = {             0xd5, 0 } }, /* 055 */
    { .mk = {      0x56, 0 }, .brk = {             0xd6, 0 } }, /* 056 */
    { .mk = {      0x57, 0 }, .brk = {             0xd7, 0 } }, /* 057 */
    { .mk = {      0x58, 0 }, .brk = {             0xd8, 0 } }, /* 058 */
    { .mk = {      0x59, 0 }, .brk = {             0xd9, 0 } }, /* 059 */
    { .mk = {      0x5a, 0 }, .brk = {             0xda, 0 } }, /* 05a */
    { .mk = {      0x5b, 0 }, .brk = {             0xdb, 0 } }, /* 05b */
    { .mk = {      0x5c, 0 }, .brk = {             0xdc, 0 } }, /* 05c */
    { .mk = {      0x5d, 0 }, .brk = {             0xdd, 0 } }, /* 05d */
    { .mk = {      0x5e, 0 }, .brk = {             0xde, 0 } }, /* 05e */
    { .mk = {      0x5f, 0 }, .brk = {             0xdf, 0 } }, /* 05f */
    { .mk = {      0x60, 0 }, .brk = {             0xe0, 0 } }, /* 060 */
    { .mk = {      0x61, 0 }, .brk = {             0xe1, 0 } }, /* 061 */
    { .mk = {      0x62, 0 }, .brk = {             0xe2, 0 } }, /* 062 */
    { .mk = {      0x63, 0 }, .brk = {             0xe3, 0 } }, /* 063 */
    { .mk = {      0x64, 0 }, .brk = {             0xe4, 0 } }, /* 064 */
    { .mk = {      0x65, 0 }, .brk = {             0xe5, 0 } }, /* 065 */
    { .mk = {      0x66, 0 }, .brk = {             0xe6, 0 } }, /* 066 */
    { .mk = {      0x67, 0 }, .brk = {             0xe7, 0 } }, /* 067 */
    { .mk = {      0x68, 0 }, .brk = {             0xe8, 0 } }, /* 068 */
    { .mk = {      0x69, 0 }, .brk = {             0xe9, 0 } }, /* 069 */
    { .mk = {      0x6a, 0 }, .brk = {             0xea, 0 } }, /* 06a */
    { .mk = {      0x6b, 0 }, .brk = {             0xeb, 0 } }, /* 06b */
    { .mk = {      0x6c, 0 }, .brk = {             0xec, 0 } }, /* 06c */
    { .mk = {      0x6d, 0 }, .brk = {             0xed, 0 } }, /* 06d */
    { .mk = {      0x6e, 0 }, .brk = {             0xee, 0 } }, /* 06e */
    { .mk = {      0x6f, 0 }, .brk = {             0xef, 0 } }, /* 06f */
    { .mk = {      0x70, 0 }, .brk = {             0xf0, 0 } }, /* 070 */
    { .mk = {      0x71, 0 }, .brk = {             0xf1, 0 } }, /* 071 */
    { .mk = {      0x72, 0 }, .brk = {             0xf2, 0 } }, /* 072 */
    { .mk = {      0x73, 0 }, .brk = {             0xf3, 0 } }, /* 073 */
    { .mk = {      0x74, 0 }, .brk = {             0xf4, 0 } }, /* 074 */
    { .mk = {      0x75, 0 }, .brk = {             0xf5, 0 } }, /* 075 */
    { .mk = {      0x76, 0 }, .brk = {             0xf6, 0 } }, /* 076 */
    { .mk = {      0x77, 0 }, .brk = {             0xf7, 0 } }, /* 077 */
    { .mk = {      0x78, 0 }, .brk = {             0xf8, 0 } }, /* 078 */
    { .mk = {      0x79, 0 }, .brk = {             0xf9, 0 } }, /* 079 */
    { .mk = {      0x7a, 0 }, .brk = {             0xfa, 0 } }, /* 07a */
    { .mk = {      0x7b, 0 }, .brk = {             0xfb, 0 } }, /* 07b */
    { .mk = {      0x7c, 0 }, .brk = {             0xfc, 0 } }, /* 07c */
    { .mk = {      0x7d, 0 }, .brk = {             0xfd, 0 } }, /* 07d */
    { .mk = {      0x7e, 0 }, .brk = {             0xfe, 0 } }, /* 07e */
    { .mk = {      0x7f, 0 }, .brk = {             0xff, 0 } }, /* 07f */
    { .mk = {      0x80, 0 }, .brk = {                   0 } }, /* 080 */
    { .mk = {      0x81, 0 }, .brk = {                   0 } }, /* 081 */
    { .mk = {      0x82, 0 }, .brk = {                   0 } }, /* 082 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 083 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 084 */
    { .mk = {      0x85, 0 }, .brk = {                   0 } }, /* 085 */
    { .mk = {      0x86, 0 }, .brk = {                   0 } }, /* 086 */
    { .mk = {      0x87, 0 }, .brk = {                   0 } }, /* 087 */
    { .mk = {      0x88, 0 }, .brk = {                   0 } }, /* 088 */
    { .mk = {      0x89, 0 }, .brk = {                   0 } }, /* 089 */
    { .mk = {      0x8a, 0 }, .brk = {                   0 } }, /* 08a */
    { .mk = {      0x8b, 0 }, .brk = {                   0 } }, /* 08b */
    { .mk = {      0x8c, 0 }, .brk = {                   0 } }, /* 08c */
    { .mk = {      0x8d, 0 }, .brk = {                   0 } }, /* 08d */
    { .mk = {      0x8e, 0 }, .brk = {                   0 } }, /* 08e */
    { .mk = {      0x8f, 0 }, .brk = {                   0 } }, /* 08f */
    { .mk = {      0x90, 0 }, .brk = {                   0 } }, /* 090 */
    { .mk = {      0x91, 0 }, .brk = {                   0 } }, /* 091 */
    { .mk = {      0x92, 0 }, .brk = {                   0 } }, /* 092 */
    { .mk = {      0x93, 0 }, .brk = {                   0 } }, /* 093 */
    { .mk = {      0x94, 0 }, .brk = {                   0 } }, /* 094 */
    { .mk = {      0x95, 0 }, .brk = {                   0 } }, /* 095 */
    { .mk = {      0x96, 0 }, .brk = {                   0 } }, /* 096 */
    { .mk = {      0x97, 0 }, .brk = {                   0 } }, /* 097 */
    { .mk = {      0x98, 0 }, .brk = {                   0 } }, /* 098 */
    { .mk = {      0x99, 0 }, .brk = {                   0 } }, /* 099 */
    { .mk = {      0x9a, 0 }, .brk = {                   0 } }, /* 09a */
    { .mk = {      0x9b, 0 }, .brk = {                   0 } }, /* 09b */
    { .mk = {      0x9c, 0 }, .brk = {                   0 } }, /* 09c */
    { .mk = {      0x9d, 0 }, .brk = {                   0 } }, /* 09d */
    { .mk = {      0x9e, 0 }, .brk = {                   0 } }, /* 09e */
    { .mk = {      0x9f, 0 }, .brk = {                   0 } }, /* 09f */
    { .mk = {      0xa0, 0 }, .brk = {                   0 } }, /* 0a0 */
    { .mk = {      0xa1, 0 }, .brk = {                   0 } }, /* 0a1 */
    { .mk = {      0xa2, 0 }, .brk = {                   0 } }, /* 0a2 */
    { .mk = {      0xa3, 0 }, .brk = {                   0 } }, /* 0a3 */
    { .mk = {      0xa4, 0 }, .brk = {                   0 } }, /* 0a4 */
    { .mk = {      0xa5, 0 }, .brk = {                   0 } }, /* 0a5 */
    { .mk = {      0xa6, 0 }, .brk = {                   0 } }, /* 0a6 */
    { .mk = {      0xa7, 0 }, .brk = {                   0 } }, /* 0a7 */
    { .mk = {      0xa8, 0 }, .brk = {                   0 } }, /* 0a8 */
    { .mk = {      0xa9, 0 }, .brk = {                   0 } }, /* 0a9 */
    { .mk = {      0xaa, 0 }, .brk = {                   0 } }, /* 0aa */
    { .mk = {      0xab, 0 }, .brk = {                   0 } }, /* 0ab */
    { .mk = {      0xac, 0 }, .brk = {                   0 } }, /* 0ac */
    { .mk = {      0xad, 0 }, .brk = {                   0 } }, /* 0ad */
    { .mk = {      0xae, 0 }, .brk = {                   0 } }, /* 0ae */
    { .mk = {      0xaf, 0 }, .brk = {                   0 } }, /* 0af */
    { .mk = {      0xb0, 0 }, .brk = {                   0 } }, /* 0b0 */
    { .mk = {      0xb1, 0 }, .brk = {                   0 } }, /* 0b1 */
    { .mk = {      0xb2, 0 }, .brk = {                   0 } }, /* 0b2 */
    { .mk = {      0xb3, 0 }, .brk = {                   0 } }, /* 0b3 */
    { .mk = {      0xb4, 0 }, .brk = {                   0 } }, /* 0b4 */
    { .mk = {      0xb5, 0 }, .brk = {                   0 } }, /* 0b5 */
    { .mk = {      0xb6, 0 }, .brk = {                   0 } }, /* 0b6 */
    { .mk = {      0xb7, 0 }, .brk = {                   0 } }, /* 0b7 */
    { .mk = {      0xb8, 0 }, .brk = {                   0 } }, /* 0b8 */
    { .mk = {      0xb9, 0 }, .brk = {                   0 } }, /* 0b9 */
    { .mk = {      0xba, 0 }, .brk = {                   0 } }, /* 0ba */
    { .mk = {      0xbb, 0 }, .brk = {                   0 } }, /* 0bb */
    { .mk = {      0xbc, 0 }, .brk = {                   0 } }, /* 0bc */
    { .mk = {      0xbd, 0 }, .brk = {                   0 } }, /* 0bd */
    { .mk = {      0xbe, 0 }, .brk = {                   0 } }, /* 0be */
    { .mk = {      0xbf, 0 }, .brk = {                   0 } }, /* 0bf */
    { .mk = {      0xc0, 0 }, .brk = {                   0 } }, /* 0c0 */
    { .mk = {      0xc1, 0 }, .brk = {                   0 } }, /* 0c1 */
    { .mk = {      0xc2, 0 }, .brk = {                   0 } }, /* 0c2 */
    { .mk = {      0xc3, 0 }, .brk = {                   0 } }, /* 0c3 */
    { .mk = {      0xc4, 0 }, .brk = {                   0 } }, /* 0c4 */
    { .mk = {      0xc5, 0 }, .brk = {                   0 } }, /* 0c5 */
    { .mk = {      0xc6, 0 }, .brk = {                   0 } }, /* 0c6 */
    { .mk = {      0xc7, 0 }, .brk = {                   0 } }, /* 0c7 */
    { .mk = {      0xc8, 0 }, .brk = {                   0 } }, /* 0c8 */
    { .mk = {      0xc9, 0 }, .brk = {                   0 } }, /* 0c9 */
    { .mk = {      0xca, 0 }, .brk = {                   0 } }, /* 0ca */
    { .mk = {      0xcb, 0 }, .brk = {                   0 } }, /* 0cb */
    { .mk = {      0xcc, 0 }, .brk = {                   0 } }, /* 0cc */
    { .mk = {      0xcd, 0 }, .brk = {                   0 } }, /* 0cd */
    { .mk = {      0xce, 0 }, .brk = {                   0 } }, /* 0ce */
    { .mk = {      0xcf, 0 }, .brk = {                   0 } }, /* 0cf */
    { .mk = {      0xd0, 0 }, .brk = {                   0 } }, /* 0d0 */
    { .mk = {      0xd1, 0 }, .brk = {                   0 } }, /* 0d1 */
    { .mk = {      0xd2, 0 }, .brk = {                   0 } }, /* 0d2 */
    { .mk = {      0xd3, 0 }, .brk = {                   0 } }, /* 0d3 */
    { .mk = {      0xd4, 0 }, .brk = {                   0 } }, /* 0d4 */
    { .mk = {      0xd5, 0 }, .brk = {                   0 } }, /* 0d5 */
    { .mk = {      0xd6, 0 }, .brk = {                   0 } }, /* 0d6 */
    { .mk = {      0xd7, 0 }, .brk = {                   0 } }, /* 0d7 */
    { .mk = {      0xd8, 0 }, .brk = {                   0 } }, /* 0d8 */
    { .mk = {      0xd9, 0 }, .brk = {                   0 } }, /* 0d9 */
    { .mk = {      0xda, 0 }, .brk = {                   0 } }, /* 0da */
    { .mk = {      0xdb, 0 }, .brk = {                   0 } }, /* 0db */
    { .mk = {      0xdc, 0 }, .brk = {                   0 } }, /* 0dc */
    { .mk = {      0xdd, 0 }, .brk = {                   0 } }, /* 0dd */
    { .mk = {      0xde, 0 }, .brk = {                   0 } }, /* 0de */
    { .mk = {      0xdf, 0 }, .brk = {                   0 } }, /* 0df */
    { .mk = {      0xe0, 0 }, .brk = {                   0 } }, /* 0e0 */
    { .mk = {      0xe1, 0 }, .brk = {                   0 } }, /* 0e1 */
    { .mk = {      0xe2, 0 }, .brk = {                   0 } }, /* 0e2 */
    { .mk = {      0xe3, 0 }, .brk = {                   0 } }, /* 0e3 */
    { .mk = {      0xe4, 0 }, .brk = {                   0 } }, /* 0e4 */
    { .mk = {      0xe5, 0 }, .brk = {                   0 } }, /* 0e5 */
    { .mk = {      0xe6, 0 }, .brk = {                   0 } }, /* 0e6 */
    { .mk = {      0xe7, 0 }, .brk = {                   0 } }, /* 0e7 */
    { .mk = {      0xe8, 0 }, .brk = {                   0 } }, /* 0e8 */
    { .mk = {      0xe9, 0 }, .brk = {                   0 } }, /* 0e9 */
    { .mk = {      0xea, 0 }, .brk = {                   0 } }, /* 0ea */
    { .mk = {      0xeb, 0 }, .brk = {                   0 } }, /* 0eb */
    { .mk = {      0xec, 0 }, .brk = {                   0 } }, /* 0ec */
    { .mk = {      0xed, 0 }, .brk = {                   0 } }, /* 0ed */
    { .mk = {      0xee, 0 }, .brk = {                   0 } }, /* 0ee */
    { .mk = {      0xef, 0 }, .brk = {                   0 } }, /* 0ef */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0f0 */
    { .mk = {      0xf1, 0 }, .brk = {                   0 } }, /* 0f1 */
    { .mk = {      0xf2, 0 }, .brk = {                   0 } }, /* 0f2 */
    { .mk = {      0xf3, 0 }, .brk = {                   0 } }, /* 0f3 */
    { .mk = {      0xf4, 0 }, .brk = {                   0 } }, /* 0f4 */
    { .mk = {      0xf5, 0 }, .brk = {                   0 } }, /* 0f5 */
    { .mk = {      0xf6, 0 }, .brk = {                   0 } }, /* 0f6 */
    { .mk = {      0xf7, 0 }, .brk = {                   0 } }, /* 0f7 */
    { .mk = {      0xf8, 0 }, .brk = {                   0 } }, /* 0f8 */
    { .mk = {      0xf9, 0 }, .brk = {                   0 } }, /* 0f9 */
    { .mk = {      0xfa, 0 }, .brk = {                   0 } }, /* 0fa */
    { .mk = {      0xfb, 0 }, .brk = {                   0 } }, /* 0fb */
    { .mk = {      0xfc, 0 }, .brk = {                   0 } }, /* 0fc */
    { .mk = {      0xfd, 0 }, .brk = {                   0 } }, /* 0fd */
    { .mk = {      0xfe, 0 }, .brk = {                   0 } }, /* 0fe */
    { .mk = {      0xff, 0 }, .brk = {                   0 } }, /* 0ff */
    { .mk = {0xe1, 0x1d, 0 }, .brk = { 0xe1,       0x9d, 0 } }, /* 100 */
    { .mk = {0xe0, 0x01, 0 }, .brk = { 0xe0,       0x81, 0 } }, /* 101 */
    { .mk = {0xe0, 0x02, 0 }, .brk = { 0xe0,       0x82, 0 } }, /* 102 */
    { .mk = {0xe0, 0x03, 0 }, .brk = { 0xe0,       0x83, 0 } }, /* 103 */
    { .mk = {0xe0, 0x04, 0 }, .brk = { 0xe0,       0x84, 0 } }, /* 104 */
    { .mk = {0xe0, 0x05, 0 }, .brk = { 0xe0,       0x85, 0 } }, /* 105 */
    { .mk = {0xe0, 0x06, 0 }, .brk = { 0xe0,       0x86, 0 } }, /* 106 */
    { .mk = {0xe0, 0x07, 0 }, .brk = { 0xe0,       0x87, 0 } }, /* 107 */
    { .mk = {0xe0, 0x08, 0 }, .brk = { 0xe0,       0x88, 0 } }, /* 108 */
    { .mk = {0xe0, 0x09, 0 }, .brk = { 0xe0,       0x89, 0 } }, /* 109 */
    { .mk = {0xe0, 0x0a, 0 }, .brk = { 0xe0,       0x8a, 0 } }, /* 10a */
    { .mk = {0xe0, 0x0b, 0 }, .brk = { 0xe0,       0x8b, 0 } }, /* 10b */
    { .mk = {0xe0, 0x0c, 0 }, .brk = { 0xe0,       0x8c, 0 } }, /* 10c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 10d */
    { .mk = {0xe0, 0x0e, 0 }, .brk = { 0xe0,       0x8e, 0 } }, /* 10e */
    { .mk = {0xe0, 0x0f, 0 }, .brk = { 0xe0,       0x8f, 0 } }, /* 10f */
    { .mk = {0xe0, 0x10, 0 }, .brk = { 0xe0,       0x90, 0 } }, /* 110 */
    { .mk = {0xe0, 0x11, 0 }, .brk = { 0xe0,       0x91, 0 } }, /* 111 */
    { .mk = {0xe0, 0x12, 0 }, .brk = { 0xe0,       0x92, 0 } }, /* 112 */
    { .mk = {0xe0, 0x13, 0 }, .brk = { 0xe0,       0x93, 0 } }, /* 113 */
    { .mk = {0xe0, 0x14, 0 }, .brk = { 0xe0,       0x94, 0 } }, /* 114 */
    { .mk = {0xe0, 0x15, 0 }, .brk = { 0xe0,       0x95, 0 } }, /* 115 */
    { .mk = {0xe0, 0x16, 0 }, .brk = { 0xe0,       0x96, 0 } }, /* 116 */
    { .mk = {0xe0, 0x17, 0 }, .brk = { 0xe0,       0x97, 0 } }, /* 117 */
    { .mk = {0xe0, 0x18, 0 }, .brk = { 0xe0,       0x98, 0 } }, /* 118 */
    { .mk = {0xe0, 0x19, 0 }, .brk = { 0xe0,       0x99, 0 } }, /* 119 */
    { .mk = {0xe0, 0x1a, 0 }, .brk = { 0xe0,       0x9a, 0 } }, /* 11a */
    { .mk = {0xe0, 0x1b, 0 }, .brk = { 0xe0,       0x9b, 0 } }, /* 11b */
    { .mk = {0xe0, 0x1c, 0 }, .brk = { 0xe0,       0x9c, 0 } }, /* 11c */
    { .mk = {0xe0, 0x1d, 0 }, .brk = { 0xe0,       0x9d, 0 } }, /* 11d */
    { .mk = {0xe0, 0x1e, 0 }, .brk = { 0xe0,       0x9e, 0 } }, /* 11e */
    { .mk = {0xe0, 0x1f, 0 }, .brk = { 0xe0,       0x9f, 0 } }, /* 11f */
    { .mk = {0xe0, 0x20, 0 }, .brk = { 0xe0,       0xa0, 0 } }, /* 120 */
    { .mk = {0xe0, 0x21, 0 }, .brk = { 0xe0,       0xa1, 0 } }, /* 121 */
    { .mk = {0xe0, 0x22, 0 }, .brk = { 0xe0,       0xa2, 0 } }, /* 122 */
    { .mk = {0xe0, 0x23, 0 }, .brk = { 0xe0,       0xa3, 0 } }, /* 123 */
    { .mk = {0xe0, 0x24, 0 }, .brk = { 0xe0,       0xa4, 0 } }, /* 124 */
    { .mk = {0xe0, 0x25, 0 }, .brk = { 0xe0,       0xa5, 0 } }, /* 125 */
    { .mk = {0xe0, 0x26, 0 }, .brk = { 0xe0,       0xa6, 0 } }, /* 126 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 127 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 128 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 129 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 12a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 12b */
    { .mk = {0xe0, 0x2c, 0 }, .brk = { 0xe0,       0xac, 0 } }, /* 12c */
    { .mk = {0xe0, 0x2d, 0 }, .brk = { 0xe0,       0xad, 0 } }, /* 12d */
    { .mk = {0xe0, 0x2e, 0 }, .brk = { 0xe0,       0xae, 0 } }, /* 12e */
    { .mk = {0xe0, 0x2f, 0 }, .brk = { 0xe0,       0xaf, 0 } }, /* 12f */
    { .mk = {0xe0, 0x30, 0 }, .brk = { 0xe0,       0xb0, 0 } }, /* 130 */
    { .mk = {0xe0, 0x31, 0 }, .brk = { 0xe0,       0xb1, 0 } }, /* 131 */
    { .mk = {0xe0, 0x32, 0 }, .brk = { 0xe0,       0xb2, 0 } }, /* 132 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 133 */
    { .mk = {0xe0, 0x34, 0 }, .brk = { 0xe0,       0xb4, 0 } }, /* 134 */
    { .mk = {0xe0, 0x35, 0 }, .brk = { 0xe0,       0xb5, 0 } }, /* 135 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 136 */
    { .mk = {0xe0, 0x37, 0 }, .brk = { 0xe0,       0xb7, 0 } }, /* 137 */
    { .mk = {0xe0, 0x38, 0 }, .brk = { 0xe0,       0xb8, 0 } }, /* 138 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 139 */
    { .mk = {0xe0, 0x3a, 0 }, .brk = { 0xe0,       0xba, 0 } }, /* 13a */
    { .mk = {0xe0, 0x3b, 0 }, .brk = { 0xe0,       0xbb, 0 } }, /* 13b */
    { .mk = {0xe0, 0x3c, 0 }, .brk = { 0xe0,       0xbc, 0 } }, /* 13c */
    { .mk = {0xe0, 0x3d, 0 }, .brk = { 0xe0,       0xbd, 0 } }, /* 13d */
    { .mk = {0xe0, 0x3e, 0 }, .brk = { 0xe0,       0xbe, 0 } }, /* 13e */
    { .mk = {0xe0, 0x3f, 0 }, .brk = { 0xe0,       0xbf, 0 } }, /* 13f */
    { .mk = {0xe0, 0x40, 0 }, .brk = { 0xe0,       0xc0, 0 } }, /* 140 */
    { .mk = {0xe0, 0x41, 0 }, .brk = { 0xe0,       0xc1, 0 } }, /* 141 */
    { .mk = {0xe0, 0x42, 0 }, .brk = { 0xe0,       0xc2, 0 } }, /* 142 */
    { .mk = {0xe0, 0x43, 0 }, .brk = { 0xe0,       0xc3, 0 } }, /* 143 */
    { .mk = {0xe0, 0x44, 0 }, .brk = { 0xe0,       0xc4, 0 } }, /* 144 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 145 */
    { .mk = {0xe0, 0x46, 0 }, .brk = { 0xe0,       0xc6, 0 } }, /* 146 */
    { .mk = {0xe0, 0x47, 0 }, .brk = { 0xe0,       0xc7, 0 } }, /* 147 */
    { .mk = {0xe0, 0x48, 0 }, .brk = { 0xe0,       0xc8, 0 } }, /* 148 */
    { .mk = {0xe0, 0x49, 0 }, .brk = { 0xe0,       0xc9, 0 } }, /* 149 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 14a */
    { .mk = {0xe0, 0x4b, 0 }, .brk = { 0xe0,       0xcb, 0 } }, /* 14b */
    { .mk = {0xe0, 0x4c, 0 }, .brk = { 0xe0,       0xcc, 0 } }, /* 14c */
    { .mk = {0xe0, 0x4d, 0 }, .brk = { 0xe0,       0xcd, 0 } }, /* 14d */
    { .mk = {0xe0, 0x4e, 0 }, .brk = { 0xe0,       0xce, 0 } }, /* 14e */
    { .mk = {0xe0, 0x4f, 0 }, .brk = { 0xe0,       0xcf, 0 } }, /* 14f */
    { .mk = {0xe0, 0x50, 0 }, .brk = { 0xe0,       0xd0, 0 } }, /* 150 */
    { .mk = {0xe0, 0x51, 0 }, .brk = { 0xe0,       0xd1, 0 } }, /* 151 */
    { .mk = {0xe0, 0x52, 0 }, .brk = { 0xe0,       0xd2, 0 } }, /* 152 */
    { .mk = {0xe0, 0x53, 0 }, .brk = { 0xe0,       0xd3, 0 } }, /* 153 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 154 */
    { .mk = {0xe0, 0x55, 0 }, .brk = { 0xe0,       0xd5, 0 } }, /* 155 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 156 */
    { .mk = {0xe0, 0x57, 0 }, .brk = { 0xe0,       0xd7, 0 } }, /* 157 */
    { .mk = {0xe0, 0x58, 0 }, .brk = { 0xe0,       0xd8, 0 } }, /* 158 */
    { .mk = {0xe0, 0x59, 0 }, .brk = { 0xe0,       0xd9, 0 } }, /* 159 */
    { .mk = {0xe0, 0x5a, 0 }, .brk = { 0xe0,       0xaa, 0 } }, /* 15a */
    { .mk = {0xe0, 0x5b, 0 }, .brk = { 0xe0,       0xdb, 0 } }, /* 15b */
    { .mk = {0xe0, 0x5c, 0 }, .brk = { 0xe0,       0xdc, 0 } }, /* 15c */
    { .mk = {0xe0, 0x5d, 0 }, .brk = { 0xe0,       0xdd, 0 } }, /* 15d */
    { .mk = {0xe0, 0x5e, 0 }, .brk = { 0xe0,       0xee, 0 } }, /* 15e */
    { .mk = {0xe0, 0x5f, 0 }, .brk = { 0xe0,       0xdf, 0 } }, /* 15f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 160 */
    { .mk = {0xe0, 0x61, 0 }, .brk = { 0xe0,       0xe1, 0 } }, /* 161 */
    { .mk = {0xe0, 0x62, 0 }, .brk = { 0xe0,       0xe2, 0 } }, /* 162 */
    { .mk = {0xe0, 0x63, 0 }, .brk = { 0xe0,       0xe3, 0 } }, /* 163 */
    { .mk = {0xe0, 0x64, 0 }, .brk = { 0xe0,       0xe4, 0 } }, /* 164 */
    { .mk = {0xe0, 0x65, 0 }, .brk = { 0xe0,       0xe5, 0 } }, /* 165 */
    { .mk = {0xe0, 0x66, 0 }, .brk = { 0xe0,       0xe6, 0 } }, /* 166 */
    { .mk = {0xe0, 0x67, 0 }, .brk = { 0xe0,       0xe7, 0 } }, /* 167 */
    { .mk = {0xe0, 0x68, 0 }, .brk = { 0xe0,       0xe8, 0 } }, /* 168 */
    { .mk = {0xe0, 0x69, 0 }, .brk = { 0xe0,       0xe9, 0 } }, /* 169 */
    { .mk = {0xe0, 0x6a, 0 }, .brk = { 0xe0,       0xea, 0 } }, /* 16a */
    { .mk = {0xe0, 0x6b, 0 }, .brk = { 0xe0,       0xeb, 0 } }, /* 16b */
    { .mk = {0xe0, 0x6c, 0 }, .brk = { 0xe0,       0xec, 0 } }, /* 16c */
    { .mk = {0xe0, 0x6d, 0 }, .brk = { 0xe0,       0xed, 0 } }, /* 16d */
    { .mk = {0xe0, 0x6e, 0 }, .brk = { 0xe0,       0xee, 0 } }, /* 16e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 16f */
    { .mk = {0xe0, 0x70, 0 }, .brk = { 0xe0,       0xf0, 0 } }, /* 170 */
    { .mk = {0xe0, 0x71, 0 }, .brk = { 0xe0,       0xf1, 0 } }, /* 171 */
    { .mk = {0xe0, 0x72, 0 }, .brk = { 0xe0,       0xf2, 0 } }, /* 172 */
    { .mk = {0xe0, 0x73, 0 }, .brk = { 0xe0,       0xf3, 0 } }, /* 173 */
    { .mk = {0xe0, 0x74, 0 }, .brk = { 0xe0,       0xf4, 0 } }, /* 174 */
    { .mk = {0xe0, 0x75, 0 }, .brk = { 0xe0,       0xf5, 0 } }, /* 175 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 176 */
    { .mk = {0xe0, 0x77, 0 }, .brk = { 0xe0,       0xf7, 0 } }, /* 177 */
    { .mk = {0xe0, 0x78, 0 }, .brk = { 0xe0,       0xf8, 0 } }, /* 178 */
    { .mk = {0xe0, 0x79, 0 }, .brk = { 0xe0,       0xf9, 0 } }, /* 179 */
    { .mk = {0xe0, 0x7a, 0 }, .brk = { 0xe0,       0xfa, 0 } }, /* 17a */
    { .mk = {0xe0, 0x7b, 0 }, .brk = { 0xe0,       0xfb, 0 } }, /* 17b */
    { .mk = {0xe0, 0x7c, 0 }, .brk = { 0xe0,       0xfc, 0 } }, /* 17c */
    { .mk = {0xe0, 0x7d, 0 }, .brk = { 0xe0,       0xfd, 0 } }, /* 17d */
    { .mk = {0xe0, 0x7e, 0 }, .brk = { 0xe0,       0xfe, 0 } }, /* 17e */
    { .mk = {0xe0, 0x7f, 0 }, .brk = { 0xe0,       0xff, 0 } }, /* 17f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 180 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 181 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 182 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 183 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 184 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 185 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 186 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 187 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 188 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 189 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 190 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 191 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 192 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 193 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 194 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 195 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 196 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 197 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 198 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 199 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1aa */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ab */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ac */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ad */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ae */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1af */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ba */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1bb */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1bc */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1bd */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1be */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1bf */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ca */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1cb */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1cc */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1cd */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ce */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1cf */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1da */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1db */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1dc */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1dd */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1de */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1df */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e0 */
    { .mk = {0xe0, 0xe1, 0 }, .brk = {                   0 } }, /* 1e1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ea */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1eb */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ec */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ed */
    { .mk = {0xe0, 0xee, 0 }, .brk = {                   0 } }, /* 1ee */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ef */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f0 */
    { .mk = {0xe0, 0xf1, 0 }, .brk = {                   0 } }, /* 1f1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1fa */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1fb */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1fc */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1fd */
    { .mk = {0xe0, 0xfe, 0 }, .brk = {                   0 } }, /* 1fe */
    { .mk = {0xe0, 0xff, 0 }, .brk = {                   0 } }  /* 1ff */
  // clang-format on
};

static const scancode scancode_set2[512] = {
  // clang-format off
    { .mk = {            0 }, .brk = {                   0 } }, /* 000 */
    { .mk = {      0x76, 0 }, .brk = {       0xF0, 0x76, 0 } }, /* 001 */
    { .mk = {      0x16, 0 }, .brk = {       0xF0, 0x16, 0 } }, /* 002 */
    { .mk = {      0x1E, 0 }, .brk = {       0xF0, 0x1E, 0 } }, /* 003 */
    { .mk = {      0x26, 0 }, .brk = {       0xF0, 0x26, 0 } }, /* 004 */
    { .mk = {      0x25, 0 }, .brk = {       0xF0, 0x25, 0 } }, /* 005 */
    { .mk = {      0x2E, 0 }, .brk = {       0xF0, 0x2E, 0 } }, /* 006 */
    { .mk = {      0x36, 0 }, .brk = {       0xF0, 0x36, 0 } }, /* 007 */
    { .mk = {      0x3D, 0 }, .brk = {       0xF0, 0x3D, 0 } }, /* 008 */
    { .mk = {      0x3E, 0 }, .brk = {       0xF0, 0x3E, 0 } }, /* 009 */
    { .mk = {      0x46, 0 }, .brk = {       0xF0, 0x46, 0 } }, /* 00a */
    { .mk = {      0x45, 0 }, .brk = {       0xF0, 0x45, 0 } }, /* 00b */
    { .mk = {      0x4E, 0 }, .brk = {       0xF0, 0x4E, 0 } }, /* 00c */
    { .mk = {      0x55, 0 }, .brk = {       0xF0, 0x55, 0 } }, /* 00d */
    { .mk = {      0x66, 0 }, .brk = {       0xF0, 0x66, 0 } }, /* 00e */
    { .mk = {      0x0D, 0 }, .brk = {       0xF0, 0x0D, 0 } }, /* 00f */
    { .mk = {      0x15, 0 }, .brk = {       0xF0, 0x15, 0 } }, /* 010 */
    { .mk = {      0x1D, 0 }, .brk = {       0xF0, 0x1D, 0 } }, /* 011 */
    { .mk = {      0x24, 0 }, .brk = {       0xF0, 0x24, 0 } }, /* 012 */
    { .mk = {      0x2D, 0 }, .brk = {       0xF0, 0x2D, 0 } }, /* 013 */
    { .mk = {      0x2C, 0 }, .brk = {       0xF0, 0x2C, 0 } }, /* 014 */
    { .mk = {      0x35, 0 }, .brk = {       0xF0, 0x35, 0 } }, /* 015 */
    { .mk = {      0x3C, 0 }, .brk = {       0xF0, 0x3C, 0 } }, /* 016 */
    { .mk = {      0x43, 0 }, .brk = {       0xF0, 0x43, 0 } }, /* 017 */
    { .mk = {      0x44, 0 }, .brk = {       0xF0, 0x44, 0 } }, /* 018 */
    { .mk = {      0x4D, 0 }, .brk = {       0xF0, 0x4D, 0 } }, /* 019 */
    { .mk = {      0x54, 0 }, .brk = {       0xF0, 0x54, 0 } }, /* 01a */
    { .mk = {      0x5B, 0 }, .brk = {       0xF0, 0x5B, 0 } }, /* 01b */
    { .mk = {      0x5A, 0 }, .brk = {       0xF0, 0x5A, 0 } }, /* 01c */
    { .mk = {      0x14, 0 }, .brk = {       0xF0, 0x14, 0 } }, /* 01d */
    { .mk = {      0x1C, 0 }, .brk = {       0xF0, 0x1C, 0 } }, /* 01e */
    { .mk = {      0x1B, 0 }, .brk = {       0xF0, 0x1B, 0 } }, /* 01f */
    { .mk = {      0x23, 0 }, .brk = {       0xF0, 0x23, 0 } }, /* 020 */
    { .mk = {      0x2B, 0 }, .brk = {       0xF0, 0x2B, 0 } }, /* 021 */
    { .mk = {      0x34, 0 }, .brk = {       0xF0, 0x34, 0 } }, /* 022 */
    { .mk = {      0x33, 0 }, .brk = {       0xF0, 0x33, 0 } }, /* 023 */
    { .mk = {      0x3B, 0 }, .brk = {       0xF0, 0x3B, 0 } }, /* 024 */
    { .mk = {      0x42, 0 }, .brk = {       0xF0, 0x42, 0 } }, /* 025 */
    { .mk = {      0x4B, 0 }, .brk = {       0xF0, 0x4B, 0 } }, /* 026 */
    { .mk = {      0x4C, 0 }, .brk = {       0xF0, 0x4C, 0 } }, /* 027 */
    { .mk = {      0x52, 0 }, .brk = {       0xF0, 0x52, 0 } }, /* 028 */
    { .mk = {      0x0E, 0 }, .brk = {       0xF0, 0x0E, 0 } }, /* 029 */
    { .mk = {      0x12, 0 }, .brk = {       0xF0, 0x12, 0 } }, /* 02a */
    { .mk = {      0x5D, 0 }, .brk = {       0xF0, 0x5D, 0 } }, /* 02b */
    { .mk = {      0x1A, 0 }, .brk = {       0xF0, 0x1A, 0 } }, /* 02c */
    { .mk = {      0x22, 0 }, .brk = {       0xF0, 0x22, 0 } }, /* 02d */
    { .mk = {      0x21, 0 }, .brk = {       0xF0, 0x21, 0 } }, /* 02e */
    { .mk = {      0x2A, 0 }, .brk = {       0xF0, 0x2A, 0 } }, /* 02f */
    { .mk = {      0x32, 0 }, .brk = {       0xF0, 0x32, 0 } }, /* 030 */
    { .mk = {      0x31, 0 }, .brk = {       0xF0, 0x31, 0 } }, /* 031 */
    { .mk = {      0x3A, 0 }, .brk = {       0xF0, 0x3A, 0 } }, /* 032 */
    { .mk = {      0x41, 0 }, .brk = {       0xF0, 0x41, 0 } }, /* 033 */
    { .mk = {      0x49, 0 }, .brk = {       0xF0, 0x49, 0 } }, /* 034 */
    { .mk = {      0x4A, 0 }, .brk = {       0xF0, 0x4A, 0 } }, /* 035 */
    { .mk = {      0x59, 0 }, .brk = {       0xF0, 0x59, 0 } }, /* 036 */
    { .mk = {      0x7C, 0 }, .brk = {       0xF0, 0x7C, 0 } }, /* 037 */
    { .mk = {      0x11, 0 }, .brk = {       0xF0, 0x11, 0 } }, /* 038 */
    { .mk = {      0x29, 0 }, .brk = {       0xF0, 0x29, 0 } }, /* 039 */
    { .mk = {      0x58, 0 }, .brk = {       0xF0, 0x58, 0 } }, /* 03a */
    { .mk = {      0x05, 0 }, .brk = {       0xF0, 0x05, 0 } }, /* 03b */
    { .mk = {      0x06, 0 }, .brk = {       0xF0, 0x06, 0 } }, /* 03c */
    { .mk = {      0x04, 0 }, .brk = {       0xF0, 0x04, 0 } }, /* 03d */
    { .mk = {      0x0C, 0 }, .brk = {       0xF0, 0x0C, 0 } }, /* 03e */
    { .mk = {      0x03, 0 }, .brk = {       0xF0, 0x03, 0 } }, /* 03f */
    { .mk = {      0x0B, 0 }, .brk = {       0xF0, 0x0B, 0 } }, /* 040 */
    { .mk = {      0x83, 0 }, .brk = {       0xF0, 0x83, 0 } }, /* 041 */
    { .mk = {      0x0A, 0 }, .brk = {       0xF0, 0x0A, 0 } }, /* 042 */
    { .mk = {      0x01, 0 }, .brk = {       0xF0, 0x01, 0 } }, /* 043 */
    { .mk = {      0x09, 0 }, .brk = {       0xF0, 0x09, 0 } }, /* 044 */
    { .mk = {      0x77, 0 }, .brk = {       0xF0, 0x77, 0 } }, /* 045 */
    { .mk = {      0x7E, 0 }, .brk = {       0xF0, 0x7E, 0 } }, /* 046 */
    { .mk = {      0x6C, 0 }, .brk = {       0xF0, 0x6C, 0 } }, /* 047 */
    { .mk = {      0x75, 0 }, .brk = {       0xF0, 0x75, 0 } }, /* 048 */
    { .mk = {      0x7D, 0 }, .brk = {       0xF0, 0x7D, 0 } }, /* 049 */
    { .mk = {      0x7B, 0 }, .brk = {       0xF0, 0x7B, 0 } }, /* 04a */
    { .mk = {      0x6B, 0 }, .brk = {       0xF0, 0x6B, 0 } }, /* 04b */
    { .mk = {      0x73, 0 }, .brk = {       0xF0, 0x73, 0 } }, /* 04c */
    { .mk = {      0x74, 0 }, .brk = {       0xF0, 0x74, 0 } }, /* 04d */
    { .mk = {      0x79, 0 }, .brk = {       0xF0, 0x79, 0 } }, /* 04e */
    { .mk = {      0x69, 0 }, .brk = {       0xF0, 0x69, 0 } }, /* 04f */
    { .mk = {      0x72, 0 }, .brk = {       0xF0, 0x72, 0 } }, /* 050 */
    { .mk = {      0x7A, 0 }, .brk = {       0xF0, 0x7A, 0 } }, /* 051 */
    { .mk = {      0x70, 0 }, .brk = {       0xF0, 0x70, 0 } }, /* 052 */
    { .mk = {      0x71, 0 }, .brk = {       0xF0, 0x71, 0 } }, /* 053 */
    { .mk = {      0x84, 0 }, .brk = {       0xF0, 0x84, 0 } }, /* 054 */
    { .mk = {      0x60, 0 }, .brk = {       0xF0, 0x60, 0 } }, /* 055 */
    { .mk = {      0x61, 0 }, .brk = {       0xF0, 0x61, 0 } }, /* 056 */
    { .mk = {      0x78, 0 }, .brk = {       0xF0, 0x78, 0 } }, /* 057 */
    { .mk = {      0x07, 0 }, .brk = {       0xF0, 0x07, 0 } }, /* 058 */
    { .mk = {      0x0F, 0 }, .brk = {       0xF0, 0x0F, 0 } }, /* 059 */
    { .mk = {      0x17, 0 }, .brk = {       0xF0, 0x17, 0 } }, /* 05a */
    { .mk = {      0x1F, 0 }, .brk = {       0xF0, 0x1F, 0 } }, /* 05b */
    { .mk = {      0x27, 0 }, .brk = {       0xF0, 0x27, 0 } }, /* 05c */
    { .mk = {      0x2F, 0 }, .brk = {       0xF0, 0x2F, 0 } }, /* 05d */
    { .mk = {      0x37, 0 }, .brk = {       0xF0, 0x37, 0 } }, /* 05e */
    { .mk = {      0x3F, 0 }, .brk = {       0xF0, 0x3F, 0 } }, /* 05f */
    { .mk = {      0x47, 0 }, .brk = {       0xF0, 0x47, 0 } }, /* 060 */
    { .mk = {      0x4F, 0 }, .brk = {       0xF0, 0x4F, 0 } }, /* 061 */
    { .mk = {      0x56, 0 }, .brk = {       0xF0, 0x56, 0 } }, /* 062 */
    { .mk = {      0x5E, 0 }, .brk = {       0xF0, 0x5E, 0 } }, /* 063 */
    { .mk = {      0x08, 0 }, .brk = {       0xF0, 0x08, 0 } }, /* 064 */
    { .mk = {      0x10, 0 }, .brk = {       0xF0, 0x10, 0 } }, /* 065 */
    { .mk = {      0x18, 0 }, .brk = {       0xF0, 0x18, 0 } }, /* 066 */
    { .mk = {      0x20, 0 }, .brk = {       0xF0, 0x20, 0 } }, /* 067 */
    { .mk = {      0x28, 0 }, .brk = {       0xF0, 0x28, 0 } }, /* 068 */
    { .mk = {      0x30, 0 }, .brk = {       0xF0, 0x30, 0 } }, /* 069 */
    { .mk = {      0x38, 0 }, .brk = {       0xF0, 0x38, 0 } }, /* 06a */
    { .mk = {      0x40, 0 }, .brk = {       0xF0, 0x40, 0 } }, /* 06b */
    { .mk = {      0x48, 0 }, .brk = {       0xF0, 0x48, 0 } }, /* 06c */
    { .mk = {      0x50, 0 }, .brk = {       0xF0, 0x50, 0 } }, /* 06d */
    { .mk = {      0x57, 0 }, .brk = {       0xF0, 0x57, 0 } }, /* 06e */
    { .mk = {      0x6F, 0 }, .brk = {       0xF0, 0x6F, 0 } }, /* 06f */
    { .mk = {      0x13, 0 }, .brk = {       0xF0, 0x13, 0 } }, /* 070 */
    { .mk = {      0x19, 0 }, .brk = {       0xF0, 0x19, 0 } }, /* 071 */
    { .mk = {      0x39, 0 }, .brk = {       0xF0, 0x39, 0 } }, /* 072 */
    { .mk = {      0x51, 0 }, .brk = {       0xF0, 0x51, 0 } }, /* 073 */
    { .mk = {      0x53, 0 }, .brk = {       0xF0, 0x53, 0 } }, /* 074 */
    { .mk = {      0x5C, 0 }, .brk = {       0xF0, 0x5C, 0 } }, /* 075 */
    { .mk = {      0x5F, 0 }, .brk = {       0xF0, 0x5F, 0 } }, /* 076 */
    { .mk = {      0x62, 0 }, .brk = {       0xF0, 0x62, 0 } }, /* 077 */
    { .mk = {      0x63, 0 }, .brk = {       0xF0, 0x63, 0 } }, /* 078 */
    { .mk = {      0x64, 0 }, .brk = {       0xF0, 0x64, 0 } }, /* 079 */
    { .mk = {      0x65, 0 }, .brk = {       0xF0, 0x65, 0 } }, /* 07a */
    { .mk = {      0x67, 0 }, .brk = {       0xF0, 0x67, 0 } }, /* 07b */
    { .mk = {      0x68, 0 }, .brk = {       0xF0, 0x68, 0 } }, /* 07c */
    { .mk = {      0x6A, 0 }, .brk = {       0xF0, 0x6A, 0 } }, /* 07d */
    { .mk = {      0x6D, 0 }, .brk = {       0xF0, 0x6D, 0 } }, /* 07e */
    { .mk = {      0x6E, 0 }, .brk = {       0xF0, 0x6E, 0 } }, /* 07f */
    { .mk = {      0x80, 0 }, .brk = {       0xf0, 0x80, 0 } }, /* 080 */
    { .mk = {      0x81, 0 }, .brk = {       0xf0, 0x81, 0 } }, /* 081 */
    { .mk = {      0x82, 0 }, .brk = {       0xf0, 0x82, 0 } }, /* 082 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 083 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 084 */
    { .mk = {      0x85, 0 }, .brk = {       0xf0, 0x54, 0 } }, /* 085 */
    { .mk = {      0x86, 0 }, .brk = {       0xf0, 0x86, 0 } }, /* 086 */
    { .mk = {      0x87, 0 }, .brk = {       0xf0, 0x87, 0 } }, /* 087 */
    { .mk = {      0x88, 0 }, .brk = {       0xf0, 0x88, 0 } }, /* 088 */
    { .mk = {      0x89, 0 }, .brk = {       0xf0, 0x89, 0 } }, /* 089 */
    { .mk = {      0x8a, 0 }, .brk = {       0xf0, 0x8a, 0 } }, /* 08a */
    { .mk = {      0x8b, 0 }, .brk = {       0xf0, 0x8b, 0 } }, /* 08b */
    { .mk = {      0x8c, 0 }, .brk = {       0xf0, 0x8c, 0 } }, /* 08c */
    { .mk = {      0x8d, 0 }, .brk = {       0xf0, 0x8d, 0 } }, /* 08d */
    { .mk = {      0x8e, 0 }, .brk = {       0xf0, 0x8e, 0 } }, /* 08e */
    { .mk = {      0x8f, 0 }, .brk = {       0xf0, 0x8f, 0 } }, /* 08f */
    { .mk = {      0x90, 0 }, .brk = {       0xf0, 0x90, 0 } }, /* 090 */
    { .mk = {      0x91, 0 }, .brk = {       0xf0, 0x91, 0 } }, /* 091 */
    { .mk = {      0x92, 0 }, .brk = {       0xf0, 0x92, 0 } }, /* 092 */
    { .mk = {      0x93, 0 }, .brk = {       0xf0, 0x93, 0 } }, /* 093 */
    { .mk = {      0x94, 0 }, .brk = {       0xf0, 0x94, 0 } }, /* 094 */
    { .mk = {      0x95, 0 }, .brk = {       0xf0, 0x95, 0 } }, /* 095 */
    { .mk = {      0x96, 0 }, .brk = {       0xf0, 0x96, 0 } }, /* 096 */
    { .mk = {      0x97, 0 }, .brk = {       0xf0, 0x97, 0 } }, /* 097 */
    { .mk = {      0x98, 0 }, .brk = {       0xf0, 0x98, 0 } }, /* 098 */
    { .mk = {      0x99, 0 }, .brk = {       0xf0, 0x99, 0 } }, /* 099 */
    { .mk = {      0x9a, 0 }, .brk = {       0xf0, 0x9a, 0 } }, /* 09a */
    { .mk = {      0x9b, 0 }, .brk = {       0xf0, 0x9b, 0 } }, /* 09b */
    { .mk = {      0x9c, 0 }, .brk = {       0xf0, 0x9c, 0 } }, /* 09c */
    { .mk = {      0x9d, 0 }, .brk = {       0xf0, 0x9d, 0 } }, /* 09d */
    { .mk = {      0x9e, 0 }, .brk = {       0xf0, 0x9e, 0 } }, /* 09e */
    { .mk = {      0x9f, 0 }, .brk = {       0xf0, 0x9f, 0 } }, /* 09f */
    { .mk = {      0xa0, 0 }, .brk = {       0xf0, 0xa0, 0 } }, /* 0a0 */
    { .mk = {      0xa1, 0 }, .brk = {       0xf0, 0xa1, 0 } }, /* 0a1 */
    { .mk = {      0xa2, 0 }, .brk = {       0xf0, 0xa2, 0 } }, /* 0a2 */
    { .mk = {      0xa3, 0 }, .brk = {       0xf0, 0xa3, 0 } }, /* 0a3 */
    { .mk = {      0xa4, 0 }, .brk = {       0xf0, 0xa4, 0 } }, /* 0a4 */
    { .mk = {      0xa5, 0 }, .brk = {       0xf0, 0xa5, 0 } }, /* 0a5 */
    { .mk = {      0xa6, 0 }, .brk = {       0xf0, 0xa6, 0 } }, /* 0a6 */
    { .mk = {      0xa7, 0 }, .brk = {       0xf0, 0xa7, 0 } }, /* 0a7 */
    { .mk = {      0xa8, 0 }, .brk = {       0xf0, 0xa8, 0 } }, /* 0a8 */
    { .mk = {      0xa9, 0 }, .brk = {       0xf0, 0xa9, 0 } }, /* 0a9 */
    { .mk = {      0xaa, 0 }, .brk = {       0xf0, 0xaa, 0 } }, /* 0aa */
    { .mk = {      0xab, 0 }, .brk = {       0xf0, 0xab, 0 } }, /* 0ab */
    { .mk = {      0xac, 0 }, .brk = {       0xf0, 0xac, 0 } }, /* 0ac */
    { .mk = {      0xad, 0 }, .brk = {       0xf0, 0xad, 0 } }, /* 0ad */
    { .mk = {      0xae, 0 }, .brk = {       0xf0, 0xae, 0 } }, /* 0ae */
    { .mk = {      0xaf, 0 }, .brk = {       0xf0, 0xaf, 0 } }, /* 0af */
    { .mk = {      0xb0, 0 }, .brk = {       0xf0, 0xb0, 0 } }, /* 0b0 */
    { .mk = {      0xb1, 0 }, .brk = {       0xf0, 0xb1, 0 } }, /* 0b1 */
    { .mk = {      0xb2, 0 }, .brk = {       0xf0, 0xb2, 0 } }, /* 0b2 */
    { .mk = {      0xb3, 0 }, .brk = {       0xf0, 0xb3, 0 } }, /* 0b3 */
    { .mk = {      0xb4, 0 }, .brk = {       0xf0, 0xb4, 0 } }, /* 0b4 */
    { .mk = {      0xb5, 0 }, .brk = {       0xf0, 0xb5, 0 } }, /* 0b5 */
    { .mk = {      0xb6, 0 }, .brk = {       0xf0, 0xb6, 0 } }, /* 0b6 */
    { .mk = {      0xb7, 0 }, .brk = {       0xf0, 0xb7, 0 } }, /* 0b7 */
    { .mk = {      0xb8, 0 }, .brk = {       0xf0, 0xb8, 0 } }, /* 0b8 */
    { .mk = {      0xb9, 0 }, .brk = {       0xf0, 0xb9, 0 } }, /* 0b9 */
    { .mk = {      0xba, 0 }, .brk = {       0xf0, 0xba, 0 } }, /* 0ba */
    { .mk = {      0xbb, 0 }, .brk = {       0xf0, 0xbb, 0 } }, /* 0bb */
    { .mk = {      0xbc, 0 }, .brk = {       0xf0, 0xbc, 0 } }, /* 0bc */
    { .mk = {      0xbd, 0 }, .brk = {       0xf0, 0xbd, 0 } }, /* 0bd */
    { .mk = {      0xbe, 0 }, .brk = {       0xf0, 0xbe, 0 } }, /* 0be */
    { .mk = {      0xbf, 0 }, .brk = {       0xf0, 0xbf, 0 } }, /* 0bf */
    { .mk = {      0xc0, 0 }, .brk = {       0xf0, 0xc0, 0 } }, /* 0c0 */
    { .mk = {      0xc1, 0 }, .brk = {       0xf0, 0xc1, 0 } }, /* 0c1 */
    { .mk = {      0xc2, 0 }, .brk = {       0xf0, 0xc2, 0 } }, /* 0c2 */
    { .mk = {      0xc3, 0 }, .brk = {       0xf0, 0xc3, 0 } }, /* 0c3 */
    { .mk = {      0xc4, 0 }, .brk = {       0xf0, 0xc4, 0 } }, /* 0c4 */
    { .mk = {      0xc5, 0 }, .brk = {       0xf0, 0xc5, 0 } }, /* 0c5 */
    { .mk = {      0xc6, 0 }, .brk = {       0xf0, 0xc6, 0 } }, /* 0c6 */
    { .mk = {      0xc7, 0 }, .brk = {       0xf0, 0xc7, 0 } }, /* 0c7 */
    { .mk = {      0xc8, 0 }, .brk = {       0xf0, 0xc8, 0 } }, /* 0c8 */
    { .mk = {      0xc9, 0 }, .brk = {       0xf0, 0xc9, 0 } }, /* 0c9 */
    { .mk = {      0xca, 0 }, .brk = {       0xf0, 0xca, 0 } }, /* 0ca */
    { .mk = {      0xcb, 0 }, .brk = {       0xf0, 0xcb, 0 } }, /* 0cb */
    { .mk = {      0xcc, 0 }, .brk = {       0xf0, 0xcc, 0 } }, /* 0cc */
    { .mk = {      0xcd, 0 }, .brk = {       0xf0, 0xcd, 0 } }, /* 0cd */
    { .mk = {      0xce, 0 }, .brk = {       0xf0, 0xce, 0 } }, /* 0ce */
    { .mk = {      0xcf, 0 }, .brk = {       0xf0, 0xcf, 0 } }, /* 0cf */
    { .mk = {      0xd0, 0 }, .brk = {       0xf0, 0xd0, 0 } }, /* 0d0 */
    { .mk = {      0xd1, 0 }, .brk = {       0xf0, 0xd0, 0 } }, /* 0d1 */
    { .mk = {      0xd2, 0 }, .brk = {       0xf0, 0xd2, 0 } }, /* 0d2 */
    { .mk = {      0xd3, 0 }, .brk = {       0xf0, 0xd3, 0 } }, /* 0d3 */
    { .mk = {      0xd4, 0 }, .brk = {       0xf0, 0xd4, 0 } }, /* 0d4 */
    { .mk = {      0xd5, 0 }, .brk = {       0xf0, 0xd5, 0 } }, /* 0d5 */
    { .mk = {      0xd6, 0 }, .brk = {       0xf0, 0xd6, 0 } }, /* 0d6 */
    { .mk = {      0xd7, 0 }, .brk = {       0xf0, 0xd7, 0 } }, /* 0d7 */
    { .mk = {      0xd8, 0 }, .brk = {       0xf0, 0xd8, 0 } }, /* 0d8 */
    { .mk = {      0xd9, 0 }, .brk = {       0xf0, 0xd9, 0 } }, /* 0d9 */
    { .mk = {      0xda, 0 }, .brk = {       0xf0, 0xda, 0 } }, /* 0da */
    { .mk = {      0xdb, 0 }, .brk = {       0xf0, 0xdb, 0 } }, /* 0db */
    { .mk = {      0xdc, 0 }, .brk = {       0xf0, 0xdc, 0 } }, /* 0dc */
    { .mk = {      0xdd, 0 }, .brk = {       0xf0, 0xdd, 0 } }, /* 0dd */
    { .mk = {      0xde, 0 }, .brk = {       0xf0, 0xde, 0 } }, /* 0de */
    { .mk = {      0xdf, 0 }, .brk = {       0xf0, 0xdf, 0 } }, /* 0df */
    { .mk = {      0xe0, 0 }, .brk = {       0xf0, 0xe0, 0 } }, /* 0e0 */
    { .mk = {      0xe1, 0 }, .brk = {       0xf0, 0xe1, 0 } }, /* 0e1 */
    { .mk = {      0xe2, 0 }, .brk = {       0xf0, 0xe2, 0 } }, /* 0e2 */
    { .mk = {      0xe3, 0 }, .brk = {       0xf0, 0xe3, 0 } }, /* 0e3 */
    { .mk = {      0xe4, 0 }, .brk = {       0xf0, 0xe4, 0 } }, /* 0e4 */
    { .mk = {      0xe5, 0 }, .brk = {       0xf0, 0xe5, 0 } }, /* 0e5 */
    { .mk = {      0xe6, 0 }, .brk = {       0xf0, 0xe6, 0 } }, /* 0e6 */
    { .mk = {      0xe7, 0 }, .brk = {       0xf0, 0xe7, 0 } }, /* 0e7 */
    { .mk = {      0xe8, 0 }, .brk = {       0xf0, 0xe8, 0 } }, /* 0e8 */
    { .mk = {      0xe9, 0 }, .brk = {       0xf0, 0xe9, 0 } }, /* 0e9 */
    { .mk = {      0xea, 0 }, .brk = {       0xf0, 0xea, 0 } }, /* 0ea */
    { .mk = {      0xeb, 0 }, .brk = {       0xf0, 0xeb, 0 } }, /* 0eb */
    { .mk = {      0xec, 0 }, .brk = {       0xf0, 0xec, 0 } }, /* 0ec */
    { .mk = {      0xed, 0 }, .brk = {       0xf0, 0xed, 0 } }, /* 0ed */
    { .mk = {      0xee, 0 }, .brk = {       0xf0, 0xee, 0 } }, /* 0ee */
    { .mk = {      0xef, 0 }, .brk = {       0xf0, 0xef, 0 } }, /* 0ef */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0f0 */
    { .mk = {      0xf1, 0 }, .brk = {       0xf0, 0xf1, 0 } }, /* 0f1 */
    { .mk = {      0xf2, 0 }, .brk = {       0xf0, 0xf2, 0 } }, /* 0f2 */
    { .mk = {      0xf3, 0 }, .brk = {       0xf0, 0xf3, 0 } }, /* 0f3 */
    { .mk = {      0xf4, 0 }, .brk = {       0xf0, 0xf4, 0 } }, /* 0f4 */
    { .mk = {      0xf5, 0 }, .brk = {       0xf0, 0xf5, 0 } }, /* 0f5 */
    { .mk = {      0xf6, 0 }, .brk = {       0xf0, 0xf6, 0 } }, /* 0f6 */
    { .mk = {      0xf7, 0 }, .brk = {       0xf0, 0xf7, 0 } }, /* 0f7 */
    { .mk = {      0xf8, 0 }, .brk = {       0xf0, 0xf8, 0 } }, /* 0f8 */
    { .mk = {      0xf9, 0 }, .brk = {       0xf0, 0xf9, 0 } }, /* 0f9 */
    { .mk = {      0xfa, 0 }, .brk = {       0xf0, 0xfa, 0 } }, /* 0fa */
    { .mk = {      0xfb, 0 }, .brk = {       0xf0, 0xfb, 0 } }, /* 0fb */
    { .mk = {      0xfc, 0 }, .brk = {       0xf0, 0xfc, 0 } }, /* 0fc */
    { .mk = {      0xfd, 0 }, .brk = {       0xf0, 0xfd, 0 } }, /* 0fd */
    { .mk = {      0xfe, 0 }, .brk = {       0xf0, 0xfe, 0 } }, /* 0fe */
    { .mk = {      0xff, 0 }, .brk = {       0xf0, 0xff, 0 } }, /* 0ff */
    { .mk = {0xe1, 0x14, 0 }, .brk = { 0xe1, 0xf0, 0x14, 0 } }, /* 100 */
    { .mk = {0xe0, 0x76, 0 }, .brk = { 0xe0, 0xF0, 0x76, 0 } }, /* 101 */
    { .mk = {0xe0, 0x16, 0 }, .brk = { 0xe0, 0xF0, 0x16, 0 } }, /* 102 */
    { .mk = {0xe0, 0x1E, 0 }, .brk = { 0xe0, 0xF0, 0x1E, 0 } }, /* 103 */
    { .mk = {0xe0, 0x26, 0 }, .brk = { 0xe0, 0xF0, 0x26, 0 } }, /* 104 */
    { .mk = {0xe0, 0x25, 0 }, .brk = { 0xe0, 0xF0, 0x25, 0 } }, /* 105 */
    { .mk = {0xe0, 0x2E, 0 }, .brk = { 0xe0, 0xF0, 0x2E, 0 } }, /* 106 */
    { .mk = {0xe0, 0x36, 0 }, .brk = { 0xe0, 0xF0, 0x36, 0 } }, /* 107 */
    { .mk = {0xe0, 0x3D, 0 }, .brk = { 0xe0, 0xF0, 0x3D, 0 } }, /* 108 */
    { .mk = {0xe0, 0x3E, 0 }, .brk = { 0xe0, 0xF0, 0x3E, 0 } }, /* 109 */
    { .mk = {0xe0, 0x46, 0 }, .brk = { 0xe0, 0xF0, 0x46, 0 } }, /* 10a */
    { .mk = {0xe0, 0x45, 0 }, .brk = { 0xe0, 0xF0, 0x45, 0 } }, /* 10b */
    { .mk = {0xe0, 0x4E, 0 }, .brk = { 0xe0, 0xF0, 0x4E, 0 } }, /* 10c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 10d */
    { .mk = {0xe0, 0x66, 0 }, .brk = { 0xe0, 0xF0, 0x66, 0 } }, /* 10e */
    { .mk = {0xe0, 0x0D, 0 }, .brk = { 0xe0, 0xF0, 0x0D, 0 } }, /* 10f */
    { .mk = {0xe0, 0x15, 0 }, .brk = { 0xe0, 0xF0, 0x15, 0 } }, /* 110 */
    { .mk = {0xe0, 0x1D, 0 }, .brk = { 0xe0, 0xF0, 0x1D, 0 } }, /* 112 */
    { .mk = {0xe0, 0x24, 0 }, .brk = { 0xe0, 0xF0, 0x24, 0 } }, /* 113 */
    { .mk = {0xe0, 0x2D, 0 }, .brk = { 0xe0, 0xF0, 0x2D, 0 } }, /* 113 */
    { .mk = {0xe0, 0x2C, 0 }, .brk = { 0xe0, 0xF0, 0x2C, 0 } }, /* 114 */
    { .mk = {0xe0, 0x35, 0 }, .brk = { 0xe0, 0xF0, 0x35, 0 } }, /* 115 */
    { .mk = {0xe0, 0x3C, 0 }, .brk = { 0xe0, 0xF0, 0x3C, 0 } }, /* 116 */
    { .mk = {0xe0, 0x43, 0 }, .brk = { 0xe0, 0xF0, 0x43, 0 } }, /* 117 */
    { .mk = {0xe0, 0x44, 0 }, .brk = { 0xe0, 0xF0, 0x44, 0 } }, /* 118 */
    { .mk = {0xe0, 0x4D, 0 }, .brk = { 0xe0, 0xF0, 0x4D, 0 } }, /* 119 */
    { .mk = {0xe0, 0x54, 0 }, .brk = { 0xe0, 0xF0, 0x54, 0 } }, /* 11a */
    { .mk = {0xe0, 0x5B, 0 }, .brk = { 0xe0, 0xF0, 0x5B, 0 } }, /* 11b */
    { .mk = {0xe0, 0x5A, 0 }, .brk = { 0xe0, 0xF0, 0x5A, 0 } }, /* 11c */
    { .mk = {0xe0, 0x14, 0 }, .brk = { 0xe0, 0xF0, 0x14, 0 } }, /* 11d */
    { .mk = {0xe0, 0x1C, 0 }, .brk = { 0xe0, 0xF0, 0x1C, 0 } }, /* 11e */
    { .mk = {0xe0, 0x1B, 0 }, .brk = { 0xe0, 0xF0, 0x1B, 0 } }, /* 11f */
    { .mk = {0xe0, 0x23, 0 }, .brk = { 0xe0, 0xF0, 0x23, 0 } }, /* 120 */
    { .mk = {0xe0, 0x2B, 0 }, .brk = { 0xe0, 0xF0, 0x2B, 0 } }, /* 121 */
    { .mk = {0xe0, 0x34, 0 }, .brk = { 0xe0, 0xF0, 0x34, 0 } }, /* 122 */
    { .mk = {0xe0, 0x33, 0 }, .brk = { 0xe0, 0xF0, 0x33, 0 } }, /* 123 */
    { .mk = {0xe0, 0x3B, 0 }, .brk = { 0xe0, 0xF0, 0x3B, 0 } }, /* 124 */
    { .mk = {0xe0, 0x42, 0 }, .brk = { 0xe0, 0xF0, 0x42, 0 } }, /* 125 */
    { .mk = {0xe0, 0x4B, 0 }, .brk = { 0xe0, 0xF0, 0x4B, 0 } }, /* 126 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 127 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 128 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 129 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 12a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 12b */
    { .mk = {0xe0, 0x1A, 0 }, .brk = { 0xe0, 0xF0, 0x1A, 0 } }, /* 12c */
    { .mk = {0xe0, 0x22, 0 }, .brk = { 0xe0, 0xF0, 0x22, 0 } }, /* 12d */
    { .mk = {0xe0, 0x21, 0 }, .brk = { 0xe0, 0xF0, 0x21, 0 } }, /* 12e */
    { .mk = {0xe0, 0x2A, 0 }, .brk = { 0xe0, 0xF0, 0x2A, 0 } }, /* 12f */
    { .mk = {0xe0, 0x32, 0 }, .brk = { 0xe0, 0xF0, 0x32, 0 } }, /* 130 */
    { .mk = {0xe0, 0x31, 0 }, .brk = { 0xe0, 0xF0, 0x31, 0 } }, /* 131 */
    { .mk = {0xe0, 0x3A, 0 }, .brk = { 0xe0, 0xF0, 0x3A, 0 } }, /* 132 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 133 */
    { .mk = {0xe0, 0x49, 0 }, .brk = { 0xe0, 0xF0, 0x49, 0 } }, /* 134 */
    { .mk = {0xe0, 0x4A, 0 }, .brk = { 0xe0, 0xF0, 0x4A, 0 } }, /* 135 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 136 */
    { .mk = {0xe0, 0x7C, 0 }, .brk = { 0xe0, 0xF0, 0x7C, 0 } }, /* 137 */
    { .mk = {0xe0, 0x11, 0 }, .brk = { 0xe0, 0xF0, 0x11, 0 } }, /* 138 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 139 */
    { .mk = {0xe0, 0x58, 0 }, .brk = { 0xe0, 0xF0, 0x58, 0 } }, /* 13a */
    { .mk = {0xe0, 0x05, 0 }, .brk = { 0xe0, 0xF0, 0x05, 0 } }, /* 13b */
    { .mk = {0xe0, 0x06, 0 }, .brk = { 0xe0, 0xF0, 0x06, 0 } }, /* 13c */
    { .mk = {0xe0, 0x04, 0 }, .brk = { 0xe0, 0xF0, 0x04, 0 } }, /* 13d */
    { .mk = {0xe0, 0x0C, 0 }, .brk = { 0xe0, 0xF0, 0x0C, 0 } }, /* 13e */
    { .mk = {0xe0, 0x03, 0 }, .brk = { 0xe0, 0xF0, 0x03, 0 } }, /* 13f */
    { .mk = {0xe0, 0x0B, 0 }, .brk = { 0xe0, 0xF0, 0x0B, 0 } }, /* 140 */
    { .mk = {0xe0, 0x02, 0 }, .brk = { 0xe0, 0xF0, 0x02, 0 } }, /* 141 */
    { .mk = {0xe0, 0x0A, 0 }, .brk = { 0xe0, 0xF0, 0x0A, 0 } }, /* 142 */
    { .mk = {0xe0, 0x01, 0 }, .brk = { 0xe0, 0xF0, 0x01, 0 } }, /* 143 */
    { .mk = {0xe0, 0x09, 0 }, .brk = { 0xe0, 0xF0, 0x09, 0 } }, /* 144 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 145 */
    { .mk = {0xe0, 0x7E, 0 }, .brk = { 0xe0, 0xF0, 0x7E, 0 } }, /* 146 */
    { .mk = {0xe0, 0x6C, 0 }, .brk = { 0xe0, 0xF0, 0x6C, 0 } }, /* 147 */
    { .mk = {0xe0, 0x75, 0 }, .brk = { 0xe0, 0xF0, 0x75, 0 } }, /* 148 */
    { .mk = {0xe0, 0x7D, 0 }, .brk = { 0xe0, 0xF0, 0x7D, 0 } }, /* 149 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 14a */
    { .mk = {0xe0, 0x6B, 0 }, .brk = { 0xe0, 0xF0, 0x6B, 0 } }, /* 14b */
    { .mk = {0xe0, 0x73, 0 }, .brk = { 0xe0, 0xF0, 0x73, 0 } }, /* 14c */
    { .mk = {0xe0, 0x74, 0 }, .brk = { 0xe0, 0xF0, 0x74, 0 } }, /* 14d */
    { .mk = {0xe0, 0x79, 0 }, .brk = { 0xe0, 0xF0, 0x79, 0 } }, /* 14e */
    { .mk = {0xe0, 0x69, 0 }, .brk = { 0xe0, 0xF0, 0x69, 0 } }, /* 14f */
    { .mk = {0xe0, 0x72, 0 }, .brk = { 0xe0, 0xF0, 0x72, 0 } }, /* 150 */
    { .mk = {0xe0, 0x7A, 0 }, .brk = { 0xe0, 0xF0, 0x7A, 0 } }, /* 151 */
    { .mk = {0xe0, 0x70, 0 }, .brk = { 0xe0, 0xF0, 0x70, 0 } }, /* 152 */
    { .mk = {0xe0, 0x71, 0 }, .brk = { 0xe0, 0xF0, 0x71, 0 } }, /* 153 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 154 */
    { .mk = {0xe0, 0x60, 0 }, .brk = { 0xe0, 0xF0, 0x60, 0 } }, /* 155 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 156 */
    { .mk = {0xe0, 0x78, 0 }, .brk = { 0xe0, 0xF0, 0x78, 0 } }, /* 157 */
    { .mk = {0xe0, 0x07, 0 }, .brk = { 0xe0, 0xF0, 0x07, 0 } }, /* 158 */
    { .mk = {0xe0, 0x0F, 0 }, .brk = { 0xe0, 0xF0, 0x0F, 0 } }, /* 159 */
    { .mk = {0xe0, 0x17, 0 }, .brk = { 0xe0, 0xF0, 0x17, 0 } }, /* 15a */
    { .mk = {0xe0, 0x1F, 0 }, .brk = { 0xe0, 0xF0, 0x1F, 0 } }, /* 15b */
    { .mk = {0xe0, 0x27, 0 }, .brk = { 0xe0, 0xF0, 0x27, 0 } }, /* 15c */
    { .mk = {0xe0, 0x2F, 0 }, .brk = { 0xe0, 0xF0, 0x2F, 0 } }, /* 15d */
    { .mk = {0xe0, 0x37, 0 }, .brk = { 0xe0, 0xF0, 0x37, 0 } }, /* 15e */
    { .mk = {0xe0, 0x3F, 0 }, .brk = { 0xe0, 0xF0, 0x3F, 0 } }, /* 15f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 160 */
    { .mk = {0xe0, 0x4F, 0 }, .brk = { 0xe0, 0xF0, 0x4F, 0 } }, /* 161 */
    { .mk = {0xe0, 0x56, 0 }, .brk = { 0xe0, 0xF0, 0x56, 0 } }, /* 162 */
    { .mk = {0xe0, 0x5E, 0 }, .brk = { 0xe0, 0xF0, 0x5E, 0 } }, /* 163 */
    { .mk = {0xe0, 0x08, 0 }, .brk = { 0xe0, 0xF0, 0x08, 0 } }, /* 164 */
    { .mk = {0xe0, 0x10, 0 }, .brk = { 0xe0, 0xF0, 0x10, 0 } }, /* 165 */
    { .mk = {0xe0, 0x18, 0 }, .brk = { 0xe0, 0xF0, 0x18, 0 } }, /* 166 */
    { .mk = {0xe0, 0x20, 0 }, .brk = { 0xe0, 0xF0, 0x20, 0 } }, /* 167 */
    { .mk = {0xe0, 0x28, 0 }, .brk = { 0xe0, 0xF0, 0x28, 0 } }, /* 168 */
    { .mk = {0xe0, 0x30, 0 }, .brk = { 0xe0, 0xF0, 0x30, 0 } }, /* 169 */
    { .mk = {0xe0, 0x38, 0 }, .brk = { 0xe0, 0xF0, 0x38, 0 } }, /* 16a */
    { .mk = {0xe0, 0x40, 0 }, .brk = { 0xe0, 0xF0, 0x40, 0 } }, /* 16b */
    { .mk = {0xe0, 0x48, 0 }, .brk = { 0xe0, 0xF0, 0x48, 0 } }, /* 16c */
    { .mk = {0xe0, 0x50, 0 }, .brk = { 0xe0, 0xF0, 0x50, 0 } }, /* 16d */
    { .mk = {0xe0, 0x57, 0 }, .brk = { 0xe0, 0xF0, 0x57, 0 } }, /* 16e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 16f */
    { .mk = {0xe0, 0x13, 0 }, .brk = { 0xe0, 0xF0, 0x13, 0 } }, /* 170 */
    { .mk = {0xe0, 0x19, 0 }, .brk = { 0xe0, 0xF0, 0x19, 0 } }, /* 171 */
    { .mk = {0xe0, 0x39, 0 }, .brk = { 0xe0, 0xF0, 0x39, 0 } }, /* 172 */
    { .mk = {0xe0, 0x51, 0 }, .brk = { 0xe0, 0xF0, 0x51, 0 } }, /* 173 */
    { .mk = {0xe0, 0x53, 0 }, .brk = { 0xe0, 0xF0, 0x53, 0 } }, /* 174 */
    { .mk = {0xe0, 0x5C, 0 }, .brk = { 0xe0, 0xF0, 0x5C, 0 } }, /* 175 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 176 */
    { .mk = {0xe0, 0x62, 0 }, .brk = { 0xe0, 0xF0, 0x62, 0 } }, /* 177 */
    { .mk = {0xe0, 0x63, 0 }, .brk = { 0xe0, 0xF0, 0x63, 0 } }, /* 178 */
    { .mk = {0xe0, 0x64, 0 }, .brk = { 0xe0, 0xF0, 0x64, 0 } }, /* 179 */
    { .mk = {0xe0, 0x65, 0 }, .brk = { 0xe0, 0xF0, 0x65, 0 } }, /* 17a */
    { .mk = {0xe0, 0x67, 0 }, .brk = { 0xe0, 0xF0, 0x67, 0 } }, /* 17b */
    { .mk = {0xe0, 0x68, 0 }, .brk = { 0xe0, 0xF0, 0x68, 0 } }, /* 17c */
    { .mk = {0xe0, 0x6A, 0 }, .brk = { 0xe0, 0xF0, 0x6A, 0 } }, /* 17d */
    { .mk = {0xe0, 0x6D, 0 }, .brk = { 0xe0, 0xF0, 0x6D, 0 } }, /* 17e */
    { .mk = {0xe0, 0x6E, 0 }, .brk = { 0xe0, 0xF0, 0x6E, 0 } }, /* 17f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 180 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 181 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 182 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 183 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 184 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 185 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 186 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 187 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 188 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 189 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 190 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 191 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 192 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 193 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 194 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 195 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 196 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 197 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 198 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 199 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1aa */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ab */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ac */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ad */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ae */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1af */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ba */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1bb */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1bc */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1bd */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1be */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1bf */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ca */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1cb */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1cv */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1cd */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ce */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1cf */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1da */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1db */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1dc */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1dd */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1de */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1df */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e0 */
    { .mk = {0xe0, 0xe1, 0 }, .brk = { 0xe0, 0xF0, 0xE1, 0 } }, /* 1e1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ea */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1eb */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ec */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ed */
    { .mk = {0xe0, 0xee, 0 }, .brk = { 0xe0, 0xF0, 0xEE, 0 } }, /* 1ee */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ef */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f0 */
    { .mk = {0xe0, 0xf1, 0 }, .brk = { 0xe0, 0xF0, 0xF1, 0 } }, /* 1f1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1fa */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1fb */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1fc */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1fd */
    { .mk = {0xe0, 0xfe, 0 }, .brk = { 0xe0, 0xF0, 0xFE, 0 } }, /* 1fe */
    { .mk = {0xe0, 0xff, 0 }, .brk = { 0xe0, 0xF0, 0xFF, 0 } }  /* 1ff */
  // clang-format on
};

static const scancode scancode_set3[512] = {
  // clang-format off
    { .mk = {            0 }, .brk = {                   0 } }, /* 000 */
    { .mk = {      0x08, 0 }, .brk = {       0xf0, 0x08, 0 } }, /* 001 */
    { .mk = {      0x16, 0 }, .brk = {       0xf0, 0x16, 0 } }, /* 002 */
    { .mk = {      0x1E, 0 }, .brk = {       0xf0, 0x1E, 0 } }, /* 003 */
    { .mk = {      0x26, 0 }, .brk = {       0xf0, 0x26, 0 } }, /* 004 */
    { .mk = {      0x25, 0 }, .brk = {       0xf0, 0x25, 0 } }, /* 005 */
    { .mk = {      0x2E, 0 }, .brk = {       0xf0, 0x2E, 0 } }, /* 006 */
    { .mk = {      0x36, 0 }, .brk = {       0xf0, 0x36, 0 } }, /* 007 */
    { .mk = {      0x3D, 0 }, .brk = {       0xf0, 0x3D, 0 } }, /* 008 */
    { .mk = {      0x3E, 0 }, .brk = {       0xf0, 0x3E, 0 } }, /* 009 */
    { .mk = {      0x46, 0 }, .brk = {       0xf0, 0x46, 0 } }, /* 00a */
    { .mk = {      0x45, 0 }, .brk = {       0xf0, 0x45, 0 } }, /* 00b */
    { .mk = {      0x4E, 0 }, .brk = {       0xf0, 0x4E, 0 } }, /* 00c */
    { .mk = {      0x55, 0 }, .brk = {       0xf0, 0x55, 0 } }, /* 00d */
    { .mk = {      0x66, 0 }, .brk = {       0xf0, 0x66, 0 } }, /* 00e */
    { .mk = {      0x0D, 0 }, .brk = {       0xf0, 0x0D, 0 } }, /* 00f */
    { .mk = {      0x15, 0 }, .brk = {       0xf0, 0x15, 0 } }, /* 010 */
    { .mk = {      0x1D, 0 }, .brk = {       0xf0, 0x1D, 0 } }, /* 011 */
    { .mk = {      0x24, 0 }, .brk = {       0xf0, 0x24, 0 } }, /* 012 */
    { .mk = {      0x2D, 0 }, .brk = {       0xf0, 0x2D, 0 } }, /* 013 */
    { .mk = {      0x2C, 0 }, .brk = {       0xf0, 0x2C, 0 } }, /* 014 */
    { .mk = {      0x35, 0 }, .brk = {       0xf0, 0x35, 0 } }, /* 015 */
    { .mk = {      0x3C, 0 }, .brk = {       0xf0, 0x3C, 0 } }, /* 016 */
    { .mk = {      0x43, 0 }, .brk = {       0xf0, 0x43, 0 } }, /* 017 */
    { .mk = {      0x44, 0 }, .brk = {       0xf0, 0x44, 0 } }, /* 018 */
    { .mk = {      0x4D, 0 }, .brk = {       0xf0, 0x4D, 0 } }, /* 019 */
    { .mk = {      0x54, 0 }, .brk = {       0xf0, 0x54, 0 } }, /* 01a */
    { .mk = {      0x5B, 0 }, .brk = {       0xf0, 0x5B, 0 } }, /* 01b */
    { .mk = {      0x5A, 0 }, .brk = {       0xf0, 0x5A, 0 } }, /* 01c */
    { .mk = {      0x11, 0 }, .brk = {       0xf0, 0x11, 0 } }, /* 01d */
    { .mk = {      0x1C, 0 }, .brk = {       0xf0, 0x1C, 0 } }, /* 01e */
    { .mk = {      0x1B, 0 }, .brk = {       0xf0, 0x1B, 0 } }, /* 01f */
    { .mk = {      0x23, 0 }, .brk = {       0xf0, 0x23, 0 } }, /* 020 */
    { .mk = {      0x2B, 0 }, .brk = {       0xf0, 0x2B, 0 } }, /* 021 */
    { .mk = {      0x34, 0 }, .brk = {       0xf0, 0x34, 0 } }, /* 022 */
    { .mk = {      0x33, 0 }, .brk = {       0xf0, 0x33, 0 } }, /* 023 */
    { .mk = {      0x3B, 0 }, .brk = {       0xf0, 0x3B, 0 } }, /* 024 */
    { .mk = {      0x42, 0 }, .brk = {       0xf0, 0x42, 0 } }, /* 025 */
    { .mk = {      0x4B, 0 }, .brk = {       0xf0, 0x4B, 0 } }, /* 026 */
    { .mk = {      0x4C, 0 }, .brk = {       0xf0, 0x4C, 0 } }, /* 027 */
    { .mk = {      0x52, 0 }, .brk = {       0xf0, 0x52, 0 } }, /* 028 */
    { .mk = {      0x0E, 0 }, .brk = {       0xf0, 0x0E, 0 } }, /* 029 */
    { .mk = {      0x12, 0 }, .brk = {       0xf0, 0x12, 0 } }, /* 02a */
    { .mk = {      0x5C, 0 }, .brk = {       0xf0, 0x5C, 0 } }, /* 02b */
    { .mk = {      0x1A, 0 }, .brk = {       0xf0, 0x1A, 0 } }, /* 02c */
    { .mk = {      0x22, 0 }, .brk = {       0xf0, 0x22, 0 } }, /* 02d */
    { .mk = {      0x21, 0 }, .brk = {       0xf0, 0x21, 0 } }, /* 02e */
    { .mk = {      0x2A, 0 }, .brk = {       0xf0, 0x2A, 0 } }, /* 02f */
    { .mk = {      0x32, 0 }, .brk = {       0xf0, 0x32, 0 } }, /* 030 */
    { .mk = {      0x31, 0 }, .brk = {       0xf0, 0x31, 0 } }, /* 031 */
    { .mk = {      0x3A, 0 }, .brk = {       0xf0, 0x3A, 0 } }, /* 032 */
    { .mk = {      0x41, 0 }, .brk = {       0xf0, 0x41, 0 } }, /* 033 */
    { .mk = {      0x49, 0 }, .brk = {       0xf0, 0x49, 0 } }, /* 034 */
    { .mk = {      0x4A, 0 }, .brk = {       0xf0, 0x4A, 0 } }, /* 035 */
    { .mk = {      0x59, 0 }, .brk = {       0xf0, 0x59, 0 } }, /* 036 */
    { .mk = {      0x7E, 0 }, .brk = {       0xf0, 0x7E, 0 } }, /* 037 */
    { .mk = {      0x19, 0 }, .brk = {       0xf0, 0x19, 0 } }, /* 038 */
    { .mk = {      0x29, 0 }, .brk = {       0xf0, 0x29, 0 } }, /* 039 */
    { .mk = {      0x14, 0 }, .brk = {       0xf0, 0x14, 0 } }, /* 03a */
    { .mk = {      0x07, 0 }, .brk = {       0xf0, 0x07, 0 } }, /* 03b */
    { .mk = {      0x0F, 0 }, .brk = {       0xf0, 0x0F, 0 } }, /* 03c */
    { .mk = {      0x17, 0 }, .brk = {       0xf0, 0x17, 0 } }, /* 03d */
    { .mk = {      0x1F, 0 }, .brk = {       0xf0, 0x1F, 0 } }, /* 03e */
    { .mk = {      0x27, 0 }, .brk = {       0xf0, 0x27, 0 } }, /* 03f */
    { .mk = {      0x2F, 0 }, .brk = {       0xf0, 0x2F, 0 } }, /* 040 */
    { .mk = {      0x37, 0 }, .brk = {       0xf0, 0x37, 0 } }, /* 041 */
    { .mk = {      0x3F, 0 }, .brk = {       0xf0, 0x3F, 0 } }, /* 042 */
    { .mk = {      0x47, 0 }, .brk = {       0xf0, 0x47, 0 } }, /* 043 */
    { .mk = {      0x4F, 0 }, .brk = {       0xf0, 0x4F, 0 } }, /* 044 */
    { .mk = {      0x76, 0 }, .brk = {       0xf0, 0x76, 0 } }, /* 045 */
    { .mk = {      0x5F, 0 }, .brk = {       0xf0, 0x5F, 0 } }, /* 046 */
    { .mk = {      0x6C, 0 }, .brk = {       0xf0, 0x6C, 0 } }, /* 047 */
    { .mk = {      0x75, 0 }, .brk = {       0xf0, 0x75, 0 } }, /* 048 */
    { .mk = {      0x7D, 0 }, .brk = {       0xf0, 0x7D, 0 } }, /* 049 */
    { .mk = {      0x84, 0 }, .brk = {       0xf0, 0x84, 0 } }, /* 04a */
    { .mk = {      0x6B, 0 }, .brk = {       0xf0, 0x6B, 0 } }, /* 04b */
    { .mk = {      0x73, 0 }, .brk = {       0xf0, 0x73, 0 } }, /* 04c */
    { .mk = {      0x74, 0 }, .brk = {       0xf0, 0x74, 0 } }, /* 04d */
    { .mk = {      0x7C, 0 }, .brk = {       0xf0, 0x7C, 0 } }, /* 04e */
    { .mk = {      0x69, 0 }, .brk = {       0xf0, 0x69, 0 } }, /* 04f */
    { .mk = {      0x72, 0 }, .brk = {       0xf0, 0x72, 0 } }, /* 050 */
    { .mk = {      0x7A, 0 }, .brk = {       0xf0, 0x7A, 0 } }, /* 051 */
    { .mk = {      0x70, 0 }, .brk = {       0xf0, 0x70, 0 } }, /* 052 */
    { .mk = {      0x71, 0 }, .brk = {       0xf0, 0x71, 0 } }, /* 053 */
    { .mk = {      0x57, 0 }, .brk = {       0xf0, 0x57, 0 } }, /* 054 */
    { .mk = {      0x60, 0 }, .brk = {       0xf0, 0x60, 0 } }, /* 055 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 056 */
    { .mk = {      0x56, 0 }, .brk = {       0xf0, 0x56, 0 } }, /* 057 */
    { .mk = {      0x5E, 0 }, .brk = {       0xf0, 0x5E, 0 } }, /* 058 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 059 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 05a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 05b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 05c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 05d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 05e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 05f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 060 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 061 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 062 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 063 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 064 */
    { .mk = {      0x10, 0 }, .brk = {       0xf0, 0x10, 0 } }, /* 065 */
    { .mk = {      0x18, 0 }, .brk = {       0xf0, 0x18, 0 } }, /* 066 */
    { .mk = {      0x20, 0 }, .brk = {       0xf0, 0x20, 0 } }, /* 067 */
    { .mk = {      0x28, 0 }, .brk = {       0xf0, 0x28, 0 } }, /* 068 */
    { .mk = {      0x30, 0 }, .brk = {       0xf0, 0x30, 0 } }, /* 069 */
    { .mk = {      0x38, 0 }, .brk = {       0xf0, 0x38, 0 } }, /* 06a */
    { .mk = {      0x40, 0 }, .brk = {       0xf0, 0x40, 0 } }, /* 06b */
    { .mk = {      0x48, 0 }, .brk = {       0xf0, 0x48, 0 } }, /* 06c */
    { .mk = {      0x50, 0 }, .brk = {       0xf0, 0x50, 0 } }, /* 06d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 06e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 06f */
    { .mk = {      0x87, 0 }, .brk = {       0xf0, 0x87, 0 } }, /* 070 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 071 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 072 */
    { .mk = {      0x51, 0 }, .brk = {       0xf0, 0x51, 0 } }, /* 073 */
    { .mk = {      0x53, 0 }, .brk = {       0xf0, 0x53, 0 } }, /* 074 */
    { .mk = {      0x5C, 0 }, .brk = {       0xf0, 0x5C, 0 } }, /* 075 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 076 */
    { .mk = {      0x62, 0 }, .brk = {       0xf0, 0x62, 0 } }, /* 077 */
    { .mk = {      0x63, 0 }, .brk = {       0xf0, 0x63, 0 } }, /* 078 */
    { .mk = {      0x86, 0 }, .brk = {       0xf0, 0x86, 0 } }, /* 079 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 07a */
    { .mk = {      0x85, 0 }, .brk = {       0xf0, 0x85, 0 } }, /* 07b */
    { .mk = {      0x68, 0 }, .brk = {       0xf0, 0x68, 0 } }, /* 07c */
    { .mk = {      0x13, 0 }, .brk = {       0xf0, 0x13, 0 } }, /* 07d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 07e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 07f */
    { .mk = {      0x80, 0 }, .brk = {       0xf0, 0x80, 0 } }, /* 080 */
    { .mk = {      0x81, 0 }, .brk = {       0xf0, 0x81, 0 } }, /* 081 */
    { .mk = {      0x82, 0 }, .brk = {       0xf0, 0x82, 0 } }, /* 082 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 083 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 084 */
    { .mk = {      0x85, 0 }, .brk = {       0xf0, 0x54, 0 } }, /* 085 */
    { .mk = {      0x86, 0 }, .brk = {       0xf0, 0x86, 0 } }, /* 086 */
    { .mk = {      0x87, 0 }, .brk = {       0xf0, 0x87, 0 } }, /* 087 */
    { .mk = {      0x88, 0 }, .brk = {       0xf0, 0x88, 0 } }, /* 087 */
    { .mk = {      0x89, 0 }, .brk = {       0xf0, 0x89, 0 } }, /* 088 */
    { .mk = {      0x8a, 0 }, .brk = {       0xf0, 0x8a, 0 } }, /* 089 */
    { .mk = {      0x8b, 0 }, .brk = {       0xf0, 0x8b, 0 } }, /* 08b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 08c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 08d */
    { .mk = {      0x8e, 0 }, .brk = {       0xf0, 0x8e, 0 } }, /* 08e */
    { .mk = {      0x8f, 0 }, .brk = {       0xf0, 0x8f, 0 } }, /* 08f */
    { .mk = {      0x90, 0 }, .brk = {       0xf0, 0x90, 0 } }, /* 090 */
    { .mk = {      0x91, 0 }, .brk = {       0xf0, 0x91, 0 } }, /* 091 */
    { .mk = {      0x92, 0 }, .brk = {       0xf0, 0x92, 0 } }, /* 092 */
    { .mk = {      0x93, 0 }, .brk = {       0xf0, 0x93, 0 } }, /* 093 */
    { .mk = {      0x94, 0 }, .brk = {       0xf0, 0x94, 0 } }, /* 094 */
    { .mk = {      0x95, 0 }, .brk = {       0xf0, 0x95, 0 } }, /* 095 */
    { .mk = {      0x96, 0 }, .brk = {       0xf0, 0x96, 0 } }, /* 096 */
    { .mk = {      0x97, 0 }, .brk = {       0xf0, 0x97, 0 } }, /* 097 */
    { .mk = {      0x98, 0 }, .brk = {       0xf0, 0x98, 0 } }, /* 098 */
    { .mk = {      0x99, 0 }, .brk = {       0xf0, 0x99, 0 } }, /* 099 */
    { .mk = {      0x9a, 0 }, .brk = {       0xf0, 0x9a, 0 } }, /* 09a */
    { .mk = {      0x9b, 0 }, .brk = {       0xf0, 0x9b, 0 } }, /* 09b */
    { .mk = {      0x9c, 0 }, .brk = {       0xf0, 0x9c, 0 } }, /* 09c */
    { .mk = {      0x9d, 0 }, .brk = {       0xf0, 0x9d, 0 } }, /* 09d */
    { .mk = {      0x9e, 0 }, .brk = {       0xf0, 0x9e, 0 } }, /* 09e */
    { .mk = {      0x9f, 0 }, .brk = {       0xf0, 0x9f, 0 } }, /* 09f */
    { .mk = {      0xa0, 0 }, .brk = {       0xf0, 0xa0, 0 } }, /* 0a0 */
    { .mk = {      0xa1, 0 }, .brk = {       0xf0, 0xa1, 0 } }, /* 0a1 */
    { .mk = {      0xa2, 0 }, .brk = {       0xf0, 0xa2, 0 } }, /* 0a2 */
    { .mk = {      0xa3, 0 }, .brk = {       0xf0, 0xa3, 0 } }, /* 0a3 */
    { .mk = {      0xa4, 0 }, .brk = {       0xf0, 0xa4, 0 } }, /* 0a4 */
    { .mk = {      0xa5, 0 }, .brk = {       0xf0, 0xa5, 0 } }, /* 0a5 */
    { .mk = {      0xa6, 0 }, .brk = {       0xf0, 0xa6, 0 } }, /* 0a6 */
    { .mk = {      0xa7, 0 }, .brk = {       0xf0, 0xa7, 0 } }, /* 0a7 */
    { .mk = {      0xa8, 0 }, .brk = {       0xf0, 0xa8, 0 } }, /* 0a8 */
    { .mk = {      0xa9, 0 }, .brk = {       0xf0, 0xa9, 0 } }, /* 0a9 */
    { .mk = {      0xaa, 0 }, .brk = {       0xf0, 0xaa, 0 } }, /* 0aa */
    { .mk = {      0xab, 0 }, .brk = {       0xf0, 0xab, 0 } }, /* 0ab */
    { .mk = {      0xac, 0 }, .brk = {       0xf0, 0xac, 0 } }, /* 0ac */
    { .mk = {      0xad, 0 }, .brk = {       0xf0, 0xad, 0 } }, /* 0ad */
    { .mk = {      0xae, 0 }, .brk = {       0xf0, 0xae, 0 } }, /* 0ae */
    { .mk = {      0xaf, 0 }, .brk = {       0xf0, 0xaf, 0 } }, /* 0af */
    { .mk = {      0xb0, 0 }, .brk = {       0xf0, 0xb0, 0 } }, /* 0b0 */
    { .mk = {      0xb1, 0 }, .brk = {       0xf0, 0xb1, 0 } }, /* 0b1 */
    { .mk = {      0xb2, 0 }, .brk = {       0xf0, 0xb2, 0 } }, /* 0b2 */
    { .mk = {      0xb3, 0 }, .brk = {       0xf0, 0xb3, 0 } }, /* 0b3 */
    { .mk = {      0xb4, 0 }, .brk = {       0xf0, 0xb4, 0 } }, /* 0b4 */
    { .mk = {      0xb5, 0 }, .brk = {       0xf0, 0xb5, 0 } }, /* 0b5 */
    { .mk = {      0xb6, 0 }, .brk = {       0xf0, 0xb6, 0 } }, /* 0b6 */
    { .mk = {      0xb7, 0 }, .brk = {       0xf0, 0xb7, 0 } }, /* 0b7 */
    { .mk = {      0xb8, 0 }, .brk = {       0xf0, 0xb8, 0 } }, /* 0b8 */
    { .mk = {      0xb9, 0 }, .brk = {       0xf0, 0xb9, 0 } }, /* 0b9 */
    { .mk = {      0xba, 0 }, .brk = {       0xf0, 0xba, 0 } }, /* 0ba */
    { .mk = {      0xbb, 0 }, .brk = {       0xf0, 0xbb, 0 } }, /* 0bb */
    { .mk = {      0xbc, 0 }, .brk = {       0xf0, 0xbc, 0 } }, /* 0bc */
    { .mk = {      0xbd, 0 }, .brk = {       0xf0, 0xbd, 0 } }, /* 0bd */
    { .mk = {      0xbe, 0 }, .brk = {       0xf0, 0xbe, 0 } }, /* 0be */
    { .mk = {      0xbf, 0 }, .brk = {       0xf0, 0xbf, 0 } }, /* 0bf */
    { .mk = {      0xc0, 0 }, .brk = {       0xf0, 0xc0, 0 } }, /* 0c0 */
    { .mk = {      0xc1, 0 }, .brk = {       0xf0, 0xc1, 0 } }, /* 0c1 */
    { .mk = {      0xc2, 0 }, .brk = {       0xf0, 0xc2, 0 } }, /* 0c2 */
    { .mk = {      0xc3, 0 }, .brk = {       0xf0, 0xc3, 0 } }, /* 0c3 */
    { .mk = {      0xc4, 0 }, .brk = {       0xf0, 0xc4, 0 } }, /* 0c4 */
    { .mk = {      0xc5, 0 }, .brk = {       0xf0, 0xc5, 0 } }, /* 0c5 */
    { .mk = {      0xc6, 0 }, .brk = {       0xf0, 0xc6, 0 } }, /* 0c6 */
    { .mk = {      0xc7, 0 }, .brk = {       0xf0, 0xc7, 0 } }, /* 0c7 */
    { .mk = {      0xc8, 0 }, .brk = {       0xf0, 0xc8, 0 } }, /* 0c8 */
    { .mk = {      0xc9, 0 }, .brk = {       0xf0, 0xc9, 0 } }, /* 0c9 */
    { .mk = {      0xca, 0 }, .brk = {       0xf0, 0xca, 0 } }, /* 0ca */
    { .mk = {      0xcb, 0 }, .brk = {       0xf0, 0xcb, 0 } }, /* 0cb */
    { .mk = {      0xcc, 0 }, .brk = {       0xf0, 0xcc, 0 } }, /* 0cc */
    { .mk = {      0xcd, 0 }, .brk = {       0xf0, 0xcd, 0 } }, /* 0cd */
    { .mk = {      0xce, 0 }, .brk = {       0xf0, 0xce, 0 } }, /* 0ce */
    { .mk = {      0xcf, 0 }, .brk = {       0xf0, 0xcf, 0 } }, /* 0cf */
    { .mk = {      0xd0, 0 }, .brk = {       0xf0, 0xd0, 0 } }, /* 0d0 */
    { .mk = {      0xd1, 0 }, .brk = {       0xf0, 0xd0, 0 } }, /* 0d1 */
    { .mk = {      0xd2, 0 }, .brk = {       0xf0, 0xd2, 0 } }, /* 0d2 */
    { .mk = {      0xd3, 0 }, .brk = {       0xf0, 0xd3, 0 } }, /* 0d3 */
    { .mk = {      0xd4, 0 }, .brk = {       0xf0, 0xd4, 0 } }, /* 0d4 */
    { .mk = {      0xd5, 0 }, .brk = {       0xf0, 0xd5, 0 } }, /* 0d5 */
    { .mk = {      0xd6, 0 }, .brk = {       0xf0, 0xd6, 0 } }, /* 0d6 */
    { .mk = {      0xd7, 0 }, .brk = {       0xf0, 0xd7, 0 } }, /* 0d7 */
    { .mk = {      0xd8, 0 }, .brk = {       0xf0, 0xd8, 0 } }, /* 0d8 */
    { .mk = {      0xd9, 0 }, .brk = {       0xf0, 0xd9, 0 } }, /* 0d9 */
    { .mk = {      0xda, 0 }, .brk = {       0xf0, 0xda, 0 } }, /* 0da */
    { .mk = {      0xdb, 0 }, .brk = {       0xf0, 0xdb, 0 } }, /* 0db */
    { .mk = {      0xdc, 0 }, .brk = {       0xf0, 0xdc, 0 } }, /* 0dc */
    { .mk = {      0xdd, 0 }, .brk = {       0xf0, 0xdd, 0 } }, /* 0dd */
    { .mk = {      0xde, 0 }, .brk = {       0xf0, 0xde, 0 } }, /* 0de */
    { .mk = {      0xdf, 0 }, .brk = {       0xf0, 0xdf, 0 } }, /* 0df */
    { .mk = {      0xe0, 0 }, .brk = {       0xf0, 0xe0, 0 } }, /* 0e0 */
    { .mk = {      0xe1, 0 }, .brk = {       0xf0, 0xe1, 0 } }, /* 0e1 */
    { .mk = {      0xe2, 0 }, .brk = {       0xf0, 0xe2, 0 } }, /* 0e2 */
    { .mk = {      0xe3, 0 }, .brk = {       0xf0, 0xe3, 0 } }, /* 0e3 */
    { .mk = {      0xe4, 0 }, .brk = {       0xf0, 0xe4, 0 } }, /* 0e4 */
    { .mk = {      0xe5, 0 }, .brk = {       0xf0, 0xe5, 0 } }, /* 0e5 */
    { .mk = {      0xe6, 0 }, .brk = {       0xf0, 0xe6, 0 } }, /* 0e6 */
    { .mk = {      0xe7, 0 }, .brk = {       0xf0, 0xe7, 0 } }, /* 0e7 */
    { .mk = {      0xe8, 0 }, .brk = {       0xf0, 0xe8, 0 } }, /* 0e7 */
    { .mk = {      0xe9, 0 }, .brk = {       0xf0, 0xe9, 0 } }, /* 0e8 */
    { .mk = {      0xea, 0 }, .brk = {       0xf0, 0xea, 0 } }, /* 0e9 */
    { .mk = {      0xeb, 0 }, .brk = {       0xf0, 0xeb, 0 } }, /* 0eb */
    { .mk = {      0xec, 0 }, .brk = {       0xf0, 0xec, 0 } }, /* 0ec */
    { .mk = {      0xed, 0 }, .brk = {       0xf0, 0xed, 0 } }, /* 0ed */
    { .mk = {      0xee, 0 }, .brk = {       0xf0, 0xee, 0 } }, /* 0ee */
    { .mk = {      0xef, 0 }, .brk = {       0xf0, 0xef, 0 } }, /* 0ef */
    { .mk = {            0 }, .brk = {                   0 } }, /* 0f0 */
    { .mk = {      0xf1, 0 }, .brk = {       0xf0, 0xf1, 0 } }, /* 0f1 */
    { .mk = {      0xf2, 0 }, .brk = {       0xf0, 0xf2, 0 } }, /* 0f2 */
    { .mk = {      0xf3, 0 }, .brk = {       0xf0, 0xf3, 0 } }, /* 0f3 */
    { .mk = {      0xf4, 0 }, .brk = {       0xf0, 0xf4, 0 } }, /* 0f4 */
    { .mk = {      0xf5, 0 }, .brk = {       0xf0, 0xf5, 0 } }, /* 0f5 */
    { .mk = {      0xf6, 0 }, .brk = {       0xf0, 0xf6, 0 } }, /* 0f6 */
    { .mk = {      0xf7, 0 }, .brk = {       0xf0, 0xf7, 0 } }, /* 0f7 */
    { .mk = {      0xf8, 0 }, .brk = {       0xf0, 0xf8, 0 } }, /* 0f8 */
    { .mk = {      0xf9, 0 }, .brk = {       0xf0, 0xf9, 0 } }, /* 0f9 */
    { .mk = {      0xfa, 0 }, .brk = {       0xf0, 0xfa, 0 } }, /* 0fa */
    { .mk = {      0xfb, 0 }, .brk = {       0xf0, 0xfb, 0 } }, /* 0fb */
    { .mk = {      0xfc, 0 }, .brk = {       0xf0, 0xfc, 0 } }, /* 0fc */
    { .mk = {      0xfd, 0 }, .brk = {       0xf0, 0xfd, 0 } }, /* 0fd */
    { .mk = {      0xfe, 0 }, .brk = {       0xf0, 0xfe, 0 } }, /* 0fe */
    { .mk = {      0xff, 0 }, .brk = {       0xf0, 0xff, 0 } }, /* 0ff */
    { .mk = {      0x62, 0 }, .brk = {       0xF0, 0x62, 0 } }, /* 100 */
    { .mk = {0xe0, 0x76, 0 }, .brk = { 0xe0, 0xF0, 0x76, 0 } }, /* 101 */
    { .mk = {0xe0, 0x16, 0 }, .brk = { 0xe0, 0xF0, 0x16, 0 } }, /* 102 */
    { .mk = {0xe0, 0x1E, 0 }, .brk = { 0xe0, 0xF0, 0x1E, 0 } }, /* 103 */
    { .mk = {0xe0, 0x26, 0 }, .brk = { 0xe0, 0xF0, 0x26, 0 } }, /* 104 */
    { .mk = {0xe0, 0x25, 0 }, .brk = { 0xe0, 0xF0, 0x25, 0 } }, /* 105 */
    { .mk = {0xe0, 0x2E, 0 }, .brk = { 0xe0, 0xF0, 0x2E, 0 } }, /* 106 */
    { .mk = {0xe0, 0x36, 0 }, .brk = { 0xe0, 0xF0, 0x36, 0 } }, /* 107 */
    { .mk = {0xe0, 0x3D, 0 }, .brk = { 0xe0, 0xF0, 0x3D, 0 } }, /* 108 */
    { .mk = {0xe0, 0x3E, 0 }, .brk = { 0xe0, 0xF0, 0x3E, 0 } }, /* 109 */
    { .mk = {0xe0, 0x46, 0 }, .brk = { 0xe0, 0xF0, 0x46, 0 } }, /* 10a */
    { .mk = {0xe0, 0x45, 0 }, .brk = { 0xe0, 0xF0, 0x45, 0 } }, /* 10b */
    { .mk = {0xe0, 0x4E, 0 }, .brk = { 0xe0, 0xF0, 0x4E, 0 } }, /* 10c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 10d */
    { .mk = {0xe0, 0x66, 0 }, .brk = { 0xe0, 0xF0, 0x66, 0 } }, /* 10e */
    { .mk = {0xe0, 0x0D, 0 }, .brk = { 0xe0, 0xF0, 0x0D, 0 } }, /* 10f */
    { .mk = {0xe0, 0x15, 0 }, .brk = { 0xe0, 0xF0, 0x15, 0 } }, /* 110 */
    { .mk = {0xe0, 0x1D, 0 }, .brk = { 0xe0, 0xF0, 0x1D, 0 } }, /* 111 */
    { .mk = {0xe0, 0x24, 0 }, .brk = { 0xe0, 0xF0, 0x24, 0 } }, /* 112 */
    { .mk = {0xe0, 0x2D, 0 }, .brk = { 0xe0, 0xF0, 0x2D, 0 } }, /* 113 */
    { .mk = {0xe0, 0x2C, 0 }, .brk = { 0xe0, 0xF0, 0x2C, 0 } }, /* 114 */
    { .mk = {0xe0, 0x35, 0 }, .brk = { 0xe0, 0xF0, 0x35, 0 } }, /* 115 */
    { .mk = {0xe0, 0x3C, 0 }, .brk = { 0xe0, 0xF0, 0x3C, 0 } }, /* 116 */
    { .mk = {0xe0, 0x43, 0 }, .brk = { 0xe0, 0xF0, 0x43, 0 } }, /* 117 */
    { .mk = {0xe0, 0x44, 0 }, .brk = { 0xe0, 0xF0, 0x44, 0 } }, /* 118 */
    { .mk = {0xe0, 0x4D, 0 }, .brk = { 0xe0, 0xF0, 0x4D, 0 } }, /* 119 */
    { .mk = {0xe0, 0x54, 0 }, .brk = { 0xe0, 0xF0, 0x54, 0 } }, /* 11a */
    { .mk = {0xe0, 0x5B, 0 }, .brk = { 0xe0, 0xF0, 0x5B, 0 } }, /* 11b */
    { .mk = {      0x79, 0 }, .brk = {       0xf0, 0x79, 0 } }, /* 11c */
    { .mk = {      0x58, 0 }, .brk = {       0xf0, 0x58, 0 } }, /* 11d */
    { .mk = {0xe0, 0x1C, 0 }, .brk = { 0xe0, 0xF0, 0x1C, 0 } }, /* 11e */
    { .mk = {0xe0, 0x1B, 0 }, .brk = { 0xe0, 0xF0, 0x1B, 0 } }, /* 11f */
    { .mk = {0xe0, 0x23, 0 }, .brk = { 0xe0, 0xF0, 0x23, 0 } }, /* 120 */
    { .mk = {0xe0, 0x2B, 0 }, .brk = { 0xe0, 0xF0, 0x2B, 0 } }, /* 121 */
    { .mk = {0xe0, 0x34, 0 }, .brk = { 0xe0, 0xF0, 0x34, 0 } }, /* 122 */
    { .mk = {0xe0, 0x33, 0 }, .brk = { 0xe0, 0xF0, 0x33, 0 } }, /* 123 */
    { .mk = {0xe0, 0x3B, 0 }, .brk = { 0xe0, 0xF0, 0x3B, 0 } }, /* 124 */
    { .mk = {0xe0, 0x42, 0 }, .brk = { 0xe0, 0xF0, 0x42, 0 } }, /* 125 */
    { .mk = {0xe0, 0x4B, 0 }, .brk = { 0xe0, 0xF0, 0x4B, 0 } }, /* 126 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 127 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 128 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 129 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 12a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 12b */
    { .mk = {0xe0, 0x1A, 0 }, .brk = { 0xe0, 0xF0, 0x1A, 0 } }, /* 12c */
    { .mk = {0xe0, 0x22, 0 }, .brk = { 0xe0, 0xF0, 0x22, 0 } }, /* 12d */
    { .mk = {0xe0, 0x21, 0 }, .brk = { 0xe0, 0xF0, 0x21, 0 } }, /* 12e */
    { .mk = {0xe0, 0x2A, 0 }, .brk = { 0xe0, 0xF0, 0x2A, 0 } }, /* 12f */
    { .mk = {0xe0, 0x32, 0 }, .brk = { 0xe0, 0xF0, 0x32, 0 } }, /* 130 */
    { .mk = {0xe0, 0x31, 0 }, .brk = { 0xe0, 0xF0, 0x31, 0 } }, /* 131 */
    { .mk = {0xe0, 0x3A, 0 }, .brk = { 0xe0, 0xF0, 0x3A, 0 } }, /* 132 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 133 */
    { .mk = {0xe0, 0x49, 0 }, .brk = { 0xe0, 0xF0, 0x49, 0 } }, /* 134 */
    { .mk = {      0x77, 0 }, .brk = {       0xf0, 0x77, 0 } }, /* 135 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 136 */
    { .mk = {      0x57, 0 }, .brk = {       0xf0, 0x57, 0 } }, /* 137 */
    { .mk = {      0x39, 0 }, .brk = {       0xf0, 0x39, 0 } }, /* 138 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 139 */
    { .mk = {0xe0, 0x58, 0 }, .brk = { 0xe0, 0xF0, 0x58, 0 } }, /* 13a */
    { .mk = {0xe0, 0x05, 0 }, .brk = { 0xe0, 0xF0, 0x05, 0 } }, /* 13b */
    { .mk = {0xe0, 0x06, 0 }, .brk = { 0xe0, 0xF0, 0x06, 0 } }, /* 13c */
    { .mk = {0xe0, 0x04, 0 }, .brk = { 0xe0, 0xF0, 0x04, 0 } }, /* 13d */
    { .mk = {0xe0, 0x0C, 0 }, .brk = { 0xe0, 0xF0, 0x0C, 0 } }, /* 13e */
    { .mk = {0xe0, 0x03, 0 }, .brk = { 0xe0, 0xF0, 0x03, 0 } }, /* 13f */
    { .mk = {0xe0, 0x0B, 0 }, .brk = { 0xe0, 0xF0, 0x0B, 0 } }, /* 140 */
    { .mk = {0xe0, 0x02, 0 }, .brk = { 0xe0, 0xF0, 0x02, 0 } }, /* 141 */
    { .mk = {0xe0, 0x0A, 0 }, .brk = { 0xe0, 0xF0, 0x0A, 0 } }, /* 142 */
    { .mk = {0xe0, 0x01, 0 }, .brk = { 0xe0, 0xF0, 0x01, 0 } }, /* 143 */
    { .mk = {0xe0, 0x09, 0 }, .brk = { 0xe0, 0xF0, 0x09, 0 } }, /* 144 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 145 */
    { .mk = {0xe0, 0x7E, 0 }, .brk = { 0xe0, 0xF0, 0x7E, 0 } }, /* 146 */
    { .mk = {      0x6E, 0 }, .brk = {       0xf0, 0x6E, 0 } }, /* 147 */
    { .mk = {      0x63, 0 }, .brk = {       0xf0, 0x63, 0 } }, /* 148 */
    { .mk = {      0x6F, 0 }, .brk = {       0xf0, 0x6F, 0 } }, /* 149 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 14a */
    { .mk = {      0x61, 0 }, .brk = {       0xf0, 0x61, 0 } }, /* 14b */
    { .mk = {0xe0, 0x73, 0 }, .brk = { 0xe0, 0xF0, 0x73, 0 } }, /* 14c */
    { .mk = {      0x6A, 0 }, .brk = {       0xf0, 0x6A, 0 } }, /* 14d */
    { .mk = {0xe0, 0x79, 0 }, .brk = { 0xe0, 0xF0, 0x79, 0 } }, /* 14e */
    { .mk = {      0x65, 0 }, .brk = {       0xf0, 0x65, 0 } }, /* 14f */
    { .mk = {      0x60, 0 }, .brk = {       0xf0, 0x60, 0 } }, /* 150 */
    { .mk = {      0x6D, 0 }, .brk = {       0xf0, 0x6D, 0 } }, /* 151 */
    { .mk = {      0x67, 0 }, .brk = {       0xf0, 0x67, 0 } }, /* 152 */
    { .mk = {      0x64, 0 }, .brk = {       0xf0, 0x64, 0 } }, /* 153 */
    { .mk = {      0xd4, 0 }, .brk = {       0xf0, 0xD4, 0 } }, /* 154 */
    { .mk = {0xe0, 0x60, 0 }, .brk = { 0xe0, 0xF0, 0x60, 0 } }, /* 155 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 156 */
    { .mk = {0xe0, 0x78, 0 }, .brk = { 0xe0, 0xF0, 0x78, 0 } }, /* 157 */
    { .mk = {0xe0, 0x07, 0 }, .brk = { 0xe0, 0xF0, 0x07, 0 } }, /* 158 */
    { .mk = {0xe0, 0x0F, 0 }, .brk = { 0xe0, 0xF0, 0x0F, 0 } }, /* 159 */
    { .mk = {0xe0, 0x17, 0 }, .brk = { 0xe0, 0xF0, 0x17, 0 } }, /* 15a */
    { .mk = {      0x8B, 0 }, .brk = {       0xf0, 0x8B, 0 } }, /* 15b */
    { .mk = {      0x8C, 0 }, .brk = {       0xf0, 0x8C, 0 } }, /* 15c */
    { .mk = {      0x8D, 0 }, .brk = {       0xf0, 0x8D, 0 } }, /* 15d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 15e */
    { .mk = {      0x7F, 0 }, .brk = {       0xf0, 0x7F, 0 } }, /* 15f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 160 */
    { .mk = {0xe0, 0x4F, 0 }, .brk = { 0xe0, 0xF0, 0x4F, 0 } }, /* 161 */
    { .mk = {0xe0, 0x56, 0 }, .brk = { 0xe0, 0xF0, 0x56, 0 } }, /* 162 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 163 */
    { .mk = {0xe0, 0x08, 0 }, .brk = { 0xe0, 0xF0, 0x08, 0 } }, /* 164 */
    { .mk = {0xe0, 0x10, 0 }, .brk = { 0xe0, 0xF0, 0x10, 0 } }, /* 165 */
    { .mk = {0xe0, 0x18, 0 }, .brk = { 0xe0, 0xF0, 0x18, 0 } }, /* 166 */
    { .mk = {0xe0, 0x20, 0 }, .brk = { 0xe0, 0xF0, 0x20, 0 } }, /* 167 */
    { .mk = {0xe0, 0x28, 0 }, .brk = { 0xe0, 0xF0, 0x28, 0 } }, /* 168 */
    { .mk = {0xe0, 0x30, 0 }, .brk = { 0xe0, 0xF0, 0x30, 0 } }, /* 169 */
    { .mk = {0xe0, 0x38, 0 }, .brk = { 0xe0, 0xF0, 0x38, 0 } }, /* 16a */
    { .mk = {0xe0, 0x40, 0 }, .brk = { 0xe0, 0xF0, 0x40, 0 } }, /* 16b */
    { .mk = {0xe0, 0x48, 0 }, .brk = { 0xe0, 0xF0, 0x48, 0 } }, /* 16c */
    { .mk = {0xe0, 0x50, 0 }, .brk = { 0xe0, 0xF0, 0x50, 0 } }, /* 16d */
    { .mk = {0xe0, 0x57, 0 }, .brk = { 0xe0, 0xF0, 0x57, 0 } }, /* 16e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 16f */
    { .mk = {0xe0, 0x13, 0 }, .brk = { 0xe0, 0xF0, 0x13, 0 } }, /* 170 */
    { .mk = {0xe0, 0x19, 0 }, .brk = { 0xe0, 0xF0, 0x19, 0 } }, /* 171 */
    { .mk = {0xe0, 0x39, 0 }, .brk = { 0xe0, 0xF0, 0x39, 0 } }, /* 172 */
    { .mk = {0xe0, 0x51, 0 }, .brk = { 0xe0, 0xF0, 0x51, 0 } }, /* 173 */
    { .mk = {0xe0, 0x53, 0 }, .brk = { 0xe0, 0xF0, 0x53, 0 } }, /* 174 */
    { .mk = {0xe0, 0x5C, 0 }, .brk = { 0xe0, 0xF0, 0x5C, 0 } }, /* 175 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 176 */
    { .mk = {0xe0, 0x62, 0 }, .brk = { 0xe0, 0xF0, 0x62, 0 } }, /* 177 */
    { .mk = {0xe0, 0x63, 0 }, .brk = { 0xe0, 0xF0, 0x63, 0 } }, /* 178 */
    { .mk = {0xe0, 0x64, 0 }, .brk = { 0xe0, 0xF0, 0x64, 0 } }, /* 179 */
    { .mk = {0xe0, 0x65, 0 }, .brk = { 0xe0, 0xF0, 0x65, 0 } }, /* 17a */
    { .mk = {0xe0, 0x67, 0 }, .brk = { 0xe0, 0xF0, 0x67, 0 } }, /* 17b */
    { .mk = {0xe0, 0x68, 0 }, .brk = { 0xe0, 0xF0, 0x68, 0 } }, /* 17c */
    { .mk = {0xe0, 0x6A, 0 }, .brk = { 0xe0, 0xF0, 0x6A, 0 } }, /* 17d */
    { .mk = {0xe0, 0x6D, 0 }, .brk = { 0xe0, 0xF0, 0x6D, 0 } }, /* 17e */
    { .mk = {0xe0, 0x6E, 0 }, .brk = { 0xe0, 0xF0, 0x6E, 0 } }, /* 17f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 180 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 181 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 182 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 183 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 184 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 185 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 186 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 187 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 188 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 189 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 18f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 190 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 191 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 192 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 193 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 194 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 195 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 196 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 197 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 198 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 199 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19a */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19b */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19c */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19d */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19e */
    { .mk = {            0 }, .brk = {                   0 } }, /* 19f */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1a9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1aa */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ab */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ac */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ad */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ae */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1af */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1b9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ba */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1bb */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1bc */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1bd */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1be */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1bf */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1c9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ca */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1cb */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1cc */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1cd */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ce */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1cf */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d0 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1d9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1da */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1db */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1dc */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1dd */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1de */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1df */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e0 */
    { .mk = {0xe0, 0xe1, 0 }, .brk = { 0xe0, 0xF0, 0xE1, 0 } }, /* 1e1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1e9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ea */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1eb */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ec */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ed */
    { .mk = {0xe0, 0xee, 0 }, .brk = { 0xe0, 0xF0, 0xEE, 0 } }, /* 1ee */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1ef */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f0 */
    { .mk = {0xe0, 0xf1, 0 }, .brk = { 0xe0, 0xF0, 0xF1, 0 } }, /* 1f1 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f2 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f3 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f4 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f5 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f6 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f7 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f8 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1f9 */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1fa */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1fb */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1fc */
    { .mk = {            0 }, .brk = {                   0 } }, /* 1fd */
    { .mk = {0xe0, 0xfe, 0 }, .brk = { 0xe0, 0xF0, 0xFE, 0 } }, /* 1fe */
    { .mk = {0xe0, 0xff, 0 }, .brk = { 0xe0, 0xF0, 0xFF, 0 } }  /* 1ff */
  // clang-format on
};

#ifdef ENABLE_KEYBOARD_AT_LOG
int keyboard_at_do_log = ENABLE_KEYBOARD_AT_LOG;

static void
keyboard_at_log(const char *fmt, ...)
{
    va_list ap;

    if (keyboard_at_do_log) {
        va_start(ap, fmt);
        pclog_ex(fmt, ap);
        va_end(ap);
    }
}
#else
#    define keyboard_at_log(fmt, ...)
#endif

static void
keyboard_at_set_scancode_set(void)
{
    switch (keyboard_mode) {
        default:
        case 0x01:
            keyboard_set_table(scancode_set1);
            break;

        case 0x02:
            keyboard_set_table(scancode_set2);
            break;

        case 0x03:
            keyboard_set_table(scancode_set3);
            break;
    }
}

static void
add_data_vals(atkbc_dev_t *dev, uint8_t *val, uint8_t len)
{
    for (uint8_t i = 0; i < len; i++)
        kbc_at_dev_queue_add(dev, val[i], 1);
}

static void
add_data_kbd(uint16_t val)
{
    atkbc_dev_t *dev = SavedKbd;
    uint8_t  fake_shift[4] = { 0 };
    uint8_t  num_lock = 0;
    uint8_t  shift_states = 0;

    dev->ignore = 1;

    keyboard_get_states(NULL, &num_lock, NULL);
    shift_states = keyboard_get_shift() & STATE_SHIFT_MASK;

    switch (val) {
        case FAKE_LSHIFT_ON:
            keyboard_at_log("%s: Fake left shift on, scan code: ", dev->name);
            if (num_lock) {
                if (shift_states) {
                    keyboard_at_log("N/A (one or both shifts on)\n");
                    break;
                } else {
                    /* Num lock on and no shifts are pressed, send non-inverted fake shift. */
                    switch (keyboard_mode & 0x02) {
                        case 1:
                            keyboard_at_log("E0 2A\n");
                            fake_shift[0] = 0xe0;
                            fake_shift[1] = 0x2a;
                            add_data_vals(dev, fake_shift, 2);
                            break;

                        case 2:
                            keyboard_at_log("E0 12\n");
                            fake_shift[0] = 0xe0;
                            fake_shift[1] = 0x12;
                            add_data_vals(dev, fake_shift, 2);
                            break;

                        default:
                            keyboard_at_log("N/A (scan code set %i)\n", keyboard_mode & 0x02);
                            break;
                    }
                }
            } else {
                if (shift_states & STATE_LSHIFT) {
                    /* Num lock off and left shift pressed. */
                    switch (keyboard_mode & 0x02) {
                        case 1:
                            keyboard_at_log("E0 AA\n");
                            fake_shift[0] = 0xe0;
                            fake_shift[1] = 0xaa;
                            add_data_vals(dev, fake_shift, 2);
                            break;

                        case 2:
                            keyboard_at_log("E0 F0 12\n");
                            fake_shift[0] = 0xe0;
                            fake_shift[1] = 0xf0;
                            fake_shift[2] = 0x12;
                            add_data_vals(dev, fake_shift, 3);
                            break;

                        default:
                            keyboard_at_log("N/A (scan code set %i)\n", keyboard_mode & 0x02);
                            break;
                    }
                }
                if (shift_states & STATE_RSHIFT) {
                    /* Num lock off and right shift pressed. */
                    switch (keyboard_mode & 0x02) {
                        case 1:
                            keyboard_at_log("E0 B6\n");
                            fake_shift[0] = 0xe0;
                            fake_shift[1] = 0xb6;
                            add_data_vals(dev, fake_shift, 2);
                            break;

                        case 2:
                            keyboard_at_log("E0 F0 59\n");
                            fake_shift[0] = 0xe0;
                            fake_shift[1] = 0xf0;
                            fake_shift[2] = 0x59;
                            add_data_vals(dev, fake_shift, 3);
                            break;

                        default:
                            keyboard_at_log("N/A (scan code set %i)\n", keyboard_mode & 0x02);
                            break;
                    }
                }
                keyboard_at_log(shift_states ? "" : "N/A (both shifts off)\n");
            }
            break;

        case FAKE_LSHIFT_OFF:
            keyboard_at_log("%s: Fake left shift on, scan code: ", dev->name);
            if (num_lock) {
                if (shift_states) {
                    keyboard_at_log("N/A (one or both shifts on)\n");
                    break;
                } else {
                    /* Num lock on and no shifts are pressed, send non-inverted fake shift. */
                    switch (keyboard_mode & 0x02) {
                        case 1:
                            keyboard_at_log("E0 AA\n");
                            fake_shift[0] = 0xe0;
                            fake_shift[1] = 0xaa;
                            add_data_vals(dev, fake_shift, 2);
                            break;

                        case 2:
                            keyboard_at_log("E0 F0 12\n");
                            fake_shift[0] = 0xe0;
                            fake_shift[1] = 0xf0;
                            fake_shift[2] = 0x12;
                            add_data_vals(dev, fake_shift, 3);
                            break;

                        default:
                            keyboard_at_log("N/A (scan code set %i)\n", keyboard_mode & 0x02);
                            break;
                    }
                }
            } else {
                if (shift_states & STATE_LSHIFT) {
                    /* Num lock off and left shift pressed. */
                    switch (keyboard_mode & 0x02) {
                        case 1:
                            keyboard_at_log("E0 2A\n");
                            fake_shift[0] = 0xe0;
                            fake_shift[1] = 0x2a;
                            add_data_vals(dev, fake_shift, 2);
                            break;

                        case 2:
                            keyboard_at_log("E0 12\n");
                            fake_shift[0] = 0xe0;
                            fake_shift[1] = 0x12;
                            add_data_vals(dev, fake_shift, 2);
                            break;

                        default:
                            keyboard_at_log("N/A (scan code set %i)\n", keyboard_mode & 0x02);
                            break;
                    }
                }
                if (shift_states & STATE_RSHIFT) {
                    /* Num lock off and right shift pressed. */
                    switch (keyboard_mode & 0x02) {
                        case 1:
                            keyboard_at_log("E0 36\n");
                            fake_shift[0] = 0xe0;
                            fake_shift[1] = 0x36;
                            add_data_vals(dev, fake_shift, 2);
                            break;

                        case 2:
                            keyboard_at_log("E0 59\n");
                            fake_shift[0] = 0xe0;
                            fake_shift[1] = 0x59;
                            add_data_vals(dev, fake_shift, 2);
                            break;

                        default:
                            keyboard_at_log("N/A (scan code set %i)\n", keyboard_mode & 0x02);
                            break;
                    }
                }
                keyboard_at_log(shift_states ? "" : "N/A (both shifts off)\n");
            }
            break;

        default:
            kbc_at_dev_queue_add(dev, val, 1);
            break;
    }

    dev->ignore = 0;
}

void
keyboard_at_clear_data(void *priv)
{
    atkbc_dev_t *dev = (atkbc_dev_t *) priv;

    dev->flags &= ~FLAG_CTRLDAT;
}

static void
keyboard_at_set_defaults(atkbc_dev_t *dev)
{
    dev->rate = 1;

    keyboard_set3_all_break  = 0;
    keyboard_set3_all_repeat = 0;
    memset(keyboard_set3_flags, 0, 512);

    keyboard_mode = 0x02;
    keyboard_at_set_scancode_set();
}

static void
keyboard_at_bat(void *priv)
{
    atkbc_dev_t *dev = (atkbc_dev_t *) priv;

    if (bat_counter == 0x0000) {
        keyboard_at_set_defaults(dev);

        keyboard_scan = 1;

        kbc_at_dev_queue_add(dev, 0xaa, 0);
    } else {
        bat_counter--;
        dev->state = DEV_STATE_EXECUTE_BAT;
    }
}

static void
keyboard_at_invalid_cmd(atkbc_dev_t *dev)
{
    keyboard_at_log("%s: Invalid command [%02X]\n", dev->name, dev->port->dat);
    kbc_at_dev_queue_add(dev, inv_cmd_response, 0);
}


static void
keyboard_at_write(void *priv)
{
    atkbc_dev_t *dev = (atkbc_dev_t *) priv;
    uint8_t val;

    if (dev->port == NULL)
        return;

    val = dev->port->dat;

    dev->state = DEV_STATE_MAIN_OUT;

    if ((val < 0xed) && (dev->flags & FLAG_CTRLDAT)) {
        dev->flags &= ~FLAG_CTRLDAT;

        switch (dev->command) {
            case 0xed: /* Set/reset LEDs */
                kbc_at_dev_queue_add(dev, 0xfa, 0);
                keyboard_at_log("%s: Set/reset LEDs [%02X]\n", dev->name, val);
                break;

            case 0xf0: /* Get/set scancode set */
                kbc_at_dev_queue_add(dev, (val > 3) ? 0xfe : 0xfa, 0);
                switch (val) {
                    case 0x00:
                        keyboard_at_log("%s: Get scan code set [%02X]\n", dev->name, keyboard_mode);
                        kbc_at_dev_queue_add(dev, keyboard_mode, 0);
                        break;
                    case 0x01 ... 0x03:
                        keyboard_mode = val;
                        keyboard_at_log("%s: Set scan code set [%02X]\n", dev->name, keyboard_mode);
                        keyboard_at_set_scancode_set();
                        break;
                    default:
                        /* Fatal so any instance of anything attempting to set scan code > 3 can be reported to us. */
                        fatal("%s: Scan code set [%02X] invalid, resend\n", dev->name, val);
                        dev->flags |= FLAG_CTRLDAT;
                        dev->state = DEV_STATE_MAIN_WANT_IN;
                        break;
                }
                break;

            case 0xf3: /* set typematic rate/delay */
                if (val & 0x80) {
                    keyboard_at_log("%s: Set typematic rate/delay [%02X] has bit 7 set - invalid\n", dev->name, val);
                    dev->flags |= FLAG_CTRLDAT;      /* Resend = keep waiting for parameter. */
                    kbc_at_dev_queue_add(dev, 0xfe, 0); /* Command response */
                    dev->state = DEV_STATE_MAIN_WANT_IN;
                } else {
                    dev->rate = val;
                    kbc_at_dev_queue_add(dev, 0xfa, 0); /* Command response */
                    keyboard_at_log("%s: Set typematic rate/delay [%02X]\n", dev->name, val);
                }
                break;

            default:
                fatal("%s: Parameter [%02X] for invalid command [%02X] - possibly memory corruption!\n", dev->name, val, dev->command);
                kbc_at_dev_queue_add(dev, 0xfe, 0);
        }
    } else {
        if (dev->flags & FLAG_CTRLDAT) {
            /* Special case - another command during another command that wants input - proceed
               as normal but do not cancel the command (so keep waiting for input), unless the
               command in progress is ED (Set/reset LEDs). */
            if (val == 0xed) {
                keyboard_scan = 1;
                dev->flags &= ~FLAG_CTRLDAT;
            } else
                dev->state = DEV_STATE_MAIN_WANT_IN;
        }

        switch (val) {
            case 0xed: /* set/reset LEDs */
                dev->command = val;
                keyboard_at_log("%s: set/reset LEDs\n", dev->name);
                dev->flags |= FLAG_CTRLDAT;
                kbc_at_dev_queue_add(dev, 0xfa, 0); /* ACK for command byte */
                dev->state = DEV_STATE_MAIN_WANT_IN;
                break;

            case 0xee: /* diagnostic echo */
                keyboard_at_log("%s: ECHO\n", dev->name);
                kbc_at_dev_queue_add(dev, 0xee, 0);
                break;

            case 0xef: /* Invalid command */
            case 0xf1: /* Invalid command */
                keyboard_at_log("%s: Invalid command [%02X]\n", dev->name, dev->port->dat);
                kbc_at_dev_queue_add(dev, inv_cmd_response, 0);
                break;

            case 0xf0: /* get/set scan code set */
                if (dev->type & FLAG_PS2) {
                    dev->command = val;
                    keyboard_at_log("%s: scan code set\n", dev->name);
                    dev->flags |= FLAG_CTRLDAT;
                    kbc_at_dev_queue_add(dev, 0xfa, 0); /* ACK for command byte */
                    dev->state = DEV_STATE_MAIN_WANT_IN;
                } else
                    keyboard_at_invalid_cmd(dev);
                break;

            case 0xf2: /* read ID */
                keyboard_at_log("%s: read keyboard id\n", dev->name);
                /* TODO: After keyboard type selection is implemented, make this
                         return the correct keyboard ID for the selected type. */
                kbc_at_dev_queue_add(dev, 0xfa, 0);
                for (uint8_t i = 0; i < 4; i++) {
                    if (id_bytes[dev->type][i] == 0)
                        break;

                    kbc_at_dev_queue_add(dev, id_bytes[dev->type][i], 0);
                }
                break;

            case 0xf3: /* set command mode */
                dev->command = val;
                keyboard_at_log("%s: set typematic rate/delay\n", dev->name);
                dev->flags |= FLAG_CTRLDAT;
                kbc_at_dev_queue_add(dev, 0xfa, 0); /* ACK for command byte */
                dev->state = DEV_STATE_MAIN_WANT_IN;
                break;

            case 0xf4: /* enable */
                keyboard_at_log("%s: enable keyboard\n", dev->name);
                keyboard_scan = 1;
                kbc_at_dev_queue_add(dev, 0xfa, 0);
                break;

            case 0xf5: /* set defaults and disable keyboard */
            case 0xf6: /* set defaults */
                keyboard_at_log("%s: set defaults%s\n",
                                dev->name, (val == 0xf6) ? "" : " and disable keyboard");
                keyboard_scan = !(val & 0x01);
                keyboard_at_log("%s: val = %02X, keyboard_scan = %i\n",
                                dev->name, val, keyboard_scan);
                kbc_at_dev_queue_add(dev, 0xfa, 0);

                keyboard_set3_all_break  = 0;
                keyboard_set3_all_repeat = 0;
                memset(keyboard_set3_flags, 0, 512);

                keyboard_mode = 0x02;
                keyboard_at_set_scancode_set();
                break;

            case 0xf7: /* set all keys to repeat */
                if (dev->type & FLAG_PS2) {
                    keyboard_at_log("%s: set all keys to repeat\n", dev->name);
                    kbc_at_dev_queue_add(dev, 0xfa, 0);
                    keyboard_set3_all_break = 1;
                } else
                    keyboard_at_invalid_cmd(dev);
                break;

            case 0xf8: /* set all keys to give make/break codes */
                if (dev->type & FLAG_PS2) {
                    keyboard_at_log("%s: set all keys to give make/break codes\n", dev->name);
                    kbc_at_dev_queue_add(dev, 0xfa, 0);
                    keyboard_set3_all_break = 1;
                } else
                    keyboard_at_invalid_cmd(dev);
                break;

            case 0xf9: /* set all keys to give make codes only */
                if (dev->type & FLAG_PS2) {
                    keyboard_at_log("%s: set all keys to give make codes only\n", dev->name);
                    kbc_at_dev_queue_add(dev, 0xfa, 0);
                    keyboard_set3_all_break = 0;
                } else
                    keyboard_at_invalid_cmd(dev);
                break;

            case 0xfa: /* set all keys to repeat and give make/break codes */
                if (dev->type & FLAG_PS2) {
                    keyboard_at_log("%s: set all keys to repeat and give make/break codes\n", dev->name);
                    kbc_at_dev_queue_add(dev, 0xfa, 0);
                    keyboard_set3_all_repeat = 1;
                    keyboard_set3_all_break  = 1;
                } else
                    keyboard_at_invalid_cmd(dev);
                break;

            /* TODO: Actually implement these commands. */
            case 0xfb: /* set some keys to repeat */
                keyboard_at_log("%s: set some keys to repeat\n", dev->name);
                kbc_at_dev_queue_add(dev, inv_cmd_response, 0);
                break;

            case 0xfc: /* set some keys to give make/break codes */
                keyboard_at_log("%s: set some keys to give make/break codes\n", dev->name);
                kbc_at_dev_queue_add(dev, inv_cmd_response, 0);
                break;

            case 0xfd: /* set some keys to give make codes only */
                keyboard_at_log("%s: set some keys to give make codes only\n", dev->name);
                kbc_at_dev_queue_add(dev, inv_cmd_response, 0);
                break;

            /* TODO: This is supposed to resend multiple bytes after some commands. */
            case 0xfe: /* resend last scan code */
                keyboard_at_log("%s: resend last scan code\n", dev->name);
                kbc_at_dev_queue_add(dev, 0xfa, 0);
                kbc_at_dev_queue_add(dev, dev->last_scan_code, 0);
                break;

            case 0xff: /* reset */
                kbc_at_dev_reset(dev, 1);
                bat_counter = 1000;
                break;

            default:
                kbc_at_dev_queue_add(dev, 0xfe, 0);
        }
    }
}

/*
 * Initialize the device for use by the user.
 *
 * We also get called from the various machines.
 */
void *
keyboard_at_init(const device_t *info)
{
    atkbc_dev_t *dev = kbc_at_dev_init(DEV_KBD);

    dev->name = info->name;
    /* Key 14  = Japanese key next to backspace, scan code:                13 (Yen 7D);
       Key 29  = US backslash, scan code:                                  5C (Backslash 2B);
       Key 42  = European backslash, scan code:                            53 (Backslash 2B);
       Key 45  = European key next to left shift, scan code:               13 (Key 56);
       Key 56  = Japanese key next to right shift, scan code:              5C (Backslash 73);
       Key 59  = Japanese key between left Ctrl and left Alt, scan code:   85 (Muhenkan 7B);
       Key 63  = Japanese key between right Ctrl and right Alt, scan code: 86 (Henkan/Zenkouho 79);
       Key 65? = Japanese key between right Ctrl and right Alt, scan code: 87 (Hiragana/Katakana 70).
     */
    dev->type = FLAG_PS2 | KBD_102_KEY /* device_get_config_int("type") */;

    keyboard_at_log("%s: type=%d\n", dev->name, dev->type);

    dev->process_cmd = keyboard_at_write;
    dev->execute_bat = keyboard_at_bat;

    dev->scan        = &keyboard_scan;

    dev->fifo_mask   = FIFO_SIZE - 1;

    if (dev->port != NULL) {
        kbc_at_dev_reset(dev, 0);
        bat_counter = 0x0000;
    }

    keyboard_send = add_data_kbd;
    SavedKbd = dev;

    inv_cmd_response = (dev->type & FLAG_PS2) ? 0xfe : 0xfa;

    /* Return our private data to the I/O layer. */
    return dev;
}

static void
keyboard_at_close(void *priv)
{
    atkbc_dev_t *dev = (atkbc_dev_t *) priv;

    keyboard_scan = 0;
    keyboard_send = NULL;

    /* Disable the scancode maps. */
    keyboard_set_table(NULL);

    SavedKbd = NULL;

    free(dev);
}

static const device_config_t keyboard_at_config[] = {
  // clang-format off
    {
        .name = "type",
        .description = "Type",
        .type = CONFIG_SELECTION,
        .default_string = "",
        .default_int = 1,
        .file_filter = "",
        .spinner = { 0 },
        .selection = {
            { .description = "AT 84-key",          .value = FLAG_AT  | KBD_84_KEY  },
            { .description = "AT 101/102/106-key", .value = FLAG_AT  | KBD_101_KEY },
            { .description = "AT Korean",          .value = FLAG_AT  | KBD_KOREAN  },
            { .description = "PS/2 101-key",       .value = FLAG_PS2 | KBD_101_KEY },
            { .description = "PS/2 102-key",       .value = FLAG_PS2 | KBD_102_KEY },
            { .description = "PS/2 106-key JIS",   .value = FLAG_PS2 | KBD_JIS     },
            { .description = "PS/2 Korean",        .value = FLAG_PS2 | KBD_KOREAN  },
            { .description = ""                                                    }
        }
    },
    {
        .name = "", .description = "", .type = CONFIG_END
    }
  // clang-format on
};

/* TODO: Add more keyboard types. */
const device_t keyboard_at_generic_device = {
    .name          = "Standard AT or PS/2 Keyboard",
    .internal_name = "ps2",
    .flags         = DEVICE_PS2,
    .local         = 0,
    .init          = keyboard_at_init,
    .close         = keyboard_at_close,
    .reset         = NULL,
    { .poll = NULL },
    .speed_changed = NULL,
    .force_redraw  = NULL,
    .config        = keyboard_at_config
};
