/* icdprog.c

    icdprog - an open source PIC programmer for use with the Microchip ICD(1)
    Copyright (C) 2001-2004  Geir Thomassen, Andrew Ferry.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
*/


#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include "icdprog.h"
#include "intel.h"
#include "icd.h"
#include "pic16f87x.h"

void usage(char *argv_0)
{
	printf("Usage: %s [-p serial_port_device] [-v] <-e eeprom_addr eeprom_data | -c config_word | hexfile>\n\n", argv_0);
	printf("Programs a Microchip PIC controller with the Microchip ICD hardware\n\n");
	printf("  options:\n");
	printf("   -p serial_port_device, default %s\n", DEFAULT_DEVICE);
	printf("   -e eeprom_write_address, eeprom write\n");                   
	printf("   -c config word write\n");                                      
	printf("   -v verify only\n");
}


/*
  read_file() loads an intel hex file into memory.

  returns 0 on success, -1 on file open errors, -2 on hex file parsing errors.
*/

int read_file(char *filename, int *mem)
{
	int i;
	int dat, datlow, dathigh, err;
	long addr;

	HEXFILE *hfp;

	for(i=0;i<MAX_FLASH_SIZE;i++) {  /* fill array with UNINITIALIZED (-1) -> no data */
		mem[i] = UNINITIALIZED;
	}

	if ((hfp = open_hexfile(filename, "r")) == NULL) {
		fprintf(stderr,"Error: Could not open %s\n", filename);
		return -1;
	}

	while ((err = read_hexfile(hfp, &datlow, &addr)) == HEX_OK) {
		if(((addr/2) < MAX_PROG_MEM_SIZE) ||
		  (((addr/2)>=ID_LOC_ADDR) && ((addr/2)<ID_LOC_ADDR + ID_LOC_LEN))) {

			err = read_hexfile(hfp, &dathigh, &addr);
			if(err != HEX_OK) {
				return -2;
			}

			dat = datlow + (dathigh << 8);
			addr = addr / 2;

			if(mem[addr] != UNINITIALIZED) {
				printf("Warning, hex file contains duplicate data\n");
			}

			if(addr==DEVICE_ID_ADDR) {
				printf("Warning, hex file contains a device ID (address = %04X)\n",DEVICE_ID_ADDR);
			}

                        if(addr==CONF_WORD_ADDR) {
				dat |= 0xC00;  /* The ICD refuses to program if these bits are wrong */
			}
			mem[addr] = dat;
		}
	}

	if (err != HEX_EOF) {
		printf("Error reading %s (Is this a valid Intel hex file ?)\n", filename);
		return -2;
	}

	close_hexfile(hfp);

	return 0;
}

void print_config(int dat) {
		
	printf("----- Device configuration: -----\n");
	printf("Oscillator selection bits: ");
	switch(dat & 0x3) {
	case 0:
		printf("LP\n");
		break;
	case 1:
		printf("XT\n");
		break;
	case 2:
		printf("HS\n");
		break;
	case 3:
		printf("RC\n");
		break;
	}
	printf("Watchdog timer:  %s\n", (dat & 0x04) ? "enabled"  : "disabled");
	printf("Power up timer:  %s\n", (dat & 0x08) ? "disabled" : "enabled");
	printf("Brown out reset: %s\n", (dat & 0x40) ? "enabled"  : "disabled");
	printf("Low voltage programming: %s\n", (dat & 0x80) ? "enabled"  : "disabled");
	printf("EE data code protection: %s\n", (dat & 0x100) ? "off"  : "on");
	printf("Flash memory write: %s\n", (dat & 0x200) ? "ok"  : "protected");

	if(((dat >>4)&0x3) != ((dat >> 12) & 0x3)) {
		fprintf(stderr,"Waring, funny code protection bits.\n");
	} else {
		switch((dat >> 4)&0x3) {
		case 0:
			printf("0x0000-0x0FFF (4K devices) or 0x0000-0x1FFF (8K devices) protected\n");
			break;
		case 1:
			printf("0x0800-0x0FFF (4K devices) or 0x1000-0x1FFF (8K devices) protected\n");
			break;
		case 2:
			printf("0x0F00-0x0FFF (4K devices) or 0x1F00-0x1FFF (8K devices) protected\n");
			break;
		case 3:
			printf("Code protection: off\n");
			break;
		}
	}

	printf("---------------------------------\n");
}


int main(int argc, char *argv[])
{
	char *filename;
	char *icd_device = "/dev/ttyS0";
	
	char *function="Programming";
	
	int mem[MAX_FLASH_SIZE];
	
	int i,option;
	int mode = PROGRAM;
	int memory = FLASH;

	int eeprom_addr=0;

	
	controller t;
	
	while ((option = getopt(argc, argv, "vrp:e:c")) != -1) {
		switch (option) {
		case 'v':
			mode = VERIFY;
			function = "Verifying";
			break;

		case 'p':
			icd_device = optarg;
			break;

		case 'e':
			memory = EEPROM;
			eeprom_addr=strtol(optarg,NULL,0) & 0xFF;
			break;
			
		case 'c':
			memory = CONFIG;
			break;

		case 'r':
			mode = READ;
			function = "Reading";
			break;

		case '?':
		default:
			usage(argv[0]);
			exit(1);
			break;
		}
	}
	
	if ((mode == READ) && (memory != FLASH)) {
		usage(argv[0]);
		exit(1);
	}
	
	if (optind != argc - 1) {	/* There should be exactly one arg left */
		usage(argv[0]);
		exit(1);
	}
		
	if (memory == EEPROM) {
		
		for(i=0;i<MAX_EEPROM_SIZE;i++) {  /* fill array with -1 -> no data */
			mem[i] = UNINITIALIZED;
		}
		
		/* The remaining argument is the eeprom data*/
		mem[eeprom_addr] = strtol(argv[optind],NULL,0) & 0xFF;
		
		
		if(icd_init(icd_device) != 0) {
		    exit(2);
		}

		t=icd_controller_type();

		if (t.type == 0) {
			fprintf(stderr,"Error: No target PIC controller connected to the ICD\n");
			fprintf(stderr,"Please check your cables\n");
			exit(2);
		}
		
		
		printf("%s EEPROM:\n",function);
		
		if(icd_prog(mem,ICD_PROG_EEPROM,0,t.eeprom_max,mode) != 0) {
			fprintf(stderr,"Error %s EEPROM\n",function);
			exit(3);
		}
		
		exit(0);	


	}
	
	if (memory == CONFIG) {

		/* The remaining argument is the config word*/
		mem[CONF_WORD_ADDR]=strtol(argv[optind],NULL,0) & 0x3FFF;

		if(icd_init(icd_device) != 0) {
		    exit(2);
		}

		printf("%s config word:\n",function);
		
		if(icd_prog_ext(mem,CONF_WORD_ADDR,CONF_WORD_ADDR,mode)!= 0) {
			fprintf(stderr,"Error %s config word\n",function);
			exit(3);
		}
			
		print_config(mem[CONF_WORD_ADDR]);
		
		exit(0);
	}
	
	/* default (memory == flash) */
		
	/* The remaining argument is the file name*/
	filename = argv[optind];
		
	if (mode != READ) {
		if (read_file(filename,mem) !=0 )
			exit(1);
		if(mem[CONF_WORD_ADDR] == UNINITIALIZED) {
			fprintf(stderr,"ERROR: Hex file does not contain a configuration word, device can not be programmed\n");
			exit(3);
		}

	}
	else { /* mode == READ */
		for(i=0;i<MAX_FLASH_SIZE;i++)    /* fill array with 0 nop */
			mem[i] = 0;
	}

	
	/* TODO: check program fits memory */

	if(icd_init(icd_device) != 0) {
		exit(2);
	}

	t=icd_controller_type();

	if (t.type == 0) {
		fprintf(stderr,"Error: No target PIC controller connected to the ICD\n");
		fprintf(stderr,"Please check your cables\n");
		exit(2);
	}

	
	if (mode == PROGRAM)
		if(icd_erase() != 0) {
		    fprintf(stderr,"Error erasing device\n");
		    exit(3);
		}
	
	printf("%s flash memory:\n",function);
	if(icd_prog(mem,ICD_PROG_FLASH,0,t.flash_max,mode) != 0) {
		fprintf(stderr,"Error %s flash memory\n",function);
		exit(3);
	}
		
	printf("%s id locations:\n",function);
	if(icd_prog_ext(mem, ID_LOC_ADDR, ID_LOC_ADDR+4,mode) != 0) {
		fprintf(stderr,"Error %s id locations\n",function);
		exit(3);
	}
	
	printf("%s  config word:\n",function);
	
	if(icd_prog_ext(mem, CONF_WORD_ADDR, CONF_WORD_ADDR,mode) != 0) {
		fprintf(stderr,"Error %s config word\n",function);
		exit(3);
	}
	
	print_config(mem[CONF_WORD_ADDR]);

	if (mode == READ)
 {
		printf ("TODO: save HEX file\n");
		/* if(write_file(mem,filename) != 0) {
			fprintf(stderr,"Error writing HEX file\n");
			exit(3);
		} */
	}
		
	exit(0);
}

