//==============================================================================
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//==============================================================================

//==============================================================================
// File: cWeapon.cpp
// Project: Shooting Star
// Author: Jarmo Hekkanen <jarski@2ndpoint.fi>
// Copyrights (c) 2003 2ndPoint ry (www.2ndpoint.fi)
//------------------------------------------------------------------------------
// Revision history
//==============================================================================

//==============================================================================
// Includes
#include "cWeapon.hpp"

#include "Debug.hpp"
#include "cParticleSystem.hpp"
#include "cTextureManager.hpp"
#include "cWorld.hpp"
#include "cMixer.hpp"
#include "cParticleEmitter.hpp"
//------------------------------------------------------------------------------
// Namespaces
using namespace ShootingStar;
//==============================================================================

//==============================================================================
//! Constructor
//------------------------------------------------------------------------------
cWeapon::cWeapon (int clipSize, Uint32 fireDelay, Uint32 reloadTime):
mState (State_Ready),
mActive (false),
mFire (false),
mClipSize (clipSize),
mCurrentClip (0),
mNumberOfClips (0),
mFireDelay (fireDelay),
mFireDelayLeft (0),
mReloadTime (reloadTime),
mReloadTimeLeft (0),
mSound (cMixer::GetInstance ().LoadSound ("empty.wav")),
mChannel(-2)
{
	// Empty
};
//==============================================================================

//==============================================================================
//! Destructor
//------------------------------------------------------------------------------
cWeapon::~cWeapon (void)
{
	// Empty
};
//==============================================================================

//==============================================================================
//! Update
//------------------------------------------------------------------------------
void 
cWeapon::Update (Uint32 deltaTime)
{
	if ( !mActive )
		return;
	
	if ( mpShooter == NULL )
		return;
		
	const cVector2f &position = mpShooter->GetEmittingPosition ();
	float angle = mpShooter->GetRotation ();
	
	switch ( mState )
	{
		case State_Reloading:
			if ( mReloadTimeLeft < deltaTime )
				mReloadTimeLeft = 0;
			else
				mReloadTimeLeft -= deltaTime;	
			if ( mReloadTimeLeft == 0 )
			{
				dbg::assertion (dbg::error, DBG_ASSERTION (mNumberOfClips > 0));
				mNumberOfClips--;
				mCurrentClip = mClipSize;
				mState = State_Ready;
				EndReload ();
			  	if ( mFire )
				  BeginFiring ();
			}
			break;
		case State_Ready:
			if ( mFireDelayLeft < deltaTime )
				mFireDelayLeft = 0;
			else
				mFireDelayLeft -= deltaTime;	
			if ( mFire && mCurrentClip > 0 && mFireDelayLeft == 0 )
			{
				mCurrentClip--;
				Fire (position, angle);
				mFireDelayLeft = mFireDelay;
			}
			if ( mCurrentClip == 0 )
			{
		
				EndFiring ();
				if ( mNumberOfClips > 0 )
				{
					mReloadTimeLeft = mReloadTime;
					mState = State_Reloading;
					BeginReload ();
				}
			}
			break;
		default:
			break;
	}
}
//==============================================================================

/*!
\todo Suuliekki -> english
*/
void 
cWeapon::SpawnSuuliekki (void)
{
	cParticleEmitter *pEmitter = dynamic_cast<cParticleEmitter*> (GetShooter ());
	
	// Spawn suuliekki
	cParticleSystem *pSystem = new cParticleSystem (10);
	pSystem->SetTexture (cTextureManager::GetInstance ().LoadTexture ("tuli2.png"));
	pSystem->SetAngleVariation (5.0f);
	pSystem->SetSpeed (0.4f, 1.0f);
	pSystem->SetBlending ();
	pSystem->SetSize (8.0f, 24.0f);
	pSystem->SetEnergy (150, 0.5f);
	pSystem->SetEmitDelay (0);
	pSystem->SetEmitter (pEmitter);
	pSystem->SetEndColor (1.0f, 1.0f, 1.0f, 0.5f);
	pSystem->SetEndColor (1.0f, 0.5f, 0.5f, 0.0f);
	pSystem->EmitAll ();
	pSystem->KillEmptySystem (true);
	cWorld::GetInstance ().SpawnObject (pSystem);
}

//==============================================================================
// EOF
//==============================================================================
