/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.importer.fetcher;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.apache.http.client.utils.URIBuilder;
import org.jabref.logic.help.HelpFile;
import org.jabref.logic.importer.FetcherException;
import org.jabref.logic.importer.FulltextFetcher;
import org.jabref.logic.importer.IdBasedFetcher;
import org.jabref.logic.importer.IdFetcher;
import org.jabref.logic.importer.ImportFormatPreferences;
import org.jabref.logic.importer.SearchBasedFetcher;
import org.jabref.logic.importer.fetcher.TrustLevel;
import org.jabref.logic.importer.util.OAI2Handler;
import org.jabref.logic.util.io.XMLUtil;
import org.jabref.logic.util.strings.StringSimilarity;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.BibtexEntryTypes;
import org.jabref.model.entry.LinkedFile;
import org.jabref.model.entry.identifier.ArXivIdentifier;
import org.jabref.model.entry.identifier.DOI;
import org.jabref.model.strings.StringUtil;
import org.jabref.model.util.OptionalUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;

public class ArXiv
implements FulltextFetcher,
SearchBasedFetcher,
IdBasedFetcher,
IdFetcher<ArXivIdentifier> {
    private static final Logger LOGGER = LoggerFactory.getLogger(ArXiv.class);
    private static final String API_URL = "https://export.arxiv.org/api/query";
    private final ImportFormatPreferences importFormatPreferences;

    public ArXiv(ImportFormatPreferences importFormatPreferences) {
        this.importFormatPreferences = importFormatPreferences;
    }

    @Override
    public Optional<URL> findFullText(BibEntry entry) throws IOException {
        Objects.requireNonNull(entry);
        try {
            Optional<URL> pdfUrl = this.searchForEntries(entry).stream().map(ArXivEntry::getPdfUrl).filter(Optional::isPresent).map(Optional::get).findFirst();
            if (pdfUrl.isPresent()) {
                LOGGER.info("Fulltext PDF found @ arXiv.");
            }
            return pdfUrl;
        }
        catch (FetcherException e) {
            LOGGER.warn("arXiv API request failed", e);
            return Optional.empty();
        }
    }

    @Override
    public TrustLevel getTrustLevel() {
        return TrustLevel.PREPRINT;
    }

    private Optional<ArXivEntry> searchForEntry(String searchQuery) throws FetcherException {
        List<ArXivEntry> entries = this.queryApi(searchQuery, Collections.emptyList(), 0, 1);
        if (entries.size() == 1) {
            return Optional.of(entries.get(0));
        }
        return Optional.empty();
    }

    private Optional<ArXivEntry> searchForEntryById(String id) throws FetcherException {
        Optional<ArXivIdentifier> identifier = ArXivIdentifier.parse(id);
        if (!identifier.isPresent()) {
            return Optional.empty();
        }
        List<ArXivEntry> entries = this.queryApi("", Collections.singletonList(identifier.get()), 0, 1);
        if (entries.size() >= 1) {
            return Optional.of(entries.get(0));
        }
        return Optional.empty();
    }

    private List<ArXivEntry> searchForEntries(BibEntry entry) throws FetcherException {
        String entryTitle;
        String arxivTitle;
        StringSimilarity match;
        String query;
        Optional<String> doi;
        Optional<String> identifier = entry.getField("eprint");
        if (StringUtil.isNotBlank(identifier)) {
            try {
                return OptionalUtil.toList(this.searchForEntryById(identifier.get()));
            }
            catch (FetcherException e) {
                LOGGER.warn("arXiv eprint API request failed", e);
            }
        }
        if ((doi = entry.getField("doi").flatMap(DOI::parse).map(DOI::getNormalized)).isPresent()) {
            query = "doi:" + doi.get();
        } else {
            Optional<String> authorQuery = entry.getField("author").map(author -> "au:" + author);
            Optional<String> titleQuery = entry.getField("title").map(title -> "ti:" + title);
            query = OptionalUtil.toList(authorQuery, titleQuery).stream().collect(Collectors.joining("+AND+"));
        }
        Optional<ArXivEntry> arxivEntry = this.searchForEntry(query);
        if (arxivEntry.isPresent() && (match = new StringSimilarity()).isSimilar(arxivTitle = arxivEntry.get().title.orElse(""), entryTitle = entry.getField("title").orElse(""))) {
            return OptionalUtil.toList(arxivEntry);
        }
        return Collections.emptyList();
    }

    private List<ArXivEntry> searchForEntries(String searchQuery) throws FetcherException {
        return this.queryApi(searchQuery, Collections.emptyList(), 0, 10);
    }

    private List<ArXivEntry> queryApi(String searchQuery, List<ArXivIdentifier> ids, int start, int maxResults) throws FetcherException {
        Document result = this.callApi(searchQuery, ids, start, maxResults);
        List<Node> entries = XMLUtil.asList(result.getElementsByTagName("entry"));
        return entries.stream().map(ArXivEntry::new).collect(Collectors.toList());
    }

    private Document callApi(String searchQuery, List<ArXivIdentifier> ids, int start, int maxResults) throws FetcherException {
        if (maxResults > 2000) {
            throw new IllegalArgumentException("The arXiv API limits the number of maximal results to be 2000");
        }
        try {
            URIBuilder uriBuilder = new URIBuilder(API_URL);
            if (StringUtil.isNotBlank(searchQuery)) {
                uriBuilder.addParameter("search_query", StringUtil.stripAccents(searchQuery));
            }
            if (!ids.isEmpty()) {
                uriBuilder.addParameter("id_list", ids.stream().map(ArXivIdentifier::getNormalized).collect(Collectors.joining(",")));
            }
            uriBuilder.addParameter("start", String.valueOf(start));
            uriBuilder.addParameter("max_results", String.valueOf(maxResults));
            URL url = uriBuilder.build().toURL();
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();
            HttpURLConnection connection = (HttpURLConnection)url.openConnection();
            if (connection.getResponseCode() == 400) {
                throw this.getException(builder.parse(connection.getErrorStream()));
            }
            return builder.parse(connection.getInputStream());
        }
        catch (IOException | URISyntaxException | ParserConfigurationException | SAXException exception) {
            throw new FetcherException("arXiv API request failed", exception);
        }
    }

    private FetcherException getException(Document error) {
        Node node;
        Optional<String> id;
        Boolean isError;
        List<Node> entries = XMLUtil.asList(error.getElementsByTagName("entry"));
        if (entries.size() == 1 && (isError = (id = XMLUtil.getNodeContent(node = entries.get(0), "id")).map(idContent -> idContent.startsWith("http://arxiv.org/api/errors")).orElse(false)).booleanValue()) {
            String errorMessage = XMLUtil.getNodeContent(node, "summary").orElse("Unknown error");
            return new FetcherException(errorMessage);
        }
        return new FetcherException("arXiv API request failed");
    }

    @Override
    public String getName() {
        return "ArXiv";
    }

    @Override
    public HelpFile getHelpPage() {
        return HelpFile.FETCHER_OAI2_ARXIV;
    }

    @Override
    public List<BibEntry> performSearch(String query) throws FetcherException {
        return this.searchForEntries(query).stream().map(arXivEntry -> arXivEntry.toBibEntry(this.importFormatPreferences.getKeywordSeparator())).collect(Collectors.toList());
    }

    @Override
    public Optional<BibEntry> performSearchById(String identifier) throws FetcherException {
        String cleanedIdentifier = identifier.trim();
        cleanedIdentifier = identifier.replaceAll(" ", "");
        return this.searchForEntryById(cleanedIdentifier).map(arXivEntry -> arXivEntry.toBibEntry(this.importFormatPreferences.getKeywordSeparator()));
    }

    @Override
    public Optional<ArXivIdentifier> findIdentifier(BibEntry entry) throws FetcherException {
        return this.searchForEntries(entry).stream().map(ArXivEntry::getId).filter(Optional::isPresent).map(Optional::get).findFirst();
    }

    @Override
    public String getIdentifierName() {
        return "ArXiv";
    }

    private static class ArXivEntry {
        private final Optional<String> title;
        private final Optional<String> urlAbstractPage;
        private final Optional<String> publishedDate;
        private final Optional<String> abstractText;
        private final List<String> authorNames;
        private final List<String> categories;
        private final Optional<URL> pdfUrl;
        private final Optional<String> doi;
        private final Optional<String> journalReferenceText;
        private final Optional<String> primaryCategory;

        public ArXivEntry(Node item) {
            this.title = XMLUtil.getNodeContent(item, "title").map(OAI2Handler::correctLineBreaks);
            this.urlAbstractPage = XMLUtil.getNodeContent(item, "id");
            this.publishedDate = XMLUtil.getNodeContent(item, "published");
            this.abstractText = XMLUtil.getNodeContent(item, "summary").map(OAI2Handler::correctLineBreaks).map(String::trim);
            this.authorNames = new ArrayList<String>();
            for (Node authorNode : XMLUtil.getNodesByName(item, "author")) {
                Optional<String> authorName = XMLUtil.getNodeContent(authorNode, "name").map(String::trim);
                authorName.ifPresent(this.authorNames::add);
            }
            this.categories = new ArrayList<String>();
            for (Node categoryNode : XMLUtil.getNodesByName(item, "category")) {
                Optional<String> category = XMLUtil.getAttributeContent(categoryNode, "term");
                category.ifPresent(this.categories::add);
            }
            Optional<Object> pdfUrlParsed = Optional.empty();
            for (Node linkNode : XMLUtil.getNodesByName(item, "link")) {
                Optional<String> linkTitle = XMLUtil.getAttributeContent(linkNode, "title");
                if (!linkTitle.equals(Optional.of("pdf"))) continue;
                pdfUrlParsed = XMLUtil.getAttributeContent(linkNode, "href").map(url -> {
                    try {
                        return new URL((String)url);
                    }
                    catch (MalformedURLException e) {
                        return null;
                    }
                });
            }
            this.pdfUrl = pdfUrlParsed;
            this.doi = XMLUtil.getNodeContent(item, "arxiv:doi");
            this.journalReferenceText = XMLUtil.getNodeContent(item, "arxiv:journal_ref");
            this.primaryCategory = XMLUtil.getNode(item, "arxiv:primary_category").flatMap(node -> XMLUtil.getAttributeContent(node, "term"));
        }

        public Optional<URL> getPdfUrl() {
            return this.pdfUrl;
        }

        public Optional<String> getIdString() {
            String prefix = "https://arxiv.org/abs/";
            return this.urlAbstractPage.map(abstractUrl -> {
                if (abstractUrl.startsWith(prefix)) {
                    return abstractUrl.substring(prefix.length());
                }
                return abstractUrl;
            });
        }

        public Optional<ArXivIdentifier> getId() {
            return this.getIdString().flatMap(ArXivIdentifier::parse);
        }

        public Optional<String> getDate() {
            return this.publishedDate.map(date -> {
                if (date.length() < 10) {
                    return null;
                }
                return date.substring(0, 10);
            });
        }

        public BibEntry toBibEntry(Character keywordDelimiter) {
            BibEntry bibEntry = new BibEntry();
            bibEntry.setType(BibtexEntryTypes.ARTICLE);
            bibEntry.setField("eprinttype", "arXiv");
            bibEntry.setField("author", String.join((CharSequence)" and ", this.authorNames));
            bibEntry.addKeywords(this.categories, keywordDelimiter);
            this.getIdString().ifPresent(id -> bibEntry.setField("eprint", (String)id));
            this.title.ifPresent(titleContent -> bibEntry.setField("title", (String)titleContent));
            this.doi.ifPresent(doiContent -> bibEntry.setField("doi", (String)doiContent));
            this.abstractText.ifPresent(abstractContent -> bibEntry.setField("abstract", (String)abstractContent));
            this.getDate().ifPresent(date -> bibEntry.setField("date", (String)date));
            this.primaryCategory.ifPresent(category -> bibEntry.setField("eprintclass", (String)category));
            this.journalReferenceText.ifPresent(journal -> bibEntry.setField("journaltitle", (String)journal));
            this.getPdfUrl().ifPresent(url -> bibEntry.setFiles(Collections.singletonList(new LinkedFile((URL)url, "PDF"))));
            return bibEntry;
        }
    }
}

