/* mkpkgutils.c
   $Id: mkpkgutils.c,v 1.3 2001/09/30 01:43:45 gwiley Exp $
   Glen Wiley, gwiley@ieee.org
  
   Copyright (c)2001, Glen Wiley
  
	Permission is hereby granted, free of charge, to any person
	obtaining a copy of this software and associated documentation
	files (the "Software"), to deal in the Software without
	restriction, including without limitation the rights to use, copy,
	modify, merge, publish, distribute, sublicense, and/or sell copies
	of the Software, and to permit persons to whom the Software is
	furnished to do so, subject to the following conditions: The above
	copyright notice and this permission notice shall be included in
	all copies or substantial portions of the Software.  THE SOFTWARE
	IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
	MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
	NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
	HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
	WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
	OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
	DEALINGS IN THE SOFTWARE.

*/

#if HAVE_CONFIG_H
 #include "config.h"
#endif
#include <errno.h>
#if HAVE_FCNTL_H
 #include <fcntl.h>
#endif
#if HAVE_GETOPT_H
 #include <getopt.h>
#endif
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#if HAVE_STRING_H
 #include <string.h>
#else
 #include <strings.h>
#endif
#include <sys/stat.h>
#include <sys/types.h>
#if HAVE_UNISTD_H
 #include <unistd.h>
#endif
#if HAVE_ZLIB_H
 #include <zlib.h>
#endif
#include "pdv.h"

extern char *g_prg;

/*---------------------------------------- findpdv
  locate the pdv stub that will be used as a prefix for the payload

  prg == argv[0], used to try to determine directory from which this
  was launched (in case path can't help us)

  hint == suggested pdv stub, if this is a valid executable then no
  search will be performed

  if showmsgs != 0 then fatal errors will print and then exit (used for
  the command line version)

  return the full path to the stub, or NULL if it was not found
  the caller is responsible for freeing the storage returned
*/
char *
findpdv(char *prg, char *hint, int showmsgs)
{
	char *fn = NULL;
	char *path;
	char *dirnm;
	char *fn_tst;
	char *p;
	long l;

	/* try the hint first */

	if(hint != NULL && *hint != '\0')
	{
		if(access(hint, R_OK | X_OK) == 0)
		{
			fn = strdup(hint);
			if(fn == NULL && showmsgs)
				fatalError(errno, "malloc (strdup) failed");

			return fn;
		}
	}

	/* get a sane value for max filename */

	l = pathconf("/", _PC_NAME_MAX);
	l = (l < 255 ? 255 : l);
	l = (l > 10000 ? 10000 : l);
	fn_tst = (char *) malloc(l + 1);
	if(fn_tst == NULL)
		fatalError(errno, "malloc failed");

	/* try the same directory that we were executed from */

	strcpy(fn_tst, prg);
	p = strrchr(fn_tst, '/');
	if(p)
		*p = '\0';
	else
		strcpy(fn_tst, ".");
	strcat(fn_tst, "/pdv");
	if(access(fn_tst, R_OK) == 0)
		fn = strdup(fn_tst);

	free(fn_tst);

	/* if we could not find it then try the search path
	   including the current directory as a last resort */

	if(fn == NULL)
	{
		path = (char *) malloc(strlen(getenv("PATH")) + 4);
		if(path == NULL)
			fatalError(errno, "malloc failed");

		sprintf(path, "%s:./", getenv("PATH"));

		dirnm = strtok(path, ":");
		while(dirnm)
		{
			sprintf(fn_tst, "%s/pdv", dirnm);
			if(access(fn_tst, R_OK) == 0)
			{
				fn = strdup(fn_tst);
				if(fn == NULL)
					fatalError(errno, "malloc failed");
				break;
			}
			dirnm = strtok(NULL, ":");
		}
	
		free(path);
	} /* if(fn == NULL) */

	return fn;
} /* findpdv */

/*---------------------------------------- writeHdrTxt
  write text to the header - used by newpackage() to add filenames and
  messages to the PDV header

  fd       file descriptor to write
  txt      text to write to the file
  showmsgs if != 0 then error messages are written to stderr
  pkg      package filename (used in error message)

  return 0 on success
*/
int
writeHdrTxt(int fd, const char *txt, int showmsgs, const char *pkg)
{
	int     retval = 0;
	int     l;
	ssize_t nbytes;

	if(txt)
	{
		l = strlen(txt);
		while((nbytes = write(fd, txt, l)) > 0)
		{
			txt += nbytes;
			l -= nbytes;
		}

		if(nbytes == -1)
			retval = errno;
	}

	if(retval == 0)
	{
		if(write(fd, "\0", 1) != 1)
			retval = errno;
	}

	if(showmsgs && retval != 0)
		fatalError(errno, "error writing package");

	return retval;
} /* writeHdrTxt */

/*---------------------------------------- newpkgfile
create a new package file, fn_pdv is the pdv stub file and
fn_pkg is the new file name, fn_pld is the name of the file
to append as a payload, value payloadstart with the offset
of the first byte of payload data

return 0 on success, errno on error
*/
int
newpkgfile(const char *fn_pdv, const char *fn_pkg, const char *fn_pld
 , struct payload_st *pld_data, int showmsgs)
{
	int    retval   = 0;
	int    allswell = 1;
	int    fd;
	int    l;
	char   c;
	char   offset[20]; /* we use this for storing a number - 20 is plenty */
	struct stat stbuf;

	if(showmsgs)
		printf("copying pdv...");

	/* create a copy of PDV with the same mode but current user/group */

	retval = filecopy(fn_pdv, fn_pkg, 1, getuid(), getgid(), 0, 0, 0);

	if(showmsgs)
	{
		if(retval == 0)
			printf("ok\n");
		else
			fatalError(retval, "error copying %s to %s", fn_pdv, fn_pkg);
	}

	/* append the payload to the package file */
	
	if(retval == 0)
	{
		stat(fn_pkg, &stbuf);
		pld_data->payloadstart = stbuf.st_size;

		if(showmsgs)
			printf("appending payload...");

		retval = filecopy(fn_pld, fn_pkg, 0, 0, 0, 0, 0, 1);

		if(showmsgs)
		{
			if(retval == 0)
				printf("ok\n");
			else
				fatalError(retval, "error appending %s to %s", fn_pld, fn_pkg);

			printf("done\n");
		}
	} /* if(retval == 0) */
	
	/* now write the meta data to the file */

	if(allswell)
	{
		fd = open(fn_pkg, O_RDWR);
		if(fd)
		{
			pld_data->metadatastart = lseek(fd, 0L, SEEK_END);

			/* write the ourput filename */

			allswell != writeHdrTxt(fd, pld_data->fn_out, showmsgs, fn_pkg);

			/* write the mode for the output file */

			if(allswell)
			{
				l = sizeof(mode_t);
				if(write(fd, &(pld_data->payldmode), l) != l)
				{
					if(showmsgs)
						fatalError(errno, "error writing %s", fn_pkg);
					retval   = errno;
					allswell = 0;
				}
			}

			/* write the command to execute */

			if(allswell)
				allswell != writeHdrTxt(fd, pld_data->cmd, showmsgs, fn_pkg);

			/* write the help message */

			if(allswell)
				allswell != writeHdrTxt(fd, pld_data->hlpmsg, showmsgs, fn_pkg);

			/* write the agreement message */

			if(allswell)
				allswell != writeHdrTxt(fd, pld_data->agrmsg, showmsgs, fn_pkg);

			/* write compressed flag */

			if(allswell)
			{
				c = (pld_data->iscompressed ? 1 : 0);
				if(write(fd, &c, 1) != 1)
				{
					if(showmsgs)
						fatalError(errno, "error writing %s", fn_pkg);
					retval   = errno;
					allswell = 0;
				}
			}

			/* write tar flag */

			if(allswell)
			{
				c = (pld_data->isatar ? 1 : 0);
				if(write(fd, &c, 1) != 1)
				{
					if(showmsgs)
						fatalError(errno, "error writing %s", fn_pkg);
					retval   = errno;
					allswell = 0;
				}
			}

			/* write payload start offset */

			if(allswell)
			{
				l = sizeof(off_t);
				if(write(fd, &(pld_data->payloadstart), l) != l)
				{
					if(showmsgs)
						fatalError(errno, "error writing %s", fn_pkg);
					retval   = errno;
					allswell = 0;
				}
			}

			/* write the metadata start offset */

			if(allswell)
			{
				l = sizeof(off_t);
				if(write(fd, &(pld_data->metadatastart), l) != l)
				{
					if(showmsgs)
						fatalError(errno, "error writing %s", fn_pkg);
					retval   = errno;
					allswell = 0;
				}
			}

			close(fd);
		} /* if(fd) */
		else
		{
			retval   = errno;
			allswell = 0;
			if(showmsgs)
				fatalError(errno, "unable to open %s for writing", fn_pkg);
		}
	} /* if(allswell) */

	return retval;
} /* newpkgfile */

/*---------------------------------------- fatalError
  display error message and exit */
void
fatalError(int err, const char *fmt, ...)
{
	char    *newfmt;
	va_list alist;

	newfmt = (char *) malloc(strlen(fmt) + 80);
	sprintf(newfmt, "\n%s: %s, %s\n\n", g_prg, fmt, strerror(err));

	va_start(alist, fmt);
	vfprintf(stderr, newfmt, alist);
	va_end(alist);

	free(newfmt);

	exit(err);
} /* fatalError */

/*---------------------------------------- readMsgFile
  read specified file as text for the help or agreement message into *msg,
  which should be freed by the caller

  return 0 on success
*/
int
readMsgFile(char *fn, char **msg)
{
	FILE *fh;
	char *p;
	int  alocd = 0;     /* total bytes allocated */
	int  totsz = 0;     /* total size of message */
	int  incsz = 1024;  /* bytes to grow buffer by */
	int  bufsz = 0;     /* bytes remaining in buffer */
	int  l;
	int  retval = 0;

	*msg = NULL;
	fh = fopen(fn, "r");
	if(fh)
	{
		*msg = (char *) malloc(incsz);
		if(*msg == NULL)
			retval = errno;
		bufsz   = incsz;
		alocd   = incsz;
		p       = *msg;

		while(retval == 0 && fgets(p, bufsz, fh) != NULL)
		{
			l = strlen(p);
			totsz += l;
			p     += l;
			bufsz -= l;
			if(bufsz < 80)
			{
				alocd += incsz;
				*msg = (char *) realloc(*msg, alocd);
				if(*msg == NULL)
					retval = errno;
				else
				{
					p     = *msg + totsz;
					bufsz += incsz;
				}
			}
		}

		fclose(fh);
	} /* if(fh) */
	else
		retval = errno;

	return retval;
} /* readMsgFile */

/*---------------------------------------- zipthefiles
  add the specified files to the package as a zip using zlib */
void
zipthefiles(const char *fn_pkg, const char **fn_pld, int nfiles)
{
	/* ummm - not quite there yet */

	return;
} /* zipthefiles */

/* mkpkgutils.c */
