/*
  Archive Diff - display differences between two archives
  Copyright (C) 2011  Christopher Howard
  
  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#define _XOPEN_SOURCE 700

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <dirent.h>
#include <string.h>
#include <getopt.h>
#include <stdbool.h>
#include <sys/wait.h>
#include <rremove.h>

#include "config.h"
#include "extract.h"
#include "concat.h"
#include "err.h"

char * skip_top_dir(const char * path);
void print_usage();
void print_version();
void die_archive_syntax_failure();
void execvp_wrap(const char * progname, char * const argv[]);

static const char optstring[] = "+bchV";

static const struct option longopts[] = {
  { "boring-mode", 0, NULL, 'b' },
  { "clean", 0, NULL, 'c' },
  { "help", 0, NULL, 'h' },
  { "version", 0, NULL, 'V' },
  { NULL, 0, NULL, 0 }
};

int main(int argc, char * argv[]) {
  bool boring_mode = false;
  bool clean_mode = false;
  while(1) {
    const int c = getopt_long(argc, argv, optstring, longopts, NULL);
    if(c == -1) { break; }
    switch(c) {
    case 'h':
      print_usage();
      exit(EXIT_SUCCESS);
    case 'V':
      print_version();
      exit(EXIT_SUCCESS);
    case 'b':
      boring_mode = true;
      break;
    case 'c':
      clean_mode = true;
      break;
    case '?':
      fprintf(stderr, "Try `archdiff --help' for more information.\n");
      exit(EXIT_FAILURE);
    }
  }  

  const char * arch1 = NULL;
  const char * arch2 = NULL;
  if(optind + 2 == argc) {
    arch1 = argv[optind++];
    arch2 = argv[optind++];
  } else {
    die_archive_syntax_failure();
  }
  if(arch1 == NULL || arch2 == NULL) {
    die_archive_syntax_failure();
  }
  /* determine temporary directory location */
  const char * tmpdir = getenv("TMPDIR");
  if(tmpdir == NULL) { tmpdir = P_tmpdir; }
  /* extract archives */
  const char * const wrkd1 = extract(arch1, tmpdir);
  const char * const wrkd2 = extract(arch2, tmpdir);
  /*
    We need to drop the top-level of the archives from our diff comparisons, if
    the top-level of the archives contain only one file file, a directory. This
    is because archive (particular source-code archives) will typically have all
    content bundled in one top-level directory, but the name of this directory
    may be version dependent. For example, two archives may contain almost
    entirely the same content, but because the top-level directory name is
    different, they will not be diff'd in a way that is useful for browsing.
   */
  const char * const modpath1 = skip_top_dir(wrkd1);
  const char * const modpath2 = skip_top_dir(wrkd2);
  /* making the external call to a diff program */
  {
    const char * const doption1 = "colordiff";
    const char * const doption2 = "diff";
    const char * dchoice;
    if(boring_mode == false) {
      dchoice = doption1;
    } else {
      dchoice = doption2;
    }
    const char * preferred_path1;
    const char * preferred_path2;
    if(modpath1 != NULL && modpath2 != NULL) {
      preferred_path1 = modpath1;
      preferred_path2 = modpath2;
    } else {
      preferred_path1 = wrkd1;
      preferred_path2 = wrkd2;
    }
    if(clean_mode == false) {
      char * const diff_cmd[] = { (char *) dchoice,
				  "-rNu",
				  (char *) preferred_path1,
				  (char *) preferred_path2,
				  (char *) NULL };
      execvp_wrap(dchoice, diff_cmd);
    } else {
      char * const diff_cmd[] = { (char *) dchoice,
				  "-rNu",
				  /* autotools, make */
				  "-x", "aclocal.m4",
				  "-x", "autogen.sh",
				  "-x", "configure",
				  "-x", "configure.ac",
				  "-x", "configure.in",
				  "-x", "config.guess",
				  "-x", "install-sh",
				  "-x", "m4",
				  "-x", "*.m4",
				  "-x", "Makefile",
				  "-x", "Makefile.am",
				  "-x", "Makefile.in",
				  "-x", "*.make",
				  "-x", "missing",
				  /* cmake */
				  "-x", "cmake",
				  "-x", "*.cmake",
				  "-x", "CMakeLists.txt",
				  (char *) preferred_path1,
				  (char *) preferred_path2,
				  (char *) NULL };
      execvp_wrap(dchoice, diff_cmd);
    }
  }
  /* deleting the temporary files */
  if(rremove(wrkd2, RR_STOP_ON_ERR, stderr) != RR_OKAY) {
    fprintf(stderr, "error: problem deleting temporary directory %s. aborting\n", wrkd2);
    exit(EXIT_FAILURE);
  }
  if(rremove(wrkd1, RR_STOP_ON_ERR, stderr) != RR_OKAY) {
    fprintf(stderr, "error: problem deleting temporary directory %s. aborting\n", wrkd1);
    exit(EXIT_FAILURE);
  }
  free((void *) wrkd2);
  free((void *) wrkd1);
  exit(EXIT_SUCCESS);
}

/*
  if PATH is a directory that contains only one directory entry, not including a
  link to the current directory or a parent directory, and that directory entry
  is itself a directory, then return a full path derived from the PATH and the
  directory entry name. PATH should not have a trailing forward-slash.
 */
char * skip_top_dir(const char * path) {
  DIR * const basedir = opendir(path);
  if(basedir == NULL) {
    const int e = errno;
    opendir_errmsg(e, path);
    exit(e);
  }
  const char * dirname_candidate = NULL;
  while(1) {
    const int e1 = errno;
    struct dirent * const entry = readdir(basedir);
    const int e2 = errno;
    if(entry == NULL) {
      if(e1 != e2) {
	fprintf(stderr, "error: problem processing temporary directory");
	readdir_errmsg(e2);
	exit(e2);
      } else {
	/* no more directories */
	if(dirname_candidate == NULL) {
	  /* empty directory? */
	  return NULL;
	} else {
	  return (char *) dirname_candidate;
	}
      }
    } else {
      /* found a file */
      const char * const current_dname = entry->d_name;
      if(strcmp(current_dname, ".") == 0 || strcmp(current_dname, "..") == 0) {
	continue;
      } else {
	if(dirname_candidate == NULL) {
	  const char * const current_dname_fullpath = concat(path, "/", current_dname, NULL);
	  struct stat s;
	  if(stat(current_dname_fullpath, &s) != 0) {
	    const int e = errno;
	    fprintf(stderr, "error: problem processing temporary directory");
	    stat_errmsg(e);
	    exit(e);
	  }
	  if(! S_ISDIR(s.st_mode)) {
	    /* there is a non-directory file at the top level */
	    free((void *) current_dname_fullpath);
	    return NULL;
	  } else {
	    free((void *) dirname_candidate); /* previous path mem may not have been released */
	    dirname_candidate = current_dname_fullpath;
	  }
	} else {
	  /* there is more than one file in the top level */
	  return NULL;
	}
      }
    }
  }
}

void print_usage() {
  printf("archdiff [-b] ARCHIVE1 ARCHIVE2" "\n"
  	 "archdiff -h|-V" "\n"
  	 "\n"
  	 "  -b, --boring-mode      no color (use regular diff instead of colordiff)" "\n"
	 "  -c, --clean            filter out some common configuration or installation files" "\n"
  	 "  -h, --help             print usage statement" "\n"
  	 "  -V, --version          print version statement" "\n"
  	 "\n"
	 "Shows the differences between the files in two archives, for browsing" "\n"
	 "purposes. Program failure will result in immediate termination with a return" "\n"
	 "value not equal to EXIT_SUCCESS, along with a message to the standard error" "\n"
	 "stream describing the nature of the failure or at least the errno value." "\n"
	 "\n"
  	 "Report bugs to: <" PACKAGE_BUGREPORT ">\n"
  	 "home page: <" PACKAGE_URL ">\n"
  	 );
}

void print_version() {
  printf(PACKAGE_STRING " Copyright (C) 2011 Christopher Howard" "\n"
	 "There is NO WARRANTY, to the extent permitted by law." "\n"
         "This is free software, and you are welcome to redistribute it" "\n"
	 "under certain conditions; see the COPYING file for details." "\n"
	 "\n"
  	 "Created and maintained by Christopher Howard, Frigid Code Enterprises." "\n"
  	 );
}

void die_archive_syntax_failure() {
  fprintf(stderr, "error: exactly two archives must be specified\n");
  exit(EXIT_FAILURE);
}

void execvp_wrap(const char * progname, char * const argv[]) {
  switch(fork()) {
  case(0):
    /* child ("Momma!") */
    if(execvp(progname, argv) == -1) {
      const int e = errno;
      exec_errmsg(e, progname);
      exit(EXIT_FAILURE);
    }
  case(-1):
    /* fork failure! */
    {
      const int e = errno;
      fprintf(stderr, "error: problem attempting to call %s\n", progname);
      fork_errmsg(e);
      exit(e);
    };
  default:
    /* parent */
    if(wait(NULL) == -1) {
      const int e = errno;
      fprintf(stderr, "error: problem attempting to call %s\n", progname);
      wait_errmsg(e);
      exit(e);
    }
  }
}
