// ampache_browser_main_window.cc
//
// Project: Ampache Browser
// License: GNU GPLv3
//
// Copyright (C) 2015 - 2024 Róbert Čerňanský



#include <Qt>
#include <QSize>
#include <QSizePolicy>
#include <QWidget>
#include <QToolBar>
#include <QStatusBar>
#include <QStyle>
#include <QAction>
#include <QLineEdit>
#include <QListView>
#include <QTreeView>
#include <QHBoxLayout>
#include <QDockWidget>
#include <QAbstractItemView>

#include "infrastructure/i18n.h"
#include "settings_dialog.h"
#include "custom_proxy_style.h"
#include "ampache_browser_main_window.h"

using namespace infrastructure;



namespace ui {

AmpacheBrowserMainWindow::AmpacheBrowserMainWindow(QWidget* parent): QMainWindow(parent) {
    setWindowFlags(Qt::Widget);

    myCustomProxyStyle = new CustomProxyStyle{};

    createAndSetupToolBar();
    createAndSetupAlbumsWidget();
    createAndSetupArtistsWidget();
    createAndSetupTracksWidget();

    statusBar()->setSizeGripEnabled(false);
}



AmpacheBrowserMainWindow::~AmpacheBrowserMainWindow() {
    delete(settingsDialog);
    delete(tracksTreeView);
    delete(artistsListView);
    delete(albumsListView);
    delete(searchLineEdit);
    delete(addToPlaylistAction);
    delete(createPlaylistAction);
    delete(playAction);
    delete(myCustomProxyStyle);
}



QSize AmpacheBrowserMainWindow::sizeHint() const {
    return QSize(570, 400);
}



void AmpacheBrowserMainWindow::createAndSetupToolBar()
{
    playAction = new QAction(style()->standardIcon(QStyle::SP_MediaPlay), _("Play"), this);
    playAction->setEnabled(false);
    createPlaylistAction = new QAction(style()->standardIcon(QStyle::SP_FileIcon), _("Create Playlist"), this);
    createPlaylistAction->setEnabled(false);
    addToPlaylistAction = new QAction(style()->standardIcon(QStyle::SP_MediaSeekForward), _("Add to Playlist"),
        this);
    addToPlaylistAction->setEnabled(false);
    searchLineEdit = new QLineEdit();
    auto spacerWidget = new QWidget();
    auto spacerWidget2 = new QWidget();
    auto mainToolBar = addToolBar(_("Main"));
    auto settingsAction = new QAction(style()->standardIcon(QStyle::SP_ComputerIcon), _("Settings"), this);

    searchLineEdit->setPlaceholderText(_("Search..."));
    searchLineEdit->setToolTip(_("Search for artists, albums and tracks."));

    spacerWidget->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    spacerWidget2->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    mainToolBar->setMovable(false);
    mainToolBar->addAction(playAction);
    mainToolBar->addAction(createPlaylistAction);
    mainToolBar->addAction(addToPlaylistAction);
    mainToolBar->addWidget(spacerWidget);
    mainToolBar->addWidget(spacerWidget2);
    mainToolBar->addWidget(searchLineEdit);
    mainToolBar->addSeparator();
    mainToolBar->addAction(settingsAction);
    
    settingsDialog = new SettingsDialog{};
    connect(settingsAction, SIGNAL(triggered()), settingsDialog, SLOT(open()));
}



void AmpacheBrowserMainWindow::createAndSetupAlbumsWidget() {
    auto albumsWidget = new QWidget{};
    auto centralLayout = new QHBoxLayout{};
    albumsListView = new QListView{};
    albumsListView->setViewMode(QListView::ViewMode::IconMode);
    albumsListView->setResizeMode(QListView::ResizeMode::Adjust);
    albumsListView->setWordWrap(true);
    albumsListView->setGridSize(QSize(ALBUM_THUMBNAIL_SIZE + 28, ALBUM_THUMBNAIL_SIZE + 92));
    albumsListView->setIconSize(QSize(ALBUM_THUMBNAIL_SIZE, ALBUM_THUMBNAIL_SIZE));
    albumsListView->setSelectionMode(QAbstractItemView::ExtendedSelection);
    albumsListView->setSelectionBehavior(QAbstractItemView::SelectRows);
    albumsListView->setStyle(myCustomProxyStyle);

    centralLayout->addWidget(albumsListView);
    albumsWidget->setLayout(centralLayout);
    setCentralWidget(albumsWidget);
}



void AmpacheBrowserMainWindow::createAndSetupArtistsWidget() {
    auto artistsDockWidget = new QDockWidget(_("Artists"));
    artistsListView = new QListView{};
    artistsListView->setResizeMode(QListView::ResizeMode::Adjust);
    artistsListView->setUniformItemSizes(true);
    artistsListView->setSelectionMode(QAbstractItemView::ExtendedSelection);
    artistsListView->setSelectionBehavior(QAbstractItemView::SelectRows);
    artistsListView->setStyle(myCustomProxyStyle);

    artistsDockWidget->setFeatures(QDockWidget::DockWidgetFloatable | QDockWidget::DockWidgetMovable);
    artistsDockWidget->setAllowedAreas(Qt::AllDockWidgetAreas);
    artistsDockWidget->setWidget(artistsListView);
    addDockWidget(Qt::RightDockWidgetArea, artistsDockWidget);
}



void AmpacheBrowserMainWindow::createAndSetupTracksWidget() {
    auto tracksDockWidget = new QDockWidget(_("Tracks"));
    tracksTreeView = new QTreeView{};
    tracksTreeView->setSelectionMode(QAbstractItemView::ExtendedSelection);
    tracksTreeView->setIndentation(0);
    tracksTreeView->setStyle(myCustomProxyStyle);

    tracksDockWidget->setFeatures(QDockWidget::DockWidgetFloatable | QDockWidget::DockWidgetMovable);
    tracksDockWidget->setAllowedAreas(Qt::AllDockWidgetAreas);
    tracksDockWidget->setWidget(tracksTreeView);
    addDockWidget(Qt::RightDockWidgetArea, tracksDockWidget);
}

}
