pub const ext = @import("ext.zig");
const tsparql = @This();

const std = @import("std");
const compat = @import("compat");
const gio = @import("gio2");
const gobject = @import("gobject2");
const glib = @import("glib2");
const gmodule = @import("gmodule2");
/// `TrackerBatch` executes a series of SPARQL updates and RDF data
/// insertions within a transaction.
///
/// A batch is created with `SparqlConnection.createBatch`.
/// To add resources use `Batch.addResource`,
/// `Batch.addSparql` or `Batch.addStatement`.
///
/// When a batch is ready for execution, use `Batch.execute`
/// or `Batch.executeAsync`. The batch is executed as a single
/// transaction, it will succeed or fail entirely.
///
/// This object has a single use, after the batch is executed it can
/// only be finished and freed.
///
/// The mapping of blank node labels is global in a `TrackerBatch`,
/// referencing the same blank node label in different operations in
/// a batch will resolve to the same resource.
pub const Batch = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = tsparql.BatchClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The `SparqlConnection` the batch belongs to.
        pub const connection = struct {
            pub const name = "connection";

            pub const Type = ?*tsparql.SparqlConnection;
        };
    };

    pub const signals = struct {};

    /// Inserts the RDF data contained in `stream` as part of `batch`.
    ///
    /// The RDF data will be inserted in the given `default_graph` if one is provided,
    /// or the anonymous graph if `default_graph` is `NULL`. Any RDF data that has a
    /// graph specified (e.g. using the `GRAPH` clause in the Trig format) will
    /// be inserted in the specified graph instead of `default_graph`.
    ///
    /// The `flags` argument is reserved for future expansions, currently
    /// `TRACKER_DESERIALIZE_FLAGS_NONE` must be passed.
    extern fn tracker_batch_add_rdf(p_batch: *Batch, p_flags: tsparql.DeserializeFlags, p_format: tsparql.RdfFormat, p_default_graph: [*:0]const u8, p_stream: *gio.InputStream) void;
    pub const addRdf = tracker_batch_add_rdf;

    /// Adds the RDF represented by `resource` to `batch`.
    extern fn tracker_batch_add_resource(p_batch: *Batch, p_graph: ?[*:0]const u8, p_resource: *tsparql.Resource) void;
    pub const addResource = tracker_batch_add_resource;

    /// Adds an SPARQL update string to `batch`.
    extern fn tracker_batch_add_sparql(p_batch: *Batch, p_sparql: [*:0]const u8) void;
    pub const addSparql = tracker_batch_add_sparql;

    /// Adds a `SparqlStatement` containing an SPARQL update. The statement will
    /// be executed once in the batch, with the parameters bound as specified in the
    /// variable arguments.
    ///
    /// The variable arguments are a NULL terminated set of variable name, type `gobject.Type`,
    /// and value. The value C type must correspond to the given `gobject.Type`. For example, for
    /// a statement that has a single `~name` parameter, it could be given a value for execution
    /// with the following code:
    ///
    /// ```c
    /// tracker_batch_add_statement (batch, stmt,
    ///                              "name", G_TYPE_STRING, "John Smith",
    ///                              NULL);
    /// ```
    ///
    /// A `SparqlStatement` may be used on multiple `Batch.addStatement`
    /// calls with the same or different values, on the same or different `TrackerBatch`
    /// objects.
    ///
    /// This function should only be called on `SparqlStatement` objects
    /// obtained through `SparqlConnection.updateStatement` or
    /// update statements loaded through `SparqlConnection.loadStatementFromGresource`.
    extern fn tracker_batch_add_statement(p_batch: *Batch, p_stmt: *tsparql.SparqlStatement, ...) void;
    pub const addStatement = tracker_batch_add_statement;

    /// Adds a `SparqlStatement` containing an SPARQL update. The statement will
    /// be executed once in the batch, with the values bound as specified by `variable_names`
    /// and `values`.
    ///
    /// For example, for a statement that has a single `~name` parameter,
    /// it could be given a value for execution with the given code:
    ///
    /// ```c
    /// const char *names = { "name" };
    /// const GValue values[G_N_ELEMENTS (names)] = { 0, };
    ///
    /// g_value_init (&values[0], G_TYPE_STRING);
    /// g_value_set_string (&values[0], "John Smith");
    /// tracker_batch_add_statementv (batch, stmt,
    ///                               G_N_ELEMENTS (names),
    ///                               names, values);
    /// ```
    /// ```python
    /// batch.add_statement(stmt, ['name'], ['John Smith']);
    /// ```
    /// ```js
    /// batch.add_statement(stmt, ['name'], ['John Smith']);
    /// ```
    ///
    /// A `SparqlStatement` may be used on multiple `Batch.addStatement`
    /// calls with the same or different values, on the same or different `TrackerBatch`
    /// objects.
    ///
    /// This function should only be called on `SparqlStatement` objects
    /// obtained through `SparqlConnection.updateStatement` or
    /// update statements loaded through `SparqlConnection.loadStatementFromGresource`.
    extern fn tracker_batch_add_statementv(p_batch: *Batch, p_stmt: *tsparql.SparqlStatement, p_n_values: c_uint, p_variable_names: [*][*:0]const u8, p_values: [*]const gobject.Value) void;
    pub const addStatementv = tracker_batch_add_statementv;

    /// Executes the batch. This operations happens synchronously.
    extern fn tracker_batch_execute(p_batch: *Batch, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const execute = tracker_batch_execute;

    /// Executes the batch. This operation happens asynchronously, when
    /// finished `callback` will be executed.
    extern fn tracker_batch_execute_async(p_batch: *Batch, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const executeAsync = tracker_batch_execute_async;

    /// Finishes the operation started with `Batch.executeAsync`.
    extern fn tracker_batch_execute_finish(p_batch: *Batch, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const executeFinish = tracker_batch_execute_finish;

    /// Returns the `SparqlConnection` that this batch was created
    /// from.
    extern fn tracker_batch_get_connection(p_batch: *Batch) *tsparql.SparqlConnection;
    pub const getConnection = tracker_batch_get_connection;

    extern fn tracker_batch_get_type() usize;
    pub const getGObjectType = tracker_batch_get_type;

    extern fn g_object_ref(p_self: *tsparql.Batch) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *tsparql.Batch) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Batch, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `TrackerEndpoint` is a helper object to make RDF triple stores represented
/// by a `SparqlConnection` publicly available to other processes/hosts.
///
/// This is a base abstract object, see `EndpointDBus` to make
/// RDF triple stores available to other processes in the same machine, and
/// `EndpointHttp` to make it available to other hosts in the
/// network.
///
/// When the RDF triple store represented by a `SparqlConnection`
/// is made public this way, other peers may connect to the database using
/// `SparqlConnection.busNew` or `SparqlConnection.remoteNew`
/// to access this endpoint exclusively, or they may use the `SERVICE <uri> { ... }` SPARQL
/// syntax from their own `SparqlConnection`s to expand their data set.
///
/// By default, and as long as the underlying `SparqlConnection`
/// allows SPARQL updates and RDF graph changes, endpoints will allow updates
/// and modifications to happen through them. Use `Endpoint.setReadonly`
/// to change this behavior.
///
/// By default, endpoints allow access to every RDF graph in the triple store
/// and further external SPARQL endpoints to the queries performed on it. Use
/// `Endpoint.setAllowedGraphs` and
/// `Endpoint.setAllowedServices` to change this behavior. Users do
/// not typically need to do this for D-Bus endpoints, as these do already have a layer
/// of protection with the Tracker portal. This is the mechanism used by the portal
/// itself. This access control API may not interoperate with other SPARQL endpoint
/// implementations than Tracker.
pub const Endpoint = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = tsparql.EndpointClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// RDF graphs that are allowed to be accessed
        /// through queries to this endpoint. See
        /// `tsparql.Endpoint.setAllowedGraphs`.
        pub const allowed_graphs = struct {
            pub const name = "allowed-graphs";

            pub const Type = ?[*][*:0]u8;
        };

        /// External SPARQL endpoints that are allowed to be
        /// accessed through queries to this endpint. See
        /// `tsparql.Endpoint.setAllowedServices`.
        pub const allowed_services = struct {
            pub const name = "allowed-services";

            pub const Type = ?[*][*:0]u8;
        };

        /// Whether the endpoint allows SPARQL updates or not. See
        /// `tsparql.Endpoint.setReadonly`.
        pub const readonly = struct {
            pub const name = "readonly";

            pub const Type = c_int;
        };

        /// The `SparqlConnection` being proxied by this endpoint.
        pub const sparql_connection = struct {
            pub const name = "sparql-connection";

            pub const Type = ?*tsparql.SparqlConnection;
        };
    };

    pub const signals = struct {};

    /// Returns the list of RDF graphs that the endpoint allows
    /// access for.
    extern fn tracker_endpoint_get_allowed_graphs(p_endpoint: *Endpoint) [*][*:0]u8;
    pub const getAllowedGraphs = tracker_endpoint_get_allowed_graphs;

    /// Returns the list of external SPARQL endpoints that are
    /// allowed to be accessed through this endpoint.
    extern fn tracker_endpoint_get_allowed_services(p_endpoint: *Endpoint) [*][*:0]u8;
    pub const getAllowedServices = tracker_endpoint_get_allowed_services;

    /// Returns whether the endpoint is readonly, thus SPARQL update
    /// queries are disallowed.
    extern fn tracker_endpoint_get_readonly(p_endpoint: *Endpoint) c_int;
    pub const getReadonly = tracker_endpoint_get_readonly;

    /// Returns the `SparqlConnection` that this endpoint proxies
    /// to a wider audience.
    extern fn tracker_endpoint_get_sparql_connection(p_endpoint: *Endpoint) *tsparql.SparqlConnection;
    pub const getSparqlConnection = tracker_endpoint_get_sparql_connection;

    /// Sets the list of RDF graphs that this endpoint will allow access
    /// for. Any explicit (e.g. `GRAPH` keyword) or implicit (e.g. through the
    /// default anonymous graph) access to RDF graphs unespecified in this
    /// list in SPARQL queries will be seen as if those graphs did not exist, or
    /// (equivalently) had an empty set. Changes to these graphs through SPARQL
    /// updates will also be disallowed.
    ///
    /// If `graphs` is `NULL`, access will be allowed to every RDF graph stored
    /// in the endpoint, this is the default behavior. If you want to forbid access
    /// to all RDF graphs, use an empty list.
    ///
    /// The empty string (`""`) is allowed as a special value, to allow access
    /// to the stock anonymous graph. All graph names are otherwise dependent
    /// on the endpoint and its contained data.
    extern fn tracker_endpoint_set_allowed_graphs(p_endpoint: *Endpoint, p_graphs: *const [*:0]const u8) void;
    pub const setAllowedGraphs = tracker_endpoint_set_allowed_graphs;

    /// Sets the list of external SPARQL endpoints that this endpoint
    /// will allow access for. Access through the `SERVICE` SPARQL syntax
    /// will fail for services not specified in this list.
    ///
    /// If `services` is `NULL`, access will be allowed to every external endpoint,
    /// this is the default behavior. If you want to forbid access to all
    /// external SPARQL endpoints, use an empty list.
    ///
    /// This affects both remote SPARQL endpoints accessed through HTTP,
    /// and external SPARQL endpoints offered through D-Bus. For the latter,
    /// the following syntax is allowed to describe them as an URI:
    ///
    /// `DBUS_URI = 'dbus:' [ ('system' | 'session') ':' ]? dbus-name [ ':' object-path ]?`
    ///
    /// If the system/session part is omitted, it will default to the session
    /// bus. If the object path is omitted, the `/org/freedesktop/Tracker3/Endpoint`
    /// `EndpointDBus` default will be assumed.
    extern fn tracker_endpoint_set_allowed_services(p_endpoint: *Endpoint, p_services: *const [*:0]const u8) void;
    pub const setAllowedServices = tracker_endpoint_set_allowed_services;

    /// Sets whether the endpoint will be readonly. Readonly endpoints
    /// will not allow SPARQL update queries. The underlying
    /// `SparqlConnection` may be readonly of its own, this
    /// method does not change its behavior in any way.
    extern fn tracker_endpoint_set_readonly(p_endpoint: *Endpoint, p_readonly: c_int) void;
    pub const setReadonly = tracker_endpoint_set_readonly;

    extern fn tracker_endpoint_get_type() usize;
    pub const getGObjectType = tracker_endpoint_get_type;

    extern fn g_object_ref(p_self: *tsparql.Endpoint) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *tsparql.Endpoint) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Endpoint, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `TrackerEndpointDBus` makes the RDF data in a `SparqlConnection`
/// accessible to other processes via DBus.
///
/// This object is a `Endpoint` subclass that exports
/// a `SparqlConnection` so its RDF data is accessible to other
/// processes through the given `gio.DBusConnection`.
///
/// ```c
/// // This process already has org.example.Endpoint bus name
/// endpoint = tracker_endpoint_dbus_new (sparql_connection,
///                                       dbus_connection,
///                                       NULL,
///                                       NULL,
///                                       &error);
///
/// // From another process
/// connection = tracker_sparql_connection_bus_new ("org.example.Endpoint",
///                                                 NULL,
///                                                 dbus_connection,
///                                                 &error);
/// ```
///
/// The `TrackerEndpointDBus` will manage a DBus object at the given path
/// with the `org.freedesktop.Tracker3.Endpoint` interface, if no path is
/// given the object will be at the default `/org/freedesktop/Tracker3/Endpoint`
/// location.
///
/// Access to D-Bus endpoints may be managed via the
/// `EndpointDBus.signals.block_call` signal, the boolean
/// return value expressing whether the request is blocked or not.
/// Inspection of the requester address is left up to the user. The
/// default value allows all requests independently of their provenance.
///
/// However, moderating access to D-Bus interfaces is typically not necessary
/// in user code, as access to public D-Bus endpoints will be transparently
/// managed through the Tracker portal service for applications sandboxed
/// via XDG portals. These already have access to D-Bus SPARQL endpoints and
/// their data naturally filtered as defined in the application manifest.
///
/// A `TrackerEndpointDBus` may be created on a different thread/main
/// context from the one that created `SparqlConnection`.
pub const EndpointDBus = opaque {
    pub const Parent = tsparql.Endpoint;
    pub const Implements = [_]type{gio.Initable};
    pub const Class = opaque {
        pub const Instance = EndpointDBus;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The `gio.DBusConnection` where the connection is proxied through.
        pub const dbus_connection = struct {
            pub const name = "dbus-connection";

            pub const Type = ?*gio.DBusConnection;
        };

        /// The DBus object path that this endpoint manages.
        pub const object_path = struct {
            pub const name = "object-path";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {
        pub const block_call = struct {
            pub const name = "block-call";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_object: [*:0]u8, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EndpointDBus, p_instance))),
                    gobject.signalLookup("block-call", EndpointDBus.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Registers a Tracker endpoint object at `object_path` on `dbus_connection`.
    /// The default object path is `/org/freedesktop/Tracker3/Endpoint`.
    extern fn tracker_endpoint_dbus_new(p_sparql_connection: *tsparql.SparqlConnection, p_dbus_connection: *gio.DBusConnection, p_object_path: ?[*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*tsparql.EndpointDBus;
    pub const new = tracker_endpoint_dbus_new;

    extern fn tracker_endpoint_dbus_get_type() usize;
    pub const getGObjectType = tracker_endpoint_dbus_get_type;

    extern fn g_object_ref(p_self: *tsparql.EndpointDBus) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *tsparql.EndpointDBus) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *EndpointDBus, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `TrackerEndpointHttp` makes the RDF data in a `SparqlConnection`
/// accessible to other hosts via HTTP.
///
/// This object is a `Endpoint` subclass that exports
/// a `SparqlConnection` so its RDF data is accessible via HTTP
/// requests on the given port. This endpoint implementation is compliant
/// with the [SPARQL protocol specifications](https://www.w3.org/TR/2013/REC-sparql11-protocol-20130321/)
/// and may interoperate with other implementations.
///
/// ```c
/// // This host has "example.local" hostname
/// endpoint = tracker_endpoint_http_new (sparql_connection,
///                                       8080,
///                                       tls_certificate,
///                                       NULL,
///                                       &error);
///
/// // From another host
/// connection = tracker_sparql_connection_remote_new ("http://example.local:8080/sparql");
/// ```
///
/// Access to HTTP endpoints may be managed via the
/// `EndpointHttp.signals.block_remote_address` signal, the boolean
/// return value expressing whether the connection is blocked or not.
/// Inspection of the requester address is left up to the user. The
/// default value allows all requests independently of their provenance,
/// users are encouraged to add a handler.
///
/// If the provided `gio.TlsCertificate` is `NULL`, the endpoint will allow
/// plain HTTP connections. Users are encouraged to provide a certificate
/// in order to use HTTPS.
///
/// As a security measure, and in compliance specifications,
/// the HTTP endpoint does not handle database updates or modifications in any
/// way. The database content is considered to be entirely managed by the
/// process that creates the HTTP endpoint and owns the `SparqlConnection`.
///
/// A `TrackerEndpointHttp` may be created on a different thread/main
/// context from the one that created `SparqlConnection`.
pub const EndpointHttp = opaque {
    pub const Parent = tsparql.Endpoint;
    pub const Implements = [_]type{gio.Initable};
    pub const Class = opaque {
        pub const Instance = EndpointHttp;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// `gio.TlsCertificate` to encrypt the communication.
        pub const http_certificate = struct {
            pub const name = "http-certificate";

            pub const Type = ?*gio.TlsCertificate;
        };

        /// HTTP port used to listen requests.
        pub const http_port = struct {
            pub const name = "http-port";

            pub const Type = c_uint;
        };
    };

    pub const signals = struct {
        /// Allows control over the connections stablished. The given
        /// address is that of the requesting peer.
        ///
        /// Returning `FALSE` in this handler allows the connection,
        /// returning `TRUE` blocks it. The default with no signal
        /// handlers connected is `FALSE`.
        pub const block_remote_address = struct {
            pub const name = "block-remote-address";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_address: *gio.SocketAddress, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(EndpointHttp, p_instance))),
                    gobject.signalLookup("block-remote-address", EndpointHttp.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Sets up a Tracker endpoint to listen via HTTP, in the given `port`.
    /// If `certificate` is not `NULL`, HTTPS may be used to connect to the
    /// endpoint.
    extern fn tracker_endpoint_http_new(p_sparql_connection: *tsparql.SparqlConnection, p_port: c_uint, p_certificate: ?*gio.TlsCertificate, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*tsparql.EndpointHttp;
    pub const new = tracker_endpoint_http_new;

    extern fn tracker_endpoint_http_get_type() usize;
    pub const getGObjectType = tracker_endpoint_http_get_type;

    extern fn g_object_ref(p_self: *tsparql.EndpointHttp) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *tsparql.EndpointHttp) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *EndpointHttp, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `TrackerNamespaceManager` object represents a mapping between namespaces and
/// their shortened prefixes.
///
/// This object keeps track of namespaces, and allows you to assign
/// short prefixes for them to avoid frequent use of full namespace IRIs. The syntax
/// used is that of [Compact URIs (CURIEs)](https://www.w3.org/TR/2010/NOTE-curie-20101216).
///
/// Usually you will want to use a namespace manager obtained through
/// `SparqlConnection.getNamespaceManager` from the
/// `SparqlConnection` that manages the RDF data, as that will
/// contain all prefixes and namespaces that are pre-defined by its ontology.
pub const NamespaceManager = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = tsparql.NamespaceManagerClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Returns the global `TrackerNamespaceManager` that contains a set of well-known
    /// namespaces and prefixes, such as `rdf:`, `rdfs:`, `nie:`, `tracker:`, etc.
    ///
    /// Note that the list of prefixes and namespaces is hardcoded in
    /// libtracker-sparql. It may not correspond with the installed set of
    /// ontologies, if they have been modified since they were installed.
    extern fn tracker_namespace_manager_get_default() *tsparql.NamespaceManager;
    pub const getDefault = tracker_namespace_manager_get_default;

    /// Creates a new, empty `TrackerNamespaceManager` instance.
    extern fn tracker_namespace_manager_new() *tsparql.NamespaceManager;
    pub const new = tracker_namespace_manager_new;

    /// Adds `prefix` as the recognised abbreviaton of `namespace`.
    ///
    /// Only one prefix is allowed for a given namespace, and all prefixes must
    /// be unique.
    ///
    /// Since 3.3, The `TrackerNamespaceManager` instances obtained through
    /// `SparqlConnection.getNamespaceManager` are "sealed",
    /// this API call should not performed on those.
    extern fn tracker_namespace_manager_add_prefix(p_self: *NamespaceManager, p_prefix: [*:0]const u8, p_ns: [*:0]const u8) void;
    pub const addPrefix = tracker_namespace_manager_add_prefix;

    /// If `uri` begins with one of the namespaces known to this
    /// `TrackerNamespaceManager`, then the return value will be the
    /// compressed URI. Otherwise, `NULL` will be returned.
    extern fn tracker_namespace_manager_compress_uri(p_self: *NamespaceManager, p_uri: [*:0]const u8) [*:0]u8;
    pub const compressUri = tracker_namespace_manager_compress_uri;

    /// If `compact_uri` begins with one of the prefixes known to this
    /// `TrackerNamespaceManager`, then the return value will be the
    /// expanded URI. Otherwise, a copy of `compact_uri` will be returned.
    extern fn tracker_namespace_manager_expand_uri(p_self: *NamespaceManager, p_compact_uri: [*:0]const u8) [*:0]u8;
    pub const expandUri = tracker_namespace_manager_expand_uri;

    /// Calls `func` for each known prefix / URI pair.
    extern fn tracker_namespace_manager_foreach(p_self: *NamespaceManager, p_func: glib.HFunc, p_user_data: ?*anyopaque) void;
    pub const foreach = tracker_namespace_manager_foreach;

    /// Returns whether `prefix` is known.
    extern fn tracker_namespace_manager_has_prefix(p_self: *NamespaceManager, p_prefix: [*:0]const u8) c_int;
    pub const hasPrefix = tracker_namespace_manager_has_prefix;

    /// Looks up the namespace URI corresponding to `prefix`, or `NULL` if the prefix
    /// is not known.
    extern fn tracker_namespace_manager_lookup_prefix(p_self: *NamespaceManager, p_prefix: [*:0]const u8) ?[*:0]const u8;
    pub const lookupPrefix = tracker_namespace_manager_lookup_prefix;

    /// Writes out all namespaces as ``prefix`` statements in
    /// the [Turtle](https://www.w3.org/TR/turtle/) RDF format.
    extern fn tracker_namespace_manager_print_turtle(p_self: *NamespaceManager) [*:0]u8;
    pub const printTurtle = tracker_namespace_manager_print_turtle;

    extern fn tracker_namespace_manager_get_type() usize;
    pub const getGObjectType = tracker_namespace_manager_get_type;

    extern fn g_object_ref(p_self: *tsparql.NamespaceManager) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *tsparql.NamespaceManager) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *NamespaceManager, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `TrackerNotifier` allows receiving notification on changes
/// in the data stored by a `SparqlConnection`.
///
/// This object may be created through `SparqlConnection.createNotifier`,
/// events can then be listened for by connecting to the
/// `Notifier.signals.events` signal.
///
/// Not every change is notified, only RDF resources with a
/// class that has the [nrl:notify](nrl-ontology.html`nrl.properties.notify`)
/// property defined by the ontology will be notified upon changes.
///
/// Database changes are communicated through `NotifierEvent` events on
/// individual graph/resource pairs. The event type obtained through
/// `NotifierEvent.getEventType` will determine the type of event.
/// Insertion of new resources is notified through
/// `TRACKER_NOTIFIER_EVENT_CREATE` events, deletion of
/// resources is notified through `TRACKER_NOTIFIER_EVENT_DELETE`
/// events, and changes on any property of the resource is notified
/// through `TRACKER_NOTIFIER_EVENT_UPDATE` events.
///
/// The events happen in reaction to database changes, after a `TrackerNotifier`
/// received an event of type `TRACKER_NOTIFIER_EVENT_DELETE`, the resource will
/// not exist anymore and only the information in the `NotifierEvent`
/// will remain.
///
/// Similarly, when receiving an event of type `TRACKER_NOTIFIER_EVENT_UPDATE`,
/// the resource will have already changed, so the data previous to the update is
/// no longer available.
///
/// The `Notifier.signals.events` signal is emitted in the thread-default
/// main context of the thread where the `TrackerNotifier` instance was created.
pub const Notifier = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = tsparql.NotifierClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// SPARQL connection to listen to.
        pub const connection = struct {
            pub const name = "connection";

            pub const Type = ?*tsparql.SparqlConnection;
        };
    };

    pub const signals = struct {
        /// Notifies of changes in the Tracker database.
        pub const events = struct {
            pub const name = "events";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_service: [*:0]u8, p_graph: [*:0]u8, p_events: [*]*tsparql.NotifierEvent, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Notifier, p_instance))),
                    gobject.signalLookup("events", Notifier.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Listens to notification events from a remote DBus SPARQL endpoint.
    ///
    /// If `connection` refers to a message bus (system/session), `service` must refer
    /// to a D-Bus name (either unique or well-known). If `connection` is a non-message
    /// bus (e.g. a peer-to-peer D-Bus connection) the `service` argument may be `NULL`.
    ///
    /// If the `object_path` argument is `NULL`, the default
    /// `/org/freedesktop/Tracker3/Endpoint` path will be
    /// used. If `graph` is `NULL`, all graphs will be listened for.
    ///
    /// The signal subscription can be removed with
    /// `Notifier.signalUnsubscribe`.
    ///
    /// Note that this call is not necessary to receive notifications on
    /// a connection obtained through `SparqlConnection.busNew`,
    /// only to listen to update notifications from additional DBus endpoints.
    extern fn tracker_notifier_signal_subscribe(p_notifier: *Notifier, p_connection: *gio.DBusConnection, p_service: ?[*:0]const u8, p_object_path: ?[*:0]const u8, p_graph: ?[*:0]const u8) c_uint;
    pub const signalSubscribe = tracker_notifier_signal_subscribe;

    /// Undoes a signal subscription done through `Notifier.signalSubscribe`.
    ///
    /// The `handler_id` argument was previously obtained during signal subscription creation.
    extern fn tracker_notifier_signal_unsubscribe(p_notifier: *Notifier, p_handler_id: c_uint) void;
    pub const signalUnsubscribe = tracker_notifier_signal_unsubscribe;

    extern fn tracker_notifier_get_type() usize;
    pub const getGObjectType = tracker_notifier_get_type;

    extern fn g_object_ref(p_self: *tsparql.Notifier) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *tsparql.Notifier) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Notifier, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `TrackerResource` is an in-memory representation of RDF data about a given resource.
///
/// This object keeps track of a set of properties for a given resource, and can
/// also link to other `TrackerResource` objects to form trees or graphs of RDF
/// data. See `Resource.setRelation` and `Resource.setUri`
/// on how to link a `TrackerResource` to other RDF data.
///
/// `TrackerResource` may also hold data about literal values, added through
/// the specialized `Resource.setInt64`, `Resource.setString`,
/// etc family of functions, or the generic `Resource.setGvalue` method.
///
/// Since RDF properties may be multi-valued, for every `set` call there exists
/// another `add` call (e.g. `Resource.addInt64`, `Resource.addString`
/// and so on). The `set` methods do also reset any previously value the
/// property might hold for the given resource.
///
/// Resources may have an IRI set at creation through `Resource.new`,
/// or set afterwards through `Resource.setIdentifier`. Resources
/// without a name will represent a blank node, and will be dealt with as such
/// during database insertions.
///
/// `TrackerResource` performs no validation on the data being coherent as per
/// any ontology. Errors will be found out at the time of using the TrackerResource
/// for e.g. database updates.
///
/// Once the RDF data is built in memory, the (tree of) `TrackerResource` may be
/// converted to a RDF format through `Resource.printRdf`, or
/// directly inserted into a database through `Batch.addResource`
/// or `SparqlConnection.updateResource`.
pub const Resource = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = tsparql.ResourceClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The URI identifier for this class, or `NULL` for a
        /// blank node.
        pub const identifier = struct {
            pub const name = "identifier";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Deserializes a `TrackerResource` previously serialized with
    /// `Resource.serialize`. It is implied that both ends
    /// use a common `NamespaceManager`.
    extern fn tracker_resource_deserialize(p_variant: *glib.Variant) ?*tsparql.Resource;
    pub const deserialize = tracker_resource_deserialize;

    /// Creates a TrackerResource instance.
    extern fn tracker_resource_new(p_identifier: ?[*:0]const u8) *tsparql.Resource;
    pub const new = tracker_resource_new;

    /// Adds a boolean property. Previous values for the same property are kept.
    ///
    /// This method is meant for RDF properties allowing multiple values, see
    /// [nrl:maxCardinality](nrl-ontology.html`nrl.properties.maxCardinality`).
    ///
    /// This method corresponds to [xsd:boolean](xsd-ontology.html`xsd.properties.boolean`).
    extern fn tracker_resource_add_boolean(p_self: *Resource, p_property_uri: [*:0]const u8, p_value: c_int) void;
    pub const addBoolean = tracker_resource_add_boolean;

    /// Adds a date property as a `glib.DateTime`. Previous values for the
    /// same property are kept.
    ///
    /// This method is meant for RDF properties allowing multiple values, see
    /// [nrl:maxCardinality](nrl-ontology.html`nrl.properties.maxCardinality`).
    ///
    /// This method corresponds to [xsd:date](xsd-ontology.html`xsd.properties.date`) and
    /// [xsd:dateTime](xsd-ontology.html`xsd.properties.dateTime`).
    extern fn tracker_resource_add_datetime(p_self: *Resource, p_property_uri: [*:0]const u8, p_value: *glib.DateTime) void;
    pub const addDatetime = tracker_resource_add_datetime;

    /// Adds a numeric property with double precision. Previous values for the same property are kept.
    ///
    /// This method is meant for RDF properties allowing multiple values, see
    /// [nrl:maxCardinality](nrl-ontology.html`nrl.properties.maxCardinality`).
    ///
    /// This method corresponds to [xsd:double](xsd-ontology.html`xsd.properties.double`).
    extern fn tracker_resource_add_double(p_self: *Resource, p_property_uri: [*:0]const u8, p_value: f64) void;
    pub const addDouble = tracker_resource_add_double;

    /// Add `value` to the list of values for given property.
    ///
    /// You can pass any kind of `gobject.Value` for `value`, but serialization functions will
    /// normally only be able to serialize URIs/relationships and fundamental value
    /// types (string, int, etc.).
    extern fn tracker_resource_add_gvalue(p_self: *Resource, p_property_uri: [*:0]const u8, p_value: *const gobject.Value) void;
    pub const addGvalue = tracker_resource_add_gvalue;

    /// Adds a numeric property with integer precision. Previous values for the same property are kept.
    ///
    /// This method is meant for RDF properties allowing multiple values, see
    /// [nrl:maxCardinality](nrl-ontology.html`nrl.properties.maxCardinality`).
    ///
    /// This method corresponds to [xsd:integer](xsd-ontology.html`xsd.properties.integer`).
    extern fn tracker_resource_add_int(p_self: *Resource, p_property_uri: [*:0]const u8, p_value: c_int) void;
    pub const addInt = tracker_resource_add_int;

    /// Adds a numeric property with 64-bit integer precision. Previous values for the same property are kept.
    ///
    /// This method is meant for RDF properties allowing multiple values, see
    /// [nrl:maxCardinality](nrl-ontology.html`nrl.properties.maxCardinality`).
    ///
    /// This method corresponds to [xsd:integer](xsd-ontology.html`xsd.properties.integer`).
    extern fn tracker_resource_add_int64(p_self: *Resource, p_property_uri: [*:0]const u8, p_value: i64) void;
    pub const addInt64 = tracker_resource_add_int64;

    /// Adds a resource property as a `TrackerResource`. Previous values for the same property are kept.
    ///
    /// This method is meant for RDF properties allowing multiple values, see
    /// [nrl:maxCardinality](nrl-ontology.html`nrl.properties.maxCardinality`).
    ///
    /// This method applies to properties with a [rdfs:range](rdf-ontology.html`rdfs.properties.range`)
    /// that points to a non-literal class (i.e. a subclass of
    /// [rdfs:Resource](rdf-ontology.html`rdfs.properties.Resource`)).
    ///
    /// This method produces similar RDF to `Resource.addUri`,
    /// although in this function the URI will depend on the identifier
    /// set on `resource`.
    extern fn tracker_resource_add_relation(p_self: *Resource, p_property_uri: [*:0]const u8, p_resource: *tsparql.Resource) void;
    pub const addRelation = tracker_resource_add_relation;

    /// Adds a string property. Previous values for the same property are kept.
    ///
    /// This method is meant for RDF properties allowing multiple values, see
    /// [nrl:maxCardinality](nrl-ontology.html`nrl.properties.maxCardinality`).
    ///
    /// This method corresponds to [xsd:string](xsd-ontology.html`xsd.properties.string`).
    extern fn tracker_resource_add_string(p_self: *Resource, p_property_uri: [*:0]const u8, p_value: [*:0]const u8) void;
    pub const addString = tracker_resource_add_string;

    /// Adds a resource property as a `TrackerResource`. Previous values for the same property are kept.
    /// Takes ownership on the given `resource`.
    ///
    /// This method is meant to RDF properties allowing multiple values, see
    /// [nrl:maxCardinality](nrl-ontology.html`nrl.properties.maxCardinality`).
    ///
    /// This method applies to properties with a [rdfs:range](rdf-ontology.html`rdfs.properties.range`)
    /// that points to a non-literal class (i.e. a subclass of
    /// [rdfs:Resource](rdf-ontology.html`rdfs.properties.Resource`)).
    ///
    /// This function produces similar RDF to `Resource.addUri`,
    /// although in this function the URI will depend on the identifier
    /// set on `resource`. This function takes ownership of `resource`.
    extern fn tracker_resource_add_take_relation(p_self: *Resource, p_property_uri: [*:0]const u8, p_resource: *tsparql.Resource) void;
    pub const addTakeRelation = tracker_resource_add_take_relation;

    /// Adds a resource property as an URI string. Previous values for the same property are kept.
    ///
    /// This method applies to properties with a [rdfs:range](rdf-ontology.html`rdfs.properties.range`)
    /// that points to a non-literal class (i.e. a subclass of
    /// [rdfs:Resource](rdf-ontology.html`rdfs.properties.Resource`)).
    ///
    /// This method is meant for RDF properties allowing multiple values, see
    /// [nrl:maxCardinality](nrl-ontology.html`nrl.properties.maxCardinality`).
    ///
    /// This function produces similar RDF to `Resource.addRelation`, although
    /// it requires that the URI is previously known.
    extern fn tracker_resource_add_uri(p_self: *Resource, p_property_uri: [*:0]const u8, p_value: [*:0]const u8) void;
    pub const addUri = tracker_resource_add_uri;

    /// Returns the first boolean object previously assigned to a property.
    extern fn tracker_resource_get_first_boolean(p_self: *Resource, p_property_uri: [*:0]const u8) c_int;
    pub const getFirstBoolean = tracker_resource_get_first_boolean;

    /// Returns the first `glib.DateTime` previously assigned to a property.
    extern fn tracker_resource_get_first_datetime(p_self: *Resource, p_property_uri: [*:0]const u8) ?*glib.DateTime;
    pub const getFirstDatetime = tracker_resource_get_first_datetime;

    /// Returns the first double object previously assigned to a property.
    extern fn tracker_resource_get_first_double(p_self: *Resource, p_property_uri: [*:0]const u8) f64;
    pub const getFirstDouble = tracker_resource_get_first_double;

    /// Returns the first integer object previously assigned to a property.
    extern fn tracker_resource_get_first_int(p_self: *Resource, p_property_uri: [*:0]const u8) c_int;
    pub const getFirstInt = tracker_resource_get_first_int;

    /// Returns the first integer object previously assigned to a property.
    extern fn tracker_resource_get_first_int64(p_self: *Resource, p_property_uri: [*:0]const u8) i64;
    pub const getFirstInt64 = tracker_resource_get_first_int64;

    /// Returns the first resource object previously assigned to a property.
    extern fn tracker_resource_get_first_relation(p_self: *Resource, p_property_uri: [*:0]const u8) ?*tsparql.Resource;
    pub const getFirstRelation = tracker_resource_get_first_relation;

    /// Returns the first string object previously assigned to a property.
    extern fn tracker_resource_get_first_string(p_self: *Resource, p_property_uri: [*:0]const u8) ?[*:0]const u8;
    pub const getFirstString = tracker_resource_get_first_string;

    /// Returns the first resource object previously assigned to a property.
    extern fn tracker_resource_get_first_uri(p_self: *Resource, p_property_uri: [*:0]const u8) ?[*:0]const u8;
    pub const getFirstUri = tracker_resource_get_first_uri;

    /// Returns the identifier of a resource.
    ///
    /// If the identifier was set to NULL, the identifier returned will be a locally
    /// unique SPARQL blank node identifier, such as `_:123`.
    extern fn tracker_resource_get_identifier(p_self: *Resource) ?[*:0]const u8;
    pub const getIdentifier = tracker_resource_get_identifier;

    /// Gets the list of properties defined in `resource`
    extern fn tracker_resource_get_properties(p_resource: *Resource) *glib.List;
    pub const getProperties = tracker_resource_get_properties;

    /// Returns whether the prior values for this property would be deleted
    /// in the SPARQL issued by `resource`.
    extern fn tracker_resource_get_property_overwrite(p_resource: *Resource, p_property_uri: [*:0]const u8) c_int;
    pub const getPropertyOverwrite = tracker_resource_get_property_overwrite;

    /// Returns the list of all known values of the given property.
    extern fn tracker_resource_get_values(p_self: *Resource, p_property_uri: [*:0]const u8) ?*glib.List;
    pub const getValues = tracker_resource_get_values;

    /// A helper function that compares a `TrackerResource` by its identifier
    /// string.
    extern fn tracker_resource_identifier_compare_func(p_resource: *Resource, p_identifier: [*:0]const u8) c_int;
    pub const identifierCompareFunc = tracker_resource_identifier_compare_func;

    /// Serialize all the information in `resource` as a JSON-LD document.
    ///
    /// See <http://www.jsonld.org/> for more information on the JSON-LD
    /// serialization format.
    ///
    /// The `namespaces` object is used to expand any compact URI values. In most
    /// cases you should pass the one returned by `SparqlConnection.getNamespaceManager`
    /// from the connection that is the intended recipient of this data.
    extern fn tracker_resource_print_jsonld(p_self: *Resource, p_namespaces: ?*tsparql.NamespaceManager) [*:0]u8;
    pub const printJsonld = tracker_resource_print_jsonld;

    /// Serialize all the information in `resource` into the selected RDF format.
    ///
    /// The `namespaces` object is used to expand any compact URI values. In most
    /// cases you should pass the one returned by `SparqlConnection.getNamespaceManager`
    /// from the connection that is the intended recipient of this data.
    extern fn tracker_resource_print_rdf(p_self: *Resource, p_namespaces: *tsparql.NamespaceManager, p_format: tsparql.RdfFormat, p_graph: ?[*:0]const u8) [*:0]u8;
    pub const printRdf = tracker_resource_print_rdf;

    /// Generates a SPARQL command to update a database with the information
    /// stored in `resource`.
    ///
    /// The `namespaces` object is used to expand any compact URI values. In most
    /// cases you should pass the one returned by `SparqlConnection.getNamespaceManager`
    /// from the connection that is the intended recipient of this data.
    extern fn tracker_resource_print_sparql_update(p_self: *Resource, p_namespaces: ?*tsparql.NamespaceManager, p_graph_id: ?[*:0]const u8) [*:0]u8;
    pub const printSparqlUpdate = tracker_resource_print_sparql_update;

    /// Serialize all the information in `resource` as a Turtle document.
    ///
    /// The generated Turtle should correspond to this standard:
    /// <https://www.w3.org/TR/2014/REC-turtle-20140225/>
    ///
    /// The `namespaces` object is used to expand any compact URI values. In most
    /// cases you should pass the one returned by `SparqlConnection.getNamespaceManager`
    /// from the connection that is the intended recipient of this data.
    extern fn tracker_resource_print_turtle(p_self: *Resource, p_namespaces: ?*tsparql.NamespaceManager) [*:0]u8;
    pub const printTurtle = tracker_resource_print_turtle;

    /// Serializes a `TrackerResource` to a `glib.Variant` in a lossless way.
    /// All child resources are subsequently serialized. It is implied
    /// that both ends use a common `NamespaceManager`.
    extern fn tracker_resource_serialize(p_resource: *Resource) ?*glib.Variant;
    pub const serialize = tracker_resource_serialize;

    /// Sets a boolean property. Replaces any previous value.
    ///
    /// This method corresponds to [xsd:boolean](xsd-ontology.html`xsd.properties.boolean`).
    extern fn tracker_resource_set_boolean(p_self: *Resource, p_property_uri: [*:0]const u8, p_value: c_int) void;
    pub const setBoolean = tracker_resource_set_boolean;

    /// Sets a date property as a `glib.DateTime`. Replaces any previous value.
    ///
    /// This method corresponds to [xsd:date](xsd-ontology.html`xsd.properties.date`) and
    /// [xsd:dateTime](xsd-ontology.html`xsd.properties.dateTime`).
    extern fn tracker_resource_set_datetime(p_self: *Resource, p_property_uri: [*:0]const u8, p_value: *glib.DateTime) void;
    pub const setDatetime = tracker_resource_set_datetime;

    /// Sets a numeric property with double precision. Replaces any previous value.
    ///
    /// This method corresponds to [xsd:double](xsd-ontology.html`xsd.properties.double`).
    extern fn tracker_resource_set_double(p_self: *Resource, p_property_uri: [*:0]const u8, p_value: f64) void;
    pub const setDouble = tracker_resource_set_double;

    /// Replace any previously existing value for `property_uri` with `value`.
    ///
    /// When serialising to SPARQL, any properties that were set with this function
    /// will get a corresponding DELETE statement to remove any existing values in
    /// the database.
    ///
    /// You can pass any kind of `gobject.Value` for `value`, but serialization functions will
    /// normally only be able to serialize URIs/relationships and fundamental value
    /// types (string, int, etc.).
    extern fn tracker_resource_set_gvalue(p_self: *Resource, p_property_uri: [*:0]const u8, p_value: *const gobject.Value) void;
    pub const setGvalue = tracker_resource_set_gvalue;

    /// Changes the identifier of a `TrackerResource`. The identifier should be a
    /// URI or compact URI, but this is not necessarily enforced. Invalid
    /// identifiers may cause errors when serializing the resource or trying to
    /// insert the results in a database.
    ///
    /// If the identifier is set to `NULL`, a SPARQL blank node identifier such as
    /// `_:123` is assigned to the resource.
    extern fn tracker_resource_set_identifier(p_self: *Resource, p_identifier: ?[*:0]const u8) void;
    pub const setIdentifier = tracker_resource_set_identifier;

    /// Sets a numeric property with integer precision. Replaces any previous value.
    ///
    /// This method corresponds to [xsd:integer](xsd-ontology.html`xsd.properties.integer`).
    extern fn tracker_resource_set_int(p_self: *Resource, p_property_uri: [*:0]const u8, p_value: c_int) void;
    pub const setInt = tracker_resource_set_int;

    /// Sets a numeric property with 64-bit integer precision. Replaces any previous value.
    ///
    /// This method corresponds to [xsd:integer](xsd-ontology.html`xsd.properties.integer`).
    extern fn tracker_resource_set_int64(p_self: *Resource, p_property_uri: [*:0]const u8, p_value: i64) void;
    pub const setInt64 = tracker_resource_set_int64;

    /// Sets a resource property as a `TrackerResource`. Replaces any previous value.
    ///
    /// This method applies to properties with a [rdfs:range](rdf-ontology.html`rdfs.properties.range`)
    /// that points to a non-literal class (i.e. a subclass of
    /// [rdfs:Resource](rdf-ontology.html`rdfs.properties.Resource`)).
    ///
    /// This function produces similar RDF to `Resource.setUri`,
    /// although in this function the URI will depend on the identifier
    /// set on `resource`.
    extern fn tracker_resource_set_relation(p_self: *Resource, p_property_uri: [*:0]const u8, p_resource: *tsparql.Resource) void;
    pub const setRelation = tracker_resource_set_relation;

    /// Sets a string property. Replaces any previous value.
    ///
    /// This method corresponds to [xsd:string](xsd-ontology.html`xsd.properties.string`).
    extern fn tracker_resource_set_string(p_self: *Resource, p_property_uri: [*:0]const u8, p_value: [*:0]const u8) void;
    pub const setString = tracker_resource_set_string;

    /// Sets a resource property as a `TrackerResource`. Replaces any previous value.
    /// Takes ownership on the given `resource`.
    ///
    /// This method applies to properties with a [rdfs:range](rdf-ontology.html`rdfs.properties.range`)
    /// that points to a non-literal class (i.e. a subclass of
    /// [rdfs:Resource](rdf-ontology.html`rdfs.properties.Resource`)).
    ///
    /// This function produces similar RDF to `Resource.setUri`,
    /// although in this function the URI will depend on the identifier
    /// set on `resource`.
    extern fn tracker_resource_set_take_relation(p_self: *Resource, p_property_uri: [*:0]const u8, p_resource: *tsparql.Resource) void;
    pub const setTakeRelation = tracker_resource_set_take_relation;

    /// Sets a resource property as an URI string. Replaces any previous value.
    ///
    /// This method applies to properties with a [rdfs:range](rdf-ontology.html`rdfs.properties.range`)
    /// that points to a non-literal class (i.e. a subclass of
    /// [rdfs:Resource](rdf-ontology.html`rdfs.properties.Resource`)).
    ///
    /// This function produces similar RDF to `Resource.setRelation`, although
    /// it requires that the URI is previously known.
    extern fn tracker_resource_set_uri(p_self: *Resource, p_property_uri: [*:0]const u8, p_value: [*:0]const u8) void;
    pub const setUri = tracker_resource_set_uri;

    extern fn tracker_resource_get_type() usize;
    pub const getGObjectType = tracker_resource_get_type;

    extern fn g_object_ref(p_self: *tsparql.Resource) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *tsparql.Resource) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Resource, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `TrackerSparqlConnection` holds a connection to a RDF triple store.
///
/// This triple store may be of three types:
///
///  - Local to the process, created through `SparqlConnection.new`.
///  - A HTTP SPARQL endpoint over the network, created through
///    `SparqlConnection.remoteNew`
///  - A DBus SPARQL endpoint owned by another process in the same machine, created
///    through `SparqlConnection.busNew`
///
/// When creating a local triple store, it is required to give details about its
/// structure. This is done by passing a location to an ontology, see more
/// on how are [ontologies defined](ontologies.html). A local database may be
/// stored in a filesystem location, or it may reside in memory.
///
/// A `TrackerSparqlConnection` is private to the calling process, it can be
/// exposed to other hosts/processes via a `Endpoint`, see
/// `EndpointDBus.new` and `EndpointHttp.new`.
///
/// When issuing SPARQL queries and updates, it is recommended that these are
/// created through `SparqlStatement` to avoid the SPARQL
/// injection class of bugs, see `SparqlConnection.queryStatement`
/// and `SparqlConnection.updateStatement`. For SPARQL updates
/// it is also possible to use a "builder" approach to generate RDF data, see
/// `Resource`. It is also possible to create `SparqlStatement`
/// objects for SPARQL queries and updates from SPARQL strings embedded in a
/// `gio.Resource`, see `SparqlConnection.loadStatementFromGresource`.
///
/// To get the best performance, it is recommended that SPARQL updates are clustered
/// through `Batch`.
///
/// `TrackerSparqlConnection` also offers a number of methods for the simple cases,
/// `SparqlConnection.query` may be used when there is a SPARQL
/// query string directly available, and the `SparqlConnection.update`
/// family of functions may be used for one-off updates. All functions have asynchronous
/// variants.
///
/// When a SPARQL query is executed, a `SparqlCursor` will be obtained
/// to iterate over the query results.
///
/// Depending on the ontology definition, `TrackerSparqlConnection` may emit
/// notifications whenever resources of certain types get insert, modified or
/// deleted from the triple store (see [nrl:notify](nrl-ontology.html`nrl.properties.notify`).
/// These notifications can be handled via a `Notifier` obtained with
/// `SparqlConnection.createNotifier`.
///
/// After done with a connection, it is recommended to call `SparqlConnection.close`
/// or `SparqlConnection.closeAsync` explicitly to cleanly close the
/// connection and prevent consistency checks on future runs. The triple store
/// connection will be implicitly closed when the `TrackerSparqlConnection` object
/// is disposed.
///
/// A `TrackerSparqlConnection` may be used from multiple threads, asynchronous
/// updates are executed sequentially on arrival order, asynchronous
/// queries are dispatched in a thread pool.
///
/// If you ever have the need to procedurally compose SPARQL query strings, consider
/// the use of `sparqlEscapeString` for literal strings and
/// the `sparqlEscapeUri` family of functions for URIs.
pub const SparqlConnection = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = tsparql.SparqlConnectionClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Connects asynchronously to a database owned by another process on the
    /// local machine via DBus.
    extern fn tracker_sparql_connection_bus_new_async(p_service_name: [*:0]const u8, p_object_path: ?[*:0]const u8, p_dbus_connection: ?*gio.DBusConnection, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const busNewAsync = tracker_sparql_connection_bus_new_async;

    /// Creates or opens a process-local database asynchronously.
    ///
    /// See `SparqlConnection.new` for more information.
    extern fn tracker_sparql_connection_new_async(p_flags: tsparql.SparqlConnectionFlags, p_store: ?*gio.File, p_ontology: ?*gio.File, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const newAsync = tracker_sparql_connection_new_async;

    /// `service_name` (nullable): The name of the D-Bus service to connect to, or `NULL` if not using a message bus.
    /// Connects to a database owned by another process on the
    /// local machine via DBus.
    ///
    /// When using a message bus (session/system), the `service_name` argument will
    /// be used to describe the remote endpoint, either by unique or well-known D-Bus
    /// names. If not using a message bus (e.g. peer-to-peer D-Bus connections) the
    /// `service_name` may be `NULL`.
    ///
    /// The D-Bus object path of the remote endpoint will be given through
    /// `object_path`, `NULL` may be used to use the default
    /// `/org/freedesktop/Tracker3/Endpoint` path.
    ///
    /// The D-Bus connection used to set up the connection may be given through
    /// the `dbus_connection` argument. Using `NULL` will resort to the default session
    /// bus.
    extern fn tracker_sparql_connection_bus_new(p_service_name: [*:0]const u8, p_object_path: ?[*:0]const u8, p_dbus_connection: ?*gio.DBusConnection, p_error: ?*?*glib.Error) ?*tsparql.SparqlConnection;
    pub const busNew = tracker_sparql_connection_bus_new;

    /// Finishes the operation started with `SparqlConnection.busNewAsync`.
    extern fn tracker_sparql_connection_bus_new_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*tsparql.SparqlConnection;
    pub const busNewFinish = tracker_sparql_connection_bus_new_finish;

    /// Creates or opens a process-local database.
    ///
    /// This method should only be used for databases owned by the current process.
    /// To connect to databases managed by other processes, use
    /// `SparqlConnection.busNew`.
    ///
    /// If `store` is `NULL`, the database will be created in memory.
    ///
    /// If defined, the `ontology` argument must point to a location containing
    /// suitable `.ontology` files in Turtle format. These define the structure of
    /// the triple store. You can learn more about [ontologies](ontologies.html),
    /// or you can use the stock Nepomuk ontologies by calling
    /// `sparqlGetOntologyNepomuk`.
    ///
    /// If opening an existing database, it is possible to pass `NULL` as the
    /// `ontology` location, the ontology will be introspected from the database.
    /// Passing a `NULL` `ontology` will raise an error if the database does not exist.
    ///
    /// If a database is opened without the `READONLY` `SparqlConnectionFlags`
    /// flag enabled, and the given `ontology` holds differences with the current
    /// data layout, migration to the new structure will be attempted. This operation
    /// may raise an error. In particular, not all migrations are possible without
    /// causing data loss and Tracker will refuse to delete data during a migration.
    /// The database is always left in a consistent state, either prior or posterior
    /// to migration.
    ///
    /// Operations on a `SparqlConnection` resulting on a
    /// `CORRUPT` `SparqlError` will have the event recorded
    /// persistently through a `.meta.corrupted` file alongside the database files.
    /// If the database is opened without the `READONLY` `SparqlConnectionFlags`
    /// flag enabled and the file is found, this constructor will attempt to repair the
    /// database. In that situation, this constructor will either return a valid
    /// `SparqlConnection` if the database was repaired successfully, or
    /// raise a `CORRUPT` `SparqlError` error if the database remains
    /// corrupted.
    ///
    /// It is considered a developer error to ship ontologies that contain format
    /// errors, or that fail at migrations.
    ///
    /// It is encouraged to use `resource:///` URI locations for `ontology` wherever
    /// possible, so the triple store structure is tied to the executable binary,
    /// and in order to minimize disk seeks during `TrackerSparqlConnection`
    /// initialization.
    extern fn tracker_sparql_connection_new(p_flags: tsparql.SparqlConnectionFlags, p_store: ?*gio.File, p_ontology: ?*gio.File, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*tsparql.SparqlConnection;
    pub const new = tracker_sparql_connection_new;

    /// Finishes the operation started with `SparqlConnection.newAsync`.
    extern fn tracker_sparql_connection_new_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*tsparql.SparqlConnection;
    pub const newFinish = tracker_sparql_connection_new_finish;

    /// Creates a connection to a remote HTTP SPARQL endpoint.
    ///
    /// The connection is made using the libsoup HTTP library. The connection will
    /// normally use the `https://` or `http://` protocols.
    extern fn tracker_sparql_connection_remote_new(p_uri_base: [*:0]const u8) *tsparql.SparqlConnection;
    pub const remoteNew = tracker_sparql_connection_remote_new;

    /// Closes a SPARQL connection.
    ///
    /// No other API calls than `gobject.Object.unref` should happen after this call.
    ///
    /// This call is blocking. All pending updates will be flushed, and the
    /// store databases will be closed orderly. All ongoing SELECT queries
    /// will be cancelled. Notifiers will no longer emit events.
    extern fn tracker_sparql_connection_close(p_connection: *SparqlConnection) void;
    pub const close = tracker_sparql_connection_close;

    /// Closes a SPARQL connection asynchronously.
    ///
    /// No other API calls than `gobject.Object.unref` should happen after this call.
    extern fn tracker_sparql_connection_close_async(p_connection: *SparqlConnection, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const closeAsync = tracker_sparql_connection_close_async;

    /// Finishes the operation started with `SparqlConnection.closeAsync`.
    extern fn tracker_sparql_connection_close_finish(p_connection: *SparqlConnection, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const closeFinish = tracker_sparql_connection_close_finish;

    /// Creates a new `Batch` to store and execute SPARQL updates.
    ///
    /// If the connection is readonly or cannot issue SPARQL updates, `NULL` will be returned.
    extern fn tracker_sparql_connection_create_batch(p_connection: *SparqlConnection) *tsparql.Batch;
    pub const createBatch = tracker_sparql_connection_create_batch;

    /// Creates a new `Notifier` to receive notifications about changes in `connection`.
    ///
    /// See `Notifier` documentation for information about how to use this
    /// object.
    ///
    /// Connections to HTTP endpoints will return `NULL`.
    extern fn tracker_sparql_connection_create_notifier(p_connection: *SparqlConnection) ?*tsparql.Notifier;
    pub const createNotifier = tracker_sparql_connection_create_notifier;

    /// Loads the RDF data contained in `stream` into the given `connection`.
    ///
    /// This is an asynchronous operation, `callback` will be invoked when the
    /// data has been fully inserted to `connection`.
    ///
    /// The RDF data will be inserted in the given `default_graph` if one is provided,
    /// or the anonymous graph if `default_graph` is `NULL`. Any RDF data that has a
    /// graph specified (e.g. using the `GRAPH` clause in the Trig format) will
    /// be inserted in the specified graph instead of `default_graph`.
    ///
    /// The `flags` argument is reserved for future expansions, currently
    /// `TRACKER_DESERIALIZE_FLAGS_NONE` must be passed.
    extern fn tracker_sparql_connection_deserialize_async(p_connection: *SparqlConnection, p_flags: tsparql.DeserializeFlags, p_format: tsparql.RdfFormat, p_default_graph: [*:0]const u8, p_stream: *gio.InputStream, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const deserializeAsync = tracker_sparql_connection_deserialize_async;

    /// Finishes the operation started with `SparqlConnection.deserializeAsync`.
    extern fn tracker_sparql_connection_deserialize_finish(p_connection: *SparqlConnection, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const deserializeFinish = tracker_sparql_connection_deserialize_finish;

    /// Returns a `NamespaceManager` that contains all
    /// prefixes in the ontology of `connection`.
    extern fn tracker_sparql_connection_get_namespace_manager(p_connection: *SparqlConnection) *tsparql.NamespaceManager;
    pub const getNamespaceManager = tracker_sparql_connection_get_namespace_manager;

    /// Prepares a `SparqlStatement` for the SPARQL contained as a `gio.Resource`
    /// file at `resource_path`.
    ///
    /// SPARQL Query files typically have the .rq extension. This will use
    /// `SparqlConnection.queryStatement` or `SparqlConnection.updateStatement`
    /// underneath to indistinctly return SPARQL query or update statements.
    extern fn tracker_sparql_connection_load_statement_from_gresource(p_connection: *SparqlConnection, p_resource_path: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*tsparql.SparqlStatement;
    pub const loadStatementFromGresource = tracker_sparql_connection_load_statement_from_gresource;

    /// Maps a `TrackerSparqlConnection` onto another through a `private:`handle_name`` URI.
    ///
    /// This can be accessed via the SERVICE SPARQL syntax in
    /// queries from `connection`. E.g.:
    ///
    /// ```c
    /// tracker_sparql_connection_map_connection (connection,
    ///                                           "other-connection",
    ///                                           other_connection);
    /// ```
    ///
    /// ```sparql
    /// SELECT ?u {
    ///   SERVICE <private:other-connection> {
    ///     ?u a rdfs:Resource
    ///   }
    /// }
    /// ```
    ///
    /// This is useful to interrelate data from multiple
    /// `TrackerSparqlConnection` instances maintained by the same process,
    /// without creating a public endpoint for `service_connection`.
    ///
    /// `connection` may only be a `TrackerSparqlConnection` created via
    /// `SparqlConnection.new` and `SparqlConnection.newAsync`.
    extern fn tracker_sparql_connection_map_connection(p_connection: *SparqlConnection, p_handle_name: [*:0]const u8, p_service_connection: *tsparql.SparqlConnection) void;
    pub const mapConnection = tracker_sparql_connection_map_connection;

    /// Executes a SPARQL query on `connection`.
    ///
    /// This method is synchronous and will block until the query
    /// is executed. See `SparqlConnection.queryAsync`
    /// for an asynchronous variant.
    ///
    /// If the query is partially built from user input or other
    /// untrusted sources, special care is required about possible
    /// SPARQL injection. In order to avoid it entirely, it is recommended
    /// to use `SparqlStatement`. The function
    /// `sparqlEscapeString` exists as a last resort,
    /// but its use is not recommended.
    extern fn tracker_sparql_connection_query(p_connection: *SparqlConnection, p_sparql: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*tsparql.SparqlCursor;
    pub const query = tracker_sparql_connection_query;

    /// Executes asynchronously a SPARQL query on `connection`
    ///
    /// If the query is partially built from user input or other
    /// untrusted sources, special care is required about possible
    /// SPARQL injection. In order to avoid it entirely, it is recommended
    /// to use `SparqlStatement`. The function
    /// `sparqlEscapeString` exists as a last resort,
    /// but its use is not recommended.
    extern fn tracker_sparql_connection_query_async(p_connection: *SparqlConnection, p_sparql: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const queryAsync = tracker_sparql_connection_query_async;

    /// Finishes the operation started with `SparqlConnection.queryAsync`.
    extern fn tracker_sparql_connection_query_finish(p_connection: *SparqlConnection, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*tsparql.SparqlCursor;
    pub const queryFinish = tracker_sparql_connection_query_finish;

    /// Prepares the given `SELECT`/`ASK`/`DESCRIBE`/`CONSTRUCT` SPARQL query as a
    /// `SparqlStatement`.
    ///
    /// This prepared statement can be executed through `SparqlStatement.execute`
    /// or `SparqlStatement.serializeAsync` families of functions.
    extern fn tracker_sparql_connection_query_statement(p_connection: *SparqlConnection, p_sparql: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*tsparql.SparqlStatement;
    pub const queryStatement = tracker_sparql_connection_query_statement;

    /// Serializes a `DESCRIBE` or `CONSTRUCT` query into the specified RDF format.
    ///
    /// This is an asynchronous operation, `callback` will be invoked when
    /// the data is available for reading.
    ///
    /// The SPARQL endpoint may not support the specified format, in that case
    /// an error will be raised.
    ///
    /// The `flags` argument is reserved for future expansions, currently
    /// `TRACKER_SERIALIZE_FLAGS_NONE` must be passed.
    extern fn tracker_sparql_connection_serialize_async(p_connection: *SparqlConnection, p_flags: tsparql.SerializeFlags, p_format: tsparql.RdfFormat, p_query: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const serializeAsync = tracker_sparql_connection_serialize_async;

    /// Finishes the operation started with `SparqlConnection.serializeAsync`.
    extern fn tracker_sparql_connection_serialize_finish(p_connection: *SparqlConnection, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.InputStream;
    pub const serializeFinish = tracker_sparql_connection_serialize_finish;

    /// Executes a SPARQL update on `connection`.
    ///
    /// This method is synchronous and will block until the update
    /// is finished. See `SparqlConnection.updateAsync`
    /// for an asynchronous variant.
    ///
    /// It is recommented to consider the usage of `Batch`
    /// to cluster database updates. Frequent isolated SPARQL updates
    /// through this method will have a degraded performance in comparison.
    ///
    /// If the query is partially built from user input or other
    /// untrusted sources, special care is required about possible
    /// SPARQL injection. In order to avoid it entirely, it is recommended
    /// to use `SparqlStatement`, or to build the SPARQL
    /// input through `Resource`. The function
    /// `sparqlEscapeString` exists as a last resort,
    /// but its use is not recommended.
    extern fn tracker_sparql_connection_update(p_connection: *SparqlConnection, p_sparql: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) void;
    pub const update = tracker_sparql_connection_update;

    /// Executes asynchronously an array of SPARQL updates. All updates in the
    /// array are handled within a single transaction.
    ///
    /// If the query is partially built from user input or other
    /// untrusted sources, special care is required about possible
    /// SPARQL injection. In order to avoid it entirely, it is recommended
    /// to use `SparqlStatement`, or to build the SPARQL
    /// input through `Resource`. The function
    /// `sparqlEscapeString` exists as a last resort,
    /// but its use is not recommended.
    extern fn tracker_sparql_connection_update_array_async(p_connection: *SparqlConnection, p_sparql: *[*:0]u8, p_sparql_length: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const updateArrayAsync = tracker_sparql_connection_update_array_async;

    /// Finishes the operation started with `SparqlConnection.updateArrayAsync`.
    extern fn tracker_sparql_connection_update_array_finish(p_connection: *SparqlConnection, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const updateArrayFinish = tracker_sparql_connection_update_array_finish;

    /// Executes asynchronously a SPARQL update.
    ///
    /// It is recommented to consider the usage of `Batch`
    /// to cluster database updates. Frequent isolated SPARQL updates
    /// through this method will have a degraded performance in comparison.
    ///
    /// If the query is partially built from user input or other
    /// untrusted sources, special care is required about possible
    /// SPARQL injection. In order to avoid it entirely, it is recommended
    /// to use `SparqlStatement`, or to build the SPARQL
    /// input through `Resource`. The function
    /// `sparqlEscapeString` exists as a last resort,
    /// but its use is not recommended.
    extern fn tracker_sparql_connection_update_async(p_connection: *SparqlConnection, p_sparql: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const updateAsync = tracker_sparql_connection_update_async;

    /// Executes a SPARQL update and returns the names of the generated blank nodes.
    ///
    /// This method is synchronous and will block until the update
    /// is finished. See `SparqlConnection.updateBlankAsync`
    /// for an asynchronous variant.
    ///
    /// The `sparql` query should be built with `Resource`, or
    /// its parts correctly escaped using `sparqlEscapeString`,
    /// otherwise SPARQL injection is possible.
    ///
    /// The format string of the `GVariant` is `aaa{ss}` (an array of an array
    /// of dictionaries). The first array represents each INSERT that may exist in
    /// the SPARQL string. The second array represents each new node for a given
    /// WHERE clause. The last array holds a string pair with the blank node name
    /// (e.g. `foo` for the blank node `_:foo`) and the URN that was generated for
    /// it. For most updates the first two outer arrays will only contain one item.
    extern fn tracker_sparql_connection_update_blank(p_connection: *SparqlConnection, p_sparql: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*glib.Variant;
    pub const updateBlank = tracker_sparql_connection_update_blank;

    /// Executes asynchronously a SPARQL update and returns the names of the generated blank nodes.
    ///
    /// See the `SparqlConnection.updateBlank` documentation to
    /// learn the differences with `SparqlConnection.update`.
    extern fn tracker_sparql_connection_update_blank_async(p_connection: *SparqlConnection, p_sparql: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const updateBlankAsync = tracker_sparql_connection_update_blank_async;

    /// Finishes the operation started with `SparqlConnection.updateBlankAsync`.
    ///
    /// This method returns the URNs of the generated nodes, if any. See the
    /// `SparqlConnection.updateBlank` documentation for the interpretation
    /// of the returned `glib.Variant`.
    extern fn tracker_sparql_connection_update_blank_finish(p_connection: *SparqlConnection, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.Variant;
    pub const updateBlankFinish = tracker_sparql_connection_update_blank_finish;

    /// Finishes the operation started with `SparqlConnection.updateAsync`.
    extern fn tracker_sparql_connection_update_finish(p_connection: *SparqlConnection, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) void;
    pub const updateFinish = tracker_sparql_connection_update_finish;

    /// Inserts a resource as described by `resource` on the given `graph`.
    ///
    /// This method is synchronous and will block until the update
    /// is finished. See `SparqlConnection.updateResourceAsync`
    /// for an asynchronous variant.
    ///
    /// It is recommented to consider the usage of `Batch`
    /// to cluster database updates. Frequent isolated SPARQL updates
    /// through this method will have a degraded performance in comparison.
    extern fn tracker_sparql_connection_update_resource(p_connection: *SparqlConnection, p_graph: ?[*:0]const u8, p_resource: *tsparql.Resource, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const updateResource = tracker_sparql_connection_update_resource;

    /// Inserts asynchronously a resource as described by `resource` on the given `graph`.
    ///
    /// It is recommented to consider the usage of `Batch`
    /// to cluster database updates. Frequent isolated SPARQL updates
    /// through this method will have a degraded performance in comparison.
    extern fn tracker_sparql_connection_update_resource_async(p_connection: *SparqlConnection, p_graph: ?[*:0]const u8, p_resource: *tsparql.Resource, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const updateResourceAsync = tracker_sparql_connection_update_resource_async;

    /// Finishes the operation started with `SparqlConnection.updateResourceAsync`.
    extern fn tracker_sparql_connection_update_resource_finish(p_connection: *SparqlConnection, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const updateResourceFinish = tracker_sparql_connection_update_resource_finish;

    /// Prepares the given `INSERT`/`DELETE` SPARQL as a `SparqlStatement`.
    ///
    /// This prepared statement can be executed through
    /// the `SparqlStatement.update` family of functions.
    extern fn tracker_sparql_connection_update_statement(p_connection: *SparqlConnection, p_sparql: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*tsparql.SparqlStatement;
    pub const updateStatement = tracker_sparql_connection_update_statement;

    extern fn tracker_sparql_connection_get_type() usize;
    pub const getGObjectType = tracker_sparql_connection_get_type;

    extern fn g_object_ref(p_self: *tsparql.SparqlConnection) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *tsparql.SparqlConnection) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SparqlConnection, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `TrackerSparqlCursor` provides the methods to iterate the results of a SPARQL query.
///
/// Cursors are obtained through e.g. `SparqlStatement.execute`
/// or `SparqlConnection.query` after the SPARQL query has been
/// executed.
///
/// When created, a cursor does not point to any element, `SparqlCursor.next`
/// is necessary to iterate one by one to the first (and following) results.
/// When the cursor iterated across all rows in the result set, `SparqlCursor.next`
/// will return `FALSE` with no error set.
///
/// On each row, it is possible to extract the result values through the
/// `SparqlCursor.getInteger`, `SparqlCursor.getString`, etc... family
/// of methods. The column index of those functions starts at 0. The number of columns is
/// dependent on the SPARQL query issued, but may be checked at runtime through the
/// `SparqlCursor.getNColumns` method.
///
/// After a cursor is iterated, it is recommended to call `SparqlCursor.close`
/// explicitly to free up resources for other users of the same `SparqlConnection`,
/// this is especially important in garbage collected languages. These resources
/// will be also implicitly freed on cursor object finalization.
///
/// It is possible to use a given `TrackerSparqlCursor` in other threads than
/// the one it was created from. It must be however used from just one thread
/// at any given time.
pub const SparqlCursor = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = tsparql.SparqlCursorClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The `SparqlConnection` used to retrieve the results.
        pub const connection = struct {
            pub const name = "connection";

            pub const Type = ?*tsparql.SparqlConnection;
        };

        /// Number of columns available in the result set.
        pub const n_columns = struct {
            pub const name = "n-columns";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Closes the cursor. The object can only be freed after this call.
    extern fn tracker_sparql_cursor_close(p_cursor: *SparqlCursor) void;
    pub const close = tracker_sparql_cursor_close;

    /// Retrieve a boolean for the current row in `column`.
    ///
    /// If the row/column do not have a boolean value, the result is
    /// undefined, see `SparqlCursor.getValueType`.
    extern fn tracker_sparql_cursor_get_boolean(p_cursor: *SparqlCursor, p_column: c_int) c_int;
    pub const getBoolean = tracker_sparql_cursor_get_boolean;

    /// Returns the `SparqlConnection` associated with this
    /// `TrackerSparqlCursor`.
    extern fn tracker_sparql_cursor_get_connection(p_cursor: *SparqlCursor) *tsparql.SparqlConnection;
    pub const getConnection = tracker_sparql_cursor_get_connection;

    /// Retrieves a `glib.DateTime` pointer for the current row in `column`.
    extern fn tracker_sparql_cursor_get_datetime(p_cursor: *SparqlCursor, p_column: c_int) ?*glib.DateTime;
    pub const getDatetime = tracker_sparql_cursor_get_datetime;

    /// Retrieve a double for the current row in `column`.
    ///
    /// If the row/column do not have a integer or double value, the result is
    /// undefined, see `SparqlCursor.getValueType`.
    extern fn tracker_sparql_cursor_get_double(p_cursor: *SparqlCursor, p_column: c_int) f64;
    pub const getDouble = tracker_sparql_cursor_get_double;

    /// Retrieve an integer for the current row in `column`.
    ///
    /// If the row/column do not have an integer value, the result is
    /// undefined, see `SparqlCursor.getValueType`.
    extern fn tracker_sparql_cursor_get_integer(p_cursor: *SparqlCursor, p_column: c_int) i64;
    pub const getInteger = tracker_sparql_cursor_get_integer;

    /// Retrieves a string representation of the data in the current
    /// row in `column`. If the string has language information (i.e. it is
    /// a `rdf:langString`](rdf-ontology.html`rdf.properties.langString`)), the language
    /// tag will be returned in the location provided through `langtag`. This
    /// language tag will typically be in a format conforming
    /// [RFC 5646](https://www.rfc-editor.org/rfc/rfc5646.html).
    extern fn tracker_sparql_cursor_get_langstring(p_cursor: *SparqlCursor, p_column: c_int, p_langtag: *[*:0]const u8, p_length: ?*c_long) ?[*:0]const u8;
    pub const getLangstring = tracker_sparql_cursor_get_langstring;

    /// Retrieves the number of columns available in the result set.
    ///
    /// This method should only be called after a successful
    /// `SparqlCursor.next`, otherwise its return value
    /// will be undefined.
    extern fn tracker_sparql_cursor_get_n_columns(p_cursor: *SparqlCursor) c_int;
    pub const getNColumns = tracker_sparql_cursor_get_n_columns;

    /// Retrieves a string representation of the data in the current
    /// row in `column`.
    ///
    /// Any type may be converted to a string. If the value is not bound
    /// (See `SparqlCursor.isBound`) this method will return `NULL`.
    extern fn tracker_sparql_cursor_get_string(p_cursor: *SparqlCursor, p_column: c_int, p_length: ?*c_long) ?[*:0]const u8;
    pub const getString = tracker_sparql_cursor_get_string;

    /// Returns the data type bound to the current row and the given `column`.
    ///
    /// If the column is unbound, the value will be `TRACKER_SPARQL_VALUE_TYPE_UNBOUND`.
    /// See also `SparqlCursor.isBound`.
    ///
    /// Values of type `TRACKER_SPARQL_VALUE_TYPE_RESOURCE` and
    /// `TRACKER_SPARQL_VALUE_TYPE_BLANK_NODE` can be considered equivalent, the
    /// difference is the resource being referenced as a named IRI or a blank
    /// node.
    ///
    /// All other `SparqlValueType` value types refer to literal values.
    extern fn tracker_sparql_cursor_get_value_type(p_cursor: *SparqlCursor, p_column: c_int) tsparql.SparqlValueType;
    pub const getValueType = tracker_sparql_cursor_get_value_type;

    /// Retrieves the name of the given `column`.
    ///
    /// This name will be defined at the SPARQL query, either
    /// implicitly from the names of the variables returned in
    /// the resultset, or explicitly through the `AS ?var` SPARQL
    /// syntax.
    extern fn tracker_sparql_cursor_get_variable_name(p_cursor: *SparqlCursor, p_column: c_int) ?[*:0]const u8;
    pub const getVariableName = tracker_sparql_cursor_get_variable_name;

    /// Returns whether the given `column` has a bound value in the current row.
    ///
    /// This may not be the case through e.g. the `OPTIONAL { }` SPARQL syntax.
    extern fn tracker_sparql_cursor_is_bound(p_cursor: *SparqlCursor, p_column: c_int) c_int;
    pub const isBound = tracker_sparql_cursor_is_bound;

    /// Iterates the cursor to the next result.
    ///
    /// If the cursor was not started, it will point to the first result after
    /// this call. This operation is completely synchronous and it may block,
    /// see `SparqlCursor.nextAsync` for an asynchronous variant.
    extern fn tracker_sparql_cursor_next(p_cursor: *SparqlCursor, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const next = tracker_sparql_cursor_next;

    /// Iterates the cursor asyncronously to the next result.
    ///
    /// If the cursor was not started, it will point to the first result after
    /// this operation completes.
    ///
    /// In the period between this call and the corresponding
    /// `SparqlCursor.nextFinish` call, the other cursor methods
    /// should not be used, nor their results trusted. The cursor should only
    /// be iterated once at a time.
    extern fn tracker_sparql_cursor_next_async(p_cursor: *SparqlCursor, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const nextAsync = tracker_sparql_cursor_next_async;

    /// Finishes the asynchronous iteration to the next result started with
    /// `SparqlCursor.nextAsync`.
    extern fn tracker_sparql_cursor_next_finish(p_cursor: *SparqlCursor, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const nextFinish = tracker_sparql_cursor_next_finish;

    /// Resets the iterator to point back to the first result.
    extern fn tracker_sparql_cursor_rewind(p_cursor: *SparqlCursor) void;
    pub const rewind = tracker_sparql_cursor_rewind;

    extern fn tracker_sparql_cursor_get_type() usize;
    pub const getGObjectType = tracker_sparql_cursor_get_type;

    extern fn g_object_ref(p_self: *tsparql.SparqlCursor) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *tsparql.SparqlCursor) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SparqlCursor, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `TrackerSparqlStatement` represents a prepared statement for a SPARQL query.
///
/// The SPARQL query will be internally compiled into the format that is most
/// optimal to execute the query many times. For connections created
/// through `SparqlConnection.new` that will be a
/// SQLite compiled statement.
///
/// The SPARQL query may contain parameterized variables expressed via the
/// `~` prefix in the SPARQL syntax (e.g. `~var`), these may happen anywhere
/// in the SPARQL where a literal or variable would typically happen. These
/// parameterized variables may be mapped to arbitrary values prior to
/// execution. The `TrackerSparqlStatement` may be reused for future
/// queries with different values.
///
/// The argument bindings may be changed through the `SparqlStatement.bindInt`,
/// `SparqlStatement.bindInt`, etc... family of functions. Those functions
/// receive a `name` argument corresponding for the variable name in the SPARQL query
/// (eg. `"var"` for `~var`) and a value to map the variable to.
///
/// Once all arguments have a value, the query may be executed through
/// `SparqlStatement.executeAsync` or `SparqlStatement.execute`.
///
/// It is possible to use any `TrackerSparqlStatement` from other threads than
/// the one it was created from. However, binding values and executing the
/// statement must only happen from one thread at a time. It is possible to reuse
/// the `TrackerSparqlStatement` right after `SparqlStatement.executeAsync`
/// was called, there is no need to wait for `SparqlStatement.executeFinish`.
///
/// In some circumstances, it is possible that the query needs to be recompiled
/// from the SPARQL source. This will happen transparently.
pub const SparqlStatement = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = tsparql.SparqlStatementClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The `SparqlConnection` the statement was created for.
        pub const connection = struct {
            pub const name = "connection";

            pub const Type = ?*tsparql.SparqlConnection;
        };

        /// SPARQL query stored in this statement.
        pub const sparql = struct {
            pub const name = "sparql";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Binds the boolean `value` to the parameterized variable given by `name`.
    extern fn tracker_sparql_statement_bind_boolean(p_stmt: *SparqlStatement, p_name: [*:0]const u8, p_value: c_int) void;
    pub const bindBoolean = tracker_sparql_statement_bind_boolean;

    /// Binds the `glib.DateTime` `value` to the parameterized variable given by `name`.
    extern fn tracker_sparql_statement_bind_datetime(p_stmt: *SparqlStatement, p_name: [*:0]const u8, p_value: *glib.DateTime) void;
    pub const bindDatetime = tracker_sparql_statement_bind_datetime;

    /// Binds the double `value` to the parameterized variable given by `name`.
    extern fn tracker_sparql_statement_bind_double(p_stmt: *SparqlStatement, p_name: [*:0]const u8, p_value: f64) void;
    pub const bindDouble = tracker_sparql_statement_bind_double;

    /// Binds the integer `value` to the parameterized variable given by `name`.
    extern fn tracker_sparql_statement_bind_int(p_stmt: *SparqlStatement, p_name: [*:0]const u8, p_value: i64) void;
    pub const bindInt = tracker_sparql_statement_bind_int;

    /// Binds the `value` to the parameterized variable given by `name`, tagged
    /// with the language defined by `langtag`. The language tag should follow
    /// [RFC 5646](https://www.rfc-editor.org/rfc/rfc5646.html). The parameter
    /// will be represented as a [`rdf:langString`](rdf-ontology.html`rdf.properties.langString`).
    extern fn tracker_sparql_statement_bind_langstring(p_stmt: *SparqlStatement, p_name: [*:0]const u8, p_value: [*:0]const u8, p_langtag: [*:0]const u8) void;
    pub const bindLangstring = tracker_sparql_statement_bind_langstring;

    /// Binds the string `value` to the parameterized variable given by `name`.
    extern fn tracker_sparql_statement_bind_string(p_stmt: *SparqlStatement, p_name: [*:0]const u8, p_value: [*:0]const u8) void;
    pub const bindString = tracker_sparql_statement_bind_string;

    /// Clears all bindings.
    extern fn tracker_sparql_statement_clear_bindings(p_stmt: *SparqlStatement) void;
    pub const clearBindings = tracker_sparql_statement_clear_bindings;

    /// Executes the `SELECT` or `ASK` SPARQL query with the currently bound values.
    ///
    /// This function also works for `DESCRIBE` and `CONSTRUCT` queries that
    /// retrieve data from the triple store. These query forms that return
    /// RDF data are however more useful together with `SparqlStatement.serializeAsync`.
    ///
    /// This function should only be called on `TrackerSparqlStatement` objects
    /// obtained through `SparqlConnection.queryStatement` or
    /// SELECT/CONSTRUCT/DESCRIBE statements loaded through
    /// `SparqlConnection.loadStatementFromGresource`.
    /// An error will be raised if this method is called on a `INSERT` or `DELETE`
    /// SPARQL query.
    extern fn tracker_sparql_statement_execute(p_stmt: *SparqlStatement, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*tsparql.SparqlCursor;
    pub const execute = tracker_sparql_statement_execute;

    /// Executes asynchronously the `SELECT` or `ASK` SPARQL query with the currently bound values.
    ///
    /// This function also works for `DESCRIBE` and `CONSTRUCT` queries that
    /// retrieve data from the triple store. These query forms that return
    /// RDF data are however more useful together with `SparqlStatement.serializeAsync`.
    ///
    /// This function should only be called on `TrackerSparqlStatement` objects
    /// obtained through `SparqlConnection.queryStatement` or
    /// SELECT/CONSTRUCT/DESCRIBE statements loaded through
    /// `SparqlConnection.loadStatementFromGresource`.
    /// An error will be raised if this method is called on a `INSERT` or `DELETE`
    /// SPARQL query.
    extern fn tracker_sparql_statement_execute_async(p_stmt: *SparqlStatement, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const executeAsync = tracker_sparql_statement_execute_async;

    /// Finishes the asynchronous operation started through
    /// `SparqlStatement.executeAsync`.
    extern fn tracker_sparql_statement_execute_finish(p_stmt: *SparqlStatement, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*tsparql.SparqlCursor;
    pub const executeFinish = tracker_sparql_statement_execute_finish;

    /// Returns the `SparqlConnection` that this statement was created for.
    extern fn tracker_sparql_statement_get_connection(p_stmt: *SparqlStatement) *tsparql.SparqlConnection;
    pub const getConnection = tracker_sparql_statement_get_connection;

    /// Returns the SPARQL string that this prepared statement holds.
    extern fn tracker_sparql_statement_get_sparql(p_stmt: *SparqlStatement) [*:0]const u8;
    pub const getSparql = tracker_sparql_statement_get_sparql;

    /// Serializes a `DESCRIBE` or `CONSTRUCT` query into the given RDF `format`.
    ///
    /// The query `stmt` was created from must be either a `DESCRIBE` or `CONSTRUCT`
    /// query, an error will be raised otherwise.
    ///
    /// This is an asynchronous operation, `callback` will be invoked when the
    /// data is available for reading.
    ///
    /// The SPARQL endpoint may not support the specified format, in that case
    /// an error will be raised.
    ///
    /// The `flags` argument is reserved for future expansions, currently
    /// `TRACKER_SERIALIZE_FLAGS_NONE` must be passed.
    extern fn tracker_sparql_statement_serialize_async(p_stmt: *SparqlStatement, p_flags: tsparql.SerializeFlags, p_format: tsparql.RdfFormat, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const serializeAsync = tracker_sparql_statement_serialize_async;

    /// Finishes the asynchronous operation started through
    /// `SparqlStatement.serializeAsync`.
    extern fn tracker_sparql_statement_serialize_finish(p_stmt: *SparqlStatement, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.InputStream;
    pub const serializeFinish = tracker_sparql_statement_serialize_finish;

    /// Executes the `INSERT`/`DELETE` SPARQL query series with the currently bound values.
    ///
    /// This function should only be called on `TrackerSparqlStatement` objects
    /// obtained through `SparqlConnection.updateStatement` or
    /// `INSERT`/`DELETE` statements loaded through
    /// `SparqlConnection.loadStatementFromGresource`.
    /// An error will be raised if this method is called on
    /// `SELECT`/`ASK`/`DESCRIBE`/`CONSTRUCT` SPARQL queries.
    extern fn tracker_sparql_statement_update(p_stmt: *SparqlStatement, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const update = tracker_sparql_statement_update;

    /// Executes asynchronously the `INSERT`/`DELETE` SPARQL query series with the currently bound values.
    ///
    /// This function should only be called on `TrackerSparqlStatement` objects
    /// obtained through `SparqlConnection.updateStatement` or
    /// `INSERT`/`DELETE` statements loaded through
    /// `SparqlConnection.loadStatementFromGresource`.
    /// An error will be raised if this method is called on
    /// `SELECT`/`ASK`/`DESCRIBE`/`CONSTRUCT` SPARQL queries.
    extern fn tracker_sparql_statement_update_async(p_stmt: *SparqlStatement, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const updateAsync = tracker_sparql_statement_update_async;

    /// Finishes the asynchronous update started through
    /// `SparqlStatement.updateAsync`.
    extern fn tracker_sparql_statement_update_finish(p_stmt: *SparqlStatement, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const updateFinish = tracker_sparql_statement_update_finish;

    extern fn tracker_sparql_statement_get_type() usize;
    pub const getGObjectType = tracker_sparql_statement_get_type;

    extern fn g_object_ref(p_self: *tsparql.SparqlStatement) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *tsparql.SparqlStatement) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SparqlStatement, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const BatchClass = opaque {
    pub const Instance = tsparql.Batch;

    pub fn as(p_instance: *BatchClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const EndpointClass = opaque {
    pub const Instance = tsparql.Endpoint;

    pub fn as(p_instance: *EndpointClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const NamespaceManagerClass = extern struct {
    pub const Instance = tsparql.NamespaceManager;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *NamespaceManagerClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const NotifierClass = opaque {
    pub const Instance = tsparql.Notifier;

    pub fn as(p_instance: *NotifierClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The <structname>TrackerNotifierEvent</structname> struct represents a
/// change event in the stored data.
pub const NotifierEvent = opaque {
    /// Returns the event type.
    extern fn tracker_notifier_event_get_event_type(p_event: *NotifierEvent) tsparql.NotifierEventType;
    pub const getEventType = tracker_notifier_event_get_event_type;

    /// Returns the tracker:id of the element being notified upon. This is a `gint64`
    /// which is used as efficient internal identifier for the resource.
    extern fn tracker_notifier_event_get_id(p_event: *NotifierEvent) i64;
    pub const getId = tracker_notifier_event_get_id;

    /// Returns the Uniform Resource Name of the element. This is Tracker's
    /// public identifier for the resource.
    ///
    /// This URN is an unique string identifier for the resource being
    /// notified upon, typically of the form `urn:uuid:...`.
    extern fn tracker_notifier_event_get_urn(p_event: *NotifierEvent) [*:0]const u8;
    pub const getUrn = tracker_notifier_event_get_urn;

    extern fn tracker_notifier_event_get_type() usize;
    pub const getGObjectType = tracker_notifier_event_get_type;
};

pub const ResourceClass = opaque {
    pub const Instance = tsparql.Resource;

    pub fn as(p_instance: *ResourceClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SparqlConnectionClass = opaque {
    pub const Instance = tsparql.SparqlConnection;

    pub fn as(p_instance: *SparqlConnectionClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SparqlCursorClass = opaque {
    pub const Instance = tsparql.SparqlCursor;

    pub fn as(p_instance: *SparqlCursorClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SparqlStatementClass = opaque {
    pub const Instance = tsparql.SparqlStatement;

    pub fn as(p_instance: *SparqlStatementClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Flags affecting deserialization from a RDF data format.
pub const DeserializeFlags = enum(c_int) {
    none = 0,
    _,

    extern fn tracker_deserialize_flags_get_type() usize;
    pub const getGObjectType = tracker_deserialize_flags_get_type;
};

/// Notifier event types.
pub const NotifierEventType = enum(c_int) {
    create = 0,
    delete = 1,
    update = 2,
    _,

    extern fn tracker_notifier_event_type_get_type() usize;
    pub const getGObjectType = tracker_notifier_event_type_get_type;
};

/// Describes a RDF format to be used in data exchange.
pub const RdfFormat = enum(c_int) {
    turtle = 0,
    trig = 1,
    json_ld = 2,
    last = 3,
    _,

    extern fn tracker_rdf_format_get_type() usize;
    pub const getGObjectType = tracker_rdf_format_get_type;
};

/// Flags affecting serialization into a RDF data format.
pub const SerializeFlags = enum(c_int) {
    none = 0,
    _,

    extern fn tracker_serialize_flags_get_type() usize;
    pub const getGObjectType = tracker_serialize_flags_get_type;
};

/// Error domain for Tracker Sparql. Errors in this domain will be from the
/// `SparqlError` enumeration. See `glib.Error` for more information on error
/// domains.
pub const SparqlError = enum(c_int) {
    constraint = 0,
    internal = 1,
    no_space = 2,
    ontology_not_found = 3,
    open_error = 4,
    parse = 5,
    query_failed = 6,
    type = 7,
    unknown_class = 8,
    unknown_graph = 9,
    unknown_property = 10,
    unsupported = 11,
    missing_last_modified_header = 12,
    incomplete_property_definition = 13,
    corrupt = 14,
    last = 15,
    _,

    extern fn tracker_sparql_error_quark() glib.Quark;
    pub const quark = tracker_sparql_error_quark;

    extern fn tracker_sparql_error_get_type() usize;
    pub const getGObjectType = tracker_sparql_error_get_type;
};

/// Enumeration with the possible types of the cursor's cells
pub const SparqlValueType = enum(c_int) {
    unbound = 0,
    uri = 1,
    string = 2,
    integer = 3,
    double = 4,
    datetime = 5,
    blank_node = 6,
    boolean = 7,
    _,
};

/// Connection flags to modify `tsparql.SparqlConnection` behavior.
pub const SparqlConnectionFlags = packed struct(c_uint) {
    readonly: bool = false,
    fts_enable_stemmer: bool = false,
    fts_enable_unaccent: bool = false,
    fts_enable_stop_words: bool = false,
    fts_ignore_numbers: bool = false,
    anonymous_bnodes: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: SparqlConnectionFlags = @bitCast(@as(c_uint, 0));
    const flags_readonly: SparqlConnectionFlags = @bitCast(@as(c_uint, 1));
    const flags_fts_enable_stemmer: SparqlConnectionFlags = @bitCast(@as(c_uint, 2));
    const flags_fts_enable_unaccent: SparqlConnectionFlags = @bitCast(@as(c_uint, 4));
    const flags_fts_enable_stop_words: SparqlConnectionFlags = @bitCast(@as(c_uint, 8));
    const flags_fts_ignore_numbers: SparqlConnectionFlags = @bitCast(@as(c_uint, 16));
    const flags_anonymous_bnodes: SparqlConnectionFlags = @bitCast(@as(c_uint, 32));
    extern fn tracker_sparql_connection_flags_get_type() usize;
    pub const getGObjectType = tracker_sparql_connection_flags_get_type;
};

/// Checks that the Tracker library in use is compatible with the given version.
///
/// Generally you would pass in the constants
/// `MAJOR_VERSION`, `MINOR_VERSION`, `MICRO_VERSION`
/// as the three arguments to this function; that produces
/// a check that the library in use is compatible with
/// the version of Tracker the application or module was compiled
/// against.
///
/// Compatibility is defined by two things: first the version
/// of the running library is newer than the version
/// `required_major`.`required_minor`.`required_micro`. Second
/// the running library must be binary compatible with the
/// version `required_major`.`required_minor`.`required_micro`
/// (same major version.)
extern fn tracker_check_version(p_required_major: c_uint, p_required_minor: c_uint, p_required_micro: c_uint) [*:0]const u8;
pub const checkVersion = tracker_check_version;

/// Escapes `literal` so it is suitable for insertion in
/// SPARQL queries as string literals.
///
/// Manual construction of query strings based user input is best
/// avoided at all cost, use of `tsparql.SparqlStatement` is recommended
/// instead.
extern fn tracker_sparql_escape_string(p_literal: [*:0]const u8) [*:0]u8;
pub const sparqlEscapeString = tracker_sparql_escape_string;

/// Escapes a string for use as a URI.
extern fn tracker_sparql_escape_uri(p_uri: [*:0]const u8) [*:0]u8;
pub const sparqlEscapeUri = tracker_sparql_escape_uri;

/// Formats and escapes a string for use as a URI. This function takes variadic arguments.
extern fn tracker_sparql_escape_uri_printf(p_format: [*:0]const u8, ...) [*:0]u8;
pub const sparqlEscapeUriPrintf = tracker_sparql_escape_uri_printf;

/// Formats and escapes a string for use as a URI. This function takes a `va_list`.
///
/// Similar to the standard C `vsprintf` function but safer, since it
/// calculates the maximum space required and allocates memory to hold
/// the result.
extern fn tracker_sparql_escape_uri_vprintf(p_format: [*:0]const u8, p_args: std.builtin.VaList) [*:0]u8;
pub const sparqlEscapeUriVprintf = tracker_sparql_escape_uri_vprintf;

/// Returns a path to the built-in Nepomuk ontologies.
extern fn tracker_sparql_get_ontology_nepomuk() *gio.File;
pub const sparqlGetOntologyNepomuk = tracker_sparql_get_ontology_nepomuk;

/// Creates a fresh UUID-based URN.
extern fn tracker_sparql_get_uuid_urn() [*:0]u8;
pub const sparqlGetUuidUrn = tracker_sparql_get_uuid_urn;

/// The major version of the Tracker library.
///
/// Like `tracker_major_version`, but intended to be used at application compile time.
pub const MAJOR_VERSION = 3;
/// The micro version of the Tracker library.
///
/// Like `tracker_micro_version`, but intended to be used at application compile time.
pub const MICRO_VERSION = 0;
/// The minor version of the Tracker library.
///
/// Like `tracker_minor_version`, but intended to be used at application compile time.
pub const MINOR_VERSION = 8;
/// The Prefix of the DC (Dublin Core) namespace
pub const PREFIX_DC = "http://purl.org/dc/elements/1.1/";
/// The Prefix of the MFO namespace
pub const PREFIX_MFO = "http://tracker.api.gnome.org/ontology/v3/mfo#";
/// The Prefix of the NAO namespace
pub const PREFIX_NAO = "http://tracker.api.gnome.org/ontology/v3/nao#";
/// The Prefix of the NCO namespace
pub const PREFIX_NCO = "http://tracker.api.gnome.org/ontology/v3/nco#";
/// The Prefix of the NFO namespace
pub const PREFIX_NFO = "http://tracker.api.gnome.org/ontology/v3/nfo#";
/// The Prefix of the NIE namespace
pub const PREFIX_NIE = "http://tracker.api.gnome.org/ontology/v3/nie#";
/// The Prefix of the RDF namespace
pub const PREFIX_NMM = "http://tracker.api.gnome.org/ontology/v3/nmm#";
/// The Prefix of the NRL namespace
pub const PREFIX_NRL = "http://tracker.api.gnome.org/ontology/v3/nrl#";
/// The Prefix of the Osinfo namespace
pub const PREFIX_OSINFO = "http://tracker.api.gnome.org/ontology/v3/osinfo#";
/// The Prefix of the RDF namespace
pub const PREFIX_RDF = "http://www.w3.org/1999/02/22-rdf-syntax-ns#";
/// The Prefix of the RDFS namespace
pub const PREFIX_RDFS = "http://www.w3.org/2000/01/rdf-schema#";
/// The Prefix of the SLO namespace
pub const PREFIX_SLO = "http://tracker.api.gnome.org/ontology/v3/slo#";
/// The Prefix of the Tracker namespace
pub const PREFIX_TRACKER = "http://tracker.api.gnome.org/ontology/v3/tracker#";
/// The Prefix of the XSD namespace
pub const PREFIX_XSD = "http://www.w3.org/2001/XMLSchema#";
