pub const ext = @import("ext.zig");
const secret = @This();

const std = @import("std");
const compat = @import("compat");
const gio = @import("gio2");
const gobject = @import("gobject2");
const glib = @import("glib2");
const gmodule = @import("gmodule2");
/// A proxy object representing a collection of secrets in the Secret Service.
///
/// `secret.Collection` represents a collection of secret items stored in the
/// Secret Service.
///
/// A collection can be in a locked or unlocked state. Use
/// `SecretService.lock` or `SecretService.unlock` to lock or
/// unlock the collection.
///
/// Use the `SecretCollection.properties.items` property or
/// `SecretCollection.getItems` to lookup the items in the collection.
/// There may not be any items exposed when the collection is locked.
pub const Collection = extern struct {
    pub const Parent = gio.DBusProxy;
    pub const Implements = [_]type{ gio.AsyncInitable, gio.DBusInterface, gio.Initable };
    pub const Class = secret.CollectionClass;
    f_parent: gio.DBusProxy,
    f_pv: ?*secret.CollectionPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The date and time (in seconds since the UNIX epoch) that this
        /// collection was created.
        pub const created = struct {
            pub const name = "created";

            pub const Type = u64;
        };

        /// A set of flags describing which parts of the secret collection have
        /// been initialized.
        pub const flags = struct {
            pub const name = "flags";

            pub const Type = secret.CollectionFlags;
        };

        /// A list of `Item` objects representing the items that are in
        /// this collection. This list will be empty if the collection is locked.
        pub const items = struct {
            pub const name = "items";

            pub const Type = @compileError("no type information available");
        };

        /// The human readable label for the collection.
        ///
        /// Setting this property will result in the label of the collection being
        /// set asynchronously. To properly track the changing of the label use the
        /// `Collection.setLabel` function.
        pub const label = struct {
            pub const name = "label";

            pub const Type = ?[*:0]u8;
        };

        /// Whether the collection is locked or not.
        ///
        /// To lock or unlock a collection use the `Service.lock` or
        /// `Service.unlock` functions.
        pub const locked = struct {
            pub const name = "locked";

            pub const Type = c_int;
        };

        /// The date and time (in seconds since the UNIX epoch) that this
        /// collection was last modified.
        pub const modified = struct {
            pub const name = "modified";

            pub const Type = u64;
        };

        /// The `Service` object that this collection is associated with and
        /// uses to interact with the actual D-Bus Secret Service.
        pub const service = struct {
            pub const name = "service";

            pub const Type = ?*secret.Service;
        };
    };

    pub const signals = struct {};

    /// Create a new collection in the secret service.
    ///
    /// This method returns immediately and completes asynchronously. The secret
    /// service may prompt the user. `Service.prompt` will be used to handle
    /// any prompts that are required.
    ///
    /// An `alias` is a well-known tag for a collection, such as 'default' (ie: the
    /// default collection to store items in). This allows other applications to
    /// easily identify and share a collection. If you specify an `alias`, and a
    /// collection with that alias already exists, then a new collection will not
    /// be created. The previous one will be returned instead.
    ///
    /// If `service` is `NULL`, then `Service.get` will be called to get the
    /// default `Service` proxy.
    extern fn secret_collection_create(p_service: ?*secret.Service, p_label: [*:0]const u8, p_alias: ?[*:0]const u8, p_flags: secret.CollectionCreateFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const create = secret_collection_create;

    /// Finish operation to create a new collection in the secret service.
    extern fn secret_collection_create_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*secret.Collection;
    pub const createFinish = secret_collection_create_finish;

    /// Create a new collection in the secret service.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads. The secret service may prompt the user. `Service.prompt`
    /// will be used to handle any prompts that are required.
    ///
    /// An `alias` is a well-known tag for a collection, such as `default` (ie: the
    /// default collection to store items in). This allows other applications to
    /// easily identify and share a collection. If you specify an `alias`, and a
    /// collection with that alias already exists, then a new collection will not
    /// be created. The previous one will be returned instead.
    ///
    /// If `service` is `NULL`, then `Service.getSync` will be called to get the
    /// default `Service` proxy.
    extern fn secret_collection_create_sync(p_service: ?*secret.Service, p_label: [*:0]const u8, p_alias: ?[*:0]const u8, p_flags: secret.CollectionCreateFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*secret.Collection;
    pub const createSync = secret_collection_create_sync;

    /// Lookup which collection is assigned to this alias. Aliases help determine
    /// well known collections, such as 'default'.
    ///
    /// If `service` is `NULL`, then `Service.get` will be called to get the
    /// default `Service` proxy.
    ///
    /// This method will return immediately and complete asynchronously.
    extern fn secret_collection_for_alias(p_service: ?*secret.Service, p_alias: [*:0]const u8, p_flags: secret.CollectionFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const forAlias = secret_collection_for_alias;

    /// Finish an asynchronous operation to lookup which collection is assigned
    /// to an alias.
    extern fn secret_collection_for_alias_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*secret.Collection;
    pub const forAliasFinish = secret_collection_for_alias_finish;

    /// Lookup which collection is assigned to this alias. Aliases help determine
    /// well known collections, such as `default`.
    ///
    /// If `service` is `NULL`, then `Service.getSync` will be called to get the
    /// default `Service` proxy.
    ///
    /// This method may block and should not be used in user interface threads.
    extern fn secret_collection_for_alias_sync(p_service: ?*secret.Service, p_alias: [*:0]const u8, p_flags: secret.CollectionFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*secret.Collection;
    pub const forAliasSync = secret_collection_for_alias_sync;

    /// Get a new collection proxy for a collection in the secret service.
    ///
    /// If `service` is `NULL`, then `Service.get` will be called to get
    /// the default `Service` proxy.
    ///
    /// This method will return immediately and complete asynchronously.
    extern fn secret_collection_new_for_dbus_path(p_service: ?*secret.Service, p_collection_path: [*:0]const u8, p_flags: secret.CollectionFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const newForDbusPath = secret_collection_new_for_dbus_path;

    /// Finish asynchronous operation to get a new collection proxy for a
    /// collection in the secret service.
    extern fn secret_collection_new_for_dbus_path_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*secret.Collection;
    pub const newForDbusPathFinish = secret_collection_new_for_dbus_path_finish;

    /// Get a new collection proxy for a collection in the secret service.
    ///
    /// If `service` is `NULL`, then `Service.getSync` will be called to get
    /// the default `Service` proxy.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads.
    extern fn secret_collection_new_for_dbus_path_sync(p_service: ?*secret.Service, p_collection_path: [*:0]const u8, p_flags: secret.CollectionFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*secret.Collection;
    pub const newForDbusPathSync = secret_collection_new_for_dbus_path_sync;

    /// Delete this collection.
    ///
    /// This method returns immediately and completes asynchronously. The secret
    /// service may prompt the user. `Service.prompt` will be used to handle
    /// any prompts that show up.
    extern fn secret_collection_delete(p_self: *Collection, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const delete = secret_collection_delete;

    /// Complete operation to delete this collection.
    extern fn secret_collection_delete_finish(p_self: *Collection, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const deleteFinish = secret_collection_delete_finish;

    /// Delete this collection.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads. The secret service may prompt the user. `Service.prompt` will
    /// be used to handle any prompts that show up.
    extern fn secret_collection_delete_sync(p_self: *Collection, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const deleteSync = secret_collection_delete_sync;

    /// Get the created date and time of the collection.
    ///
    /// The return value is the number of seconds since the unix epoch, January 1st
    /// 1970.
    extern fn secret_collection_get_created(p_self: *Collection) u64;
    pub const getCreated = secret_collection_get_created;

    /// Get the flags representing what features of the `secret.Collection` proxy
    /// have been initialized.
    ///
    /// Use `Collection.loadItems` to initialize further features and change
    /// the flags.
    extern fn secret_collection_get_flags(p_self: *Collection) secret.CollectionFlags;
    pub const getFlags = secret_collection_get_flags;

    /// Get the list of items in this collection.
    extern fn secret_collection_get_items(p_self: *Collection) *glib.List;
    pub const getItems = secret_collection_get_items;

    /// Get the label of this collection.
    extern fn secret_collection_get_label(p_self: *Collection) [*:0]u8;
    pub const getLabel = secret_collection_get_label;

    /// Get whether the collection is locked or not.
    ///
    /// Use `Service.lock` or `Service.unlock` to lock or unlock the
    /// collection.
    extern fn secret_collection_get_locked(p_self: *Collection) c_int;
    pub const getLocked = secret_collection_get_locked;

    /// Get the modified date and time of the collection.
    ///
    /// The return value is the number of seconds since the unix epoch, January 1st
    /// 1970.
    extern fn secret_collection_get_modified(p_self: *Collection) u64;
    pub const getModified = secret_collection_get_modified;

    /// Get the Secret Service object that this collection was created with.
    extern fn secret_collection_get_service(p_self: *Collection) *secret.Service;
    pub const getService = secret_collection_get_service;

    /// Ensure that the `secret.Collection` proxy has loaded all the items present
    /// in the Secret Service.
    ///
    /// This affects the result of `Collection.getItems`.
    ///
    /// For collections returned from `Service.getCollections` the items will
    /// have already been loaded.
    ///
    /// This method will return immediately and complete asynchronously.
    extern fn secret_collection_load_items(p_self: *Collection, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const loadItems = secret_collection_load_items;

    /// Complete an asynchronous operation to ensure that the `secret.Collection` proxy
    /// has loaded all the items present in the Secret Service.
    extern fn secret_collection_load_items_finish(p_self: *Collection, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const loadItemsFinish = secret_collection_load_items_finish;

    /// Ensure that the `secret.Collection` proxy has loaded all the items present
    /// in the Secret Service. This affects the result of
    /// `Collection.getItems`.
    ///
    /// For collections returned from `Service.getCollections` the items
    /// will have already been loaded.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads.
    extern fn secret_collection_load_items_sync(p_self: *Collection, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const loadItemsSync = secret_collection_load_items_sync;

    /// Refresh the properties on this collection. This fires off a request to
    /// refresh, and the properties will be updated later.
    ///
    /// Calling this method is not normally necessary, as the secret service
    /// will notify the client when properties change.
    extern fn secret_collection_refresh(p_self: *Collection) void;
    pub const refresh = secret_collection_refresh;

    /// Search for items matching the `attributes` in the `collection`.
    /// The `attributes` should be a table of string keys and string values.
    ///
    /// If `SECRET_SEARCH_ALL` is set in `flags`, then all the items matching the
    /// search will be returned. Otherwise only the first item will be returned.
    /// This is almost always the unlocked item that was most recently stored.
    ///
    /// If `SECRET_SEARCH_UNLOCK` is set in `flags`, then items will be unlocked
    /// if necessary. In either case, locked and unlocked items will match the
    /// search and be returned. If the unlock fails, the search does not fail.
    ///
    /// If `SECRET_SEARCH_LOAD_SECRETS` is set in `flags`, then the items will have
    /// their secret values loaded and available via `Item.getSecret`.
    ///
    /// This function returns immediately and completes asynchronously.
    extern fn secret_collection_search(p_self: *Collection, p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_flags: secret.SearchFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const search = secret_collection_search;

    /// Complete asynchronous operation to search for items in a collection.
    extern fn secret_collection_search_finish(p_self: *Collection, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.List;
    pub const searchFinish = secret_collection_search_finish;

    /// Search for items in `collection` matching the `attributes`, and return their
    /// DBus object paths.
    ///
    /// Only the specified collection is searched. The `attributes` should be a table
    /// of string keys and string values.
    ///
    /// This function returns immediately and completes asynchronously.
    ///
    /// When your callback is called use `Collection.searchForDbusPathsFinish`
    /// to get the results of this function. Only the DBus object paths of the
    /// items will be returned. If you would like `Item` objects to be returned
    /// instead, then use the `Collection.search` function.
    extern fn secret_collection_search_for_dbus_paths(p_collection: *Collection, p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const searchForDbusPaths = secret_collection_search_for_dbus_paths;

    /// Complete asynchronous operation to search for items in a collection.
    ///
    /// DBus object paths of the items will be returned. If you would to have
    /// `Item` objects to be returned instead, then use the
    /// `Collection.search` and `Collection.searchFinish` functions.
    extern fn secret_collection_search_for_dbus_paths_finish(p_collection: *Collection, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?[*][*:0]u8;
    pub const searchForDbusPathsFinish = secret_collection_search_for_dbus_paths_finish;

    /// Search for items matching the `attributes` in `collection`, and return their
    /// DBus object paths.
    ///
    /// The `attributes` should be a table of string keys and string values.
    ///
    /// This function may block indefinitely. Use the asynchronous version
    /// in user interface threads.
    ///
    /// DBus object paths of the items will be returned. If you would to have
    /// `Item` objects to be returned instead, then use the
    /// `Collection.searchSync` function.
    extern fn secret_collection_search_for_dbus_paths_sync(p_collection: *Collection, p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?[*][*:0]u8;
    pub const searchForDbusPathsSync = secret_collection_search_for_dbus_paths_sync;

    /// Search for items matching the `attributes` in the `collection`.
    /// The `attributes` should be a table of string keys and string values.
    ///
    /// If `SECRET_SEARCH_ALL` is set in `flags`, then all the items matching the
    /// search will be returned. Otherwise only the first item will be returned.
    /// This is almost always the unlocked item that was most recently stored.
    ///
    /// If `SECRET_SEARCH_UNLOCK` is set in `flags`, then items will be unlocked
    /// if necessary. In either case, locked and unlocked items will match the
    /// search and be returned. If the unlock fails, the search does not fail.
    ///
    /// If `SECRET_SEARCH_LOAD_SECRETS` is set in `flags`, then the items will have
    /// their secret values loaded and available via `Item.getSecret`.
    ///
    /// This function may block indefinitely. Use the asynchronous version
    /// in user interface threads.
    extern fn secret_collection_search_sync(p_self: *Collection, p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_flags: secret.SearchFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*glib.List;
    pub const searchSync = secret_collection_search_sync;

    /// Set the label of this collection.
    ///
    /// This function returns immediately and completes asynchronously.
    extern fn secret_collection_set_label(p_self: *Collection, p_label: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const setLabel = secret_collection_set_label;

    /// Complete asynchronous operation to set the label of this collection.
    extern fn secret_collection_set_label_finish(p_self: *Collection, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const setLabelFinish = secret_collection_set_label_finish;

    /// Set the label of this collection.
    ///
    /// This function may block indefinitely. Use the asynchronous version
    /// in user interface threads.
    extern fn secret_collection_set_label_sync(p_self: *Collection, p_label: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const setLabelSync = secret_collection_set_label_sync;

    extern fn secret_collection_get_type() usize;
    pub const getGObjectType = secret_collection_get_type;

    extern fn g_object_ref(p_self: *secret.Collection) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *secret.Collection) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Collection, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A secret item
///
/// `secret.Item` represents a secret item stored in the Secret Service.
///
/// Each item has a value, represented by a `Value`, which can be
/// retrieved by `Item.getSecret` or set by `Item.setSecret`.
/// The item is only available when the item is not locked.
///
/// Items can be locked or unlocked using the `Service.lock` or
/// `Service.unlock` functions. The Secret Service may not be able to
/// unlock individual items, and may unlock an entire collection when a single
/// item is unlocked.
///
/// Each item has a set of attributes, which are used to locate the item later.
/// These are not stored or transferred in a secure manner. Each attribute has
/// a string name and a string value. Use `Service.search` to search for
/// items based on their attributes, and `Item.setAttributes` to change
/// the attributes associated with an item.
///
/// Items can be created with `Item.create` or `Service.store`.
pub const Item = extern struct {
    pub const Parent = gio.DBusProxy;
    pub const Implements = [_]type{ gio.AsyncInitable, gio.DBusInterface, gio.Initable, secret.Retrievable };
    pub const Class = secret.ItemClass;
    f_parent_instance: gio.DBusProxy,
    f_pv: ?*secret.ItemPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// A set of flags describing which parts of the secret item have
        /// been initialized.
        pub const flags = struct {
            pub const name = "flags";

            pub const Type = secret.ItemFlags;
        };

        /// Whether the item is locked or not.
        ///
        /// An item may not be independently lockable separate from other items in
        /// its collection.
        ///
        /// To lock or unlock a item use the `Service.lock` or
        /// `Service.unlock` functions.
        pub const locked = struct {
            pub const name = "locked";

            pub const Type = c_int;
        };

        /// The `Service` object that this item is associated with and
        /// uses to interact with the actual D-Bus Secret Service.
        pub const service = struct {
            pub const name = "service";

            pub const Type = ?*secret.Service;
        };
    };

    pub const signals = struct {};

    /// Create a new item in the secret service.
    ///
    /// If the `flags` contains `SECRET_ITEM_CREATE_REPLACE`, then the secret
    /// service will search for an item matching the `attributes`, and update that item
    /// instead of creating a new one.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads. The secret service may prompt the user. `Service.prompt`
    /// will be used to handle any prompts that are required.
    extern fn secret_item_create(p_collection: *secret.Collection, p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_label: [*:0]const u8, p_value: *secret.Value, p_flags: secret.ItemCreateFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const create = secret_item_create;

    /// Finish operation to create a new item in the secret service.
    extern fn secret_item_create_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*secret.Item;
    pub const createFinish = secret_item_create_finish;

    /// Create a new item in the secret service.
    ///
    /// If the `flags` contains `SECRET_ITEM_CREATE_REPLACE`, then the secret
    /// service will search for an item matching the `attributes`, and update that item
    /// instead of creating a new one.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads. The secret service may prompt the user. `Service.prompt`
    /// will be used to handle any prompts that are required.
    extern fn secret_item_create_sync(p_collection: *secret.Collection, p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_label: [*:0]const u8, p_value: *secret.Value, p_flags: secret.ItemCreateFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*secret.Item;
    pub const createSync = secret_item_create_sync;

    /// Load the secret values for a secret item stored in the service.
    ///
    /// The `items` must all have the same `Item.properties.service` property.
    ///
    /// This function returns immediately and completes asynchronously.
    extern fn secret_item_load_secrets(p_items: *glib.List, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const loadSecrets = secret_item_load_secrets;

    /// Complete asynchronous operation to load the secret values for
    /// secret items stored in the service.
    ///
    /// Items that are locked will not have their secrets loaded.
    extern fn secret_item_load_secrets_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const loadSecretsFinish = secret_item_load_secrets_finish;

    /// Load the secret values for a secret item stored in the service.
    ///
    /// The `items` must all have the same `Item.properties.service` property.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads.
    ///
    /// Items that are locked will not have their secrets loaded.
    extern fn secret_item_load_secrets_sync(p_items: *glib.List, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const loadSecretsSync = secret_item_load_secrets_sync;

    /// Get a new item proxy for a secret item in the secret service.
    ///
    /// If `service` is `NULL`, then `Service.get` will be called to get
    /// the default `Service` proxy.
    ///
    /// This method will return immediately and complete asynchronously.
    extern fn secret_item_new_for_dbus_path(p_service: ?*secret.Service, p_item_path: [*:0]const u8, p_flags: secret.ItemFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const newForDbusPath = secret_item_new_for_dbus_path;

    /// Finish asynchronous operation to get a new item proxy for a secret
    /// item in the secret service.
    extern fn secret_item_new_for_dbus_path_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*secret.Item;
    pub const newForDbusPathFinish = secret_item_new_for_dbus_path_finish;

    /// Get a new item proxy for a secret item in the secret service.
    ///
    /// If `service` is `NULL`, then `Service.getSync` will be called to get
    /// the default `Service` proxy.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads.
    extern fn secret_item_new_for_dbus_path_sync(p_service: ?*secret.Service, p_item_path: [*:0]const u8, p_flags: secret.ItemFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*secret.Item;
    pub const newForDbusPathSync = secret_item_new_for_dbus_path_sync;

    /// Delete this item.
    ///
    /// This method returns immediately and completes asynchronously. The secret
    /// service may prompt the user. `Service.prompt` will be used to handle
    /// any prompts that show up.
    extern fn secret_item_delete(p_self: *Item, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const delete = secret_item_delete;

    /// Complete asynchronous operation to delete the secret item.
    extern fn secret_item_delete_finish(p_self: *Item, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const deleteFinish = secret_item_delete_finish;

    /// Delete this secret item.
    ///
    /// This method may block indefinitely and should not be used in user
    /// interface threads. The secret service may prompt the user.
    /// `Service.prompt` will be used to handle any prompts that show up.
    extern fn secret_item_delete_sync(p_self: *Item, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const deleteSync = secret_item_delete_sync;

    /// Set the attributes of this item.
    ///
    /// The `attributes` are a mapping of string keys to string values.
    /// Attributes are used to search for items. Attributes are not stored
    /// or transferred securely by the secret service.
    ///
    /// Do not modify the attributes returned by this method. Use
    /// `Item.setAttributes` instead.
    extern fn secret_item_get_attributes(p_self: *Item) *glib.HashTable;
    pub const getAttributes = secret_item_get_attributes;

    /// Get the created date and time of the item.
    ///
    /// The return value is the number of seconds since the unix epoch, January 1st
    /// 1970.
    extern fn secret_item_get_created(p_self: *Item) u64;
    pub const getCreated = secret_item_get_created;

    /// Get the flags representing what features of the `secret.Item` proxy
    /// have been initialized.
    ///
    /// Use `Item.loadSecret` to initialize further features
    /// and change the flags.
    extern fn secret_item_get_flags(p_self: *Item) secret.ItemFlags;
    pub const getFlags = secret_item_get_flags;

    /// Get the label of this item.
    extern fn secret_item_get_label(p_self: *Item) [*:0]u8;
    pub const getLabel = secret_item_get_label;

    /// Get whether the item is locked or not.
    ///
    /// Depending on the secret service an item may not be able to be locked
    /// independently from the collection that it is in.
    extern fn secret_item_get_locked(p_self: *Item) c_int;
    pub const getLocked = secret_item_get_locked;

    /// Get the modified date and time of the item.
    ///
    /// The return value is the number of seconds since the unix epoch, January 1st
    /// 1970.
    extern fn secret_item_get_modified(p_self: *Item) u64;
    pub const getModified = secret_item_get_modified;

    /// Gets the name of the schema that this item was stored with. This is also
    /// available at the `xdg:schema` attribute.
    extern fn secret_item_get_schema_name(p_self: *Item) ?[*:0]u8;
    pub const getSchemaName = secret_item_get_schema_name;

    /// Get the secret value of this item.
    ///
    /// If this item is locked or the secret has not yet been loaded then this will
    /// return `NULL`.
    ///
    /// To load the secret call the `Item.loadSecret` method.
    extern fn secret_item_get_secret(p_self: *Item) ?*secret.Value;
    pub const getSecret = secret_item_get_secret;

    /// Get the Secret Service object that this item was created with.
    extern fn secret_item_get_service(p_self: *Item) *secret.Service;
    pub const getService = secret_item_get_service;

    /// Load the secret value of this item.
    ///
    /// Each item has a single secret which might be a password or some
    /// other secret binary value.
    ///
    /// This function will fail if the secret item is locked.
    ///
    /// This function returns immediately and completes asynchronously.
    extern fn secret_item_load_secret(p_self: *Item, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const loadSecret = secret_item_load_secret;

    /// Complete asynchronous operation to load the secret value of this item.
    ///
    /// The newly loaded secret value can be accessed by calling
    /// `Item.getSecret`.
    extern fn secret_item_load_secret_finish(p_self: *Item, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const loadSecretFinish = secret_item_load_secret_finish;

    /// Load the secret value of this item.
    ///
    /// Each item has a single secret which might be a password or some
    /// other secret binary value.
    ///
    /// This function may block indefinitely. Use the asynchronous version
    /// in user interface threads.
    extern fn secret_item_load_secret_sync(p_self: *Item, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const loadSecretSync = secret_item_load_secret_sync;

    /// Refresh the properties on this item.
    ///
    /// This fires off a request to refresh, and the properties will be updated
    /// later.
    ///
    /// Calling this method is not normally necessary, as the secret service
    /// will notify the client when properties change.
    extern fn secret_item_refresh(p_self: *Item) void;
    pub const refresh = secret_item_refresh;

    /// Set the attributes of this item.
    ///
    /// The `attributes` are a mapping of string keys to string values.
    /// Attributes are used to search for items. Attributes are not stored
    /// or transferred securely by the secret service.
    ///
    /// This function returns immediately and completes asynchronously.
    extern fn secret_item_set_attributes(p_self: *Item, p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const setAttributes = secret_item_set_attributes;

    /// Complete operation to set the attributes of this item.
    extern fn secret_item_set_attributes_finish(p_self: *Item, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const setAttributesFinish = secret_item_set_attributes_finish;

    /// Set the attributes of this item.
    ///
    /// The `attributes` are a mapping of string keys to string values.
    /// Attributes are used to search for items. Attributes are not stored
    /// or transferred securely by the secret service.
    ///
    /// This function may block indefinitely. Use the asynchronous version
    /// in user interface threads.
    extern fn secret_item_set_attributes_sync(p_self: *Item, p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const setAttributesSync = secret_item_set_attributes_sync;

    /// Set the label of this item.
    ///
    /// This function returns immediately and completes asynchronously.
    extern fn secret_item_set_label(p_self: *Item, p_label: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const setLabel = secret_item_set_label;

    /// Complete asynchronous operation to set the label of this collection.
    extern fn secret_item_set_label_finish(p_self: *Item, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const setLabelFinish = secret_item_set_label_finish;

    /// Set the label of this item.
    ///
    /// This function may block indefinitely. Use the asynchronous version
    /// in user interface threads.
    extern fn secret_item_set_label_sync(p_self: *Item, p_label: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const setLabelSync = secret_item_set_label_sync;

    /// Set the secret value of this item.
    ///
    /// Each item has a single secret which might be a password or some
    /// other secret binary value.
    ///
    /// This function returns immediately and completes asynchronously.
    extern fn secret_item_set_secret(p_self: *Item, p_value: *secret.Value, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const setSecret = secret_item_set_secret;

    /// Complete asynchronous operation to set the secret value of this item.
    extern fn secret_item_set_secret_finish(p_self: *Item, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const setSecretFinish = secret_item_set_secret_finish;

    /// Set the secret value of this item.
    ///
    /// Each item has a single secret which might be a password or some
    /// other secret binary value.
    ///
    /// This function may block indefinitely. Use the asynchronous version
    /// in user interface threads.
    extern fn secret_item_set_secret_sync(p_self: *Item, p_value: *secret.Value, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const setSecretSync = secret_item_set_secret_sync;

    extern fn secret_item_get_type() usize;
    pub const getGObjectType = secret_item_get_type;

    extern fn g_object_ref(p_self: *secret.Item) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *secret.Item) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Item, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A prompt in the Service
///
/// A proxy object representing a prompt that the Secret Service will display
/// to the user.
///
/// Certain actions on the Secret Service require user prompting to complete,
/// such as creating a collection, or unlocking a collection. When such a prompt
/// is necessary, then a `secret.Prompt` object is created by this library, and
/// passed to the `Service.prompt` method. In this way it is handled
/// automatically.
///
/// In order to customize prompt handling, override the
/// `Service.virtual_methods.prompt_async` and `Service.virtual_methods.prompt_finish` virtual
/// methods of the `Service` class.
pub const Prompt = extern struct {
    pub const Parent = gio.DBusProxy;
    pub const Implements = [_]type{ gio.AsyncInitable, gio.DBusInterface, gio.Initable };
    pub const Class = secret.PromptClass;
    f_parent_instance: gio.DBusProxy,
    f_pv: ?*secret.PromptPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Runs a prompt and performs the prompting.
    ///
    /// Returns `TRUE` if the prompt was completed and not dismissed.
    ///
    /// If `window_id` is non-null then it is used as an XWindow id on Linux. The API
    /// expects this id to be converted to a string using the ``d`` printf format. The
    /// Secret Service can make its prompt transient for the window with this id. In
    /// some Secret Service implementations this is not possible, so the behavior
    /// depending on this should degrade gracefully.
    ///
    /// This method will return immediately and complete asynchronously.
    extern fn secret_prompt_perform(p_self: *Prompt, p_window_id: ?[*:0]const u8, p_return_type: *const glib.VariantType, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const perform = secret_prompt_perform;

    /// Complete asynchronous operation to run a prompt and perform the prompting.
    ///
    /// Returns a variant result if the prompt was completed and not dismissed. The
    /// type of result depends on the action the prompt is completing, and is
    /// defined in the Secret Service DBus API specification.
    extern fn secret_prompt_perform_finish(p_self: *Prompt, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.Variant;
    pub const performFinish = secret_prompt_perform_finish;

    /// Runs a prompt and performs the prompting.
    ///
    /// Returns a variant result if the prompt was completed and not dismissed. The
    /// type of result depends on the action the prompt is completing, and is defined
    /// in the Secret Service DBus API specification.
    ///
    /// If `window_id` is non-null then it is used as an XWindow id on Linux. The API
    /// expects this id to be converted to a string using the ``d`` printf format. The
    /// Secret Service can make its prompt transient for the window with this id. In
    /// some Secret Service implementations this is not possible, so the behavior
    /// depending on this should degrade gracefully.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads.
    extern fn secret_prompt_perform_sync(p_self: *Prompt, p_window_id: ?[*:0]const u8, p_cancellable: ?*gio.Cancellable, p_return_type: *const glib.VariantType, p_error: ?*?*glib.Error) ?*glib.Variant;
    pub const performSync = secret_prompt_perform_sync;

    /// Runs a prompt and performs the prompting.
    ///
    /// Returns a variant result if the prompt was completed and not dismissed. The
    /// type of result depends on the action the prompt is completing, and is defined
    /// in the Secret Service DBus API specification.
    ///
    /// If `window_id` is non-null then it is used as an XWindow id on Linux. The API
    /// expects this id to be converted to a string using the ``d`` printf format. The
    /// Secret Service can make its prompt transient for the window with this id. In
    /// some Secret Service implementations this is not possible, so the behavior
    /// depending on this should degrade gracefully.
    ///
    /// This runs the dialog in a recursive mainloop. When run from a user interface
    /// thread, this means the user interface will remain responsive. Care should be
    /// taken that appropriate user interface actions are disabled while running the
    /// prompt.
    extern fn secret_prompt_run(p_self: *Prompt, p_window_id: ?[*:0]const u8, p_cancellable: ?*gio.Cancellable, p_return_type: *const glib.VariantType, p_error: ?*?*glib.Error) ?*glib.Variant;
    pub const run = secret_prompt_run;

    extern fn secret_prompt_get_type() usize;
    pub const getGObjectType = secret_prompt_get_type;

    extern fn g_object_ref(p_self: *secret.Prompt) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *secret.Prompt) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Prompt, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A proxy object representing the Secret Service.
///
/// A `secret.Service` object represents the Secret Service implementation which
/// runs as a D-Bus service.
///
/// Normally a single `secret.Service` object can be shared between multiple
/// callers. The `Service.get` method is used to access this `secret.Service`
/// object. If a new independent `secret.Service` object is required, use
/// `Service.open`.
///
/// In order to securely transfer secrets to the Sercret Service, a session
/// is established. This session can be established while initializing a
/// `secret.Service` object by passing the `SECRET_SERVICE_OPEN_SESSION` flag
/// to the `Service.get` or `Service.open` functions. In order to
/// establish a session on an already existing `secret.Service`, use the
/// `Service.ensureSession` function.
///
/// To search for items, use the `Service.search` method.
///
/// Multiple collections can exist in the Secret Service, each of which contains
/// secret items. In order to instantiate `Collection` objects which
/// represent those collections while initializing a `secret.Service` then pass
/// the `SECRET_SERVICE_LOAD_COLLECTIONS` flag to the `Service.get` or
/// `Service.open` functions. In order to establish a session on an already
/// existing `secret.Service`, use the `Service.loadCollections` function.
/// To access the list of collections use `Service.getCollections`.
///
/// Certain actions on the Secret Service require user prompting to complete,
/// such as creating a collection, or unlocking a collection. When such a prompt
/// is necessary, then a `Prompt` object is created by this library, and
/// passed to the `Service.prompt` method. In this way it is handled
/// automatically.
///
/// In order to customize prompt handling, override the
/// `Service.virtual_methods.prompt_async` and `Service.virtual_methods.prompt_finish` virtual
/// methods of the `secret.Service` class.
pub const Service = extern struct {
    pub const Parent = gio.DBusProxy;
    pub const Implements = [_]type{ gio.AsyncInitable, gio.DBusInterface, gio.Initable, secret.Backend };
    pub const Class = secret.ServiceClass;
    f_parent: gio.DBusProxy,
    f_pv: ?*secret.ServicePrivate,

    pub const virtual_methods = struct {
        /// Get the GObject type for collections instantiated by this service.
        ///
        /// This will always be either `Collection` or derived from it.
        pub const get_collection_gtype = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) usize {
                return gobject.ext.as(Service.Class, p_class).f_get_collection_gtype.?(gobject.ext.as(Service, p_self));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) usize) void {
                gobject.ext.as(Service.Class, p_class).f_get_collection_gtype = @ptrCast(p_implementation);
            }
        };

        /// Get the GObject type for items instantiated by this service.
        ///
        /// This will always be either `Item` or derived from it.
        pub const get_item_gtype = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) usize {
                return gobject.ext.as(Service.Class, p_class).f_get_item_gtype.?(gobject.ext.as(Service, p_self));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) usize) void {
                gobject.ext.as(Service.Class, p_class).f_get_item_gtype = @ptrCast(p_implementation);
            }
        };

        /// called to perform asynchronous prompting when necessary
        pub const prompt_async = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_prompt: *secret.Prompt, p_return_type: *const glib.VariantType, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(Service.Class, p_class).f_prompt_async.?(gobject.ext.as(Service, p_self), p_prompt, p_return_type, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_prompt: *secret.Prompt, p_return_type: *const glib.VariantType, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(Service.Class, p_class).f_prompt_async = @ptrCast(p_implementation);
            }
        };

        /// Complete asynchronous operation to perform prompting for a `Prompt`.
        ///
        /// Returns a variant result if the prompt was completed and not dismissed. The
        /// type of result depends on the action the prompt is completing, and is defined
        /// in the Secret Service DBus API specification.
        pub const prompt_finish = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.Variant {
                return gobject.ext.as(Service.Class, p_class).f_prompt_finish.?(gobject.ext.as(Service, p_self), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?*glib.Variant) void {
                gobject.ext.as(Service.Class, p_class).f_prompt_finish = @ptrCast(p_implementation);
            }
        };

        /// Perform prompting for a `Prompt`.
        ///
        /// Runs a prompt and performs the prompting. Returns a variant result if the
        /// prompt was completed and not dismissed. The type of result depends on the
        /// action the prompt is completing, and is defined in the Secret Service DBus
        /// API specification.
        ///
        /// This function is called by other parts of this library to handle prompts
        /// for the various actions that can require prompting.
        ///
        /// Override the `secret.ServiceClass` `Service.virtual_methods.prompt_sync` virtual method
        /// to change the behavior of the prompting. The default behavior is to simply
        /// run `Prompt.performSync` on the prompt with a `NULL` `window_id`.
        pub const prompt_sync = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_prompt: *secret.Prompt, p_cancellable: ?*gio.Cancellable, p_return_type: *const glib.VariantType, p_error: ?*?*glib.Error) ?*glib.Variant {
                return gobject.ext.as(Service.Class, p_class).f_prompt_sync.?(gobject.ext.as(Service, p_self), p_prompt, p_cancellable, p_return_type, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_prompt: *secret.Prompt, p_cancellable: ?*gio.Cancellable, p_return_type: *const glib.VariantType, p_error: ?*?*glib.Error) callconv(.C) ?*glib.Variant) void {
                gobject.ext.as(Service.Class, p_class).f_prompt_sync = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// A list of `Collection` objects representing the collections in
        /// the Secret Service.
        ///
        /// This list may be `NULL` if the collections have not been loaded.
        ///
        /// To load the collections, specify the `SECRET_SERVICE_LOAD_COLLECTIONS`
        /// initialization flag when calling the `Service.get` or
        /// `Service.open` functions. Or call the `Service.loadCollections`
        /// method.
        pub const collections = struct {
            pub const name = "collections";

            pub const Type = @compileError("no type information available");
        };
    };

    pub const signals = struct {};

    /// Disconnect the default `secret.Service` proxy returned by `Service.get`
    /// and `Service.getSync`.
    ///
    /// It is not necessary to call this function, but you may choose to do so at
    /// program exit. It is useful for testing that memory is not leaked.
    ///
    /// This function is safe to call at any time. But if other objects in this
    /// library are still referenced, then this will not result in all memory
    /// being freed.
    extern fn secret_service_disconnect() void;
    pub const disconnect = secret_service_disconnect;

    /// Get a `secret.Service` proxy for the Secret Service.
    ///
    /// If such a proxy object already exists, then the same proxy is returned.
    ///
    /// If `flags` contains any flags of which parts of the secret service to
    /// ensure are initialized, then those will be initialized before completing.
    ///
    /// This method will return immediately and complete asynchronously.
    extern fn secret_service_get(p_flags: secret.ServiceFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const get = secret_service_get;

    /// Complete an asynchronous operation to get a `secret.Service` proxy for the
    /// Secret Service.
    extern fn secret_service_get_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*secret.Service;
    pub const getFinish = secret_service_get_finish;

    /// Get a `secret.Service` proxy for the Secret Service.
    ///
    /// If such a proxy object already exists, then the same proxy is returned.
    ///
    /// If `flags` contains any flags of which parts of the secret service to
    /// ensure are initialized, then those will be initialized before returning.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads.
    extern fn secret_service_get_sync(p_flags: secret.ServiceFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*secret.Service;
    pub const getSync = secret_service_get_sync;

    /// Create a new `secret.Service` proxy for the Secret Service.
    ///
    /// This function is rarely used, see `Service.get` instead.
    ///
    /// The `service_gtype` argument should be set to `SECRET_TYPE_SERVICE` or a the type
    /// of a derived class.
    ///
    /// If `flags` contains any flags of which parts of the secret service to
    /// ensure are initialized, then those will be initialized before returning.
    ///
    /// If `service_bus_name` is `NULL` then the default is used.
    ///
    /// This method will return immediately and complete asynchronously.
    extern fn secret_service_open(p_service_gtype: usize, p_service_bus_name: ?[*:0]const u8, p_flags: secret.ServiceFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const open = secret_service_open;

    /// Complete an asynchronous operation to create a new `secret.Service` proxy for
    /// the Secret Service.
    extern fn secret_service_open_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*secret.Service;
    pub const openFinish = secret_service_open_finish;

    /// Create a new `secret.Service` proxy for the Secret Service.
    ///
    /// This function is rarely used, see `Service.getSync` instead.
    ///
    /// The `service_gtype` argument should be set to `SECRET_TYPE_SERVICE` or a the
    /// type of a derived class.
    ///
    /// If `flags` contains any flags of which parts of the secret service to
    /// ensure are initialized, then those will be initialized before returning.
    ///
    /// If `service_bus_name` is `NULL` then the default is used.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads.
    extern fn secret_service_open_sync(p_service_gtype: usize, p_service_bus_name: ?[*:0]const u8, p_flags: secret.ServiceFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*secret.Service;
    pub const openSync = secret_service_open_sync;

    /// Remove unlocked items which match the attributes from the secret service.
    ///
    /// The `attributes` should be a set of key and value string pairs.
    ///
    /// If `service` is `NULL`, then `Service.get` will be called to get
    /// the default `Service` proxy.
    ///
    /// This method will return immediately and complete asynchronously.
    extern fn secret_service_clear(p_service: ?*Service, p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const clear = secret_service_clear;

    /// Finish asynchronous operation to remove items from the secret
    /// service.
    extern fn secret_service_clear_finish(p_service: ?*Service, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const clearFinish = secret_service_clear_finish;

    /// Remove unlocked items which match the attributes from the secret service.
    ///
    /// The `attributes` should be a set of key and value string pairs.
    ///
    /// If `service` is `NULL`, then `Service.getSync` will be called to get
    /// the default `Service` proxy.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads.
    extern fn secret_service_clear_sync(p_service: ?*Service, p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const clearSync = secret_service_clear_sync;

    /// Create a new collection in the secret service, and return its path.
    ///
    /// Using this method requires that you setup a correct hash table of D-Bus
    /// properties for the new collection. You may prefer to use
    /// `Collection.create` which does handles this for you.
    ///
    /// An `alias` is a well-known tag for a collection, such as 'default' (ie: the
    /// default collection to store items in). This allows other applications to
    /// easily identify and share a collection. If a collection with the `alias`
    /// already exists, then instead of creating a new collection, the existing
    /// collection will be returned. If no collection with this alias exists, then a
    /// new collection will be created and this alias will be assigned to it.
    ///
    /// `properties` is a set of properties for the new collection. The keys in the
    /// hash table should be interface.property strings like
    /// `org.freedesktop.Secret.Collection.Label`. The values
    /// in the hash table should be `glib.Variant` values of the properties.
    ///
    /// If you wish to have a
    ///
    /// This method will return immediately and complete asynchronously. The secret
    /// service may prompt the user. `Service.prompt` will be used to handle
    /// any prompts that are required.
    extern fn secret_service_create_collection_dbus_path(p_self: *Service, p_properties: *glib.HashTable, p_alias: ?[*:0]const u8, p_flags: secret.CollectionCreateFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const createCollectionDbusPath = secret_service_create_collection_dbus_path;

    /// Finish asynchronous operation to create a new collection in the secret
    /// service.
    extern fn secret_service_create_collection_dbus_path_finish(p_self: *Service, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const createCollectionDbusPathFinish = secret_service_create_collection_dbus_path_finish;

    /// Create a new collection in the secret service and return its path.
    ///
    /// Using this method requires that you setup a correct hash table of D-Bus
    /// properties for the new collection. You may prefer to use
    /// `Collection.create` which does handles this for you.
    ///
    /// An `alias` is a well-known tag for a collection, such as 'default' (ie: the
    /// default collection to store items in). This allows other applications to
    /// easily identify and share a collection. If a collection with the `alias`
    /// already exists, then instead of creating a new collection, the existing
    /// collection will be returned. If no collection with this alias exists, then
    /// a new collection will be created and this alias will be assigned to it.
    ///
    /// `properties` is a set of properties for the new collection. The keys in the
    /// hash table should be interface.property strings like
    /// `org.freedesktop.Secret.Collection.Label`. The values
    /// in the hash table should be `glib.Variant` values of the properties.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads. The secret service may prompt the user. `Service.prompt`
    /// will be used to handle any prompts that are required.
    extern fn secret_service_create_collection_dbus_path_sync(p_self: *Service, p_properties: *glib.HashTable, p_alias: ?[*:0]const u8, p_flags: secret.CollectionCreateFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const createCollectionDbusPathSync = secret_service_create_collection_dbus_path_sync;

    /// Create a new item in a secret service collection and return its D-Bus
    /// object path.
    ///
    /// It is often easier to use `passwordStore` or `Item.create`
    /// rather than using this function. Using this method requires that you setup
    /// a correct hash table of D-Bus `properties` for the new collection.
    ///
    /// If the `flags` contains `SECRET_ITEM_CREATE_REPLACE`, then the secret
    /// service will search for an item matching the `attributes`, and update that item
    /// instead of creating a new one.
    ///
    /// `properties` is a set of properties for the new collection. The keys in the
    /// hash table should be interface.property strings like
    /// `org.freedesktop.Secret.Item.Label`. The values
    /// in the hash table should be `glib.Variant` values of the properties.
    ///
    /// This method will return immediately and complete asynchronously. The secret
    /// service may prompt the user. `Service.prompt` will be used to handle
    /// any prompts that are required.
    extern fn secret_service_create_item_dbus_path(p_self: *Service, p_collection_path: [*:0]const u8, p_properties: *glib.HashTable, p_value: *secret.Value, p_flags: secret.ItemCreateFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const createItemDbusPath = secret_service_create_item_dbus_path;

    /// Finish asynchronous operation to create a new item in the secret
    /// service.
    extern fn secret_service_create_item_dbus_path_finish(p_self: *Service, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const createItemDbusPathFinish = secret_service_create_item_dbus_path_finish;

    /// Create a new item in a secret service collection and return its D-Bus
    /// object path.
    ///
    /// It is often easier to use `passwordStoreSync` or `Item.createSync`
    /// rather than using this function. Using this method requires that you setup
    /// a correct hash table of D-Bus `properties` for the new collection.
    ///
    /// If the `flags` contains `SECRET_ITEM_CREATE_REPLACE`, then the secret
    /// service will search for an item matching the `attributes`, and update that item
    /// instead of creating a new one.
    ///
    /// `properties` is a set of properties for the new collection. The keys in the
    /// hash table should be interface.property strings like
    /// `org.freedesktop.Secret.Item.Label`. The values
    /// in the hash table should be `glib.Variant` values of the properties.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads. The secret service may prompt the user. `Service.prompt`
    /// will be used to handle any prompts that are required.
    extern fn secret_service_create_item_dbus_path_sync(p_self: *Service, p_collection_path: [*:0]const u8, p_properties: *glib.HashTable, p_value: *secret.Value, p_flags: secret.ItemCreateFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const createItemDbusPathSync = secret_service_create_item_dbus_path_sync;

    /// Decode a `Value` into `glib.Variant` received with the Secret Service
    /// DBus API.
    ///
    /// The `glib.Variant` should have a `(oayays)` signature.
    ///
    /// A session must have already been established by the `Service`, and
    /// the encoded secret must be valid for that session.
    extern fn secret_service_decode_dbus_secret(p_service: *Service, p_value: *glib.Variant) *secret.Value;
    pub const decodeDbusSecret = secret_service_decode_dbus_secret;

    /// Delete a secret item from the secret service.
    ///
    /// The item is represented by its D-Bus object path. If you already have a
    /// `Item` proxy objects, use use `Item.delete` instead.
    ///
    /// This method will return immediately and complete asynchronously.
    extern fn secret_service_delete_item_dbus_path(p_self: *Service, p_item_path: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const deleteItemDbusPath = secret_service_delete_item_dbus_path;

    /// Complete an asynchronous operation to delete a secret item from the secret
    /// service.
    extern fn secret_service_delete_item_dbus_path_finish(p_self: *Service, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const deleteItemDbusPathFinish = secret_service_delete_item_dbus_path_finish;

    /// Delete a secret item from the secret service.
    ///
    /// The item is represented by its D-Bus object path. If you already have a
    /// `Item` proxy objects, use use `Item.deleteSync` instead.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads.
    extern fn secret_service_delete_item_dbus_path_sync(p_self: *Service, p_item_path: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const deleteItemDbusPathSync = secret_service_delete_item_dbus_path_sync;

    /// Encodes a `Value` into `glib.Variant` for use with the Secret
    /// Service DBus API.
    ///
    /// The resulting `glib.Variant` will have a `(oayays)` signature.
    ///
    /// A session must have already been established by the `Service`.
    extern fn secret_service_encode_dbus_secret(p_service: *Service, p_value: *secret.Value) *glib.Variant;
    pub const encodeDbusSecret = secret_service_encode_dbus_secret;

    /// Ensure that the `secret.Service` proxy has established a session with the
    /// Secret Service.
    ///
    /// This session is used to transfer secrets.
    ///
    /// It is not normally necessary to call this method, as the session is
    /// established as necessary. You can also pass the `SECRET_SERVICE_OPEN_SESSION`
    /// to `Service.get` in order to ensure that a session has been established
    /// by the time you get the `secret.Service` proxy.
    ///
    /// This method will return immediately and complete asynchronously.
    extern fn secret_service_ensure_session(p_self: *Service, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const ensureSession = secret_service_ensure_session;

    /// Finish an asynchronous operation to ensure that the `secret.Service` proxy
    /// has established a session with the Secret Service.
    extern fn secret_service_ensure_session_finish(p_self: *Service, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const ensureSessionFinish = secret_service_ensure_session_finish;

    /// Ensure that the `secret.Service` proxy has established a session with the
    /// Secret Service.
    ///
    /// This session is used to transfer secrets.
    ///
    /// It is not normally necessary to call this method, as the session is
    /// established as necessary. You can also pass the `SECRET_SERVICE_OPEN_SESSION`
    /// to `Service.getSync` in order to ensure that a session has been
    /// established by the time you get the `secret.Service` proxy.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads.
    extern fn secret_service_ensure_session_sync(p_self: *Service, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const ensureSessionSync = secret_service_ensure_session_sync;

    /// Get the GObject type for collections instantiated by this service.
    ///
    /// This will always be either `Collection` or derived from it.
    extern fn secret_service_get_collection_gtype(p_self: *Service) usize;
    pub const getCollectionGtype = secret_service_get_collection_gtype;

    /// Get a list of `Collection` objects representing all the collections
    /// in the secret service.
    ///
    /// If the `SECRET_SERVICE_LOAD_COLLECTIONS` flag was not specified when
    /// initializing `secret.Service` proxy object, then this method will return
    /// `NULL`. Use `Service.loadCollections` to load the collections.
    extern fn secret_service_get_collections(p_self: *Service) ?*glib.List;
    pub const getCollections = secret_service_get_collections;

    /// Get the flags representing what features of the `secret.Service` proxy
    /// have been initialized.
    ///
    /// Use `Service.ensureSession` or `Service.loadCollections`
    /// to initialize further features and change the flags.
    extern fn secret_service_get_flags(p_self: *Service) secret.ServiceFlags;
    pub const getFlags = secret_service_get_flags;

    /// Get the GObject type for items instantiated by this service.
    ///
    /// This will always be either `Item` or derived from it.
    extern fn secret_service_get_item_gtype(p_self: *Service) usize;
    pub const getItemGtype = secret_service_get_item_gtype;

    /// Get the secret value for a secret item stored in the service.
    ///
    /// The item is represented by its D-Bus object path. If you already have a
    /// `Item` proxy object, use use `Item.getSecret` to more simply
    /// get its secret value.
    ///
    /// This function returns immediately and completes asynchronously.
    extern fn secret_service_get_secret_for_dbus_path(p_self: *Service, p_item_path: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const getSecretForDbusPath = secret_service_get_secret_for_dbus_path;

    /// Complete asynchronous operation to get the secret value for an
    /// secret item stored in the service.
    ///
    /// Will return `NULL` if the item is locked.
    extern fn secret_service_get_secret_for_dbus_path_finish(p_self: *Service, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*secret.Value;
    pub const getSecretForDbusPathFinish = secret_service_get_secret_for_dbus_path_finish;

    /// Get the secret value for a secret item stored in the service.
    ///
    /// The item is represented by its D-Bus object path. If you already have a
    /// `Item` proxy object, use use `Item.loadSecretSync` to more simply
    /// get its secret value.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads.
    ///
    /// Will return `NULL` if the item is locked.
    extern fn secret_service_get_secret_for_dbus_path_sync(p_self: *Service, p_item_path: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*secret.Value;
    pub const getSecretForDbusPathSync = secret_service_get_secret_for_dbus_path_sync;

    /// Get the secret values for a secret item stored in the service.
    ///
    /// The items are represented by their D-Bus object paths. If you already have
    /// `Item` proxy objects, use use `Item.loadSecrets` to more simply
    /// get their secret values.
    ///
    /// This function returns immediately and completes asynchronously.
    extern fn secret_service_get_secrets_for_dbus_paths(p_self: *Service, p_item_paths: *[*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const getSecretsForDbusPaths = secret_service_get_secrets_for_dbus_paths;

    /// Complete asynchronous operation to get the secret values for an
    /// secret items stored in the service.
    ///
    /// Items that are locked will not be included the results.
    extern fn secret_service_get_secrets_for_dbus_paths_finish(p_self: *Service, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.HashTable;
    pub const getSecretsForDbusPathsFinish = secret_service_get_secrets_for_dbus_paths_finish;

    /// Get the secret values for a secret item stored in the service.
    ///
    /// The items are represented by their D-Bus object paths. If you already have
    /// `Item` proxy objects, use use `Item.loadSecretsSync` to more
    /// simply get their secret values.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads.
    ///
    /// Items that are locked will not be included the results.
    extern fn secret_service_get_secrets_for_dbus_paths_sync(p_self: *Service, p_item_paths: *[*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*glib.HashTable;
    pub const getSecretsForDbusPathsSync = secret_service_get_secrets_for_dbus_paths_sync;

    /// Get the set of algorithms being used to transfer secrets between this
    /// secret service proxy and the Secret Service itself.
    ///
    /// This will be `NULL` if no session has been established. Use
    /// `Service.ensureSession` to establish a session.
    extern fn secret_service_get_session_algorithms(p_self: *Service) ?[*:0]const u8;
    pub const getSessionAlgorithms = secret_service_get_session_algorithms;

    /// Get the D-Bus object path of the session object being used to transfer
    /// secrets between this secret service proxy and the Secret Service itself.
    ///
    /// This will be `NULL` if no session has been established. Use
    /// `Service.ensureSession` to establish a session.
    extern fn secret_service_get_session_dbus_path(p_self: *Service) ?[*:0]const u8;
    pub const getSessionDbusPath = secret_service_get_session_dbus_path;

    /// Ensure that the `secret.Service` proxy has loaded all the collections present
    /// in the Secret Service.
    ///
    /// This affects the result of `Service.getCollections`.
    ///
    /// You can also pass the `SECRET_SERVICE_LOAD_COLLECTIONS` to
    /// `Service.getSync` in order to ensure that the collections have been
    /// loaded by the time you get the `secret.Service` proxy.
    ///
    /// This method will return immediately and complete asynchronously.
    extern fn secret_service_load_collections(p_self: *Service, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const loadCollections = secret_service_load_collections;

    /// Complete an asynchronous operation to ensure that the `secret.Service` proxy
    /// has loaded all the collections present in the Secret Service.
    extern fn secret_service_load_collections_finish(p_self: *Service, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const loadCollectionsFinish = secret_service_load_collections_finish;

    /// Ensure that the `secret.Service` proxy has loaded all the collections present
    /// in the Secret Service.
    ///
    /// This affects the result of `Service.getCollections`.
    ///
    /// You can also pass the `SECRET_SERVICE_LOAD_COLLECTIONS` to
    /// `Service.getSync` in order to ensure that the collections have been
    /// loaded by the time you get the `secret.Service` proxy.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads.
    extern fn secret_service_load_collections_sync(p_self: *Service, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const loadCollectionsSync = secret_service_load_collections_sync;

    /// Lock items or collections in the secret service.
    ///
    /// The secret service may not be able to lock items individually, and may
    /// lock an entire collection instead.
    ///
    /// If `service` is `NULL`, then `Service.get` will be called to get
    /// the default `Service` proxy.
    ///
    /// This method returns immediately and completes asynchronously. The secret
    /// service may prompt the user. `Service.prompt` will be used to handle
    /// any prompts that show up.
    extern fn secret_service_lock(p_service: ?*Service, p_objects: *glib.List, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const lock = secret_service_lock;

    /// Lock items or collections in the secret service.
    ///
    /// The items or collections are represented by their D-Bus object paths. If you
    /// already have `Item` and `Collection` proxy objects, use use
    /// `Service.lock` instead.
    ///
    /// The secret service may not be able to lock items individually, and may
    /// lock an entire collection instead.
    ///
    /// This method returns immediately and completes asynchronously. The secret
    /// service may prompt the user. `Service.prompt` will be used to handle
    /// any prompts that show up.
    extern fn secret_service_lock_dbus_paths(p_self: *Service, p_paths: [*][*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const lockDbusPaths = secret_service_lock_dbus_paths;

    /// Complete asynchronous operation to lock items or collections in the secret
    /// service.
    ///
    /// The secret service may not be able to lock items individually, and may
    /// lock an entire collection instead.
    extern fn secret_service_lock_dbus_paths_finish(p_self: *Service, p_result: *gio.AsyncResult, p_locked: ?[*]*[*:0]u8, p_error: ?*?*glib.Error) c_int;
    pub const lockDbusPathsFinish = secret_service_lock_dbus_paths_finish;

    /// Lock items or collections in the secret service.
    ///
    /// The items or collections are represented by their D-Bus object paths. If you
    /// already have `Item` and `Collection` proxy objects, use use
    /// `Service.lockSync` instead.
    ///
    /// The secret service may not be able to lock items individually, and may
    /// lock an entire collection instead.
    ///
    /// This method may block indefinitely and should not be used in user
    /// interface threads. The secret service may prompt the user.
    /// `Service.prompt` will be used to handle any prompts that show up.
    extern fn secret_service_lock_dbus_paths_sync(p_self: *Service, p_paths: [*][*:0]const u8, p_cancellable: ?*gio.Cancellable, p_locked: ?[*]*[*:0]u8, p_error: ?*?*glib.Error) c_int;
    pub const lockDbusPathsSync = secret_service_lock_dbus_paths_sync;

    /// Complete asynchronous operation to lock items or collections in the secret
    /// service.
    ///
    /// The secret service may not be able to lock items individually, and may
    /// lock an entire collection instead.
    extern fn secret_service_lock_finish(p_service: ?*Service, p_result: *gio.AsyncResult, p_locked: ?**glib.List, p_error: ?*?*glib.Error) c_int;
    pub const lockFinish = secret_service_lock_finish;

    /// Lock items or collections in the secret service.
    ///
    /// The secret service may not be able to lock items individually, and may
    /// lock an entire collection instead.
    ///
    /// If `service` is `NULL`, then `Service.getSync` will be called to get
    /// the default `Service` proxy.
    ///
    /// This method may block indefinitely and should not be used in user
    /// interface threads. The secret service may prompt the user.
    /// `Service.prompt` will be used to handle any prompts that show up.
    extern fn secret_service_lock_sync(p_service: ?*Service, p_objects: *glib.List, p_cancellable: ?*gio.Cancellable, p_locked: ?**glib.List, p_error: ?*?*glib.Error) c_int;
    pub const lockSync = secret_service_lock_sync;

    /// Lookup a secret value in the secret service.
    ///
    /// The `attributes` should be a set of key and value string pairs.
    ///
    /// If `service` is `NULL`, then `Service.get` will be called to get
    /// the default `Service` proxy.
    ///
    /// This method will return immediately and complete asynchronously.
    extern fn secret_service_lookup(p_service: ?*Service, p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const lookup = secret_service_lookup;

    /// Finish asynchronous operation to lookup a secret value in the secret service.
    ///
    /// If no secret is found then `NULL` is returned.
    extern fn secret_service_lookup_finish(p_service: ?*Service, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*secret.Value;
    pub const lookupFinish = secret_service_lookup_finish;

    /// Lookup a secret value in the secret service.
    ///
    /// The `attributes` should be a set of key and value string pairs.
    ///
    /// If `service` is `NULL`, then `Service.getSync` will be called to get
    /// the default `Service` proxy.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads.
    extern fn secret_service_lookup_sync(p_service: ?*Service, p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*secret.Value;
    pub const lookupSync = secret_service_lookup_sync;

    /// Perform prompting for a `Prompt`.
    ///
    /// This function is called by other parts of this library to handle prompts
    /// for the various actions that can require prompting.
    ///
    /// Override the `secret.ServiceClass` `Service.virtual_methods.prompt_async` virtual method
    /// to change the behavior of the prompting. The default behavior is to simply
    /// run `Prompt.perform` on the prompt.
    extern fn secret_service_prompt(p_self: *Service, p_prompt: *secret.Prompt, p_return_type: ?*const glib.VariantType, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const prompt = secret_service_prompt;

    /// Perform prompting for a `Prompt`.
    ///
    /// This function is called by other parts of this library to handle prompts
    /// for the various actions that can require prompting.
    ///
    /// Override the `secret.ServiceClass` `Service.virtual_methods.prompt_async` virtual method
    /// to change the behavior of the propmting. The default behavior is to simply
    /// run `Prompt.perform` on the prompt.
    extern fn secret_service_prompt_at_dbus_path(p_self: *Service, p_prompt_path: [*:0]const u8, p_return_type: ?*const glib.VariantType, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const promptAtDbusPath = secret_service_prompt_at_dbus_path;

    /// Complete asynchronous operation to perform prompting for a `Prompt`.
    ///
    /// Returns a variant result if the prompt was completed and not dismissed. The
    /// type of result depends on the action the prompt is completing, and is defined
    /// in the Secret Service DBus API specification.
    extern fn secret_service_prompt_at_dbus_path_finish(p_self: *Service, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.Variant;
    pub const promptAtDbusPathFinish = secret_service_prompt_at_dbus_path_finish;

    /// Perform prompting for a `Prompt`.
    ///
    /// Override the `secret.ServiceClass` `Service.virtual_methods.prompt_async` virtual method
    /// to change the behavior of the propmting. The default behavior is to simply
    /// run `Prompt.perform` on the prompt.
    ///
    /// Returns a variant result if the prompt was completed and not dismissed. The
    /// type of result depends on the action the prompt is completing, and is defined
    /// in the Secret Service DBus API specification.
    ///
    /// This method may block and should not be used in user interface threads.
    extern fn secret_service_prompt_at_dbus_path_sync(p_self: *Service, p_prompt_path: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_return_type: ?*const glib.VariantType, p_error: ?*?*glib.Error) ?*glib.Variant;
    pub const promptAtDbusPathSync = secret_service_prompt_at_dbus_path_sync;

    /// Complete asynchronous operation to perform prompting for a `Prompt`.
    ///
    /// Returns a variant result if the prompt was completed and not dismissed. The
    /// type of result depends on the action the prompt is completing, and is defined
    /// in the Secret Service DBus API specification.
    extern fn secret_service_prompt_finish(p_self: *Service, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.Variant;
    pub const promptFinish = secret_service_prompt_finish;

    /// Perform prompting for a `Prompt`.
    ///
    /// Runs a prompt and performs the prompting. Returns a variant result if the
    /// prompt was completed and not dismissed. The type of result depends on the
    /// action the prompt is completing, and is defined in the Secret Service DBus
    /// API specification.
    ///
    /// This function is called by other parts of this library to handle prompts
    /// for the various actions that can require prompting.
    ///
    /// Override the `secret.ServiceClass` `Service.virtual_methods.prompt_sync` virtual method
    /// to change the behavior of the prompting. The default behavior is to simply
    /// run `Prompt.performSync` on the prompt with a `NULL` `window_id`.
    extern fn secret_service_prompt_sync(p_self: *Service, p_prompt: *secret.Prompt, p_cancellable: ?*gio.Cancellable, p_return_type: *const glib.VariantType, p_error: ?*?*glib.Error) ?*glib.Variant;
    pub const promptSync = secret_service_prompt_sync;

    /// Lookup which collection is assigned to this alias.
    ///
    /// Aliases help determine well known collections, such as 'default'. This method
    /// looks up the dbus object path of the well known collection.
    ///
    /// This method will return immediately and complete asynchronously.
    extern fn secret_service_read_alias_dbus_path(p_self: *Service, p_alias: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const readAliasDbusPath = secret_service_read_alias_dbus_path;

    /// Finish an asynchronous operation to lookup which collection is assigned
    /// to an alias.
    ///
    /// This method returns the DBus object path of the collection
    extern fn secret_service_read_alias_dbus_path_finish(p_self: *Service, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const readAliasDbusPathFinish = secret_service_read_alias_dbus_path_finish;

    /// Lookup which collection is assigned to this alias.
    ///
    /// Aliases help determine well known collections, such as 'default'. This method
    /// returns the dbus object path of the collection.
    ///
    /// This method may block and should not be used in user interface threads.
    extern fn secret_service_read_alias_dbus_path_sync(p_self: *Service, p_alias: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const readAliasDbusPathSync = secret_service_read_alias_dbus_path_sync;

    /// Search for items matching the `attributes`.
    ///
    /// All collections are searched. The `attributes` should be a table of string
    /// keys and string values.
    ///
    /// If `service` is `NULL`, then `Service.get` will be called to get
    /// the default `Service` proxy.
    ///
    /// If `SECRET_SEARCH_ALL` is set in `flags`, then all the items matching the
    /// search will be returned. Otherwise only the first item will be returned.
    /// This is almost always the unlocked item that was most recently stored.
    ///
    /// If `SECRET_SEARCH_UNLOCK` is set in `flags`, then items will be unlocked
    /// if necessary. In either case, locked and unlocked items will match the
    /// search and be returned. If the unlock fails, the search does not fail.
    ///
    /// If `SECRET_SEARCH_LOAD_SECRETS` is set in `flags`, then the items will have
    /// their secret values loaded and available via `Item.getSecret`.
    ///
    /// This function returns immediately and completes asynchronously.
    extern fn secret_service_search(p_service: ?*Service, p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_flags: secret.SearchFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const search = secret_service_search;

    /// Complete asynchronous operation to search for items.
    extern fn secret_service_search_finish(p_service: ?*Service, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.List;
    pub const searchFinish = secret_service_search_finish;

    /// Search for items matching the `attributes`, and return their D-Bus object paths.
    ///
    /// All collections are searched. The `attributes` should be a table of string keys
    /// and string values.
    ///
    /// This function returns immediately and completes asynchronously.
    ///
    /// When your callback is called use `Service.searchForDbusPathsFinish`
    /// to get the results of this function. Only the D-Bus object paths of the
    /// items will be returned. If you would like `Item` objects to be returned
    /// instead, then use the `Service.search` function.
    extern fn secret_service_search_for_dbus_paths(p_self: *Service, p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const searchForDbusPaths = secret_service_search_for_dbus_paths;

    /// Complete asynchronous operation to search for items, and return their
    /// D-Bus object paths.
    ///
    /// Matching items that are locked or unlocked, have their D-Bus paths placed
    /// in the `locked` or `unlocked` arrays respectively.
    ///
    /// D-Bus object paths of the items will be returned in the `unlocked` or
    /// `locked` arrays. If you would to have `Item` objects to be returned
    /// instead, then us the `Service.search` and
    /// `Service.searchFinish` functions.
    extern fn secret_service_search_for_dbus_paths_finish(p_self: *Service, p_result: *gio.AsyncResult, p_unlocked: ?[*]*[*:0]u8, p_locked: ?[*]*[*:0]u8, p_error: ?*?*glib.Error) c_int;
    pub const searchForDbusPathsFinish = secret_service_search_for_dbus_paths_finish;

    /// Search for items matching the `attributes`, and return their D-Bus object
    /// paths.
    ///
    /// All collections are searched. The `attributes` should be a table of string
    /// keys and string values.
    ///
    /// This function may block indefinitely. Use the asynchronous version
    /// in user interface threads.
    ///
    /// Matching items that are locked or unlocked, have their D-Bus paths placed
    /// in the `locked` or `unlocked` arrays respectively.
    ///
    /// D-Bus object paths of the items will be returned in the `unlocked` or
    /// `locked` arrays. If you would to have `Item` objects to be returned
    /// instead, then use the `Service.searchSync` function.
    extern fn secret_service_search_for_dbus_paths_sync(p_self: *Service, p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_unlocked: ?[*]*[*:0]u8, p_locked: ?[*]*[*:0]u8, p_error: ?*?*glib.Error) c_int;
    pub const searchForDbusPathsSync = secret_service_search_for_dbus_paths_sync;

    /// Search for items matching the `attributes`.
    ///
    /// All collections are searched. The `attributes` should be a table of string
    /// keys and string values.
    ///
    /// If `service` is `NULL`, then `Service.getSync` will be called to get
    /// the default `Service` proxy.
    ///
    /// If `SECRET_SEARCH_ALL` is set in `flags`, then all the items matching the
    /// search will be returned. Otherwise only the first item will be returned.
    /// This is almost always the unlocked item that was most recently stored.
    ///
    /// If `SECRET_SEARCH_UNLOCK` is set in `flags`, then items will be unlocked
    /// if necessary. In either case, locked and unlocked items will match the
    /// search and be returned. If the unlock fails, the search does not fail.
    ///
    /// If `SECRET_SEARCH_LOAD_SECRETS` is set in `flags`, then the items' secret
    /// values will be loaded for any unlocked items. Loaded item secret values
    /// are available via `Item.getSecret`. If the load of a secret values
    /// fail, then the
    ///
    /// This function may block indefinitely. Use the asynchronous version
    /// in user interface threads.
    extern fn secret_service_search_sync(p_service: ?*Service, p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_flags: secret.SearchFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*glib.List;
    pub const searchSync = secret_service_search_sync;

    /// Assign a collection to this alias.
    ///
    /// Aliases help determine well known collections, such as 'default'.
    ///
    /// If `service` is `NULL`, then `Service.get` will be called to get
    /// the default `Service` proxy.
    ///
    /// This method will return immediately and complete asynchronously.
    extern fn secret_service_set_alias(p_service: ?*Service, p_alias: [*:0]const u8, p_collection: ?*secret.Collection, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const setAlias = secret_service_set_alias;

    /// Finish an asynchronous operation to assign a collection to an alias.
    extern fn secret_service_set_alias_finish(p_service: ?*Service, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const setAliasFinish = secret_service_set_alias_finish;

    /// Assign a collection to this alias. Aliases help determine
    /// well known collections, such as 'default'.
    ///
    /// If `service` is `NULL`, then `Service.getSync` will be called to get
    /// the default `Service` proxy.
    ///
    /// This method may block and should not be used in user interface threads.
    extern fn secret_service_set_alias_sync(p_service: ?*Service, p_alias: [*:0]const u8, p_collection: ?*secret.Collection, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const setAliasSync = secret_service_set_alias_sync;

    /// Assign a collection to this alias. Aliases help determine
    /// well known collections, such as 'default'. This method takes the dbus object
    /// path of the collection to assign to the alias.
    ///
    /// This method will return immediately and complete asynchronously.
    extern fn secret_service_set_alias_to_dbus_path(p_self: *Service, p_alias: [*:0]const u8, p_collection_path: ?[*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const setAliasToDbusPath = secret_service_set_alias_to_dbus_path;

    /// Finish an asynchronous operation to assign a collection to an alias.
    extern fn secret_service_set_alias_to_dbus_path_finish(p_self: *Service, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const setAliasToDbusPathFinish = secret_service_set_alias_to_dbus_path_finish;

    /// Assign a collection to this alias.
    ///
    /// Aliases help determine well known collections, such as 'default'. This method
    /// takes the dbus object path of the collection to assign to the alias.
    ///
    /// This method may block and should not be used in user interface threads.
    extern fn secret_service_set_alias_to_dbus_path_sync(p_self: *Service, p_alias: [*:0]const u8, p_collection_path: ?[*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const setAliasToDbusPathSync = secret_service_set_alias_to_dbus_path_sync;

    /// Store a secret value in the secret service.
    ///
    /// The `attributes` should be a set of key and value string pairs.
    ///
    /// If the attributes match a secret item already stored in the collection, then
    /// the item will be updated with these new values.
    ///
    /// If `service` is `NULL`, then `Service.get` will be called to get
    /// the default `Service` proxy.
    ///
    /// If `collection` is not specified, then the default collection will be
    /// used. Use `COLLECTION_SESSION` to store the password in the session
    /// collection, which doesn't get stored across login sessions.
    ///
    /// This method will return immediately and complete asynchronously.
    extern fn secret_service_store(p_service: ?*Service, p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_collection: ?[*:0]const u8, p_label: [*:0]const u8, p_value: *secret.Value, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const store = secret_service_store;

    /// Finish asynchronous operation to store a secret value in the secret service.
    extern fn secret_service_store_finish(p_service: ?*Service, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const storeFinish = secret_service_store_finish;

    /// Store a secret value in the secret service.
    ///
    /// The `attributes` should be a set of key and value string pairs.
    ///
    /// If the attributes match a secret item already stored in the collection, then
    /// the item will be updated with these new values.
    ///
    /// If `collection` is `NULL`, then the default collection will be
    /// used. Use `COLLECTION_SESSION` to store the password in the session
    /// collection, which doesn't get stored across login sessions.
    ///
    /// If `service` is `NULL`, then `Service.getSync` will be called to get
    /// the default `Service` proxy.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads.
    extern fn secret_service_store_sync(p_service: ?*Service, p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_collection: ?[*:0]const u8, p_label: [*:0]const u8, p_value: *secret.Value, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const storeSync = secret_service_store_sync;

    /// Unlock items or collections in the secret service.
    ///
    /// The secret service may not be able to unlock items individually, and may
    /// unlock an entire collection instead.
    ///
    /// If `service` is `NULL`, then `Service.get` will be called to get
    /// the default `Service` proxy.
    ///
    /// This method may block indefinitely and should not be used in user
    /// interface threads. The secret service may prompt the user.
    /// `Service.prompt` will be used to handle any prompts that show up.
    extern fn secret_service_unlock(p_service: ?*Service, p_objects: *glib.List, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const unlock = secret_service_unlock;

    /// Unlock items or collections in the secret service.
    ///
    /// The items or collections are represented by their D-Bus object paths. If you
    /// already have `Item` and `Collection` proxy objects, use use
    /// `Service.unlock` instead.
    ///
    /// The secret service may not be able to unlock items individually, and may
    /// unlock an entire collection instead.
    ///
    /// This method returns immediately and completes asynchronously. The secret
    /// service may prompt the user. `Service.prompt` will be used to handle
    /// any prompts that show up.
    extern fn secret_service_unlock_dbus_paths(p_self: *Service, p_paths: [*][*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const unlockDbusPaths = secret_service_unlock_dbus_paths;

    /// Complete asynchronous operation to unlock items or collections in the secret
    /// service.
    ///
    /// The secret service may not be able to unlock items individually, and may
    /// unlock an entire collection instead.
    extern fn secret_service_unlock_dbus_paths_finish(p_self: *Service, p_result: *gio.AsyncResult, p_unlocked: ?[*]*[*:0]u8, p_error: ?*?*glib.Error) c_int;
    pub const unlockDbusPathsFinish = secret_service_unlock_dbus_paths_finish;

    /// Unlock items or collections in the secret service.
    ///
    /// The items or collections are represented by their D-Bus object paths. If you
    /// already have `Item` and `Collection` proxy objects, use use
    /// `Service.unlockSync` instead.
    ///
    /// The secret service may not be able to unlock items individually, and may
    /// unlock an entire collection instead.
    ///
    /// This method may block indefinitely and should not be used in user
    /// interface threads. The secret service may prompt the user.
    /// `Service.prompt` will be used to handle any prompts that show up.
    extern fn secret_service_unlock_dbus_paths_sync(p_self: *Service, p_paths: [*][*:0]const u8, p_cancellable: ?*gio.Cancellable, p_unlocked: ?[*]*[*:0]u8, p_error: ?*?*glib.Error) c_int;
    pub const unlockDbusPathsSync = secret_service_unlock_dbus_paths_sync;

    /// Complete asynchronous operation to unlock items or collections in the secret
    /// service.
    ///
    /// The secret service may not be able to unlock items individually, and may
    /// unlock an entire collection instead.
    extern fn secret_service_unlock_finish(p_service: ?*Service, p_result: *gio.AsyncResult, p_unlocked: ?**glib.List, p_error: ?*?*glib.Error) c_int;
    pub const unlockFinish = secret_service_unlock_finish;

    /// Unlock items or collections in the secret service.
    ///
    /// The secret service may not be able to unlock items individually, and may
    /// unlock an entire collection instead.
    ///
    /// If `service` is `NULL`, then `Service.getSync` will be called to get
    /// the default `Service` proxy.
    ///
    /// This method may block indefinitely and should not be used in user
    /// interface threads. The secret service may prompt the user.
    /// `Service.prompt` will be used to handle any prompts that show up.
    extern fn secret_service_unlock_sync(p_service: ?*Service, p_objects: *glib.List, p_cancellable: ?*gio.Cancellable, p_unlocked: ?**glib.List, p_error: ?*?*glib.Error) c_int;
    pub const unlockSync = secret_service_unlock_sync;

    extern fn secret_service_get_type() usize;
    pub const getGObjectType = secret_service_get_type;

    extern fn g_object_ref(p_self: *secret.Service) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *secret.Service) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Service, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `secret.Backend` represents a backend implementation of password
/// storage.
pub const Backend = opaque {
    pub const Prerequisites = [_]type{gio.AsyncInitable};
    pub const Iface = secret.BackendInterface;
    pub const virtual_methods = struct {
        /// implementation of `passwordClear`, required
        pub const clear = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_schema: *const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(Backend.Iface, p_class).f_clear.?(gobject.ext.as(Backend, p_self), p_schema, p_attributes, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_schema: *const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(Backend.Iface, p_class).f_clear = @ptrCast(p_implementation);
            }
        };

        /// implementation of `passwordClearFinish`, required
        pub const clear_finish = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(Backend.Iface, p_class).f_clear_finish.?(gobject.ext.as(Backend, p_self), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(Backend.Iface, p_class).f_clear_finish = @ptrCast(p_implementation);
            }
        };

        /// implementation of reinitialization step in constructor, optional
        pub const ensure_for_flags = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_flags: secret.BackendFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(Backend.Iface, p_class).f_ensure_for_flags.?(gobject.ext.as(Backend, p_self), p_flags, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_flags: secret.BackendFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(Backend.Iface, p_class).f_ensure_for_flags = @ptrCast(p_implementation);
            }
        };

        /// implementation of reinitialization step in constructor, optional
        pub const ensure_for_flags_finish = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(Backend.Iface, p_class).f_ensure_for_flags_finish.?(gobject.ext.as(Backend, p_self), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(Backend.Iface, p_class).f_ensure_for_flags_finish = @ptrCast(p_implementation);
            }
        };

        /// implementation of `passwordLookup`, required
        pub const lookup = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_schema: *const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(Backend.Iface, p_class).f_lookup.?(gobject.ext.as(Backend, p_self), p_schema, p_attributes, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_schema: *const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(Backend.Iface, p_class).f_lookup = @ptrCast(p_implementation);
            }
        };

        /// implementation of `passwordLookupFinish`, required
        pub const lookup_finish = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*secret.Value {
                return gobject.ext.as(Backend.Iface, p_class).f_lookup_finish.?(gobject.ext.as(Backend, p_self), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?*secret.Value) void {
                gobject.ext.as(Backend.Iface, p_class).f_lookup_finish = @ptrCast(p_implementation);
            }
        };

        /// implementation of `passwordSearch`, required
        pub const search = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_schema: *const secret.Schema, p_attributes: *glib.HashTable, p_flags: secret.SearchFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(Backend.Iface, p_class).f_search.?(gobject.ext.as(Backend, p_self), p_schema, p_attributes, p_flags, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_schema: *const secret.Schema, p_attributes: *glib.HashTable, p_flags: secret.SearchFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(Backend.Iface, p_class).f_search = @ptrCast(p_implementation);
            }
        };

        /// implementation of `passwordSearchFinish`, required
        pub const search_finish = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.List {
                return gobject.ext.as(Backend.Iface, p_class).f_search_finish.?(gobject.ext.as(Backend, p_self), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?*glib.List) void {
                gobject.ext.as(Backend.Iface, p_class).f_search_finish = @ptrCast(p_implementation);
            }
        };

        /// implementation of `passwordStore`, required
        pub const store = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_schema: *const secret.Schema, p_attributes: *glib.HashTable, p_collection: [*:0]const u8, p_label: [*:0]const u8, p_value: *secret.Value, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(Backend.Iface, p_class).f_store.?(gobject.ext.as(Backend, p_self), p_schema, p_attributes, p_collection, p_label, p_value, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_schema: *const secret.Schema, p_attributes: *glib.HashTable, p_collection: [*:0]const u8, p_label: [*:0]const u8, p_value: *secret.Value, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(Backend.Iface, p_class).f_store = @ptrCast(p_implementation);
            }
        };

        /// implementation of `passwordStoreFinish`, required
        pub const store_finish = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(Backend.Iface, p_class).f_store_finish.?(gobject.ext.as(Backend, p_self), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(Backend.Iface, p_class).f_store_finish = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// A set of flags describing which parts of the secret backend have
        /// been initialized.
        pub const flags = struct {
            pub const name = "flags";

            pub const Type = secret.ServiceFlags;
        };
    };

    pub const signals = struct {};

    /// Get a `secret.Backend` instance.
    ///
    /// If such a backend already exists, then the same backend is returned.
    ///
    /// If `flags` contains any flags of which parts of the secret backend to
    /// ensure are initialized, then those will be initialized before completing.
    ///
    /// This method will return immediately and complete asynchronously.
    extern fn secret_backend_get(p_flags: secret.BackendFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const get = secret_backend_get;

    /// Complete an asynchronous operation to get a `secret.Backend`.
    extern fn secret_backend_get_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*secret.Backend;
    pub const getFinish = secret_backend_get_finish;

    extern fn secret_backend_get_type() usize;
    pub const getGObjectType = secret_backend_get_type;

    extern fn g_object_ref(p_self: *secret.Backend) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *secret.Backend) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Backend, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A read-only view of a secret item in the Secret Service.
///
/// `secret.Retrievable` provides a read-only view of a secret item
/// stored in the Secret Service.
///
/// Each item has a value, represented by a `Value`, which can be
/// retrieved by `Retrievable.retrieveSecret` and
/// `Retrievable.retrieveSecretFinish`.
pub const Retrievable = opaque {
    pub const Prerequisites = [_]type{gobject.Object};
    pub const Iface = secret.RetrievableInterface;
    pub const virtual_methods = struct {
        /// Retrieve the secret value of this object.
        ///
        /// Each retrievable object has a single secret which might be a
        /// password or some other secret binary value.
        ///
        /// This function returns immediately and completes asynchronously.
        pub const retrieve_secret = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(Retrievable.Iface, p_class).f_retrieve_secret.?(gobject.ext.as(Retrievable, p_self), p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(Retrievable.Iface, p_class).f_retrieve_secret = @ptrCast(p_implementation);
            }
        };

        /// Complete asynchronous operation to retrieve the secret value of this object.
        pub const retrieve_secret_finish = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*secret.Value {
                return gobject.ext.as(Retrievable.Iface, p_class).f_retrieve_secret_finish.?(gobject.ext.as(Retrievable, p_self), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?*secret.Value) void {
                gobject.ext.as(Retrievable.Iface, p_class).f_retrieve_secret_finish = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The attributes set on this item.
        ///
        /// Attributes are used to locate an item. They are not guaranteed to be
        /// stored or transferred securely.
        pub const attributes = struct {
            pub const name = "attributes";

            pub const Type = ?*glib.HashTable;
        };

        /// The date and time (in seconds since the UNIX epoch) that this
        /// item was created.
        pub const created = struct {
            pub const name = "created";

            pub const Type = u64;
        };

        /// The human readable label for the item.
        pub const label = struct {
            pub const name = "label";

            pub const Type = ?[*:0]u8;
        };

        /// The date and time (in seconds since the UNIX epoch) that this
        /// item was last modified.
        pub const modified = struct {
            pub const name = "modified";

            pub const Type = u64;
        };
    };

    pub const signals = struct {};

    /// Get the attributes of this object.
    ///
    /// The attributes are a mapping of string keys to string values.
    /// Attributes are used to search for items. Attributes are not stored
    /// or transferred securely by the secret service.
    ///
    /// Do not modify the attribute returned by this method.
    extern fn secret_retrievable_get_attributes(p_self: *Retrievable) *glib.HashTable;
    pub const getAttributes = secret_retrievable_get_attributes;

    /// Get the created date and time of the object.
    ///
    /// The return value is the number of seconds since the unix epoch, January 1st
    /// 1970.
    extern fn secret_retrievable_get_created(p_self: *Retrievable) u64;
    pub const getCreated = secret_retrievable_get_created;

    /// Get the label of this item.
    extern fn secret_retrievable_get_label(p_self: *Retrievable) [*:0]u8;
    pub const getLabel = secret_retrievable_get_label;

    /// Get the modified date and time of the object.
    ///
    /// The return value is the number of seconds since the unix epoch, January 1st
    /// 1970.
    extern fn secret_retrievable_get_modified(p_self: *Retrievable) u64;
    pub const getModified = secret_retrievable_get_modified;

    /// Retrieve the secret value of this object.
    ///
    /// Each retrievable object has a single secret which might be a
    /// password or some other secret binary value.
    ///
    /// This function returns immediately and completes asynchronously.
    extern fn secret_retrievable_retrieve_secret(p_self: *Retrievable, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const retrieveSecret = secret_retrievable_retrieve_secret;

    /// Complete asynchronous operation to retrieve the secret value of this object.
    extern fn secret_retrievable_retrieve_secret_finish(p_self: *Retrievable, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*secret.Value;
    pub const retrieveSecretFinish = secret_retrievable_retrieve_secret_finish;

    /// Retrieve the secret value of this object synchronously.
    ///
    /// Each retrievable object has a single secret which might be a
    /// password or some other secret binary value.
    ///
    /// This method may block indefinitely and should not be used in user interface
    /// threads.
    extern fn secret_retrievable_retrieve_secret_sync(p_self: *Retrievable, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*secret.Value;
    pub const retrieveSecretSync = secret_retrievable_retrieve_secret_sync;

    extern fn secret_retrievable_get_type() usize;
    pub const getGObjectType = secret_retrievable_get_type;

    extern fn g_object_ref(p_self: *secret.Retrievable) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *secret.Retrievable) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Retrievable, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The interface for `secret.Backend`.
pub const BackendInterface = extern struct {
    pub const Instance = secret.Backend;

    /// the parent interface
    f_parent_iface: gobject.TypeInterface,
    /// implementation of reinitialization step in constructor, optional
    f_ensure_for_flags: ?*const fn (p_self: *secret.Backend, p_flags: secret.BackendFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void,
    /// implementation of reinitialization step in constructor, optional
    f_ensure_for_flags_finish: ?*const fn (p_self: *secret.Backend, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) c_int,
    /// implementation of `passwordStore`, required
    f_store: ?*const fn (p_self: *secret.Backend, p_schema: *const secret.Schema, p_attributes: *glib.HashTable, p_collection: [*:0]const u8, p_label: [*:0]const u8, p_value: *secret.Value, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void,
    /// implementation of `passwordStoreFinish`, required
    f_store_finish: ?*const fn (p_self: *secret.Backend, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) c_int,
    /// implementation of `passwordLookup`, required
    f_lookup: ?*const fn (p_self: *secret.Backend, p_schema: *const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void,
    /// implementation of `passwordLookupFinish`, required
    f_lookup_finish: ?*const fn (p_self: *secret.Backend, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?*secret.Value,
    /// implementation of `passwordClear`, required
    f_clear: ?*const fn (p_self: *secret.Backend, p_schema: *const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void,
    /// implementation of `passwordClearFinish`, required
    f_clear_finish: ?*const fn (p_self: *secret.Backend, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) c_int,
    /// implementation of `passwordSearch`, required
    f_search: ?*const fn (p_self: *secret.Backend, p_schema: *const secret.Schema, p_attributes: *glib.HashTable, p_flags: secret.SearchFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void,
    /// implementation of `passwordSearchFinish`, required
    f_search_finish: ?*const fn (p_self: *secret.Backend, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?*glib.List,

    pub fn as(p_instance: *BackendInterface, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The class for `secret.Collection`.
pub const CollectionClass = extern struct {
    pub const Instance = secret.Collection;

    /// the parent class
    f_parent_class: gio.DBusProxyClass,
    f_padding: [8]*anyopaque,

    pub fn as(p_instance: *CollectionClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const CollectionPrivate = opaque {};

/// The class for `secret.Item`.
pub const ItemClass = extern struct {
    pub const Instance = secret.Item;

    /// the parent class
    f_parent_class: gio.DBusProxyClass,
    f_padding: [4]*anyopaque,

    pub fn as(p_instance: *ItemClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ItemPrivate = opaque {};

/// The class for `secret.Prompt`.
pub const PromptClass = extern struct {
    pub const Instance = secret.Prompt;

    /// the parent class
    f_parent_class: gio.DBusProxyClass,
    f_padding: [8]*anyopaque,

    pub fn as(p_instance: *PromptClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const PromptPrivate = opaque {};

/// The interface for `secret.Retrievable`.
pub const RetrievableInterface = extern struct {
    pub const Instance = secret.Retrievable;

    /// the parent interface
    f_parent_iface: gobject.TypeInterface,
    /// implementation of `Retrievable.retrieveSecret`,
    ///   required
    f_retrieve_secret: ?*const fn (p_self: *secret.Retrievable, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void,
    /// implementation of
    ///   `Retrievable.retrieveSecretFinish`, required
    f_retrieve_secret_finish: ?*const fn (p_self: *secret.Retrievable, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?*secret.Value,

    pub fn as(p_instance: *RetrievableInterface, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Represents a set of attributes that are stored with an item.
///
/// These schemas are used for interoperability between various services storing
/// the same types of items.
///
/// Each schema has a name like `org.gnome.keyring.NetworkPassword`, and defines a
/// set of attributes, and types (string, integer, boolean) for those attributes.
///
/// Attributes are stored as strings in the Secret Service, and the attribute types
/// simply define standard ways to store integer and boolean values as strings.
/// Attributes are represented in libsecret via a `glib.HashTable` with
/// string keys and values. Even for values that defined as an integer or boolean in
/// the schema, the attribute values in the `glib.HashTable` are strings.
/// Boolean values are stored as the strings 'true' and 'false'. Integer values are
/// stored in decimal, with a preceding negative sign for negative integers.
///
/// Schemas are handled entirely on the client side by this library. The name of the
/// schema is automatically stored as an attribute on the item.
///
/// Normally when looking up passwords only those with matching schema names are
/// returned. If the schema `flags` contain the `SECRET_SCHEMA_DONT_MATCH_NAME` flag,
/// then lookups will not check that the schema name matches that on the item, only
/// the schema's attributes are matched. This is useful when you are looking up
/// items that are not stored by the libsecret library. Other libraries such as
/// libgnome-keyring don't store the schema name.
///
/// Additional schemas can be defined via the `Schema` structure like this:
///
/// ```c
/// // in a header:
///
/// const SecretSchema * example_get_schema (void) G_GNUC_CONST;
///
/// `define` EXAMPLE_SCHEMA  example_get_schema ()
///
///
/// // in a .c file
///
/// const SecretSchema *
/// example_get_schema (void)
/// {
///     static const SecretSchema the_schema = {
///         "org.example.Password", SECRET_SCHEMA_NONE,
///         {
///             {  "number", SECRET_SCHEMA_ATTRIBUTE_INTEGER },
///             {  "string", SECRET_SCHEMA_ATTRIBUTE_STRING },
///             {  "even", SECRET_SCHEMA_ATTRIBUTE_BOOLEAN },
///             {  NULL, 0 },
///         }
///     };
///     return &the_schema;
/// }
/// ```
pub const Schema = extern struct {
    /// the dotted name of the schema
    f_name: ?[*:0]const u8,
    /// flags for the schema
    f_flags: secret.SchemaFlags,
    /// the attribute names and types of those attributes
    f_attributes: [32]secret.SchemaAttribute,
    f_reserved: c_int,
    f_reserved1: ?*anyopaque,
    f_reserved2: ?*anyopaque,
    f_reserved3: ?*anyopaque,
    f_reserved4: ?*anyopaque,
    f_reserved5: ?*anyopaque,
    f_reserved6: ?*anyopaque,
    f_reserved7: ?*anyopaque,

    /// Using this function is not normally necessary from C code.
    ///
    /// A schema represents a set of attributes that are stored with an item. These
    /// schemas are used for interoperability between various services storing the
    /// same types of items.
    ///
    /// Each schema has an `name` like `org.gnome.keyring.NetworkPassword`, and
    /// defines a set of attributes names, and types (string, integer, boolean) for
    /// those attributes.
    ///
    /// The variable argument list should contain pairs of a) The attribute name as
    /// a null-terminated string, followed by b) integers from the
    /// `SchemaAttributeType` enumeration, representing the attribute type for
    /// each attribute name. The list of attributes should be terminated with a `NULL`.
    ///
    /// Normally when looking up passwords only those with matching schema names are
    /// returned. If the schema `flags` contain the `SECRET_SCHEMA_DONT_MATCH_NAME` flag,
    /// then lookups will not check that the schema name matches that on the item, only
    /// the schema's attributes are matched. This is useful when you are looking up items
    /// that are not stored by the libsecret library. Other libraries such as libgnome-keyring
    /// don't store the schema name.
    extern fn secret_schema_new(p_name: [*:0]const u8, p_flags: secret.SchemaFlags, ...) *secret.Schema;
    pub const new = secret_schema_new;

    /// Using this function is not normally necessary from C code. This is useful
    /// for constructing `secret.Schema` structures in bindings.
    ///
    /// A schema represents a set of attributes that are stored with an item. These
    /// schemas are used for interoperability between various services storing the
    /// same types of items.
    ///
    /// Each schema has an `name` like `org.gnome.keyring.NetworkPassword`, and
    /// defines a set of attributes names, and types (string, integer, boolean) for
    /// those attributes.
    ///
    /// Each key in the `attributes` table should be a attribute name strings, and
    /// the values in the table should be integers from the `SchemaAttributeType`
    /// enumeration, representing the attribute type for each attribute name.
    ///
    /// Normally when looking up passwords only those with matching schema names are
    /// returned. If the schema `flags` contain the `SECRET_SCHEMA_DONT_MATCH_NAME` flag,
    /// then lookups will not check that the schema name matches that on the item, only
    /// the schema's attributes are matched. This is useful when you are looking up items
    /// that are not stored by the libsecret library. Other libraries such as libgnome-keyring
    /// don't store the schema name.
    extern fn secret_schema_newv(p_name: [*:0]const u8, p_flags: secret.SchemaFlags, p_attribute_names_and_types: *glib.HashTable) *secret.Schema;
    pub const newv = secret_schema_newv;

    /// Adds a reference to the `secret.Schema`.
    ///
    /// It is not normally necessary to call this function from C code, and is
    /// mainly present for the sake of bindings. If the `schema` was statically
    /// allocated, then this function will copy the schema.
    extern fn secret_schema_ref(p_schema: *Schema) *secret.Schema;
    pub const ref = secret_schema_ref;

    /// Releases a reference to the `secret.Schema`.
    ///
    /// If the last reference is released then the schema will be freed.
    ///
    /// It is not normally necessary to call this function from C code, and is
    /// mainly present for the sake of bindings. It is an error to call this for
    /// a `schema` that was statically allocated.
    extern fn secret_schema_unref(p_schema: *Schema) void;
    pub const unref = secret_schema_unref;

    extern fn secret_schema_get_type() usize;
    pub const getGObjectType = secret_schema_get_type;
};

/// An attribute in a `secret.Schema`.
pub const SchemaAttribute = extern struct {
    /// name of the attribute
    f_name: ?[*:0]const u8,
    /// the type of the attribute
    f_type: secret.SchemaAttributeType,

    extern fn secret_schema_attribute_get_type() usize;
    pub const getGObjectType = secret_schema_attribute_get_type;
};

/// The class for `secret.Service`.
pub const ServiceClass = extern struct {
    pub const Instance = secret.Service;

    /// the parent class
    f_parent_class: gio.DBusProxyClass,
    /// the `glib.Type` of the `Collection` objects instantiated
    ///   by the `secret.Service` proxy
    f_collection_gtype: usize,
    /// the `glib.Type` of the `Item` objects instantiated by the
    ///   `secret.Service` proxy
    f_item_gtype: usize,
    /// called to perform synchronous prompting when necessary
    f_prompt_sync: ?*const fn (p_self: *secret.Service, p_prompt: *secret.Prompt, p_cancellable: ?*gio.Cancellable, p_return_type: *const glib.VariantType, p_error: ?*?*glib.Error) callconv(.C) ?*glib.Variant,
    /// called to perform asynchronous prompting when necessary
    f_prompt_async: ?*const fn (p_self: *secret.Service, p_prompt: *secret.Prompt, p_return_type: *const glib.VariantType, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void,
    /// called to complete an asynchronous prompt operation
    f_prompt_finish: ?*const fn (p_self: *secret.Service, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?*glib.Variant,
    /// called to get the GObject type for collections
    ///   instantiated by the `secret.Service` proxy
    f_get_collection_gtype: ?*const fn (p_self: *secret.Service) callconv(.C) usize,
    /// called to get the GObject type for collections
    ///   instantiated by the `secret.Service` proxy
    f_get_item_gtype: ?*const fn (p_self: *secret.Service) callconv(.C) usize,
    f_padding: [14]*anyopaque,

    pub fn as(p_instance: *ServiceClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ServicePrivate = opaque {};

/// A value containing a secret
///
/// A `secret.Value` contains a password or other secret value.
///
/// Use `Value.get` to get the actual secret data, such as a password.
/// The secret data is not necessarily null-terminated, unless the content type
/// is "text/plain".
///
/// Each `secret.Value` has a content type. For passwords, this is `text/plain`.
/// Use `Value.getContentType` to look at the content type.
///
/// `secret.Value` is reference counted and immutable. The secret data is only
/// freed when all references have been released via `Value.unref`.
pub const Value = opaque {
    /// Create a `secret.Value` for the secret data passed in.
    ///
    /// The secret data is copied into non-pageable 'secure' memory.
    ///
    /// If the length is less than zero, then `secret` is assumed to be
    /// null-terminated.
    extern fn secret_value_new(p_secret: [*:0]const u8, p_length: isize, p_content_type: [*:0]const u8) *secret.Value;
    pub const new = secret_value_new;

    /// Create a `secret.Value` for the secret data passed in.
    ///
    /// The secret data is not copied, and will later be freed with the `destroy`
    /// function.
    ///
    /// If the length is less than zero, then `secret` is assumed to be
    /// null-terminated.
    extern fn secret_value_new_full(p_secret: [*:0]u8, p_length: isize, p_content_type: [*:0]const u8, p_destroy: glib.DestroyNotify) *secret.Value;
    pub const newFull = secret_value_new_full;

    /// Get the secret data in the `secret.Value`.
    ///
    /// The value is not necessarily null-terminated unless it was created with
    /// `Value.new` or a null-terminated string was passed to
    /// `Value.newFull`.
    extern fn secret_value_get(p_value: *Value, p_length: *usize) [*]const u8;
    pub const get = secret_value_get;

    /// Get the content type of the secret value, such as
    /// `text/plain`.
    extern fn secret_value_get_content_type(p_value: *Value) [*:0]const u8;
    pub const getContentType = secret_value_get_content_type;

    /// Get the secret data in the `secret.Value` if it contains a textual
    /// value.
    ///
    /// The content type must be `text/plain`.
    extern fn secret_value_get_text(p_value: *Value) ?[*:0]const u8;
    pub const getText = secret_value_get_text;

    /// Add another reference to the `secret.Value`.
    ///
    /// For each reference `Value.unref` should be called to unreference the
    /// value.
    extern fn secret_value_ref(p_value: *Value) *secret.Value;
    pub const ref = secret_value_ref;

    /// Unreference a `secret.Value`.
    ///
    /// When the last reference is gone, then the value will be freed.
    extern fn secret_value_unref(p_value: *Value) void;
    pub const unref = secret_value_unref;

    /// Unreference a `secret.Value` and steal the secret data in
    /// `secret.Value` as nonpageable memory.
    extern fn secret_value_unref_to_password(p_value: *Value, p_length: *usize) [*:0]u8;
    pub const unrefToPassword = secret_value_unref_to_password;

    extern fn secret_value_get_type() usize;
    pub const getGObjectType = secret_value_get_type;
};

/// Flags which determine which parts of the `secret.Backend` are initialized.
pub const BackendFlags = enum(c_int) {
    none = 0,
    open_session = 2,
    load_collections = 4,
    _,

    extern fn secret_backend_flags_get_type() usize;
    pub const getGObjectType = secret_backend_flags_get_type;
};

/// Errors returned by the Secret Service.
///
/// None of the errors are appropriate for display to the user. It is up to the
/// application to handle them appropriately.
pub const Error = enum(c_int) {
    protocol = 1,
    is_locked = 2,
    no_such_object = 3,
    already_exists = 4,
    invalid_file_format = 5,
    mismatched_schema = 6,
    no_matching_attribute = 7,
    wrong_type = 8,
    empty_table = 9,
    _,

    /// Get the error quark.
    extern fn secret_error_get_quark() glib.Quark;
    pub const getQuark = secret_error_get_quark;

    extern fn secret_error_get_type() usize;
    pub const getGObjectType = secret_error_get_type;
};

/// The type of an attribute in a `SecretSchema`.
///
/// Attributes are stored as strings in the Secret Service, and the attribute
/// types simply define standard ways to store integer and boolean values as
/// strings.
pub const SchemaAttributeType = enum(c_int) {
    string = 0,
    integer = 1,
    boolean = 2,
    _,

    extern fn secret_schema_attribute_type_get_type() usize;
    pub const getGObjectType = secret_schema_attribute_type_get_type;
};

/// Different types of schemas for storing secrets, intended for use with
/// `getSchema`.
///
/// ## @SECRET_SCHEMA_NOTE
///
/// A predefined schema for personal passwords stored by the user in the
/// password manager. This schema has no attributes, and the items are not
/// meant to be used automatically by applications.
///
/// When used to search for items using this schema, it will only match
/// items that have the same schema. Items stored via libgnome-keyring with the
/// `GNOME_KEYRING_ITEM_NOTE` item type will match.
///
/// ## @SECRET_SCHEMA_COMPAT_NETWORK
///
/// A predefined schema that is compatible with items stored via the
/// libgnome-keyring 'network password' functions. This is meant to be used by
/// applications migrating from libgnome-keyring which stored their secrets as
/// 'network passwords'. It is not recommended that new code use this schema.
///
/// When used to search for items using this schema, it will only match
/// items that have the same schema. Items stored via libgnome-keyring with the
/// `GNOME_KEYRING_ITEM_NETWORK_PASSWORD` item type will match.
///
/// The following attributes exist in the schema:
///
/// ### Attributes:
///
/// <table>
///     <tr>
///         <td><tt>user</tt>:</td>
///         <td>The user name (string).</td>
///     </tr>
///     <tr>
///         <td><tt>domain</tt>:</td>
///         <td>The login domain or realm (string).</td></tr>
///     <tr>
///         <td><tt>object</tt>:</td>
///         <td>The object or path (string).</td>
///     </tr>
///     <tr>
///         <td><tt>protocol</tt>:</td>
///         <td>The protocol (a string like 'http').</td>
///     </tr>
///     <tr>
///         <td><tt>port</tt>:</td>
///         <td>The network port (integer).</td>
///     </tr>
///     <tr>
///         <td><tt>server</tt>:</td>
///         <td>The hostname or server (string).</td>
///     </tr>
///     <tr>
///         <td><tt>authtype</tt>:</td>
///         <td>The authentication type (string).</td>
///     </tr>
/// </table>
pub const SchemaType = enum(c_int) {
    note = 0,
    compat_network = 1,
    _,

    extern fn secret_schema_type_get_type() usize;
    pub const getGObjectType = secret_schema_type_get_type;
};

/// Flags for `Collection.create`.
pub const CollectionCreateFlags = packed struct(c_uint) {
    _padding0: bool = false,
    _padding1: bool = false,
    _padding2: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: CollectionCreateFlags = @bitCast(@as(c_uint, 0));
    extern fn secret_collection_create_flags_get_type() usize;
    pub const getGObjectType = secret_collection_create_flags_get_type;
};

/// Flags which determine which parts of the `secret.Collection` proxy are initialized.
pub const CollectionFlags = packed struct(c_uint) {
    _padding0: bool = false,
    load_items: bool = false,
    _padding2: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: CollectionFlags = @bitCast(@as(c_uint, 0));
    const flags_load_items: CollectionFlags = @bitCast(@as(c_uint, 2));
    extern fn secret_collection_flags_get_type() usize;
    pub const getGObjectType = secret_collection_flags_get_type;
};

/// Flags for `Item.create`.
pub const ItemCreateFlags = packed struct(c_uint) {
    _padding0: bool = false,
    replace: bool = false,
    _padding2: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: ItemCreateFlags = @bitCast(@as(c_uint, 0));
    const flags_replace: ItemCreateFlags = @bitCast(@as(c_uint, 2));
    extern fn secret_item_create_flags_get_type() usize;
    pub const getGObjectType = secret_item_create_flags_get_type;
};

/// Flags which determine which parts of the `secret.Item` proxy are initialized.
pub const ItemFlags = packed struct(c_uint) {
    _padding0: bool = false,
    load_secret: bool = false,
    _padding2: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: ItemFlags = @bitCast(@as(c_uint, 0));
    const flags_load_secret: ItemFlags = @bitCast(@as(c_uint, 2));
    extern fn secret_item_flags_get_type() usize;
    pub const getGObjectType = secret_item_flags_get_type;
};

/// Flags for a `secret.Schema` definition.
pub const SchemaFlags = packed struct(c_uint) {
    _padding0: bool = false,
    dont_match_name: bool = false,
    _padding2: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: SchemaFlags = @bitCast(@as(c_uint, 0));
    const flags_dont_match_name: SchemaFlags = @bitCast(@as(c_uint, 2));
    extern fn secret_schema_flags_get_type() usize;
    pub const getGObjectType = secret_schema_flags_get_type;
};

/// Various flags to be used with `Service.search` and `Service.searchSync`.
pub const SearchFlags = packed struct(c_uint) {
    _padding0: bool = false,
    all: bool = false,
    unlock: bool = false,
    load_secrets: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: SearchFlags = @bitCast(@as(c_uint, 0));
    const flags_all: SearchFlags = @bitCast(@as(c_uint, 2));
    const flags_unlock: SearchFlags = @bitCast(@as(c_uint, 4));
    const flags_load_secrets: SearchFlags = @bitCast(@as(c_uint, 8));
    extern fn secret_search_flags_get_type() usize;
    pub const getGObjectType = secret_search_flags_get_type;
};

/// Flags which determine which parts of the `secret.Service` proxy are initialized
/// during a `Service.get` or `Service.open` operation.
pub const ServiceFlags = packed struct(c_uint) {
    _padding0: bool = false,
    open_session: bool = false,
    load_collections: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: ServiceFlags = @bitCast(@as(c_uint, 0));
    const flags_open_session: ServiceFlags = @bitCast(@as(c_uint, 2));
    const flags_load_collections: ServiceFlags = @bitCast(@as(c_uint, 4));
    extern fn secret_service_flags_get_type() usize;
    pub const getGObjectType = secret_service_flags_get_type;
};

/// Build up a hash table of attribute values.
///
/// The variable argument list should contain pairs of a) The attribute name as
/// a null-terminated string, followed by b) attribute value, either a character
/// string, an int number, or a gboolean value, as defined in the password
/// `schema`. The list of attributes should be terminated with a `NULL`.
extern fn secret_attributes_build(p_schema: *const secret.Schema, ...) *glib.HashTable;
pub const attributesBuild = secret_attributes_build;

/// Build up a hash table of attribute values.
///
/// The variable argument list should contain pairs of a) The attribute name as
/// a null-terminated string, followed by b) attribute value, either a character
/// string, an int number, or a gboolean value, as defined in the password
/// `schema`. The list of attributes should be terminated with a `NULL`.
extern fn secret_attributes_buildv(p_schema: *const secret.Schema, p_va: std.builtin.VaList) *glib.HashTable;
pub const attributesBuildv = secret_attributes_buildv;

/// Check if attributes are valid according to the provided schema.
///
/// Verifies schema name if available, attribute names and parsing
/// of attribute values.
extern fn secret_attributes_validate(p_schema: *const secret.Schema, p_attributes: *glib.HashTable, p_error: ?*?*glib.Error) c_int;
pub const attributesValidate = secret_attributes_validate;

/// Get a secret storage schema of the given `type`.
///
/// C code may access the schemas (such as `SECRET_SCHEMA_NOTE`) directly, but
/// language bindings cannot, and must use this accessor.
extern fn secret_get_schema(p_type: secret.SchemaType) *const secret.Schema;
pub const getSchema = secret_get_schema;

/// Clear unlocked matching passwords from the secret service.
///
/// The variable argument list should contain pairs of a) The attribute name as
/// a null-terminated string, followed by b) attribute value, either a character
/// string, an int number, or a gboolean value, as defined in the password
/// `schema`. The list of attributes should be terminated with a `NULL`.
///
/// All unlocked items that match the attributes will be deleted.
///
/// This method will return immediately and complete asynchronously.
extern fn secret_password_clear(p_schema: *const secret.Schema, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque, ...) void;
pub const passwordClear = secret_password_clear;

/// Finish an asynchronous operation to remove passwords from the secret
/// service.
extern fn secret_password_clear_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
pub const passwordClearFinish = secret_password_clear_finish;

/// Remove unlocked matching passwords from the secret service.
///
/// The variable argument list should contain pairs of a) The attribute name as
/// a null-terminated string, followed by b) attribute value, either a character
/// string, an int number, or a gboolean value, as defined in the password
/// `schema`. The list of attributes should be terminated with a `NULL`.
///
/// All unlocked items that match the attributes will be deleted.
///
/// This method may block indefinitely and should not be used in user interface
/// threads.
extern fn secret_password_clear_sync(p_schema: *const secret.Schema, p_cancellable: ?*gio.Cancellable, p_error: **glib.Error, ...) c_int;
pub const passwordClearSync = secret_password_clear_sync;

/// Remove unlocked matching passwords from the secret service.
///
/// The `attributes` should be a set of key and value string pairs.
///
/// All unlocked items that match the attributes will be deleted.
///
/// This method will return immediately and complete asynchronously.
extern fn secret_password_clearv(p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
pub const passwordClearv = secret_password_clearv;

/// Remove unlocked matching passwords from the secret service.
///
/// The `attributes` should be a set of key and value string pairs.
///
/// All unlocked items that match the attributes will be deleted.
///
/// This method may block indefinitely and should not be used in user interface
/// threads.
extern fn secret_password_clearv_sync(p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
pub const passwordClearvSync = secret_password_clearv_sync;

/// Clear the memory used by a password, and then free it.
///
/// This function must be used to free nonpageable memory returned by
/// `passwordLookupNonpageableFinish`,
/// `passwordLookupNonpageableSync` or
/// `passwordLookupvNonpageableSync`.
extern fn secret_password_free(p_password: ?[*:0]u8) void;
pub const passwordFree = secret_password_free;

/// Lookup a password in the secret service.
///
/// The variable argument list should contain pairs of a) The attribute name as
/// a null-terminated string, followed by b) attribute value, either a character
/// string, an int number, or a gboolean value, as defined in the password
/// `schema`. The list of attributes should be terminated with a `NULL`.
///
/// If no secret is found then `NULL` is returned.
///
/// This method will return immediately and complete asynchronously.
extern fn secret_password_lookup(p_schema: *const secret.Schema, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque, ...) void;
pub const passwordLookup = secret_password_lookup;

/// Finish an asynchronous operation to lookup a password in the secret service.
extern fn secret_password_lookup_binary_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*secret.Value;
pub const passwordLookupBinaryFinish = secret_password_lookup_binary_finish;

/// Lookup a password in the secret service.
///
/// This is similar to `passwordLookupSync`, but returns a
/// `Value` instead of a null-terminated password.
///
/// This method may block indefinitely and should not be used in user interface
/// threads.
extern fn secret_password_lookup_binary_sync(p_schema: *const secret.Schema, p_cancellable: ?*gio.Cancellable, p_error: **glib.Error, ...) *secret.Value;
pub const passwordLookupBinarySync = secret_password_lookup_binary_sync;

/// Finish an asynchronous operation to lookup a password in the secret service.
extern fn secret_password_lookup_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?[*:0]u8;
pub const passwordLookupFinish = secret_password_lookup_finish;

/// Finish an asynchronous operation to lookup a password in the secret service.
extern fn secret_password_lookup_nonpageable_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?[*:0]u8;
pub const passwordLookupNonpageableFinish = secret_password_lookup_nonpageable_finish;

/// Lookup a password in the secret service.
///
/// The variable argument list should contain pairs of a) The attribute name as
/// a null-terminated string, followed by b) attribute value, either a character
/// string, an int number, or a gboolean value, as defined in the password
/// `schema`. The list of attributes should be terminated with a `NULL`.
///
/// If no secret is found then `NULL` is returned.
///
/// This method may block indefinitely and should not be used in user interface
/// threads.
extern fn secret_password_lookup_nonpageable_sync(p_schema: *const secret.Schema, p_cancellable: ?*gio.Cancellable, p_error: **glib.Error, ...) [*:0]u8;
pub const passwordLookupNonpageableSync = secret_password_lookup_nonpageable_sync;

/// Lookup a password in the secret service.
///
/// The variable argument list should contain pairs of a) The attribute name as
/// a null-terminated string, followed by b) attribute value, either a character
/// string, an int number, or a gboolean value, as defined in the password
/// `schema`. The list of attributes should be terminated with a `NULL`.
///
/// If no secret is found then `NULL` is returned.
///
/// This method may block indefinitely and should not be used in user interface
/// threads.
extern fn secret_password_lookup_sync(p_schema: *const secret.Schema, p_cancellable: ?*gio.Cancellable, p_error: **glib.Error, ...) [*:0]u8;
pub const passwordLookupSync = secret_password_lookup_sync;

/// Lookup a password in the secret service.
///
/// The `attributes` should be a set of key and value string pairs.
///
/// If no secret is found then `NULL` is returned.
///
/// This method will return immediately and complete asynchronously.
extern fn secret_password_lookupv(p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
pub const passwordLookupv = secret_password_lookupv;

/// Lookup a password in the secret service.
///
/// This is similar to `passwordLookupvSync`, but returns a
/// `Value` instead of a null-terminated password.
///
/// This method may block indefinitely and should not be used in user interface
/// threads.
extern fn secret_password_lookupv_binary_sync(p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*secret.Value;
pub const passwordLookupvBinarySync = secret_password_lookupv_binary_sync;

/// Lookup a password in the secret service.
///
/// The `attributes` should be a set of key and value string pairs.
///
/// If no secret is found then `NULL` is returned.
///
/// This method may block indefinitely and should not be used in user interface
/// threads.
extern fn secret_password_lookupv_nonpageable_sync(p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?[*:0]u8;
pub const passwordLookupvNonpageableSync = secret_password_lookupv_nonpageable_sync;

/// Lookup a password in the secret service.
///
/// The `attributes` should be a set of key and value string pairs.
///
/// If no secret is found then `NULL` is returned.
///
/// This method may block indefinitely and should not be used in user interface
/// threads.
extern fn secret_password_lookupv_sync(p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?[*:0]u8;
pub const passwordLookupvSync = secret_password_lookupv_sync;

/// Search for items in the secret service.
///
/// The variable argument list should contain pairs of a) The attribute name as
/// a null-terminated string, followed by b) attribute value, either a character
/// string, an int number, or a gboolean value, as defined in the password
/// `schema`. The list of attributes should be terminated with a `NULL`.
///
/// This method will return immediately and complete asynchronously.
extern fn secret_password_search(p_schema: *const secret.Schema, p_flags: secret.SearchFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque, ...) void;
pub const passwordSearch = secret_password_search;

/// Finish an asynchronous operation to search for items in the secret service.
extern fn secret_password_search_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.List;
pub const passwordSearchFinish = secret_password_search_finish;

/// Search for items in the secret service.
///
/// The variable argument list should contain pairs of a) The attribute name as
/// a null-terminated string, followed by b) attribute value, either a character
/// string, an int number, or a gboolean value, as defined in the password
/// `schema`. The list of attributes should be terminated with a `NULL`.
///
/// If no secret is found then `NULL` is returned.
///
/// This method may block indefinitely and should not be used in user interface
/// threads.
extern fn secret_password_search_sync(p_schema: *const secret.Schema, p_flags: secret.SearchFlags, p_cancellable: ?*gio.Cancellable, p_error: **glib.Error, ...) *glib.List;
pub const passwordSearchSync = secret_password_search_sync;

/// Search for items in the secret service.
///
/// The `attributes` should be a set of key and value string pairs.
///
/// This method will return immediately and complete asynchronously.
extern fn secret_password_searchv(p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_flags: secret.SearchFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
pub const passwordSearchv = secret_password_searchv;

/// Search for items in the secret service.
///
/// The `attributes` should be a set of key and value string pairs.
///
/// If no secret is found then `NULL` is returned.
///
/// This method may block indefinitely and should not be used in user interface
/// threads.
extern fn secret_password_searchv_sync(p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_flags: secret.SearchFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*glib.List;
pub const passwordSearchvSync = secret_password_searchv_sync;

/// Store a password in the secret service.
///
/// The variable argument list should contain pairs of a) The attribute name as
/// a null-terminated string, followed by b) attribute value, either a character
/// string, an int number, or a gboolean value, as defined in the `schema`.
/// The list of attributes should be terminated with a `NULL`.
///
/// If the attributes match a secret item already stored in the collection, then
/// the item will be updated with these new values.
///
/// If `collection` is `NULL`, then the default collection will be
/// used. Use `COLLECTION_SESSION` to store the password in the session
/// collection, which doesn't get stored across login sessions.
///
/// This method will return immediately and complete asynchronously.
extern fn secret_password_store(p_schema: *const secret.Schema, p_collection: ?[*:0]const u8, p_label: [*:0]const u8, p_password: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque, ...) void;
pub const passwordStore = secret_password_store;

/// Store a password in the secret service.
///
/// This is similar to `passwordStore`, but takes a
/// `Value` as the argument instead of a null-terminated password.
///
/// This method will return immediately and complete asynchronously.
extern fn secret_password_store_binary(p_schema: *const secret.Schema, p_collection: ?[*:0]const u8, p_label: [*:0]const u8, p_value: *secret.Value, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque, ...) void;
pub const passwordStoreBinary = secret_password_store_binary;

/// Store a password in the secret service.
///
/// This is similar to `passwordStoreSync`, but takes a
/// `Value` as the argument instead of a null terminated password.
///
/// This method may block indefinitely and should not be used in user interface
/// threads.
extern fn secret_password_store_binary_sync(p_schema: *const secret.Schema, p_collection: ?[*:0]const u8, p_label: [*:0]const u8, p_value: *secret.Value, p_cancellable: ?*gio.Cancellable, p_error: **glib.Error, ...) c_int;
pub const passwordStoreBinarySync = secret_password_store_binary_sync;

/// Finish asynchronous operation to store a password in the secret service.
extern fn secret_password_store_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
pub const passwordStoreFinish = secret_password_store_finish;

/// Store a password in the secret service.
///
/// The variable argument list should contain pairs of a) The attribute name as
/// a null-terminated string, followed by b) attribute value, either a character
/// string, an int number, or a gboolean value, as defined in the `schema`.
/// The list of attributes should be terminated with a `NULL`.
///
/// If the attributes match a secret item already stored in the collection, then
/// the item will be updated with these new values.
///
/// If `collection` is `NULL`, then the default collection will be
/// used. Use `COLLECTION_SESSION` to store the password in the session
/// collection, which doesn't get stored across login sessions.
///
/// This method may block indefinitely and should not be used in user interface
/// threads.
extern fn secret_password_store_sync(p_schema: *const secret.Schema, p_collection: ?[*:0]const u8, p_label: [*:0]const u8, p_password: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: **glib.Error, ...) c_int;
pub const passwordStoreSync = secret_password_store_sync;

/// Store a password in the secret service.
///
/// The `attributes` should be a set of key and value string pairs.
///
/// If the attributes match a secret item already stored in the collection, then
/// the item will be updated with these new values.
///
/// If `collection` is `NULL`, then the default collection will be
/// used. Use `COLLECTION_SESSION` to store the password in the session
/// collection, which doesn't get stored across login sessions.
///
/// This method will return immediately and complete asynchronously.
extern fn secret_password_storev(p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_collection: ?[*:0]const u8, p_label: [*:0]const u8, p_password: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
pub const passwordStorev = secret_password_storev;

/// Store a password in the secret service.
///
/// This is similar to `passwordStorev`, but takes a
/// `Value` as the argument instead of a null-terminated password.
///
/// This method will return immediately and complete asynchronously.
extern fn secret_password_storev_binary(p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_collection: ?[*:0]const u8, p_label: [*:0]const u8, p_value: *secret.Value, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
pub const passwordStorevBinary = secret_password_storev_binary;

/// Store a password in the secret service.
///
/// This is similar to `passwordStorevSync`, but takes a `Value` as
/// the argument instead of a null-terminated passwords.
///
/// This method may block indefinitely and should not be used in user interface
/// threads.
extern fn secret_password_storev_binary_sync(p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_collection: ?[*:0]const u8, p_label: [*:0]const u8, p_value: *secret.Value, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
pub const passwordStorevBinarySync = secret_password_storev_binary_sync;

/// Store a password in the secret service.
///
/// The `attributes` should be a set of key and value string pairs.
///
/// If the attributes match a secret item already stored in the collection, then
/// the item will be updated with these new values.
///
/// If `collection` is `NULL`, then the default collection will be
/// used. Use `COLLECTION_SESSION` to store the password in the session
/// collection, which doesn't get stored across login sessions.
///
/// This method may block indefinitely and should not be used in user interface
/// threads.
extern fn secret_password_storev_sync(p_schema: ?*const secret.Schema, p_attributes: *glib.HashTable, p_collection: ?[*:0]const u8, p_label: [*:0]const u8, p_password: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
pub const passwordStorevSync = secret_password_storev_sync;

/// Clear the memory used by a password.
extern fn secret_password_wipe(p_password: ?[*:0]u8) void;
pub const passwordWipe = secret_password_wipe;

/// Extension point for the secret backend.
pub const BACKEND_EXTENSION_POINT_NAME = "secret-backend";
/// An alias to the default collection.
///
/// This can be passed to `passwordStore` `Collection.forAlias`.
pub const COLLECTION_DEFAULT = "default";
/// An alias to the session collection, which will be cleared when the user ends
/// the session.
///
/// This can be passed to `passwordStore`, `Collection.forAlias` or
/// similar functions.
pub const COLLECTION_SESSION = "session";
/// The major version of libsecret.
pub const MAJOR_VERSION = 0;
/// The micro version of libsecret.
pub const MICRO_VERSION = 4;
/// The minor version of libsecret.
pub const MINOR_VERSION = 21;
