pub const ext = @import("ext.zig");
const gio = @This();

const std = @import("std");
const compat = @import("compat");
const gobject = @import("gobject2");
const glib = @import("glib2");
const gmodule = @import("gmodule2");
/// `GAppInfoMonitor` monitors application information for changes.
///
/// `GAppInfoMonitor` is a very simple object used for monitoring the app
/// info database for changes (newly installed or removed applications).
///
/// Call `gio.AppInfoMonitor.get` to get a `GAppInfoMonitor` and connect
/// to the `gio.AppInfoMonitor.signals.changed` signal. The signal will be emitted once when
/// the app info database changes, and will not be emitted again until after the
/// next call to `gio.AppInfo.getAll` or another `g_app_info_*()` function.
/// This is because monitoring the app info database for changes is expensive.
///
/// The following functions will re-arm the `gio.AppInfoMonitor.signals.changed`
/// signal so it can be emitted again:
///
///  - `gio.AppInfo.getAll`
///  - `gio.AppInfo.getAllForType`
///  - `gio.AppInfo.getDefaultForType`
///  - `gio.AppInfo.getFallbackForType`
///  - `gio.AppInfo.getRecommendedForType`
///  - [``gio.DesktopAppInfo.getImplementations``](../gio-unix/type_func.DesktopAppInfo.get_implementation.html)
///  - [``gio.DesktopAppInfo.new``](../gio-unix/ctor.DesktopAppInfo.new.html)
///  - [``gio.DesktopAppInfo.newFromFilename``](../gio-unix/ctor.DesktopAppInfo.new_from_filename.html)
///  - [``gio.DesktopAppInfo.newFromKeyfile``](../gio-unix/ctor.DesktopAppInfo.new_from_keyfile.html)
///  - [``gio.DesktopAppInfo.search``](../gio-unix/type_func.DesktopAppInfo.search.html)
///
/// The latter functions are available if using
/// [`GDesktopAppInfo`](../gio-unix/class.DesktopAppInfo.html) from
/// `gio-unix-2.0.pc` (GIR namespace `GioUnix-2.0`).
///
/// In the usual case, applications should try to make note of the change
/// (doing things like invalidating caches) but not act on it. In
/// particular, applications should avoid making calls to `GAppInfo` APIs
/// in response to the change signal, deferring these until the time that
/// the updated data is actually required. The exception to this case is when
/// application information is actually being displayed on the screen
/// (for example, during a search or when the list of all applications is shown).
/// The reason for this is that changes to the list of installed applications
/// often come in groups (like during system updates) and rescanning the list
/// on every change is pointless and expensive.
pub const AppInfoMonitor = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = opaque {
        pub const Instance = AppInfoMonitor;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {
        /// Signal emitted when the app info database changes, when applications are
        /// installed or removed.
        pub const changed = struct {
            pub const name = "changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(AppInfoMonitor, p_instance))),
                    gobject.signalLookup("changed", AppInfoMonitor.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Gets the `gio.AppInfoMonitor` for the current thread-default main
    /// context.
    ///
    /// The `gio.AppInfoMonitor` will emit a “changed” signal in the
    /// thread-default main context whenever the list of installed
    /// applications (as reported by `gio.appInfoGetAll`) may have changed.
    ///
    /// The `gio.AppInfoMonitor.signals.changed` signal will only be emitted once until
    /// `gio.appInfoGetAll` (or another `g_app_info_*()` function) is called. Doing
    /// so will re-arm the signal ready to notify about the next change.
    ///
    /// You must only call `gobject.Object.unref` on the return value from under
    /// the same main context as you created it.
    extern fn g_app_info_monitor_get() *gio.AppInfoMonitor;
    pub const get = g_app_info_monitor_get;

    extern fn g_app_info_monitor_get_type() usize;
    pub const getGObjectType = g_app_info_monitor_get_type;

    extern fn g_object_ref(p_self: *gio.AppInfoMonitor) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.AppInfoMonitor) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *AppInfoMonitor, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Integrating the launch with the launching application. This is used to
/// handle for instance startup notification and launching the new application
/// on the same screen as the launching window.
pub const AppLaunchContext = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.AppLaunchContextClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.AppLaunchContextPrivate,

    pub const virtual_methods = struct {
        /// Gets the display string for the `context`. This is used to ensure new
        /// applications are started on the same display as the launching
        /// application, by setting the `DISPLAY` environment variable.
        pub const get_display = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_info: *gio.AppInfo, p_files: *glib.List) ?[*:0]u8 {
                return gobject.ext.as(AppLaunchContext.Class, p_class).f_get_display.?(gobject.ext.as(AppLaunchContext, p_context), p_info, p_files);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_info: *gio.AppInfo, p_files: *glib.List) callconv(.C) ?[*:0]u8) void {
                gobject.ext.as(AppLaunchContext.Class, p_class).f_get_display = @ptrCast(p_implementation);
            }
        };

        /// Initiates startup notification for the application and returns the
        /// `XDG_ACTIVATION_TOKEN` or `DESKTOP_STARTUP_ID` for the launched operation,
        /// if supported.
        ///
        /// The returned token may be referred to equivalently as an ‘activation token’
        /// (using Wayland terminology) or a ‘startup sequence ID’ (using X11 terminology).
        /// The two [are interoperable](https://gitlab.freedesktop.org/wayland/wayland-protocols/-/blob/main/staging/xdg-activation/x11-interoperation.rst).
        ///
        /// Activation tokens are defined in the [XDG Activation Protocol](https://wayland.app/protocols/xdg-activation-v1),
        /// and startup notification IDs are defined in the
        /// [freedesktop.org Startup Notification Protocol](http://standards.freedesktop.org/startup-notification-spec/startup-notification-latest.txt).
        ///
        /// Support for the XDG Activation Protocol was added in GLib 2.76.
        /// Since GLib 2.82 `info` and `files` can be `NULL`. If that’s not supported by the backend,
        /// the returned token will be `NULL`.
        pub const get_startup_notify_id = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_info: ?*gio.AppInfo, p_files: ?*glib.List) ?[*:0]u8 {
                return gobject.ext.as(AppLaunchContext.Class, p_class).f_get_startup_notify_id.?(gobject.ext.as(AppLaunchContext, p_context), p_info, p_files);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_info: ?*gio.AppInfo, p_files: ?*glib.List) callconv(.C) ?[*:0]u8) void {
                gobject.ext.as(AppLaunchContext.Class, p_class).f_get_startup_notify_id = @ptrCast(p_implementation);
            }
        };

        /// Called when an application has failed to launch, so that it can cancel
        /// the application startup notification started in
        /// `gio.AppLaunchContext.getStartupNotifyId`.
        pub const launch_failed = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_startup_notify_id: [*:0]const u8) void {
                return gobject.ext.as(AppLaunchContext.Class, p_class).f_launch_failed.?(gobject.ext.as(AppLaunchContext, p_context), p_startup_notify_id);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_startup_notify_id: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(AppLaunchContext.Class, p_class).f_launch_failed = @ptrCast(p_implementation);
            }
        };

        pub const launch_started = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_info: *gio.AppInfo, p_platform_data: *glib.Variant) void {
                return gobject.ext.as(AppLaunchContext.Class, p_class).f_launch_started.?(gobject.ext.as(AppLaunchContext, p_context), p_info, p_platform_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_info: *gio.AppInfo, p_platform_data: *glib.Variant) callconv(.C) void) void {
                gobject.ext.as(AppLaunchContext.Class, p_class).f_launch_started = @ptrCast(p_implementation);
            }
        };

        pub const launched = struct {
            pub fn call(p_class: anytype, p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_info: *gio.AppInfo, p_platform_data: *glib.Variant) void {
                return gobject.ext.as(AppLaunchContext.Class, p_class).f_launched.?(gobject.ext.as(AppLaunchContext, p_context), p_info, p_platform_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_context: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_info: *gio.AppInfo, p_platform_data: *glib.Variant) callconv(.C) void) void {
                gobject.ext.as(AppLaunchContext.Class, p_class).f_launched = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {
        /// The `gio.AppLaunchContext.signals.launch_failed` signal is emitted when a
        /// `gio.AppInfo` launch fails. The startup notification id is provided,
        /// so that the launcher can cancel the startup notification.
        ///
        /// Because a launch operation may involve spawning multiple instances of the
        /// target application, you should expect this signal to be emitted multiple
        /// times, one for each spawned instance.
        pub const launch_failed = struct {
            pub const name = "launch-failed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_startup_notify_id: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(AppLaunchContext, p_instance))),
                    gobject.signalLookup("launch-failed", AppLaunchContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The `gio.AppLaunchContext.signals.launch_started` signal is emitted when a
        /// `gio.AppInfo` is about to be launched. If non-null the
        /// `platform_data` is an GVariant dictionary mapping strings to variants
        /// (ie `a{sv}`), which contains additional, platform-specific data about this
        /// launch. On UNIX, at least the `startup-notification-id` keys will be
        /// present.
        ///
        /// The value of the `startup-notification-id` key (type `s`) is a startup
        /// notification ID corresponding to the format from the [startup-notification
        /// specification](https://specifications.freedesktop.org/startup-notification-spec/startup-notification-0.1.txt).
        /// It allows tracking the progress of the launchee through startup.
        ///
        /// It is guaranteed that this signal is followed by either a
        /// `gio.AppLaunchContext.signals.launched` or
        /// `gio.AppLaunchContext.signals.launch_failed` signal.
        ///
        /// Because a launch operation may involve spawning multiple instances of the
        /// target application, you should expect this signal to be emitted multiple
        /// times, one for each spawned instance.
        pub const launch_started = struct {
            pub const name = "launch-started";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_info: *gio.AppInfo, p_platform_data: ?*glib.Variant, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(AppLaunchContext, p_instance))),
                    gobject.signalLookup("launch-started", AppLaunchContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The `gio.AppLaunchContext.signals.launched` signal is emitted when a
        /// `gio.AppInfo` is successfully launched.
        ///
        /// Because a launch operation may involve spawning multiple instances of the
        /// target application, you should expect this signal to be emitted multiple
        /// times, one time for each spawned instance.
        ///
        /// The `platform_data` is an GVariant dictionary mapping
        /// strings to variants (ie `a{sv}`), which contains additional,
        /// platform-specific data about this launch. On UNIX, at least the
        /// `pid` and `startup-notification-id` keys will be present.
        ///
        /// Since 2.72 the `pid` may be 0 if the process id wasn’t known (for
        /// example if the process was launched via D-Bus). The `pid` may not be
        /// set at all in subsequent releases.
        ///
        /// On Windows, `pid` is guaranteed to be valid only for the duration of the
        /// `gio.AppLaunchContext.signals.launched` signal emission; after the signal
        /// is emitted, GLib will call `glib.spawnClosePid`. If you need to
        /// keep the `glib.Pid` after the signal has been emitted, then you can
        /// duplicate `pid` using ``DuplicateHandle``.
        pub const launched = struct {
            pub const name = "launched";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_info: *gio.AppInfo, p_platform_data: *glib.Variant, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(AppLaunchContext, p_instance))),
                    gobject.signalLookup("launched", AppLaunchContext.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new application launch context. This is not normally used,
    /// instead you instantiate a subclass of this, such as
    /// [`GdkAppLaunchContext`](https://docs.gtk.org/gdk4/class.AppLaunchContext.html).
    extern fn g_app_launch_context_new() *gio.AppLaunchContext;
    pub const new = g_app_launch_context_new;

    /// Gets the display string for the `context`. This is used to ensure new
    /// applications are started on the same display as the launching
    /// application, by setting the `DISPLAY` environment variable.
    extern fn g_app_launch_context_get_display(p_context: *AppLaunchContext, p_info: *gio.AppInfo, p_files: *glib.List) ?[*:0]u8;
    pub const getDisplay = g_app_launch_context_get_display;

    /// Gets the complete environment variable list to be passed to
    /// the child process when `context` is used to launch an application.
    /// This is a `NULL`-terminated array of strings, where each string has
    /// the form `KEY=VALUE`.
    extern fn g_app_launch_context_get_environment(p_context: *AppLaunchContext) [*][*:0]u8;
    pub const getEnvironment = g_app_launch_context_get_environment;

    /// Initiates startup notification for the application and returns the
    /// `XDG_ACTIVATION_TOKEN` or `DESKTOP_STARTUP_ID` for the launched operation,
    /// if supported.
    ///
    /// The returned token may be referred to equivalently as an ‘activation token’
    /// (using Wayland terminology) or a ‘startup sequence ID’ (using X11 terminology).
    /// The two [are interoperable](https://gitlab.freedesktop.org/wayland/wayland-protocols/-/blob/main/staging/xdg-activation/x11-interoperation.rst).
    ///
    /// Activation tokens are defined in the [XDG Activation Protocol](https://wayland.app/protocols/xdg-activation-v1),
    /// and startup notification IDs are defined in the
    /// [freedesktop.org Startup Notification Protocol](http://standards.freedesktop.org/startup-notification-spec/startup-notification-latest.txt).
    ///
    /// Support for the XDG Activation Protocol was added in GLib 2.76.
    /// Since GLib 2.82 `info` and `files` can be `NULL`. If that’s not supported by the backend,
    /// the returned token will be `NULL`.
    extern fn g_app_launch_context_get_startup_notify_id(p_context: *AppLaunchContext, p_info: ?*gio.AppInfo, p_files: ?*glib.List) ?[*:0]u8;
    pub const getStartupNotifyId = g_app_launch_context_get_startup_notify_id;

    /// Called when an application has failed to launch, so that it can cancel
    /// the application startup notification started in
    /// `gio.AppLaunchContext.getStartupNotifyId`.
    extern fn g_app_launch_context_launch_failed(p_context: *AppLaunchContext, p_startup_notify_id: [*:0]const u8) void;
    pub const launchFailed = g_app_launch_context_launch_failed;

    /// Arranges for `variable` to be set to `value` in the child’s environment when
    /// `context` is used to launch an application.
    extern fn g_app_launch_context_setenv(p_context: *AppLaunchContext, p_variable: [*:0]const u8, p_value: [*:0]const u8) void;
    pub const setenv = g_app_launch_context_setenv;

    /// Arranges for `variable` to be unset in the child’s environment when `context`
    /// is used to launch an application.
    extern fn g_app_launch_context_unsetenv(p_context: *AppLaunchContext, p_variable: [*:0]const u8) void;
    pub const unsetenv = g_app_launch_context_unsetenv;

    extern fn g_app_launch_context_get_type() usize;
    pub const getGObjectType = g_app_launch_context_get_type;

    extern fn g_object_ref(p_self: *gio.AppLaunchContext) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.AppLaunchContext) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *AppLaunchContext, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GApplication` is the core class for application support.
///
/// A `GApplication` is the foundation of an application. It wraps some
/// low-level platform-specific services and is intended to act as the
/// foundation for higher-level application classes such as
/// `GtkApplication` or `MxApplication`. In general, you should not use
/// this class outside of a higher level framework.
///
/// `GApplication` provides convenient life-cycle management by maintaining
/// a "use count" for the primary application instance. The use count can
/// be changed using `gio.Application.hold` and
/// `gio.Application.release`. If it drops to zero, the application
/// exits. Higher-level classes such as `GtkApplication` employ the use count
/// to ensure that the application stays alive as long as it has any opened
/// windows.
///
/// Another feature that `GApplication` (optionally) provides is process
/// uniqueness. Applications can make use of this functionality by
/// providing a unique application ID. If given, only one application
/// with this ID can be running at a time per session. The session
/// concept is platform-dependent, but corresponds roughly to a graphical
/// desktop login. When your application is launched again, its
/// arguments are passed through platform communication to the already
/// running program. The already running instance of the program is
/// called the "primary instance"; for non-unique applications this is
/// always the current instance. On Linux, the D-Bus session bus
/// is used for communication.
///
/// The use of `GApplication` differs from some other commonly-used
/// uniqueness libraries (such as libunique) in important ways. The
/// application is not expected to manually register itself and check
/// if it is the primary instance. Instead, the `main` function of a
/// `GApplication` should do very little more than instantiating the
/// application instance, possibly connecting signal handlers, then
/// calling `gio.Application.run`. All checks for uniqueness are done
/// internally. If the application is the primary instance then the
/// startup signal is emitted and the mainloop runs. If the application
/// is not the primary instance then a signal is sent to the primary
/// instance and `gio.Application.run` promptly returns. See the code
/// examples below.
///
/// If used, the expected form of an application identifier is the
/// same as that of a
/// [D-Bus well-known bus name](https://dbus.freedesktop.org/doc/dbus-specification.html`message`-protocol-names-bus).
/// Examples include: `com.example.MyApp`, `org.example.internal_apps.Calculator`,
/// `org._7_zip.Archiver`.
/// For details on valid application identifiers, see `gio.Application.idIsValid`.
///
/// On Linux, the application identifier is claimed as a well-known bus name
/// on the user's session bus. This means that the uniqueness of your
/// application is scoped to the current session. It also means that your
/// application may provide additional services (through registration of other
/// object paths) at that bus name. The registration of these object paths
/// should be done with the shared GDBus session bus. Note that due to the
/// internal architecture of GDBus, method calls can be dispatched at any time
/// (even if a main loop is not running). For this reason, you must ensure that
/// any object paths that you wish to register are registered before `gio.Application`
/// attempts to acquire the bus name of your application (which happens in
/// `gio.Application.register`). Unfortunately, this means that you cannot
/// use `gio.Application.properties.is_remote` to decide if you want to register
/// object paths.
///
/// `GApplication` also implements the `gio.ActionGroup` and `gio.ActionMap`
/// interfaces and lets you easily export actions by adding them with
/// `gio.ActionMap.addAction`. When invoking an action by calling
/// `gio.ActionGroup.activateAction` on the application, it is always
/// invoked in the primary instance. The actions are also exported on
/// the session bus, and GIO provides the `gio.DBusActionGroup` wrapper to
/// conveniently access them remotely. GIO provides a `gio.DBusMenuModel` wrapper
/// for remote access to exported `gio.MenuModel`s.
///
/// Note: Due to the fact that actions are exported on the session bus,
/// using `maybe` parameters is not supported, since D-Bus does not support
/// `maybe` types.
///
/// There is a number of different entry points into a `GApplication`:
///
/// - via 'Activate' (i.e. just starting the application)
///
/// - via 'Open' (i.e. opening some files)
///
/// - by handling a command-line
///
/// - via activating an action
///
/// The `gio.Application.signals.startup` signal lets you handle the application
/// initialization for all of these in a single place.
///
/// Regardless of which of these entry points is used to start the
/// application, `GApplication` passes some ‘platform data’ from the
/// launching instance to the primary instance, in the form of a
/// `glib.Variant` dictionary mapping strings to variants. To use platform
/// data, override the `gio.Application.virtual_methods.before_emit` or
/// `gio.Application.virtual_methods.after_emit` virtual functions
/// in your `GApplication` subclass. When dealing with
/// `gio.ApplicationCommandLine` objects, the platform data is
/// directly available via `gio.ApplicationCommandLine.getCwd`,
/// `gio.ApplicationCommandLine.getEnviron` and
/// `gio.ApplicationCommandLine.getPlatformData`.
///
/// As the name indicates, the platform data may vary depending on the
/// operating system, but it always includes the current directory (key
/// `cwd`), and optionally the environment (ie the set of environment
/// variables and their values) of the calling process (key `environ`).
/// The environment is only added to the platform data if the
/// `G_APPLICATION_SEND_ENVIRONMENT` flag is set. `GApplication` subclasses
/// can add their own platform data by overriding the
/// `gio.Application.virtual_methods.add_platform_data` virtual function. For instance,
/// `GtkApplication` adds startup notification data in this way.
///
/// To parse commandline arguments you may handle the
/// `gio.Application.signals.command_line` signal or override the
/// `gio.Application.virtual_methods.local_command_line` virtual function, to parse them in
/// either the primary instance or the local instance, respectively.
///
/// For an example of opening files with a `GApplication`, see
/// [gapplication-example-open.c](https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gapplication-example-open.c).
///
/// For an example of using actions with `GApplication`, see
/// [gapplication-example-actions.c](https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gapplication-example-actions.c).
///
/// For an example of using extra D-Bus hooks with `GApplication`, see
/// [gapplication-example-dbushooks.c](https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gapplication-example-dbushooks.c).
pub const Application = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{ gio.ActionGroup, gio.ActionMap };
    pub const Class = gio.ApplicationClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.ApplicationPrivate,

    pub const virtual_methods = struct {
        /// Activates the application.
        ///
        /// In essence, this results in the `gio.Application.signals.activate` signal being
        /// emitted in the primary instance.
        ///
        /// The application must be registered before calling this function.
        pub const activate = struct {
            pub fn call(p_class: anytype, p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Application.Class, p_class).f_activate.?(gobject.ext.as(Application, p_application));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Application.Class, p_class).f_activate = @ptrCast(p_implementation);
            }
        };

        /// invoked (locally) to add 'platform data' to be sent to
        ///     the primary instance when activating, opening or invoking actions
        pub const add_platform_data = struct {
            pub fn call(p_class: anytype, p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_builder: *glib.VariantBuilder) void {
                return gobject.ext.as(Application.Class, p_class).f_add_platform_data.?(gobject.ext.as(Application, p_application), p_builder);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_builder: *glib.VariantBuilder) callconv(.C) void) void {
                gobject.ext.as(Application.Class, p_class).f_add_platform_data = @ptrCast(p_implementation);
            }
        };

        /// invoked on the primary instance after 'activate', 'open',
        ///     'command-line' or any action invocation, gets the 'platform data' from
        ///     the calling instance
        pub const after_emit = struct {
            pub fn call(p_class: anytype, p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_platform_data: *glib.Variant) void {
                return gobject.ext.as(Application.Class, p_class).f_after_emit.?(gobject.ext.as(Application, p_application), p_platform_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_platform_data: *glib.Variant) callconv(.C) void) void {
                gobject.ext.as(Application.Class, p_class).f_after_emit = @ptrCast(p_implementation);
            }
        };

        /// invoked on the primary instance before 'activate', 'open',
        ///     'command-line' or any action invocation, gets the 'platform data' from
        ///     the calling instance
        pub const before_emit = struct {
            pub fn call(p_class: anytype, p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_platform_data: *glib.Variant) void {
                return gobject.ext.as(Application.Class, p_class).f_before_emit.?(gobject.ext.as(Application, p_application), p_platform_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_platform_data: *glib.Variant) callconv(.C) void) void {
                gobject.ext.as(Application.Class, p_class).f_before_emit = @ptrCast(p_implementation);
            }
        };

        /// invoked on the primary instance when a command-line is
        ///   not handled locally
        pub const command_line = struct {
            pub fn call(p_class: anytype, p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_command_line: *gio.ApplicationCommandLine) c_int {
                return gobject.ext.as(Application.Class, p_class).f_command_line.?(gobject.ext.as(Application, p_application), p_command_line);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_command_line: *gio.ApplicationCommandLine) callconv(.C) c_int) void {
                gobject.ext.as(Application.Class, p_class).f_command_line = @ptrCast(p_implementation);
            }
        };

        /// invoked locally during registration, if the application is
        ///     using its D-Bus backend. You can use this to export extra objects on the
        ///     bus, that need to exist before the application tries to own the bus name.
        ///     The function is passed the `gio.DBusConnection` to to session bus, and the
        ///     object path that `gio.Application` will use to export its D-Bus API.
        ///     If this function returns `TRUE`, registration will proceed; otherwise
        ///     registration will abort. Since: 2.34
        pub const dbus_register = struct {
            pub fn call(p_class: anytype, p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_connection: *gio.DBusConnection, p_object_path: [*:0]const u8, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(Application.Class, p_class).f_dbus_register.?(gobject.ext.as(Application, p_application), p_connection, p_object_path, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_connection: *gio.DBusConnection, p_object_path: [*:0]const u8, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(Application.Class, p_class).f_dbus_register = @ptrCast(p_implementation);
            }
        };

        /// invoked locally during unregistration, if the application
        ///     is using its D-Bus backend. Use this to undo anything done by
        ///     the `dbus_register` vfunc. Since: 2.34
        pub const dbus_unregister = struct {
            pub fn call(p_class: anytype, p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_connection: *gio.DBusConnection, p_object_path: [*:0]const u8) void {
                return gobject.ext.as(Application.Class, p_class).f_dbus_unregister.?(gobject.ext.as(Application, p_application), p_connection, p_object_path);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_connection: *gio.DBusConnection, p_object_path: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(Application.Class, p_class).f_dbus_unregister = @ptrCast(p_implementation);
            }
        };

        /// invoked locally after the parsing of the commandline
        ///  options has occurred. Since: 2.40
        pub const handle_local_options = struct {
            pub fn call(p_class: anytype, p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_options: *glib.VariantDict) c_int {
                return gobject.ext.as(Application.Class, p_class).f_handle_local_options.?(gobject.ext.as(Application, p_application), p_options);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_options: *glib.VariantDict) callconv(.C) c_int) void {
                gobject.ext.as(Application.Class, p_class).f_handle_local_options = @ptrCast(p_implementation);
            }
        };

        /// This virtual function is always invoked in the local instance. It
        /// gets passed a pointer to a `NULL`-terminated copy of `argv` and is
        /// expected to remove arguments that it handled (shifting up remaining
        /// arguments).
        ///
        /// The last argument to `local_command_line` is a pointer to the `status`
        /// variable which can used to set the exit status that is returned from
        /// `gio.Application.run`.
        ///
        /// See `gio.Application.run` for more details on `gio.Application` startup.
        pub const local_command_line = struct {
            pub fn call(p_class: anytype, p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_arguments: [*]*[*:0]u8, p_exit_status: *c_int) c_int {
                return gobject.ext.as(Application.Class, p_class).f_local_command_line.?(gobject.ext.as(Application, p_application), p_arguments, p_exit_status);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_arguments: [*]*[*:0]u8, p_exit_status: *c_int) callconv(.C) c_int) void {
                gobject.ext.as(Application.Class, p_class).f_local_command_line = @ptrCast(p_implementation);
            }
        };

        /// invoked when another instance is taking over the name. Since: 2.60
        pub const name_lost = struct {
            pub fn call(p_class: anytype, p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) c_int {
                return gobject.ext.as(Application.Class, p_class).f_name_lost.?(gobject.ext.as(Application, p_application));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) c_int) void {
                gobject.ext.as(Application.Class, p_class).f_name_lost = @ptrCast(p_implementation);
            }
        };

        /// Opens the given files.
        ///
        /// In essence, this results in the `gio.Application.signals.open` signal being emitted
        /// in the primary instance.
        ///
        /// `n_files` must be greater than zero.
        ///
        /// `hint` is simply passed through to the ::open signal.  It is
        /// intended to be used by applications that have multiple modes for
        /// opening files (eg: "view" vs "edit", etc).  Unless you have a need
        /// for this functionality, you should use "".
        ///
        /// The application must be registered before calling this function
        /// and it must have the `G_APPLICATION_HANDLES_OPEN` flag set.
        pub const open = struct {
            pub fn call(p_class: anytype, p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_files: [*]*gio.File, p_n_files: c_int, p_hint: [*:0]const u8) void {
                return gobject.ext.as(Application.Class, p_class).f_open.?(gobject.ext.as(Application, p_application), p_files, p_n_files, p_hint);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_files: [*]*gio.File, p_n_files: c_int, p_hint: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(Application.Class, p_class).f_open = @ptrCast(p_implementation);
            }
        };

        /// Used to be invoked on the primary instance when the use
        ///     count of the application drops to zero (and after any inactivity
        ///     timeout, if requested). Not used anymore since 2.32
        pub const quit_mainloop = struct {
            pub fn call(p_class: anytype, p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Application.Class, p_class).f_quit_mainloop.?(gobject.ext.as(Application, p_application));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Application.Class, p_class).f_quit_mainloop = @ptrCast(p_implementation);
            }
        };

        /// Used to be invoked on the primary instance from
        ///     `gio.Application.run` if the use-count is non-zero. Since 2.32,
        ///     GApplication is iterating the main context directly and is not
        ///     using `run_mainloop` anymore
        pub const run_mainloop = struct {
            pub fn call(p_class: anytype, p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Application.Class, p_class).f_run_mainloop.?(gobject.ext.as(Application, p_application));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Application.Class, p_class).f_run_mainloop = @ptrCast(p_implementation);
            }
        };

        /// invoked only on the registered primary instance immediately
        ///      after the main loop terminates
        pub const shutdown = struct {
            pub fn call(p_class: anytype, p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Application.Class, p_class).f_shutdown.?(gobject.ext.as(Application, p_application));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Application.Class, p_class).f_shutdown = @ptrCast(p_implementation);
            }
        };

        /// invoked on the primary instance immediately after registration
        pub const startup = struct {
            pub fn call(p_class: anytype, p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Application.Class, p_class).f_startup.?(gobject.ext.as(Application, p_application));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_application: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Application.Class, p_class).f_startup = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The group of actions that the application exports.
        pub const action_group = struct {
            pub const name = "action-group";

            pub const Type = ?*gio.ActionGroup;
        };

        /// The unique identifier for the application.
        pub const application_id = struct {
            pub const name = "application-id";

            pub const Type = ?[*:0]u8;
        };

        /// Flags specifying the behaviour of the application.
        pub const flags = struct {
            pub const name = "flags";

            pub const Type = gio.ApplicationFlags;
        };

        /// Time (in milliseconds) to stay alive after becoming idle.
        pub const inactivity_timeout = struct {
            pub const name = "inactivity-timeout";

            pub const Type = c_uint;
        };

        /// Whether the application is currently marked as busy through
        /// `gio.Application.markBusy` or `gio.Application.bindBusyProperty`.
        pub const is_busy = struct {
            pub const name = "is-busy";

            pub const Type = c_int;
        };

        /// Whether `gio.Application.register` has been called.
        pub const is_registered = struct {
            pub const name = "is-registered";

            pub const Type = c_int;
        };

        /// Whether this application instance is remote.
        pub const is_remote = struct {
            pub const name = "is-remote";

            pub const Type = c_int;
        };

        /// The base resource path for the application.
        pub const resource_base_path = struct {
            pub const name = "resource-base-path";

            pub const Type = ?[*:0]u8;
        };

        /// The human-readable version number of the application.
        pub const version = struct {
            pub const name = "version";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {
        /// The ::activate signal is emitted on the primary instance when an
        /// activation occurs. See `gio.Application.activate`.
        pub const activate = struct {
            pub const name = "activate";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Application, p_instance))),
                    gobject.signalLookup("activate", Application.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::command-line signal is emitted on the primary instance when
        /// a commandline is not handled locally. See `gio.Application.run` and
        /// the `gio.ApplicationCommandLine` documentation for more information.
        pub const command_line = struct {
            pub const name = "command-line";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_command_line: *gio.ApplicationCommandLine, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Application, p_instance))),
                    gobject.signalLookup("command-line", Application.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::handle-local-options signal is emitted on the local instance
        /// after the parsing of the commandline options has occurred.
        ///
        /// You can add options to be recognised during commandline option
        /// parsing using `gio.Application.addMainOptionEntries` and
        /// `gio.Application.addOptionGroup`.
        ///
        /// Signal handlers can inspect `options` (along with values pointed to
        /// from the `arg_data` of an installed `GOptionEntrys`) in order to
        /// decide to perform certain actions, including direct local handling
        /// (which may be useful for options like --version).
        ///
        /// In the event that the application is marked
        /// `G_APPLICATION_HANDLES_COMMAND_LINE` the "normal processing" will
        /// send the `options` dictionary to the primary instance where it can be
        /// read with `gio.ApplicationCommandLine.getOptionsDict`.  The signal
        /// handler can modify the dictionary before returning, and the
        /// modified dictionary will be sent.
        ///
        /// In the event that `G_APPLICATION_HANDLES_COMMAND_LINE` is not set,
        /// "normal processing" will treat the remaining uncollected command
        /// line arguments as filenames or URIs.  If there are no arguments,
        /// the application is activated by `gio.Application.activate`.  One or
        /// more arguments results in a call to `gio.Application.open`.
        ///
        /// If you want to handle the local commandline arguments for yourself
        /// by converting them to calls to `gio.Application.open` or
        /// `gio.ActionGroup.activateAction` then you must be sure to register
        /// the application first.  You should probably not call
        /// `gio.Application.activate` for yourself, however: just return -1 and
        /// allow the default handler to do it for you.  This will ensure that
        /// the `--gapplication-service` switch works properly (i.e. no activation
        /// in that case).
        ///
        /// Note that this signal is emitted from the default implementation of
        /// `local_command_line`.  If you override that function and don't
        /// chain up then this signal will never be emitted.
        ///
        /// You can override `local_command_line` if you need more powerful
        /// capabilities than what is provided here, but this should not
        /// normally be required.
        pub const handle_local_options = struct {
            pub const name = "handle-local-options";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_options: *glib.VariantDict, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Application, p_instance))),
                    gobject.signalLookup("handle-local-options", Application.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::name-lost signal is emitted only on the registered primary instance
        /// when a new instance has taken over. This can only happen if the application
        /// is using the `G_APPLICATION_ALLOW_REPLACEMENT` flag.
        ///
        /// The default handler for this signal calls `gio.Application.quit`.
        pub const name_lost = struct {
            pub const name = "name-lost";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Application, p_instance))),
                    gobject.signalLookup("name-lost", Application.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::open signal is emitted on the primary instance when there are
        /// files to open. See `gio.Application.open` for more information.
        pub const open = struct {
            pub const name = "open";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_files: [*]*gio.File, p_n_files: c_int, p_hint: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Application, p_instance))),
                    gobject.signalLookup("open", Application.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::shutdown signal is emitted only on the registered primary instance
        /// immediately after the main loop terminates.
        pub const shutdown = struct {
            pub const name = "shutdown";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Application, p_instance))),
                    gobject.signalLookup("shutdown", Application.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The ::startup signal is emitted on the primary instance immediately
        /// after registration. See `gio.Application.register`.
        pub const startup = struct {
            pub const name = "startup";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Application, p_instance))),
                    gobject.signalLookup("startup", Application.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Returns the default `gio.Application` instance for this process.
    ///
    /// Normally there is only one `gio.Application` per process and it becomes
    /// the default when it is created.  You can exercise more control over
    /// this by using `gio.Application.setDefault`.
    ///
    /// If there is no default application then `NULL` is returned.
    extern fn g_application_get_default() ?*gio.Application;
    pub const getDefault = g_application_get_default;

    /// Checks if `application_id` is a valid application identifier.
    ///
    /// A valid ID is required for calls to `gio.Application.new` and
    /// `gio.Application.setApplicationId`.
    ///
    /// Application identifiers follow the same format as
    /// [D-Bus well-known bus names](https://dbus.freedesktop.org/doc/dbus-specification.html`message`-protocol-names-bus).
    /// For convenience, the restrictions on application identifiers are
    /// reproduced here:
    ///
    /// - Application identifiers are composed of 1 or more elements separated by a
    ///   period (`.`) character. All elements must contain at least one character.
    ///
    /// - Each element must only contain the ASCII characters `[A-Z][a-z][0-9]_-`,
    ///   with `-` discouraged in new application identifiers. Each element must not
    ///   begin with a digit.
    ///
    /// - Application identifiers must contain at least one `.` (period) character
    ///   (and thus at least two elements).
    ///
    /// - Application identifiers must not begin with a `.` (period) character.
    ///
    /// - Application identifiers must not exceed 255 characters.
    ///
    /// Note that the hyphen (`-`) character is allowed in application identifiers,
    /// but is problematic or not allowed in various specifications and APIs that
    /// refer to D-Bus, such as
    /// [Flatpak application IDs](http://docs.flatpak.org/en/latest/introduction.html`identifiers`),
    /// the
    /// [`DBusActivatable` interface in the Desktop Entry Specification](https://specifications.freedesktop.org/desktop-entry-spec/desktop-entry-spec-latest.html`dbus`),
    /// and the convention that an application's "main" interface and object path
    /// resemble its application identifier and bus name. To avoid situations that
    /// require special-case handling, it is recommended that new application
    /// identifiers consistently replace hyphens with underscores.
    ///
    /// Like D-Bus interface names, application identifiers should start with the
    /// reversed DNS domain name of the author of the interface (in lower-case), and
    /// it is conventional for the rest of the application identifier to consist of
    /// words run together, with initial capital letters.
    ///
    /// As with D-Bus interface names, if the author's DNS domain name contains
    /// hyphen/minus characters they should be replaced by underscores, and if it
    /// contains leading digits they should be escaped by prepending an underscore.
    /// For example, if the owner of 7-zip.org used an application identifier for an
    /// archiving application, it might be named `org._7_zip.Archiver`.
    extern fn g_application_id_is_valid(p_application_id: [*:0]const u8) c_int;
    pub const idIsValid = g_application_id_is_valid;

    /// Creates a new `gio.Application` instance.
    ///
    /// If non-`NULL`, the application id must be valid.  See
    /// `gio.Application.idIsValid`.
    ///
    /// If no application ID is given then some features of `gio.Application`
    /// (most notably application uniqueness) will be disabled.
    extern fn g_application_new(p_application_id: ?[*:0]const u8, p_flags: gio.ApplicationFlags) *gio.Application;
    pub const new = g_application_new;

    /// Activates the application.
    ///
    /// In essence, this results in the `gio.Application.signals.activate` signal being
    /// emitted in the primary instance.
    ///
    /// The application must be registered before calling this function.
    extern fn g_application_activate(p_application: *Application) void;
    pub const activate = g_application_activate;

    /// Add an option to be handled by `application`.
    ///
    /// Calling this function is the equivalent of calling
    /// `gio.Application.addMainOptionEntries` with a single `glib.OptionEntry`
    /// that has its arg_data member set to `NULL`.
    ///
    /// The parsed arguments will be packed into a `glib.VariantDict` which
    /// is passed to `gio.Application.signals.handle`-local-options. If
    /// `G_APPLICATION_HANDLES_COMMAND_LINE` is set, then it will also
    /// be sent to the primary instance. See
    /// `gio.Application.addMainOptionEntries` for more details.
    ///
    /// See `glib.OptionEntry` for more documentation of the arguments.
    extern fn g_application_add_main_option(p_application: *Application, p_long_name: [*:0]const u8, p_short_name: u8, p_flags: glib.OptionFlags, p_arg: glib.OptionArg, p_description: [*:0]const u8, p_arg_description: ?[*:0]const u8) void;
    pub const addMainOption = g_application_add_main_option;

    /// Adds main option entries to be handled by `application`.
    ///
    /// This function is comparable to `glib.OptionContext.addMainEntries`.
    ///
    /// After the commandline arguments are parsed, the
    /// `gio.Application.signals.handle`-local-options signal will be emitted.  At this
    /// point, the application can inspect the values pointed to by `arg_data`
    /// in the given `GOptionEntrys`.
    ///
    /// Unlike `glib.OptionContext`, `gio.Application` supports giving a `NULL`
    /// `arg_data` for a non-callback `glib.OptionEntry`.  This results in the
    /// argument in question being packed into a `glib.VariantDict` which is also
    /// passed to `gio.Application.signals.handle`-local-options, where it can be
    /// inspected and modified.  If `G_APPLICATION_HANDLES_COMMAND_LINE` is
    /// set, then the resulting dictionary is sent to the primary instance,
    /// where `gio.ApplicationCommandLine.getOptionsDict` will return it.
    /// As it has been passed outside the process at this point, the types of all
    /// values in the options dict must be checked before being used.
    /// This "packing" is done according to the type of the argument --
    /// booleans for normal flags, strings for strings, bytestrings for
    /// filenames, etc.  The packing only occurs if the flag is given (ie: we
    /// do not pack a "false" `glib.Variant` in the case that a flag is missing).
    ///
    /// In general, it is recommended that all commandline arguments are
    /// parsed locally.  The options dictionary should then be used to
    /// transmit the result of the parsing to the primary instance, where
    /// `glib.VariantDict.lookup` can be used.  For local options, it is
    /// possible to either use `arg_data` in the usual way, or to consult (and
    /// potentially remove) the option from the options dictionary.
    ///
    /// This function is new in GLib 2.40.  Before then, the only real choice
    /// was to send all of the commandline arguments (options and all) to the
    /// primary instance for handling.  `gio.Application` ignored them completely
    /// on the local side.  Calling this function "opts in" to the new
    /// behaviour, and in particular, means that unrecognized options will be
    /// treated as errors.  Unrecognized options have never been ignored when
    /// `G_APPLICATION_HANDLES_COMMAND_LINE` is unset.
    ///
    /// If `gio.Application.signals.handle`-local-options needs to see the list of
    /// filenames, then the use of `G_OPTION_REMAINING` is recommended.  If
    /// `arg_data` is `NULL` then `G_OPTION_REMAINING` can be used as a key into
    /// the options dictionary.  If you do use `G_OPTION_REMAINING` then you
    /// need to handle these arguments for yourself because once they are
    /// consumed, they will no longer be visible to the default handling
    /// (which treats them as filenames to be opened).
    ///
    /// It is important to use the proper GVariant format when retrieving
    /// the options with `glib.VariantDict.lookup`:
    /// - for `G_OPTION_ARG_NONE`, use `b`
    /// - for `G_OPTION_ARG_STRING`, use `&s`
    /// - for `G_OPTION_ARG_INT`, use `i`
    /// - for `G_OPTION_ARG_INT64`, use `x`
    /// - for `G_OPTION_ARG_DOUBLE`, use `d`
    /// - for `G_OPTION_ARG_FILENAME`, use `^&ay`
    /// - for `G_OPTION_ARG_STRING_ARRAY`, use `^a&s`
    /// - for `G_OPTION_ARG_FILENAME_ARRAY`, use `^a&ay`
    extern fn g_application_add_main_option_entries(p_application: *Application, p_entries: [*]const glib.OptionEntry) void;
    pub const addMainOptionEntries = g_application_add_main_option_entries;

    /// Adds a `glib.OptionGroup` to the commandline handling of `application`.
    ///
    /// This function is comparable to `glib.OptionContext.addGroup`.
    ///
    /// Unlike `gio.Application.addMainOptionEntries`, this function does
    /// not deal with `NULL` `arg_data` and never transmits options to the
    /// primary instance.
    ///
    /// The reason for that is because, by the time the options arrive at the
    /// primary instance, it is typically too late to do anything with them.
    /// Taking the GTK option group as an example: GTK will already have been
    /// initialised by the time the `gio.Application.signals.command`-line handler runs.
    /// In the case that this is not the first-running instance of the
    /// application, the existing instance may already have been running for
    /// a very long time.
    ///
    /// This means that the options from `glib.OptionGroup` are only really usable
    /// in the case that the instance of the application being run is the
    /// first instance.  Passing options like `--display=` or `--gdk-debug=`
    /// on future runs will have no effect on the existing primary instance.
    ///
    /// Calling this function will cause the options in the supplied option
    /// group to be parsed, but it does not cause you to be "opted in" to the
    /// new functionality whereby unrecognized options are rejected even if
    /// `G_APPLICATION_HANDLES_COMMAND_LINE` was given.
    extern fn g_application_add_option_group(p_application: *Application, p_group: *glib.OptionGroup) void;
    pub const addOptionGroup = g_application_add_option_group;

    /// Marks `application` as busy (see `gio.Application.markBusy`) while
    /// `property` on `object` is `TRUE`.
    ///
    /// The binding holds a reference to `application` while it is active, but
    /// not to `object`. Instead, the binding is destroyed when `object` is
    /// finalized.
    extern fn g_application_bind_busy_property(p_application: *Application, p_object: *gobject.Object, p_property: [*:0]const u8) void;
    pub const bindBusyProperty = g_application_bind_busy_property;

    /// Gets the unique identifier for `application`.
    extern fn g_application_get_application_id(p_application: *Application) ?[*:0]const u8;
    pub const getApplicationId = g_application_get_application_id;

    /// Gets the `gio.DBusConnection` being used by the application, or `NULL`.
    ///
    /// If `gio.Application` is using its D-Bus backend then this function will
    /// return the `gio.DBusConnection` being used for uniqueness and
    /// communication with the desktop environment and other instances of the
    /// application.
    ///
    /// If `gio.Application` is not using D-Bus then this function will return
    /// `NULL`.  This includes the situation where the D-Bus backend would
    /// normally be in use but we were unable to connect to the bus.
    ///
    /// This function must not be called before the application has been
    /// registered.  See `gio.Application.getIsRegistered`.
    extern fn g_application_get_dbus_connection(p_application: *Application) ?*gio.DBusConnection;
    pub const getDbusConnection = g_application_get_dbus_connection;

    /// Gets the D-Bus object path being used by the application, or `NULL`.
    ///
    /// If `gio.Application` is using its D-Bus backend then this function will
    /// return the D-Bus object path that `gio.Application` is using.  If the
    /// application is the primary instance then there is an object published
    /// at this path.  If the application is not the primary instance then
    /// the result of this function is undefined.
    ///
    /// If `gio.Application` is not using D-Bus then this function will return
    /// `NULL`.  This includes the situation where the D-Bus backend would
    /// normally be in use but we were unable to connect to the bus.
    ///
    /// This function must not be called before the application has been
    /// registered.  See `gio.Application.getIsRegistered`.
    extern fn g_application_get_dbus_object_path(p_application: *Application) ?[*:0]const u8;
    pub const getDbusObjectPath = g_application_get_dbus_object_path;

    /// Gets the flags for `application`.
    ///
    /// See `gio.ApplicationFlags`.
    extern fn g_application_get_flags(p_application: *Application) gio.ApplicationFlags;
    pub const getFlags = g_application_get_flags;

    /// Gets the current inactivity timeout for the application.
    ///
    /// This is the amount of time (in milliseconds) after the last call to
    /// `gio.Application.release` before the application stops running.
    extern fn g_application_get_inactivity_timeout(p_application: *Application) c_uint;
    pub const getInactivityTimeout = g_application_get_inactivity_timeout;

    /// Gets the application's current busy state, as set through
    /// `gio.Application.markBusy` or `gio.Application.bindBusyProperty`.
    extern fn g_application_get_is_busy(p_application: *Application) c_int;
    pub const getIsBusy = g_application_get_is_busy;

    /// Checks if `application` is registered.
    ///
    /// An application is registered if `gio.Application.register` has been
    /// successfully called.
    extern fn g_application_get_is_registered(p_application: *Application) c_int;
    pub const getIsRegistered = g_application_get_is_registered;

    /// Checks if `application` is remote.
    ///
    /// If `application` is remote then it means that another instance of
    /// application already exists (the 'primary' instance).  Calls to
    /// perform actions on `application` will result in the actions being
    /// performed by the primary instance.
    ///
    /// The value of this property cannot be accessed before
    /// `gio.Application.register` has been called.  See
    /// `gio.Application.getIsRegistered`.
    extern fn g_application_get_is_remote(p_application: *Application) c_int;
    pub const getIsRemote = g_application_get_is_remote;

    /// Gets the resource base path of `application`.
    ///
    /// See `gio.Application.setResourceBasePath` for more information.
    extern fn g_application_get_resource_base_path(p_application: *Application) ?[*:0]const u8;
    pub const getResourceBasePath = g_application_get_resource_base_path;

    /// Gets the version of `application`.
    extern fn g_application_get_version(p_application: *Application) ?[*:0]const u8;
    pub const getVersion = g_application_get_version;

    /// Increases the use count of `application`.
    ///
    /// Use this function to indicate that the application has a reason to
    /// continue to run.  For example, `gio.Application.hold` is called by GTK
    /// when a toplevel window is on the screen.
    ///
    /// To cancel the hold, call `gio.Application.release`.
    extern fn g_application_hold(p_application: *Application) void;
    pub const hold = g_application_hold;

    /// Increases the busy count of `application`.
    ///
    /// Use this function to indicate that the application is busy, for instance
    /// while a long running operation is pending.
    ///
    /// The busy state will be exposed to other processes, so a session shell will
    /// use that information to indicate the state to the user (e.g. with a
    /// spinner).
    ///
    /// To cancel the busy indication, use `gio.Application.unmarkBusy`.
    ///
    /// The application must be registered before calling this function.
    extern fn g_application_mark_busy(p_application: *Application) void;
    pub const markBusy = g_application_mark_busy;

    /// Opens the given files.
    ///
    /// In essence, this results in the `gio.Application.signals.open` signal being emitted
    /// in the primary instance.
    ///
    /// `n_files` must be greater than zero.
    ///
    /// `hint` is simply passed through to the ::open signal.  It is
    /// intended to be used by applications that have multiple modes for
    /// opening files (eg: "view" vs "edit", etc).  Unless you have a need
    /// for this functionality, you should use "".
    ///
    /// The application must be registered before calling this function
    /// and it must have the `G_APPLICATION_HANDLES_OPEN` flag set.
    extern fn g_application_open(p_application: *Application, p_files: [*]*gio.File, p_n_files: c_int, p_hint: [*:0]const u8) void;
    pub const open = g_application_open;

    /// Immediately quits the application.
    ///
    /// Upon return to the mainloop, `gio.Application.run` will return,
    /// calling only the 'shutdown' function before doing so.
    ///
    /// The hold count is ignored.
    /// Take care if your code has called `gio.Application.hold` on the application and
    /// is therefore still expecting it to exist.
    /// (Note that you may have called `gio.Application.hold` indirectly, for example
    /// through `gtk_application_add_window`.)
    ///
    /// The result of calling `gio.Application.run` again after it returns is
    /// unspecified.
    extern fn g_application_quit(p_application: *Application) void;
    pub const quit = g_application_quit;

    /// Attempts registration of the application.
    ///
    /// This is the point at which the application discovers if it is the
    /// primary instance or merely acting as a remote for an already-existing
    /// primary instance.  This is implemented by attempting to acquire the
    /// application identifier as a unique bus name on the session bus using
    /// GDBus.
    ///
    /// If there is no application ID or if `G_APPLICATION_NON_UNIQUE` was
    /// given, then this process will always become the primary instance.
    ///
    /// Due to the internal architecture of GDBus, method calls can be
    /// dispatched at any time (even if a main loop is not running).  For
    /// this reason, you must ensure that any object paths that you wish to
    /// register are registered before calling this function.
    ///
    /// If the application has already been registered then `TRUE` is
    /// returned with no work performed.
    ///
    /// The `gio.Application.signals.startup` signal is emitted if registration succeeds
    /// and `application` is the primary instance (including the non-unique
    /// case).
    ///
    /// In the event of an error (such as `cancellable` being cancelled, or a
    /// failure to connect to the session bus), `FALSE` is returned and `error`
    /// is set appropriately.
    ///
    /// Note: the return value of this function is not an indicator that this
    /// instance is or is not the primary instance of the application.  See
    /// `gio.Application.getIsRemote` for that.
    extern fn g_application_register(p_application: *Application, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const register = g_application_register;

    /// Decrease the use count of `application`.
    ///
    /// When the use count reaches zero, the application will stop running.
    ///
    /// Never call this function except to cancel the effect of a previous
    /// call to `gio.Application.hold`.
    extern fn g_application_release(p_application: *Application) void;
    pub const release = g_application_release;

    /// Runs the application.
    ///
    /// This function is intended to be run from `main` and its return value
    /// is intended to be returned by `main`. Although you are expected to pass
    /// the `argc`, `argv` parameters from `main` to this function, it is possible
    /// to pass `NULL` if `argv` is not available or commandline handling is not
    /// required.  Note that on Windows, `argc` and `argv` are ignored, and
    /// `g_win32_get_command_line` is called internally (for proper support
    /// of Unicode commandline arguments).
    ///
    /// `gio.Application` will attempt to parse the commandline arguments.  You
    /// can add commandline flags to the list of recognised options by way of
    /// `gio.Application.addMainOptionEntries`.  After this, the
    /// `gio.Application.signals.handle`-local-options signal is emitted, from which the
    /// application can inspect the values of its `GOptionEntrys`.
    ///
    /// `gio.Application.signals.handle`-local-options is a good place to handle options
    /// such as `--version`, where an immediate reply from the local process is
    /// desired (instead of communicating with an already-running instance).
    /// A `gio.Application.signals.handle`-local-options handler can stop further processing
    /// by returning a non-negative value, which then becomes the exit status of
    /// the process.
    ///
    /// What happens next depends on the flags: if
    /// `G_APPLICATION_HANDLES_COMMAND_LINE` was specified then the remaining
    /// commandline arguments are sent to the primary instance, where a
    /// `gio.Application.signals.command`-line signal is emitted.  Otherwise, the
    /// remaining commandline arguments are assumed to be a list of files.
    /// If there are no files listed, the application is activated via the
    /// `gio.Application.signals.activate` signal.  If there are one or more files, and
    /// `G_APPLICATION_HANDLES_OPEN` was specified then the files are opened
    /// via the `gio.Application.signals.open` signal.
    ///
    /// If you are interested in doing more complicated local handling of the
    /// commandline then you should implement your own `gio.Application` subclass
    /// and override `local_command_line`. In this case, you most likely want
    /// to return `TRUE` from your `local_command_line` implementation to
    /// suppress the default handling. See
    /// [gapplication-example-cmdline2.c][https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gapplication-example-cmdline2.c]
    /// for an example.
    ///
    /// If, after the above is done, the use count of the application is zero
    /// then the exit status is returned immediately.  If the use count is
    /// non-zero then the default main context is iterated until the use count
    /// falls to zero, at which point 0 is returned.
    ///
    /// If the `G_APPLICATION_IS_SERVICE` flag is set, then the service will
    /// run for as much as 10 seconds with a use count of zero while waiting
    /// for the message that caused the activation to arrive.  After that,
    /// if the use count falls to zero the application will exit immediately,
    /// except in the case that `gio.Application.setInactivityTimeout` is in
    /// use.
    ///
    /// This function sets the prgname (`glib.setPrgname`), if not already set,
    /// to the basename of argv[0].
    ///
    /// Much like `glib.MainLoop.run`, this function will acquire the main context
    /// for the duration that the application is running.
    ///
    /// Since 2.40, applications that are not explicitly flagged as services
    /// or launchers (ie: neither `G_APPLICATION_IS_SERVICE` or
    /// `G_APPLICATION_IS_LAUNCHER` are given as flags) will check (from the
    /// default handler for local_command_line) if "--gapplication-service"
    /// was given in the command line.  If this flag is present then normal
    /// commandline processing is interrupted and the
    /// `G_APPLICATION_IS_SERVICE` flag is set.  This provides a "compromise"
    /// solution whereby running an application directly from the commandline
    /// will invoke it in the normal way (which can be useful for debugging)
    /// while still allowing applications to be D-Bus activated in service
    /// mode.  The D-Bus service file should invoke the executable with
    /// "--gapplication-service" as the sole commandline argument.  This
    /// approach is suitable for use by most graphical applications but
    /// should not be used from applications like editors that need precise
    /// control over when processes invoked via the commandline will exit and
    /// what their exit status will be.
    extern fn g_application_run(p_application: *Application, p_argc: c_int, p_argv: ?[*][*:0]u8) c_int;
    pub const run = g_application_run;

    /// Sends a notification on behalf of `application` to the desktop shell.
    /// There is no guarantee that the notification is displayed immediately,
    /// or even at all.
    ///
    /// Notifications may persist after the application exits. It will be
    /// D-Bus-activated when the notification or one of its actions is
    /// activated.
    ///
    /// Modifying `notification` after this call has no effect. However, the
    /// object can be reused for a later call to this function.
    ///
    /// `id` may be any string that uniquely identifies the event for the
    /// application. It does not need to be in any special format. For
    /// example, "new-message" might be appropriate for a notification about
    /// new messages.
    ///
    /// If a previous notification was sent with the same `id`, it will be
    /// replaced with `notification` and shown again as if it was a new
    /// notification. This works even for notifications sent from a previous
    /// execution of the application, as long as `id` is the same string.
    ///
    /// `id` may be `NULL`, but it is impossible to replace or withdraw
    /// notifications without an id.
    ///
    /// If `notification` is no longer relevant, it can be withdrawn with
    /// `gio.Application.withdrawNotification`.
    ///
    /// It is an error to call this function if `application` has no
    /// application ID.
    extern fn g_application_send_notification(p_application: *Application, p_id: ?[*:0]const u8, p_notification: *gio.Notification) void;
    pub const sendNotification = g_application_send_notification;

    /// This used to be how actions were associated with a `gio.Application`.
    /// Now there is `gio.ActionMap` for that.
    extern fn g_application_set_action_group(p_application: *Application, p_action_group: ?*gio.ActionGroup) void;
    pub const setActionGroup = g_application_set_action_group;

    /// Sets the unique identifier for `application`.
    ///
    /// The application id can only be modified if `application` has not yet
    /// been registered.
    ///
    /// If non-`NULL`, the application id must be valid.  See
    /// `gio.Application.idIsValid`.
    extern fn g_application_set_application_id(p_application: *Application, p_application_id: ?[*:0]const u8) void;
    pub const setApplicationId = g_application_set_application_id;

    /// Sets or unsets the default application for the process, as returned
    /// by `gio.Application.getDefault`.
    ///
    /// This function does not take its own reference on `application`.  If
    /// `application` is destroyed then the default application will revert
    /// back to `NULL`.
    extern fn g_application_set_default(p_application: ?*Application) void;
    pub const setDefault = g_application_set_default;

    /// Sets the flags for `application`.
    ///
    /// The flags can only be modified if `application` has not yet been
    /// registered.
    ///
    /// See `gio.ApplicationFlags`.
    extern fn g_application_set_flags(p_application: *Application, p_flags: gio.ApplicationFlags) void;
    pub const setFlags = g_application_set_flags;

    /// Sets the current inactivity timeout for the application.
    ///
    /// This is the amount of time (in milliseconds) after the last call to
    /// `gio.Application.release` before the application stops running.
    ///
    /// This call has no side effects of its own.  The value set here is only
    /// used for next time `gio.Application.release` drops the use count to
    /// zero.  Any timeouts currently in progress are not impacted.
    extern fn g_application_set_inactivity_timeout(p_application: *Application, p_inactivity_timeout: c_uint) void;
    pub const setInactivityTimeout = g_application_set_inactivity_timeout;

    /// Adds a description to the `application` option context.
    ///
    /// See `glib.OptionContext.setDescription` for more information.
    extern fn g_application_set_option_context_description(p_application: *Application, p_description: ?[*:0]const u8) void;
    pub const setOptionContextDescription = g_application_set_option_context_description;

    /// Sets the parameter string to be used by the commandline handling of `application`.
    ///
    /// This function registers the argument to be passed to `glib.OptionContext.new`
    /// when the internal `glib.OptionContext` of `application` is created.
    ///
    /// See `glib.OptionContext.new` for more information about `parameter_string`.
    extern fn g_application_set_option_context_parameter_string(p_application: *Application, p_parameter_string: ?[*:0]const u8) void;
    pub const setOptionContextParameterString = g_application_set_option_context_parameter_string;

    /// Adds a summary to the `application` option context.
    ///
    /// See `glib.OptionContext.setSummary` for more information.
    extern fn g_application_set_option_context_summary(p_application: *Application, p_summary: ?[*:0]const u8) void;
    pub const setOptionContextSummary = g_application_set_option_context_summary;

    /// Sets (or unsets) the base resource path of `application`.
    ///
    /// The path is used to automatically load various [application
    /// resources][gresource] such as menu layouts and action descriptions.
    /// The various types of resources will be found at fixed names relative
    /// to the given base path.
    ///
    /// By default, the resource base path is determined from the application
    /// ID by prefixing '/' and replacing each '.' with '/'.  This is done at
    /// the time that the `gio.Application` object is constructed.  Changes to
    /// the application ID after that point will not have an impact on the
    /// resource base path.
    ///
    /// As an example, if the application has an ID of "org.example.app" then
    /// the default resource base path will be "/org/example/app".  If this
    /// is a `GtkApplication` (and you have not manually changed the path)
    /// then Gtk will then search for the menus of the application at
    /// "/org/example/app/gtk/menus.ui".
    ///
    /// See `gio.Resource` for more information about adding resources to your
    /// application.
    ///
    /// You can disable automatic resource loading functionality by setting
    /// the path to `NULL`.
    ///
    /// Changing the resource base path once the application is running is
    /// not recommended.  The point at which the resource path is consulted
    /// for forming paths for various purposes is unspecified.  When writing
    /// a sub-class of `gio.Application` you should either set the
    /// `gio.Application.properties.resource`-base-path property at construction time, or call
    /// this function during the instance initialization. Alternatively, you
    /// can call this function in the `gio.ApplicationClass.startup` virtual function,
    /// before chaining up to the parent implementation.
    extern fn g_application_set_resource_base_path(p_application: *Application, p_resource_path: ?[*:0]const u8) void;
    pub const setResourceBasePath = g_application_set_resource_base_path;

    /// Sets the version number of `application`. This will be used to implement
    /// a `--version` command line argument
    ///
    /// The application version can only be modified if `application` has not yet
    /// been registered.
    extern fn g_application_set_version(p_application: *Application, p_version: [*:0]const u8) void;
    pub const setVersion = g_application_set_version;

    /// Destroys a binding between `property` and the busy state of
    /// `application` that was previously created with
    /// `gio.Application.bindBusyProperty`.
    extern fn g_application_unbind_busy_property(p_application: *Application, p_object: *gobject.Object, p_property: [*:0]const u8) void;
    pub const unbindBusyProperty = g_application_unbind_busy_property;

    /// Decreases the busy count of `application`.
    ///
    /// When the busy count reaches zero, the new state will be propagated
    /// to other processes.
    ///
    /// This function must only be called to cancel the effect of a previous
    /// call to `gio.Application.markBusy`.
    extern fn g_application_unmark_busy(p_application: *Application) void;
    pub const unmarkBusy = g_application_unmark_busy;

    /// Withdraws a notification that was sent with
    /// `gio.Application.sendNotification`.
    ///
    /// This call does nothing if a notification with `id` doesn't exist or
    /// the notification was never sent.
    ///
    /// This function works even for notifications sent in previous
    /// executions of this application, as long `id` is the same as it was for
    /// the sent notification.
    ///
    /// Note that notifications are dismissed when the user clicks on one
    /// of the buttons in a notification or triggers its default action, so
    /// there is no need to explicitly withdraw the notification in that case.
    extern fn g_application_withdraw_notification(p_application: *Application, p_id: [*:0]const u8) void;
    pub const withdrawNotification = g_application_withdraw_notification;

    extern fn g_application_get_type() usize;
    pub const getGObjectType = g_application_get_type;

    extern fn g_object_ref(p_self: *gio.Application) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.Application) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Application, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GApplicationCommandLine` represents a command-line invocation of
/// an application.
///
/// It is created by `gio.Application` and emitted
/// in the `gio.Application.signals.command_line` signal and virtual function.
///
/// The class contains the list of arguments that the program was invoked
/// with. It is also possible to query if the commandline invocation was
/// local (ie: the current process is running in direct response to the
/// invocation) or remote (ie: some other process forwarded the
/// commandline to this process).
///
/// The `GApplicationCommandLine` object can provide the `argc` and `argv`
/// parameters for use with the `glib.OptionContext` command-line parsing API,
/// with the `gio.ApplicationCommandLine.getArguments` function. See
/// [gapplication-example-cmdline3.c][gapplication-example-cmdline3]
/// for an example.
///
/// The exit status of the originally-invoked process may be set and
/// messages can be printed to stdout or stderr of that process.
///
/// For remote invocation, the originally-invoked process exits when
/// `gio.ApplicationCommandLine.done` method is called. This method is
/// also automatically called when the object is disposed.
///
/// The main use for `GApplicationCommandLine` (and the
/// `gio.Application.signals.command_line` signal) is 'Emacs server' like use cases:
/// You can set the `EDITOR` environment variable to have e.g. git use
/// your favourite editor to edit commit messages, and if you already
/// have an instance of the editor running, the editing will happen
/// in the running instance, instead of opening a new one. An important
/// aspect of this use case is that the process that gets started by git
/// does not return until the editing is done.
///
/// Normally, the commandline is completely handled in the
/// `gio.Application.signals.command_line` handler. The launching instance exits
/// once the signal handler in the primary instance has returned, and
/// the return value of the signal handler becomes the exit status
/// of the launching instance.
///
/// ```c
/// static int
/// command_line (GApplication            *application,
///               GApplicationCommandLine *cmdline)
/// {
///   gchar **argv;
///   gint argc;
///   gint i;
///
///   argv = g_application_command_line_get_arguments (cmdline, &argc);
///
///   g_application_command_line_print (cmdline,
///                                     "This text is written back\n"
///                                     "to stdout of the caller\n");
///
///   for (i = 0; i < argc; i++)
///     g_print ("argument `d`: `s`\n", i, argv[i]);
///
///   g_strfreev (argv);
///
///   return 0;
/// }
/// ```
///
/// The complete example can be found here:
/// [gapplication-example-cmdline.c](https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gapplication-example-cmdline.c)
///
/// In more complicated cases, the handling of the commandline can be
/// split between the launcher and the primary instance.
///
/// ```c
/// static gboolean
///  test_local_cmdline (GApplication   *application,
///                      gchar        ***arguments,
///                      gint           *exit_status)
/// {
///   gint i, j;
///   gchar **argv;
///
///   argv = *arguments;
///
///   if (argv[0] == NULL)
///     {
///       *exit_status = 0;
///       return FALSE;
///     }
///
///   i = 1;
///   while (argv[i])
///     {
///       if (g_str_has_prefix (argv[i], "--local-"))
///         {
///           g_print ("handling argument `s` locally\n", argv[i]);
///           g_free (argv[i]);
///           for (j = i; argv[j]; j++)
///             argv[j] = argv[j + 1];
///         }
///       else
///         {
///           g_print ("not handling argument `s` locally\n", argv[i]);
///           i++;
///         }
///     }
///
///   *exit_status = 0;
///
///   return FALSE;
/// }
///
/// static void
/// test_application_class_init (TestApplicationClass *class)
/// {
///   G_APPLICATION_CLASS (class)->local_command_line = test_local_cmdline;
///
///   ...
/// }
/// ```
///
/// In this example of split commandline handling, options that start
/// with `--local-` are handled locally, all other options are passed
/// to the `gio.Application.signals.command_line` handler which runs in the primary
/// instance.
///
/// The complete example can be found here:
/// [gapplication-example-cmdline2.c](https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gapplication-example-cmdline2.c)
///
/// If handling the commandline requires a lot of work, it may be better to defer it.
///
/// ```c
/// static gboolean
/// my_cmdline_handler (gpointer data)
/// {
///   GApplicationCommandLine *cmdline = data;
///
///   // do the heavy lifting in an idle
///
///   g_application_command_line_set_exit_status (cmdline, 0);
///   g_object_unref (cmdline); // this releases the application
///
///   return G_SOURCE_REMOVE;
/// }
///
/// static int
/// command_line (GApplication            *application,
///               GApplicationCommandLine *cmdline)
/// {
///   // keep the application running until we are done with this commandline
///   g_application_hold (application);
///
///   g_object_set_data_full (G_OBJECT (cmdline),
///                           "application", application,
///                           (GDestroyNotify)g_application_release);
///
///   g_object_ref (cmdline);
///   g_idle_add (my_cmdline_handler, cmdline);
///
///   return 0;
/// }
/// ```
///
/// In this example the commandline is not completely handled before
/// the `gio.Application.signals.command_line` handler returns. Instead, we keep
/// a reference to the `GApplicationCommandLine` object and handle it
/// later (in this example, in an idle). Note that it is necessary to
/// hold the application until you are done with the commandline.
///
/// The complete example can be found here:
/// [gapplication-example-cmdline3.c](https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gapplication-example-cmdline3.c)
pub const ApplicationCommandLine = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.ApplicationCommandLineClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.ApplicationCommandLinePrivate,

    pub const virtual_methods = struct {
        /// Signals that command line processing is completed.
        ///
        /// For remote invocation, it causes the invoking process to terminate.
        ///
        /// For local invocation, it does nothing.
        ///
        /// This method should be called in the `gio.Application.signals.command_line`
        /// handler, after the exit status is set and all messages are printed.
        ///
        /// After this call, `gio.ApplicationCommandLine.setExitStatus` has no effect.
        /// Subsequent calls to this method are no-ops.
        ///
        /// This method is automatically called when the `gio.ApplicationCommandLine`
        /// object is disposed — so you can omit the call in non-garbage collected
        /// languages.
        pub const done = struct {
            pub fn call(p_class: anytype, p_cmdline: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(ApplicationCommandLine.Class, p_class).f_done.?(gobject.ext.as(ApplicationCommandLine, p_cmdline));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_cmdline: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(ApplicationCommandLine.Class, p_class).f_done = @ptrCast(p_implementation);
            }
        };

        /// Gets the stdin of the invoking process.
        ///
        /// The `gio.InputStream` can be used to read data passed to the standard
        /// input of the invoking process.
        /// This doesn't work on all platforms.  Presently, it is only available
        /// on UNIX when using a D-Bus daemon capable of passing file descriptors.
        /// If stdin is not available then `NULL` will be returned.  In the
        /// future, support may be expanded to other platforms.
        ///
        /// You must only call this function once per commandline invocation.
        pub const get_stdin = struct {
            pub fn call(p_class: anytype, p_cmdline: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) ?*gio.InputStream {
                return gobject.ext.as(ApplicationCommandLine.Class, p_class).f_get_stdin.?(gobject.ext.as(ApplicationCommandLine, p_cmdline));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_cmdline: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) ?*gio.InputStream) void {
                gobject.ext.as(ApplicationCommandLine.Class, p_class).f_get_stdin = @ptrCast(p_implementation);
            }
        };

        /// Prints a message using the stdout print handler in the invoking process.
        ///
        /// Unlike `gio.ApplicationCommandLine.print`, `message` is not a ``printf``-style
        /// format string. Use this function if `message` contains text you don't have
        /// control over, that could include ``printf`` escape sequences.
        pub const print_literal = struct {
            pub fn call(p_class: anytype, p_cmdline: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_message: [*:0]const u8) void {
                return gobject.ext.as(ApplicationCommandLine.Class, p_class).f_print_literal.?(gobject.ext.as(ApplicationCommandLine, p_cmdline), p_message);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_cmdline: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_message: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(ApplicationCommandLine.Class, p_class).f_print_literal = @ptrCast(p_implementation);
            }
        };

        /// Prints a message using the stderr print handler in the invoking process.
        ///
        /// Unlike `gio.ApplicationCommandLine.printerr`, `message` is not
        /// a ``printf``-style format string. Use this function if `message` contains text
        /// you don't have control over, that could include ``printf`` escape sequences.
        pub const printerr_literal = struct {
            pub fn call(p_class: anytype, p_cmdline: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_message: [*:0]const u8) void {
                return gobject.ext.as(ApplicationCommandLine.Class, p_class).f_printerr_literal.?(gobject.ext.as(ApplicationCommandLine, p_cmdline), p_message);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_cmdline: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_message: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(ApplicationCommandLine.Class, p_class).f_printerr_literal = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The commandline that caused this `gio.Application.signals.command_line`
        /// signal emission.
        pub const arguments = struct {
            pub const name = "arguments";

            pub const Type = ?*glib.Variant;
        };

        /// Whether this is a remote commandline.
        pub const is_remote = struct {
            pub const name = "is-remote";

            pub const Type = c_int;
        };

        /// The options sent along with the commandline.
        pub const options = struct {
            pub const name = "options";

            pub const Type = ?*glib.Variant;
        };

        /// Platform-specific data for the commandline.
        pub const platform_data = struct {
            pub const name = "platform-data";

            pub const Type = ?*glib.Variant;
        };
    };

    pub const signals = struct {};

    /// Creates a `gio.File` corresponding to a filename that was given as part
    /// of the invocation of `cmdline`.
    ///
    /// This differs from `gio.fileNewForCommandlineArg` in that it
    /// resolves relative pathnames using the current working directory of
    /// the invoking process rather than the local process.
    extern fn g_application_command_line_create_file_for_arg(p_cmdline: *ApplicationCommandLine, p_arg: [*:0]const u8) *gio.File;
    pub const createFileForArg = g_application_command_line_create_file_for_arg;

    /// Signals that command line processing is completed.
    ///
    /// For remote invocation, it causes the invoking process to terminate.
    ///
    /// For local invocation, it does nothing.
    ///
    /// This method should be called in the `gio.Application.signals.command_line`
    /// handler, after the exit status is set and all messages are printed.
    ///
    /// After this call, `gio.ApplicationCommandLine.setExitStatus` has no effect.
    /// Subsequent calls to this method are no-ops.
    ///
    /// This method is automatically called when the `gio.ApplicationCommandLine`
    /// object is disposed — so you can omit the call in non-garbage collected
    /// languages.
    extern fn g_application_command_line_done(p_cmdline: *ApplicationCommandLine) void;
    pub const done = g_application_command_line_done;

    /// Gets the list of arguments that was passed on the command line.
    ///
    /// The strings in the array may contain non-UTF-8 data on UNIX (such as
    /// filenames or arguments given in the system locale) but are always in
    /// UTF-8 on Windows.
    ///
    /// If you wish to use the return value with `glib.OptionContext`, you must
    /// use `glib.OptionContext.parseStrv`.
    ///
    /// The return value is `NULL`-terminated and should be freed using
    /// `glib.strfreev`.
    extern fn g_application_command_line_get_arguments(p_cmdline: *ApplicationCommandLine, p_argc: ?*c_int) [*][*:0]u8;
    pub const getArguments = g_application_command_line_get_arguments;

    /// Gets the working directory of the command line invocation.
    /// The string may contain non-utf8 data.
    ///
    /// It is possible that the remote application did not send a working
    /// directory, so this may be `NULL`.
    ///
    /// The return value should not be modified or freed and is valid for as
    /// long as `cmdline` exists.
    extern fn g_application_command_line_get_cwd(p_cmdline: *ApplicationCommandLine) ?[*:0]const u8;
    pub const getCwd = g_application_command_line_get_cwd;

    /// Gets the contents of the 'environ' variable of the command line
    /// invocation, as would be returned by `glib.getEnviron`, ie as a
    /// `NULL`-terminated list of strings in the form 'NAME=VALUE'.
    /// The strings may contain non-utf8 data.
    ///
    /// The remote application usually does not send an environment.  Use
    /// `G_APPLICATION_SEND_ENVIRONMENT` to affect that.  Even with this flag
    /// set it is possible that the environment is still not available (due
    /// to invocation messages from other applications).
    ///
    /// The return value should not be modified or freed and is valid for as
    /// long as `cmdline` exists.
    ///
    /// See `gio.ApplicationCommandLine.getenv` if you are only interested
    /// in the value of a single environment variable.
    extern fn g_application_command_line_get_environ(p_cmdline: *ApplicationCommandLine) [*]const [*:0]const u8;
    pub const getEnviron = g_application_command_line_get_environ;

    /// Gets the exit status of `cmdline`.  See
    /// `gio.ApplicationCommandLine.setExitStatus` for more information.
    extern fn g_application_command_line_get_exit_status(p_cmdline: *ApplicationCommandLine) c_int;
    pub const getExitStatus = g_application_command_line_get_exit_status;

    /// Determines if `cmdline` represents a remote invocation.
    extern fn g_application_command_line_get_is_remote(p_cmdline: *ApplicationCommandLine) c_int;
    pub const getIsRemote = g_application_command_line_get_is_remote;

    /// Gets the options that were passed to `g_application_command_line`.
    ///
    /// If you did not override `local_command_line` then these are the same
    /// options that were parsed according to the `GOptionEntrys` added to the
    /// application with `gio.Application.addMainOptionEntries` and possibly
    /// modified from your GApplication::handle-local-options handler.
    ///
    /// If no options were sent then an empty dictionary is returned so that
    /// you don't need to check for `NULL`.
    ///
    /// The data has been passed via an untrusted external process, so the types of
    /// all values must be checked before being used.
    extern fn g_application_command_line_get_options_dict(p_cmdline: *ApplicationCommandLine) *glib.VariantDict;
    pub const getOptionsDict = g_application_command_line_get_options_dict;

    /// Gets the platform data associated with the invocation of `cmdline`.
    ///
    /// This is a `glib.Variant` dictionary containing information about the
    /// context in which the invocation occurred.  It typically contains
    /// information like the current working directory and the startup
    /// notification ID.
    ///
    /// It comes from an untrusted external process and hence the types of all
    /// values must be validated before being used.
    ///
    /// For local invocation, it will be `NULL`.
    extern fn g_application_command_line_get_platform_data(p_cmdline: *ApplicationCommandLine) ?*glib.Variant;
    pub const getPlatformData = g_application_command_line_get_platform_data;

    /// Gets the stdin of the invoking process.
    ///
    /// The `gio.InputStream` can be used to read data passed to the standard
    /// input of the invoking process.
    /// This doesn't work on all platforms.  Presently, it is only available
    /// on UNIX when using a D-Bus daemon capable of passing file descriptors.
    /// If stdin is not available then `NULL` will be returned.  In the
    /// future, support may be expanded to other platforms.
    ///
    /// You must only call this function once per commandline invocation.
    extern fn g_application_command_line_get_stdin(p_cmdline: *ApplicationCommandLine) ?*gio.InputStream;
    pub const getStdin = g_application_command_line_get_stdin;

    /// Gets the value of a particular environment variable of the command
    /// line invocation, as would be returned by `glib.getenv`.  The strings may
    /// contain non-utf8 data.
    ///
    /// The remote application usually does not send an environment.  Use
    /// `G_APPLICATION_SEND_ENVIRONMENT` to affect that.  Even with this flag
    /// set it is possible that the environment is still not available (due
    /// to invocation messages from other applications).
    ///
    /// The return value should not be modified or freed and is valid for as
    /// long as `cmdline` exists.
    extern fn g_application_command_line_getenv(p_cmdline: *ApplicationCommandLine, p_name: [*:0]const u8) ?[*:0]const u8;
    pub const getenv = g_application_command_line_getenv;

    /// Formats a message and prints it using the stdout print handler in the
    /// invoking process.
    ///
    /// If `cmdline` is a local invocation then this is exactly equivalent to
    /// `glib.print`.  If `cmdline` is remote then this is equivalent to calling
    /// `glib.print` in the invoking process.
    extern fn g_application_command_line_print(p_cmdline: *ApplicationCommandLine, p_format: [*:0]const u8, ...) void;
    pub const print = g_application_command_line_print;

    /// Prints a message using the stdout print handler in the invoking process.
    ///
    /// Unlike `gio.ApplicationCommandLine.print`, `message` is not a ``printf``-style
    /// format string. Use this function if `message` contains text you don't have
    /// control over, that could include ``printf`` escape sequences.
    extern fn g_application_command_line_print_literal(p_cmdline: *ApplicationCommandLine, p_message: [*:0]const u8) void;
    pub const printLiteral = g_application_command_line_print_literal;

    /// Formats a message and prints it using the stderr print handler in the
    /// invoking process.
    ///
    /// If `cmdline` is a local invocation then this is exactly equivalent to
    /// `glib.printerr`.  If `cmdline` is remote then this is equivalent to
    /// calling `glib.printerr` in the invoking process.
    extern fn g_application_command_line_printerr(p_cmdline: *ApplicationCommandLine, p_format: [*:0]const u8, ...) void;
    pub const printerr = g_application_command_line_printerr;

    /// Prints a message using the stderr print handler in the invoking process.
    ///
    /// Unlike `gio.ApplicationCommandLine.printerr`, `message` is not
    /// a ``printf``-style format string. Use this function if `message` contains text
    /// you don't have control over, that could include ``printf`` escape sequences.
    extern fn g_application_command_line_printerr_literal(p_cmdline: *ApplicationCommandLine, p_message: [*:0]const u8) void;
    pub const printerrLiteral = g_application_command_line_printerr_literal;

    /// Sets the exit status that will be used when the invoking process
    /// exits.
    ///
    /// The return value of the `gio.Application.signals.command`-line signal is
    /// passed to this function when the handler returns.  This is the usual
    /// way of setting the exit status.
    ///
    /// In the event that you want the remote invocation to continue running
    /// and want to decide on the exit status in the future, you can use this
    /// call.  For the case of a remote invocation, the remote process will
    /// typically exit when the last reference is dropped on `cmdline`.  The
    /// exit status of the remote process will be equal to the last value
    /// that was set with this function.
    ///
    /// In the case that the commandline invocation is local, the situation
    /// is slightly more complicated.  If the commandline invocation results
    /// in the mainloop running (ie: because the use-count of the application
    /// increased to a non-zero value) then the application is considered to
    /// have been 'successful' in a certain sense, and the exit status is
    /// always zero.  If the application use count is zero, though, the exit
    /// status of the local `gio.ApplicationCommandLine` is used.
    ///
    /// This method is a no-op if `gio.ApplicationCommandLine.done` has
    /// been called.
    extern fn g_application_command_line_set_exit_status(p_cmdline: *ApplicationCommandLine, p_exit_status: c_int) void;
    pub const setExitStatus = g_application_command_line_set_exit_status;

    extern fn g_application_command_line_get_type() usize;
    pub const getGObjectType = g_application_command_line_get_type;

    extern fn g_object_ref(p_self: *gio.ApplicationCommandLine) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.ApplicationCommandLine) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ApplicationCommandLine, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Buffered input stream implements `gio.FilterInputStream` and provides
/// for buffered reads.
///
/// By default, `GBufferedInputStream`'s buffer size is set at 4 kilobytes.
///
/// To create a buffered input stream, use `gio.BufferedInputStream.new`,
/// or `gio.BufferedInputStream.newSized` to specify the buffer's size at
/// construction.
///
/// To get the size of a buffer within a buffered input stream, use
/// `gio.BufferedInputStream.getBufferSize`. To change the size of a
/// buffered input stream's buffer, use `gio.BufferedInputStream.setBufferSize`.
/// Note that the buffer's size cannot be reduced below the size of the data within the buffer.
pub const BufferedInputStream = extern struct {
    pub const Parent = gio.FilterInputStream;
    pub const Implements = [_]type{gio.Seekable};
    pub const Class = gio.BufferedInputStreamClass;
    f_parent_instance: gio.FilterInputStream,
    f_priv: ?*gio.BufferedInputStreamPrivate,

    pub const virtual_methods = struct {
        /// Tries to read `count` bytes from the stream into the buffer.
        /// Will block during this read.
        ///
        /// If `count` is zero, returns zero and does nothing. A value of `count`
        /// larger than `G_MAXSSIZE` will cause a
        /// `gio.@"IOErrorEnum.INVALID_ARGUMENT"` error.
        ///
        /// On success, the number of bytes read into the buffer is returned.
        /// It is not an error if this is not the same as the requested size, as it
        /// can happen e.g. near the end of a file. Zero is returned on end of file
        /// (or if `count` is zero),  but never otherwise.
        ///
        /// If `count` is -1 then the attempted read size is equal to the number of
        /// bytes that are required to fill the buffer.
        ///
        /// If `cancellable` is not `NULL`, then the operation can be cancelled by
        /// triggering the cancellable object from another thread. If the operation
        /// was cancelled, the error `gio.@"IOErrorEnum.CANCELLED"` will be returned.
        /// If an operation was partially finished when the operation was cancelled the
        /// partial result will be returned, without an error.
        ///
        /// On error `-1` is returned and `error` is set accordingly.
        ///
        /// For the asynchronous, non-blocking, version of this function, see
        /// `gio.BufferedInputStream.fillAsync`.
        pub const fill = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_count: isize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) isize {
                return gobject.ext.as(BufferedInputStream.Class, p_class).f_fill.?(gobject.ext.as(BufferedInputStream, p_stream), p_count, p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_count: isize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) isize) void {
                gobject.ext.as(BufferedInputStream.Class, p_class).f_fill = @ptrCast(p_implementation);
            }
        };

        /// Reads data into `stream`'s buffer asynchronously, up to `count` size.
        /// `io_priority` can be used to prioritize reads. For the synchronous
        /// version of this function, see `gio.BufferedInputStream.fill`.
        ///
        /// If `count` is `-1` then the attempted read size is equal to the number
        /// of bytes that are required to fill the buffer.
        pub const fill_async = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_count: isize, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(BufferedInputStream.Class, p_class).f_fill_async.?(gobject.ext.as(BufferedInputStream, p_stream), p_count, p_io_priority, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_count: isize, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(BufferedInputStream.Class, p_class).f_fill_async = @ptrCast(p_implementation);
            }
        };

        /// Finishes an asynchronous read.
        pub const fill_finish = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) isize {
                return gobject.ext.as(BufferedInputStream.Class, p_class).f_fill_finish.?(gobject.ext.as(BufferedInputStream, p_stream), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) isize) void {
                gobject.ext.as(BufferedInputStream.Class, p_class).f_fill_finish = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The size of the backend buffer, in bytes.
        pub const buffer_size = struct {
            pub const name = "buffer-size";

            pub const Type = c_uint;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gio.InputStream` from the given `base_stream`, with
    /// a buffer set to the default size (4 kilobytes).
    extern fn g_buffered_input_stream_new(p_base_stream: *gio.InputStream) *gio.BufferedInputStream;
    pub const new = g_buffered_input_stream_new;

    /// Creates a new `gio.BufferedInputStream` from the given `base_stream`,
    /// with a buffer set to `size`.
    extern fn g_buffered_input_stream_new_sized(p_base_stream: *gio.InputStream, p_size: usize) *gio.BufferedInputStream;
    pub const newSized = g_buffered_input_stream_new_sized;

    /// Tries to read `count` bytes from the stream into the buffer.
    /// Will block during this read.
    ///
    /// If `count` is zero, returns zero and does nothing. A value of `count`
    /// larger than `G_MAXSSIZE` will cause a
    /// `gio.@"IOErrorEnum.INVALID_ARGUMENT"` error.
    ///
    /// On success, the number of bytes read into the buffer is returned.
    /// It is not an error if this is not the same as the requested size, as it
    /// can happen e.g. near the end of a file. Zero is returned on end of file
    /// (or if `count` is zero),  but never otherwise.
    ///
    /// If `count` is -1 then the attempted read size is equal to the number of
    /// bytes that are required to fill the buffer.
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `gio.@"IOErrorEnum.CANCELLED"` will be returned.
    /// If an operation was partially finished when the operation was cancelled the
    /// partial result will be returned, without an error.
    ///
    /// On error `-1` is returned and `error` is set accordingly.
    ///
    /// For the asynchronous, non-blocking, version of this function, see
    /// `gio.BufferedInputStream.fillAsync`.
    extern fn g_buffered_input_stream_fill(p_stream: *BufferedInputStream, p_count: isize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) isize;
    pub const fill = g_buffered_input_stream_fill;

    /// Reads data into `stream`'s buffer asynchronously, up to `count` size.
    /// `io_priority` can be used to prioritize reads. For the synchronous
    /// version of this function, see `gio.BufferedInputStream.fill`.
    ///
    /// If `count` is `-1` then the attempted read size is equal to the number
    /// of bytes that are required to fill the buffer.
    extern fn g_buffered_input_stream_fill_async(p_stream: *BufferedInputStream, p_count: isize, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const fillAsync = g_buffered_input_stream_fill_async;

    /// Finishes an asynchronous read.
    extern fn g_buffered_input_stream_fill_finish(p_stream: *BufferedInputStream, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) isize;
    pub const fillFinish = g_buffered_input_stream_fill_finish;

    /// Gets the size of the available data within the stream.
    extern fn g_buffered_input_stream_get_available(p_stream: *BufferedInputStream) usize;
    pub const getAvailable = g_buffered_input_stream_get_available;

    /// Gets the size of the input buffer.
    extern fn g_buffered_input_stream_get_buffer_size(p_stream: *BufferedInputStream) usize;
    pub const getBufferSize = g_buffered_input_stream_get_buffer_size;

    /// Peeks in the buffer, copying data of size `count` into `buffer`,
    /// offset `offset` bytes.
    extern fn g_buffered_input_stream_peek(p_stream: *BufferedInputStream, p_buffer: [*]u8, p_offset: usize, p_count: usize) usize;
    pub const peek = g_buffered_input_stream_peek;

    /// Returns the buffer with the currently available bytes. The returned
    /// buffer must not be modified and will become invalid when reading from
    /// the stream or filling the buffer.
    extern fn g_buffered_input_stream_peek_buffer(p_stream: *BufferedInputStream, p_count: *usize) [*]u8;
    pub const peekBuffer = g_buffered_input_stream_peek_buffer;

    /// Tries to read a single byte from the stream or the buffer. Will block
    /// during this read.
    ///
    /// On success, the byte read from the stream is returned. On end of stream
    /// `-1` is returned but it's not an exceptional error and `error` is not set.
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `gio.@"IOErrorEnum.CANCELLED"` will be returned.
    /// If an operation was partially finished when the operation was cancelled the
    /// partial result will be returned, without an error.
    ///
    /// On error `-1` is returned and `error` is set accordingly.
    extern fn g_buffered_input_stream_read_byte(p_stream: *BufferedInputStream, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const readByte = g_buffered_input_stream_read_byte;

    /// Sets the size of the internal buffer of `stream` to `size`, or to the
    /// size of the contents of the buffer. The buffer can never be resized
    /// smaller than its current contents.
    extern fn g_buffered_input_stream_set_buffer_size(p_stream: *BufferedInputStream, p_size: usize) void;
    pub const setBufferSize = g_buffered_input_stream_set_buffer_size;

    extern fn g_buffered_input_stream_get_type() usize;
    pub const getGObjectType = g_buffered_input_stream_get_type;

    extern fn g_object_ref(p_self: *gio.BufferedInputStream) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.BufferedInputStream) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *BufferedInputStream, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Buffered output stream implements `gio.FilterOutputStream` and provides
/// for buffered writes.
///
/// By default, `GBufferedOutputStream`'s buffer size is set at 4 kilobytes.
///
/// To create a buffered output stream, use `gio.BufferedOutputStream.new`,
/// or `gio.BufferedOutputStream.newSized` to specify the buffer's size
/// at construction.
///
/// To get the size of a buffer within a buffered input stream, use
/// `gio.BufferedOutputStream.getBufferSize`. To change the size of a
/// buffered output stream's buffer, use `gio.BufferedOutputStream.setBufferSize`.
/// Note that the buffer's size cannot be reduced below the size of the data within the buffer.
pub const BufferedOutputStream = extern struct {
    pub const Parent = gio.FilterOutputStream;
    pub const Implements = [_]type{gio.Seekable};
    pub const Class = gio.BufferedOutputStreamClass;
    f_parent_instance: gio.FilterOutputStream,
    f_priv: ?*gio.BufferedOutputStreamPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Whether the buffer should automatically grow.
        pub const auto_grow = struct {
            pub const name = "auto-grow";

            pub const Type = c_int;
        };

        /// The size of the backend buffer, in bytes.
        pub const buffer_size = struct {
            pub const name = "buffer-size";

            pub const Type = c_uint;
        };
    };

    pub const signals = struct {};

    /// Creates a new buffered output stream for a base stream.
    extern fn g_buffered_output_stream_new(p_base_stream: *gio.OutputStream) *gio.BufferedOutputStream;
    pub const new = g_buffered_output_stream_new;

    /// Creates a new buffered output stream with a given buffer size.
    extern fn g_buffered_output_stream_new_sized(p_base_stream: *gio.OutputStream, p_size: usize) *gio.BufferedOutputStream;
    pub const newSized = g_buffered_output_stream_new_sized;

    /// Checks if the buffer automatically grows as data is added.
    extern fn g_buffered_output_stream_get_auto_grow(p_stream: *BufferedOutputStream) c_int;
    pub const getAutoGrow = g_buffered_output_stream_get_auto_grow;

    /// Gets the size of the buffer in the `stream`.
    extern fn g_buffered_output_stream_get_buffer_size(p_stream: *BufferedOutputStream) usize;
    pub const getBufferSize = g_buffered_output_stream_get_buffer_size;

    /// Sets whether or not the `stream`'s buffer should automatically grow.
    /// If `auto_grow` is true, then each write will just make the buffer
    /// larger, and you must manually flush the buffer to actually write out
    /// the data to the underlying stream.
    extern fn g_buffered_output_stream_set_auto_grow(p_stream: *BufferedOutputStream, p_auto_grow: c_int) void;
    pub const setAutoGrow = g_buffered_output_stream_set_auto_grow;

    /// Sets the size of the internal buffer to `size`.
    extern fn g_buffered_output_stream_set_buffer_size(p_stream: *BufferedOutputStream, p_size: usize) void;
    pub const setBufferSize = g_buffered_output_stream_set_buffer_size;

    extern fn g_buffered_output_stream_get_type() usize;
    pub const getGObjectType = g_buffered_output_stream_get_type;

    extern fn g_object_ref(p_self: *gio.BufferedOutputStream) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.BufferedOutputStream) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *BufferedOutputStream, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GBytesIcon` specifies an image held in memory in a common format (usually
/// PNG) to be used as icon.
pub const BytesIcon = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{ gio.Icon, gio.LoadableIcon };
    pub const Class = opaque {
        pub const Instance = BytesIcon;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The bytes containing the icon.
        pub const bytes = struct {
            pub const name = "bytes";

            pub const Type = ?*glib.Bytes;
        };
    };

    pub const signals = struct {};

    /// Creates a new icon for a bytes.
    ///
    /// This cannot fail, but loading and interpreting the bytes may fail later on
    /// (for example, if `gio.LoadableIcon.load` is called) if the image is invalid.
    extern fn g_bytes_icon_new(p_bytes: *glib.Bytes) *gio.BytesIcon;
    pub const new = g_bytes_icon_new;

    /// Gets the `glib.Bytes` associated with the given `icon`.
    extern fn g_bytes_icon_get_bytes(p_icon: *BytesIcon) *glib.Bytes;
    pub const getBytes = g_bytes_icon_get_bytes;

    extern fn g_bytes_icon_get_type() usize;
    pub const getGObjectType = g_bytes_icon_get_type;

    extern fn g_object_ref(p_self: *gio.BytesIcon) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.BytesIcon) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *BytesIcon, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GCancellable` allows operations to be cancelled.
///
/// `GCancellable` is a thread-safe operation cancellation stack used
/// throughout GIO to allow for cancellation of synchronous and
/// asynchronous operations.
pub const Cancellable = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.CancellableClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.CancellablePrivate,

    pub const virtual_methods = struct {
        pub const cancelled = struct {
            pub fn call(p_class: anytype, p_cancellable: ?*compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Cancellable.Class, p_class).f_cancelled.?(gobject.ext.as(Cancellable, p_cancellable));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_cancellable: ?*compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Cancellable.Class, p_class).f_cancelled = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {
        /// Emitted when the operation has been cancelled.
        ///
        /// Can be used by implementations of cancellable operations. If the
        /// operation is cancelled from another thread, the signal will be
        /// emitted in the thread that cancelled the operation, not the
        /// thread that is running the operation.
        ///
        /// Note that disconnecting from this signal (or any signal) in a
        /// multi-threaded program is prone to race conditions. For instance
        /// it is possible that a signal handler may be invoked even after
        /// a call to `gobject.signalHandlerDisconnect` for that handler has
        /// already returned.
        ///
        /// There is also a problem when cancellation happens right before
        /// connecting to the signal. If this happens the signal will
        /// unexpectedly not be emitted, and checking before connecting to
        /// the signal leaves a race condition where this is still happening.
        ///
        /// In order to make it safe and easy to connect handlers there
        /// are two helper functions: `gio.Cancellable.connect` and
        /// `gio.Cancellable.disconnect` which protect against problems
        /// like this.
        ///
        /// An example of how to us this:
        /// ```
        ///     // Make sure we don't do unnecessary work if already cancelled
        ///     if (g_cancellable_set_error_if_cancelled (cancellable, error))
        ///       return;
        ///
        ///     // Set up all the data needed to be able to handle cancellation
        ///     // of the operation
        ///     my_data = my_data_new (...);
        ///
        ///     id = 0;
        ///     if (cancellable)
        ///       id = g_cancellable_connect (cancellable,
        ///                       G_CALLBACK (cancelled_handler)
        ///                       data, NULL);
        ///
        ///     // cancellable operation here...
        ///
        ///     g_cancellable_disconnect (cancellable, id);
        ///
        ///     // cancelled_handler is never called after this, it is now safe
        ///     // to free the data
        ///     my_data_free (my_data);
        /// ```
        ///
        /// Note that the cancelled signal is emitted in the thread that
        /// the user cancelled from, which may be the main thread. So, the
        /// cancellable signal should not do something that can block.
        pub const cancelled = struct {
            pub const name = "cancelled";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Cancellable, p_instance))),
                    gobject.signalLookup("cancelled", Cancellable.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Gets the top cancellable from the stack.
    extern fn g_cancellable_get_current() ?*gio.Cancellable;
    pub const getCurrent = g_cancellable_get_current;

    /// Creates a new `gio.Cancellable` object.
    ///
    /// Applications that want to start one or more operations
    /// that should be cancellable should create a `gio.Cancellable`
    /// and pass it to the operations.
    ///
    /// One `gio.Cancellable` can be used in multiple consecutive
    /// operations or in multiple concurrent operations.
    extern fn g_cancellable_new() *gio.Cancellable;
    pub const new = g_cancellable_new;

    /// Will set `cancellable` to cancelled, and will emit the
    /// `gio.Cancellable.signals.cancelled` signal. (However, see the warning about
    /// race conditions in the documentation for that signal if you are
    /// planning to connect to it.)
    ///
    /// This function is thread-safe. In other words, you can safely call
    /// it from a thread other than the one running the operation that was
    /// passed the `cancellable`.
    ///
    /// If `cancellable` is `NULL`, this function returns immediately for convenience.
    ///
    /// The convention within GIO is that cancelling an asynchronous
    /// operation causes it to complete asynchronously. That is, if you
    /// cancel the operation from the same thread in which it is running,
    /// then the operation's `gio.AsyncReadyCallback` will not be invoked until
    /// the application returns to the main loop.
    extern fn g_cancellable_cancel(p_cancellable: ?*Cancellable) void;
    pub const cancel = g_cancellable_cancel;

    /// Convenience function to connect to the `gio.Cancellable.signals.cancelled`
    /// signal. Also handles the race condition that may happen
    /// if the cancellable is cancelled right before connecting.
    ///
    /// `callback` is called at most once, either directly at the
    /// time of the connect if `cancellable` is already cancelled,
    /// or when `cancellable` is cancelled in some thread.
    ///
    /// `data_destroy_func` will be called when the handler is
    /// disconnected, or immediately if the cancellable is already
    /// cancelled.
    ///
    /// See `gio.Cancellable.signals.cancelled` for details on how to use this.
    ///
    /// Since GLib 2.40, the lock protecting `cancellable` is not held when
    /// `callback` is invoked.  This lifts a restriction in place for
    /// earlier GLib versions which now makes it easier to write cleanup
    /// code that unconditionally invokes e.g. `gio.Cancellable.cancel`.
    extern fn g_cancellable_connect(p_cancellable: ?*Cancellable, p_callback: gobject.Callback, p_data: ?*anyopaque, p_data_destroy_func: ?glib.DestroyNotify) c_ulong;
    pub const connect = g_cancellable_connect;

    /// Disconnects a handler from a cancellable instance similar to
    /// `gobject.signalHandlerDisconnect`.  Additionally, in the event that a
    /// signal handler is currently running, this call will block until the
    /// handler has finished.  Calling this function from a
    /// `gio.Cancellable.signals.cancelled` signal handler will therefore result in a
    /// deadlock.
    ///
    /// This avoids a race condition where a thread cancels at the
    /// same time as the cancellable operation is finished and the
    /// signal handler is removed. See `gio.Cancellable.signals.cancelled` for
    /// details on how to use this.
    ///
    /// If `cancellable` is `NULL` or `handler_id` is `0` this function does
    /// nothing.
    extern fn g_cancellable_disconnect(p_cancellable: ?*Cancellable, p_handler_id: c_ulong) void;
    pub const disconnect = g_cancellable_disconnect;

    /// Gets the file descriptor for a cancellable job. This can be used to
    /// implement cancellable operations on Unix systems. The returned fd will
    /// turn readable when `cancellable` is cancelled.
    ///
    /// You are not supposed to read from the fd yourself, just check for
    /// readable status. Reading to unset the readable status is done
    /// with `gio.Cancellable.reset`.
    ///
    /// After a successful return from this function, you should use
    /// `gio.Cancellable.releaseFd` to free up resources allocated for
    /// the returned file descriptor.
    ///
    /// See also `gio.Cancellable.makePollfd`.
    extern fn g_cancellable_get_fd(p_cancellable: ?*Cancellable) c_int;
    pub const getFd = g_cancellable_get_fd;

    /// Checks if a cancellable job has been cancelled.
    extern fn g_cancellable_is_cancelled(p_cancellable: ?*Cancellable) c_int;
    pub const isCancelled = g_cancellable_is_cancelled;

    /// Creates a `glib.PollFD` corresponding to `cancellable`; this can be passed
    /// to `glib.poll` and used to poll for cancellation. This is useful both
    /// for unix systems without a native poll and for portability to
    /// windows.
    ///
    /// When this function returns `TRUE`, you should use
    /// `gio.Cancellable.releaseFd` to free up resources allocated for the
    /// `pollfd`. After a `FALSE` return, do not call `gio.Cancellable.releaseFd`.
    ///
    /// If this function returns `FALSE`, either no `cancellable` was given or
    /// resource limits prevent this function from allocating the necessary
    /// structures for polling. (On Linux, you will likely have reached
    /// the maximum number of file descriptors.) The suggested way to handle
    /// these cases is to ignore the `cancellable`.
    ///
    /// You are not supposed to read from the fd yourself, just check for
    /// readable status. Reading to unset the readable status is done
    /// with `gio.Cancellable.reset`.
    extern fn g_cancellable_make_pollfd(p_cancellable: ?*Cancellable, p_pollfd: *glib.PollFD) c_int;
    pub const makePollfd = g_cancellable_make_pollfd;

    /// Pops `cancellable` off the cancellable stack (verifying that `cancellable`
    /// is on the top of the stack).
    extern fn g_cancellable_pop_current(p_cancellable: ?*Cancellable) void;
    pub const popCurrent = g_cancellable_pop_current;

    /// Pushes `cancellable` onto the cancellable stack. The current
    /// cancellable can then be received using `gio.Cancellable.getCurrent`.
    ///
    /// This is useful when implementing cancellable operations in
    /// code that does not allow you to pass down the cancellable object.
    ///
    /// This is typically called automatically by e.g. `gio.File` operations,
    /// so you rarely have to call this yourself.
    extern fn g_cancellable_push_current(p_cancellable: ?*Cancellable) void;
    pub const pushCurrent = g_cancellable_push_current;

    /// Releases a resources previously allocated by `gio.Cancellable.getFd`
    /// or `gio.Cancellable.makePollfd`.
    ///
    /// For compatibility reasons with older releases, calling this function
    /// is not strictly required, the resources will be automatically freed
    /// when the `cancellable` is finalized. However, the `cancellable` will
    /// block scarce file descriptors until it is finalized if this function
    /// is not called. This can cause the application to run out of file
    /// descriptors when many `GCancellables` are used at the same time.
    extern fn g_cancellable_release_fd(p_cancellable: ?*Cancellable) void;
    pub const releaseFd = g_cancellable_release_fd;

    /// Resets `cancellable` to its uncancelled state.
    ///
    /// If cancellable is currently in use by any cancellable operation
    /// then the behavior of this function is undefined.
    ///
    /// Note that it is generally not a good idea to reuse an existing
    /// cancellable for more operations after it has been cancelled once,
    /// as this function might tempt you to do. The recommended practice
    /// is to drop the reference to a cancellable after cancelling it,
    /// and let it die with the outstanding async operations. You should
    /// create a fresh cancellable for further async operations.
    extern fn g_cancellable_reset(p_cancellable: ?*Cancellable) void;
    pub const reset = g_cancellable_reset;

    /// If the `cancellable` is cancelled, sets the error to notify
    /// that the operation was cancelled.
    extern fn g_cancellable_set_error_if_cancelled(p_cancellable: ?*Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const setErrorIfCancelled = g_cancellable_set_error_if_cancelled;

    /// Creates a source that triggers if `cancellable` is cancelled and
    /// calls its callback of type `gio.CancellableSourceFunc`. This is
    /// primarily useful for attaching to another (non-cancellable) source
    /// with `glib.Source.addChildSource` to add cancellability to it.
    ///
    /// For convenience, you can call this with a `NULL` `gio.Cancellable`,
    /// in which case the source will never trigger.
    ///
    /// The new `glib.Source` will hold a reference to the `gio.Cancellable`.
    extern fn g_cancellable_source_new(p_cancellable: ?*Cancellable) *glib.Source;
    pub const sourceNew = g_cancellable_source_new;

    extern fn g_cancellable_get_type() usize;
    pub const getGObjectType = g_cancellable_get_type;

    extern fn g_object_ref(p_self: *gio.Cancellable) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.Cancellable) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Cancellable, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GCharsetConverter` is an implementation of `gio.Converter` based on
/// `glib.IConv`.
pub const CharsetConverter = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{ gio.Converter, gio.Initable };
    pub const Class = gio.CharsetConverterClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The character encoding to convert from.
        pub const from_charset = struct {
            pub const name = "from-charset";

            pub const Type = ?[*:0]u8;
        };

        /// The character encoding to convert to.
        pub const to_charset = struct {
            pub const name = "to-charset";

            pub const Type = ?[*:0]u8;
        };

        /// Use fallback (of form `\<hexval>`) for invalid bytes.
        pub const use_fallback = struct {
            pub const name = "use-fallback";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gio.CharsetConverter`.
    extern fn g_charset_converter_new(p_to_charset: [*:0]const u8, p_from_charset: [*:0]const u8, p_error: ?*?*glib.Error) ?*gio.CharsetConverter;
    pub const new = g_charset_converter_new;

    /// Gets the number of fallbacks that `converter` has applied so far.
    extern fn g_charset_converter_get_num_fallbacks(p_converter: *CharsetConverter) c_uint;
    pub const getNumFallbacks = g_charset_converter_get_num_fallbacks;

    /// Gets the `gio.CharsetConverter.properties.use`-fallback property.
    extern fn g_charset_converter_get_use_fallback(p_converter: *CharsetConverter) c_int;
    pub const getUseFallback = g_charset_converter_get_use_fallback;

    /// Sets the `gio.CharsetConverter.properties.use`-fallback property.
    extern fn g_charset_converter_set_use_fallback(p_converter: *CharsetConverter, p_use_fallback: c_int) void;
    pub const setUseFallback = g_charset_converter_set_use_fallback;

    extern fn g_charset_converter_get_type() usize;
    pub const getGObjectType = g_charset_converter_get_type;

    extern fn g_object_ref(p_self: *gio.CharsetConverter) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.CharsetConverter) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CharsetConverter, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Converter input stream implements `gio.InputStream` and allows
/// conversion of data of various types during reading.
///
/// As of GLib 2.34, `GConverterInputStream` implements
/// `gio.PollableInputStream`.
pub const ConverterInputStream = extern struct {
    pub const Parent = gio.FilterInputStream;
    pub const Implements = [_]type{gio.PollableInputStream};
    pub const Class = gio.ConverterInputStreamClass;
    f_parent_instance: gio.FilterInputStream,
    f_priv: ?*gio.ConverterInputStreamPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The converter object.
        pub const converter = struct {
            pub const name = "converter";

            pub const Type = ?*gio.Converter;
        };
    };

    pub const signals = struct {};

    /// Creates a new converter input stream for the `base_stream`.
    extern fn g_converter_input_stream_new(p_base_stream: *gio.InputStream, p_converter: *gio.Converter) *gio.ConverterInputStream;
    pub const new = g_converter_input_stream_new;

    /// Gets the `gio.Converter` that is used by `converter_stream`.
    extern fn g_converter_input_stream_get_converter(p_converter_stream: *ConverterInputStream) *gio.Converter;
    pub const getConverter = g_converter_input_stream_get_converter;

    extern fn g_converter_input_stream_get_type() usize;
    pub const getGObjectType = g_converter_input_stream_get_type;

    extern fn g_object_ref(p_self: *gio.ConverterInputStream) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.ConverterInputStream) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ConverterInputStream, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Converter output stream implements `gio.OutputStream` and allows
/// conversion of data of various types during reading.
///
/// As of GLib 2.34, `GConverterOutputStream` implements
/// `gio.PollableOutputStream`.
pub const ConverterOutputStream = extern struct {
    pub const Parent = gio.FilterOutputStream;
    pub const Implements = [_]type{gio.PollableOutputStream};
    pub const Class = gio.ConverterOutputStreamClass;
    f_parent_instance: gio.FilterOutputStream,
    f_priv: ?*gio.ConverterOutputStreamPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The converter object.
        pub const converter = struct {
            pub const name = "converter";

            pub const Type = ?*gio.Converter;
        };
    };

    pub const signals = struct {};

    /// Creates a new converter output stream for the `base_stream`.
    extern fn g_converter_output_stream_new(p_base_stream: *gio.OutputStream, p_converter: *gio.Converter) *gio.ConverterOutputStream;
    pub const new = g_converter_output_stream_new;

    /// Gets the `gio.Converter` that is used by `converter_stream`.
    extern fn g_converter_output_stream_get_converter(p_converter_stream: *ConverterOutputStream) *gio.Converter;
    pub const getConverter = g_converter_output_stream_get_converter;

    extern fn g_converter_output_stream_get_type() usize;
    pub const getGObjectType = g_converter_output_stream_get_type;

    extern fn g_object_ref(p_self: *gio.ConverterOutputStream) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.ConverterOutputStream) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ConverterOutputStream, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `GCredentials` type is a reference-counted wrapper for native
/// credentials.
///
/// The information in `GCredentials` is typically used for identifying,
/// authenticating and authorizing other processes.
///
/// Some operating systems supports looking up the credentials of the remote
/// peer of a communication endpoint - see e.g. `gio.Socket.getCredentials`.
///
/// Some operating systems supports securely sending and receiving
/// credentials over a Unix Domain Socket, see `gio.UnixCredentialsMessage`,
/// `gio.UnixConnection.sendCredentials` and
/// `gio.UnixConnection.receiveCredentials` for details.
///
/// On Linux, the native credential type is a `struct ucred` - see the
/// [`unix(7)` man page](man:unix(7)) for details. This corresponds to
/// `G_CREDENTIALS_TYPE_LINUX_UCRED`.
///
/// On Apple operating systems (including iOS, tvOS, and macOS), the native credential
/// type is a `struct xucred`. This corresponds to `G_CREDENTIALS_TYPE_APPLE_XUCRED`.
///
/// On FreeBSD, Debian GNU/kFreeBSD, and GNU/Hurd, the native credential type is a
/// `struct cmsgcred`. This corresponds to `G_CREDENTIALS_TYPE_FREEBSD_CMSGCRED`.
///
/// On NetBSD, the native credential type is a `struct unpcbid`.
/// This corresponds to `G_CREDENTIALS_TYPE_NETBSD_UNPCBID`.
///
/// On OpenBSD, the native credential type is a `struct sockpeercred`.
/// This corresponds to `G_CREDENTIALS_TYPE_OPENBSD_SOCKPEERCRED`.
///
/// On Solaris (including OpenSolaris and its derivatives), the native credential type
/// is a `ucred_t`. This corresponds to `G_CREDENTIALS_TYPE_SOLARIS_UCRED`.
///
/// Since GLib 2.72, on Windows, the native credentials may contain the PID of a
/// process. This corresponds to `G_CREDENTIALS_TYPE_WIN32_PID`.
pub const Credentials = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.CredentialsClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `gio.Credentials` object with credentials matching the
    /// the current process.
    extern fn g_credentials_new() *gio.Credentials;
    pub const new = g_credentials_new;

    /// Gets a pointer to native credentials of type `native_type` from
    /// `credentials`.
    ///
    /// It is a programming error (which will cause a warning to be
    /// logged) to use this method if there is no `gio.Credentials` support for
    /// the OS or if `native_type` isn't supported by the OS.
    extern fn g_credentials_get_native(p_credentials: *Credentials, p_native_type: gio.CredentialsType) ?*anyopaque;
    pub const getNative = g_credentials_get_native;

    /// Tries to get the UNIX process identifier from `credentials`. This
    /// method is only available on UNIX platforms.
    ///
    /// This operation can fail if `gio.Credentials` is not supported on the
    /// OS or if the native credentials type does not contain information
    /// about the UNIX process ID.
    extern fn g_credentials_get_unix_pid(p_credentials: *Credentials, p_error: ?*?*glib.Error) std.posix.pid_t;
    pub const getUnixPid = g_credentials_get_unix_pid;

    /// Tries to get the UNIX user identifier from `credentials`. This
    /// method is only available on UNIX platforms.
    ///
    /// This operation can fail if `gio.Credentials` is not supported on the
    /// OS or if the native credentials type does not contain information
    /// about the UNIX user.
    extern fn g_credentials_get_unix_user(p_credentials: *Credentials, p_error: ?*?*glib.Error) std.posix.uid_t;
    pub const getUnixUser = g_credentials_get_unix_user;

    /// Checks if `credentials` and `other_credentials` is the same user.
    ///
    /// This operation can fail if `gio.Credentials` is not supported on the
    /// the OS.
    extern fn g_credentials_is_same_user(p_credentials: *Credentials, p_other_credentials: *gio.Credentials, p_error: ?*?*glib.Error) c_int;
    pub const isSameUser = g_credentials_is_same_user;

    /// Copies the native credentials of type `native_type` from `native`
    /// into `credentials`.
    ///
    /// It is a programming error (which will cause a warning to be
    /// logged) to use this method if there is no `gio.Credentials` support for
    /// the OS or if `native_type` isn't supported by the OS.
    extern fn g_credentials_set_native(p_credentials: *Credentials, p_native_type: gio.CredentialsType, p_native: *anyopaque) void;
    pub const setNative = g_credentials_set_native;

    /// Tries to set the UNIX user identifier on `credentials`. This method
    /// is only available on UNIX platforms.
    ///
    /// This operation can fail if `gio.Credentials` is not supported on the
    /// OS or if the native credentials type does not contain information
    /// about the UNIX user. It can also fail if the OS does not allow the
    /// use of "spoofed" credentials.
    extern fn g_credentials_set_unix_user(p_credentials: *Credentials, p_uid: std.posix.uid_t, p_error: ?*?*glib.Error) c_int;
    pub const setUnixUser = g_credentials_set_unix_user;

    /// Creates a human-readable textual representation of `credentials`
    /// that can be used in logging and debug messages. The format of the
    /// returned string may change in future GLib release.
    extern fn g_credentials_to_string(p_credentials: *Credentials) [*:0]u8;
    pub const toString = g_credentials_to_string;

    extern fn g_credentials_get_type() usize;
    pub const getGObjectType = g_credentials_get_type;

    extern fn g_object_ref(p_self: *gio.Credentials) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.Credentials) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Credentials, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GDBusActionGroup` is an implementation of the `gio.ActionGroup`
/// interface.
///
/// `GDBusActionGroup` can be used as a proxy for an action group
/// that is exported over D-Bus with `gio.DBusConnection.exportActionGroup`.
pub const DBusActionGroup = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{ gio.ActionGroup, gio.RemoteActionGroup };
    pub const Class = opaque {
        pub const Instance = DBusActionGroup;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Obtains a `gio.DBusActionGroup` for the action group which is exported at
    /// the given `bus_name` and `object_path`.
    ///
    /// The thread default main context is taken at the time of this call.
    /// All signals on the menu model (and any linked models) are reported
    /// with respect to this context.  All calls on the returned menu model
    /// (and linked models) must also originate from this same context, with
    /// the thread default main context unchanged.
    ///
    /// This call is non-blocking.  The returned action group may or may not
    /// already be filled in.  The correct thing to do is connect the signals
    /// for the action group to monitor for changes and then to call
    /// `gio.ActionGroup.listActions` to get the initial list.
    extern fn g_dbus_action_group_get(p_connection: *gio.DBusConnection, p_bus_name: ?[*:0]const u8, p_object_path: [*:0]const u8) *gio.DBusActionGroup;
    pub const get = g_dbus_action_group_get;

    extern fn g_dbus_action_group_get_type() usize;
    pub const getGObjectType = g_dbus_action_group_get_type;

    extern fn g_object_ref(p_self: *gio.DBusActionGroup) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.DBusActionGroup) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *DBusActionGroup, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GDBusAuthObserver` provides a mechanism for participating
/// in how a `gio.DBusServer` (or a `gio.DBusConnection`)
/// authenticates remote peers.
///
/// Simply instantiate a `GDBusAuthObserver` and connect to the
/// signals you are interested in. Note that new signals may be added
/// in the future.
///
/// ## Controlling Authentication Mechanisms
///
/// By default, a `GDBusServer` or server-side `GDBusConnection` will allow
/// any authentication mechanism to be used. If you only want to allow D-Bus
/// connections with the `EXTERNAL` mechanism, which makes use of credentials
/// passing and is the recommended mechanism for modern Unix platforms such
/// as Linux and the BSD family, you would use a signal handler like this:
///
/// ```c
/// static gboolean
/// on_allow_mechanism (GDBusAuthObserver *observer,
///                     const gchar       *mechanism,
///                     gpointer           user_data)
/// {
///   if (g_strcmp0 (mechanism, "EXTERNAL") == 0)
///     {
///       return TRUE;
///     }
///
///   return FALSE;
/// }
/// ```
///
/// ## Controlling Authorization
///
/// By default, a `GDBusServer` or server-side `GDBusConnection` will accept
/// connections from any successfully authenticated user (but not from
/// anonymous connections using the `ANONYMOUS` mechanism). If you only
/// want to allow D-Bus connections from processes owned by the same uid
/// as the server, since GLib 2.68, you should use the
/// `G_DBUS_SERVER_FLAGS_AUTHENTICATION_REQUIRE_SAME_USER` flag. It’s equivalent
/// to the following signal handler:
///
/// ```c
/// static gboolean
/// on_authorize_authenticated_peer (GDBusAuthObserver *observer,
///                                  GIOStream         *stream,
///                                  GCredentials      *credentials,
///                                  gpointer           user_data)
/// {
///   gboolean authorized;
///
///   authorized = FALSE;
///   if (credentials != NULL)
///     {
///       GCredentials *own_credentials;
///       own_credentials = g_credentials_new ();
///       if (g_credentials_is_same_user (credentials, own_credentials, NULL))
///         authorized = TRUE;
///       g_object_unref (own_credentials);
///     }
///
///   return authorized;
/// }
/// ```
pub const DBusAuthObserver = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = opaque {
        pub const Instance = DBusAuthObserver;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {
        /// Emitted to check if `mechanism` is allowed to be used.
        pub const allow_mechanism = struct {
            pub const name = "allow-mechanism";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_mechanism: [*:0]u8, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(DBusAuthObserver, p_instance))),
                    gobject.signalLookup("allow-mechanism", DBusAuthObserver.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted to check if a peer that is successfully authenticated
        /// is authorized.
        pub const authorize_authenticated_peer = struct {
            pub const name = "authorize-authenticated-peer";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_stream: *gio.IOStream, p_credentials: ?*gio.Credentials, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(DBusAuthObserver, p_instance))),
                    gobject.signalLookup("authorize-authenticated-peer", DBusAuthObserver.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gio.DBusAuthObserver` object.
    extern fn g_dbus_auth_observer_new() *gio.DBusAuthObserver;
    pub const new = g_dbus_auth_observer_new;

    /// Emits the `gio.DBusAuthObserver.signals.allow`-mechanism signal on `observer`.
    extern fn g_dbus_auth_observer_allow_mechanism(p_observer: *DBusAuthObserver, p_mechanism: [*:0]const u8) c_int;
    pub const allowMechanism = g_dbus_auth_observer_allow_mechanism;

    /// Emits the `gio.DBusAuthObserver.signals.authorize`-authenticated-peer signal on `observer`.
    extern fn g_dbus_auth_observer_authorize_authenticated_peer(p_observer: *DBusAuthObserver, p_stream: *gio.IOStream, p_credentials: ?*gio.Credentials) c_int;
    pub const authorizeAuthenticatedPeer = g_dbus_auth_observer_authorize_authenticated_peer;

    extern fn g_dbus_auth_observer_get_type() usize;
    pub const getGObjectType = g_dbus_auth_observer_get_type;

    extern fn g_object_ref(p_self: *gio.DBusAuthObserver) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.DBusAuthObserver) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *DBusAuthObserver, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `GDBusConnection` type is used for D-Bus connections to remote
/// peers such as a message buses.
///
/// It is a low-level API that offers a lot of flexibility. For instance,
/// it lets you establish a connection over any transport that can by represented
/// as a `gio.IOStream`.
///
/// This class is rarely used directly in D-Bus clients. If you are writing
/// a D-Bus client, it is often easier to use the `gio.busOwnName`,
/// `gio.busWatchName` or `gio.DBusProxy.newForBus` APIs.
///
/// As an exception to the usual GLib rule that a particular object must not
/// be used by two threads at the same time, `GDBusConnection`s methods may be
/// called from any thread. This is so that `gio.busGet` and
/// `gio.busGetSync` can safely return the same `GDBusConnection` when
/// called from any thread.
///
/// Most of the ways to obtain a `GDBusConnection` automatically initialize it
/// (i.e. connect to D-Bus): for instance, `gio.DBusConnection.new` and
/// `gio.busGet`, and the synchronous versions of those methods, give you
/// an initialized connection. Language bindings for GIO should use
/// `gio.Initable.new` or `gio.AsyncInitable.newAsync`, which also
/// initialize the connection.
///
/// If you construct an uninitialized `GDBusConnection`, such as via
/// `gobject.Object.new`, you must initialize it via `gio.Initable.init` or
/// `gio.AsyncInitable.initAsync` before using its methods or properties.
/// Calling methods or accessing properties on a `GDBusConnection` that has not
/// completed initialization successfully is considered to be invalid, and leads
/// to undefined behaviour. In particular, if initialization fails with a
/// `GError`, the only valid thing you can do with that `GDBusConnection` is to
/// free it with `gobject.Object.unref`.
///
/// ## An example D-Bus server
///
/// Here is an example for a D-Bus server:
/// [gdbus-example-server.c](https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gdbus-example-server.c)
///
/// ## An example for exporting a subtree
///
/// Here is an example for exporting a subtree:
/// [gdbus-example-subtree.c](https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gdbus-example-subtree.c)
///
/// ## An example for file descriptor passing
///
/// Here is an example for passing UNIX file descriptors:
/// [gdbus-unix-fd-client.c](https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gdbus-example-unix-fd-client.c)
///
/// ## An example for exporting a GObject
///
/// Here is an example for exporting a `gobject.Object`:
/// [gdbus-example-export.c](https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gdbus-example-export.c)
pub const DBusConnection = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{ gio.AsyncInitable, gio.Initable };
    pub const Class = opaque {
        pub const Instance = DBusConnection;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// A D-Bus address specifying potential endpoints that can be used
        /// when establishing the connection.
        pub const address = struct {
            pub const name = "address";

            pub const Type = ?[*:0]u8;
        };

        /// A `gio.DBusAuthObserver` object to assist in the authentication process or `NULL`.
        pub const authentication_observer = struct {
            pub const name = "authentication-observer";

            pub const Type = ?*gio.DBusAuthObserver;
        };

        /// Flags from the `gio.DBusCapabilityFlags` enumeration
        /// representing connection features negotiated with the other peer.
        pub const capabilities = struct {
            pub const name = "capabilities";

            pub const Type = gio.DBusCapabilityFlags;
        };

        /// A boolean specifying whether the connection has been closed.
        pub const closed = struct {
            pub const name = "closed";

            pub const Type = c_int;
        };

        /// A boolean specifying whether the process will be terminated (by
        /// calling `raise(SIGTERM)`) if the connection is closed by the
        /// remote peer.
        ///
        /// Note that `gio.DBusConnection` objects returned by `gio.busGetFinish`
        /// and `gio.busGetSync` will (usually) have this property set to `TRUE`.
        pub const exit_on_close = struct {
            pub const name = "exit-on-close";

            pub const Type = c_int;
        };

        /// Flags from the `gio.DBusConnectionFlags` enumeration.
        pub const flags = struct {
            pub const name = "flags";

            pub const Type = gio.DBusConnectionFlags;
        };

        /// The GUID of the peer performing the role of server when
        /// authenticating.
        ///
        /// If you are constructing a `gio.DBusConnection` and pass
        /// `G_DBUS_CONNECTION_FLAGS_AUTHENTICATION_SERVER` in the
        /// `gio.DBusConnection.properties.flags` property then you **must** also set this
        /// property to a valid guid.
        ///
        /// If you are constructing a `gio.DBusConnection` and pass
        /// `G_DBUS_CONNECTION_FLAGS_AUTHENTICATION_CLIENT` in the
        /// `gio.DBusConnection.properties.flags` property you will be able to read the GUID
        /// of the other peer here after the connection has been successfully
        /// initialized.
        ///
        /// Note that the
        /// [D-Bus specification](https://dbus.freedesktop.org/doc/dbus-specification.html`addresses`)
        /// uses the term ‘UUID’ to refer to this, whereas GLib consistently uses the
        /// term ‘GUID’ for historical reasons.
        ///
        /// Despite its name, the format of `gio.DBusConnection.properties.guid` does not follow
        /// [RFC 4122](https://datatracker.ietf.org/doc/html/rfc4122) or the Microsoft
        /// GUID format.
        pub const guid = struct {
            pub const name = "guid";

            pub const Type = ?[*:0]u8;
        };

        /// The underlying `gio.IOStream` used for I/O.
        ///
        /// If this is passed on construction and is a `gio.SocketConnection`,
        /// then the corresponding `gio.Socket` will be put into non-blocking mode.
        ///
        /// While the `gio.DBusConnection` is active, it will interact with this
        /// stream from a worker thread, so it is not safe to interact with
        /// the stream directly.
        pub const stream = struct {
            pub const name = "stream";

            pub const Type = ?*gio.IOStream;
        };

        /// The unique name as assigned by the message bus or `NULL` if the
        /// connection is not open or not a message bus connection.
        pub const unique_name = struct {
            pub const name = "unique-name";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {
        /// Emitted when the connection is closed.
        ///
        /// The cause of this event can be
        ///
        /// - If `gio.DBusConnection.close` is called. In this case
        ///   `remote_peer_vanished` is set to `FALSE` and `error` is `NULL`.
        ///
        /// - If the remote peer closes the connection. In this case
        ///   `remote_peer_vanished` is set to `TRUE` and `error` is set.
        ///
        /// - If the remote peer sends invalid or malformed data. In this
        ///   case `remote_peer_vanished` is set to `FALSE` and `error` is set.
        ///
        /// Upon receiving this signal, you should give up your reference to
        /// `connection`. You are guaranteed that this signal is emitted only
        /// once.
        pub const closed = struct {
            pub const name = "closed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_remote_peer_vanished: c_int, p_error: ?*glib.Error, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(DBusConnection, p_instance))),
                    gobject.signalLookup("closed", DBusConnection.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Asynchronously sets up a D-Bus connection for exchanging D-Bus messages
    /// with the end represented by `stream`.
    ///
    /// If `stream` is a `gio.SocketConnection`, then the corresponding `gio.Socket`
    /// will be put into non-blocking mode.
    ///
    /// The D-Bus connection will interact with `stream` from a worker thread.
    /// As a result, the caller should not interact with `stream` after this
    /// method has been called, except by calling `gobject.Object.unref` on it.
    ///
    /// If `observer` is not `NULL` it may be used to control the
    /// authentication process.
    ///
    /// When the operation is finished, `callback` will be invoked. You can
    /// then call `gio.DBusConnection.newFinish` to get the result of the
    /// operation.
    ///
    /// This is an asynchronous failable constructor. See
    /// `gio.DBusConnection.newSync` for the synchronous
    /// version.
    extern fn g_dbus_connection_new(p_stream: *gio.IOStream, p_guid: ?[*:0]const u8, p_flags: gio.DBusConnectionFlags, p_observer: ?*gio.DBusAuthObserver, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const new = g_dbus_connection_new;

    /// Asynchronously connects and sets up a D-Bus client connection for
    /// exchanging D-Bus messages with an endpoint specified by `address`
    /// which must be in the
    /// [D-Bus address format](https://dbus.freedesktop.org/doc/dbus-specification.html`addresses`).
    ///
    /// This constructor can only be used to initiate client-side
    /// connections - use `gio.DBusConnection.new` if you need to act as the
    /// server. In particular, `flags` cannot contain the
    /// `G_DBUS_CONNECTION_FLAGS_AUTHENTICATION_SERVER`,
    /// `G_DBUS_CONNECTION_FLAGS_AUTHENTICATION_ALLOW_ANONYMOUS` or
    /// `G_DBUS_CONNECTION_FLAGS_AUTHENTICATION_REQUIRE_SAME_USER` flags.
    ///
    /// When the operation is finished, `callback` will be invoked. You can
    /// then call `gio.DBusConnection.newForAddressFinish` to get the result of
    /// the operation.
    ///
    /// If `observer` is not `NULL` it may be used to control the
    /// authentication process.
    ///
    /// This is an asynchronous failable constructor. See
    /// `gio.DBusConnection.newForAddressSync` for the synchronous
    /// version.
    extern fn g_dbus_connection_new_for_address(p_address: [*:0]const u8, p_flags: gio.DBusConnectionFlags, p_observer: ?*gio.DBusAuthObserver, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const newForAddress = g_dbus_connection_new_for_address;

    /// Finishes an operation started with `gio.DBusConnection.new`.
    extern fn g_dbus_connection_new_finish(p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.DBusConnection;
    pub const newFinish = g_dbus_connection_new_finish;

    /// Finishes an operation started with `gio.DBusConnection.newForAddress`.
    extern fn g_dbus_connection_new_for_address_finish(p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.DBusConnection;
    pub const newForAddressFinish = g_dbus_connection_new_for_address_finish;

    /// Synchronously connects and sets up a D-Bus client connection for
    /// exchanging D-Bus messages with an endpoint specified by `address`
    /// which must be in the
    /// [D-Bus address format](https://dbus.freedesktop.org/doc/dbus-specification.html`addresses`).
    ///
    /// This constructor can only be used to initiate client-side
    /// connections - use `gio.DBusConnection.newSync` if you need to act
    /// as the server. In particular, `flags` cannot contain the
    /// `G_DBUS_CONNECTION_FLAGS_AUTHENTICATION_SERVER`,
    /// `G_DBUS_CONNECTION_FLAGS_AUTHENTICATION_ALLOW_ANONYMOUS` or
    /// `G_DBUS_CONNECTION_FLAGS_AUTHENTICATION_REQUIRE_SAME_USER` flags.
    ///
    /// This is a synchronous failable constructor. See
    /// `gio.DBusConnection.newForAddress` for the asynchronous version.
    ///
    /// If `observer` is not `NULL` it may be used to control the
    /// authentication process.
    extern fn g_dbus_connection_new_for_address_sync(p_address: [*:0]const u8, p_flags: gio.DBusConnectionFlags, p_observer: ?*gio.DBusAuthObserver, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.DBusConnection;
    pub const newForAddressSync = g_dbus_connection_new_for_address_sync;

    /// Synchronously sets up a D-Bus connection for exchanging D-Bus messages
    /// with the end represented by `stream`.
    ///
    /// If `stream` is a `gio.SocketConnection`, then the corresponding `gio.Socket`
    /// will be put into non-blocking mode.
    ///
    /// The D-Bus connection will interact with `stream` from a worker thread.
    /// As a result, the caller should not interact with `stream` after this
    /// method has been called, except by calling `gobject.Object.unref` on it.
    ///
    /// If `observer` is not `NULL` it may be used to control the
    /// authentication process.
    ///
    /// This is a synchronous failable constructor. See
    /// `gio.DBusConnection.new` for the asynchronous version.
    extern fn g_dbus_connection_new_sync(p_stream: *gio.IOStream, p_guid: ?[*:0]const u8, p_flags: gio.DBusConnectionFlags, p_observer: ?*gio.DBusAuthObserver, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.DBusConnection;
    pub const newSync = g_dbus_connection_new_sync;

    /// Adds a message filter. Filters are handlers that are run on all
    /// incoming and outgoing messages, prior to standard dispatch. Filters
    /// are run in the order that they were added.  The same handler can be
    /// added as a filter more than once, in which case it will be run more
    /// than once.  Filters added during a filter callback won't be run on
    /// the message being processed. Filter functions are allowed to modify
    /// and even drop messages.
    ///
    /// Note that filters are run in a dedicated message handling thread so
    /// they can't block and, generally, can't do anything but signal a
    /// worker thread. Also note that filters are rarely needed - use API
    /// such as `gio.DBusConnection.sendMessageWithReply`,
    /// `gio.DBusConnection.signalSubscribe` or `gio.DBusConnection.call` instead.
    ///
    /// If a filter consumes an incoming message the message is not
    /// dispatched anywhere else - not even the standard dispatch machinery
    /// (that API such as `gio.DBusConnection.signalSubscribe` and
    /// `gio.DBusConnection.sendMessageWithReply` relies on) will see the
    /// message. Similarly, if a filter consumes an outgoing message, the
    /// message will not be sent to the other peer.
    ///
    /// If `user_data_free_func` is non-`NULL`, it will be called (in the
    /// thread-default main context of the thread you are calling this
    /// method from) at some point after `user_data` is no longer
    /// needed. (It is not guaranteed to be called synchronously when the
    /// filter is removed, and may be called after `connection` has been
    /// destroyed.)
    extern fn g_dbus_connection_add_filter(p_connection: *DBusConnection, p_filter_function: gio.DBusMessageFilterFunction, p_user_data: ?*anyopaque, p_user_data_free_func: ?glib.DestroyNotify) c_uint;
    pub const addFilter = g_dbus_connection_add_filter;

    /// Asynchronously invokes the `method_name` method on the
    /// `interface_name` D-Bus interface on the remote object at
    /// `object_path` owned by `bus_name`.
    ///
    /// If `connection` is closed then the operation will fail with
    /// `G_IO_ERROR_CLOSED`. If `cancellable` is canceled, the operation will
    /// fail with `G_IO_ERROR_CANCELLED`. If `parameters` contains a value
    /// not compatible with the D-Bus protocol, the operation fails with
    /// `G_IO_ERROR_INVALID_ARGUMENT`.
    ///
    /// If `reply_type` is non-`NULL` then the reply will be checked for having this type and an
    /// error will be raised if it does not match.  Said another way, if you give a `reply_type`
    /// then any non-`NULL` return value will be of this type. Unless it’s
    /// `G_VARIANT_TYPE_UNIT`, the `reply_type` will be a tuple containing one or more
    /// values.
    ///
    /// If the `parameters` `glib.Variant` is floating, it is consumed. This allows
    /// convenient 'inline' use of `glib.Variant.new`, e.g.:
    /// ```
    ///  g_dbus_connection_call (connection,
    ///                          "org.freedesktop.StringThings",
    ///                          "/org/freedesktop/StringThings",
    ///                          "org.freedesktop.StringThings",
    ///                          "TwoStrings",
    ///                          g_variant_new ("(ss)",
    ///                                         "Thing One",
    ///                                         "Thing Two"),
    ///                          NULL,
    ///                          G_DBUS_CALL_FLAGS_NONE,
    ///                          -1,
    ///                          NULL,
    ///                          (GAsyncReadyCallback) two_strings_done,
    ///                          NULL);
    /// ```
    ///
    /// This is an asynchronous method. When the operation is finished,
    /// `callback` will be invoked in the
    /// [thread-default main context][g-main-context-push-thread-default]
    /// of the thread you are calling this method from. You can then call
    /// `gio.DBusConnection.callFinish` to get the result of the operation.
    /// See `gio.DBusConnection.callSync` for the synchronous version of this
    /// function.
    ///
    /// If `callback` is `NULL` then the D-Bus method call message will be sent with
    /// the `G_DBUS_MESSAGE_FLAGS_NO_REPLY_EXPECTED` flag set.
    extern fn g_dbus_connection_call(p_connection: *DBusConnection, p_bus_name: ?[*:0]const u8, p_object_path: [*:0]const u8, p_interface_name: [*:0]const u8, p_method_name: [*:0]const u8, p_parameters: ?*glib.Variant, p_reply_type: ?*const glib.VariantType, p_flags: gio.DBusCallFlags, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const call = g_dbus_connection_call;

    /// Finishes an operation started with `gio.DBusConnection.call`.
    extern fn g_dbus_connection_call_finish(p_connection: *DBusConnection, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.Variant;
    pub const callFinish = g_dbus_connection_call_finish;

    /// Synchronously invokes the `method_name` method on the
    /// `interface_name` D-Bus interface on the remote object at
    /// `object_path` owned by `bus_name`.
    ///
    /// If `connection` is closed then the operation will fail with
    /// `G_IO_ERROR_CLOSED`. If `cancellable` is canceled, the
    /// operation will fail with `G_IO_ERROR_CANCELLED`. If `parameters`
    /// contains a value not compatible with the D-Bus protocol, the operation
    /// fails with `G_IO_ERROR_INVALID_ARGUMENT`.
    ///
    /// If `reply_type` is non-`NULL` then the reply will be checked for having
    /// this type and an error will be raised if it does not match.  Said
    /// another way, if you give a `reply_type` then any non-`NULL` return
    /// value will be of this type.
    ///
    /// If the `parameters` `glib.Variant` is floating, it is consumed.
    /// This allows convenient 'inline' use of `glib.Variant.new`, e.g.:
    /// ```
    ///  g_dbus_connection_call_sync (connection,
    ///                               "org.freedesktop.StringThings",
    ///                               "/org/freedesktop/StringThings",
    ///                               "org.freedesktop.StringThings",
    ///                               "TwoStrings",
    ///                               g_variant_new ("(ss)",
    ///                                              "Thing One",
    ///                                              "Thing Two"),
    ///                               NULL,
    ///                               G_DBUS_CALL_FLAGS_NONE,
    ///                               -1,
    ///                               NULL,
    ///                               &error);
    /// ```
    ///
    /// The calling thread is blocked until a reply is received. See
    /// `gio.DBusConnection.call` for the asynchronous version of
    /// this method.
    extern fn g_dbus_connection_call_sync(p_connection: *DBusConnection, p_bus_name: ?[*:0]const u8, p_object_path: [*:0]const u8, p_interface_name: [*:0]const u8, p_method_name: [*:0]const u8, p_parameters: ?*glib.Variant, p_reply_type: ?*const glib.VariantType, p_flags: gio.DBusCallFlags, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*glib.Variant;
    pub const callSync = g_dbus_connection_call_sync;

    /// Like `gio.DBusConnection.call` but also takes a `gio.UnixFDList` object.
    ///
    /// The file descriptors normally correspond to `G_VARIANT_TYPE_HANDLE`
    /// values in the body of the message. For example, if a message contains
    /// two file descriptors, `fd_list` would have length 2, and
    /// `g_variant_new_handle (0)` and `g_variant_new_handle (1)` would appear
    /// somewhere in the body of the message (not necessarily in that order!)
    /// to represent the file descriptors at indexes 0 and 1 respectively.
    ///
    /// When designing D-Bus APIs that are intended to be interoperable,
    /// please note that non-GDBus implementations of D-Bus can usually only
    /// access file descriptors if they are referenced in this way by a
    /// value of type `G_VARIANT_TYPE_HANDLE` in the body of the message.
    ///
    /// This method is only available on UNIX.
    extern fn g_dbus_connection_call_with_unix_fd_list(p_connection: *DBusConnection, p_bus_name: ?[*:0]const u8, p_object_path: [*:0]const u8, p_interface_name: [*:0]const u8, p_method_name: [*:0]const u8, p_parameters: ?*glib.Variant, p_reply_type: ?*const glib.VariantType, p_flags: gio.DBusCallFlags, p_timeout_msec: c_int, p_fd_list: ?*gio.UnixFDList, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const callWithUnixFdList = g_dbus_connection_call_with_unix_fd_list;

    /// Finishes an operation started with `gio.DBusConnection.callWithUnixFdList`.
    ///
    /// The file descriptors normally correspond to `G_VARIANT_TYPE_HANDLE`
    /// values in the body of the message. For example,
    /// if `glib.Variant.getHandle` returns 5, that is intended to be a reference
    /// to the file descriptor that can be accessed by
    /// `g_unix_fd_list_get (*out_fd_list, 5, ...)`.
    ///
    /// When designing D-Bus APIs that are intended to be interoperable,
    /// please note that non-GDBus implementations of D-Bus can usually only
    /// access file descriptors if they are referenced in this way by a
    /// value of type `G_VARIANT_TYPE_HANDLE` in the body of the message.
    extern fn g_dbus_connection_call_with_unix_fd_list_finish(p_connection: *DBusConnection, p_out_fd_list: ?**gio.UnixFDList, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.Variant;
    pub const callWithUnixFdListFinish = g_dbus_connection_call_with_unix_fd_list_finish;

    /// Like `gio.DBusConnection.callSync` but also takes and returns `gio.UnixFDList` objects.
    /// See `gio.DBusConnection.callWithUnixFdList` and
    /// `gio.DBusConnection.callWithUnixFdListFinish` for more details.
    ///
    /// This method is only available on UNIX.
    extern fn g_dbus_connection_call_with_unix_fd_list_sync(p_connection: *DBusConnection, p_bus_name: ?[*:0]const u8, p_object_path: [*:0]const u8, p_interface_name: [*:0]const u8, p_method_name: [*:0]const u8, p_parameters: ?*glib.Variant, p_reply_type: ?*const glib.VariantType, p_flags: gio.DBusCallFlags, p_timeout_msec: c_int, p_fd_list: ?*gio.UnixFDList, p_out_fd_list: ?**gio.UnixFDList, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*glib.Variant;
    pub const callWithUnixFdListSync = g_dbus_connection_call_with_unix_fd_list_sync;

    /// Closes `connection`. Note that this never causes the process to
    /// exit (this might only happen if the other end of a shared message
    /// bus connection disconnects, see `gio.DBusConnection.properties.exit`-on-close).
    ///
    /// Once the connection is closed, operations such as sending a message
    /// will return with the error `G_IO_ERROR_CLOSED`. Closing a connection
    /// will not automatically flush the connection so queued messages may
    /// be lost. Use `gio.DBusConnection.flush` if you need such guarantees.
    ///
    /// If `connection` is already closed, this method fails with
    /// `G_IO_ERROR_CLOSED`.
    ///
    /// When `connection` has been closed, the `gio.DBusConnection.signals.closed`
    /// signal is emitted in the
    /// [thread-default main context][g-main-context-push-thread-default]
    /// of the thread that `connection` was constructed in.
    ///
    /// This is an asynchronous method. When the operation is finished,
    /// `callback` will be invoked in the
    /// [thread-default main context][g-main-context-push-thread-default]
    /// of the thread you are calling this method from. You can
    /// then call `gio.DBusConnection.closeFinish` to get the result of the
    /// operation. See `gio.DBusConnection.closeSync` for the synchronous
    /// version.
    extern fn g_dbus_connection_close(p_connection: *DBusConnection, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const close = g_dbus_connection_close;

    /// Finishes an operation started with `gio.DBusConnection.close`.
    extern fn g_dbus_connection_close_finish(p_connection: *DBusConnection, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const closeFinish = g_dbus_connection_close_finish;

    /// Synchronously closes `connection`. The calling thread is blocked
    /// until this is done. See `gio.DBusConnection.close` for the
    /// asynchronous version of this method and more details about what it
    /// does.
    extern fn g_dbus_connection_close_sync(p_connection: *DBusConnection, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const closeSync = g_dbus_connection_close_sync;

    /// Emits a signal.
    ///
    /// If the parameters GVariant is floating, it is consumed.
    ///
    /// This can only fail if `parameters` is not compatible with the D-Bus protocol
    /// (`G_IO_ERROR_INVALID_ARGUMENT`), or if `connection` has been closed
    /// (`G_IO_ERROR_CLOSED`).
    extern fn g_dbus_connection_emit_signal(p_connection: *DBusConnection, p_destination_bus_name: ?[*:0]const u8, p_object_path: [*:0]const u8, p_interface_name: [*:0]const u8, p_signal_name: [*:0]const u8, p_parameters: ?*glib.Variant, p_error: ?*?*glib.Error) c_int;
    pub const emitSignal = g_dbus_connection_emit_signal;

    /// Exports `action_group` on `connection` at `object_path`.
    ///
    /// The implemented D-Bus API should be considered private.  It is
    /// subject to change in the future.
    ///
    /// A given object path can only have one action group exported on it.
    /// If this constraint is violated, the export will fail and 0 will be
    /// returned (with `error` set accordingly).
    ///
    /// You can unexport the action group using
    /// `gio.DBusConnection.unexportActionGroup` with the return value of
    /// this function.
    ///
    /// The thread default main context is taken at the time of this call.
    /// All incoming action activations and state change requests are
    /// reported from this context.  Any changes on the action group that
    /// cause it to emit signals must also come from this same context.
    /// Since incoming action activations and state change requests are
    /// rather likely to cause changes on the action group, this effectively
    /// limits a given action group to being exported from only one main
    /// context.
    extern fn g_dbus_connection_export_action_group(p_connection: *DBusConnection, p_object_path: [*:0]const u8, p_action_group: *gio.ActionGroup, p_error: ?*?*glib.Error) c_uint;
    pub const exportActionGroup = g_dbus_connection_export_action_group;

    /// Exports `menu` on `connection` at `object_path`.
    ///
    /// The implemented D-Bus API should be considered private.
    /// It is subject to change in the future.
    ///
    /// An object path can only have one menu model exported on it. If this
    /// constraint is violated, the export will fail and 0 will be
    /// returned (with `error` set accordingly).
    ///
    /// Exporting menus with sections containing more than
    /// `G_MENU_EXPORTER_MAX_SECTION_SIZE` items is not supported and results in
    /// undefined behavior.
    ///
    /// You can unexport the menu model using
    /// `gio.DBusConnection.unexportMenuModel` with the return value of
    /// this function.
    extern fn g_dbus_connection_export_menu_model(p_connection: *DBusConnection, p_object_path: [*:0]const u8, p_menu: *gio.MenuModel, p_error: ?*?*glib.Error) c_uint;
    pub const exportMenuModel = g_dbus_connection_export_menu_model;

    /// Asynchronously flushes `connection`, that is, writes all queued
    /// outgoing message to the transport and then flushes the transport
    /// (using `gio.OutputStream.flushAsync`). This is useful in programs
    /// that wants to emit a D-Bus signal and then exit immediately. Without
    /// flushing the connection, there is no guaranteed that the message has
    /// been sent to the networking buffers in the OS kernel.
    ///
    /// This is an asynchronous method. When the operation is finished,
    /// `callback` will be invoked in the
    /// [thread-default main context][g-main-context-push-thread-default]
    /// of the thread you are calling this method from. You can
    /// then call `gio.DBusConnection.flushFinish` to get the result of the
    /// operation. See `gio.DBusConnection.flushSync` for the synchronous
    /// version.
    extern fn g_dbus_connection_flush(p_connection: *DBusConnection, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const flush = g_dbus_connection_flush;

    /// Finishes an operation started with `gio.DBusConnection.flush`.
    extern fn g_dbus_connection_flush_finish(p_connection: *DBusConnection, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const flushFinish = g_dbus_connection_flush_finish;

    /// Synchronously flushes `connection`. The calling thread is blocked
    /// until this is done. See `gio.DBusConnection.flush` for the
    /// asynchronous version of this method and more details about what it
    /// does.
    extern fn g_dbus_connection_flush_sync(p_connection: *DBusConnection, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const flushSync = g_dbus_connection_flush_sync;

    /// Gets the capabilities negotiated with the remote peer
    extern fn g_dbus_connection_get_capabilities(p_connection: *DBusConnection) gio.DBusCapabilityFlags;
    pub const getCapabilities = g_dbus_connection_get_capabilities;

    /// Gets whether the process is terminated when `connection` is
    /// closed by the remote peer. See
    /// `gio.DBusConnection.properties.exit`-on-close for more details.
    extern fn g_dbus_connection_get_exit_on_close(p_connection: *DBusConnection) c_int;
    pub const getExitOnClose = g_dbus_connection_get_exit_on_close;

    /// Gets the flags used to construct this connection
    extern fn g_dbus_connection_get_flags(p_connection: *DBusConnection) gio.DBusConnectionFlags;
    pub const getFlags = g_dbus_connection_get_flags;

    /// The GUID of the peer performing the role of server when
    /// authenticating. See `gio.DBusConnection.properties.guid` for more details.
    extern fn g_dbus_connection_get_guid(p_connection: *DBusConnection) [*:0]const u8;
    pub const getGuid = g_dbus_connection_get_guid;

    /// Retrieves the last serial number assigned to a `gio.DBusMessage` on
    /// the current thread. This includes messages sent via both low-level
    /// API such as `gio.DBusConnection.sendMessage` as well as
    /// high-level API such as `gio.DBusConnection.emitSignal`,
    /// `gio.DBusConnection.call` or `gio.DBusProxy.call`.
    extern fn g_dbus_connection_get_last_serial(p_connection: *DBusConnection) u32;
    pub const getLastSerial = g_dbus_connection_get_last_serial;

    /// Gets the credentials of the authenticated peer. This will always
    /// return `NULL` unless `connection` acted as a server
    /// (e.g. `G_DBUS_CONNECTION_FLAGS_AUTHENTICATION_SERVER` was passed)
    /// when set up and the client passed credentials as part of the
    /// authentication process.
    ///
    /// In a message bus setup, the message bus is always the server and
    /// each application is a client. So this method will always return
    /// `NULL` for message bus clients.
    extern fn g_dbus_connection_get_peer_credentials(p_connection: *DBusConnection) ?*gio.Credentials;
    pub const getPeerCredentials = g_dbus_connection_get_peer_credentials;

    /// Gets the underlying stream used for IO.
    ///
    /// While the `gio.DBusConnection` is active, it will interact with this
    /// stream from a worker thread, so it is not safe to interact with
    /// the stream directly.
    extern fn g_dbus_connection_get_stream(p_connection: *DBusConnection) *gio.IOStream;
    pub const getStream = g_dbus_connection_get_stream;

    /// Gets the unique name of `connection` as assigned by the message
    /// bus. This can also be used to figure out if `connection` is a
    /// message bus connection.
    extern fn g_dbus_connection_get_unique_name(p_connection: *DBusConnection) ?[*:0]const u8;
    pub const getUniqueName = g_dbus_connection_get_unique_name;

    /// Gets whether `connection` is closed.
    extern fn g_dbus_connection_is_closed(p_connection: *DBusConnection) c_int;
    pub const isClosed = g_dbus_connection_is_closed;

    /// Registers callbacks for exported objects at `object_path` with the
    /// D-Bus interface that is described in `interface_info`.
    ///
    /// Calls to functions in `vtable` (and `user_data_free_func`) will happen
    /// in the
    /// [thread-default main context][g-main-context-push-thread-default]
    /// of the thread you are calling this method from.
    ///
    /// Note that all `glib.Variant` values passed to functions in `vtable` will match
    /// the signature given in `interface_info` - if a remote caller passes
    /// incorrect values, the `org.freedesktop.DBus.Error.InvalidArgs`
    /// is returned to the remote caller.
    ///
    /// Additionally, if the remote caller attempts to invoke methods or
    /// access properties not mentioned in `interface_info` the
    /// `org.freedesktop.DBus.Error.UnknownMethod` resp.
    /// `org.freedesktop.DBus.Error.InvalidArgs` errors
    /// are returned to the caller.
    ///
    /// It is considered a programming error if the
    /// `gio.DBusInterfaceGetPropertyFunc` function in `vtable` returns a
    /// `glib.Variant` of incorrect type.
    ///
    /// If an existing callback is already registered at `object_path` and
    /// `interface_name`, then `error` is set to `G_IO_ERROR_EXISTS`.
    ///
    /// GDBus automatically implements the standard D-Bus interfaces
    /// org.freedesktop.DBus.Properties, org.freedesktop.DBus.Introspectable
    /// and org.freedesktop.Peer, so you don't have to implement those for the
    /// objects you export. You can implement org.freedesktop.DBus.Properties
    /// yourself, e.g. to handle getting and setting of properties asynchronously.
    ///
    /// Note that the reference count on `interface_info` will be
    /// incremented by 1 (unless allocated statically, e.g. if the
    /// reference count is -1, see `gio.DBusInterfaceInfo.ref`) for as long
    /// as the object is exported. Also note that `vtable` will be copied.
    ///
    /// See this [server][gdbus-server] for an example of how to use this method.
    extern fn g_dbus_connection_register_object(p_connection: *DBusConnection, p_object_path: [*:0]const u8, p_interface_info: *gio.DBusInterfaceInfo, p_vtable: ?*const gio.DBusInterfaceVTable, p_user_data: ?*anyopaque, p_user_data_free_func: glib.DestroyNotify, p_error: ?*?*glib.Error) c_uint;
    pub const registerObject = g_dbus_connection_register_object;

    /// Version of `gio.DBusConnection.registerObject` using closures instead of a
    /// `gio.DBusInterfaceVTable` for easier binding in other languages.
    extern fn g_dbus_connection_register_object_with_closures(p_connection: *DBusConnection, p_object_path: [*:0]const u8, p_interface_info: *gio.DBusInterfaceInfo, p_method_call_closure: ?*gobject.Closure, p_get_property_closure: ?*gobject.Closure, p_set_property_closure: ?*gobject.Closure, p_error: ?*?*glib.Error) c_uint;
    pub const registerObjectWithClosures = g_dbus_connection_register_object_with_closures;

    /// Registers a whole subtree of dynamic objects.
    ///
    /// The `enumerate` and `introspection` functions in `vtable` are used to
    /// convey, to remote callers, what nodes exist in the subtree rooted
    /// by `object_path`.
    ///
    /// When handling remote calls into any node in the subtree, first the
    /// `enumerate` function is used to check if the node exists. If the node exists
    /// or the `G_DBUS_SUBTREE_FLAGS_DISPATCH_TO_UNENUMERATED_NODES` flag is set
    /// the `introspection` function is used to check if the node supports the
    /// requested method. If so, the `dispatch` function is used to determine
    /// where to dispatch the call. The collected `gio.DBusInterfaceVTable` and
    /// `gpointer` will be used to call into the interface vtable for processing
    /// the request.
    ///
    /// All calls into user-provided code will be invoked in the
    /// [thread-default main context][g-main-context-push-thread-default]
    /// of the thread you are calling this method from.
    ///
    /// If an existing subtree is already registered at `object_path` or
    /// then `error` is set to `G_IO_ERROR_EXISTS`.
    ///
    /// Note that it is valid to register regular objects (using
    /// `gio.DBusConnection.registerObject`) in a subtree registered with
    /// `gio.DBusConnection.registerSubtree` - if so, the subtree handler
    /// is tried as the last resort. One way to think about a subtree
    /// handler is to consider it a fallback handler for object paths not
    /// registered via `gio.DBusConnection.registerObject` or other bindings.
    ///
    /// Note that `vtable` will be copied so you cannot change it after
    /// registration.
    ///
    /// See this [server][gdbus-subtree-server] for an example of how to use
    /// this method.
    extern fn g_dbus_connection_register_subtree(p_connection: *DBusConnection, p_object_path: [*:0]const u8, p_vtable: *const gio.DBusSubtreeVTable, p_flags: gio.DBusSubtreeFlags, p_user_data: ?*anyopaque, p_user_data_free_func: glib.DestroyNotify, p_error: ?*?*glib.Error) c_uint;
    pub const registerSubtree = g_dbus_connection_register_subtree;

    /// Removes a filter.
    ///
    /// Note that since filters run in a different thread, there is a race
    /// condition where it is possible that the filter will be running even
    /// after calling `gio.DBusConnection.removeFilter`, so you cannot just
    /// free data that the filter might be using. Instead, you should pass
    /// a `glib.DestroyNotify` to `gio.DBusConnection.addFilter`, which will be
    /// called when it is guaranteed that the data is no longer needed.
    extern fn g_dbus_connection_remove_filter(p_connection: *DBusConnection, p_filter_id: c_uint) void;
    pub const removeFilter = g_dbus_connection_remove_filter;

    /// Asynchronously sends `message` to the peer represented by `connection`.
    ///
    /// Unless `flags` contain the
    /// `G_DBUS_SEND_MESSAGE_FLAGS_PRESERVE_SERIAL` flag, the serial number
    /// will be assigned by `connection` and set on `message` via
    /// `gio.DBusMessage.setSerial`. If `out_serial` is not `NULL`, then the
    /// serial number used will be written to this location prior to
    /// submitting the message to the underlying transport. While it has a `volatile`
    /// qualifier, this is a historical artifact and the argument passed to it should
    /// not be `volatile`.
    ///
    /// If `connection` is closed then the operation will fail with
    /// `G_IO_ERROR_CLOSED`. If `message` is not well-formed,
    /// the operation fails with `G_IO_ERROR_INVALID_ARGUMENT`.
    ///
    /// See this [server][gdbus-server] and [client][gdbus-unix-fd-client]
    /// for an example of how to use this low-level API to send and receive
    /// UNIX file descriptors.
    ///
    /// Note that `message` must be unlocked, unless `flags` contain the
    /// `G_DBUS_SEND_MESSAGE_FLAGS_PRESERVE_SERIAL` flag.
    extern fn g_dbus_connection_send_message(p_connection: *DBusConnection, p_message: *gio.DBusMessage, p_flags: gio.DBusSendMessageFlags, p_out_serial: ?*u32, p_error: ?*?*glib.Error) c_int;
    pub const sendMessage = g_dbus_connection_send_message;

    /// Asynchronously sends `message` to the peer represented by `connection`.
    ///
    /// Unless `flags` contain the
    /// `G_DBUS_SEND_MESSAGE_FLAGS_PRESERVE_SERIAL` flag, the serial number
    /// will be assigned by `connection` and set on `message` via
    /// `gio.DBusMessage.setSerial`. If `out_serial` is not `NULL`, then the
    /// serial number used will be written to this location prior to
    /// submitting the message to the underlying transport. While it has a `volatile`
    /// qualifier, this is a historical artifact and the argument passed to it should
    /// not be `volatile`.
    ///
    /// If `connection` is closed then the operation will fail with
    /// `G_IO_ERROR_CLOSED`. If `cancellable` is canceled, the operation will
    /// fail with `G_IO_ERROR_CANCELLED`. If `message` is not well-formed,
    /// the operation fails with `G_IO_ERROR_INVALID_ARGUMENT`.
    ///
    /// This is an asynchronous method. When the operation is finished, `callback`
    /// will be invoked in the
    /// [thread-default main context][g-main-context-push-thread-default]
    /// of the thread you are calling this method from. You can then call
    /// `gio.DBusConnection.sendMessageWithReplyFinish` to get the result of the operation.
    /// See `gio.DBusConnection.sendMessageWithReplySync` for the synchronous version.
    ///
    /// Note that `message` must be unlocked, unless `flags` contain the
    /// `G_DBUS_SEND_MESSAGE_FLAGS_PRESERVE_SERIAL` flag.
    ///
    /// See this [server][gdbus-server] and [client][gdbus-unix-fd-client]
    /// for an example of how to use this low-level API to send and receive
    /// UNIX file descriptors.
    extern fn g_dbus_connection_send_message_with_reply(p_connection: *DBusConnection, p_message: *gio.DBusMessage, p_flags: gio.DBusSendMessageFlags, p_timeout_msec: c_int, p_out_serial: ?*u32, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const sendMessageWithReply = g_dbus_connection_send_message_with_reply;

    /// Finishes an operation started with `gio.DBusConnection.sendMessageWithReply`.
    ///
    /// Note that `error` is only set if a local in-process error
    /// occurred. That is to say that the returned `gio.DBusMessage` object may
    /// be of type `G_DBUS_MESSAGE_TYPE_ERROR`. Use
    /// `gio.DBusMessage.toGerror` to transcode this to a `glib.Error`.
    ///
    /// See this [server][gdbus-server] and [client][gdbus-unix-fd-client]
    /// for an example of how to use this low-level API to send and receive
    /// UNIX file descriptors.
    extern fn g_dbus_connection_send_message_with_reply_finish(p_connection: *DBusConnection, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.DBusMessage;
    pub const sendMessageWithReplyFinish = g_dbus_connection_send_message_with_reply_finish;

    /// Synchronously sends `message` to the peer represented by `connection`
    /// and blocks the calling thread until a reply is received or the
    /// timeout is reached. See `gio.DBusConnection.sendMessageWithReply`
    /// for the asynchronous version of this method.
    ///
    /// Unless `flags` contain the
    /// `G_DBUS_SEND_MESSAGE_FLAGS_PRESERVE_SERIAL` flag, the serial number
    /// will be assigned by `connection` and set on `message` via
    /// `gio.DBusMessage.setSerial`. If `out_serial` is not `NULL`, then the
    /// serial number used will be written to this location prior to
    /// submitting the message to the underlying transport. While it has a `volatile`
    /// qualifier, this is a historical artifact and the argument passed to it should
    /// not be `volatile`.
    ///
    /// If `connection` is closed then the operation will fail with
    /// `G_IO_ERROR_CLOSED`. If `cancellable` is canceled, the operation will
    /// fail with `G_IO_ERROR_CANCELLED`. If `message` is not well-formed,
    /// the operation fails with `G_IO_ERROR_INVALID_ARGUMENT`.
    ///
    /// Note that `error` is only set if a local in-process error
    /// occurred. That is to say that the returned `gio.DBusMessage` object may
    /// be of type `G_DBUS_MESSAGE_TYPE_ERROR`. Use
    /// `gio.DBusMessage.toGerror` to transcode this to a `glib.Error`.
    ///
    /// See this [server][gdbus-server] and [client][gdbus-unix-fd-client]
    /// for an example of how to use this low-level API to send and receive
    /// UNIX file descriptors.
    ///
    /// Note that `message` must be unlocked, unless `flags` contain the
    /// `G_DBUS_SEND_MESSAGE_FLAGS_PRESERVE_SERIAL` flag.
    extern fn g_dbus_connection_send_message_with_reply_sync(p_connection: *DBusConnection, p_message: *gio.DBusMessage, p_flags: gio.DBusSendMessageFlags, p_timeout_msec: c_int, p_out_serial: ?*u32, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.DBusMessage;
    pub const sendMessageWithReplySync = g_dbus_connection_send_message_with_reply_sync;

    /// Sets whether the process should be terminated when `connection` is
    /// closed by the remote peer. See `gio.DBusConnection.properties.exit`-on-close for
    /// more details.
    ///
    /// Note that this function should be used with care. Most modern UNIX
    /// desktops tie the notion of a user session with the session bus, and expect
    /// all of a user's applications to quit when their bus connection goes away.
    /// If you are setting `exit_on_close` to `FALSE` for the shared session
    /// bus connection, you should make sure that your application exits
    /// when the user session ends.
    extern fn g_dbus_connection_set_exit_on_close(p_connection: *DBusConnection, p_exit_on_close: c_int) void;
    pub const setExitOnClose = g_dbus_connection_set_exit_on_close;

    /// Subscribes to signals on `connection` and invokes `callback` whenever
    /// the signal is received. Note that `callback` will be invoked in the
    /// [thread-default main context][g-main-context-push-thread-default]
    /// of the thread you are calling this method from.
    ///
    /// If `connection` is not a message bus connection, `sender` must be
    /// `NULL`.
    ///
    /// If `sender` is a well-known name note that `callback` is invoked with
    /// the unique name for the owner of `sender`, not the well-known name
    /// as one would expect. This is because the message bus rewrites the
    /// name. As such, to avoid certain race conditions, users should be
    /// tracking the name owner of the well-known name and use that when
    /// processing the received signal.
    ///
    /// If one of `G_DBUS_SIGNAL_FLAGS_MATCH_ARG0_NAMESPACE` or
    /// `G_DBUS_SIGNAL_FLAGS_MATCH_ARG0_PATH` are given, `arg0` is
    /// interpreted as part of a namespace or path.  The first argument
    /// of a signal is matched against that part as specified by D-Bus.
    ///
    /// If `user_data_free_func` is non-`NULL`, it will be called (in the
    /// thread-default main context of the thread you are calling this
    /// method from) at some point after `user_data` is no longer
    /// needed. (It is not guaranteed to be called synchronously when the
    /// signal is unsubscribed from, and may be called after `connection`
    /// has been destroyed.)
    ///
    /// As `callback` is potentially invoked in a different thread from where it’s
    /// emitted, it’s possible for this to happen after
    /// `gio.DBusConnection.signalUnsubscribe` has been called in another thread.
    /// Due to this, `user_data` should have a strong reference which is freed with
    /// `user_data_free_func`, rather than pointing to data whose lifecycle is tied
    /// to the signal subscription. For example, if a `gobject.Object` is used to store the
    /// subscription ID from `gio.DBusConnection.signalSubscribe`, a strong reference
    /// to that `gobject.Object` must be passed to `user_data`, and `gobject.Object.unref` passed to
    /// `user_data_free_func`. You are responsible for breaking the resulting
    /// reference count cycle by explicitly unsubscribing from the signal when
    /// dropping the last external reference to the `gobject.Object`. Alternatively, a weak
    /// reference may be used.
    ///
    /// It is guaranteed that if you unsubscribe from a signal using
    /// `gio.DBusConnection.signalUnsubscribe` from the same thread which made the
    /// corresponding `gio.DBusConnection.signalSubscribe` call, `callback` will not
    /// be invoked after `gio.DBusConnection.signalUnsubscribe` returns.
    ///
    /// The returned subscription identifier is an opaque value which is guaranteed
    /// to never be zero.
    ///
    /// This function can never fail.
    extern fn g_dbus_connection_signal_subscribe(p_connection: *DBusConnection, p_sender: ?[*:0]const u8, p_interface_name: ?[*:0]const u8, p_member: ?[*:0]const u8, p_object_path: ?[*:0]const u8, p_arg0: ?[*:0]const u8, p_flags: gio.DBusSignalFlags, p_callback: gio.DBusSignalCallback, p_user_data: ?*anyopaque, p_user_data_free_func: ?glib.DestroyNotify) c_uint;
    pub const signalSubscribe = g_dbus_connection_signal_subscribe;

    /// Unsubscribes from signals.
    ///
    /// Note that there may still be D-Bus traffic to process (relating to this
    /// signal subscription) in the current thread-default `glib.MainContext` after this
    /// function has returned. You should continue to iterate the `glib.MainContext`
    /// until the `glib.DestroyNotify` function passed to
    /// `gio.DBusConnection.signalSubscribe` is called, in order to avoid memory
    /// leaks through callbacks queued on the `glib.MainContext` after it’s stopped being
    /// iterated.
    /// Alternatively, any idle source with a priority lower than `G_PRIORITY_DEFAULT`
    /// that was scheduled after unsubscription, also indicates that all resources
    /// of this subscription are released.
    extern fn g_dbus_connection_signal_unsubscribe(p_connection: *DBusConnection, p_subscription_id: c_uint) void;
    pub const signalUnsubscribe = g_dbus_connection_signal_unsubscribe;

    /// If `connection` was created with
    /// `G_DBUS_CONNECTION_FLAGS_DELAY_MESSAGE_PROCESSING`, this method
    /// starts processing messages. Does nothing on if `connection` wasn't
    /// created with this flag or if the method has already been called.
    extern fn g_dbus_connection_start_message_processing(p_connection: *DBusConnection) void;
    pub const startMessageProcessing = g_dbus_connection_start_message_processing;

    /// Reverses the effect of a previous call to
    /// `gio.DBusConnection.exportActionGroup`.
    ///
    /// It is an error to call this function with an ID that wasn’t returned from
    /// `gio.DBusConnection.exportActionGroup` or to call it with the same
    /// ID more than once.
    extern fn g_dbus_connection_unexport_action_group(p_connection: *DBusConnection, p_export_id: c_uint) void;
    pub const unexportActionGroup = g_dbus_connection_unexport_action_group;

    /// Reverses the effect of a previous call to
    /// `gio.DBusConnection.exportMenuModel`.
    ///
    /// It is an error to call this function with an ID that wasn't returned
    /// from `gio.DBusConnection.exportMenuModel` or to call it with the
    /// same ID more than once.
    extern fn g_dbus_connection_unexport_menu_model(p_connection: *DBusConnection, p_export_id: c_uint) void;
    pub const unexportMenuModel = g_dbus_connection_unexport_menu_model;

    /// Unregisters an object.
    extern fn g_dbus_connection_unregister_object(p_connection: *DBusConnection, p_registration_id: c_uint) c_int;
    pub const unregisterObject = g_dbus_connection_unregister_object;

    /// Unregisters a subtree.
    extern fn g_dbus_connection_unregister_subtree(p_connection: *DBusConnection, p_registration_id: c_uint) c_int;
    pub const unregisterSubtree = g_dbus_connection_unregister_subtree;

    extern fn g_dbus_connection_get_type() usize;
    pub const getGObjectType = g_dbus_connection_get_type;

    extern fn g_object_ref(p_self: *gio.DBusConnection) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.DBusConnection) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *DBusConnection, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Abstract base class for D-Bus interfaces on the service side.
pub const DBusInterfaceSkeleton = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gio.DBusInterface};
    pub const Class = gio.DBusInterfaceSkeletonClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.DBusInterfaceSkeletonPrivate,

    pub const virtual_methods = struct {
        /// If `interface_` has outstanding changes, request for these changes to be
        /// emitted immediately.
        ///
        /// For example, an exported D-Bus interface may queue up property
        /// changes and emit the
        /// `org.freedesktop.DBus.Properties.PropertiesChanged`
        /// signal later (e.g. in an idle handler). This technique is useful
        /// for collapsing multiple property changes into one.
        pub const flush = struct {
            pub fn call(p_class: anytype, p_interface_: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(DBusInterfaceSkeleton.Class, p_class).f_flush.?(gobject.ext.as(DBusInterfaceSkeleton, p_interface_));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_interface_: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(DBusInterfaceSkeleton.Class, p_class).f_flush = @ptrCast(p_implementation);
            }
        };

        /// Signal class handler for the `gio.DBusInterfaceSkeleton.signals.g`-authorize-method signal.
        pub const g_authorize_method = struct {
            pub fn call(p_class: anytype, p_interface_: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_invocation: *gio.DBusMethodInvocation) c_int {
                return gobject.ext.as(DBusInterfaceSkeleton.Class, p_class).f_g_authorize_method.?(gobject.ext.as(DBusInterfaceSkeleton, p_interface_), p_invocation);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_interface_: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_invocation: *gio.DBusMethodInvocation) callconv(.C) c_int) void {
                gobject.ext.as(DBusInterfaceSkeleton.Class, p_class).f_g_authorize_method = @ptrCast(p_implementation);
            }
        };

        /// Gets D-Bus introspection information for the D-Bus interface
        /// implemented by `interface_`.
        pub const get_info = struct {
            pub fn call(p_class: anytype, p_interface_: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) *gio.DBusInterfaceInfo {
                return gobject.ext.as(DBusInterfaceSkeleton.Class, p_class).f_get_info.?(gobject.ext.as(DBusInterfaceSkeleton, p_interface_));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_interface_: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) *gio.DBusInterfaceInfo) void {
                gobject.ext.as(DBusInterfaceSkeleton.Class, p_class).f_get_info = @ptrCast(p_implementation);
            }
        };

        /// Gets all D-Bus properties for `interface_`.
        pub const get_properties = struct {
            pub fn call(p_class: anytype, p_interface_: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) *glib.Variant {
                return gobject.ext.as(DBusInterfaceSkeleton.Class, p_class).f_get_properties.?(gobject.ext.as(DBusInterfaceSkeleton, p_interface_));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_interface_: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) *glib.Variant) void {
                gobject.ext.as(DBusInterfaceSkeleton.Class, p_class).f_get_properties = @ptrCast(p_implementation);
            }
        };

        /// Gets the interface vtable for the D-Bus interface implemented by
        /// `interface_`. The returned function pointers should expect `interface_`
        /// itself to be passed as `user_data`.
        pub const get_vtable = struct {
            pub fn call(p_class: anytype, p_interface_: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) *gio.DBusInterfaceVTable {
                return gobject.ext.as(DBusInterfaceSkeleton.Class, p_class).f_get_vtable.?(gobject.ext.as(DBusInterfaceSkeleton, p_interface_));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_interface_: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) *gio.DBusInterfaceVTable) void {
                gobject.ext.as(DBusInterfaceSkeleton.Class, p_class).f_get_vtable = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// Flags from the `gio.DBusInterfaceSkeletonFlags` enumeration.
        pub const g_flags = struct {
            pub const name = "g-flags";

            pub const Type = gio.DBusInterfaceSkeletonFlags;
        };
    };

    pub const signals = struct {
        /// Emitted when a method is invoked by a remote caller and used to
        /// determine if the method call is authorized.
        ///
        /// Note that this signal is emitted in a thread dedicated to
        /// handling the method call so handlers are allowed to perform
        /// blocking IO. This means that it is appropriate to call e.g.
        /// [`polkit_authority_check_authorization_sync`](http://hal.freedesktop.org/docs/polkit/PolkitAuthority.html`polkit`-authority-check-authorization-sync)
        /// with the
        /// [POLKIT_CHECK_AUTHORIZATION_FLAGS_ALLOW_USER_INTERACTION](http://hal.freedesktop.org/docs/polkit/PolkitAuthority.html`POLKIT`-CHECK-AUTHORIZATION-FLAGS-ALLOW-USER-INTERACTION:CAPS)
        /// flag set.
        ///
        /// If `FALSE` is returned then no further handlers are run and the
        /// signal handler must take a reference to `invocation` and finish
        /// handling the call (e.g. return an error via
        /// `gio.DBusMethodInvocation.returnError`).
        ///
        /// Otherwise, if `TRUE` is returned, signal emission continues. If no
        /// handlers return `FALSE`, then the method is dispatched. If
        /// `interface` has an enclosing `gio.DBusObjectSkeleton`, then the
        /// `gio.DBusObjectSkeleton.signals.authorize`-method signal handlers run before
        /// the handlers for this signal.
        ///
        /// The default class handler just returns `TRUE`.
        ///
        /// Please note that the common case is optimized: if no signals
        /// handlers are connected and the default class handler isn't
        /// overridden (for both `interface` and the enclosing
        /// `gio.DBusObjectSkeleton`, if any) and `gio.DBusInterfaceSkeleton.properties.g`-flags does
        /// not have the
        /// `G_DBUS_INTERFACE_SKELETON_FLAGS_HANDLE_METHOD_INVOCATIONS_IN_THREAD`
        /// flags set, no dedicated thread is ever used and the call will be
        /// handled in the same thread as the object that `interface` belongs
        /// to was exported in.
        pub const g_authorize_method = struct {
            pub const name = "g-authorize-method";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_invocation: *gio.DBusMethodInvocation, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(DBusInterfaceSkeleton, p_instance))),
                    gobject.signalLookup("g-authorize-method", DBusInterfaceSkeleton.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Exports `interface_` at `object_path` on `connection`.
    ///
    /// This can be called multiple times to export the same `interface_`
    /// onto multiple connections however the `object_path` provided must be
    /// the same for all connections.
    ///
    /// Use `gio.DBusInterfaceSkeleton.unexport` to unexport the object.
    extern fn g_dbus_interface_skeleton_export(p_interface_: *DBusInterfaceSkeleton, p_connection: *gio.DBusConnection, p_object_path: [*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const @"export" = g_dbus_interface_skeleton_export;

    /// If `interface_` has outstanding changes, request for these changes to be
    /// emitted immediately.
    ///
    /// For example, an exported D-Bus interface may queue up property
    /// changes and emit the
    /// `org.freedesktop.DBus.Properties.PropertiesChanged`
    /// signal later (e.g. in an idle handler). This technique is useful
    /// for collapsing multiple property changes into one.
    extern fn g_dbus_interface_skeleton_flush(p_interface_: *DBusInterfaceSkeleton) void;
    pub const flush = g_dbus_interface_skeleton_flush;

    /// Gets the first connection that `interface_` is exported on, if any.
    extern fn g_dbus_interface_skeleton_get_connection(p_interface_: *DBusInterfaceSkeleton) ?*gio.DBusConnection;
    pub const getConnection = g_dbus_interface_skeleton_get_connection;

    /// Gets a list of the connections that `interface_` is exported on.
    extern fn g_dbus_interface_skeleton_get_connections(p_interface_: *DBusInterfaceSkeleton) *glib.List;
    pub const getConnections = g_dbus_interface_skeleton_get_connections;

    /// Gets the `gio.DBusInterfaceSkeletonFlags` that describes what the behavior
    /// of `interface_`
    extern fn g_dbus_interface_skeleton_get_flags(p_interface_: *DBusInterfaceSkeleton) gio.DBusInterfaceSkeletonFlags;
    pub const getFlags = g_dbus_interface_skeleton_get_flags;

    /// Gets D-Bus introspection information for the D-Bus interface
    /// implemented by `interface_`.
    extern fn g_dbus_interface_skeleton_get_info(p_interface_: *DBusInterfaceSkeleton) *gio.DBusInterfaceInfo;
    pub const getInfo = g_dbus_interface_skeleton_get_info;

    /// Gets the object path that `interface_` is exported on, if any.
    extern fn g_dbus_interface_skeleton_get_object_path(p_interface_: *DBusInterfaceSkeleton) ?[*:0]const u8;
    pub const getObjectPath = g_dbus_interface_skeleton_get_object_path;

    /// Gets all D-Bus properties for `interface_`.
    extern fn g_dbus_interface_skeleton_get_properties(p_interface_: *DBusInterfaceSkeleton) *glib.Variant;
    pub const getProperties = g_dbus_interface_skeleton_get_properties;

    /// Gets the interface vtable for the D-Bus interface implemented by
    /// `interface_`. The returned function pointers should expect `interface_`
    /// itself to be passed as `user_data`.
    extern fn g_dbus_interface_skeleton_get_vtable(p_interface_: *DBusInterfaceSkeleton) *gio.DBusInterfaceVTable;
    pub const getVtable = g_dbus_interface_skeleton_get_vtable;

    /// Checks if `interface_` is exported on `connection`.
    extern fn g_dbus_interface_skeleton_has_connection(p_interface_: *DBusInterfaceSkeleton, p_connection: *gio.DBusConnection) c_int;
    pub const hasConnection = g_dbus_interface_skeleton_has_connection;

    /// Sets flags describing what the behavior of `skeleton` should be.
    extern fn g_dbus_interface_skeleton_set_flags(p_interface_: *DBusInterfaceSkeleton, p_flags: gio.DBusInterfaceSkeletonFlags) void;
    pub const setFlags = g_dbus_interface_skeleton_set_flags;

    /// Stops exporting `interface_` on all connections it is exported on.
    ///
    /// To unexport `interface_` from only a single connection, use
    /// `gio.DBusInterfaceSkeleton.unexportFromConnection`
    extern fn g_dbus_interface_skeleton_unexport(p_interface_: *DBusInterfaceSkeleton) void;
    pub const unexport = g_dbus_interface_skeleton_unexport;

    /// Stops exporting `interface_` on `connection`.
    ///
    /// To stop exporting on all connections the interface is exported on,
    /// use `gio.DBusInterfaceSkeleton.unexport`.
    extern fn g_dbus_interface_skeleton_unexport_from_connection(p_interface_: *DBusInterfaceSkeleton, p_connection: *gio.DBusConnection) void;
    pub const unexportFromConnection = g_dbus_interface_skeleton_unexport_from_connection;

    extern fn g_dbus_interface_skeleton_get_type() usize;
    pub const getGObjectType = g_dbus_interface_skeleton_get_type;

    extern fn g_object_ref(p_self: *gio.DBusInterfaceSkeleton) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.DBusInterfaceSkeleton) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *DBusInterfaceSkeleton, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GDBusMenuModel` is an implementation of `gio.MenuModel` that can be
/// used as a proxy for a menu model that is exported over D-Bus with
/// `gio.DBusConnection.exportMenuModel`.
pub const DBusMenuModel = opaque {
    pub const Parent = gio.MenuModel;
    pub const Implements = [_]type{};
    pub const Class = opaque {
        pub const Instance = DBusMenuModel;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Obtains a `gio.DBusMenuModel` for the menu model which is exported
    /// at the given `bus_name` and `object_path`.
    ///
    /// The thread default main context is taken at the time of this call.
    /// All signals on the menu model (and any linked models) are reported
    /// with respect to this context.  All calls on the returned menu model
    /// (and linked models) must also originate from this same context, with
    /// the thread default main context unchanged.
    extern fn g_dbus_menu_model_get(p_connection: *gio.DBusConnection, p_bus_name: ?[*:0]const u8, p_object_path: [*:0]const u8) *gio.DBusMenuModel;
    pub const get = g_dbus_menu_model_get;

    extern fn g_dbus_menu_model_get_type() usize;
    pub const getGObjectType = g_dbus_menu_model_get_type;

    extern fn g_object_ref(p_self: *gio.DBusMenuModel) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.DBusMenuModel) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *DBusMenuModel, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A type for representing D-Bus messages that can be sent or received
/// on a `gio.DBusConnection`.
pub const DBusMessage = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = opaque {
        pub const Instance = DBusMessage;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const locked = struct {
            pub const name = "locked";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Utility function to calculate how many bytes are needed to
    /// completely deserialize the D-Bus message stored at `blob`.
    extern fn g_dbus_message_bytes_needed(p_blob: [*]u8, p_blob_len: usize, p_error: ?*?*glib.Error) isize;
    pub const bytesNeeded = g_dbus_message_bytes_needed;

    /// Creates a new empty `gio.DBusMessage`.
    extern fn g_dbus_message_new() *gio.DBusMessage;
    pub const new = g_dbus_message_new;

    /// Creates a new `gio.DBusMessage` from the data stored at `blob`. The byte
    /// order that the message was in can be retrieved using
    /// `gio.DBusMessage.getByteOrder`.
    ///
    /// If the `blob` cannot be parsed, contains invalid fields, or contains invalid
    /// headers, `G_IO_ERROR_INVALID_ARGUMENT` will be returned.
    extern fn g_dbus_message_new_from_blob(p_blob: [*]u8, p_blob_len: usize, p_capabilities: gio.DBusCapabilityFlags, p_error: ?*?*glib.Error) ?*gio.DBusMessage;
    pub const newFromBlob = g_dbus_message_new_from_blob;

    /// Creates a new `gio.DBusMessage` for a method call.
    extern fn g_dbus_message_new_method_call(p_name: ?[*:0]const u8, p_path: [*:0]const u8, p_interface_: ?[*:0]const u8, p_method: [*:0]const u8) *gio.DBusMessage;
    pub const newMethodCall = g_dbus_message_new_method_call;

    /// Creates a new `gio.DBusMessage` for a signal emission.
    extern fn g_dbus_message_new_signal(p_path: [*:0]const u8, p_interface_: [*:0]const u8, p_signal: [*:0]const u8) *gio.DBusMessage;
    pub const newSignal = g_dbus_message_new_signal;

    /// Copies `message`. The copy is a deep copy and the returned
    /// `gio.DBusMessage` is completely identical except that it is guaranteed
    /// to not be locked.
    ///
    /// This operation can fail if e.g. `message` contains file descriptors
    /// and the per-process or system-wide open files limit is reached.
    extern fn g_dbus_message_copy(p_message: *DBusMessage, p_error: ?*?*glib.Error) ?*gio.DBusMessage;
    pub const copy = g_dbus_message_copy;

    /// Convenience to get the first item in the body of `message`.
    ///
    /// See `gio.DBusMessage.getArg0Path` for returning object-path-typed
    /// arg0 values.
    extern fn g_dbus_message_get_arg0(p_message: *DBusMessage) ?[*:0]const u8;
    pub const getArg0 = g_dbus_message_get_arg0;

    /// Convenience to get the first item in the body of `message`.
    ///
    /// See `gio.DBusMessage.getArg0` for returning string-typed arg0 values.
    extern fn g_dbus_message_get_arg0_path(p_message: *DBusMessage) ?[*:0]const u8;
    pub const getArg0Path = g_dbus_message_get_arg0_path;

    /// Gets the body of a message.
    extern fn g_dbus_message_get_body(p_message: *DBusMessage) ?*glib.Variant;
    pub const getBody = g_dbus_message_get_body;

    /// Gets the byte order of `message`.
    extern fn g_dbus_message_get_byte_order(p_message: *DBusMessage) gio.DBusMessageByteOrder;
    pub const getByteOrder = g_dbus_message_get_byte_order;

    /// Convenience getter for the `G_DBUS_MESSAGE_HEADER_FIELD_DESTINATION` header field.
    extern fn g_dbus_message_get_destination(p_message: *DBusMessage) ?[*:0]const u8;
    pub const getDestination = g_dbus_message_get_destination;

    /// Convenience getter for the `G_DBUS_MESSAGE_HEADER_FIELD_ERROR_NAME` header field.
    extern fn g_dbus_message_get_error_name(p_message: *DBusMessage) ?[*:0]const u8;
    pub const getErrorName = g_dbus_message_get_error_name;

    /// Gets the flags for `message`.
    extern fn g_dbus_message_get_flags(p_message: *DBusMessage) gio.DBusMessageFlags;
    pub const getFlags = g_dbus_message_get_flags;

    /// Gets a header field on `message`.
    ///
    /// The caller is responsible for checking the type of the returned `glib.Variant`
    /// matches what is expected.
    extern fn g_dbus_message_get_header(p_message: *DBusMessage, p_header_field: gio.DBusMessageHeaderField) ?*glib.Variant;
    pub const getHeader = g_dbus_message_get_header;

    /// Gets an array of all header fields on `message` that are set.
    extern fn g_dbus_message_get_header_fields(p_message: *DBusMessage) [*]u8;
    pub const getHeaderFields = g_dbus_message_get_header_fields;

    /// Convenience getter for the `G_DBUS_MESSAGE_HEADER_FIELD_INTERFACE` header field.
    extern fn g_dbus_message_get_interface(p_message: *DBusMessage) ?[*:0]const u8;
    pub const getInterface = g_dbus_message_get_interface;

    /// Checks whether `message` is locked. To monitor changes to this
    /// value, conncet to the `gobject.Object.signals.notify` signal to listen for changes
    /// on the `gio.DBusMessage.properties.locked` property.
    extern fn g_dbus_message_get_locked(p_message: *DBusMessage) c_int;
    pub const getLocked = g_dbus_message_get_locked;

    /// Convenience getter for the `G_DBUS_MESSAGE_HEADER_FIELD_MEMBER` header field.
    extern fn g_dbus_message_get_member(p_message: *DBusMessage) ?[*:0]const u8;
    pub const getMember = g_dbus_message_get_member;

    /// Gets the type of `message`.
    extern fn g_dbus_message_get_message_type(p_message: *DBusMessage) gio.DBusMessageType;
    pub const getMessageType = g_dbus_message_get_message_type;

    /// Convenience getter for the `G_DBUS_MESSAGE_HEADER_FIELD_NUM_UNIX_FDS` header field.
    extern fn g_dbus_message_get_num_unix_fds(p_message: *DBusMessage) u32;
    pub const getNumUnixFds = g_dbus_message_get_num_unix_fds;

    /// Convenience getter for the `G_DBUS_MESSAGE_HEADER_FIELD_PATH` header field.
    extern fn g_dbus_message_get_path(p_message: *DBusMessage) ?[*:0]const u8;
    pub const getPath = g_dbus_message_get_path;

    /// Convenience getter for the `G_DBUS_MESSAGE_HEADER_FIELD_REPLY_SERIAL` header field.
    extern fn g_dbus_message_get_reply_serial(p_message: *DBusMessage) u32;
    pub const getReplySerial = g_dbus_message_get_reply_serial;

    /// Convenience getter for the `G_DBUS_MESSAGE_HEADER_FIELD_SENDER` header field.
    extern fn g_dbus_message_get_sender(p_message: *DBusMessage) ?[*:0]const u8;
    pub const getSender = g_dbus_message_get_sender;

    /// Gets the serial for `message`.
    extern fn g_dbus_message_get_serial(p_message: *DBusMessage) u32;
    pub const getSerial = g_dbus_message_get_serial;

    /// Convenience getter for the `G_DBUS_MESSAGE_HEADER_FIELD_SIGNATURE` header field.
    ///
    /// This will always be non-`NULL`, but may be an empty string.
    extern fn g_dbus_message_get_signature(p_message: *DBusMessage) [*:0]const u8;
    pub const getSignature = g_dbus_message_get_signature;

    /// Gets the UNIX file descriptors associated with `message`, if any.
    ///
    /// This method is only available on UNIX.
    ///
    /// The file descriptors normally correspond to `G_VARIANT_TYPE_HANDLE`
    /// values in the body of the message. For example,
    /// if `glib.Variant.getHandle` returns 5, that is intended to be a reference
    /// to the file descriptor that can be accessed by
    /// `g_unix_fd_list_get (list, 5, ...)`.
    extern fn g_dbus_message_get_unix_fd_list(p_message: *DBusMessage) ?*gio.UnixFDList;
    pub const getUnixFdList = g_dbus_message_get_unix_fd_list;

    /// If `message` is locked, does nothing. Otherwise locks the message.
    extern fn g_dbus_message_lock(p_message: *DBusMessage) void;
    pub const lock = g_dbus_message_lock;

    /// Creates a new `gio.DBusMessage` that is an error reply to `method_call_message`.
    extern fn g_dbus_message_new_method_error(p_method_call_message: *DBusMessage, p_error_name: [*:0]const u8, p_error_message_format: [*:0]const u8, ...) *gio.DBusMessage;
    pub const newMethodError = g_dbus_message_new_method_error;

    /// Creates a new `gio.DBusMessage` that is an error reply to `method_call_message`.
    extern fn g_dbus_message_new_method_error_literal(p_method_call_message: *DBusMessage, p_error_name: [*:0]const u8, p_error_message: [*:0]const u8) *gio.DBusMessage;
    pub const newMethodErrorLiteral = g_dbus_message_new_method_error_literal;

    /// Like `gio.DBusMessage.newMethodError` but intended for language bindings.
    extern fn g_dbus_message_new_method_error_valist(p_method_call_message: *DBusMessage, p_error_name: [*:0]const u8, p_error_message_format: [*:0]const u8, p_var_args: std.builtin.VaList) *gio.DBusMessage;
    pub const newMethodErrorValist = g_dbus_message_new_method_error_valist;

    /// Creates a new `gio.DBusMessage` that is a reply to `method_call_message`.
    extern fn g_dbus_message_new_method_reply(p_method_call_message: *DBusMessage) *gio.DBusMessage;
    pub const newMethodReply = g_dbus_message_new_method_reply;

    /// Produces a human-readable multi-line description of `message`.
    ///
    /// The contents of the description has no ABI guarantees, the contents
    /// and formatting is subject to change at any time. Typical output
    /// looks something like this:
    /// ```
    /// Flags:   none
    /// Version: 0
    /// Serial:  4
    /// Headers:
    ///   path -> objectpath '/org/gtk/GDBus/TestObject'
    ///   interface -> 'org.gtk.GDBus.TestInterface'
    ///   member -> 'GimmeStdout'
    ///   destination -> ':1.146'
    /// Body: ()
    /// UNIX File Descriptors:
    ///   (none)
    /// ```
    /// or
    /// ```
    /// Flags:   no-reply-expected
    /// Version: 0
    /// Serial:  477
    /// Headers:
    ///   reply-serial -> uint32 4
    ///   destination -> ':1.159'
    ///   sender -> ':1.146'
    ///   num-unix-fds -> uint32 1
    /// Body: ()
    /// UNIX File Descriptors:
    ///   fd 12: dev=0:10,mode=020620,ino=5,uid=500,gid=5,rdev=136:2,size=0,atime=1273085037,mtime=1273085851,ctime=1272982635
    /// ```
    extern fn g_dbus_message_print(p_message: *DBusMessage, p_indent: c_uint) [*:0]u8;
    pub const print = g_dbus_message_print;

    /// Sets the body `message`. As a side-effect the
    /// `G_DBUS_MESSAGE_HEADER_FIELD_SIGNATURE` header field is set to the
    /// type string of `body` (or cleared if `body` is `NULL`).
    ///
    /// If `body` is floating, `message` assumes ownership of `body`.
    extern fn g_dbus_message_set_body(p_message: *DBusMessage, p_body: *glib.Variant) void;
    pub const setBody = g_dbus_message_set_body;

    /// Sets the byte order of `message`.
    extern fn g_dbus_message_set_byte_order(p_message: *DBusMessage, p_byte_order: gio.DBusMessageByteOrder) void;
    pub const setByteOrder = g_dbus_message_set_byte_order;

    /// Convenience setter for the `G_DBUS_MESSAGE_HEADER_FIELD_DESTINATION` header field.
    extern fn g_dbus_message_set_destination(p_message: *DBusMessage, p_value: ?[*:0]const u8) void;
    pub const setDestination = g_dbus_message_set_destination;

    /// Convenience setter for the `G_DBUS_MESSAGE_HEADER_FIELD_ERROR_NAME` header field.
    extern fn g_dbus_message_set_error_name(p_message: ?*DBusMessage, p_value: [*:0]const u8) void;
    pub const setErrorName = g_dbus_message_set_error_name;

    /// Sets the flags to set on `message`.
    extern fn g_dbus_message_set_flags(p_message: *DBusMessage, p_flags: gio.DBusMessageFlags) void;
    pub const setFlags = g_dbus_message_set_flags;

    /// Sets a header field on `message`.
    ///
    /// If `value` is floating, `message` assumes ownership of `value`.
    extern fn g_dbus_message_set_header(p_message: *DBusMessage, p_header_field: gio.DBusMessageHeaderField, p_value: ?*glib.Variant) void;
    pub const setHeader = g_dbus_message_set_header;

    /// Convenience setter for the `G_DBUS_MESSAGE_HEADER_FIELD_INTERFACE` header field.
    extern fn g_dbus_message_set_interface(p_message: *DBusMessage, p_value: ?[*:0]const u8) void;
    pub const setInterface = g_dbus_message_set_interface;

    /// Convenience setter for the `G_DBUS_MESSAGE_HEADER_FIELD_MEMBER` header field.
    extern fn g_dbus_message_set_member(p_message: *DBusMessage, p_value: ?[*:0]const u8) void;
    pub const setMember = g_dbus_message_set_member;

    /// Sets `message` to be of `type`.
    extern fn g_dbus_message_set_message_type(p_message: *DBusMessage, p_type: gio.DBusMessageType) void;
    pub const setMessageType = g_dbus_message_set_message_type;

    /// Convenience setter for the `G_DBUS_MESSAGE_HEADER_FIELD_NUM_UNIX_FDS` header field.
    extern fn g_dbus_message_set_num_unix_fds(p_message: *DBusMessage, p_value: u32) void;
    pub const setNumUnixFds = g_dbus_message_set_num_unix_fds;

    /// Convenience setter for the `G_DBUS_MESSAGE_HEADER_FIELD_PATH` header field.
    extern fn g_dbus_message_set_path(p_message: *DBusMessage, p_value: ?[*:0]const u8) void;
    pub const setPath = g_dbus_message_set_path;

    /// Convenience setter for the `G_DBUS_MESSAGE_HEADER_FIELD_REPLY_SERIAL` header field.
    extern fn g_dbus_message_set_reply_serial(p_message: *DBusMessage, p_value: u32) void;
    pub const setReplySerial = g_dbus_message_set_reply_serial;

    /// Convenience setter for the `G_DBUS_MESSAGE_HEADER_FIELD_SENDER` header field.
    extern fn g_dbus_message_set_sender(p_message: *DBusMessage, p_value: ?[*:0]const u8) void;
    pub const setSender = g_dbus_message_set_sender;

    /// Sets the serial for `message`.
    extern fn g_dbus_message_set_serial(p_message: *DBusMessage, p_serial: u32) void;
    pub const setSerial = g_dbus_message_set_serial;

    /// Convenience setter for the `G_DBUS_MESSAGE_HEADER_FIELD_SIGNATURE` header field.
    extern fn g_dbus_message_set_signature(p_message: *DBusMessage, p_value: ?[*:0]const u8) void;
    pub const setSignature = g_dbus_message_set_signature;

    /// Sets the UNIX file descriptors associated with `message`. As a
    /// side-effect the `G_DBUS_MESSAGE_HEADER_FIELD_NUM_UNIX_FDS` header
    /// field is set to the number of fds in `fd_list` (or cleared if
    /// `fd_list` is `NULL`).
    ///
    /// This method is only available on UNIX.
    ///
    /// When designing D-Bus APIs that are intended to be interoperable,
    /// please note that non-GDBus implementations of D-Bus can usually only
    /// access file descriptors if they are referenced by a value of type
    /// `G_VARIANT_TYPE_HANDLE` in the body of the message.
    extern fn g_dbus_message_set_unix_fd_list(p_message: *DBusMessage, p_fd_list: ?*gio.UnixFDList) void;
    pub const setUnixFdList = g_dbus_message_set_unix_fd_list;

    /// Serializes `message` to a blob. The byte order returned by
    /// `gio.DBusMessage.getByteOrder` will be used.
    extern fn g_dbus_message_to_blob(p_message: *DBusMessage, p_out_size: *usize, p_capabilities: gio.DBusCapabilityFlags, p_error: ?*?*glib.Error) ?[*]u8;
    pub const toBlob = g_dbus_message_to_blob;

    /// If `message` is not of type `G_DBUS_MESSAGE_TYPE_ERROR` does
    /// nothing and returns `FALSE`.
    ///
    /// Otherwise this method encodes the error in `message` as a `glib.Error`
    /// using `gio.DBusError.setDbusError` using the information in the
    /// `G_DBUS_MESSAGE_HEADER_FIELD_ERROR_NAME` header field of `message` as
    /// well as the first string item in `message`'s body.
    extern fn g_dbus_message_to_gerror(p_message: *DBusMessage, p_error: ?*?*glib.Error) c_int;
    pub const toGerror = g_dbus_message_to_gerror;

    extern fn g_dbus_message_get_type() usize;
    pub const getGObjectType = g_dbus_message_get_type;

    extern fn g_object_ref(p_self: *gio.DBusMessage) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.DBusMessage) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *DBusMessage, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Instances of the `GDBusMethodInvocation` class are used when
/// handling D-Bus method calls. It provides a way to asynchronously
/// return results and errors.
///
/// The normal way to obtain a `GDBusMethodInvocation` object is to receive
/// it as an argument to the ``handle_method_call`` function in a
/// `gio.DBusInterfaceVTable` that was passed to
/// `gio.DBusConnection.registerObject`.
pub const DBusMethodInvocation = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = opaque {
        pub const Instance = DBusMethodInvocation;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Gets the `gio.DBusConnection` the method was invoked on.
    extern fn g_dbus_method_invocation_get_connection(p_invocation: *DBusMethodInvocation) *gio.DBusConnection;
    pub const getConnection = g_dbus_method_invocation_get_connection;

    /// Gets the name of the D-Bus interface the method was invoked on.
    ///
    /// If this method call is a property Get, Set or GetAll call that has
    /// been redirected to the method call handler then
    /// "org.freedesktop.DBus.Properties" will be returned.  See
    /// `gio.DBusInterfaceVTable` for more information.
    extern fn g_dbus_method_invocation_get_interface_name(p_invocation: *DBusMethodInvocation) [*:0]const u8;
    pub const getInterfaceName = g_dbus_method_invocation_get_interface_name;

    /// Gets the `gio.DBusMessage` for the method invocation. This is useful if
    /// you need to use low-level protocol features, such as UNIX file
    /// descriptor passing, that cannot be properly expressed in the
    /// `glib.Variant` API.
    ///
    /// See this [server][gdbus-server] and [client][gdbus-unix-fd-client]
    /// for an example of how to use this low-level API to send and receive
    /// UNIX file descriptors.
    extern fn g_dbus_method_invocation_get_message(p_invocation: *DBusMethodInvocation) *gio.DBusMessage;
    pub const getMessage = g_dbus_method_invocation_get_message;

    /// Gets information about the method call, if any.
    ///
    /// If this method invocation is a property Get, Set or GetAll call that
    /// has been redirected to the method call handler then `NULL` will be
    /// returned.  See `gio.DBusMethodInvocation.getPropertyInfo` and
    /// `gio.DBusInterfaceVTable` for more information.
    extern fn g_dbus_method_invocation_get_method_info(p_invocation: *DBusMethodInvocation) ?*const gio.DBusMethodInfo;
    pub const getMethodInfo = g_dbus_method_invocation_get_method_info;

    /// Gets the name of the method that was invoked.
    extern fn g_dbus_method_invocation_get_method_name(p_invocation: *DBusMethodInvocation) [*:0]const u8;
    pub const getMethodName = g_dbus_method_invocation_get_method_name;

    /// Gets the object path the method was invoked on.
    extern fn g_dbus_method_invocation_get_object_path(p_invocation: *DBusMethodInvocation) [*:0]const u8;
    pub const getObjectPath = g_dbus_method_invocation_get_object_path;

    /// Gets the parameters of the method invocation. If there are no input
    /// parameters then this will return a GVariant with 0 children rather than NULL.
    extern fn g_dbus_method_invocation_get_parameters(p_invocation: *DBusMethodInvocation) *glib.Variant;
    pub const getParameters = g_dbus_method_invocation_get_parameters;

    /// Gets information about the property that this method call is for, if
    /// any.
    ///
    /// This will only be set in the case of an invocation in response to a
    /// property Get or Set call that has been directed to the method call
    /// handler for an object on account of its `property_get` or
    /// `property_set` vtable pointers being unset.
    ///
    /// See `gio.DBusInterfaceVTable` for more information.
    ///
    /// If the call was GetAll, `NULL` will be returned.
    extern fn g_dbus_method_invocation_get_property_info(p_invocation: *DBusMethodInvocation) ?*const gio.DBusPropertyInfo;
    pub const getPropertyInfo = g_dbus_method_invocation_get_property_info;

    /// Gets the bus name that invoked the method.
    extern fn g_dbus_method_invocation_get_sender(p_invocation: *DBusMethodInvocation) [*:0]const u8;
    pub const getSender = g_dbus_method_invocation_get_sender;

    /// Gets the `user_data` `gpointer` passed to `gio.DBusConnection.registerObject`.
    extern fn g_dbus_method_invocation_get_user_data(p_invocation: *DBusMethodInvocation) ?*anyopaque;
    pub const getUserData = g_dbus_method_invocation_get_user_data;

    /// Finishes handling a D-Bus method call by returning an error.
    ///
    /// This method will take ownership of `invocation`. See
    /// `gio.DBusInterfaceVTable` for more information about the ownership of
    /// `invocation`.
    extern fn g_dbus_method_invocation_return_dbus_error(p_invocation: *DBusMethodInvocation, p_error_name: [*:0]const u8, p_error_message: [*:0]const u8) void;
    pub const returnDbusError = g_dbus_method_invocation_return_dbus_error;

    /// Finishes handling a D-Bus method call by returning an error.
    ///
    /// See `gio.dbusErrorEncodeGerror` for details about what error name
    /// will be returned on the wire. In a nutshell, if the given error is
    /// registered using `gio.dbusErrorRegisterError` the name given
    /// during registration is used. Otherwise, a name of the form
    /// `org.gtk.GDBus.UnmappedGError.Quark...` is used. This provides
    /// transparent mapping of `glib.Error` between applications using GDBus.
    ///
    /// If you are writing an application intended to be portable,
    /// always register errors with `gio.dbusErrorRegisterError`
    /// or use `gio.DBusMethodInvocation.returnDbusError`.
    ///
    /// This method will take ownership of `invocation`. See
    /// `gio.DBusInterfaceVTable` for more information about the ownership of
    /// `invocation`.
    ///
    /// Since 2.48, if the method call requested for a reply not to be sent
    /// then this call will free `invocation` but otherwise do nothing (as per
    /// the recommendations of the D-Bus specification).
    extern fn g_dbus_method_invocation_return_error(p_invocation: *DBusMethodInvocation, p_domain: glib.Quark, p_code: c_int, p_format: [*:0]const u8, ...) void;
    pub const returnError = g_dbus_method_invocation_return_error;

    /// Like `gio.DBusMethodInvocation.returnError` but without `printf`-style formatting.
    ///
    /// This method will take ownership of `invocation`. See
    /// `gio.DBusInterfaceVTable` for more information about the ownership of
    /// `invocation`.
    extern fn g_dbus_method_invocation_return_error_literal(p_invocation: *DBusMethodInvocation, p_domain: glib.Quark, p_code: c_int, p_message: [*:0]const u8) void;
    pub const returnErrorLiteral = g_dbus_method_invocation_return_error_literal;

    /// Like `gio.DBusMethodInvocation.returnError` but intended for
    /// language bindings.
    ///
    /// This method will take ownership of `invocation`. See
    /// `gio.DBusInterfaceVTable` for more information about the ownership of
    /// `invocation`.
    extern fn g_dbus_method_invocation_return_error_valist(p_invocation: *DBusMethodInvocation, p_domain: glib.Quark, p_code: c_int, p_format: [*:0]const u8, p_var_args: std.builtin.VaList) void;
    pub const returnErrorValist = g_dbus_method_invocation_return_error_valist;

    /// Like `gio.DBusMethodInvocation.returnError` but takes a `glib.Error`
    /// instead of the error domain, error code and message.
    ///
    /// This method will take ownership of `invocation`. See
    /// `gio.DBusInterfaceVTable` for more information about the ownership of
    /// `invocation`.
    extern fn g_dbus_method_invocation_return_gerror(p_invocation: *DBusMethodInvocation, p_error: *const glib.Error) void;
    pub const returnGerror = g_dbus_method_invocation_return_gerror;

    /// Finishes handling a D-Bus method call by returning `parameters`.
    /// If the `parameters` GVariant is floating, it is consumed.
    ///
    /// It is an error if `parameters` is not of the right format: it must be a tuple
    /// containing the out-parameters of the D-Bus method. Even if the method has a
    /// single out-parameter, it must be contained in a tuple. If the method has no
    /// out-parameters, `parameters` may be `NULL` or an empty tuple.
    ///
    /// ```
    /// GDBusMethodInvocation *invocation = some_invocation;
    /// g_autofree gchar *result_string = NULL;
    /// g_autoptr (GError) error = NULL;
    ///
    /// result_string = calculate_result (&error);
    ///
    /// if (error != NULL)
    ///   g_dbus_method_invocation_return_gerror (invocation, error);
    /// else
    ///   g_dbus_method_invocation_return_value (invocation,
    ///                                          g_variant_new ("(s)", result_string));
    ///
    /// // Do not free `invocation` here; returning a value does that
    /// ```
    ///
    /// This method will take ownership of `invocation`. See
    /// `gio.DBusInterfaceVTable` for more information about the ownership of
    /// `invocation`.
    ///
    /// Since 2.48, if the method call requested for a reply not to be sent
    /// then this call will sink `parameters` and free `invocation`, but
    /// otherwise do nothing (as per the recommendations of the D-Bus
    /// specification).
    extern fn g_dbus_method_invocation_return_value(p_invocation: *DBusMethodInvocation, p_parameters: ?*glib.Variant) void;
    pub const returnValue = g_dbus_method_invocation_return_value;

    /// Like `gio.DBusMethodInvocation.returnValue` but also takes a `gio.UnixFDList`.
    ///
    /// This method is only available on UNIX.
    ///
    /// This method will take ownership of `invocation`. See
    /// `gio.DBusInterfaceVTable` for more information about the ownership of
    /// `invocation`.
    extern fn g_dbus_method_invocation_return_value_with_unix_fd_list(p_invocation: *DBusMethodInvocation, p_parameters: ?*glib.Variant, p_fd_list: ?*gio.UnixFDList) void;
    pub const returnValueWithUnixFdList = g_dbus_method_invocation_return_value_with_unix_fd_list;

    /// Like `gio.DBusMethodInvocation.returnGerror` but takes ownership
    /// of `error` so the caller does not need to free it.
    ///
    /// This method will take ownership of `invocation`. See
    /// `gio.DBusInterfaceVTable` for more information about the ownership of
    /// `invocation`.
    extern fn g_dbus_method_invocation_take_error(p_invocation: *DBusMethodInvocation, p_error: *glib.Error) void;
    pub const takeError = g_dbus_method_invocation_take_error;

    extern fn g_dbus_method_invocation_get_type() usize;
    pub const getGObjectType = g_dbus_method_invocation_get_type;

    extern fn g_object_ref(p_self: *gio.DBusMethodInvocation) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.DBusMethodInvocation) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *DBusMethodInvocation, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GDBusObjectManagerClient` is used to create, monitor and delete object
/// proxies for remote objects exported by a `gio.DBusObjectManagerServer`
/// (or any code implementing the
/// [org.freedesktop.DBus.ObjectManager](http://dbus.freedesktop.org/doc/dbus-specification.html`standard`-interfaces-objectmanager)
/// interface).
///
/// Once an instance of this type has been created, you can connect to
/// the `gio.DBusObjectManager.signals.object_added` and
/// `gio.DBusObjectManager.signals.@"object_removed signals"` and inspect the
/// `gio.DBusObjectProxy` objects returned by
/// `gio.DBusObjectManager.getObjects`.
///
/// If the name for a `GDBusObjectManagerClient` is not owned by anyone at
/// object construction time, the default behavior is to request the
/// message bus to launch an owner for the name. This behavior can be
/// disabled using the `G_DBUS_OBJECT_MANAGER_CLIENT_FLAGS_DO_NOT_AUTO_START`
/// flag. It’s also worth noting that this only works if the name of
/// interest is activatable in the first place. E.g. in some cases it
/// is not possible to launch an owner for the requested name. In this
/// case, `GDBusObjectManagerClient` object construction still succeeds but
/// there will be no object proxies
/// (e.g. `gio.DBusObjectManager.getObjects` returns the empty list) and
/// the `gio.DBusObjectManagerClient.properties.name_owner` property is `NULL`.
///
/// The owner of the requested name can come and go (for example
/// consider a system service being restarted) – `GDBusObjectManagerClient`
/// handles this case too; simply connect to the `gobject.Object.signals.notify`
/// signal to watch for changes on the
/// `gio.DBusObjectManagerClient.properties.name_owner` property. When the name
/// owner vanishes, the behavior is that
/// `gio.DBusObjectManagerClient.properties.name_owner` is set to `NULL` (this
/// includes emission of the `gobject.Object.signals.notify` signal) and then
/// `gio.DBusObjectManager.signals.object_removed` signals are synthesized
/// for all currently existing object proxies. Since
/// `gio.DBusObjectManagerClient.properties.name_owner` is `NULL` when this
/// happens, you can use this information to disambiguate a synthesized signal
/// from a genuine signal caused by object removal on the remote
/// `gio.DBusObjectManager`. Similarly, when a new name owner appears,
/// `gio.DBusObjectManager.signals.object_added` signals are synthesized
/// while `gio.DBusObjectManagerClient.properties.name_owner` is still `NULL`. Only
/// when all object proxies have been added, the
/// `gio.DBusObjectManagerClient.properties.name_owner` is set to the new name
/// owner (this includes emission of the `gobject.Object.signals.notify` signal).
/// Furthermore, you are guaranteed that
/// `gio.DBusObjectManagerClient.properties.name_owner` will alternate between a
/// name owner (e.g. `:1.42`) and `NULL` even in the case where
/// the name of interest is atomically replaced
///
/// Ultimately, `GDBusObjectManagerClient` is used to obtain
/// `gio.DBusProxy` instances. All signals (including the
/// `org.freedesktop.DBus.Properties::PropertiesChanged` signal)
/// delivered to `gio.DBusProxy` instances are guaranteed to originate
/// from the name owner. This guarantee along with the behavior
/// described above, means that certain race conditions including the
/// “half the proxy is from the old owner and the other half is from
/// the new owner” problem cannot happen.
///
/// To avoid having the application connect to signals on the returned
/// `gio.DBusObjectProxy` and `gio.DBusProxy` objects, the
/// `gio.DBusObject.signals.interface_added`,
/// `gio.DBusObject.signals.interface_removed`,
/// `gio.DBusProxy.signals.g_properties_changed` and
/// `gio.DBusProxy.signals.g_signal` signals
/// are also emitted on the `GDBusObjectManagerClient` instance managing these
/// objects. The signals emitted are
/// `gio.DBusObjectManager.signals.interface_added`,
/// `gio.DBusObjectManager.signals.interface_removed`,
/// `gio.DBusObjectManagerClient.signals.interface_proxy_properties_changed` and
/// `gio.DBusObjectManagerClient.signals.interface_proxy_signal`.
///
/// Note that all callbacks and signals are emitted in the
/// thread-default main context (see
/// `glib.MainContext.pushThreadDefault`) that the
/// `GDBusObjectManagerClient` object was constructed in. Additionally, the
/// `gio.DBusObjectProxy` and `gio.DBusProxy` objects
/// originating from the `GDBusObjectManagerClient` object will be created in
/// the same context and, consequently, will deliver signals in the
/// same main loop.
pub const DBusObjectManagerClient = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{ gio.AsyncInitable, gio.DBusObjectManager, gio.Initable };
    pub const Class = gio.DBusObjectManagerClientClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.DBusObjectManagerClientPrivate,

    pub const virtual_methods = struct {
        /// Signal class handler for the `gio.DBusObjectManagerClient.signals.interface`-proxy-properties-changed signal.
        pub const interface_proxy_properties_changed = struct {
            pub fn call(p_class: anytype, p_manager: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_object_proxy: *gio.DBusObjectProxy, p_interface_proxy: *gio.DBusProxy, p_changed_properties: *glib.Variant, p_invalidated_properties: *const [*:0]const u8) void {
                return gobject.ext.as(DBusObjectManagerClient.Class, p_class).f_interface_proxy_properties_changed.?(gobject.ext.as(DBusObjectManagerClient, p_manager), p_object_proxy, p_interface_proxy, p_changed_properties, p_invalidated_properties);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_manager: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_object_proxy: *gio.DBusObjectProxy, p_interface_proxy: *gio.DBusProxy, p_changed_properties: *glib.Variant, p_invalidated_properties: *const [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(DBusObjectManagerClient.Class, p_class).f_interface_proxy_properties_changed = @ptrCast(p_implementation);
            }
        };

        /// Signal class handler for the `gio.DBusObjectManagerClient.signals.interface`-proxy-signal signal.
        pub const interface_proxy_signal = struct {
            pub fn call(p_class: anytype, p_manager: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_object_proxy: *gio.DBusObjectProxy, p_interface_proxy: *gio.DBusProxy, p_sender_name: [*:0]const u8, p_signal_name: [*:0]const u8, p_parameters: *glib.Variant) void {
                return gobject.ext.as(DBusObjectManagerClient.Class, p_class).f_interface_proxy_signal.?(gobject.ext.as(DBusObjectManagerClient, p_manager), p_object_proxy, p_interface_proxy, p_sender_name, p_signal_name, p_parameters);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_manager: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_object_proxy: *gio.DBusObjectProxy, p_interface_proxy: *gio.DBusProxy, p_sender_name: [*:0]const u8, p_signal_name: [*:0]const u8, p_parameters: *glib.Variant) callconv(.C) void) void {
                gobject.ext.as(DBusObjectManagerClient.Class, p_class).f_interface_proxy_signal = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// If this property is not `G_BUS_TYPE_NONE`, then
        /// `gio.DBusObjectManagerClient.properties.connection` must be `NULL` and will be set to the
        /// `gio.DBusConnection` obtained by calling `gio.busGet` with the value
        /// of this property.
        pub const bus_type = struct {
            pub const name = "bus-type";

            pub const Type = gio.BusType;
        };

        /// The `gio.DBusConnection` to use.
        pub const connection = struct {
            pub const name = "connection";

            pub const Type = ?*gio.DBusConnection;
        };

        /// Flags from the `gio.DBusObjectManagerClientFlags` enumeration.
        pub const flags = struct {
            pub const name = "flags";

            pub const Type = gio.DBusObjectManagerClientFlags;
        };

        /// A `glib.DestroyNotify` for the `gpointer` user_data in `gio.DBusObjectManagerClient.properties.get`-proxy-type-user-data.
        pub const get_proxy_type_destroy_notify = struct {
            pub const name = "get-proxy-type-destroy-notify";

            pub const Type = ?*anyopaque;
        };

        /// The `gio.DBusProxyTypeFunc` to use when determining what `gobject.Type` to
        /// use for interface proxies or `NULL`.
        pub const get_proxy_type_func = struct {
            pub const name = "get-proxy-type-func";

            pub const Type = ?*anyopaque;
        };

        /// The `gpointer` user_data to pass to `gio.DBusObjectManagerClient.properties.get`-proxy-type-func.
        pub const get_proxy_type_user_data = struct {
            pub const name = "get-proxy-type-user-data";

            pub const Type = ?*anyopaque;
        };

        /// The well-known name or unique name that the manager is for.
        pub const name = struct {
            pub const name = "name";

            pub const Type = ?[*:0]u8;
        };

        /// The unique name that owns `gio.DBusObjectManagerClient.properties.name` or `NULL` if
        /// no-one is currently owning the name. Connect to the
        /// `gobject.Object.signals.notify` signal to track changes to this property.
        pub const name_owner = struct {
            pub const name = "name-owner";

            pub const Type = ?[*:0]u8;
        };

        /// The object path the manager is for.
        pub const object_path = struct {
            pub const name = "object-path";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {
        /// Emitted when one or more D-Bus properties on proxy changes. The
        /// local cache has already been updated when this signal fires. Note
        /// that both `changed_properties` and `invalidated_properties` are
        /// guaranteed to never be `NULL` (either may be empty though).
        ///
        /// This signal exists purely as a convenience to avoid having to
        /// connect signals to all interface proxies managed by `manager`.
        ///
        /// This signal is emitted in the
        /// [thread-default main context][g-main-context-push-thread-default]
        /// that `manager` was constructed in.
        pub const interface_proxy_properties_changed = struct {
            pub const name = "interface-proxy-properties-changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_object_proxy: *gio.DBusObjectProxy, p_interface_proxy: *gio.DBusProxy, p_changed_properties: *glib.Variant, p_invalidated_properties: [*][*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(DBusObjectManagerClient, p_instance))),
                    gobject.signalLookup("interface-proxy-properties-changed", DBusObjectManagerClient.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when a D-Bus signal is received on `interface_proxy`.
        ///
        /// This signal exists purely as a convenience to avoid having to
        /// connect signals to all interface proxies managed by `manager`.
        ///
        /// This signal is emitted in the
        /// [thread-default main context][g-main-context-push-thread-default]
        /// that `manager` was constructed in.
        pub const interface_proxy_signal = struct {
            pub const name = "interface-proxy-signal";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_object_proxy: *gio.DBusObjectProxy, p_interface_proxy: *gio.DBusProxy, p_sender_name: [*:0]u8, p_signal_name: [*:0]u8, p_parameters: *glib.Variant, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(DBusObjectManagerClient, p_instance))),
                    gobject.signalLookup("interface-proxy-signal", DBusObjectManagerClient.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Asynchronously creates a new `gio.DBusObjectManagerClient` object.
    ///
    /// This is an asynchronous failable constructor. When the result is
    /// ready, `callback` will be invoked in the
    /// [thread-default main context][g-main-context-push-thread-default]
    /// of the thread you are calling this method from. You can
    /// then call `gio.DBusObjectManagerClient.newFinish` to get the result. See
    /// `gio.DBusObjectManagerClient.newSync` for the synchronous version.
    extern fn g_dbus_object_manager_client_new(p_connection: *gio.DBusConnection, p_flags: gio.DBusObjectManagerClientFlags, p_name: [*:0]const u8, p_object_path: [*:0]const u8, p_get_proxy_type_func: ?gio.DBusProxyTypeFunc, p_get_proxy_type_user_data: ?*anyopaque, p_get_proxy_type_destroy_notify: ?glib.DestroyNotify, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const new = g_dbus_object_manager_client_new;

    /// Like `gio.DBusObjectManagerClient.new` but takes a `gio.BusType` instead of a
    /// `gio.DBusConnection`.
    ///
    /// This is an asynchronous failable constructor. When the result is
    /// ready, `callback` will be invoked in the
    /// [thread-default main loop][g-main-context-push-thread-default]
    /// of the thread you are calling this method from. You can
    /// then call `gio.DBusObjectManagerClient.newForBusFinish` to get the result. See
    /// `gio.DBusObjectManagerClient.newForBusSync` for the synchronous version.
    extern fn g_dbus_object_manager_client_new_for_bus(p_bus_type: gio.BusType, p_flags: gio.DBusObjectManagerClientFlags, p_name: [*:0]const u8, p_object_path: [*:0]const u8, p_get_proxy_type_func: ?gio.DBusProxyTypeFunc, p_get_proxy_type_user_data: ?*anyopaque, p_get_proxy_type_destroy_notify: ?glib.DestroyNotify, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const newForBus = g_dbus_object_manager_client_new_for_bus;

    /// Finishes an operation started with `gio.DBusObjectManagerClient.new`.
    extern fn g_dbus_object_manager_client_new_finish(p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.DBusObjectManagerClient;
    pub const newFinish = g_dbus_object_manager_client_new_finish;

    /// Finishes an operation started with `gio.DBusObjectManagerClient.newForBus`.
    extern fn g_dbus_object_manager_client_new_for_bus_finish(p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.DBusObjectManagerClient;
    pub const newForBusFinish = g_dbus_object_manager_client_new_for_bus_finish;

    /// Like `gio.DBusObjectManagerClient.newSync` but takes a `gio.BusType` instead
    /// of a `gio.DBusConnection`.
    ///
    /// This is a synchronous failable constructor - the calling thread is
    /// blocked until a reply is received. See `gio.DBusObjectManagerClient.newForBus`
    /// for the asynchronous version.
    extern fn g_dbus_object_manager_client_new_for_bus_sync(p_bus_type: gio.BusType, p_flags: gio.DBusObjectManagerClientFlags, p_name: [*:0]const u8, p_object_path: [*:0]const u8, p_get_proxy_type_func: ?gio.DBusProxyTypeFunc, p_get_proxy_type_user_data: ?*anyopaque, p_get_proxy_type_destroy_notify: ?glib.DestroyNotify, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.DBusObjectManagerClient;
    pub const newForBusSync = g_dbus_object_manager_client_new_for_bus_sync;

    /// Creates a new `gio.DBusObjectManagerClient` object.
    ///
    /// This is a synchronous failable constructor - the calling thread is
    /// blocked until a reply is received. See `gio.DBusObjectManagerClient.new`
    /// for the asynchronous version.
    extern fn g_dbus_object_manager_client_new_sync(p_connection: *gio.DBusConnection, p_flags: gio.DBusObjectManagerClientFlags, p_name: ?[*:0]const u8, p_object_path: [*:0]const u8, p_get_proxy_type_func: ?gio.DBusProxyTypeFunc, p_get_proxy_type_user_data: ?*anyopaque, p_get_proxy_type_destroy_notify: ?glib.DestroyNotify, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.DBusObjectManagerClient;
    pub const newSync = g_dbus_object_manager_client_new_sync;

    /// Gets the `gio.DBusConnection` used by `manager`.
    extern fn g_dbus_object_manager_client_get_connection(p_manager: *DBusObjectManagerClient) *gio.DBusConnection;
    pub const getConnection = g_dbus_object_manager_client_get_connection;

    /// Gets the flags that `manager` was constructed with.
    extern fn g_dbus_object_manager_client_get_flags(p_manager: *DBusObjectManagerClient) gio.DBusObjectManagerClientFlags;
    pub const getFlags = g_dbus_object_manager_client_get_flags;

    /// Gets the name that `manager` is for, or `NULL` if not a message bus
    /// connection.
    extern fn g_dbus_object_manager_client_get_name(p_manager: *DBusObjectManagerClient) [*:0]const u8;
    pub const getName = g_dbus_object_manager_client_get_name;

    /// The unique name that owns the name that `manager` is for or `NULL` if
    /// no-one currently owns that name. You can connect to the
    /// `gobject.Object.signals.notify` signal to track changes to the
    /// `gio.DBusObjectManagerClient.properties.name`-owner property.
    extern fn g_dbus_object_manager_client_get_name_owner(p_manager: *DBusObjectManagerClient) ?[*:0]u8;
    pub const getNameOwner = g_dbus_object_manager_client_get_name_owner;

    extern fn g_dbus_object_manager_client_get_type() usize;
    pub const getGObjectType = g_dbus_object_manager_client_get_type;

    extern fn g_object_ref(p_self: *gio.DBusObjectManagerClient) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.DBusObjectManagerClient) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *DBusObjectManagerClient, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GDBusObjectManagerServer` is used to export `gio.DBusObject` instances
/// using the standardized
/// [`org.freedesktop.DBus.ObjectManager`](http://dbus.freedesktop.org/doc/dbus-specification.html`standard`-interfaces-objectmanager)
/// interface. For example, remote D-Bus clients can get all objects
/// and properties in a single call. Additionally, any change in the
/// object hierarchy is broadcast using signals. This means that D-Bus
/// clients can keep caches up to date by only listening to D-Bus
/// signals.
///
/// The recommended path to export an object manager at is the path form of the
/// well-known name of a D-Bus service, or below. For example, if a D-Bus service
/// is available at the well-known name `net.example.ExampleService1`, the object
/// manager should typically be exported at `/net/example/ExampleService1`, or
/// below (to allow for multiple object managers in a service).
///
/// It is supported, but not recommended, to export an object manager at the root
/// path, `/`.
///
/// See `gio.DBusObjectManagerClient` for the client-side code that is
/// intended to be used with `GDBusObjectManagerServer` or any D-Bus
/// object implementing the `org.freedesktop.DBus.ObjectManager` interface.
pub const DBusObjectManagerServer = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gio.DBusObjectManager};
    pub const Class = gio.DBusObjectManagerServerClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.DBusObjectManagerServerPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The `gio.DBusConnection` to export objects on.
        pub const connection = struct {
            pub const name = "connection";

            pub const Type = ?*gio.DBusConnection;
        };

        /// The object path to register the manager object at.
        pub const object_path = struct {
            pub const name = "object-path";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gio.DBusObjectManagerServer` object.
    ///
    /// The returned server isn't yet exported on any connection. To do so,
    /// use `gio.DBusObjectManagerServer.setConnection`. Normally you
    /// want to export all of your objects before doing so to avoid
    /// [InterfacesAdded](http://dbus.freedesktop.org/doc/dbus-specification.html`standard`-interfaces-objectmanager)
    /// signals being emitted.
    extern fn g_dbus_object_manager_server_new(p_object_path: [*:0]const u8) *gio.DBusObjectManagerServer;
    pub const new = g_dbus_object_manager_server_new;

    /// Exports `object` on `manager`.
    ///
    /// If there is already a `gio.DBusObject` exported at the object path,
    /// then the old object is removed.
    ///
    /// The object path for `object` must be in the hierarchy rooted by the
    /// object path for `manager`.
    ///
    /// Note that `manager` will take a reference on `object` for as long as
    /// it is exported.
    extern fn g_dbus_object_manager_server_export(p_manager: *DBusObjectManagerServer, p_object: *gio.DBusObjectSkeleton) void;
    pub const @"export" = g_dbus_object_manager_server_export;

    /// Like `gio.DBusObjectManagerServer.@"export"` but appends a string of
    /// the form _N (with N being a natural number) to `object`'s object path
    /// if an object with the given path already exists. As such, the
    /// `gio.DBusObjectProxy.properties.g`-object-path property of `object` may be modified.
    extern fn g_dbus_object_manager_server_export_uniquely(p_manager: *DBusObjectManagerServer, p_object: *gio.DBusObjectSkeleton) void;
    pub const exportUniquely = g_dbus_object_manager_server_export_uniquely;

    /// Gets the `gio.DBusConnection` used by `manager`.
    extern fn g_dbus_object_manager_server_get_connection(p_manager: *DBusObjectManagerServer) ?*gio.DBusConnection;
    pub const getConnection = g_dbus_object_manager_server_get_connection;

    /// Returns whether `object` is currently exported on `manager`.
    extern fn g_dbus_object_manager_server_is_exported(p_manager: *DBusObjectManagerServer, p_object: *gio.DBusObjectSkeleton) c_int;
    pub const isExported = g_dbus_object_manager_server_is_exported;

    /// Exports all objects managed by `manager` on `connection`. If
    /// `connection` is `NULL`, stops exporting objects.
    extern fn g_dbus_object_manager_server_set_connection(p_manager: *DBusObjectManagerServer, p_connection: ?*gio.DBusConnection) void;
    pub const setConnection = g_dbus_object_manager_server_set_connection;

    /// If `manager` has an object at `path`, removes the object. Otherwise
    /// does nothing.
    ///
    /// Note that `object_path` must be in the hierarchy rooted by the
    /// object path for `manager`.
    extern fn g_dbus_object_manager_server_unexport(p_manager: *DBusObjectManagerServer, p_object_path: [*:0]const u8) c_int;
    pub const unexport = g_dbus_object_manager_server_unexport;

    extern fn g_dbus_object_manager_server_get_type() usize;
    pub const getGObjectType = g_dbus_object_manager_server_get_type;

    extern fn g_object_ref(p_self: *gio.DBusObjectManagerServer) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.DBusObjectManagerServer) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *DBusObjectManagerServer, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `GDBusObjectProxy` is an object used to represent a remote object
/// with one or more D-Bus interfaces. Normally, you don’t instantiate
/// a `GDBusObjectProxy` yourself — typically `gio.DBusObjectManagerClient`
/// is used to obtain it.
pub const DBusObjectProxy = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gio.DBusObject};
    pub const Class = gio.DBusObjectProxyClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.DBusObjectProxyPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The connection of the proxy.
        pub const g_connection = struct {
            pub const name = "g-connection";

            pub const Type = ?*gio.DBusConnection;
        };

        /// The object path of the proxy.
        pub const g_object_path = struct {
            pub const name = "g-object-path";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gio.DBusObjectProxy` for the given connection and
    /// object path.
    extern fn g_dbus_object_proxy_new(p_connection: *gio.DBusConnection, p_object_path: [*:0]const u8) *gio.DBusObjectProxy;
    pub const new = g_dbus_object_proxy_new;

    /// Gets the connection that `proxy` is for.
    extern fn g_dbus_object_proxy_get_connection(p_proxy: *DBusObjectProxy) *gio.DBusConnection;
    pub const getConnection = g_dbus_object_proxy_get_connection;

    extern fn g_dbus_object_proxy_get_type() usize;
    pub const getGObjectType = g_dbus_object_proxy_get_type;

    extern fn g_object_ref(p_self: *gio.DBusObjectProxy) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.DBusObjectProxy) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *DBusObjectProxy, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `GDBusObjectSkeleton` instance is essentially a group of D-Bus
/// interfaces. The set of exported interfaces on the object may be
/// dynamic and change at runtime.
///
/// This type is intended to be used with `gio.DBusObjectManager`.
pub const DBusObjectSkeleton = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gio.DBusObject};
    pub const Class = gio.DBusObjectSkeletonClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.DBusObjectSkeletonPrivate,

    pub const virtual_methods = struct {
        /// Signal class handler for the `gio.DBusObjectSkeleton.signals.authorize`-method signal.
        pub const authorize_method = struct {
            pub fn call(p_class: anytype, p_object: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_interface_: *gio.DBusInterfaceSkeleton, p_invocation: *gio.DBusMethodInvocation) c_int {
                return gobject.ext.as(DBusObjectSkeleton.Class, p_class).f_authorize_method.?(gobject.ext.as(DBusObjectSkeleton, p_object), p_interface_, p_invocation);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_object: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_interface_: *gio.DBusInterfaceSkeleton, p_invocation: *gio.DBusMethodInvocation) callconv(.C) c_int) void {
                gobject.ext.as(DBusObjectSkeleton.Class, p_class).f_authorize_method = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The object path where the object is exported.
        pub const g_object_path = struct {
            pub const name = "g-object-path";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {
        /// Emitted when a method is invoked by a remote caller and used to
        /// determine if the method call is authorized.
        ///
        /// This signal is like `gio.DBusInterfaceSkeleton`'s
        /// `gio.DBusInterfaceSkeleton.signals.g`-authorize-method signal,
        /// except that it is for the enclosing object.
        ///
        /// The default class handler just returns `TRUE`.
        pub const authorize_method = struct {
            pub const name = "authorize-method";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_interface: *gio.DBusInterfaceSkeleton, p_invocation: *gio.DBusMethodInvocation, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(DBusObjectSkeleton, p_instance))),
                    gobject.signalLookup("authorize-method", DBusObjectSkeleton.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gio.DBusObjectSkeleton`.
    extern fn g_dbus_object_skeleton_new(p_object_path: [*:0]const u8) *gio.DBusObjectSkeleton;
    pub const new = g_dbus_object_skeleton_new;

    /// Adds `interface_` to `object`.
    ///
    /// If `object` already contains a `gio.DBusInterfaceSkeleton` with the same
    /// interface name, it is removed before `interface_` is added.
    ///
    /// Note that `object` takes its own reference on `interface_` and holds
    /// it until removed.
    extern fn g_dbus_object_skeleton_add_interface(p_object: *DBusObjectSkeleton, p_interface_: *gio.DBusInterfaceSkeleton) void;
    pub const addInterface = g_dbus_object_skeleton_add_interface;

    /// This method simply calls `gio.DBusInterfaceSkeleton.flush` on all
    /// interfaces belonging to `object`. See that method for when flushing
    /// is useful.
    extern fn g_dbus_object_skeleton_flush(p_object: *DBusObjectSkeleton) void;
    pub const flush = g_dbus_object_skeleton_flush;

    /// Removes `interface_` from `object`.
    extern fn g_dbus_object_skeleton_remove_interface(p_object: *DBusObjectSkeleton, p_interface_: *gio.DBusInterfaceSkeleton) void;
    pub const removeInterface = g_dbus_object_skeleton_remove_interface;

    /// Removes the `gio.DBusInterface` with `interface_name` from `object`.
    ///
    /// If no D-Bus interface of the given interface exists, this function
    /// does nothing.
    extern fn g_dbus_object_skeleton_remove_interface_by_name(p_object: *DBusObjectSkeleton, p_interface_name: [*:0]const u8) void;
    pub const removeInterfaceByName = g_dbus_object_skeleton_remove_interface_by_name;

    /// Sets the object path for `object`.
    extern fn g_dbus_object_skeleton_set_object_path(p_object: *DBusObjectSkeleton, p_object_path: [*:0]const u8) void;
    pub const setObjectPath = g_dbus_object_skeleton_set_object_path;

    extern fn g_dbus_object_skeleton_get_type() usize;
    pub const getGObjectType = g_dbus_object_skeleton_get_type;

    extern fn g_object_ref(p_self: *gio.DBusObjectSkeleton) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.DBusObjectSkeleton) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *DBusObjectSkeleton, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GDBusProxy` is a base class used for proxies to access a D-Bus
/// interface on a remote object. A `GDBusProxy` can be constructed for
/// both well-known and unique names.
///
/// By default, `GDBusProxy` will cache all properties (and listen to
/// changes) of the remote object, and proxy all signals that get
/// emitted. This behaviour can be changed by passing suitable
/// `gio.DBusProxyFlags` when the proxy is created. If the proxy is for a
/// well-known name, the property cache is flushed when the name owner
/// vanishes and reloaded when a name owner appears.
///
/// The unique name owner of the proxy’s name is tracked and can be read from
/// `gio.DBusProxy.properties.g_name_owner`. Connect to the
/// `gobject.Object.signals.notify` signal to get notified of changes.
/// Additionally, only signals and property changes emitted from the current name
/// owner are considered and calls are always sent to the current name owner.
/// This avoids a number of race conditions when the name is lost by one owner
/// and claimed by another. However, if no name owner currently exists,
/// then calls will be sent to the well-known name which may result in
/// the message bus launching an owner (unless
/// `G_DBUS_PROXY_FLAGS_DO_NOT_AUTO_START` is set).
///
/// If the proxy is for a stateless D-Bus service, where the name owner may
/// be started and stopped between calls, the
/// `gio.DBusProxy.properties.g_name_owner` tracking of `GDBusProxy` will cause the
/// proxy to drop signal and property changes from the service after it has
/// restarted for the first time. When interacting with a stateless D-Bus
/// service, do not use `GDBusProxy` — use direct D-Bus method calls and signal
/// connections.
///
/// The generic `gio.DBusProxy.signals.g_properties_changed` and
/// `gio.DBusProxy.signals.g_signal` signals are not very convenient to work
/// with. Therefore, the recommended way of working with proxies is to subclass
/// `GDBusProxy`, and have more natural properties and signals in your derived
/// class. This [example](migrating-gdbus.html`using`-gdbus-codegen) shows how
/// this can easily be done using the [`gdbus-codegen`](gdbus-codegen.html) tool.
///
/// A `GDBusProxy` instance can be used from multiple threads but note
/// that all signals (e.g. `gio.DBusProxy.signals.g_signal`,
/// `gio.DBusProxy.signals.g_properties_changed` and
/// `gobject.Object.signals.notify`) are emitted in the thread-default main
/// context (see `glib.MainContext.pushThreadDefault`) of the thread
/// where the instance was constructed.
///
/// An example using a proxy for a well-known name can be found in
/// [`gdbus-example-watch-proxy.c`](https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gdbus-example-watch-proxy.c).
pub const DBusProxy = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{ gio.AsyncInitable, gio.DBusInterface, gio.Initable };
    pub const Class = gio.DBusProxyClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.DBusProxyPrivate,

    pub const virtual_methods = struct {
        /// Signal class handler for the `gio.DBusProxy.signals.g`-properties-changed signal.
        pub const g_properties_changed = struct {
            pub fn call(p_class: anytype, p_proxy: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_changed_properties: *glib.Variant, p_invalidated_properties: *const [*:0]const u8) void {
                return gobject.ext.as(DBusProxy.Class, p_class).f_g_properties_changed.?(gobject.ext.as(DBusProxy, p_proxy), p_changed_properties, p_invalidated_properties);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_proxy: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_changed_properties: *glib.Variant, p_invalidated_properties: *const [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(DBusProxy.Class, p_class).f_g_properties_changed = @ptrCast(p_implementation);
            }
        };

        /// Signal class handler for the `gio.DBusProxy.signals.g`-signal signal.
        pub const g_signal = struct {
            pub fn call(p_class: anytype, p_proxy: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_sender_name: [*:0]const u8, p_signal_name: [*:0]const u8, p_parameters: *glib.Variant) void {
                return gobject.ext.as(DBusProxy.Class, p_class).f_g_signal.?(gobject.ext.as(DBusProxy, p_proxy), p_sender_name, p_signal_name, p_parameters);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_proxy: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_sender_name: [*:0]const u8, p_signal_name: [*:0]const u8, p_parameters: *glib.Variant) callconv(.C) void) void {
                gobject.ext.as(DBusProxy.Class, p_class).f_g_signal = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// If this property is not `G_BUS_TYPE_NONE`, then
        /// `gio.DBusProxy.properties.g`-connection must be `NULL` and will be set to the
        /// `gio.DBusConnection` obtained by calling `gio.busGet` with the value
        /// of this property.
        pub const g_bus_type = struct {
            pub const name = "g-bus-type";

            pub const Type = gio.BusType;
        };

        /// The `gio.DBusConnection` the proxy is for.
        pub const g_connection = struct {
            pub const name = "g-connection";

            pub const Type = ?*gio.DBusConnection;
        };

        /// The timeout to use if -1 (specifying default timeout) is passed
        /// as `timeout_msec` in the `gio.DBusProxy.call` and
        /// `gio.DBusProxy.callSync` functions.
        ///
        /// This allows applications to set a proxy-wide timeout for all
        /// remote method invocations on the proxy. If this property is -1,
        /// the default timeout (typically 25 seconds) is used. If set to
        /// `G_MAXINT`, then no timeout is used.
        pub const g_default_timeout = struct {
            pub const name = "g-default-timeout";

            pub const Type = c_int;
        };

        /// Flags from the `gio.DBusProxyFlags` enumeration.
        pub const g_flags = struct {
            pub const name = "g-flags";

            pub const Type = gio.DBusProxyFlags;
        };

        /// Ensure that interactions with this proxy conform to the given
        /// interface. This is mainly to ensure that malformed data received
        /// from the other peer is ignored. The given `gio.DBusInterfaceInfo` is
        /// said to be the "expected interface".
        ///
        /// The checks performed are:
        /// - When completing a method call, if the type signature of
        ///   the reply message isn't what's expected, the reply is
        ///   discarded and the `glib.Error` is set to `G_IO_ERROR_INVALID_ARGUMENT`.
        ///
        /// - Received signals that have a type signature mismatch are dropped and
        ///   a warning is logged via `g_warning`.
        ///
        /// - Properties received via the initial ``GetAll`` call or via the
        ///   `::PropertiesChanged` signal (on the
        ///   [org.freedesktop.DBus.Properties](http://dbus.freedesktop.org/doc/dbus-specification.html`standard`-interfaces-properties)
        ///   interface) or set using `gio.DBusProxy.setCachedProperty`
        ///   with a type signature mismatch are ignored and a warning is
        ///   logged via `g_warning`.
        ///
        /// Note that these checks are never done on methods, signals and
        /// properties that are not referenced in the given
        /// `gio.DBusInterfaceInfo`, since extending a D-Bus interface on the
        /// service-side is not considered an ABI break.
        pub const g_interface_info = struct {
            pub const name = "g-interface-info";

            pub const Type = ?*gio.DBusInterfaceInfo;
        };

        /// The D-Bus interface name the proxy is for.
        pub const g_interface_name = struct {
            pub const name = "g-interface-name";

            pub const Type = ?[*:0]u8;
        };

        /// The well-known or unique name that the proxy is for.
        pub const g_name = struct {
            pub const name = "g-name";

            pub const Type = ?[*:0]u8;
        };

        /// The unique name that owns `gio.DBusProxy.properties.g`-name or `NULL` if no-one
        /// currently owns that name. You may connect to `gobject.Object.signals.notify` signal to
        /// track changes to this property.
        pub const g_name_owner = struct {
            pub const name = "g-name-owner";

            pub const Type = ?[*:0]u8;
        };

        /// The object path the proxy is for.
        pub const g_object_path = struct {
            pub const name = "g-object-path";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {
        /// Emitted when one or more D-Bus properties on `proxy` changes. The
        /// local cache has already been updated when this signal fires. Note
        /// that both `changed_properties` and `invalidated_properties` are
        /// guaranteed to never be `NULL` (either may be empty though).
        ///
        /// If the proxy has the flag
        /// `G_DBUS_PROXY_FLAGS_GET_INVALIDATED_PROPERTIES` set, then
        /// `invalidated_properties` will always be empty.
        ///
        /// This signal corresponds to the
        /// `PropertiesChanged` D-Bus signal on the
        /// `org.freedesktop.DBus.Properties` interface.
        pub const g_properties_changed = struct {
            pub const name = "g-properties-changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_changed_properties: *glib.Variant, p_invalidated_properties: [*][*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(DBusProxy, p_instance))),
                    gobject.signalLookup("g-properties-changed", DBusProxy.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when a signal from the remote object and interface that `proxy` is for, has been received.
        ///
        /// Since 2.72 this signal supports detailed connections. You can connect to
        /// the detailed signal `g-signal::x` in order to receive callbacks only when
        /// signal `x` is received from the remote object.
        pub const g_signal = struct {
            pub const name = "g-signal";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_sender_name: ?[*:0]u8, p_signal_name: [*:0]u8, p_parameters: *glib.Variant, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(DBusProxy, p_instance))),
                    gobject.signalLookup("g-signal", DBusProxy.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a proxy for accessing `interface_name` on the remote object
    /// at `object_path` owned by `name` at `connection` and asynchronously
    /// loads D-Bus properties unless the
    /// `G_DBUS_PROXY_FLAGS_DO_NOT_LOAD_PROPERTIES` flag is used. Connect to
    /// the `gio.DBusProxy.signals.g`-properties-changed signal to get notified about
    /// property changes.
    ///
    /// If the `G_DBUS_PROXY_FLAGS_DO_NOT_CONNECT_SIGNALS` flag is not set, also sets up
    /// match rules for signals. Connect to the `gio.DBusProxy.signals.g`-signal signal
    /// to handle signals from the remote object.
    ///
    /// If both `G_DBUS_PROXY_FLAGS_DO_NOT_LOAD_PROPERTIES` and
    /// `G_DBUS_PROXY_FLAGS_DO_NOT_CONNECT_SIGNALS` are set, this constructor is
    /// guaranteed to complete immediately without blocking.
    ///
    /// If `name` is a well-known name and the
    /// `G_DBUS_PROXY_FLAGS_DO_NOT_AUTO_START` and `G_DBUS_PROXY_FLAGS_DO_NOT_AUTO_START_AT_CONSTRUCTION`
    /// flags aren't set and no name owner currently exists, the message bus
    /// will be requested to launch a name owner for the name.
    ///
    /// This is a failable asynchronous constructor - when the proxy is
    /// ready, `callback` will be invoked and you can use
    /// `gio.DBusProxy.newFinish` to get the result.
    ///
    /// See `gio.DBusProxy.newSync` and for a synchronous version of this constructor.
    ///
    /// `gio.DBusProxy` is used in this [example][gdbus-wellknown-proxy].
    extern fn g_dbus_proxy_new(p_connection: *gio.DBusConnection, p_flags: gio.DBusProxyFlags, p_info: ?*gio.DBusInterfaceInfo, p_name: ?[*:0]const u8, p_object_path: [*:0]const u8, p_interface_name: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const new = g_dbus_proxy_new;

    /// Like `gio.DBusProxy.new` but takes a `gio.BusType` instead of a `gio.DBusConnection`.
    ///
    /// `gio.DBusProxy` is used in this [example][gdbus-wellknown-proxy].
    extern fn g_dbus_proxy_new_for_bus(p_bus_type: gio.BusType, p_flags: gio.DBusProxyFlags, p_info: ?*gio.DBusInterfaceInfo, p_name: [*:0]const u8, p_object_path: [*:0]const u8, p_interface_name: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const newForBus = g_dbus_proxy_new_for_bus;

    /// Finishes creating a `gio.DBusProxy`.
    extern fn g_dbus_proxy_new_finish(p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.DBusProxy;
    pub const newFinish = g_dbus_proxy_new_finish;

    /// Finishes creating a `gio.DBusProxy`.
    extern fn g_dbus_proxy_new_for_bus_finish(p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.DBusProxy;
    pub const newForBusFinish = g_dbus_proxy_new_for_bus_finish;

    /// Like `gio.DBusProxy.newSync` but takes a `gio.BusType` instead of a `gio.DBusConnection`.
    ///
    /// `gio.DBusProxy` is used in this [example][gdbus-wellknown-proxy].
    extern fn g_dbus_proxy_new_for_bus_sync(p_bus_type: gio.BusType, p_flags: gio.DBusProxyFlags, p_info: ?*gio.DBusInterfaceInfo, p_name: [*:0]const u8, p_object_path: [*:0]const u8, p_interface_name: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.DBusProxy;
    pub const newForBusSync = g_dbus_proxy_new_for_bus_sync;

    /// Creates a proxy for accessing `interface_name` on the remote object
    /// at `object_path` owned by `name` at `connection` and synchronously
    /// loads D-Bus properties unless the
    /// `G_DBUS_PROXY_FLAGS_DO_NOT_LOAD_PROPERTIES` flag is used.
    ///
    /// If the `G_DBUS_PROXY_FLAGS_DO_NOT_CONNECT_SIGNALS` flag is not set, also sets up
    /// match rules for signals. Connect to the `gio.DBusProxy.signals.g`-signal signal
    /// to handle signals from the remote object.
    ///
    /// If both `G_DBUS_PROXY_FLAGS_DO_NOT_LOAD_PROPERTIES` and
    /// `G_DBUS_PROXY_FLAGS_DO_NOT_CONNECT_SIGNALS` are set, this constructor is
    /// guaranteed to return immediately without blocking.
    ///
    /// If `name` is a well-known name and the
    /// `G_DBUS_PROXY_FLAGS_DO_NOT_AUTO_START` and `G_DBUS_PROXY_FLAGS_DO_NOT_AUTO_START_AT_CONSTRUCTION`
    /// flags aren't set and no name owner currently exists, the message bus
    /// will be requested to launch a name owner for the name.
    ///
    /// This is a synchronous failable constructor. See `gio.DBusProxy.new`
    /// and `gio.DBusProxy.newFinish` for the asynchronous version.
    ///
    /// `gio.DBusProxy` is used in this [example][gdbus-wellknown-proxy].
    extern fn g_dbus_proxy_new_sync(p_connection: *gio.DBusConnection, p_flags: gio.DBusProxyFlags, p_info: ?*gio.DBusInterfaceInfo, p_name: ?[*:0]const u8, p_object_path: [*:0]const u8, p_interface_name: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.DBusProxy;
    pub const newSync = g_dbus_proxy_new_sync;

    /// Asynchronously invokes the `method_name` method on `proxy`.
    ///
    /// If `method_name` contains any dots, then `name` is split into interface and
    /// method name parts. This allows using `proxy` for invoking methods on
    /// other interfaces.
    ///
    /// If the `gio.DBusConnection` associated with `proxy` is closed then
    /// the operation will fail with `G_IO_ERROR_CLOSED`. If
    /// `cancellable` is canceled, the operation will fail with
    /// `G_IO_ERROR_CANCELLED`. If `parameters` contains a value not
    /// compatible with the D-Bus protocol, the operation fails with
    /// `G_IO_ERROR_INVALID_ARGUMENT`.
    ///
    /// If the `parameters` `glib.Variant` is floating, it is consumed. This allows
    /// convenient 'inline' use of `glib.Variant.new`, e.g.:
    /// ```
    ///  g_dbus_proxy_call (proxy,
    ///                     "TwoStrings",
    ///                     g_variant_new ("(ss)",
    ///                                    "Thing One",
    ///                                    "Thing Two"),
    ///                     G_DBUS_CALL_FLAGS_NONE,
    ///                     -1,
    ///                     NULL,
    ///                     (GAsyncReadyCallback) two_strings_done,
    ///                     &data);
    /// ```
    ///
    /// If `proxy` has an expected interface (see
    /// `gio.DBusProxy.properties.g`-interface-info) and `method_name` is referenced by it,
    /// then the return value is checked against the return type.
    ///
    /// This is an asynchronous method. When the operation is finished,
    /// `callback` will be invoked in the
    /// [thread-default main context][g-main-context-push-thread-default]
    /// of the thread you are calling this method from.
    /// You can then call `gio.DBusProxy.callFinish` to get the result of
    /// the operation. See `gio.DBusProxy.callSync` for the synchronous
    /// version of this method.
    ///
    /// If `callback` is `NULL` then the D-Bus method call message will be sent with
    /// the `G_DBUS_MESSAGE_FLAGS_NO_REPLY_EXPECTED` flag set.
    extern fn g_dbus_proxy_call(p_proxy: *DBusProxy, p_method_name: [*:0]const u8, p_parameters: ?*glib.Variant, p_flags: gio.DBusCallFlags, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const call = g_dbus_proxy_call;

    /// Finishes an operation started with `gio.DBusProxy.call`.
    extern fn g_dbus_proxy_call_finish(p_proxy: *DBusProxy, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.Variant;
    pub const callFinish = g_dbus_proxy_call_finish;

    /// Synchronously invokes the `method_name` method on `proxy`.
    ///
    /// If `method_name` contains any dots, then `name` is split into interface and
    /// method name parts. This allows using `proxy` for invoking methods on
    /// other interfaces.
    ///
    /// If the `gio.DBusConnection` associated with `proxy` is disconnected then
    /// the operation will fail with `G_IO_ERROR_CLOSED`. If
    /// `cancellable` is canceled, the operation will fail with
    /// `G_IO_ERROR_CANCELLED`. If `parameters` contains a value not
    /// compatible with the D-Bus protocol, the operation fails with
    /// `G_IO_ERROR_INVALID_ARGUMENT`.
    ///
    /// If the `parameters` `glib.Variant` is floating, it is consumed. This allows
    /// convenient 'inline' use of `glib.Variant.new`, e.g.:
    /// ```
    ///  g_dbus_proxy_call_sync (proxy,
    ///                          "TwoStrings",
    ///                          g_variant_new ("(ss)",
    ///                                         "Thing One",
    ///                                         "Thing Two"),
    ///                          G_DBUS_CALL_FLAGS_NONE,
    ///                          -1,
    ///                          NULL,
    ///                          &error);
    /// ```
    ///
    /// The calling thread is blocked until a reply is received. See
    /// `gio.DBusProxy.call` for the asynchronous version of this
    /// method.
    ///
    /// If `proxy` has an expected interface (see
    /// `gio.DBusProxy.properties.g`-interface-info) and `method_name` is referenced by it,
    /// then the return value is checked against the return type.
    extern fn g_dbus_proxy_call_sync(p_proxy: *DBusProxy, p_method_name: [*:0]const u8, p_parameters: ?*glib.Variant, p_flags: gio.DBusCallFlags, p_timeout_msec: c_int, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*glib.Variant;
    pub const callSync = g_dbus_proxy_call_sync;

    /// Like `gio.DBusProxy.call` but also takes a `gio.UnixFDList` object.
    ///
    /// This method is only available on UNIX.
    extern fn g_dbus_proxy_call_with_unix_fd_list(p_proxy: *DBusProxy, p_method_name: [*:0]const u8, p_parameters: ?*glib.Variant, p_flags: gio.DBusCallFlags, p_timeout_msec: c_int, p_fd_list: ?*gio.UnixFDList, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const callWithUnixFdList = g_dbus_proxy_call_with_unix_fd_list;

    /// Finishes an operation started with `gio.DBusProxy.callWithUnixFdList`.
    extern fn g_dbus_proxy_call_with_unix_fd_list_finish(p_proxy: *DBusProxy, p_out_fd_list: ?**gio.UnixFDList, p_res: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.Variant;
    pub const callWithUnixFdListFinish = g_dbus_proxy_call_with_unix_fd_list_finish;

    /// Like `gio.DBusProxy.callSync` but also takes and returns `gio.UnixFDList` objects.
    ///
    /// This method is only available on UNIX.
    extern fn g_dbus_proxy_call_with_unix_fd_list_sync(p_proxy: *DBusProxy, p_method_name: [*:0]const u8, p_parameters: ?*glib.Variant, p_flags: gio.DBusCallFlags, p_timeout_msec: c_int, p_fd_list: ?*gio.UnixFDList, p_out_fd_list: ?**gio.UnixFDList, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*glib.Variant;
    pub const callWithUnixFdListSync = g_dbus_proxy_call_with_unix_fd_list_sync;

    /// Looks up the value for a property from the cache. This call does no
    /// blocking IO.
    ///
    /// If `proxy` has an expected interface (see
    /// `gio.DBusProxy.properties.g`-interface-info) and `property_name` is referenced by
    /// it, then `value` is checked against the type of the property.
    extern fn g_dbus_proxy_get_cached_property(p_proxy: *DBusProxy, p_property_name: [*:0]const u8) ?*glib.Variant;
    pub const getCachedProperty = g_dbus_proxy_get_cached_property;

    /// Gets the names of all cached properties on `proxy`.
    extern fn g_dbus_proxy_get_cached_property_names(p_proxy: *DBusProxy) ?[*][*:0]u8;
    pub const getCachedPropertyNames = g_dbus_proxy_get_cached_property_names;

    /// Gets the connection `proxy` is for.
    extern fn g_dbus_proxy_get_connection(p_proxy: *DBusProxy) *gio.DBusConnection;
    pub const getConnection = g_dbus_proxy_get_connection;

    /// Gets the timeout to use if -1 (specifying default timeout) is
    /// passed as `timeout_msec` in the `gio.DBusProxy.call` and
    /// `gio.DBusProxy.callSync` functions.
    ///
    /// See the `gio.DBusProxy.properties.g`-default-timeout property for more details.
    extern fn g_dbus_proxy_get_default_timeout(p_proxy: *DBusProxy) c_int;
    pub const getDefaultTimeout = g_dbus_proxy_get_default_timeout;

    /// Gets the flags that `proxy` was constructed with.
    extern fn g_dbus_proxy_get_flags(p_proxy: *DBusProxy) gio.DBusProxyFlags;
    pub const getFlags = g_dbus_proxy_get_flags;

    /// Returns the `gio.DBusInterfaceInfo`, if any, specifying the interface
    /// that `proxy` conforms to. See the `gio.DBusProxy.properties.g`-interface-info
    /// property for more details.
    extern fn g_dbus_proxy_get_interface_info(p_proxy: *DBusProxy) ?*gio.DBusInterfaceInfo;
    pub const getInterfaceInfo = g_dbus_proxy_get_interface_info;

    /// Gets the D-Bus interface name `proxy` is for.
    extern fn g_dbus_proxy_get_interface_name(p_proxy: *DBusProxy) [*:0]const u8;
    pub const getInterfaceName = g_dbus_proxy_get_interface_name;

    /// Gets the name that `proxy` was constructed for.
    ///
    /// When connected to a message bus, this will usually be non-`NULL`.
    /// However, it may be `NULL` for a proxy that communicates using a peer-to-peer
    /// pattern.
    extern fn g_dbus_proxy_get_name(p_proxy: *DBusProxy) ?[*:0]const u8;
    pub const getName = g_dbus_proxy_get_name;

    /// The unique name that owns the name that `proxy` is for or `NULL` if
    /// no-one currently owns that name. You may connect to the
    /// `gobject.Object.signals.notify` signal to track changes to the
    /// `gio.DBusProxy.properties.g`-name-owner property.
    extern fn g_dbus_proxy_get_name_owner(p_proxy: *DBusProxy) ?[*:0]u8;
    pub const getNameOwner = g_dbus_proxy_get_name_owner;

    /// Gets the object path `proxy` is for.
    extern fn g_dbus_proxy_get_object_path(p_proxy: *DBusProxy) [*:0]const u8;
    pub const getObjectPath = g_dbus_proxy_get_object_path;

    /// If `value` is not `NULL`, sets the cached value for the property with
    /// name `property_name` to the value in `value`.
    ///
    /// If `value` is `NULL`, then the cached value is removed from the
    /// property cache.
    ///
    /// If `proxy` has an expected interface (see
    /// `gio.DBusProxy.properties.g`-interface-info) and `property_name` is referenced by
    /// it, then `value` is checked against the type of the property.
    ///
    /// If the `value` `glib.Variant` is floating, it is consumed. This allows
    /// convenient 'inline' use of `glib.Variant.new`, e.g.
    /// ```
    ///  g_dbus_proxy_set_cached_property (proxy,
    ///                                    "SomeProperty",
    ///                                    g_variant_new ("(si)",
    ///                                                  "A String",
    ///                                                  42));
    /// ```
    ///
    /// Normally you will not need to use this method since `proxy`
    /// is tracking changes using the
    /// `org.freedesktop.DBus.Properties.PropertiesChanged`
    /// D-Bus signal. However, for performance reasons an object may
    /// decide to not use this signal for some properties and instead
    /// use a proprietary out-of-band mechanism to transmit changes.
    ///
    /// As a concrete example, consider an object with a property
    /// `ChatroomParticipants` which is an array of strings. Instead of
    /// transmitting the same (long) array every time the property changes,
    /// it is more efficient to only transmit the delta using e.g. signals
    /// `ChatroomParticipantJoined(String name)` and
    /// `ChatroomParticipantParted(String name)`.
    extern fn g_dbus_proxy_set_cached_property(p_proxy: *DBusProxy, p_property_name: [*:0]const u8, p_value: ?*glib.Variant) void;
    pub const setCachedProperty = g_dbus_proxy_set_cached_property;

    /// Sets the timeout to use if -1 (specifying default timeout) is
    /// passed as `timeout_msec` in the `gio.DBusProxy.call` and
    /// `gio.DBusProxy.callSync` functions.
    ///
    /// See the `gio.DBusProxy.properties.g`-default-timeout property for more details.
    extern fn g_dbus_proxy_set_default_timeout(p_proxy: *DBusProxy, p_timeout_msec: c_int) void;
    pub const setDefaultTimeout = g_dbus_proxy_set_default_timeout;

    /// Ensure that interactions with `proxy` conform to the given
    /// interface. See the `gio.DBusProxy.properties.g`-interface-info property for more
    /// details.
    extern fn g_dbus_proxy_set_interface_info(p_proxy: *DBusProxy, p_info: ?*gio.DBusInterfaceInfo) void;
    pub const setInterfaceInfo = g_dbus_proxy_set_interface_info;

    extern fn g_dbus_proxy_get_type() usize;
    pub const getGObjectType = g_dbus_proxy_get_type;

    extern fn g_object_ref(p_self: *gio.DBusProxy) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.DBusProxy) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *DBusProxy, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GDBusServer` is a helper for listening to and accepting D-Bus
/// connections. This can be used to create a new D-Bus server, allowing two
/// peers to use the D-Bus protocol for their own specialized communication.
/// A server instance provided in this way will not perform message routing or
/// implement the
/// [`org.freedesktop.DBus` interface](https://dbus.freedesktop.org/doc/dbus-specification.html`message`-bus-messages).
///
/// To just export an object on a well-known name on a message bus, such as the
/// session or system bus, you should instead use `gio.busOwnName`.
///
/// An example of peer-to-peer communication with GDBus can be found
/// in [gdbus-example-peer.c](https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gdbus-example-peer.c).
///
/// Note that a minimal `GDBusServer` will accept connections from any
/// peer. In many use-cases it will be necessary to add a
/// `gio.DBusAuthObserver` that only accepts connections that have
/// successfully authenticated as the same user that is running the
/// `GDBusServer`. Since GLib 2.68 this can be achieved more simply by passing
/// the `G_DBUS_SERVER_FLAGS_AUTHENTICATION_REQUIRE_SAME_USER` flag to the
/// server.
pub const DBusServer = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gio.Initable};
    pub const Class = opaque {
        pub const Instance = DBusServer;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Whether the server is currently active.
        pub const active = struct {
            pub const name = "active";

            pub const Type = c_int;
        };

        /// The D-Bus address to listen on.
        pub const address = struct {
            pub const name = "address";

            pub const Type = ?[*:0]u8;
        };

        /// A `gio.DBusAuthObserver` object to assist in the authentication process or `NULL`.
        pub const authentication_observer = struct {
            pub const name = "authentication-observer";

            pub const Type = ?*gio.DBusAuthObserver;
        };

        /// The D-Bus address that clients can use.
        pub const client_address = struct {
            pub const name = "client-address";

            pub const Type = ?[*:0]u8;
        };

        /// Flags from the `gio.DBusServerFlags` enumeration.
        pub const flags = struct {
            pub const name = "flags";

            pub const Type = gio.DBusServerFlags;
        };

        /// The GUID of the server.
        ///
        /// See `gio.DBusConnection.properties.guid` for more details.
        pub const guid = struct {
            pub const name = "guid";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {
        /// Emitted when a new authenticated connection has been made. Use
        /// `gio.DBusConnection.getPeerCredentials` to figure out what
        /// identity (if any), was authenticated.
        ///
        /// If you want to accept the connection, take a reference to the
        /// `connection` object and return `TRUE`. When you are done with the
        /// connection call `gio.DBusConnection.close` and give up your
        /// reference. Note that the other peer may disconnect at any time -
        /// a typical thing to do when accepting a connection is to listen to
        /// the `gio.DBusConnection.signals.closed` signal.
        ///
        /// If `gio.DBusServer.properties.flags` contains `G_DBUS_SERVER_FLAGS_RUN_IN_THREAD`
        /// then the signal is emitted in a new thread dedicated to the
        /// connection. Otherwise the signal is emitted in the
        /// [thread-default main context][g-main-context-push-thread-default]
        /// of the thread that `server` was constructed in.
        ///
        /// You are guaranteed that signal handlers for this signal runs
        /// before incoming messages on `connection` are processed. This means
        /// that it's suitable to call `gio.DBusConnection.registerObject` or
        /// similar from the signal handler.
        pub const new_connection = struct {
            pub const name = "new-connection";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_connection: *gio.DBusConnection, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(DBusServer, p_instance))),
                    gobject.signalLookup("new-connection", DBusServer.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new D-Bus server that listens on the first address in
    /// `address` that works.
    ///
    /// Once constructed, you can use `gio.DBusServer.getClientAddress` to
    /// get a D-Bus address string that clients can use to connect.
    ///
    /// To have control over the available authentication mechanisms and
    /// the users that are authorized to connect, it is strongly recommended
    /// to provide a non-`NULL` `gio.DBusAuthObserver`.
    ///
    /// Connect to the `gio.DBusServer.signals.new`-connection signal to handle
    /// incoming connections.
    ///
    /// The returned `gio.DBusServer` isn't active - you have to start it with
    /// `gio.DBusServer.start`.
    ///
    /// `gio.DBusServer` is used in this [example][gdbus-peer-to-peer].
    ///
    /// This is a synchronous failable constructor. There is currently no
    /// asynchronous version.
    extern fn g_dbus_server_new_sync(p_address: [*:0]const u8, p_flags: gio.DBusServerFlags, p_guid: [*:0]const u8, p_observer: ?*gio.DBusAuthObserver, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.DBusServer;
    pub const newSync = g_dbus_server_new_sync;

    /// Gets a
    /// [D-Bus address](https://dbus.freedesktop.org/doc/dbus-specification.html`addresses`)
    /// string that can be used by clients to connect to `server`.
    ///
    /// This is valid and non-empty if initializing the `gio.DBusServer` succeeded.
    extern fn g_dbus_server_get_client_address(p_server: *DBusServer) [*:0]const u8;
    pub const getClientAddress = g_dbus_server_get_client_address;

    /// Gets the flags for `server`.
    extern fn g_dbus_server_get_flags(p_server: *DBusServer) gio.DBusServerFlags;
    pub const getFlags = g_dbus_server_get_flags;

    /// Gets the GUID for `server`, as provided to `gio.DBusServer.newSync`.
    extern fn g_dbus_server_get_guid(p_server: *DBusServer) [*:0]const u8;
    pub const getGuid = g_dbus_server_get_guid;

    /// Gets whether `server` is active.
    extern fn g_dbus_server_is_active(p_server: *DBusServer) c_int;
    pub const isActive = g_dbus_server_is_active;

    /// Starts `server`.
    extern fn g_dbus_server_start(p_server: *DBusServer) void;
    pub const start = g_dbus_server_start;

    /// Stops `server`.
    extern fn g_dbus_server_stop(p_server: *DBusServer) void;
    pub const stop = g_dbus_server_stop;

    extern fn g_dbus_server_get_type() usize;
    pub const getGObjectType = g_dbus_server_get_type;

    extern fn g_object_ref(p_self: *gio.DBusServer) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.DBusServer) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *DBusServer, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Data input stream implements `gio.InputStream` and includes functions
/// for reading structured data directly from a binary input stream.
pub const DataInputStream = extern struct {
    pub const Parent = gio.BufferedInputStream;
    pub const Implements = [_]type{gio.Seekable};
    pub const Class = gio.DataInputStreamClass;
    f_parent_instance: gio.BufferedInputStream,
    f_priv: ?*gio.DataInputStreamPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The :byte-order property determines the byte ordering that
        /// is used when reading multi-byte entities (such as integers)
        /// from the stream.
        pub const byte_order = struct {
            pub const name = "byte-order";

            pub const Type = gio.DataStreamByteOrder;
        };

        /// The :newline-type property determines what is considered
        /// as a line ending when reading complete lines from the stream.
        pub const newline_type = struct {
            pub const name = "newline-type";

            pub const Type = gio.DataStreamNewlineType;
        };
    };

    pub const signals = struct {};

    /// Creates a new data input stream for the `base_stream`.
    extern fn g_data_input_stream_new(p_base_stream: *gio.InputStream) *gio.DataInputStream;
    pub const new = g_data_input_stream_new;

    /// Gets the byte order for the data input stream.
    extern fn g_data_input_stream_get_byte_order(p_stream: *DataInputStream) gio.DataStreamByteOrder;
    pub const getByteOrder = g_data_input_stream_get_byte_order;

    /// Gets the current newline type for the `stream`.
    extern fn g_data_input_stream_get_newline_type(p_stream: *DataInputStream) gio.DataStreamNewlineType;
    pub const getNewlineType = g_data_input_stream_get_newline_type;

    /// Reads an unsigned 8-bit/1-byte value from `stream`.
    extern fn g_data_input_stream_read_byte(p_stream: *DataInputStream, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) u8;
    pub const readByte = g_data_input_stream_read_byte;

    /// Reads a 16-bit/2-byte value from `stream`.
    ///
    /// In order to get the correct byte order for this read operation,
    /// see `gio.DataInputStream.getByteOrder` and `gio.DataInputStream.setByteOrder`.
    extern fn g_data_input_stream_read_int16(p_stream: *DataInputStream, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) i16;
    pub const readInt16 = g_data_input_stream_read_int16;

    /// Reads a signed 32-bit/4-byte value from `stream`.
    ///
    /// In order to get the correct byte order for this read operation,
    /// see `gio.DataInputStream.getByteOrder` and `gio.DataInputStream.setByteOrder`.
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned.
    extern fn g_data_input_stream_read_int32(p_stream: *DataInputStream, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) i32;
    pub const readInt32 = g_data_input_stream_read_int32;

    /// Reads a 64-bit/8-byte value from `stream`.
    ///
    /// In order to get the correct byte order for this read operation,
    /// see `gio.DataInputStream.getByteOrder` and `gio.DataInputStream.setByteOrder`.
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned.
    extern fn g_data_input_stream_read_int64(p_stream: *DataInputStream, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) i64;
    pub const readInt64 = g_data_input_stream_read_int64;

    /// Reads a line from the data input stream.  Note that no encoding
    /// checks or conversion is performed; the input is not guaranteed to
    /// be UTF-8, and may in fact have embedded NUL characters.
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned.
    extern fn g_data_input_stream_read_line(p_stream: *DataInputStream, p_length: ?*usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?[*]u8;
    pub const readLine = g_data_input_stream_read_line;

    /// The asynchronous version of `gio.DataInputStream.readLine`.  It is
    /// an error to have two outstanding calls to this function.
    ///
    /// When the operation is finished, `callback` will be called. You
    /// can then call `gio.DataInputStream.readLineFinish` to get
    /// the result of the operation.
    extern fn g_data_input_stream_read_line_async(p_stream: *DataInputStream, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const readLineAsync = g_data_input_stream_read_line_async;

    /// Finish an asynchronous call started by
    /// `gio.DataInputStream.readLineAsync`.  Note the warning about
    /// string encoding in `gio.DataInputStream.readLine` applies here as
    /// well.
    extern fn g_data_input_stream_read_line_finish(p_stream: *DataInputStream, p_result: *gio.AsyncResult, p_length: ?*usize, p_error: ?*?*glib.Error) ?[*]u8;
    pub const readLineFinish = g_data_input_stream_read_line_finish;

    /// Finish an asynchronous call started by
    /// `gio.DataInputStream.readLineAsync`.
    extern fn g_data_input_stream_read_line_finish_utf8(p_stream: *DataInputStream, p_result: *gio.AsyncResult, p_length: ?*usize, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const readLineFinishUtf8 = g_data_input_stream_read_line_finish_utf8;

    /// Reads a UTF-8 encoded line from the data input stream.
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned.
    extern fn g_data_input_stream_read_line_utf8(p_stream: *DataInputStream, p_length: ?*usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const readLineUtf8 = g_data_input_stream_read_line_utf8;

    /// Reads an unsigned 16-bit/2-byte value from `stream`.
    ///
    /// In order to get the correct byte order for this read operation,
    /// see `gio.DataInputStream.getByteOrder` and `gio.DataInputStream.setByteOrder`.
    extern fn g_data_input_stream_read_uint16(p_stream: *DataInputStream, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) u16;
    pub const readUint16 = g_data_input_stream_read_uint16;

    /// Reads an unsigned 32-bit/4-byte value from `stream`.
    ///
    /// In order to get the correct byte order for this read operation,
    /// see `gio.DataInputStream.getByteOrder` and `gio.DataInputStream.setByteOrder`.
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned.
    extern fn g_data_input_stream_read_uint32(p_stream: *DataInputStream, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) u32;
    pub const readUint32 = g_data_input_stream_read_uint32;

    /// Reads an unsigned 64-bit/8-byte value from `stream`.
    ///
    /// In order to get the correct byte order for this read operation,
    /// see `gio.DataInputStream.getByteOrder`.
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned.
    extern fn g_data_input_stream_read_uint64(p_stream: *DataInputStream, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) u64;
    pub const readUint64 = g_data_input_stream_read_uint64;

    /// Reads a string from the data input stream, up to the first
    /// occurrence of any of the stop characters.
    ///
    /// Note that, in contrast to `gio.DataInputStream.readUntilAsync`,
    /// this function consumes the stop character that it finds.
    ///
    /// Don't use this function in new code.  Its functionality is
    /// inconsistent with `gio.DataInputStream.readUntilAsync`.  Both
    /// functions will be marked as deprecated in a future release.  Use
    /// `gio.DataInputStream.readUpto` instead, but note that that function
    /// does not consume the stop character.
    extern fn g_data_input_stream_read_until(p_stream: *DataInputStream, p_stop_chars: [*:0]const u8, p_length: ?*usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const readUntil = g_data_input_stream_read_until;

    /// The asynchronous version of `gio.DataInputStream.readUntil`.
    /// It is an error to have two outstanding calls to this function.
    ///
    /// Note that, in contrast to `gio.DataInputStream.readUntil`,
    /// this function does not consume the stop character that it finds.  You
    /// must read it for yourself.
    ///
    /// When the operation is finished, `callback` will be called. You
    /// can then call `gio.DataInputStream.readUntilFinish` to get
    /// the result of the operation.
    ///
    /// Don't use this function in new code.  Its functionality is
    /// inconsistent with `gio.DataInputStream.readUntil`.  Both functions
    /// will be marked as deprecated in a future release.  Use
    /// `gio.DataInputStream.readUptoAsync` instead.
    extern fn g_data_input_stream_read_until_async(p_stream: *DataInputStream, p_stop_chars: [*:0]const u8, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const readUntilAsync = g_data_input_stream_read_until_async;

    /// Finish an asynchronous call started by
    /// `gio.DataInputStream.readUntilAsync`.
    extern fn g_data_input_stream_read_until_finish(p_stream: *DataInputStream, p_result: *gio.AsyncResult, p_length: ?*usize, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const readUntilFinish = g_data_input_stream_read_until_finish;

    /// Reads a string from the data input stream, up to the first
    /// occurrence of any of the stop characters.
    ///
    /// In contrast to `gio.DataInputStream.readUntil`, this function
    /// does not consume the stop character. You have to use
    /// `gio.DataInputStream.readByte` to get it before calling
    /// `gio.DataInputStream.readUpto` again.
    ///
    /// Note that `stop_chars` may contain '\0' if `stop_chars_len` is
    /// specified.
    ///
    /// The returned string will always be nul-terminated on success.
    extern fn g_data_input_stream_read_upto(p_stream: *DataInputStream, p_stop_chars: [*:0]const u8, p_stop_chars_len: isize, p_length: ?*usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const readUpto = g_data_input_stream_read_upto;

    /// The asynchronous version of `gio.DataInputStream.readUpto`.
    /// It is an error to have two outstanding calls to this function.
    ///
    /// In contrast to `gio.DataInputStream.readUntil`, this function
    /// does not consume the stop character. You have to use
    /// `gio.DataInputStream.readByte` to get it before calling
    /// `gio.DataInputStream.readUpto` again.
    ///
    /// Note that `stop_chars` may contain '\0' if `stop_chars_len` is
    /// specified.
    ///
    /// When the operation is finished, `callback` will be called. You
    /// can then call `gio.DataInputStream.readUptoFinish` to get
    /// the result of the operation.
    extern fn g_data_input_stream_read_upto_async(p_stream: *DataInputStream, p_stop_chars: [*:0]const u8, p_stop_chars_len: isize, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const readUptoAsync = g_data_input_stream_read_upto_async;

    /// Finish an asynchronous call started by
    /// `gio.DataInputStream.readUptoAsync`.
    ///
    /// Note that this function does not consume the stop character. You
    /// have to use `gio.DataInputStream.readByte` to get it before calling
    /// `gio.DataInputStream.readUptoAsync` again.
    ///
    /// The returned string will always be nul-terminated on success.
    extern fn g_data_input_stream_read_upto_finish(p_stream: *DataInputStream, p_result: *gio.AsyncResult, p_length: ?*usize, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const readUptoFinish = g_data_input_stream_read_upto_finish;

    /// This function sets the byte order for the given `stream`. All subsequent
    /// reads from the `stream` will be read in the given `order`.
    extern fn g_data_input_stream_set_byte_order(p_stream: *DataInputStream, p_order: gio.DataStreamByteOrder) void;
    pub const setByteOrder = g_data_input_stream_set_byte_order;

    /// Sets the newline type for the `stream`.
    ///
    /// Note that using G_DATA_STREAM_NEWLINE_TYPE_ANY is slightly unsafe. If a read
    /// chunk ends in "CR" we must read an additional byte to know if this is "CR" or
    /// "CR LF", and this might block if there is no more data available.
    extern fn g_data_input_stream_set_newline_type(p_stream: *DataInputStream, p_type: gio.DataStreamNewlineType) void;
    pub const setNewlineType = g_data_input_stream_set_newline_type;

    extern fn g_data_input_stream_get_type() usize;
    pub const getGObjectType = g_data_input_stream_get_type;

    extern fn g_object_ref(p_self: *gio.DataInputStream) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.DataInputStream) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *DataInputStream, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Data output stream implements `gio.OutputStream` and includes functions
/// for writing data directly to an output stream.
pub const DataOutputStream = extern struct {
    pub const Parent = gio.FilterOutputStream;
    pub const Implements = [_]type{gio.Seekable};
    pub const Class = gio.DataOutputStreamClass;
    f_parent_instance: gio.FilterOutputStream,
    f_priv: ?*gio.DataOutputStreamPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Determines the byte ordering that is used when writing
        /// multi-byte entities (such as integers) to the stream.
        pub const byte_order = struct {
            pub const name = "byte-order";

            pub const Type = gio.DataStreamByteOrder;
        };
    };

    pub const signals = struct {};

    /// Creates a new data output stream for `base_stream`.
    extern fn g_data_output_stream_new(p_base_stream: *gio.OutputStream) *gio.DataOutputStream;
    pub const new = g_data_output_stream_new;

    /// Gets the byte order for the stream.
    extern fn g_data_output_stream_get_byte_order(p_stream: *DataOutputStream) gio.DataStreamByteOrder;
    pub const getByteOrder = g_data_output_stream_get_byte_order;

    /// Puts a byte into the output stream.
    extern fn g_data_output_stream_put_byte(p_stream: *DataOutputStream, p_data: u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const putByte = g_data_output_stream_put_byte;

    /// Puts a signed 16-bit integer into the output stream.
    extern fn g_data_output_stream_put_int16(p_stream: *DataOutputStream, p_data: i16, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const putInt16 = g_data_output_stream_put_int16;

    /// Puts a signed 32-bit integer into the output stream.
    extern fn g_data_output_stream_put_int32(p_stream: *DataOutputStream, p_data: i32, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const putInt32 = g_data_output_stream_put_int32;

    /// Puts a signed 64-bit integer into the stream.
    extern fn g_data_output_stream_put_int64(p_stream: *DataOutputStream, p_data: i64, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const putInt64 = g_data_output_stream_put_int64;

    /// Puts a string into the output stream.
    extern fn g_data_output_stream_put_string(p_stream: *DataOutputStream, p_str: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const putString = g_data_output_stream_put_string;

    /// Puts an unsigned 16-bit integer into the output stream.
    extern fn g_data_output_stream_put_uint16(p_stream: *DataOutputStream, p_data: u16, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const putUint16 = g_data_output_stream_put_uint16;

    /// Puts an unsigned 32-bit integer into the stream.
    extern fn g_data_output_stream_put_uint32(p_stream: *DataOutputStream, p_data: u32, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const putUint32 = g_data_output_stream_put_uint32;

    /// Puts an unsigned 64-bit integer into the stream.
    extern fn g_data_output_stream_put_uint64(p_stream: *DataOutputStream, p_data: u64, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const putUint64 = g_data_output_stream_put_uint64;

    /// Sets the byte order of the data output stream to `order`.
    extern fn g_data_output_stream_set_byte_order(p_stream: *DataOutputStream, p_order: gio.DataStreamByteOrder) void;
    pub const setByteOrder = g_data_output_stream_set_byte_order;

    extern fn g_data_output_stream_get_type() usize;
    pub const getGObjectType = g_data_output_stream_get_type;

    extern fn g_object_ref(p_self: *gio.DataOutputStream) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.DataOutputStream) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *DataOutputStream, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GDebugControllerDBus` is an implementation of `gio.DebugController`
/// which exposes debug settings as a D-Bus object.
///
/// It is a `gio.Initable` object, and will register an object at
/// `/org/gtk/Debugging` on the bus given as
/// `gio.DebugControllerDBus.properties.connection` once it’s initialized. The
/// object will be unregistered when the last reference to the
/// `GDebugControllerDBus` is dropped.
///
/// This D-Bus object can be used by remote processes to enable or disable debug
/// output in this process. Remote processes calling
/// `org.gtk.Debugging.`SetDebugEnabled`` will affect the value of
/// `gio.DebugController.properties.debug_enabled` and, by default,
/// `glib.logGetDebugEnabled`.
///
/// By default, no processes are allowed to call ``SetDebugEnabled`` unless a
/// `gio.DebugControllerDBus.signals.authorize` signal handler is installed. This
/// is because the process may be privileged, or might expose sensitive
/// information in its debug output. You may want to restrict the ability to
/// enable debug output to privileged users or processes.
///
/// One option is to install a D-Bus security policy which restricts access to
/// ``SetDebugEnabled``, installing something like the following in
/// `$datadir/dbus-1/system.d/`:
///
/// ```xml
/// <?xml version="1.0"?> <!--*-nxml-*-->
/// <!DOCTYPE busconfig PUBLIC "-//freedesktop//DTD D-BUS Bus Configuration 1.0//EN"
///      "http://www.freedesktop.org/standards/dbus/1.0/busconfig.dtd">
/// <busconfig>
///   <policy user="root">
///     <allow send_destination="com.example.MyService" send_interface="org.gtk.Debugging"/>
///   </policy>
///   <policy context="default">
///     <deny send_destination="com.example.MyService" send_interface="org.gtk.Debugging"/>
///   </policy>
/// </busconfig>
/// ```
///
/// This will prevent the ``SetDebugEnabled`` method from being called by all
/// except root. It will not prevent the `DebugEnabled` property from being read,
/// as it’s accessed through the `org.freedesktop.DBus.Properties` interface.
///
/// Another option is to use polkit to allow or deny requests on a case-by-case
/// basis, allowing for the possibility of dynamic authorisation. To do this,
/// connect to the `gio.DebugControllerDBus.signals.authorize` signal and query
/// polkit in it:
///
/// ```c
///   g_autoptr(GError) child_error = NULL;
///   g_autoptr(GDBusConnection) connection = g_bus_get_sync (G_BUS_TYPE_SYSTEM, NULL, NULL);
///   gulong debug_controller_authorize_id = 0;
///
///   // Set up the debug controller.
///   debug_controller = G_DEBUG_CONTROLLER (g_debug_controller_dbus_new (priv->connection, NULL, &child_error));
///   if (debug_controller == NULL)
///     {
///       g_error ("Could not register debug controller on bus: `s`"),
///                child_error->message);
///     }
///
///   debug_controller_authorize_id = g_signal_connect (debug_controller,
///                                                     "authorize",
///                                                     G_CALLBACK (debug_controller_authorize_cb),
///                                                     self);
///
///   static gboolean
///   debug_controller_authorize_cb (GDebugControllerDBus  *debug_controller,
///                                  GDBusMethodInvocation *invocation,
///                                  gpointer               user_data)
///   {
///     g_autoptr(PolkitAuthority) authority = NULL;
///     g_autoptr(PolkitSubject) subject = NULL;
///     g_autoptr(PolkitAuthorizationResult) auth_result = NULL;
///     g_autoptr(GError) local_error = NULL;
///     GDBusMessage *message;
///     GDBusMessageFlags message_flags;
///     PolkitCheckAuthorizationFlags flags = POLKIT_CHECK_AUTHORIZATION_FLAGS_NONE;
///
///     message = g_dbus_method_invocation_get_message (invocation);
///     message_flags = g_dbus_message_get_flags (message);
///
///     authority = polkit_authority_get_sync (NULL, &local_error);
///     if (authority == NULL)
///       {
///         g_warning ("Failed to get polkit authority: `s`", local_error->message);
///         return FALSE;
///       }
///
///     if (message_flags & G_DBUS_MESSAGE_FLAGS_ALLOW_INTERACTIVE_AUTHORIZATION)
///       flags |= POLKIT_CHECK_AUTHORIZATION_FLAGS_ALLOW_USER_INTERACTION;
///
///     subject = polkit_system_bus_name_new (g_dbus_method_invocation_get_sender (invocation));
///
///     auth_result = polkit_authority_check_authorization_sync (authority,
///                                                              subject,
///                                                              "com.example.MyService.set-debug-enabled",
///                                                              NULL,
///                                                              flags,
///                                                              NULL,
///                                                              &local_error);
///     if (auth_result == NULL)
///       {
///         g_warning ("Failed to get check polkit authorization: `s`", local_error->message);
///         return FALSE;
///       }
///
///     return polkit_authorization_result_get_is_authorized (auth_result);
///   }
/// ```
pub const DebugControllerDBus = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{ gio.DebugController, gio.Initable };
    pub const Class = gio.DebugControllerDBusClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {
        /// Default handler for the `gio.DebugControllerDBus.signals.authorize` signal.
        pub const authorize = struct {
            pub fn call(p_class: anytype, p_controller: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_invocation: *gio.DBusMethodInvocation) c_int {
                return gobject.ext.as(DebugControllerDBus.Class, p_class).f_authorize.?(gobject.ext.as(DebugControllerDBus, p_controller), p_invocation);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_controller: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_invocation: *gio.DBusMethodInvocation) callconv(.C) c_int) void {
                gobject.ext.as(DebugControllerDBus.Class, p_class).f_authorize = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The D-Bus connection to expose the debugging interface on.
        ///
        /// Typically this will be the same connection (to the system or session bus)
        /// which the rest of the application or service’s D-Bus objects are registered
        /// on.
        pub const connection = struct {
            pub const name = "connection";

            pub const Type = ?*gio.DBusConnection;
        };
    };

    pub const signals = struct {
        /// Emitted when a D-Bus peer is trying to change the debug settings and used
        /// to determine if that is authorized.
        ///
        /// This signal is emitted in a dedicated worker thread, so handlers are
        /// allowed to perform blocking I/O. This means that, for example, it is
        /// appropriate to call ``polkit_authority_check_authorization_sync`` to check
        /// authorization using polkit.
        ///
        /// If `FALSE` is returned then no further handlers are run and the request to
        /// change the debug settings is rejected.
        ///
        /// Otherwise, if `TRUE` is returned, signal emission continues. If no handlers
        /// return `FALSE`, then the debug settings are allowed to be changed.
        ///
        /// Signal handlers must not modify `invocation`, or cause it to return a value.
        ///
        /// The default class handler just returns `TRUE`.
        pub const authorize = struct {
            pub const name = "authorize";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_invocation: *gio.DBusMethodInvocation, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(DebugControllerDBus, p_instance))),
                    gobject.signalLookup("authorize", DebugControllerDBus.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Create a new `gio.DebugControllerDBus` and synchronously initialize it.
    ///
    /// Initializing the object will export the debug object on `connection`. The
    /// object will remain registered until the last reference to the
    /// `gio.DebugControllerDBus` is dropped.
    ///
    /// Initialization may fail if registering the object on `connection` fails.
    extern fn g_debug_controller_dbus_new(p_connection: *gio.DBusConnection, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.DebugControllerDBus;
    pub const new = g_debug_controller_dbus_new;

    /// Stop the debug controller, unregistering its object from the bus.
    ///
    /// Any pending method calls to the object will complete successfully, but new
    /// ones will return an error. This method will block until all pending
    /// `gio.DebugControllerDBus.signals.authorize` signals have been handled. This is expected
    /// to not take long, as it will just be waiting for threads to join. If any
    /// `gio.DebugControllerDBus.signals.authorize` signal handlers are still executing in other
    /// threads, this will block until after they have returned.
    ///
    /// This method will be called automatically when the final reference to the
    /// `gio.DebugControllerDBus` is dropped. You may want to call it explicitly to know
    /// when the controller has been fully removed from the bus, or to break
    /// reference count cycles.
    ///
    /// Calling this method from within a `gio.DebugControllerDBus.signals.authorize` signal
    /// handler will cause a deadlock and must not be done.
    extern fn g_debug_controller_dbus_stop(p_self: *DebugControllerDBus) void;
    pub const stop = g_debug_controller_dbus_stop;

    extern fn g_debug_controller_dbus_get_type() usize;
    pub const getGObjectType = g_debug_controller_dbus_get_type;

    extern fn g_object_ref(p_self: *gio.DebugControllerDBus) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.DebugControllerDBus) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *DebugControllerDBus, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GDesktopAppInfo` is an implementation of `gio.AppInfo` based on
/// desktop files.
///
/// Note that `<gio/gdesktopappinfo.h>` belongs to the UNIX-specific
/// GIO interfaces, thus you have to use the `gio-unix-2.0.pc` pkg-config
/// file or the `GioUnix-2.0` GIR namespace when using it.
pub const DesktopAppInfo = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gio.AppInfo};
    pub const Class = gio.DesktopAppInfoClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The origin filename of this `gio.DesktopAppInfo`
        pub const filename = struct {
            pub const name = "filename";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Gets all applications that implement `interface`.
    ///
    /// An application implements an interface if that interface is listed in
    /// the `Implements` line of the desktop file of the application.
    extern fn g_desktop_app_info_get_implementations(p_interface: [*:0]const u8) *glib.List;
    pub const getImplementations = g_desktop_app_info_get_implementations;

    /// Searches desktop files for ones that match `search_string`.
    ///
    /// The return value is an array of strvs.  Each strv contains a list of
    /// applications that matched `search_string` with an equal score.  The
    /// outer list is sorted by score so that the first strv contains the
    /// best-matching applications, and so on.
    /// The algorithm for determining matches is undefined and may change at
    /// any time.
    ///
    /// None of the search results are subjected to the normal validation
    /// checks performed by `gio.DesktopAppInfo.new` (for example, checking that
    /// the executable referenced by a result exists), and so it is possible for
    /// `gio.DesktopAppInfo.new` to return `NULL` when passed an app ID returned
    /// by this function. It is expected that calling code will do this when
    /// subsequently creating a `gio.DesktopAppInfo` for each result.
    extern fn g_desktop_app_info_search(p_search_string: [*:0]const u8) [*][*][*:0]u8;
    pub const search = g_desktop_app_info_search;

    /// Sets the name of the desktop that the application is running in.
    ///
    /// This is used by `gio.AppInfo.shouldShow` and
    /// `gio.DesktopAppInfo.getShowIn` to evaluate the
    /// [`OnlyShowIn`](https://specifications.freedesktop.org/desktop-entry-spec/latest/ar01s06.html`key`-onlyshowin)
    /// and [`NotShowIn`](https://specifications.freedesktop.org/desktop-entry-spec/latest/ar01s06.html`key`-notshowin)
    /// keys.
    ///
    /// Should be called only once; subsequent calls are ignored.
    extern fn g_desktop_app_info_set_desktop_env(p_desktop_env: [*:0]const u8) void;
    pub const setDesktopEnv = g_desktop_app_info_set_desktop_env;

    /// Creates a new `gio.DesktopAppInfo` based on a desktop file ID.
    ///
    /// A desktop file ID is the basename of the desktop file, including the
    /// `.desktop` extension. GIO is looking for a desktop file with this name
    /// in the `applications` subdirectories of the XDG
    /// data directories (i.e. the directories specified in the `XDG_DATA_HOME`
    /// and `XDG_DATA_DIRS` environment variables). GIO also supports the
    /// prefix-to-subdirectory mapping that is described in the
    /// [Menu Spec](http://standards.freedesktop.org/menu-spec/latest/)
    /// (i.e. a desktop ID of `kde-foo.desktop` will match
    /// `/usr/share/applications/kde/foo.desktop`).
    extern fn g_desktop_app_info_new(p_desktop_id: [*:0]const u8) ?*gio.DesktopAppInfo;
    pub const new = g_desktop_app_info_new;

    /// Creates a new `gio.DesktopAppInfo`.
    extern fn g_desktop_app_info_new_from_filename(p_filename: [*:0]const u8) ?*gio.DesktopAppInfo;
    pub const newFromFilename = g_desktop_app_info_new_from_filename;

    /// Creates a new `gio.DesktopAppInfo`.
    extern fn g_desktop_app_info_new_from_keyfile(p_key_file: *glib.KeyFile) ?*gio.DesktopAppInfo;
    pub const newFromKeyfile = g_desktop_app_info_new_from_keyfile;

    /// Gets the user-visible display name of the
    /// [‘additional application actions’](https://specifications.freedesktop.org/desktop-entry-spec/latest/ar01s11.html)
    /// specified by `action_name`.
    ///
    /// This corresponds to the `Name` key within the keyfile group for the
    /// action.
    extern fn g_desktop_app_info_get_action_name(p_info: *DesktopAppInfo, p_action_name: [*:0]const u8) [*:0]u8;
    pub const getActionName = g_desktop_app_info_get_action_name;

    /// Looks up a boolean value in the keyfile backing `info`.
    ///
    /// The `key` is looked up in the `Desktop Entry` group.
    extern fn g_desktop_app_info_get_boolean(p_info: *DesktopAppInfo, p_key: [*:0]const u8) c_int;
    pub const getBoolean = g_desktop_app_info_get_boolean;

    /// Gets the categories from the desktop file.
    extern fn g_desktop_app_info_get_categories(p_info: *DesktopAppInfo) ?[*:0]const u8;
    pub const getCategories = g_desktop_app_info_get_categories;

    /// When `info` was created from a known filename, return it.  In some
    /// situations such as a `gio.DesktopAppInfo` returned from
    /// `gio.DesktopAppInfo.newFromKeyfile`, this function will return `NULL`.
    extern fn g_desktop_app_info_get_filename(p_info: *DesktopAppInfo) ?[*:0]const u8;
    pub const getFilename = g_desktop_app_info_get_filename;

    /// Gets the generic name from the desktop file.
    extern fn g_desktop_app_info_get_generic_name(p_info: *DesktopAppInfo) ?[*:0]const u8;
    pub const getGenericName = g_desktop_app_info_get_generic_name;

    /// A desktop file is hidden if the
    /// [`Hidden` key](https://specifications.freedesktop.org/desktop-entry-spec/latest/ar01s06.html`key`-hidden)
    /// in it is set to `True`.
    extern fn g_desktop_app_info_get_is_hidden(p_info: *DesktopAppInfo) c_int;
    pub const getIsHidden = g_desktop_app_info_get_is_hidden;

    /// Gets the keywords from the desktop file.
    extern fn g_desktop_app_info_get_keywords(p_info: *DesktopAppInfo) [*]const [*:0]const u8;
    pub const getKeywords = g_desktop_app_info_get_keywords;

    /// Looks up a localized string value in the keyfile backing `info`
    /// translated to the current locale.
    ///
    /// The `key` is looked up in the `Desktop Entry` group.
    extern fn g_desktop_app_info_get_locale_string(p_info: *DesktopAppInfo, p_key: [*:0]const u8) ?[*:0]u8;
    pub const getLocaleString = g_desktop_app_info_get_locale_string;

    /// Gets the value of the
    /// [`NoDisplay` key](https://specifications.freedesktop.org/desktop-entry-spec/latest/ar01s06.html`key`-nodisplay)
    ///  which helps determine if the application info should be shown in menus. See
    /// `G_KEY_FILE_DESKTOP_KEY_NO_DISPLAY` and `gio.AppInfo.shouldShow`.
    extern fn g_desktop_app_info_get_nodisplay(p_info: *DesktopAppInfo) c_int;
    pub const getNodisplay = g_desktop_app_info_get_nodisplay;

    /// Checks if the application info should be shown in menus that list available
    /// applications for a specific name of the desktop, based on the
    /// [`OnlyShowIn`](https://specifications.freedesktop.org/desktop-entry-spec/latest/ar01s06.html`key`-onlyshowin)
    /// and [`NotShowIn`](https://specifications.freedesktop.org/desktop-entry-spec/latest/ar01s06.html`key`-notshowin)
    /// keys.
    ///
    /// `desktop_env` should typically be given as `NULL`, in which case the
    /// `XDG_CURRENT_DESKTOP` environment variable is consulted.  If you want
    /// to override the default mechanism then you may specify `desktop_env`,
    /// but this is not recommended.
    ///
    /// Note that `gio.AppInfo.shouldShow` for `info` will include this check
    /// (with `NULL` for `desktop_env`) as well as additional checks.
    extern fn g_desktop_app_info_get_show_in(p_info: *DesktopAppInfo, p_desktop_env: ?[*:0]const u8) c_int;
    pub const getShowIn = g_desktop_app_info_get_show_in;

    /// Retrieves the `StartupWMClass` field from `info`. This represents the
    /// `WM_CLASS` property of the main window of the application, if launched
    /// through `info`.
    extern fn g_desktop_app_info_get_startup_wm_class(p_info: *DesktopAppInfo) ?[*:0]const u8;
    pub const getStartupWmClass = g_desktop_app_info_get_startup_wm_class;

    /// Looks up a string value in the keyfile backing `info`.
    ///
    /// The `key` is looked up in the `Desktop Entry` group.
    extern fn g_desktop_app_info_get_string(p_info: *DesktopAppInfo, p_key: [*:0]const u8) ?[*:0]u8;
    pub const getString = g_desktop_app_info_get_string;

    /// Looks up a string list value in the keyfile backing `info`.
    ///
    /// The `key` is looked up in the `Desktop Entry` group.
    extern fn g_desktop_app_info_get_string_list(p_info: *DesktopAppInfo, p_key: [*:0]const u8, p_length: ?*usize) [*:null]?[*:0]u8;
    pub const getStringList = g_desktop_app_info_get_string_list;

    /// Returns whether `key` exists in the `Desktop Entry` group
    /// of the keyfile backing `info`.
    extern fn g_desktop_app_info_has_key(p_info: *DesktopAppInfo, p_key: [*:0]const u8) c_int;
    pub const hasKey = g_desktop_app_info_has_key;

    /// Activates the named application action.
    ///
    /// You may only call this function on action names that were
    /// returned from `gio.DesktopAppInfo.listActions`.
    ///
    /// Note that if the main entry of the desktop file indicates that the
    /// application supports startup notification, and `launch_context` is
    /// non-`NULL`, then startup notification will be used when activating the
    /// action (and as such, invocation of the action on the receiving side
    /// must signal the end of startup notification when it is completed).
    /// This is the expected behaviour of applications declaring additional
    /// actions, as per the
    /// [desktop file specification](https://specifications.freedesktop.org/desktop-entry-spec/latest/ar01s11.html).
    ///
    /// As with `gio.AppInfo.launch` there is no way to detect failures that
    /// occur while using this function.
    extern fn g_desktop_app_info_launch_action(p_info: *DesktopAppInfo, p_action_name: [*:0]const u8, p_launch_context: ?*gio.AppLaunchContext) void;
    pub const launchAction = g_desktop_app_info_launch_action;

    /// This function performs the equivalent of `gio.AppInfo.launchUris`,
    /// but is intended primarily for operating system components that
    /// launch applications.  Ordinary applications should use
    /// `gio.AppInfo.launchUris`.
    ///
    /// If the application is launched via GSpawn, then `spawn_flags`, `user_setup`
    /// and `user_setup_data` are used for the call to `glib.spawnAsync`.
    /// Additionally, `pid_callback` (with `pid_callback_data`) will be called to
    /// inform about the PID of the created process. See
    /// `glib.spawnAsyncWithPipes` for information on certain parameter
    /// conditions that can enable an optimized [``posix_spawn``](man:posix_spawn(3))
    /// code path to be used.
    ///
    /// If application launching occurs via some other mechanism (for example, D-Bus
    /// activation) then `spawn_flags`, `user_setup`, `user_setup_data`,
    /// `pid_callback` and `pid_callback_data` are ignored.
    extern fn g_desktop_app_info_launch_uris_as_manager(p_appinfo: *DesktopAppInfo, p_uris: *glib.List, p_launch_context: ?*gio.AppLaunchContext, p_spawn_flags: glib.SpawnFlags, p_user_setup: ?glib.SpawnChildSetupFunc, p_user_setup_data: ?*anyopaque, p_pid_callback: ?gio.DesktopAppLaunchCallback, p_pid_callback_data: ?*anyopaque, p_error: ?*?*glib.Error) c_int;
    pub const launchUrisAsManager = g_desktop_app_info_launch_uris_as_manager;

    /// Equivalent to `gio.DesktopAppInfo.launchUrisAsManager` but allows
    /// you to pass in file descriptors for the stdin, stdout and stderr streams
    /// of the launched process.
    ///
    /// If application launching occurs via some non-spawn mechanism (e.g. D-Bus
    /// activation) then `stdin_fd`, `stdout_fd` and `stderr_fd` are ignored.
    extern fn g_desktop_app_info_launch_uris_as_manager_with_fds(p_appinfo: *DesktopAppInfo, p_uris: *glib.List, p_launch_context: ?*gio.AppLaunchContext, p_spawn_flags: glib.SpawnFlags, p_user_setup: ?glib.SpawnChildSetupFunc, p_user_setup_data: ?*anyopaque, p_pid_callback: ?gio.DesktopAppLaunchCallback, p_pid_callback_data: ?*anyopaque, p_stdin_fd: c_int, p_stdout_fd: c_int, p_stderr_fd: c_int, p_error: ?*?*glib.Error) c_int;
    pub const launchUrisAsManagerWithFds = g_desktop_app_info_launch_uris_as_manager_with_fds;

    /// Returns the list of
    /// [‘additional application actions’](https://specifications.freedesktop.org/desktop-entry-spec/latest/ar01s11.html)
    /// supported on the desktop file, as per the desktop file specification.
    ///
    /// As per the specification, this is the list of actions that are
    /// explicitly listed in the `Actions` key of the `Desktop Entry` group.
    extern fn g_desktop_app_info_list_actions(p_info: *DesktopAppInfo) [*]const [*:0]const u8;
    pub const listActions = g_desktop_app_info_list_actions;

    extern fn g_desktop_app_info_get_type() usize;
    pub const getGObjectType = g_desktop_app_info_get_type;

    extern fn g_object_ref(p_self: *gio.DesktopAppInfo) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.DesktopAppInfo) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *DesktopAppInfo, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GEmblem` is an implementation of `gio.Icon` that supports
/// having an emblem, which is an icon with additional properties.
/// It can than be added to a `gio.EmblemedIcon`.
///
/// Currently, only metainformation about the emblem's origin is
/// supported. More may be added in the future.
pub const Emblem = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gio.Icon};
    pub const Class = gio.EmblemClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The actual icon of the emblem.
        pub const icon = struct {
            pub const name = "icon";

            pub const Type = ?*gobject.Object;
        };

        /// The origin the emblem is derived from.
        pub const origin = struct {
            pub const name = "origin";

            pub const Type = gio.EmblemOrigin;
        };
    };

    pub const signals = struct {};

    /// Creates a new emblem for `icon`.
    extern fn g_emblem_new(p_icon: *gio.Icon) *gio.Emblem;
    pub const new = g_emblem_new;

    /// Creates a new emblem for `icon`.
    extern fn g_emblem_new_with_origin(p_icon: *gio.Icon, p_origin: gio.EmblemOrigin) *gio.Emblem;
    pub const newWithOrigin = g_emblem_new_with_origin;

    /// Gives back the icon from `emblem`.
    extern fn g_emblem_get_icon(p_emblem: *Emblem) *gio.Icon;
    pub const getIcon = g_emblem_get_icon;

    /// Gets the origin of the emblem.
    extern fn g_emblem_get_origin(p_emblem: *Emblem) gio.EmblemOrigin;
    pub const getOrigin = g_emblem_get_origin;

    extern fn g_emblem_get_type() usize;
    pub const getGObjectType = g_emblem_get_type;

    extern fn g_object_ref(p_self: *gio.Emblem) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.Emblem) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Emblem, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GEmblemedIcon` is an implementation of `gio.Icon` that supports
/// adding an emblem to an icon. Adding multiple emblems to an
/// icon is ensured via `gio.EmblemedIcon.addEmblem`.
///
/// Note that `GEmblemedIcon` allows no control over the position
/// of the emblems. See also `gio.Emblem` for more information.
pub const EmblemedIcon = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gio.Icon};
    pub const Class = gio.EmblemedIconClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.EmblemedIconPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The `gio.Icon` to attach emblems to.
        pub const gicon = struct {
            pub const name = "gicon";

            pub const Type = ?*gio.Icon;
        };
    };

    pub const signals = struct {};

    /// Creates a new emblemed icon for `icon` with the emblem `emblem`.
    extern fn g_emblemed_icon_new(p_icon: *gio.Icon, p_emblem: ?*gio.Emblem) *gio.EmblemedIcon;
    pub const new = g_emblemed_icon_new;

    /// Adds `emblem` to the `glib.List` of `GEmblems`.
    extern fn g_emblemed_icon_add_emblem(p_emblemed: *EmblemedIcon, p_emblem: *gio.Emblem) void;
    pub const addEmblem = g_emblemed_icon_add_emblem;

    /// Removes all the emblems from `icon`.
    extern fn g_emblemed_icon_clear_emblems(p_emblemed: *EmblemedIcon) void;
    pub const clearEmblems = g_emblemed_icon_clear_emblems;

    /// Gets the list of emblems for the `icon`.
    extern fn g_emblemed_icon_get_emblems(p_emblemed: *EmblemedIcon) *glib.List;
    pub const getEmblems = g_emblemed_icon_get_emblems;

    /// Gets the main icon for `emblemed`.
    extern fn g_emblemed_icon_get_icon(p_emblemed: *EmblemedIcon) *gio.Icon;
    pub const getIcon = g_emblemed_icon_get_icon;

    extern fn g_emblemed_icon_get_type() usize;
    pub const getGObjectType = g_emblemed_icon_get_type;

    extern fn g_object_ref(p_self: *gio.EmblemedIcon) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.EmblemedIcon) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *EmblemedIcon, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GFileEnumerator` allows you to operate on a set of `gio.File` objects,
/// returning a `gio.FileInfo` structure for each file enumerated (e.g.
/// `gio.File.enumerateChildren` will return a `GFileEnumerator` for each
/// of the children within a directory).
///
/// To get the next file's information from a `GFileEnumerator`, use
/// `gio.FileEnumerator.nextFile` or its asynchronous version,
/// `gio.FileEnumerator.nextFilesAsync`. Note that the asynchronous
/// version will return a list of `gio.FileInfo` objects, whereas the
/// synchronous will only return the next file in the enumerator.
///
/// The ordering of returned files is unspecified for non-Unix
/// platforms; for more information, see `glib.Dir.readName`.  On Unix,
/// when operating on local files, returned files will be sorted by
/// inode number.  Effectively you can assume that the ordering of
/// returned files will be stable between successive calls (and
/// applications) assuming the directory is unchanged.
///
/// If your application needs a specific ordering, such as by name or
/// modification time, you will have to implement that in your
/// application code.
///
/// To close a `GFileEnumerator`, use `gio.FileEnumerator.close`, or
/// its asynchronous version, `gio.FileEnumerator.closeAsync`. Once
/// a `GFileEnumerator` is closed, no further actions may be performed
/// on it, and it should be freed with `gobject.Object.unref`.
pub const FileEnumerator = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.FileEnumeratorClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.FileEnumeratorPrivate,

    pub const virtual_methods = struct {
        /// Asynchronously closes the file enumerator.
        ///
        /// If `cancellable` is not `NULL`, then the operation can be cancelled by
        /// triggering the cancellable object from another thread. If the operation
        /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned in
        /// `gio.FileEnumerator.closeFinish`.
        pub const close_async = struct {
            pub fn call(p_class: anytype, p_enumerator: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(FileEnumerator.Class, p_class).f_close_async.?(gobject.ext.as(FileEnumerator, p_enumerator), p_io_priority, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_enumerator: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(FileEnumerator.Class, p_class).f_close_async = @ptrCast(p_implementation);
            }
        };

        /// Finishes closing a file enumerator, started from `gio.FileEnumerator.closeAsync`.
        ///
        /// If the file enumerator was already closed when `gio.FileEnumerator.closeAsync`
        /// was called, then this function will report `G_IO_ERROR_CLOSED` in `error`, and
        /// return `FALSE`. If the file enumerator had pending operation when the close
        /// operation was started, then this function will report `G_IO_ERROR_PENDING`, and
        /// return `FALSE`.  If `cancellable` was not `NULL`, then the operation may have been
        /// cancelled by triggering the cancellable object from another thread. If the operation
        /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be set, and `FALSE` will be
        /// returned.
        pub const close_finish = struct {
            pub fn call(p_class: anytype, p_enumerator: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(FileEnumerator.Class, p_class).f_close_finish.?(gobject.ext.as(FileEnumerator, p_enumerator), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_enumerator: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(FileEnumerator.Class, p_class).f_close_finish = @ptrCast(p_implementation);
            }
        };

        pub const close_fn = struct {
            pub fn call(p_class: anytype, p_enumerator: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(FileEnumerator.Class, p_class).f_close_fn.?(gobject.ext.as(FileEnumerator, p_enumerator), p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_enumerator: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(FileEnumerator.Class, p_class).f_close_fn = @ptrCast(p_implementation);
            }
        };

        /// Returns information for the next file in the enumerated object.
        /// Will block until the information is available. The `gio.FileInfo`
        /// returned from this function will contain attributes that match the
        /// attribute string that was passed when the `gio.FileEnumerator` was created.
        ///
        /// See the documentation of `gio.FileEnumerator` for information about the
        /// order of returned files.
        ///
        /// On error, returns `NULL` and sets `error` to the error. If the
        /// enumerator is at the end, `NULL` will be returned and `error` will
        /// be unset.
        pub const next_file = struct {
            pub fn call(p_class: anytype, p_enumerator: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.FileInfo {
                return gobject.ext.as(FileEnumerator.Class, p_class).f_next_file.?(gobject.ext.as(FileEnumerator, p_enumerator), p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_enumerator: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) ?*gio.FileInfo) void {
                gobject.ext.as(FileEnumerator.Class, p_class).f_next_file = @ptrCast(p_implementation);
            }
        };

        /// Request information for a number of files from the enumerator asynchronously.
        /// When all I/O for the operation is finished the `callback` will be called with
        /// the requested information.
        ///
        /// See the documentation of `gio.FileEnumerator` for information about the
        /// order of returned files.
        ///
        /// Once the end of the enumerator is reached, or if an error occurs, the
        /// `callback` will be called with an empty list. In this case, the previous call
        /// to `gio.FileEnumerator.nextFilesAsync` will typically have returned fewer
        /// than `num_files` items.
        ///
        /// If a request is cancelled the callback will be called with
        /// `G_IO_ERROR_CANCELLED`.
        ///
        /// This leads to the following pseudo-code usage:
        /// ```
        /// g_autoptr(GFile) dir = get_directory ();
        /// g_autoptr(GFileEnumerator) enumerator = NULL;
        /// g_autolist(GFileInfo) files = NULL;
        /// g_autoptr(GError) local_error = NULL;
        ///
        /// enumerator = yield g_file_enumerate_children_async (dir,
        ///                                                     G_FILE_ATTRIBUTE_STANDARD_NAME ","
        ///                                                     G_FILE_ATTRIBUTE_STANDARD_TYPE,
        ///                                                     G_FILE_QUERY_INFO_NONE,
        ///                                                     G_PRIORITY_DEFAULT,
        ///                                                     cancellable,
        ///                                                     …,
        ///                                                     &local_error);
        /// if (enumerator == NULL)
        ///   g_error ("Error enumerating: `s`", local_error->message);
        ///
        /// // Loop until no files are returned, either because the end of the enumerator
        /// // has been reached, or an error was returned.
        /// do
        ///   {
        ///     files = yield g_file_enumerator_next_files_async (enumerator,
        ///                                                       5,  // number of files to request
        ///                                                       G_PRIORITY_DEFAULT,
        ///                                                       cancellable,
        ///                                                       …,
        ///                                                       &local_error);
        ///
        ///     // Process the returned files, but don’t assume that exactly 5 were returned.
        ///     for (GList *l = files; l != NULL; l = l->next)
        ///       {
        ///         GFileInfo *info = l->data;
        ///         handle_file_info (info);
        ///       }
        ///   }
        /// while (files != NULL);
        ///
        /// if (local_error != NULL &&
        ///     !g_error_matches (local_error, G_IO_ERROR, G_IO_ERROR_CANCELLED))
        ///   g_error ("Error while enumerating: `s`", local_error->message);
        /// ```
        ///
        /// During an async request no other sync and async calls are allowed, and will
        /// result in `G_IO_ERROR_PENDING` errors.
        ///
        /// Any outstanding I/O request with higher priority (lower numerical value) will
        /// be executed before an outstanding request with lower priority. Default
        /// priority is `G_PRIORITY_DEFAULT`.
        pub const next_files_async = struct {
            pub fn call(p_class: anytype, p_enumerator: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_num_files: c_int, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(FileEnumerator.Class, p_class).f_next_files_async.?(gobject.ext.as(FileEnumerator, p_enumerator), p_num_files, p_io_priority, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_enumerator: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_num_files: c_int, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(FileEnumerator.Class, p_class).f_next_files_async = @ptrCast(p_implementation);
            }
        };

        /// Finishes the asynchronous operation started with `gio.FileEnumerator.nextFilesAsync`.
        pub const next_files_finish = struct {
            pub fn call(p_class: anytype, p_enumerator: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.List {
                return gobject.ext.as(FileEnumerator.Class, p_class).f_next_files_finish.?(gobject.ext.as(FileEnumerator, p_enumerator), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_enumerator: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?*glib.List) void {
                gobject.ext.as(FileEnumerator.Class, p_class).f_next_files_finish = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The container that is being enumerated.
        pub const container = struct {
            pub const name = "container";

            pub const Type = ?*gio.File;
        };
    };

    pub const signals = struct {};

    /// Releases all resources used by this enumerator, making the
    /// enumerator return `G_IO_ERROR_CLOSED` on all calls.
    ///
    /// This will be automatically called when the last reference
    /// is dropped, but you might want to call this function to make
    /// sure resources are released as early as possible.
    extern fn g_file_enumerator_close(p_enumerator: *FileEnumerator, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const close = g_file_enumerator_close;

    /// Asynchronously closes the file enumerator.
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned in
    /// `gio.FileEnumerator.closeFinish`.
    extern fn g_file_enumerator_close_async(p_enumerator: *FileEnumerator, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const closeAsync = g_file_enumerator_close_async;

    /// Finishes closing a file enumerator, started from `gio.FileEnumerator.closeAsync`.
    ///
    /// If the file enumerator was already closed when `gio.FileEnumerator.closeAsync`
    /// was called, then this function will report `G_IO_ERROR_CLOSED` in `error`, and
    /// return `FALSE`. If the file enumerator had pending operation when the close
    /// operation was started, then this function will report `G_IO_ERROR_PENDING`, and
    /// return `FALSE`.  If `cancellable` was not `NULL`, then the operation may have been
    /// cancelled by triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be set, and `FALSE` will be
    /// returned.
    extern fn g_file_enumerator_close_finish(p_enumerator: *FileEnumerator, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const closeFinish = g_file_enumerator_close_finish;

    /// Return a new `gio.File` which refers to the file named by `info` in the source
    /// directory of `enumerator`.  This function is primarily intended to be used
    /// inside loops with `gio.FileEnumerator.nextFile`.
    ///
    /// To use this, `G_FILE_ATTRIBUTE_STANDARD_NAME` must have been listed in the
    /// attributes list used when creating the `gio.FileEnumerator`.
    ///
    /// This is a convenience method that's equivalent to:
    /// ```
    ///   gchar *name = g_file_info_get_name (info);
    ///   GFile *child = g_file_get_child (g_file_enumerator_get_container (enumr),
    ///                                    name);
    /// ```
    extern fn g_file_enumerator_get_child(p_enumerator: *FileEnumerator, p_info: *gio.FileInfo) *gio.File;
    pub const getChild = g_file_enumerator_get_child;

    /// Get the `gio.File` container which is being enumerated.
    extern fn g_file_enumerator_get_container(p_enumerator: *FileEnumerator) *gio.File;
    pub const getContainer = g_file_enumerator_get_container;

    /// Checks if the file enumerator has pending operations.
    extern fn g_file_enumerator_has_pending(p_enumerator: *FileEnumerator) c_int;
    pub const hasPending = g_file_enumerator_has_pending;

    /// Checks if the file enumerator has been closed.
    extern fn g_file_enumerator_is_closed(p_enumerator: *FileEnumerator) c_int;
    pub const isClosed = g_file_enumerator_is_closed;

    /// This is a version of `gio.FileEnumerator.nextFile` that's easier to
    /// use correctly from C programs.  With `gio.FileEnumerator.nextFile`,
    /// the gboolean return value signifies "end of iteration or error", which
    /// requires allocation of a temporary `glib.Error`.
    ///
    /// In contrast, with this function, a `FALSE` return from
    /// `gio.FileEnumerator.iterate` *always* means
    /// "error".  End of iteration is signaled by `out_info` or `out_child` being `NULL`.
    ///
    /// Another crucial difference is that the references for `out_info` and
    /// `out_child` are owned by `direnum` (they are cached as hidden
    /// properties).  You must not unref them in your own code.  This makes
    /// memory management significantly easier for C code in combination
    /// with loops.
    ///
    /// Finally, this function optionally allows retrieving a `gio.File` as
    /// well.
    ///
    /// You must specify at least one of `out_info` or `out_child`.
    ///
    /// The code pattern for correctly using `gio.FileEnumerator.iterate` from C
    /// is:
    ///
    /// ```
    /// direnum = g_file_enumerate_children (file, ...);
    /// while (TRUE)
    ///   {
    ///     GFileInfo *info;
    ///     if (!g_file_enumerator_iterate (direnum, &info, NULL, cancellable, error))
    ///       goto out;
    ///     if (!info)
    ///       break;
    ///     ... do stuff with "info"; do not unref it! ...
    ///   }
    ///
    /// out:
    ///   g_object_unref (direnum); // Note: frees the last `info`
    /// ```
    extern fn g_file_enumerator_iterate(p_direnum: *FileEnumerator, p_out_info: ?**gio.FileInfo, p_out_child: ?**gio.File, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const iterate = g_file_enumerator_iterate;

    /// Returns information for the next file in the enumerated object.
    /// Will block until the information is available. The `gio.FileInfo`
    /// returned from this function will contain attributes that match the
    /// attribute string that was passed when the `gio.FileEnumerator` was created.
    ///
    /// See the documentation of `gio.FileEnumerator` for information about the
    /// order of returned files.
    ///
    /// On error, returns `NULL` and sets `error` to the error. If the
    /// enumerator is at the end, `NULL` will be returned and `error` will
    /// be unset.
    extern fn g_file_enumerator_next_file(p_enumerator: *FileEnumerator, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.FileInfo;
    pub const nextFile = g_file_enumerator_next_file;

    /// Request information for a number of files from the enumerator asynchronously.
    /// When all I/O for the operation is finished the `callback` will be called with
    /// the requested information.
    ///
    /// See the documentation of `gio.FileEnumerator` for information about the
    /// order of returned files.
    ///
    /// Once the end of the enumerator is reached, or if an error occurs, the
    /// `callback` will be called with an empty list. In this case, the previous call
    /// to `gio.FileEnumerator.nextFilesAsync` will typically have returned fewer
    /// than `num_files` items.
    ///
    /// If a request is cancelled the callback will be called with
    /// `G_IO_ERROR_CANCELLED`.
    ///
    /// This leads to the following pseudo-code usage:
    /// ```
    /// g_autoptr(GFile) dir = get_directory ();
    /// g_autoptr(GFileEnumerator) enumerator = NULL;
    /// g_autolist(GFileInfo) files = NULL;
    /// g_autoptr(GError) local_error = NULL;
    ///
    /// enumerator = yield g_file_enumerate_children_async (dir,
    ///                                                     G_FILE_ATTRIBUTE_STANDARD_NAME ","
    ///                                                     G_FILE_ATTRIBUTE_STANDARD_TYPE,
    ///                                                     G_FILE_QUERY_INFO_NONE,
    ///                                                     G_PRIORITY_DEFAULT,
    ///                                                     cancellable,
    ///                                                     …,
    ///                                                     &local_error);
    /// if (enumerator == NULL)
    ///   g_error ("Error enumerating: `s`", local_error->message);
    ///
    /// // Loop until no files are returned, either because the end of the enumerator
    /// // has been reached, or an error was returned.
    /// do
    ///   {
    ///     files = yield g_file_enumerator_next_files_async (enumerator,
    ///                                                       5,  // number of files to request
    ///                                                       G_PRIORITY_DEFAULT,
    ///                                                       cancellable,
    ///                                                       …,
    ///                                                       &local_error);
    ///
    ///     // Process the returned files, but don’t assume that exactly 5 were returned.
    ///     for (GList *l = files; l != NULL; l = l->next)
    ///       {
    ///         GFileInfo *info = l->data;
    ///         handle_file_info (info);
    ///       }
    ///   }
    /// while (files != NULL);
    ///
    /// if (local_error != NULL &&
    ///     !g_error_matches (local_error, G_IO_ERROR, G_IO_ERROR_CANCELLED))
    ///   g_error ("Error while enumerating: `s`", local_error->message);
    /// ```
    ///
    /// During an async request no other sync and async calls are allowed, and will
    /// result in `G_IO_ERROR_PENDING` errors.
    ///
    /// Any outstanding I/O request with higher priority (lower numerical value) will
    /// be executed before an outstanding request with lower priority. Default
    /// priority is `G_PRIORITY_DEFAULT`.
    extern fn g_file_enumerator_next_files_async(p_enumerator: *FileEnumerator, p_num_files: c_int, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const nextFilesAsync = g_file_enumerator_next_files_async;

    /// Finishes the asynchronous operation started with `gio.FileEnumerator.nextFilesAsync`.
    extern fn g_file_enumerator_next_files_finish(p_enumerator: *FileEnumerator, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.List;
    pub const nextFilesFinish = g_file_enumerator_next_files_finish;

    /// Sets the file enumerator as having pending operations.
    extern fn g_file_enumerator_set_pending(p_enumerator: *FileEnumerator, p_pending: c_int) void;
    pub const setPending = g_file_enumerator_set_pending;

    extern fn g_file_enumerator_get_type() usize;
    pub const getGObjectType = g_file_enumerator_get_type;

    extern fn g_object_ref(p_self: *gio.FileEnumerator) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.FileEnumerator) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FileEnumerator, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GFileIOStream` provides I/O streams that both read and write to the same
/// file handle.
///
/// `GFileIOStream` implements `gio.Seekable`, which allows the I/O
/// stream to jump to arbitrary positions in the file and to truncate
/// the file, provided the filesystem of the file supports these
/// operations.
///
/// To find the position of a file I/O stream, use `gio.Seekable.tell`.
///
/// To find out if a file I/O stream supports seeking, use
/// `gio.Seekable.canSeek`. To position a file I/O stream, use
/// `gio.Seekable.seek`. To find out if a file I/O stream supports
/// truncating, use `gio.Seekable.canTruncate`. To truncate a file I/O
/// stream, use `gio.Seekable.truncate`.
///
/// The default implementation of all the `GFileIOStream` operations
/// and the implementation of `gio.Seekable` just call into the same
/// operations on the output stream.
pub const FileIOStream = extern struct {
    pub const Parent = gio.IOStream;
    pub const Implements = [_]type{gio.Seekable};
    pub const Class = gio.FileIOStreamClass;
    f_parent_instance: gio.IOStream,
    f_priv: ?*gio.FileIOStreamPrivate,

    pub const virtual_methods = struct {
        pub const can_seek = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) c_int {
                return gobject.ext.as(FileIOStream.Class, p_class).f_can_seek.?(gobject.ext.as(FileIOStream, p_stream));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) c_int) void {
                gobject.ext.as(FileIOStream.Class, p_class).f_can_seek = @ptrCast(p_implementation);
            }
        };

        pub const can_truncate = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) c_int {
                return gobject.ext.as(FileIOStream.Class, p_class).f_can_truncate.?(gobject.ext.as(FileIOStream, p_stream));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) c_int) void {
                gobject.ext.as(FileIOStream.Class, p_class).f_can_truncate = @ptrCast(p_implementation);
            }
        };

        /// Gets the entity tag for the file when it has been written.
        /// This must be called after the stream has been written
        /// and closed, as the etag can change while writing.
        pub const get_etag = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) ?[*:0]u8 {
                return gobject.ext.as(FileIOStream.Class, p_class).f_get_etag.?(gobject.ext.as(FileIOStream, p_stream));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) ?[*:0]u8) void {
                gobject.ext.as(FileIOStream.Class, p_class).f_get_etag = @ptrCast(p_implementation);
            }
        };

        /// Queries a file io stream for the given `attributes`.
        /// This function blocks while querying the stream. For the asynchronous
        /// version of this function, see `gio.FileIOStream.queryInfoAsync`.
        /// While the stream is blocked, the stream will set the pending flag
        /// internally, and any other operations on the stream will fail with
        /// `G_IO_ERROR_PENDING`.
        ///
        /// Can fail if the stream was already closed (with `error` being set to
        /// `G_IO_ERROR_CLOSED`), the stream has pending operations (with `error` being
        /// set to `G_IO_ERROR_PENDING`), or if querying info is not supported for
        /// the stream's interface (with `error` being set to `G_IO_ERROR_NOT_SUPPORTED`). I
        /// all cases of failure, `NULL` will be returned.
        ///
        /// If `cancellable` is not `NULL`, then the operation can be cancelled by
        /// triggering the cancellable object from another thread. If the operation
        /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be set, and `NULL` will
        /// be returned.
        pub const query_info = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_attributes: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.FileInfo {
                return gobject.ext.as(FileIOStream.Class, p_class).f_query_info.?(gobject.ext.as(FileIOStream, p_stream), p_attributes, p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_attributes: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) ?*gio.FileInfo) void {
                gobject.ext.as(FileIOStream.Class, p_class).f_query_info = @ptrCast(p_implementation);
            }
        };

        /// Asynchronously queries the `stream` for a `gio.FileInfo`. When completed,
        /// `callback` will be called with a `gio.AsyncResult` which can be used to
        /// finish the operation with `gio.FileIOStream.queryInfoFinish`.
        ///
        /// For the synchronous version of this function, see
        /// `gio.FileIOStream.queryInfo`.
        pub const query_info_async = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_attributes: [*:0]const u8, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(FileIOStream.Class, p_class).f_query_info_async.?(gobject.ext.as(FileIOStream, p_stream), p_attributes, p_io_priority, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_attributes: [*:0]const u8, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(FileIOStream.Class, p_class).f_query_info_async = @ptrCast(p_implementation);
            }
        };

        /// Finalizes the asynchronous query started
        /// by `gio.FileIOStream.queryInfoAsync`.
        pub const query_info_finish = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.FileInfo {
                return gobject.ext.as(FileIOStream.Class, p_class).f_query_info_finish.?(gobject.ext.as(FileIOStream, p_stream), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?*gio.FileInfo) void {
                gobject.ext.as(FileIOStream.Class, p_class).f_query_info_finish = @ptrCast(p_implementation);
            }
        };

        pub const seek = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_offset: i64, p_type: glib.SeekType, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(FileIOStream.Class, p_class).f_seek.?(gobject.ext.as(FileIOStream, p_stream), p_offset, p_type, p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_offset: i64, p_type: glib.SeekType, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(FileIOStream.Class, p_class).f_seek = @ptrCast(p_implementation);
            }
        };

        pub const tell = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) i64 {
                return gobject.ext.as(FileIOStream.Class, p_class).f_tell.?(gobject.ext.as(FileIOStream, p_stream));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) i64) void {
                gobject.ext.as(FileIOStream.Class, p_class).f_tell = @ptrCast(p_implementation);
            }
        };

        pub const truncate_fn = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_size: i64, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(FileIOStream.Class, p_class).f_truncate_fn.?(gobject.ext.as(FileIOStream, p_stream), p_size, p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_size: i64, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(FileIOStream.Class, p_class).f_truncate_fn = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {};

    /// Gets the entity tag for the file when it has been written.
    /// This must be called after the stream has been written
    /// and closed, as the etag can change while writing.
    extern fn g_file_io_stream_get_etag(p_stream: *FileIOStream) ?[*:0]u8;
    pub const getEtag = g_file_io_stream_get_etag;

    /// Queries a file io stream for the given `attributes`.
    /// This function blocks while querying the stream. For the asynchronous
    /// version of this function, see `gio.FileIOStream.queryInfoAsync`.
    /// While the stream is blocked, the stream will set the pending flag
    /// internally, and any other operations on the stream will fail with
    /// `G_IO_ERROR_PENDING`.
    ///
    /// Can fail if the stream was already closed (with `error` being set to
    /// `G_IO_ERROR_CLOSED`), the stream has pending operations (with `error` being
    /// set to `G_IO_ERROR_PENDING`), or if querying info is not supported for
    /// the stream's interface (with `error` being set to `G_IO_ERROR_NOT_SUPPORTED`). I
    /// all cases of failure, `NULL` will be returned.
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be set, and `NULL` will
    /// be returned.
    extern fn g_file_io_stream_query_info(p_stream: *FileIOStream, p_attributes: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.FileInfo;
    pub const queryInfo = g_file_io_stream_query_info;

    /// Asynchronously queries the `stream` for a `gio.FileInfo`. When completed,
    /// `callback` will be called with a `gio.AsyncResult` which can be used to
    /// finish the operation with `gio.FileIOStream.queryInfoFinish`.
    ///
    /// For the synchronous version of this function, see
    /// `gio.FileIOStream.queryInfo`.
    extern fn g_file_io_stream_query_info_async(p_stream: *FileIOStream, p_attributes: [*:0]const u8, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const queryInfoAsync = g_file_io_stream_query_info_async;

    /// Finalizes the asynchronous query started
    /// by `gio.FileIOStream.queryInfoAsync`.
    extern fn g_file_io_stream_query_info_finish(p_stream: *FileIOStream, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.FileInfo;
    pub const queryInfoFinish = g_file_io_stream_query_info_finish;

    extern fn g_file_io_stream_get_type() usize;
    pub const getGObjectType = g_file_io_stream_get_type;

    extern fn g_object_ref(p_self: *gio.FileIOStream) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.FileIOStream) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FileIOStream, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GFileIcon` specifies an icon by pointing to an image file
/// to be used as icon.
///
/// It implements `gio.LoadableIcon`.
pub const FileIcon = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{ gio.Icon, gio.LoadableIcon };
    pub const Class = gio.FileIconClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The file containing the icon.
        pub const file = struct {
            pub const name = "file";

            pub const Type = ?*gio.File;
        };
    };

    pub const signals = struct {};

    /// Creates a new icon for a file.
    extern fn g_file_icon_new(p_file: *gio.File) *gio.FileIcon;
    pub const new = g_file_icon_new;

    /// Gets the `gio.File` associated with the given `icon`.
    extern fn g_file_icon_get_file(p_icon: *FileIcon) *gio.File;
    pub const getFile = g_file_icon_get_file;

    extern fn g_file_icon_get_type() usize;
    pub const getGObjectType = g_file_icon_get_type;

    extern fn g_object_ref(p_self: *gio.FileIcon) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.FileIcon) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FileIcon, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Stores information about a file system object referenced by a `gio.File`.
///
/// Functionality for manipulating basic metadata for files. `GFileInfo`
/// implements methods for getting information that all files should
/// contain, and allows for manipulation of extended attributes.
///
/// See the [file attributes](file-attributes.html) document for more
/// information on how GIO handles file attributes.
///
/// To obtain a `GFileInfo` for a `gio.File`, use
/// `gio.File.queryInfo` (or its async variant). To obtain a `GFileInfo`
/// for a file input or output stream, use `gio.FileInputStream.queryInfo`
/// or `gio.FileOutputStream.queryInfo` (or their async variants).
///
/// To change the actual attributes of a file, you should then set the
/// attribute in the `GFileInfo` and call `gio.File.setAttributesFromInfo`
/// or `gio.File.setAttributesAsync` on a `GFile`.
///
/// However, not all attributes can be changed in the file. For instance,
/// the actual size of a file cannot be changed via `gio.FileInfo.setSize`.
/// You may call `gio.File.querySettableAttributes` and
/// `gio.File.queryWritableNamespaces` to discover the settable attributes
/// of a particular file at runtime.
///
/// The direct accessors, such as `gio.FileInfo.getName`, are slightly more
/// optimized than the generic attribute accessors, such as
/// `gio.FileInfo.getAttributeByteString`.This optimization will matter
/// only if calling the API in a tight loop.
///
/// It is an error to call these accessors without specifying their required file
/// attributes when creating the `GFileInfo`. Use
/// `gio.FileInfo.hasAttribute` or `gio.FileInfo.listAttributes`
/// to check what attributes are specified for a `GFileInfo`.
///
/// `gio.FileAttributeMatcher` allows for searching through a `GFileInfo`
/// for attributes.
pub const FileInfo = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.FileInfoClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new file info structure.
    extern fn g_file_info_new() *gio.FileInfo;
    pub const new = g_file_info_new;

    /// Clears the status information from `info`.
    extern fn g_file_info_clear_status(p_info: *FileInfo) void;
    pub const clearStatus = g_file_info_clear_status;

    /// First clears all of the [GFileAttribute][gio-GFileAttribute] of `dest_info`,
    /// and then copies all of the file attributes from `src_info` to `dest_info`.
    extern fn g_file_info_copy_into(p_src_info: *FileInfo, p_dest_info: *gio.FileInfo) void;
    pub const copyInto = g_file_info_copy_into;

    /// Duplicates a file info structure.
    extern fn g_file_info_dup(p_other: *FileInfo) *gio.FileInfo;
    pub const dup = g_file_info_dup;

    /// Gets the access time of the current `info` and returns it as a
    /// `glib.DateTime`.
    ///
    /// It is an error to call this if the `gio.FileInfo` does not contain
    /// `G_FILE_ATTRIBUTE_TIME_ACCESS`. If `G_FILE_ATTRIBUTE_TIME_ACCESS_USEC` is
    /// provided, the resulting `glib.DateTime` will additionally have microsecond
    /// precision.
    ///
    /// If nanosecond precision is needed, `G_FILE_ATTRIBUTE_TIME_ACCESS_NSEC` must
    /// be queried separately using `gio.FileInfo.getAttributeUint32`.
    extern fn g_file_info_get_access_date_time(p_info: *FileInfo) ?*glib.DateTime;
    pub const getAccessDateTime = g_file_info_get_access_date_time;

    /// Gets the value of an attribute, formatted as a string.
    /// This escapes things as needed to make the string valid
    /// UTF-8.
    extern fn g_file_info_get_attribute_as_string(p_info: *FileInfo, p_attribute: [*:0]const u8) ?[*:0]u8;
    pub const getAttributeAsString = g_file_info_get_attribute_as_string;

    /// Gets the value of a boolean attribute. If the attribute does not
    /// contain a boolean value, `FALSE` will be returned.
    extern fn g_file_info_get_attribute_boolean(p_info: *FileInfo, p_attribute: [*:0]const u8) c_int;
    pub const getAttributeBoolean = g_file_info_get_attribute_boolean;

    /// Gets the value of a byte string attribute. If the attribute does
    /// not contain a byte string, `NULL` will be returned.
    extern fn g_file_info_get_attribute_byte_string(p_info: *FileInfo, p_attribute: [*:0]const u8) ?[*:0]const u8;
    pub const getAttributeByteString = g_file_info_get_attribute_byte_string;

    /// Gets the attribute type, value and status for an attribute key.
    extern fn g_file_info_get_attribute_data(p_info: *FileInfo, p_attribute: [*:0]const u8, p_type: ?*gio.FileAttributeType, p_value_pp: ?*anyopaque, p_status: ?*gio.FileAttributeStatus) c_int;
    pub const getAttributeData = g_file_info_get_attribute_data;

    /// Gets the value of a byte string attribute as a file path.
    ///
    /// If the attribute does not contain a byte string, `NULL` will be returned.
    ///
    /// This function is meant to be used by language bindings that have specific
    /// handling for Unix paths.
    extern fn g_file_info_get_attribute_file_path(p_info: *FileInfo, p_attribute: [*:0]const u8) ?[*:0]const u8;
    pub const getAttributeFilePath = g_file_info_get_attribute_file_path;

    /// Gets a signed 32-bit integer contained within the attribute. If the
    /// attribute does not contain a signed 32-bit integer, or is invalid,
    /// 0 will be returned.
    extern fn g_file_info_get_attribute_int32(p_info: *FileInfo, p_attribute: [*:0]const u8) i32;
    pub const getAttributeInt32 = g_file_info_get_attribute_int32;

    /// Gets a signed 64-bit integer contained within the attribute. If the
    /// attribute does not contain a signed 64-bit integer, or is invalid,
    /// 0 will be returned.
    extern fn g_file_info_get_attribute_int64(p_info: *FileInfo, p_attribute: [*:0]const u8) i64;
    pub const getAttributeInt64 = g_file_info_get_attribute_int64;

    /// Gets the value of a `gobject.Object` attribute. If the attribute does
    /// not contain a `gobject.Object`, `NULL` will be returned.
    extern fn g_file_info_get_attribute_object(p_info: *FileInfo, p_attribute: [*:0]const u8) ?*gobject.Object;
    pub const getAttributeObject = g_file_info_get_attribute_object;

    /// Gets the attribute status for an attribute key.
    extern fn g_file_info_get_attribute_status(p_info: *FileInfo, p_attribute: [*:0]const u8) gio.FileAttributeStatus;
    pub const getAttributeStatus = g_file_info_get_attribute_status;

    /// Gets the value of a string attribute. If the attribute does
    /// not contain a string, `NULL` will be returned.
    extern fn g_file_info_get_attribute_string(p_info: *FileInfo, p_attribute: [*:0]const u8) ?[*:0]const u8;
    pub const getAttributeString = g_file_info_get_attribute_string;

    /// Gets the value of a stringv attribute. If the attribute does
    /// not contain a stringv, `NULL` will be returned.
    extern fn g_file_info_get_attribute_stringv(p_info: *FileInfo, p_attribute: [*:0]const u8) ?[*][*:0]u8;
    pub const getAttributeStringv = g_file_info_get_attribute_stringv;

    /// Gets the attribute type for an attribute key.
    extern fn g_file_info_get_attribute_type(p_info: *FileInfo, p_attribute: [*:0]const u8) gio.FileAttributeType;
    pub const getAttributeType = g_file_info_get_attribute_type;

    /// Gets an unsigned 32-bit integer contained within the attribute. If the
    /// attribute does not contain an unsigned 32-bit integer, or is invalid,
    /// 0 will be returned.
    extern fn g_file_info_get_attribute_uint32(p_info: *FileInfo, p_attribute: [*:0]const u8) u32;
    pub const getAttributeUint32 = g_file_info_get_attribute_uint32;

    /// Gets a unsigned 64-bit integer contained within the attribute. If the
    /// attribute does not contain an unsigned 64-bit integer, or is invalid,
    /// 0 will be returned.
    extern fn g_file_info_get_attribute_uint64(p_info: *FileInfo, p_attribute: [*:0]const u8) u64;
    pub const getAttributeUint64 = g_file_info_get_attribute_uint64;

    /// Gets the file's content type.
    ///
    /// It is an error to call this if the `gio.FileInfo` does not contain
    /// `G_FILE_ATTRIBUTE_STANDARD_CONTENT_TYPE`.
    extern fn g_file_info_get_content_type(p_info: *FileInfo) ?[*:0]const u8;
    pub const getContentType = g_file_info_get_content_type;

    /// Gets the creation time of the current `info` and returns it as a
    /// `glib.DateTime`.
    ///
    /// It is an error to call this if the `gio.FileInfo` does not contain
    /// `G_FILE_ATTRIBUTE_TIME_CREATED`. If `G_FILE_ATTRIBUTE_TIME_CREATED_USEC` is
    /// provided, the resulting `glib.DateTime` will additionally have microsecond
    /// precision.
    ///
    /// If nanosecond precision is needed, `G_FILE_ATTRIBUTE_TIME_CREATED_NSEC` must
    /// be queried separately using `gio.FileInfo.getAttributeUint32`.
    extern fn g_file_info_get_creation_date_time(p_info: *FileInfo) ?*glib.DateTime;
    pub const getCreationDateTime = g_file_info_get_creation_date_time;

    /// Returns the `glib.DateTime` representing the deletion date of the file, as
    /// available in `G_FILE_ATTRIBUTE_TRASH_DELETION_DATE`. If the
    /// `G_FILE_ATTRIBUTE_TRASH_DELETION_DATE` attribute is unset, `NULL` is returned.
    extern fn g_file_info_get_deletion_date(p_info: *FileInfo) ?*glib.DateTime;
    pub const getDeletionDate = g_file_info_get_deletion_date;

    /// Gets a display name for a file. This is guaranteed to always be set.
    ///
    /// It is an error to call this if the `gio.FileInfo` does not contain
    /// `G_FILE_ATTRIBUTE_STANDARD_DISPLAY_NAME`.
    extern fn g_file_info_get_display_name(p_info: *FileInfo) [*:0]const u8;
    pub const getDisplayName = g_file_info_get_display_name;

    /// Gets the edit name for a file.
    ///
    /// It is an error to call this if the `gio.FileInfo` does not contain
    /// `G_FILE_ATTRIBUTE_STANDARD_EDIT_NAME`.
    extern fn g_file_info_get_edit_name(p_info: *FileInfo) [*:0]const u8;
    pub const getEditName = g_file_info_get_edit_name;

    /// Gets the [entity tag](iface.File.html`entity`-tags) for a given
    /// `gio.FileInfo`. See `G_FILE_ATTRIBUTE_ETAG_VALUE`.
    ///
    /// It is an error to call this if the `gio.FileInfo` does not contain
    /// `G_FILE_ATTRIBUTE_ETAG_VALUE`.
    extern fn g_file_info_get_etag(p_info: *FileInfo) ?[*:0]const u8;
    pub const getEtag = g_file_info_get_etag;

    /// Gets a file's type (whether it is a regular file, symlink, etc).
    /// This is different from the file's content type, see `gio.FileInfo.getContentType`.
    ///
    /// It is an error to call this if the `gio.FileInfo` does not contain
    /// `G_FILE_ATTRIBUTE_STANDARD_TYPE`.
    extern fn g_file_info_get_file_type(p_info: *FileInfo) gio.FileType;
    pub const getFileType = g_file_info_get_file_type;

    /// Gets the icon for a file.
    ///
    /// It is an error to call this if the `gio.FileInfo` does not contain
    /// `G_FILE_ATTRIBUTE_STANDARD_ICON`.
    extern fn g_file_info_get_icon(p_info: *FileInfo) ?*gio.Icon;
    pub const getIcon = g_file_info_get_icon;

    /// Checks if a file is a backup file.
    ///
    /// It is an error to call this if the `gio.FileInfo` does not contain
    /// `G_FILE_ATTRIBUTE_STANDARD_IS_BACKUP`.
    extern fn g_file_info_get_is_backup(p_info: *FileInfo) c_int;
    pub const getIsBackup = g_file_info_get_is_backup;

    /// Checks if a file is hidden.
    ///
    /// It is an error to call this if the `gio.FileInfo` does not contain
    /// `G_FILE_ATTRIBUTE_STANDARD_IS_HIDDEN`.
    extern fn g_file_info_get_is_hidden(p_info: *FileInfo) c_int;
    pub const getIsHidden = g_file_info_get_is_hidden;

    /// Checks if a file is a symlink.
    ///
    /// It is an error to call this if the `gio.FileInfo` does not contain
    /// `G_FILE_ATTRIBUTE_STANDARD_IS_SYMLINK`.
    extern fn g_file_info_get_is_symlink(p_info: *FileInfo) c_int;
    pub const getIsSymlink = g_file_info_get_is_symlink;

    /// Gets the modification time of the current `info` and returns it as a
    /// `glib.DateTime`.
    ///
    /// It is an error to call this if the `gio.FileInfo` does not contain
    /// `G_FILE_ATTRIBUTE_TIME_MODIFIED`. If `G_FILE_ATTRIBUTE_TIME_MODIFIED_USEC` is
    /// provided, the resulting `glib.DateTime` will additionally have microsecond
    /// precision.
    ///
    /// If nanosecond precision is needed, `G_FILE_ATTRIBUTE_TIME_MODIFIED_NSEC` must
    /// be queried separately using `gio.FileInfo.getAttributeUint32`.
    extern fn g_file_info_get_modification_date_time(p_info: *FileInfo) ?*glib.DateTime;
    pub const getModificationDateTime = g_file_info_get_modification_date_time;

    /// Gets the modification time of the current `info` and sets it
    /// in `result`.
    ///
    /// It is an error to call this if the `gio.FileInfo` does not contain
    /// `G_FILE_ATTRIBUTE_TIME_MODIFIED`. If `G_FILE_ATTRIBUTE_TIME_MODIFIED_USEC` is
    /// provided it will be used too.
    extern fn g_file_info_get_modification_time(p_info: *FileInfo, p_result: *glib.TimeVal) void;
    pub const getModificationTime = g_file_info_get_modification_time;

    /// Gets the name for a file. This is guaranteed to always be set.
    ///
    /// It is an error to call this if the `gio.FileInfo` does not contain
    /// `G_FILE_ATTRIBUTE_STANDARD_NAME`.
    extern fn g_file_info_get_name(p_info: *FileInfo) [*:0]const u8;
    pub const getName = g_file_info_get_name;

    /// Gets the file's size (in bytes). The size is retrieved through the value of
    /// the `G_FILE_ATTRIBUTE_STANDARD_SIZE` attribute and is converted
    /// from `guint64` to `goffset` before returning the result.
    ///
    /// It is an error to call this if the `gio.FileInfo` does not contain
    /// `G_FILE_ATTRIBUTE_STANDARD_SIZE`.
    extern fn g_file_info_get_size(p_info: *FileInfo) i64;
    pub const getSize = g_file_info_get_size;

    /// Gets the value of the sort_order attribute from the `gio.FileInfo`.
    /// See `G_FILE_ATTRIBUTE_STANDARD_SORT_ORDER`.
    ///
    /// It is an error to call this if the `gio.FileInfo` does not contain
    /// `G_FILE_ATTRIBUTE_STANDARD_SORT_ORDER`.
    extern fn g_file_info_get_sort_order(p_info: *FileInfo) i32;
    pub const getSortOrder = g_file_info_get_sort_order;

    /// Gets the symbolic icon for a file.
    ///
    /// It is an error to call this if the `gio.FileInfo` does not contain
    /// `G_FILE_ATTRIBUTE_STANDARD_SYMBOLIC_ICON`.
    extern fn g_file_info_get_symbolic_icon(p_info: *FileInfo) ?*gio.Icon;
    pub const getSymbolicIcon = g_file_info_get_symbolic_icon;

    /// Gets the symlink target for a given `gio.FileInfo`.
    ///
    /// It is an error to call this if the `gio.FileInfo` does not contain
    /// `G_FILE_ATTRIBUTE_STANDARD_SYMLINK_TARGET`.
    extern fn g_file_info_get_symlink_target(p_info: *FileInfo) ?[*:0]const u8;
    pub const getSymlinkTarget = g_file_info_get_symlink_target;

    /// Checks if a file info structure has an attribute named `attribute`.
    extern fn g_file_info_has_attribute(p_info: *FileInfo, p_attribute: [*:0]const u8) c_int;
    pub const hasAttribute = g_file_info_has_attribute;

    /// Checks if a file info structure has an attribute in the
    /// specified `name_space`.
    extern fn g_file_info_has_namespace(p_info: *FileInfo, p_name_space: [*:0]const u8) c_int;
    pub const hasNamespace = g_file_info_has_namespace;

    /// Lists the file info structure's attributes.
    extern fn g_file_info_list_attributes(p_info: *FileInfo, p_name_space: ?[*:0]const u8) ?[*][*:0]u8;
    pub const listAttributes = g_file_info_list_attributes;

    /// Removes all cases of `attribute` from `info` if it exists.
    extern fn g_file_info_remove_attribute(p_info: *FileInfo, p_attribute: [*:0]const u8) void;
    pub const removeAttribute = g_file_info_remove_attribute;

    /// Sets the `G_FILE_ATTRIBUTE_TIME_ACCESS` and
    /// `G_FILE_ATTRIBUTE_TIME_ACCESS_USEC` attributes in the file info to the
    /// given date/time value.
    ///
    /// `G_FILE_ATTRIBUTE_TIME_ACCESS_NSEC` will be cleared.
    extern fn g_file_info_set_access_date_time(p_info: *FileInfo, p_atime: *glib.DateTime) void;
    pub const setAccessDateTime = g_file_info_set_access_date_time;

    /// Sets the `attribute` to contain the given value, if possible. To unset the
    /// attribute, use `G_FILE_ATTRIBUTE_TYPE_INVALID` for `type`.
    extern fn g_file_info_set_attribute(p_info: *FileInfo, p_attribute: [*:0]const u8, p_type: gio.FileAttributeType, p_value_p: *anyopaque) void;
    pub const setAttribute = g_file_info_set_attribute;

    /// Sets the `attribute` to contain the given `attr_value`,
    /// if possible.
    extern fn g_file_info_set_attribute_boolean(p_info: *FileInfo, p_attribute: [*:0]const u8, p_attr_value: c_int) void;
    pub const setAttributeBoolean = g_file_info_set_attribute_boolean;

    /// Sets the `attribute` to contain the given `attr_value`,
    /// if possible.
    extern fn g_file_info_set_attribute_byte_string(p_info: *FileInfo, p_attribute: [*:0]const u8, p_attr_value: [*:0]const u8) void;
    pub const setAttributeByteString = g_file_info_set_attribute_byte_string;

    /// Sets the `attribute` to contain the given `attr_value`,
    /// if possible.
    ///
    /// This function is meant to be used by language bindings that have specific
    /// handling for Unix paths.
    extern fn g_file_info_set_attribute_file_path(p_info: *FileInfo, p_attribute: [*:0]const u8, p_attr_value: [*:0]const u8) void;
    pub const setAttributeFilePath = g_file_info_set_attribute_file_path;

    /// Sets the `attribute` to contain the given `attr_value`,
    /// if possible.
    extern fn g_file_info_set_attribute_int32(p_info: *FileInfo, p_attribute: [*:0]const u8, p_attr_value: i32) void;
    pub const setAttributeInt32 = g_file_info_set_attribute_int32;

    /// Sets the `attribute` to contain the given `attr_value`,
    /// if possible.
    extern fn g_file_info_set_attribute_int64(p_info: *FileInfo, p_attribute: [*:0]const u8, p_attr_value: i64) void;
    pub const setAttributeInt64 = g_file_info_set_attribute_int64;

    /// Sets `mask` on `info` to match specific attribute types.
    extern fn g_file_info_set_attribute_mask(p_info: *FileInfo, p_mask: *gio.FileAttributeMatcher) void;
    pub const setAttributeMask = g_file_info_set_attribute_mask;

    /// Sets the `attribute` to contain the given `attr_value`,
    /// if possible.
    extern fn g_file_info_set_attribute_object(p_info: *FileInfo, p_attribute: [*:0]const u8, p_attr_value: *gobject.Object) void;
    pub const setAttributeObject = g_file_info_set_attribute_object;

    /// Sets the attribute status for an attribute key. This is only
    /// needed by external code that implement `gio.File.setAttributesFromInfo`
    /// or similar functions.
    ///
    /// The attribute must exist in `info` for this to work. Otherwise `FALSE`
    /// is returned and `info` is unchanged.
    extern fn g_file_info_set_attribute_status(p_info: *FileInfo, p_attribute: [*:0]const u8, p_status: gio.FileAttributeStatus) c_int;
    pub const setAttributeStatus = g_file_info_set_attribute_status;

    /// Sets the `attribute` to contain the given `attr_value`,
    /// if possible.
    extern fn g_file_info_set_attribute_string(p_info: *FileInfo, p_attribute: [*:0]const u8, p_attr_value: [*:0]const u8) void;
    pub const setAttributeString = g_file_info_set_attribute_string;

    /// Sets the `attribute` to contain the given `attr_value`,
    /// if possible.
    ///
    /// Sinze: 2.22
    extern fn g_file_info_set_attribute_stringv(p_info: *FileInfo, p_attribute: [*:0]const u8, p_attr_value: [*][*:0]u8) void;
    pub const setAttributeStringv = g_file_info_set_attribute_stringv;

    /// Sets the `attribute` to contain the given `attr_value`,
    /// if possible.
    extern fn g_file_info_set_attribute_uint32(p_info: *FileInfo, p_attribute: [*:0]const u8, p_attr_value: u32) void;
    pub const setAttributeUint32 = g_file_info_set_attribute_uint32;

    /// Sets the `attribute` to contain the given `attr_value`,
    /// if possible.
    extern fn g_file_info_set_attribute_uint64(p_info: *FileInfo, p_attribute: [*:0]const u8, p_attr_value: u64) void;
    pub const setAttributeUint64 = g_file_info_set_attribute_uint64;

    /// Sets the content type attribute for a given `gio.FileInfo`.
    /// See `G_FILE_ATTRIBUTE_STANDARD_CONTENT_TYPE`.
    extern fn g_file_info_set_content_type(p_info: *FileInfo, p_content_type: [*:0]const u8) void;
    pub const setContentType = g_file_info_set_content_type;

    /// Sets the `G_FILE_ATTRIBUTE_TIME_CREATED` and
    /// `G_FILE_ATTRIBUTE_TIME_CREATED_USEC` attributes in the file info to the
    /// given date/time value.
    ///
    /// `G_FILE_ATTRIBUTE_TIME_CREATED_NSEC` will be cleared.
    extern fn g_file_info_set_creation_date_time(p_info: *FileInfo, p_creation_time: *glib.DateTime) void;
    pub const setCreationDateTime = g_file_info_set_creation_date_time;

    /// Sets the display name for the current `gio.FileInfo`.
    /// See `G_FILE_ATTRIBUTE_STANDARD_DISPLAY_NAME`.
    extern fn g_file_info_set_display_name(p_info: *FileInfo, p_display_name: [*:0]const u8) void;
    pub const setDisplayName = g_file_info_set_display_name;

    /// Sets the edit name for the current file.
    /// See `G_FILE_ATTRIBUTE_STANDARD_EDIT_NAME`.
    extern fn g_file_info_set_edit_name(p_info: *FileInfo, p_edit_name: [*:0]const u8) void;
    pub const setEditName = g_file_info_set_edit_name;

    /// Sets the file type in a `gio.FileInfo` to `type`.
    /// See `G_FILE_ATTRIBUTE_STANDARD_TYPE`.
    extern fn g_file_info_set_file_type(p_info: *FileInfo, p_type: gio.FileType) void;
    pub const setFileType = g_file_info_set_file_type;

    /// Sets the icon for a given `gio.FileInfo`.
    /// See `G_FILE_ATTRIBUTE_STANDARD_ICON`.
    extern fn g_file_info_set_icon(p_info: *FileInfo, p_icon: *gio.Icon) void;
    pub const setIcon = g_file_info_set_icon;

    /// Sets the "is_hidden" attribute in a `gio.FileInfo` according to `is_hidden`.
    /// See `G_FILE_ATTRIBUTE_STANDARD_IS_HIDDEN`.
    extern fn g_file_info_set_is_hidden(p_info: *FileInfo, p_is_hidden: c_int) void;
    pub const setIsHidden = g_file_info_set_is_hidden;

    /// Sets the "is_symlink" attribute in a `gio.FileInfo` according to `is_symlink`.
    /// See `G_FILE_ATTRIBUTE_STANDARD_IS_SYMLINK`.
    extern fn g_file_info_set_is_symlink(p_info: *FileInfo, p_is_symlink: c_int) void;
    pub const setIsSymlink = g_file_info_set_is_symlink;

    /// Sets the `G_FILE_ATTRIBUTE_TIME_MODIFIED` and
    /// `G_FILE_ATTRIBUTE_TIME_MODIFIED_USEC` attributes in the file info to the
    /// given date/time value.
    ///
    /// `G_FILE_ATTRIBUTE_TIME_MODIFIED_NSEC` will be cleared.
    extern fn g_file_info_set_modification_date_time(p_info: *FileInfo, p_mtime: *glib.DateTime) void;
    pub const setModificationDateTime = g_file_info_set_modification_date_time;

    /// Sets the `G_FILE_ATTRIBUTE_TIME_MODIFIED` and
    /// `G_FILE_ATTRIBUTE_TIME_MODIFIED_USEC` attributes in the file info to the
    /// given time value.
    ///
    /// `G_FILE_ATTRIBUTE_TIME_MODIFIED_NSEC` will be cleared.
    extern fn g_file_info_set_modification_time(p_info: *FileInfo, p_mtime: *glib.TimeVal) void;
    pub const setModificationTime = g_file_info_set_modification_time;

    /// Sets the name attribute for the current `gio.FileInfo`.
    /// See `G_FILE_ATTRIBUTE_STANDARD_NAME`.
    extern fn g_file_info_set_name(p_info: *FileInfo, p_name: [*:0]const u8) void;
    pub const setName = g_file_info_set_name;

    /// Sets the `G_FILE_ATTRIBUTE_STANDARD_SIZE` attribute in the file info
    /// to the given size.
    extern fn g_file_info_set_size(p_info: *FileInfo, p_size: i64) void;
    pub const setSize = g_file_info_set_size;

    /// Sets the sort order attribute in the file info structure. See
    /// `G_FILE_ATTRIBUTE_STANDARD_SORT_ORDER`.
    extern fn g_file_info_set_sort_order(p_info: *FileInfo, p_sort_order: i32) void;
    pub const setSortOrder = g_file_info_set_sort_order;

    /// Sets the symbolic icon for a given `gio.FileInfo`.
    /// See `G_FILE_ATTRIBUTE_STANDARD_SYMBOLIC_ICON`.
    extern fn g_file_info_set_symbolic_icon(p_info: *FileInfo, p_icon: *gio.Icon) void;
    pub const setSymbolicIcon = g_file_info_set_symbolic_icon;

    /// Sets the `G_FILE_ATTRIBUTE_STANDARD_SYMLINK_TARGET` attribute in the file info
    /// to the given symlink target.
    extern fn g_file_info_set_symlink_target(p_info: *FileInfo, p_symlink_target: [*:0]const u8) void;
    pub const setSymlinkTarget = g_file_info_set_symlink_target;

    /// Unsets a mask set by `gio.FileInfo.setAttributeMask`, if one
    /// is set.
    extern fn g_file_info_unset_attribute_mask(p_info: *FileInfo) void;
    pub const unsetAttributeMask = g_file_info_unset_attribute_mask;

    extern fn g_file_info_get_type() usize;
    pub const getGObjectType = g_file_info_get_type;

    extern fn g_object_ref(p_self: *gio.FileInfo) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.FileInfo) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FileInfo, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GFileInputStream` provides input streams that take their
/// content from a file.
///
/// `GFileInputStream` implements `gio.Seekable`, which allows the input
/// stream to jump to arbitrary positions in the file, provided the
/// filesystem of the file allows it. To find the position of a file
/// input stream, use `gio.Seekable.tell`. To find out if a file input
/// stream supports seeking, use `gio.Seekable.virtual_methods.can_seek`.
/// To position a file input stream, use `gio.Seekable.virtual_methods.seek`.
pub const FileInputStream = extern struct {
    pub const Parent = gio.InputStream;
    pub const Implements = [_]type{gio.Seekable};
    pub const Class = gio.FileInputStreamClass;
    f_parent_instance: gio.InputStream,
    f_priv: ?*gio.FileInputStreamPrivate,

    pub const virtual_methods = struct {
        pub const can_seek = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) c_int {
                return gobject.ext.as(FileInputStream.Class, p_class).f_can_seek.?(gobject.ext.as(FileInputStream, p_stream));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) c_int) void {
                gobject.ext.as(FileInputStream.Class, p_class).f_can_seek = @ptrCast(p_implementation);
            }
        };

        /// Queries a file input stream the given `attributes`. This function blocks
        /// while querying the stream. For the asynchronous (non-blocking) version
        /// of this function, see `gio.FileInputStream.queryInfoAsync`. While the
        /// stream is blocked, the stream will set the pending flag internally, and
        /// any other operations on the stream will fail with `G_IO_ERROR_PENDING`.
        pub const query_info = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_attributes: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.FileInfo {
                return gobject.ext.as(FileInputStream.Class, p_class).f_query_info.?(gobject.ext.as(FileInputStream, p_stream), p_attributes, p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_attributes: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) ?*gio.FileInfo) void {
                gobject.ext.as(FileInputStream.Class, p_class).f_query_info = @ptrCast(p_implementation);
            }
        };

        /// Queries the stream information asynchronously.
        /// When the operation is finished `callback` will be called.
        /// You can then call `gio.FileInputStream.queryInfoFinish`
        /// to get the result of the operation.
        ///
        /// For the synchronous version of this function,
        /// see `gio.FileInputStream.queryInfo`.
        ///
        /// If `cancellable` is not `NULL`, then the operation can be cancelled by
        /// triggering the cancellable object from another thread. If the operation
        /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be set
        pub const query_info_async = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_attributes: [*:0]const u8, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(FileInputStream.Class, p_class).f_query_info_async.?(gobject.ext.as(FileInputStream, p_stream), p_attributes, p_io_priority, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_attributes: [*:0]const u8, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(FileInputStream.Class, p_class).f_query_info_async = @ptrCast(p_implementation);
            }
        };

        /// Finishes an asynchronous info query operation.
        pub const query_info_finish = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.FileInfo {
                return gobject.ext.as(FileInputStream.Class, p_class).f_query_info_finish.?(gobject.ext.as(FileInputStream, p_stream), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?*gio.FileInfo) void {
                gobject.ext.as(FileInputStream.Class, p_class).f_query_info_finish = @ptrCast(p_implementation);
            }
        };

        pub const seek = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_offset: i64, p_type: glib.SeekType, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(FileInputStream.Class, p_class).f_seek.?(gobject.ext.as(FileInputStream, p_stream), p_offset, p_type, p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_offset: i64, p_type: glib.SeekType, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(FileInputStream.Class, p_class).f_seek = @ptrCast(p_implementation);
            }
        };

        pub const tell = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) i64 {
                return gobject.ext.as(FileInputStream.Class, p_class).f_tell.?(gobject.ext.as(FileInputStream, p_stream));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) i64) void {
                gobject.ext.as(FileInputStream.Class, p_class).f_tell = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {};

    /// Queries a file input stream the given `attributes`. This function blocks
    /// while querying the stream. For the asynchronous (non-blocking) version
    /// of this function, see `gio.FileInputStream.queryInfoAsync`. While the
    /// stream is blocked, the stream will set the pending flag internally, and
    /// any other operations on the stream will fail with `G_IO_ERROR_PENDING`.
    extern fn g_file_input_stream_query_info(p_stream: *FileInputStream, p_attributes: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.FileInfo;
    pub const queryInfo = g_file_input_stream_query_info;

    /// Queries the stream information asynchronously.
    /// When the operation is finished `callback` will be called.
    /// You can then call `gio.FileInputStream.queryInfoFinish`
    /// to get the result of the operation.
    ///
    /// For the synchronous version of this function,
    /// see `gio.FileInputStream.queryInfo`.
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be set
    extern fn g_file_input_stream_query_info_async(p_stream: *FileInputStream, p_attributes: [*:0]const u8, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const queryInfoAsync = g_file_input_stream_query_info_async;

    /// Finishes an asynchronous info query operation.
    extern fn g_file_input_stream_query_info_finish(p_stream: *FileInputStream, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.FileInfo;
    pub const queryInfoFinish = g_file_input_stream_query_info_finish;

    extern fn g_file_input_stream_get_type() usize;
    pub const getGObjectType = g_file_input_stream_get_type;

    extern fn g_object_ref(p_self: *gio.FileInputStream) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.FileInputStream) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FileInputStream, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Monitors a file or directory for changes.
///
/// To obtain a `GFileMonitor` for a file or directory, use
/// `gio.File.monitor`, `gio.File.monitorFile`, or
/// `gio.File.monitorDirectory`.
///
/// To get informed about changes to the file or directory you are
/// monitoring, connect to the `gio.FileMonitor.signals.changed` signal. The
/// signal will be emitted in the thread-default main context (see
/// `glib.MainContext.pushThreadDefault`) of the thread that the monitor
/// was created in (though if the global default main context is blocked, this
/// may cause notifications to be blocked even if the thread-default
/// context is still running).
pub const FileMonitor = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.FileMonitorClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.FileMonitorPrivate,

    pub const virtual_methods = struct {
        /// Cancels a file monitor.
        pub const cancel = struct {
            pub fn call(p_class: anytype, p_monitor: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) c_int {
                return gobject.ext.as(FileMonitor.Class, p_class).f_cancel.?(gobject.ext.as(FileMonitor, p_monitor));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_monitor: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) c_int) void {
                gobject.ext.as(FileMonitor.Class, p_class).f_cancel = @ptrCast(p_implementation);
            }
        };

        pub const changed = struct {
            pub fn call(p_class: anytype, p_monitor: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_file: *gio.File, p_other_file: *gio.File, p_event_type: gio.FileMonitorEvent) void {
                return gobject.ext.as(FileMonitor.Class, p_class).f_changed.?(gobject.ext.as(FileMonitor, p_monitor), p_file, p_other_file, p_event_type);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_monitor: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_file: *gio.File, p_other_file: *gio.File, p_event_type: gio.FileMonitorEvent) callconv(.C) void) void {
                gobject.ext.as(FileMonitor.Class, p_class).f_changed = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// Whether the monitor has been cancelled.
        pub const cancelled = struct {
            pub const name = "cancelled";

            pub const Type = c_int;
        };

        /// The limit of the monitor to watch for changes, in milliseconds.
        pub const rate_limit = struct {
            pub const name = "rate-limit";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// Emitted when `file` has been changed.
        ///
        /// If using `G_FILE_MONITOR_WATCH_MOVES` on a directory monitor, and
        /// the information is available (and if supported by the backend),
        /// `event_type` may be `G_FILE_MONITOR_EVENT_RENAMED`,
        /// `G_FILE_MONITOR_EVENT_MOVED_IN` or `G_FILE_MONITOR_EVENT_MOVED_OUT`.
        ///
        /// In all cases `file` will be a child of the monitored directory.  For
        /// renames, `file` will be the old name and `other_file` is the new
        /// name.  For "moved in" events, `file` is the name of the file that
        /// appeared and `other_file` is the old name that it was moved from (in
        /// another directory).  For "moved out" events, `file` is the name of
        /// the file that used to be in this directory and `other_file` is the
        /// name of the file at its new location.
        ///
        /// It makes sense to treat `G_FILE_MONITOR_EVENT_MOVED_IN` as
        /// equivalent to `G_FILE_MONITOR_EVENT_CREATED` and
        /// `G_FILE_MONITOR_EVENT_MOVED_OUT` as equivalent to
        /// `G_FILE_MONITOR_EVENT_DELETED`, with extra information.
        /// `G_FILE_MONITOR_EVENT_RENAMED` is equivalent to a delete/create
        /// pair.  This is exactly how the events will be reported in the case
        /// that the `G_FILE_MONITOR_WATCH_MOVES` flag is not in use.
        ///
        /// If using the deprecated flag `G_FILE_MONITOR_SEND_MOVED` flag and `event_type` is
        /// `G_FILE_MONITOR_EVENT_MOVED`, `file` will be set to a `gio.File` containing the
        /// old path, and `other_file` will be set to a `gio.File` containing the new path.
        ///
        /// In all the other cases, `other_file` will be set to `NULL`.
        pub const changed = struct {
            pub const name = "changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_file: *gio.File, p_other_file: ?*gio.File, p_event_type: gio.FileMonitorEvent, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FileMonitor, p_instance))),
                    gobject.signalLookup("changed", FileMonitor.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Cancels a file monitor.
    extern fn g_file_monitor_cancel(p_monitor: *FileMonitor) c_int;
    pub const cancel = g_file_monitor_cancel;

    /// Emits the `gio.FileMonitor.signals.changed` signal if a change
    /// has taken place. Should be called from file monitor
    /// implementations only.
    ///
    /// Implementations are responsible to call this method from the
    /// [thread-default main context][g-main-context-push-thread-default] of the
    /// thread that the monitor was created in.
    extern fn g_file_monitor_emit_event(p_monitor: *FileMonitor, p_child: *gio.File, p_other_file: *gio.File, p_event_type: gio.FileMonitorEvent) void;
    pub const emitEvent = g_file_monitor_emit_event;

    /// Returns whether the monitor is canceled.
    extern fn g_file_monitor_is_cancelled(p_monitor: *FileMonitor) c_int;
    pub const isCancelled = g_file_monitor_is_cancelled;

    /// Sets the rate limit to which the `monitor` will report
    /// consecutive change events to the same file.
    extern fn g_file_monitor_set_rate_limit(p_monitor: *FileMonitor, p_limit_msecs: c_int) void;
    pub const setRateLimit = g_file_monitor_set_rate_limit;

    extern fn g_file_monitor_get_type() usize;
    pub const getGObjectType = g_file_monitor_get_type;

    extern fn g_object_ref(p_self: *gio.FileMonitor) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.FileMonitor) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FileMonitor, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GFileOutputStream` provides output streams that write their
/// content to a file.
///
/// `GFileOutputStream` implements `gio.Seekable`, which allows the output
/// stream to jump to arbitrary positions in the file and to truncate
/// the file, provided the filesystem of the file supports these
/// operations.
///
/// To find the position of a file output stream, use `gio.Seekable.tell`.
/// To find out if a file output stream supports seeking, use
/// `gio.Seekable.canSeek`.To position a file output stream, use
/// `gio.Seekable.seek`. To find out if a file output stream supports
/// truncating, use `gio.Seekable.canTruncate`. To truncate a file output
/// stream, use `gio.Seekable.truncate`.
pub const FileOutputStream = extern struct {
    pub const Parent = gio.OutputStream;
    pub const Implements = [_]type{gio.Seekable};
    pub const Class = gio.FileOutputStreamClass;
    f_parent_instance: gio.OutputStream,
    f_priv: ?*gio.FileOutputStreamPrivate,

    pub const virtual_methods = struct {
        pub const can_seek = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) c_int {
                return gobject.ext.as(FileOutputStream.Class, p_class).f_can_seek.?(gobject.ext.as(FileOutputStream, p_stream));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) c_int) void {
                gobject.ext.as(FileOutputStream.Class, p_class).f_can_seek = @ptrCast(p_implementation);
            }
        };

        pub const can_truncate = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) c_int {
                return gobject.ext.as(FileOutputStream.Class, p_class).f_can_truncate.?(gobject.ext.as(FileOutputStream, p_stream));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) c_int) void {
                gobject.ext.as(FileOutputStream.Class, p_class).f_can_truncate = @ptrCast(p_implementation);
            }
        };

        /// Gets the entity tag for the file when it has been written.
        /// This must be called after the stream has been written
        /// and closed, as the etag can change while writing.
        pub const get_etag = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) ?[*:0]u8 {
                return gobject.ext.as(FileOutputStream.Class, p_class).f_get_etag.?(gobject.ext.as(FileOutputStream, p_stream));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) ?[*:0]u8) void {
                gobject.ext.as(FileOutputStream.Class, p_class).f_get_etag = @ptrCast(p_implementation);
            }
        };

        /// Queries a file output stream for the given `attributes`.
        /// This function blocks while querying the stream. For the asynchronous
        /// version of this function, see `gio.FileOutputStream.queryInfoAsync`.
        /// While the stream is blocked, the stream will set the pending flag
        /// internally, and any other operations on the stream will fail with
        /// `G_IO_ERROR_PENDING`.
        ///
        /// Can fail if the stream was already closed (with `error` being set to
        /// `G_IO_ERROR_CLOSED`), the stream has pending operations (with `error` being
        /// set to `G_IO_ERROR_PENDING`), or if querying info is not supported for
        /// the stream's interface (with `error` being set to `G_IO_ERROR_NOT_SUPPORTED`). In
        /// all cases of failure, `NULL` will be returned.
        ///
        /// If `cancellable` is not `NULL`, then the operation can be cancelled by
        /// triggering the cancellable object from another thread. If the operation
        /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be set, and `NULL` will
        /// be returned.
        pub const query_info = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_attributes: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.FileInfo {
                return gobject.ext.as(FileOutputStream.Class, p_class).f_query_info.?(gobject.ext.as(FileOutputStream, p_stream), p_attributes, p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_attributes: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) ?*gio.FileInfo) void {
                gobject.ext.as(FileOutputStream.Class, p_class).f_query_info = @ptrCast(p_implementation);
            }
        };

        /// Asynchronously queries the `stream` for a `gio.FileInfo`. When completed,
        /// `callback` will be called with a `gio.AsyncResult` which can be used to
        /// finish the operation with `gio.FileOutputStream.queryInfoFinish`.
        ///
        /// For the synchronous version of this function, see
        /// `gio.FileOutputStream.queryInfo`.
        pub const query_info_async = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_attributes: [*:0]const u8, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(FileOutputStream.Class, p_class).f_query_info_async.?(gobject.ext.as(FileOutputStream, p_stream), p_attributes, p_io_priority, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_attributes: [*:0]const u8, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(FileOutputStream.Class, p_class).f_query_info_async = @ptrCast(p_implementation);
            }
        };

        /// Finalizes the asynchronous query started
        /// by `gio.FileOutputStream.queryInfoAsync`.
        pub const query_info_finish = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.FileInfo {
                return gobject.ext.as(FileOutputStream.Class, p_class).f_query_info_finish.?(gobject.ext.as(FileOutputStream, p_stream), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?*gio.FileInfo) void {
                gobject.ext.as(FileOutputStream.Class, p_class).f_query_info_finish = @ptrCast(p_implementation);
            }
        };

        pub const seek = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_offset: i64, p_type: glib.SeekType, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(FileOutputStream.Class, p_class).f_seek.?(gobject.ext.as(FileOutputStream, p_stream), p_offset, p_type, p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_offset: i64, p_type: glib.SeekType, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(FileOutputStream.Class, p_class).f_seek = @ptrCast(p_implementation);
            }
        };

        pub const tell = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) i64 {
                return gobject.ext.as(FileOutputStream.Class, p_class).f_tell.?(gobject.ext.as(FileOutputStream, p_stream));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) i64) void {
                gobject.ext.as(FileOutputStream.Class, p_class).f_tell = @ptrCast(p_implementation);
            }
        };

        pub const truncate_fn = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_size: i64, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(FileOutputStream.Class, p_class).f_truncate_fn.?(gobject.ext.as(FileOutputStream, p_stream), p_size, p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_size: i64, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(FileOutputStream.Class, p_class).f_truncate_fn = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {};

    /// Gets the entity tag for the file when it has been written.
    /// This must be called after the stream has been written
    /// and closed, as the etag can change while writing.
    extern fn g_file_output_stream_get_etag(p_stream: *FileOutputStream) ?[*:0]u8;
    pub const getEtag = g_file_output_stream_get_etag;

    /// Queries a file output stream for the given `attributes`.
    /// This function blocks while querying the stream. For the asynchronous
    /// version of this function, see `gio.FileOutputStream.queryInfoAsync`.
    /// While the stream is blocked, the stream will set the pending flag
    /// internally, and any other operations on the stream will fail with
    /// `G_IO_ERROR_PENDING`.
    ///
    /// Can fail if the stream was already closed (with `error` being set to
    /// `G_IO_ERROR_CLOSED`), the stream has pending operations (with `error` being
    /// set to `G_IO_ERROR_PENDING`), or if querying info is not supported for
    /// the stream's interface (with `error` being set to `G_IO_ERROR_NOT_SUPPORTED`). In
    /// all cases of failure, `NULL` will be returned.
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be set, and `NULL` will
    /// be returned.
    extern fn g_file_output_stream_query_info(p_stream: *FileOutputStream, p_attributes: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.FileInfo;
    pub const queryInfo = g_file_output_stream_query_info;

    /// Asynchronously queries the `stream` for a `gio.FileInfo`. When completed,
    /// `callback` will be called with a `gio.AsyncResult` which can be used to
    /// finish the operation with `gio.FileOutputStream.queryInfoFinish`.
    ///
    /// For the synchronous version of this function, see
    /// `gio.FileOutputStream.queryInfo`.
    extern fn g_file_output_stream_query_info_async(p_stream: *FileOutputStream, p_attributes: [*:0]const u8, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const queryInfoAsync = g_file_output_stream_query_info_async;

    /// Finalizes the asynchronous query started
    /// by `gio.FileOutputStream.queryInfoAsync`.
    extern fn g_file_output_stream_query_info_finish(p_stream: *FileOutputStream, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.FileInfo;
    pub const queryInfoFinish = g_file_output_stream_query_info_finish;

    extern fn g_file_output_stream_get_type() usize;
    pub const getGObjectType = g_file_output_stream_get_type;

    extern fn g_object_ref(p_self: *gio.FileOutputStream) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.FileOutputStream) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FileOutputStream, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Completes partial file and directory names given a partial string by
/// looking in the file system for clues. Can return a list of possible
/// completion strings for widget implementations.
pub const FilenameCompleter = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.FilenameCompleterClass;
    pub const virtual_methods = struct {
        pub const got_completion_data = struct {
            pub fn call(p_class: anytype, p_filename_completer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(FilenameCompleter.Class, p_class).f_got_completion_data.?(gobject.ext.as(FilenameCompleter, p_filename_completer));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_filename_completer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(FilenameCompleter.Class, p_class).f_got_completion_data = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {
        /// Emitted when the file name completion information comes available.
        pub const got_completion_data = struct {
            pub const name = "got-completion-data";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(FilenameCompleter, p_instance))),
                    gobject.signalLookup("got-completion-data", FilenameCompleter.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new filename completer.
    extern fn g_filename_completer_new() *gio.FilenameCompleter;
    pub const new = g_filename_completer_new;

    /// Obtains a completion for `initial_text` from `completer`.
    extern fn g_filename_completer_get_completion_suffix(p_completer: *FilenameCompleter, p_initial_text: [*:0]const u8) ?[*:0]u8;
    pub const getCompletionSuffix = g_filename_completer_get_completion_suffix;

    /// Gets an array of completion strings for a given initial text.
    extern fn g_filename_completer_get_completions(p_completer: *FilenameCompleter, p_initial_text: [*:0]const u8) [*][*:0]u8;
    pub const getCompletions = g_filename_completer_get_completions;

    /// If `dirs_only` is `TRUE`, `completer` will only
    /// complete directory names, and not file names.
    extern fn g_filename_completer_set_dirs_only(p_completer: *FilenameCompleter, p_dirs_only: c_int) void;
    pub const setDirsOnly = g_filename_completer_set_dirs_only;

    extern fn g_filename_completer_get_type() usize;
    pub const getGObjectType = g_filename_completer_get_type;

    extern fn g_object_ref(p_self: *gio.FilenameCompleter) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.FilenameCompleter) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FilenameCompleter, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Base class for input stream implementations that perform some
/// kind of filtering operation on a base stream. Typical examples
/// of filtering operations are character set conversion, compression
/// and byte order flipping.
pub const FilterInputStream = extern struct {
    pub const Parent = gio.InputStream;
    pub const Implements = [_]type{};
    pub const Class = gio.FilterInputStreamClass;
    f_parent_instance: gio.InputStream,
    f_base_stream: ?*gio.InputStream,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The underlying base stream on which the I/O ops will be done.
        pub const base_stream = struct {
            pub const name = "base-stream";

            pub const Type = ?*gio.InputStream;
        };

        /// Whether the base stream should be closed when the filter stream is closed.
        pub const close_base_stream = struct {
            pub const name = "close-base-stream";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Gets the base stream for the filter stream.
    extern fn g_filter_input_stream_get_base_stream(p_stream: *FilterInputStream) *gio.InputStream;
    pub const getBaseStream = g_filter_input_stream_get_base_stream;

    /// Returns whether the base stream will be closed when `stream` is
    /// closed.
    extern fn g_filter_input_stream_get_close_base_stream(p_stream: *FilterInputStream) c_int;
    pub const getCloseBaseStream = g_filter_input_stream_get_close_base_stream;

    /// Sets whether the base stream will be closed when `stream` is closed.
    extern fn g_filter_input_stream_set_close_base_stream(p_stream: *FilterInputStream, p_close_base: c_int) void;
    pub const setCloseBaseStream = g_filter_input_stream_set_close_base_stream;

    extern fn g_filter_input_stream_get_type() usize;
    pub const getGObjectType = g_filter_input_stream_get_type;

    extern fn g_object_ref(p_self: *gio.FilterInputStream) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.FilterInputStream) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FilterInputStream, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Base class for output stream implementations that perform some
/// kind of filtering operation on a base stream. Typical examples
/// of filtering operations are character set conversion, compression
/// and byte order flipping.
pub const FilterOutputStream = extern struct {
    pub const Parent = gio.OutputStream;
    pub const Implements = [_]type{};
    pub const Class = gio.FilterOutputStreamClass;
    f_parent_instance: gio.OutputStream,
    f_base_stream: ?*gio.OutputStream,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const base_stream = struct {
            pub const name = "base-stream";

            pub const Type = ?*gio.OutputStream;
        };

        /// Whether the base stream should be closed when the filter stream is closed.
        pub const close_base_stream = struct {
            pub const name = "close-base-stream";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Gets the base stream for the filter stream.
    extern fn g_filter_output_stream_get_base_stream(p_stream: *FilterOutputStream) *gio.OutputStream;
    pub const getBaseStream = g_filter_output_stream_get_base_stream;

    /// Returns whether the base stream will be closed when `stream` is
    /// closed.
    extern fn g_filter_output_stream_get_close_base_stream(p_stream: *FilterOutputStream) c_int;
    pub const getCloseBaseStream = g_filter_output_stream_get_close_base_stream;

    /// Sets whether the base stream will be closed when `stream` is closed.
    extern fn g_filter_output_stream_set_close_base_stream(p_stream: *FilterOutputStream, p_close_base: c_int) void;
    pub const setCloseBaseStream = g_filter_output_stream_set_close_base_stream;

    extern fn g_filter_output_stream_get_type() usize;
    pub const getGObjectType = g_filter_output_stream_get_type;

    extern fn g_object_ref(p_self: *gio.FilterOutputStream) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.FilterOutputStream) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *FilterOutputStream, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Provides an interface and default functions for loading and unloading
/// modules. This is used internally to make GIO extensible, but can also
/// be used by others to implement module loading.
pub const IOModule = opaque {
    pub const Parent = gobject.TypeModule;
    pub const Implements = [_]type{gobject.TypePlugin};
    pub const Class = gio.IOModuleClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Optional API for GIO modules to implement.
    ///
    /// Should return a list of all the extension points that may be
    /// implemented in this module.
    ///
    /// This method will not be called in normal use, however it may be
    /// called when probing existing modules and recording which extension
    /// points that this model is used for. This means we won't have to
    /// load and initialize this module unless its needed.
    ///
    /// If this function is not implemented by the module the module will
    /// always be loaded, initialized and then unloaded on application
    /// startup so that it can register its extension points during init.
    ///
    /// Note that a module need not actually implement all the extension
    /// points that `gio.IOModule.query` returns, since the exact list of
    /// extension may depend on runtime issues. However all extension
    /// points actually implemented must be returned by `gio.IOModule.query`
    /// (if defined).
    ///
    /// When installing a module that implements `gio.IOModule.query` you must
    /// run gio-querymodules in order to build the cache files required for
    /// lazy loading.
    ///
    /// Since 2.56, this function should be named `g_io_<modulename>_query`, where
    /// `modulename` is the plugin’s filename with the `lib` or `libgio` prefix and
    /// everything after the first dot removed, and with `-` replaced with `_`
    /// throughout. For example, `libgiognutls-helper.so` becomes `gnutls_helper`.
    /// Using the new symbol names avoids name clashes when building modules
    /// statically. The old symbol names continue to be supported, but cannot be used
    /// for static builds.
    extern fn g_io_module_query() [*][*:0]u8;
    pub const query = g_io_module_query;

    /// Creates a new GIOModule that will load the specific
    /// shared library when in use.
    extern fn g_io_module_new(p_filename: [*:0]const u8) *gio.IOModule;
    pub const new = g_io_module_new;

    /// Required API for GIO modules to implement.
    ///
    /// This function is run after the module has been loaded into GIO,
    /// to initialize the module. Typically, this function will call
    /// `gio.ioExtensionPointImplement`.
    ///
    /// Since 2.56, this function should be named `g_io_<modulename>_load`, where
    /// `modulename` is the plugin’s filename with the `lib` or `libgio` prefix and
    /// everything after the first dot removed, and with `-` replaced with `_`
    /// throughout. For example, `libgiognutls-helper.so` becomes `gnutls_helper`.
    /// Using the new symbol names avoids name clashes when building modules
    /// statically. The old symbol names continue to be supported, but cannot be used
    /// for static builds.
    extern fn g_io_module_load(p_module: *IOModule) void;
    pub const load = g_io_module_load;

    /// Required API for GIO modules to implement.
    ///
    /// This function is run when the module is being unloaded from GIO,
    /// to finalize the module.
    ///
    /// Since 2.56, this function should be named `g_io_<modulename>_unload`, where
    /// `modulename` is the plugin’s filename with the `lib` or `libgio` prefix and
    /// everything after the first dot removed, and with `-` replaced with `_`
    /// throughout. For example, `libgiognutls-helper.so` becomes `gnutls_helper`.
    /// Using the new symbol names avoids name clashes when building modules
    /// statically. The old symbol names continue to be supported, but cannot be used
    /// for static builds.
    extern fn g_io_module_unload(p_module: *IOModule) void;
    pub const unload = g_io_module_unload;

    extern fn g_io_module_get_type() usize;
    pub const getGObjectType = g_io_module_get_type;

    extern fn g_object_ref(p_self: *gio.IOModule) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.IOModule) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *IOModule, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GIOStream` represents an object that has both read and write streams.
/// Generally the two streams act as separate input and output streams,
/// but they share some common resources and state. For instance, for
/// seekable streams, both streams may use the same position.
///
/// Examples of `GIOStream` objects are `gio.SocketConnection`, which represents
/// a two-way network connection; and `gio.FileIOStream`, which represents a
/// file handle opened in read-write mode.
///
/// To do the actual reading and writing you need to get the substreams
/// with `gio.IOStream.getInputStream` and
/// `gio.IOStream.getOutputStream`.
///
/// The `GIOStream` object owns the input and the output streams, not the other
/// way around, so keeping the substreams alive will not keep the `GIOStream`
/// object alive. If the `GIOStream` object is freed it will be closed, thus
/// closing the substreams, so even if the substreams stay alive they will
/// always return `G_IO_ERROR_CLOSED` for all operations.
///
/// To close a stream use `gio.IOStream.close` which will close the common
/// stream object and also the individual substreams. You can also close
/// the substreams themselves. In most cases this only marks the
/// substream as closed, so further I/O on it fails but common state in the
/// `GIOStream` may still be open. However, some streams may support
/// ‘half-closed’ states where one direction of the stream is actually shut down.
///
/// Operations on `GIOStream`s cannot be started while another operation on the
/// `GIOStream` or its substreams is in progress. Specifically, an application can
/// read from the `gio.InputStream` and write to the
/// `gio.OutputStream` simultaneously (either in separate threads, or as
/// asynchronous operations in the same thread), but an application cannot start
/// any `GIOStream` operation while there is a `GIOStream`, `GInputStream` or
/// `GOutputStream` operation in progress, and an application can’t start any
/// `GInputStream` or `GOutputStream` operation while there is a `GIOStream`
/// operation in progress.
///
/// This is a product of individual stream operations being associated with a
/// given `glib.MainContext` (the thread-default context at the time the
/// operation was started), rather than entire streams being associated with a
/// single `GMainContext`.
///
/// GIO may run operations on `GIOStream`s from other (worker) threads, and this
/// may be exposed to application code in the behaviour of wrapper streams, such
/// as `gio.BufferedInputStream` or `gio.TlsConnection`. With such
/// wrapper APIs, application code may only run operations on the base (wrapped)
/// stream when the wrapper stream is idle. Note that the semantics of such
/// operations may not be well-defined due to the state the wrapper stream leaves
/// the base stream in (though they are guaranteed not to crash).
pub const IOStream = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.IOStreamClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.IOStreamPrivate,

    pub const virtual_methods = struct {
        /// Requests an asynchronous close of the stream, releasing resources
        /// related to it. When the operation is finished `callback` will be
        /// called. You can then call `gio.IOStream.closeFinish` to get
        /// the result of the operation.
        ///
        /// For behaviour details see `gio.IOStream.close`.
        ///
        /// The asynchronous methods have a default fallback that uses threads
        /// to implement asynchronicity, so they are optional for inheriting
        /// classes. However, if you override one you must override all.
        pub const close_async = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(IOStream.Class, p_class).f_close_async.?(gobject.ext.as(IOStream, p_stream), p_io_priority, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(IOStream.Class, p_class).f_close_async = @ptrCast(p_implementation);
            }
        };

        /// Closes a stream.
        pub const close_finish = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(IOStream.Class, p_class).f_close_finish.?(gobject.ext.as(IOStream, p_stream), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(IOStream.Class, p_class).f_close_finish = @ptrCast(p_implementation);
            }
        };

        pub const close_fn = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(IOStream.Class, p_class).f_close_fn.?(gobject.ext.as(IOStream, p_stream), p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(IOStream.Class, p_class).f_close_fn = @ptrCast(p_implementation);
            }
        };

        /// Gets the input stream for this object. This is used
        /// for reading.
        pub const get_input_stream = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) *gio.InputStream {
                return gobject.ext.as(IOStream.Class, p_class).f_get_input_stream.?(gobject.ext.as(IOStream, p_stream));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) *gio.InputStream) void {
                gobject.ext.as(IOStream.Class, p_class).f_get_input_stream = @ptrCast(p_implementation);
            }
        };

        /// Gets the output stream for this object. This is used for
        /// writing.
        pub const get_output_stream = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) *gio.OutputStream {
                return gobject.ext.as(IOStream.Class, p_class).f_get_output_stream.?(gobject.ext.as(IOStream, p_stream));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) *gio.OutputStream) void {
                gobject.ext.as(IOStream.Class, p_class).f_get_output_stream = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// Whether the stream is closed.
        pub const closed = struct {
            pub const name = "closed";

            pub const Type = c_int;
        };

        /// The `gio.InputStream` to read from.
        pub const input_stream = struct {
            pub const name = "input-stream";

            pub const Type = ?*gio.InputStream;
        };

        /// The `gio.OutputStream` to write to.
        pub const output_stream = struct {
            pub const name = "output-stream";

            pub const Type = ?*gio.OutputStream;
        };
    };

    pub const signals = struct {};

    /// Finishes an asynchronous io stream splice operation.
    extern fn g_io_stream_splice_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const spliceFinish = g_io_stream_splice_finish;

    /// Clears the pending flag on `stream`.
    extern fn g_io_stream_clear_pending(p_stream: *IOStream) void;
    pub const clearPending = g_io_stream_clear_pending;

    /// Closes the stream, releasing resources related to it. This will also
    /// close the individual input and output streams, if they are not already
    /// closed.
    ///
    /// Once the stream is closed, all other operations will return
    /// `G_IO_ERROR_CLOSED`. Closing a stream multiple times will not
    /// return an error.
    ///
    /// Closing a stream will automatically flush any outstanding buffers
    /// in the stream.
    ///
    /// Streams will be automatically closed when the last reference
    /// is dropped, but you might want to call this function to make sure
    /// resources are released as early as possible.
    ///
    /// Some streams might keep the backing store of the stream (e.g. a file
    /// descriptor) open after the stream is closed. See the documentation for
    /// the individual stream for details.
    ///
    /// On failure the first error that happened will be reported, but the
    /// close operation will finish as much as possible. A stream that failed
    /// to close will still return `G_IO_ERROR_CLOSED` for all operations.
    /// Still, it is important to check and report the error to the user,
    /// otherwise there might be a loss of data as all data might not be written.
    ///
    /// If `cancellable` is not NULL, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned.
    /// Cancelling a close will still leave the stream closed, but some streams
    /// can use a faster close that doesn't block to e.g. check errors.
    ///
    /// The default implementation of this method just calls close on the
    /// individual input/output streams.
    extern fn g_io_stream_close(p_stream: *IOStream, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const close = g_io_stream_close;

    /// Requests an asynchronous close of the stream, releasing resources
    /// related to it. When the operation is finished `callback` will be
    /// called. You can then call `gio.IOStream.closeFinish` to get
    /// the result of the operation.
    ///
    /// For behaviour details see `gio.IOStream.close`.
    ///
    /// The asynchronous methods have a default fallback that uses threads
    /// to implement asynchronicity, so they are optional for inheriting
    /// classes. However, if you override one you must override all.
    extern fn g_io_stream_close_async(p_stream: *IOStream, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const closeAsync = g_io_stream_close_async;

    /// Closes a stream.
    extern fn g_io_stream_close_finish(p_stream: *IOStream, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const closeFinish = g_io_stream_close_finish;

    /// Gets the input stream for this object. This is used
    /// for reading.
    extern fn g_io_stream_get_input_stream(p_stream: *IOStream) *gio.InputStream;
    pub const getInputStream = g_io_stream_get_input_stream;

    /// Gets the output stream for this object. This is used for
    /// writing.
    extern fn g_io_stream_get_output_stream(p_stream: *IOStream) *gio.OutputStream;
    pub const getOutputStream = g_io_stream_get_output_stream;

    /// Checks if a stream has pending actions.
    extern fn g_io_stream_has_pending(p_stream: *IOStream) c_int;
    pub const hasPending = g_io_stream_has_pending;

    /// Checks if a stream is closed.
    extern fn g_io_stream_is_closed(p_stream: *IOStream) c_int;
    pub const isClosed = g_io_stream_is_closed;

    /// Sets `stream` to have actions pending. If the pending flag is
    /// already set or `stream` is closed, it will return `FALSE` and set
    /// `error`.
    extern fn g_io_stream_set_pending(p_stream: *IOStream, p_error: ?*?*glib.Error) c_int;
    pub const setPending = g_io_stream_set_pending;

    /// Asynchronously splice the output stream of `stream1` to the input stream of
    /// `stream2`, and splice the output stream of `stream2` to the input stream of
    /// `stream1`.
    ///
    /// When the operation is finished `callback` will be called.
    /// You can then call `gio.IOStream.spliceFinish` to get the
    /// result of the operation.
    extern fn g_io_stream_splice_async(p_stream1: *IOStream, p_stream2: *gio.IOStream, p_flags: gio.IOStreamSpliceFlags, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const spliceAsync = g_io_stream_splice_async;

    extern fn g_io_stream_get_type() usize;
    pub const getGObjectType = g_io_stream_get_type;

    extern fn g_object_ref(p_self: *gio.IOStream) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.IOStream) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *IOStream, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GInetAddress` represents an IPv4 or IPv6 internet address. Use
/// `gio.Resolver.lookupByName` or
/// `gio.Resolver.lookupByNameAsync` to look up the `GInetAddress` for
/// a hostname. Use `gio.Resolver.lookupByAddress` or
/// `gio.Resolver.lookupByAddressAsync` to look up the hostname for a
/// `GInetAddress`.
///
/// To actually connect to a remote host, you will need a
/// `gio.InetSocketAddress` (which includes a `GInetAddress` as well as a
/// port number).
pub const InetAddress = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.InetAddressClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.InetAddressPrivate,

    pub const virtual_methods = struct {
        /// Gets the raw binary address data from `address`.
        pub const to_bytes = struct {
            pub fn call(p_class: anytype, p_address: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) *const u8 {
                return gobject.ext.as(InetAddress.Class, p_class).f_to_bytes.?(gobject.ext.as(InetAddress, p_address));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_address: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) *const u8) void {
                gobject.ext.as(InetAddress.Class, p_class).f_to_bytes = @ptrCast(p_implementation);
            }
        };

        /// Converts `address` to string form.
        pub const to_string = struct {
            pub fn call(p_class: anytype, p_address: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) [*:0]u8 {
                return gobject.ext.as(InetAddress.Class, p_class).f_to_string.?(gobject.ext.as(InetAddress, p_address));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_address: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) [*:0]u8) void {
                gobject.ext.as(InetAddress.Class, p_class).f_to_string = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The raw address data.
        pub const bytes = struct {
            pub const name = "bytes";

            pub const Type = ?*anyopaque;
        };

        /// The address family (IPv4 or IPv6).
        pub const family = struct {
            pub const name = "family";

            pub const Type = gio.SocketFamily;
        };

        /// Whether this is the "any" address for its family.
        /// See `gio.InetAddress.getIsAny`.
        pub const is_any = struct {
            pub const name = "is-any";

            pub const Type = c_int;
        };

        /// Whether this is a link-local address.
        /// See `gio.InetAddress.getIsLinkLocal`.
        pub const is_link_local = struct {
            pub const name = "is-link-local";

            pub const Type = c_int;
        };

        /// Whether this is the loopback address for its family.
        /// See `gio.InetAddress.getIsLoopback`.
        pub const is_loopback = struct {
            pub const name = "is-loopback";

            pub const Type = c_int;
        };

        /// Whether this is a global multicast address.
        /// See `gio.InetAddress.getIsMcGlobal`.
        pub const is_mc_global = struct {
            pub const name = "is-mc-global";

            pub const Type = c_int;
        };

        /// Whether this is a link-local multicast address.
        /// See `gio.InetAddress.getIsMcLinkLocal`.
        pub const is_mc_link_local = struct {
            pub const name = "is-mc-link-local";

            pub const Type = c_int;
        };

        /// Whether this is a node-local multicast address.
        /// See `gio.InetAddress.getIsMcNodeLocal`.
        pub const is_mc_node_local = struct {
            pub const name = "is-mc-node-local";

            pub const Type = c_int;
        };

        /// Whether this is an organization-local multicast address.
        /// See `gio.InetAddress.getIsMcOrgLocal`.
        pub const is_mc_org_local = struct {
            pub const name = "is-mc-org-local";

            pub const Type = c_int;
        };

        /// Whether this is a site-local multicast address.
        /// See `gio.InetAddress.getIsMcSiteLocal`.
        pub const is_mc_site_local = struct {
            pub const name = "is-mc-site-local";

            pub const Type = c_int;
        };

        /// Whether this is a multicast address.
        /// See `gio.InetAddress.getIsMulticast`.
        pub const is_multicast = struct {
            pub const name = "is-multicast";

            pub const Type = c_int;
        };

        /// Whether this is a site-local address.
        /// See `gio.InetAddress.getIsLoopback`.
        pub const is_site_local = struct {
            pub const name = "is-site-local";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a `gio.InetAddress` for the "any" address (unassigned/"don't
    /// care") for `family`.
    extern fn g_inet_address_new_any(p_family: gio.SocketFamily) *gio.InetAddress;
    pub const newAny = g_inet_address_new_any;

    /// Creates a new `gio.InetAddress` from the given `family` and `bytes`.
    /// `bytes` should be 4 bytes for `G_SOCKET_FAMILY_IPV4` and 16 bytes for
    /// `G_SOCKET_FAMILY_IPV6`.
    extern fn g_inet_address_new_from_bytes(p_bytes: [*]const u8, p_family: gio.SocketFamily) *gio.InetAddress;
    pub const newFromBytes = g_inet_address_new_from_bytes;

    /// Parses `string` as an IP address and creates a new `gio.InetAddress`.
    extern fn g_inet_address_new_from_string(p_string: [*:0]const u8) ?*gio.InetAddress;
    pub const newFromString = g_inet_address_new_from_string;

    /// Creates a `gio.InetAddress` for the loopback address for `family`.
    extern fn g_inet_address_new_loopback(p_family: gio.SocketFamily) *gio.InetAddress;
    pub const newLoopback = g_inet_address_new_loopback;

    /// Checks if two `gio.InetAddress` instances are equal, e.g. the same address.
    extern fn g_inet_address_equal(p_address: *InetAddress, p_other_address: *gio.InetAddress) c_int;
    pub const equal = g_inet_address_equal;

    /// Gets `address`'s family
    extern fn g_inet_address_get_family(p_address: *InetAddress) gio.SocketFamily;
    pub const getFamily = g_inet_address_get_family;

    /// Tests whether `address` is the "any" address for its family.
    extern fn g_inet_address_get_is_any(p_address: *InetAddress) c_int;
    pub const getIsAny = g_inet_address_get_is_any;

    /// Tests whether `address` is a link-local address (that is, if it
    /// identifies a host on a local network that is not connected to the
    /// Internet).
    extern fn g_inet_address_get_is_link_local(p_address: *InetAddress) c_int;
    pub const getIsLinkLocal = g_inet_address_get_is_link_local;

    /// Tests whether `address` is the loopback address for its family.
    extern fn g_inet_address_get_is_loopback(p_address: *InetAddress) c_int;
    pub const getIsLoopback = g_inet_address_get_is_loopback;

    /// Tests whether `address` is a global multicast address.
    extern fn g_inet_address_get_is_mc_global(p_address: *InetAddress) c_int;
    pub const getIsMcGlobal = g_inet_address_get_is_mc_global;

    /// Tests whether `address` is a link-local multicast address.
    extern fn g_inet_address_get_is_mc_link_local(p_address: *InetAddress) c_int;
    pub const getIsMcLinkLocal = g_inet_address_get_is_mc_link_local;

    /// Tests whether `address` is a node-local multicast address.
    extern fn g_inet_address_get_is_mc_node_local(p_address: *InetAddress) c_int;
    pub const getIsMcNodeLocal = g_inet_address_get_is_mc_node_local;

    /// Tests whether `address` is an organization-local multicast address.
    extern fn g_inet_address_get_is_mc_org_local(p_address: *InetAddress) c_int;
    pub const getIsMcOrgLocal = g_inet_address_get_is_mc_org_local;

    /// Tests whether `address` is a site-local multicast address.
    extern fn g_inet_address_get_is_mc_site_local(p_address: *InetAddress) c_int;
    pub const getIsMcSiteLocal = g_inet_address_get_is_mc_site_local;

    /// Tests whether `address` is a multicast address.
    extern fn g_inet_address_get_is_multicast(p_address: *InetAddress) c_int;
    pub const getIsMulticast = g_inet_address_get_is_multicast;

    /// Tests whether `address` is a site-local address such as 10.0.0.1
    /// (that is, the address identifies a host on a local network that can
    /// not be reached directly from the Internet, but which may have
    /// outgoing Internet connectivity via a NAT or firewall).
    extern fn g_inet_address_get_is_site_local(p_address: *InetAddress) c_int;
    pub const getIsSiteLocal = g_inet_address_get_is_site_local;

    /// Gets the size of the native raw binary address for `address`. This
    /// is the size of the data that you get from `gio.InetAddress.toBytes`.
    extern fn g_inet_address_get_native_size(p_address: *InetAddress) usize;
    pub const getNativeSize = g_inet_address_get_native_size;

    /// Gets the raw binary address data from `address`.
    extern fn g_inet_address_to_bytes(p_address: *InetAddress) *const u8;
    pub const toBytes = g_inet_address_to_bytes;

    /// Converts `address` to string form.
    extern fn g_inet_address_to_string(p_address: *InetAddress) [*:0]u8;
    pub const toString = g_inet_address_to_string;

    extern fn g_inet_address_get_type() usize;
    pub const getGObjectType = g_inet_address_get_type;

    extern fn g_object_ref(p_self: *gio.InetAddress) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.InetAddress) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *InetAddress, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GInetAddressMask` represents a range of IPv4 or IPv6 addresses
/// described by a base address and a length indicating how many bits
/// of the base address are relevant for matching purposes. These are
/// often given in string form. For example, `10.0.0.0/8`, or `fe80::/10`.
pub const InetAddressMask = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gio.Initable};
    pub const Class = gio.InetAddressMaskClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.InetAddressMaskPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The base address.
        pub const address = struct {
            pub const name = "address";

            pub const Type = ?*gio.InetAddress;
        };

        /// The address family (IPv4 or IPv6).
        pub const family = struct {
            pub const name = "family";

            pub const Type = gio.SocketFamily;
        };

        /// The prefix length, in bytes.
        pub const length = struct {
            pub const name = "length";

            pub const Type = c_uint;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gio.InetAddressMask` representing all addresses whose
    /// first `length` bits match `addr`.
    extern fn g_inet_address_mask_new(p_addr: *gio.InetAddress, p_length: c_uint, p_error: ?*?*glib.Error) ?*gio.InetAddressMask;
    pub const new = g_inet_address_mask_new;

    /// Parses `mask_string` as an IP address and (optional) length, and
    /// creates a new `gio.InetAddressMask`. The length, if present, is
    /// delimited by a "/". If it is not present, then the length is
    /// assumed to be the full length of the address.
    extern fn g_inet_address_mask_new_from_string(p_mask_string: [*:0]const u8, p_error: ?*?*glib.Error) ?*gio.InetAddressMask;
    pub const newFromString = g_inet_address_mask_new_from_string;

    /// Tests if `mask` and `mask2` are the same mask.
    extern fn g_inet_address_mask_equal(p_mask: *InetAddressMask, p_mask2: *gio.InetAddressMask) c_int;
    pub const equal = g_inet_address_mask_equal;

    /// Gets `mask`'s base address
    extern fn g_inet_address_mask_get_address(p_mask: *InetAddressMask) *gio.InetAddress;
    pub const getAddress = g_inet_address_mask_get_address;

    /// Gets the `gio.SocketFamily` of `mask`'s address
    extern fn g_inet_address_mask_get_family(p_mask: *InetAddressMask) gio.SocketFamily;
    pub const getFamily = g_inet_address_mask_get_family;

    /// Gets `mask`'s length
    extern fn g_inet_address_mask_get_length(p_mask: *InetAddressMask) c_uint;
    pub const getLength = g_inet_address_mask_get_length;

    /// Tests if `address` falls within the range described by `mask`.
    extern fn g_inet_address_mask_matches(p_mask: *InetAddressMask, p_address: *gio.InetAddress) c_int;
    pub const matches = g_inet_address_mask_matches;

    /// Converts `mask` back to its corresponding string form.
    extern fn g_inet_address_mask_to_string(p_mask: *InetAddressMask) [*:0]u8;
    pub const toString = g_inet_address_mask_to_string;

    extern fn g_inet_address_mask_get_type() usize;
    pub const getGObjectType = g_inet_address_mask_get_type;

    extern fn g_object_ref(p_self: *gio.InetAddressMask) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.InetAddressMask) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *InetAddressMask, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An IPv4 or IPv6 socket address. That is, the combination of a
/// `gio.InetAddress` and a port number.
///
/// In UNIX terms, `GInetSocketAddress` corresponds to a
/// [`struct sockaddr_in` or `struct sockaddr_in6`](man:sockaddr(3type)).
pub const InetSocketAddress = extern struct {
    pub const Parent = gio.SocketAddress;
    pub const Implements = [_]type{gio.SocketConnectable};
    pub const Class = gio.InetSocketAddressClass;
    f_parent_instance: gio.SocketAddress,
    f_priv: ?*gio.InetSocketAddressPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The address.
        pub const address = struct {
            pub const name = "address";

            pub const Type = ?*gio.InetAddress;
        };

        /// The `sin6_flowinfo` field, for IPv6 addresses.
        pub const flowinfo = struct {
            pub const name = "flowinfo";

            pub const Type = c_uint;
        };

        /// The port.
        pub const port = struct {
            pub const name = "port";

            pub const Type = c_uint;
        };

        /// The `sin6_scope_id` field, for IPv6 addresses.
        pub const scope_id = struct {
            pub const name = "scope-id";

            pub const Type = c_uint;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gio.InetSocketAddress` for `address` and `port`.
    extern fn g_inet_socket_address_new(p_address: *gio.InetAddress, p_port: u16) *gio.InetSocketAddress;
    pub const new = g_inet_socket_address_new;

    /// Creates a new `gio.InetSocketAddress` for `address` and `port`.
    ///
    /// If `address` is an IPv6 address, it can also contain a scope ID
    /// (separated from the address by a `%`).
    extern fn g_inet_socket_address_new_from_string(p_address: [*:0]const u8, p_port: c_uint) ?*gio.InetSocketAddress;
    pub const newFromString = g_inet_socket_address_new_from_string;

    /// Gets `address`'s `gio.InetAddress`.
    extern fn g_inet_socket_address_get_address(p_address: *InetSocketAddress) *gio.InetAddress;
    pub const getAddress = g_inet_socket_address_get_address;

    /// Gets the `sin6_flowinfo` field from `address`,
    /// which must be an IPv6 address.
    extern fn g_inet_socket_address_get_flowinfo(p_address: *InetSocketAddress) u32;
    pub const getFlowinfo = g_inet_socket_address_get_flowinfo;

    /// Gets `address`'s port.
    extern fn g_inet_socket_address_get_port(p_address: *InetSocketAddress) u16;
    pub const getPort = g_inet_socket_address_get_port;

    /// Gets the `sin6_scope_id` field from `address`,
    /// which must be an IPv6 address.
    extern fn g_inet_socket_address_get_scope_id(p_address: *InetSocketAddress) u32;
    pub const getScopeId = g_inet_socket_address_get_scope_id;

    extern fn g_inet_socket_address_get_type() usize;
    pub const getGObjectType = g_inet_socket_address_get_type;

    extern fn g_object_ref(p_self: *gio.InetSocketAddress) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.InetSocketAddress) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *InetSocketAddress, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GInputStream` is a base class for implementing streaming input.
///
/// It has functions to read from a stream (`gio.InputStream.read`),
/// to close a stream (`gio.InputStream.close`) and to skip some content
/// (`gio.InputStream.skip`).
///
/// To copy the content of an input stream to an output stream without
/// manually handling the reads and writes, use `gio.OutputStream.splice`.
///
/// See the documentation for `gio.IOStream` for details of thread safety
/// of streaming APIs.
///
/// All of these functions have async variants too.
pub const InputStream = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.InputStreamClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.InputStreamPrivate,

    pub const virtual_methods = struct {
        /// Requests an asynchronous closes of the stream, releasing resources related to it.
        /// When the operation is finished `callback` will be called.
        /// You can then call `gio.InputStream.closeFinish` to get the result of the
        /// operation.
        ///
        /// For behaviour details see `gio.InputStream.close`.
        ///
        /// The asynchronous methods have a default fallback that uses threads to implement
        /// asynchronicity, so they are optional for inheriting classes. However, if you
        /// override one you must override all.
        pub const close_async = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(InputStream.Class, p_class).f_close_async.?(gobject.ext.as(InputStream, p_stream), p_io_priority, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(InputStream.Class, p_class).f_close_async = @ptrCast(p_implementation);
            }
        };

        /// Finishes closing a stream asynchronously, started from `gio.InputStream.closeAsync`.
        pub const close_finish = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(InputStream.Class, p_class).f_close_finish.?(gobject.ext.as(InputStream, p_stream), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(InputStream.Class, p_class).f_close_finish = @ptrCast(p_implementation);
            }
        };

        pub const close_fn = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(InputStream.Class, p_class).f_close_fn.?(gobject.ext.as(InputStream, p_stream), p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(InputStream.Class, p_class).f_close_fn = @ptrCast(p_implementation);
            }
        };

        /// Request an asynchronous read of `count` bytes from the stream into the buffer
        /// starting at `buffer`. When the operation is finished `callback` will be called.
        /// You can then call `gio.InputStream.readFinish` to get the result of the
        /// operation.
        ///
        /// During an async request no other sync and async calls are allowed on `stream`, and will
        /// result in `G_IO_ERROR_PENDING` errors.
        ///
        /// A value of `count` larger than `G_MAXSSIZE` will cause a `G_IO_ERROR_INVALID_ARGUMENT` error.
        ///
        /// On success, the number of bytes read into the buffer will be passed to the
        /// callback. It is not an error if this is not the same as the requested size, as it
        /// can happen e.g. near the end of a file, but generally we try to read
        /// as many bytes as requested. Zero is returned on end of file
        /// (or if `count` is zero),  but never otherwise.
        ///
        /// Any outstanding i/o request with higher priority (lower numerical value) will
        /// be executed before an outstanding request with lower priority. Default
        /// priority is `G_PRIORITY_DEFAULT`.
        ///
        /// The asynchronous methods have a default fallback that uses threads to implement
        /// asynchronicity, so they are optional for inheriting classes. However, if you
        /// override one you must override all.
        pub const read_async = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_buffer: ?[*]u8, p_count: usize, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(InputStream.Class, p_class).f_read_async.?(gobject.ext.as(InputStream, p_stream), p_buffer, p_count, p_io_priority, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_buffer: ?[*]u8, p_count: usize, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(InputStream.Class, p_class).f_read_async = @ptrCast(p_implementation);
            }
        };

        /// Finishes an asynchronous stream read operation.
        pub const read_finish = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) isize {
                return gobject.ext.as(InputStream.Class, p_class).f_read_finish.?(gobject.ext.as(InputStream, p_stream), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) isize) void {
                gobject.ext.as(InputStream.Class, p_class).f_read_finish = @ptrCast(p_implementation);
            }
        };

        pub const read_fn = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_buffer: ?*anyopaque, p_count: usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) isize {
                return gobject.ext.as(InputStream.Class, p_class).f_read_fn.?(gobject.ext.as(InputStream, p_stream), p_buffer, p_count, p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_buffer: ?*anyopaque, p_count: usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) isize) void {
                gobject.ext.as(InputStream.Class, p_class).f_read_fn = @ptrCast(p_implementation);
            }
        };

        /// Tries to skip `count` bytes from the stream. Will block during the operation.
        ///
        /// This is identical to `gio.InputStream.read`, from a behaviour standpoint,
        /// but the bytes that are skipped are not returned to the user. Some
        /// streams have an implementation that is more efficient than reading the data.
        ///
        /// This function is optional for inherited classes, as the default implementation
        /// emulates it using read.
        ///
        /// If `cancellable` is not `NULL`, then the operation can be cancelled by
        /// triggering the cancellable object from another thread. If the operation
        /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned. If an
        /// operation was partially finished when the operation was cancelled the
        /// partial result will be returned, without an error.
        pub const skip = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_count: usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) isize {
                return gobject.ext.as(InputStream.Class, p_class).f_skip.?(gobject.ext.as(InputStream, p_stream), p_count, p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_count: usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) isize) void {
                gobject.ext.as(InputStream.Class, p_class).f_skip = @ptrCast(p_implementation);
            }
        };

        /// Request an asynchronous skip of `count` bytes from the stream.
        /// When the operation is finished `callback` will be called.
        /// You can then call `gio.InputStream.skipFinish` to get the result
        /// of the operation.
        ///
        /// During an async request no other sync and async calls are allowed,
        /// and will result in `G_IO_ERROR_PENDING` errors.
        ///
        /// A value of `count` larger than `G_MAXSSIZE` will cause a `G_IO_ERROR_INVALID_ARGUMENT` error.
        ///
        /// On success, the number of bytes skipped will be passed to the callback.
        /// It is not an error if this is not the same as the requested size, as it
        /// can happen e.g. near the end of a file, but generally we try to skip
        /// as many bytes as requested. Zero is returned on end of file
        /// (or if `count` is zero), but never otherwise.
        ///
        /// Any outstanding i/o request with higher priority (lower numerical value)
        /// will be executed before an outstanding request with lower priority.
        /// Default priority is `G_PRIORITY_DEFAULT`.
        ///
        /// The asynchronous methods have a default fallback that uses threads to
        /// implement asynchronicity, so they are optional for inheriting classes.
        /// However, if you override one, you must override all.
        pub const skip_async = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_count: usize, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(InputStream.Class, p_class).f_skip_async.?(gobject.ext.as(InputStream, p_stream), p_count, p_io_priority, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_count: usize, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(InputStream.Class, p_class).f_skip_async = @ptrCast(p_implementation);
            }
        };

        /// Finishes a stream skip operation.
        pub const skip_finish = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) isize {
                return gobject.ext.as(InputStream.Class, p_class).f_skip_finish.?(gobject.ext.as(InputStream, p_stream), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) isize) void {
                gobject.ext.as(InputStream.Class, p_class).f_skip_finish = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {};

    /// Clears the pending flag on `stream`.
    extern fn g_input_stream_clear_pending(p_stream: *InputStream) void;
    pub const clearPending = g_input_stream_clear_pending;

    /// Closes the stream, releasing resources related to it.
    ///
    /// Once the stream is closed, all other operations will return `G_IO_ERROR_CLOSED`.
    /// Closing a stream multiple times will not return an error.
    ///
    /// Streams will be automatically closed when the last reference
    /// is dropped, but you might want to call this function to make sure
    /// resources are released as early as possible.
    ///
    /// Some streams might keep the backing store of the stream (e.g. a file descriptor)
    /// open after the stream is closed. See the documentation for the individual
    /// stream for details.
    ///
    /// On failure the first error that happened will be reported, but the close
    /// operation will finish as much as possible. A stream that failed to
    /// close will still return `G_IO_ERROR_CLOSED` for all operations. Still, it
    /// is important to check and report the error to the user.
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned.
    /// Cancelling a close will still leave the stream closed, but some streams
    /// can use a faster close that doesn't block to e.g. check errors.
    extern fn g_input_stream_close(p_stream: *InputStream, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const close = g_input_stream_close;

    /// Requests an asynchronous closes of the stream, releasing resources related to it.
    /// When the operation is finished `callback` will be called.
    /// You can then call `gio.InputStream.closeFinish` to get the result of the
    /// operation.
    ///
    /// For behaviour details see `gio.InputStream.close`.
    ///
    /// The asynchronous methods have a default fallback that uses threads to implement
    /// asynchronicity, so they are optional for inheriting classes. However, if you
    /// override one you must override all.
    extern fn g_input_stream_close_async(p_stream: *InputStream, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const closeAsync = g_input_stream_close_async;

    /// Finishes closing a stream asynchronously, started from `gio.InputStream.closeAsync`.
    extern fn g_input_stream_close_finish(p_stream: *InputStream, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const closeFinish = g_input_stream_close_finish;

    /// Checks if an input stream has pending actions.
    extern fn g_input_stream_has_pending(p_stream: *InputStream) c_int;
    pub const hasPending = g_input_stream_has_pending;

    /// Checks if an input stream is closed.
    extern fn g_input_stream_is_closed(p_stream: *InputStream) c_int;
    pub const isClosed = g_input_stream_is_closed;

    /// Tries to read `count` bytes from the stream into the buffer starting at
    /// `buffer`. Will block during this read.
    ///
    /// If count is zero returns zero and does nothing. A value of `count`
    /// larger than `G_MAXSSIZE` will cause a `G_IO_ERROR_INVALID_ARGUMENT` error.
    ///
    /// On success, the number of bytes read into the buffer is returned.
    /// It is not an error if this is not the same as the requested size, as it
    /// can happen e.g. near the end of a file. Zero is returned on end of file
    /// (or if `count` is zero),  but never otherwise.
    ///
    /// The returned `buffer` is not a nul-terminated string, it can contain nul bytes
    /// at any position, and this function doesn't nul-terminate the `buffer`.
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned. If an
    /// operation was partially finished when the operation was cancelled the
    /// partial result will be returned, without an error.
    ///
    /// On error -1 is returned and `error` is set accordingly.
    extern fn g_input_stream_read(p_stream: *InputStream, p_buffer: [*]u8, p_count: usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) isize;
    pub const read = g_input_stream_read;

    /// Tries to read `count` bytes from the stream into the buffer starting at
    /// `buffer`. Will block during this read.
    ///
    /// This function is similar to `gio.InputStream.read`, except it tries to
    /// read as many bytes as requested, only stopping on an error or end of stream.
    ///
    /// On a successful read of `count` bytes, or if we reached the end of the
    /// stream,  `TRUE` is returned, and `bytes_read` is set to the number of bytes
    /// read into `buffer`.
    ///
    /// If there is an error during the operation `FALSE` is returned and `error`
    /// is set to indicate the error status.
    ///
    /// As a special exception to the normal conventions for functions that
    /// use `glib.Error`, if this function returns `FALSE` (and sets `error`) then
    /// `bytes_read` will be set to the number of bytes that were successfully
    /// read before the error was encountered.  This functionality is only
    /// available from C.  If you need it from another language then you must
    /// write your own loop around `gio.InputStream.read`.
    extern fn g_input_stream_read_all(p_stream: *InputStream, p_buffer: [*]u8, p_count: usize, p_bytes_read: *usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const readAll = g_input_stream_read_all;

    /// Request an asynchronous read of `count` bytes from the stream into the
    /// buffer starting at `buffer`.
    ///
    /// This is the asynchronous equivalent of `gio.InputStream.readAll`.
    ///
    /// Call `gio.InputStream.readAllFinish` to collect the result.
    ///
    /// Any outstanding I/O request with higher priority (lower numerical
    /// value) will be executed before an outstanding request with lower
    /// priority. Default priority is `G_PRIORITY_DEFAULT`.
    extern fn g_input_stream_read_all_async(p_stream: *InputStream, p_buffer: [*]u8, p_count: usize, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const readAllAsync = g_input_stream_read_all_async;

    /// Finishes an asynchronous stream read operation started with
    /// `gio.InputStream.readAllAsync`.
    ///
    /// As a special exception to the normal conventions for functions that
    /// use `glib.Error`, if this function returns `FALSE` (and sets `error`) then
    /// `bytes_read` will be set to the number of bytes that were successfully
    /// read before the error was encountered.  This functionality is only
    /// available from C.  If you need it from another language then you must
    /// write your own loop around `gio.InputStream.readAsync`.
    extern fn g_input_stream_read_all_finish(p_stream: *InputStream, p_result: *gio.AsyncResult, p_bytes_read: *usize, p_error: ?*?*glib.Error) c_int;
    pub const readAllFinish = g_input_stream_read_all_finish;

    /// Request an asynchronous read of `count` bytes from the stream into the buffer
    /// starting at `buffer`. When the operation is finished `callback` will be called.
    /// You can then call `gio.InputStream.readFinish` to get the result of the
    /// operation.
    ///
    /// During an async request no other sync and async calls are allowed on `stream`, and will
    /// result in `G_IO_ERROR_PENDING` errors.
    ///
    /// A value of `count` larger than `G_MAXSSIZE` will cause a `G_IO_ERROR_INVALID_ARGUMENT` error.
    ///
    /// On success, the number of bytes read into the buffer will be passed to the
    /// callback. It is not an error if this is not the same as the requested size, as it
    /// can happen e.g. near the end of a file, but generally we try to read
    /// as many bytes as requested. Zero is returned on end of file
    /// (or if `count` is zero),  but never otherwise.
    ///
    /// Any outstanding i/o request with higher priority (lower numerical value) will
    /// be executed before an outstanding request with lower priority. Default
    /// priority is `G_PRIORITY_DEFAULT`.
    ///
    /// The asynchronous methods have a default fallback that uses threads to implement
    /// asynchronicity, so they are optional for inheriting classes. However, if you
    /// override one you must override all.
    extern fn g_input_stream_read_async(p_stream: *InputStream, p_buffer: [*]u8, p_count: usize, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const readAsync = g_input_stream_read_async;

    /// Like `gio.InputStream.read`, this tries to read `count` bytes from
    /// the stream in a blocking fashion. However, rather than reading into
    /// a user-supplied buffer, this will create a new `glib.Bytes` containing
    /// the data that was read. This may be easier to use from language
    /// bindings.
    ///
    /// If count is zero, returns a zero-length `glib.Bytes` and does nothing. A
    /// value of `count` larger than `G_MAXSSIZE` will cause a
    /// `G_IO_ERROR_INVALID_ARGUMENT` error.
    ///
    /// On success, a new `glib.Bytes` is returned. It is not an error if the
    /// size of this object is not the same as the requested size, as it
    /// can happen e.g. near the end of a file. A zero-length `glib.Bytes` is
    /// returned on end of file (or if `count` is zero), but never
    /// otherwise.
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned. If an
    /// operation was partially finished when the operation was cancelled the
    /// partial result will be returned, without an error.
    ///
    /// On error `NULL` is returned and `error` is set accordingly.
    extern fn g_input_stream_read_bytes(p_stream: *InputStream, p_count: usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*glib.Bytes;
    pub const readBytes = g_input_stream_read_bytes;

    /// Request an asynchronous read of `count` bytes from the stream into a
    /// new `glib.Bytes`. When the operation is finished `callback` will be
    /// called. You can then call `gio.InputStream.readBytesFinish` to get the
    /// result of the operation.
    ///
    /// During an async request no other sync and async calls are allowed
    /// on `stream`, and will result in `G_IO_ERROR_PENDING` errors.
    ///
    /// A value of `count` larger than `G_MAXSSIZE` will cause a
    /// `G_IO_ERROR_INVALID_ARGUMENT` error.
    ///
    /// On success, the new `glib.Bytes` will be passed to the callback. It is
    /// not an error if this is smaller than the requested size, as it can
    /// happen e.g. near the end of a file, but generally we try to read as
    /// many bytes as requested. Zero is returned on end of file (or if
    /// `count` is zero), but never otherwise.
    ///
    /// Any outstanding I/O request with higher priority (lower numerical
    /// value) will be executed before an outstanding request with lower
    /// priority. Default priority is `G_PRIORITY_DEFAULT`.
    extern fn g_input_stream_read_bytes_async(p_stream: *InputStream, p_count: usize, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const readBytesAsync = g_input_stream_read_bytes_async;

    /// Finishes an asynchronous stream read-into-`glib.Bytes` operation.
    extern fn g_input_stream_read_bytes_finish(p_stream: *InputStream, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.Bytes;
    pub const readBytesFinish = g_input_stream_read_bytes_finish;

    /// Finishes an asynchronous stream read operation.
    extern fn g_input_stream_read_finish(p_stream: *InputStream, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) isize;
    pub const readFinish = g_input_stream_read_finish;

    /// Sets `stream` to have actions pending. If the pending flag is
    /// already set or `stream` is closed, it will return `FALSE` and set
    /// `error`.
    extern fn g_input_stream_set_pending(p_stream: *InputStream, p_error: ?*?*glib.Error) c_int;
    pub const setPending = g_input_stream_set_pending;

    /// Tries to skip `count` bytes from the stream. Will block during the operation.
    ///
    /// This is identical to `gio.InputStream.read`, from a behaviour standpoint,
    /// but the bytes that are skipped are not returned to the user. Some
    /// streams have an implementation that is more efficient than reading the data.
    ///
    /// This function is optional for inherited classes, as the default implementation
    /// emulates it using read.
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned. If an
    /// operation was partially finished when the operation was cancelled the
    /// partial result will be returned, without an error.
    extern fn g_input_stream_skip(p_stream: *InputStream, p_count: usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) isize;
    pub const skip = g_input_stream_skip;

    /// Request an asynchronous skip of `count` bytes from the stream.
    /// When the operation is finished `callback` will be called.
    /// You can then call `gio.InputStream.skipFinish` to get the result
    /// of the operation.
    ///
    /// During an async request no other sync and async calls are allowed,
    /// and will result in `G_IO_ERROR_PENDING` errors.
    ///
    /// A value of `count` larger than `G_MAXSSIZE` will cause a `G_IO_ERROR_INVALID_ARGUMENT` error.
    ///
    /// On success, the number of bytes skipped will be passed to the callback.
    /// It is not an error if this is not the same as the requested size, as it
    /// can happen e.g. near the end of a file, but generally we try to skip
    /// as many bytes as requested. Zero is returned on end of file
    /// (or if `count` is zero), but never otherwise.
    ///
    /// Any outstanding i/o request with higher priority (lower numerical value)
    /// will be executed before an outstanding request with lower priority.
    /// Default priority is `G_PRIORITY_DEFAULT`.
    ///
    /// The asynchronous methods have a default fallback that uses threads to
    /// implement asynchronicity, so they are optional for inheriting classes.
    /// However, if you override one, you must override all.
    extern fn g_input_stream_skip_async(p_stream: *InputStream, p_count: usize, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const skipAsync = g_input_stream_skip_async;

    /// Finishes a stream skip operation.
    extern fn g_input_stream_skip_finish(p_stream: *InputStream, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) isize;
    pub const skipFinish = g_input_stream_skip_finish;

    extern fn g_input_stream_get_type() usize;
    pub const getGObjectType = g_input_stream_get_type;

    extern fn g_object_ref(p_self: *gio.InputStream) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.InputStream) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *InputStream, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GListStore` is a simple implementation of `gio.ListModel` that stores
/// all items in memory.
///
/// It provides insertions, deletions, and lookups in logarithmic time
/// with a fast path for the common case of iterating the list linearly.
pub const ListStore = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gio.ListModel};
    pub const Class = gio.ListStoreClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The type of items contained in this list store. Items must be
        /// subclasses of `gobject.Object`.
        pub const item_type = struct {
            pub const name = "item-type";

            pub const Type = usize;
        };

        /// The number of items contained in this list store.
        pub const n_items = struct {
            pub const name = "n-items";

            pub const Type = c_uint;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gio.ListStore` with items of type `item_type`. `item_type`
    /// must be a subclass of `gobject.Object`.
    extern fn g_list_store_new(p_item_type: usize) *gio.ListStore;
    pub const new = g_list_store_new;

    /// Appends `item` to `store`. `item` must be of type `gio.ListStore.properties.item`-type.
    ///
    /// This function takes a ref on `item`.
    ///
    /// Use `gio.ListStore.splice` to append multiple items at the same time
    /// efficiently.
    extern fn g_list_store_append(p_store: *ListStore, p_item: *gobject.Object) void;
    pub const append = g_list_store_append;

    /// Looks up the given `item` in the list store by looping over the items until
    /// the first occurrence of `item`. If `item` was not found, then `position` will
    /// not be set, and this method will return `FALSE`.
    ///
    /// If you need to compare the two items with a custom comparison function, use
    /// `gio.ListStore.findWithEqualFunc` with a custom `glib.EqualFunc` instead.
    extern fn g_list_store_find(p_store: *ListStore, p_item: *gobject.Object, p_position: ?*c_uint) c_int;
    pub const find = g_list_store_find;

    /// Looks up the given `item` in the list store by looping over the items and
    /// comparing them with `equal_func` until the first occurrence of `item` which
    /// matches. If `item` was not found, then `position` will not be set, and this
    /// method will return `FALSE`.
    ///
    /// `item` is always passed as second parameter to `equal_func`.
    ///
    /// Since GLib 2.76 it is possible to pass `NULL` for `item`.
    extern fn g_list_store_find_with_equal_func(p_store: *ListStore, p_item: ?*gobject.Object, p_equal_func: glib.EqualFunc, p_position: ?*c_uint) c_int;
    pub const findWithEqualFunc = g_list_store_find_with_equal_func;

    /// Like `gio.ListStore.findWithEqualFunc` but with an additional `user_data`
    /// that is passed to `equal_func`.
    ///
    /// `item` is always passed as second parameter to `equal_func`.
    ///
    /// Since GLib 2.76 it is possible to pass `NULL` for `item`.
    extern fn g_list_store_find_with_equal_func_full(p_store: *ListStore, p_item: ?*gobject.Object, p_equal_func: glib.EqualFuncFull, p_user_data: ?*anyopaque, p_position: ?*c_uint) c_int;
    pub const findWithEqualFuncFull = g_list_store_find_with_equal_func_full;

    /// Inserts `item` into `store` at `position`. `item` must be of type
    /// `gio.ListStore.properties.item`-type or derived from it. `position` must be smaller
    /// than the length of the list, or equal to it to append.
    ///
    /// This function takes a ref on `item`.
    ///
    /// Use `gio.ListStore.splice` to insert multiple items at the same time
    /// efficiently.
    extern fn g_list_store_insert(p_store: *ListStore, p_position: c_uint, p_item: *gobject.Object) void;
    pub const insert = g_list_store_insert;

    /// Inserts `item` into `store` at a position to be determined by the
    /// `compare_func`.
    ///
    /// The list must already be sorted before calling this function or the
    /// result is undefined.  Usually you would approach this by only ever
    /// inserting items by way of this function.
    ///
    /// This function takes a ref on `item`.
    extern fn g_list_store_insert_sorted(p_store: *ListStore, p_item: *gobject.Object, p_compare_func: glib.CompareDataFunc, p_user_data: ?*anyopaque) c_uint;
    pub const insertSorted = g_list_store_insert_sorted;

    /// Removes the item from `store` that is at `position`. `position` must be
    /// smaller than the current length of the list.
    ///
    /// Use `gio.ListStore.splice` to remove multiple items at the same time
    /// efficiently.
    extern fn g_list_store_remove(p_store: *ListStore, p_position: c_uint) void;
    pub const remove = g_list_store_remove;

    /// Removes all items from `store`.
    extern fn g_list_store_remove_all(p_store: *ListStore) void;
    pub const removeAll = g_list_store_remove_all;

    /// Sort the items in `store` according to `compare_func`.
    extern fn g_list_store_sort(p_store: *ListStore, p_compare_func: glib.CompareDataFunc, p_user_data: ?*anyopaque) void;
    pub const sort = g_list_store_sort;

    /// Changes `store` by removing `n_removals` items and adding `n_additions`
    /// items to it. `additions` must contain `n_additions` items of type
    /// `gio.ListStore.properties.item`-type.  `NULL` is not permitted.
    ///
    /// This function is more efficient than `gio.ListStore.insert` and
    /// `gio.ListStore.remove`, because it only emits
    /// `gio.ListModel.signals.items`-changed once for the change.
    ///
    /// This function takes a ref on each item in `additions`.
    ///
    /// The parameters `position` and `n_removals` must be correct (ie:
    /// `position` + `n_removals` must be less than or equal to the length of
    /// the list at the time this function is called).
    extern fn g_list_store_splice(p_store: *ListStore, p_position: c_uint, p_n_removals: c_uint, p_additions: [*]*gobject.Object, p_n_additions: c_uint) void;
    pub const splice = g_list_store_splice;

    extern fn g_list_store_get_type() usize;
    pub const getGObjectType = g_list_store_get_type;

    extern fn g_object_ref(p_self: *gio.ListStore) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.ListStore) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ListStore, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GMemoryInputStream` is a class for using arbitrary
/// memory chunks as input for GIO streaming input operations.
///
/// As of GLib 2.34, `GMemoryInputStream` implements
/// `gio.PollableInputStream`.
pub const MemoryInputStream = extern struct {
    pub const Parent = gio.InputStream;
    pub const Implements = [_]type{ gio.PollableInputStream, gio.Seekable };
    pub const Class = gio.MemoryInputStreamClass;
    f_parent_instance: gio.InputStream,
    f_priv: ?*gio.MemoryInputStreamPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new empty `gio.MemoryInputStream`.
    extern fn g_memory_input_stream_new() *gio.MemoryInputStream;
    pub const new = g_memory_input_stream_new;

    /// Creates a new `gio.MemoryInputStream` with data from the given `bytes`.
    extern fn g_memory_input_stream_new_from_bytes(p_bytes: *glib.Bytes) *gio.MemoryInputStream;
    pub const newFromBytes = g_memory_input_stream_new_from_bytes;

    /// Creates a new `gio.MemoryInputStream` with data in memory of a given size.
    extern fn g_memory_input_stream_new_from_data(p_data: [*]u8, p_len: isize, p_destroy: ?glib.DestroyNotify) *gio.MemoryInputStream;
    pub const newFromData = g_memory_input_stream_new_from_data;

    /// Appends `bytes` to data that can be read from the input stream.
    extern fn g_memory_input_stream_add_bytes(p_stream: *MemoryInputStream, p_bytes: *glib.Bytes) void;
    pub const addBytes = g_memory_input_stream_add_bytes;

    /// Appends `data` to data that can be read from the input stream
    extern fn g_memory_input_stream_add_data(p_stream: *MemoryInputStream, p_data: [*]u8, p_len: isize, p_destroy: ?glib.DestroyNotify) void;
    pub const addData = g_memory_input_stream_add_data;

    extern fn g_memory_input_stream_get_type() usize;
    pub const getGObjectType = g_memory_input_stream_get_type;

    extern fn g_object_ref(p_self: *gio.MemoryInputStream) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.MemoryInputStream) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *MemoryInputStream, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GMemoryOutputStream` is a class for using arbitrary
/// memory chunks as output for GIO streaming output operations.
///
/// As of GLib 2.34, `GMemoryOutputStream` trivially implements
/// `gio.PollableOutputStream`: it always polls as ready.
pub const MemoryOutputStream = extern struct {
    pub const Parent = gio.OutputStream;
    pub const Implements = [_]type{ gio.PollableOutputStream, gio.Seekable };
    pub const Class = gio.MemoryOutputStreamClass;
    f_parent_instance: gio.OutputStream,
    f_priv: ?*gio.MemoryOutputStreamPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Pointer to buffer where data will be written.
        pub const data = struct {
            pub const name = "data";

            pub const Type = ?*anyopaque;
        };

        /// Size of data written to the buffer.
        pub const data_size = struct {
            pub const name = "data-size";

            pub const Type = c_ulong;
        };

        /// Function called with the buffer as argument when the stream is destroyed.
        pub const destroy_function = struct {
            pub const name = "destroy-function";

            pub const Type = ?*anyopaque;
        };

        /// Function with realloc semantics called to enlarge the buffer.
        pub const realloc_function = struct {
            pub const name = "realloc-function";

            pub const Type = ?*anyopaque;
        };

        /// Current size of the data buffer.
        pub const size = struct {
            pub const name = "size";

            pub const Type = c_ulong;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gio.MemoryOutputStream`.
    ///
    /// In most cases this is not the function you want.  See
    /// `gio.MemoryOutputStream.newResizable` instead.
    ///
    /// If `data` is non-`NULL`, the stream will use that for its internal storage.
    ///
    /// If `realloc_fn` is non-`NULL`, it will be used for resizing the internal
    /// storage when necessary and the stream will be considered resizable.
    /// In that case, the stream will start out being (conceptually) empty.
    /// `size` is used only as a hint for how big `data` is.  Specifically,
    /// seeking to the end of a newly-created stream will seek to zero, not
    /// `size`.  Seeking past the end of the stream and then writing will
    /// introduce a zero-filled gap.
    ///
    /// If `realloc_fn` is `NULL` then the stream is fixed-sized.  Seeking to
    /// the end will seek to `size` exactly.  Writing past the end will give
    /// an 'out of space' error.  Attempting to seek past the end will fail.
    /// Unlike the resizable case, seeking to an offset within the stream and
    /// writing will preserve the bytes passed in as `data` before that point
    /// and will return them as part of `gio.MemoryOutputStream.stealData`.
    /// If you intend to seek you should probably therefore ensure that `data`
    /// is properly initialised.
    ///
    /// It is probably only meaningful to provide `data` and `size` in the case
    /// that you want a fixed-sized stream.  Put another way: if `realloc_fn`
    /// is non-`NULL` then it makes most sense to give `data` as `NULL` and
    /// `size` as 0 (allowing `gio.MemoryOutputStream` to do the initial
    /// allocation for itself).
    ///
    /// ```
    /// // a stream that can grow
    /// stream = g_memory_output_stream_new (NULL, 0, realloc, free);
    ///
    /// // another stream that can grow
    /// stream2 = g_memory_output_stream_new (NULL, 0, g_realloc, g_free);
    ///
    /// // a fixed-size stream
    /// data = malloc (200);
    /// stream3 = g_memory_output_stream_new (data, 200, NULL, free);
    /// ```
    extern fn g_memory_output_stream_new(p_data: ?*anyopaque, p_size: usize, p_realloc_function: ?gio.ReallocFunc, p_destroy_function: ?glib.DestroyNotify) *gio.MemoryOutputStream;
    pub const new = g_memory_output_stream_new;

    /// Creates a new `gio.MemoryOutputStream`, using `glib.realloc` and `glib.free`
    /// for memory allocation.
    extern fn g_memory_output_stream_new_resizable() *gio.MemoryOutputStream;
    pub const newResizable = g_memory_output_stream_new_resizable;

    /// Gets any loaded data from the `ostream`.
    ///
    /// Note that the returned pointer may become invalid on the next
    /// write or truncate operation on the stream.
    extern fn g_memory_output_stream_get_data(p_ostream: *MemoryOutputStream) ?*anyopaque;
    pub const getData = g_memory_output_stream_get_data;

    /// Returns the number of bytes from the start up to including the last
    /// byte written in the stream that has not been truncated away.
    extern fn g_memory_output_stream_get_data_size(p_ostream: *MemoryOutputStream) usize;
    pub const getDataSize = g_memory_output_stream_get_data_size;

    /// Gets the size of the currently allocated data area (available from
    /// `gio.MemoryOutputStream.getData`).
    ///
    /// You probably don't want to use this function on resizable streams.
    /// See `gio.MemoryOutputStream.getDataSize` instead.  For resizable
    /// streams the size returned by this function is an implementation
    /// detail and may be change at any time in response to operations on the
    /// stream.
    ///
    /// If the stream is fixed-sized (ie: no realloc was passed to
    /// `gio.MemoryOutputStream.new`) then this is the maximum size of the
    /// stream and further writes will return `G_IO_ERROR_NO_SPACE`.
    ///
    /// In any case, if you want the number of bytes currently written to the
    /// stream, use `gio.MemoryOutputStream.getDataSize`.
    extern fn g_memory_output_stream_get_size(p_ostream: *MemoryOutputStream) usize;
    pub const getSize = g_memory_output_stream_get_size;

    /// Returns data from the `ostream` as a `glib.Bytes`. `ostream` must be
    /// closed before calling this function.
    extern fn g_memory_output_stream_steal_as_bytes(p_ostream: *MemoryOutputStream) *glib.Bytes;
    pub const stealAsBytes = g_memory_output_stream_steal_as_bytes;

    /// Gets any loaded data from the `ostream`. Ownership of the data
    /// is transferred to the caller; when no longer needed it must be
    /// freed using the free function set in `ostream`'s
    /// `gio.MemoryOutputStream.properties.destroy`-function property.
    ///
    /// `ostream` must be closed before calling this function.
    extern fn g_memory_output_stream_steal_data(p_ostream: *MemoryOutputStream) ?*anyopaque;
    pub const stealData = g_memory_output_stream_steal_data;

    extern fn g_memory_output_stream_get_type() usize;
    pub const getGObjectType = g_memory_output_stream_get_type;

    extern fn g_object_ref(p_self: *gio.MemoryOutputStream) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.MemoryOutputStream) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *MemoryOutputStream, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GMenu` is a simple implementation of `gio.MenuModel`.
/// You populate a `GMenu` by adding `gio.MenuItem` instances to it.
///
/// There are some convenience functions to allow you to directly
/// add items (avoiding `gio.MenuItem`) for the common cases. To add
/// a regular item, use `gio.Menu.insert`. To add a section, use
/// `gio.Menu.insertSection`. To add a submenu, use
/// `gio.Menu.insertSubmenu`.
pub const Menu = opaque {
    pub const Parent = gio.MenuModel;
    pub const Implements = [_]type{};
    pub const Class = opaque {
        pub const Instance = Menu;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `gio.Menu`.
    ///
    /// The new menu has no items.
    extern fn g_menu_new() *gio.Menu;
    pub const new = g_menu_new;

    /// Convenience function for appending a normal menu item to the end of
    /// `menu`.  Combine `gio.MenuItem.new` and `gio.Menu.insertItem` for a more
    /// flexible alternative.
    extern fn g_menu_append(p_menu: *Menu, p_label: ?[*:0]const u8, p_detailed_action: ?[*:0]const u8) void;
    pub const append = g_menu_append;

    /// Appends `item` to the end of `menu`.
    ///
    /// See `gio.Menu.insertItem` for more information.
    extern fn g_menu_append_item(p_menu: *Menu, p_item: *gio.MenuItem) void;
    pub const appendItem = g_menu_append_item;

    /// Convenience function for appending a section menu item to the end of
    /// `menu`.  Combine `gio.MenuItem.newSection` and `gio.Menu.insertItem` for a
    /// more flexible alternative.
    extern fn g_menu_append_section(p_menu: *Menu, p_label: ?[*:0]const u8, p_section: *gio.MenuModel) void;
    pub const appendSection = g_menu_append_section;

    /// Convenience function for appending a submenu menu item to the end of
    /// `menu`.  Combine `gio.MenuItem.newSubmenu` and `gio.Menu.insertItem` for a
    /// more flexible alternative.
    extern fn g_menu_append_submenu(p_menu: *Menu, p_label: ?[*:0]const u8, p_submenu: *gio.MenuModel) void;
    pub const appendSubmenu = g_menu_append_submenu;

    /// Marks `menu` as frozen.
    ///
    /// After the menu is frozen, it is an error to attempt to make any
    /// changes to it.  In effect this means that the `gio.Menu` API must no
    /// longer be used.
    ///
    /// This function causes `gio.MenuModel.isMutable` to begin returning
    /// `FALSE`, which has some positive performance implications.
    extern fn g_menu_freeze(p_menu: *Menu) void;
    pub const freeze = g_menu_freeze;

    /// Convenience function for inserting a normal menu item into `menu`.
    /// Combine `gio.MenuItem.new` and `gio.Menu.insertItem` for a more flexible
    /// alternative.
    extern fn g_menu_insert(p_menu: *Menu, p_position: c_int, p_label: ?[*:0]const u8, p_detailed_action: ?[*:0]const u8) void;
    pub const insert = g_menu_insert;

    /// Inserts `item` into `menu`.
    ///
    /// The "insertion" is actually done by copying all of the attribute and
    /// link values of `item` and using them to form a new item within `menu`.
    /// As such, `item` itself is not really inserted, but rather, a menu item
    /// that is exactly the same as the one presently described by `item`.
    ///
    /// This means that `item` is essentially useless after the insertion
    /// occurs.  Any changes you make to it are ignored unless it is inserted
    /// again (at which point its updated values will be copied).
    ///
    /// You should probably just free `item` once you're done.
    ///
    /// There are many convenience functions to take care of common cases.
    /// See `gio.Menu.insert`, `gio.Menu.insertSection` and
    /// `gio.Menu.insertSubmenu` as well as "prepend" and "append" variants of
    /// each of these functions.
    extern fn g_menu_insert_item(p_menu: *Menu, p_position: c_int, p_item: *gio.MenuItem) void;
    pub const insertItem = g_menu_insert_item;

    /// Convenience function for inserting a section menu item into `menu`.
    /// Combine `gio.MenuItem.newSection` and `gio.Menu.insertItem` for a more
    /// flexible alternative.
    extern fn g_menu_insert_section(p_menu: *Menu, p_position: c_int, p_label: ?[*:0]const u8, p_section: *gio.MenuModel) void;
    pub const insertSection = g_menu_insert_section;

    /// Convenience function for inserting a submenu menu item into `menu`.
    /// Combine `gio.MenuItem.newSubmenu` and `gio.Menu.insertItem` for a more
    /// flexible alternative.
    extern fn g_menu_insert_submenu(p_menu: *Menu, p_position: c_int, p_label: ?[*:0]const u8, p_submenu: *gio.MenuModel) void;
    pub const insertSubmenu = g_menu_insert_submenu;

    /// Convenience function for prepending a normal menu item to the start
    /// of `menu`.  Combine `gio.MenuItem.new` and `gio.Menu.insertItem` for a more
    /// flexible alternative.
    extern fn g_menu_prepend(p_menu: *Menu, p_label: ?[*:0]const u8, p_detailed_action: ?[*:0]const u8) void;
    pub const prepend = g_menu_prepend;

    /// Prepends `item` to the start of `menu`.
    ///
    /// See `gio.Menu.insertItem` for more information.
    extern fn g_menu_prepend_item(p_menu: *Menu, p_item: *gio.MenuItem) void;
    pub const prependItem = g_menu_prepend_item;

    /// Convenience function for prepending a section menu item to the start
    /// of `menu`.  Combine `gio.MenuItem.newSection` and `gio.Menu.insertItem` for
    /// a more flexible alternative.
    extern fn g_menu_prepend_section(p_menu: *Menu, p_label: ?[*:0]const u8, p_section: *gio.MenuModel) void;
    pub const prependSection = g_menu_prepend_section;

    /// Convenience function for prepending a submenu menu item to the start
    /// of `menu`.  Combine `gio.MenuItem.newSubmenu` and `gio.Menu.insertItem` for
    /// a more flexible alternative.
    extern fn g_menu_prepend_submenu(p_menu: *Menu, p_label: ?[*:0]const u8, p_submenu: *gio.MenuModel) void;
    pub const prependSubmenu = g_menu_prepend_submenu;

    /// Removes an item from the menu.
    ///
    /// `position` gives the index of the item to remove.
    ///
    /// It is an error if position is not in range the range from 0 to one
    /// less than the number of items in the menu.
    ///
    /// It is not possible to remove items by identity since items are added
    /// to the menu simply by copying their links and attributes (ie:
    /// identity of the item itself is not preserved).
    extern fn g_menu_remove(p_menu: *Menu, p_position: c_int) void;
    pub const remove = g_menu_remove;

    /// Removes all items in the menu.
    extern fn g_menu_remove_all(p_menu: *Menu) void;
    pub const removeAll = g_menu_remove_all;

    extern fn g_menu_get_type() usize;
    pub const getGObjectType = g_menu_get_type;

    extern fn g_object_ref(p_self: *gio.Menu) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.Menu) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Menu, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gio.MenuAttributeIter` is an opaque structure type.  You must access it
/// using the functions below.
pub const MenuAttributeIter = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.MenuAttributeIterClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.MenuAttributeIterPrivate,

    pub const virtual_methods = struct {
        /// This function combines `gio.MenuAttributeIter.next` with
        /// `gio.MenuAttributeIter.getName` and `gio.MenuAttributeIter.getValue`.
        ///
        /// First the iterator is advanced to the next (possibly first) attribute.
        /// If that fails, then `FALSE` is returned and there are no other
        /// effects.
        ///
        /// If successful, `name` and `value` are set to the name and value of the
        /// attribute that has just been advanced to.  At this point,
        /// `gio.MenuAttributeIter.getName` and `gio.MenuAttributeIter.getValue` will
        /// return the same values again.
        ///
        /// The value returned in `name` remains valid for as long as the iterator
        /// remains at the current position.  The value returned in `value` must
        /// be unreffed using `glib.Variant.unref` when it is no longer in use.
        pub const get_next = struct {
            pub fn call(p_class: anytype, p_iter: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_out_name: ?*[*:0]const u8, p_value: ?**glib.Variant) c_int {
                return gobject.ext.as(MenuAttributeIter.Class, p_class).f_get_next.?(gobject.ext.as(MenuAttributeIter, p_iter), p_out_name, p_value);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_iter: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_out_name: ?*[*:0]const u8, p_value: ?**glib.Variant) callconv(.C) c_int) void {
                gobject.ext.as(MenuAttributeIter.Class, p_class).f_get_next = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {};

    /// Gets the name of the attribute at the current iterator position, as
    /// a string.
    ///
    /// The iterator is not advanced.
    extern fn g_menu_attribute_iter_get_name(p_iter: *MenuAttributeIter) [*:0]const u8;
    pub const getName = g_menu_attribute_iter_get_name;

    /// This function combines `gio.MenuAttributeIter.next` with
    /// `gio.MenuAttributeIter.getName` and `gio.MenuAttributeIter.getValue`.
    ///
    /// First the iterator is advanced to the next (possibly first) attribute.
    /// If that fails, then `FALSE` is returned and there are no other
    /// effects.
    ///
    /// If successful, `name` and `value` are set to the name and value of the
    /// attribute that has just been advanced to.  At this point,
    /// `gio.MenuAttributeIter.getName` and `gio.MenuAttributeIter.getValue` will
    /// return the same values again.
    ///
    /// The value returned in `name` remains valid for as long as the iterator
    /// remains at the current position.  The value returned in `value` must
    /// be unreffed using `glib.Variant.unref` when it is no longer in use.
    extern fn g_menu_attribute_iter_get_next(p_iter: *MenuAttributeIter, p_out_name: ?*[*:0]const u8, p_value: ?**glib.Variant) c_int;
    pub const getNext = g_menu_attribute_iter_get_next;

    /// Gets the value of the attribute at the current iterator position.
    ///
    /// The iterator is not advanced.
    extern fn g_menu_attribute_iter_get_value(p_iter: *MenuAttributeIter) *glib.Variant;
    pub const getValue = g_menu_attribute_iter_get_value;

    /// Attempts to advance the iterator to the next (possibly first)
    /// attribute.
    ///
    /// `TRUE` is returned on success, or `FALSE` if there are no more
    /// attributes.
    ///
    /// You must call this function when you first acquire the iterator
    /// to advance it to the first attribute (and determine if the first
    /// attribute exists at all).
    extern fn g_menu_attribute_iter_next(p_iter: *MenuAttributeIter) c_int;
    pub const next = g_menu_attribute_iter_next;

    extern fn g_menu_attribute_iter_get_type() usize;
    pub const getGObjectType = g_menu_attribute_iter_get_type;

    extern fn g_object_ref(p_self: *gio.MenuAttributeIter) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.MenuAttributeIter) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *MenuAttributeIter, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gio.MenuItem` is an opaque structure type.  You must access it using the
/// functions below.
pub const MenuItem = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = opaque {
        pub const Instance = MenuItem;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `gio.MenuItem`.
    ///
    /// If `label` is non-`NULL` it is used to set the "label" attribute of the
    /// new item.
    ///
    /// If `detailed_action` is non-`NULL` it is used to set the "action" and
    /// possibly the "target" attribute of the new item.  See
    /// `gio.MenuItem.setDetailedAction` for more information.
    extern fn g_menu_item_new(p_label: ?[*:0]const u8, p_detailed_action: ?[*:0]const u8) *gio.MenuItem;
    pub const new = g_menu_item_new;

    /// Creates a `gio.MenuItem` as an exact copy of an existing menu item in a
    /// `gio.MenuModel`.
    ///
    /// `item_index` must be valid (ie: be sure to call
    /// `gio.MenuModel.getNItems` first).
    extern fn g_menu_item_new_from_model(p_model: *gio.MenuModel, p_item_index: c_int) *gio.MenuItem;
    pub const newFromModel = g_menu_item_new_from_model;

    /// Creates a new `gio.MenuItem` representing a section.
    ///
    /// This is a convenience API around `gio.MenuItem.new` and
    /// `gio.MenuItem.setSection`.
    ///
    /// The effect of having one menu appear as a section of another is
    /// exactly as it sounds: the items from `section` become a direct part of
    /// the menu that `menu_item` is added to.
    ///
    /// Visual separation is typically displayed between two non-empty
    /// sections.  If `label` is non-`NULL` then it will be incorporated into
    /// this visual indication.  This allows for labeled subsections of a
    /// menu.
    ///
    /// As a simple example, consider a typical "Edit" menu from a simple
    /// program.  It probably contains an "Undo" and "Redo" item, followed by
    /// a separator, followed by "Cut", "Copy" and "Paste".
    ///
    /// This would be accomplished by creating three `gio.Menu` instances.  The
    /// first would be populated with the "Undo" and "Redo" items, and the
    /// second with the "Cut", "Copy" and "Paste" items.  The first and
    /// second menus would then be added as submenus of the third.  In XML
    /// format, this would look something like the following:
    /// ```
    /// <menu id='edit-menu'>
    ///   <section>
    ///     <item label='Undo'/>
    ///     <item label='Redo'/>
    ///   </section>
    ///   <section>
    ///     <item label='Cut'/>
    ///     <item label='Copy'/>
    ///     <item label='Paste'/>
    ///   </section>
    /// </menu>
    /// ```
    ///
    /// The following example is exactly equivalent.  It is more illustrative
    /// of the exact relationship between the menus and items (keeping in
    /// mind that the 'link' element defines a new menu that is linked to the
    /// containing one).  The style of the second example is more verbose and
    /// difficult to read (and therefore not recommended except for the
    /// purpose of understanding what is really going on).
    /// ```
    /// <menu id='edit-menu'>
    ///   <item>
    ///     <link name='section'>
    ///       <item label='Undo'/>
    ///       <item label='Redo'/>
    ///     </link>
    ///   </item>
    ///   <item>
    ///     <link name='section'>
    ///       <item label='Cut'/>
    ///       <item label='Copy'/>
    ///       <item label='Paste'/>
    ///     </link>
    ///   </item>
    /// </menu>
    /// ```
    extern fn g_menu_item_new_section(p_label: ?[*:0]const u8, p_section: *gio.MenuModel) *gio.MenuItem;
    pub const newSection = g_menu_item_new_section;

    /// Creates a new `gio.MenuItem` representing a submenu.
    ///
    /// This is a convenience API around `gio.MenuItem.new` and
    /// `gio.MenuItem.setSubmenu`.
    extern fn g_menu_item_new_submenu(p_label: ?[*:0]const u8, p_submenu: *gio.MenuModel) *gio.MenuItem;
    pub const newSubmenu = g_menu_item_new_submenu;

    /// Queries the named `attribute` on `menu_item`.
    ///
    /// If the attribute exists and matches the `glib.VariantType` corresponding
    /// to `format_string` then `format_string` is used to deconstruct the
    /// value into the positional parameters and `TRUE` is returned.
    ///
    /// If the attribute does not exist, or it does exist but has the wrong
    /// type, then the positional parameters are ignored and `FALSE` is
    /// returned.
    extern fn g_menu_item_get_attribute(p_menu_item: *MenuItem, p_attribute: [*:0]const u8, p_format_string: [*:0]const u8, ...) c_int;
    pub const getAttribute = g_menu_item_get_attribute;

    /// Queries the named `attribute` on `menu_item`.
    ///
    /// If `expected_type` is specified and the attribute does not have this
    /// type, `NULL` is returned.  `NULL` is also returned if the attribute
    /// simply does not exist.
    extern fn g_menu_item_get_attribute_value(p_menu_item: *MenuItem, p_attribute: [*:0]const u8, p_expected_type: ?*const glib.VariantType) ?*glib.Variant;
    pub const getAttributeValue = g_menu_item_get_attribute_value;

    /// Queries the named `link` on `menu_item`.
    extern fn g_menu_item_get_link(p_menu_item: *MenuItem, p_link: [*:0]const u8) ?*gio.MenuModel;
    pub const getLink = g_menu_item_get_link;

    /// Sets or unsets the "action" and "target" attributes of `menu_item`.
    ///
    /// If `action` is `NULL` then both the "action" and "target" attributes
    /// are unset (and `format_string` is ignored along with the positional
    /// parameters).
    ///
    /// If `action` is non-`NULL` then the "action" attribute is set.
    /// `format_string` is then inspected.  If it is non-`NULL` then the proper
    /// position parameters are collected to create a `glib.Variant` instance to
    /// use as the target value.  If it is `NULL` then the positional
    /// parameters are ignored and the "target" attribute is unset.
    ///
    /// See also `gio.MenuItem.setActionAndTargetValue` for an equivalent
    /// call that directly accepts a `glib.Variant`.  See
    /// `gio.MenuItem.setDetailedAction` for a more convenient version that
    /// works with string-typed targets.
    ///
    /// See also `gio.MenuItem.setActionAndTargetValue` for a
    /// description of the semantics of the action and target attributes.
    extern fn g_menu_item_set_action_and_target(p_menu_item: *MenuItem, p_action: ?[*:0]const u8, p_format_string: ?[*:0]const u8, ...) void;
    pub const setActionAndTarget = g_menu_item_set_action_and_target;

    /// Sets or unsets the "action" and "target" attributes of `menu_item`.
    ///
    /// If `action` is `NULL` then both the "action" and "target" attributes
    /// are unset (and `target_value` is ignored).
    ///
    /// If `action` is non-`NULL` then the "action" attribute is set.  The
    /// "target" attribute is then set to the value of `target_value` if it is
    /// non-`NULL` or unset otherwise.
    ///
    /// Normal menu items (ie: not submenu, section or other custom item
    /// types) are expected to have the "action" attribute set to identify
    /// the action that they are associated with.  The state type of the
    /// action help to determine the disposition of the menu item.  See
    /// `gio.Action` and `gio.ActionGroup` for an overview of actions.
    ///
    /// In general, clicking on the menu item will result in activation of
    /// the named action with the "target" attribute given as the parameter
    /// to the action invocation.  If the "target" attribute is not set then
    /// the action is invoked with no parameter.
    ///
    /// If the action has no state then the menu item is usually drawn as a
    /// plain menu item (ie: with no additional decoration).
    ///
    /// If the action has a boolean state then the menu item is usually drawn
    /// as a toggle menu item (ie: with a checkmark or equivalent
    /// indication).  The item should be marked as 'toggled' or 'checked'
    /// when the boolean state is `TRUE`.
    ///
    /// If the action has a string state then the menu item is usually drawn
    /// as a radio menu item (ie: with a radio bullet or equivalent
    /// indication).  The item should be marked as 'selected' when the string
    /// state is equal to the value of the `target` property.
    ///
    /// See `gio.MenuItem.setActionAndTarget` or
    /// `gio.MenuItem.setDetailedAction` for two equivalent calls that are
    /// probably more convenient for most uses.
    extern fn g_menu_item_set_action_and_target_value(p_menu_item: *MenuItem, p_action: ?[*:0]const u8, p_target_value: ?*glib.Variant) void;
    pub const setActionAndTargetValue = g_menu_item_set_action_and_target_value;

    /// Sets or unsets an attribute on `menu_item`.
    ///
    /// The attribute to set or unset is specified by `attribute`. This
    /// can be one of the standard attribute names `G_MENU_ATTRIBUTE_LABEL`,
    /// `G_MENU_ATTRIBUTE_ACTION`, `G_MENU_ATTRIBUTE_TARGET`, or a custom
    /// attribute name.
    /// Attribute names are restricted to lowercase characters, numbers
    /// and '-'. Furthermore, the names must begin with a lowercase character,
    /// must not end with a '-', and must not contain consecutive dashes.
    ///
    /// If `format_string` is non-`NULL` then the proper position parameters
    /// are collected to create a `glib.Variant` instance to use as the attribute
    /// value.  If it is `NULL` then the positional parameterrs are ignored
    /// and the named attribute is unset.
    ///
    /// See also `gio.MenuItem.setAttributeValue` for an equivalent call
    /// that directly accepts a `glib.Variant`.
    extern fn g_menu_item_set_attribute(p_menu_item: *MenuItem, p_attribute: [*:0]const u8, p_format_string: ?[*:0]const u8, ...) void;
    pub const setAttribute = g_menu_item_set_attribute;

    /// Sets or unsets an attribute on `menu_item`.
    ///
    /// The attribute to set or unset is specified by `attribute`. This
    /// can be one of the standard attribute names `G_MENU_ATTRIBUTE_LABEL`,
    /// `G_MENU_ATTRIBUTE_ACTION`, `G_MENU_ATTRIBUTE_TARGET`, or a custom
    /// attribute name.
    /// Attribute names are restricted to lowercase characters, numbers
    /// and '-'. Furthermore, the names must begin with a lowercase character,
    /// must not end with a '-', and must not contain consecutive dashes.
    ///
    /// must consist only of lowercase
    /// ASCII characters, digits and '-'.
    ///
    /// If `value` is non-`NULL` then it is used as the new value for the
    /// attribute.  If `value` is `NULL` then the attribute is unset. If
    /// the `value` `glib.Variant` is floating, it is consumed.
    ///
    /// See also `gio.MenuItem.setAttribute` for a more convenient way to do
    /// the same.
    extern fn g_menu_item_set_attribute_value(p_menu_item: *MenuItem, p_attribute: [*:0]const u8, p_value: ?*glib.Variant) void;
    pub const setAttributeValue = g_menu_item_set_attribute_value;

    /// Sets the "action" and possibly the "target" attribute of `menu_item`.
    ///
    /// The format of `detailed_action` is the same format parsed by
    /// `gio.actionParseDetailedName`.
    ///
    /// See `gio.MenuItem.setActionAndTarget` or
    /// `gio.MenuItem.setActionAndTargetValue` for more flexible (but
    /// slightly less convenient) alternatives.
    ///
    /// See also `gio.MenuItem.setActionAndTargetValue` for a description of
    /// the semantics of the action and target attributes.
    extern fn g_menu_item_set_detailed_action(p_menu_item: *MenuItem, p_detailed_action: [*:0]const u8) void;
    pub const setDetailedAction = g_menu_item_set_detailed_action;

    /// Sets (or unsets) the icon on `menu_item`.
    ///
    /// This call is the same as calling `gio.Icon.serialize` and using the
    /// result as the value to `gio.MenuItem.setAttributeValue` for
    /// `G_MENU_ATTRIBUTE_ICON`.
    ///
    /// This API is only intended for use with "noun" menu items; things like
    /// bookmarks or applications in an "Open With" menu.  Don't use it on
    /// menu items corresponding to verbs (eg: stock icons for 'Save' or
    /// 'Quit').
    ///
    /// If `icon` is `NULL` then the icon is unset.
    extern fn g_menu_item_set_icon(p_menu_item: *MenuItem, p_icon: *gio.Icon) void;
    pub const setIcon = g_menu_item_set_icon;

    /// Sets or unsets the "label" attribute of `menu_item`.
    ///
    /// If `label` is non-`NULL` it is used as the label for the menu item.  If
    /// it is `NULL` then the label attribute is unset.
    extern fn g_menu_item_set_label(p_menu_item: *MenuItem, p_label: ?[*:0]const u8) void;
    pub const setLabel = g_menu_item_set_label;

    /// Creates a link from `menu_item` to `model` if non-`NULL`, or unsets it.
    ///
    /// Links are used to establish a relationship between a particular menu
    /// item and another menu.  For example, `G_MENU_LINK_SUBMENU` is used to
    /// associate a submenu with a particular menu item, and `G_MENU_LINK_SECTION`
    /// is used to create a section. Other types of link can be used, but there
    /// is no guarantee that clients will be able to make sense of them.
    /// Link types are restricted to lowercase characters, numbers
    /// and '-'. Furthermore, the names must begin with a lowercase character,
    /// must not end with a '-', and must not contain consecutive dashes.
    extern fn g_menu_item_set_link(p_menu_item: *MenuItem, p_link: [*:0]const u8, p_model: ?*gio.MenuModel) void;
    pub const setLink = g_menu_item_set_link;

    /// Sets or unsets the "section" link of `menu_item` to `section`.
    ///
    /// The effect of having one menu appear as a section of another is
    /// exactly as it sounds: the items from `section` become a direct part of
    /// the menu that `menu_item` is added to.  See `gio.MenuItem.newSection`
    /// for more information about what it means for a menu item to be a
    /// section.
    extern fn g_menu_item_set_section(p_menu_item: *MenuItem, p_section: ?*gio.MenuModel) void;
    pub const setSection = g_menu_item_set_section;

    /// Sets or unsets the "submenu" link of `menu_item` to `submenu`.
    ///
    /// If `submenu` is non-`NULL`, it is linked to.  If it is `NULL` then the
    /// link is unset.
    ///
    /// The effect of having one menu appear as a submenu of another is
    /// exactly as it sounds.
    extern fn g_menu_item_set_submenu(p_menu_item: *MenuItem, p_submenu: ?*gio.MenuModel) void;
    pub const setSubmenu = g_menu_item_set_submenu;

    extern fn g_menu_item_get_type() usize;
    pub const getGObjectType = g_menu_item_get_type;

    extern fn g_object_ref(p_self: *gio.MenuItem) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.MenuItem) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *MenuItem, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `gio.MenuLinkIter` is an opaque structure type.  You must access it using
/// the functions below.
pub const MenuLinkIter = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.MenuLinkIterClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.MenuLinkIterPrivate,

    pub const virtual_methods = struct {
        /// This function combines `gio.MenuLinkIter.next` with
        /// `gio.MenuLinkIter.getName` and `gio.MenuLinkIter.getValue`.
        ///
        /// First the iterator is advanced to the next (possibly first) link.
        /// If that fails, then `FALSE` is returned and there are no other effects.
        ///
        /// If successful, `out_link` and `value` are set to the name and `gio.MenuModel`
        /// of the link that has just been advanced to.  At this point,
        /// `gio.MenuLinkIter.getName` and `gio.MenuLinkIter.getValue` will return the
        /// same values again.
        ///
        /// The value returned in `out_link` remains valid for as long as the iterator
        /// remains at the current position.  The value returned in `value` must
        /// be unreffed using `gobject.Object.unref` when it is no longer in use.
        pub const get_next = struct {
            pub fn call(p_class: anytype, p_iter: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_out_link: ?*[*:0]const u8, p_value: ?**gio.MenuModel) c_int {
                return gobject.ext.as(MenuLinkIter.Class, p_class).f_get_next.?(gobject.ext.as(MenuLinkIter, p_iter), p_out_link, p_value);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_iter: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_out_link: ?*[*:0]const u8, p_value: ?**gio.MenuModel) callconv(.C) c_int) void {
                gobject.ext.as(MenuLinkIter.Class, p_class).f_get_next = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {};

    /// Gets the name of the link at the current iterator position.
    ///
    /// The iterator is not advanced.
    extern fn g_menu_link_iter_get_name(p_iter: *MenuLinkIter) [*:0]const u8;
    pub const getName = g_menu_link_iter_get_name;

    /// This function combines `gio.MenuLinkIter.next` with
    /// `gio.MenuLinkIter.getName` and `gio.MenuLinkIter.getValue`.
    ///
    /// First the iterator is advanced to the next (possibly first) link.
    /// If that fails, then `FALSE` is returned and there are no other effects.
    ///
    /// If successful, `out_link` and `value` are set to the name and `gio.MenuModel`
    /// of the link that has just been advanced to.  At this point,
    /// `gio.MenuLinkIter.getName` and `gio.MenuLinkIter.getValue` will return the
    /// same values again.
    ///
    /// The value returned in `out_link` remains valid for as long as the iterator
    /// remains at the current position.  The value returned in `value` must
    /// be unreffed using `gobject.Object.unref` when it is no longer in use.
    extern fn g_menu_link_iter_get_next(p_iter: *MenuLinkIter, p_out_link: ?*[*:0]const u8, p_value: ?**gio.MenuModel) c_int;
    pub const getNext = g_menu_link_iter_get_next;

    /// Gets the linked `gio.MenuModel` at the current iterator position.
    ///
    /// The iterator is not advanced.
    extern fn g_menu_link_iter_get_value(p_iter: *MenuLinkIter) *gio.MenuModel;
    pub const getValue = g_menu_link_iter_get_value;

    /// Attempts to advance the iterator to the next (possibly first)
    /// link.
    ///
    /// `TRUE` is returned on success, or `FALSE` if there are no more links.
    ///
    /// You must call this function when you first acquire the iterator to
    /// advance it to the first link (and determine if the first link exists
    /// at all).
    extern fn g_menu_link_iter_next(p_iter: *MenuLinkIter) c_int;
    pub const next = g_menu_link_iter_next;

    extern fn g_menu_link_iter_get_type() usize;
    pub const getGObjectType = g_menu_link_iter_get_type;

    extern fn g_object_ref(p_self: *gio.MenuLinkIter) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.MenuLinkIter) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *MenuLinkIter, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GMenuModel` represents the contents of a menu — an ordered list of
/// menu items. The items are associated with actions, which can be
/// activated through them. Items can be grouped in sections, and may
/// have submenus associated with them. Both items and sections usually
/// have some representation data, such as labels or icons. The type of
/// the associated action (ie whether it is stateful, and what kind of
/// state it has) can influence the representation of the item.
///
/// The conceptual model of menus in `GMenuModel` is hierarchical:
/// sections and submenus are again represented by `GMenuModel`s.
/// Menus themselves do not define their own roles. Rather, the role
/// of a particular `GMenuModel` is defined by the item that references
/// it (or, in the case of the ‘root’ menu, is defined by the context
/// in which it is used).
///
/// As an example, consider the visible portions of this menu:
///
/// ## An example menu
///
/// ![](menu-example.png)
///
/// While this kind of deeply nested menu is no longer considered good UI
/// practice, it serves as a good example of the concepts in `GMenuModel`.
/// There are 8 ‘menus’ visible in the screenshot: one menubar, two
/// submenus and 5 sections:
///
/// - the toplevel menubar (containing 4 items)
/// - the View submenu (containing 3 sections)
/// - the first section of the View submenu (containing 2 items)
/// - the second section of the View submenu (containing 1 item)
/// - the final section of the View submenu (containing 1 item)
/// - the Highlight Mode submenu (containing 2 sections)
/// - the Sources section (containing 2 items)
/// - the Markup section (containing 2 items)
///
/// The [example](`a`-menu-example) illustrates the conceptual connection between
/// these 8 menus. Each large block in the figure represents a menu and the
/// smaller blocks within the large block represent items in that menu. Some
/// items contain references to other menus.
///
/// ## A menu example
///
/// <picture>
///   <source srcset="menu-model-dark.svg" media="(prefers-color-scheme: dark)">
///   <img src="menu-model-light.svg" alt="menu model">
/// </picture>
///
/// Notice that the separators visible in the [example](`an`-example-menu)
/// appear nowhere in the [menu model](`a`-menu-example). This is because
/// separators are not explicitly represented in the menu model. Instead,
/// a separator is inserted between any two non-empty sections of a menu.
/// Section items can have labels just like any other item. In that case,
/// a display system may show a section header instead of a separator.
///
/// The motivation for this abstract model of application controls is
/// that modern user interfaces tend to make these controls available
/// outside the application. Examples include global menus, jumplists,
/// dash boards, etc. To support such uses, it is necessary to ‘export’
/// information about actions and their representation in menus, which
/// is exactly what the action group exporter and the menu model exporter do for
/// `gio.ActionGroup` and `gio.MenuModel`. The client-side
/// counterparts to make use of the exported information are
/// `gio.DBusActionGroup` and `gio.DBusMenuModel`.
///
/// The API of `GMenuModel` is very generic, with iterators for the
/// attributes and links of an item, see
/// `gio.MenuModel.iterateItemAttributes` and
/// `gio.MenuModel.iterateItemLinks`. The ‘standard’ attributes and
/// link types have predefined names: `G_MENU_ATTRIBUTE_LABEL`,
/// `G_MENU_ATTRIBUTE_ACTION`, `G_MENU_ATTRIBUTE_TARGET`, `G_MENU_LINK_SECTION`
/// and `G_MENU_LINK_SUBMENU`.
///
/// Items in a `GMenuModel` represent active controls if they refer to
/// an action that can get activated when the user interacts with the
/// menu item. The reference to the action is encoded by the string ID
/// in the `G_MENU_ATTRIBUTE_ACTION` attribute. An action ID uniquely
/// identifies an action in an action group. Which action group(s) provide
/// actions depends on the context in which the menu model is used.
/// E.g. when the model is exported as the application menu of a
/// [`GtkApplication`](https://docs.gtk.org/gtk4/class.Application.html),
/// actions can be application-wide or window-specific (and thus come from
/// two different action groups). By convention, the application-wide actions
/// have names that start with `app.`, while the names of window-specific
/// actions start with `win.`.
///
/// While a wide variety of stateful actions is possible, the following
/// is the minimum that is expected to be supported by all users of exported
/// menu information:
/// - an action with no parameter type and no state
/// - an action with no parameter type and boolean state
/// - an action with string parameter type and string state
///
/// ## Stateless
///
/// A stateless action typically corresponds to an ordinary menu item.
///
/// Selecting such a menu item will activate the action (with no parameter).
///
/// ## Boolean State
///
/// An action with a boolean state will most typically be used with a ‘toggle’
/// or ‘switch’ menu item. The state can be set directly, but activating the
/// action (with no parameter) results in the state being toggled.
///
/// Selecting a toggle menu item will activate the action. The menu item should
/// be rendered as ‘checked’ when the state is true.
///
/// ## String Parameter and State
///
/// Actions with string parameters and state will most typically be used to
/// represent an enumerated choice over the items available for a group of
/// radio menu items. Activating the action with a string parameter is
/// equivalent to setting that parameter as the state.
///
/// Radio menu items, in addition to being associated with the action, will
/// have a target value. Selecting that menu item will result in activation
/// of the action with the target value as the parameter. The menu item should
/// be rendered as ‘selected’ when the state of the action is equal to the
/// target value of the menu item.
pub const MenuModel = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.MenuModelClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.MenuModelPrivate,

    pub const virtual_methods = struct {
        /// Queries the item at position `item_index` in `model` for the attribute
        /// specified by `attribute`.
        ///
        /// If `expected_type` is non-`NULL` then it specifies the expected type of
        /// the attribute.  If it is `NULL` then any type will be accepted.
        ///
        /// If the attribute exists and matches `expected_type` (or if the
        /// expected type is unspecified) then the value is returned.
        ///
        /// If the attribute does not exist, or does not match the expected type
        /// then `NULL` is returned.
        pub const get_item_attribute_value = struct {
            pub fn call(p_class: anytype, p_model: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_item_index: c_int, p_attribute: [*:0]const u8, p_expected_type: ?*const glib.VariantType) ?*glib.Variant {
                return gobject.ext.as(MenuModel.Class, p_class).f_get_item_attribute_value.?(gobject.ext.as(MenuModel, p_model), p_item_index, p_attribute, p_expected_type);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_model: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_item_index: c_int, p_attribute: [*:0]const u8, p_expected_type: ?*const glib.VariantType) callconv(.C) ?*glib.Variant) void {
                gobject.ext.as(MenuModel.Class, p_class).f_get_item_attribute_value = @ptrCast(p_implementation);
            }
        };

        /// Gets all the attributes associated with the item in the menu model.
        pub const get_item_attributes = struct {
            pub fn call(p_class: anytype, p_model: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_item_index: c_int, p_attributes: **glib.HashTable) void {
                return gobject.ext.as(MenuModel.Class, p_class).f_get_item_attributes.?(gobject.ext.as(MenuModel, p_model), p_item_index, p_attributes);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_model: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_item_index: c_int, p_attributes: **glib.HashTable) callconv(.C) void) void {
                gobject.ext.as(MenuModel.Class, p_class).f_get_item_attributes = @ptrCast(p_implementation);
            }
        };

        /// Queries the item at position `item_index` in `model` for the link
        /// specified by `link`.
        ///
        /// If the link exists, the linked `gio.MenuModel` is returned.  If the link
        /// does not exist, `NULL` is returned.
        pub const get_item_link = struct {
            pub fn call(p_class: anytype, p_model: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_item_index: c_int, p_link: [*:0]const u8) ?*gio.MenuModel {
                return gobject.ext.as(MenuModel.Class, p_class).f_get_item_link.?(gobject.ext.as(MenuModel, p_model), p_item_index, p_link);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_model: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_item_index: c_int, p_link: [*:0]const u8) callconv(.C) ?*gio.MenuModel) void {
                gobject.ext.as(MenuModel.Class, p_class).f_get_item_link = @ptrCast(p_implementation);
            }
        };

        /// Gets all the links associated with the item in the menu model.
        pub const get_item_links = struct {
            pub fn call(p_class: anytype, p_model: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_item_index: c_int, p_links: **glib.HashTable) void {
                return gobject.ext.as(MenuModel.Class, p_class).f_get_item_links.?(gobject.ext.as(MenuModel, p_model), p_item_index, p_links);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_model: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_item_index: c_int, p_links: **glib.HashTable) callconv(.C) void) void {
                gobject.ext.as(MenuModel.Class, p_class).f_get_item_links = @ptrCast(p_implementation);
            }
        };

        /// Query the number of items in `model`.
        pub const get_n_items = struct {
            pub fn call(p_class: anytype, p_model: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) c_int {
                return gobject.ext.as(MenuModel.Class, p_class).f_get_n_items.?(gobject.ext.as(MenuModel, p_model));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_model: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) c_int) void {
                gobject.ext.as(MenuModel.Class, p_class).f_get_n_items = @ptrCast(p_implementation);
            }
        };

        /// Queries if `model` is mutable.
        ///
        /// An immutable `gio.MenuModel` will never emit the `gio.MenuModel.signals.items`-changed
        /// signal. Consumers of the model may make optimisations accordingly.
        pub const is_mutable = struct {
            pub fn call(p_class: anytype, p_model: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) c_int {
                return gobject.ext.as(MenuModel.Class, p_class).f_is_mutable.?(gobject.ext.as(MenuModel, p_model));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_model: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) c_int) void {
                gobject.ext.as(MenuModel.Class, p_class).f_is_mutable = @ptrCast(p_implementation);
            }
        };

        /// Creates a `gio.MenuAttributeIter` to iterate over the attributes of
        /// the item at position `item_index` in `model`.
        ///
        /// You must free the iterator with `gobject.Object.unref` when you are done.
        pub const iterate_item_attributes = struct {
            pub fn call(p_class: anytype, p_model: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_item_index: c_int) *gio.MenuAttributeIter {
                return gobject.ext.as(MenuModel.Class, p_class).f_iterate_item_attributes.?(gobject.ext.as(MenuModel, p_model), p_item_index);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_model: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_item_index: c_int) callconv(.C) *gio.MenuAttributeIter) void {
                gobject.ext.as(MenuModel.Class, p_class).f_iterate_item_attributes = @ptrCast(p_implementation);
            }
        };

        /// Creates a `gio.MenuLinkIter` to iterate over the links of the item at
        /// position `item_index` in `model`.
        ///
        /// You must free the iterator with `gobject.Object.unref` when you are done.
        pub const iterate_item_links = struct {
            pub fn call(p_class: anytype, p_model: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_item_index: c_int) *gio.MenuLinkIter {
                return gobject.ext.as(MenuModel.Class, p_class).f_iterate_item_links.?(gobject.ext.as(MenuModel, p_model), p_item_index);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_model: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_item_index: c_int) callconv(.C) *gio.MenuLinkIter) void {
                gobject.ext.as(MenuModel.Class, p_class).f_iterate_item_links = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {
        /// Emitted when a change has occurred to the menu.
        ///
        /// The only changes that can occur to a menu is that items are removed
        /// or added.  Items may not change (except by being removed and added
        /// back in the same location).  This signal is capable of describing
        /// both of those changes (at the same time).
        ///
        /// The signal means that starting at the index `position`, `removed`
        /// items were removed and `added` items were added in their place.  If
        /// `removed` is zero then only items were added.  If `added` is zero
        /// then only items were removed.
        ///
        /// As an example, if the menu contains items a, b, c, d (in that
        /// order) and the signal (2, 1, 3) occurs then the new composition of
        /// the menu will be a, b, _, _, _, d (with each _ representing some
        /// new item).
        ///
        /// Signal handlers may query the model (particularly the added items)
        /// and expect to see the results of the modification that is being
        /// reported.  The signal is emitted after the modification.
        pub const items_changed = struct {
            pub const name = "items-changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_position: c_int, p_removed: c_int, p_added: c_int, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(MenuModel, p_instance))),
                    gobject.signalLookup("items-changed", MenuModel.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Queries item at position `item_index` in `model` for the attribute
    /// specified by `attribute`.
    ///
    /// If the attribute exists and matches the `glib.VariantType` corresponding
    /// to `format_string` then `format_string` is used to deconstruct the
    /// value into the positional parameters and `TRUE` is returned.
    ///
    /// If the attribute does not exist, or it does exist but has the wrong
    /// type, then the positional parameters are ignored and `FALSE` is
    /// returned.
    ///
    /// This function is a mix of `gio.MenuModel.getItemAttributeValue` and
    /// `glib.Variant.get`, followed by a `glib.Variant.unref`.  As such,
    /// `format_string` must make a complete copy of the data (since the
    /// `glib.Variant` may go away after the call to `glib.Variant.unref`).  In
    /// particular, no '&' characters are allowed in `format_string`.
    extern fn g_menu_model_get_item_attribute(p_model: *MenuModel, p_item_index: c_int, p_attribute: [*:0]const u8, p_format_string: [*:0]const u8, ...) c_int;
    pub const getItemAttribute = g_menu_model_get_item_attribute;

    /// Queries the item at position `item_index` in `model` for the attribute
    /// specified by `attribute`.
    ///
    /// If `expected_type` is non-`NULL` then it specifies the expected type of
    /// the attribute.  If it is `NULL` then any type will be accepted.
    ///
    /// If the attribute exists and matches `expected_type` (or if the
    /// expected type is unspecified) then the value is returned.
    ///
    /// If the attribute does not exist, or does not match the expected type
    /// then `NULL` is returned.
    extern fn g_menu_model_get_item_attribute_value(p_model: *MenuModel, p_item_index: c_int, p_attribute: [*:0]const u8, p_expected_type: ?*const glib.VariantType) ?*glib.Variant;
    pub const getItemAttributeValue = g_menu_model_get_item_attribute_value;

    /// Queries the item at position `item_index` in `model` for the link
    /// specified by `link`.
    ///
    /// If the link exists, the linked `gio.MenuModel` is returned.  If the link
    /// does not exist, `NULL` is returned.
    extern fn g_menu_model_get_item_link(p_model: *MenuModel, p_item_index: c_int, p_link: [*:0]const u8) ?*gio.MenuModel;
    pub const getItemLink = g_menu_model_get_item_link;

    /// Query the number of items in `model`.
    extern fn g_menu_model_get_n_items(p_model: *MenuModel) c_int;
    pub const getNItems = g_menu_model_get_n_items;

    /// Queries if `model` is mutable.
    ///
    /// An immutable `gio.MenuModel` will never emit the `gio.MenuModel.signals.items`-changed
    /// signal. Consumers of the model may make optimisations accordingly.
    extern fn g_menu_model_is_mutable(p_model: *MenuModel) c_int;
    pub const isMutable = g_menu_model_is_mutable;

    /// Requests emission of the `gio.MenuModel.signals.items`-changed signal on `model`.
    ///
    /// This function should never be called except by `gio.MenuModel`
    /// subclasses.  Any other calls to this function will very likely lead
    /// to a violation of the interface of the model.
    ///
    /// The implementation should update its internal representation of the
    /// menu before emitting the signal.  The implementation should further
    /// expect to receive queries about the new state of the menu (and
    /// particularly added menu items) while signal handlers are running.
    ///
    /// The implementation must dispatch this call directly from a mainloop
    /// entry and not in response to calls -- particularly those from the
    /// `gio.MenuModel` API.  Said another way: the menu must not change while
    /// user code is running without returning to the mainloop.
    extern fn g_menu_model_items_changed(p_model: *MenuModel, p_position: c_int, p_removed: c_int, p_added: c_int) void;
    pub const itemsChanged = g_menu_model_items_changed;

    /// Creates a `gio.MenuAttributeIter` to iterate over the attributes of
    /// the item at position `item_index` in `model`.
    ///
    /// You must free the iterator with `gobject.Object.unref` when you are done.
    extern fn g_menu_model_iterate_item_attributes(p_model: *MenuModel, p_item_index: c_int) *gio.MenuAttributeIter;
    pub const iterateItemAttributes = g_menu_model_iterate_item_attributes;

    /// Creates a `gio.MenuLinkIter` to iterate over the links of the item at
    /// position `item_index` in `model`.
    ///
    /// You must free the iterator with `gobject.Object.unref` when you are done.
    extern fn g_menu_model_iterate_item_links(p_model: *MenuModel, p_item_index: c_int) *gio.MenuLinkIter;
    pub const iterateItemLinks = g_menu_model_iterate_item_links;

    extern fn g_menu_model_get_type() usize;
    pub const getGObjectType = g_menu_model_get_type;

    extern fn g_object_ref(p_self: *gio.MenuModel) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.MenuModel) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *MenuModel, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GMountOperation` provides a mechanism for interacting with the user.
/// It can be used for authenticating mountable operations, such as loop
/// mounting files, hard drive partitions or server locations. It can
/// also be used to ask the user questions or show a list of applications
/// preventing unmount or eject operations from completing.
///
/// Note that `GMountOperation` is used for more than just `gio.Mount`
/// objects – for example it is also used in `gio.Drive.start` and
/// `gio.Drive.stop`.
///
/// Users should instantiate a subclass of this that implements all the
/// various callbacks to show the required dialogs, such as
/// [`GtkMountOperation`](https://docs.gtk.org/gtk4/class.MountOperation.html).
/// If no user interaction is desired (for example when automounting
/// filesystems at login time), usually `NULL` can be passed, see each method
/// taking a `GMountOperation` for details.
///
/// Throughout the API, the term ‘TCRYPT’ is used to mean ‘compatible with TrueCrypt and VeraCrypt’.
/// [TrueCrypt](https://en.wikipedia.org/wiki/TrueCrypt) is a discontinued system for
/// encrypting file containers, partitions or whole disks, typically used with Windows.
/// [VeraCrypt](https://www.veracrypt.fr/) is a maintained fork of TrueCrypt with various
/// improvements and auditing fixes.
pub const MountOperation = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.MountOperationClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.MountOperationPrivate,

    pub const virtual_methods = struct {
        pub const aborted = struct {
            pub fn call(p_class: anytype, p_op: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(MountOperation.Class, p_class).f_aborted.?(gobject.ext.as(MountOperation, p_op));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_op: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(MountOperation.Class, p_class).f_aborted = @ptrCast(p_implementation);
            }
        };

        pub const ask_password = struct {
            pub fn call(p_class: anytype, p_op: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_message: [*:0]const u8, p_default_user: [*:0]const u8, p_default_domain: [*:0]const u8, p_flags: gio.AskPasswordFlags) void {
                return gobject.ext.as(MountOperation.Class, p_class).f_ask_password.?(gobject.ext.as(MountOperation, p_op), p_message, p_default_user, p_default_domain, p_flags);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_op: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_message: [*:0]const u8, p_default_user: [*:0]const u8, p_default_domain: [*:0]const u8, p_flags: gio.AskPasswordFlags) callconv(.C) void) void {
                gobject.ext.as(MountOperation.Class, p_class).f_ask_password = @ptrCast(p_implementation);
            }
        };

        /// Virtual implementation of `gio.MountOperation.signals.ask`-question.
        pub const ask_question = struct {
            pub fn call(p_class: anytype, p_op: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_message: [*:0]const u8, p_choices: [*][*:0]const u8) void {
                return gobject.ext.as(MountOperation.Class, p_class).f_ask_question.?(gobject.ext.as(MountOperation, p_op), p_message, p_choices);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_op: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_message: [*:0]const u8, p_choices: [*][*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(MountOperation.Class, p_class).f_ask_question = @ptrCast(p_implementation);
            }
        };

        /// Emits the `gio.MountOperation.signals.reply` signal.
        pub const reply = struct {
            pub fn call(p_class: anytype, p_op: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: gio.MountOperationResult) void {
                return gobject.ext.as(MountOperation.Class, p_class).f_reply.?(gobject.ext.as(MountOperation, p_op), p_result);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_op: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: gio.MountOperationResult) callconv(.C) void) void {
                gobject.ext.as(MountOperation.Class, p_class).f_reply = @ptrCast(p_implementation);
            }
        };

        /// Virtual implementation of `gio.MountOperation.signals.show`-processes.
        pub const show_processes = struct {
            pub fn call(p_class: anytype, p_op: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_message: [*:0]const u8, p_processes: *glib.Array, p_choices: [*][*:0]const u8) void {
                return gobject.ext.as(MountOperation.Class, p_class).f_show_processes.?(gobject.ext.as(MountOperation, p_op), p_message, p_processes, p_choices);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_op: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_message: [*:0]const u8, p_processes: *glib.Array, p_choices: [*][*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(MountOperation.Class, p_class).f_show_processes = @ptrCast(p_implementation);
            }
        };

        pub const show_unmount_progress = struct {
            pub fn call(p_class: anytype, p_op: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_message: [*:0]const u8, p_time_left: i64, p_bytes_left: i64) void {
                return gobject.ext.as(MountOperation.Class, p_class).f_show_unmount_progress.?(gobject.ext.as(MountOperation, p_op), p_message, p_time_left, p_bytes_left);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_op: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_message: [*:0]const u8, p_time_left: i64, p_bytes_left: i64) callconv(.C) void) void {
                gobject.ext.as(MountOperation.Class, p_class).f_show_unmount_progress = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// Whether to use an anonymous user when authenticating.
        pub const anonymous = struct {
            pub const name = "anonymous";

            pub const Type = c_int;
        };

        /// The index of the user's choice when a question is asked during the
        /// mount operation. See the `gio.MountOperation.signals.ask`-question signal.
        pub const choice = struct {
            pub const name = "choice";

            pub const Type = c_int;
        };

        /// The domain to use for the mount operation.
        pub const domain = struct {
            pub const name = "domain";

            pub const Type = ?[*:0]u8;
        };

        /// Whether the device to be unlocked is a TCRYPT hidden volume.
        /// See [the VeraCrypt documentation](https://www.veracrypt.fr/en/Hidden`@"20Volume"`.html).
        pub const is_tcrypt_hidden_volume = struct {
            pub const name = "is-tcrypt-hidden-volume";

            pub const Type = c_int;
        };

        /// Whether the device to be unlocked is a TCRYPT system volume.
        /// In this context, a system volume is a volume with a bootloader
        /// and operating system installed. This is only supported for Windows
        /// operating systems. For further documentation, see
        /// [the VeraCrypt documentation](https://www.veracrypt.fr/en/System`@"20Encryption"`.html).
        pub const is_tcrypt_system_volume = struct {
            pub const name = "is-tcrypt-system-volume";

            pub const Type = c_int;
        };

        /// The password that is used for authentication when carrying out
        /// the mount operation.
        pub const password = struct {
            pub const name = "password";

            pub const Type = ?[*:0]u8;
        };

        /// Determines if and how the password information should be saved.
        pub const password_save = struct {
            pub const name = "password-save";

            pub const Type = gio.PasswordSave;
        };

        /// The VeraCrypt PIM value, when unlocking a VeraCrypt volume. See
        /// [the VeraCrypt documentation](https://www.veracrypt.fr/en/Personal`@"20Iterations"``@"20Multiplier"``@"20"`(PIM).html).
        pub const pim = struct {
            pub const name = "pim";

            pub const Type = c_uint;
        };

        /// The user name that is used for authentication when carrying out
        /// the mount operation.
        pub const username = struct {
            pub const name = "username";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {
        /// Emitted by the backend when e.g. a device becomes unavailable
        /// while a mount operation is in progress.
        ///
        /// Implementations of GMountOperation should handle this signal
        /// by dismissing open password dialogs.
        pub const aborted = struct {
            pub const name = "aborted";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(MountOperation, p_instance))),
                    gobject.signalLookup("aborted", MountOperation.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when a mount operation asks the user for a password.
        ///
        /// If the message contains a line break, the first line should be
        /// presented as a heading. For example, it may be used as the
        /// primary text in a `GtkMessageDialog`.
        pub const ask_password = struct {
            pub const name = "ask-password";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_message: [*:0]u8, p_default_user: [*:0]u8, p_default_domain: [*:0]u8, p_flags: gio.AskPasswordFlags, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(MountOperation, p_instance))),
                    gobject.signalLookup("ask-password", MountOperation.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when asking the user a question and gives a list of
        /// choices for the user to choose from.
        ///
        /// If the message contains a line break, the first line should be
        /// presented as a heading. For example, it may be used as the
        /// primary text in a `GtkMessageDialog`.
        pub const ask_question = struct {
            pub const name = "ask-question";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_message: [*:0]u8, p_choices: [*][*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(MountOperation, p_instance))),
                    gobject.signalLookup("ask-question", MountOperation.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when the user has replied to the mount operation.
        pub const reply = struct {
            pub const name = "reply";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_result: gio.MountOperationResult, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(MountOperation, p_instance))),
                    gobject.signalLookup("reply", MountOperation.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when one or more processes are blocking an operation
        /// e.g. unmounting/ejecting a `gio.Mount` or stopping a `gio.Drive`.
        ///
        /// Note that this signal may be emitted several times to update the
        /// list of blocking processes as processes close files. The
        /// application should only respond with `gio.MountOperation.reply` to
        /// the latest signal (setting `gio.MountOperation.properties.choice` to the choice
        /// the user made).
        ///
        /// If the message contains a line break, the first line should be
        /// presented as a heading. For example, it may be used as the
        /// primary text in a `GtkMessageDialog`.
        pub const show_processes = struct {
            pub const name = "show-processes";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_message: [*:0]u8, p_processes: [*]glib.Pid, p_choices: [*][*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(MountOperation, p_instance))),
                    gobject.signalLookup("show-processes", MountOperation.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Emitted when an unmount operation has been busy for more than some time
        /// (typically 1.5 seconds).
        ///
        /// When unmounting or ejecting a volume, the kernel might need to flush
        /// pending data in its buffers to the volume stable storage, and this operation
        /// can take a considerable amount of time. This signal may be emitted several
        /// times as long as the unmount operation is outstanding, and then one
        /// last time when the operation is completed, with `bytes_left` set to zero.
        ///
        /// Implementations of GMountOperation should handle this signal by
        /// showing an UI notification, and then dismiss it, or show another notification
        /// of completion, when `bytes_left` reaches zero.
        ///
        /// If the message contains a line break, the first line should be
        /// presented as a heading. For example, it may be used as the
        /// primary text in a `GtkMessageDialog`.
        pub const show_unmount_progress = struct {
            pub const name = "show-unmount-progress";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_message: [*:0]u8, p_time_left: i64, p_bytes_left: i64, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(MountOperation, p_instance))),
                    gobject.signalLookup("show-unmount-progress", MountOperation.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new mount operation.
    extern fn g_mount_operation_new() *gio.MountOperation;
    pub const new = g_mount_operation_new;

    /// Check to see whether the mount operation is being used
    /// for an anonymous user.
    extern fn g_mount_operation_get_anonymous(p_op: *MountOperation) c_int;
    pub const getAnonymous = g_mount_operation_get_anonymous;

    /// Gets a choice from the mount operation.
    extern fn g_mount_operation_get_choice(p_op: *MountOperation) c_int;
    pub const getChoice = g_mount_operation_get_choice;

    /// Gets the domain of the mount operation.
    extern fn g_mount_operation_get_domain(p_op: *MountOperation) ?[*:0]const u8;
    pub const getDomain = g_mount_operation_get_domain;

    /// Check to see whether the mount operation is being used
    /// for a TCRYPT hidden volume.
    extern fn g_mount_operation_get_is_tcrypt_hidden_volume(p_op: *MountOperation) c_int;
    pub const getIsTcryptHiddenVolume = g_mount_operation_get_is_tcrypt_hidden_volume;

    /// Check to see whether the mount operation is being used
    /// for a TCRYPT system volume.
    extern fn g_mount_operation_get_is_tcrypt_system_volume(p_op: *MountOperation) c_int;
    pub const getIsTcryptSystemVolume = g_mount_operation_get_is_tcrypt_system_volume;

    /// Gets a password from the mount operation.
    extern fn g_mount_operation_get_password(p_op: *MountOperation) ?[*:0]const u8;
    pub const getPassword = g_mount_operation_get_password;

    /// Gets the state of saving passwords for the mount operation.
    extern fn g_mount_operation_get_password_save(p_op: *MountOperation) gio.PasswordSave;
    pub const getPasswordSave = g_mount_operation_get_password_save;

    /// Gets a PIM from the mount operation.
    extern fn g_mount_operation_get_pim(p_op: *MountOperation) c_uint;
    pub const getPim = g_mount_operation_get_pim;

    /// Get the user name from the mount operation.
    extern fn g_mount_operation_get_username(p_op: *MountOperation) ?[*:0]const u8;
    pub const getUsername = g_mount_operation_get_username;

    /// Emits the `gio.MountOperation.signals.reply` signal.
    extern fn g_mount_operation_reply(p_op: *MountOperation, p_result: gio.MountOperationResult) void;
    pub const reply = g_mount_operation_reply;

    /// Sets the mount operation to use an anonymous user if `anonymous` is `TRUE`.
    extern fn g_mount_operation_set_anonymous(p_op: *MountOperation, p_anonymous: c_int) void;
    pub const setAnonymous = g_mount_operation_set_anonymous;

    /// Sets a default choice for the mount operation.
    extern fn g_mount_operation_set_choice(p_op: *MountOperation, p_choice: c_int) void;
    pub const setChoice = g_mount_operation_set_choice;

    /// Sets the mount operation's domain.
    extern fn g_mount_operation_set_domain(p_op: *MountOperation, p_domain: ?[*:0]const u8) void;
    pub const setDomain = g_mount_operation_set_domain;

    /// Sets the mount operation to use a hidden volume if `hidden_volume` is `TRUE`.
    extern fn g_mount_operation_set_is_tcrypt_hidden_volume(p_op: *MountOperation, p_hidden_volume: c_int) void;
    pub const setIsTcryptHiddenVolume = g_mount_operation_set_is_tcrypt_hidden_volume;

    /// Sets the mount operation to use a system volume if `system_volume` is `TRUE`.
    extern fn g_mount_operation_set_is_tcrypt_system_volume(p_op: *MountOperation, p_system_volume: c_int) void;
    pub const setIsTcryptSystemVolume = g_mount_operation_set_is_tcrypt_system_volume;

    /// Sets the mount operation's password to `password`.
    extern fn g_mount_operation_set_password(p_op: *MountOperation, p_password: ?[*:0]const u8) void;
    pub const setPassword = g_mount_operation_set_password;

    /// Sets the state of saving passwords for the mount operation.
    extern fn g_mount_operation_set_password_save(p_op: *MountOperation, p_save: gio.PasswordSave) void;
    pub const setPasswordSave = g_mount_operation_set_password_save;

    /// Sets the mount operation's PIM to `pim`.
    extern fn g_mount_operation_set_pim(p_op: *MountOperation, p_pim: c_uint) void;
    pub const setPim = g_mount_operation_set_pim;

    /// Sets the user name within `op` to `username`.
    extern fn g_mount_operation_set_username(p_op: *MountOperation, p_username: ?[*:0]const u8) void;
    pub const setUsername = g_mount_operation_set_username;

    extern fn g_mount_operation_get_type() usize;
    pub const getGObjectType = g_mount_operation_get_type;

    extern fn g_object_ref(p_self: *gio.MountOperation) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.MountOperation) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *MountOperation, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A socket address of some unknown native type.
///
/// This corresponds to a general `struct sockaddr` of a type not otherwise
/// handled by GLib.
pub const NativeSocketAddress = extern struct {
    pub const Parent = gio.SocketAddress;
    pub const Implements = [_]type{gio.SocketConnectable};
    pub const Class = gio.NativeSocketAddressClass;
    f_parent_instance: gio.SocketAddress,
    f_priv: ?*gio.NativeSocketAddressPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `gio.NativeSocketAddress` for `native` and `len`.
    extern fn g_native_socket_address_new(p_native: ?*anyopaque, p_len: usize) *gio.NativeSocketAddress;
    pub const new = g_native_socket_address_new;

    extern fn g_native_socket_address_get_type() usize;
    pub const getGObjectType = g_native_socket_address_get_type;

    extern fn g_object_ref(p_self: *gio.NativeSocketAddress) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.NativeSocketAddress) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *NativeSocketAddress, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const NativeVolumeMonitor = extern struct {
    pub const Parent = gio.VolumeMonitor;
    pub const Implements = [_]type{};
    pub const Class = gio.NativeVolumeMonitorClass;
    f_parent_instance: gio.VolumeMonitor,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    extern fn g_native_volume_monitor_get_type() usize;
    pub const getGObjectType = g_native_volume_monitor_get_type;

    extern fn g_object_ref(p_self: *gio.NativeVolumeMonitor) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.NativeVolumeMonitor) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *NativeVolumeMonitor, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GNetworkAddress` provides an easy way to resolve a hostname and
/// then attempt to connect to that host, handling the possibility of
/// multiple IP addresses and multiple address families.
///
/// The enumeration results of resolved addresses *may* be cached as long
/// as this object is kept alive which may have unexpected results if
/// alive for too long.
///
/// See `gio.SocketConnectable` for an example of using the connectable
/// interface.
pub const NetworkAddress = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gio.SocketConnectable};
    pub const Class = gio.NetworkAddressClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.NetworkAddressPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Hostname to resolve.
        pub const hostname = struct {
            pub const name = "hostname";

            pub const Type = ?[*:0]u8;
        };

        /// Network port.
        pub const port = struct {
            pub const name = "port";

            pub const Type = c_uint;
        };

        /// URI scheme.
        pub const scheme = struct {
            pub const name = "scheme";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gio.SocketConnectable` for connecting to the given
    /// `hostname` and `port`. May fail and return `NULL` in case
    /// parsing `host_and_port` fails.
    ///
    /// `host_and_port` may be in any of a number of recognised formats; an IPv6
    /// address, an IPv4 address, or a domain name (in which case a DNS
    /// lookup is performed). Quoting with [] is supported for all address
    /// types. A port override may be specified in the usual way with a
    /// colon.
    ///
    /// If no port is specified in `host_and_port` then `default_port` will be
    /// used as the port number to connect to.
    ///
    /// In general, `host_and_port` is expected to be provided by the user
    /// (allowing them to give the hostname, and a port override if necessary)
    /// and `default_port` is expected to be provided by the application.
    ///
    /// (The port component of `host_and_port` can also be specified as a
    /// service name rather than as a numeric port, but this functionality
    /// is deprecated, because it depends on the contents of /etc/services,
    /// which is generally quite sparse on platforms other than Linux.)
    extern fn g_network_address_parse(p_host_and_port: [*:0]const u8, p_default_port: u16, p_error: ?*?*glib.Error) ?*gio.NetworkAddress;
    pub const parse = g_network_address_parse;

    /// Creates a new `gio.SocketConnectable` for connecting to the given
    /// `uri`. May fail and return `NULL` in case parsing `uri` fails.
    ///
    /// Using this rather than `gio.NetworkAddress.new` or
    /// `gio.NetworkAddress.parse` allows `gio.SocketClient` to determine
    /// when to use application-specific proxy protocols.
    extern fn g_network_address_parse_uri(p_uri: [*:0]const u8, p_default_port: u16, p_error: ?*?*glib.Error) ?*gio.NetworkAddress;
    pub const parseUri = g_network_address_parse_uri;

    /// Creates a new `gio.SocketConnectable` for connecting to the given
    /// `hostname` and `port`.
    ///
    /// Note that depending on the configuration of the machine, a
    /// `hostname` of `localhost` may refer to the IPv4 loopback address
    /// only, or to both IPv4 and IPv6; use
    /// `gio.NetworkAddress.newLoopback` to create a `gio.NetworkAddress` that
    /// is guaranteed to resolve to both addresses.
    extern fn g_network_address_new(p_hostname: [*:0]const u8, p_port: u16) *gio.NetworkAddress;
    pub const new = g_network_address_new;

    /// Creates a new `gio.SocketConnectable` for connecting to the local host
    /// over a loopback connection to the given `port`. This is intended for
    /// use in connecting to local services which may be running on IPv4 or
    /// IPv6.
    ///
    /// The connectable will return IPv4 and IPv6 loopback addresses,
    /// regardless of how the host resolves `localhost`. By contrast,
    /// `gio.NetworkAddress.new` will often only return an IPv4 address when
    /// resolving `localhost`, and an IPv6 address for `localhost6`.
    ///
    /// `gio.NetworkAddress.getHostname` will always return `localhost` for
    /// a `gio.NetworkAddress` created with this constructor.
    extern fn g_network_address_new_loopback(p_port: u16) *gio.NetworkAddress;
    pub const newLoopback = g_network_address_new_loopback;

    /// Gets `addr`'s hostname. This might be either UTF-8 or ASCII-encoded,
    /// depending on what `addr` was created with.
    extern fn g_network_address_get_hostname(p_addr: *NetworkAddress) [*:0]const u8;
    pub const getHostname = g_network_address_get_hostname;

    /// Gets `addr`'s port number
    extern fn g_network_address_get_port(p_addr: *NetworkAddress) u16;
    pub const getPort = g_network_address_get_port;

    /// Gets `addr`'s scheme
    extern fn g_network_address_get_scheme(p_addr: *NetworkAddress) ?[*:0]const u8;
    pub const getScheme = g_network_address_get_scheme;

    extern fn g_network_address_get_type() usize;
    pub const getGObjectType = g_network_address_get_type;

    extern fn g_object_ref(p_self: *gio.NetworkAddress) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.NetworkAddress) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *NetworkAddress, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Like `gio.NetworkAddress` does with hostnames, `GNetworkService`
/// provides an easy way to resolve a SRV record, and then attempt to
/// connect to one of the hosts that implements that service, handling
/// service priority/weighting, multiple IP addresses, and multiple
/// address families.
///
/// See `gio.SrvTarget` for more information about SRV records, and see
/// `gio.SocketConnectable` for an example of using the connectable
/// interface.
pub const NetworkService = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gio.SocketConnectable};
    pub const Class = gio.NetworkServiceClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.NetworkServicePrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Network domain, for example `example.com`.
        pub const domain = struct {
            pub const name = "domain";

            pub const Type = ?[*:0]u8;
        };

        /// Network protocol, for example `tcp`.
        pub const protocol = struct {
            pub const name = "protocol";

            pub const Type = ?[*:0]u8;
        };

        /// Network scheme (default is to use service).
        pub const scheme = struct {
            pub const name = "scheme";

            pub const Type = ?[*:0]u8;
        };

        /// Service name, for example `ldap`.
        pub const service = struct {
            pub const name = "service";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gio.NetworkService` representing the given `service`,
    /// `protocol`, and `domain`. This will initially be unresolved; use the
    /// `gio.SocketConnectable` interface to resolve it.
    extern fn g_network_service_new(p_service: [*:0]const u8, p_protocol: [*:0]const u8, p_domain: [*:0]const u8) *gio.NetworkService;
    pub const new = g_network_service_new;

    /// Gets the domain that `srv` serves. This might be either UTF-8 or
    /// ASCII-encoded, depending on what `srv` was created with.
    extern fn g_network_service_get_domain(p_srv: *NetworkService) [*:0]const u8;
    pub const getDomain = g_network_service_get_domain;

    /// Gets `srv`'s protocol name (eg, "tcp").
    extern fn g_network_service_get_protocol(p_srv: *NetworkService) [*:0]const u8;
    pub const getProtocol = g_network_service_get_protocol;

    /// Gets the URI scheme used to resolve proxies. By default, the service name
    /// is used as scheme.
    extern fn g_network_service_get_scheme(p_srv: *NetworkService) [*:0]const u8;
    pub const getScheme = g_network_service_get_scheme;

    /// Gets `srv`'s service name (eg, "ldap").
    extern fn g_network_service_get_service(p_srv: *NetworkService) [*:0]const u8;
    pub const getService = g_network_service_get_service;

    /// Set's the URI scheme used to resolve proxies. By default, the service name
    /// is used as scheme.
    extern fn g_network_service_set_scheme(p_srv: *NetworkService, p_scheme: [*:0]const u8) void;
    pub const setScheme = g_network_service_set_scheme;

    extern fn g_network_service_get_type() usize;
    pub const getGObjectType = g_network_service_get_type;

    extern fn g_object_ref(p_self: *gio.NetworkService) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.NetworkService) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *NetworkService, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GNotification` is a mechanism for creating a notification to be shown
/// to the user — typically as a pop-up notification presented by the
/// desktop environment shell.
///
/// The key difference between `GNotification` and other similar APIs is
/// that, if supported by the desktop environment, notifications sent
/// with `GNotification` will persist after the application has exited,
/// and even across system reboots.
///
/// Since the user may click on a notification while the application is
/// not running, applications using `GNotification` should be able to be
/// started as a D-Bus service, using `gio.Application`.
///
/// In order for `GNotification` to work, the application must have installed
/// a `.desktop` file. For example:
/// ```
/// [Desktop Entry]
/// Name=Test Application
/// Comment=Description of what Test Application does
/// Exec=gnome-test-application
/// Icon=org.gnome.TestApplication
/// Terminal=false
/// Type=Application
/// Categories=GNOME;GTK;TestApplication Category;
/// StartupNotify=true
/// DBusActivatable=true
/// X-GNOME-UsesNotifications=true
/// ```
///
/// The `X-GNOME-UsesNotifications` key indicates to GNOME Control Center
/// that this application uses notifications, so it can be listed in the
/// Control Center’s ‘Notifications’ panel.
///
/// The `.desktop` file must be named as `org.gnome.TestApplication.desktop`,
/// where `org.gnome.TestApplication` is the ID passed to
/// `gio.Application.new`.
///
/// User interaction with a notification (either the default action, or
/// buttons) must be associated with actions on the application (ie:
/// `app.` actions).  It is not possible to route user interaction
/// through the notification itself, because the object will not exist if
/// the application is autostarted as a result of a notification being
/// clicked.
///
/// A notification can be sent with `gio.Application.sendNotification`.
pub const Notification = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = opaque {
        pub const Instance = Notification;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `gio.Notification` with `title` as its title.
    ///
    /// After populating `notification` with more details, it can be sent to
    /// the desktop shell with `gio.Application.sendNotification`. Changing
    /// any properties after this call will not have any effect until
    /// resending `notification`.
    extern fn g_notification_new(p_title: [*:0]const u8) *gio.Notification;
    pub const new = g_notification_new;

    /// Adds a button to `notification` that activates the action in
    /// `detailed_action` when clicked. That action must be an
    /// application-wide action (starting with "app."). If `detailed_action`
    /// contains a target, the action will be activated with that target as
    /// its parameter.
    ///
    /// See `gio.actionParseDetailedName` for a description of the format
    /// for `detailed_action`.
    extern fn g_notification_add_button(p_notification: *Notification, p_label: [*:0]const u8, p_detailed_action: [*:0]const u8) void;
    pub const addButton = g_notification_add_button;

    /// Adds a button to `notification` that activates `action` when clicked.
    /// `action` must be an application-wide action (it must start with "app.").
    ///
    /// If `target_format` is given, it is used to collect remaining
    /// positional parameters into a `glib.Variant` instance, similar to
    /// `glib.Variant.new`. `action` will be activated with that `glib.Variant` as its
    /// parameter.
    extern fn g_notification_add_button_with_target(p_notification: *Notification, p_label: [*:0]const u8, p_action: [*:0]const u8, p_target_format: ?[*:0]const u8, ...) void;
    pub const addButtonWithTarget = g_notification_add_button_with_target;

    /// Adds a button to `notification` that activates `action` when clicked.
    /// `action` must be an application-wide action (it must start with "app.").
    ///
    /// If `target` is non-`NULL`, `action` will be activated with `target` as
    /// its parameter.
    extern fn g_notification_add_button_with_target_value(p_notification: *Notification, p_label: [*:0]const u8, p_action: [*:0]const u8, p_target: ?*glib.Variant) void;
    pub const addButtonWithTargetValue = g_notification_add_button_with_target_value;

    /// Sets the body of `notification` to `body`.
    extern fn g_notification_set_body(p_notification: *Notification, p_body: ?[*:0]const u8) void;
    pub const setBody = g_notification_set_body;

    /// Sets the type of `notification` to `category`. Categories have a main
    /// type like `email`, `im` or `device` and can have a detail separated
    /// by a `.`, e.g. `im.received` or `email.arrived`. Setting the category
    /// helps the notification server to select proper feedback to the user.
    ///
    /// Standard categories are [listed in the specification](https://specifications.freedesktop.org/notification-spec/latest/ar01s06.html).
    extern fn g_notification_set_category(p_notification: *Notification, p_category: ?[*:0]const u8) void;
    pub const setCategory = g_notification_set_category;

    /// Sets the default action of `notification` to `detailed_action`. This
    /// action is activated when the notification is clicked on.
    ///
    /// The action in `detailed_action` must be an application-wide action (it
    /// must start with "app."). If `detailed_action` contains a target, the
    /// given action will be activated with that target as its parameter.
    /// See `gio.actionParseDetailedName` for a description of the format
    /// for `detailed_action`.
    ///
    /// When no default action is set, the application that the notification
    /// was sent on is activated.
    extern fn g_notification_set_default_action(p_notification: *Notification, p_detailed_action: [*:0]const u8) void;
    pub const setDefaultAction = g_notification_set_default_action;

    /// Sets the default action of `notification` to `action`. This action is
    /// activated when the notification is clicked on. It must be an
    /// application-wide action (it must start with "app.").
    ///
    /// If `target_format` is given, it is used to collect remaining
    /// positional parameters into a `glib.Variant` instance, similar to
    /// `glib.Variant.new`. `action` will be activated with that `glib.Variant` as its
    /// parameter.
    ///
    /// When no default action is set, the application that the notification
    /// was sent on is activated.
    extern fn g_notification_set_default_action_and_target(p_notification: *Notification, p_action: [*:0]const u8, p_target_format: ?[*:0]const u8, ...) void;
    pub const setDefaultActionAndTarget = g_notification_set_default_action_and_target;

    /// Sets the default action of `notification` to `action`. This action is
    /// activated when the notification is clicked on. It must be an
    /// application-wide action (start with "app.").
    ///
    /// If `target` is non-`NULL`, `action` will be activated with `target` as
    /// its parameter. If `target` is floating, it will be consumed.
    ///
    /// When no default action is set, the application that the notification
    /// was sent on is activated.
    extern fn g_notification_set_default_action_and_target_value(p_notification: *Notification, p_action: [*:0]const u8, p_target: ?*glib.Variant) void;
    pub const setDefaultActionAndTargetValue = g_notification_set_default_action_and_target_value;

    /// Sets the icon of `notification` to `icon`.
    extern fn g_notification_set_icon(p_notification: *Notification, p_icon: *gio.Icon) void;
    pub const setIcon = g_notification_set_icon;

    /// Sets the priority of `notification` to `priority`. See
    /// `gio.NotificationPriority` for possible values.
    extern fn g_notification_set_priority(p_notification: *Notification, p_priority: gio.NotificationPriority) void;
    pub const setPriority = g_notification_set_priority;

    /// Sets the title of `notification` to `title`.
    extern fn g_notification_set_title(p_notification: *Notification, p_title: [*:0]const u8) void;
    pub const setTitle = g_notification_set_title;

    /// Deprecated in favor of `gio.Notification.setPriority`.
    extern fn g_notification_set_urgent(p_notification: *Notification, p_urgent: c_int) void;
    pub const setUrgent = g_notification_set_urgent;

    extern fn g_notification_get_type() usize;
    pub const getGObjectType = g_notification_get_type;

    extern fn g_object_ref(p_self: *gio.Notification) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.Notification) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Notification, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GOutputStream` is a base class for implementing streaming output.
///
/// It has functions to write to a stream (`gio.OutputStream.write`),
/// to close a stream (`gio.OutputStream.close`) and to flush pending
/// writes (`gio.OutputStream.flush`).
///
/// To copy the content of an input stream to an output stream without
/// manually handling the reads and writes, use `gio.OutputStream.splice`.
///
/// See the documentation for `gio.IOStream` for details of thread safety
/// of streaming APIs.
///
/// All of these functions have async variants too.
///
/// All classes derived from `GOutputStream` *should* implement synchronous
/// writing, splicing, flushing and closing streams, but *may* implement
/// asynchronous versions.
pub const OutputStream = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.OutputStreamClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.OutputStreamPrivate,

    pub const virtual_methods = struct {
        /// Requests an asynchronous close of the stream, releasing resources
        /// related to it. When the operation is finished `callback` will be
        /// called. You can then call `gio.OutputStream.closeFinish` to get
        /// the result of the operation.
        ///
        /// For behaviour details see `gio.OutputStream.close`.
        ///
        /// The asynchronous methods have a default fallback that uses threads
        /// to implement asynchronicity, so they are optional for inheriting
        /// classes. However, if you override one you must override all.
        pub const close_async = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(OutputStream.Class, p_class).f_close_async.?(gobject.ext.as(OutputStream, p_stream), p_io_priority, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(OutputStream.Class, p_class).f_close_async = @ptrCast(p_implementation);
            }
        };

        /// Closes an output stream.
        pub const close_finish = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(OutputStream.Class, p_class).f_close_finish.?(gobject.ext.as(OutputStream, p_stream), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(OutputStream.Class, p_class).f_close_finish = @ptrCast(p_implementation);
            }
        };

        pub const close_fn = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(OutputStream.Class, p_class).f_close_fn.?(gobject.ext.as(OutputStream, p_stream), p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(OutputStream.Class, p_class).f_close_fn = @ptrCast(p_implementation);
            }
        };

        /// Forces a write of all user-space buffered data for the given
        /// `stream`. Will block during the operation. Closing the stream will
        /// implicitly cause a flush.
        ///
        /// This function is optional for inherited classes.
        ///
        /// If `cancellable` is not `NULL`, then the operation can be cancelled by
        /// triggering the cancellable object from another thread. If the operation
        /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned.
        pub const flush = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(OutputStream.Class, p_class).f_flush.?(gobject.ext.as(OutputStream, p_stream), p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(OutputStream.Class, p_class).f_flush = @ptrCast(p_implementation);
            }
        };

        /// Forces an asynchronous write of all user-space buffered data for
        /// the given `stream`.
        /// For behaviour details see `gio.OutputStream.flush`.
        ///
        /// When the operation is finished `callback` will be
        /// called. You can then call `gio.OutputStream.flushFinish` to get the
        /// result of the operation.
        pub const flush_async = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(OutputStream.Class, p_class).f_flush_async.?(gobject.ext.as(OutputStream, p_stream), p_io_priority, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(OutputStream.Class, p_class).f_flush_async = @ptrCast(p_implementation);
            }
        };

        /// Finishes flushing an output stream.
        pub const flush_finish = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(OutputStream.Class, p_class).f_flush_finish.?(gobject.ext.as(OutputStream, p_stream), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(OutputStream.Class, p_class).f_flush_finish = @ptrCast(p_implementation);
            }
        };

        /// Splices an input stream into an output stream.
        pub const splice = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_source: *gio.InputStream, p_flags: gio.OutputStreamSpliceFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) isize {
                return gobject.ext.as(OutputStream.Class, p_class).f_splice.?(gobject.ext.as(OutputStream, p_stream), p_source, p_flags, p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_source: *gio.InputStream, p_flags: gio.OutputStreamSpliceFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) isize) void {
                gobject.ext.as(OutputStream.Class, p_class).f_splice = @ptrCast(p_implementation);
            }
        };

        /// Splices a stream asynchronously.
        /// When the operation is finished `callback` will be called.
        /// You can then call `gio.OutputStream.spliceFinish` to get the
        /// result of the operation.
        ///
        /// For the synchronous, blocking version of this function, see
        /// `gio.OutputStream.splice`.
        pub const splice_async = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_source: *gio.InputStream, p_flags: gio.OutputStreamSpliceFlags, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(OutputStream.Class, p_class).f_splice_async.?(gobject.ext.as(OutputStream, p_stream), p_source, p_flags, p_io_priority, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_source: *gio.InputStream, p_flags: gio.OutputStreamSpliceFlags, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(OutputStream.Class, p_class).f_splice_async = @ptrCast(p_implementation);
            }
        };

        /// Finishes an asynchronous stream splice operation.
        pub const splice_finish = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) isize {
                return gobject.ext.as(OutputStream.Class, p_class).f_splice_finish.?(gobject.ext.as(OutputStream, p_stream), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) isize) void {
                gobject.ext.as(OutputStream.Class, p_class).f_splice_finish = @ptrCast(p_implementation);
            }
        };

        /// Request an asynchronous write of `count` bytes from `buffer` into
        /// the stream. When the operation is finished `callback` will be called.
        /// You can then call `gio.OutputStream.writeFinish` to get the result of the
        /// operation.
        ///
        /// During an async request no other sync and async calls are allowed,
        /// and will result in `G_IO_ERROR_PENDING` errors.
        ///
        /// A value of `count` larger than `G_MAXSSIZE` will cause a
        /// `G_IO_ERROR_INVALID_ARGUMENT` error.
        ///
        /// On success, the number of bytes written will be passed to the
        /// `callback`. It is not an error if this is not the same as the
        /// requested size, as it can happen e.g. on a partial I/O error,
        /// but generally we try to write as many bytes as requested.
        ///
        /// You are guaranteed that this method will never fail with
        /// `G_IO_ERROR_WOULD_BLOCK` - if `stream` can't accept more data, the
        /// method will just wait until this changes.
        ///
        /// Any outstanding I/O request with higher priority (lower numerical
        /// value) will be executed before an outstanding request with lower
        /// priority. Default priority is `G_PRIORITY_DEFAULT`.
        ///
        /// The asynchronous methods have a default fallback that uses threads
        /// to implement asynchronicity, so they are optional for inheriting
        /// classes. However, if you override one you must override all.
        ///
        /// For the synchronous, blocking version of this function, see
        /// `gio.OutputStream.write`.
        ///
        /// Note that no copy of `buffer` will be made, so it must stay valid
        /// until `callback` is called. See `gio.OutputStream.writeBytesAsync`
        /// for a `glib.Bytes` version that will automatically hold a reference to
        /// the contents (without copying) for the duration of the call.
        pub const write_async = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_buffer: ?[*]u8, p_count: usize, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(OutputStream.Class, p_class).f_write_async.?(gobject.ext.as(OutputStream, p_stream), p_buffer, p_count, p_io_priority, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_buffer: ?[*]u8, p_count: usize, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(OutputStream.Class, p_class).f_write_async = @ptrCast(p_implementation);
            }
        };

        /// Finishes a stream write operation.
        pub const write_finish = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) isize {
                return gobject.ext.as(OutputStream.Class, p_class).f_write_finish.?(gobject.ext.as(OutputStream, p_stream), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) isize) void {
                gobject.ext.as(OutputStream.Class, p_class).f_write_finish = @ptrCast(p_implementation);
            }
        };

        /// Tries to write `count` bytes from `buffer` into the stream. Will block
        /// during the operation.
        ///
        /// If count is 0, returns 0 and does nothing. A value of `count`
        /// larger than `G_MAXSSIZE` will cause a `G_IO_ERROR_INVALID_ARGUMENT` error.
        ///
        /// On success, the number of bytes written to the stream is returned.
        /// It is not an error if this is not the same as the requested size, as it
        /// can happen e.g. on a partial I/O error, or if there is not enough
        /// storage in the stream. All writes block until at least one byte
        /// is written or an error occurs; 0 is never returned (unless
        /// `count` is 0).
        ///
        /// If `cancellable` is not `NULL`, then the operation can be cancelled by
        /// triggering the cancellable object from another thread. If the operation
        /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned. If an
        /// operation was partially finished when the operation was cancelled the
        /// partial result will be returned, without an error.
        ///
        /// On error -1 is returned and `error` is set accordingly.
        pub const write_fn = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_buffer: ?[*]u8, p_count: usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) isize {
                return gobject.ext.as(OutputStream.Class, p_class).f_write_fn.?(gobject.ext.as(OutputStream, p_stream), p_buffer, p_count, p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_buffer: ?[*]u8, p_count: usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) isize) void {
                gobject.ext.as(OutputStream.Class, p_class).f_write_fn = @ptrCast(p_implementation);
            }
        };

        /// Request an asynchronous write of the bytes contained in `n_vectors` `vectors` into
        /// the stream. When the operation is finished `callback` will be called.
        /// You can then call `gio.OutputStream.writevFinish` to get the result of the
        /// operation.
        ///
        /// During an async request no other sync and async calls are allowed,
        /// and will result in `G_IO_ERROR_PENDING` errors.
        ///
        /// On success, the number of bytes written will be passed to the
        /// `callback`. It is not an error if this is not the same as the
        /// requested size, as it can happen e.g. on a partial I/O error,
        /// but generally we try to write as many bytes as requested.
        ///
        /// You are guaranteed that this method will never fail with
        /// `G_IO_ERROR_WOULD_BLOCK` — if `stream` can't accept more data, the
        /// method will just wait until this changes.
        ///
        /// Any outstanding I/O request with higher priority (lower numerical
        /// value) will be executed before an outstanding request with lower
        /// priority. Default priority is `G_PRIORITY_DEFAULT`.
        ///
        /// The asynchronous methods have a default fallback that uses threads
        /// to implement asynchronicity, so they are optional for inheriting
        /// classes. However, if you override one you must override all.
        ///
        /// For the synchronous, blocking version of this function, see
        /// `gio.OutputStream.writev`.
        ///
        /// Note that no copy of `vectors` will be made, so it must stay valid
        /// until `callback` is called.
        pub const writev_async = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_vectors: [*]const gio.OutputVector, p_n_vectors: usize, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(OutputStream.Class, p_class).f_writev_async.?(gobject.ext.as(OutputStream, p_stream), p_vectors, p_n_vectors, p_io_priority, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_vectors: [*]const gio.OutputVector, p_n_vectors: usize, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(OutputStream.Class, p_class).f_writev_async = @ptrCast(p_implementation);
            }
        };

        /// Finishes a stream writev operation.
        pub const writev_finish = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_bytes_written: ?*usize, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(OutputStream.Class, p_class).f_writev_finish.?(gobject.ext.as(OutputStream, p_stream), p_result, p_bytes_written, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_bytes_written: ?*usize, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(OutputStream.Class, p_class).f_writev_finish = @ptrCast(p_implementation);
            }
        };

        /// Tries to write the bytes contained in the `n_vectors` `vectors` into the
        /// stream. Will block during the operation.
        ///
        /// If `n_vectors` is 0 or the sum of all bytes in `vectors` is 0, returns 0 and
        /// does nothing.
        ///
        /// On success, the number of bytes written to the stream is returned.
        /// It is not an error if this is not the same as the requested size, as it
        /// can happen e.g. on a partial I/O error, or if there is not enough
        /// storage in the stream. All writes block until at least one byte
        /// is written or an error occurs; 0 is never returned (unless
        /// `n_vectors` is 0 or the sum of all bytes in `vectors` is 0).
        ///
        /// If `cancellable` is not `NULL`, then the operation can be cancelled by
        /// triggering the cancellable object from another thread. If the operation
        /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned. If an
        /// operation was partially finished when the operation was cancelled the
        /// partial result will be returned, without an error.
        ///
        /// Some implementations of `gio.OutputStream.writev` may have limitations on the
        /// aggregate buffer size, and will return `G_IO_ERROR_INVALID_ARGUMENT` if these
        /// are exceeded. For example, when writing to a local file on UNIX platforms,
        /// the aggregate buffer size must not exceed `G_MAXSSIZE` bytes.
        pub const writev_fn = struct {
            pub fn call(p_class: anytype, p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_vectors: [*]const gio.OutputVector, p_n_vectors: usize, p_bytes_written: ?*usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(OutputStream.Class, p_class).f_writev_fn.?(gobject.ext.as(OutputStream, p_stream), p_vectors, p_n_vectors, p_bytes_written, p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_stream: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_vectors: [*]const gio.OutputVector, p_n_vectors: usize, p_bytes_written: ?*usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(OutputStream.Class, p_class).f_writev_fn = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {};

    /// Clears the pending flag on `stream`.
    extern fn g_output_stream_clear_pending(p_stream: *OutputStream) void;
    pub const clearPending = g_output_stream_clear_pending;

    /// Closes the stream, releasing resources related to it.
    ///
    /// Once the stream is closed, all other operations will return `G_IO_ERROR_CLOSED`.
    /// Closing a stream multiple times will not return an error.
    ///
    /// Closing a stream will automatically flush any outstanding buffers in the
    /// stream.
    ///
    /// Streams will be automatically closed when the last reference
    /// is dropped, but you might want to call this function to make sure
    /// resources are released as early as possible.
    ///
    /// Some streams might keep the backing store of the stream (e.g. a file descriptor)
    /// open after the stream is closed. See the documentation for the individual
    /// stream for details.
    ///
    /// On failure the first error that happened will be reported, but the close
    /// operation will finish as much as possible. A stream that failed to
    /// close will still return `G_IO_ERROR_CLOSED` for all operations. Still, it
    /// is important to check and report the error to the user, otherwise
    /// there might be a loss of data as all data might not be written.
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned.
    /// Cancelling a close will still leave the stream closed, but there some streams
    /// can use a faster close that doesn't block to e.g. check errors. On
    /// cancellation (as with any error) there is no guarantee that all written
    /// data will reach the target.
    extern fn g_output_stream_close(p_stream: *OutputStream, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const close = g_output_stream_close;

    /// Requests an asynchronous close of the stream, releasing resources
    /// related to it. When the operation is finished `callback` will be
    /// called. You can then call `gio.OutputStream.closeFinish` to get
    /// the result of the operation.
    ///
    /// For behaviour details see `gio.OutputStream.close`.
    ///
    /// The asynchronous methods have a default fallback that uses threads
    /// to implement asynchronicity, so they are optional for inheriting
    /// classes. However, if you override one you must override all.
    extern fn g_output_stream_close_async(p_stream: *OutputStream, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const closeAsync = g_output_stream_close_async;

    /// Closes an output stream.
    extern fn g_output_stream_close_finish(p_stream: *OutputStream, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const closeFinish = g_output_stream_close_finish;

    /// Forces a write of all user-space buffered data for the given
    /// `stream`. Will block during the operation. Closing the stream will
    /// implicitly cause a flush.
    ///
    /// This function is optional for inherited classes.
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned.
    extern fn g_output_stream_flush(p_stream: *OutputStream, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const flush = g_output_stream_flush;

    /// Forces an asynchronous write of all user-space buffered data for
    /// the given `stream`.
    /// For behaviour details see `gio.OutputStream.flush`.
    ///
    /// When the operation is finished `callback` will be
    /// called. You can then call `gio.OutputStream.flushFinish` to get the
    /// result of the operation.
    extern fn g_output_stream_flush_async(p_stream: *OutputStream, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const flushAsync = g_output_stream_flush_async;

    /// Finishes flushing an output stream.
    extern fn g_output_stream_flush_finish(p_stream: *OutputStream, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const flushFinish = g_output_stream_flush_finish;

    /// Checks if an output stream has pending actions.
    extern fn g_output_stream_has_pending(p_stream: *OutputStream) c_int;
    pub const hasPending = g_output_stream_has_pending;

    /// Checks if an output stream has already been closed.
    extern fn g_output_stream_is_closed(p_stream: *OutputStream) c_int;
    pub const isClosed = g_output_stream_is_closed;

    /// Checks if an output stream is being closed. This can be
    /// used inside e.g. a flush implementation to see if the
    /// flush (or other i/o operation) is called from within
    /// the closing operation.
    extern fn g_output_stream_is_closing(p_stream: *OutputStream) c_int;
    pub const isClosing = g_output_stream_is_closing;

    /// This is a utility function around `gio.OutputStream.writeAll`. It
    /// uses `glib.strdupVprintf` to turn `format` and @... into a string that
    /// is then written to `stream`.
    ///
    /// See the documentation of `gio.OutputStream.writeAll` about the
    /// behavior of the actual write operation.
    ///
    /// Note that partial writes cannot be properly checked with this
    /// function due to the variable length of the written string, if you
    /// need precise control over partial write failures, you need to
    /// create you own `printf`-like wrapper around `gio.OutputStream.write`
    /// or `gio.OutputStream.writeAll`.
    extern fn g_output_stream_printf(p_stream: *OutputStream, p_bytes_written: ?*usize, p_cancellable: ?*gio.Cancellable, p_error: **glib.Error, p_format: [*:0]const u8, ...) c_int;
    pub const printf = g_output_stream_printf;

    /// Sets `stream` to have actions pending. If the pending flag is
    /// already set or `stream` is closed, it will return `FALSE` and set
    /// `error`.
    extern fn g_output_stream_set_pending(p_stream: *OutputStream, p_error: ?*?*glib.Error) c_int;
    pub const setPending = g_output_stream_set_pending;

    /// Splices an input stream into an output stream.
    extern fn g_output_stream_splice(p_stream: *OutputStream, p_source: *gio.InputStream, p_flags: gio.OutputStreamSpliceFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) isize;
    pub const splice = g_output_stream_splice;

    /// Splices a stream asynchronously.
    /// When the operation is finished `callback` will be called.
    /// You can then call `gio.OutputStream.spliceFinish` to get the
    /// result of the operation.
    ///
    /// For the synchronous, blocking version of this function, see
    /// `gio.OutputStream.splice`.
    extern fn g_output_stream_splice_async(p_stream: *OutputStream, p_source: *gio.InputStream, p_flags: gio.OutputStreamSpliceFlags, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const spliceAsync = g_output_stream_splice_async;

    /// Finishes an asynchronous stream splice operation.
    extern fn g_output_stream_splice_finish(p_stream: *OutputStream, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) isize;
    pub const spliceFinish = g_output_stream_splice_finish;

    /// This is a utility function around `gio.OutputStream.writeAll`. It
    /// uses `glib.strdupVprintf` to turn `format` and `args` into a string that
    /// is then written to `stream`.
    ///
    /// See the documentation of `gio.OutputStream.writeAll` about the
    /// behavior of the actual write operation.
    ///
    /// Note that partial writes cannot be properly checked with this
    /// function due to the variable length of the written string, if you
    /// need precise control over partial write failures, you need to
    /// create you own `printf`-like wrapper around `gio.OutputStream.write`
    /// or `gio.OutputStream.writeAll`.
    extern fn g_output_stream_vprintf(p_stream: *OutputStream, p_bytes_written: ?*usize, p_cancellable: ?*gio.Cancellable, p_error: **glib.Error, p_format: [*:0]const u8, p_args: std.builtin.VaList) c_int;
    pub const vprintf = g_output_stream_vprintf;

    /// Tries to write `count` bytes from `buffer` into the stream. Will block
    /// during the operation.
    ///
    /// If count is 0, returns 0 and does nothing. A value of `count`
    /// larger than `G_MAXSSIZE` will cause a `G_IO_ERROR_INVALID_ARGUMENT` error.
    ///
    /// On success, the number of bytes written to the stream is returned.
    /// It is not an error if this is not the same as the requested size, as it
    /// can happen e.g. on a partial I/O error, or if there is not enough
    /// storage in the stream. All writes block until at least one byte
    /// is written or an error occurs; 0 is never returned (unless
    /// `count` is 0).
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned. If an
    /// operation was partially finished when the operation was cancelled the
    /// partial result will be returned, without an error.
    ///
    /// On error -1 is returned and `error` is set accordingly.
    extern fn g_output_stream_write(p_stream: *OutputStream, p_buffer: [*]u8, p_count: usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) isize;
    pub const write = g_output_stream_write;

    /// Tries to write `count` bytes from `buffer` into the stream. Will block
    /// during the operation.
    ///
    /// This function is similar to `gio.OutputStream.write`, except it tries to
    /// write as many bytes as requested, only stopping on an error.
    ///
    /// On a successful write of `count` bytes, `TRUE` is returned, and `bytes_written`
    /// is set to `count`.
    ///
    /// If there is an error during the operation `FALSE` is returned and `error`
    /// is set to indicate the error status.
    ///
    /// As a special exception to the normal conventions for functions that
    /// use `glib.Error`, if this function returns `FALSE` (and sets `error`) then
    /// `bytes_written` will be set to the number of bytes that were
    /// successfully written before the error was encountered.  This
    /// functionality is only available from C.  If you need it from another
    /// language then you must write your own loop around
    /// `gio.OutputStream.write`.
    extern fn g_output_stream_write_all(p_stream: *OutputStream, p_buffer: [*]u8, p_count: usize, p_bytes_written: ?*usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const writeAll = g_output_stream_write_all;

    /// Request an asynchronous write of `count` bytes from `buffer` into
    /// the stream. When the operation is finished `callback` will be called.
    /// You can then call `gio.OutputStream.writeAllFinish` to get the result of the
    /// operation.
    ///
    /// This is the asynchronous version of `gio.OutputStream.writeAll`.
    ///
    /// Call `gio.OutputStream.writeAllFinish` to collect the result.
    ///
    /// Any outstanding I/O request with higher priority (lower numerical
    /// value) will be executed before an outstanding request with lower
    /// priority. Default priority is `G_PRIORITY_DEFAULT`.
    ///
    /// Note that no copy of `buffer` will be made, so it must stay valid
    /// until `callback` is called.
    extern fn g_output_stream_write_all_async(p_stream: *OutputStream, p_buffer: [*]u8, p_count: usize, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const writeAllAsync = g_output_stream_write_all_async;

    /// Finishes an asynchronous stream write operation started with
    /// `gio.OutputStream.writeAllAsync`.
    ///
    /// As a special exception to the normal conventions for functions that
    /// use `glib.Error`, if this function returns `FALSE` (and sets `error`) then
    /// `bytes_written` will be set to the number of bytes that were
    /// successfully written before the error was encountered.  This
    /// functionality is only available from C.  If you need it from another
    /// language then you must write your own loop around
    /// `gio.OutputStream.writeAsync`.
    extern fn g_output_stream_write_all_finish(p_stream: *OutputStream, p_result: *gio.AsyncResult, p_bytes_written: ?*usize, p_error: ?*?*glib.Error) c_int;
    pub const writeAllFinish = g_output_stream_write_all_finish;

    /// Request an asynchronous write of `count` bytes from `buffer` into
    /// the stream. When the operation is finished `callback` will be called.
    /// You can then call `gio.OutputStream.writeFinish` to get the result of the
    /// operation.
    ///
    /// During an async request no other sync and async calls are allowed,
    /// and will result in `G_IO_ERROR_PENDING` errors.
    ///
    /// A value of `count` larger than `G_MAXSSIZE` will cause a
    /// `G_IO_ERROR_INVALID_ARGUMENT` error.
    ///
    /// On success, the number of bytes written will be passed to the
    /// `callback`. It is not an error if this is not the same as the
    /// requested size, as it can happen e.g. on a partial I/O error,
    /// but generally we try to write as many bytes as requested.
    ///
    /// You are guaranteed that this method will never fail with
    /// `G_IO_ERROR_WOULD_BLOCK` - if `stream` can't accept more data, the
    /// method will just wait until this changes.
    ///
    /// Any outstanding I/O request with higher priority (lower numerical
    /// value) will be executed before an outstanding request with lower
    /// priority. Default priority is `G_PRIORITY_DEFAULT`.
    ///
    /// The asynchronous methods have a default fallback that uses threads
    /// to implement asynchronicity, so they are optional for inheriting
    /// classes. However, if you override one you must override all.
    ///
    /// For the synchronous, blocking version of this function, see
    /// `gio.OutputStream.write`.
    ///
    /// Note that no copy of `buffer` will be made, so it must stay valid
    /// until `callback` is called. See `gio.OutputStream.writeBytesAsync`
    /// for a `glib.Bytes` version that will automatically hold a reference to
    /// the contents (without copying) for the duration of the call.
    extern fn g_output_stream_write_async(p_stream: *OutputStream, p_buffer: [*]u8, p_count: usize, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const writeAsync = g_output_stream_write_async;

    /// A wrapper function for `gio.OutputStream.write` which takes a
    /// `glib.Bytes` as input.  This can be more convenient for use by language
    /// bindings or in other cases where the refcounted nature of `glib.Bytes`
    /// is helpful over a bare pointer interface.
    ///
    /// However, note that this function may still perform partial writes,
    /// just like `gio.OutputStream.write`.  If that occurs, to continue
    /// writing, you will need to create a new `glib.Bytes` containing just the
    /// remaining bytes, using `glib.Bytes.newFromBytes`. Passing the same
    /// `glib.Bytes` instance multiple times potentially can result in duplicated
    /// data in the output stream.
    extern fn g_output_stream_write_bytes(p_stream: *OutputStream, p_bytes: *glib.Bytes, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) isize;
    pub const writeBytes = g_output_stream_write_bytes;

    /// This function is similar to `gio.OutputStream.writeAsync`, but
    /// takes a `glib.Bytes` as input.  Due to the refcounted nature of `glib.Bytes`,
    /// this allows the stream to avoid taking a copy of the data.
    ///
    /// However, note that this function may still perform partial writes,
    /// just like `gio.OutputStream.writeAsync`. If that occurs, to continue
    /// writing, you will need to create a new `glib.Bytes` containing just the
    /// remaining bytes, using `glib.Bytes.newFromBytes`. Passing the same
    /// `glib.Bytes` instance multiple times potentially can result in duplicated
    /// data in the output stream.
    ///
    /// For the synchronous, blocking version of this function, see
    /// `gio.OutputStream.writeBytes`.
    extern fn g_output_stream_write_bytes_async(p_stream: *OutputStream, p_bytes: *glib.Bytes, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const writeBytesAsync = g_output_stream_write_bytes_async;

    /// Finishes a stream write-from-`glib.Bytes` operation.
    extern fn g_output_stream_write_bytes_finish(p_stream: *OutputStream, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) isize;
    pub const writeBytesFinish = g_output_stream_write_bytes_finish;

    /// Finishes a stream write operation.
    extern fn g_output_stream_write_finish(p_stream: *OutputStream, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) isize;
    pub const writeFinish = g_output_stream_write_finish;

    /// Tries to write the bytes contained in the `n_vectors` `vectors` into the
    /// stream. Will block during the operation.
    ///
    /// If `n_vectors` is 0 or the sum of all bytes in `vectors` is 0, returns 0 and
    /// does nothing.
    ///
    /// On success, the number of bytes written to the stream is returned.
    /// It is not an error if this is not the same as the requested size, as it
    /// can happen e.g. on a partial I/O error, or if there is not enough
    /// storage in the stream. All writes block until at least one byte
    /// is written or an error occurs; 0 is never returned (unless
    /// `n_vectors` is 0 or the sum of all bytes in `vectors` is 0).
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned. If an
    /// operation was partially finished when the operation was cancelled the
    /// partial result will be returned, without an error.
    ///
    /// Some implementations of `gio.OutputStream.writev` may have limitations on the
    /// aggregate buffer size, and will return `G_IO_ERROR_INVALID_ARGUMENT` if these
    /// are exceeded. For example, when writing to a local file on UNIX platforms,
    /// the aggregate buffer size must not exceed `G_MAXSSIZE` bytes.
    extern fn g_output_stream_writev(p_stream: *OutputStream, p_vectors: [*]const gio.OutputVector, p_n_vectors: usize, p_bytes_written: ?*usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const writev = g_output_stream_writev;

    /// Tries to write the bytes contained in the `n_vectors` `vectors` into the
    /// stream. Will block during the operation.
    ///
    /// This function is similar to `gio.OutputStream.writev`, except it tries to
    /// write as many bytes as requested, only stopping on an error.
    ///
    /// On a successful write of all `n_vectors` vectors, `TRUE` is returned, and
    /// `bytes_written` is set to the sum of all the sizes of `vectors`.
    ///
    /// If there is an error during the operation `FALSE` is returned and `error`
    /// is set to indicate the error status.
    ///
    /// As a special exception to the normal conventions for functions that
    /// use `glib.Error`, if this function returns `FALSE` (and sets `error`) then
    /// `bytes_written` will be set to the number of bytes that were
    /// successfully written before the error was encountered.  This
    /// functionality is only available from C. If you need it from another
    /// language then you must write your own loop around
    /// `gio.OutputStream.write`.
    ///
    /// The content of the individual elements of `vectors` might be changed by this
    /// function.
    extern fn g_output_stream_writev_all(p_stream: *OutputStream, p_vectors: [*]gio.OutputVector, p_n_vectors: usize, p_bytes_written: ?*usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const writevAll = g_output_stream_writev_all;

    /// Request an asynchronous write of the bytes contained in the `n_vectors` `vectors` into
    /// the stream. When the operation is finished `callback` will be called.
    /// You can then call `gio.OutputStream.writevAllFinish` to get the result of the
    /// operation.
    ///
    /// This is the asynchronous version of `gio.OutputStream.writevAll`.
    ///
    /// Call `gio.OutputStream.writevAllFinish` to collect the result.
    ///
    /// Any outstanding I/O request with higher priority (lower numerical
    /// value) will be executed before an outstanding request with lower
    /// priority. Default priority is `G_PRIORITY_DEFAULT`.
    ///
    /// Note that no copy of `vectors` will be made, so it must stay valid
    /// until `callback` is called. The content of the individual elements
    /// of `vectors` might be changed by this function.
    extern fn g_output_stream_writev_all_async(p_stream: *OutputStream, p_vectors: [*]gio.OutputVector, p_n_vectors: usize, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const writevAllAsync = g_output_stream_writev_all_async;

    /// Finishes an asynchronous stream write operation started with
    /// `gio.OutputStream.writevAllAsync`.
    ///
    /// As a special exception to the normal conventions for functions that
    /// use `glib.Error`, if this function returns `FALSE` (and sets `error`) then
    /// `bytes_written` will be set to the number of bytes that were
    /// successfully written before the error was encountered.  This
    /// functionality is only available from C.  If you need it from another
    /// language then you must write your own loop around
    /// `gio.OutputStream.writevAsync`.
    extern fn g_output_stream_writev_all_finish(p_stream: *OutputStream, p_result: *gio.AsyncResult, p_bytes_written: ?*usize, p_error: ?*?*glib.Error) c_int;
    pub const writevAllFinish = g_output_stream_writev_all_finish;

    /// Request an asynchronous write of the bytes contained in `n_vectors` `vectors` into
    /// the stream. When the operation is finished `callback` will be called.
    /// You can then call `gio.OutputStream.writevFinish` to get the result of the
    /// operation.
    ///
    /// During an async request no other sync and async calls are allowed,
    /// and will result in `G_IO_ERROR_PENDING` errors.
    ///
    /// On success, the number of bytes written will be passed to the
    /// `callback`. It is not an error if this is not the same as the
    /// requested size, as it can happen e.g. on a partial I/O error,
    /// but generally we try to write as many bytes as requested.
    ///
    /// You are guaranteed that this method will never fail with
    /// `G_IO_ERROR_WOULD_BLOCK` — if `stream` can't accept more data, the
    /// method will just wait until this changes.
    ///
    /// Any outstanding I/O request with higher priority (lower numerical
    /// value) will be executed before an outstanding request with lower
    /// priority. Default priority is `G_PRIORITY_DEFAULT`.
    ///
    /// The asynchronous methods have a default fallback that uses threads
    /// to implement asynchronicity, so they are optional for inheriting
    /// classes. However, if you override one you must override all.
    ///
    /// For the synchronous, blocking version of this function, see
    /// `gio.OutputStream.writev`.
    ///
    /// Note that no copy of `vectors` will be made, so it must stay valid
    /// until `callback` is called.
    extern fn g_output_stream_writev_async(p_stream: *OutputStream, p_vectors: [*]const gio.OutputVector, p_n_vectors: usize, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const writevAsync = g_output_stream_writev_async;

    /// Finishes a stream writev operation.
    extern fn g_output_stream_writev_finish(p_stream: *OutputStream, p_result: *gio.AsyncResult, p_bytes_written: ?*usize, p_error: ?*?*glib.Error) c_int;
    pub const writevFinish = g_output_stream_writev_finish;

    extern fn g_output_stream_get_type() usize;
    pub const getGObjectType = g_output_stream_get_type;

    extern fn g_object_ref(p_self: *gio.OutputStream) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.OutputStream) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *OutputStream, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `GPermission` represents the status of the caller’s permission to
/// perform a certain action.
///
/// You can query if the action is currently allowed and if it is
/// possible to acquire the permission so that the action will be allowed
/// in the future.
///
/// There is also an API to actually acquire the permission and one to
/// release it.
///
/// As an example, a `GPermission` might represent the ability for the
/// user to write to a `gio.Settings` object.  This `GPermission` object
/// could then be used to decide if it is appropriate to show a “Click here to
/// unlock” button in a dialog and to provide the mechanism to invoke
/// when that button is clicked.
pub const Permission = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.PermissionClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.PermissionPrivate,

    pub const virtual_methods = struct {
        /// Attempts to acquire the permission represented by `permission`.
        ///
        /// The precise method by which this happens depends on the permission
        /// and the underlying authentication mechanism.  A simple example is
        /// that a dialog may appear asking the user to enter their password.
        ///
        /// You should check with `gio.Permission.getCanAcquire` before calling
        /// this function.
        ///
        /// If the permission is acquired then `TRUE` is returned.  Otherwise,
        /// `FALSE` is returned and `error` is set appropriately.
        ///
        /// This call is blocking, likely for a very long time (in the case that
        /// user interaction is required).  See `gio.Permission.acquireAsync` for
        /// the non-blocking version.
        pub const acquire = struct {
            pub fn call(p_class: anytype, p_permission: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(Permission.Class, p_class).f_acquire.?(gobject.ext.as(Permission, p_permission), p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_permission: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(Permission.Class, p_class).f_acquire = @ptrCast(p_implementation);
            }
        };

        /// Attempts to acquire the permission represented by `permission`.
        ///
        /// This is the first half of the asynchronous version of
        /// `gio.Permission.acquire`.
        pub const acquire_async = struct {
            pub fn call(p_class: anytype, p_permission: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(Permission.Class, p_class).f_acquire_async.?(gobject.ext.as(Permission, p_permission), p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_permission: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(Permission.Class, p_class).f_acquire_async = @ptrCast(p_implementation);
            }
        };

        /// Collects the result of attempting to acquire the permission
        /// represented by `permission`.
        ///
        /// This is the second half of the asynchronous version of
        /// `gio.Permission.acquire`.
        pub const acquire_finish = struct {
            pub fn call(p_class: anytype, p_permission: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(Permission.Class, p_class).f_acquire_finish.?(gobject.ext.as(Permission, p_permission), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_permission: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(Permission.Class, p_class).f_acquire_finish = @ptrCast(p_implementation);
            }
        };

        /// Attempts to release the permission represented by `permission`.
        ///
        /// The precise method by which this happens depends on the permission
        /// and the underlying authentication mechanism.  In most cases the
        /// permission will be dropped immediately without further action.
        ///
        /// You should check with `gio.Permission.getCanRelease` before calling
        /// this function.
        ///
        /// If the permission is released then `TRUE` is returned.  Otherwise,
        /// `FALSE` is returned and `error` is set appropriately.
        ///
        /// This call is blocking, likely for a very long time (in the case that
        /// user interaction is required).  See `gio.Permission.releaseAsync` for
        /// the non-blocking version.
        pub const release = struct {
            pub fn call(p_class: anytype, p_permission: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(Permission.Class, p_class).f_release.?(gobject.ext.as(Permission, p_permission), p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_permission: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(Permission.Class, p_class).f_release = @ptrCast(p_implementation);
            }
        };

        /// Attempts to release the permission represented by `permission`.
        ///
        /// This is the first half of the asynchronous version of
        /// `gio.Permission.release`.
        pub const release_async = struct {
            pub fn call(p_class: anytype, p_permission: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(Permission.Class, p_class).f_release_async.?(gobject.ext.as(Permission, p_permission), p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_permission: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(Permission.Class, p_class).f_release_async = @ptrCast(p_implementation);
            }
        };

        /// Collects the result of attempting to release the permission
        /// represented by `permission`.
        ///
        /// This is the second half of the asynchronous version of
        /// `gio.Permission.release`.
        pub const release_finish = struct {
            pub fn call(p_class: anytype, p_permission: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(Permission.Class, p_class).f_release_finish.?(gobject.ext.as(Permission, p_permission), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_permission: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(Permission.Class, p_class).f_release_finish = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// `TRUE` if the caller currently has permission to perform the action that
        /// `permission` represents the permission to perform.
        pub const allowed = struct {
            pub const name = "allowed";

            pub const Type = c_int;
        };

        /// `TRUE` if it is generally possible to acquire the permission by calling
        /// `gio.Permission.acquire`.
        pub const can_acquire = struct {
            pub const name = "can-acquire";

            pub const Type = c_int;
        };

        /// `TRUE` if it is generally possible to release the permission by calling
        /// `gio.Permission.release`.
        pub const can_release = struct {
            pub const name = "can-release";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Attempts to acquire the permission represented by `permission`.
    ///
    /// The precise method by which this happens depends on the permission
    /// and the underlying authentication mechanism.  A simple example is
    /// that a dialog may appear asking the user to enter their password.
    ///
    /// You should check with `gio.Permission.getCanAcquire` before calling
    /// this function.
    ///
    /// If the permission is acquired then `TRUE` is returned.  Otherwise,
    /// `FALSE` is returned and `error` is set appropriately.
    ///
    /// This call is blocking, likely for a very long time (in the case that
    /// user interaction is required).  See `gio.Permission.acquireAsync` for
    /// the non-blocking version.
    extern fn g_permission_acquire(p_permission: *Permission, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const acquire = g_permission_acquire;

    /// Attempts to acquire the permission represented by `permission`.
    ///
    /// This is the first half of the asynchronous version of
    /// `gio.Permission.acquire`.
    extern fn g_permission_acquire_async(p_permission: *Permission, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const acquireAsync = g_permission_acquire_async;

    /// Collects the result of attempting to acquire the permission
    /// represented by `permission`.
    ///
    /// This is the second half of the asynchronous version of
    /// `gio.Permission.acquire`.
    extern fn g_permission_acquire_finish(p_permission: *Permission, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const acquireFinish = g_permission_acquire_finish;

    /// Gets the value of the 'allowed' property.  This property is `TRUE` if
    /// the caller currently has permission to perform the action that
    /// `permission` represents the permission to perform.
    extern fn g_permission_get_allowed(p_permission: *Permission) c_int;
    pub const getAllowed = g_permission_get_allowed;

    /// Gets the value of the 'can-acquire' property.  This property is `TRUE`
    /// if it is generally possible to acquire the permission by calling
    /// `gio.Permission.acquire`.
    extern fn g_permission_get_can_acquire(p_permission: *Permission) c_int;
    pub const getCanAcquire = g_permission_get_can_acquire;

    /// Gets the value of the 'can-release' property.  This property is `TRUE`
    /// if it is generally possible to release the permission by calling
    /// `gio.Permission.release`.
    extern fn g_permission_get_can_release(p_permission: *Permission) c_int;
    pub const getCanRelease = g_permission_get_can_release;

    /// This function is called by the `gio.Permission` implementation to update
    /// the properties of the permission.  You should never call this
    /// function except from a `gio.Permission` implementation.
    ///
    /// GObject notify signals are generated, as appropriate.
    extern fn g_permission_impl_update(p_permission: *Permission, p_allowed: c_int, p_can_acquire: c_int, p_can_release: c_int) void;
    pub const implUpdate = g_permission_impl_update;

    /// Attempts to release the permission represented by `permission`.
    ///
    /// The precise method by which this happens depends on the permission
    /// and the underlying authentication mechanism.  In most cases the
    /// permission will be dropped immediately without further action.
    ///
    /// You should check with `gio.Permission.getCanRelease` before calling
    /// this function.
    ///
    /// If the permission is released then `TRUE` is returned.  Otherwise,
    /// `FALSE` is returned and `error` is set appropriately.
    ///
    /// This call is blocking, likely for a very long time (in the case that
    /// user interaction is required).  See `gio.Permission.releaseAsync` for
    /// the non-blocking version.
    extern fn g_permission_release(p_permission: *Permission, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const release = g_permission_release;

    /// Attempts to release the permission represented by `permission`.
    ///
    /// This is the first half of the asynchronous version of
    /// `gio.Permission.release`.
    extern fn g_permission_release_async(p_permission: *Permission, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const releaseAsync = g_permission_release_async;

    /// Collects the result of attempting to release the permission
    /// represented by `permission`.
    ///
    /// This is the second half of the asynchronous version of
    /// `gio.Permission.release`.
    extern fn g_permission_release_finish(p_permission: *Permission, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const releaseFinish = g_permission_release_finish;

    extern fn g_permission_get_type() usize;
    pub const getGObjectType = g_permission_get_type;

    extern fn g_object_ref(p_self: *gio.Permission) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.Permission) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Permission, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `GPropertyAction` is a way to get a `gio.Action` with a state value
/// reflecting and controlling the value of a `gobject.Object` property.
///
/// The state of the action will correspond to the value of the property.
/// Changing it will change the property (assuming the requested value
/// matches the requirements as specified in the `gobject.ParamSpec`).
///
/// Only the most common types are presently supported.  Booleans are
/// mapped to booleans, strings to strings, signed/unsigned integers to
/// int32/uint32 and floats and doubles to doubles.
///
/// If the property is an enum then the state will be string-typed and
/// conversion will automatically be performed between the enum value and
/// ‘nick’ string as per the `gobject.EnumValue` table.
///
/// Flags types are not currently supported.
///
/// Properties of object types, boxed types and pointer types are not
/// supported and probably never will be.
///
/// Properties of `glib.Variant` types are not currently supported.
///
/// If the property is boolean-valued then the action will have a `NULL`
/// parameter type, and activating the action (with no parameter) will
/// toggle the value of the property.
///
/// In all other cases, the parameter type will correspond to the type of
/// the property.
///
/// The general idea here is to reduce the number of locations where a
/// particular piece of state is kept (and therefore has to be synchronised
/// between). `GPropertyAction` does not have a separate state that is kept
/// in sync with the property value — its state is the property value.
///
/// For example, it might be useful to create a `gio.Action` corresponding
/// to the `visible-child-name` property of a [`GtkStack`](https://docs.gtk.org/gtk4/class.Stack.html)
/// so that the current page can be switched from a menu.  The active radio
/// indication in the menu is then directly determined from the active page of
/// the `GtkStack`.
///
/// An anti-example would be binding the `active-id` property on a
/// [`GtkComboBox`](https://docs.gtk.org/gtk4/class.ComboBox.html). This is
/// because the state of the combo box itself is probably uninteresting and is
/// actually being used to control something else.
///
/// Another anti-example would be to bind to the `visible-child-name`
/// property of a [`GtkStack`](https://docs.gtk.org/gtk4/class.Stack.html) if
/// this value is actually stored in `gio.Settings`.  In that case, the
/// real source of the value is* `gio.Settings`.  If you want
/// a `gio.Action` to control a setting stored in `gio.Settings`,
/// see `gio.Settings.createAction` instead, and possibly combine its
/// use with `gio.Settings.bind`.
pub const PropertyAction = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gio.Action};
    pub const Class = opaque {
        pub const Instance = PropertyAction;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// If `action` is currently enabled.
        ///
        /// If the action is disabled then calls to `gio.Action.activate` and
        /// `gio.Action.changeState` have no effect.
        pub const enabled = struct {
            pub const name = "enabled";

            pub const Type = c_int;
        };

        /// If `TRUE`, the state of the action will be the negation of the
        /// property value, provided the property is boolean.
        pub const invert_boolean = struct {
            pub const name = "invert-boolean";

            pub const Type = c_int;
        };

        /// The name of the action.  This is mostly meaningful for identifying
        /// the action once it has been added to a `gio.ActionMap`.
        pub const name = struct {
            pub const name = "name";

            pub const Type = ?[*:0]u8;
        };

        /// The object to wrap a property on.
        ///
        /// The object must be a non-`NULL` `gobject.Object` with properties.
        pub const object = struct {
            pub const name = "object";

            pub const Type = ?*gobject.Object;
        };

        /// The type of the parameter that must be given when activating the
        /// action.
        pub const parameter_type = struct {
            pub const name = "parameter-type";

            pub const Type = ?*glib.VariantType;
        };

        /// The name of the property to wrap on the object.
        ///
        /// The property must exist on the passed-in object and it must be
        /// readable and writable (and not construct-only).
        pub const property_name = struct {
            pub const name = "property-name";

            pub const Type = ?[*:0]u8;
        };

        /// The state of the action, or `NULL` if the action is stateless.
        pub const state = struct {
            pub const name = "state";

            pub const Type = ?*glib.Variant;
        };

        /// The `glib.VariantType` of the state that the action has, or `NULL` if the
        /// action is stateless.
        pub const state_type = struct {
            pub const name = "state-type";

            pub const Type = ?*glib.VariantType;
        };
    };

    pub const signals = struct {};

    /// Creates a `gio.Action` corresponding to the value of property
    /// `property_name` on `object`.
    ///
    /// The property must be existent and readable and writable (and not
    /// construct-only).
    ///
    /// This function takes a reference on `object` and doesn't release it
    /// until the action is destroyed.
    extern fn g_property_action_new(p_name: [*:0]const u8, p_object: *gobject.Object, p_property_name: [*:0]const u8) *gio.PropertyAction;
    pub const new = g_property_action_new;

    extern fn g_property_action_get_type() usize;
    pub const getGObjectType = g_property_action_get_type;

    extern fn g_object_ref(p_self: *gio.PropertyAction) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.PropertyAction) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *PropertyAction, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `gio.InetSocketAddress` representing a connection via a proxy server.
pub const ProxyAddress = extern struct {
    pub const Parent = gio.InetSocketAddress;
    pub const Implements = [_]type{gio.SocketConnectable};
    pub const Class = gio.ProxyAddressClass;
    f_parent_instance: gio.InetSocketAddress,
    f_priv: ?*gio.ProxyAddressPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The proxy destination hostname.
        pub const destination_hostname = struct {
            pub const name = "destination-hostname";

            pub const Type = ?[*:0]u8;
        };

        /// The proxy destination port.
        pub const destination_port = struct {
            pub const name = "destination-port";

            pub const Type = c_uint;
        };

        /// The protocol being spoke to the destination host, or `NULL` if
        /// the `gio.ProxyAddress` doesn't know.
        pub const destination_protocol = struct {
            pub const name = "destination-protocol";

            pub const Type = ?[*:0]u8;
        };

        /// The proxy password.
        pub const password = struct {
            pub const name = "password";

            pub const Type = ?[*:0]u8;
        };

        /// The proxy protocol.
        pub const protocol = struct {
            pub const name = "protocol";

            pub const Type = ?[*:0]u8;
        };

        /// The URI string that the proxy was constructed from (or `NULL`
        /// if the creator didn't specify this).
        pub const uri = struct {
            pub const name = "uri";

            pub const Type = ?[*:0]u8;
        };

        /// The proxy username.
        pub const username = struct {
            pub const name = "username";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gio.ProxyAddress` for `inetaddr` with `protocol` that should
    /// tunnel through `dest_hostname` and `dest_port`.
    ///
    /// (Note that this method doesn't set the `gio.ProxyAddress.properties.uri` or
    /// `gio.ProxyAddress.properties.destination`-protocol fields; use `gobject.Object.new`
    /// directly if you want to set those.)
    extern fn g_proxy_address_new(p_inetaddr: *gio.InetAddress, p_port: u16, p_protocol: [*:0]const u8, p_dest_hostname: [*:0]const u8, p_dest_port: u16, p_username: ?[*:0]const u8, p_password: ?[*:0]const u8) *gio.ProxyAddress;
    pub const new = g_proxy_address_new;

    /// Gets `proxy`'s destination hostname; that is, the name of the host
    /// that will be connected to via the proxy, not the name of the proxy
    /// itself.
    extern fn g_proxy_address_get_destination_hostname(p_proxy: *ProxyAddress) [*:0]const u8;
    pub const getDestinationHostname = g_proxy_address_get_destination_hostname;

    /// Gets `proxy`'s destination port; that is, the port on the
    /// destination host that will be connected to via the proxy, not the
    /// port number of the proxy itself.
    extern fn g_proxy_address_get_destination_port(p_proxy: *ProxyAddress) u16;
    pub const getDestinationPort = g_proxy_address_get_destination_port;

    /// Gets the protocol that is being spoken to the destination
    /// server; eg, "http" or "ftp".
    extern fn g_proxy_address_get_destination_protocol(p_proxy: *ProxyAddress) [*:0]const u8;
    pub const getDestinationProtocol = g_proxy_address_get_destination_protocol;

    /// Gets `proxy`'s password.
    extern fn g_proxy_address_get_password(p_proxy: *ProxyAddress) ?[*:0]const u8;
    pub const getPassword = g_proxy_address_get_password;

    /// Gets `proxy`'s protocol. eg, "socks" or "http"
    extern fn g_proxy_address_get_protocol(p_proxy: *ProxyAddress) [*:0]const u8;
    pub const getProtocol = g_proxy_address_get_protocol;

    /// Gets the proxy URI that `proxy` was constructed from.
    extern fn g_proxy_address_get_uri(p_proxy: *ProxyAddress) ?[*:0]const u8;
    pub const getUri = g_proxy_address_get_uri;

    /// Gets `proxy`'s username.
    extern fn g_proxy_address_get_username(p_proxy: *ProxyAddress) ?[*:0]const u8;
    pub const getUsername = g_proxy_address_get_username;

    extern fn g_proxy_address_get_type() usize;
    pub const getGObjectType = g_proxy_address_get_type;

    extern fn g_object_ref(p_self: *gio.ProxyAddress) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.ProxyAddress) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ProxyAddress, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GProxyAddressEnumerator` is a wrapper around
/// `gio.SocketAddressEnumerator` which takes the `gio.SocketAddress`
/// instances returned by the `gio.SocketAddressEnumerator`
/// and wraps them in `gio.ProxyAddress` instances, using the given
/// `gio.ProxyAddressEnumerator.properties.proxy_resolver`.
///
/// This enumerator will be returned (for example, by
/// `gio.SocketConnectable.enumerate`) as appropriate when a proxy is
/// configured; there should be no need to manually wrap a
/// `gio.SocketAddressEnumerator` instance with one.
pub const ProxyAddressEnumerator = extern struct {
    pub const Parent = gio.SocketAddressEnumerator;
    pub const Implements = [_]type{};
    pub const Class = gio.ProxyAddressEnumeratorClass;
    f_parent_instance: gio.SocketAddressEnumerator,
    f_priv: ?*gio.ProxyAddressEnumeratorPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The connectable being enumerated.
        pub const connectable = struct {
            pub const name = "connectable";

            pub const Type = ?*gio.SocketConnectable;
        };

        /// The default port to use if `gio.ProxyAddressEnumerator.properties.uri` does not
        /// specify one.
        pub const default_port = struct {
            pub const name = "default-port";

            pub const Type = c_uint;
        };

        /// The proxy resolver to use.
        pub const proxy_resolver = struct {
            pub const name = "proxy-resolver";

            pub const Type = ?*gio.ProxyResolver;
        };

        /// The destination URI. Use `none://` for a generic socket.
        pub const uri = struct {
            pub const name = "uri";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    extern fn g_proxy_address_enumerator_get_type() usize;
    pub const getGObjectType = g_proxy_address_enumerator_get_type;

    extern fn g_object_ref(p_self: *gio.ProxyAddressEnumerator) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.ProxyAddressEnumerator) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ProxyAddressEnumerator, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The object that handles DNS resolution. Use `gio.Resolver.getDefault`
/// to get the default resolver.
///
/// `GResolver` provides cancellable synchronous and asynchronous DNS
/// resolution, for hostnames (`gio.Resolver.lookupByAddress`,
/// `gio.Resolver.lookupByName` and their async variants) and SRV
/// (service) records (`gio.Resolver.lookupService`).
///
/// `gio.NetworkAddress` and `gio.NetworkService` provide wrappers
/// around `GResolver` functionality that also implement
/// `gio.SocketConnectable`, making it easy to connect to a remote
/// host/service.
///
/// The default resolver (see `gio.Resolver.getDefault`) has a timeout of
/// 30s set on it since GLib 2.78. Earlier versions of GLib did not support
/// resolver timeouts.
///
/// This is an abstract type; subclasses of it implement different resolvers for
/// different platforms and situations.
pub const Resolver = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.ResolverClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.ResolverPrivate,

    pub const virtual_methods = struct {
        /// Synchronously reverse-resolves `address` to determine its
        /// associated hostname.
        ///
        /// If the DNS resolution fails, `error` (if non-`NULL`) will be set to
        /// a value from `gio.ResolverError`.
        ///
        /// If `cancellable` is non-`NULL`, it can be used to cancel the
        /// operation, in which case `error` (if non-`NULL`) will be set to
        /// `G_IO_ERROR_CANCELLED`.
        pub const lookup_by_address = struct {
            pub fn call(p_class: anytype, p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_address: *gio.InetAddress, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?[*:0]u8 {
                return gobject.ext.as(Resolver.Class, p_class).f_lookup_by_address.?(gobject.ext.as(Resolver, p_resolver), p_address, p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_address: *gio.InetAddress, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) ?[*:0]u8) void {
                gobject.ext.as(Resolver.Class, p_class).f_lookup_by_address = @ptrCast(p_implementation);
            }
        };

        /// Begins asynchronously reverse-resolving `address` to determine its
        /// associated hostname, and eventually calls `callback`, which must
        /// call `gio.Resolver.lookupByAddressFinish` to get the final result.
        pub const lookup_by_address_async = struct {
            pub fn call(p_class: anytype, p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_address: *gio.InetAddress, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(Resolver.Class, p_class).f_lookup_by_address_async.?(gobject.ext.as(Resolver, p_resolver), p_address, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_address: *gio.InetAddress, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(Resolver.Class, p_class).f_lookup_by_address_async = @ptrCast(p_implementation);
            }
        };

        /// Retrieves the result of a previous call to
        /// `gio.Resolver.lookupByAddressAsync`.
        ///
        /// If the DNS resolution failed, `error` (if non-`NULL`) will be set to
        /// a value from `gio.ResolverError`. If the operation was cancelled,
        /// `error` will be set to `G_IO_ERROR_CANCELLED`.
        pub const lookup_by_address_finish = struct {
            pub fn call(p_class: anytype, p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?[*:0]u8 {
                return gobject.ext.as(Resolver.Class, p_class).f_lookup_by_address_finish.?(gobject.ext.as(Resolver, p_resolver), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?[*:0]u8) void {
                gobject.ext.as(Resolver.Class, p_class).f_lookup_by_address_finish = @ptrCast(p_implementation);
            }
        };

        /// Synchronously resolves `hostname` to determine its associated IP
        /// address(es). `hostname` may be an ASCII-only or UTF-8 hostname, or
        /// the textual form of an IP address (in which case this just becomes
        /// a wrapper around `gio.InetAddress.newFromString`).
        ///
        /// On success, `gio.Resolver.lookupByName` will return a non-empty `glib.List` of
        /// `gio.InetAddress`, sorted in order of preference and guaranteed to not
        /// contain duplicates. That is, if using the result to connect to
        /// `hostname`, you should attempt to connect to the first address
        /// first, then the second if the first fails, etc. If you are using
        /// the result to listen on a socket, it is appropriate to add each
        /// result using e.g. `gio.SocketListener.addAddress`.
        ///
        /// If the DNS resolution fails, `error` (if non-`NULL`) will be set to a
        /// value from `gio.ResolverError` and `NULL` will be returned.
        ///
        /// If `cancellable` is non-`NULL`, it can be used to cancel the
        /// operation, in which case `error` (if non-`NULL`) will be set to
        /// `G_IO_ERROR_CANCELLED`.
        ///
        /// If you are planning to connect to a socket on the resolved IP
        /// address, it may be easier to create a `gio.NetworkAddress` and use its
        /// `gio.SocketConnectable` interface.
        pub const lookup_by_name = struct {
            pub fn call(p_class: anytype, p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_hostname: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*glib.List {
                return gobject.ext.as(Resolver.Class, p_class).f_lookup_by_name.?(gobject.ext.as(Resolver, p_resolver), p_hostname, p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_hostname: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) ?*glib.List) void {
                gobject.ext.as(Resolver.Class, p_class).f_lookup_by_name = @ptrCast(p_implementation);
            }
        };

        /// Begins asynchronously resolving `hostname` to determine its
        /// associated IP address(es), and eventually calls `callback`, which
        /// must call `gio.Resolver.lookupByNameFinish` to get the result.
        /// See `gio.Resolver.lookupByName` for more details.
        pub const lookup_by_name_async = struct {
            pub fn call(p_class: anytype, p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_hostname: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(Resolver.Class, p_class).f_lookup_by_name_async.?(gobject.ext.as(Resolver, p_resolver), p_hostname, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_hostname: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(Resolver.Class, p_class).f_lookup_by_name_async = @ptrCast(p_implementation);
            }
        };

        /// Retrieves the result of a call to
        /// `gio.Resolver.lookupByNameAsync`.
        ///
        /// If the DNS resolution failed, `error` (if non-`NULL`) will be set to
        /// a value from `gio.ResolverError`. If the operation was cancelled,
        /// `error` will be set to `G_IO_ERROR_CANCELLED`.
        pub const lookup_by_name_finish = struct {
            pub fn call(p_class: anytype, p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.List {
                return gobject.ext.as(Resolver.Class, p_class).f_lookup_by_name_finish.?(gobject.ext.as(Resolver, p_resolver), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?*glib.List) void {
                gobject.ext.as(Resolver.Class, p_class).f_lookup_by_name_finish = @ptrCast(p_implementation);
            }
        };

        /// This differs from `gio.Resolver.lookupByName` in that you can modify
        /// the lookup behavior with `flags`. For example this can be used to limit
        /// results with `G_RESOLVER_NAME_LOOKUP_FLAGS_IPV4_ONLY`.
        pub const lookup_by_name_with_flags = struct {
            pub fn call(p_class: anytype, p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_hostname: [*:0]const u8, p_flags: gio.ResolverNameLookupFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*glib.List {
                return gobject.ext.as(Resolver.Class, p_class).f_lookup_by_name_with_flags.?(gobject.ext.as(Resolver, p_resolver), p_hostname, p_flags, p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_hostname: [*:0]const u8, p_flags: gio.ResolverNameLookupFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) ?*glib.List) void {
                gobject.ext.as(Resolver.Class, p_class).f_lookup_by_name_with_flags = @ptrCast(p_implementation);
            }
        };

        /// Begins asynchronously resolving `hostname` to determine its
        /// associated IP address(es), and eventually calls `callback`, which
        /// must call `gio.Resolver.lookupByNameWithFlagsFinish` to get the result.
        /// See `gio.Resolver.lookupByName` for more details.
        pub const lookup_by_name_with_flags_async = struct {
            pub fn call(p_class: anytype, p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_hostname: [*:0]const u8, p_flags: gio.ResolverNameLookupFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(Resolver.Class, p_class).f_lookup_by_name_with_flags_async.?(gobject.ext.as(Resolver, p_resolver), p_hostname, p_flags, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_hostname: [*:0]const u8, p_flags: gio.ResolverNameLookupFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(Resolver.Class, p_class).f_lookup_by_name_with_flags_async = @ptrCast(p_implementation);
            }
        };

        /// Retrieves the result of a call to
        /// `gio.Resolver.lookupByNameWithFlagsAsync`.
        ///
        /// If the DNS resolution failed, `error` (if non-`NULL`) will be set to
        /// a value from `gio.ResolverError`. If the operation was cancelled,
        /// `error` will be set to `G_IO_ERROR_CANCELLED`.
        pub const lookup_by_name_with_flags_finish = struct {
            pub fn call(p_class: anytype, p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.List {
                return gobject.ext.as(Resolver.Class, p_class).f_lookup_by_name_with_flags_finish.?(gobject.ext.as(Resolver, p_resolver), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?*glib.List) void {
                gobject.ext.as(Resolver.Class, p_class).f_lookup_by_name_with_flags_finish = @ptrCast(p_implementation);
            }
        };

        /// Synchronously performs a DNS record lookup for the given `rrname` and returns
        /// a list of records as `glib.Variant` tuples. See `gio.ResolverRecordType` for
        /// information on what the records contain for each `record_type`.
        ///
        /// If the DNS resolution fails, `error` (if non-`NULL`) will be set to
        /// a value from `gio.ResolverError` and `NULL` will be returned.
        ///
        /// If `cancellable` is non-`NULL`, it can be used to cancel the
        /// operation, in which case `error` (if non-`NULL`) will be set to
        /// `G_IO_ERROR_CANCELLED`.
        pub const lookup_records = struct {
            pub fn call(p_class: anytype, p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_rrname: [*:0]const u8, p_record_type: gio.ResolverRecordType, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*glib.List {
                return gobject.ext.as(Resolver.Class, p_class).f_lookup_records.?(gobject.ext.as(Resolver, p_resolver), p_rrname, p_record_type, p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_rrname: [*:0]const u8, p_record_type: gio.ResolverRecordType, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) ?*glib.List) void {
                gobject.ext.as(Resolver.Class, p_class).f_lookup_records = @ptrCast(p_implementation);
            }
        };

        /// Begins asynchronously performing a DNS lookup for the given
        /// `rrname`, and eventually calls `callback`, which must call
        /// `gio.Resolver.lookupRecordsFinish` to get the final result. See
        /// `gio.Resolver.lookupRecords` for more details.
        pub const lookup_records_async = struct {
            pub fn call(p_class: anytype, p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_rrname: [*:0]const u8, p_record_type: gio.ResolverRecordType, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(Resolver.Class, p_class).f_lookup_records_async.?(gobject.ext.as(Resolver, p_resolver), p_rrname, p_record_type, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_rrname: [*:0]const u8, p_record_type: gio.ResolverRecordType, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(Resolver.Class, p_class).f_lookup_records_async = @ptrCast(p_implementation);
            }
        };

        /// Retrieves the result of a previous call to
        /// `gio.Resolver.lookupRecordsAsync`. Returns a non-empty list of records as
        /// `glib.Variant` tuples. See `gio.ResolverRecordType` for information on what the
        /// records contain.
        ///
        /// If the DNS resolution failed, `error` (if non-`NULL`) will be set to
        /// a value from `gio.ResolverError`. If the operation was cancelled,
        /// `error` will be set to `G_IO_ERROR_CANCELLED`.
        pub const lookup_records_finish = struct {
            pub fn call(p_class: anytype, p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.List {
                return gobject.ext.as(Resolver.Class, p_class).f_lookup_records_finish.?(gobject.ext.as(Resolver, p_resolver), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?*glib.List) void {
                gobject.ext.as(Resolver.Class, p_class).f_lookup_records_finish = @ptrCast(p_implementation);
            }
        };

        pub const lookup_service = struct {
            pub fn call(p_class: anytype, p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_rrname: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*glib.List {
                return gobject.ext.as(Resolver.Class, p_class).f_lookup_service.?(gobject.ext.as(Resolver, p_resolver), p_rrname, p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_rrname: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) ?*glib.List) void {
                gobject.ext.as(Resolver.Class, p_class).f_lookup_service = @ptrCast(p_implementation);
            }
        };

        pub const lookup_service_async = struct {
            pub fn call(p_class: anytype, p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_rrname: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(Resolver.Class, p_class).f_lookup_service_async.?(gobject.ext.as(Resolver, p_resolver), p_rrname, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_rrname: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(Resolver.Class, p_class).f_lookup_service_async = @ptrCast(p_implementation);
            }
        };

        /// Retrieves the result of a previous call to
        /// `gio.Resolver.lookupServiceAsync`.
        ///
        /// If the DNS resolution failed, `error` (if non-`NULL`) will be set to
        /// a value from `gio.ResolverError`. If the operation was cancelled,
        /// `error` will be set to `G_IO_ERROR_CANCELLED`.
        pub const lookup_service_finish = struct {
            pub fn call(p_class: anytype, p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.List {
                return gobject.ext.as(Resolver.Class, p_class).f_lookup_service_finish.?(gobject.ext.as(Resolver, p_resolver), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?*glib.List) void {
                gobject.ext.as(Resolver.Class, p_class).f_lookup_service_finish = @ptrCast(p_implementation);
            }
        };

        pub const reload = struct {
            pub fn call(p_class: anytype, p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Resolver.Class, p_class).f_reload.?(gobject.ext.as(Resolver, p_resolver));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_resolver: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Resolver.Class, p_class).f_reload = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The timeout applied to all resolver lookups, in milliseconds.
        ///
        /// This may be changed through the lifetime of the `gio.Resolver`. The new value
        /// will apply to any lookups started after the change, but not to any
        /// already-ongoing lookups.
        ///
        /// If this is `0`, no timeout is applied to lookups.
        ///
        /// No timeout was applied to lookups before this property was added in
        /// GLib 2.78.
        pub const timeout = struct {
            pub const name = "timeout";

            pub const Type = c_uint;
        };
    };

    pub const signals = struct {
        /// Emitted when the resolver notices that the system resolver
        /// configuration has changed.
        pub const reload = struct {
            pub const name = "reload";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Resolver, p_instance))),
                    gobject.signalLookup("reload", Resolver.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Frees `addresses` (which should be the return value from
    /// `gio.Resolver.lookupByName` or `gio.Resolver.lookupByNameFinish`).
    /// (This is a convenience method; you can also simply free the results
    /// by hand.)
    extern fn g_resolver_free_addresses(p_addresses: *glib.List) void;
    pub const freeAddresses = g_resolver_free_addresses;

    /// Frees `targets` (which should be the return value from
    /// `gio.Resolver.lookupService` or `gio.Resolver.lookupServiceFinish`).
    /// (This is a convenience method; you can also simply free the
    /// results by hand.)
    extern fn g_resolver_free_targets(p_targets: *glib.List) void;
    pub const freeTargets = g_resolver_free_targets;

    /// Gets the default `gio.Resolver`. You should unref it when you are done
    /// with it. `gio.Resolver` may use its reference count as a hint about how
    /// many threads it should allocate for concurrent DNS resolutions.
    extern fn g_resolver_get_default() *gio.Resolver;
    pub const getDefault = g_resolver_get_default;

    /// Get the timeout applied to all resolver lookups. See `gio.Resolver.properties.timeout`.
    extern fn g_resolver_get_timeout(p_resolver: *Resolver) c_uint;
    pub const getTimeout = g_resolver_get_timeout;

    /// Synchronously reverse-resolves `address` to determine its
    /// associated hostname.
    ///
    /// If the DNS resolution fails, `error` (if non-`NULL`) will be set to
    /// a value from `gio.ResolverError`.
    ///
    /// If `cancellable` is non-`NULL`, it can be used to cancel the
    /// operation, in which case `error` (if non-`NULL`) will be set to
    /// `G_IO_ERROR_CANCELLED`.
    extern fn g_resolver_lookup_by_address(p_resolver: *Resolver, p_address: *gio.InetAddress, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const lookupByAddress = g_resolver_lookup_by_address;

    /// Begins asynchronously reverse-resolving `address` to determine its
    /// associated hostname, and eventually calls `callback`, which must
    /// call `gio.Resolver.lookupByAddressFinish` to get the final result.
    extern fn g_resolver_lookup_by_address_async(p_resolver: *Resolver, p_address: *gio.InetAddress, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const lookupByAddressAsync = g_resolver_lookup_by_address_async;

    /// Retrieves the result of a previous call to
    /// `gio.Resolver.lookupByAddressAsync`.
    ///
    /// If the DNS resolution failed, `error` (if non-`NULL`) will be set to
    /// a value from `gio.ResolverError`. If the operation was cancelled,
    /// `error` will be set to `G_IO_ERROR_CANCELLED`.
    extern fn g_resolver_lookup_by_address_finish(p_resolver: *Resolver, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const lookupByAddressFinish = g_resolver_lookup_by_address_finish;

    /// Synchronously resolves `hostname` to determine its associated IP
    /// address(es). `hostname` may be an ASCII-only or UTF-8 hostname, or
    /// the textual form of an IP address (in which case this just becomes
    /// a wrapper around `gio.InetAddress.newFromString`).
    ///
    /// On success, `gio.Resolver.lookupByName` will return a non-empty `glib.List` of
    /// `gio.InetAddress`, sorted in order of preference and guaranteed to not
    /// contain duplicates. That is, if using the result to connect to
    /// `hostname`, you should attempt to connect to the first address
    /// first, then the second if the first fails, etc. If you are using
    /// the result to listen on a socket, it is appropriate to add each
    /// result using e.g. `gio.SocketListener.addAddress`.
    ///
    /// If the DNS resolution fails, `error` (if non-`NULL`) will be set to a
    /// value from `gio.ResolverError` and `NULL` will be returned.
    ///
    /// If `cancellable` is non-`NULL`, it can be used to cancel the
    /// operation, in which case `error` (if non-`NULL`) will be set to
    /// `G_IO_ERROR_CANCELLED`.
    ///
    /// If you are planning to connect to a socket on the resolved IP
    /// address, it may be easier to create a `gio.NetworkAddress` and use its
    /// `gio.SocketConnectable` interface.
    extern fn g_resolver_lookup_by_name(p_resolver: *Resolver, p_hostname: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*glib.List;
    pub const lookupByName = g_resolver_lookup_by_name;

    /// Begins asynchronously resolving `hostname` to determine its
    /// associated IP address(es), and eventually calls `callback`, which
    /// must call `gio.Resolver.lookupByNameFinish` to get the result.
    /// See `gio.Resolver.lookupByName` for more details.
    extern fn g_resolver_lookup_by_name_async(p_resolver: *Resolver, p_hostname: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const lookupByNameAsync = g_resolver_lookup_by_name_async;

    /// Retrieves the result of a call to
    /// `gio.Resolver.lookupByNameAsync`.
    ///
    /// If the DNS resolution failed, `error` (if non-`NULL`) will be set to
    /// a value from `gio.ResolverError`. If the operation was cancelled,
    /// `error` will be set to `G_IO_ERROR_CANCELLED`.
    extern fn g_resolver_lookup_by_name_finish(p_resolver: *Resolver, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.List;
    pub const lookupByNameFinish = g_resolver_lookup_by_name_finish;

    /// This differs from `gio.Resolver.lookupByName` in that you can modify
    /// the lookup behavior with `flags`. For example this can be used to limit
    /// results with `G_RESOLVER_NAME_LOOKUP_FLAGS_IPV4_ONLY`.
    extern fn g_resolver_lookup_by_name_with_flags(p_resolver: *Resolver, p_hostname: [*:0]const u8, p_flags: gio.ResolverNameLookupFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*glib.List;
    pub const lookupByNameWithFlags = g_resolver_lookup_by_name_with_flags;

    /// Begins asynchronously resolving `hostname` to determine its
    /// associated IP address(es), and eventually calls `callback`, which
    /// must call `gio.Resolver.lookupByNameWithFlagsFinish` to get the result.
    /// See `gio.Resolver.lookupByName` for more details.
    extern fn g_resolver_lookup_by_name_with_flags_async(p_resolver: *Resolver, p_hostname: [*:0]const u8, p_flags: gio.ResolverNameLookupFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const lookupByNameWithFlagsAsync = g_resolver_lookup_by_name_with_flags_async;

    /// Retrieves the result of a call to
    /// `gio.Resolver.lookupByNameWithFlagsAsync`.
    ///
    /// If the DNS resolution failed, `error` (if non-`NULL`) will be set to
    /// a value from `gio.ResolverError`. If the operation was cancelled,
    /// `error` will be set to `G_IO_ERROR_CANCELLED`.
    extern fn g_resolver_lookup_by_name_with_flags_finish(p_resolver: *Resolver, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.List;
    pub const lookupByNameWithFlagsFinish = g_resolver_lookup_by_name_with_flags_finish;

    /// Synchronously performs a DNS record lookup for the given `rrname` and returns
    /// a list of records as `glib.Variant` tuples. See `gio.ResolverRecordType` for
    /// information on what the records contain for each `record_type`.
    ///
    /// If the DNS resolution fails, `error` (if non-`NULL`) will be set to
    /// a value from `gio.ResolverError` and `NULL` will be returned.
    ///
    /// If `cancellable` is non-`NULL`, it can be used to cancel the
    /// operation, in which case `error` (if non-`NULL`) will be set to
    /// `G_IO_ERROR_CANCELLED`.
    extern fn g_resolver_lookup_records(p_resolver: *Resolver, p_rrname: [*:0]const u8, p_record_type: gio.ResolverRecordType, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*glib.List;
    pub const lookupRecords = g_resolver_lookup_records;

    /// Begins asynchronously performing a DNS lookup for the given
    /// `rrname`, and eventually calls `callback`, which must call
    /// `gio.Resolver.lookupRecordsFinish` to get the final result. See
    /// `gio.Resolver.lookupRecords` for more details.
    extern fn g_resolver_lookup_records_async(p_resolver: *Resolver, p_rrname: [*:0]const u8, p_record_type: gio.ResolverRecordType, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const lookupRecordsAsync = g_resolver_lookup_records_async;

    /// Retrieves the result of a previous call to
    /// `gio.Resolver.lookupRecordsAsync`. Returns a non-empty list of records as
    /// `glib.Variant` tuples. See `gio.ResolverRecordType` for information on what the
    /// records contain.
    ///
    /// If the DNS resolution failed, `error` (if non-`NULL`) will be set to
    /// a value from `gio.ResolverError`. If the operation was cancelled,
    /// `error` will be set to `G_IO_ERROR_CANCELLED`.
    extern fn g_resolver_lookup_records_finish(p_resolver: *Resolver, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.List;
    pub const lookupRecordsFinish = g_resolver_lookup_records_finish;

    /// Synchronously performs a DNS SRV lookup for the given `service` and
    /// `protocol` in the given `domain` and returns an array of `gio.SrvTarget`.
    /// `domain` may be an ASCII-only or UTF-8 hostname. Note also that the
    /// `service` and `protocol` arguments do not include the leading underscore
    /// that appears in the actual DNS entry.
    ///
    /// On success, `gio.Resolver.lookupService` will return a non-empty `glib.List` of
    /// `gio.SrvTarget`, sorted in order of preference. (That is, you should
    /// attempt to connect to the first target first, then the second if
    /// the first fails, etc.)
    ///
    /// If the DNS resolution fails, `error` (if non-`NULL`) will be set to
    /// a value from `gio.ResolverError` and `NULL` will be returned.
    ///
    /// If `cancellable` is non-`NULL`, it can be used to cancel the
    /// operation, in which case `error` (if non-`NULL`) will be set to
    /// `G_IO_ERROR_CANCELLED`.
    ///
    /// If you are planning to connect to the service, it is usually easier
    /// to create a `gio.NetworkService` and use its `gio.SocketConnectable`
    /// interface.
    extern fn g_resolver_lookup_service(p_resolver: *Resolver, p_service: [*:0]const u8, p_protocol: [*:0]const u8, p_domain: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*glib.List;
    pub const lookupService = g_resolver_lookup_service;

    /// Begins asynchronously performing a DNS SRV lookup for the given
    /// `service` and `protocol` in the given `domain`, and eventually calls
    /// `callback`, which must call `gio.Resolver.lookupServiceFinish` to
    /// get the final result. See `gio.Resolver.lookupService` for more
    /// details.
    extern fn g_resolver_lookup_service_async(p_resolver: *Resolver, p_service: [*:0]const u8, p_protocol: [*:0]const u8, p_domain: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const lookupServiceAsync = g_resolver_lookup_service_async;

    /// Retrieves the result of a previous call to
    /// `gio.Resolver.lookupServiceAsync`.
    ///
    /// If the DNS resolution failed, `error` (if non-`NULL`) will be set to
    /// a value from `gio.ResolverError`. If the operation was cancelled,
    /// `error` will be set to `G_IO_ERROR_CANCELLED`.
    extern fn g_resolver_lookup_service_finish(p_resolver: *Resolver, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*glib.List;
    pub const lookupServiceFinish = g_resolver_lookup_service_finish;

    /// Sets `resolver` to be the application's default resolver (reffing
    /// `resolver`, and unreffing the previous default resolver, if any).
    /// Future calls to `gio.Resolver.getDefault` will return this resolver.
    ///
    /// This can be used if an application wants to perform any sort of DNS
    /// caching or "pinning"; it can implement its own `gio.Resolver` that
    /// calls the original default resolver for DNS operations, and
    /// implements its own cache policies on top of that, and then set
    /// itself as the default resolver for all later code to use.
    extern fn g_resolver_set_default(p_resolver: *Resolver) void;
    pub const setDefault = g_resolver_set_default;

    /// Set the timeout applied to all resolver lookups. See `gio.Resolver.properties.timeout`.
    extern fn g_resolver_set_timeout(p_resolver: *Resolver, p_timeout_ms: c_uint) void;
    pub const setTimeout = g_resolver_set_timeout;

    extern fn g_resolver_get_type() usize;
    pub const getGObjectType = g_resolver_get_type;

    extern fn g_object_ref(p_self: *gio.Resolver) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.Resolver) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Resolver, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `GSettings` class provides a convenient API for storing and retrieving
/// application settings.
///
/// Reads and writes can be considered to be non-blocking.  Reading
/// settings with `GSettings` is typically extremely fast: on
/// approximately the same order of magnitude (but slower than) a
/// `glib.HashTable` lookup.  Writing settings is also extremely fast in
/// terms of time to return to your application, but can be extremely expensive
/// for other threads and other processes.  Many settings backends
/// (including dconf) have lazy initialisation which means in the common
/// case of the user using their computer without modifying any settings
/// a lot of work can be avoided.  For dconf, the D-Bus service doesn’t
/// even need to be started in this case.  For this reason, you should
/// only ever modify `GSettings` keys in response to explicit user action.
/// Particular care should be paid to ensure that modifications are not
/// made during startup — for example, when setting the initial value
/// of preferences widgets.  The built-in `gio.Settings.bind`
/// functionality is careful not to write settings in response to notify signals
/// as a result of modifications that it makes to widgets.
///
/// When creating a `GSettings` instance, you have to specify a schema
/// that describes the keys in your settings and their types and default
/// values, as well as some other information.
///
/// Normally, a schema has a fixed path that determines where the settings
/// are stored in the conceptual global tree of settings. However, schemas
/// can also be ‘[relocatable](`relocatable`-schemas)’, i.e. not equipped with
/// a fixed path. This is
/// useful e.g. when the schema describes an ‘account’, and you want to be
/// able to store a arbitrary number of accounts.
///
/// Paths must start with and end with a forward slash character (`/`)
/// and must not contain two sequential slash characters.  Paths should
/// be chosen based on a domain name associated with the program or
/// library to which the settings belong.  Examples of paths are
/// `/org/gtk/settings/file-chooser/` and `/ca/desrt/dconf-editor/`.
/// Paths should not start with `/apps/`, `/desktop/` or `/system/` as
/// they often did in GConf.
///
/// Unlike other configuration systems (like GConf), GSettings does not
/// restrict keys to basic types like strings and numbers. GSettings stores
/// values as `glib.Variant`, and allows any `glib.VariantType` for
/// keys. Key names are restricted to lowercase characters, numbers and `-`.
/// Furthermore, the names must begin with a lowercase character, must not end
/// with a `-`, and must not contain consecutive dashes.
///
/// Similar to GConf, the default values in GSettings schemas can be
/// localized, but the localized values are stored in gettext catalogs
/// and looked up with the domain that is specified in the
/// `gettext-domain` attribute of the `<schemalist>` or `<schema>`
/// elements and the category that is specified in the `l10n` attribute of
/// the `<default>` element. The string which is translated includes all text in
/// the `<default>` element, including any surrounding quotation marks.
///
/// The `l10n` attribute must be set to `messages` or `time`, and sets the
/// [locale category for
/// translation](https://www.gnu.org/software/gettext/manual/html_node/Aspects.html`index`-locale-categories-1).
/// The `messages` category should be used by default; use `time` for
/// translatable date or time formats. A translation comment can be added as an
/// XML comment immediately above the `<default>` element — it is recommended to
/// add these comments to aid translators understand the meaning and
/// implications of the default value. An optional translation `context`
/// attribute can be set on the `<default>` element to disambiguate multiple
/// defaults which use the same string.
///
/// For example:
/// ```xml
///  <!-- Translators: A list of words which are not allowed to be typed, in
///       GVariant serialization syntax.
///       See: https://developer.gnome.org/glib/stable/gvariant-text.html -->
///  <default l10n='messages' context='Banned words'>['bad', 'words']</default>
/// ```
///
/// Translations of default values must remain syntactically valid serialized
/// `glib.Variant`s (e.g. retaining any surrounding quotation marks) or
/// runtime errors will occur.
///
/// GSettings uses schemas in a compact binary form that is created
/// by the [`glib-compile-schemas`](glib-compile-schemas.html)
/// utility. The input is a schema description in an XML format.
///
/// A DTD for the gschema XML format can be found here:
/// [gschema.dtd](https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/gschema.dtd)
///
/// The [`glib-compile-schemas`](glib-compile-schemas.html) tool expects schema
/// files to have the extension `.gschema.xml`.
///
/// At runtime, schemas are identified by their ID (as specified in the
/// `id` attribute of the `<schema>` element). The convention for schema
/// IDs is to use a dotted name, similar in style to a D-Bus bus name,
/// e.g. `org.gnome.SessionManager`. In particular, if the settings are
/// for a specific service that owns a D-Bus bus name, the D-Bus bus name
/// and schema ID should match. For schemas which deal with settings not
/// associated with one named application, the ID should not use
/// StudlyCaps, e.g. `org.gnome.font-rendering`.
///
/// In addition to `glib.Variant` types, keys can have types that have
/// enumerated types. These can be described by a `<choice>`,
/// `<enum>` or `<flags>` element, as seen in the
/// second example below. The underlying type of such a key
/// is string, but you can use `gio.Settings.getEnum`,
/// `gio.Settings.setEnum`, `gio.Settings.getFlags`,
/// `gio.Settings.setFlags` access the numeric values corresponding to
/// the string value of enum and flags keys.
///
/// An example for default value:
/// ```xml
/// <schemalist>
///   <schema id="org.gtk.Test" path="/org/gtk/Test/" gettext-domain="test">
///
///     <key name="greeting" type="s">
///       <default l10n="messages">"Hello, earthlings"</default>
///       <summary>A greeting</summary>
///       <description>
///         Greeting of the invading martians
///       </description>
///     </key>
///
///     <key name="box" type="(ii)">
///       <default>(20,30)</default>
///     </key>
///
///     <key name="empty-string" type="s">
///       <default>""</default>
///       <summary>Empty strings have to be provided in GVariant form</summary>
///     </key>
///
///   </schema>
/// </schemalist>
/// ```
///
/// An example for ranges, choices and enumerated types:
/// ```xml
/// <schemalist>
///
///   <enum id="org.gtk.Test.myenum">
///     <value nick="first" value="1"/>
///     <value nick="second" value="2"/>
///   </enum>
///
///   <flags id="org.gtk.Test.myflags">
///     <value nick="flag1" value="1"/>
///     <value nick="flag2" value="2"/>
///     <value nick="flag3" value="4"/>
///   </flags>
///
///   <schema id="org.gtk.Test">
///
///     <key name="key-with-range" type="i">
///       <range min="1" max="100"/>
///       <default>10</default>
///     </key>
///
///     <key name="key-with-choices" type="s">
///       <choices>
///         <choice value='Elisabeth'/>
///         <choice value='Annabeth'/>
///         <choice value='Joe'/>
///       </choices>
///       <aliases>
///         <alias value='Anna' target='Annabeth'/>
///         <alias value='Beth' target='Elisabeth'/>
///       </aliases>
///       <default>'Joe'</default>
///     </key>
///
///     <key name='enumerated-key' enum='org.gtk.Test.myenum'>
///       <default>'first'</default>
///     </key>
///
///     <key name='flags-key' flags='org.gtk.Test.myflags'>
///       <default>["flag1","flag2"]</default>
///     </key>
///   </schema>
/// </schemalist>
/// ```
///
/// ## Vendor overrides
///
/// Default values are defined in the schemas that get installed by
/// an application. Sometimes, it is necessary for a vendor or distributor
/// to adjust these defaults. Since patching the XML source for the schema
/// is inconvenient and error-prone,
/// [`glib-compile-schemas`](glib-compile-schemas.html) reads so-called ‘vendor
/// override’ files. These are keyfiles in the same directory as the XML
/// schema sources which can override default values. The schema ID serves
/// as the group name in the key file, and the values are expected in
/// serialized `glib.Variant` form, as in the following example:
/// ```
/// [org.gtk.Example]
/// key1='string'
/// key2=1.5
/// ```
///
/// `glib-compile-schemas` expects schema files to have the extension
/// `.gschema.override`.
///
/// ## Binding
///
/// A very convenient feature of GSettings lets you bind `gobject.Object`
/// properties directly to settings, using `gio.Settings.bind`. Once a
/// `gobject.Object` property has been bound to a setting, changes on
/// either side are automatically propagated to the other side. GSettings handles
/// details like mapping between `gobject.Object` and `glib.Variant`
/// types, and preventing infinite cycles.
///
/// This makes it very easy to hook up a preferences dialog to the
/// underlying settings. To make this even more convenient, GSettings
/// looks for a boolean property with the name `sensitivity` and
/// automatically binds it to the writability of the bound setting.
/// If this ‘magic’ gets in the way, it can be suppressed with the
/// `G_SETTINGS_BIND_NO_SENSITIVITY` flag.
///
/// ## Relocatable schemas
///
/// A relocatable schema is one with no `path` attribute specified on its
/// `<schema>` element. By using `gio.Settings.newWithPath`, a `GSettings`
/// object can be instantiated for a relocatable schema, assigning a path to the
/// instance. Paths passed to `gio.Settings.newWithPath` will typically be
/// constructed dynamically from a constant prefix plus some form of instance
/// identifier; but they must still be valid GSettings paths. Paths could also
/// be constant and used with a globally installed schema originating from a
/// dependency library.
///
/// For example, a relocatable schema could be used to store geometry information
/// for different windows in an application. If the schema ID was
/// `org.foo.MyApp.Window`, it could be instantiated for paths
/// `/org/foo/MyApp/main/`, `/org/foo/MyApp/document-1/`,
/// `/org/foo/MyApp/document-2/`, etc. If any of the paths are well-known
/// they can be specified as `<child>` elements in the parent schema, e.g.:
/// ```xml
/// <schema id="org.foo.MyApp" path="/org/foo/MyApp/">
///   <child name="main" schema="org.foo.MyApp.Window"/>
/// </schema>
/// ```
///
/// ## Build system integration
///
/// GSettings comes with autotools integration to simplify compiling and
/// installing schemas. To add GSettings support to an application, add the
/// following to your `configure.ac`:
/// ```
/// GLIB_GSETTINGS
/// ```
///
/// In the appropriate `Makefile.am`, use the following snippet to compile and
/// install the named schema:
/// ```
/// gsettings_SCHEMAS = org.foo.MyApp.gschema.xml
/// EXTRA_DIST = $(gsettings_SCHEMAS)
///
/// `GSETTINGS_RULES`@
/// ```
///
/// No changes are needed to the build system to mark a schema XML file for
/// translation. Assuming it sets the `gettext-domain` attribute, a schema may
/// be marked for translation by adding it to `POTFILES.in`, assuming gettext
/// 0.19 is in use (the preferred method for translation):
/// ```
/// data/org.foo.MyApp.gschema.xml
/// ```
///
/// Alternatively, if intltool 0.50.1 is in use:
/// ```
/// [type: gettext/gsettings]data/org.foo.MyApp.gschema.xml
/// ```
///
/// GSettings will use gettext to look up translations for the `<summary>` and
/// `<description>` elements, and also any `<default>` elements which have a
/// `l10n` attribute set. Translations must not be included in the `.gschema.xml`
/// file by the build system, for example by using intltool XML rules with a
/// `.gschema.xml.in` template.
///
/// If an enumerated type defined in a C header file is to be used in a GSettings
/// schema, it can either be defined manually using an `<enum>` element in the
/// schema XML, or it can be extracted automatically from the C header. This
/// approach is preferred, as it ensures the two representations are always
/// synchronised. To do so, add the following to the relevant `Makefile.am`:
/// ```
/// gsettings_ENUM_NAMESPACE = org.foo.MyApp
/// gsettings_ENUM_FILES = my-app-enums.h my-app-misc.h
/// ```
///
/// `gsettings_ENUM_NAMESPACE` specifies the schema namespace for the enum files,
/// which are specified in `gsettings_ENUM_FILES`. This will generate a
/// `org.foo.MyApp.enums.xml` file containing the extracted enums, which will be
/// automatically included in the schema compilation, install and uninstall
/// rules. It should not be committed to version control or included in
/// `EXTRA_DIST`.
pub const Settings = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.SettingsClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.SettingsPrivate,

    pub const virtual_methods = struct {
        pub const change_event = struct {
            pub fn call(p_class: anytype, p_settings: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_keys: *const glib.Quark, p_n_keys: c_int) c_int {
                return gobject.ext.as(Settings.Class, p_class).f_change_event.?(gobject.ext.as(Settings, p_settings), p_keys, p_n_keys);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_settings: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_keys: *const glib.Quark, p_n_keys: c_int) callconv(.C) c_int) void {
                gobject.ext.as(Settings.Class, p_class).f_change_event = @ptrCast(p_implementation);
            }
        };

        pub const changed = struct {
            pub fn call(p_class: anytype, p_settings: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_key: [*:0]const u8) void {
                return gobject.ext.as(Settings.Class, p_class).f_changed.?(gobject.ext.as(Settings, p_settings), p_key);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_settings: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_key: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(Settings.Class, p_class).f_changed = @ptrCast(p_implementation);
            }
        };

        pub const writable_change_event = struct {
            pub fn call(p_class: anytype, p_settings: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_key: glib.Quark) c_int {
                return gobject.ext.as(Settings.Class, p_class).f_writable_change_event.?(gobject.ext.as(Settings, p_settings), p_key);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_settings: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_key: glib.Quark) callconv(.C) c_int) void {
                gobject.ext.as(Settings.Class, p_class).f_writable_change_event = @ptrCast(p_implementation);
            }
        };

        pub const writable_changed = struct {
            pub fn call(p_class: anytype, p_settings: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_key: [*:0]const u8) void {
                return gobject.ext.as(Settings.Class, p_class).f_writable_changed.?(gobject.ext.as(Settings, p_settings), p_key);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_settings: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_key: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(Settings.Class, p_class).f_writable_changed = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The name of the context that the settings are stored in.
        pub const backend = struct {
            pub const name = "backend";

            pub const Type = ?*gio.SettingsBackend;
        };

        /// Whether the `gio.Settings` object is in 'delay-apply' mode. See
        /// `gio.Settings.delay` for details.
        pub const delay_apply = struct {
            pub const name = "delay-apply";

            pub const Type = c_int;
        };

        /// If this property is `TRUE`, the `gio.Settings` object has outstanding
        /// changes that will be applied when `gio.Settings.apply` is called.
        pub const has_unapplied = struct {
            pub const name = "has-unapplied";

            pub const Type = c_int;
        };

        /// The path within the backend where the settings are stored.
        pub const path = struct {
            pub const name = "path";

            pub const Type = ?[*:0]u8;
        };

        /// The name of the schema that describes the types of keys
        /// for this `gio.Settings` object.
        ///
        /// The type of this property is *not* `gio.SettingsSchema`.
        /// `gio.SettingsSchema` has only existed since version 2.32 and
        /// unfortunately this name was used in previous versions to refer to
        /// the schema ID rather than the schema itself.  Take care to use the
        /// 'settings-schema' property if you wish to pass in a
        /// `gio.SettingsSchema`.
        pub const schema = struct {
            pub const name = "schema";

            pub const Type = ?[*:0]u8;
        };

        /// The name of the schema that describes the types of keys
        /// for this `gio.Settings` object.
        pub const schema_id = struct {
            pub const name = "schema-id";

            pub const Type = ?[*:0]u8;
        };

        /// The `gio.SettingsSchema` describing the types of keys for this
        /// `gio.Settings` object.
        ///
        /// Ideally, this property would be called 'schema'.  `gio.SettingsSchema`
        /// has only existed since version 2.32, however, and before then the
        /// 'schema' property was used to refer to the ID of the schema rather
        /// than the schema itself.  Take care.
        pub const settings_schema = struct {
            pub const name = "settings-schema";

            pub const Type = ?*gio.SettingsSchema;
        };
    };

    pub const signals = struct {
        /// The "change-event" signal is emitted once per change event that
        /// affects this settings object.  You should connect to this signal
        /// only if you are interested in viewing groups of changes before they
        /// are split out into multiple emissions of the "changed" signal.
        /// For most use cases it is more appropriate to use the "changed" signal.
        ///
        /// In the event that the change event applies to one or more specified
        /// keys, `keys` will be an array of `glib.Quark` of length `n_keys`.  In the
        /// event that the change event applies to the `gio.Settings` object as a
        /// whole (ie: potentially every key has been changed) then `keys` will
        /// be `NULL` and `n_keys` will be 0.
        ///
        /// The default handler for this signal invokes the "changed" signal
        /// for each affected key.  If any other connected handler returns
        /// `TRUE` then this default functionality will be suppressed.
        pub const change_event = struct {
            pub const name = "change-event";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_keys: ?[*]glib.Quark, p_n_keys: c_int, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Settings, p_instance))),
                    gobject.signalLookup("change-event", Settings.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The "changed" signal is emitted when a key has potentially changed.
        /// You should call one of the `gio.Settings.get` calls to check the new
        /// value.
        ///
        /// This signal supports detailed connections.  You can connect to the
        /// detailed signal "changed::x" in order to only receive callbacks
        /// when key "x" changes.
        ///
        /// Note that `settings` only emits this signal if you have read `key` at
        /// least once while a signal handler was already connected for `key`.
        pub const changed = struct {
            pub const name = "changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_key: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Settings, p_instance))),
                    gobject.signalLookup("changed", Settings.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The "writable-change-event" signal is emitted once per writability
        /// change event that affects this settings object.  You should connect
        /// to this signal if you are interested in viewing groups of changes
        /// before they are split out into multiple emissions of the
        /// "writable-changed" signal.  For most use cases it is more
        /// appropriate to use the "writable-changed" signal.
        ///
        /// In the event that the writability change applies only to a single
        /// key, `key` will be set to the `glib.Quark` for that key.  In the event
        /// that the writability change affects the entire settings object,
        /// `key` will be 0.
        ///
        /// The default handler for this signal invokes the "writable-changed"
        /// and "changed" signals for each affected key.  This is done because
        /// changes in writability might also imply changes in value (if for
        /// example, a new mandatory setting is introduced).  If any other
        /// connected handler returns `TRUE` then this default functionality
        /// will be suppressed.
        pub const writable_change_event = struct {
            pub const name = "writable-change-event";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_key: c_uint, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Settings, p_instance))),
                    gobject.signalLookup("writable-change-event", Settings.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// The "writable-changed" signal is emitted when the writability of a
        /// key has potentially changed.  You should call
        /// `gio.Settings.isWritable` in order to determine the new status.
        ///
        /// This signal supports detailed connections.  You can connect to the
        /// detailed signal "writable-changed::x" in order to only receive
        /// callbacks when the writability of "x" changes.
        pub const writable_changed = struct {
            pub const name = "writable-changed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_key: [*:0]u8, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Settings, p_instance))),
                    gobject.signalLookup("writable-changed", Settings.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Deprecated.
    extern fn g_settings_list_relocatable_schemas() [*]const [*:0]const u8;
    pub const listRelocatableSchemas = g_settings_list_relocatable_schemas;

    /// Deprecated.
    extern fn g_settings_list_schemas() [*]const [*:0]const u8;
    pub const listSchemas = g_settings_list_schemas;

    /// Ensures that all pending operations are complete for the default backend.
    ///
    /// Writes made to a `gio.Settings` are handled asynchronously.  For this
    /// reason, it is very unlikely that the changes have it to disk by the
    /// time `gio.Settings.set` returns.
    ///
    /// This call will block until all of the writes have made it to the
    /// backend.  Since the mainloop is not running, no change notifications
    /// will be dispatched during this call (but some may be queued by the
    /// time the call is done).
    extern fn g_settings_sync() void;
    pub const sync = g_settings_sync;

    /// Removes an existing binding for `property` on `object`.
    ///
    /// Note that bindings are automatically removed when the
    /// object is finalized, so it is rarely necessary to call this
    /// function.
    extern fn g_settings_unbind(p_object: *gobject.Object, p_property: [*:0]const u8) void;
    pub const unbind = g_settings_unbind;

    /// Creates a new `gio.Settings` object with the schema specified by
    /// `schema_id`.
    ///
    /// It is an error for the schema to not exist: schemas are an
    /// essential part of a program, as they provide type information.
    /// If schemas need to be dynamically loaded (for example, from an
    /// optional runtime dependency), `gio.SettingsSchemaSource.lookup`
    /// can be used to test for their existence before loading them.
    ///
    /// Signals on the newly created `gio.Settings` object will be dispatched
    /// via the thread-default `glib.MainContext` in effect at the time of the
    /// call to `gio.Settings.new`.  The new `gio.Settings` will hold a reference
    /// on the context.  See `glib.MainContext.pushThreadDefault`.
    extern fn g_settings_new(p_schema_id: [*:0]const u8) *gio.Settings;
    pub const new = g_settings_new;

    /// Creates a new `gio.Settings` object with a given schema, backend and
    /// path.
    ///
    /// It should be extremely rare that you ever want to use this function.
    /// It is made available for advanced use-cases (such as plugin systems
    /// that want to provide access to schemas loaded from custom locations,
    /// etc).
    ///
    /// At the most basic level, a `gio.Settings` object is a pure composition of
    /// 4 things: a `gio.SettingsSchema`, a `gio.SettingsBackend`, a path within that
    /// backend, and a `glib.MainContext` to which signals are dispatched.
    ///
    /// This constructor therefore gives you full control over constructing
    /// `gio.Settings` instances.  The first 3 parameters are given directly as
    /// `schema`, `backend` and `path`, and the main context is taken from the
    /// thread-default (as per `gio.Settings.new`).
    ///
    /// If `backend` is `NULL` then the default backend is used.
    ///
    /// If `path` is `NULL` then the path from the schema is used.  It is an
    /// error if `path` is `NULL` and the schema has no path of its own or if
    /// `path` is non-`NULL` and not equal to the path that the schema does
    /// have.
    extern fn g_settings_new_full(p_schema: *gio.SettingsSchema, p_backend: ?*gio.SettingsBackend, p_path: ?[*:0]const u8) *gio.Settings;
    pub const newFull = g_settings_new_full;

    /// Creates a new `gio.Settings` object with the schema specified by
    /// `schema_id` and a given `gio.SettingsBackend`.
    ///
    /// Creating a `gio.Settings` object with a different backend allows accessing
    /// settings from a database other than the usual one. For example, it may make
    /// sense to pass a backend corresponding to the "defaults" settings database on
    /// the system to get a settings object that modifies the system default
    /// settings instead of the settings for this user.
    extern fn g_settings_new_with_backend(p_schema_id: [*:0]const u8, p_backend: *gio.SettingsBackend) *gio.Settings;
    pub const newWithBackend = g_settings_new_with_backend;

    /// Creates a new `gio.Settings` object with the schema specified by
    /// `schema_id` and a given `gio.SettingsBackend` and path.
    ///
    /// This is a mix of `gio.Settings.newWithBackend` and
    /// `gio.Settings.newWithPath`.
    extern fn g_settings_new_with_backend_and_path(p_schema_id: [*:0]const u8, p_backend: *gio.SettingsBackend, p_path: [*:0]const u8) *gio.Settings;
    pub const newWithBackendAndPath = g_settings_new_with_backend_and_path;

    /// Creates a new `gio.Settings` object with the relocatable schema specified
    /// by `schema_id` and a given path.
    ///
    /// You only need to do this if you want to directly create a settings
    /// object with a schema that doesn't have a specified path of its own.
    /// That's quite rare.
    ///
    /// It is a programmer error to call this function for a schema that
    /// has an explicitly specified path.
    ///
    /// It is a programmer error if `path` is not a valid path.  A valid path
    /// begins and ends with '/' and does not contain two consecutive '/'
    /// characters.
    extern fn g_settings_new_with_path(p_schema_id: [*:0]const u8, p_path: [*:0]const u8) *gio.Settings;
    pub const newWithPath = g_settings_new_with_path;

    /// Applies any changes that have been made to the settings.  This
    /// function does nothing unless `settings` is in 'delay-apply' mode;
    /// see `gio.Settings.delay`.  In the normal case settings are always
    /// applied immediately.
    extern fn g_settings_apply(p_settings: *Settings) void;
    pub const apply = g_settings_apply;

    /// Create a binding between the `key` in the `settings` object
    /// and the property `property` of `object`.
    ///
    /// The binding uses the default GIO mapping functions to map
    /// between the settings and property values. These functions
    /// handle booleans, numeric types and string types in a
    /// straightforward way. Use `gio.Settings.bindWithMapping` if
    /// you need a custom mapping, or map between types that are not
    /// supported by the default mapping functions.
    ///
    /// Unless the `flags` include `G_SETTINGS_BIND_NO_SENSITIVITY`, this
    /// function also establishes a binding between the writability of
    /// `key` and the "sensitive" property of `object` (if `object` has
    /// a boolean property by that name). See `gio.Settings.bindWritable`
    /// for more details about writable bindings.
    ///
    /// Note that the lifecycle of the binding is tied to `object`,
    /// and that you can have only one binding per object property.
    /// If you bind the same property twice on the same object, the second
    /// binding overrides the first one.
    extern fn g_settings_bind(p_settings: *Settings, p_key: [*:0]const u8, p_object: *gobject.Object, p_property: [*:0]const u8, p_flags: gio.SettingsBindFlags) void;
    pub const bind = g_settings_bind;

    /// Create a binding between the `key` in the `settings` object
    /// and the property `property` of `object`.
    ///
    /// The binding uses the provided mapping functions to map between
    /// settings and property values.
    ///
    /// Note that the lifecycle of the binding is tied to `object`,
    /// and that you can have only one binding per object property.
    /// If you bind the same property twice on the same object, the second
    /// binding overrides the first one.
    extern fn g_settings_bind_with_mapping(p_settings: *Settings, p_key: [*:0]const u8, p_object: *gobject.Object, p_property: [*:0]const u8, p_flags: gio.SettingsBindFlags, p_get_mapping: gio.SettingsBindGetMapping, p_set_mapping: gio.SettingsBindSetMapping, p_user_data: ?*anyopaque, p_destroy: ?glib.DestroyNotify) void;
    pub const bindWithMapping = g_settings_bind_with_mapping;

    /// Version of `gio.Settings.bindWithMapping` using closures instead of callbacks
    /// for easier binding in other languages.
    extern fn g_settings_bind_with_mapping_closures(p_settings: *Settings, p_key: [*:0]const u8, p_object: *gobject.Object, p_property: [*:0]const u8, p_flags: gio.SettingsBindFlags, p_get_mapping: ?*gobject.Closure, p_set_mapping: ?*gobject.Closure) void;
    pub const bindWithMappingClosures = g_settings_bind_with_mapping_closures;

    /// Create a binding between the writability of `key` in the
    /// `settings` object and the property `property` of `object`.
    /// The property must be boolean; "sensitive" or "visible"
    /// properties of widgets are the most likely candidates.
    ///
    /// Writable bindings are always uni-directional; changes of the
    /// writability of the setting will be propagated to the object
    /// property, not the other way.
    ///
    /// When the `inverted` argument is `TRUE`, the binding inverts the
    /// value as it passes from the setting to the object, i.e. `property`
    /// will be set to `TRUE` if the key is not writable.
    ///
    /// Note that the lifecycle of the binding is tied to `object`,
    /// and that you can have only one binding per object property.
    /// If you bind the same property twice on the same object, the second
    /// binding overrides the first one.
    extern fn g_settings_bind_writable(p_settings: *Settings, p_key: [*:0]const u8, p_object: *gobject.Object, p_property: [*:0]const u8, p_inverted: c_int) void;
    pub const bindWritable = g_settings_bind_writable;

    /// Creates a `gio.Action` corresponding to a given `gio.Settings` key.
    ///
    /// The action has the same name as the key.
    ///
    /// The value of the key becomes the state of the action and the action
    /// is enabled when the key is writable.  Changing the state of the
    /// action results in the key being written to.  Changes to the value or
    /// writability of the key cause appropriate change notifications to be
    /// emitted for the action.
    ///
    /// For boolean-valued keys, action activations take no parameter and
    /// result in the toggling of the value.  For all other types,
    /// activations take the new value for the key (which must have the
    /// correct type).
    extern fn g_settings_create_action(p_settings: *Settings, p_key: [*:0]const u8) *gio.Action;
    pub const createAction = g_settings_create_action;

    /// Changes the `gio.Settings` object into 'delay-apply' mode. In this
    /// mode, changes to `settings` are not immediately propagated to the
    /// backend, but kept locally until `gio.Settings.apply` is called.
    extern fn g_settings_delay(p_settings: *Settings) void;
    pub const delay = g_settings_delay;

    /// Gets the value that is stored at `key` in `settings`.
    ///
    /// A convenience function that combines `gio.Settings.getValue` with
    /// `glib.Variant.get`.
    ///
    /// It is a programmer error to give a `key` that isn't contained in the
    /// schema for `settings` or for the `glib.VariantType` of `format` to mismatch
    /// the type given in the schema.
    extern fn g_settings_get(p_settings: *Settings, p_key: [*:0]const u8, p_format: [*:0]const u8, ...) void;
    pub const get = g_settings_get;

    /// Gets the value that is stored at `key` in `settings`.
    ///
    /// A convenience variant of `gio.Settings.get` for booleans.
    ///
    /// It is a programmer error to give a `key` that isn't specified as
    /// having a boolean type in the schema for `settings`.
    extern fn g_settings_get_boolean(p_settings: *Settings, p_key: [*:0]const u8) c_int;
    pub const getBoolean = g_settings_get_boolean;

    /// Creates a child settings object which has a base path of
    /// `base-path/`name``, where `base-path` is the base path of
    /// `settings`.
    ///
    /// The schema for the child settings object must have been declared
    /// in the schema of `settings` using a `<child>` element.
    ///
    /// The created child settings object will inherit the `gio.Settings.properties.delay`-apply
    /// mode from `settings`.
    extern fn g_settings_get_child(p_settings: *Settings, p_name: [*:0]const u8) *gio.Settings;
    pub const getChild = g_settings_get_child;

    /// Gets the "default value" of a key.
    ///
    /// This is the value that would be read if `gio.Settings.reset` were to be
    /// called on the key.
    ///
    /// Note that this may be a different value than returned by
    /// `gio.SettingsSchemaKey.getDefaultValue` if the system administrator
    /// has provided a default value.
    ///
    /// Comparing the return values of `gio.Settings.getDefaultValue` and
    /// `gio.Settings.getValue` is not sufficient for determining if a value
    /// has been set because the user may have explicitly set the value to
    /// something that happens to be equal to the default.  The difference
    /// here is that if the default changes in the future, the user's key
    /// will still be set.
    ///
    /// This function may be useful for adding an indication to a UI of what
    /// the default value was before the user set it.
    ///
    /// It is a programmer error to give a `key` that isn't contained in the
    /// schema for `settings`.
    extern fn g_settings_get_default_value(p_settings: *Settings, p_key: [*:0]const u8) ?*glib.Variant;
    pub const getDefaultValue = g_settings_get_default_value;

    /// Gets the value that is stored at `key` in `settings`.
    ///
    /// A convenience variant of `gio.Settings.get` for doubles.
    ///
    /// It is a programmer error to give a `key` that isn't specified as
    /// having a 'double' type in the schema for `settings`.
    extern fn g_settings_get_double(p_settings: *Settings, p_key: [*:0]const u8) f64;
    pub const getDouble = g_settings_get_double;

    /// Gets the value that is stored in `settings` for `key` and converts it
    /// to the enum value that it represents.
    ///
    /// In order to use this function the type of the value must be a string
    /// and it must be marked in the schema file as an enumerated type.
    ///
    /// It is a programmer error to give a `key` that isn't contained in the
    /// schema for `settings` or is not marked as an enumerated type.
    ///
    /// If the value stored in the configuration database is not a valid
    /// value for the enumerated type then this function will return the
    /// default value.
    extern fn g_settings_get_enum(p_settings: *Settings, p_key: [*:0]const u8) c_int;
    pub const getEnum = g_settings_get_enum;

    /// Gets the value that is stored in `settings` for `key` and converts it
    /// to the flags value that it represents.
    ///
    /// In order to use this function the type of the value must be an array
    /// of strings and it must be marked in the schema file as a flags type.
    ///
    /// It is a programmer error to give a `key` that isn't contained in the
    /// schema for `settings` or is not marked as a flags type.
    ///
    /// If the value stored in the configuration database is not a valid
    /// value for the flags type then this function will return the default
    /// value.
    extern fn g_settings_get_flags(p_settings: *Settings, p_key: [*:0]const u8) c_uint;
    pub const getFlags = g_settings_get_flags;

    /// Returns whether the `gio.Settings` object has any unapplied
    /// changes.  This can only be the case if it is in 'delayed-apply' mode.
    extern fn g_settings_get_has_unapplied(p_settings: *Settings) c_int;
    pub const getHasUnapplied = g_settings_get_has_unapplied;

    /// Gets the value that is stored at `key` in `settings`.
    ///
    /// A convenience variant of `gio.Settings.get` for 32-bit integers.
    ///
    /// It is a programmer error to give a `key` that isn't specified as
    /// having a int32 type in the schema for `settings`.
    extern fn g_settings_get_int(p_settings: *Settings, p_key: [*:0]const u8) c_int;
    pub const getInt = g_settings_get_int;

    /// Gets the value that is stored at `key` in `settings`.
    ///
    /// A convenience variant of `gio.Settings.get` for 64-bit integers.
    ///
    /// It is a programmer error to give a `key` that isn't specified as
    /// having a int64 type in the schema for `settings`.
    extern fn g_settings_get_int64(p_settings: *Settings, p_key: [*:0]const u8) i64;
    pub const getInt64 = g_settings_get_int64;

    /// Gets the value that is stored at `key` in `settings`, subject to
    /// application-level validation/mapping.
    ///
    /// You should use this function when the application needs to perform
    /// some processing on the value of the key (for example, parsing).  The
    /// `mapping` function performs that processing.  If the function
    /// indicates that the processing was unsuccessful (due to a parse error,
    /// for example) then the mapping is tried again with another value.
    ///
    /// This allows a robust 'fall back to defaults' behaviour to be
    /// implemented somewhat automatically.
    ///
    /// The first value that is tried is the user's setting for the key.  If
    /// the mapping function fails to map this value, other values may be
    /// tried in an unspecified order (system or site defaults, translated
    /// schema default values, untranslated schema default values, etc).
    ///
    /// If the mapping function fails for all possible values, one additional
    /// attempt is made: the mapping function is called with a `NULL` value.
    /// If the mapping function still indicates failure at this point then
    /// the application will be aborted.
    ///
    /// The result parameter for the `mapping` function is pointed to a
    /// `gpointer` which is initially set to `NULL`.  The same pointer is given
    /// to each invocation of `mapping`.  The final value of that `gpointer` is
    /// what is returned by this function.  `NULL` is valid; it is returned
    /// just as any other value would be.
    extern fn g_settings_get_mapped(p_settings: *Settings, p_key: [*:0]const u8, p_mapping: gio.SettingsGetMapping, p_user_data: ?*anyopaque) ?*anyopaque;
    pub const getMapped = g_settings_get_mapped;

    /// Queries the range of a key.
    extern fn g_settings_get_range(p_settings: *Settings, p_key: [*:0]const u8) *glib.Variant;
    pub const getRange = g_settings_get_range;

    /// Gets the value that is stored at `key` in `settings`.
    ///
    /// A convenience variant of `gio.Settings.get` for strings.
    ///
    /// It is a programmer error to give a `key` that isn't specified as
    /// having a string type in the schema for `settings`.
    extern fn g_settings_get_string(p_settings: *Settings, p_key: [*:0]const u8) [*:0]u8;
    pub const getString = g_settings_get_string;

    /// A convenience variant of `gio.Settings.get` for string arrays.
    ///
    /// It is a programmer error to give a `key` that isn't specified as
    /// having an array of strings type in the schema for `settings`.
    extern fn g_settings_get_strv(p_settings: *Settings, p_key: [*:0]const u8) [*][*:0]u8;
    pub const getStrv = g_settings_get_strv;

    /// Gets the value that is stored at `key` in `settings`.
    ///
    /// A convenience variant of `gio.Settings.get` for 32-bit unsigned
    /// integers.
    ///
    /// It is a programmer error to give a `key` that isn't specified as
    /// having a uint32 type in the schema for `settings`.
    extern fn g_settings_get_uint(p_settings: *Settings, p_key: [*:0]const u8) c_uint;
    pub const getUint = g_settings_get_uint;

    /// Gets the value that is stored at `key` in `settings`.
    ///
    /// A convenience variant of `gio.Settings.get` for 64-bit unsigned
    /// integers.
    ///
    /// It is a programmer error to give a `key` that isn't specified as
    /// having a uint64 type in the schema for `settings`.
    extern fn g_settings_get_uint64(p_settings: *Settings, p_key: [*:0]const u8) u64;
    pub const getUint64 = g_settings_get_uint64;

    /// Checks the "user value" of a key, if there is one.
    ///
    /// The user value of a key is the last value that was set by the user.
    ///
    /// After calling `gio.Settings.reset` this function should always return
    /// `NULL` (assuming something is not wrong with the system
    /// configuration).
    ///
    /// It is possible that `gio.Settings.getValue` will return a different
    /// value than this function.  This can happen in the case that the user
    /// set a value for a key that was subsequently locked down by the system
    /// administrator -- this function will return the user's old value.
    ///
    /// This function may be useful for adding a "reset" option to a UI or
    /// for providing indication that a particular value has been changed.
    ///
    /// It is a programmer error to give a `key` that isn't contained in the
    /// schema for `settings`.
    extern fn g_settings_get_user_value(p_settings: *Settings, p_key: [*:0]const u8) ?*glib.Variant;
    pub const getUserValue = g_settings_get_user_value;

    /// Gets the value that is stored in `settings` for `key`.
    ///
    /// It is a programmer error to give a `key` that isn't contained in the
    /// schema for `settings`.
    extern fn g_settings_get_value(p_settings: *Settings, p_key: [*:0]const u8) *glib.Variant;
    pub const getValue = g_settings_get_value;

    /// Finds out if a key can be written or not
    extern fn g_settings_is_writable(p_settings: *Settings, p_name: [*:0]const u8) c_int;
    pub const isWritable = g_settings_is_writable;

    /// Gets the list of children on `settings`.
    ///
    /// The list is exactly the list of strings for which it is not an error
    /// to call `gio.Settings.getChild`.
    ///
    /// There is little reason to call this function from "normal" code, since
    /// you should already know what children are in your schema. This function
    /// may still be useful there for introspection reasons, however.
    ///
    /// You should free the return value with `glib.strfreev` when you are done
    /// with it.
    extern fn g_settings_list_children(p_settings: *Settings) [*][*:0]u8;
    pub const listChildren = g_settings_list_children;

    /// Introspects the list of keys on `settings`.
    ///
    /// You should probably not be calling this function from "normal" code
    /// (since you should already know what keys are in your schema).  This
    /// function is intended for introspection reasons.
    ///
    /// You should free the return value with `glib.strfreev` when you are done
    /// with it.
    extern fn g_settings_list_keys(p_settings: *Settings) [*][*:0]u8;
    pub const listKeys = g_settings_list_keys;

    /// Checks if the given `value` is of the correct type and within the
    /// permitted range for `key`.
    extern fn g_settings_range_check(p_settings: *Settings, p_key: [*:0]const u8, p_value: *glib.Variant) c_int;
    pub const rangeCheck = g_settings_range_check;

    /// Resets `key` to its default value.
    ///
    /// This call resets the key, as much as possible, to its default value.
    /// That might be the value specified in the schema or the one set by the
    /// administrator.
    extern fn g_settings_reset(p_settings: *Settings, p_key: [*:0]const u8) void;
    pub const reset = g_settings_reset;

    /// Reverts all non-applied changes to the settings.  This function
    /// does nothing unless `settings` is in 'delay-apply' mode; see
    /// `gio.Settings.delay`.  In the normal case settings are always applied
    /// immediately.
    ///
    /// Change notifications will be emitted for affected keys.
    extern fn g_settings_revert(p_settings: *Settings) void;
    pub const revert = g_settings_revert;

    /// Sets `key` in `settings` to `value`.
    ///
    /// A convenience function that combines `gio.Settings.setValue` with
    /// `glib.Variant.new`.
    ///
    /// It is a programmer error to give a `key` that isn't contained in the
    /// schema for `settings` or for the `glib.VariantType` of `format` to mismatch
    /// the type given in the schema.
    extern fn g_settings_set(p_settings: *Settings, p_key: [*:0]const u8, p_format: [*:0]const u8, ...) c_int;
    pub const set = g_settings_set;

    /// Sets `key` in `settings` to `value`.
    ///
    /// A convenience variant of `gio.Settings.set` for booleans.
    ///
    /// It is a programmer error to give a `key` that isn't specified as
    /// having a boolean type in the schema for `settings`.
    extern fn g_settings_set_boolean(p_settings: *Settings, p_key: [*:0]const u8, p_value: c_int) c_int;
    pub const setBoolean = g_settings_set_boolean;

    /// Sets `key` in `settings` to `value`.
    ///
    /// A convenience variant of `gio.Settings.set` for doubles.
    ///
    /// It is a programmer error to give a `key` that isn't specified as
    /// having a 'double' type in the schema for `settings`.
    extern fn g_settings_set_double(p_settings: *Settings, p_key: [*:0]const u8, p_value: f64) c_int;
    pub const setDouble = g_settings_set_double;

    /// Looks up the enumerated type nick for `value` and writes it to `key`,
    /// within `settings`.
    ///
    /// It is a programmer error to give a `key` that isn't contained in the
    /// schema for `settings` or is not marked as an enumerated type, or for
    /// `value` not to be a valid value for the named type.
    ///
    /// After performing the write, accessing `key` directly with
    /// `gio.Settings.getString` will return the 'nick' associated with
    /// `value`.
    extern fn g_settings_set_enum(p_settings: *Settings, p_key: [*:0]const u8, p_value: c_int) c_int;
    pub const setEnum = g_settings_set_enum;

    /// Looks up the flags type nicks for the bits specified by `value`, puts
    /// them in an array of strings and writes the array to `key`, within
    /// `settings`.
    ///
    /// It is a programmer error to give a `key` that isn't contained in the
    /// schema for `settings` or is not marked as a flags type, or for `value`
    /// to contain any bits that are not value for the named type.
    ///
    /// After performing the write, accessing `key` directly with
    /// `gio.Settings.getStrv` will return an array of 'nicks'; one for each
    /// bit in `value`.
    extern fn g_settings_set_flags(p_settings: *Settings, p_key: [*:0]const u8, p_value: c_uint) c_int;
    pub const setFlags = g_settings_set_flags;

    /// Sets `key` in `settings` to `value`.
    ///
    /// A convenience variant of `gio.Settings.set` for 32-bit integers.
    ///
    /// It is a programmer error to give a `key` that isn't specified as
    /// having a int32 type in the schema for `settings`.
    extern fn g_settings_set_int(p_settings: *Settings, p_key: [*:0]const u8, p_value: c_int) c_int;
    pub const setInt = g_settings_set_int;

    /// Sets `key` in `settings` to `value`.
    ///
    /// A convenience variant of `gio.Settings.set` for 64-bit integers.
    ///
    /// It is a programmer error to give a `key` that isn't specified as
    /// having a int64 type in the schema for `settings`.
    extern fn g_settings_set_int64(p_settings: *Settings, p_key: [*:0]const u8, p_value: i64) c_int;
    pub const setInt64 = g_settings_set_int64;

    /// Sets `key` in `settings` to `value`.
    ///
    /// A convenience variant of `gio.Settings.set` for strings.
    ///
    /// It is a programmer error to give a `key` that isn't specified as
    /// having a string type in the schema for `settings`.
    extern fn g_settings_set_string(p_settings: *Settings, p_key: [*:0]const u8, p_value: [*:0]const u8) c_int;
    pub const setString = g_settings_set_string;

    /// Sets `key` in `settings` to `value`.
    ///
    /// A convenience variant of `gio.Settings.set` for string arrays.  If
    /// `value` is `NULL`, then `key` is set to be the empty array.
    ///
    /// It is a programmer error to give a `key` that isn't specified as
    /// having an array of strings type in the schema for `settings`.
    extern fn g_settings_set_strv(p_settings: *Settings, p_key: [*:0]const u8, p_value: ?[*]const [*:0]const u8) c_int;
    pub const setStrv = g_settings_set_strv;

    /// Sets `key` in `settings` to `value`.
    ///
    /// A convenience variant of `gio.Settings.set` for 32-bit unsigned
    /// integers.
    ///
    /// It is a programmer error to give a `key` that isn't specified as
    /// having a uint32 type in the schema for `settings`.
    extern fn g_settings_set_uint(p_settings: *Settings, p_key: [*:0]const u8, p_value: c_uint) c_int;
    pub const setUint = g_settings_set_uint;

    /// Sets `key` in `settings` to `value`.
    ///
    /// A convenience variant of `gio.Settings.set` for 64-bit unsigned
    /// integers.
    ///
    /// It is a programmer error to give a `key` that isn't specified as
    /// having a uint64 type in the schema for `settings`.
    extern fn g_settings_set_uint64(p_settings: *Settings, p_key: [*:0]const u8, p_value: u64) c_int;
    pub const setUint64 = g_settings_set_uint64;

    /// Sets `key` in `settings` to `value`.
    ///
    /// It is a programmer error to give a `key` that isn't contained in the
    /// schema for `settings` or for `value` to have the incorrect type, per
    /// the schema.
    ///
    /// If `value` is floating then this function consumes the reference.
    extern fn g_settings_set_value(p_settings: *Settings, p_key: [*:0]const u8, p_value: *glib.Variant) c_int;
    pub const setValue = g_settings_set_value;

    extern fn g_settings_get_type() usize;
    pub const getGObjectType = g_settings_get_type;

    extern fn g_object_ref(p_self: *gio.Settings) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.Settings) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Settings, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The `GSettingsBackend` interface defines a generic interface for
/// non-strictly-typed data that is stored in a hierarchy. To implement
/// an alternative storage backend for `gio.Settings`, you need to
/// implement the `GSettingsBackend` interface and then make it implement the
/// extension point `G_SETTINGS_BACKEND_EXTENSION_POINT_NAME`.
///
/// The interface defines methods for reading and writing values, a
/// method for determining if writing of certain values will fail
/// (lockdown) and a change notification mechanism.
///
/// The semantics of the interface are very precisely defined and
/// implementations must carefully adhere to the expectations of
/// callers that are documented on each of the interface methods.
///
/// Some of the `GSettingsBackend` functions accept or return a
/// `glib.Tree`. These trees always have strings as keys and
/// `glib.Variant` as values.
///
/// The `GSettingsBackend` API is exported to allow third-party
/// implementations, but does not carry the same stability guarantees
/// as the public GIO API. For this reason, you have to define the
/// C preprocessor symbol `G_SETTINGS_ENABLE_BACKEND` before including
/// `gio/gsettingsbackend.h`.
pub const SettingsBackend = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.SettingsBackendClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.SettingsBackendPrivate,

    pub const virtual_methods = struct {
        /// virtual method to get permission of a key
        pub const get_permission = struct {
            pub fn call(p_class: anytype, p_backend: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_path: [*:0]const u8) *gio.Permission {
                return gobject.ext.as(SettingsBackend.Class, p_class).f_get_permission.?(gobject.ext.as(SettingsBackend, p_backend), p_path);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_backend: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_path: [*:0]const u8) callconv(.C) *gio.Permission) void {
                gobject.ext.as(SettingsBackend.Class, p_class).f_get_permission = @ptrCast(p_implementation);
            }
        };

        /// virtual method to get if a key is writable
        pub const get_writable = struct {
            pub fn call(p_class: anytype, p_backend: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_key: [*:0]const u8) c_int {
                return gobject.ext.as(SettingsBackend.Class, p_class).f_get_writable.?(gobject.ext.as(SettingsBackend, p_backend), p_key);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_backend: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_key: [*:0]const u8) callconv(.C) c_int) void {
                gobject.ext.as(SettingsBackend.Class, p_class).f_get_writable = @ptrCast(p_implementation);
            }
        };

        /// virtual method to read a key's value
        pub const read = struct {
            pub fn call(p_class: anytype, p_backend: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_key: [*:0]const u8, p_expected_type: *const glib.VariantType, p_default_value: c_int) *glib.Variant {
                return gobject.ext.as(SettingsBackend.Class, p_class).f_read.?(gobject.ext.as(SettingsBackend, p_backend), p_key, p_expected_type, p_default_value);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_backend: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_key: [*:0]const u8, p_expected_type: *const glib.VariantType, p_default_value: c_int) callconv(.C) *glib.Variant) void {
                gobject.ext.as(SettingsBackend.Class, p_class).f_read = @ptrCast(p_implementation);
            }
        };

        /// virtual method to read user's key value
        pub const read_user_value = struct {
            pub fn call(p_class: anytype, p_backend: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_key: [*:0]const u8, p_expected_type: *const glib.VariantType) *glib.Variant {
                return gobject.ext.as(SettingsBackend.Class, p_class).f_read_user_value.?(gobject.ext.as(SettingsBackend, p_backend), p_key, p_expected_type);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_backend: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_key: [*:0]const u8, p_expected_type: *const glib.VariantType) callconv(.C) *glib.Variant) void {
                gobject.ext.as(SettingsBackend.Class, p_class).f_read_user_value = @ptrCast(p_implementation);
            }
        };

        /// virtual method to reset state
        pub const reset = struct {
            pub fn call(p_class: anytype, p_backend: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_key: [*:0]const u8, p_origin_tag: ?*anyopaque) void {
                return gobject.ext.as(SettingsBackend.Class, p_class).f_reset.?(gobject.ext.as(SettingsBackend, p_backend), p_key, p_origin_tag);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_backend: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_key: [*:0]const u8, p_origin_tag: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(SettingsBackend.Class, p_class).f_reset = @ptrCast(p_implementation);
            }
        };

        /// virtual method to subscribe to key changes
        pub const subscribe = struct {
            pub fn call(p_class: anytype, p_backend: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_name: [*:0]const u8) void {
                return gobject.ext.as(SettingsBackend.Class, p_class).f_subscribe.?(gobject.ext.as(SettingsBackend, p_backend), p_name);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_backend: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_name: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(SettingsBackend.Class, p_class).f_subscribe = @ptrCast(p_implementation);
            }
        };

        /// virtual method to sync state
        pub const sync = struct {
            pub fn call(p_class: anytype, p_backend: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(SettingsBackend.Class, p_class).f_sync.?(gobject.ext.as(SettingsBackend, p_backend));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_backend: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(SettingsBackend.Class, p_class).f_sync = @ptrCast(p_implementation);
            }
        };

        /// virtual method to unsubscribe to key changes
        pub const unsubscribe = struct {
            pub fn call(p_class: anytype, p_backend: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_name: [*:0]const u8) void {
                return gobject.ext.as(SettingsBackend.Class, p_class).f_unsubscribe.?(gobject.ext.as(SettingsBackend, p_backend), p_name);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_backend: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_name: [*:0]const u8) callconv(.C) void) void {
                gobject.ext.as(SettingsBackend.Class, p_class).f_unsubscribe = @ptrCast(p_implementation);
            }
        };

        /// virtual method to change key's value
        pub const write = struct {
            pub fn call(p_class: anytype, p_backend: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_key: [*:0]const u8, p_value: *glib.Variant, p_origin_tag: ?*anyopaque) c_int {
                return gobject.ext.as(SettingsBackend.Class, p_class).f_write.?(gobject.ext.as(SettingsBackend, p_backend), p_key, p_value, p_origin_tag);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_backend: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_key: [*:0]const u8, p_value: *glib.Variant, p_origin_tag: ?*anyopaque) callconv(.C) c_int) void {
                gobject.ext.as(SettingsBackend.Class, p_class).f_write = @ptrCast(p_implementation);
            }
        };

        /// virtual method to change a tree of keys
        pub const write_tree = struct {
            pub fn call(p_class: anytype, p_backend: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_tree: *glib.Tree, p_origin_tag: ?*anyopaque) c_int {
                return gobject.ext.as(SettingsBackend.Class, p_class).f_write_tree.?(gobject.ext.as(SettingsBackend, p_backend), p_tree, p_origin_tag);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_backend: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_tree: *glib.Tree, p_origin_tag: ?*anyopaque) callconv(.C) c_int) void {
                gobject.ext.as(SettingsBackend.Class, p_class).f_write_tree = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {};

    /// Calculate the longest common prefix of all keys in a tree and write
    /// out an array of the key names relative to that prefix and,
    /// optionally, the value to store at each of those keys.
    ///
    /// You must free the value returned in `path`, `keys` and `values` using
    /// `glib.free`.  You should not attempt to free or unref the contents of
    /// `keys` or `values`.
    extern fn g_settings_backend_flatten_tree(p_tree: *glib.Tree, p_path: *[*:0]u8, p_keys: [*]*[*:0]const u8, p_values: ?[*]**glib.Variant) void;
    pub const flattenTree = g_settings_backend_flatten_tree;

    /// Returns the default `gio.SettingsBackend`. It is possible to override
    /// the default by setting the `GSETTINGS_BACKEND` environment variable
    /// to the name of a settings backend.
    ///
    /// The user gets a reference to the backend.
    extern fn g_settings_backend_get_default() *gio.SettingsBackend;
    pub const getDefault = g_settings_backend_get_default;

    /// Signals that a single key has possibly changed.  Backend
    /// implementations should call this if a key has possibly changed its
    /// value.
    ///
    /// `key` must be a valid key (ie starting with a slash, not containing
    /// '//', and not ending with a slash).
    ///
    /// The implementation must call this function during any call to
    /// `g_settings_backend_write`, before the call returns (except in the
    /// case that no keys are actually changed and it cares to detect this
    /// fact).  It may not rely on the existence of a mainloop for
    /// dispatching the signal later.
    ///
    /// The implementation may call this function at any other time it likes
    /// in response to other events (such as changes occurring outside of the
    /// program).  These calls may originate from a mainloop or may originate
    /// in response to any other action (including from calls to
    /// `g_settings_backend_write`).
    ///
    /// In the case that this call is in response to a call to
    /// `g_settings_backend_write` then `origin_tag` must be set to the same
    /// value that was passed to that call.
    extern fn g_settings_backend_changed(p_backend: *SettingsBackend, p_key: [*:0]const u8, p_origin_tag: ?*anyopaque) void;
    pub const changed = g_settings_backend_changed;

    /// This call is a convenience wrapper.  It gets the list of changes from
    /// `tree`, computes the longest common prefix and calls
    /// `gio.SettingsBackend.changed`.
    extern fn g_settings_backend_changed_tree(p_backend: *SettingsBackend, p_tree: *glib.Tree, p_origin_tag: ?*anyopaque) void;
    pub const changedTree = g_settings_backend_changed_tree;

    /// Signals that a list of keys have possibly changed.  Backend
    /// implementations should call this if keys have possibly changed their
    /// values.
    ///
    /// `path` must be a valid path (ie starting and ending with a slash and
    /// not containing '//').  Each string in `items` must form a valid key
    /// name when `path` is prefixed to it (ie: each item must not start or
    /// end with '/' and must not contain '//').
    ///
    /// The meaning of this signal is that any of the key names resulting
    /// from the contatenation of `path` with each item in `items` may have
    /// changed.
    ///
    /// The same rules for when notifications must occur apply as per
    /// `gio.SettingsBackend.changed`.  These two calls can be used
    /// interchangeably if exactly one item has changed (although in that
    /// case `gio.SettingsBackend.changed` is definitely preferred).
    ///
    /// For efficiency reasons, the implementation should strive for `path` to
    /// be as long as possible (ie: the longest common prefix of all of the
    /// keys that were changed) but this is not strictly required.
    extern fn g_settings_backend_keys_changed(p_backend: *SettingsBackend, p_path: [*:0]const u8, p_items: [*]const [*:0]const u8, p_origin_tag: ?*anyopaque) void;
    pub const keysChanged = g_settings_backend_keys_changed;

    /// Signals that all keys below a given path may have possibly changed.
    /// Backend implementations should call this if an entire path of keys
    /// have possibly changed their values.
    ///
    /// `path` must be a valid path (ie starting and ending with a slash and
    /// not containing '//').
    ///
    /// The meaning of this signal is that any of the key which has a name
    /// starting with `path` may have changed.
    ///
    /// The same rules for when notifications must occur apply as per
    /// `gio.SettingsBackend.changed`.  This call might be an appropriate
    /// reasponse to a 'reset' call but implementations are also free to
    /// explicitly list the keys that were affected by that call if they can
    /// easily do so.
    ///
    /// For efficiency reasons, the implementation should strive for `path` to
    /// be as long as possible (ie: the longest common prefix of all of the
    /// keys that were changed) but this is not strictly required.  As an
    /// example, if this function is called with the path of "/" then every
    /// single key in the application will be notified of a possible change.
    extern fn g_settings_backend_path_changed(p_backend: *SettingsBackend, p_path: [*:0]const u8, p_origin_tag: ?*anyopaque) void;
    pub const pathChanged = g_settings_backend_path_changed;

    /// Signals that the writability of all keys below a given path may have
    /// changed.
    ///
    /// Since GSettings performs no locking operations for itself, this call
    /// will always be made in response to external events.
    extern fn g_settings_backend_path_writable_changed(p_backend: *SettingsBackend, p_path: [*:0]const u8) void;
    pub const pathWritableChanged = g_settings_backend_path_writable_changed;

    /// Signals that the writability of a single key has possibly changed.
    ///
    /// Since GSettings performs no locking operations for itself, this call
    /// will always be made in response to external events.
    extern fn g_settings_backend_writable_changed(p_backend: *SettingsBackend, p_key: [*:0]const u8) void;
    pub const writableChanged = g_settings_backend_writable_changed;

    extern fn g_settings_backend_get_type() usize;
    pub const getGObjectType = g_settings_backend_get_type;

    extern fn g_object_ref(p_self: *gio.SettingsBackend) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.SettingsBackend) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SettingsBackend, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `GSimpleAction` is the obvious simple implementation of the
/// `gio.Action` interface. This is the easiest way to create an action for
/// purposes of adding it to a `gio.SimpleActionGroup`.
pub const SimpleAction = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gio.Action};
    pub const Class = opaque {
        pub const Instance = SimpleAction;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// If `action` is currently enabled.
        ///
        /// If the action is disabled then calls to `gio.Action.activate` and
        /// `gio.Action.changeState` have no effect.
        pub const enabled = struct {
            pub const name = "enabled";

            pub const Type = c_int;
        };

        /// The name of the action. This is mostly meaningful for identifying
        /// the action once it has been added to a `gio.SimpleActionGroup`.
        pub const name = struct {
            pub const name = "name";

            pub const Type = ?[*:0]u8;
        };

        /// The type of the parameter that must be given when activating the
        /// action.
        pub const parameter_type = struct {
            pub const name = "parameter-type";

            pub const Type = ?*glib.VariantType;
        };

        /// The state of the action, or `NULL` if the action is stateless.
        pub const state = struct {
            pub const name = "state";

            pub const Type = ?*glib.Variant;
        };

        /// The `glib.VariantType` of the state that the action has, or `NULL` if the
        /// action is stateless.
        pub const state_type = struct {
            pub const name = "state-type";

            pub const Type = ?*glib.VariantType;
        };
    };

    pub const signals = struct {
        /// Indicates that the action was just activated.
        ///
        /// `parameter` will always be of the expected type, i.e. the parameter type
        /// specified when the action was created. If an incorrect type is given when
        /// activating the action, this signal is not emitted.
        ///
        /// Since GLib 2.40, if no handler is connected to this signal then the
        /// default behaviour for boolean-stated actions with a `NULL` parameter
        /// type is to toggle them via the `gio.SimpleAction.signals.change`-state signal.
        /// For stateful actions where the state type is equal to the parameter
        /// type, the default is to forward them directly to
        /// `gio.SimpleAction.signals.change`-state.  This should allow almost all users
        /// of `gio.SimpleAction` to connect only one handler or the other.
        pub const activate = struct {
            pub const name = "activate";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_parameter: ?*glib.Variant, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(SimpleAction, p_instance))),
                    gobject.signalLookup("activate", SimpleAction.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// Indicates that the action just received a request to change its
        /// state.
        ///
        /// `value` will always be of the correct state type, i.e. the type of the
        /// initial state passed to `gio.SimpleAction.newStateful`. If an incorrect
        /// type is given when requesting to change the state, this signal is not
        /// emitted.
        ///
        /// If no handler is connected to this signal then the default
        /// behaviour is to call `gio.SimpleAction.setState` to set the state
        /// to the requested value. If you connect a signal handler then no
        /// default action is taken. If the state should change then you must
        /// call `gio.SimpleAction.setState` from the handler.
        ///
        /// An example of a 'change-state' handler:
        /// ```
        /// static void
        /// change_volume_state (GSimpleAction *action,
        ///                      GVariant      *value,
        ///                      gpointer       user_data)
        /// {
        ///   gint requested;
        ///
        ///   requested = g_variant_get_int32 (value);
        ///
        ///   // Volume only goes from 0 to 10
        ///   if (0 <= requested && requested <= 10)
        ///     g_simple_action_set_state (action, value);
        /// }
        /// ```
        ///
        /// The handler need not set the state to the requested value.
        /// It could set it to any value at all, or take some other action.
        pub const change_state = struct {
            pub const name = "change-state";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_value: ?*glib.Variant, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(SimpleAction, p_instance))),
                    gobject.signalLookup("change-state", SimpleAction.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new action.
    ///
    /// The created action is stateless. See `gio.SimpleAction.newStateful` to create
    /// an action that has state.
    extern fn g_simple_action_new(p_name: [*:0]const u8, p_parameter_type: ?*const glib.VariantType) *gio.SimpleAction;
    pub const new = g_simple_action_new;

    /// Creates a new stateful action.
    ///
    /// All future state values must have the same `glib.VariantType` as the initial
    /// `state`.
    ///
    /// If the `state` `glib.Variant` is floating, it is consumed.
    extern fn g_simple_action_new_stateful(p_name: [*:0]const u8, p_parameter_type: ?*const glib.VariantType, p_state: *glib.Variant) *gio.SimpleAction;
    pub const newStateful = g_simple_action_new_stateful;

    /// Sets the action as enabled or not.
    ///
    /// An action must be enabled in order to be activated or in order to
    /// have its state changed from outside callers.
    ///
    /// This should only be called by the implementor of the action.  Users
    /// of the action should not attempt to modify its enabled flag.
    extern fn g_simple_action_set_enabled(p_simple: *SimpleAction, p_enabled: c_int) void;
    pub const setEnabled = g_simple_action_set_enabled;

    /// Sets the state of the action.
    ///
    /// This directly updates the 'state' property to the given value.
    ///
    /// This should only be called by the implementor of the action.  Users
    /// of the action should not attempt to directly modify the 'state'
    /// property.  Instead, they should call `gio.Action.changeState` to
    /// request the change.
    ///
    /// If the `value` GVariant is floating, it is consumed.
    extern fn g_simple_action_set_state(p_simple: *SimpleAction, p_value: *glib.Variant) void;
    pub const setState = g_simple_action_set_state;

    /// Sets the state hint for the action.
    ///
    /// See `gio.Action.getStateHint` for more information about
    /// action state hints.
    extern fn g_simple_action_set_state_hint(p_simple: *SimpleAction, p_state_hint: ?*glib.Variant) void;
    pub const setStateHint = g_simple_action_set_state_hint;

    extern fn g_simple_action_get_type() usize;
    pub const getGObjectType = g_simple_action_get_type;

    extern fn g_object_ref(p_self: *gio.SimpleAction) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.SimpleAction) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SimpleAction, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GSimpleActionGroup` is a hash table filled with `gio.Action` objects,
/// implementing the `gio.ActionGroup` and `gio.ActionMap`
/// interfaces.
pub const SimpleActionGroup = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{ gio.ActionGroup, gio.ActionMap };
    pub const Class = gio.SimpleActionGroupClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.SimpleActionGroupPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new, empty, `gio.SimpleActionGroup`.
    extern fn g_simple_action_group_new() *gio.SimpleActionGroup;
    pub const new = g_simple_action_group_new;

    /// A convenience function for creating multiple `gio.SimpleAction` instances
    /// and adding them to the action group.
    extern fn g_simple_action_group_add_entries(p_simple: *SimpleActionGroup, p_entries: [*]const gio.ActionEntry, p_n_entries: c_int, p_user_data: ?*anyopaque) void;
    pub const addEntries = g_simple_action_group_add_entries;

    /// Adds an action to the action group.
    ///
    /// If the action group already contains an action with the same name as
    /// `action` then the old action is dropped from the group.
    ///
    /// The action group takes its own reference on `action`.
    extern fn g_simple_action_group_insert(p_simple: *SimpleActionGroup, p_action: *gio.Action) void;
    pub const insert = g_simple_action_group_insert;

    /// Looks up the action with the name `action_name` in the group.
    ///
    /// If no such action exists, returns `NULL`.
    extern fn g_simple_action_group_lookup(p_simple: *SimpleActionGroup, p_action_name: [*:0]const u8) *gio.Action;
    pub const lookup = g_simple_action_group_lookup;

    /// Removes the named action from the action group.
    ///
    /// If no action of this name is in the group then nothing happens.
    extern fn g_simple_action_group_remove(p_simple: *SimpleActionGroup, p_action_name: [*:0]const u8) void;
    pub const remove = g_simple_action_group_remove;

    extern fn g_simple_action_group_get_type() usize;
    pub const getGObjectType = g_simple_action_group_get_type;

    extern fn g_object_ref(p_self: *gio.SimpleActionGroup) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.SimpleActionGroup) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SimpleActionGroup, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// As of GLib 2.46, `GSimpleAsyncResult` is deprecated in favor of
/// `gio.Task`, which provides a simpler API.
///
/// `GSimpleAsyncResult` implements `gio.AsyncResult`.
///
/// `GSimpleAsyncResult` handles `gio.AsyncReadyCallback`s, error
/// reporting, operation cancellation and the final state of an operation,
/// completely transparent to the application. Results can be returned
/// as a pointer e.g. for functions that return data that is collected
/// asynchronously, a boolean value for checking the success or failure
/// of an operation, or a `gssize` for operations which return the number
/// of bytes modified by the operation; all of the simple return cases
/// are covered.
///
/// Most of the time, an application will not need to know of the details
/// of this API; it is handled transparently, and any necessary operations
/// are handled by `gio.AsyncResult`’s interface. However, if implementing
/// a new GIO module, for writing language bindings, or for complex
/// applications that need better control of how asynchronous operations
/// are completed, it is important to understand this functionality.
///
/// `GSimpleAsyncResult`s are tagged with the calling function to ensure
/// that asynchronous functions and their finishing functions are used
/// together correctly.
///
/// To create a new `GSimpleAsyncResult`, call `gio.SimpleAsyncResult.new`.
/// If the result needs to be created for a `GError`, use
/// `gio.SimpleAsyncResult.newFromError` or
/// `gio.SimpleAsyncResult.newTakeError`. If a `GError` is not available
/// (e.g. the asynchronous operation doesn’t take a `GError` argument),
/// but the result still needs to be created for an error condition, use
/// `gio.SimpleAsyncResult.newError` (or
/// `gio.SimpleAsyncResult.setErrorVa` if your application or binding
/// requires passing a variable argument list directly), and the error can then
/// be propagated through the use of
/// `gio.SimpleAsyncResult.propagateError`.
///
/// An asynchronous operation can be made to ignore a cancellation event by
/// calling `gio.SimpleAsyncResult.setHandleCancellation` with a
/// `GSimpleAsyncResult` for the operation and `FALSE`. This is useful for
/// operations that are dangerous to cancel, such as close (which would
/// cause a leak if cancelled before being run).
///
/// `GSimpleAsyncResult` can integrate into GLib’s event loop,
/// `glib.MainLoop`, or it can use `glib.Thread`s.
/// `gio.SimpleAsyncResult.complete` will finish an I/O task directly
/// from the point where it is called.
/// `gio.SimpleAsyncResult.completeInIdle` will finish it from an idle
/// handler in the  thread-default main context (see
/// `glib.MainContext.pushThreadDefault`) where the `GSimpleAsyncResult`
/// was created. `gio.SimpleAsyncResult.runInThread` will run the job in
/// a separate thread and then use
/// `gio.SimpleAsyncResult.completeInIdle` to deliver the result.
///
/// To set the results of an asynchronous function,
/// `gio.SimpleAsyncResult.setOpResGpointer`,
/// `gio.SimpleAsyncResult.setOpResGboolean`, and
/// `gio.SimpleAsyncResult.setOpResGssize`
/// are provided, setting the operation's result to a `gpointer`, `gboolean`, or
/// `gssize`, respectively.
///
/// Likewise, to get the result of an asynchronous function,
/// `gio.SimpleAsyncResult.getOpResGpointer`,
/// `gio.SimpleAsyncResult.getOpResGboolean`, and
/// `gio.SimpleAsyncResult.getOpResGssize` are
/// provided, getting the operation’s result as a `gpointer`, `gboolean`, and
/// `gssize`, respectively.
///
/// For the details of the requirements implementations must respect, see
/// `gio.AsyncResult`.  A typical implementation of an asynchronous
/// operation using `GSimpleAsyncResult` looks something like this:
///
/// ```c
/// static void
/// baked_cb (Cake    *cake,
///           gpointer user_data)
/// {
///   // In this example, this callback is not given a reference to the cake,
///   // so the GSimpleAsyncResult has to take a reference to it.
///   GSimpleAsyncResult *result = user_data;
///
///   if (cake == NULL)
///     g_simple_async_result_set_error (result,
///                                      BAKER_ERRORS,
///                                      BAKER_ERROR_NO_FLOUR,
///                                      "Go to the supermarket");
///   else
///     g_simple_async_result_set_op_res_gpointer (result,
///                                                g_object_ref (cake),
///                                                g_object_unref);
///
///
///   // In this example, we assume that baked_cb is called as a callback from
///   // the mainloop, so it's safe to complete the operation synchronously here.
///   // If, however, _baker_prepare_cake () might call its callback without
///   // first returning to the mainloop — inadvisable, but some APIs do so —
///   // we would need to use `gio.SimpleAsyncResult.completeInIdle`.
///   g_simple_async_result_complete (result);
///   g_object_unref (result);
/// }
///
/// void
/// baker_bake_cake_async (Baker              *self,
///                        guint               radius,
///                        GAsyncReadyCallback callback,
///                        gpointer            user_data)
/// {
///   GSimpleAsyncResult *simple;
///   Cake               *cake;
///
///   if (radius < 3)
///     {
///       g_simple_async_report_error_in_idle (G_OBJECT (self),
///                                            callback,
///                                            user_data,
///                                            BAKER_ERRORS,
///                                            BAKER_ERROR_TOO_SMALL,
///                                            "`ucm` radius cakes are silly",
///                                            radius);
///       return;
///     }
///
///   simple = g_simple_async_result_new (G_OBJECT (self),
///                                       callback,
///                                       user_data,
///                                       baker_bake_cake_async);
///   cake = _baker_get_cached_cake (self, radius);
///
///   if (cake != NULL)
///     {
///       g_simple_async_result_set_op_res_gpointer (simple,
///                                                  g_object_ref (cake),
///                                                  g_object_unref);
///       g_simple_async_result_complete_in_idle (simple);
///       g_object_unref (simple);
///       // Drop the reference returned by `_baker_get_cached_cake`;
///       // the GSimpleAsyncResult has taken its own reference.
///       g_object_unref (cake);
///       return;
///     }
///
///   _baker_prepare_cake (self, radius, baked_cb, simple);
/// }
///
/// Cake *
/// baker_bake_cake_finish (Baker        *self,
///                         GAsyncResult *result,
///                         GError      **error)
/// {
///   GSimpleAsyncResult *simple;
///   Cake               *cake;
///
///   g_return_val_if_fail (g_simple_async_result_is_valid (result,
///                                                         G_OBJECT (self),
///                                                         baker_bake_cake_async),
///                         NULL);
///
///   simple = (GSimpleAsyncResult *) result;
///
///   if (g_simple_async_result_propagate_error (simple, error))
///     return NULL;
///
///   cake = CAKE (g_simple_async_result_get_op_res_gpointer (simple));
///   return g_object_ref (cake);
/// }
/// ```
pub const SimpleAsyncResult = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gio.AsyncResult};
    pub const Class = gio.SimpleAsyncResultClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Ensures that the data passed to the _finish function of an async
    /// operation is consistent.  Three checks are performed.
    ///
    /// First, `result` is checked to ensure that it is really a
    /// `gio.SimpleAsyncResult`.  Second, `source` is checked to ensure that it
    /// matches the source object of `result`.  Third, `source_tag` is
    /// checked to ensure that it is equal to the `source_tag` argument given
    /// to `gio.SimpleAsyncResult.new` (which, by convention, is a pointer
    /// to the _async function corresponding to the _finish function from
    /// which this function is called).  (Alternatively, if either
    /// `source_tag` or `result`'s source tag is `NULL`, then the source tag
    /// check is skipped.)
    extern fn g_simple_async_result_is_valid(p_result: *gio.AsyncResult, p_source: ?*gobject.Object, p_source_tag: ?*anyopaque) c_int;
    pub const isValid = g_simple_async_result_is_valid;

    /// Creates a `gio.SimpleAsyncResult`.
    ///
    /// The common convention is to create the `gio.SimpleAsyncResult` in the
    /// function that starts the asynchronous operation and use that same
    /// function as the `source_tag`.
    ///
    /// If your operation supports cancellation with `gio.Cancellable` (which it
    /// probably should) then you should provide the user's cancellable to
    /// `gio.SimpleAsyncResult.setCheckCancellable` immediately after
    /// this function returns.
    extern fn g_simple_async_result_new(p_source_object: ?*gobject.Object, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque, p_source_tag: ?*anyopaque) *gio.SimpleAsyncResult;
    pub const new = g_simple_async_result_new;

    /// Creates a new `gio.SimpleAsyncResult` with a set error.
    extern fn g_simple_async_result_new_error(p_source_object: ?*gobject.Object, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque, p_domain: glib.Quark, p_code: c_int, p_format: [*:0]const u8, ...) *gio.SimpleAsyncResult;
    pub const newError = g_simple_async_result_new_error;

    /// Creates a `gio.SimpleAsyncResult` from an error condition.
    extern fn g_simple_async_result_new_from_error(p_source_object: ?*gobject.Object, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque, p_error: *const glib.Error) *gio.SimpleAsyncResult;
    pub const newFromError = g_simple_async_result_new_from_error;

    /// Creates a `gio.SimpleAsyncResult` from an error condition, and takes over the
    /// caller's ownership of `error`, so the caller does not need to free it anymore.
    extern fn g_simple_async_result_new_take_error(p_source_object: ?*gobject.Object, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque, p_error: *glib.Error) *gio.SimpleAsyncResult;
    pub const newTakeError = g_simple_async_result_new_take_error;

    /// Completes an asynchronous I/O job immediately. Must be called in
    /// the thread where the asynchronous result was to be delivered, as it
    /// invokes the callback directly. If you are in a different thread use
    /// `gio.SimpleAsyncResult.completeInIdle`.
    ///
    /// Calling this function takes a reference to `simple` for as long as
    /// is needed to complete the call.
    extern fn g_simple_async_result_complete(p_simple: *SimpleAsyncResult) void;
    pub const complete = g_simple_async_result_complete;

    /// Completes an asynchronous function in an idle handler in the
    /// [thread-default main context][g-main-context-push-thread-default]
    /// of the thread that `simple` was initially created in
    /// (and re-pushes that context around the invocation of the callback).
    ///
    /// Calling this function takes a reference to `simple` for as long as
    /// is needed to complete the call.
    extern fn g_simple_async_result_complete_in_idle(p_simple: *SimpleAsyncResult) void;
    pub const completeInIdle = g_simple_async_result_complete_in_idle;

    /// Gets the operation result boolean from within the asynchronous result.
    extern fn g_simple_async_result_get_op_res_gboolean(p_simple: *SimpleAsyncResult) c_int;
    pub const getOpResGboolean = g_simple_async_result_get_op_res_gboolean;

    /// Gets a pointer result as returned by the asynchronous function.
    extern fn g_simple_async_result_get_op_res_gpointer(p_simple: *SimpleAsyncResult) ?*anyopaque;
    pub const getOpResGpointer = g_simple_async_result_get_op_res_gpointer;

    /// Gets a gssize from the asynchronous result.
    extern fn g_simple_async_result_get_op_res_gssize(p_simple: *SimpleAsyncResult) isize;
    pub const getOpResGssize = g_simple_async_result_get_op_res_gssize;

    /// Gets the source tag for the `gio.SimpleAsyncResult`.
    extern fn g_simple_async_result_get_source_tag(p_simple: *SimpleAsyncResult) ?*anyopaque;
    pub const getSourceTag = g_simple_async_result_get_source_tag;

    /// Propagates an error from within the simple asynchronous result to
    /// a given destination.
    ///
    /// If the `gio.Cancellable` given to a prior call to
    /// `gio.SimpleAsyncResult.setCheckCancellable` is cancelled then this
    /// function will return `TRUE` with `dest` set appropriately.
    extern fn g_simple_async_result_propagate_error(p_simple: *SimpleAsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const propagateError = g_simple_async_result_propagate_error;

    /// Runs the asynchronous job in a separate thread and then calls
    /// `gio.SimpleAsyncResult.completeInIdle` on `simple` to return
    /// the result to the appropriate main loop.
    ///
    /// Calling this function takes a reference to `simple` for as long as
    /// is needed to run the job and report its completion.
    extern fn g_simple_async_result_run_in_thread(p_simple: *SimpleAsyncResult, p_func: gio.SimpleAsyncThreadFunc, p_io_priority: c_int, p_cancellable: ?*gio.Cancellable) void;
    pub const runInThread = g_simple_async_result_run_in_thread;

    /// Sets a `gio.Cancellable` to check before dispatching results.
    ///
    /// This function has one very specific purpose: the provided cancellable
    /// is checked at the time of `gio.SimpleAsyncResult.propagateError` If
    /// it is cancelled, these functions will return an "Operation was
    /// cancelled" error (`G_IO_ERROR_CANCELLED`).
    ///
    /// Implementors of cancellable asynchronous functions should use this in
    /// order to provide a guarantee to their callers that cancelling an
    /// async operation will reliably result in an error being returned for
    /// that operation (even if a positive result for the operation has
    /// already been sent as an idle to the main context to be dispatched).
    ///
    /// The checking described above is done regardless of any call to the
    /// unrelated `gio.SimpleAsyncResult.setHandleCancellation` function.
    extern fn g_simple_async_result_set_check_cancellable(p_simple: *SimpleAsyncResult, p_check_cancellable: ?*gio.Cancellable) void;
    pub const setCheckCancellable = g_simple_async_result_set_check_cancellable;

    /// Sets an error within the asynchronous result without a `glib.Error`.
    extern fn g_simple_async_result_set_error(p_simple: *SimpleAsyncResult, p_domain: glib.Quark, p_code: c_int, p_format: [*:0]const u8, ...) void;
    pub const setError = g_simple_async_result_set_error;

    /// Sets an error within the asynchronous result without a `glib.Error`.
    /// Unless writing a binding, see `gio.SimpleAsyncResult.setError`.
    extern fn g_simple_async_result_set_error_va(p_simple: *SimpleAsyncResult, p_domain: glib.Quark, p_code: c_int, p_format: [*:0]const u8, p_args: std.builtin.VaList) void;
    pub const setErrorVa = g_simple_async_result_set_error_va;

    /// Sets the result from a `glib.Error`.
    extern fn g_simple_async_result_set_from_error(p_simple: *SimpleAsyncResult, p_error: *const glib.Error) void;
    pub const setFromError = g_simple_async_result_set_from_error;

    /// Sets whether to handle cancellation within the asynchronous operation.
    ///
    /// This function has nothing to do with
    /// `gio.SimpleAsyncResult.setCheckCancellable`.  It only refers to the
    /// `gio.Cancellable` passed to `gio.SimpleAsyncResult.runInThread`.
    extern fn g_simple_async_result_set_handle_cancellation(p_simple: *SimpleAsyncResult, p_handle_cancellation: c_int) void;
    pub const setHandleCancellation = g_simple_async_result_set_handle_cancellation;

    /// Sets the operation result to a boolean within the asynchronous result.
    extern fn g_simple_async_result_set_op_res_gboolean(p_simple: *SimpleAsyncResult, p_op_res: c_int) void;
    pub const setOpResGboolean = g_simple_async_result_set_op_res_gboolean;

    /// Sets the operation result within the asynchronous result to a pointer.
    extern fn g_simple_async_result_set_op_res_gpointer(p_simple: *SimpleAsyncResult, p_op_res: ?*anyopaque, p_destroy_op_res: glib.DestroyNotify) void;
    pub const setOpResGpointer = g_simple_async_result_set_op_res_gpointer;

    /// Sets the operation result within the asynchronous result to
    /// the given `op_res`.
    extern fn g_simple_async_result_set_op_res_gssize(p_simple: *SimpleAsyncResult, p_op_res: isize) void;
    pub const setOpResGssize = g_simple_async_result_set_op_res_gssize;

    /// Sets the result from `error`, and takes over the caller's ownership
    /// of `error`, so the caller does not need to free it any more.
    extern fn g_simple_async_result_take_error(p_simple: *SimpleAsyncResult, p_error: *glib.Error) void;
    pub const takeError = g_simple_async_result_take_error;

    extern fn g_simple_async_result_get_type() usize;
    pub const getGObjectType = g_simple_async_result_get_type;

    extern fn g_object_ref(p_self: *gio.SimpleAsyncResult) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.SimpleAsyncResult) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SimpleAsyncResult, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GSimpleIOStream` creates a `gio.IOStream` from an arbitrary
/// `gio.InputStream` and `gio.OutputStream`. This allows any pair of
/// input and output streams to be used with `gio.IOStream` methods.
///
/// This is useful when you obtained a `gio.InputStream` and a
/// `gio.OutputStream` by other means, for instance creating them with
/// platform specific methods as
/// [``gio.UnixInputStream.new``](../gio-unix/ctor.UnixInputStream.new.html)
/// (from `gio-unix-2.0.pc` / `GioUnix-2.0`), and you want to
/// take advantage of the methods provided by `gio.IOStream`.
pub const SimpleIOStream = opaque {
    pub const Parent = gio.IOStream;
    pub const Implements = [_]type{};
    pub const Class = opaque {
        pub const Instance = SimpleIOStream;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The `gio.InputStream` to read from.
        pub const input_stream = struct {
            pub const name = "input-stream";

            pub const Type = ?*gio.InputStream;
        };

        /// The `gio.OutputStream` to write to.
        pub const output_stream = struct {
            pub const name = "output-stream";

            pub const Type = ?*gio.OutputStream;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gio.SimpleIOStream` wrapping `input_stream` and `output_stream`.
    /// See also `gio.IOStream`.
    extern fn g_simple_io_stream_new(p_input_stream: *gio.InputStream, p_output_stream: *gio.OutputStream) *gio.SimpleIOStream;
    pub const new = g_simple_io_stream_new;

    extern fn g_simple_io_stream_get_type() usize;
    pub const getGObjectType = g_simple_io_stream_get_type;

    extern fn g_object_ref(p_self: *gio.SimpleIOStream) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.SimpleIOStream) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SimpleIOStream, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GSimplePermission` is a trivial implementation of `gio.Permission`
/// that represents a permission that is either always or never allowed.  The
/// value is given at construction and doesn’t change.
///
/// Calling `gio.Permission.acquire` or `gio.Permission.release`
/// on a `GSimplePermission` will result in errors.
pub const SimplePermission = opaque {
    pub const Parent = gio.Permission;
    pub const Implements = [_]type{};
    pub const Class = opaque {
        pub const Instance = SimplePermission;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new `gio.Permission` instance that represents an action that is
    /// either always or never allowed.
    extern fn g_simple_permission_new(p_allowed: c_int) *gio.SimplePermission;
    pub const new = g_simple_permission_new;

    extern fn g_simple_permission_get_type() usize;
    pub const getGObjectType = g_simple_permission_get_type;

    extern fn g_object_ref(p_self: *gio.SimplePermission) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.SimplePermission) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SimplePermission, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GSimpleProxyResolver` is a simple `gio.ProxyResolver` implementation
/// that handles a single default proxy, multiple URI-scheme-specific
/// proxies, and a list of hosts that proxies should not be used for.
///
/// `GSimpleProxyResolver` is never the default proxy resolver, but it
/// can be used as the base class for another proxy resolver
/// implementation, or it can be created and used manually, such as
/// with `gio.SocketClient.setProxyResolver`.
pub const SimpleProxyResolver = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gio.ProxyResolver};
    pub const Class = gio.SimpleProxyResolverClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.SimpleProxyResolverPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The default proxy URI that will be used for any URI that doesn't
        /// match `gio.SimpleProxyResolver.properties.ignore`-hosts, and doesn't match any
        /// of the schemes set with `gio.SimpleProxyResolver.setUriProxy`.
        ///
        /// Note that as a special case, if this URI starts with
        /// "socks://", `gio.SimpleProxyResolver` will treat it as referring
        /// to all three of the socks5, socks4a, and socks4 proxy types.
        pub const default_proxy = struct {
            pub const name = "default-proxy";

            pub const Type = ?[*:0]u8;
        };

        /// A list of hostnames and IP addresses that the resolver should
        /// allow direct connections to.
        ///
        /// Entries can be in one of 4 formats:
        ///
        /// - A hostname, such as "example.com", ".example.com", or
        ///   "*.example.com", any of which match "example.com" or
        ///   any subdomain of it.
        ///
        /// - An IPv4 or IPv6 address, such as "192.168.1.1",
        ///   which matches only that address.
        ///
        /// - A hostname or IP address followed by a port, such as
        ///   "example.com:80", which matches whatever the hostname or IP
        ///   address would match, but only for URLs with the (explicitly)
        ///   indicated port. In the case of an IPv6 address, the address
        ///   part must appear in brackets: "[::1]:443"
        ///
        /// - An IP address range, given by a base address and prefix length,
        ///   such as "fe80::/10", which matches any address in that range.
        ///
        /// Note that when dealing with Unicode hostnames, the matching is
        /// done against the ASCII form of the name.
        ///
        /// Also note that hostname exclusions apply only to connections made
        /// to hosts identified by name, and IP address exclusions apply only
        /// to connections made to hosts identified by address. That is, if
        /// example.com has an address of 192.168.1.1, and the :ignore-hosts list
        /// contains only "192.168.1.1", then a connection to "example.com"
        /// (eg, via a `gio.NetworkAddress`) will use the proxy, and a connection to
        /// "192.168.1.1" (eg, via a `gio.InetSocketAddress`) will not.
        ///
        /// These rules match the "ignore-hosts"/"noproxy" rules most
        /// commonly used by other applications.
        pub const ignore_hosts = struct {
            pub const name = "ignore-hosts";

            pub const Type = ?[*][*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gio.SimpleProxyResolver`. See
    /// `gio.SimpleProxyResolver.properties.default`-proxy and
    /// `gio.SimpleProxyResolver.properties.ignore`-hosts for more details on how the
    /// arguments are interpreted.
    extern fn g_simple_proxy_resolver_new(p_default_proxy: ?[*:0]const u8, p_ignore_hosts: ?[*][*:0]u8) *gio.ProxyResolver;
    pub const new = g_simple_proxy_resolver_new;

    /// Sets the default proxy on `resolver`, to be used for any URIs that
    /// don't match `gio.SimpleProxyResolver.properties.ignore`-hosts or a proxy set
    /// via `gio.SimpleProxyResolver.setUriProxy`.
    ///
    /// If `default_proxy` starts with "socks://",
    /// `gio.SimpleProxyResolver` will treat it as referring to all three of
    /// the socks5, socks4a, and socks4 proxy types.
    extern fn g_simple_proxy_resolver_set_default_proxy(p_resolver: *SimpleProxyResolver, p_default_proxy: ?[*:0]const u8) void;
    pub const setDefaultProxy = g_simple_proxy_resolver_set_default_proxy;

    /// Sets the list of ignored hosts.
    ///
    /// See `gio.SimpleProxyResolver.properties.ignore`-hosts for more details on how the
    /// `ignore_hosts` argument is interpreted.
    extern fn g_simple_proxy_resolver_set_ignore_hosts(p_resolver: *SimpleProxyResolver, p_ignore_hosts: [*][*:0]u8) void;
    pub const setIgnoreHosts = g_simple_proxy_resolver_set_ignore_hosts;

    /// Adds a URI-scheme-specific proxy to `resolver`; URIs whose scheme
    /// matches `uri_scheme` (and which don't match
    /// `gio.SimpleProxyResolver.properties.ignore`-hosts) will be proxied via `proxy`.
    ///
    /// As with `gio.SimpleProxyResolver.properties.default`-proxy, if `proxy` starts with
    /// "socks://", `gio.SimpleProxyResolver` will treat it
    /// as referring to all three of the socks5, socks4a, and socks4 proxy
    /// types.
    extern fn g_simple_proxy_resolver_set_uri_proxy(p_resolver: *SimpleProxyResolver, p_uri_scheme: [*:0]const u8, p_proxy: [*:0]const u8) void;
    pub const setUriProxy = g_simple_proxy_resolver_set_uri_proxy;

    extern fn g_simple_proxy_resolver_get_type() usize;
    pub const getGObjectType = g_simple_proxy_resolver_get_type;

    extern fn g_object_ref(p_self: *gio.SimpleProxyResolver) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.SimpleProxyResolver) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SimpleProxyResolver, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `GSocket` is a low-level networking primitive. It is a more or less
/// direct mapping of the BSD socket API in a portable GObject based API.
/// It supports both the UNIX socket implementations and winsock2 on Windows.
///
/// `GSocket` is the platform independent base upon which the higher level
/// network primitives are based. Applications are not typically meant to
/// use it directly, but rather through classes like `gio.SocketClient`,
/// `gio.SocketService` and `gio.SocketConnection`. However there may
/// be cases where direct use of `GSocket` is useful.
///
/// `GSocket` implements the `gio.Initable` interface, so if it is manually
/// constructed by e.g. `gobject.Object.new` you must call
/// `gio.Initable.init` and check the results before using the object.
/// This is done automatically in `gio.Socket.new` and
/// `gio.Socket.newFromFd`, so these functions can return `NULL`.
///
/// Sockets operate in two general modes, blocking or non-blocking. When
/// in blocking mode all operations (which don’t take an explicit blocking
/// parameter) block until the requested operation
/// is finished or there is an error. In non-blocking mode all calls that
/// would block return immediately with a `G_IO_ERROR_WOULD_BLOCK` error.
/// To know when a call would successfully run you can call
/// `gio.Socket.conditionCheck`, or `gio.Socket.conditionWait`.
/// You can also use `gio.Socket.createSource` and attach it to a
/// `glib.MainContext` to get callbacks when I/O is possible.
/// Note that all sockets are always set to non blocking mode in the system, and
/// blocking mode is emulated in `GSocket`.
///
/// When working in non-blocking mode applications should always be able to
/// handle getting a `G_IO_ERROR_WOULD_BLOCK` error even when some other
/// function said that I/O was possible. This can easily happen in case
/// of a race condition in the application, but it can also happen for other
/// reasons. For instance, on Windows a socket is always seen as writable
/// until a write returns `G_IO_ERROR_WOULD_BLOCK`.
///
/// `GSocket`s can be either connection oriented or datagram based.
/// For connection oriented types you must first establish a connection by
/// either connecting to an address or accepting a connection from another
/// address. For connectionless socket types the target/source address is
/// specified or received in each I/O operation.
///
/// All socket file descriptors are set to be close-on-exec.
///
/// Note that creating a `GSocket` causes the signal `SIGPIPE` to be
/// ignored for the remainder of the program. If you are writing a
/// command-line utility that uses `GSocket`, you may need to take into
/// account the fact that your program will not automatically be killed
/// if it tries to write to `stdout` after it has been closed.
///
/// Like most other APIs in GLib, `GSocket` is not inherently thread safe. To use
/// a `GSocket` concurrently from multiple threads, you must implement your own
/// locking.
///
/// ## Nagle’s algorithm
///
/// Since GLib 2.80, `GSocket` will automatically set the `TCP_NODELAY` option on
/// all `G_SOCKET_TYPE_STREAM` sockets. This disables
/// [Nagle’s algorithm](https://en.wikipedia.org/wiki/Nagle`@"27s_algorithm"`) as it
/// typically does more harm than good on modern networks.
///
/// If your application needs Nagle’s algorithm enabled, call
/// `gio.Socket.setOption` after constructing a `GSocket` to enable it:
/// ```c
/// socket = g_socket_new (…, G_SOCKET_TYPE_STREAM, …);
/// if (socket != NULL)
///   {
///     g_socket_set_option (socket, IPPROTO_TCP, TCP_NODELAY, FALSE, &local_error);
///     // handle error if needed
///   }
/// ```
pub const Socket = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{ gio.DatagramBased, gio.Initable };
    pub const Class = gio.SocketClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.SocketPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Whether I/O on this socket is blocking.
        pub const blocking = struct {
            pub const name = "blocking";

            pub const Type = c_int;
        };

        /// Whether the socket should allow sending to broadcast addresses.
        pub const broadcast = struct {
            pub const name = "broadcast";

            pub const Type = c_int;
        };

        /// The socket’s address family.
        pub const family = struct {
            pub const name = "family";

            pub const Type = gio.SocketFamily;
        };

        /// The socket’s file descriptor.
        pub const fd = struct {
            pub const name = "fd";

            pub const Type = c_int;
        };

        /// Whether to keep the connection alive by sending periodic pings.
        pub const keepalive = struct {
            pub const name = "keepalive";

            pub const Type = c_int;
        };

        /// The number of outstanding connections in the listen queue.
        pub const listen_backlog = struct {
            pub const name = "listen-backlog";

            pub const Type = c_int;
        };

        /// The local address the socket is bound to.
        pub const local_address = struct {
            pub const name = "local-address";

            pub const Type = ?*gio.SocketAddress;
        };

        /// Whether outgoing multicast packets loop back to the local host.
        pub const multicast_loopback = struct {
            pub const name = "multicast-loopback";

            pub const Type = c_int;
        };

        /// Time-to-live out outgoing multicast packets
        pub const multicast_ttl = struct {
            pub const name = "multicast-ttl";

            pub const Type = c_uint;
        };

        /// The ID of the protocol to use, or `-1` for unknown.
        pub const protocol = struct {
            pub const name = "protocol";

            pub const Type = gio.SocketProtocol;
        };

        /// The remote address the socket is connected to.
        pub const remote_address = struct {
            pub const name = "remote-address";

            pub const Type = ?*gio.SocketAddress;
        };

        /// The timeout in seconds on socket I/O
        pub const timeout = struct {
            pub const name = "timeout";

            pub const Type = c_uint;
        };

        /// Time-to-live for outgoing unicast packets
        pub const ttl = struct {
            pub const name = "ttl";

            pub const Type = c_uint;
        };

        /// The socket’s type.
        pub const @"type" = struct {
            pub const name = "type";

            pub const Type = gio.SocketType;
        };
    };

    pub const signals = struct {};

    /// Creates a new `gio.Socket` with the defined family, type and protocol.
    /// If `protocol` is 0 (`G_SOCKET_PROTOCOL_DEFAULT`) the default protocol type
    /// for the family and type is used.
    ///
    /// The `protocol` is a family and type specific int that specifies what
    /// kind of protocol to use. `gio.SocketProtocol` lists several common ones.
    /// Many families only support one protocol, and use 0 for this, others
    /// support several and using 0 means to use the default protocol for
    /// the family and type.
    ///
    /// The protocol id is passed directly to the operating
    /// system, so you can use protocols not listed in `gio.SocketProtocol` if you
    /// know the protocol number used for it.
    extern fn g_socket_new(p_family: gio.SocketFamily, p_type: gio.SocketType, p_protocol: gio.SocketProtocol, p_error: ?*?*glib.Error) ?*gio.Socket;
    pub const new = g_socket_new;

    /// Creates a new `gio.Socket` from a native file descriptor
    /// or winsock SOCKET handle.
    ///
    /// This reads all the settings from the file descriptor so that
    /// all properties should work. Note that the file descriptor
    /// will be set to non-blocking mode, independent on the blocking
    /// mode of the `gio.Socket`.
    ///
    /// On success, the returned `gio.Socket` takes ownership of `fd`. On failure, the
    /// caller must close `fd` themselves.
    ///
    /// Since GLib 2.46, it is no longer a fatal error to call this on a non-socket
    /// descriptor.  Instead, a GError will be set with code `G_IO_ERROR_FAILED`
    extern fn g_socket_new_from_fd(p_fd: c_int, p_error: ?*?*glib.Error) ?*gio.Socket;
    pub const newFromFd = g_socket_new_from_fd;

    /// Accept incoming connections on a connection-based socket. This removes
    /// the first outstanding connection request from the listening socket and
    /// creates a `gio.Socket` object for it.
    ///
    /// The `socket` must be bound to a local address with `gio.Socket.bind` and
    /// must be listening for incoming connections (`gio.Socket.listen`).
    ///
    /// If there are no outstanding connections then the operation will block
    /// or return `G_IO_ERROR_WOULD_BLOCK` if non-blocking I/O is enabled.
    /// To be notified of an incoming connection, wait for the `G_IO_IN` condition.
    extern fn g_socket_accept(p_socket: *Socket, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.Socket;
    pub const accept = g_socket_accept;

    /// When a socket is created it is attached to an address family, but it
    /// doesn't have an address in this family. `gio.Socket.bind` assigns the
    /// address (sometimes called name) of the socket.
    ///
    /// It is generally required to bind to a local address before you can
    /// receive connections. (See `gio.Socket.listen` and `gio.Socket.accept` ).
    /// In certain situations, you may also want to bind a socket that will be
    /// used to initiate connections, though this is not normally required.
    ///
    /// If `socket` is a TCP socket, then `allow_reuse` controls the setting
    /// of the `SO_REUSEADDR` socket option; normally it should be `TRUE` for
    /// server sockets (sockets that you will eventually call
    /// `gio.Socket.accept` on), and `FALSE` for client sockets. (Failing to
    /// set this flag on a server socket may cause `gio.Socket.bind` to return
    /// `G_IO_ERROR_ADDRESS_IN_USE` if the server program is stopped and then
    /// immediately restarted.)
    ///
    /// If `socket` is a UDP socket, then `allow_reuse` determines whether or
    /// not other UDP sockets can be bound to the same address at the same
    /// time. In particular, you can have several UDP sockets bound to the
    /// same address, and they will all receive all of the multicast and
    /// broadcast packets sent to that address. (The behavior of unicast
    /// UDP packets to an address with multiple listeners is not defined.)
    extern fn g_socket_bind(p_socket: *Socket, p_address: *gio.SocketAddress, p_allow_reuse: c_int, p_error: ?*?*glib.Error) c_int;
    pub const bind = g_socket_bind;

    /// Checks and resets the pending connect error for the socket.
    /// This is used to check for errors when `gio.Socket.connect` is
    /// used in non-blocking mode.
    extern fn g_socket_check_connect_result(p_socket: *Socket, p_error: ?*?*glib.Error) c_int;
    pub const checkConnectResult = g_socket_check_connect_result;

    /// Closes the socket, shutting down any active connection.
    ///
    /// Closing a socket does not wait for all outstanding I/O operations
    /// to finish, so the caller should not rely on them to be guaranteed
    /// to complete even if the close returns with no error.
    ///
    /// Once the socket is closed, all other operations will return
    /// `G_IO_ERROR_CLOSED`. Closing a socket multiple times will not
    /// return an error.
    ///
    /// Sockets will be automatically closed when the last reference
    /// is dropped, but you might want to call this function to make sure
    /// resources are released as early as possible.
    ///
    /// Beware that due to the way that TCP works, it is possible for
    /// recently-sent data to be lost if either you close a socket while the
    /// `G_IO_IN` condition is set, or else if the remote connection tries to
    /// send something to you after you close the socket but before it has
    /// finished reading all of the data you sent. There is no easy generic
    /// way to avoid this problem; the easiest fix is to design the network
    /// protocol such that the client will never send data "out of turn".
    /// Another solution is for the server to half-close the connection by
    /// calling `gio.Socket.shutdown` with only the `shutdown_write` flag set,
    /// and then wait for the client to notice this and close its side of the
    /// connection, after which the server can safely call `gio.Socket.close`.
    /// (This is what `gio.TcpConnection` does if you call
    /// `gio.TcpConnection.setGracefulDisconnect`. But of course, this
    /// only works if the client will close its connection after the server
    /// does.)
    extern fn g_socket_close(p_socket: *Socket, p_error: ?*?*glib.Error) c_int;
    pub const close = g_socket_close;

    /// Checks on the readiness of `socket` to perform operations.
    /// The operations specified in `condition` are checked for and masked
    /// against the currently-satisfied conditions on `socket`. The result
    /// is returned.
    ///
    /// Note that on Windows, it is possible for an operation to return
    /// `G_IO_ERROR_WOULD_BLOCK` even immediately after
    /// `gio.Socket.conditionCheck` has claimed that the socket is ready for
    /// writing. Rather than calling `gio.Socket.conditionCheck` and then
    /// writing to the socket if it succeeds, it is generally better to
    /// simply try writing to the socket right away, and try again later if
    /// the initial attempt returns `G_IO_ERROR_WOULD_BLOCK`.
    ///
    /// It is meaningless to specify `G_IO_ERR` or `G_IO_HUP` in condition;
    /// these conditions will always be set in the output if they are true.
    ///
    /// This call never blocks.
    extern fn g_socket_condition_check(p_socket: *Socket, p_condition: glib.IOCondition) glib.IOCondition;
    pub const conditionCheck = g_socket_condition_check;

    /// Waits for up to `timeout_us` microseconds for `condition` to become true
    /// on `socket`. If the condition is met, `TRUE` is returned.
    ///
    /// If `cancellable` is cancelled before the condition is met, or if
    /// `timeout_us` (or the socket's `gio.Socket.properties.timeout`) is reached before the
    /// condition is met, then `FALSE` is returned and `error`, if non-`NULL`,
    /// is set to the appropriate value (`G_IO_ERROR_CANCELLED` or
    /// `G_IO_ERROR_TIMED_OUT`).
    ///
    /// If you don't want a timeout, use `gio.Socket.conditionWait`.
    /// (Alternatively, you can pass -1 for `timeout_us`.)
    ///
    /// Note that although `timeout_us` is in microseconds for consistency with
    /// other GLib APIs, this function actually only has millisecond
    /// resolution, and the behavior is undefined if `timeout_us` is not an
    /// exact number of milliseconds.
    extern fn g_socket_condition_timed_wait(p_socket: *Socket, p_condition: glib.IOCondition, p_timeout_us: i64, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const conditionTimedWait = g_socket_condition_timed_wait;

    /// Waits for `condition` to become true on `socket`. When the condition
    /// is met, `TRUE` is returned.
    ///
    /// If `cancellable` is cancelled before the condition is met, or if the
    /// socket has a timeout set and it is reached before the condition is
    /// met, then `FALSE` is returned and `error`, if non-`NULL`, is set to
    /// the appropriate value (`G_IO_ERROR_CANCELLED` or
    /// `G_IO_ERROR_TIMED_OUT`).
    ///
    /// See also `gio.Socket.conditionTimedWait`.
    extern fn g_socket_condition_wait(p_socket: *Socket, p_condition: glib.IOCondition, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const conditionWait = g_socket_condition_wait;

    /// Connect the socket to the specified remote address.
    ///
    /// For connection oriented socket this generally means we attempt to make
    /// a connection to the `address`. For a connection-less socket it sets
    /// the default address for `gio.Socket.send` and discards all incoming datagrams
    /// from other sources.
    ///
    /// Generally connection oriented sockets can only connect once, but
    /// connection-less sockets can connect multiple times to change the
    /// default address.
    ///
    /// If the connect call needs to do network I/O it will block, unless
    /// non-blocking I/O is enabled. Then `G_IO_ERROR_PENDING` is returned
    /// and the user can be notified of the connection finishing by waiting
    /// for the G_IO_OUT condition. The result of the connection must then be
    /// checked with `gio.Socket.checkConnectResult`.
    extern fn g_socket_connect(p_socket: *Socket, p_address: *gio.SocketAddress, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const connect = g_socket_connect;

    /// Creates a `gio.SocketConnection` subclass of the right type for
    /// `socket`.
    extern fn g_socket_connection_factory_create_connection(p_socket: *Socket) *gio.SocketConnection;
    pub const connectionFactoryCreateConnection = g_socket_connection_factory_create_connection;

    /// Creates a `glib.Source` that can be attached to a `glib.MainContext` to monitor
    /// for the availability of the specified `condition` on the socket. The `glib.Source`
    /// keeps a reference to the `socket`.
    ///
    /// The callback on the source is of the `gio.SocketSourceFunc` type.
    ///
    /// It is meaningless to specify `G_IO_ERR` or `G_IO_HUP` in `condition`;
    /// these conditions will always be reported output if they are true.
    ///
    /// `cancellable` if not `NULL` can be used to cancel the source, which will
    /// cause the source to trigger, reporting the current condition (which
    /// is likely 0 unless cancellation happened at the same time as a
    /// condition change). You can check for this in the callback using
    /// `gio.Cancellable.isCancelled`.
    ///
    /// If `socket` has a timeout set, and it is reached before `condition`
    /// occurs, the source will then trigger anyway, reporting `G_IO_IN` or
    /// `G_IO_OUT` depending on `condition`. However, `socket` will have been
    /// marked as having had a timeout, and so the next `gio.Socket` I/O method
    /// you call will then fail with a `G_IO_ERROR_TIMED_OUT`.
    extern fn g_socket_create_source(p_socket: *Socket, p_condition: glib.IOCondition, p_cancellable: ?*gio.Cancellable) *glib.Source;
    pub const createSource = g_socket_create_source;

    /// Get the amount of data pending in the OS input buffer, without blocking.
    ///
    /// If `socket` is a UDP or SCTP socket, this will return the size of
    /// just the next packet, even if additional packets are buffered after
    /// that one.
    ///
    /// Note that on Windows, this function is rather inefficient in the
    /// UDP case, and so if you know any plausible upper bound on the size
    /// of the incoming packet, it is better to just do a
    /// `gio.Socket.receive` with a buffer of that size, rather than calling
    /// `gio.Socket.getAvailableBytes` first and then doing a receive of
    /// exactly the right size.
    extern fn g_socket_get_available_bytes(p_socket: *Socket) isize;
    pub const getAvailableBytes = g_socket_get_available_bytes;

    /// Gets the blocking mode of the socket. For details on blocking I/O,
    /// see `gio.Socket.setBlocking`.
    extern fn g_socket_get_blocking(p_socket: *Socket) c_int;
    pub const getBlocking = g_socket_get_blocking;

    /// Gets the broadcast setting on `socket`; if `TRUE`,
    /// it is possible to send packets to broadcast
    /// addresses.
    extern fn g_socket_get_broadcast(p_socket: *Socket) c_int;
    pub const getBroadcast = g_socket_get_broadcast;

    /// Returns the credentials of the foreign process connected to this
    /// socket, if any (e.g. it is only supported for `G_SOCKET_FAMILY_UNIX`
    /// sockets).
    ///
    /// If this operation isn't supported on the OS, the method fails with
    /// the `G_IO_ERROR_NOT_SUPPORTED` error. On Linux this is implemented
    /// by reading the `SO_PEERCRED` option on the underlying socket.
    ///
    /// This method can be expected to be available on the following platforms:
    ///
    /// - Linux since GLib 2.26
    /// - OpenBSD since GLib 2.30
    /// - Solaris, Illumos and OpenSolaris since GLib 2.40
    /// - NetBSD since GLib 2.42
    /// - macOS, tvOS, iOS since GLib 2.66
    ///
    /// Other ways to obtain credentials from a foreign peer includes the
    /// `gio.UnixCredentialsMessage` type and
    /// `gio.UnixConnection.sendCredentials` /
    /// `gio.UnixConnection.receiveCredentials` functions.
    extern fn g_socket_get_credentials(p_socket: *Socket, p_error: ?*?*glib.Error) ?*gio.Credentials;
    pub const getCredentials = g_socket_get_credentials;

    /// Gets the socket family of the socket.
    extern fn g_socket_get_family(p_socket: *Socket) gio.SocketFamily;
    pub const getFamily = g_socket_get_family;

    /// Returns the underlying OS socket object. On unix this
    /// is a socket file descriptor, and on Windows this is
    /// a Winsock2 SOCKET handle. This may be useful for
    /// doing platform specific or otherwise unusual operations
    /// on the socket.
    extern fn g_socket_get_fd(p_socket: *Socket) c_int;
    pub const getFd = g_socket_get_fd;

    /// Gets the keepalive mode of the socket. For details on this,
    /// see `gio.Socket.setKeepalive`.
    extern fn g_socket_get_keepalive(p_socket: *Socket) c_int;
    pub const getKeepalive = g_socket_get_keepalive;

    /// Gets the listen backlog setting of the socket. For details on this,
    /// see `gio.Socket.setListenBacklog`.
    extern fn g_socket_get_listen_backlog(p_socket: *Socket) c_int;
    pub const getListenBacklog = g_socket_get_listen_backlog;

    /// Try to get the local address of a bound socket. This is only
    /// useful if the socket has been bound to a local address,
    /// either explicitly or implicitly when connecting.
    extern fn g_socket_get_local_address(p_socket: *Socket, p_error: ?*?*glib.Error) ?*gio.SocketAddress;
    pub const getLocalAddress = g_socket_get_local_address;

    /// Gets the multicast loopback setting on `socket`; if `TRUE` (the
    /// default), outgoing multicast packets will be looped back to
    /// multicast listeners on the same host.
    extern fn g_socket_get_multicast_loopback(p_socket: *Socket) c_int;
    pub const getMulticastLoopback = g_socket_get_multicast_loopback;

    /// Gets the multicast time-to-live setting on `socket`; see
    /// `gio.Socket.setMulticastTtl` for more details.
    extern fn g_socket_get_multicast_ttl(p_socket: *Socket) c_uint;
    pub const getMulticastTtl = g_socket_get_multicast_ttl;

    /// Gets the value of an integer-valued option on `socket`, as with
    /// `getsockopt`. (If you need to fetch a  non-integer-valued option,
    /// you will need to call `getsockopt` directly.)
    ///
    /// The [`<gio/gnetworking.h>`](networking.html)
    /// header pulls in system headers that will define most of the
    /// standard/portable socket options. For unusual socket protocols or
    /// platform-dependent options, you may need to include additional
    /// headers.
    ///
    /// Note that even for socket options that are a single byte in size,
    /// `value` is still a pointer to a `gint` variable, not a `guchar`;
    /// `gio.Socket.getOption` will handle the conversion internally.
    extern fn g_socket_get_option(p_socket: *Socket, p_level: c_int, p_optname: c_int, p_value: *c_int, p_error: ?*?*glib.Error) c_int;
    pub const getOption = g_socket_get_option;

    /// Gets the socket protocol id the socket was created with.
    /// In case the protocol is unknown, -1 is returned.
    extern fn g_socket_get_protocol(p_socket: *Socket) gio.SocketProtocol;
    pub const getProtocol = g_socket_get_protocol;

    /// Try to get the remote address of a connected socket. This is only
    /// useful for connection oriented sockets that have been connected.
    extern fn g_socket_get_remote_address(p_socket: *Socket, p_error: ?*?*glib.Error) ?*gio.SocketAddress;
    pub const getRemoteAddress = g_socket_get_remote_address;

    /// Gets the socket type of the socket.
    extern fn g_socket_get_socket_type(p_socket: *Socket) gio.SocketType;
    pub const getSocketType = g_socket_get_socket_type;

    /// Gets the timeout setting of the socket. For details on this, see
    /// `gio.Socket.setTimeout`.
    extern fn g_socket_get_timeout(p_socket: *Socket) c_uint;
    pub const getTimeout = g_socket_get_timeout;

    /// Gets the unicast time-to-live setting on `socket`; see
    /// `gio.Socket.setTtl` for more details.
    extern fn g_socket_get_ttl(p_socket: *Socket) c_uint;
    pub const getTtl = g_socket_get_ttl;

    /// Checks whether a socket is closed.
    extern fn g_socket_is_closed(p_socket: *Socket) c_int;
    pub const isClosed = g_socket_is_closed;

    /// Check whether the socket is connected. This is only useful for
    /// connection-oriented sockets.
    ///
    /// If using `gio.Socket.shutdown`, this function will return `TRUE` until the
    /// socket has been shut down for reading and writing. If you do a non-blocking
    /// connect, this function will not return `TRUE` until after you call
    /// `gio.Socket.checkConnectResult`.
    extern fn g_socket_is_connected(p_socket: *Socket) c_int;
    pub const isConnected = g_socket_is_connected;

    /// Registers `socket` to receive multicast messages sent to `group`.
    /// `socket` must be a `G_SOCKET_TYPE_DATAGRAM` socket, and must have
    /// been bound to an appropriate interface and port with
    /// `gio.Socket.bind`.
    ///
    /// If `iface` is `NULL`, the system will automatically pick an interface
    /// to bind to based on `group`.
    ///
    /// If `source_specific` is `TRUE`, source-specific multicast as defined
    /// in RFC 4604 is used. Note that on older platforms this may fail
    /// with a `G_IO_ERROR_NOT_SUPPORTED` error.
    ///
    /// To bind to a given source-specific multicast address, use
    /// `gio.Socket.joinMulticastGroupSsm` instead.
    extern fn g_socket_join_multicast_group(p_socket: *Socket, p_group: *gio.InetAddress, p_source_specific: c_int, p_iface: ?[*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const joinMulticastGroup = g_socket_join_multicast_group;

    /// Registers `socket` to receive multicast messages sent to `group`.
    /// `socket` must be a `G_SOCKET_TYPE_DATAGRAM` socket, and must have
    /// been bound to an appropriate interface and port with
    /// `gio.Socket.bind`.
    ///
    /// If `iface` is `NULL`, the system will automatically pick an interface
    /// to bind to based on `group`.
    ///
    /// If `source_specific` is not `NULL`, use source-specific multicast as
    /// defined in RFC 4604. Note that on older platforms this may fail
    /// with a `G_IO_ERROR_NOT_SUPPORTED` error.
    ///
    /// Note that this function can be called multiple times for the same
    /// `group` with different `source_specific` in order to receive multicast
    /// packets from more than one source.
    extern fn g_socket_join_multicast_group_ssm(p_socket: *Socket, p_group: *gio.InetAddress, p_source_specific: ?*gio.InetAddress, p_iface: ?[*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const joinMulticastGroupSsm = g_socket_join_multicast_group_ssm;

    /// Removes `socket` from the multicast group defined by `group`, `iface`,
    /// and `source_specific` (which must all have the same values they had
    /// when you joined the group).
    ///
    /// `socket` remains bound to its address and port, and can still receive
    /// unicast messages after calling this.
    ///
    /// To unbind to a given source-specific multicast address, use
    /// `gio.Socket.leaveMulticastGroupSsm` instead.
    extern fn g_socket_leave_multicast_group(p_socket: *Socket, p_group: *gio.InetAddress, p_source_specific: c_int, p_iface: ?[*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const leaveMulticastGroup = g_socket_leave_multicast_group;

    /// Removes `socket` from the multicast group defined by `group`, `iface`,
    /// and `source_specific` (which must all have the same values they had
    /// when you joined the group).
    ///
    /// `socket` remains bound to its address and port, and can still receive
    /// unicast messages after calling this.
    extern fn g_socket_leave_multicast_group_ssm(p_socket: *Socket, p_group: *gio.InetAddress, p_source_specific: ?*gio.InetAddress, p_iface: ?[*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const leaveMulticastGroupSsm = g_socket_leave_multicast_group_ssm;

    /// Marks the socket as a server socket, i.e. a socket that is used
    /// to accept incoming requests using `gio.Socket.accept`.
    ///
    /// Before calling this the socket must be bound to a local address using
    /// `gio.Socket.bind`.
    ///
    /// To set the maximum amount of outstanding clients, use
    /// `gio.Socket.setListenBacklog`.
    extern fn g_socket_listen(p_socket: *Socket, p_error: ?*?*glib.Error) c_int;
    pub const listen = g_socket_listen;

    /// Receive data (up to `size` bytes) from a socket. This is mainly used by
    /// connection-oriented sockets; it is identical to `gio.Socket.receiveFrom`
    /// with `address` set to `NULL`.
    ///
    /// For `G_SOCKET_TYPE_DATAGRAM` and `G_SOCKET_TYPE_SEQPACKET` sockets,
    /// `gio.Socket.receive` will always read either 0 or 1 complete messages from
    /// the socket. If the received message is too large to fit in `buffer`, then
    /// the data beyond `size` bytes will be discarded, without any explicit
    /// indication that this has occurred.
    ///
    /// For `G_SOCKET_TYPE_STREAM` sockets, `gio.Socket.receive` can return any
    /// number of bytes, up to `size`. If more than `size` bytes have been
    /// received, the additional data will be returned in future calls to
    /// `gio.Socket.receive`.
    ///
    /// If the socket is in blocking mode the call will block until there
    /// is some data to receive, the connection is closed, or there is an
    /// error. If there is no data available and the socket is in
    /// non-blocking mode, a `G_IO_ERROR_WOULD_BLOCK` error will be
    /// returned. To be notified when data is available, wait for the
    /// `G_IO_IN` condition.
    ///
    /// On error -1 is returned and `error` is set accordingly.
    extern fn g_socket_receive(p_socket: *Socket, p_buffer: [*]u8, p_size: usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) isize;
    pub const receive = g_socket_receive;

    /// Receives data (up to `size` bytes) from a socket.
    ///
    /// This function is a variant of `gio.Socket.receive` which returns a
    /// `glib.Bytes` rather than a plain buffer.
    ///
    /// Pass `-1` to `timeout_us` to block indefinitely until data is received (or
    /// the connection is closed, or there is an error). Pass `0` to use the default
    /// timeout from `gio.Socket.properties.timeout`, or pass a positive number to wait
    /// for that many microseconds for data before returning `G_IO_ERROR_TIMED_OUT`.
    extern fn g_socket_receive_bytes(p_socket: *Socket, p_size: usize, p_timeout_us: i64, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*glib.Bytes;
    pub const receiveBytes = g_socket_receive_bytes;

    /// Receive data (up to `size` bytes) from a socket.
    ///
    /// This function is a variant of `gio.Socket.receiveFrom` which returns
    /// a `glib.Bytes` rather than a plain buffer.
    ///
    /// If `address` is non-`NULL` then `address` will be set equal to the
    /// source address of the received packet.
    ///
    /// The `address` is owned by the caller.
    ///
    /// Pass `-1` to `timeout_us` to block indefinitely until data is received (or
    /// the connection is closed, or there is an error). Pass `0` to use the default
    /// timeout from `gio.Socket.properties.timeout`, or pass a positive number to wait
    /// for that many microseconds for data before returning `G_IO_ERROR_TIMED_OUT`.
    extern fn g_socket_receive_bytes_from(p_socket: *Socket, p_address: ?**gio.SocketAddress, p_size: usize, p_timeout_us: i64, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*glib.Bytes;
    pub const receiveBytesFrom = g_socket_receive_bytes_from;

    /// Receive data (up to `size` bytes) from a socket.
    ///
    /// If `address` is non-`NULL` then `address` will be set equal to the
    /// source address of the received packet.
    /// `address` is owned by the caller.
    ///
    /// See `gio.Socket.receive` for additional information.
    extern fn g_socket_receive_from(p_socket: *Socket, p_address: ?**gio.SocketAddress, p_buffer: [*]u8, p_size: usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) isize;
    pub const receiveFrom = g_socket_receive_from;

    /// Receive data from a socket.  For receiving multiple messages, see
    /// `gio.Socket.receiveMessages`; for easier use, see
    /// `gio.Socket.receive` and `gio.Socket.receiveFrom`.
    ///
    /// If `address` is non-`NULL` then `address` will be set equal to the
    /// source address of the received packet.
    /// `address` is owned by the caller.
    ///
    /// `vector` must point to an array of `gio.InputVector` structs and
    /// `num_vectors` must be the length of this array.  These structs
    /// describe the buffers that received data will be scattered into.
    /// If `num_vectors` is -1, then `vectors` is assumed to be terminated
    /// by a `gio.InputVector` with a `NULL` buffer pointer.
    ///
    /// As a special case, if `num_vectors` is 0 (in which case, `vectors`
    /// may of course be `NULL`), then a single byte is received and
    /// discarded. This is to facilitate the common practice of sending a
    /// single '\0' byte for the purposes of transferring ancillary data.
    ///
    /// `messages`, if non-`NULL`, will be set to point to a newly-allocated
    /// array of `gio.SocketControlMessage` instances or `NULL` if no such
    /// messages was received. These correspond to the control messages
    /// received from the kernel, one `gio.SocketControlMessage` per message
    /// from the kernel. This array is `NULL`-terminated and must be freed
    /// by the caller using `glib.free` after calling `gobject.Object.unref` on each
    /// element. If `messages` is `NULL`, any control messages received will
    /// be discarded.
    ///
    /// `num_messages`, if non-`NULL`, will be set to the number of control
    /// messages received.
    ///
    /// If both `messages` and `num_messages` are non-`NULL`, then
    /// `num_messages` gives the number of `gio.SocketControlMessage` instances
    /// in `messages` (ie: not including the `NULL` terminator).
    ///
    /// `flags` is an in/out parameter. The commonly available arguments
    /// for this are available in the `gio.SocketMsgFlags` enum, but the
    /// values there are the same as the system values, and the flags
    /// are passed in as-is, so you can pass in system-specific flags too
    /// (and `gio.Socket.receiveMessage` may pass system-specific flags out).
    /// Flags passed in to the parameter affect the receive operation; flags returned
    /// out of it are relevant to the specific returned message.
    ///
    /// As with `gio.Socket.receive`, data may be discarded if `socket` is
    /// `G_SOCKET_TYPE_DATAGRAM` or `G_SOCKET_TYPE_SEQPACKET` and you do not
    /// provide enough buffer space to read a complete message. You can pass
    /// `G_SOCKET_MSG_PEEK` in `flags` to peek at the current message without
    /// removing it from the receive queue, but there is no portable way to find
    /// out the length of the message other than by reading it into a
    /// sufficiently-large buffer.
    ///
    /// If the socket is in blocking mode the call will block until there
    /// is some data to receive, the connection is closed, or there is an
    /// error. If there is no data available and the socket is in
    /// non-blocking mode, a `G_IO_ERROR_WOULD_BLOCK` error will be
    /// returned. To be notified when data is available, wait for the
    /// `G_IO_IN` condition.
    ///
    /// On error -1 is returned and `error` is set accordingly.
    extern fn g_socket_receive_message(p_socket: *Socket, p_address: ?**gio.SocketAddress, p_vectors: [*]gio.InputVector, p_num_vectors: c_int, p_messages: ?[*]**gio.SocketControlMessage, p_num_messages: *c_int, p_flags: *c_int, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) isize;
    pub const receiveMessage = g_socket_receive_message;

    /// Receive multiple data messages from `socket` in one go.  This is the most
    /// complicated and fully-featured version of this call. For easier use, see
    /// `gio.Socket.receive`, `gio.Socket.receiveFrom`, and `gio.Socket.receiveMessage`.
    ///
    /// `messages` must point to an array of `gio.InputMessage` structs and
    /// `num_messages` must be the length of this array. Each `gio.InputMessage`
    /// contains a pointer to an array of `gio.InputVector` structs describing the
    /// buffers that the data received in each message will be written to. Using
    /// multiple `GInputVectors` is more memory-efficient than manually copying data
    /// out of a single buffer to multiple sources, and more system-call-efficient
    /// than making multiple calls to `gio.Socket.receive`, such as in scenarios where
    /// a lot of data packets need to be received (e.g. high-bandwidth video
    /// streaming over RTP/UDP).
    ///
    /// `flags` modify how all messages are received. The commonly available
    /// arguments for this are available in the `gio.SocketMsgFlags` enum, but the
    /// values there are the same as the system values, and the flags
    /// are passed in as-is, so you can pass in system-specific flags too. These
    /// flags affect the overall receive operation. Flags affecting individual
    /// messages are returned in `gio.InputMessage.flags`.
    ///
    /// The other members of `gio.InputMessage` are treated as described in its
    /// documentation.
    ///
    /// If `gio.Socket.properties.blocking` is `TRUE` the call will block until `num_messages` have
    /// been received, or the end of the stream is reached.
    ///
    /// If `gio.Socket.properties.blocking` is `FALSE` the call will return up to `num_messages`
    /// without blocking, or `G_IO_ERROR_WOULD_BLOCK` if no messages are queued in the
    /// operating system to be received.
    ///
    /// In blocking mode, if `gio.Socket.properties.timeout` is positive and is reached before any
    /// messages are received, `G_IO_ERROR_TIMED_OUT` is returned, otherwise up to
    /// `num_messages` are returned. (Note: This is effectively the
    /// behaviour of `MSG_WAITFORONE` with `recvmmsg`.)
    ///
    /// To be notified when messages are available, wait for the
    /// `G_IO_IN` condition. Note though that you may still receive
    /// `G_IO_ERROR_WOULD_BLOCK` from `gio.Socket.receiveMessages` even if you were
    /// previously notified of a `G_IO_IN` condition.
    ///
    /// If the remote peer closes the connection, any messages queued in the
    /// operating system will be returned, and subsequent calls to
    /// `gio.Socket.receiveMessages` will return 0 (with no error set).
    ///
    /// On error -1 is returned and `error` is set accordingly. An error will only
    /// be returned if zero messages could be received; otherwise the number of
    /// messages successfully received before the error will be returned.
    extern fn g_socket_receive_messages(p_socket: *Socket, p_messages: [*]gio.InputMessage, p_num_messages: c_uint, p_flags: c_int, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const receiveMessages = g_socket_receive_messages;

    /// This behaves exactly the same as `gio.Socket.receive`, except that
    /// the choice of blocking or non-blocking behavior is determined by
    /// the `blocking` argument rather than by `socket`'s properties.
    extern fn g_socket_receive_with_blocking(p_socket: *Socket, p_buffer: [*]u8, p_size: usize, p_blocking: c_int, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) isize;
    pub const receiveWithBlocking = g_socket_receive_with_blocking;

    /// Tries to send `size` bytes from `buffer` on the socket. This is
    /// mainly used by connection-oriented sockets; it is identical to
    /// `gio.Socket.sendTo` with `address` set to `NULL`.
    ///
    /// If the socket is in blocking mode the call will block until there is
    /// space for the data in the socket queue. If there is no space available
    /// and the socket is in non-blocking mode a `G_IO_ERROR_WOULD_BLOCK` error
    /// will be returned. To be notified when space is available, wait for the
    /// `G_IO_OUT` condition. Note though that you may still receive
    /// `G_IO_ERROR_WOULD_BLOCK` from `gio.Socket.send` even if you were previously
    /// notified of a `G_IO_OUT` condition. (On Windows in particular, this is
    /// very common due to the way the underlying APIs work.)
    ///
    /// On error -1 is returned and `error` is set accordingly.
    extern fn g_socket_send(p_socket: *Socket, p_buffer: [*]const u8, p_size: usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) isize;
    pub const send = g_socket_send;

    /// Send data to `address` on `socket`.  For sending multiple messages see
    /// `gio.Socket.sendMessages`; for easier use, see
    /// `gio.Socket.send` and `gio.Socket.sendTo`.
    ///
    /// If `address` is `NULL` then the message is sent to the default receiver
    /// (set by `gio.Socket.connect`).
    ///
    /// `vectors` must point to an array of `gio.OutputVector` structs and
    /// `num_vectors` must be the length of this array. (If `num_vectors` is -1,
    /// then `vectors` is assumed to be terminated by a `gio.OutputVector` with a
    /// `NULL` buffer pointer.) The `gio.OutputVector` structs describe the buffers
    /// that the sent data will be gathered from. Using multiple
    /// `GOutputVectors` is more memory-efficient than manually copying
    /// data from multiple sources into a single buffer, and more
    /// network-efficient than making multiple calls to `gio.Socket.send`.
    ///
    /// `messages`, if non-`NULL`, is taken to point to an array of `num_messages`
    /// `gio.SocketControlMessage` instances. These correspond to the control
    /// messages to be sent on the socket.
    /// If `num_messages` is -1 then `messages` is treated as a `NULL`-terminated
    /// array.
    ///
    /// `flags` modify how the message is sent. The commonly available arguments
    /// for this are available in the `gio.SocketMsgFlags` enum, but the
    /// values there are the same as the system values, and the flags
    /// are passed in as-is, so you can pass in system-specific flags too.
    ///
    /// If the socket is in blocking mode the call will block until there is
    /// space for the data in the socket queue. If there is no space available
    /// and the socket is in non-blocking mode a `G_IO_ERROR_WOULD_BLOCK` error
    /// will be returned. To be notified when space is available, wait for the
    /// `G_IO_OUT` condition. Note though that you may still receive
    /// `G_IO_ERROR_WOULD_BLOCK` from `gio.Socket.send` even if you were previously
    /// notified of a `G_IO_OUT` condition. (On Windows in particular, this is
    /// very common due to the way the underlying APIs work.)
    ///
    /// The sum of the sizes of each `gio.OutputVector` in vectors must not be
    /// greater than `G_MAXSSIZE`. If the message can be larger than this,
    /// then it is mandatory to use the `gio.Socket.sendMessageWithTimeout`
    /// function.
    ///
    /// On error -1 is returned and `error` is set accordingly.
    extern fn g_socket_send_message(p_socket: *Socket, p_address: ?*gio.SocketAddress, p_vectors: [*]gio.OutputVector, p_num_vectors: c_int, p_messages: ?[*]*gio.SocketControlMessage, p_num_messages: c_int, p_flags: c_int, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) isize;
    pub const sendMessage = g_socket_send_message;

    /// This behaves exactly the same as `gio.Socket.sendMessage`, except that
    /// the choice of timeout behavior is determined by the `timeout_us` argument
    /// rather than by `socket`'s properties.
    ///
    /// On error `G_POLLABLE_RETURN_FAILED` is returned and `error` is set accordingly, or
    /// if the socket is currently not writable `G_POLLABLE_RETURN_WOULD_BLOCK` is
    /// returned. `bytes_written` will contain 0 in both cases.
    extern fn g_socket_send_message_with_timeout(p_socket: *Socket, p_address: ?*gio.SocketAddress, p_vectors: [*]const gio.OutputVector, p_num_vectors: c_int, p_messages: ?[*]*gio.SocketControlMessage, p_num_messages: c_int, p_flags: c_int, p_timeout_us: i64, p_bytes_written: ?*usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) gio.PollableReturn;
    pub const sendMessageWithTimeout = g_socket_send_message_with_timeout;

    /// Send multiple data messages from `socket` in one go.  This is the most
    /// complicated and fully-featured version of this call. For easier use, see
    /// `gio.Socket.send`, `gio.Socket.sendTo`, and `gio.Socket.sendMessage`.
    ///
    /// `messages` must point to an array of `gio.OutputMessage` structs and
    /// `num_messages` must be the length of this array. Each `gio.OutputMessage`
    /// contains an address to send the data to, and a pointer to an array of
    /// `gio.OutputVector` structs to describe the buffers that the data to be sent
    /// for each message will be gathered from. Using multiple `GOutputVectors` is
    /// more memory-efficient than manually copying data from multiple sources
    /// into a single buffer, and more network-efficient than making multiple
    /// calls to `gio.Socket.send`. Sending multiple messages in one go avoids the
    /// overhead of making a lot of syscalls in scenarios where a lot of data
    /// packets need to be sent (e.g. high-bandwidth video streaming over RTP/UDP),
    /// or where the same data needs to be sent to multiple recipients.
    ///
    /// `flags` modify how the message is sent. The commonly available arguments
    /// for this are available in the `gio.SocketMsgFlags` enum, but the
    /// values there are the same as the system values, and the flags
    /// are passed in as-is, so you can pass in system-specific flags too.
    ///
    /// If the socket is in blocking mode the call will block until there is
    /// space for all the data in the socket queue. If there is no space available
    /// and the socket is in non-blocking mode a `G_IO_ERROR_WOULD_BLOCK` error
    /// will be returned if no data was written at all, otherwise the number of
    /// messages sent will be returned. To be notified when space is available,
    /// wait for the `G_IO_OUT` condition. Note though that you may still receive
    /// `G_IO_ERROR_WOULD_BLOCK` from `gio.Socket.send` even if you were previously
    /// notified of a `G_IO_OUT` condition. (On Windows in particular, this is
    /// very common due to the way the underlying APIs work.)
    ///
    /// On error -1 is returned and `error` is set accordingly. An error will only
    /// be returned if zero messages could be sent; otherwise the number of messages
    /// successfully sent before the error will be returned.
    extern fn g_socket_send_messages(p_socket: *Socket, p_messages: [*]gio.OutputMessage, p_num_messages: c_uint, p_flags: c_int, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const sendMessages = g_socket_send_messages;

    /// Tries to send `size` bytes from `buffer` to `address`. If `address` is
    /// `NULL` then the message is sent to the default receiver (set by
    /// `gio.Socket.connect`).
    ///
    /// See `gio.Socket.send` for additional information.
    extern fn g_socket_send_to(p_socket: *Socket, p_address: ?*gio.SocketAddress, p_buffer: [*]const u8, p_size: usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) isize;
    pub const sendTo = g_socket_send_to;

    /// This behaves exactly the same as `gio.Socket.send`, except that
    /// the choice of blocking or non-blocking behavior is determined by
    /// the `blocking` argument rather than by `socket`'s properties.
    extern fn g_socket_send_with_blocking(p_socket: *Socket, p_buffer: [*]const u8, p_size: usize, p_blocking: c_int, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) isize;
    pub const sendWithBlocking = g_socket_send_with_blocking;

    /// Sets the blocking mode of the socket. In blocking mode
    /// all operations (which don’t take an explicit blocking parameter) block until
    /// they succeed or there is an error. In
    /// non-blocking mode all functions return results immediately or
    /// with a `G_IO_ERROR_WOULD_BLOCK` error.
    ///
    /// All sockets are created in blocking mode. However, note that the
    /// platform level socket is always non-blocking, and blocking mode
    /// is a GSocket level feature.
    extern fn g_socket_set_blocking(p_socket: *Socket, p_blocking: c_int) void;
    pub const setBlocking = g_socket_set_blocking;

    /// Sets whether `socket` should allow sending to broadcast addresses.
    /// This is `FALSE` by default.
    extern fn g_socket_set_broadcast(p_socket: *Socket, p_broadcast: c_int) void;
    pub const setBroadcast = g_socket_set_broadcast;

    /// Sets or unsets the `SO_KEEPALIVE` flag on the underlying socket. When
    /// this flag is set on a socket, the system will attempt to verify that the
    /// remote socket endpoint is still present if a sufficiently long period of
    /// time passes with no data being exchanged. If the system is unable to
    /// verify the presence of the remote endpoint, it will automatically close
    /// the connection.
    ///
    /// This option is only functional on certain kinds of sockets. (Notably,
    /// `G_SOCKET_PROTOCOL_TCP` sockets.)
    ///
    /// The exact time between pings is system- and protocol-dependent, but will
    /// normally be at least two hours. Most commonly, you would set this flag
    /// on a server socket if you want to allow clients to remain idle for long
    /// periods of time, but also want to ensure that connections are eventually
    /// garbage-collected if clients crash or become unreachable.
    extern fn g_socket_set_keepalive(p_socket: *Socket, p_keepalive: c_int) void;
    pub const setKeepalive = g_socket_set_keepalive;

    /// Sets the maximum number of outstanding connections allowed
    /// when listening on this socket. If more clients than this are
    /// connecting to the socket and the application is not handling them
    /// on time then the new connections will be refused.
    ///
    /// Note that this must be called before `gio.Socket.listen` and has no
    /// effect if called after that.
    extern fn g_socket_set_listen_backlog(p_socket: *Socket, p_backlog: c_int) void;
    pub const setListenBacklog = g_socket_set_listen_backlog;

    /// Sets whether outgoing multicast packets will be received by sockets
    /// listening on that multicast address on the same host. This is `TRUE`
    /// by default.
    extern fn g_socket_set_multicast_loopback(p_socket: *Socket, p_loopback: c_int) void;
    pub const setMulticastLoopback = g_socket_set_multicast_loopback;

    /// Sets the time-to-live for outgoing multicast datagrams on `socket`.
    /// By default, this is 1, meaning that multicast packets will not leave
    /// the local network.
    extern fn g_socket_set_multicast_ttl(p_socket: *Socket, p_ttl: c_uint) void;
    pub const setMulticastTtl = g_socket_set_multicast_ttl;

    /// Sets the value of an integer-valued option on `socket`, as with
    /// `setsockopt`. (If you need to set a non-integer-valued option,
    /// you will need to call `setsockopt` directly.)
    ///
    /// The [`<gio/gnetworking.h>`](networking.html)
    /// header pulls in system headers that will define most of the
    /// standard/portable socket options. For unusual socket protocols or
    /// platform-dependent options, you may need to include additional
    /// headers.
    extern fn g_socket_set_option(p_socket: *Socket, p_level: c_int, p_optname: c_int, p_value: c_int, p_error: ?*?*glib.Error) c_int;
    pub const setOption = g_socket_set_option;

    /// Sets the time in seconds after which I/O operations on `socket` will
    /// time out if they have not yet completed.
    ///
    /// On a blocking socket, this means that any blocking `gio.Socket`
    /// operation will time out after `timeout` seconds of inactivity,
    /// returning `G_IO_ERROR_TIMED_OUT`.
    ///
    /// On a non-blocking socket, calls to `gio.Socket.conditionWait` will
    /// also fail with `G_IO_ERROR_TIMED_OUT` after the given time. Sources
    /// created with `gio.Socket.createSource` will trigger after
    /// `timeout` seconds of inactivity, with the requested condition
    /// set, at which point calling `gio.Socket.receive`, `gio.Socket.send`,
    /// `gio.Socket.checkConnectResult`, etc, will fail with
    /// `G_IO_ERROR_TIMED_OUT`.
    ///
    /// If `timeout` is 0 (the default), operations will never time out
    /// on their own.
    ///
    /// Note that if an I/O operation is interrupted by a signal, this may
    /// cause the timeout to be reset.
    extern fn g_socket_set_timeout(p_socket: *Socket, p_timeout: c_uint) void;
    pub const setTimeout = g_socket_set_timeout;

    /// Sets the time-to-live for outgoing unicast packets on `socket`.
    /// By default the platform-specific default value is used.
    extern fn g_socket_set_ttl(p_socket: *Socket, p_ttl: c_uint) void;
    pub const setTtl = g_socket_set_ttl;

    /// Shut down part or all of a full-duplex connection.
    ///
    /// If `shutdown_read` is `TRUE` then the receiving side of the connection
    /// is shut down, and further reading is disallowed.
    ///
    /// If `shutdown_write` is `TRUE` then the sending side of the connection
    /// is shut down, and further writing is disallowed.
    ///
    /// It is allowed for both `shutdown_read` and `shutdown_write` to be `TRUE`.
    ///
    /// One example where it is useful to shut down only one side of a connection is
    /// graceful disconnect for TCP connections where you close the sending side,
    /// then wait for the other side to close the connection, thus ensuring that the
    /// other side saw all sent data.
    extern fn g_socket_shutdown(p_socket: *Socket, p_shutdown_read: c_int, p_shutdown_write: c_int, p_error: ?*?*glib.Error) c_int;
    pub const shutdown = g_socket_shutdown;

    /// Checks if a socket is capable of speaking IPv4.
    ///
    /// IPv4 sockets are capable of speaking IPv4.  On some operating systems
    /// and under some combinations of circumstances IPv6 sockets are also
    /// capable of speaking IPv4.  See RFC 3493 section 3.7 for more
    /// information.
    ///
    /// No other types of sockets are currently considered as being capable
    /// of speaking IPv4.
    extern fn g_socket_speaks_ipv4(p_socket: *Socket) c_int;
    pub const speaksIpv4 = g_socket_speaks_ipv4;

    extern fn g_socket_get_type() usize;
    pub const getGObjectType = g_socket_get_type;

    extern fn g_object_ref(p_self: *gio.Socket) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.Socket) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Socket, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GSocketAddress` is the equivalent of
/// [`struct sockaddr`](man:sockaddr(3type)) and its subtypes in the BSD sockets
/// API. This is an abstract class; use `gio.InetSocketAddress` for
/// internet sockets, or `gio.UnixSocketAddress` for UNIX domain sockets.
pub const SocketAddress = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gio.SocketConnectable};
    pub const Class = gio.SocketAddressClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {
        /// Gets the socket family type of `address`.
        pub const get_family = struct {
            pub fn call(p_class: anytype, p_address: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) gio.SocketFamily {
                return gobject.ext.as(SocketAddress.Class, p_class).f_get_family.?(gobject.ext.as(SocketAddress, p_address));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_address: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) gio.SocketFamily) void {
                gobject.ext.as(SocketAddress.Class, p_class).f_get_family = @ptrCast(p_implementation);
            }
        };

        /// Gets the size of `address`'s native struct sockaddr.
        /// You can use this to allocate memory to pass to
        /// `gio.SocketAddress.toNative`.
        pub const get_native_size = struct {
            pub fn call(p_class: anytype, p_address: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) isize {
                return gobject.ext.as(SocketAddress.Class, p_class).f_get_native_size.?(gobject.ext.as(SocketAddress, p_address));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_address: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) isize) void {
                gobject.ext.as(SocketAddress.Class, p_class).f_get_native_size = @ptrCast(p_implementation);
            }
        };

        /// Converts a `gio.SocketAddress` to a native struct sockaddr, which can
        /// be passed to low-level functions like `connect` or `bind`.
        ///
        /// If not enough space is available, a `G_IO_ERROR_NO_SPACE` error
        /// is returned. If the address type is not known on the system
        /// then a `G_IO_ERROR_NOT_SUPPORTED` error is returned.
        pub const to_native = struct {
            pub fn call(p_class: anytype, p_address: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_dest: ?*anyopaque, p_destlen: usize, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(SocketAddress.Class, p_class).f_to_native.?(gobject.ext.as(SocketAddress, p_address), p_dest, p_destlen, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_address: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_dest: ?*anyopaque, p_destlen: usize, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(SocketAddress.Class, p_class).f_to_native = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The family of the socket address.
        pub const family = struct {
            pub const name = "family";

            pub const Type = gio.SocketFamily;
        };
    };

    pub const signals = struct {};

    /// Creates a `gio.SocketAddress` subclass corresponding to the native
    /// struct sockaddr `native`.
    extern fn g_socket_address_new_from_native(p_native: *anyopaque, p_len: usize) *gio.SocketAddress;
    pub const newFromNative = g_socket_address_new_from_native;

    /// Gets the socket family type of `address`.
    extern fn g_socket_address_get_family(p_address: *SocketAddress) gio.SocketFamily;
    pub const getFamily = g_socket_address_get_family;

    /// Gets the size of `address`'s native struct sockaddr.
    /// You can use this to allocate memory to pass to
    /// `gio.SocketAddress.toNative`.
    extern fn g_socket_address_get_native_size(p_address: *SocketAddress) isize;
    pub const getNativeSize = g_socket_address_get_native_size;

    /// Converts a `gio.SocketAddress` to a native struct sockaddr, which can
    /// be passed to low-level functions like `connect` or `bind`.
    ///
    /// If not enough space is available, a `G_IO_ERROR_NO_SPACE` error
    /// is returned. If the address type is not known on the system
    /// then a `G_IO_ERROR_NOT_SUPPORTED` error is returned.
    extern fn g_socket_address_to_native(p_address: *SocketAddress, p_dest: ?*anyopaque, p_destlen: usize, p_error: ?*?*glib.Error) c_int;
    pub const toNative = g_socket_address_to_native;

    extern fn g_socket_address_get_type() usize;
    pub const getGObjectType = g_socket_address_get_type;

    extern fn g_object_ref(p_self: *gio.SocketAddress) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.SocketAddress) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SocketAddress, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GSocketAddressEnumerator` is an enumerator type for
/// `gio.SocketAddress` instances. It is returned by enumeration functions
/// such as `gio.SocketConnectable.enumerate`, which returns a
/// `GSocketAddressEnumerator` to list each `gio.SocketAddress` which could
/// be used to connect to that `gio.SocketConnectable`.
///
/// Enumeration is typically a blocking operation, so the asynchronous methods
/// `gio.SocketAddressEnumerator.nextAsync` and
/// `gio.SocketAddressEnumerator.nextFinish` should be used where
/// possible.
///
/// Each `GSocketAddressEnumerator` can only be enumerated once. Once
/// `gio.SocketAddressEnumerator.next` has returned `NULL`, further
/// enumeration with that `GSocketAddressEnumerator` is not possible, and it can
/// be unreffed.
pub const SocketAddressEnumerator = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.SocketAddressEnumeratorClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {
        /// Retrieves the next `gio.SocketAddress` from `enumerator`. Note that this
        /// may block for some amount of time. (Eg, a `gio.NetworkAddress` may need
        /// to do a DNS lookup before it can return an address.) Use
        /// `gio.SocketAddressEnumerator.nextAsync` if you need to avoid
        /// blocking.
        ///
        /// If `enumerator` is expected to yield addresses, but for some reason
        /// is unable to (eg, because of a DNS error), then the first call to
        /// `gio.SocketAddressEnumerator.next` will return an appropriate error
        /// in *`error`. However, if the first call to
        /// `gio.SocketAddressEnumerator.next` succeeds, then any further
        /// internal errors (other than `cancellable` being triggered) will be
        /// ignored.
        pub const next = struct {
            pub fn call(p_class: anytype, p_enumerator: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.SocketAddress {
                return gobject.ext.as(SocketAddressEnumerator.Class, p_class).f_next.?(gobject.ext.as(SocketAddressEnumerator, p_enumerator), p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_enumerator: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) ?*gio.SocketAddress) void {
                gobject.ext.as(SocketAddressEnumerator.Class, p_class).f_next = @ptrCast(p_implementation);
            }
        };

        /// Asynchronously retrieves the next `gio.SocketAddress` from `enumerator`
        /// and then calls `callback`, which must call
        /// `gio.SocketAddressEnumerator.nextFinish` to get the result.
        ///
        /// It is an error to call this multiple times before the previous callback has finished.
        pub const next_async = struct {
            pub fn call(p_class: anytype, p_enumerator: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(SocketAddressEnumerator.Class, p_class).f_next_async.?(gobject.ext.as(SocketAddressEnumerator, p_enumerator), p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_enumerator: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(SocketAddressEnumerator.Class, p_class).f_next_async = @ptrCast(p_implementation);
            }
        };

        /// Retrieves the result of a completed call to
        /// `gio.SocketAddressEnumerator.nextAsync`. See
        /// `gio.SocketAddressEnumerator.next` for more information about
        /// error handling.
        pub const next_finish = struct {
            pub fn call(p_class: anytype, p_enumerator: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.SocketAddress {
                return gobject.ext.as(SocketAddressEnumerator.Class, p_class).f_next_finish.?(gobject.ext.as(SocketAddressEnumerator, p_enumerator), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_enumerator: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?*gio.SocketAddress) void {
                gobject.ext.as(SocketAddressEnumerator.Class, p_class).f_next_finish = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {};

    /// Retrieves the next `gio.SocketAddress` from `enumerator`. Note that this
    /// may block for some amount of time. (Eg, a `gio.NetworkAddress` may need
    /// to do a DNS lookup before it can return an address.) Use
    /// `gio.SocketAddressEnumerator.nextAsync` if you need to avoid
    /// blocking.
    ///
    /// If `enumerator` is expected to yield addresses, but for some reason
    /// is unable to (eg, because of a DNS error), then the first call to
    /// `gio.SocketAddressEnumerator.next` will return an appropriate error
    /// in *`error`. However, if the first call to
    /// `gio.SocketAddressEnumerator.next` succeeds, then any further
    /// internal errors (other than `cancellable` being triggered) will be
    /// ignored.
    extern fn g_socket_address_enumerator_next(p_enumerator: *SocketAddressEnumerator, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.SocketAddress;
    pub const next = g_socket_address_enumerator_next;

    /// Asynchronously retrieves the next `gio.SocketAddress` from `enumerator`
    /// and then calls `callback`, which must call
    /// `gio.SocketAddressEnumerator.nextFinish` to get the result.
    ///
    /// It is an error to call this multiple times before the previous callback has finished.
    extern fn g_socket_address_enumerator_next_async(p_enumerator: *SocketAddressEnumerator, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const nextAsync = g_socket_address_enumerator_next_async;

    /// Retrieves the result of a completed call to
    /// `gio.SocketAddressEnumerator.nextAsync`. See
    /// `gio.SocketAddressEnumerator.next` for more information about
    /// error handling.
    extern fn g_socket_address_enumerator_next_finish(p_enumerator: *SocketAddressEnumerator, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.SocketAddress;
    pub const nextFinish = g_socket_address_enumerator_next_finish;

    extern fn g_socket_address_enumerator_get_type() usize;
    pub const getGObjectType = g_socket_address_enumerator_get_type;

    extern fn g_object_ref(p_self: *gio.SocketAddressEnumerator) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.SocketAddressEnumerator) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SocketAddressEnumerator, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GSocketClient` is a lightweight high-level utility class for connecting to
/// a network host using a connection oriented socket type.
///
/// You create a `GSocketClient` object, set any options you want, and then
/// call a sync or async connect operation, which returns a
/// `gio.SocketConnection` subclass on success.
///
/// The type of the `gio.SocketConnection` object returned depends on the
/// type of the underlying socket that is in use. For instance, for a TCP/IP
/// connection it will be a `gio.TcpConnection`.
///
/// As `GSocketClient` is a lightweight object, you don't need to cache it. You
/// can just create a new one any time you need one.
pub const SocketClient = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.SocketClientClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.SocketClientPrivate,

    pub const virtual_methods = struct {
        pub const event = struct {
            pub fn call(p_class: anytype, p_client: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_event: gio.SocketClientEvent, p_connectable: *gio.SocketConnectable, p_connection: *gio.IOStream) void {
                return gobject.ext.as(SocketClient.Class, p_class).f_event.?(gobject.ext.as(SocketClient, p_client), p_event, p_connectable, p_connection);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_client: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_event: gio.SocketClientEvent, p_connectable: *gio.SocketConnectable, p_connection: *gio.IOStream) callconv(.C) void) void {
                gobject.ext.as(SocketClient.Class, p_class).f_event = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// Enable proxy support.
        pub const enable_proxy = struct {
            pub const name = "enable-proxy";

            pub const Type = c_int;
        };

        /// The address family to use for socket construction.
        pub const family = struct {
            pub const name = "family";

            pub const Type = gio.SocketFamily;
        };

        /// The local address constructed sockets will be bound to.
        pub const local_address = struct {
            pub const name = "local-address";

            pub const Type = ?*gio.SocketAddress;
        };

        /// The protocol to use for socket construction, or `0` for default.
        pub const protocol = struct {
            pub const name = "protocol";

            pub const Type = gio.SocketProtocol;
        };

        /// The proxy resolver to use
        pub const proxy_resolver = struct {
            pub const name = "proxy-resolver";

            pub const Type = ?*gio.ProxyResolver;
        };

        /// The I/O timeout for sockets, in seconds, or `0` for none.
        pub const timeout = struct {
            pub const name = "timeout";

            pub const Type = c_uint;
        };

        /// Whether to create TLS connections.
        pub const tls = struct {
            pub const name = "tls";

            pub const Type = c_int;
        };

        /// The TLS validation flags used when creating TLS connections. The
        /// default value is `G_TLS_CERTIFICATE_VALIDATE_ALL`.
        ///
        /// GLib guarantees that if certificate verification fails, at least one
        /// flag will be set, but it does not guarantee that all possible flags
        /// will be set. Accordingly, you may not safely decide to ignore any
        /// particular type of error. For example, it would be incorrect to mask
        /// `G_TLS_CERTIFICATE_EXPIRED` if you want to allow expired certificates,
        /// because this could potentially be the only error flag set even if
        /// other problems exist with the certificate. Therefore, there is no
        /// safe way to use this property. This is not a horrible problem,
        /// though, because you should not be attempting to ignore validation
        /// errors anyway. If you really must ignore TLS certificate errors,
        /// connect to the `gio.SocketClient.signals.event` signal, wait for it to be
        /// emitted with `G_SOCKET_CLIENT_TLS_HANDSHAKING`, and use that to
        /// connect to `gio.TlsConnection.signals.accept`-certificate.
        pub const tls_validation_flags = struct {
            pub const name = "tls-validation-flags";

            pub const Type = gio.TlsCertificateFlags;
        };

        /// The type to use for socket construction.
        pub const @"type" = struct {
            pub const name = "type";

            pub const Type = gio.SocketType;
        };
    };

    pub const signals = struct {
        /// Emitted when `client`'s activity on `connectable` changes state.
        /// Among other things, this can be used to provide progress
        /// information about a network connection in the UI. The meanings of
        /// the different `event` values are as follows:
        ///
        /// - `G_SOCKET_CLIENT_RESOLVING`: `client` is about to look up `connectable`
        ///   in DNS. `connection` will be `NULL`.
        ///
        /// - `G_SOCKET_CLIENT_RESOLVED`:  `client` has successfully resolved
        ///   `connectable` in DNS. `connection` will be `NULL`.
        ///
        /// - `G_SOCKET_CLIENT_CONNECTING`: `client` is about to make a connection
        ///   to a remote host; either a proxy server or the destination server
        ///   itself. `connection` is the `gio.SocketConnection`, which is not yet
        ///   connected.  Since GLib 2.40, you can access the remote
        ///   address via `gio.SocketConnection.getRemoteAddress`.
        ///
        /// - `G_SOCKET_CLIENT_CONNECTED`: `client` has successfully connected
        ///   to a remote host. `connection` is the connected `gio.SocketConnection`.
        ///
        /// - `G_SOCKET_CLIENT_PROXY_NEGOTIATING`: `client` is about to negotiate
        ///   with a proxy to get it to connect to `connectable`. `connection` is
        ///   the `gio.SocketConnection` to the proxy server.
        ///
        /// - `G_SOCKET_CLIENT_PROXY_NEGOTIATED`: `client` has negotiated a
        ///   connection to `connectable` through a proxy server. `connection` is
        ///   the stream returned from `gio.Proxy.connect`, which may or may not
        ///   be a `gio.SocketConnection`.
        ///
        /// - `G_SOCKET_CLIENT_TLS_HANDSHAKING`: `client` is about to begin a TLS
        ///   handshake. `connection` is a `gio.TlsClientConnection`.
        ///
        /// - `G_SOCKET_CLIENT_TLS_HANDSHAKED`: `client` has successfully completed
        ///   the TLS handshake. `connection` is a `gio.TlsClientConnection`.
        ///
        /// - `G_SOCKET_CLIENT_COMPLETE`: `client` has either successfully connected
        ///   to `connectable` (in which case `connection` is the `gio.SocketConnection`
        ///   that it will be returning to the caller) or has failed (in which
        ///   case `connection` is `NULL` and the client is about to return an error).
        ///
        /// Each event except `G_SOCKET_CLIENT_COMPLETE` may be emitted
        /// multiple times (or not at all) for a given connectable (in
        /// particular, if `client` ends up attempting to connect to more than
        /// one address). However, if `client` emits the `gio.SocketClient.signals.event`
        /// signal at all for a given connectable, then it will always emit
        /// it with `G_SOCKET_CLIENT_COMPLETE` when it is done.
        ///
        /// Note that there may be additional `gio.SocketClientEvent` values in
        /// the future; unrecognized `event` values should be ignored.
        pub const event = struct {
            pub const name = "event";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_event: gio.SocketClientEvent, p_connectable: *gio.SocketConnectable, p_connection: ?*gio.IOStream, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(SocketClient, p_instance))),
                    gobject.signalLookup("event", SocketClient.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gio.SocketClient` with the default options.
    extern fn g_socket_client_new() *gio.SocketClient;
    pub const new = g_socket_client_new;

    /// Enable proxy protocols to be handled by the application. When the
    /// indicated proxy protocol is returned by the `gio.ProxyResolver`,
    /// `gio.SocketClient` will consider this protocol as supported but will
    /// not try to find a `gio.Proxy` instance to handle handshaking. The
    /// application must check for this case by calling
    /// `gio.SocketConnection.getRemoteAddress` on the returned
    /// `gio.SocketConnection`, and seeing if it's a `gio.ProxyAddress` of the
    /// appropriate type, to determine whether or not it needs to handle
    /// the proxy handshaking itself.
    ///
    /// This should be used for proxy protocols that are dialects of
    /// another protocol such as HTTP proxy. It also allows cohabitation of
    /// proxy protocols that are reused between protocols. A good example
    /// is HTTP. It can be used to proxy HTTP, FTP and Gopher and can also
    /// be use as generic socket proxy through the HTTP CONNECT method.
    ///
    /// When the proxy is detected as being an application proxy, TLS handshake
    /// will be skipped. This is required to let the application do the proxy
    /// specific handshake.
    extern fn g_socket_client_add_application_proxy(p_client: *SocketClient, p_protocol: [*:0]const u8) void;
    pub const addApplicationProxy = g_socket_client_add_application_proxy;

    /// Tries to resolve the `connectable` and make a network connection to it.
    ///
    /// Upon a successful connection, a new `gio.SocketConnection` is constructed
    /// and returned.  The caller owns this new object and must drop their
    /// reference to it when finished with it.
    ///
    /// The type of the `gio.SocketConnection` object returned depends on the type of
    /// the underlying socket that is used. For instance, for a TCP/IP connection
    /// it will be a `gio.TcpConnection`.
    ///
    /// The socket created will be the same family as the address that the
    /// `connectable` resolves to, unless family is set with `gio.SocketClient.setFamily`
    /// or indirectly via `gio.SocketClient.setLocalAddress`. The socket type
    /// defaults to `G_SOCKET_TYPE_STREAM` but can be set with
    /// `gio.SocketClient.setSocketType`.
    ///
    /// If a local address is specified with `gio.SocketClient.setLocalAddress` the
    /// socket will be bound to this address before connecting.
    extern fn g_socket_client_connect(p_client: *SocketClient, p_connectable: *gio.SocketConnectable, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.SocketConnection;
    pub const connect = g_socket_client_connect;

    /// This is the asynchronous version of `gio.SocketClient.connect`.
    ///
    /// You may wish to prefer the asynchronous version even in synchronous
    /// command line programs because, since 2.60, it implements
    /// [RFC 8305](https://tools.ietf.org/html/rfc8305) "Happy Eyeballs"
    /// recommendations to work around long connection timeouts in networks
    /// where IPv6 is broken by performing an IPv4 connection simultaneously
    /// without waiting for IPv6 to time out, which is not supported by the
    /// synchronous call. (This is not an API guarantee, and may change in
    /// the future.)
    ///
    /// When the operation is finished `callback` will be
    /// called. You can then call `gio.SocketClient.connectFinish` to get
    /// the result of the operation.
    extern fn g_socket_client_connect_async(p_client: *SocketClient, p_connectable: *gio.SocketConnectable, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const connectAsync = g_socket_client_connect_async;

    /// Finishes an async connect operation. See `gio.SocketClient.connectAsync`
    extern fn g_socket_client_connect_finish(p_client: *SocketClient, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.SocketConnection;
    pub const connectFinish = g_socket_client_connect_finish;

    /// This is a helper function for `gio.SocketClient.connect`.
    ///
    /// Attempts to create a TCP connection to the named host.
    ///
    /// `host_and_port` may be in any of a number of recognized formats; an IPv6
    /// address, an IPv4 address, or a domain name (in which case a DNS
    /// lookup is performed).  Quoting with [] is supported for all address
    /// types.  A port override may be specified in the usual way with a
    /// colon.  Ports may be given as decimal numbers or symbolic names (in
    /// which case an /etc/services lookup is performed).
    ///
    /// If no port override is given in `host_and_port` then `default_port` will be
    /// used as the port number to connect to.
    ///
    /// In general, `host_and_port` is expected to be provided by the user (allowing
    /// them to give the hostname, and a port override if necessary) and
    /// `default_port` is expected to be provided by the application.
    ///
    /// In the case that an IP address is given, a single connection
    /// attempt is made.  In the case that a name is given, multiple
    /// connection attempts may be made, in turn and according to the
    /// number of address records in DNS, until a connection succeeds.
    ///
    /// Upon a successful connection, a new `gio.SocketConnection` is constructed
    /// and returned.  The caller owns this new object and must drop their
    /// reference to it when finished with it.
    ///
    /// In the event of any failure (DNS error, service not found, no hosts
    /// connectable) `NULL` is returned and `error` (if non-`NULL`) is set
    /// accordingly.
    extern fn g_socket_client_connect_to_host(p_client: *SocketClient, p_host_and_port: [*:0]const u8, p_default_port: u16, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.SocketConnection;
    pub const connectToHost = g_socket_client_connect_to_host;

    /// This is the asynchronous version of `gio.SocketClient.connectToHost`.
    ///
    /// When the operation is finished `callback` will be
    /// called. You can then call `gio.SocketClient.connectToHostFinish` to get
    /// the result of the operation.
    extern fn g_socket_client_connect_to_host_async(p_client: *SocketClient, p_host_and_port: [*:0]const u8, p_default_port: u16, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const connectToHostAsync = g_socket_client_connect_to_host_async;

    /// Finishes an async connect operation. See `gio.SocketClient.connectToHostAsync`
    extern fn g_socket_client_connect_to_host_finish(p_client: *SocketClient, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.SocketConnection;
    pub const connectToHostFinish = g_socket_client_connect_to_host_finish;

    /// Attempts to create a TCP connection to a service.
    ///
    /// This call looks up the SRV record for `service` at `domain` for the
    /// "tcp" protocol.  It then attempts to connect, in turn, to each of
    /// the hosts providing the service until either a connection succeeds
    /// or there are no hosts remaining.
    ///
    /// Upon a successful connection, a new `gio.SocketConnection` is constructed
    /// and returned.  The caller owns this new object and must drop their
    /// reference to it when finished with it.
    ///
    /// In the event of any failure (DNS error, service not found, no hosts
    /// connectable) `NULL` is returned and `error` (if non-`NULL`) is set
    /// accordingly.
    extern fn g_socket_client_connect_to_service(p_client: *SocketClient, p_domain: [*:0]const u8, p_service: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.SocketConnection;
    pub const connectToService = g_socket_client_connect_to_service;

    /// This is the asynchronous version of
    /// `gio.SocketClient.connectToService`.
    extern fn g_socket_client_connect_to_service_async(p_client: *SocketClient, p_domain: [*:0]const u8, p_service: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const connectToServiceAsync = g_socket_client_connect_to_service_async;

    /// Finishes an async connect operation. See `gio.SocketClient.connectToServiceAsync`
    extern fn g_socket_client_connect_to_service_finish(p_client: *SocketClient, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.SocketConnection;
    pub const connectToServiceFinish = g_socket_client_connect_to_service_finish;

    /// This is a helper function for `gio.SocketClient.connect`.
    ///
    /// Attempts to create a TCP connection with a network URI.
    ///
    /// `uri` may be any valid URI containing an "authority" (hostname/port)
    /// component. If a port is not specified in the URI, `default_port`
    /// will be used. TLS will be negotiated if `gio.SocketClient.properties.tls` is `TRUE`.
    /// (`gio.SocketClient` does not know to automatically assume TLS for
    /// certain URI schemes.)
    ///
    /// Using this rather than `gio.SocketClient.connect` or
    /// `gio.SocketClient.connectToHost` allows `gio.SocketClient` to
    /// determine when to use application-specific proxy protocols.
    ///
    /// Upon a successful connection, a new `gio.SocketConnection` is constructed
    /// and returned.  The caller owns this new object and must drop their
    /// reference to it when finished with it.
    ///
    /// In the event of any failure (DNS error, service not found, no hosts
    /// connectable) `NULL` is returned and `error` (if non-`NULL`) is set
    /// accordingly.
    extern fn g_socket_client_connect_to_uri(p_client: *SocketClient, p_uri: [*:0]const u8, p_default_port: u16, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.SocketConnection;
    pub const connectToUri = g_socket_client_connect_to_uri;

    /// This is the asynchronous version of `gio.SocketClient.connectToUri`.
    ///
    /// When the operation is finished `callback` will be
    /// called. You can then call `gio.SocketClient.connectToUriFinish` to get
    /// the result of the operation.
    extern fn g_socket_client_connect_to_uri_async(p_client: *SocketClient, p_uri: [*:0]const u8, p_default_port: u16, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const connectToUriAsync = g_socket_client_connect_to_uri_async;

    /// Finishes an async connect operation. See `gio.SocketClient.connectToUriAsync`
    extern fn g_socket_client_connect_to_uri_finish(p_client: *SocketClient, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gio.SocketConnection;
    pub const connectToUriFinish = g_socket_client_connect_to_uri_finish;

    /// Gets the proxy enable state; see `gio.SocketClient.setEnableProxy`
    extern fn g_socket_client_get_enable_proxy(p_client: *SocketClient) c_int;
    pub const getEnableProxy = g_socket_client_get_enable_proxy;

    /// Gets the socket family of the socket client.
    ///
    /// See `gio.SocketClient.setFamily` for details.
    extern fn g_socket_client_get_family(p_client: *SocketClient) gio.SocketFamily;
    pub const getFamily = g_socket_client_get_family;

    /// Gets the local address of the socket client.
    ///
    /// See `gio.SocketClient.setLocalAddress` for details.
    extern fn g_socket_client_get_local_address(p_client: *SocketClient) ?*gio.SocketAddress;
    pub const getLocalAddress = g_socket_client_get_local_address;

    /// Gets the protocol name type of the socket client.
    ///
    /// See `gio.SocketClient.setProtocol` for details.
    extern fn g_socket_client_get_protocol(p_client: *SocketClient) gio.SocketProtocol;
    pub const getProtocol = g_socket_client_get_protocol;

    /// Gets the `gio.ProxyResolver` being used by `client`. Normally, this will
    /// be the resolver returned by `gio.proxyResolverGetDefault`, but you
    /// can override it with `gio.SocketClient.setProxyResolver`.
    extern fn g_socket_client_get_proxy_resolver(p_client: *SocketClient) *gio.ProxyResolver;
    pub const getProxyResolver = g_socket_client_get_proxy_resolver;

    /// Gets the socket type of the socket client.
    ///
    /// See `gio.SocketClient.setSocketType` for details.
    extern fn g_socket_client_get_socket_type(p_client: *SocketClient) gio.SocketType;
    pub const getSocketType = g_socket_client_get_socket_type;

    /// Gets the I/O timeout time for sockets created by `client`.
    ///
    /// See `gio.SocketClient.setTimeout` for details.
    extern fn g_socket_client_get_timeout(p_client: *SocketClient) c_uint;
    pub const getTimeout = g_socket_client_get_timeout;

    /// Gets whether `client` creates TLS connections. See
    /// `gio.SocketClient.setTls` for details.
    extern fn g_socket_client_get_tls(p_client: *SocketClient) c_int;
    pub const getTls = g_socket_client_get_tls;

    /// Gets the TLS validation flags used creating TLS connections via
    /// `client`.
    ///
    /// This function does not work as originally designed and is impossible
    /// to use correctly. See `gio.SocketClient.properties.tls`-validation-flags for more
    /// information.
    extern fn g_socket_client_get_tls_validation_flags(p_client: *SocketClient) gio.TlsCertificateFlags;
    pub const getTlsValidationFlags = g_socket_client_get_tls_validation_flags;

    /// Sets whether or not `client` attempts to make connections via a
    /// proxy server. When enabled (the default), `gio.SocketClient` will use a
    /// `gio.ProxyResolver` to determine if a proxy protocol such as SOCKS is
    /// needed, and automatically do the necessary proxy negotiation.
    ///
    /// See also `gio.SocketClient.setProxyResolver`.
    extern fn g_socket_client_set_enable_proxy(p_client: *SocketClient, p_enable: c_int) void;
    pub const setEnableProxy = g_socket_client_set_enable_proxy;

    /// Sets the socket family of the socket client.
    /// If this is set to something other than `G_SOCKET_FAMILY_INVALID`
    /// then the sockets created by this object will be of the specified
    /// family.
    ///
    /// This might be useful for instance if you want to force the local
    /// connection to be an ipv4 socket, even though the address might
    /// be an ipv6 mapped to ipv4 address.
    extern fn g_socket_client_set_family(p_client: *SocketClient, p_family: gio.SocketFamily) void;
    pub const setFamily = g_socket_client_set_family;

    /// Sets the local address of the socket client.
    /// The sockets created by this object will bound to the
    /// specified address (if not `NULL`) before connecting.
    ///
    /// This is useful if you want to ensure that the local
    /// side of the connection is on a specific port, or on
    /// a specific interface.
    extern fn g_socket_client_set_local_address(p_client: *SocketClient, p_address: ?*gio.SocketAddress) void;
    pub const setLocalAddress = g_socket_client_set_local_address;

    /// Sets the protocol of the socket client.
    /// The sockets created by this object will use of the specified
    /// protocol.
    ///
    /// If `protocol` is `G_SOCKET_PROTOCOL_DEFAULT` that means to use the default
    /// protocol for the socket family and type.
    extern fn g_socket_client_set_protocol(p_client: *SocketClient, p_protocol: gio.SocketProtocol) void;
    pub const setProtocol = g_socket_client_set_protocol;

    /// Overrides the `gio.ProxyResolver` used by `client`. You can call this if
    /// you want to use specific proxies, rather than using the system
    /// default proxy settings.
    ///
    /// Note that whether or not the proxy resolver is actually used
    /// depends on the setting of `gio.SocketClient.properties.enable`-proxy, which is not
    /// changed by this function (but which is `TRUE` by default)
    extern fn g_socket_client_set_proxy_resolver(p_client: *SocketClient, p_proxy_resolver: ?*gio.ProxyResolver) void;
    pub const setProxyResolver = g_socket_client_set_proxy_resolver;

    /// Sets the socket type of the socket client.
    /// The sockets created by this object will be of the specified
    /// type.
    ///
    /// It doesn't make sense to specify a type of `G_SOCKET_TYPE_DATAGRAM`,
    /// as GSocketClient is used for connection oriented services.
    extern fn g_socket_client_set_socket_type(p_client: *SocketClient, p_type: gio.SocketType) void;
    pub const setSocketType = g_socket_client_set_socket_type;

    /// Sets the I/O timeout for sockets created by `client`. `timeout` is a
    /// time in seconds, or 0 for no timeout (the default).
    ///
    /// The timeout value affects the initial connection attempt as well,
    /// so setting this may cause calls to `gio.SocketClient.connect`, etc,
    /// to fail with `G_IO_ERROR_TIMED_OUT`.
    extern fn g_socket_client_set_timeout(p_client: *SocketClient, p_timeout: c_uint) void;
    pub const setTimeout = g_socket_client_set_timeout;

    /// Sets whether `client` creates TLS (aka SSL) connections. If `tls` is
    /// `TRUE`, `client` will wrap its connections in a `gio.TlsClientConnection`
    /// and perform a TLS handshake when connecting.
    ///
    /// Note that since `gio.SocketClient` must return a `gio.SocketConnection`,
    /// but `gio.TlsClientConnection` is not a `gio.SocketConnection`, this
    /// actually wraps the resulting `gio.TlsClientConnection` in a
    /// `gio.TcpWrapperConnection` when returning it. You can use
    /// `gio.TcpWrapperConnection.getBaseIoStream` on the return value
    /// to extract the `gio.TlsClientConnection`.
    ///
    /// If you need to modify the behavior of the TLS handshake (eg, by
    /// setting a client-side certificate to use, or connecting to the
    /// `gio.TlsConnection.signals.accept`-certificate signal), you can connect to
    /// `client`'s `gio.SocketClient.signals.event` signal and wait for it to be
    /// emitted with `G_SOCKET_CLIENT_TLS_HANDSHAKING`, which will give you
    /// a chance to see the `gio.TlsClientConnection` before the handshake
    /// starts.
    extern fn g_socket_client_set_tls(p_client: *SocketClient, p_tls: c_int) void;
    pub const setTls = g_socket_client_set_tls;

    /// Sets the TLS validation flags used when creating TLS connections
    /// via `client`. The default value is `G_TLS_CERTIFICATE_VALIDATE_ALL`.
    ///
    /// This function does not work as originally designed and is impossible
    /// to use correctly. See `gio.SocketClient.properties.tls`-validation-flags for more
    /// information.
    extern fn g_socket_client_set_tls_validation_flags(p_client: *SocketClient, p_flags: gio.TlsCertificateFlags) void;
    pub const setTlsValidationFlags = g_socket_client_set_tls_validation_flags;

    extern fn g_socket_client_get_type() usize;
    pub const getGObjectType = g_socket_client_get_type;

    extern fn g_object_ref(p_self: *gio.SocketClient) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.SocketClient) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SocketClient, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GSocketConnection` is a `gio.IOStream` for a connected socket. They
/// can be created either by `gio.SocketClient` when connecting to a host,
/// or by `gio.SocketListener` when accepting a new client.
///
/// The type of the `GSocketConnection` object returned from these calls
/// depends on the type of the underlying socket that is in use. For
/// instance, for a TCP/IP connection it will be a `gio.TcpConnection`.
///
/// Choosing what type of object to construct is done with the socket
/// connection factory, and it is possible for third parties to register
/// custom socket connection types for specific combination of socket
/// family/type/protocol using `gio.SocketConnection.factoryRegisterType`.
///
/// To close a `GSocketConnection`, use `gio.IOStream.close`. Closing both
/// substreams of the `gio.IOStream` separately will not close the
/// underlying `gio.Socket`.
pub const SocketConnection = extern struct {
    pub const Parent = gio.IOStream;
    pub const Implements = [_]type{};
    pub const Class = gio.SocketConnectionClass;
    f_parent_instance: gio.IOStream,
    f_priv: ?*gio.SocketConnectionPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The underlying `gio.Socket`.
        pub const socket = struct {
            pub const name = "socket";

            pub const Type = ?*gio.Socket;
        };
    };

    pub const signals = struct {};

    /// Looks up the `gobject.Type` to be used when creating socket connections on
    /// sockets with the specified `family`, `type` and `protocol_id`.
    ///
    /// If no type is registered, the `gio.SocketConnection` base type is returned.
    extern fn g_socket_connection_factory_lookup_type(p_family: gio.SocketFamily, p_type: gio.SocketType, p_protocol_id: c_int) usize;
    pub const factoryLookupType = g_socket_connection_factory_lookup_type;

    /// Looks up the `gobject.Type` to be used when creating socket connections on
    /// sockets with the specified `family`, `type` and `protocol`.
    ///
    /// If no type is registered, the `gio.SocketConnection` base type is returned.
    extern fn g_socket_connection_factory_register_type(p_g_type: usize, p_family: gio.SocketFamily, p_type: gio.SocketType, p_protocol: c_int) void;
    pub const factoryRegisterType = g_socket_connection_factory_register_type;

    /// Connect `connection` to the specified remote address.
    extern fn g_socket_connection_connect(p_connection: *SocketConnection, p_address: *gio.SocketAddress, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const connect = g_socket_connection_connect;

    /// Asynchronously connect `connection` to the specified remote address.
    ///
    /// This clears the `gio.Socket.properties.blocking` flag on `connection`'s underlying
    /// socket if it is currently set.
    ///
    /// If `gio.Socket.properties.timeout` is set, the operation will time out and return
    /// `G_IO_ERROR_TIMED_OUT` after that period. Otherwise, it will continue
    /// indefinitely until operating system timeouts (if any) are hit.
    ///
    /// Use `gio.SocketConnection.connectFinish` to retrieve the result.
    extern fn g_socket_connection_connect_async(p_connection: *SocketConnection, p_address: *gio.SocketAddress, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const connectAsync = g_socket_connection_connect_async;

    /// Gets the result of a `gio.SocketConnection.connectAsync` call.
    extern fn g_socket_connection_connect_finish(p_connection: *SocketConnection, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const connectFinish = g_socket_connection_connect_finish;

    /// Try to get the local address of a socket connection.
    extern fn g_socket_connection_get_local_address(p_connection: *SocketConnection, p_error: ?*?*glib.Error) ?*gio.SocketAddress;
    pub const getLocalAddress = g_socket_connection_get_local_address;

    /// Try to get the remote address of a socket connection.
    ///
    /// Since GLib 2.40, when used with `gio.SocketClient.connect` or
    /// `gio.SocketClient.connectAsync`, during emission of
    /// `G_SOCKET_CLIENT_CONNECTING`, this function will return the remote
    /// address that will be used for the connection.  This allows
    /// applications to print e.g. "Connecting to example.com
    /// (10.42.77.3)...".
    extern fn g_socket_connection_get_remote_address(p_connection: *SocketConnection, p_error: ?*?*glib.Error) ?*gio.SocketAddress;
    pub const getRemoteAddress = g_socket_connection_get_remote_address;

    /// Gets the underlying `gio.Socket` object of the connection.
    /// This can be useful if you want to do something unusual on it
    /// not supported by the `gio.SocketConnection` APIs.
    extern fn g_socket_connection_get_socket(p_connection: *SocketConnection) *gio.Socket;
    pub const getSocket = g_socket_connection_get_socket;

    /// Checks if `connection` is connected. This is equivalent to calling
    /// `gio.Socket.isConnected` on `connection`'s underlying `gio.Socket`.
    extern fn g_socket_connection_is_connected(p_connection: *SocketConnection) c_int;
    pub const isConnected = g_socket_connection_is_connected;

    extern fn g_socket_connection_get_type() usize;
    pub const getGObjectType = g_socket_connection_get_type;

    extern fn g_object_ref(p_self: *gio.SocketConnection) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.SocketConnection) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SocketConnection, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `GSocketControlMessage` is a special-purpose utility message that
/// can be sent to or received from a `gio.Socket`. These types of
/// messages are often called ‘ancillary data’.
///
/// The message can represent some sort of special instruction to or
/// information from the socket or can represent a special kind of
/// transfer to the peer (for example, sending a file descriptor over
/// a UNIX socket).
///
/// These messages are sent with `gio.Socket.sendMessage` and received
/// with `gio.Socket.receiveMessage`.
///
/// To extend the set of control message that can be sent, subclass this
/// class and override the `get_size`, `get_level`, `get_type` and `serialize`
/// methods.
///
/// To extend the set of control messages that can be received, subclass
/// this class and implement the `deserialize` method. Also, make sure your
/// class is registered with the `gobject.Type` type system before calling
/// `gio.Socket.receiveMessage` to read such a message.
pub const SocketControlMessage = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.SocketControlMessageClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.SocketControlMessagePrivate,

    pub const virtual_methods = struct {
        /// Returns the "level" (i.e. the originating protocol) of the control message.
        /// This is often SOL_SOCKET.
        pub const get_level = struct {
            pub fn call(p_class: anytype, p_message: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) c_int {
                return gobject.ext.as(SocketControlMessage.Class, p_class).f_get_level.?(gobject.ext.as(SocketControlMessage, p_message));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_message: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) c_int) void {
                gobject.ext.as(SocketControlMessage.Class, p_class).f_get_level = @ptrCast(p_implementation);
            }
        };

        /// Returns the space required for the control message, not including
        /// headers or alignment.
        pub const get_size = struct {
            pub fn call(p_class: anytype, p_message: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) usize {
                return gobject.ext.as(SocketControlMessage.Class, p_class).f_get_size.?(gobject.ext.as(SocketControlMessage, p_message));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_message: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) usize) void {
                gobject.ext.as(SocketControlMessage.Class, p_class).f_get_size = @ptrCast(p_implementation);
            }
        };

        /// gets the protocol specific type of the message.
        pub const get_type = struct {
            pub fn call(p_class: anytype, p_message: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) c_int {
                return gobject.ext.as(SocketControlMessage.Class, p_class).f_get_type.?(gobject.ext.as(SocketControlMessage, p_message));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_message: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) c_int) void {
                gobject.ext.as(SocketControlMessage.Class, p_class).f_get_type = @ptrCast(p_implementation);
            }
        };

        /// Converts the data in the message to bytes placed in the
        /// message.
        ///
        /// `data` is guaranteed to have enough space to fit the size
        /// returned by `gio.SocketControlMessage.getSize` on this
        /// object.
        pub const serialize = struct {
            pub fn call(p_class: anytype, p_message: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_data: *anyopaque) void {
                return gobject.ext.as(SocketControlMessage.Class, p_class).f_serialize.?(gobject.ext.as(SocketControlMessage, p_message), p_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_message: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_data: *anyopaque) callconv(.C) void) void {
                gobject.ext.as(SocketControlMessage.Class, p_class).f_serialize = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {};

    /// Tries to deserialize a socket control message of a given
    /// `level` and `type`. This will ask all known (to GType) subclasses
    /// of `gio.SocketControlMessage` if they can understand this kind
    /// of message and if so deserialize it into a `gio.SocketControlMessage`.
    ///
    /// If there is no implementation for this kind of control message, `NULL`
    /// will be returned.
    extern fn g_socket_control_message_deserialize(p_level: c_int, p_type: c_int, p_size: usize, p_data: [*]u8) ?*gio.SocketControlMessage;
    pub const deserialize = g_socket_control_message_deserialize;

    /// Returns the "level" (i.e. the originating protocol) of the control message.
    /// This is often SOL_SOCKET.
    extern fn g_socket_control_message_get_level(p_message: *SocketControlMessage) c_int;
    pub const getLevel = g_socket_control_message_get_level;

    /// Returns the protocol specific type of the control message.
    /// For instance, for UNIX fd passing this would be SCM_RIGHTS.
    extern fn g_socket_control_message_get_msg_type(p_message: *SocketControlMessage) c_int;
    pub const getMsgType = g_socket_control_message_get_msg_type;

    /// Returns the space required for the control message, not including
    /// headers or alignment.
    extern fn g_socket_control_message_get_size(p_message: *SocketControlMessage) usize;
    pub const getSize = g_socket_control_message_get_size;

    /// Converts the data in the message to bytes placed in the
    /// message.
    ///
    /// `data` is guaranteed to have enough space to fit the size
    /// returned by `gio.SocketControlMessage.getSize` on this
    /// object.
    extern fn g_socket_control_message_serialize(p_message: *SocketControlMessage, p_data: *anyopaque) void;
    pub const serialize = g_socket_control_message_serialize;

    extern fn g_socket_control_message_get_type() usize;
    pub const getGObjectType = g_socket_control_message_get_type;

    extern fn g_object_ref(p_self: *gio.SocketControlMessage) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.SocketControlMessage) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SocketControlMessage, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `GSocketListener` is an object that keeps track of a set
/// of server sockets and helps you accept sockets from any of the
/// socket, either sync or async.
///
/// Add addresses and ports to listen on using
/// `gio.SocketListener.addAddress` and
/// `gio.SocketListener.addInetPort`. These will be listened on until
/// `gio.SocketListener.close` is called. Dropping your final reference to
/// the `GSocketListener` will not cause `gio.SocketListener.close` to be
/// called implicitly, as some references to the `GSocketListener` may be held
/// internally.
///
/// If you want to implement a network server, also look at
/// `gio.SocketService` and `gio.ThreadedSocketService` which are
/// subclasses of `GSocketListener` that make this even easier.
pub const SocketListener = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gio.SocketListenerClass;
    f_parent_instance: gobject.Object,
    f_priv: ?*gio.SocketListenerPrivate,

    pub const virtual_methods = struct {
        /// virtual method called when the set of socket listened to changes
        pub const changed = struct {
            pub fn call(p_class: anytype, p_listener: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(SocketListener.Class, p_class).f_changed.?(gobject.ext.as(SocketListener, p_listener));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_listener: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(SocketListener.Class, p_class).f_changed = @ptrCast(p_implementation);
            }
        };

        pub const event = struct {
            pub fn call(p_class: anytype, p_listener: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_event: gio.SocketListenerEvent, p_socket: *gio.Socket) void {
                return gobject.ext.as(SocketListener.Class, p_class).f_event.?(gobject.ext.as(SocketListener, p_listener), p_event, p_socket);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_listener: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_event: gio.SocketListenerEvent, p_socket: *gio.Socket) callconv(.C) void) void {
                gobject.ext.as(SocketListener.Class, p_class).f_event = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The number of outstanding connections in the listen queue.
        pub const listen_backlog = struct {
            pub const name = "listen-backlog";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// Emitted when `listener`'s activity on `socket` changes state.
        /// Note that when `listener` is used to listen on both IPv4 and
        /// IPv6, a separate set of signals will be emitted for each, and
        /// the order they happen in is undefined.
        pub const event = struct {
            pub const name = "event";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_event: gio.SocketListenerEvent, p_socket: *gio.Socket, P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(SocketListener, p_instance))),
                    gobject.signalLookup("event", SocketListener.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gio.SocketListener` with no sockets to listen for.
    /// New listeners can be added with e.g. `gio.SocketListener.addAddress`
    /// or `gio.SocketListener.addInetPort`.
    extern fn g_socket_listener_new() *gio.SocketListener;
    pub const new = g_socket_listener_new;

    /// Blocks waiting for a client to connect to any of the sockets added
    /// to the listener. Returns a `gio.SocketConnection` for the socket that was
    /// accepted.
    ///
    /// If `source_object` is not `NULL` it will be filled out with the source
    /// object specified when the corresponding socket or address was added
    /// to the listener.
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned.
    extern fn g_socket_listener_accept(p_listener: *SocketListener, p_source_object: ?**gobject.Object, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.SocketConnection;
    pub const accept = g_socket_listener_accept;

    /// This is the asynchronous version of `gio.SocketListener.accept`.
    ///
    /// When the operation is finished `callback` will be
    /// called. You can then call `gio.SocketListener.acceptFinish`
    /// to get the result of the operation.
    extern fn g_socket_listener_accept_async(p_listener: *SocketListener, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const acceptAsync = g_socket_listener_accept_async;

    /// Finishes an async accept operation. See `gio.SocketListener.acceptAsync`
    extern fn g_socket_listener_accept_finish(p_listener: *SocketListener, p_result: *gio.AsyncResult, p_source_object: ?**gobject.Object, p_error: ?*?*glib.Error) ?*gio.SocketConnection;
    pub const acceptFinish = g_socket_listener_accept_finish;

    /// Blocks waiting for a client to connect to any of the sockets added
    /// to the listener. Returns the `gio.Socket` that was accepted.
    ///
    /// If you want to accept the high-level `gio.SocketConnection`, not a `gio.Socket`,
    /// which is often the case, then you should use `gio.SocketListener.accept`
    /// instead.
    ///
    /// If `source_object` is not `NULL` it will be filled out with the source
    /// object specified when the corresponding socket or address was added
    /// to the listener.
    ///
    /// If `cancellable` is not `NULL`, then the operation can be cancelled by
    /// triggering the cancellable object from another thread. If the operation
    /// was cancelled, the error `G_IO_ERROR_CANCELLED` will be returned.
    extern fn g_socket_listener_accept_socket(p_listener: *SocketListener, p_source_object: ?**gobject.Object, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gio.Socket;
    pub const acceptSocket = g_socket_listener_accept_socket;

    /// This is the asynchronous version of `gio.SocketListener.acceptSocket`.
    ///
    /// When the operation is finished `callback` will be
    /// called. You can then call `gio.SocketListener.acceptSocketFinish`
    /// to get the result of the operation.
    extern fn g_socket_listener_accept_socket_async(p_listener: *SocketListener, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const acceptSocketAsync = g_socket_listener_accept_socket_async;

    /// Finishes an async accept operation. See `gio.SocketListener.acceptSocketAsync`
    extern fn g_socket_listener_accept_socket_finish(p_listener: *SocketListener, p_result: *gio.AsyncResult, p_source_object: ?**gobject.Object, p_error: ?*?*glib.Error) ?*gio.Socket;
    pub const acceptSocketFinish = g_socket_listener_accept_socket_finish;

    /// Creates a socket of type `type` and protocol `protocol`, binds
    /// it to `address` and adds it to the set of sockets we're accepting
    /// sockets from.
    ///
    /// Note that adding an IPv6 address, depending on the platform,
    /// may or may not result in a listener that also accepts IPv4
    /// connections.  For more deterministic behavior, see
    /// `gio.SocketListener.addInetPort`.
    ///
    /// `source_object` will be passed out in the various calls
    /// to accept to identify this particular source, which is
    /// useful if you're listening on multiple addresses and do
    /// different things depending on what address is connected to.
    ///
    /// If successful and `effective_address` is non-`NULL` then it will
    /// be set to the address that the binding actually occurred at.  This
    /// is helpful for determining the port number that was used for when
    /// requesting a binding to port 0 (ie: "any port").  This address, if
    /// requested, belongs to the caller and must be freed.
    ///
    /// Call `gio.SocketListener.close` to stop listening on `address`; this will not
    /// be done automatically when you drop your final reference to `listener`, as
    /// references may be held internally.
    extern fn g_socket_listener_add_address(p_listener: *SocketListener, p_address: *gio.SocketAddress, p_type: gio.SocketType, p_protocol: gio.SocketProtocol, p_source_object: ?*gobject.Object, p_effective_address: ?**gio.SocketAddress, p_error: ?*?*glib.Error) c_int;
    pub const addAddress = g_socket_listener_add_address;

    /// Listens for TCP connections on any available port number for both
    /// IPv6 and IPv4 (if each is available).
    ///
    /// This is useful if you need to have a socket for incoming connections
    /// but don't care about the specific port number.
    ///
    /// `source_object` will be passed out in the various calls
    /// to accept to identify this particular source, which is
    /// useful if you're listening on multiple addresses and do
    /// different things depending on what address is connected to.
    extern fn g_socket_listener_add_any_inet_port(p_listener: *SocketListener, p_source_object: ?*gobject.Object, p_error: ?*?*glib.Error) u16;
    pub const addAnyInetPort = g_socket_listener_add_any_inet_port;

    /// Helper function for `gio.SocketListener.addAddress` that
    /// creates a TCP/IP socket listening on IPv4 and IPv6 (if
    /// supported) on the specified port on all interfaces.
    ///
    /// `source_object` will be passed out in the various calls
    /// to accept to identify this particular source, which is
    /// useful if you're listening on multiple addresses and do
    /// different things depending on what address is connected to.
    ///
    /// Call `gio.SocketListener.close` to stop listening on `port`; this will not
    /// be done automatically when you drop your final reference to `listener`, as
    /// references may be held internally.
    extern fn g_socket_listener_add_inet_port(p_listener: *SocketListener, p_port: u16, p_source_object: ?*gobject.Object, p_error: ?*?*glib.Error) c_int;
    pub const addInetPort = g_socket_listener_add_inet_port;

    /// Adds `socket` to the set of sockets that we try to accept
    /// new clients from. The socket must be bound to a local
    /// address and listened to.
    ///
    /// `source_object` will be passed out in the various calls
    /// to accept to identify this particular source, which is
    /// useful if you're listening on multiple addresses and do
    /// different things depending on what address is connected to.
    ///
    /// The `socket` will not be automatically closed when the `listener` is finalized
    /// unless the listener held the final reference to the socket. Before GLib 2.42,
    /// the `socket` was automatically closed on finalization of the `listener`, even
    /// if references to it were held elsewhere.
    extern fn g_socket_listener_add_socket(p_listener: *SocketListener, p_socket: *gio.Socket, p_source_object: ?*gobject.Object, p_error: ?*?*glib.Error) c_int;
    pub const addSocket = g_socket_listener_add_socket;

    /// Closes all the sockets in the listener.
    extern fn g_socket_listener_close(p_listener: *SocketListener) void;
    pub const close = g_socket_listener_close;

    /// Sets the listen backlog on the sockets in the listener. This must be called
    /// before adding any sockets, addresses or ports to the `gio.SocketListener` (for
    /// example, by calling `gio.SocketListener.addInetPort`) to be effective.
    ///
    /// See `gio.Socket.setListenBacklog` for details
    extern fn g_socket_listener_set_backlog(p_listener: *SocketListener, p_listen_backlog: c_int) void;
    pub const setBacklog = g_socket_listener_set_backlog;

    extern fn g_socket_listener_get_type() usize;
    pub const getGObjectType = g_socket_listener_get_type;

    extern fn g_object_ref(p_self: *gio.SocketListener) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.SocketListener) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SocketListener, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `GSocketService` is an object that represents a service that
/// is provided to the network or over local sockets.  When a new
/// connection is made to the service the `gio.SocketService.signals.incoming`
/// signal is emitted.
///
/// A `GSocketService` is a subclass of `gio.SocketListener` and you need
/// to add the addresses you want to accept connections on with the
/// `gio.SocketListener` APIs.
///
/// There are two options for implementing a network service based on
/// `GSocketService`. The first is to create the service using
/// `gio.SocketService.new` and to connect to the
/// `gio.SocketService.signals.incoming` signal. The second is to subclass
/// `GSocketService` and override the default signal handler implementation.
///
/// In either case, the handler must immediately return, or else it
/// will block additional incoming connections from being serviced.
/// If you are interested in writing connection handlers that contain
/// blocking code then see `gio.ThreadedSocketService`.
///
/// The socket service runs on the main loop of the
/// thread-default context (see
/// `glib.MainContext.pushThreadDefault`) of the thread it is
/// created in, and is not threadsafe in general. However, the calls to start and
/// stop the service are thread-safe so these can be used from threads that
/// handle incoming clients.
pub const SocketService = extern struct {
    pub const Parent = gio.SocketListener;
    pub const Implements = [_]type{};
    pub const Class = gio.SocketServiceClass;
    f_parent_instance: gio.SocketListener,
    f_priv: ?*gio.SocketServicePrivate,

    pub const virtual_methods = struct {
        /// signal emitted when new connections are accepted
        pub const incoming = struct {
            pub fn call(p_class: anytype, p_service: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_connection: *gio.SocketConnection, p_source_object: *gobject.Object) c_int {
                return gobject.ext.as(SocketService.Class, p_class).f_incoming.?(gobject.ext.as(SocketService, p_service), p_connection, p_source_object);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_service: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_connection: *gio.SocketConnection, p_source_object: *gobject.Object) callconv(.C) c_int) void {
                gobject.ext.as(SocketService.Class, p_class).f_incoming = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// Whether the service is currently accepting connections.
        pub const active = struct {
            pub const name = "active";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// The ::incoming signal is emitted when a new incoming connection
        /// to `service` needs to be handled. The handler must initiate the
        /// handling of `connection`, but may not block; in essence,
        /// asynchronous operations must be used.
        ///
        /// `connection` will be unreffed once the signal handler returns,
        /// so you need to ref it yourself if you are planning to use it.
        pub const incoming = struct {
            pub const name = "incoming";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_connection: *gio.SocketConnection, p_source_object: ?*gobject.Object, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(SocketService, p_instance))),
                    gobject.signalLookup("incoming", SocketService.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Creates a new `gio.SocketService` with no sockets to listen for.
    /// New listeners can be added with e.g. `gio.SocketListener.addAddress`
    /// or `gio.SocketListener.addInetPort`.
    ///
    /// New services are created active, there is no need to call
    /// `gio.SocketService.start`, unless `gio.SocketService.stop` has been
    /// called before.
    extern fn g_socket_service_new() *gio.SocketService;
    pub const new = g_socket_service_new;

    /// Check whether the service is active or not. An active
    /// service will accept new clients that connect, while
    /// a non-active service will let connecting clients queue
    /// up until the service is started.
    extern fn g_socket_service_is_active(p_service: *SocketService) c_int;
    pub const isActive = g_socket_service_is_active;

    /// Restarts the service, i.e. start accepting connections
    /// from the added sockets when the mainloop runs. This only needs
    /// to be called after the service has been stopped from
    /// `gio.SocketService.stop`.
    ///
    /// This call is thread-safe, so it may be called from a thread
    /// handling an incoming client request.
    extern fn g_socket_service_start(p_service: *SocketService) void;
    pub const start = g_socket_service_start;

    /// Stops the service, i.e. stops accepting connections
    /// from the added sockets when the mainloop runs.
    ///
    /// This call is thread-safe, so it may be called from a thread
    /// handling an incoming client request.
    ///
    /// Note that this only stops accepting new connections; it does not
    /// close the listening sockets, and you can call
    /// `gio.SocketService.start` again later to begin listening again. To
    /// close the listening sockets, call `gio.SocketListener.close`. (This
    /// will happen automatically when the `gio.SocketService` is finalized.)
    ///
    /// This must be called before calling `gio.SocketListener.close` as
    /// the socket service will start accepting connections immediately
    /// when a new socket is added.
    extern fn g_socket_service_stop(p_service: *SocketService) void;
    pub const stop = g_socket_service_stop;

    extern fn g_socket_service_get_type() usize;
    pub const getGObjectType = g_socket_service_get_type;

    extern fn g_object_ref(p_self: *gio.SocketService) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gio.SocketService) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SocketService, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// `GSubprocess` allows the creation of and interaction with child
/// processes.
///
/// Processes can be communicated with using standard GIO-style APIs (ie:
/// `gio.InputStream`, `gio.OutputStream`). There are GIO-style APIs
/// to wait for process termination (ie: cancellable and with an asynchronous
/// variant).
///
/// There is an API to force a process to terminate, as well as a
/// race-free API for sending UNIX signals to a subprocess.
///
/// One major advantage that GIO brings over the core GLib library is
/// comprehensive API for asynchronous I/O, such
/// `gio.OutputStream.spliceAsync`.  This makes `GSubprocess`
/// significantly more powerful and flexible than equivalent APIs in
/// some other languages such as the `subprocess.py`
/// included with Python.  For example, using `GSubprocess` one could
/// create two child processes, reading standard output from the first,
/// processing it, and writing to the input stream of the second, all
/// without blocking the main loop.
///
/// A powerful `gio.Subprocess.communicate` API is provided similar to the
/// ``communicate`` method of `subprocess.py`. This enables very easy
/// interaction with a subprocess that has been opened with pipes.
///
/// `GSubprocess` defaults to tight control over the file descriptors open
/// in the child process, avoiding dangling-FD issues that are caused by
/// a simple ``fork``/``exec``.  The only open file descriptors in the
/// spawned process are ones that were explicitly specified by the
/// `GSubprocess` API (unless `G_SUBPROCESS_FLAGS_INHERIT_FDS` was
/// specified).
///
/// `GSubprocess` will quickly reap all child processes as they exit,
/// avoiding ‘zombie processes’ remaining around for long periods of
/// time.  `gio.Subprocess.wait` can be used to wait for this to happen,
/// but it will happen even without the call being explicitly made.
///
/// As a matter of principle, `GSubprocess` has no API that accepts
/// shell-style space-separated strings.  It will, however, match the
/// typical shell behaviour of searching the `PATH` for executables that do
/// not contain a directory separator in their name. By default, the `PATH`
/// of the current process is used.  You can specify
/// `G_SUBPROCESS_FLAGS_SEARCH_PATH_FROM_ENVP` to use the `PATH` of the
/// launcher environment instead.
///
/// `GSubprocess` attempts to have a very simple API for most uses (ie:
/// spawning a subprocess with arguments and support for most typical
/// kinds of input and output redirection).  See `gio.Subprocess.new`. The
/// `gio.SubprocessLauncher` API is provided for more complicated cases
/// (advanced types of redirection, environment variable manipulation,
/// change of working directory, child setup functions, etc).
///
/// A typical use of `GSubprocess` will involve calling
/// `gio.Subprocess.new`, followed by `gio.Subprocess.waitAsync` or
/// `gio.Subprocess.wait`.  After the process exits, the status can be
/// checked using functions such as `gio.Subprocess.getIfExited` (which
/// are similar to the familiar `WIFEXITED`-style POSIX macros).
///
/// Note that as of GLib 2.82, creating a `GSubprocess` causes the signal
/// `SIGPIPE` to be ignored for the remainder of the program. If you are writing
/// a command-line utility that uses `GSubprocess`, you may need to take into
/// account the fact that your program will not automatically be killed
/// if it tries to write to `stdout` after it has been closed.
pub const Subprocess = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gio.Initable};
    pub const Class = opaque {
        pub const Instance = Subprocess;
    };
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Argument vector.
        pub const argv = struct {
            pub const name = "argv";

            pub const Type = ?[*][*:0]u8;
        };

        /// Subprocess flags.
        pub const flags = struct {
            pub const name = "flags";

            pub const Type = gio.SubprocessFlags;
        };
    };

    pub const signals = struct {};

    /// Create a new process with the given flags and varargs argument
    /// list.  By default, matching the `glib.spawnAsync` defaults, the
    /// child's stdin will be set to the system null device, and
    /// stdout/stderr will be inherited from the parent.  You can use
    /// `flags` to control this behavior.
    ///
    /// The argument list must be terminated with `NULL`.
    extern fn g_subprocess_new(p_flags: gio.SubprocessFlags, p_error: ?**glib.Error, p_argv0: [*:0]const u8, ...) *gio.Subprocess;
    pub const new = g_subprocess_new;

    /// Create a new process with the given flags and argument list.
    ///
    /// The argument list is expected to be `NULL`-terminated.
    extern fn g_subprocess_newv(p_argv: [*]const [*:0]const u8, p_flags: gio.SubprocessFlags, p_error: ?*?*glib.Error) ?*gio.Subprocess;
    pub const newv = g_subprocess_newv;

    /// Communicate with the subprocess until it terminates, and all input
    /// and output has been completed.
    ///
    /// If `stdin_buf` is given, the subprocess must have been created with
    /// `G_SUBPROCESS_FLAGS_STDIN_PIPE`.  The given data is fed to the
    /// stdin of the subprocess and the pipe is closed (ie: EOF).
    ///
    /// At the same time (as not to cause blocking when dealing with large
    /// amounts of data), if `G_SUBPROCESS_FLAGS_STDOUT_PIPE` or
    /// `G_SUBPROCESS_FLAGS_STDERR_PIPE` were used, reads from those
    /// streams.  The data that was read is returned in `stdout` and/or
    /// the `stderr`.
    ///
    /// If the subprocess was created with `G_SUBPROCESS_FLAGS_STDOUT_PIPE`,
    /// `stdout_buf` will contain the data read from stdout.  Otherwise, for
    /// subprocesses not created with `G_SUBPROCESS_FLAGS_STDOUT_PIPE`,
    /// `stdout_buf` will be set to `NULL`.  Similar provisions apply to
    /// `stderr_buf` and `G_SUBPROCESS_FLAGS_STDERR_PIPE`.
    ///
    /// As usual, any output variable may be given as `NULL` to ignore it.
    ///
    /// If you desire the stdout and stderr data to be interleaved, create
    /// the subprocess with `G_SUBPROCESS_FLAGS_STDOUT_PIPE` and
    /// `G_SUBPROCESS_FLAGS_STDERR_MERGE`.  The merged result will be returned
    /// in `stdout_buf` and `stderr_buf` will be set to `NULL`.
    ///
    /// In case of any error (including cancellation), `FALSE` will be
    /// returned with `error` set.  Some or all of the stdin data may have
    /// been written.  Any stdout or stderr data that has been read will be
    /// discarded. None of the out variables (aside from `error`) will have
    /// been set to anything in particular and should not be inspected.
    ///
    /// In the case that `TRUE` is returned, the subprocess has exited and the
    /// exit status inspection APIs (eg: `gio.Subprocess.getIfExited`,
    /// `gio.Subprocess.getExitStatus`) may be used.
    ///
    /// You should not attempt to use any of the subprocess pipes after
    /// starting this function, since they may be left in strange states,
    /// even if the operation was cancelled.  You should especially not
    /// attempt to interact with the pipes while the operation is in progress
    /// (either from another thread or if using the asynchronous version).
    extern fn g_subprocess_communicate(p_subprocess: *Subprocess, p_stdin_buf: ?*glib.Bytes, p_cancellable: ?*gio.Cancellable, p_stdout_buf: ?**glib.Bytes, p_stderr_buf: ?**glib.Bytes, p_error: ?*?*glib.Error) c_int;
    pub const communicate = g_subprocess_communicate;

    ///