/*
 * Copyright (c) 2010-2011, Jan Friesse <honzaf@users.sourceforge.net>
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#define __STDC_LIMIT_MACROS
#define __STDC_FORMAT_MACROS

#include <sys/types.h>

#include <sys/socket.h>

#include <arpa/inet.h>
#include <netinet/in.h>

#include <inttypes.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "opt.h"

/*
 * Defines
 */

/*
 * Used in opt decode func array to donate fact, that this options is decoded
 * directly inside opt_decode function
 */
#define OPT_DECODE_DIRECT	NULL

/*
 * Typedefs
 */
typedef void (*opt_decode_func)(const char *str, const char *msg_p,
    uint8_t opt_len);

/*
 * Structs
 */
struct opt_decode_struct {
	opt_decode_func func;
	const char *desc;
};

/*
 * Prototypes
 */
static void	opt_decode_bfso(const char *str, const char *msg_p,
    uint8_t opt_len);

static void	opt_decode_bool(const char *str, const char *msg_p,
    uint8_t opt_len);

static void	opt_decode_class_i(const char *str, const char *msg_p,
    uint8_t opt_len);

static void	opt_decode_dhcp_mt(const char *str, const char *msg_p,
    uint8_t opt_len, enum opt_msg_type *msg_type, int verbose);

static void	opt_decode_hex(const char *str, const char *msg_p,
    uint8_t opt_len);

static void	opt_decode_ip(const char *str, const char *msg_p,
    uint8_t opt_len);

static void	opt_decode_ip_pairs(const char *str, const char *msg_p,
    uint8_t opt_len, const char *separator);

static void	opt_decode_ips(const char *str, const char *msg_p,
    uint8_t opt_len);

static void	opt_decode_nbios_nto(const char *str, const char *msg_p,
    uint8_t opt_len);

static void	opt_decode_opt_overload(const char *str, const char *msg_p,
    uint8_t opt_len, enum opt_overload *overload, int verbose);

static void	opt_decode_pfo(const char *str, const char *msg_p,
    uint8_t opt_len);

static void	opt_decode_pname(const char *str, const char *msg_p,
    uint8_t opt_len);

static void	opt_decode_prl(const char *str, const char *msg_p,
    uint8_t opt_len);

static void	opt_decode_sro(const char *str, const char *msg_p,
    uint8_t opt_len);

static void	opt_decode_s32(const char *str, const char *msg_p,
    uint8_t opt_len);

static void	opt_decode_str(const char *str, const char *msg_p,
    uint8_t opt_len);

static void	opt_decode_u16(const char *str, const char *msg_p,
    uint8_t opt_len);

static void	opt_decode_u16s(const char *str, const char *msg_p,
    uint8_t opt_len);

static void	opt_decode_u32(const char *str, const char *msg_p,
    uint8_t opt_len);

static void	opt_decode_u8(const char *str, const char *msg_p,
    uint8_t opt_len);

/*
 * Decode table.
 * Table length is enough to handle all possible 1-byte (8-bit) values. We are
 * also using feature of compiler to fill static values by zero.
 */
static struct opt_decode_struct opt_decode_table[UINT8_MAX + 1] = {
	{opt_decode_pname, "Pad Option"},			/*  0 */
	{opt_decode_ip, "Subnet Mask"},				/*  1 */
	{opt_decode_s32, "Time Offset"},			/*  2 */
	{opt_decode_ips, "Router Option"},			/*  3 */
	{opt_decode_ips, "Time Server Option"},			/*  4 */
	{opt_decode_ips, "Name Server Option"},			/*  5 */
	{opt_decode_ips, "Domain Name Server Option"},		/*  6 */
	{opt_decode_ips, "Log Server Option"},			/*  7 */
	{opt_decode_ips, "Cookie Server Option"},		/*  8 */
	{opt_decode_ips, "LPR Server Option"},			/*  9 */
	{opt_decode_ips, "Impress Server Option"},		/* 10 */
	{opt_decode_ips, "Resource Location Server Option"},	/* 11 */
	{opt_decode_str, "Host Name Option"},			/* 12 */
	{opt_decode_bfso, "Boot File Size Option"},		/* 13 */
	{opt_decode_str, "Merit Dump File"},			/* 14 */
	{opt_decode_str, "Domain Name"},			/* 15 */
	{opt_decode_ip, "Swap Server"},				/* 16 */
	{opt_decode_str, "Root Path"},				/* 17 */
	{opt_decode_str, "Extensions Path"},			/* 18 */
	{opt_decode_bool, "IP Forwarding E/D Option"},		/* 19 */
	{opt_decode_bool, "Non-Local Source Routing E/D Opt"},	/* 20 */
	{opt_decode_pfo, "Policy Filter Option"},		/* 21 */
	{opt_decode_u16, "Maximum Datagram Reassembly Size"},	/* 22 */
	{opt_decode_u8, "Default IP Time-to-live"},		/* 23 */
	{opt_decode_u32, "Path MTU Aging Timeout Option"},	/* 24 */
	{opt_decode_u16s, "Path MTU Plateau Table Option"},	/* 25 */
	{opt_decode_u16, "Interface MTU Option"},		/* 26 */
	{opt_decode_bool, "All Subnets are Local Option"},	/* 27 */
	{opt_decode_ip, "Broadcast Address Option"},		/* 28 */
	{opt_decode_bool, "Perform Mask Discovery Option"},	/* 29 */
	{opt_decode_bool, "Mask Supplier Option"},		/* 30 */
	{opt_decode_bool, "Perform Router Discovery Option"},	/* 31 */
	{opt_decode_ip, "Router Solicitation Address Option"},	/* 32 */
	{opt_decode_sro, "Static Route Option"},		/* 33 */
	{opt_decode_bool, "Trailer Encapsulation Option"},	/* 34 */
	{opt_decode_u32, "ARP Cache Timeout Option"},		/* 35 */
	{opt_decode_bool, "Ethernet Encapsulation Option"},	/* 36 */
	{opt_decode_u8, "TCP Default TTL Option"},		/* 37 */
	{opt_decode_u32, "TCP Keepalive Interval Option"},	/* 38 */
	{opt_decode_bool,"TCP Keepalive Garbage Option"},	/* 39 */
	{opt_decode_str,
	    "Network Information Service Domain Option"},	/* 40 */
	{opt_decode_ips, "Network Information Servers Option"},	/* 41 */
	{opt_decode_ips,
	    "Network Time Protocol Servers Option"},		/* 42 */
	{opt_decode_pname, "Vendor Specific Information"},	/* 43 */
	{opt_decode_ips,
	    "NetBIOS over TCP/IP Name Server Option"},		/* 44 */
	{opt_decode_ips,
	    "NetBIOS over TCP/IP Datagram Distribution Server "
	    "Option"},						/* 45 */
	{opt_decode_nbios_nto,
		"NetBIOS over TCP/IP Node Type Option"},	/* 46 */
	{opt_decode_str, "NetBIOS over TCP/IP Scope Option"},	/* 47 */
	{opt_decode_ips, "X Window System Font Server Option"},	/* 48 */
	{opt_decode_ips,
	    "X Window System Display Manager Option"},		/* 49 */
	{opt_decode_ip, "Requested IP Address"},		/* 50 */
	{opt_decode_u32, "IP Address Lease Time"},		/* 51 */
	{OPT_DECODE_DIRECT, "Option Overload"},			/* 52 */
	{OPT_DECODE_DIRECT, "DHCP Message Type"},		/* 53 */
	{opt_decode_ip, "Server Identifier"},			/* 54 */
	{opt_decode_prl, "Parameter Request List"},		/* 55 */
	{opt_decode_str, "Message"},				/* 56 */
	{opt_decode_u16, "Maximum DHCP Message Size"},		/* 57 */
	{opt_decode_u32, "Renewal (T1) Time Value"},		/* 58 */
	{opt_decode_u32, "Rebinding (T2) Time Value"},		/* 59 */
	{opt_decode_hex, "Class-identifier"},			/* 60 */
	{opt_decode_class_i, "Client-identifier"},		/* 61 */
};

/*
 * Decode option.
 * opt is type of option, opt_len is length of option, msg_p is pointer to
 * option data. end_reached is filled if end_option (with code 255) is found
 * and verbose is boolean flag indicating if verbose packet printing should be
 * processed.
 */
void
opt_decode(uint8_t opt, uint8_t opt_len, const char *msg_p, int *end_reached,
    enum opt_msg_type *msg_type, enum opt_overload *overload, int verbose)
{
	struct opt_decode_struct opt_decode_s;

	/*
	 * Process options which needs to be processed independently on verbose
	 * flag
	 */
	switch (opt) {
	case 52:
		opt_decode_opt_overload(opt_decode_table[opt].desc, msg_p,
		    opt_len, overload, verbose);

		return ;
		break; /* NOTREACHED */
	case 53:
		opt_decode_dhcp_mt(opt_decode_table[opt].desc, msg_p, opt_len,
		    msg_type, verbose);

		return ;
		break; /* NOTREACHED */
	case 255:
		if (verbose)
			printf("End Option");
		*end_reached = 1;

		return ;
		break; /* NOTREACHED */
	}

	if (!verbose || *end_reached)
		return ;

	/*
	 * Here print options which are dependant on verbose flag
	 */

	/*
	 * Look to table
	 */
	opt_decode_s = opt_decode_table[opt];
	if (opt_decode_s.func != NULL) {
		opt_decode_s.func(opt_decode_s.desc, msg_p, opt_len);

		return ;
	} else {
		printf("Private");
	}
}

/*
 * Decode boot file size option.
 * str is description of option, msg_p is buffer with option data and opt_len
 * is length of option.
 */
static void
opt_decode_bfso(const char *str, const char *msg_p, uint8_t opt_len)
{
	uint16_t u16;

	printf("%s - ", str);

	if (opt_len == 2) {
		memcpy(&u16, msg_p, sizeof(u16));
		u16 = ntohs(u16);
		printf("%"PRIu16" 512-octet block(s)", u16);
	} else {
		printf("invalid");
	}
}

/*
 * Decode boolean like option.
 * str is description of option, msg_p is buffer with option data and opt_len
 * is length of option. opt_len must be 1. *msg_p value 0 is decoded as
 * disable, value 1 as enable. Other values are decoded as invalid.
 */
static void
opt_decode_bool(const char *str, const char *msg_p, uint8_t opt_len)
{
	char enable;

	printf("%s - ", str);

	if (opt_len != 1) {
		printf("invalid");
	} else {
		enable = *msg_p;

		switch (enable) {
		case 0:
			printf("disable");
			break;
		case 1:
			printf("enable");
			break;
		default:
			printf("invalid (0x%02"PRIx8")", enable);
		}
	}
}

/*
 * Decode Class-identifier option.
 * str is description of option, msg_p is buffer with option data and opt_len
 * is length of option. opt_len must be larger then 1. First byte of msg_p is
 * interpreted as hardware type <1, 16> and rest of the bytes as hardware
 * address. If type is outside <1, 16> interval, msg_p is interpreted as invalid
 * hardware type.
 */
static void
opt_decode_class_i(const char *str, const char *msg_p, uint8_t opt_len)
{
	const char *ht_str[] = {
		"Unknown",					/*  0 */
		"Ethernet",					/*  1 */
		"Experimental Ethernet",			/*  2 */
		"Amateur Radio AX.25",				/*  3 */
		"Proteon ProNET Token Ring",			/*  4 */
		"Chaos",					/*  5 */
		"IEEE 802 Networks",				/*  6 */
		"ARCNET",					/*  7 */
		"Hyperchannel",					/*  8 */
		"Lanstar",					/*  9 */
		"Autonet Short Address",			/* 10 */
		"LocalTalk",					/* 11 */
		"LocalNet (IBM PCNet or SYTEK LocalNET)",	/* 12 */
		"Ultra link",					/* 13 */
		"SMDS",						/* 14 */
		"Frame Relay",					/* 15 */
		"Asynchronous Transmission Mode (ATM)",		/* 16 */
	};
	int i;
	uint8_t type;

	printf("%s - ", str);

	if (opt_len < 2) {
		printf("invalid");
	} else {
		type = msg_p[0];
		if (type > 0 && type < 17) {
			printf("Hardware Type: %s, Hardware Address: ",
			    ht_str[type]);
			for (i = 1; i < opt_len; i++) {
				if (i != 1) {
					printf(":");
				}
				printf("%02"PRIx8, msg_p[i]);
			}
		} else {
			printf("invalid Hardware Type (0x%02"PRIx8")", type);
		}
	}
}

/*
 * Decode message type option.
 * str is description of option, msg_p is buffer with option data and opt_len
 * is length of option. opt_len must be 1. *msg_p values are decoded as message
 * type. Valid message type is in interval <1, 7>. If value is in this
 * interval, string representation is displayed, otherwise hex value of byte is
 * displayed.
 */
static void
opt_decode_dhcp_mt(const char *str, const char *msg_p, uint8_t opt_len,
    enum opt_msg_type *msg_type, int verbose)
{
	const char *mt_str[] = {
		"None",		/* 0 */
		"Discover",	/* 1 */
		"Offer",	/* 2 */
		"Request",	/* 3 */
		"Decline",	/* 4 */
		"Ack",		/* 5 */
		"Nak",		/* 6 */
		"Release",	/* 7 */
		"Unknown",	/* 8 */
	};
	uint8_t u8;

	if (verbose)
		printf("%s - ", str);

	if (opt_len != 1) {
		if (verbose)
			printf("invalid");
	} else {
		u8 = *msg_p;

		if (u8 > 0 && u8 <= 8) {
			if (verbose) {
				printf("DHCP %s", mt_str[u8]);
			}
			*msg_type = (enum opt_msg_type)u8;
		} else {
			if (verbose) {
				printf("invalid DHCPs type (0x%02"PRIx8")",
				    u8);
			}
			*msg_type = OPT_MSG_TYPE_UNKNOWN;
		}
	}
}

/*
 * Decode hex like option. Used for options which are vendor dependant
 * (currently only option 60 - Class-identifier).
 * str is description of option, msg_p is buffer with option data and opt_len
 * is length of option. opt_len must be larger then 1. *msg_p value is decoded
 * hex printed byte values.
 */
static void
opt_decode_hex(const char *str, const char *msg_p, uint8_t opt_len)
{
	int i;

	printf("%s - ", str);

	if (opt_len < 1) {
		printf("invalid");
	} else {
		printf("0x");
		for (i = 0; i < opt_len; i++) {
			printf("%02"PRIx8, (uint8_t)msg_p[i]);
		}
	}
}

/*
 * Decode ip like option.
 * str is description of option, msg_p is buffer with option data and opt_len
 * is length of option. opt_len must be 4. *msg_p value is decoded as IPv4
 * address.
 */
static void
opt_decode_ip(const char *str, const char *msg_p, uint8_t opt_len)
{
	char ipaddr_s[INET_ADDRSTRLEN];

	printf("%s - ", str);

	if (opt_len != 4) {
		strcpy(ipaddr_s, "invalid");
	} else {
		inet_ntop(AF_INET, msg_p, ipaddr_s, INET_ADDRSTRLEN);
	}
	printf("%s", ipaddr_s);
}

/*
 * Decode list of IP pairs
 * str is description of option, msg_p is buffer with option data and opt_len
 * is length of option. opt_len must be => 8 and must be multiply of 8. *msg_p
 * value is decoded as list of IPv4 address + IP mask pairs. Pair is separated
 * by separator string and pairs are separated by ", ".
 */
static void
opt_decode_ip_pairs(const char *str, const char *msg_p, uint8_t opt_len,
    const char *separator)
{
	char ipaddr_s[INET_ADDRSTRLEN];
	int i;

	printf("%s - ", str);

	if (opt_len < 8 || opt_len % 8 != 0) {
		printf("invalid");
		return ;
	}

	for (i = 0; i < opt_len / 4; i++) {
		inet_ntop(AF_INET, msg_p + (i * 4), ipaddr_s, INET_ADDRSTRLEN);
		if (i % 2 == 0 && i != 0)
			printf(", ");

		if (i % 2 != 0)
			printf("%s", separator);

		printf("%s", ipaddr_s);
	}
}

/*
 * Decode ip list like option.
 * str is description of option, msg_p is buffer with option data and opt_len
 * is length of option. opt_len must be => 4 and must be multiply of 4. *msg_p
 * value is decoded as list of IPv4 addresses, with ", " as splitter between
 * two items.
 */
static void
opt_decode_ips(const char *str, const char *msg_p, uint8_t opt_len)
{
	char ipaddr_s[INET_ADDRSTRLEN];
	int i;

	printf("%s - ", str);

	if (opt_len < 4 || opt_len % 4 != 0) {
		printf("invalid");
		return ;
	}

	for (i = 0; i < opt_len / 4; i++) {
		inet_ntop(AF_INET, msg_p + (i * 4), ipaddr_s, INET_ADDRSTRLEN);
		if (i != 0)
			printf(", ");
		printf("%s", ipaddr_s);
	}
}

/*
 * Decode NetBIOS over TCP/IP Node Type Option
 * str is description of option, msg_p is buffer with option data and opt_len
 * is length of option. opt_len must be 1. *msg_p value is decoded as Node
 * Type.
 */
static void
opt_decode_nbios_nto(const char *str, const char *msg_p, uint8_t opt_len)
{
	char ntype;
	uint8_t u8;

	printf("%s - ", str);

	if (opt_len != 1) {
		printf("invalid");
	} else {
		u8 = *msg_p;
		ntype = 0x0;

		switch (u8) {
		case 0x1:
			ntype = 'B';
			break;
		case 0x2:
			ntype = 'P';
			break;
		case 0x4:
			ntype = 'M';
			break;
		case 0x8:
			ntype = 'H';
			break;
		}

		if (ntype != 0x0) {
			printf("%c-node", ntype);
		} else {
			printf("invalid node type (0x%02"PRIx8")", ntype);
		}
	}
}

/*
 * Decode option overload options
 * str is description of option, msg_p is buffer with option data and opt_len
 * is length of option. opt_len must be 1. *msg_p value is decoded as byte,
 * where only values 1, 2 or 3 are valid.
 */
static void
opt_decode_opt_overload(const char *str, const char *msg_p, uint8_t opt_len,
    enum opt_overload *overload, int verbose)
{
	const char *val_str;
	uint8_t u8;

	if (verbose)
		printf("%s - ", str);

	if (opt_len != 1) {
		if (verbose)
			printf("invalid");
	} else {
		u8 = *msg_p;
		val_str = NULL;

		switch (u8) {
		case 1:
			if (overload)
				*overload = OPT_OVERLOAD_FILE;
			val_str = "Boot file field hold options";
			break;
		case 2:
			if (overload)
				*overload = OPT_OVERLOAD_SNAME;
			val_str = "Server host name field hold options";
			break;
		case 3:
			if (overload)
				*overload = OPT_OVERLOAD_BOTH;
			val_str = "Boot file and server host name fields hold "
			    "options";
			break;
		}

		if (verbose) {
			if (val_str == NULL) {
				printf("invalid opt overload type (0x%02"PRIx8
				    ")", u8);
			} else {
				printf("%s", val_str);
			}
		}
	}
}

/*
 * Decode policy filter option
 * str is description of option, msg_p is buffer with option data and opt_len is
 * length of option. opt_len must be => 8 and must be multiply of 8. *msg_p
 * value is decoded as list of IPv4 address + IP mask pairs. Pair is separated
 * by "/" character and pairs are separated by ", ".
 */
static void
opt_decode_pfo(const char *str, const char *msg_p, uint8_t opt_len)
{

	opt_decode_ip_pairs(str, msg_p, opt_len, "/");
}

/*
 * Decode name only option. This is used for options without specific data
 * values in them.
 * str is description of option, msg_p is buffer with option data and opt_len is
 * length of option. msg_p and opt_len are not used.
 */
static void
opt_decode_pname(const char *str, const char *msg_p, uint8_t opt_len)
{

	printf("%s", str);
}

/*
 * Decode parameter request list option.
 * str is description of option, msg_p is buffer with option data and opt_len is
 * length of option. opt_len must be >= 1. *msg_p is then decoded as list of
 * request options. Each option is printed as hex value (byte) and also it's
 * string representation.
 */
static void
opt_decode_prl(const char *str, const char *msg_p, uint8_t opt_len)
{
	struct opt_decode_struct opt_decode_s;
	const char *desc;
	int i;
	uint8_t u8;

	printf("%s - ", str);

	if (opt_len < 1) {
		printf("invalid");
	} else {
		for (i = 0; i < opt_len; i++) {
			u8 = msg_p[i];

			if (i != 0) {
				printf(", ");
			}

			opt_decode_s = opt_decode_table[u8];
			if (opt_decode_s.desc == NULL) {
				desc = "Private";
			} else {
				desc = opt_decode_s.desc;
			}

			printf("0x%02"PRIx8" (%s)", u8, desc);
		}
	}
}

/*
 * Decode Statis Route option.
 * str is description of option, msg_p is buffer with option data and opt_len is
 * length of option. opt_len must be => 8 and must be multiply of 8. *msg_p
 * value is decoded as list of IPv4 address + IP mask pairs. Pair is separated
 * by " -> " character and pairs are separated by ", ".
 */
static void
opt_decode_sro(const char *str, const char *msg_p, uint8_t opt_len)
{

	opt_decode_ip_pairs(str, msg_p, opt_len, " -> ");
}

/*
 * Decode signed 32-bit int option.
 * str is description of option, msg_p is buffer with option data and opt_len is
 * length of option. opt_len must be 4. *msg_p value is decoded as signed 32-bit
 * integer.
 */
static void
opt_decode_s32(const char *str, const char *msg_p, uint8_t opt_len)
{
	int32_t s32;

	printf("%s - ", str);

	if (opt_len != 4) {
		printf("invalid");
	} else {
		memcpy(&s32, msg_p, sizeof(s32));
		s32 = ntohl(s32);
		printf("%"PRId32, s32);
	}
}

/*
 * Decode string option.
 * str is description of option, msg_p is buffer with option data and opt_len is
 * length of option. opt_len is length of string. String doesn't need to end
 * with byte with 0x0 value (so string is Pascal type rather then C one).
 */
static void
opt_decode_str(const char *str, const char *msg_p, uint8_t opt_len)
{
	char tmp_str[512];

	memset(tmp_str, 0, sizeof(tmp_str));
	memcpy(tmp_str, msg_p, opt_len);
	tmp_str[opt_len] = 0;

	printf("%s - %s", str, tmp_str);
}


/*
 * Decode unsigned 16-bit integer option.
 * str is description of option, msg_p is buffer with option data and opt_len
 * is length of option. opt_len must be 2. *msg_p value is decoded as unsigned
 * 16-bit integer.
 */
static void
opt_decode_u16(const char *str, const char *msg_p, uint8_t opt_len)
{
	uint16_t u16;

	printf("%s - ", str);

	if (opt_len != 2) {
		printf("invalid");
	} else {
		memcpy(&u16, msg_p, sizeof(u16));
		u16 = ntohs(u16);
		printf("%"PRIu16, u16);
	}
}

/*
 * Decode list of unsigned 16-bit integers option.
 * str is description of option, msg_p is buffer with option data and opt_len
 * is length of option. opt_len must be >=2 and must be multiply of 2.
 * msg_p value is decoded as list of unsigned 16-bit integers divided by ", "
 * string.
 */
static void
opt_decode_u16s(const char *str, const char *msg_p, uint8_t opt_len)
{
	int i;
	uint16_t u16;

	printf("%s - ", str);

	if (opt_len < 2 || opt_len % 2 != 0) {
		printf("invalid");
		return ;
	}

	for (i = 0; i < opt_len / 2; i++) {
		memcpy(&u16, msg_p + (i * 2), sizeof(u16));
		u16 = ntohs(u16);
		if (i != 0)
			printf(", ");
		printf("%"PRIu16, u16);
	}
}

/*
 * Decode unsigned 32-bit integer option.
 * str is description of option, msg_p is buffer with option data and opt_len
 * is length of option. opt_len must be 4. *msg_p value is decoded as unsigned
 * 32-bit integer.
 */
static void
opt_decode_u32(const char *str, const char *msg_p, uint8_t opt_len)
{
	uint32_t u32;

	printf("%s - ", str);

	if (opt_len != 4) {
		printf("invalid");
	} else {
		memcpy(&u32, msg_p, sizeof(u32));
		u32 = ntohl(u32);
		printf("%"PRIu32, u32);
	}
}

/*
 * Decode unsigned 8-bit integer option.
 * str is description of option, msg_p is buffer with option data and opt_len
 * is length of option. opt_len must be . *msg_p value is decoded as unsigned
 * 8-bit integer.
 */
static void
opt_decode_u8(const char *str, const char *msg_p, uint8_t opt_len)
{

	printf("%s - ", str);

	if (opt_len != 1) {
		printf("invalid");
	} else {
		printf("%"PRIu8, *msg_p);
	}
}
