/*
 * GQradio
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * I must also give credit to those that
 * contributed all the code for non Linux platforms...
 * ...and apologize when I mangle it ;)
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

/*
 * This file stolen from GQmpeg, and the unsupported systems (below) were removed
 * for simplicity. To regain support for those systems, simply copy the mixer.c from
 * GQmpeg over this one (at this writing, GQmpeg was 0.10.0) and change #include "gqmpeg.h"
 */

/* current systems with supported mixer functions:
 *   Linux
 *   FreeBSD (left in because it uses the same API as Linux)
 *
 * unsupported (removed) because, well, GQradio currently only supports video4linux ...
 *   NetBSD, OpenBSD
 *   SPARC Ultra 1
 *   SGI
 *   Tru64 (OSF1)
 *   HPUX (10.X & 11.X)
 *
 * Each platform interface now has it's own section.
 */

#include "gqradio.h"
#include "mixer.h"

#include <fcntl.h>
#include <sys/ioctl.h>

#ifdef linux
#include <linux/soundcard.h>
#endif

#ifdef __FreeBSD__
#include <sys/soundcard.h>
#endif


typedef struct _DeviceData DeviceData;
struct _DeviceData
{
	gint device_id;
	gchar *device_name;
	gint stereo;
	gint recordable;
};

static void mixer_set_vol(DeviceData *device, gint vol);
static gint mixer_get_vol(DeviceData *device);

static gint mixer_enabled = FALSE;
static gint current_vol = 0;
static gint current_bal = 50;

static GList *device_list = NULL;
static DeviceData *current_device = NULL;

/*
 *--------------------------------------------------------------------
 * Linux (and FreeBSD uses the same interface)
 *--------------------------------------------------------------------
 */

#if defined (linux) || defined (__FreeBSD__)

#define LEFT  1
#define RIGHT 256
#define LEFT_MASK 0x00ff
#define RIGHT_MASK 0xff00

void mixer_init(gint init_device_id)
{
	char *device_names[] = SOUND_DEVICE_NAMES;
	int dev_mask = 0;
	int rec_mask = 0;
	int stereo_mask = 0;
	char *mixer_device;
	int i;
	int mf = -1;
	int t;

	mixer_device = getenv("MIXERDEVICE");
	if (mixer_device == NULL) mixer_device = "/dev/mixer";

	mf = open(mixer_device, O_RDWR);
	if(mf < 0)
		{
		printf(_("unable to open %s\n"), mixer_device);
		mixer_enabled = FALSE;
		return;
		}

	t = ioctl(mf, SOUND_MIXER_READ_DEVMASK, &dev_mask);
	t|= ioctl(mf, SOUND_MIXER_READ_RECMASK, &rec_mask);
	t|= ioctl(mf, SOUND_MIXER_READ_STEREODEVS, &stereo_mask);
	if (t!=0)
		{
		printf(_("Unable to determine mixer devices\n"));
		close(mf);
		mixer_enabled = FALSE;
		return;
		}

	current_device = NULL;

	/* get device listing */
	for (i=0; i<SOUND_MIXER_NRDEVICES; i++)
		{
		if ( (dev_mask & (1<<i)) ) /* skip unsupported devices */
 			{
			DeviceData *device = g_new0(DeviceData, 1);
			device->device_id = i;
			device->device_name = device_names[i];
			device->stereo = (stereo_mask & (1<<i));
			device->recordable = (rec_mask & (1<<i));
			device_list = g_list_append(device_list, device);

			if (debug_mode) printf("Mixer device added to list: %d, %s, %d, %d\n",
					device->device_id, device->device_name, device->stereo, device->recordable);

			if (init_device_id == i) current_device = device;
			}
		}

	close (mf);

	if (device_list)
		{
		mixer_enabled = TRUE;
		if (!current_device) current_device = device_list->data;
		current_vol = mixer_get_vol(current_device);
		}
	else
		{
		mixer_enabled = FALSE;
		}
}

static void mixer_set_vol(DeviceData *device, gint vol)
{
	int mf = -1;
	gint l, r;
	gint new_vol = 0x0000;
	char *mixer_device;

	if (!mixer_enabled) return;
	if (!device) return;

	if (current_bal == 50)
		{
		l = vol;
		r = vol;
		}
	else if (current_bal > 50)
		{
		l = (float)vol * (100 - current_bal) / 50;
		r = vol;
		}
	else
		{
		l = vol;
		r = (float)vol * current_bal / 50;
		}

	new_vol = l * LEFT + r * RIGHT;

	mixer_device = getenv("MIXERDEVICE");
	if (mixer_device == NULL) mixer_device = "/dev/mixer";

	mf = open(mixer_device, O_RDWR);
	if(mf < 0)
		{
		printf(_("unable to open %s\n"), mixer_device);
		return;
		}

	if(ioctl(mf, MIXER_WRITE(device->device_id), &new_vol)<0)
		{
		printf(_("failed to set volume on %d\n"), device->device_id);
		}

	close(mf);

	if (debug_mode) printf("volume set to %d,%d\n", l, r);
}

static gint mixer_get_vol(DeviceData *device)
{
	int mf = -1;
	gint vol = 0;
	gint l;
	gint r;
	char *mixer_device;

	if (!mixer_enabled) return 0;
	if (!device) return 0;

	mixer_device = getenv("MIXERDEVICE");
	if (mixer_device == NULL) mixer_device = "/dev/mixer";

	mf = open(mixer_device, O_RDWR);
	if(mf < 0)
		{
		printf(_("unable to open %s\n"), mixer_device);
		return 0;
		}

	if(ioctl(mf, MIXER_READ(device->device_id), &vol)<0)
		{
		printf(_("failed to get volume on %d\n"), device->device_id);
		vol = 0;
		}

	close(mf);

	l = vol & LEFT_MASK;
	r = (vol & RIGHT_MASK) >> 8;

	if (l == r)
		{
		current_bal = 50;
		}
	else if (l<r)
		{
		current_bal = 50 + (float)(r - l) / r * 50;
		}
	else
		{
		current_bal = 50 - (float)(l - r) / l * 50;
		}

	if (debug_mode) printf("volume is %d,%d\n", l, r);

	if (l > r)
		{
		return l;
		}
	else
		{
		return r;
		}
}

/*
 *--------------------------------------------------------------------
 * Empty functions for unsupported platforms
 *--------------------------------------------------------------------
 */

#else

void mixer_init(gint init_device_id)
{
	mixer_enabled = FALSE;
}

static void mixer_set_vol(DeviceData *device, gint vol)
{
	return;
}

static gint mixer_get_vol(DeviceData *device)
{
	return 0;
}

#endif

/*
 *--------------------------------------------------------------------
 * these are the functions that the app should call
 *--------------------------------------------------------------------
 */

/* do _not_ free the data contained in the list, only the list */
GList *get_mixer_device_list(void)
{
	GList *list = NULL;
	GList *work = device_list;

	while(work)
		{
		DeviceData *device = work->data;
		list = g_list_append(list, device->device_name);
		work = work->next;
		}
	return list;
}

gint get_mixer_device_id(gchar *device_name)
{
	GList *list = device_list;
	if (!device_name) return -1;

	while(list)
		{
		DeviceData *device = list->data;
		if (strcmp(device->device_name, device_name) == 0) return device->device_id;
		list = list->next;
		}

	return -1;
}

void set_mixer_device(gint device_id)
{
	GList *list = device_list;

	while(list)
		{
		DeviceData *device = list->data;
		if (device->device_id == device_id)
			{
			current_device = device;
			mixer_get_vol(device);
			return;
			}
		list = list->next;
		}
}

/* 0 through 100 */
void set_volume(gint vol)
{
	if (vol < 0) vol = 0;
	if (vol > 100) vol = 100;

	if (current_device && !current_device->stereo)  current_bal = 50;

	mixer_set_vol(current_device, vol);

	current_vol = vol;
}

gint get_volume(void)
{
	current_vol = mixer_get_vol(current_device);
	return current_vol;
}

/* from 0 through 100: 0 left, 50 center, 100 right */
void set_balance(gint bal)
{  
	
	if (current_device && !current_device->stereo)
		{
		current_bal = 50;
		return;
		}
	
	if (bal < 0) bal = 0;
	if (bal > 100) bal = 100;

	current_bal = bal;
	
	if (debug_mode) printf("balance set to %d\n", bal);

	mixer_set_vol(current_device, current_vol);
}

gint get_balance(void)
{
	return current_bal;
}

