/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Copyright (C) 2007 University of Waikato 
 */

package weka.core;

import weka.filters.unsupervised.attribute.StringToWordVector;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

/**
 * Tests Utils. Run from the command line with:<p/>
 * java weka.core.UtilsTest
 *
 * @author  FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 11410 $
 */
public class UtilsTest 
  extends TestCase {
  
  private static final double TOLERANCE = 1e-14;
  
  /**
   * Constructs the <code>UtilsTest</code>.
   *
   * @param name 	the name of the test class
   */
  public UtilsTest(String name) { 
    super(name); 
  }
  
  /**
   * Called by JUnit before each test method.
   *
   * @throws Exception 	if an error occurs
   */
  protected void setUp() throws Exception {
    super.setUp();
  }

  /**
   * Called by JUnit after each test method
   * 
   * @throws Exception 	if an error occurs
   */
  protected void tearDown() throws Exception {
    super.tearDown();
  }
  
  /**
   * Tests the accuracy and behavior of the {@link Utils.variance} method.
   */
  public void testVariance() {
    
    assertTrue("Incorrect behaviour when count <= 1!",
        Double.isNaN(Utils.variance(new double[]{})));
    assertTrue("Incorrect behaviour when count <= 1!",
        Double.isNaN(Utils.variance(new double[]{3})));
    
    checkAccuracy(generatedVar1, generatedValues1);
    checkAccuracy(generatedVar2, generatedValues2, 1e-2);
    checkAccuracy(generatedVar3, generatedValues3);
    checkAccuracy(generatedVar4, generatedValues4);
    checkAccuracy(generatedVar5, generatedValues5);
    checkAccuracy(generatedVar6, generatedValues6, 2);
    checkAccuracy(generatedVar7, generatedValues7);
    checkAccuracy(generatedVar8, generatedValues8, 2);
    checkAccuracy(generatedVar9, generatedValues9);
    checkAccuracy(generatedVar10, generatedValues10);
  }
  
  private void checkAccuracy(long var, long[] values) {
    checkAccuracy(var, values, TOLERANCE);
  }
  
  private void checkAccuracy(long var, long[] values, double tol) {
    double ref = Double.longBitsToDouble(var);
    double test = Utils.variance(convert(values));
    assertEquals("Inaccurate variance calculation!", ref, test, Math.abs(tol*ref));
  }

  private double[] convert(long[] values) {
    double[] valuesDouble = new double[values.length];
    for (int i = 0; i < values.length; i++) {
      valuesDouble[i] = Double.longBitsToDouble(values[i]);
    }
    return valuesDouble;
  }

  /**
   * tests splitOptions and joinOptions
   * 
   * @see Utils#splitOptions(String)
   * @see Utils#joinOptions(String[])
   */
  public void testSplittingAndJoining() {
    String[] 	options;
    String[] 	newOptions;
    String 	joined;
    int		i;
    
    try {
      options    = new StringToWordVector().getOptions();
      joined     = Utils.joinOptions(options);
      newOptions = Utils.splitOptions(joined);
      assertEquals("Same number of options", options.length, newOptions.length);
      for (i = 0; i < options.length; i++) {
	if (!options[i].equals(newOptions[i]))
	  fail("Option " + (i+1) + " differs");
      }
    }
    catch (Exception e) {
      fail("Exception: " + e);
    }
  }
  
  /**
   * tests quote and unquote
   * 
   * @see Utils#quote(String)
   * @see Utils#unquote(String)
   */
  public void testQuoting() {
    String 	input;
    String 	output;
    
    input  = "blahblah";
    output = Utils.quote(input);
    assertTrue("No quoting necessary", !output.startsWith("'") && !output.endsWith("'"));
    
    input  = "";
    output = Utils.quote(input);
    assertTrue("Empty string quoted", output.startsWith("'") && output.endsWith("'"));
    assertTrue("Empty string restored", input.equals(Utils.unquote(output)));
    
    input  = " ";
    output = Utils.quote(input);
    assertTrue("Blank quoted", output.startsWith("'") && output.endsWith("'"));
    assertTrue("Blank restored", input.equals(Utils.unquote(output)));
    
    input  = "{";
    output = Utils.quote(input);
    assertTrue(">" + input + "< quoted", output.startsWith("'") && output.endsWith("'"));
    assertTrue(">" + input + "< restored", input.equals(Utils.unquote(output)));
    
    input  = "}";
    output = Utils.quote(input);
    assertTrue(">" + input + "< quoted", output.startsWith("'") && output.endsWith("'"));
    assertTrue(">" + input + "< restored", input.equals(Utils.unquote(output)));
    
    input  = ",";
    output = Utils.quote(input);
    assertTrue(">" + input + "< quoted", output.startsWith("'") && output.endsWith("'"));
    assertTrue(">" + input + "< restored", input.equals(Utils.unquote(output)));
    
    input  = "?";
    output = Utils.quote(input);
    assertTrue(">" + input + "< quoted", output.startsWith("'") && output.endsWith("'"));
    assertTrue(">" + input + "< restored", input.equals(Utils.unquote(output)));
    
    input  = "\r\n\t'\"%";
    output = Utils.quote(input);
    assertTrue(">" + input + "< quoted", output.startsWith("'") && output.endsWith("'"));
    assertTrue(">" + input + "< restored", input.equals(Utils.unquote(output)));
  }
  
  /**
   * tests backQuoteChars and unbackQuoteChars
   * 
   * @see Utils#backQuoteChars(String)
   * @see Utils#unbackQuoteChars(String)
   */
  public void testBackQuoting() {
    String 	input;
    String 	output;
    
    input  = "blahblah";
    output = Utils.backQuoteChars(input);
    assertTrue("No backquoting necessary", input.equals(output));
    
    input  = "\r\n\t'\"%";
    output = Utils.backQuoteChars(input);
    assertTrue(">" + input + "< restored", input.equals(Utils.unbackQuoteChars(output)));
    
    input  = "\\r\\n\\t\\'\\\"\\%";
    output = Utils.backQuoteChars(input);
    assertTrue(">" + input + "< restored", input.equals(Utils.unbackQuoteChars(output)));
    
    input  = Utils.joinOptions(new StringToWordVector().getOptions());
    output = Utils.backQuoteChars(input);
    assertTrue(">" + input + "< restored", input.equals(Utils.unbackQuoteChars(output)));
  }
  
  public static Test suite() {
    return new TestSuite(UtilsTest.class);
  }

  public static void main(String[] args){
    junit.textui.TestRunner.run(suite());
  }

  //Generated values with parameters:
  //Count = 100
  //values_mean = 1.000000e+08; values_stdDev = 1.000000e+08; values_ordering = Random
  private static final long generatedVar1 = 4846379797248880081L; // approx ~ 1.002040e+16
  private static final long[] generatedValues1 = {-4502108233396493964L, 4723728974583900862L, 4724352188189308360L, 4728115190960860804L, 4729651617148795053L, 4725637613972895537L, 4730983312112713362L, 4732984965466902608L, 4733348328003874764L, 4721867348739709532L, -4515915348804967712L, -4504475538151643424L, 4720620978861520689L, 4729253181485441422L, 4728924315319343309L, 4730860883401456904L, 4729624414008534920L, 4721944814894706417L, -4502718654540998012L, -4505760338916381024L, 4732325452438913325L, -4503995118402924976L, -4499171420353650292L, 4701274311466297152L, 4728911828384797328L, -4504173214362173328L, -4500037646772938368L, 4722659557372032140L, 4721662633398364149L, 4726865972667491760L, 4726785560101146141L, -4508794473662502776L, 4729003032045369746L, 4731873093211098583L, 4703857113551828096L, 4722843885751083471L, 4713949557851769080L, -4501334979361403136L, 4726971853446339409L, -4501110925457140188L, 4728810432137718965L, 4728195121463203742L, 4729725337545897655L, 4720378352228018261L, 4727690759409108414L, 4732055580878571254L, 4727132649379861353L, 4726514625839732792L, 4719931711826427412L, 4725825440236714718L, -4502653151060615824L, 4732441086397906553L, -4499679870917995464L, 4726125560497166743L, 4732288574983369587L, -4503495536230517904L, 4731674699063872490L, -4509686744296375680L, -4502573192816411356L, 4729128740459696730L, 4725207374250699774L, 4729338924827077800L, 4732668192408843973L, 4732931336961679433L, -4504326372865511840L, -4499658508274424552L, -4499458222199525156L, 4727707496571224177L, 4730997225583310543L, -4515381666377829712L, 4729539307593296511L, 4704104525801341344L, 4732145306997181814L, 4715809951873630108L, 4729311235884579739L, 4731345354127246072L, 4725707175693491160L, 4730630464779307854L, 4730133941199409854L, 4731328290963192479L, 4731361316139689106L, 4714698125210484040L, 4732133740508962216L, 4729140741926166706L, -4508102649237681760L, 4719218275732095672L, 4721787841851815648L, 4729112873433246780L, -4506546851958645244L, 4733246230484764212L, 4730842947051650822L, -4508306899685325664L, 4729152439012063403L, 4732892633054335396L, 4732062715198881078L, 4722256675761560790L, 4721095628932846213L, 4718724670462699848L, 4728211531090703109L, 4704495152248023264L};

  //Generated values with parameters:
  //Count = 100
  //values_mean = 1.000000e+08; values_stdDev = 1.000000e-08; values_ordering = Random
  private static final long generatedVar2 = 4369282270483963940L; // approx ~ 1.305129e-16
  private static final long[] generatedValues2 = {4726483295884279809L, 4726483295884279808L, 4726483295884279809L, 4726483295884279807L, 4726483295884279809L, 4726483295884279808L, 4726483295884279807L, 4726483295884279809L, 4726483295884279808L, 4726483295884279808L, 4726483295884279807L, 4726483295884279807L, 4726483295884279807L, 4726483295884279807L, 4726483295884279809L, 4726483295884279809L, 4726483295884279808L, 4726483295884279809L, 4726483295884279808L, 4726483295884279808L, 4726483295884279809L, 4726483295884279808L, 4726483295884279809L, 4726483295884279809L, 4726483295884279808L, 4726483295884279809L, 4726483295884279807L, 4726483295884279809L, 4726483295884279809L, 4726483295884279809L, 4726483295884279809L, 4726483295884279808L, 4726483295884279807L, 4726483295884279808L, 4726483295884279809L, 4726483295884279807L, 4726483295884279809L, 4726483295884279809L, 4726483295884279807L, 4726483295884279807L, 4726483295884279807L, 4726483295884279807L, 4726483295884279808L, 4726483295884279808L, 4726483295884279808L, 4726483295884279808L, 4726483295884279807L, 4726483295884279808L, 4726483295884279808L, 4726483295884279807L, 4726483295884279808L, 4726483295884279809L, 4726483295884279809L, 4726483295884279808L, 4726483295884279808L, 4726483295884279808L, 4726483295884279807L, 4726483295884279807L, 4726483295884279808L, 4726483295884279808L, 4726483295884279809L, 4726483295884279808L, 4726483295884279807L, 4726483295884279808L, 4726483295884279807L, 4726483295884279809L, 4726483295884279809L, 4726483295884279809L, 4726483295884279809L, 4726483295884279809L, 4726483295884279808L, 4726483295884279808L, 4726483295884279808L, 4726483295884279809L, 4726483295884279808L, 4726483295884279808L, 4726483295884279807L, 4726483295884279808L, 4726483295884279809L, 4726483295884279808L, 4726483295884279808L, 4726483295884279808L, 4726483295884279807L, 4726483295884279807L, 4726483295884279808L, 4726483295884279807L, 4726483295884279807L, 4726483295884279808L, 4726483295884279808L, 4726483295884279808L, 4726483295884279809L, 4726483295884279809L, 4726483295884279809L, 4726483295884279809L, 4726483295884279808L, 4726483295884279809L, 4726483295884279808L, 4726483295884279807L, 4726483295884279809L, 4726483295884279808L};

  //Generated values with parameters:
  //Count = 100
  //values_mean = -1.000000e-08; values_stdDev = 1.000000e+08; values_ordering = Random
  private static final long generatedVar3 = 4844570879730511607L; // approx ~ 7.704880e+15
  private static final long[] generatedValues3 = {4727614085264422387L, 4729822904144993430L, -4501533623300706899L, 4725550860350631780L, 4707831857130637288L, -4493831279798689701L, 4690350546008583362L, 4715250026626502695L, -4500723130024346213L, -4506104426821601215L, -4499647392731248643L, -4502492504827419791L, 4714995792120462637L, 4729143799547126471L, -4496530082912807058L, -4516495068366770762L, -4502844105319543054L, 4715754168521955409L, 4707535500818617910L, -4538932597234654826L, 4724336516246955826L, -4496367369090342063L, 4728543336485433632L, -4511177646709752083L, 4728144163944883492L, 4726029013994137935L, -4497479371383437700L, 4724020269267902956L, 4729978563936381012L, -4502521620806730359L, 4724125513427702137L, 4723493393721426096L, 4724533706444586895L, 4722439303169502110L, -4502497299786702930L, 4724809461201778872L, -4505034711994511426L, -4502162673658398504L, 4716984000072350202L, -4494574756088018883L, -4537228024645647354L, 4723880341987935573L, -4494567814509031927L, 4725961287861098272L, -4495166446833190090L, -4509133419599434981L, 4702845236393422636L, 4724526448640755611L, -4494675726122808919L, 4728147557186292566L, 4728470461460451601L, 4725535951488418303L, 4729271455468345782L, 4727087873814575776L, 4728158588186642127L, -4501254022379924972L, -4494197173216948786L, 4729030252304087590L, 4717828167491930196L, 4708729658997642194L, -4493824896020691984L, 4728326903494815884L, 4721560995342556147L, -4495067790032209144L, -4493875482175190724L, 4725016035147302199L, 4725036647699806016L, 4728232732195143901L, -4496630962002429254L, 4720450510954109921L, -4496736254019316830L, -4507058793038450992L, -4502022642592921783L, -4508754849778200138L, 4725373534790589372L, 4725680292683889571L, -4518902258473409101L, -4517474472222193740L, 4725784253634528052L, 4728482916516002507L, 4721765014155612542L, 4715653595921436055L, -4505706194065392789L, 4704283850301763462L, 4717498068933105177L, -4497477773517110812L, -4493356434378926840L, 4727839907154884133L, 4727489954514901048L, 4717002805459010187L, -4496151925035974466L, 4725869328512287560L, 4724410623171322973L, -4497597528424304799L, -4503484818442129578L, -4499516236247547924L, -4496106139519803354L, 4717097421922224635L, 4725713595518978797L, -4506676387566201673L};

  //Generated values with parameters:
  //Count = 100
  //values_mean = -1.000000e-08; values_stdDev = 1.000000e-08; values_ordering = Random
  private static final long generatedVar4 = 4364942961993454131L; // approx ~ 6.728149e-17
  private static final long[] generatedValues4 = {-4732349234744891292L, -4732764153069503801L, -4731544356145578862L, 4468568715091447808L, -4731530721302521152L, -4733807168040689408L, -4733310824566269017L, -4730126204321226147L, -4735511974915456215L, -4736290680996051437L, -4734819959696855997L, -4741655705708219572L, -4736799697507971543L, -4734510704765379322L, -4735231819551601746L, -4732886695229309825L, -4730640277889541612L, -4736684019329934081L, -4735765430657243202L, -4731332157453076144L, -4738947895009155298L, -4731200856510489866L, -4731971430368897513L, -4730426272880812804L, -4745495158457200588L, 4468154805568426016L, -4731212578161773993L, -4730186188501065544L, -4731091262002054770L, -4740431048420181034L, 4467413179796375712L, -4732032338465530323L, -4731469107935286147L, -4736982586081345857L, -4735306150957214138L, -4732837520346473777L, -4742081919565457476L, -4732847911870317075L, -4732352981636993974L, -4730807901525864194L, -4731055052453529050L, -4735146311823665969L, -4739501547049354382L, 4481157913111773566L, -4733008534555692671L, -4731771917095641020L, -4732532919878115979L, -4744819749819817374L, -4730455765112876112L, -4730417945763269048L, -4731234324365921540L, -4730605407528700866L, -4752063342983935136L, -4735055061107730540L, -4734278101861304946L, -4753932182896083184L, -4736097779917929790L, -4738529420189981829L, -4729964457452431733L, -4736789922572968900L, -4736931565190904092L, -4732066755953478650L, -4731074700743653222L, -4730361484119683196L, -4741984057750141413L, -4730199867826006657L, -4741565110816140723L, -4733861538425538390L, 4473610356461396976L, -4730429174613584778L, -4732301848712660890L, -4739101733655202930L, -4744230848897304682L, 4442747765734639616L, -4731275273888975450L, 4477615042456604740L, -4736490219644851704L, -4735150349658359566L, -4734853071177605441L, -4731827295810397163L, -4732155364265618487L, 4481808266510177504L, -4735416841210442270L, -4732611876429199408L, -4731335471170417399L, -4732057957170056876L, -4740135310457232794L, -4735178479569557102L, -4732260946064420250L, -4738290718949096975L, 4474490408149131392L, -4733038422987526050L, -4731464189857122527L, -4735217500533836421L, -4737713700100642568L, -4729961959293741809L, -4734151738701183188L, -4742545162452425432L, -4731860572560781248L, -4733679049866745466L};

  //Generated values with parameters:
  //Count = 50
  //values_mean = 1.000000e+08; values_stdDev = 1.000000e+08; values_ordering = Sorted
  private static final long generatedVar5 = 4846974017524636543L; // approx ~ 1.120884e+16
  private static final long[] generatedValues5 = {-4498912892414487052L, -4499398190319717868L, -4500297290314489324L, -4500387474705298668L, -4503250593641512088L, -4503472082898395048L, -4505446264199319328L, -4506226164228556148L, -4507335111092290224L, -4509814722879662680L, -4512047619108749048L, -4518505355305333984L, 4675635140899745792L, 4704718470733712608L, 4706649608693031136L, 4710103337520561648L, 4712449697085350064L, 4717916270725686172L, 4719329868832357312L, 4720941195373143007L, 4723169652029697694L, 4723885068325350574L, 4724991248018793708L, 4726043776248824682L, 4727503349390206675L, 4727899223121157425L, 4728209384497727132L, 4728470263533433045L, 4728563778294844192L, 4728939506247005870L, 4728955523076082793L, 4729012780241202296L, 4729049080837647391L, 4729197137554604722L, 4729698074603569364L, 4729728348461888755L, 4730039405925020419L, 4730304008457155474L, 4730577809882478181L, 4730723620479735931L, 4731147338656098480L, 4731949470484435370L, 4732019617746427948L, 4732492511208269189L, 4732632465318108953L, 4732751002550804032L, 4732882842086225281L, 4732958476178190786L, 4733094865267973824L, 4733347240363570364L};

  //Generated values with parameters:
  //Count = 50
  //values_mean = -1.000000e+08; values_stdDev = 1.000000e-08; values_ordering = Sorted
  private static final long generatedVar6 = 4369315153909814582L; // approx ~ 1.313235e-16
  private static final long[] generatedValues6 = {-4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L};

  //Generated values with parameters:
  //Count = 50
  //values_mean = 1.000000e+08; values_stdDev = 1.000000e+08; values_ordering = Reverse Sorted
  private static final long generatedVar7 = 4847454532936292183L; // approx ~ 1.216987e+16
  private static final long[] generatedValues7 = {4733009454853646059L, 4732778079177426550L, 4732770666383580907L, 4732679762989759714L, 4732523447425619714L, 4732499834143549220L, 4732437188696910055L, 4732398191634574027L, 4732318612597552267L, 4731993543423580910L, 4731951886386944301L, 4731853143936732039L, 4731673248238447634L, 4731425190595213750L, 4731413678370342506L, 4730871568547959082L, 4730778505442458782L, 4730682748768909922L, 4730342546771223910L, 4730246797010979373L, 4730002093450321382L, 4729950167036488148L, 4729792583697966228L, 4729515262082537643L, 4728280546845563080L, 4728212970581925712L, 4727001108054173316L, 4726910685091916084L, 4726004979163352446L, 4724170136956617870L, 4723130604756323361L, 4721936433058408405L, 4720375336153118950L, 4718557835155273104L, 4717979851059892296L, 4708226753123151808L, 4708100312712094544L, 4703785533515119040L, 4702657646822909056L, 4700520647910821376L, 4697770635769605376L, -4512668076497743472L, -4508042234396100400L, -4506187882317570828L, -4502539088583846808L, -4502191864566717704L, -4501457165522751436L, -4501055850020964732L, -4499000017400300796L, -4498962852921562214L};

  //Generated values with parameters:
  //Count = 50
  //values_mean = -1.000000e+08; values_stdDev = 1.000000e-08; values_ordering = Reverse Sorted
  private static final long generatedVar8 = 4369653383432849754L; // approx ~ 1.396615e-16
  private static final long[] generatedValues8 = {-4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496001L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970496000L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L, -4496888740970495999L};

  //Generated values with parameters:
  //Count = 1000
  //values_mean = 1.000000e-08; values_stdDev = 1.000000e-08; values_ordering = Random
  private static final long generatedVar9 = 4367851248197333864L; // approx ~ 1.031289e-16
  private static final long[] generatedValues9 = {-4738419636726645504L, 4492175665015393186L, 4488258910959072805L, 4483659132122450276L, 4473809822363678304L, 4472295060365326336L, 4480144253704473416L, 4491715550788379772L, 4491424924140582190L, 4450852845036980480L, 4492324902983775220L, 4490891950315874246L, -4753209675777271152L, 4487486927113199309L, 4489364297420713556L, 4491582443109948642L, 4487965888685123080L, -4744548937805613196L, 4491842350486422884L, 4477760817624520948L, 4493536100406859797L, -4751089311002456392L, 4491282513277698248L, 4490083818285853317L, 4493821069156187300L, -4739509144415512064L, 4490636020019082751L, 4490417219504175040L, 4487462787550777907L, 4471511234110194832L, 4485623585476111243L, 4473358937767471224L, -4741928211895608914L, 4490907312698072876L, 4490730368279754090L, 4491115755337810584L, 4478554586150829478L, 4467305337853165888L, 4490198366954839338L, 4490133672270448563L, -4740690110944371400L, 4493763113103889236L, 4491978065071772982L, 4491177035137313700L, 4493330908104736662L, 4486912224753986021L, 4493433093954402962L, 4491656962803659342L, 4485031697763056364L, 4489306296395091609L, 4492441885381865204L, -4741795324019083760L, 4484713186356930114L, 4493382869020057051L, 4481055611009526082L, 4486146239753006748L, 4491461311916272024L, 4493274659068501020L, -4741963897534605626L, 4475943266658929176L, -4739670497335440792L, -4740270320310813360L, 4480649110253017870L, 4493750643572590074L, -4741606657505111608L, 4490431338429006232L, 4493696454621636284L, 4478526805392806634L, 4491863797596186163L, 4493824805427322812L, -4750691753180739376L, 4473563551525034408L, -4744181807906227696L, 4487791043970071804L, 4492573983387531606L, 4490120803987896248L, 4481375389976416697L, 4473872997532327736L, 4493561128420984414L, 4492903449788442506L, 4485377890915217204L, 4476096869356000096L, 4480403467784042062L, -4740807533297582692L, 4487518009098158412L, 4485628121262600199L, -4739653906706558720L, 4484648994169042837L, 4488977505376010410L, 4491077837239056628L, 4492354010039650002L, 4492463367023557252L, 4488277489889052528L, -4740666045767863876L, -4738011138086479484L, 4491672862038080552L, 4489182502244692763L, 4491356521638199526L, -4750949653537341552L, 4489605718529020366L, 4492174149485192305L, 4491564034274338428L, -4741208176686768418L, 4478127065716103648L, -4742621156294086704L, 4493491688395215063L, 4489740017967315557L, -4745024546119324932L, 4491960111245650162L, -4738699669949874060L, 4490769169919961488L, 4489950454249466355L, 4492293281050067882L, 4472021615909008464L, 4492204589230035858L, -4746258940683644696L, 4480410504959118704L, -4748203256864571672L, 4487823860792322022L, -4738263902411284292L, 4486205642352942946L, 4489638497817337820L, 4491441226640176390L, 4485966851425270185L, 4491894800007302240L, 4483153409628284948L, -4747954691399652024L, 4489970063404639961L, 4474469630031764880L, 4493215216953164386L, 4487068111477669061L, 4490527454288221001L, 4493026542210654294L, -4741783326504028104L, 4489448017822258494L, -4740160923834096496L, 4483791754062770140L, 4487974493212710496L, -4750074906337805688L, 4487935438034468872L, 4486010598231279131L, 4487539431762492413L, 4483853230219070988L, 4490555240174092691L, -4742512372507585900L, 4487004169051857187L, 4491150743627372932L, 4481506092533428470L, 4493252374173990854L, 4492550361164401056L, -4739022371157937212L, 4482138549890892779L, 4485442354676506211L, 4477788425412411084L, 4487455707510770986L, 4464204464062446400L, 4493127819942737606L, 4493355024781198591L, 4479757051764420860L, -4740727396208376236L, -4740920541665652326L, 4486327112799608140L, 4490594788769807915L, -4739098393498752480L, 4486256155696218756L, 4484791996410841030L, 4491117329093606311L, 4491508098678383597L, 4492557184316628306L, 4491653027010028808L, 4483183897189402758L, 4483838775706277797L, 4492425660550931194L, 4482646563861285001L, 4490987370380680576L, 4489825899773658880L, 4477874275337888488L, -4742062895075302032L, 4489313486413896021L, 4492477465371361394L, 4490321550870869144L, 4478775530393672858L, 4473331564506844888L, 4489861331734254965L, 4485872632809531898L, 4491235113013202102L, -4741434245931590962L, 4478023235489640200L, -4745409424670643860L, 4489718036319190814L, -4750303917285075920L, 4493539759527802268L, 4488077510879436508L, 4481168661591128911L, 4491954481596004996L, 4490234028936334250L, 4492884129501618292L, 4493679602435275268L, -4748486446821485680L, -4751230848315531768L, 4487532816183796536L, 4489586021817617286L, 4489483130760329212L, -4738900644041048344L, 4492872837037736539L, -4743880384894149248L, -4746138415462672912L, 4486855629220748638L, 4486469792229496800L, 4491328510089991215L, 4488084547061853336L, 4491611304443713166L, 4485605596517613404L, 4492082880746247588L, 4489450486150247058L, 4488060552423607628L, 4487367017659554075L, -4740362207584496772L, -4748884242731872688L, 4472864051043952632L, 4492626788781888464L, -4743534049187562276L, 4490463631399078405L, 4467226496257848704L, 4487424866439215850L, 4491548758859031000L, 4491946774956351811L, 4493686195938726506L, 4485694849886294127L, -4745381828185299748L, -4751609812690492880L, 4492077860824272744L, 4486515340187672333L, 4486953743226223549L, 4491848024005329240L, 4492476894691801962L, 4487732674675322621L, 4488460368673097884L, 4485435168432862872L, 4493801137505774737L, 4491657649916771082L, 4483428093093258838L, 4491268715430302856L, 4493295901721842968L, 4492976880971192842L, -4756004375496464320L, 4490198113667636606L, 4491944605173251048L, 4491994666630714394L, 4489005810340970099L, 4450854308488643072L, 4486283858395204469L, 4480344687716620180L, 4488134981463258557L, -4743162971342014524L, 4493768040720348022L, 4489290912935989310L, 4490517910777466916L, 4487048849673518419L, 4484844710603976664L, 4475953812288785784L, 4477405508728617824L, 4482240604367271778L, 4490054504233268223L, 4490349301765754832L, 4492146883148207186L, -4738426540942713392L, 4490835532681919908L, 4486932588999410890L, 4492997498224580696L, 4480189292892351684L, 4493565274034840901L, 4492097894789382986L, 4477172810139635500L, -4755726257402764848L, 4485491711448590506L, 4481891373860373838L, 4490754244057987304L, 4486998754157706432L, 4485960563553171566L, 4491736774166139847L, 4485556980230192988L, 4491917124580350200L, 4460210245208864960L, 4493171119804721665L, 4467747097152079504L, 4485597377620580592L, -4746557158613243008L, -4753291892275247632L, -4739526532023489484L, 4493751319359119486L, 4481159833535261557L, 4488044138751575906L, 4492694666458735452L, 4491190436601811067L, -4738208360804266232L, 4483065373037405231L, -4743848339304368760L, 4491304031536172718L, -4738405900758131640L, 4488948236512452329L, 4490701809411997106L, 4489524574027152730L, 4481262289618424241L, 4486114931941957882L, 4477410668054121416L, 4493828419389295990L, 4491325466955725512L, 4487931285317470988L, -4751843420820902560L, 4491568435086420657L, 4492740191072835112L, 4490480780931971480L, 4489944024641233516L, 4485723956589787279L, -4740160686419523764L, 4493241032994629144L, 4483237114928042236L, 4492730499861183842L, 4485708999915538045L, 4492653354710838988L, 4492942007190967092L, -4742001246479670122L, -4738923678481803344L, 4461160109199754944L, 4492909806200128302L, -4769148764784330752L, -4747314746672136928L, 4487282514238255350L, 4484509026370890707L, -4744737748140498352L, 4485013324409471872L, 4489876605488239336L, 4492954874219455450L, -4741865117965172966L, 4481526734606110653L, 4487576830280903213L, 4487443401986924640L, 4493399500560605462L, 4483897257961067542L, -4753602899578454704L, 4476446205988091416L, -4744890277361339696L, -4743682641122846504L, 4493740883416202345L, -4741485719544541560L, 4484285725503643302L, 4484758495000816892L, 4476927731133184844L, 4493092662682563707L, 4481219608593843709L, -4739543578135219336L, 4491723286793815498L, 4489617068832482740L, 4493397150013821592L, 4492366151389650366L, -4739580517479364692L, 4493752949653238935L, 4490406459228770402L, 4476795982454934664L, 4492432832338050708L, 4493551617985508963L, -4741482120336135148L, -4741733188874148674L, 4484696673018200634L, 4475292253475641840L, 4476797673946608548L, 4479377706215230378L, 4492293724863913018L, -4744550412275520740L, 4473037987339408960L, 4491542854030633486L, 4474693624188253336L, 4491731372419366718L, -4741031955108595526L, 4486090892628455777L, 4492839605043160859L, 4480936015810799088L, 4491662429481808804L, -4745105954148409620L, 4462196622940749056L, 4487639053863939636L, 4490212718448662934L, -4738618554712964780L, 4490943045808682106L, 4490980901788255887L, -4741945408638273292L, 4484122262955456436L, 4490086963318208172L, 4485630159720385504L, -4749224508337207688L, 4493766791411415235L, 4483291545550809851L, 4492282268523667800L, 4493069013137631464L, 4492024655712705586L, -4744212003722187244L, 4490584552270988003L, 4432027389012733952L, 4491587103820651420L, 4493036755594751820L, 4492756313641868758L, 4485645943626104175L, 4483149797129175026L, 4488958508719319907L, 4493396048453211442L, 4493761531562477257L, 4492879574337964524L, 4493812113013049312L, 4490481013494962164L, 4486052966709630144L, -4740931764587872846L, 4491491740537233224L, -4745976134874651236L, -4741052994911173078L, 4458926700793355968L, 4487824507706544931L, 4490647327564988908L, 4486035835715866590L, 4480891271010816036L, 4480588026317234672L, 4490528216126797606L, 4493077608536147064L, 4493253408792599015L, 4493773992036428833L, 4493139630255769480L, 4472242475069388112L, 4491812301586960272L, 4488505642704461416L, 4488720695330383269L, 4491121809457585437L, 4486740597251719084L, 4486901904134311318L, 4486020572503868762L, 4492018901985138470L, 4489701228587636508L, 4479958287505182670L, 4478971665539748718L, 4489708619743367090L, 4489939380173238062L, 4488720024709711711L, 4481199146181719160L, -4747288153841698624L, 4493616412716009863L, -4749257479270878328L, -4739421150756860544L, 4492551315898184077L, 4487597928313495133L, 4492587320770403006L, -4750884516181366576L, 4490154673118447096L, 4484722293440416322L, 4482333972752939435L, -4754934373889104336L, -4740715603936778952L, 4481739716245438708L, 4492103810193151426L, 4491409536418269073L, -4739238954200326284L, 4487642474858951159L, 4482281792943555316L, -4739506342674123624L, 4481241089352434646L, 4490970051757628474L, 4485241745863276584L, 4479700257990165532L, -4741406081725758924L, -4739291151708559116L, 4489807377016858504L, -4745608603523186684L, 4490246918431162657L, 4485391388379355904L, 4482161089814320975L, 4476503035108426304L, 4489142234524691508L, 4483183708088070198L, 4475248245208024904L, 4491236506782903206L, 4491128843099250878L, 4486779048291630834L, 4491610475952086926L, 4472969270034234768L, 4492263215370280683L, -4744767896493871924L, 4490036675280124434L, -4741504328718552686L, 4491426871719510341L, 4493132123957360432L, 4491306281461287276L, 4492701066750397034L, 4490721274346523918L, 4485418539916162478L, 4491879403196899254L, -4744346573436784728L, 4487983940603229500L, -4749432322588480552L, -4741010521340088054L, -4747467074858467888L, 4483619287392436169L, 4482987539051946504L, 4487368723011121815L, -4737947021067855736L, 4456249096660353024L, 4492989319652367234L, 4484554148115451571L, 4490582077594808608L, 4486351448563979215L, 4487663548249839646L, -4749177403808066256L, -4739794899569657104L, 4481897342390752358L, -4744498723932584520L, 4488712088472495508L, 4493452138404303483L, 4492594225949247124L, -4737956031861209564L, 4491558698794513084L, 4490679665342308447L, 4472340932648112664L, 4490742024495129642L, 4492773452001138364L, 4485217616781019155L, 4491034551193980477L, 4488183351212424225L, -4740560756833188604L, 4480996161015509542L, 4492243502739046428L, 4490154165024730238L, 4490778106199910338L, 4487980200033333361L, 4478576479754788434L, 4489114392209985249L, 4492068186909534990L, 4491007457694192391L, 4493252126018002416L, 4488719353741864751L, 4492713741300212812L, 4489465412351257442L, 4487280446390847725L, -4743527665830878280L, 4487637085670451091L, 4468982169509963744L, -4756100669187336672L, 4492011584622952757L, 4493161064405536800L, 4489058548698910765L, 4493558042706038303L, 4485994053900931468L, -4749042631015487960L, 4482205113525024795L, 4487863721550515556L, 4489517181712487852L, 4492138326464583178L, 4477415532909897832L, 4489763201851985668L, 4491898156689024113L, -4745595744213528944L, 4489490714450439802L, 4490051530118865480L, 4492021005049256317L, 4487008707693704349L, -4750183756875082560L, 4493577984900298808L, 4482286415633451129L, 4483378965367157807L, -4745928549951587184L, 4492280505010494170L, 4489672142101198834L, 4473485855331190056L, 4487718251417274448L, 4492094864261063760L, 4485291081995978454L, 4489156235498521513L, -4742283384581850860L, 4483466884630124184L, 4483700626383250186L, 4488179343521822168L, 4491185724312558624L, -4738796478974017116L, 4492866405168986433L, 4486793330681736884L, 4488246895347548900L, 4489561155349998998L, -4741860388342054636L, 4490133453357187200L, 4488889167130329382L, 4490092646440439396L, -4741067932142466136L, 4488745282375581664L, 4485754245137538624L, 4493103062476848713L, -4749253463895667728L, 4493337916813115926L, 4493316512313296654L, 4491060193381742153L, 4492792390914119652L, 4482417125217940187L, 4485573926876126166L, 4482126320036460306L, 4474546850906731728L, 4483179137600277486L, 4472236734419970168L, 4479037507381007188L, 4485950161151592218L, 4488579447768495756L, 4490817759255792877L, 4491552600507647214L, -4739234765544910928L, 4492852000652797043L, 4477217380151913372L, -4739356300713318384L, -4746453382729743548L, 4481229475601921924L, -4737949251760174044L, -4739821744055818032L, 4493002835279258913L, 4482515563763777202L, 4489182188396472214L, 4492581330573750621L, 4486341361892797790L, 4493482799580466612L, 4493190456167634410L, 4479807894891740100L, 4479473246724346360L, 4488840477378088146L, 4489319079252582623L, 4493797132599730290L, -4739799782588410520L, 4479225188605413190L, 4490583345185370439L, 4488160423581538358L, 4487899587253423169L, 4492306203198839401L, -4739229433145415948L, 4488666249155767987L, 4489432236259821576L, 4486188747704020400L, 4491561413202654960L, 4483241288808030510L, 4490323160810382500L, -4745752286188337764L, 4469539023740890560L, -4740500749522231140L, 4485081172145418449L, -4738207371539309704L, -4754657663247044800L, -4738963049944443484L, -4738258782729007676L, -4738768072856330588L, 4488399402277903221L, 4493543224544019525L, 4485040440836413844L, 4490255545128250240L, -4740093914537973896L, 4483331590864075520L, 4463095189629404800L, 4488298052146682334L, -4746067958968426692L, -4738425227670759128L, 4481527983511362098L, 4489659734123720612L, 4492853515568155419L, 4490561935488963534L, 4491781867211524459L, 4492425657091622494L, -4742252896577264864L, 4483293480243083635L, 4482160353915225004L, 4480855972018984722L, 4491953683249908715L, 4492085078812050554L, 4493730640561006473L, 4490227410421277400L, 4486974104546627174L, 4493175729071150851L, 4493223920062006650L, 4490238149832815140L, 4493542576592939685L, 4493471873266860282L, -4740300425655646112L, 4486435012757234844L, -4738398931341845364L, 4472079072672628136L, 4484185288105315733L, 4486866264133520618L, 4488348213723304771L, 4489185914827047832L, 4487752205993497333L, 4440614105134065664L, 4486573878350919243L, -4768412805482431232L, -4743354040458719028L, 4487650310592439285L, 4493695954077492622L, 4488941163223398546L, 4490463232761977226L, 4490200068168618590L, 4491300112705541449L, -4748954773834029528L, 4484543707847372075L, 4491313268243263767L, -4744364629044287876L, 4493111893577998692L, -4740294874200037456L, -4746336231782435252L, 4491599723803773582L, -4741026978935458384L, 4493263297683010552L, 4488645216255098625L, -4738497773751996752L, 4491736762117238264L, -4739359448047042200L, 4489163191552505686L, -4740216086188499584L, 4485716352083334272L, -4744641797867389852L, 4473999634395457376L, -4742081926498540150L, 4492325750080363871L, 4487353480882271237L, -4746219969430550124L, 4483394768721842705L, 4485002955667009129L, 4488354355570253230L, 4489291328373740285L, -4739433198514021388L, 4486039362289531092L, 4489603869317250117L, 4493422630284075473L, -4738844036446426504L, 4487830173669322725L, -4741915873500138422L, 4488266838036711684L, 4492163025512113696L, 4474464758653045672L, 4484584374906530070L, 4492261365783767052L, 4491628821610081109L, -4743666255145432032L, 4490761863867262737L, 4487773582832370182L, 4490670937345944564L, -4740276147636243676L, 4493732114168622227L, 4485973750847660738L, 4480956351443401874L, 4493321991947078154L, 4488118764339063971L, -4739319236466080380L, 4492897588210511978L, 4486491508459652323L, -4746048894634152992L, 4488321791501795071L, -4752668101668435968L, 4482258154566685953L, 4485869588716200328L, 4488816470075299053L, 4493264794112314849L, 4493622289307586036L, -4743877729391006892L, 4490212554552895422L, -4739514581045101172L, 4485960102871154433L, 4488192010225456556L, 4468153078262788512L, 4482297853238834086L, 4479375970556968798L, 4490340995772870041L, 4478425727082027028L, 4493283340075891670L, -4739184368145816948L, 4477432471746205852L, 4487432116100532324L, 4489201652948648104L, 4491440712722285996L, 4491905040643752955L, 4492870089354415816L, -4741461357458890106L, 4486783298953734476L, 4483974470876098314L, -4743751817831414048L, 4490956311625808395L, 4488235358665065426L, 4492719864405533267L, 4484777776465912920L, -4742182804458241342L, -4741804370154168204L, 4487109489386484023L, -4749818480983445040L, 4470782641152544320L, 4491831906360838380L, 4490313361857118398L, 4486705862441510062L, 4487734190403102759L, 4493789541857186476L, 4492376476251782486L, -4739569471660555556L, 4483592779779981046L, 4487746184973257467L, -4738273292207170908L, 4481364621901931315L, 4481203225799944990L, 4484049149441413078L, 4489616548577067135L, 4482102076518888760L, 4482387569586809345L, 4481946328498374614L, 4492100909234379554L, -4743845830035935372L, 4492852897367381609L, 4493348288985217400L, 4490442585385639080L, 4490110926657508503L, 4488989858050598562L, 4493320981576803175L, 4487537674537319031L, 4488670246122389066L, 4489151314660875949L, 4490429264546274977L, 4492032255824540940L, 4488771044745439432L, 4491619075556548974L, 4481356473894539134L, 4476173173049616008L, 4469765541035555408L, -4740848814284866696L, 4491661946400776260L, 4460453396985703680L, 4492733119347984352L, 4486978939967925553L, 4487146764622456112L, -4744589395249508964L, -4752990346154369072L, 4493664694355272373L, 4483661973389260227L, 4487107798211112159L, 4482186754398233973L, -4771647410519041280L, 4492990921325962655L, 4490117389268737605L, 4455216821044146816L, 4490955945818776551L, 4485645907856564062L, 4493171892788962772L, 4492269622912145478L, 4492284145737418650L, -4748382771225865144L, 4477932644054109320L, -4754885623887630912L, -4747961247863239632L, 4482886952963992238L, 4489952880779384738L, 4490477537497184259L, 4493383585733304996L, -4746789466983090596L, 4493637775079788698L, 4490450688069862668L, -4755535039456803200L, 4488056416944495143L, -4750004253532810240L, 4491697124995406490L, 4484837441223576163L, 4487832795784462613L, 4487600521612581615L, 4493783869941590862L, 4474568306079285368L, 4492457186040924207L, 4491666638558780020L, 4491327313225760828L, 4488347786233700646L, -4742923281392803356L, 4492891627985151190L, 4491847209576663140L, 4483164965738036614L, 4489265345532228071L, 4490084361701825621L, 4476045234340269576L, -4774067668770541568L, -4741235968374674570L, 4492901739426377208L, 4492075557425467817L, 4456891220708536320L, 4491167447050323455L, -4746075152549385600L, 4490546872042077020L, 4487851453788176118L, 4482510617194707509L, 4486658308401349800L, 4488388409918311379L, 4490113376022845616L, 4478062428769917508L, 4459304702601832640L, 4488686587049387109L, 4478891824280456456L, 4491913221329901354L, -4740377930800393436L, 4492061272306149720L, 4490728704636524547L, -4741931459913900320L, 4493582459682426406L, 4493466141016371687L, 4493085571947381557L, 4493283257275064593L, 4481937282059412210L, 4491687709473063586L, 4485461639197004814L, 4485662156961911112L, 4491381920147281726L, 4465574660681016640L, 4490850141564787169L, 4481354519444328197L, 4489693720856915275L, -4742799149589261904L, 4481013004081834574L, 4462279217853429312L, 4491580100862826263L, 4493019576237157099L, 4485679943281028042L, 4492233826214832682L, 4482156981863178230L, 4491855372698928920L, 4473183702961737224L, 4459250931152066304L, -4739403943360613816L, 4493347243322299271L, 4491817921890141293L, -4740833651214976156L, 4478717944307802040L, 4487517903258803494L, 4490268793867920847L, -4744930608686068500L, 4487680184834161879L, 4489708391829985736L, 4491010115918058100L, -4743814484555606544L, 4488381983054554221L, 4491277559010913953L, -4749586483961786800L, 4490630645219831274L, 4477078016133427284L, 4490406853925275947L, -4749082655005394776L, 4492368277227179096L, 4493057386423116384L, 4491881288629545264L, 4491557989067175480L, 4489879290652193698L, 4490263286399235284L, 4478257935956889712L, 4491938775846033568L, 4492940930583316482L, 4472561547190051272L, 4481629669236458906L, -4745298132121208124L, 4491389964910516960L, 4480084832994774926L, -4749688257861423488L, -4744743218973296800L, 4485811426232899631L, 4487560066309128571L, 4490876386614590330L, 4492499128869325932L, -4741564231929371658L, 4492519639657766054L, 4490762402001581546L, 4491116514358539110L, -4750750120147323072L, 4487834967535340797L, 4491504686080357120L, -4745416615322907120L, -4743794669044169808L, 4491281349542821575L, 4491847635324286362L, 4489290927164749617L, -4741297462929931052L, 4493375746148237692L, 4431304946945490944L, 4493000258124723542L, 4492363082792856487L, -4738132437798223044L, -4741941503221990434L, 4484004327753072241L, 4491784561952664806L, 4492975878607098960L, 4493685714487655064L, 4493456835412243215L, -4744753423996542400L, 4474908364106698616L, -4739796847966775724L, 4493095805824362770L, 4491304554735707648L, 4491436335447459080L, 4481681063884551771L, 4493677675427297147L, 4468719205397793792L, 4485979932558481085L, -4739949811656596844L, 4483874391721807895L, 4491296639751481863L, -4746038574299999548L};

  //Generated values with parameters:
  //Count = 1000
  //values_mean = 1.000000e+08; values_stdDev = 1.000000e+08; values_ordering = Random
  private static final long generatedVar10 = 4846448545458212038L; // approx ~ 1.015789e+16
  private static final long[] generatedValues10 = {4726305093217328403L, 4728301712653950436L, 4732299636837554056L, 4725161354565719134L, 4732614497658023050L, 4733180902453502024L, -4518955170106959328L, 4711342899335350392L, 4730178250168465871L, 4732186964866654949L, -4505530167405462404L, 4726730350030026949L, 4719297119779261156L, 4707241874311085824L, 4710291344785539920L, 4719787250245366003L, 4718572294166050972L, 4720354108902189025L, 4732995929590491255L, 4716067522669513164L, 4728869598154933752L, 4732046540390491645L, 4716119610391606640L, 4731936421063223358L, -4504565184376312408L, 4729210531050679314L, 4728575877033274452L, -4501208130969796496L, 4726140755292162630L, -4499222277935724016L, 4716071137999598412L, -4503543181538790272L, 4728731564018821217L, 4729388028612520054L, 4732503612133526384L, 4732578018483641724L, 4723137628831025304L, 4726481936870989475L, 4725333961252182029L, 4733202188874214393L, 4726807989420673808L, 4728341996333775743L, 4729329697699061797L, -4506175596584532388L, 4725203479087621166L, 4729009135180134248L, 4728628980105871800L, 4727739531248506664L, -4499668944473837416L, -4521073088417234304L, 4733344441726236182L, 4723703141787036333L, 4729156345534641423L, -4524192289049968576L, 4733299969393245495L, -4502569866620148356L, 4730349976815702596L, -4500476225557881760L, 4724671808393715755L, -4507235007254579960L, 4730724285219420626L, 4726840232337226833L, 4730988902981055268L, 4724101395268142728L, -4505766110125567828L, 4732130918907827826L, 4728010937281649874L, 4726818170804007578L, 4725423527509697539L, -4506448462239182556L, -4512981044579987408L, 4726276210368655143L, 4732077247052920006L, 4722980823253458449L, 4730479275912551131L, 4729625803945110582L, -4515617535977056976L, 4715311847891681100L, 4728594737902663249L, 4732865759156626975L, 4728850579224398245L, -4500385086123083000L, 4730312805219227676L, 4725994451452977830L, 4729320428349745949L, 4711335758176901072L, 4729766393808261575L, -4506134814857464760L, 4732149129068304857L, -4506800089695920172L, 4721712052710480112L, 4732873884049523187L, 4725009165250642285L, 4730804367705852654L, -4500003163846968204L, 4726274954943157042L, 4733073036684339968L, 4730702753348069255L, -4505513984613716916L, 4724516228395319067L, 4730315241042566614L, 4726994561592888278L, 4729157076144136058L, 4726873973676774111L, 4732412443070188786L, 4726954781541881061L, -4501351845553765868L, 4730363227119117410L, 4724834079860319422L, 4730771928783792512L, 4722070495509169941L, 4719793484897149148L, 4724534628842828321L, 4732497222378863807L, 4723494814177942773L, 4729620217832513836L, 4731040594872272734L, 4731747984850205252L, 4732932885387355144L, 4717274536870020264L, 4729649982222253918L, -4499264900325836288L, 4725855298585470112L, 4727895074267770646L, 4731163559109554194L, -4501926702336426232L, -4500818826023393348L, 4720693230399163817L, 4733282726770894421L, 4725362592217479040L, -4501766645717489528L, 4724695293175459910L, 4724512508197209437L, 4730715366692931910L, 4733190665090829293L, 4733261635886578341L, -4504026044349046992L, -4512270673716571184L, 4729915732713820449L, 4712361350407392288L, 4732147697922382408L, 4731529507812330617L, -4501675402697909088L, 4728991344768250416L, 4718237858461606944L, 4711506710555742024L, 4731857229696776677L, 4731957624591648038L, 4724849416821602863L, 4724976423778164827L, 4730371741007446687L, -4498719069112722866L, 4730482368739928188L, -4507804485923660200L, -4499756020976593588L, 4728815590050136560L, 4715949750905120552L, 4731713331566065538L, -4498943132235105222L, 4713015205368227136L, 4731482386556608230L, 4728911731921153918L, 4729966733935947364L, -4501573731829108960L, 4731916547016984606L, 4731228036630744113L, 4729628371770817684L, 4727940012697991177L, -4501025331316315492L, 4724361464967222790L, 4719912413187042625L, 4728802628560633895L, 4728690028650188193L, 4728888966195467114L, 4730715011342930672L, -4499172205810057544L, -4505932350903780340L, 4728024163943326150L, -4509686785962507376L, -4505553020810899600L, -4509132363132521856L, 4730204036614738978L, 4733308319895586810L, -4502941165510239776L, 4728868625702261640L, 4719854657234628760L, 4731222536277349701L, 4732142306787731450L, 4729071251869892219L, 4723375946849493690L, 4723081979570656746L, 4721831126747907926L, 4728538704455260151L, 4729604403410356179L, 4733348512662808182L, 4723257404021123648L, 4731575156390945216L, 4725201120513853052L, 4733262867375845469L, 4732356252540239903L, 4729790504086692244L, -4502891662614057536L, 4714574089315798656L, 4720029069748083110L, 4723666086184754233L, 4726962438133777538L, 4721209752291080687L, 4729519292481351139L, 4732474823848433275L, -4499068188203523928L, -4503386671528865612L, -4503116479962487160L, -4499531399225117440L, 4733186012520658215L, 4729039007274601292L, 4722025987036420851L, -4502832264718536540L, 4716500421628478668L, 4727105213973279817L, 4711518103290685160L, 4732074135123043540L, 4725550059664392879L, -4506977423211775200L, 4732816760928576973L, 4729266865676811733L, -4502903867526572352L, 4729910437846651606L, 4729745124175653456L, 4732913234694436837L, -4507907659477137300L, 4727955424243981502L, 4715510241446319636L, 4729256845411335830L, 4726506910917389224L, 4726362036221708583L, 4727231644108283258L, -4501034660851827824L, 4724444088088127892L, 4725004275212699335L, -4524465137362330304L, -4499666925223866584L, 4725965636193223871L, 4732061964918091450L, 4699181741923521664L, 4722605554018964150L, -4500861402372010876L, 4731736195150016660L, 4727149340595273696L, 4731105631641233996L, 4724461811357421091L, 4715645366442866864L, 4727142052554758652L, 4729910675042656903L, 4720134163943328543L, 4732147089315649604L, 4724526127308729171L, -4516730080718400896L, 4728332553182433782L, 4717044217540168488L, 4732049319904499674L, 4730420753373455580L, 4722993973515757658L, 4729361575095442210L, -4514573915366545520L, 4713980519128260200L, -4499291439929147084L, -4519869447552575136L, 4730520320403544692L, 4717992403558097476L, 4733327680072889084L, -4503449262096153152L, 4714294602493828040L, 4733232156059356596L, -4501610811516595628L, 4732614948547887121L, 4731661066553496808L, 4732660262905367131L, 4715977918141746400L, 4730526311403336676L, 4728898264375603782L, -4505274040769509684L, 4725088382503120877L, 4725861030844069868L, 4730946907988441597L, 4721700364247655342L, 4695915044456972672L, 4731389786093651086L, 4721704424224969642L, 4718398271319223804L, 4722576571587817051L, 4729727510547606551L, 4732120891450220115L, 4731993045236125538L, 4718328695634556004L, 4729102611173138538L, -4499531033830553932L, -4505259736704409984L, 4732918604276374397L, 4732535992051818696L, 4732660967133897372L, 4729540419991549198L, -4499034412905968440L, -4500739666054591528L, -4502282018909593892L, 4724830502078474712L, -4508429240284916192L, 4728932834654211460L, 4719377437174116896L, 4732718013763675155L, -4514395538788855744L, 4731098616441468266L, 4727375850519262616L, 4723779259326967805L, 4713762533914721024L, 4730890336198211965L, 4732049491936246748L, -4498745962704438342L, -4501438686602703884L, 4726736597973297922L, 4729745934140652626L, 4730599594841544000L, 4730145108046443402L, 4728533944817093668L, 4708165150694749936L, -4501538263839940724L, 4727330134008430761L, 4727321908230349391L, 4731076736554666396L, 4731361731552611062L, -4501948647876081036L, 4731152805681356945L, -4510608209341525272L, 4722455309635985272L, -4503066416160704756L, -4506029006506219780L, 4712169746545098952L, 4718267890530402660L, 4728998693922935276L, -4504532634231898692L, 4728928870169586018L, 4728812620468115862L, 4732814525704522248L, 4731956521022384904L, 4731348383733118063L, -4502380021003944928L, 4731295450427802446L, 4725998437639722667L, 4732314808314908871L, 4727075680706552932L, 4731184354111786964L, 4721776831779890786L, -4508488858057910176L, -4503329599402000408L, 4713499899675853336L, 4720264270780287861L, 4729560786683282338L, 4720882463017558798L, -4523364754188876352L, -4508504468901604448L, 4701931038417448640L, 4731366020875960966L, 4732642092362272074L, 4729358514508786140L, 4720750937497253178L, 4732522190998065154L, -4507585634042327392L, 4733132511644497548L, 4727932616645327803L, 4716723187115250396L, -4502578311912791528L, 4727218415661154052L, -4503594662215605818L, 4733115155405527758L, 4725626862685371231L, 4698258169110508864L, -4500102626416676492L, 4726870979538767576L, 4721270855459332413L, -4502834364422805620L, 4715449533142261564L, 4731425308473683909L, 4733291045051613187L, 4716546999064756772L, 4712703577375756496L, 4731865367050957154L, 4730756138049989959L, 4731237408404269077L, 4731075393636465660L, 4717493496045888292L, 4718972234167752348L, -4503003365750249544L, 4729089784392428476L, -4514564344706242448L, 4733042154480684632L, 4728684539108102844L, 4733291344879618261L, 4724589712152136188L, 4732941583974366324L, 4731454227322879858L, 4733284979226929914L, -4501513881955074716L, -4499939220894389260L, 4730278822617265246L, 4692824833467869568L, 4730434922315851697L, 4714640590991252544L, -4512602461488981192L, -4529579548221865600L, 4728555769139112372L, 4720933626177913864L, 4722805517474026349L, 4727878475048446544L, 4726304269546232752L, 4726553488624881542L, 4718210478250638028L, 4730671237941805036L, 4723412951898236061L, 4729385428296567013L, -4503109271095077184L, -4502765329791804812L, 4729569415338893712L, 4726365621281750444L, -4511792245097698456L, 4730014456978678792L, 4720139539510872021L, -4502930058436657232L, 4720634164086467192L, -4511868994599557104L, 4725594294745969363L, -4513170484948065216L, -4505600408644133240L, -4503325921946420604L, -4501884383258776864L, 4719367580241893560L, 4719818374642874337L, 4731260970226674716L, 4729521469283417419L, -4499391843343407232L, 4728091940371811969L, 4729810576915484828L, 4727099368752222644L, -4503452032068640904L, 4724705468391376939L, -4499258873386755356L, -4507705035682830584L, 4718042601423705068L, 4724972494597893972L, 4727593794302013646L, 4728835822475577193L, 4729079544052062128L, -4505320556907907988L, 4718949877451082104L, 4730187405078096060L, 4733298395822006716L, 4728997740765043732L, -4501483891435492800L, 4732966534076401184L, 4731924911655099796L, -4504006424022425668L, 4733302988471331582L, 4731776687018325205L, 4729211827444808457L, 4732868453517683321L, 4727040884939080473L, -4501622875923477716L, -4502709513306380428L, -4500785169638145668L, 4722830343320655407L, 4731219959056982630L, -4499689080116681632L, 4719358274865112792L, 4731595539687661086L, 4729847579034522270L, 4732963821579759652L, 4733313446911300878L, 4732240777504376733L, -4504182486045080756L, 4720177157641855374L, 4729843156516093764L, -4501134683148199292L, 4730886966512888124L, 4725275922344335733L, -4504186709452355856L, 4719498439610694964L, 4731367156840136594L, -4510826941073444424L, 4730347330021161398L, 4732534607214584016L, 4717302940033723952L, -4521106013226729568L, 4731833836372102962L, 4731173594608355366L, -4500922534568195804L, 4733125177342602991L, 4729271772690994895L, 4729290517773799968L, 4726460386772718383L, 4708442267784290528L, 4731820252013097456L, 4731299090092804956L, 4720980997630103911L, 4711041877158487808L, 4710449823949013136L, 4725920903016972200L, 4723202466795483865L, 4729869686341568791L, 4708848064177041840L, 4721220518544890629L, 4722993616395661104L, 4718904413429769472L, 4724334458395080250L, 4731973010407763584L, -4507381551248060312L, 4731303797949329352L, 4730188825435393738L, 4729257065542004527L, -4499790399622588204L, -4503188067813232056L, 4731783181691994170L, 4721502913593500434L, 4731829445027090819L, -4499197034956299432L, 4721263467351741677L, 4733041063738471479L, 4719204062819006968L, -4500706162346933276L, 4729169326841673021L, 4732747969995332321L, 4731525372973880647L, 4717990434567771176L, -4503059186909837984L, 4726106973355914073L, -4512554996101083856L, 4726626098293537792L, -4502385687161117760L, 4733139233330493550L, 4723464931070723813L, 4728421276785473204L, 4728595062386986110L, -4500312147705296636L, 4727554797449722376L, -4507237753073866032L, 4726745833480406614L, 4730595683723122394L, 4730774014748839452L, 4725796148419161655L, 4730862963793853430L, 4720034513827313374L, 4731037496636796125L, 4731747778966199442L, 4732637572559367113L, 4729917204939191814L, 4715719242117129448L, 4732360413289480183L, 4731334567269812784L, -4500818828682150112L, 4718082038506584116L, 4730368360729684682L, 4728876921031197184L, 4724714758075901298L, 4732249336635559374L, 4732109400691126776L, 4730634692818334050L, 4731393778021124058L, 4731974983513788035L, 4729736740976210105L, 4727256023151434750L, 4724119128950348522L, 4728130652564976475L, 4733287797698129667L, -4506262697051914908L, 4731644905133273288L, 4729773543454377676L, 4731064674107372080L, 4730379025800664277L, 4727876621191796728L, 4720819459840905622L, 4728835196646329390L, 4726483910313805633L, -4501990205475839376L, 4731236686455494364L, -4501344687019388944L, 4716100057796554928L, 4731168017630067336L, 4729242056807131494L, 4729781876586568815L, 4732237134959178941L, 4726269606943719869L, 4729739376470060866L, 4731973265090122624L, 4725251690098462149L, -4506448726866688984L, 4727363898578386426L, 4726788527310213474L, 4726174270938809437L, 4707157981082677904L, 4720091770821278220L, 4732903637243219637L, 4722827336938326456L, -4507659079306865008L, 4729188655223113972L, 4730885211381669607L, 4726144332080712836L, 4725701356768730811L, 4713969657432795120L, -4504482380817914020L, 4728871540096657850L, 4728905883652530064L, 4723395655883137209L, 4733143776782036927L, 4730319682447457740L, 4729726640811315377L, 4701187947220836096L, 4729040153117711731L, 4726187787745294331L, 4728424247002315485L, 4730642359279784388L, 4710791498600410856L, 4715276637890429432L, -4507240471108658712L, -4502708986410360588L, -4500426935057361632L, 4728769197410421187L, -4498883693564272702L, -4500192386594668016L, 4729620839832489171L, 4733262625120080688L, -4508976912460818112L, 4711887766785003536L, -4501734327588522276L, 4726391198558495304L, 4730923311292800293L, 4728795385996002238L, 4733294259382620114L, 4727916296186941928L, -4498900758536585034L, -4501948979459521036L, 4698445140822946240L, -4502945458588102236L, 4725687933913524376L, 4725054952151216660L, 4731624219885317630L, -4499762381647707920L, 4730711517715883054L, 4731899130988270176L, 4728008207532289212L, -4517298504244091648L, 4726784415397669389L, -4500805249103039612L, -4507305507638659764L, -4506747925867711540L, 4725699794766804436L, 4724982296909367439L, 4728020826907027249L, 4715719082254790684L, 4723127867692109360L, 4730333229909301558L, 4728932451435116877L, 4728084892016099058L, -4503089132899767068L, 4731930339249852782L, 4732104343756186981L, 4726988682937672034L, 4720172508461955383L, 4685822354812346368L, 4729302500653593267L, 4725872000089709040L, 4720395834398898015L, 4727187334869848305L, 4729191544104878659L, -4504685892813560368L, -4504642315733716472L, 4723792463937932390L, 4732942298684822196L, -4503080045053633504L, 4731223623890358228L, 4717849306000818132L, -4499612374689429980L, 4731563418937552190L, 4731560094847594642L, 4732514546294277050L, -4501796481369679732L, 4722477493779664432L, -4501668342805227188L, 4730709333600737101L, 4733295518568878492L, 4731340147347473870L, -4502256411840087728L, 4731039112374754254L, 4732404932509538689L, 4730590412331363900L, 4731066111539602225L, -4503709171352336868L, 4730348488406051810L, 4707104488261987344L, 4722420000590082751L, 4726188230147429374L, -4502513096874636488L, 4727022397323209433L, 4722856512731873714L, 4728769440796304424L, 4717580961182703916L, 4720018313590675926L, -4500809629353056904L, -4504089651260153460L, -4498715476514652984L, -4499938035115248344L, -4506721850046402004L, 4727356568808462904L, 4730128673760250561L, 4733186585364043275L, 4731844387682494154L, -4498957297122414594L, 4728997129671705154L, 4724624964369798280L, 4724121887975971410L, 4711125091220211360L, 4730516798791953360L, 4726016138429714687L, 4726541721468508361L, 4726080157734681096L, 4726122522993728238L, 4732581493624141155L, 4728441941828240181L, 4726005186184562976L, 4730444946547802266L, 4732464919864173374L, 4724213699206136364L, 4724059292149495068L, 4731693896825925671L, 4726117021282195419L, -4500443850758879736L, -4501620148109734268L, 4728867131014492223L, 4732250183636813624L, 4726945672744843492L, 4732665070013572306L, 4730059541872803404L, 4714679411279128864L, 4728443288936103990L, 4732094083101186740L, -4506123158564204108L, 4730496568715276211L, 4732926087464365927L, 4714904536149623664L, 4725572088781481328L, 4730642307496445438L, 4723568788844967743L, 4729404334593592666L, 4711832364678756040L, 4731407701501443482L, 4731031980345754826L, 4717773732825500320L, 4724305884104615522L, 4730264034982210470L, -4502645737436702456L, -4516808621078263440L, -4514872752383831456L, 4726820358693779375L, 4733296680944651947L, -4508372831328843432L, 4724889269575148135L, -4500710833548696920L, 4709969420702388832L, 4724973741952209724L, -4500116959858941196L, 4731950098528400750L, 4717791230461107236L, -4500487335525795892L, 4729541717108542429L, 4725557917578511818L, 4730079163671565964L, 4731147342596059726L, -4501147086376406156L, 4727107480048384476L, 4730184806989672120L, 4724617267089005244L, -4505035275753103228L, 4726211793900352797L, 4724339594442087571L, -4530946116638438656L, 4732724999839581436L, -4501384493002566492L, 4731132411525037811L, 4731454395361213736L, 4729938051820818482L, 4731740587239151268L, 4732090951095659246L, 4720223691339154953L, 4733332002922909682L, 4720532835457908831L, 4732324203371751581L, 4715574964196739372L, -4498876708257845374L, 4726900966139247830L, 4730856281940503674L, 4714119974562894088L, 4719804251198983131L, -4501929686806537816L, -4505501893731287260L, -4501640288858348176L, -4502075718640115520L, 4722222058540713159L, -4506887820471273356L, 4730806030648192838L, -4498716974169668726L, 4732035499462606780L, 4725060042501079842L, 4726072004077965723L, 4730318822401405608L, 4717565730901674296L, 4725585451052588537L, 4726559144053007047L, 4729272619025793211L, -4507466653702795696L, -4499659743045759696L, 4731470563485560030L, -4504749350950886188L, 4720012555324126706L, 4729930706588635016L, 4726517648798724446L, 4729293032906301214L, 4728890444176632325L, 4723597591653449939L, 4708656900610695280L, 4720726652125928464L, 4731631953497164118L, 4724339763438159362L, 4728793184017798099L, 4731612755938730626L, 4731352762511294310L, 4725043928573914593L, 4731194919253522321L, -4502375605753881284L, 4733040238101097481L, 4731376053510791491L, -4502864252386476316L, -4501550117487123956L, 4724281054506557978L, 4731857289882095188L, 4720855117003906803L, 4724949891134549626L, 4731634023193938184L, 4733288133801142218L, 4730570202117561228L, 4732806786104825930L, 4731714930464682803L, 4716008404470211520L, 4724664919826023428L, 4726199366161573562L, 4720850601168850362L, 4730454925222813542L, 4732434437571427975L, -4510237860437832288L, 4727139267043271630L, -4501591979362579744L, 4731255733721404650L, 4730476526366316209L, 4733096617491920377L, 4731025711909914961L, 4730727378542326585L, 4730141767647658841L, 4723070804458872050L, 4725685265568606590L, 4718808063636445228L, 4731985879324105124L, 4728987736242191774L, 4730273467776346620L, 4725150376039565439L, 4726887510826389028L, 4715436448942597344L, 4722861091446710383L, 4720239569550625914L, -4500700245969416952L, -4511343883471156704L, 4714465305137861168L, 4727861125726415503L, 4721892848730016015L, 4727683983250852271L, 4721708043220043507L, 4728379572300280691L, 4703962228353086944L, 4724284575800701357L, 4727187983664508524L, 4726972152637426088L, 4727668621655357013L, 4725827501952105136L, 4730417368859016051L, 4733319395803348658L, 4728850986644072755L, 4715699846325660476L, 4731291378563076338L, -4499232435928837752L, 4726640122787460035L, 4721264193541122072L, 4727381319781531035L, 4712934253699303248L, 4709438212534265968L, 4729098344145664450L, -4501102096500933960L, 4727634360988556052L, 4726467215142123947L, -4500818091282018152L, 4733263300640370804L, 4733197442377263483L, 4726285132294158990L, -4502161586323806852L, -4503257405554703308L, 4731331632951858384L, -4508900022684301024L, 4730320512063965266L, -4501015352319216000L, -4507330119498258048L, 4721292240402374804L, 4717416707471389928L, 4727921404336127106L, 4731934358780472035L, 4731701481860450906L, 4732090937922553982L, 4733234062924484062L, 4731235762442429537L, 4727143967439824833L, -4506206204903228520L, 4728165583114501777L, 4725103356924420698L, 4729043935895273725L, 4715129055815331376L, 4724996252667422196L, 4727688567048361337L, 4732135794823539195L, 4732293753571736960L, 4729953045931144096L, 4730569979995735958L, 4724952444380988294L, 4725269271642203001L, 4732719648630151784L, -4511227112069877456L, 4719298499149450896L, 4723973713969550187L, 4731437208452190514L, 4723611075012794973L, 4717680841582821608L, 4729283510633900679L, -4503479367945631648L, 4724583034361092960L, 4721133830953221585L, -4503302050608640420L, 4725608150105259222L, 4731004285042885590L, -4500037195983320344L, 4716313740248790024L, 4730091317323936907L, 4728855743527375623L, 4725496202246698953L, 4723632209472722589L, 4701898134961873216L, 4712872717073993952L, 4726880508595434415L, 4725473181066704826L, -4507050795053389464L, 4725735421463018087L, 4728791690770165386L, 4706926921653289072L, 4733253963224653430L, -4513864434470575936L, 4729097137210825138L, 4733159234089934752L, -4503039279438943016L, 4731381929464254406L, 4723031696134126780L, 4730457625319571310L, 4727918052374087244L, 4723967970932386208L, 4733104378719400422L, -4512630305282382000L, -4510568905927747584L, 4719010883961407408L, 4733059046030084288L, 4728551618565994908L, 4733246281296050264L, 4718168291582448888L, 4730897937468886088L, 4731412751216185265L, 4729065652574605374L, -4508737336241168672L, 4720367504881214349L, 4724811748029214810L, 4707474077973401312L, 4721746116379578110L, 4732989932645069085L, 4718169648049786588L, 4724679596307199879L, -4507368095516513328L, 4730360248212051586L, 4716883879700727936L, 4732784825277057175L, 4731083349587396282L, 4733222826793052638L, 4729323247362338780L, -4499441158581931156L};

}
