/*  Pasang Emas. Enjoy a unique traditional game of Brunei.
    Copyright (C) 2010  Nor Jaidi Tuah

    This file is part of Pasang Emas.
      
    Pasang Emas is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
namespace Pasang {

enum Mode {
    USER_VS_AI,
    USER_VS_USER,
    AI_VS_AI,
    NETWORK
}

/**
 * This is the main content of the window rather than the window.
 */
class Window : Gtk.Stack {
    private Gtk.ApplicationWindow main_window;
    private Gtk.HeaderBar    header_bar;
    private Gtk.Widget       theme_menu_button;
    private Gtk.Widget       go_back_button;
    private Gtk.Widget       pattern_go_back_button;
    private GameSeries       game = new GameSeries ();
    private GameMenu         game_menu;
    private GameView         game_view;
    private PatternSelector  pattern_selector = new PatternSelector ();
    private PatternEditor    pattern_editor;
    private Brain            brain = new Brain ();
    private Client           client;

    /**
     * mode is public to enable notify
     */
    public Mode mode {get; set;}

    /**
     * Virtual players when mode == Mode.AI_VS_AI.
     */
    private BrainSpec demo_player = BrainSpec ();

    /**
     * Ticket from Brain.request_think. Used to ensure we don't submit a move from aborted thinking.
     */
    private long request_ticket;

    /**
     * Ticket for async methods to effect delay in demo. Demo is aborted when the
     * ticket is no longer valid.
     */
    private int demo_ticket = 0;

    public Window (Gtk.ApplicationWindow main_window) {
        this.main_window = main_window;
        main_window.delete_event.connect (() => {
            on_quit ();
            return false;
        });
        game_view = new GameView (game);
        client = new Client (main_window);
        game_menu = new GameMenu (client);
        create_header_bar ();
        pattern_editor = new PatternEditor (pattern_selector);

        // Contents
        set_transition_type (Gtk.StackTransitionType.SLIDE_LEFT_RIGHT);
        set_transition_duration (500);
        add (game_menu);
        add (pattern_selector);
        add (pattern_editor);
        add (game_view);
        game_view.set_hexpand (true);
        game_view.set_vexpand (true);

        realize.connect ((s) => {
            // Internal signals
            pattern_selector.pattern_changed_signal.connect (on_pattern_changed);
            brain.got_move_signal.connect ((response_ticket, response) => {
                Idle.add_full (Priority.HIGH, () => {  // Commit the response within the gui thread
                    if (request_ticket == response_ticket)    // Dismiss move from aborted thinking
                        game_view.simulate (response);
                    return false;
                });
            });
            game_view.next_move_signal.connect ((s, move) => {
                if (game.stage == Stage.OPENING && move == null) {
                    // The user decides to let the machine move first
                    stop_thinking ();
                    game_view.set_turn (Player.AI, Player.HUMAN, 0);
                }
                // When a null move is made when the game is over (by simply touching
                // the screen), we should start a new round of the game
                prepare_for_next_move (game.stage == Stage.GAME_OVER && move == null);
            });

            // Defaults
            show_game_menu ();
            on_status_changed ();
            client.update_status ();
        });//endconnect

        // Update to be done whenever the interaction moves from one page to another
        // or then the mode changes
        notify["visible-child"].connect (on_status_changed);
        notify["mode"].connect (on_status_changed);

        // Network game signals
        client.on_join_game.connect (join_network_game);
        client.on_leave_game.connect (leave_network_game);
        client.on_move_received.connect (receive_network_move);
        game_view.next_move_signal.connect (send_network_move);

        // Game menu signals (Yucky!)
        game_menu.activated.connect ((n) => {
            if (n == 0)      { on_newgame_ai (); }
            else if (n == 1) { on_newgame_2p (); }
            else if (n == 2) { on_demo (); }
        });
    }

    /**
     * Join network game
     */
    private void join_network_game () {
        mode = Mode.NETWORK;
        game.start_new_series ();
        stop_thinking ();
        pattern_selector.set_sensitive (true);
        visible_child = pattern_selector;
    }

    /**
     * Leave network game
     */
    private void leave_network_game () {
        if (mode == Mode.NETWORK) show_game_menu ();
    }

    /**
     * Send move to server, if appropriate
     */
    private void send_network_move (Move? move) {
        if (move == null) return;  // This happens when the user clicks on "Game over" board
        if (mode == Mode.NETWORK && game_view.player_type == Player.REMOTE) {
            // The player to play next is REMOTE. So, the move that has
            // just been performed must have been done by the local player.
            if (game.stage == Stage.OPENING) {  // We are just starting...
                string starting_pattern = game.starting_pattern ();
                client.send_move (0, starting_pattern);    // ... so, send the opening pattern as well
            }
            client.send_move (game.seq, move.notation);
        }
    }

    /**
     * Receive move from remote opponent
     */
    private void receive_network_move (int seq, string move_notation) {
        if (seq == 0) {  // Starting pattern
            if (Game.is_valid_pattern (move_notation)) {
                pattern_selector.find_and_click (move_notation);
                pattern_selector.set_sensitive (false);
            }
            else {
                message ("Invalid pattern received: %s", move_notation);
            }
        }
        else {
            // Check the sequence number
            if (seq != game.seq + 1) {
                message ("Invalid move sequence number: %d. Expected: %d", seq, game.seq + 1);
                return;
            }
            // Check if the remote player is making the very first move in the game
            if (game.seq == 0) {
                game_view.set_turn (Player.REMOTE, Player.HUMAN, 0);
                prepare_for_next_move ();
            }
            // Ensure that the move is valid
            var move = game.get_move (move_notation);
            if (move == null) {
                message ("Invalid move received: %s", move_notation);
                return;
            }
            game_view.simulate (move);
        }   
    }

    /**
     * Abort any current thinking and ensure that result from aborted
     * thinking is dismissed (by setting request_ticket to an impossible value).
     */
    private void stop_thinking () {
        request_ticket = -1;
        brain.request_stop ();
    }

    private void prepare_for_next_move (bool next_round = false) {
        Stage stage = game.stage;
        if (next_round) {
            visible_child = pattern_selector;
        }
        else {
            game.update_num_wins ();
            game_view.queue_draw ();
        }
        // Ready for next round
        if (stage == Stage.GAME_OVER) {
            pattern_selector.set_sensitive (mode != Mode.AI_VS_AI);
            pattern_selector.unselect ();
            if (mode == Mode.AI_VS_AI) {
                end_demo_round.begin ();
            }
            return;
        }
        if (game_view.player_type == Player.AI) {
            //game_view.theme.random (); // jaidi : uncomment this for stress test
            brain.request_think (
                game,
                mode == Mode.AI_VS_AI ? demo_player : game_menu.brain_setter.brain_spec,
                out request_ticket);
        }
    }

    public void on_pattern_changed (string? starting_pattern) {
        if (starting_pattern == null) {
            visible_child = pattern_editor;
            return;
        }
        if (mode == Mode.USER_VS_AI) {
            game_view.set_turn (Player.AI, Player.HUMAN, 1);
        }
        if (mode == Mode.USER_VS_USER) {
            game_view.set_turn (Player.HUMAN, Player.HUMAN, 1);
        }
        else if (mode == Mode.NETWORK) {
            game_view.set_turn (Player.REMOTE, Player.HUMAN, 1);
        }
        stop_thinking ();
        game_view.start_game (starting_pattern);
        if (mode != Mode.AI_VS_AI) {
            // For AI_VS_AI, the following is done in start_demo_round
            visible_child = game_view;
            prepare_for_next_move ();
        }
    }

    /**
     * Do the following, aborting whenever the ticket becomes invalid
     * or the play mode changes to non-demo :
     *      Get a ticket
     *      Show patterns
     *      Wait
     *      Show selected pattern
     *      Wait
     *      Show initial game board
     *      Wait
     *      Really start
     */
    private async void start_demo_round () {
        var my_demo_ticket = ++demo_ticket;
        game_view.set_turn (Player.AI, Player.AI, game.num_rounds % 2);  // Alternate first player
        pattern_selector.set_sensitive (false);
        pattern_selector.random ();
        visible_child = pattern_selector;

        yield Util.wait_async (1000);
        if (my_demo_ticket != demo_ticket || mode != Mode.AI_VS_AI) return;

        pattern_selector.highlight_selected ();

        yield Util.wait_async (3000);
        if (my_demo_ticket != demo_ticket || mode != Mode.AI_VS_AI) return;

        // Change the theme when the board is still hidden
        game_view.theme_switch.random ();
        visible_child = game_view;

        yield Util.wait_async (4000);
        if (my_demo_ticket != demo_ticket || mode != Mode.AI_VS_AI) return;

        prepare_for_next_move ();
    }

    private async void end_demo_round () {
        var my_demo_ticket = ++demo_ticket;
        yield Util.wait_async (4000);
        if (my_demo_ticket != demo_ticket || mode != Mode.AI_VS_AI) return;
        yield start_demo_round ();
    }

    /**
     * Create and populate a header bar
     */
    private void create_header_bar () {
        header_bar = new Gtk.HeaderBar ();
        main_window.set_titlebar (header_bar);
        header_bar.show_close_button = true;
        header_bar.pack_end (create_primary_menu ());
        header_bar.pack_end (create_theme_menu_button ());
        header_bar.pack_start (create_pattern_go_back_button ());
        header_bar.pack_start (create_go_back_button ());
    }

    /**
     * Create primary menu
     */
    private Gtk.Widget create_primary_menu () {
        var button = new Gtk.MenuButton ();
        button.image = new Gtk.Image.from_icon_name ("open-menu", Gtk.IconSize.BUTTON);
        button.relief = Gtk.ReliefStyle.NONE;
        button.menu_model = new MenuMaker (main_window)
            .add ("Help",   _("_Help"),               on_help,  "F1")
            .add ("About",  _("_About Pasang Emas"),  on_about)
        .end ();
        return button;
   }

    /**
     * Attach theme_switch to a MenuButton
     */
    private Gtk.Widget create_theme_menu_button () {
        var button = new Gtk.MenuButton ();
        // Note: icon_name is listed in
        // http://standards.freedesktop.org/icon-naming-spec/icon-naming-spec-latest.html
        button.image = new Gtk.Image.from_icon_name ("preferences-color", Gtk.IconSize.BUTTON);
        button.relief = Gtk.ReliefStyle.NONE;
        button.tooltip_text = _("Choose theme");
        var popover = new Gtk.Popover (button);
        popover.add (game_view.theme_switch);
        button.popover = popover;
        return theme_menu_button = button;
    }

    /**
     * Create a go-back button to apply or ignore pattern edit
     */
    private Gtk.Widget create_pattern_go_back_button () {
        var button = new Gtk.MenuButton ();
        button.image = new Gtk.Image.from_icon_name ("go-previous", Gtk.IconSize.BUTTON);
        button.relief = Gtk.ReliefStyle.NONE;
        button.menu_model = new MenuMaker (main_window)
            .add ("UndoPattern",  _("Ignore Changes"),  on_undo_pattern)
            .add ("SavePattern",  _("Apply Changes"),   on_save_pattern)
        .end ();
        return pattern_go_back_button = button;
    }

    /**
     * Create a go-back button
     */
    private Gtk.Widget create_go_back_button () {
        var button = new Gtk.MenuButton ();
        button.image = new Gtk.Image.from_icon_name ("go-previous", Gtk.IconSize.BUTTON);
        button.relief = Gtk.ReliefStyle.NONE;
        button.menu_model = new MenuMaker (main_window)
            .add ("Undo",      _("_Undo Move"),    on_undo,      "<control>Z")
            .add ("Retreat",   _("Back to Menu"),  on_retreat)
        .end ();
        return go_back_button = button;
    }

    /**
     * Things to do when visible_child changes, or game mode changes
     */
    private void on_status_changed () {
        // Actions
        set_action_enabled ("Undo", mode == Mode.USER_VS_AI && visible_child == game_view);
        // Header bar title
        header_bar.title =
            visible_child == game_menu ? _(Config.PACKAGE_NAME) :
            mode == Mode.USER_VS_AI ?   _("Playing Against Machine") :
            mode == Mode.USER_VS_USER ? _("Playing Face to Face") :
            mode == Mode.AI_VS_AI ?     _("Demo") :
            mode == Mode.NETWORK ?      _("Playing Online") : "?? if you see this, complain ???";
        header_bar.subtitle = mode == Mode.NETWORK ? client.opponent_name : null;
        // Header bar buttons
        theme_menu_button.visible = visible_child == game_view;
        go_back_button.visible = (visible_child != game_menu && visible_child != pattern_editor);
        pattern_go_back_button.visible = visible_child == pattern_editor;
    }

    /**
     * Functions performed when some user Action is triggered
     */
    private void show_game_menu () {
        visible_child = game_menu;
        stop_thinking ();
        game_view.stop_simulation ();
        // Stop demo by changing mode
        mode = Mode.USER_VS_USER;
    }

    private void on_newgame_ai () {
        mode = Mode.USER_VS_AI;
        init_game ();
    }

    private void on_newgame_2p () {
        mode = Mode.USER_VS_USER;
        init_game ();
    }

    private void init_game () {
        game.start_new_series ();
        stop_thinking ();
        pattern_selector.set_sensitive (true);
        visible_child = pattern_selector;
    }

    private void on_demo () {
        game.start_new_series ();
        stop_thinking ();
        game_view.stop_simulation ();
        mode = Mode.AI_VS_AI;
        start_demo_round.begin ();
    }

    private void on_undo () {
        stop_thinking ();
        if (mode != Mode.USER_VS_AI) {
            stderr.printf ("Unexpected event: 'Undo'\n");
            return;
        }
        game_view.stop_simulation ();
        if (game.player == 0 && game.stage == Stage.OPENING) {
            // Undo to game start
            game_view.set_turn (Player.AI, Player.HUMAN, 1);
            pattern_selector.set_sensitive (true);
            visible_child = pattern_selector;
        }
        else {
            game.undo ();
            if (game_view.player_type == Player.AI) game.undo ();  // Again!
            game_view.hang_message ();
            prepare_for_next_move ();
        }
    }

    private void on_retreat () {
        if (mode == Mode.NETWORK) {
            client.request_retreat ();
        }
        show_game_menu ();
    }

    private void on_undo_pattern () {
        visible_child = pattern_selector;
    }

    private void on_save_pattern () {
        pattern_editor.apply ();
        visible_child = pattern_selector;
    }

    private void on_quit () {
        client.request_logout ();
        stop_thinking ();
        main_window.hide ();
        main_window.destroy ();
    }

    private void on_help () {
        //var schemes = Vfs.get_default () .get_supported_uri_schemes ();
        //foreach (var s in schemes) {
        //    print ("---- %s\n", s);
        //}
        var app = AppInfo.get_default_for_uri_scheme ("help");
        var uri = "help:pasang-emas";
        if (app == null) {
            print ("No support for \"help\" uri. Falling back to \"http\".\n");
            app = AppInfo.get_default_for_uri_scheme ("http");
            uri = "http://pasang-emas.sourceforge.net/how-to-play/index.xhtml";
        }
        try {
            print ("Showing help using %s\n", app.get_commandline ());
            Gtk.show_uri_on_window (main_window, uri, Gdk.CURRENT_TIME);
        }
        catch (Error e) {
            stderr.printf ("Error: %s \n", e.message);
        }
    }

    private void on_about () {
        Gdk.Pixbuf logo = null;
        try {
            var file = Reloc.flash_dir + "/pasang-emas-flash.png";
            logo = new Gdk.Pixbuf.from_file (file);
        }
        catch (Error e) {
            stderr.printf ("Cannot load logo: %s\n", e.message);
        }
        Gtk.show_about_dialog (main_window,
            "program-name",  _(Config.PACKAGE_NAME),
            "version", Config.PACKAGE_VERSION,
            "comments", _("A traditional board game of Brunei"),
            "copyright", "Copyright (C) 2008-2019 Nor Jaidi Tuah",
            "logo", logo,
            "authors", new string[]{
                        "Nor Jaidi Tuah <norjaidi.tuah@ubd.edu.bn> (programmer)",
                        "Mohd Abdoh bin Haji Awang Damit (initiator)", null },
            "artists", new string[]{"Nor Jaidi Tuah <norjaidi.tuah@ubd.edu.bn>", null},
            "translator_credits", _("translator_credits"),
            "license-type", Gtk.License.GPL_3_0,
            "website", "http://pasang-emas.sourceforge.net/",
            "website-label", "pasang-emas.sourceforge.net"
        );
    }

    /**
     * A helper method to call GLib.SimpleAction.set_enabled
     */
    private void set_action_enabled (string action, bool enabled) {
        var simple_action = (main_window. lookup_action (action) as GLib.SimpleAction);
        if (simple_action != null) simple_action.set_enabled (enabled);
    }
}//class

}//namespace


// vim: tabstop=4: expandtab: textwidth=100: autoindent:
